/*
 * Copyright (c) 1989 by Apollo Computer, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The authors
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the authors be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Copyright (c) 1989 Carnegie-Mellon University
 * Copyright (c) 1988 Carnegie-Mellon University
 * Copyright (c) 1987 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 *
 *
 * Bug reports/fixes, comments and enhancements should be sent to:
 *       nawaf@osf.org
 *       Nawaf Bitar
 *       Open Software Foundation
 *       11 Cambridge Center
 *       Cambridge, MA 02142
 *
 * Contributors:  Nawaf Bitar
 *		  Tom Lemaire
 *		  Vidas Neverauskas
 *		  Ed Sharpe
 *
 */


/*
 *	Manages physical address maps.
 *
 *	In addition to hardware address maps, this
 *	module is called upon to provide software-use-only
 *	maps which may or may not be stored in the same
 *	form as hardware maps.  These pseudo-maps are
 *	used to store intermediate results from copy
 *	operations to and from address spaces.
 *
 *	Since the information managed by this module is
 *	also stored by the logical address mapping module,
 *	this module may throw away valid virtual-to-physical
 *	mappings at almost any time.  However, invalidations
 *	of virtual-to-physical mappings must be done as
 *	requested.
 *
 *	In order to cope with hardware architectures which
 *	make virtual-to-physical map invalidates expensive,
 *	this module may delay invalidate or reduced protection
 *	operations until such time as they are actually
 *	necessary.  This module is given full information as
 *	to which processors are currently using which maps,
 *	and to when physical maps must be made correct.
 */

#include "cpus.h"
#include "cputypes.h"
#include "ramdisk.h"

#include "mach/machine/vm_types.h"

#include "mach/boolean.h"
#include "kern/thread.h"
#include "kern/zalloc.h"
#include "kern/lock.h"

#include "mach/vm_statistics.h"
#include "vm/vm_map.h"
#include "vm/vm_kern.h"
#include "mach/vm_param.h"
#include "mach/vm_prot.h"
#include "vm/vm_page.h"
#include "vm/vm_user.h"

#include "mach/machine/vm_param.h"
#include "machine/pcb.h"
#include "machine/memory.h"

/*
 * Determine how to allocate page tables - from zones or via kmem_alloc() .
 * Page tables must be wired, as they really live in physical, not virtual
 * memory, and they must be aligned on 16-byte boundaries.
 *
 * The zone package at MACH2.5 does not guarantee that either of the above
 * conditions are satisfied for non-pageable zones [- in particular, you can
 * get a memory allocation which crosses a page boundary, and therefore the
 * physical memory is not necessarily contiguous - very confusing to the PMMU!]
 * so the code does not currently use zones for allocation of page tables.
 *
 * The strategy for kmem_alloc()-based allocation of page tables is to allocate
 * Level 1 page tables from a free list of Level 1 page tables, but level 2 and 3
 * tables are handled differently.  Level 2 and 3 page tables are allocated in
 * 'groups', where a 'group' is defined as the number of page tables which exactly
 * fill one VM page.  When we create a new page table at level 2 or 3, we set up
 * all the page table entries for the 'group' at the next higher level.  cf. mmu.h .
 * 
 * [mmu.h depends upon the following definitions.
 */
#define L1_ZONE 0 /* zone-based storage for level 1 page tables */
#define L2_ZONE 0 /* zone-based storage for level 2 page tables */
#define PT_ZONE 0 /* zone-based storage for level 3 page tables */
		  /* XXX - warning, zone based PT allocation not tested */
#include "machine/mmu.h"

#include "machine/pte.h"
#include "machine/pmap.h"

#if	L1_ZONE
static struct zone	*lvl1_zone;	/* zone of level1 (long) table descriptors */
#endif
#if	L2_ZONE
static struct zone	*lvl2_zone;	/* zone of level2 (short)table descriptors */
#endif
#if	PT_ZONE
static struct zone	*lvl3_zone;	/* zone of level3 (long) table descriptors */
#endif

/*
 * Options for printing debug info to console.
 */
#ifdef	DBG
#define PD_PRINT           1
#define PD_FULL            2
#define PD_WORRY           4
#define PD_CHECK           8
#define PD_CHECK_KERNEL 0x10 /* keeps PMEM list in synch with kernel pmap, cf. pmap_move_page() */
int	pmap_dbg = PD_CHECK_KERNEL; 
#endif	DBG

#ifdef	DBG
#define static /* makes it easier to find things using kernel debugger */
#endif	DBG

/*
 *	The Physical Memory (PMEM) List.
 *
 * For each vm_page_t, pmap keeps a list of all currently valid virtual
 * mappings of that page.  An entry is a pv_entry_t; the list is the pmem_list.
 * This is used by things like pmap_remove, when we must find and remove all
 * mappings for a particular physical page.
 *
 * The PMEM list is allocated from physical memory at pmap_bootstrap() time,
 * so there is always one pv_entry for each physical page.  Additional pv_entries
 * for different VA's mapping to the same PA are allocated from the pv_list_zone.
 */
typedef struct pv_entry {
	struct pv_entry	*next;		/* next pv_entry */
	pmap_t		pmap;		/* pmap where mapping lies : 0 ==> null pv_entry */
	vm_offset_t	va;		/* virtual address for mapping */
} *pv_entry_t;

#define	PV_ENTRY_NULL	((pv_entry_t) 0)

static pv_entry_t	pmem_list = PV_ENTRY_NULL; /* array of entries, one per page */
static zone_t		pv_list_zone;		/* zone of pv_entry structures */

/*
 * Index into pmem_list.
 */
#define	PFIDX(pa)            PMAP_BTOP(pa - pmap_phys_start)
#define	PFIDX_TO_PVH(pfidx)  (&pmem_list[pfidx])

/*
 * Each entry in the pmem_list is locked by a bit in the pv_lock_table.
 * The lock bits are accessed by the physical address of the page they lock.
 */
static char	*pv_lock_table;		/* pointer to array of bits */

#if	NCPUS == 1                                           

#define	LOCK_PVH_PFI(pfi)	{ ; }
#define	UNLOCK_PVH_PFI(pfi)	{ ; }
#define	PV_LOCK_TABLE_SIZE(n)	0

#else	NCPUS == 1

/* No multi-processor support yet.  Line below from vax/pmap.c. */
#define	PV_LOCK_TABLE_SIZE(n)	(((n)+BYTE_SIZE-1)/BYTE_SIZE)

#endif	NCPUS == 1


/*
 * First and last physical addresses for which we maintain any information in the PMEM list.
 * Initialized at pmap_bootstrap() time.
 */           
static vm_offset_t pmap_phys_start = (vm_offset_t) 0;
static vm_offset_t pmap_phys_end   = (vm_offset_t) 0;
#define	PMAP_MANAGED(pa) ((pa) >= pmap_phys_start && (pa) < pmap_phys_end)

/*
 * Consistency checks (make the system run like a pig).
 * These checks are disabled by default; enabled by setting PD_CHECK in pmap_dbg.
 */
#ifdef	DBG
#define CHECK_PV_LIST(phys,pv_h,who)               \
	if (pmap_dbg & PD_CHECK)                   \
		check_pv_list(phys,pv_h,who)
#define CHECK_PMAP_CONSISTENCY(who)                \
	if (pmap_dbg & PD_CHECK)                   \
		check_pmap_consistency(who)
#else	DBG
#define CHECK_PV_LIST(phys,pv_h,who)
#define CHECK_PMAP_CONSISTENCY(who)
#endif	DBG


/*
 *	The KLVL3 table.
 * 
 * Physical memory in which to store the kernel's level 3 page tables
 * is allocated at bootstrap time.  The memory is broken into L3_SIZE
 * size segments and linked onto a queue.
 */
typedef struct klvl3_entry *klvl3_entry_t;
struct klvl3_entry {
	klvl3_entry_t	next;
	vm_offset_t	phys;
};
#define	KLVL3_ENTRY_NULL	((klvl3_entry_t)0)

static klvl3_entry_t	klvl3_free;

/*
 * MAX_KERNEL_VA_SIZE is the largest address space which the kernel will
 * require, and must be on a machine page boundary.  It must fit into the
 * virtual address space between VM_MIN_KERNEL_ADDRESS and VM_MAX_KERNEL_ADDRESS.
 */
#define MAX_KERNEL_VA_SIZE (128*1024*1024)
int max_kernel_va_size = MAX_KERNEL_VA_SIZE;

/* Amount of (physical) memory required for kernel's lvl3 page tables. */
#define	MAX_KERNEL_L3_SIZE  (PMAP_BTOP(MAX_KERNEL_VA_SIZE) * sizeof(pt_entry_t))

/* XXX should use TASK_MAX here, instead of 512 */
#define	PMAP_MAX 512  /* system-wide maximum number of pmaps */


/*
 * The Modify List
 * 
 * This is an array, one byte per physical page, which keeps track
 * of modified flags for pages which are no longer contained in any
 * pmap. (for mapped pages, the modified flags are in the PTE.)
 */
char	*pmap_modify_list;

static int pmap_inited = 0;

/*
 * The kernel's pmap is needed right at the start in pmap_bootstrap,
 * so it is allocated here.
 */
static struct pmap    kernel_pmap_store;
pmap_t kernel_pmap = &kernel_pmap_store;

/*
 * Two virtual pages of scratch space.
 * Used to map random physical page frames to a kernel vaddr
 * in copy_to_phys(), copy_from_phys(), pmap_copy_page() and pmap_zero_page().
 */
vm_offset_t	pmap_scratch_va1,
          	pmap_scratch_va2;
pt_entry_t	*pmap_scratch_pte1,
          	*pmap_scratch_pte2;

/*
 * another scratch page for the disk driver
 */
vm_offset_t	disk_va; 
pt_entry_t      *disk_pte;

int		ptes_per_vm_page;	/* number of M68K ptes required to map one VM page */

static struct zone	*pmap_zone;	/* zone of pmap structures */






/*
 *	Convert machine-independent protection codes to M68K protection bits.
 */
static int
m68k_protection(map, prot)
	pmap_t		map;
	vm_prot_t       prot;
{
	register pte_template_t p;

	p.bits = 0;

	if (map == kernel_pmap) {
		switch(prot) {
		case VM_PROT_NONE | VM_PROT_NONE | VM_PROT_NONE:
                case VM_PROT_NONE | VM_PROT_NONE | VM_PROT_EXECUTE:
		case VM_PROT_READ | VM_PROT_NONE | VM_PROT_NONE:
		case VM_PROT_READ | VM_PROT_NONE | VM_PROT_EXECUTE:
			p.pte.prot = 1;
			break;
		case VM_PROT_NONE | VM_PROT_WRITE | VM_PROT_NONE:
		case VM_PROT_NONE | VM_PROT_WRITE | VM_PROT_EXECUTE:
		case VM_PROT_READ | VM_PROT_WRITE | VM_PROT_NONE:
		case VM_PROT_READ | VM_PROT_WRITE | VM_PROT_EXECUTE:
			p.pte.prot = 0;
			break;
		}
	}         
	else {    
		switch(prot) {
		case VM_PROT_NONE | VM_PROT_NONE | VM_PROT_NONE:
                case VM_PROT_NONE | VM_PROT_NONE | VM_PROT_EXECUTE:
		case VM_PROT_READ | VM_PROT_NONE | VM_PROT_NONE:
		case VM_PROT_READ | VM_PROT_NONE | VM_PROT_EXECUTE:
			p.pte.prot = 1;
			break;
		case VM_PROT_NONE | VM_PROT_WRITE | VM_PROT_NONE:
		case VM_PROT_NONE | VM_PROT_WRITE | VM_PROT_EXECUTE:
		case VM_PROT_READ | VM_PROT_WRITE | VM_PROT_NONE:
		case VM_PROT_READ | VM_PROT_WRITE | VM_PROT_EXECUTE:
			p.pte.prot = 0;
			break;
		}
	}         

        return(p.bits);

} /* m68k_protection() */
   

/*
 *	Routine:	PMAP_LVL2 (internal)
 *
 *	Function:
 *		Given an offset and a map, compute the (virtual) address of the
 *		level 2 descriptor.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		virt		virtual address for which the level 2 entry is desired
 *
 *	Calls:
 *		L1ENT macro
 *
 *	Special Assumptions:
 *		That the level 1 table is always present.
 *
 *    The level 1 table is indexed by the first L1_BITS of the virtual
 * address.  If the level 1 entry is invalid, the function returns
 * L2_ENTRY_NULL.  Otherwise, the (virtual) address of the level 2 table
 * is extracted from the level 1 shadow table entry, the level 2 index is
 * added, and the result returned.
 */
static l2_entry_t *
pmap_lvl2(map, virt)
	pmap_t		map;
	vm_offset_t	virt;
{                         
	l1_entry_t	*lvl1 = L1ENT(map,virt);

	if (!L1_VALID(lvl1)) 
		return (L2_ENTRY_NULL);

	return((l2_entry_t *) (((lvl1 + L1_ENTRIES)->table_addr)<<4) + L2IDX(virt));

} /* pmap_lvl2() */          


/*
 *	Routine:	PMAP_PTE
 *
 *	Function:
 *		Given a map and a virtual address, compute a (virtual) pointer to
 *		the level 3 table entry (PTE) which maps the address.  If the level 3
 *		table associated with the address does not exist), PT_ENTRY_NULL is
 *		returned (and the map may need to grow).
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		virt		virtual address for which level 3 table entry
 *				is desired
 *
 *	Calls:
 *		pmap_lvl2
 *
 *    This routine first calls pmap_lvl2 to obtain a (virtual) pointer
 * to the level 2 table entry associated with the given virtual address.
 * If it returns L2_ENTRY_NULL, or if the level 2 entry is invalid, this
 * function will simply return PT_ENTRY_NULL.  Otherwise, the level 3
 * table address is extracted from the level 2 shadow entry, the level 3
 * index is added, and the result is returned.
 */
pt_entry_t *
pmap_pte(map, virt)
	pmap_t		map;
	vm_offset_t	virt;
{                 
	l2_entry_t	*lvl2;

	lvl2 = pmap_lvl2(map, virt);
	if (lvl2==L2_ENTRY_NULL || !L2_VALID(lvl2))
		return(PT_ENTRY_NULL);	
	else
		return((pt_entry_t *)(((lvl2 + L2_ENTRIES)->table_addr)<<4) + L3IDX(virt));

} /* pmap_pte() */


/*
 *	Routine:	PMAP_CREATE_KMAP
 *
 *	Function:
 *		Initializes the level 1 table descriptors for the kernel pmap.
 *
 *	Parameters:
 *		none.
 *
 *	Extern/Global:
 *		kernel_pmap
 *
 *     The virtual address of the kernel's level 1 table is extracted
 * from the kernel pmap structure.  The level 2 translation tables needed
 * for the kernel virtual address range will immediately follow the level
 * 1 table in contiguous virtual and physical memory locations.
 * 
 *     All level 2 table entries initialized to zero (invalidated).  The
 * level 1 table are initialized to point to their respective level 2
 * tables, and are validated.
 */
static void
pmap_create_kmap()
{                               
	pt_ldesc_template_t	*lvl1_paddr, *lvl1_vaddr;
	l2_entry_t		*lvl2_paddr, *lvl2_vaddr;
	pt_ldesc_template_t	l1_ktemplate;
	int			cnt;

	l1_ktemplate.bits.lo = 0;
	l1_ktemplate.pt_ldesc.lu    = 1;
#ifdef	M68851
	l1_ktemplate.pt_ldesc.share = 1;
#endif	M68851
	l1_ktemplate.pt_ldesc.sup   = 1;
	l1_ktemplate.pt_ldesc.dtype = DT_SHORT;

	lvl2_paddr = (l2_entry_t *)(kernel_pmap->lvl1_paddr + 2*L1_ENTRIES);
	lvl2_vaddr = (l2_entry_t *)(kernel_pmap->lvl1_vaddr + 2*L1_ENTRIES);
	lvl1_vaddr = (pt_ldesc_template_t *)kernel_pmap->lvl1_vaddr + USER_L1_ENTRIES;

	/*
	 * Initialize KERNEL_L1_ENTRIES to point to physical memory which
	 * pmap_bootstrap() allocated for the level2 page tables.
	 * The level2 tables will be initialized by pmap_expand_kmap().
	 * [USER_L1_ENTRIES are never referenced in the kernel pmap.]
	 */
	for (cnt = 0; cnt < KERNEL_L1_ENTRIES; cnt++) {
		lvl1_vaddr->bits.hi = (unsigned int)lvl2_paddr;
		(lvl1_vaddr + L1_ENTRIES)->bits.hi = (unsigned int)lvl2_vaddr;
		lvl1_vaddr->bits.lo = l1_ktemplate.bits.lo;
		(lvl1_vaddr + L1_ENTRIES)->bits.lo = l1_ktemplate.bits.lo;
		lvl1_vaddr++;
		lvl2_paddr += 2*L2_ENTRIES;
		lvl2_vaddr += 2*L2_ENTRIES;
	}   

} /* pmap_create_kmap() */



/*
 *	Routine:	PMAP_EXPAND_KMAP (internal)
 *
 *	Function:
 *		Allocate a level 3 table (pte_table) and validate associated level
 *		2 entry, returning pointer to level 3 entry.  This is much like
 *		'pmap_expand', except that table space is acquired from an area set up
 *		by pmap_bootstrap, instead of through kmem_alloc.  (Obviously, because
 *		kmem_alloc uses the kernel map for allocation - which we can't do when
 *		trying to expand the kernel map!)
 *
 *		Note that level 2 tables for the kernel map were all allocated at
 *		pmap_bootstrap time, so we only need to worry about the level 3 table here.
 *
 *	Parameters:
 *		virt		VA for which translation tables are needed
 *		prot		protection attributes for level 2 entries
 *
 *	Extern/Global:
 *		klvl3_free, klvl3 list
 *
 *	Calls:
 *		pmap_lvl2
 *
 *	This routine simply dequeues a table from the kl3_free list,
 *	initializes all its entries (invalidates them), and sets the
 *	corresponding level 2 table entry to point to it.  If the kl3_free
 *	list is empty - we panic (no other place to get memory, sorry).  (Such
 *	a panic indicates that pmap_bootstrap is not allocating enough table
 *	space for the kernel virtual address space).
 */
static pt_entry_t * 
pmap_expand_kmap (virt, prot)
	vm_offset_t	virt;
	int		prot;
{ 
	int		aprot;
	l1_entry_t	*lvl1;
	l2_entry_t	*lvl2;
	pt_entry_t	*pte;
	klvl3_entry_t	klvl3_ent;
	pmap_t		map = kernel_pmap;

#if	DBG
	if (pmap_dbg & PD_PRINT)
		printf("(pmap_expand_kmap) v %x\n", virt);
#endif	DBG

	aprot = m68k_protection (map, prot);
                         
	lvl2 = pmap_lvl2 (map, virt);

	if (lvl2 == L2_ENTRY_NULL)
		panic ("pmap_expand_kmap: lvl1 entry INVALID");

	if (L2_VALID(lvl2))
		panic ("pmap_expand_kmap: lvl2 entry VALID");
                                  
	klvl3_ent = klvl3_free;
	if (klvl3_ent == KLVL3_ENTRY_NULL) {
		panic ("pmap_expand_kmap:  Ran out of lvl3 page tables");
		return(PT_ENTRY_NULL);
	}
	klvl3_free = klvl3_free->next;

	((pt_sdesc_template_t *)lvl2)->bits = klvl3_ent->phys | aprot | DT_SHORT;
	((pt_sdesc_template_t *)(lvl2 + L2_ENTRIES))->bits = (vm_offset_t)klvl3_ent | aprot | DT_SHORT;
	return((pt_entry_t *)(klvl3_ent) + L3IDX(virt));

} /* pmap_expand_kmap() */


/*
 *	Routine:	PMAP_MAP
 *
 *	Function:
 *		Map memory at initialization.  The physical addresses being mapped
 *		are not managed and are never unmapped.
 *
 *	Parameters:
 *		virt		virtual address of range to map
 *		start		physical address of range to map
 *		end		physical address of end of range
 *		prot		protection attributes
 *
 *	Calls:
 *		pmap_pte
 *		pmap_expand_kmap
 *
 *	Special Assumptions:
 *		For now, VM is already on, we only need to map the specified memory.
 *		Used only by pmap_bootstrap().
 *
 *    For each page that needs mapping:
 *	pmap_pte is called to obtain the address of the level
 *	3 table entry (PTE).  If the level 3 table does not exist,
 *	pmap_expand_kmap is called to allocate it.  Finally, the level
 *	3 entry is set to point to the physical page.
 */
static vm_offset_t
pmap_map (virt, start, end, prot)
	register vm_offset_t	virt;
	register vm_offset_t	start;
	register vm_offset_t	end;
	register int		prot;
{
	int		aprot;
	int		num_phys_pages;
	pt_entry_t	*pte;
	pte_template_t	template;

#if	DBG
	if (pmap_dbg & PD_PRINT)
		printf ("(pmap_map) phys address from %x to %x mapped at virtual %x, prot %x (%x)\n",
			start, end, virt, prot, pmap_phys_start);
#endif	DBG
	aprot = m68k_protection (kernel_pmap, prot);
	template.bits = PMAP_TRUNC_PAGE(start) | aprot | DT_PAGE;
	for (num_phys_pages = PMAP_BTOP(PMAP_ROUND_PAGE(end - start));
	     num_phys_pages > 0; num_phys_pages--) {

		if ((pte = pmap_pte (kernel_pmap, virt)) == PT_ENTRY_NULL)
			if ((pte = pmap_expand_kmap (virt, VM_PROT_READ|VM_PROT_WRITE)) == PT_ENTRY_NULL)
				panic ("pmap_map:  Cannot allocate pte table");
		*pte = template.pte;
		virt += PMAP_PGBYTES;
		template.bits += PMAP_PGBYTES;
	}

	return(virt);

} /* pmap_map() */                  



/*
 *	Routine:	PMAP_VALID_PAGE
 *
 *	Function:
 *		Tell the VM system whether the given physical page actually exists.
 *		This is for machines/configurations where there actually are holes
 *		in the address range we said was available (e.g. Suns).
 *
 *	Parameters:
 *		p		physical address
 *
 */

boolean_t
pmap_valid_page(p)
	vm_offset_t p;
{
	return TRUE;
}
                        


/*
 *	Routine:	PMAP_BOOTSTRAP
 *
 *	Function:
 *			Bootstrap the system enough to run with virtual memory.
 *			Map the kernel's code and data, allocate the kernel
 *			translation table space, and map control registers
 *			and other IO addresses.
 *
 *	Parameters:
 *
 *		load_start	PA where kernel was loaded (IN)
 *		&phys_start	PA of first available physical page (IN/OUT)
 *		&phys_end	PA of last available physical page (IN)
 *		&virtual_avail	VA of first available page (after kernel image) (OUT)
 *		&virtual_end	VA of last available page (end of kernel space) (OUT)
 *
 *	Extern/Global:
 *
 *		page_size	VM (software) page size (IN)
 *		start		start of kernel text (IN)
 *		etext		end of kernel text (IN)
 *		pmap_scratch_va1	VA of page mapped arbitrarily for debug/IO (OUT)
 *		pmap_scratch_pte1	... addr of its level 3 table entry (PTE) (OUT)
 *		pmap_scratch_va2	VA of page mapped arbitrarily for debug/IO (OUT)
 *		pmap_scratch_pte2	... addr of its level 3 table entry (PTE) (OUT)
 *
 *	Calls:
 *		simple_lock_init
 *		pmap_create_kmap
 *		pmap_map
 *
 *    The physical address 'load_start' is mapped at
 * VM_MIN_KERNEL_ADDRESS, which maps the kernel code and data at the
 * virtual address for which it was (presumably) linked.  Immediately
 * following the end of the kernel code/data, sufficient pages of
 * physical memory are reserved to hold translation tables for the kernel
 * address space.  The 'phys_start' parameter is adjusted upward to
 * reflect this allocation.  This space is mapped in virtual memory
 * immediately following the kernel code/data map.
 * 
 *    A pair of virtual pages are reserved for debugging and IO
 * purposes.  They are arbitrarily mapped when needed.  They are used,
 * for example, by pmap_copy_page and pmap_zero_page.
 *
 *    Finally, device control registers and other physical addresses are
 * mapped into kernel virtual address space at locations prescribed by
 * constants contained in memory.h.
 */
void
pmap_bootstrap (load_start, phys_start, phys_end, virt_start, virt_end)
	vm_offset_t	load_start;
	vm_offset_t	*phys_start;	/* IN/OUT */
	vm_offset_t	*phys_end;	/* IN  */
	vm_offset_t	*virt_start;	/* OUT */
	vm_offset_t	*virt_end;	/* OUT */
{                  
	klvl3_entry_t	klvl3_virt;
	l1_entry_t	*kmap;
	vm_size_t	s;
	vm_offset_t	vaddr,
	                klvl3_phys,
	                s_text,
	                e_text,
			vstart,
	                kernel_pmap_size;
	int		npages,
			i;
	extern char	start[], etext[];
	struct crp_bits {
		unsigned int lo;
		l1_ptr_t hi;
	} croot_ptr;
	extern void load_mmu();
#ifdef	BOOTPROM
	extern unsigned int prom_code;
	unsigned int prom_code_phys, prom_data_phys;
#endif	BOOTPROM
#if	RAMDISK
	extern int ramdisk_phys_addr;  /* locore init code sets this up */
	extern int ramdisk_disksize;
	extern short boot_ctype;
#endif	RAMDISK

	ptes_per_vm_page = PAGE_SIZE>>PMAP_PGSHIFT;
	if (ptes_per_vm_page == 0)
		panic ("pmap_bootstrap: VM page size < MACHINE page size");
	if (ptes_per_vm_page > 1)
		printf ("pmap_bootstrap: VM page size > MACHINE page size\n");

	simple_lock_init (&kernel_pmap->lock);

	/*
	 * Allocate the kernel page table from the front of available physical memory,
	 * i.e. just after where the kernel image was loaded.
	 * XXX - only need to align on a 16 byte boundary, but do it on page boundary for debugging.
	 */
	pmap_phys_start = round_page(*phys_start);
	vstart = pmap_phys_start + ((unsigned int)start - load_start);

	/* 
	 * Reserve physical memory for the kernel pmap's table and page descriptors.
	 */
	kernel_pmap->ref_count = 1;            
	kernel_pmap->lvl1_paddr = kmap = (l1_entry_t *)(pmap_phys_start);
	kernel_pmap->lvl1_vaddr        = (l1_entry_t *)(vstart);
#ifdef	DBG
	/* init double-linked list of pmap structs */
	kernel_pmap->next = kernel_pmap;
	kernel_pmap->prev = kernel_pmap;
#endif	DBG
	/* reserve space for lvl1 and lvl2 entries */
	kernel_pmap_size = 2*L1_SIZE + 2*KERNEL_L1_ENTRIES*L2_SIZE;
	/* save pointers to where lvl3 entries start in physical memory */
	klvl3_phys = (pmap_phys_start + kernel_pmap_size);
	klvl3_virt = (klvl3_entry_t)(vstart + kernel_pmap_size);
	kernel_pmap_size += MAX_KERNEL_L3_SIZE;
	pmap_phys_start += kernel_pmap_size;
	vstart += kernel_pmap_size;
	/* init all lvl 1,2, and 3 descriptors to zero */
	blkclr (kernel_pmap->lvl1_vaddr, kernel_pmap_size);

	/*
	 * Init the klvl3 queue.
	 */
	klvl3_free = klvl3_virt;
	for (i = MAX_KERNEL_L3_SIZE/L3_SIZE; i>0;  i--) {
		(vm_offset_t) klvl3_virt->next = (vm_offset_t)klvl3_virt + L3_SIZE;
		klvl3_virt->phys = klvl3_phys;        
		klvl3_virt = klvl3_virt->next;
		klvl3_phys += L3_SIZE;
	}
	klvl3_virt->next = KLVL3_ENTRY_NULL; /* terminate the list */

	/*
	 * Initialize level 1 table descriptors for the kernel pmap.
	 */
	pmap_create_kmap();

	s_text = load_start;                     /* paddr of text */
	e_text = load_start + (etext - start);   /* paddr of end of text section */
	s_text = round_page(s_text);
	e_text = trunc_page(e_text);

	/*
	 * Map the kernel image into VM.
	 */
	vaddr = pmap_map (VM_MIN_KERNEL_ADDRESS, load_start, s_text, VM_PROT_WRITE|VM_PROT_READ);
	vaddr = pmap_map (&start, s_text, e_text, VM_PROT_WRITE|VM_PROT_READ);
	vaddr = pmap_map (vaddr, e_text, kmap, VM_PROT_WRITE|VM_PROT_READ);

	/*
	 * Map system page tables.
	 */
	while (vaddr < (vstart - kernel_pmap_size))
		vaddr = PMAP_ROUND_PAGE(vaddr + 1);
	if (vaddr != (vstart - kernel_pmap_size)) {
		printf("pmap_bootstrap: vstart - kpmap_size != vaddr !!.\n");
		printf("                vstart = %x, vaddr = %x, kp_size = %x\n",
					vstart, vaddr, kernel_pmap_size);
	}
	vaddr = pmap_map (vaddr, kmap, pmap_phys_start, VM_PROT_WRITE|VM_PROT_READ);

	if (vaddr != vstart) {
		vstart = vaddr;
		pmap_phys_start = round_page(pmap_phys_start);
	}

	/*
	 * Allocate memory for the pmem_list, its lock bits, and the modify array.
	 */
	npages = PMAP_BTOP(*phys_end - pmap_phys_start);

	s = (vm_size_t) (npages * sizeof(struct pv_entry)  /* PMEM list        */
	               + PV_LOCK_TABLE_SIZE(npages)        /* pv_lock_table    */
	               + npages * sizeof(char));           /* pmap_modify_list */
	s = round_page(s);

	pmap_map (vaddr, pmap_phys_start, pmap_phys_start+s, VM_PROT_WRITE|VM_PROT_READ);
	bzero (vaddr, s);

	pmem_list = (pv_entry_t) vaddr;
	vaddr = (vm_offset_t) (pmem_list + npages);
	pv_lock_table = (char *) vaddr;
	vaddr = (vm_offset_t) (pv_lock_table + PV_LOCK_TABLE_SIZE(npages));
	pmap_modify_list = (char *) vaddr;
	vstart += s;

	/*
	 * Define the area of physical memory which is PMAP_MANAGED,
	 * that is, the physical memory which is tracked in the PMEM list.
	 * All memory which has been allocated already is not managed;
	 * all successive allocations will be managed.
	 */
	pmap_phys_end    = *phys_end;
	pmap_phys_start += s;
	*phys_start      = pmap_phys_start;

	/*
	 * Allocate some virtual memory & PTEs for phys routines, disk driver, etc.
	 */
#define	ALLOC_VIRT(va, pte)                       \
	pte     = pmap_pte (kernel_pmap, vstart); \
	va      = vstart;                         \
	vstart += PAGE_SIZE;

	ALLOC_VIRT(pmap_scratch_va1, pmap_scratch_pte1);
	ALLOC_VIRT(pmap_scratch_va2, pmap_scratch_pte2);
	ALLOC_VIRT(disk_va, disk_pte);

	*virt_start = round_page(vstart);
	*virt_end   = VM_MAX_KERNEL_ADDRESS;

	/*
	 * Map devices, et al.
	 */
	pmap_map (CR_BASE, CR_BASE_P, CR_BASE_P + 32*1024 - 1, VM_PROT_READ|VM_PROT_WRITE);
	pmap_map (IO_BASE, IO_BASE_P, IO_BASE_P + 3*65536 - 1, VM_PROT_READ|VM_PROT_WRITE);
	pmap_map (AT_MEM_BASE, AT_MEM_BASE_P, AT_MEM_BASE_P + 1024*1024, VM_PROT_READ|VM_PROT_WRITE);

#ifdef DN2500
	/* DN2500 mono memory not covered by AT Memory mapping */
	pmap_map (MONO_MEM_BASE, MONO_MEM_BASE_P, MONO_MEM_BASE_P + 256*1024, VM_PROT_READ|VM_PROT_WRITE);
#endif DN2500

#if	TOMLRAMDISK
	if (boot_ctype) {
		CHECK_PAGE_ALIGN (ramdisk_phys_addr, "pmap_bootstrap - ramdisk addr");
		pmap_map (RAMDISK_BASE, ramdisk_phys_addr, ramdisk_phys_addr+ramdisk_disksize, VM_PROT_READ|VM_PROT_WRITE);
		if (pmap_dbg & PD_PRINT)
			printf("(pmap_bootstrap) RAMDISK at physical %x  virtual %x  size %x\n",
				ramdisk_phys_addr, RAMDISK_BASE, ramdisk_disksize);
	}
#endif  TOMLRAMDISK

#ifdef	BOOTPROM
	prom_code_phys = (unsigned int)(&prom_code) - (unsigned int)start + STARTLOAD;
	prom_code_phys = PMAP_ROUND_PAGE(prom_code_phys);
	prom_data_phys = prom_code_phys + 32*1024;
	vaddr = pmap_map (0x00000000, prom_code_phys, prom_code_phys+32*1024-1, VM_PROT_READ | VM_PROT_WRITE);
 	pmap_map (0x00008000, 0x00100000, 0x00102000, VM_PROT_READ | VM_PROT_WRITE);
#endif	BOOTPROM

	/*
	 * Switch to using new page tables.
	 */
	croot_ptr.lo = 0x80000000+DT_LONG; /* level 1 entries are M68K long format table descriptors. */
	croot_ptr.hi = kernel_pmap->lvl1_paddr;
	load_mmu (&croot_ptr, TC);

} /* pmap_bootstrap() */



/*
 *	Routine:	PMAP_INIT
 *
 *	Function:
 *		Initialize the pmap module.  It is called by vm_init, to initialize
 *		any structures that the pmap system needs to map virtual memory.
 *
 *	Parameters:
 *		phys_start	physical address of first available page
 *		                (was last set by pmap_bootstrap)
 *		phys_end	physical address of last available page
 *
 *	Extern/Global:
 *		pmem_list (OUT)
 *		pv_lock_table (OUT)
 *		pmap_modify_list (OUT)
 *		pmap_phys_start (OUT)
 *		pmap_phys_end (OUT)
 *
 *	Calls:
 *		kmem_alloc
 *		zinit
 *
 *    This routine does not really have much to do. 
 * It initializes zones for pmap structures, the pv_entry zone, and
 * zones for page tables, if needed.
 *
 */
void
pmap_init(phys_start, phys_end)
	vm_offset_t	phys_start, phys_end;
{                                            
	register vm_offset_t	addr, va, pa;
	register vm_size_t	s;
	int			i;
	klvl3_entry_t		klvl3_ent;

#ifdef	DBG
	if (pmap_dbg & PD_PRINT)
		printf("(pmap_init) phys_start %x  phys_end %x  pmap_phys_start %x\n",
			phys_start, phys_end, pmap_phys_start);
#endif	DBG

	/*
	 * Create the zone of physical maps,
	 * of the physical-to-virtual entries,
	 * of pt_ldesc, and of pt_sdesc (ie. lvl1 and lvl2 tables).
	 * XXX - make sure allocation size is not smaller than element size 
	 */
	s = (vm_size_t) sizeof(struct pmap);
	pmap_zone = zinit(s, PMAP_MAX*s, 4096, FALSE, "pmap");

	s = (vm_size_t) sizeof(struct pv_entry);
	pv_list_zone = zinit(s, 10000*s, 4096, FALSE, "pv_list");

	/* due to table address field handling in the 68030 MMU,
	   we need to 16-byte align the trans tables */
#if	L1_ZONE
	s = (vm_size_t) 2*L1_SIZE;
	lvl1_zone = zinit(s+16, PMAP_MAX*s, 4096, FALSE, "lvl1_map");
#endif
#if	L2_ZONE
	s = (vm_size_t) 2*L2_SIZE;
	lvl2_zone = zinit(s+16, USER_L1_ENTRIES*PMAP_MAX*s, 8192, FALSE, "lvl2_map_set");
#endif
#if	L3_ZONE
	error - L3 table zones not implemented
#endif

#ifdef	DBG
	if (pmap_dbg & PD_PRINT)
		printf("(pmap_init) done\n");
#endif	DBG

	pmap_inited++;

} /* pmap_init() */


#if	!L1_ZONE
/*
 *	Routine:	PMAP_ALLOC_L1_TABLE
 *
 *	Function:
 *		Allocate a level 1 page table.
 *
 *	Parameters:
 *		none
 *
 *	Calls:
 *		kmem_alloc
 *
 *	This routine allocates space for a level 1 table from a
 *	statically maintained free list.  If the free list is exhausted,
 *	space is allocated by kmem_alloc().
 */
#define FREELIST_NULL ((vm_offset_t) 0)
static	vm_offset_t	*l1_freelist = FREELIST_NULL;

static l1_entry_t *
pmap_alloc_l1_table()
{
	vm_offset_t	*newtbls;
	l1_entry_t	*lvl1;
	int		i;

	if (l1_freelist == FREELIST_NULL) {
		newtbls = (vm_offset_t *) kmem_alloc(user_pt_map, PAGE_SIZE);
		/* add new table space to free list */
		for (i=0; i<(PAGE_SIZE/(2*L1_SIZE)); i++) {
			*newtbls = (vm_offset_t) l1_freelist;
			l1_freelist = newtbls;
			newtbls += ((2*L1_SIZE) / sizeof(vm_offset_t));
		}
	}

	lvl1 = (l1_entry_t *) l1_freelist;
	l1_freelist = (vm_offset_t *) *l1_freelist;
	bzero(lvl1, 2*L1_SIZE);		/* make sure new tbl is all zeros */

	return(lvl1);

} /* pmap_alloc_l1_table() */


/*
 *	Routine:	PMAP_FREE_L1_TABLE
 *
 *	Function:
 *		Free a level 1 page table.
 *
 *	Parameters:
 *		none
 *
 *	Calls:
 *		kmem_free
 *
 *	This routine frees space for a level 1 table to a
 *	statically maintained free list.
 *	If all level 1 tables drawn from a specific VM page
 *	are freed, the whole page is returned to the system.
 *	[[[This coalesing of free space is not yet implemented.]]]
 */
static void
pmap_free_l1_table(lvl1)
l1_entry_t	*lvl1;
{
	vm_offset_t	*freetbl;

	freetbl = (vm_offset_t *) lvl1;
	*freetbl = (vm_offset_t) l1_freelist;
	l1_freelist = freetbl;
} /* pmap_free_l1_table() */

#endif	!L1_ZONE



/*
 *	Routine:	PMAP_CREATE
 *
 *	Function:
 *		Create and return a physical map.  If the size specified for the
 *		map is zero, the map is an actual physical map, and may be referenced
 *		by the hardware.  If the size specified is non-zero, the map will be
 *		used in software only, and is bounded by that size.
 *
 *	Parameters:
 *		size		size of the map
 *
 *	Calls:
 *		zalloc
 *		simple_lock_init
 *
 *    This routines allocates a pmap structure and a level 1 translation
 * table from the zones set up by pmap_init.  The level 1 table entries
 * for user space addresses are initialized to zero (invalid), those for
 * kernel space addresses are copied from the kernel pmap's level 1
 * table.  The pmap structure is initialized with the virtual and
 * physical addresses of the level 1 table.  The address (virtual) of the
 * pmap structure is returned.
 */
pmap_t
pmap_create(size)
	vm_size_t	size;
{
	register pmap_t			p;
	register pmap_statistics_t	stats;
	l1_entry_t	*lvl1;
	pt_ldesc_template_t *l1, *kl1;
	int		cnt;
	pt_ldesc_template_t	l1_utemplate;

	l1_utemplate.bits.lo = 0;
	l1_utemplate.pt_ldesc.lu    = 1;

	/*
	 * A software use-only map doesn't even need a map.
	 */
	if (size != 0)
		return(PMAP_NULL);

	CHECK_PMAP_CONSISTENCY ("pmap_create");

	p = (pmap_t) zalloc(pmap_zone);
	if (p == PMAP_NULL) {
		panic ("pmap_create: cannot allocate a pmap");
	}               

#if	L1_ZONE
#error this code has a bug where the memory allocated is not necessarily *physically* contiguous
	lvl1 = (l1_entry_t *) zalloc(lvl1_zone);
	/* adjust alignment to a 16-byte boundary */
	lvl1 = (l1_entry_t *) (((int)lvl1)+15 & ~15);
#else
	lvl1 = pmap_alloc_l1_table();
#endif

	/* 
	 * Point the pmap at the allocated level 1 table.
	 */ 
	p->lvl1_vaddr = lvl1;                              
	p->lvl1_paddr = (l1_entry_t *) pmap_extract(kernel_pmap, lvl1);
	if (!PT_ALIGNED(p->lvl1_paddr))
		panic ("pmap_create: lvl1_table not aligned on 16 byte boundary");


	/*	
	 * Initialize the user space level 1 entries.
	 */
	l1 = (pt_ldesc_template_t *)lvl1;
	for (cnt = USER_L1_ENTRIES; cnt>0; cnt--) {
		l1->bits.hi = 0;
		l1->bits.lo = l1_utemplate.bits.lo;
		(l1 + L1_ENTRIES)->bits.hi = (vm_offset_t) L2_ENTRY_NULL;
		(l1 + L1_ENTRIES)->bits.lo = l1_utemplate.bits.lo;
		l1++;
	}

	/*	
	 * Initialize the kernel space level 1 entries in the user pmap
	 * by copying them from the kernel pmap.
	 */
	kl1 = (pt_ldesc_template_t *)(kernel_pmap->lvl1_vaddr + USER_L1_ENTRIES);
	for (cnt = KERNEL_L1_ENTRIES; cnt>0; cnt--) {
		*(l1+L1_ENTRIES) = *(kl1+L1_ENTRIES);
		*l1++ = *kl1++;
	}

	p->ref_count  = 1;       
	simple_lock_init(&p->lock);

	/*
	 *	Initialize statistics.
	 */
	stats = &p->stats;
	stats->resident_count = 0;
	stats->wired_count = 0;

#ifdef	DBG
	/* link into list of pmaps, just after kernel_pmap */
	p->next = kernel_pmap->next;
	p->prev = kernel_pmap;
	kernel_pmap->next = p;
	p->next->prev = p;
#endif	DBG

	return(p);

} /* pmap_create() */



/*
 *	Routine:	PMAP_FREE_TABLES (internal)
 *
 *	Function:
 *		Internal procedure used by pmap_destroy() to actually deallocate
 *		the page tables.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *
 *	Calls:
 *		pmap_lvl2
 *		pmap_pte
 *		kmem_free
 *		zfree
 *
 *	Special Assumptions:
 *		No locking is needed, since this is only called which the
 *		ref_count field of the pmap structure goes to zero.
 *
 *    This routine sequences through the user address space, releasing
 * all translation table space back to their zones or the system, as
 * appropriate.  The loops are indexed by the virtual address space
 * ranges represented by the nominal table or group sizes.
 */
static void
pmap_free_tables(pmap)
	pmap_t		pmap;
{
	vm_offset_t	l1_va,	/* outer loop index */
	                l2_va;	/* inner loop index */
	l1_entry_t	*l1tbl;	/* ptr to first entry in the level 1 table */
	l2_entry_t	*l2tbl;	/* ptr to first entry in a level 2 table */
	pt_entry_t	*l3tbl;	/* ptr to first entry in a level 3 table */


#if	DBG
	if (pmap_dbg & PD_FULL)
		printf ("(pmap_free_tables) pmap %x\n", pmap);
#endif	DBG

	l1tbl = pmap->lvl1_vaddr;	/* addr of level 1 table */

	/*
	 * Loop once for each Level 2 Table Group.
	 */
	for (l1_va = VM_MIN_ADDRESS; l1_va < VM_MIN_KERNEL_ADDRESS; l1_va += L2_GROUP_VA_SPACE) {

		if ((l2tbl = pmap_lvl2(pmap, l1_va)) != L2_ENTRY_NULL) {
			/*
			 * Loop once for each Level 3 Table Group.
			 */
			for (l2_va = l1_va; l2_va < (l1_va+L2_GROUP_VA_SPACE) /* toml: was L2_VA_SPACE */; l2_va += L3_GROUP_VA_SPACE)
				if ((l3tbl = pmap_pte(pmap, l2_va)) != PT_ENTRY_NULL)
					/* XXX - should panic if any l3 entries are valid */
					PT_FREE(l3tbl);
	
			L2_FREE(l2tbl);
		}

	} /* Level 1 Loop */

#if	L1_ZONE
	zfree(lvl1_zone, (vm_offset_t) l1tbl);
#else
	pmap_free_l1_table(l1tbl);
#endif

} /* pmap_free_tables() */



/*
 *	Routine:	PMAP_DESTROY
 *
 *	Function:
 *		Retire the given physical map from service.  Should only be called
 *		if the map contains no valid mappings.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *
 *	Calls:
 *		pmap_free_tables
 *		zfree
 *
 *	Special Assumptions:
 *		Map contains no valid mappings.
 *
 *     This routine decrements the reference count in the pmap
 * structure.  If it goes to zero, pmap_free_tables is called to release
 * the memory space to the system.  Then, zfree is called to return the
 * pmap structure to its zone.
 */
void
pmap_destroy(p)
	register pmap_t	p;
{
	register int		c, s;

	if (p == PMAP_NULL)
		return;

	if (p == kernel_pmap) {
		panic ("pmap_destroy:  Attempt to destroy kernel pmap");
		return;
	}             

	CHECK_PMAP_CONSISTENCY ("pmap_destroy");

	PMAP_READ_LOCK(p,s);
	c = --p->ref_count;
	PMAP_READ_UNLOCK(p,s);
        
	if (c == 0) {
		pmap_free_tables(p);
#ifdef	DBG
		/* unlink from list of pmap structs */
		p->prev->next = p->next;
		p->next->prev = p->prev;
#endif	DBG
		zfree(pmap_zone, (vm_offset_t) p);
	}

} /* pmap_destroy() */


/*
 *	Routine:	PMAP_REFERENCE
 *
 *	Function:
 *		Add a reference to the specified pmap.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *
 *     Under a pmap read lock, the ref_count field of the pmap structure
 * is incremented.  Thefunction then returns.
 */
void
pmap_reference(p)
	register pmap_t	p;
{
	int	s;

	if (p != PMAP_NULL) {
		PMAP_READ_LOCK(p,s);
		p->ref_count++;
		PMAP_READ_UNLOCK(p,s);
	}

} /* pmap_reference() */


/*
 *	Routine:	PMAP_REMOVE_RANGE
 *
 *	Function:
 *		Invalidate level 3 translation table entries associated with the
 *		given virtual address range.  The entries given are the first
 *		(inclusive) and last (exclusive) entries for the VM pages.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		s		virtual address of start of range to remove
 *		e		virtual address of end of range to remove
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *		pmap_modify_list
 *
 *	Calls:
 *		pmap_pte
 *		zfree
 *
 *	Special Assumptions:
 *		The pmap must be locked.
 *
 *    This routine sequences through the pages defined by the given
 * range.  For each page, pmap_pte is called to obtain a (virtual)
 * pointer to the level 3 table entry (PTE) associated with the page's
 * virtual address.  If the level 3 entry does not exist, or is invalid,
 * nothing need be done.
 *
 *     If the PTE is valid, the routine must invalidated the entry.  The
 * 'modified' bit, if on, is reflected to the VM through the
 * 'vm_page_set_modified' macro, and into the appropriate entry in the
 * pmap_modify_list.  Next, the function must find the PV list entry
 * associated with this pmap/va (if it doesn't exist - the function
 * panics).  The PV list entry is unlinked from the list, and returned to
 * its zone.
 * XXX
 * XXX- This routines needs some performance improvements.  In
 * XXX- particular, it should index loops by the VA space represented by whole
 * XXX- translation tables or table groups.  This will save a lot of repeated
 * XXX- pointer chasing from the root of the translation tree.]]]
 * XXX 
 * XXX- OLD COMMENT: If the pmap is not the kernel pmap, the range must
 * XXX- lie entirely within one pte-page.  This is NOT checked.  - WHY? doe VM
 * XXX- know what a PTE page is???]]]
 */
static void
pmap_remove_range (map, s, e)
	pmap_t		map;
	vm_offset_t	s, e;
{           
	int			pfi;
	int			pfn;
	int			num_removed=0,
				num_unwired=0;
	register int		i;
	pt_entry_t		*pte;
	pv_entry_t		prev, cur;
	pv_entry_t		pvl;
	l2_entry_t		*lvl2;
	vm_offset_t		pa, va;
        boolean_t               was_modified;

#if	DBG
	if (pmap_dbg & PD_FULL)
		printf ("(pmap_remove_range) map %x  s %x  e %x\n", map, s, e);
#endif	DBG

	FLUSH_ATC_ENTRIES(map, s, e);

	/*	
	 *	Loop through the range in vm_page_size increments.
	 *	Do not assume that either start or end fall on any
	 *	kind of page boundary (though this may be true !?).
	 */

	CHECK_PAGE_ALIGN (s, "pmap_remove_range - start addr");

	for (va = s; va < e; va += PAGE_SIZE) {

		if (!L1_VALID(L1ENT(map, va))) {
			va = L1_NEXT(va)-PAGE_SIZE; /* no lvl2 table, skip to next lvl1 entry */
			continue;
		}

		pte = pmap_pte(map, va);

		if (pte == PT_ENTRY_NULL) {
			va = L2_NEXT(va)-PAGE_SIZE; /* no lvl3 table, skip to next lvl2 entry */
			continue;
		}

		if (!L3_VALID(pte))
			continue;		/* no page mapping */

		num_removed++;

		if (pte->wired)
			num_unwired++;

		pfn = pte->pfn; 
		pa = PMAP_PTOB(pfn);

		if (PMAP_MANAGED(pa)) {

			pfi = PFIDX(pa);

			LOCK_PVH(pfi);
			/*
			 *	Remove the mapping from the pvlist for
			 *	this physical page.
			 */
			pvl = PFIDX_TO_PVH(pfi);
	
			CHECK_PV_LIST (pa, pvl, "pmap_remove_range before");
	
			if (pvl->pmap == PMAP_NULL)
				panic ("pmap_remove: null pv_list");

			if (pvl->va == va && pvl->pmap == map) {
				/*
				 * Header is the pv_entry.  Copy the next one
				 * to header and free the next one (we can't
				 * free the header).
				 */
				cur = pvl->next;
				if (cur != PV_ENTRY_NULL) {
					*pvl = *cur;
					zfree(pv_list_zone, (vm_offset_t) cur);
				}
				else
				    	pvl->pmap = PMAP_NULL;
			}
			else {

				for (prev = pvl; (cur = prev->next) != PV_ENTRY_NULL; prev = cur)
					if (cur->va == va  &&  cur->pmap == map)
						break;

				if (cur == PV_ENTRY_NULL) {
					printf("pmap_remove_range: looking for VA 0x%X PV list at 0x%X\n", va, pvl);
					panic ("pmap_remove_range: mapping not in pv_list");
				}

				prev->next = cur->next;

				zfree(pv_list_zone, (vm_offset_t) cur);
			}

			CHECK_PV_LIST (pa, pvl, "pmap_remove_range after");

			UNLOCK_PVH(pfi);

		} /* if PAGE_MANAGED */


		/*
		 * For each pte in vm_page (NOTE: vm_page, not
		 * M68K (machine dependent) page !! ), set
		 * modify bits and invalidate the pte entry.
		 */
		for (i = ptes_per_vm_page; i>0; i--) {
			if (pte->modified && PMAP_MANAGED(pa))
				pmap_modify_list[pfi] = 1;
			*(int *)pte++ = 0;
		}
	}

	/*
	 *	Update the counts
	 */
	map->stats.resident_count -= num_removed;
	map->stats.wired_count -= num_unwired;

} /* pmap_remove_range() */


/*
 *	Routine:	PMAP_REMOVE
 *
 *	Function:
 *		Remove the given range of addresses from the specified map.
 *		It is assumed that the start is properly rounded to the VM page size.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *
 *	Special Assumptions:
 *		Assumes not all entries must be valid in specified range.
 *
 *	Calls:
 *		pmap_remove_range
 *
 *    The start and end addresses are checked to make sure they fall
 * within the user address space range for maps other than the kernel
 * map; and that they fall within the kernel address space range for the
 * kernel map.  If all is OK, the pmap read lock is taken, and
 * pmap_remove_range is called to do the real work.
 */
/*  XXX - need to update stats ,  and invalidate tlb entries */
void
pmap_remove(map, s, e)
	pmap_t		map;
	vm_offset_t	s, e;
{
	int		spl;

	if (map == PMAP_NULL)
		return;

#if	DBG
	if (pmap_dbg & PD_FULL)
		printf ("(pmap_remove) map %x  s %x  e %x\n", map, s, e);
#endif	DBG

	CHECK_PAGE_ALIGN (s, "pmap_remove start addr");

	if (s>e)
		panic ("pmap_remove: start greater than end address");

	if (map == kernel_pmap) {
		if (s < VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_remove: kernel start address in user va space");
	}
	else {
		if (e > VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_remove: user start address in kernel va space");
	}

	PMAP_READ_LOCK(map, spl);
	pmap_remove_range(map, s, e);
	PMAP_READ_UNLOCK(map, spl);

} /* pmap_remove() */


/*
 *	Routine:	PMAP_REMOVE_ALL
 *
 *	Function:
 *		Removes this physical page from all physical maps in which it
 *		resides.  Reflects back modify bits to the pager.
 *
 *	Parameters:
 *		phys		physical address of pages which is to
 *				be removed from all maps
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *		pmap_modify_list
 *
 *	Calls:
 *		pmap_pte
 *		zfree
 *
 *    If the page specified by the given address is not a managed page,
 * this routine simply returns.  Otherwise, the PV list associated with
 * that page is traversed.  For each pmap/va pair pmap_pte is called to
 * obtain a pointer to the level 3 table entry (PTE) associated with the
 * va (the PTE must exist and be valid, otherwise the routine panics).
 * The hardware 'modified' bit in the PTE is examined.  If it is on, the
 * pmap_modify_list entry corresponding to the physical page is set to 1.
 * Then, the PTE is invalidated, and the PV list entry is unlinked and
 * freed.
 *
 *    At the end of this function, the PV list for the specified page
 * will be null.
 */
void
pmap_remove_all(phys)
	vm_offset_t	phys;
{                          
	pv_entry_t		pvl, cur;
	register pt_entry_t	*pte;
	int			pfi;
	register int		i;
	register vm_offset_t	va;
	register pmap_t		pmap;
	int			spl;
	int			dbgcnt = 0;

	if (!PMAP_MANAGED(phys))
		/* Not a managed page. */
		return;

	/*
	 *	Lock the pmap system first, since we will be changing
	 *	several pmaps.
	 */
	PMAP_WRITE_LOCK(spl);

	/*
	 * Walk down PV list, removing all mappings.
	 * We have to do the same work as in pmap_remove_pte_page
	 * since that routine locks the pv_head.  We don't have
	 * to lock the pv_head, since we have the entire pmap system.
	 */
	pfi = PFIDX(phys);
	pvl = PFIDX_TO_PVH(pfi);
	CHECK_PV_LIST (phys, pvl, "pmap_remove_all before");

	/*
	 * Loop for each entry on the pv list.
	 */
	while ((pmap = pvl->pmap) != PMAP_NULL) {
		va = pvl->va;

		simple_lock(&pmap->lock);

		pte = pmap_pte(pmap, va);

		/*
		 * Do a few consistency checks to make sure 
		 * the pv list and the pmap are in synch.
		 */
		if (pte == PT_ENTRY_NULL) {
			printf ("(pmap_remove_all) phys %x pmap %x va %x dbgcnt %x\n",
				phys, pmap, va, dbgcnt);
			panic ("pmap_remove_all: pte NULL");
		}
		if (!L3_VALID(pte))
			panic ("pmap_remove_all: pte invalid");
		if (PMAP_PTOB(pte->pfn) != phys)
			panic ("pmap_remove_all: pte doesn't point to page");
		if (pte->wired)
			panic ("pmap_remove_all: removing a wired page");

		pmap->stats.resident_count--;

		/*
		 * Tell CPU using pmap to invalidate its TLB.
		 */
		FLUSH_ATC_ENTRIES(pmap, va, va + PAGE_SIZE);

		if ((cur = pvl->next) != PV_ENTRY_NULL) {
			*pvl = *cur;
			zfree(pv_list_zone, (vm_offset_t) cur);
		}
		else
			pvl->pmap = PMAP_NULL;
		
		/*
		 * Reflect modified pages to pager.
		 */
		for (i = ptes_per_vm_page; i>0; i--) {
			if (pte->modified)
				pmap_modify_list[pfi] = 1;
			*(unsigned long *)pte++ = 0;
		}

		/*
		 * Do not free any level 3 page tables,
		 * leave that for when VM calls pmap_collect().
		 */
		simple_unlock(&pmap->lock);
		dbgcnt++;
	}
	CHECK_PV_LIST (phys, pvl, "pmap_remove_all after");
	PMAP_WRITE_UNLOCK(spl);

} /* pmap_remove_all() */



/*
 *	Routine:	PMAP_COPY_ON_WRITE
 *
 *	Function:
 *		Remove write privileges from all physical maps for this physical page.
 *
 *	Parameters:
 *		phys		physical address of page to be read-protected.
 *
 *	Calls:
 *		pmap_pte
 *
 *	Special Assumptions:
 *		All mappings of the page are user-space mappings.
 *
 *    This routine walks the PV list.  For each pmap/va pair it locates
 * the level 3 table entry (the PTE), and sets the hardware enforced
 * read-only bit.  The ATC is appropriately flushed.
 */
/* XXX
 * XXX - is it ok for pte to be invalid here ??? May be ok later, when
 * XXX		changing prot on other pages in vm_page, but here...?
 * XXX - can the first pte be invalid, but others in vm_page be ok ?? 
 * XXX
 * XXX  - must we check for kernel vs user ptes (ie. using kernel pmap) ?? 
 * XXX	  also, is ATC invalidation different for kernel_pmap
 * XXX
 * XXX - don't need to distinguish between kernel and user protections ??
 * XXX	change M68K_protections to be generic for kernel and user...
   XXX */
void
pmap_copy_on_write(phys)
	vm_offset_t	phys;
{
	register pv_entry_t	pv_e;
	register pt_entry_t	*pte;
	register int		i;
	int			spl;

	/*
	 *	Lock the entire pmap system, since we may be changing
	 *	several maps.
	 */
	PMAP_WRITE_LOCK(spl);

	pv_e = PFIDX_TO_PVH (PFIDX(phys));
	CHECK_PV_LIST (phys, pv_e, "pmap_copy_on_write before");
	if (pv_e->pmap == PMAP_NULL) {
		PMAP_WRITE_UNLOCK(spl);
		return;		/* no mappings */
	}

	/*
	 * Run down the list of mappings to this physical page,
	 * disabling write privileges on each one.
	 */
	while (pv_e != PV_ENTRY_NULL) {
		pmap_t		pmap;
		vm_offset_t	va;

		pmap = pv_e->pmap;
		va = pv_e->va;

		simple_lock(&pmap->lock);

		/*
		 *	Check for existing and valid pte.
		 */
		if (va > VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_copy_on_write: attempted on kernel page");
		pte = pmap_pte(pmap, va);
		if (pte == PT_ENTRY_NULL)
			panic ("pmap_copy_on_write: pte from pv_list not in map");
		if (!L3_VALID(pte))	
			panic ("pmap_copy_on_write: invalid pte");
		if (PMAP_PTOB(pte->pfn) != phys)
			panic ("pmap_copy_on_write: pte doesn't point to page");
             
		/*
		 *	Ask cpus using pmap to invalidate their TLBs
		 */
		FLUSH_ATC_ENTRIES(pmap, va, va + PAGE_SIZE);

		for (i=ptes_per_vm_page; i>0; i++) {
			pte->prot = M68K_UR;
			pte++;
		}

		simple_unlock(&pmap->lock);
		pv_e = pv_e->next;
	}
	CHECK_PV_LIST (phys, PFIDX_TO_PVH(PFIDX(phys)), "pmap_copy_on_write after");
	PMAP_WRITE_UNLOCK(spl);

} /* pmap_copy_on_write() */




/*
 *	Routine:	PMAP_PAGE_PROTECT
 *
 *	Function:
 *		Lower the permission for all mappings to a given page.
 *
 *	Parameters:
 *		phys		physical address of page
 *		prot		new protection attributes
 *
 *	Calls:
 *		pmap_copy_on_write
 *		pmap_remove_all
 */
void
pmap_page_protect(phys, prot)
	vm_offset_t	phys;
	vm_prot_t	prot;
{
	switch (prot) {
		case VM_PROT_READ:
		case VM_PROT_READ|VM_PROT_EXECUTE:
			pmap_copy_on_write(phys);
			break;
		case VM_PROT_ALL:
			break;
		default:
			pmap_remove_all(phys);
			break;
	}
}

/*
 *	Routine:	PMAP_PROTECT
 *
 *	Function:
 *		Set the physical protection on the specified range of this map as requested.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		s		virtual address of start of range
 *		e		virtual address of end of range
 *		prot		desired protection attributes
 *
 *	Calls:
 *		pmap_pte
 *
 *    This routine sequences through the pages of the specified range.
 * For each, it calls pmap_pte to acquire a pointer to the level 3 table
 * entry (PTE).  If the PTE is invalid, or non-existant, nothing is done.
 * Otherwise, the PTE's protection attributes are adjusted as specified.
 */
void
pmap_protect(map, s, e, prot)
	pmap_t		map;
	vm_offset_t	s, e;
	vm_prot_t	prot;
{
	pte_template_t		maprot;
	int			spl;
	register int		i;
	pt_entry_t		*pte;
	l2_entry_t		*lvl2;
	vm_offset_t		va;

	if (map == PMAP_NULL)
		return;
	
	if (s>e)
		panic ("pmap_protect: start greater than end address");

	maprot.bits = m68k_protection (map, prot);

	PMAP_READ_LOCK(map, spl);

	/*
	 *	Invalidate the transaltion buffer first
	 */
	FLUSH_ATC_ENTRIES(map, s, e);

	/*
 	 * Check range for proper bounds.
	 */
	if (map == kernel_pmap) {
		if (s < VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_protect: kernel start address in user va space"); 
	} else {
		if (e > VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_protect: user end address in kernel va space"); 
	}

	CHECK_PAGE_ALIGN (s, "pmap_protect");

	/*	
	 *	Loop through the range in vm_page_size increments.
	 *	Do not assume that either start or end fall on any
	 *	kind of page boundary (though this may be true !?).
	 */
	for (va = s; va < e; va += PAGE_SIZE) {

		if (!L1_VALID(L1ENT(map, va))) {
			va = L1_NEXT(va)-PAGE_SIZE; /* no lvl2 table, skip to next lvl1 entry */
			continue;
		}

		pte = pmap_pte(map, va);

		if (pte == PT_ENTRY_NULL) {
			va = L2_NEXT(va)-PAGE_SIZE; /* no lvl3 table, skip to next lvl2 entry */
			continue;
		}

		if (!L3_VALID(pte))
			continue;		/* no page mapping */

		for (i = ptes_per_vm_page; i>0; i--) {
			pte->prot = maprot.pte.prot;
                        pte++;
                }
	}

	PMAP_READ_UNLOCK(map, spl);

} /* pmap_protect() */


/*
 *	Routine:	PMAP_EXPAND
 *
 *	Function:
 *		Expands a pmap to be able to map the specified virtual address.
 *		New kernel virtual memory is allocated for a level 3 and perhaps a
 *		level 2 translation table.
 *
 *		Must be called with the pmap system and the pmap unlocked, since
 *		these must be unlocked to use vm_allocate or vm_deallocate (via
 *		kmem_alloc, zalloc).  Thus it must be called in a unlock/lock loop
 *		that checks whether the map has been expanded enough.  (We won't loop
 *		forever, since page tables aren't shrunk.)
 *
 *	Parameters:
 *		map		pointer to pmap structure
 *		v		VA indicating which tables are needed
 *
 *	Extern/Global:
 *		none
 *
 *	Calls:
 *		pmap_lvl2
 *		pmap_pte
 *		kmem_alloc
 *		kmem_free
 *		zalloc
 *		zfree
 *		pmap_extract
 *
 *	Special Assumptions:
 *		no pmap locks held
 *
 *	1: This routine immeditately allocates space for a level 3
 *	   translation table.  This will be used a little later in the function.
 *
 *	2: It then obtains a pointer to the level 2 translation table for
 *	   the given VA.  It the table pointer is valid, skip to step #4.
 *
 *	3: If the level 2 table does not exist, the pmap lock is released,
 *	   and space is allocated for the level 2 table.  The lock then is
 *	   re-acquired, and once again, an attempt is made to obtain the level 2
 *	   table pointer.  If, at this point, a valid pointer is returned, the
 *	   space allocated for a new level 2 table is released to the system
 *	   (some other thread expanded the map).  Otherwise, the new table is
 *	   initialized (all entries invalidated) and the appropriate level 1
 *	   table entry is set to point to this new table.
 *
 *	4: An attempt is made to obtain a pointer to the level 3 table
 *	   associated with the given virtual address.  If a valid pointer is
 *	   returned, the space allocated for the new level 3 table (see step #1)
 *	   is released to the system, and pmap_expand returns to its caller (some
 *	   other thread expanded the map).
 *
 *	5: Otherwise, the level 3 table entries (PTEs) are initialized
 *	   (set invalid), and the corresponding level 2 entry is set to point to
 *	   this new table.  (Note, if tables are grouped together into the
 *	   allocated page - i.e., non-zone table allocation - a number of level 2
 *	   entries will be set.)
 */
static 
pmap_expand(map, v)
	register pmap_t		map;
	register vm_offset_t	v;
{
	int		i,
	                spl;
	vm_offset_t	l3_vaddr,
	                l3_paddr,
	                l2_vaddr,
	                l2_paddr;
	pt_ldesc_template_t *l1;
	l2_entry_t	*lvl2;
	pt_entry_t	*pte;

	if (map == PMAP_NULL)
		return;

#if	DBG
	if (pmap_dbg & PD_PRINT)
		printf ("(pmap_expand) map %x  v %x\n", map, v);
#endif	DBG

        CHECK_PAGE_ALIGN (v, "pmap_expand");
        
	/*
	 * Handle kernel pmap in pmap_expand_kmap().
	 */
	if (map == kernel_pmap) {
		PMAP_READ_LOCK(map, spl);
		pmap_expand_kmap (v, VM_PROT_READ|VM_PROT_WRITE);
		PMAP_READ_UNLOCK(map, spl);
		return;
	}

	l3_vaddr = kmem_alloc (user_pt_map, PAGE_SIZE);

	PMAP_READ_LOCK(map, spl);

	if ((lvl2 = pmap_lvl2(map, v)) == L2_ENTRY_NULL) {
		PMAP_READ_UNLOCK(map, spl);

#if	L2_ZONE
#error this code has a bug where the memory is not necessarily *physically* contiguous
		lvl2 = (l2_entry_t *) zalloc (lvl2_zone);
		/* adjust alignment to a 16-byte boundary */
		lvl2 = (l2_entry_t *) (((int)lvl2)+15 & ~15);
#else
		lvl2 = (l2_entry_t *) kmem_alloc(user_pt_map, PAGE_SIZE);
#endif	L2_ZONE

		PMAP_READ_LOCK(map, spl);

		/*
		 * somebody else expanded to level 2 during our vm_allocate.
		 */
		if (pmap_lvl2(map, v) != L2_ENTRY_NULL)

			L2_FREE(lvl2);

		else {
			pt_ldesc_template_t	l1_template;
			vm_offset_t		l2g_va;

			/*
			 * Apply a mask to V to obtain the vaddr of the beginning of
			 * its containing level2 'table group', i.e. the group of 
			 * level 2 tables that fit within a single VM page.
			 * Using that, obtain the lvl1 pointer that references the
			 * first level2 table in the group, and initialize all the
			 * level1 table descriptors for the level2 'table group'.
			 */
			l2g_va = v & ~(L2_GROUP_VA_SPACE-1);

			/* fill in the level 2 table */
			l1 = (pt_ldesc_template_t *) L1ENT(map, l2g_va);

			l1_template.bits.lo = 0;
			l1_template.pt_ldesc.lu    = 1;
			l1_template.pt_ldesc.dtype = DT_SHORT;
#if	L2_ZONE
			bzero(lvl2, 2*L2_SIZE);
#else
			/* kmem_alloc already zeros the page. */
#endif	L2_ZONE

			l2_vaddr = (vm_offset_t) lvl2;
			l2_paddr = pmap_extract (kernel_pmap, l2_vaddr);
			for (i = L2_GROUP_SIZE; i>0; i--) {
				l1->bits.hi = l2_paddr;
				l1->bits.lo = l1_template.bits.lo;
				(l1 + L1_ENTRIES)->bits.hi = l2_vaddr;
				(l1 + L1_ENTRIES)->bits.lo = l1_template.bits.lo;
				l2_vaddr += L2_SIZE*2;
				l2_paddr += L2_SIZE*2;
				l1++;
			}
		}
	}


	if ((pte = pmap_pte(map, v)) != PT_ENTRY_NULL) {
		/*
		 * Someone else caused us to expand
		 * during our vm_allocate.
		 */
		PMAP_READ_UNLOCK(map, spl);
		kmem_free (user_pt_map, l3_vaddr, PAGE_SIZE);
		return;
	}

	l3_paddr = pmap_extract(kernel_pmap, l3_vaddr);

	/*
	 * Apply a mask to V to obtain the vaddr of the beginning of
	 * its containing level3 'table group', i.e. the group of 
	 * level 3 tables that fit within a single VM page.
	 * Using that, obtain the lvl2 pointer that references the
	 * first level3 table in the group, and initialize all the
	 * level2 table descriptors for the level3 'table group'.
	 */
	v &= ~(L3_GROUP_VA_SPACE-1);

	lvl2 = pmap_lvl2 (map, v);
	if (lvl2 == L2_ENTRY_NULL)
		panic ("pmap_expand: no level 2 entry");

	/*
	 * Init each of the L2 entries to point to the freshly allocated L3 tables.
	 */
	for (i = L3_GROUP_SIZE; i>0; i--) {
		/* XXX - protection */
		((pt_sdesc_template_t *)lvl2)->bits = l3_paddr | M68K_UW | DT_SHORT;
		((pt_sdesc_template_t *)(lvl2 + L2_ENTRIES))->bits = l3_vaddr | M68K_UW | DT_SHORT;
		lvl2++;
		l3_paddr += L3_SIZE;
		l3_vaddr += L3_SIZE;
	}

	/* XXX - is this the right place? */
	PMAP_READ_UNLOCK(map, spl);

} /* pmap_expand() */


/*
 *	Routine:	PMAP_ENTER
 *
 *	Function:
 *		Insert the given physical page (p) at the specified virtual
 *		address (v) in the target physical map with the protection requested.
 *		If specified, the page will be wired down, meaning that the
 *		related pte can not be reclaimed.
 *
 *	N.B.:	This is the only routine which MAY NOT lazy-evaluate or lose information.
 *		That is, this routine must actually insert this page into the given map NOW.
 *
 *	N.B.B.: This call is recursive, either via pmap_expand calling kmem_alloc,
 *		or via zalloc calling kmem_alloc.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		v		VA of page to be mapped
 *		pa		PA of page to be mapped
 *		prot		protection attributes for page
 *		wired		wired attribute for page
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *		pmap_modify_list
 *
 *	Calls:
 *		pmap_pte
 *		pmap_expand
 *		pmap_remove_range
 *		zfree
 *
 *    This routine starts off by calling pmap_pte to obtain a (virtual)
 * pointer to the level 3 table entry corresponding to the given virtual
 * address.  If the level 3 table itself does not exist, pmap_expand is
 * called to allocate it.
 *
 *     If the level 3 entry (PTE) already maps the given physical page,
 * all that is needed is to set the protection and wired attributes as
 * given.  ATC entries are flushed and pmap_enter returns.
 *
 *    If the level 3 entry (PTE) maps a different physical page than
 * that given, the old mapping is removed by a call to pmap_remove_range.
 * And execution of pmap_enter continues.
 *
 *    To map the new physical page, the routine first inserts a new
 * entry in the PV list exhibiting the given pmap and virtual address.
 * It then inserts the physical page address, protection attributes, and
 * wired attributes into the level 3 table entry (PTE).
 */
void
pmap_enter(pmap, v, pa, prot, wired)
	register pmap_t		pmap;
	vm_offset_t		v;
	register vm_offset_t	pa;
	vm_prot_t		prot;
	boolean_t		wired;
{    
	int		ap,
	                spl,
	                i,
	               	pfi;
	pv_entry_t	pvl,
	                pv_entry;
	vm_offset_t	old_pa;
	pte_template_t	template;
	pt_entry_t	*pte;

	if (pmap == PMAP_NULL)
		return;     

	CHECK_PAGE_ALIGN (v, "pmap_entry - VA");
	CHECK_PAGE_ALIGN (pa, "pmap_entry - PA");

	/*
 	 *	Check range for proper bounds within user/kernel va space.
	 */
	if (pmap == kernel_pmap) {
		if (v < VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_enter: Virtual address outside kernel_pmap bounds"); 
	}
        else {
		if (v > VM_MIN_KERNEL_ADDRESS)
			panic ("pmap_enter: Virtual address outside user_pmap bounds"); 
	}

#if	DBG
	if (pmap_dbg & PD_FULL) {
		if (pmap == kernel_pmap)
			printf ("(pmap_enter) pmap kernel");
		else
			printf ("(pmap_enter) pmap %x", pmap);
		printf ("  v %x  pa %x\n", v, pa);
	}
#endif	DBG

	ap = m68k_protection (pmap, prot);

	/*
	 * Must allocate a new pvlist entry while we're unlocked;
	 * zalloc may cause pageout (which will lock the pmap system).
	 * If we determine we need a pvlist entry, we will unlock
	 * and allocate one.  Then we will retry, throwing away
	 * the allocated entry later (if we no longer need it).
	 */
	pv_entry = PV_ENTRY_NULL;
Retry:
		
	PMAP_READ_LOCK(pmap, spl);

	/*
	 * Expand pmap to include this pte.  Assume that
	 * pmap is always expanded to include enough M68K
	 * pages to map one VM page.
	 */
	while ((pte = pmap_pte(pmap, v)) == PT_ENTRY_NULL) {
		/*
		 * Must unlock to expand the pmap.
		 */
		PMAP_READ_UNLOCK (pmap, spl);
		pmap_expand (pmap, v);
		PMAP_READ_LOCK (pmap, spl);
	}

	/*
	 *	Special case if the physical page is already mapped
	 *	at this address.
	 */
	old_pa = PMAP_PTOB(pte->pfn);
	if (old_pa == pa) {

#if	DBG
		if (pmap_dbg & PD_PRINT)
			printf("(pmap_enter) map %x pa %x already mapped at va %x\n", pmap, pa, v);
#endif	DBG

		/*
		 * May be changing its wired attribute or protection
		 */
		if (wired && !pte->wired) 
			pmap->stats.wired_count++;
		else if (!wired && pte->wired)
			pmap->stats.wired_count--;

		/* XXX
		 * XXX - Need to enter valid values into BOTH phys AND virt lvl2 maps
		 * XXX      since may use either to check for validity...
		   XXX */
		template.bits = DT_PAGE | ap | PMAP_TRUNC_PAGE(pa);
		if (wired)
			template.pte.wired = 1;
		FLUSH_ATC_ENTRIES(pmap, v, v + PAGE_SIZE);
		for (i = ptes_per_vm_page; i>0; i--) {
			template.pte.modified = pte->modified;
			*pte++ = template.pte;
			template.bits += PMAP_PGBYTES;
		}
	}
	else {

		/*
		 * Remove old mapping from the PV list if necessary.
		 */
		if (old_pa != (vm_offset_t) 0) {
			/*
			 *	Invalidate the translation buffer,
			 *	then remove the mapping.
			 */
#if	DBG
			if (pmap_dbg & PD_PRINT)
				printf ("    ");
#endif	DBG
			pmap_remove_range(pmap, v, v + PAGE_SIZE);
		}

		if (PMAP_MANAGED(pa)) {

			/*
			 *	Enter the mapping in the PV list for this
			 *	physical page.
			 */
			pfi = PFIDX(pa);
			LOCK_PVH(pfi);
			pvl = PFIDX_TO_PVH(pfi);
			CHECK_PV_LIST (pa, pvl, "pmap_enter before");

			if (pvl->pmap == PMAP_NULL) {
				/*
				 *	No mappings yet
				 */
				pvl->va = v;
				pvl->pmap = pmap;
				pvl->next = PV_ENTRY_NULL;
				/* xxx - vax code does not do this. could be a serious bug. zzz */
				pmap_modify_list[pfi] = 0;
			}
			else {
#ifdef	DBG
				pv_entry_t pve;

				/*
				 * check that this mapping is not already there
				 */
				for (pve = pvl; pve != PV_ENTRY_NULL; pve = pve->next)
					if (pve->pmap == pmap && pve->va == v)
						panic ("pmap_enter: already in pv_list");
#endif	DBG
		    
			    	/*
				 *	Add new pv_entry after header.
				 */
				if (pv_entry == PV_ENTRY_NULL) {
					UNLOCK_PVH(pfi);
					PMAP_READ_UNLOCK(pmap, spl);
					pv_entry = (pv_entry_t) zalloc(pv_list_zone);
					goto Retry;
				}
				pv_entry->va = v;
				pv_entry->pmap = pmap;
				pv_entry->next = pvl->next;
				pvl->next = pv_entry;
				/* Remember that we used the pvlist entry. */
				pv_entry = PV_ENTRY_NULL;
			}
			UNLOCK_PVH(pfi);
		}


		/*
		 * And count the mapping.
		 */
		pmap->stats.resident_count++;
		if (wired) 
			pmap->stats.wired_count++;

		/* XXX
		 * XXX - Need to enter valid values into BOTH phys AND virt lvl2 maps
		 * XXX      since may use either to check for validity...
		   XXX */
		template.bits = DT_PAGE | ap | PMAP_TRUNC_PAGE(pa);
		if (wired)
			template.pte.wired = 1;

		DO_PTES (pte, template.bits);
	}

	PMAP_READ_UNLOCK(pmap, spl);

	if (pv_entry != PV_ENTRY_NULL)
		zfree(pv_list_zone, (vm_offset_t) pv_entry);

} /* pmap_enter() */


/*
 *	Routine:	pmap_change_wiring
 *
 *	Function:	Change the wiring attribute for a map/virtual-address
 *			pair.
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		v		virtual address of page to be wired/unwired
 *		wired		flag indicating new wired state
 *
 *	Special Assumptions:
 *		The mapping must already exist in the pmap.
 */
void
pmap_change_wiring (map, v, wired)
	register pmap_t	map;
	vm_offset_t	v;
	boolean_t	wired;
{

	register pt_entry_t	*pte;
	register int		i;
	int			spl;

	PMAP_READ_LOCK(map, spl);

	if ((pte = pmap_pte(map, v)) == PT_ENTRY_NULL)
		panic ("pmap_change_wiring: pte missing");

	if (wired && !pte->wired)
		/*
		 *	wiring down mapping
		 */
		map->stats.wired_count++;

	else if (!wired && pte->wired)
		/*
		 *	unwiring mapping
		 */
		map->stats.wired_count--;

	for (i = ptes_per_vm_page; i>0; i--)
		(pte++)->wired = wired;

	FLUSH_ATC_ENTRIES(pmap, v, v + PAGE_SIZE);

	PMAP_READ_UNLOCK(map, spl);

} /* pmap_change_wiring() */


/*
 *	Routine:	pmap_change_caching
 *
 *	Function:	Change the caching attribute for a map/virtual-address
 *			pair.
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		v		virtual address of page to be cached/uncached
 *		cached		flag indicating new cached state
 *
 *	Special Assumptions:
 *		The mapping must already exist in the pmap.
 */
void
pmap_change_caching (map, v, cached)
	register pmap_t	map;
	vm_offset_t	v;
	boolean_t	cached;
{

	register pt_entry_t	*pte;
	register int		i;
	int			spl;

	PMAP_READ_LOCK(map, spl);

	if ((pte = pmap_pte(map, v)) == PT_ENTRY_NULL)
		panic ("pmap_change_caching: pte missing");

	for (i = ptes_per_vm_page; i>0; i--)
		(pte++)->no_cache = !cached;

	FLUSH_ATC_ENTRIES(pmap, v, v + PAGE_SIZE);

	PMAP_READ_UNLOCK(map, spl);

} /* pmap_change_caching() */


/*
 *	Routine:	PMAP_EXTRACT
 *
 *	Function:
 *		Extract the physical page address associated
 *		with the given map/virtual_address pair.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		va		virtual address
 *
 *	Calls:
 *		pmap_pte
 *
 *	Special Assumptions:
 *		It assumes that invalid level 3 table entries have a zero page address.
 *
 *    This routine simply calls pmap_pte to get a (virtual) pointer to
 * the level 3 table entry (PTE) associated with the given virtual
 * address.  If the level 3 table does not exist, a 0 address is
 * returned.  Otherwise, the physical page address from the PTE is returned.
 */
/*      XXX
 *	XXX - See vax:  they check pte_page valid...
 *	XXX - pte_page_valid vs. PT_ENTRY_NULL vs. pte_valid
        XXX */
vm_offset_t
pmap_extract(pmap, va)
	register pmap_t	pmap;
	vm_offset_t	va;
{
	register pt_entry_t	*pte;
	register vm_offset_t	pa;
	int			spl;

	PMAP_READ_LOCK(pmap, spl);
	if ((pte = pmap_pte(pmap, va)) == PT_ENTRY_NULL)
		pa = (vm_offset_t) 0;
	else
		pa = PMAP_PTOB(pte->pfn);

	if (pa)
		pa |= (va & PMAP_PGOFSET);	/* offset within page */

	PMAP_READ_UNLOCK(pmap, spl);

	return(pa);

} /* pmap_extract() */


/*
 *	Routine:	PMAP_COPY
 *
 *	Function:
 *		Copy the range specified by src_addr/len from the source map
 *		to the range dst_addr/len in the destination map.  This routine
 *		is only advisory and need not do anything.
 *
 *	Parameters:
 *		dst_pmap	pointer to destination pmap structure
 *		src_pmap	pointer to source pmap structure
 *		dst_addr	VA in destination map
 *		len		length of address space being copied
 *		src_addr	VA in source map
 *
 *    At this time, the 68030 pmap implementation does nothing in this
 * function.  Translation tables in the destination map will be allocated
 * at VM fault time.
 */
void
pmap_copy(dst_pmap, src_pmap, dst_addr, len, src_addr)
	pmap_t		dst_pmap;
	pmap_t		src_pmap;
	vm_offset_t	dst_addr;
	vm_size_t	len;
	vm_offset_t	src_addr;
{
#ifdef	lint
	dst_pmap++; src_pmap++; dst_addr++; len++; src_addr++;
#endif	lint
}


/*
 *	Routine:	PMAP_UPDATE
 *
 *	Function:
 *		Require that all active physical maps contain no incorrect entries
 *		NOW.  [This update includes forcing updates of any address map
 *		caching.]
 *		Generally used to ensure that a thread about to run will see a
 *		semantically correct world.
 *
 *	Parameters:
 *		none
 *
 *    The 68030 pmap implementation does not defer any operations.
 * Therefore, the translation table trees are always consistent while the
 * pmap lock is not held.  Therefore, there is really no work to do in
 * this function other than to flush the ATC.
 */
void
pmap_update()
{
	FLUSH_ATC_ALL();
}


/*
 *	Routine:	PMAP_COLLECT
 *
 *	Function:
 *		Garbage collects the physical map system for pages which are
 *		no longer used.  There may well be pages which are not
 *		referenced, but others may be collected as well.
 *		Called by the pageout daemon when pages are scarce.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *
 *	Calls:
 *		kmem_free
 *		zfree
 *		pmap_lvl2
 *		pmap_pte
 *		pmap_remove_range
 *
 *     The intent of this routine is to release memory pages being used
 * by translation tables.  They can be release only if they contain no
 * valid mappings, and their parent table entry has been invalidated.
 *
 *     The routine sequences through the entire user address space,
 * inspecting page-sized groups of level 3 tables for wired entries.  If
 * a full page of tables has no wired entries, any otherwise valid
 * entries are invalidated (via pmap_remove_range).  Then, the level 2
 * table entries corresponding to this group of level 3 tables are
 * invalidated.  Finally, kmem_free is called to return the page to the
 * system.
 *
 *     If all entries in a level 2 table are invalidated, it too can
 * be returned to the system.  The corresponding level 1 table entry
 * is invalidated, and zfree called to return the space to the zone.
 * 
 *     [Note: depending upon compilation options, tables may be in zones
 * or allocated through kmem_alloc.  In the later case, the module deals
 * with groups of tables that fill a page.  In the former case, the
 * module deals with a single table at a time.]
 */
void
pmap_collect(pmap)
     pmap_t		pmap;
{

	vm_offset_t	l1va;	/* outer loop index */
	vm_offset_t	l2va;	/* inner loop index */

	l1_entry_t	*l1grp;		/* ptr to first entry in the level 1 table */
	l1_entry_t	*l1p;		/* ptr to index into level 1 table */
	l1_entry_t	*l1;		/* ptr to index into level 1 table */
	
	l2_entry_t	*l2grp;		/* ptr to first entry in a level 2 table */
	l2_entry_t	*l2p;		/* ptr to index into a level 2 table */
	l2_entry_t	*l2;		/* ptr to index into a level 2 table */

	pt_entry_t	*l3grp;		/* ptr to first entry in a level 3 table */
	pt_entry_t	*l3grpend;	/* ptr to byte after last entry in table group */
	pt_entry_t	*l3p;		/* ptr to index into a level 3 table */

	boolean_t	l2_wired;	/* flag indicating a wired page exists in a
					level 2 table's address range */
	boolean_t	l3_wired;	/* flag indicating a wired page exists in a
					level 3 table's address range */
	int		spl;


	if (pmap == PMAP_NULL)
		return;

	CHECK_PMAP_CONSISTENCY ("pmap_collect");

	if (pmap == kernel_pmap)
		panic("pmap_collect attempted on kernel pmap");

#if	DBG
	if (pmap_dbg & PD_PRINT)
		printf ("(pmap_collect) pmap %x\n", pmap);
#endif	DBG

	PMAP_READ_LOCK(pmap, spl);

	l1grp = pmap->lvl1_vaddr;	/* addr of level 1 table */
	l1p = l1grp;

	/*
	 * for each Level 2 Table Group,
	 * see if we can free the VM page holding the associated L2 page tables.
	 */
	for (l1va = VM_MIN_ADDRESS;
	     l1va < VM_MIN_KERNEL_ADDRESS;
	     l1va += L2_GROUP_VA_SPACE, l1p += L2_GROUP_SIZE) {

		l2grp = pmap_lvl2(pmap, l1va);
		if (l2grp == L2_ENTRY_NULL)
			continue;	/* no page tables in this range */
		l2p = l2grp;

		/* keep track of whether any pages in the range are wired */
		l2_wired = FALSE;

		/*
		 * Loop through the VA's in this L2 Group.
		 */
		for (l2va = l1va;
		     l2va < (l1va + L2_GROUP_VA_SPACE); /* toml: was L2_VA_SPACE */
		     l2va += L3_GROUP_VA_SPACE, l2p += L3_GROUP_SIZE) {

			l3grp = pmap_pte(pmap, l2va);

			if (l3grp == PT_ENTRY_NULL)
				continue;	/* no maps in this range */

			l3grpend = l3grp + (L3_ENTRIES * L3_GROUP_SIZE);

			/*
			 * Loop through the pte's in this Level 3 Page Table Group.
			 */
			l3_wired = FALSE;
			for (l3p=l3grp; l3p < l3grpend; l3p++) {
				if (l3p->wired) {
					l3_wired = TRUE;
					l2_wired = TRUE;
					break;
				}
			}

			/*
			 * Free the Level 3 Page Table Group.
			 */
			if (!l3_wired) {

				/* remove all page tables in this range */
				pmap_remove_range (pmap, l2va, l2va + L3_GROUP_VA_SPACE);

				/* invalidate the proper l2 entry(s) */
				for (l2 = l2p; l2 < (l2p+L3_GROUP_SIZE); l2++) {
					((pt_sdesc_template_t *) l2)->bits = 0;
					((pt_sdesc_template_t *) l2+L2_ENTRIES)->bits = 0;
				}

				/*
				 * now that we invalidated associated level 2 entry(s),
				 * we can safely deallocate the level 3 map(s)
				 */
				PT_FREE(l3grp);
			}

		} /* Level 2 Loop */

		/*
		 * Free the Level 2 Page Table Group.
		 */
		if (!l2_wired) {
			for (l1 = l1p; l1 < (l1p+L2_GROUP_SIZE); l1++) {
				((pt_ldesc_template_t *) l1) -> bits.lo = 0;
				((pt_ldesc_template_t *) l1) -> bits.hi = 0;
				((pt_ldesc_template_t *) l1+L1_ENTRIES) -> bits.lo = 0;
				((pt_ldesc_template_t *) l1+L1_ENTRIES) -> bits.hi = 0;
			}
			L2_FREE(l2grp);
		}

	} /* Level 1 Loop */

	PMAP_READ_UNLOCK(pmap, spl);

#if	DBG
	if (pmap_dbg & PD_PRINT)
		printf ("(pmap_collect) done\n");
#endif	DBG

} /* pmap_collect() */


/*
 *	Routine:	PMAP_ACTIVATE
 *
 *	Function:
 *		Binds the given physical map to the given
 *		processor, and returns a hardware map description.
 *		In a mono-processor implementation the my_cpu
 *		argument is ignored, and the PMAP_ACTIVATE macro
 *		simply sets the MMU root pointer element of the PCB
 *		to the physical address of the level 1 page table.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		th		pointer to thread structure
 *		cpu		CPU number
 */
void
pmap_activate(my_pmap, th, my_cpu)
	register pmap_t	my_pmap;
	thread_t	th;
	int		my_cpu;
{
#ifdef	lint
	/* XXX - careful !! */
	my_cpu++;
#endif	lint	
	PMAP_ACTIVATE(my_pmap, th, my_cpu);
}


/*
 *	Routine:	PMAP_DEACTIVATE
 *
 *	Function:
 *		Unbinds the given physical map from the given processor,
 *		i.e. the pmap is no longer is use on the processor.
 *		In a mono-processor the PMAP_DEACTIVATE macro is null.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		th		pointer to thread structure
 *		cpu		CPU number
 */
void
pmap_deactivate(pmap, th, which_cpu)
	pmap_t		pmap;
	thread_t	th;
	int		which_cpu;
{
#ifdef	lint
	pmap++; th++; which_cpu++;
#endif	lint
	PMAP_DEACTIVATE(pmap, th, which_cpu);
}


/*
 *	Routine:	PMAP_KERNEL
 *
 *	Function:
 *		Returns a pointer to the kernel pmap.
 */
pmap_t
pmap_kernel()
{
    	return (kernel_pmap);
}


/*
 *	Routine:	PMAP_ZERO_PAGE
 *
 *	Function:
 *		Zeros the specified (machine independent) page.
 *
 *	Parameters:
 *		phys		PA of page to zero
 *
 *	Extern/Global:
 *		pmap_scratch_pte1
 *		pmap_scratch_va1
 *
 *	Special Assumptions:
 *		no locking required
 *
 *    This routine maps the physical pages at the 'phys_map' virtual
 * address set up in pmap_bootstrap.  It flushes the ATC to make the new
 * mappings effective, and zeros all the bits.
 */
pmap_zero_page(phys)
	register vm_offset_t	phys;
{
	unsigned long	template;
	pt_entry_t	*pte;
	int		i;

	template = PMAP_TRUNC_PAGE(phys)
	           | m68k_protection (kernel_pmap, VM_PROT_READ | VM_PROT_WRITE)
	           | DT_PAGE;

	/* XXX use PLOAD */
	FLUSH_ATC_ALL();
        DO_PTES (pmap_scratch_pte1, template);
	bzero (pmap_scratch_va1, PAGE_SIZE);

} /* pmap_zero_page() */



/*
 *	Routine:	PMAP_COPY_PAGE
 *
 *	Function:
 *		Copies the specified (machine independent) pages.
 *
 *	Parameters:
 *		src		PA of source page
 *		dst		PA of destination page
 *
 *	Extern/Global:
 *		pmap_scratch_pte1
 *		pmap_scratch_va1
 *		pmap_scratch_pte2
 *		pmap_scratch_va2
 *
 *	Special Assumptions:
 *		no locking required
 *
 *    This routine maps the physical pages at the 'phys_map' virtual
 * addresses set up in pmap_bootstrap.  It flushes the ATC to make the
 * new mappings effective, and performs the copy.
 */
pmap_copy_page(src, dst)
	vm_offset_t	src, dst;
{
	int		aprot;
	pte_template_t	template;

	/*
	 *	Map source physical address.
	 */
	aprot = m68k_protection (kernel_pmap, VM_PROT_READ | VM_PROT_WRITE);
	/* XXX - PROTECTION ?! */
	template.bits = PMAP_TRUNC_PAGE(src) | aprot | DT_PAGE;
	/* XXX use PLOAD */
	FLUSH_ATC_ALL();
	DO_PTES (pmap_scratch_pte1, template.bits);

	/*
	 *	Map destination physical address.
	 */
	/* XXX - PROTECTION ?! */
	template.bits = PMAP_TRUNC_PAGE(dst) | aprot | DT_PAGE;
	DO_PTES (pmap_scratch_pte2, template.bits);

	bcopy(pmap_scratch_va1, pmap_scratch_va2, PAGE_SIZE);

} /* pmap_copy_page() */


                                                                                              
/*
 *	Routine:	PMAP_MOVE_PAGE
 *
 *	Function:
 *		Moves the specified (machine independent) pages
 *		in the kernel's VA space by tweaking the page tables.
 *
 */
pmap_move_page (from, to, size)
	register vm_offset_t from, to;
	int size;
{
	register pt_entry_t *fpte, *tpte;

	if (size % PAGE_SIZE)
		panic("pmap_move_page");

	for ( ; size>0; size -= PMAP_PGBYTES) {
		fpte = pmap_pte(kernel_pmap, from);
		tpte = pmap_pte(kernel_pmap, to);
		bcopy (fpte, tpte, sizeof(pt_entry_t));
		bzero (fpte, sizeof(pt_entry_t));

		{
			/*
			 * Keep the PMEM list in synch.
			 * Right now pmap_move_page is only called from
			 * allocbuf(), and since the buffer cache is
			 * permanently wired, inconsistencies between
			 * the PMEM list and the kernel pmap do not
			 * currently cause problems.
			 * But this could be called from elsewhere.
			 */
			pv_entry_t  pve;
			vm_offset_t phys;
			int found = 0;

			phys = PMAP_PTOB(tpte->pfn);
			for (pve = PFIDX_TO_PVH(PFIDX(phys)); pve && pve->pmap; pve = pve->next)
				if (pve->pmap == kernel_pmap && pve->va == from) {
					found++;
					pve->va = to;
					break;
				}
			if (!found)
				panic ("pmap_move_page: kernel pmap not in synch");
			else if (pmap_dbg & PD_PRINT)
				printf ("pmap_move_page: moved from %x to %x, phys %x\n",
					from, to, phys);
		}

		from += PMAP_PGBYTES;
		to   += PMAP_PGBYTES;
	}

	FLUSH_ATC_ALL();

} /* pmap_move_page() */


/*
 *	Routine:	PMAP_PAGEABLE
 *
 *	Function:
 *		Make the specified pages (by pmap, offset) pageable (or not) as
 *		requested.  A page which is not pageable may not take a fault;
 *		therefore, its page table entry must remain valid for the duration.
 *		This routine is merely advisory; pmap_enter will specify that
 *		these pages are to be wired down (or not) as appropriate.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		start		virtual address of start of range
 *		end		virtual address of end of range
 *		pageable	flag indicating whether range is to be pageable.
 *
 *	This routine currently does nothing in the 68030 implemetnation.
 */
pmap_pageable(pmap, start, end, pageable)
	pmap_t		pmap;
	vm_offset_t	start;
	vm_offset_t	end;
	boolean_t	pageable;
{
#ifdef	lint
	pmap++; start++; end++; pageable++;
#endif	lint
}

/*
 *	Routine:	PMAP_REDZONE
 *
 *	Function:
 *		Give the kernel read-only access to the specified address.  This
 *		is used to detect stack overflows.  It is assumed that the address
 *		specified is the last possible kernel stack address.  Therefore, we
 *		round up to the nearest machine dependent page.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		addr		virtual address of page to which access should
 *				be restricted to read-only
 *
 *	Calls:
 *		pmap_pte
 *
 *    This function calls pmap_pte to obtain a pointer to the level 3
 * table entry associated with the given virtual address.  If there is a
 * level 3 entry, and it is valid, its write protect bit will be set.
 */
void
pmap_redzone(pmap, addr)
	pmap_t		pmap;
	vm_offset_t	addr;
{
	pt_entry_t	*pte;
	int		spl;

	addr = PMAP_ROUND_PAGE(addr);
	PMAP_READ_LOCK(pmap,spl);

	if ((pte = pmap_pte(pmap, addr)) != PT_ENTRY_NULL  &&  L3_VALID(pte))
		pte->prot = M68K_KR;
	FLUSH_ATC_ENTRIES(pmap, addr, addr + PAGE_SIZE);

	PMAP_READ_UNLOCK(pmap,spl);
}



/*
 *	Routine:	PMAP_CLEAR_MODIFY
 *
 *	Function:
 *		Clear the modify bits on the specified physical page.
 *
 *	Parameters:
 *		phys		physical address of page
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *		pmap_modify_list
 *
 *    For managed pages, the modify_list entry corresponding to the
 * page's frame index will be zeroed.  The PV list will be traversed.
 * For each pmap/va the hardware 'modified' bit in the level 3 table
 * entry inspected - and turned off if necessary.  If any of the
 * inspected bits were found on, an ATC flush will be performed.
 */
void
pmap_clear_modify(phys)
	register vm_offset_t	phys;
{
	pv_entry_t	pvl;
	int		pfi;
	pv_entry_t	pvep;
	pt_entry_t	*ptep;
	int		spl;
	boolean_t	need_flush;
	int		i;

	if (!PMAP_MANAGED(phys))
		/* XXX or should it PANIC? */
		return;

	PMAP_WRITE_LOCK(spl);

	pfi = PFIDX(phys);
	pvl = PFIDX_TO_PVH(pfi);
	CHECK_PV_LIST (phys, pvl, "pmap_clear_modify");

	/* update corresponding pmap_modify_list element */
	pmap_modify_list[pfi] = 0;

	if (pvl->pmap == PMAP_NULL) {
		PMAP_WRITE_UNLOCK(spl);
		return;
	}

	/* for each listed pmap, turn off the page modified bit */
	pvep = pvl;
	while (pvep != PV_ENTRY_NULL) {
		simple_lock(&pvep->pmap->lock);
	  	ptep = pmap_pte(pvep->pmap, pvep->va);
		if (ptep == PT_ENTRY_NULL)
		  	panic("pmap_clear_modify: bad pv list entry.\n");
		need_flush = FALSE;
		for (i = ptes_per_vm_page; i>0; i--) {
			if (ptep->modified) {
				ptep->modified = 0;	/* CLEAR USED BIT */
				need_flush = TRUE;
				/* XXX actually, only need to flush entries
				   XXX if pvep->pmap is the current pmap */
			}
			ptep++;
		}

		if (need_flush)
			FLUSH_ATC_ENTRIES(pvep->pmap, pvep->va, pvep->va+PAGE_SIZE);

		simple_unlock(&pvep->pmap->lock);

		pvep = pvep->next;
	}

	PMAP_WRITE_UNLOCK(spl);

} /* pmap_clear_modify() */



/*
 *	Routine:	PMAP_IS_MODIFIED
 *
 *	Function:
 *		Return whether or not the specified physical page is modified by any
 *		physical maps.  That is, whether the hardware has stored data into the page.
 *
 *	Parameters:
 *		phys		physical address of a page
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *		pmap_modify_list
 *
 *	Calls:
 *		pmap_pte
 *
 *    If the physical address specified is not a managed page, this
 * routine simply returns TRUE.
 *
 *    If the entry in the modify list, corresponding to the given page,
 * is TRUE, this routine returns TRUE.  (This means at least one mapping
 * has been invalidated where the MMU had set the modified bit in the
 * level 3 table entry (PTE).
 * 
 *    Otherwise, this routine walks the PV list corresponding to the
 * given page.  For each pmap/va pair, the level 3 table entry is
 * examined.  If a modified bit is found on, the function returns TRUE
 * immediately (doesn't need to walk remainder of list).
 */
boolean_t
pmap_is_modified(phys)
	register vm_offset_t	phys;
{
	pv_entry_t	pvl;
	int		pfi;
	pv_entry_t	pvep;
	pt_entry_t	*ptep;
	int		spl;
	int		i;
	boolean_t	modified_flag;

	if (!PMAP_MANAGED(phys))
		/* XXX or should it be TRUE? */
		return(FALSE);

	PMAP_WRITE_LOCK(spl);

	pfi = PFIDX(phys);
	pvl = PFIDX_TO_PVH(pfi);
	CHECK_PV_LIST (phys, pvl, "pmap_is_modified");

	if ((boolean_t) pmap_modify_list[pfi]) {
		/* we've already cached a modify flag for this page,
			no use looking further... */
		PMAP_WRITE_UNLOCK(spl);
		return(TRUE);
	}

	if (pvl->pmap == PMAP_NULL) {
		/* unmapped page - get info from page_modified array
			maintained by pmap_remove_range/ pmap_remove_all */
		modified_flag = (boolean_t) pmap_modify_list[pfi];
		PMAP_WRITE_UNLOCK(spl);
		return(modified_flag);
	}

	/* for each listed pmap, check modified bit for given page */
	pvep = pvl;
	while (pvep != PV_ENTRY_NULL) {
		simple_lock(&pvep->pmap->lock);
	  	ptep = pmap_pte(pvep->pmap, pvep->va);
		if (ptep == PT_ENTRY_NULL)
		  	panic("pmap_is_modified: bad pv list entry.\n");
		for (i=ptes_per_vm_page; i>0; i--) {
			if (ptep->modified) {
				simple_unlock(&pvep->pmap->lock);
			  	PMAP_WRITE_UNLOCK(spl);
				return(TRUE);
			}
			ptep++;
		}
		simple_unlock(&pvep->pmap->lock);

		pvep = pvep->next;
	}

	PMAP_WRITE_UNLOCK(spl);
	return(FALSE);

} /* pmap_is_modified() */



/*
 *	Routine:	PMAP_CLEAR_REFERENCE
 *
 *	Function:
 *		Clear the reference bits on the specified physical page.
 *
 *	Parameters:
 *		phys		physical address of page
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *
 *    For managed pages, the corresponding PV list will be traversed.
 * For each pmap/va the hardware 'used' bit in the level 3 table entry
 * inspected - and turned off if necessary.  If any of the inspected bits
 * were found on, an ATC flush will be performed.
 */
void
pmap_clear_reference(phys)
	vm_offset_t	phys;
{
	pv_entry_t	pvl;
	int		pfi;
	pv_entry_t	pvep;
	pt_entry_t	*ptep;
	int		spl;
	boolean_t	need_flush;
	int		i;

	if (!PMAP_MANAGED(phys))
		/* XXX or should it PANIC? */
		return;

	PMAP_WRITE_LOCK(spl);

	pfi = PFIDX(phys);
	pvl = PFIDX_TO_PVH(pfi);
	CHECK_PV_LIST (phys, pvl, "pmap_clear_reference");

	if (pvl->pmap == PMAP_NULL) {
		PMAP_WRITE_UNLOCK(spl);
		return;
	}

	/* for each listed pmap, turn off the page referenced bit */
	pvep = pvl;
	while (pvep != PV_ENTRY_NULL) {
		simple_lock(&pvep->pmap->lock);
	  	ptep = pmap_pte(pvep->pmap, pvep->va);
		if (ptep == PT_ENTRY_NULL)
		  	panic("pmap_clear_reference: bad pv list entry.\n");
		need_flush = FALSE;
		for (i = ptes_per_vm_page; i>0; i--) {
			if (ptep->pg_used) {
				ptep->pg_used = 0;	/* CLEAR USED BIT */
				need_flush = TRUE;
				/* XXX actually, only need to flush entries
				   XXX if pvep->pmap is the current pmap */
			}
			ptep++;
		}

		if (need_flush)
			FLUSH_ATC_ENTRIES(pvep->pmap, pvep->va, pvep->va+PAGE_SIZE);

		simple_unlock(&pvep->pmap->lock);

		pvep = pvep->next;
	}

	PMAP_WRITE_UNLOCK(spl);

} /* pmap_clear_reference() */


/*
 *	Routine:	PMAP_IS_REFERENCED
 *
 *	Function:
 *		Return whether or not the specified physical page is referenced by
 *		any physical maps.  That is, whether the hardware has touched the page.
 *
 *	Parameters:
 *		phys		physical address of a page
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *
 *	Calls:
 *		pmap_pte
 *
 *    If the physical address specified is not a managed page, this
 * routine simply returns TRUE.
 *
 *    Otherwise, this routine walks the PV list corresponding to the
 * given page.  For each pmap/va pair, the level 3 table entry is
 * examined.  If a used bit is found on, the function returns TRUE
 * immediately (doesn't need to walk remainder of list).
 */
boolean_t
pmap_is_referenced(phys)
	vm_offset_t	phys;
{
	pv_entry_t	pvl;
	int		pfi;
	pv_entry_t	pvep;
	pt_entry_t	*ptep;
	int		spl;
	int		i;

	if (!PMAP_MANAGED(phys))
		/* XXX or should it be TRUE? */
		return(FALSE);

	PMAP_WRITE_LOCK(spl);

	pfi = PFIDX(phys);
	pvl = PFIDX_TO_PVH(pfi);
	CHECK_PV_LIST (phys, pvl, "pmap_is_referenced");

	if (pvl->pmap == PMAP_NULL) {
		PMAP_WRITE_UNLOCK(spl);
		return(FALSE);
	}

	/* for each listed pmap, check used bit for given page */
	pvep = pvl;
	while (pvep != PV_ENTRY_NULL) {
		simple_lock(&pvep->pmap->lock);
	  	ptep = pmap_pte(pvep->pmap, pvep->va);
		if (ptep == PT_ENTRY_NULL)
		  	panic("pmap_is_referenced: bad pv list entry.\n");
		for (i=ptes_per_vm_page; i>0; i--) {
			if (ptep->pg_used) {
				simple_unlock(&pvep->pmap->lock);
			  	PMAP_WRITE_UNLOCK(spl);
				return(TRUE);
			}
			ptep++;
		}
		simple_unlock(&pvep->pmap->lock);

		pvep = pvep->next;
	}

	PMAP_WRITE_UNLOCK(spl);
	return(FALSE);

} /* pmap_is_referenced() */



/*
 *	Routine:	PMAP_VERIFY_FREE
 *
 *	Function:
 *		Check whether the given page is really not mapped in any pmap
 *
 *	Parameters:
 *		phys		physical address of page in question
 */
boolean_t
pmap_verify_free( phys )
	vm_offset_t phys;
{
	pv_entry_t	pvl;
	int		pfi;

	/* XXX is this really needed (was part of cut/paste from mips pmap.c) */
	if (pmap_phys_end == (vm_offset_t)0)	/* are we initialized? */
		return(TRUE);			/* NO, everything's free */

	if (!PMAP_MANAGED(phys))
		return(FALSE);


	pfi = PFIDX(phys);
	pvl = PFIDX_TO_PVH(pfi);
	CHECK_PV_LIST (phys, pvl, "pmap_verify_free");

	/* this looks exactly backwards to me - toml */
	if (pvl->pmap == PMAP_NULL)
		return(FALSE);
	else
		return(TRUE);
} /* pmap_verify_free() */



#ifdef	DBG

/*
 * DEBUG ROUTINES - check_pv_list and check_pmap_consistency are used
 *			only for debugging.  They are invoked only
 *			through the macros CHECK_PV_LIST and CHECK_PMAP_CONSISTENCY
 *			defined early in this sourcefile.
 */

#define WORRY(who)                                        \
	if (pmap_dbg & PD_WORRY)                          \
		printf ("%s: WARNING ...\n", who);        \
	else                                              \
		panic (who)

/*
 *	Routine:	CHECK_PV_LIST (internal)
 *
 *	Function:
 *		Debug-mode routine to check consistency of a PV list.
 *		For the given paddr, it checks each pmap that should
 *		map that paddr, making sure it does map it at that same paddr.
 *
 *	Parameters:
 *		phys		physical address of page whose PV list is
 *				to be checked
 *		pv_h		pointer to head ot the PV list
 *		who		string containing caller's name to be
 *				printed if a panic arises
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *
 *	Calls:
 *		pmap_extract
 *
 *	Special Assumptions:
 *		No locking is required.
 *
 *    This function walks the given PV list.  For each pmap/va pair,
 * pmap_extract is called to obtain the physical address of the page from
 * the pmap in question.  If the returned physical address does not match
 * that for the PV list being perused, the function panics.
 */
static
check_pv_list(phys, pv_h, who)
	vm_offset_t	phys;
	pv_entry_t	pv_h;
	char		*who;
{
	vm_offset_t	pa;
	pv_entry_t	pv_e;

	if (pv_h != PFIDX_TO_PVH(PFIDX(phys))) {
		printf("check_pv_list: incorrect pv_h supplied.\n");
		WORRY(who);
	}

	if (!PAGE_ALIGNED(phys)) {
		printf("check_pv_list: supplied phys addr not page aligned.\n");
		WORRY(who);
	}

	if (pv_h->pmap == PMAP_NULL) {
		if (pv_h->next != PV_ENTRY_NULL) {
			printf("check_pv_list: first entry has null pmap, but list non-empty.\n");
			WORRY(who);
		}
		else
			return;		/* proper empty list */
	}

	for (pv_e = pv_h; pv_e != PV_ENTRY_NULL; pv_e = pv_e->next) {
		if (!PAGE_ALIGNED(pv_e->va)) {
			printf("check_pv_list: non-aligned VA, entry %x  va %x\n", pv_e, pv_e->va);
			WORRY(who);
		}
		if ((pmap_dbg & PD_CHECK_KERNEL) || (pv_e->pmap != kernel_pmap)) {
			pa = pmap_extract(pv_e->pmap, pv_e->va);
			if (pa != phys) {
				printf("check_pv_list: phys addr diff, entry %X  phys %x  pa %x\n", pv_e, phys, pa);
				WORRY(who);
			}
		}
	}

} /* check_pv_list() */


/*
 *	Routine:	CHECK_MAP (internal)
 *
 *	Function:
 *		Debug mode routine to check consistency of a map.
 *		Called by check_pmap_consistency only.
 *
 *	Parameters:
 *		p		pointer to pmap structure
 *		s		start of range to be checked
 *		e		end of range to be checked
 *		who		string containing caller's name to be
 *				printed if a panic arises
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *         
 *	Calls:
 *		pmap_pte
 *
 *	Special Assumptions:
 *		No locking required.
 *
 *    This function sequences through the given range of addresses.  For
 * each page, pmap_pte is called to obtain the level 3 table entry.  If
 * its valid, and the physical page it maps is managed, the PV list is
 * searched for the corresponding pmap/va entry.  If not found, the
 * function panics.  If duplicate PV list entries are found, the function
 * panics.
 * XXX
 * XXX- check modify list entry associated with each page
 */
static
check_map(map, s, e, who)
	pmap_t		map;
	vm_offset_t	s;
	vm_offset_t	e;
	char		*who;
{
	vm_offset_t	va,
	                old_va,
	                phys;
	pv_entry_t      pv_h,
	                pv_e,
	                saved_pv_e;
	pt_entry_t	*ptep;
	boolean_t	found;
	int		loopcnt;


	/*
	 * for each page in the address space, check to see if there's
	 * a valid mapping.  If so, make sure it's listed in the pv_list.
	 */
	if (pmap_dbg & PD_PRINT)
		printf("(check_map) map %X\n", map);

	old_va = s;
	for (va = s; va < e; va += PAGE_SIZE) {

		if (va < old_va)	/* check for overflow - happens if e=0xffffffff */
			break;
		else
			old_va = va;

		/*
		 * These two pages of kernel VA space are not managed in the pv list.
		 */
		if (va == pmap_scratch_va1 || va == pmap_scratch_va2)
			continue;

		if (!L1_VALID(L1ENT(map, va))) {
			va = L1_NEXT(va)-PAGE_SIZE; /* no lvl2 table, skip to next lvl1 entry */
			continue;
		}

		ptep = pmap_pte(map, va);

		if (ptep == PT_ENTRY_NULL) {
			va = L2_NEXT(va)-PAGE_SIZE; /* no lvl3 table, skip to next lvl2 entry */
			continue;
		}

		if (!L3_VALID(ptep))
			continue;		/* no page mapping */

		phys = PMAP_PTOB(ptep->pfn);	/* pick up phys addr */

		if (!PMAP_MANAGED(phys))
			continue;		/* no PV list */

		pv_h = PFIDX_TO_PVH(PFIDX(phys));
		found = FALSE;

		if (pv_h->pmap != PMAP_NULL) {

			loopcnt = 10000;	/* loop limit */
			pv_e = pv_h;
			while (pv_e != PV_ENTRY_NULL) {

				if (loopcnt-- < 0) {
					printf("check_map: loop in PV list at PVH 0x%X (for phys 0x%X)\n", pv_h, phys);
					WORRY(who);
				}

				if (pv_e->pmap == map && pv_e->va == va) {
					if (found) {
						printf("check_map: Duplicate PV list entries at %x and %x in PV list %x.\n", saved_pv_e, pv_e, pv_h);
						printf("check_map: for pmap %x  VA %x  Phys %x.\n", map, va, phys);
						WORRY(who);
					}
					else {
						found = TRUE;
						saved_pv_e = pv_e;
					}
				}
				pv_e = pv_e->next;
			}
		}

		if (!found) {
			printf("check_map: pmap %x VA %x Phys %x not in PV list %x\n", map, va, phys, pv_h);
			WORRY(who);
		}
	}

	if (pmap_dbg & PD_PRINT)
		printf("(check_map) done\n");

} /* check_map() */


/*
 *	Routine:	CHECK_PMAP_CONSISTENCY (internal)
 *
 *	Function:
 *		Debug mode routine which walks all pmaps, checking for internal
 *		consistency.  We are called UNLOCKED, so we'll take the write lock.
 *
 *	Parameters:
 *		who		string containing caller's name to be
 *				printed if a panic arises
 *
 *	Extern/Global:
 *		list of pmap structures
 *
 *	Calls:
 *		check_map
 *		check_pv_list
 *
 *    This function obtains the pmap write lock.  Then, for each pmap
 * structure in the pmap struct queue, it calls check_map to verify the
 * consistency of its translation table hierarchy.
 *
 *    Once all pmaps have been checked, check_pv_list is called to check
 * consistency of the PV lists for each managed page.
 */
static
check_pmap_consistency(who)
	char		*who;
{
	pmap_t		p;
	int		i;
	vm_offset_t	phys;
	pv_entry_t	pv_h;
	int		spl;


	if (pmap_dbg & PD_PRINT)
		printf("check_pmap_consistency (%s): in.\n", who);

	if (pmem_list == PV_ENTRY_NULL) {
		if (pmap_dbg & PD_PRINT)
			printf("check_pmap_consistency (%s) PMEM list not initialized.\n", who);
		return;
	}

	PMAP_WRITE_LOCK(spl);

	if (pmap_dbg & PD_CHECK_KERNEL)
		check_map(kernel_pmap, VM_MIN_KERNEL_ADDRESS, VM_MAX_KERNEL_ADDRESS, who);

	/* run through all pmaps.  check consistency of each one... */
	i = PMAP_MAX;
	for (p = kernel_pmap->next; p != kernel_pmap; p = p->next) {
		if (i-- < 0) {
			printf("check_pmap_consistency: pmap struct loop error.\n");
			WORRY(who);
		}
		check_map(p, VM_MIN_ADDRESS, VM_MAX_ADDRESS, who);
	}

	/* run through all managed pages, check pv_list for each one */
	for (phys = pmap_phys_start; phys < pmap_phys_end; phys += PAGE_SIZE) {
		pv_h = PFIDX_TO_PVH(PFIDX(phys));
		check_pv_list(phys, pv_h, who);
	}

	PMAP_WRITE_UNLOCK(spl);

	if (pmap_dbg & PD_PRINT)
		printf("check_pmap_consistency (%s): out.\n", who);

} /* check_pmap_consistency() */

#endif	DBG


/*
 * PMAP PRINT MACROS AND ROUTINES FOR DEBUGGING
 * These routines are called only from the debuger.
 * (No locking required.)
 */
static char *dtype_names[4] = {"INVALID", "PTE", "SPTD", "LPTD"};

#define PRINT_LONG_DESC(p) 	/* print a long format table descriptor */	\
		printf("%08X %08X : ",		\
			((pt_ldesc_template_t *)p)-> bits.lo,			\
			((pt_ldesc_template_t *)p)-> bits.hi );			\
		printf("lu=%d, limit=0x%X, sup=%d, used=%d, prot=%d, dtype=%s, addr=0x%X\n",   \
			/* ral, wal, and share fields not available on 68030 */	\
			p->lu,			\
			p->limit,		\
			p->sup,			\
			p->pg_used,		\
			p->prot,		\
			dtype_names[p->dtype],	\
			p->table_addr<<4 );


#define PRINT_SHORT_DESC(p) 			\
		printf("%08X : ",		\
			((pt_sdesc_template_t *)p)-> bits );	   \
		printf("addr=0x%X, used=%d, prot=%d, dtype=%s\n",  \
			p->table_addr<<4,	\
			p->pg_used,		\
			p->prot,		\
			dtype_names[p->dtype] );


#define PRINT_PTE_DESC(p)			\
		printf("%08X : ",		\
			((pte_template_t *)p)-> bits );	   \
		printf("pfn=0x%X, wired=%d, cache_inhibit=%d, modified=%d, used=%d, prot=%d, dtype=%s\n",  \
			/* gate and lock fields not available on 68030 */ \
			p->pfn,			\
			p->wired,		\
			p->no_cache,		\
			p->modified,		\
			p->pg_used,		\
			p->prot,		\
			dtype_names[p->dtype] );


/*
 *	Routine:	PMAP_PRINT
 *
 *	Function:
 *		Print pmap structure, including level 1 table.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *
 *	Special Assumptions:
 *		No locking required.
 *
 *    This function prints the fields of the pmap structure, then
 * iterates through the level 1 translation table, printing each entry.
 */
void
pmap_print (pmap)
pmap_t		pmap;
{
	pt_ldesc_t	*l1p;
	pt_ldesc_t	*l1v;
	int		i;

	printf("Pmap @ 0x%X:\n", pmap);
	l1p = pmap->lvl1_paddr;
	l1v = pmap->lvl1_vaddr;
	printf("    lvl1_paddr: 0x%X; lvl1_vaddr: 0x%X; ref_count: %d; lock: 0x%X\n",
		l1p, l1v, pmap->ref_count, * ( (long *) (&pmap->lock)) );

#ifdef	statistics_not_yet_maintained
	printf("    Statistics: pagesize %d; free_count %d; active_count %d; inactive_count %d; wire_count %d\n",
		pmap->stats.pagesize,
		pmap->stats.free_count,
		pmap->stats.active_count,
		pmap->stats.inactive_count,
		pmap->stats.wire_count);
	printf("        zero_fill_count %d; reactivations %d; pageins %d; pageouts %d; faults %d\n",
		pmap->stats.zero_fill_count,
		pmap->stats.reactivations,
		pmap->stats.pageins,
		pmap->stats.pageouts,
		pmap->stats.faults);
	printf("        cow_faults %d, lookups %d, hits %d\n",
		pmap->stats.cow_faults,
		pmap->stats.lookups,
		pmap->stats.hits);
#endif	statistics_not_yet_maintained

	l1p = (pt_ldesc_t *) pmap->lvl1_vaddr;	/* addr of physical table */
	l1v = l1p + L1_ENTRIES;			/* shadow table with virt addrs */
	if (l1p == (pt_ldesc_t *) 0) {
		printf("Error in pmap - lvl1_paddr is null.\n");
		}
	else {
		printf("    Level 1 table at 0x%X (0x%X):\n", l1p, l1v);
		for (i = 0; i < L1_ENTRIES; i++, l1p++, l1v++) {
			printf("    (%x)phys: ", i);
			PRINT_LONG_DESC(l1p);
			printf("    (%x)virt: ", i);
			PRINT_LONG_DESC(l1v);
		}
	}

} /* pmap_print() */


/*
 *	Routine:	PMAP_PRINT_TRACE
 *
 *	Function:
 *		Using virt addr, derive phys addr, printing pmap tables along the way.
 *
 *	Parameters:
 *		pmap		pointer to pmap structure
 *		va		virtual address whose translation is to be traced
 *		long_format	flag indicating long form output is desired
 *
 *	Special Assumptions:
 *		No locking required.
 *
 *     This function chases down through the translation tree as
 * appropriate for the given virtual address.  Each table entry
 * encountered is printed.  If the long_format is desired, all entries of
 * each table are printed, with special indication of the entries used in
 * the translation.
 */
void
pmap_print_trace (pmap, va, long_format)
pmap_t		pmap;
vm_offset_t	va;
boolean_t	long_format;
{
	pt_ldesc_t	*l1p;	/* ptr to level 1 table of physical addresses */
	pt_ldesc_t	*l1v;	/* ptr to level 1 shadow table of virtual addresses */
	pt_sdesc_t	*l2p;	/* ptr to level 2 table of physical addresses */
	pt_sdesc_t	*l2v;	/* ptr to level 2 shadow table of virtual addresses */
	pt_entry_t	*l3p;	/* ptr to level 3 table of physical page addresses */

	int		i;		/* table loop index */
	unsigned long	prev_entry_lo;	/* keep track of value of previous table entry */
	unsigned long	prev_entry_hi;	/* keep track of value of previous table entry */
	unsigned long	prev_entry;	/* keep track of value of previous table entry */
	int		n_dup_entries;	/* count contiguous duplicate entries */

	printf("Trace of virtual address 0x%X.  Pmap @ 0x%X.\n", va, pmap);

	/*** LEVEL 1 TABLES ***/
	/* get addrs of lvl1 tables */
	l1p = (pt_ldesc_t *) pmap->lvl1_vaddr;
	l1v = l1p + L1_ENTRIES;

	if (l1p == (pt_ldesc_t *) 0) {
		printf("    level 1 table pointer (pmap.lvl1_paddr) null, trace stops.\n");
		return;
	}

	n_dup_entries = 0;
	prev_entry_lo = 0xFFFFFFFF;
	prev_entry_hi = 0xFFFFFFFF;
	if (long_format) {
		printf("    level 1 table at 0x%X (virt shadow at 0x%X)\n", l1p, l1v);
		for (i = 0; i < L1_ENTRIES; i++, l1p++, l1v++) {
			if (prev_entry_lo == ((pt_ldesc_template_t *)l1p)->bits.lo
			    && prev_entry_hi == ((pt_ldesc_template_t *)l1p)->bits.hi
			    && L1IDX(va) != i  &&  i != L1_ENTRIES-1) {
				n_dup_entries++;
				continue;	/* suppress duplicate entry */
			}
			if (n_dup_entries != 0) {
				printf("      - %d duplicate entries skipped -\n",
					n_dup_entries);
				n_dup_entries = 0;
			}
			prev_entry_lo = ((pt_ldesc_template_t *)l1p)->bits.lo;
			prev_entry_hi = ((pt_ldesc_template_t *)l1p)->bits.hi;
			if (L1IDX(va) == i)
				printf(" >> (%x)phys: ", i);
			else	printf("    (%x)phys: ", i);
			PRINT_LONG_DESC(l1p);
			if (L1IDX(va) == i)
				printf(" >> (%x)virt: ", i);
			else	printf("    (%x)virt: ", i);
			PRINT_LONG_DESC(l1v);
		} /* for */
	}
	else {
		/* index into both tables to entry for given VA */
		l1p += L1IDX(va);
		l1v += L1IDX(va);
		printf("    level 1 entry index 0x%X at 0x%X (virt shadow at 0x%X)\n", L1IDX(va), l1p, l1v);
		printf("    phys: ");
		PRINT_LONG_DESC(l1p);
		printf("    virt: ");
		PRINT_LONG_DESC(l1v);
	}


	/*** LEVEL 2 TABLES */
	/* get addrs of lvl2 tables */
	l1p = (pt_ldesc_t *) pmap->lvl1_vaddr;
	l1v = l1p + L1_ENTRIES;
	l2p = (pt_sdesc_t *) ((l1v+L1IDX(va))->table_addr << 4);
	l2v = l2p + L2_ENTRIES;

	if ((l1p+L1IDX(va))->dtype != DT_SHORT) {
		printf("    level 1 table entry invalid, trace stops.\n");
		return;
	}

	n_dup_entries = 0;
	prev_entry = 0xFFFFFFFF;
	if (long_format) {
		printf("        level 2 table at 0x%X (virt shadow at 0x%X)\n", l2p, l2v);
		for (i = 0; i < L2_ENTRIES; i++, l2p++, l2v++) {
			if (prev_entry == ((pt_sdesc_template_t *)l2p)->bits
			    && L2IDX(va) != i  &&  i != L2_ENTRIES-1) {
				n_dup_entries++;
				continue;	/* suppress duplicate entry */
			}
			if (n_dup_entries != 0) {
				printf("         - %d duplicate entries skipped -\n",
					n_dup_entries);
				n_dup_entries = 0;
			}
			prev_entry = ((pt_sdesc_template_t *)l2p)->bits;
			if (L2IDX(va) == i)
				printf("     >> (%x)phys: ", i);
			else	printf("        (%x)phys: ", i);
			PRINT_SHORT_DESC(l2p);
			if (L2IDX(va) == i)
				printf("     >> (%x)virt: ", i);
			else	printf("        (%x)virt: ", i);
			PRINT_SHORT_DESC(l2v);
		} /* for */
	}
	else {
		/* index into both tables for given VA */
		l2p += L2IDX(va);
		l2v += L2IDX(va);
		printf("        level 2 entry index 0x%X at 0x%X (virt shadow at 0x%X)\n", L2IDX(va), l2p, l2v);
		printf("        phys: ");
		PRINT_SHORT_DESC(l2p);
		printf("        virt: ");
		PRINT_SHORT_DESC(l2v);
	}


	/*** LEVEL 3 TABLES ***/
	/* get addrs of lvl3 (pte) table (no shadow table) */
	l1p = (pt_ldesc_t *) pmap->lvl1_vaddr;
	l1v = l1p + L1_ENTRIES;
	l2p = (pt_sdesc_t *) ((l1v+L1IDX(va))->table_addr << 4);
	l2v = l2p + L2_ENTRIES;
	l3p = (pt_entry_t *) ((l2v+L2IDX(va))->table_addr << 4);

	if ((l2p+L2IDX(va))->dtype != DT_SHORT) {
		printf("        level 2 table entry invalid, trace stops.\n");
		return;
	}

	n_dup_entries = 0;
	prev_entry = 0xffffffff;
	if (long_format) {
		printf("            level 3 table (ptes) at 0x%X\n", l3p);
		for (i = 0; i < L3_ENTRIES; i++, l3p++) {
			if (prev_entry == ((pte_template_t *)l3p)->bits
			    && L3IDX(va) != i  &&  i != L3_ENTRIES-1) {
				n_dup_entries++;
				continue;	/* suppress duplicate entry */
			}
			if (n_dup_entries != 0) {
				printf("              - %d duplicate entries skipped -\n",
					n_dup_entries);
				n_dup_entries = 0;
			}
			prev_entry = ((pte_template_t *)l3p)->bits;
			if (L3IDX(va) == i)
				printf("         >> (%x)pte: ", i);
			else	printf("            (%x)pte: ", i);
			PRINT_PTE_DESC(l3p);
		} /* for */
	}
	else {
		/* index into page table */
		l3p += L3IDX(va);
		printf("            pte index 0x%X at 0x%X\n", L3IDX(va), l3p);
		printf("            pte: ");
		PRINT_PTE_DESC(l3p);
	}

} /* pmap_print_trace() */

/*
 *	Routine:	PMAP_PRINT_PAGE_USAGE
 *
 *	Function:
 *		Print contents of a PV list.
 *
 *	Parameters:
 *		pa		physical address of page for which the PV list
 *				should be printed
 *
 *	Extern/Global:
 *		pv_head_array, pv lists
 *
 *	Special Assumptions:
 *		No locking required.
 *
 *    If the address specified is not a managed page, this function
 * prints only an error message.  Otherwise, each entry in the
 * associated PV list is printed.
 */

#include "sys/proc.h"
#define PPL 8

void
pmap_print_page_usage (pa)
	vm_offset_t	pa;	/* physical address */
{

	pv_entry_t	pvl;
	int		pfi;


	if (!pa) {
		int pt_len, newline, repeatline, idx,
		              nulls,      /* number of nulls on current line */
		              nulllines,  /* number of lines that were all null */
		              i, pid;
		int		modified;
		pmap_t          pmap;
		thread_t	thread;
		task_t		task;
		extern struct proc *allproc;
		struct proc	*procp;

		newline = PPL-1;
		repeatline = 0;
		nulls = 0;
		for (pa = pmap_phys_start; pa < pmap_phys_end; pa += PAGE_SIZE) {

			/*
			 * Check if we want to start a new line.
			 */
			if (++newline == PPL) {
				newline = 0;
				if (repeatline) {
					repeatline = 0;
					pa -= PPL*PAGE_SIZE;
					printf ("\n           ");
					idx++;
				} else {
					idx = 0;
					if (nulls == PPL)
						nulllines++;
					else {
						nulllines = 0;
						printf ("\n%x: ", pa);
					}
					nulls = 0;
				}
			}

			pfi = PFIDX(pa);
			pvl = PFIDX_TO_PVH(pfi);
			modified = pmap_modify_list[pfi];
			for (i=idx; pvl != 0  &&  i>0; i--)
				pvl = pvl->next;
			if (!modified && (pvl == 0  ||  (pmap = pvl->pmap) == 0)) {
				if (nulllines == 0)
					printf ("              ");
				nulls++;
				continue;
			}
			/* flush any pending nulls to stdout */
			if (nulllines) {
				if (nulllines > 1)
					printf ("\n  --- %d pages unused --- ", (nulllines-1)*PPL);
				nulllines = 0;
				pa -= (nulls+1)*PAGE_SIZE;
				nulls = 0;
				newline = PPL-1;
				repeatline = 0;
				continue;
			}
				
			/*
			 * Page is not in any pmap, but is in the modify list.
			 */
			if (pvl == 0  ||  pmap == 0) {
				printf (" ../........ #");
				continue;
			}

			/* if the page is shared, do another line */
			if (pvl->next)
				repeatline++;

			/*
			 * Find the PID for the process which owns this page.
			 */
			pid = -1;
			for (procp = allproc; procp != NULL; procp = procp->p_nxt)
				if (pmap == procp->task->map->pmap) {
					pid = procp->p_pid;
					break;
				}

			printf ("%3d/%8x%c%c", pid, pvl->va,
				repeatline ? '*':' ',
				pmap_modify_list[pfi] ? '#':' ');

			/* do not display other owning tasks/processes */
			repeatline = 0;

		} /* do next physical page */

		if (nulllines > 1)
			printf ("\n  --- %d pages unused --- ", (nulllines-1)*PPL + nulls);
		printf ("\n\n");
		return;
	}

	if (pa <= 0x1000) {

		int saved = pmap_dbg;
		pmap_dbg  = pa | PD_WORRY;
		CHECK_PMAP_CONSISTENCY ("user");
		pmap_dbg = saved;

	} else if (!PMAP_MANAGED(pa)) {
		printf("%08x is not a managed page.\n", pa);

	} else {

		/* a managed page */
		pfi = PFIDX(pa);
		printf("page at physaddr 0x%x:  pageframe index 0x%x", pa, pfi);
		if (pmap_modify_list[pfi])
			printf ("  modified");
		printf ("\n");
		for (pvl = PFIDX_TO_PVH(pfi); pvl != PV_ENTRY_NULL; pvl = pvl->next)
			printf("     PV entry  %x   next %x   pmap %x   va %x\n",
				pvl, pvl->next, pvl->pmap, pvl->va);
	}

} /* pmap_print_page_usage() */
