(*$Vector : GeneralTypes Vector' List InStreamType OutStreamType *)

loadSig "Vector";

structure Vector: Vector =

(* CONSTANT VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	4 Oct 1989

   Maintenance: Author


   DESCRIPTION

   This is a reference implementation.  Most systems will provide most of
   the following as built-in functions.


   SEE ALSO

   RefVector.
*)

struct

  open Vector'

  val version = 0.1

  type 'a T = 'a Vector

  exception Sep of string * string * string * string

  fun stringSep start finish sep p (Vector (l, _)) =
	List.stringSep start finish sep p l
	handle List.Sep x => raise Sep x

  fun string p v = stringSep "" "" " " p v

  (* The parse, parse' and read functions assume that entries
     are separated by formatting characters. *)

  fun parseSepN' start finish sep p n l =
        ( case List.parseSepN' start finish sep p n l of
	    OK ([], s) => OK (emptyVector, s)
	  | OK (l', s) => OK (Vector (l', List.size l'), s)
	  | Fail (Some l', s) => Fail (Some (Vector (l', List.size l')), s)
	  | Fail (None, s) => Fail (None, s)
        )
	handle List.Sep x => raise Sep x

  fun parseSep' start finish sep p l =
	( case List.parseSep' start finish sep p l of
	    OK ([], s) => OK (emptyVector, s)
	  | OK (l', s) => OK (Vector (l', List.size l'), s)
	  | Fail (Some l', s) => Fail (Some (Vector (l', List.size l')), s)
	  | Fail (None, s) => Fail (None, s)
	)
	handle List.Sep x => raise Sep x

  fun parseN' p n l =
	if n < 0 then raise General.Nat ("parseN'", n)
	else parseSepN' "" "" "" p n l

  fun parse' p l = parseSep' "" "" "" p l

  fun parseSepN start finish sep p n s =
	if n < 0 then raise General.Nat ("parseSepN", n)
	else
	  case parseSepN' start finish sep p n (explode s) of
	    OK (v, _) => OK v
	  | Fail (x, _) => Fail x

  fun parseSep start finish sep p s =
	case parseSep' start finish sep p (explode s) of
	  OK (v, _) => OK v
	| Fail (x, _) => Fail x

  fun parseN p n s =
	if n < 0 then raise General.Nat ("parseN", n)
	else parseSepN "" "" "" p n s

  fun parse p s = parseSep "" "" "" p s

  fun readSepN start finish sep p n i =
	( case List.readSepN start finish sep p n i of
	    OK [] => OK emptyVector
	  | OK l  => OK (Vector (l, List.size l))
	  | Fail (Some l) => Fail (Some (Vector (l, List.size l)))
	  | Fail None => Fail None
	)
	handle List.Sep x => raise Sep x

  fun readSep start finish sep p i =
	case List.readSep start finish sep p i of
	  OK [] => OK emptyVector
	| OK l  => OK (Vector (l, List.size l))
	| Fail (Some l) => Fail (Some (Vector (l, List.size l)))
	| Fail None => Fail None
	handle List.Sep x => raise Sep x

  fun readN p n i =
	if n < 0 then raise General.Nat ("readN", n)
	else readSepN "" "" "" p n i

  fun read p i = readSep "" "" "" p i

  fun fromFile p name =
	let fun readList i =
		  case p i
		  of Fail _ => (InStream.closeIn i; [])
		  |  OK x => x :: readList i
	in fromList (readList (InStream.openIn name))
	end

  fun file p (Vector (l, _)) name =
	let val os = OutStream.openOut name
	    fun out s = OutStream.output' os s
	in List.apply (out o p) l;
	   OutStream.closeOut os
	end

end
