signature RefVector =
sig

(* VARIABLE VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
   Date:	30 Oct 1989

   Based on the Array signatures provided by Edinburgh ML, NJML and Poly ML.

   Maintenance:	Author


   DESCRIPTION

   A RefVector is a single-dimensional polymorphic array with elements that
   be updated in place.  Different RefVectors are not equal, even if they hold
   the same elements.  RefVectors always admit equality.  There is only one
   empty RefVector.  The first element of any RefVector has the index 0.

   Elements can be updated by the "update" function; the conventional
   "a sub i := v" notation doesn't work because the "a sub i" would have to
   return a reference, and this would either cause problems for the garbage
   collector or add an extra level of indirection.  If this extra level of
   indirection is acceptable, you can create constant vectors of references
   using the Vector type.

   To create a RefVector of RefVectors, either use a generate function
   or first create a RefVector of empty RefVectors and then update each
   element with the desired initial value.

   Unlike lists, this signature doesn't provide the  subLast  and  extractLast
   functions.  The "size" function should take constant time, so this is not a
   handicap.


   SEE ALSO

   Vector, MonoRefVector.


   NOTES

   The name was chosen so that there could also be types for matrices (type
   Matrix), updateable matrices (type RefMatrix), tensors, etc.  There is
   also scope for a type Array of abitrary dimension, indexed by an integer
   list of the appropriate length.

   A possible implementation would be to view a random access file as a
   RefVector.
*)


  val version: real
   (* version = 0.1 *)


(* TYPES *)

  type 'a RefVector

  type 'a T 
    sharing type T = RefVector


(* CREATORS *)

  val create: Nat -> '_a -> '_a RefVector
  (* create n i; create a RefVector of n locations, each containing i.
     Raise (Nat ("vreate", n)) if n < 0. *)

  val generate: Nat -> (int -> '_a) -> '_a RefVector
  (* generate n f; create a RefVector v of n locations, (v sub 0) to
     (v sub (n-1)), with (v sub i) initialised to (f i).
     Raise (Nat ("generate", n)) if n < 0. *)

  val generate': Nat -> ('b -> '_a * 'b) -> 'b -> '_a RefVector
   (* generate' n f base; create a RefVector of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub 0) initialised to (# 1 (f base)) and
      (v sub i (i > 0)) initialised to (# 1 (f (# 2 f_i))), where f_i is
      the result of the i^th application of f.
      Raise (Nat ("generate'", n)) if n < 0. *)


(* REFSEQUENCE *)

  val stringSep: string -> string -> string ->
                 ('_a -> string) -> '_a RefVector -> string

  val string: ('_a -> string) -> '_a RefVector -> string

  exception Sep of string * string * string * string

  val parseSepN: string -> string -> string ->
                 (string list -> ('_a * string list, 'b) Result) ->
                 Nat -> string -> ('_a RefVector, '_a RefVector Option) Result

  val parseSep: string -> string -> string ->
                (string list -> ('_a * string list, 'b) Result) ->
                string -> ('_a RefVector, '_a RefVector Option) Result

  val parseN: (string list -> ('_a * string list, 'b) Result) ->
              Nat -> string -> ('_a RefVector, '_a RefVector Option) Result

  val parse: (string list -> ('_a * string list, 'b) Result) ->
	     string -> ('_a RefVector, '_a RefVector Option) Result

  val parseSepN': string -> string -> string ->
                 (string list -> ('_a * string list, 'b) Result) ->
                 Nat -> string list ->
                 ('_a RefVector * string list,
		  '_a RefVector Option * string list) Result

  val parseSep': string -> string -> string ->
                 (string list -> ('_a * string list, 'b) Result) ->
                 string list ->
                 ('_a RefVector * string list,
		  '_a RefVector Option * string list) Result

  val parseN': (string list -> ('_a * string list, 'b) Result) ->
               Nat -> string list ->
               ('_a RefVector * string list,
		'_a RefVector Option * string list) Result

  val parse': (string list -> ('_a * string list, 'b) Result) ->
	      string list ->
	      ('_a RefVector * string list,
	       '_a RefVector Option * string list) Result

  val readSep: string -> string -> string ->
               (InStream -> ('_a, 'b) Result) -> InStream ->
               ('_a RefVector, '_a RefVector Option) Result

  val readSepN: string -> string -> string ->
               (InStream -> ('_a, 'b) Result) -> Nat -> InStream ->
               ('_a RefVector, '_a RefVector Option) Result

  val read: (InStream -> ('_a, 'b) Result) -> InStream ->
	    ('_a RefVector, '_a RefVector Option) Result

  val readN: (InStream -> ('_a, 'b) Result) -> Nat -> InStream ->
             ('_a RefVector, '_a RefVector Option) Result

  val same: 'a RefVector -> 'a RefVector -> bool

  val different: 'a RefVector -> 'a RefVector -> bool

  val eq: ('_a -> '_a -> bool) -> '_a RefVector -> '_a RefVector -> bool

  val ne: ('_a -> '_a -> bool) -> '_a RefVector -> '_a RefVector -> bool

  val lt: ('_a -> '_a -> bool) -> '_a RefVector -> '_a RefVector -> bool

  val le: ('_a -> '_a -> bool) -> '_a RefVector -> '_a RefVector -> bool

  val gt: ('_a -> '_a -> bool) -> '_a RefVector -> '_a RefVector -> bool

  val ge: ('_a -> '_a -> bool) -> '_a RefVector -> '_a RefVector -> bool


(* CONVERTERS *)

  val list: '_a RefVector -> '_a list
   (* list v; make a list containing (only) the elements of v, in
      the same order. *)

  val fromList: '_a list -> '_a RefVector
   (* fromList l; make a RefVector containing (only) the elements of l, in
      the same order. *)

  val fromFile: (InStream -> ('_a, 'b) Result) -> string -> '_a RefVector
   (* fromFile p name; read the contents of the file called name into a vector.
      Stops reading from the file as soon as p returns Fail.
      Raises General.Io if something goes wrong. *)

  val file: ('_a -> string) -> '_a RefVector -> string -> unit
   (* file p v name; write the contents of v to the new file called name.
      Raises General.Io if something goes wrong. *)


(* OBSERVERS *)

  val empty: 'a RefVector -> bool
   (* empty v; returns true if v is empty. *)

  val size: 'a RefVector -> Nat
   (* size v; return the number of elements in v. *)


(* SELECTORS *)

  exception Sub of string * int
  (* infix 9 sub *)
  val sub: '_a RefVector * int -> '_a
   (* v sub n; return the n+1'th element of v.
      Raise (Sub ("sub", n)) if not (0 <= n <= size v). *)

  val nth: int -> '_a RefVector -> '_a
   (* nth n v; return the n+1'th element of v.
      Raise (Sub ("nth", n)) if not (0 <= n <= size v). *)

  exception Extract of int * int
  val extract: int -> int -> '_a RefVector -> '_a RefVector
   (* extract start finish v; returns the sub-vector of v starting with
      (v sub start) and ending with (v sub (finish - 1)).
      Returns the empty vector if (start = finish).
      Raise (Extract (start, finish)) if not (0 <= start,finish <= size v). *)


(* MANIPULATORS *)

  val rev: '_a RefVector -> '_a RefVector
   (* rev v; builds a new RefVector containing the elements of v in
      reverse order. *)

  (* infix 6 ^ *)
  val ^ : '_a RefVector * '_a RefVector -> '_a RefVector
   (* v ^ v'; builds a RefVector containing the elements of v' appended to those
      of v. *)

  exception Update of int
  val update: int -> 'a -> 'a RefVector -> unit
   (* update n i v; replace (v sub n) with i.
      Raise (Update n) if not (0 <= n <= size v). *)

  exception Copy of int * int * int
  val copy: int -> int -> '_a RefVector -> int -> '_a RefVector -> unit
   (* copy start finish v start' v'; copies the sub-vector of v starting with
      (v sub start) and ending with (v sub (finish - 1)) to the RefVector v',
      starting with (v' sub start').  Has no effect if (start = finish).
      Raises (Copy (start, finish, start')) if
      not (0 <= start,finish <= size v) or if
      not (0 <= start',start'+finish-start <= size v'). *)
 
  exception UpdateRange of int * int
  val updateRange: int -> int -> 'a -> 'a RefVector -> unit
   (* updateRange start finish i v; update the elements of v starting with
      (v sub start) and ending with (v sub (finish - 1)) with i.  Has no effect
      if (start = finish).  Raises (UpdateRange (start, finish)) if
      not (0 <= start,finish <= size v). *)

  val sort: ('_a -> '_a -> bool) -> '_a RefVector -> unit
   (* sort p v; sorts v in place by p. *)


(* REDUCERS *)

  val foldR: ('_a -> 'b -> 'b) -> 'b -> '_a RefVector -> 'b
   (* foldR f base v; folds using f associating to the right over the
      base element.
      foldR f [a1,a2,...,an] base = f(a1,f(a2,...,f(an,base)...)). *)

  val foldL: ('_a -> 'b -> 'b) -> 'b -> '_a RefVector -> 'b
   (* foldL f v base; folds using f associating to the left over the
      base element.
      foldL f [a1,a2,...,an] base = f(an,...,f(a2,f(a1,base))...). *)

  exception Empty of string
  val foldR': ('_a -> '_a -> '_a) -> '_a RefVector -> '_a
   (* foldR' f v; folds using f associating to the right over the
      last element of v.  Raises (Empty "foldR'") if v is empty. *)

  val foldL': ('_a -> '_a -> '_a) -> '_a RefVector -> '_a
   (* foldL' f v; folds using f associating to the left over the
      first element of v.  Raises (Empty "foldL'") if v is empty. *)

  val pairwise: ('_a -> '_a -> bool) -> '_a RefVector -> bool
   (* pairwise f v; true if (f (v sub i) (v sub (i + 1))) is true for all
      0 <= i < size v, or if v is empty. *)


(* ITERATORS *)

  val map: ('_a -> '_b) -> '_a RefVector -> '_b RefVector
   (* map f v; builds a new RefVector by applying f to each element of v. *)

  val apply: ('_a -> unit) -> '_a RefVector -> unit
   (* apply f v; applies f to each element of v. *)

  val iterate: ('_a * int -> '_b) -> '_a RefVector -> '_b RefVector
   (* iterate f v; builds a new RefVector by applying f to each element of v
      paired with its index. *)

  val iterateApply: ('_a * int -> unit) -> '_a RefVector -> unit
   (* iterateApply f v; applies f to each element of v paired with its index. *)
end
