(*  Title: 	CTT/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Constructive Type Theory syntax
*)

signature CTT_SYNTAX =
sig
  structure Syntax : PURE_SYNTAX
  val Aterm: typ
  val Atype: typ
  val const_decs: (string list * typ) list
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor CTT_SyntaxFun (Syntax: PURE_SYNTAX) : CTT_SYNTAX = 
struct
structure Syntax = Syntax;
local open Syntax
in

(*Meta-types for terms and types*)
val Aterm  = Ground "term";
val Atype  = Ground "type";

(** Translation of lambda-abstraction **)

fun lam_tr [Free(id,T),b] = Const("lambda",Adummy) $ absfree(id,T,b);

fun lam_tr' (_ $ Abs(id,T,b)) =
    let val (id',b') = variant_abs(id,T,b)
    in Const(" lam",Adummy) $ Free(id',T) $ b' end;

(** Translation of 'dependent' type operators **)

fun dependent_tr q [Free(id,T),A,B] = Const(q,Adummy) $ A $ absfree(id,T,B);

(*Translation of * and --> to Sum and Prod*)
fun ndependent_tr q [A,B] = 
    Const(q,Adummy) $ A $ Abs("x",Ground "dummy",incr_boundvars 1 B);

(*Is there a dependence or not? Use r if not. *)
fun dependent_tr' (q,r) (_$A$Abs(id,T,B)) =
    if 0 mem (loose_bnos B) 
    then 
      let val (id',B') = variant_abs(id,T,B)
      in Const(q,Adummy) $ Free(id',T) $ A $ B' end
    else Const(r,Adummy) $ A $ B;

val mixfix =
[   (*Judgements*)
  Mixfix("(_ type)", Atype --> Aprop, "Type", [10], 5),
  Mixfix("(3_ =/ _)", [Atype,Atype]--->Aprop, "Eqtype", [10,10],5),
  Mixfix("(_ /: _)", [Aterm, Atype]--->Aprop, "Elem", [10,10], 5),
  Mixfix("(3_ =/ _ :/ _)", [Aterm,Aterm,Atype]--->Aprop, "Eqelem",
			   [10,10,10],5),
  Delimfix("Reduce'(_,_')", [Aterm,Aterm]--->Aprop, "Reduce"),
    (*Types*)
  Mixfix("(3PROD _:_./ _)",    [SId,Atype,Atype]--->Atype, " PROD", [], 10),
  Mixfix("(3SUM _:_./ _)", [SId,Atype,Atype]--->Atype, " SUM", [], 10),
  Infixr("+", [Atype,Atype]--->Atype, 30),
  (*Invisible infixes!*)
  Mixfix("(_ */ _)", [Atype,Atype]--->Atype, " *", [36,35], 35),
  Mixfix("(_ -->/ _)", [Atype,Atype]--->Atype, " -->", [26,25], 25),
    (*Functions*)
  Mixfix("(3lam _./ _)", [SId, Aterm]--->Aterm, " lam", [], 10),
  Infixl("`", [Aterm,Aterm]--->Aterm, 55),
    (*Natural numbers*)
  Delimfix("0", Aterm, "0"),
    (*Pairing*)
  Delimfix("(1<_,/_>)", [Aterm,Aterm]--->Aterm, "pair") ];

val ext = {logical_types=[Aterm,Atype],
  mixfix=mixfix,
  parse_translation=
    [(" PROD", dependent_tr "Prod"),
     (" SUM",  dependent_tr "Sum"),
     (" lam", lam_tr),
     (" -->", ndependent_tr "Prod"),
     (" *",  ndependent_tr "Sum") ],
  print_translation=
    [("Prod", dependent_tr' (" PROD"," -->")),
     ("Sum",  dependent_tr' (" SUM"," *")),
     ("lambda", lam_tr')]};

val const_decs = constants mixfix @
[(*Types*)
 (["F","T"],	Atype),	(*F is empty, T contains one element*)
 (["contr"],		Aterm-->Aterm ),
 (["tt"],		Aterm),
    (*Natural numbers*)
 (["N"],	Atype),
 (["succ"],	Aterm-->Aterm),
 (["rec"],	[Aterm, Aterm, [Aterm,Aterm]--->Aterm] ---> Aterm),
    (*Unions*)
 (["inl","inr"],	[Aterm]--->Aterm),
 (["when"],	[Aterm, Aterm-->Aterm, Aterm-->Aterm]--->Aterm),
    (*General Sum and Binary Product*)
 (["Sum"],	[Atype, Aterm-->Atype]--->Atype),
 (["fst","snd"],	Aterm-->Aterm),
 (["split"],	[Aterm, [Aterm,Aterm]--->Aterm] --->Aterm),
    (*General Product and Function Space*)
 (["Prod"],	[Atype, Aterm-->Atype]--->Atype),
 (["lambda"],	[Aterm-->Aterm]--->Aterm),
    (*Equality type*)
 (["Eq"],		[Atype,Aterm,Aterm]--->Atype ),
 (["eq"],		Aterm ) ];

val syn = Syntax.extend pure ext;

fun read a = Syntax.read syn Any a;
fun prin t = Syntax.print_top_level syn t;

end;
end;
