(*  Title: 	Syntax Definition
    Author: 	Tobias Nipkow
*)

signature SYNTAX_DEF0 =
sig
  datatype mixfix = Mixfix of string * typ * string * int list * int
		  | Delimfix of string * typ * string
		  | Infixl of string * typ * int
		  | Infixr of string * typ * int
  val constants : mixfix list -> (string list * typ) list
  val max_pri: int
  val Any: typ
  val SId: typ
  val SVar: typ
end;

signature SYNTAX_DEF =
sig
  include SYNTAX_DEF0
  structure XGram : XGRAM
  val empty : string XGram.XGram
  val extend: string XGram.XGram ->
	{logical_types: typ list, mixfix: mixfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term -> term)) list}
	-> string XGram.XGram
  val typ_to_nt: typ -> string
  val Afn: typ
  val Args: typ
  val STyp: typ
  val appl_const': string
end;

functor SYNTAX_DEF_FUN(XGram:XGRAM) : SYNTAX_DEF =
struct

structure XGram = XGram;
structure Symtab = XGram.Symtab;

local open XGram in

val max_pri = 1000; (* maximum legal priority *)


val fnnt = "_-->_";
val Afn = Ground fnnt;
fun typ_to_nt (Ground s) = s | typ_to_nt _ = fnnt;

(* Terminal types *)

val SId    = Ground "_Id";
val SVar   = Ground "_Var";
val Aterminals = [SId,SVar];

fun nonts syn = itlist_left (fn (i,"_") => i+1 | (i,_) => i) (0,explode syn);

val meta_chs = ["(",")","/","_"];

fun mk_term(pref,[]) = (pref,[])
  | mk_term(pref,"'"::c::cl) = mk_term(pref^c,cl)
  | mk_term(pref,l as c::cl) = if is_blank(c) orelse c mem meta_chs
	then (pref,l) else mk_term(pref^c,cl);

fun mk_space(sp,[]) = (sp,[]) |
    mk_space(sp,cl as c::cl') =
      if is_blank(c) then mk_space(sp^c,cl') else (sp,cl);

exception ARG_EX;

fun mk_syntax([],ar,_,sy) = (sy,ar)
  | mk_syntax("_"::cl,ar-->ar',[],sy) =
	mk_syntax(cl,ar',[],sy@[Nonterminal(typ_to_nt ar,0)])
  | mk_syntax("_"::cl,ar-->ar',p::pl,sy) =
	mk_syntax(cl,ar',pl,sy@[Nonterminal(typ_to_nt ar,p)])
  | mk_syntax("_"::cl,_,_,_) = raise ARG_EX
  | mk_syntax("("::cl,ar,pl,sy) = let val (i,cl') = scan_int cl
	in mk_syntax(cl',ar,pl,sy@[Bg(i)]) end
  | mk_syntax(")"::cl,ar,pl,sy) = mk_syntax(cl,ar,pl,sy@[En])
  | mk_syntax("/"::cl,ar,pl,sy) = let val (sp,cl') = take_prefix is_blank cl
	in mk_syntax(cl',ar,pl,sy@[Brk(length sp)]) end
  | mk_syntax(c::cl,ar,pl,sy) =	let val (term,rest) =
	   if is_blank(c)
	   then let val (sp,cl') = mk_space(c,cl) in (Space(sp),cl') end
	   else let val (tk,cl') = mk_term("",c::cl) in(Terminal(tk),cl') end
	in mk_syntax(rest,ar,pl,sy@[term]) end;

fun pri_test1 p = if 0 <= p andalso p <= max_pri then ()
	else error("Priority out of range: " ^ string_of_int p ^ "\n")
fun pri_test(pl,p) = (pri_test1 p; seq pri_test1 pl);

fun mk_prod2(sy,ty,opn,pl,p) =
    let val (syn,ty') = mk_syntax(explode sy, ty, pl, []) handle ARG_EX =>
	error("More arguments in "^sy^" than in corresponding type\n")
    in Prod(typ_to_nt ty',syn,opn,p) end;

fun mk_prod1(sy,ty,opn,pl,p) = (pri_test(pl,p); mk_prod2(sy,ty,opn,pl,p));

datatype Assoc = rightA | leftA;

datatype mixfix = Mixfix of string * typ * string * int list * int
		| Delimfix of string * typ * string
		| Infixl of string * typ * int
		| Infixr of string * typ * int;

fun terminal1(ty as _-->_) = hd(binder_types ty) mem Aterminals
  | terminal1 _ = false;

fun mk_prod(Delimfix(sy,ty,const)) = mk_prod(Mixfix(sy,ty,const,[],max_pri))
  | mk_prod(Mixfix(sy,ty,"",pl,p)) = if nonts sy <> 1
	then error"Copy op must have exactly one argument.\n" else
	if filter_out is_blank (explode sy) = ["_"] andalso
	   not(terminal1 ty)
	then [mk_prod2(sy,ty,"",[copy_pri],copy_pri)]
	else [mk_prod1(sy,ty,"",pl,p)]
  | mk_prod(Mixfix(sy,ty,const,pl,p)) = [mk_prod1(sy,ty,const,pl,p)]
  | mk_prod(Infixr(sy,ty,p)) = [mk_prod1("op "^sy,ty,sy,[],max_pri),
	mk_prod1("(_ "^sy^"/ _)",ty,sy,[p+1,p],p)]
  | mk_prod(Infixl(sy,ty,p)) = [mk_prod1("op "^sy,ty,sy,[],max_pri),
	mk_prod1("(_ "^sy^"/ _)",ty,sy,[p,p+1],p)];

(* The pure syntax *)

val Any = Ground "_???";

val appl_const' = "_F(...)";

val Args = Ground "_Arg List";

fun descend(from,to) = Delimfix("_",to-->from,"");

fun parents(ty) = Delimfix("(1'(_'))",ty-->ty,"");

fun mkappl(ty) =
    Mixfix("_(1'(_'))", [Afn,Args]--->ty, appl_const', [max_pri,0],max_pri);

fun mkid(ty) = Delimfix("_",SId-->ty,"");
fun mkvar(ty) = Delimfix("_",SVar-->ty,"");

val STyp = Ground"_type";
fun constrain ty = Mixfix("_ $$ _", [ty,STyp]--->ty, "_constrain", [1,0], 0);

fun get_string(Ground s) = s
  | get_string _ = error("Function type among logical types.\n");

val empty = XGram{Roots=[], TrTab1=Symtab.null, TrTab2=Symtab.null, Prods=[]};

fun extend (XGram{Roots=Roots,TrTab1=TrTab1,TrTab2=TrTab2,Prods=Prods})
           {logical_types=Atypes,mixfix=fixl,
	    parse_translation=trl1,print_translation=trl2} =
    let val Anames = map get_string Atypes;
	val Atypes' = Any rem Atypes;
	val Atypes'' = Aprop rem Atypes';
        val roots = Anames union Roots;(*
            if exists (apr(op mem,Roots)) Anames
            then (prs"Some type in "; print_list("[","]",prs) Anames;
                  prs"\nis already in "; print_list("[","]",prs) Roots;
                  raise ERROR)
            else Anames union Roots;*)

        val fixl' = fixl @ map parents Atypes' @ map mkappl Atypes'' @
                    map mkid Atypes'' @ map mkvar Atypes'' @
		    map constrain Atypes'' @
                    map (apl(Any,descend)) Atypes';

    in XGram{Roots = roots,
             TrTab1 = Symtab.balance(Symtab.st_of_alist(trl1,TrTab1)),
             TrTab2 = Symtab.balance(Symtab.st_of_alist(trl2,TrTab2)),
             Prods = Prods @ flat(map mk_prod fixl')}
    end;

fun const(Delimfix(_,T,c)) = ([c],T)
  | const(Mixfix(_,T,c,_,_)) = ([c],T)
  | const(Infixl(c,T,_)) = ([c],T)
  | const(Infixr(c,T,_)) = ([c],T);

fun syntactic([s],_) = s="" orelse hd(explode s)=" ";

fun constants(ml) = distinct(filter_out syntactic (map const ml));

end;

end;
