(*  Title: 	FOL/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

First-order logic (Natural Deduction). 

Should allow multiple quantification like ALL x y.P
*)


signature FOL_SYNTAX =
sig
  structure Syntax : PURE_SYNTAX
  val Aform: typ
  val Aterm: typ
  val const_decs: (string list * typ) list
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor FOL_SyntaxFun (Syntax: PURE_SYNTAX) : FOL_SYNTAX = 
struct
structure Syntax = Syntax;
local open Syntax
in


(*Meta-types for terms, formulae, types*)
val Aterm  = Ground "term";
val Aform  = Ground "form";

(** Quantifier translations: variable binding **)

fun qnt_tr q [idl,P] = abs_list_tr(Const(q,Adummy), idl, P);

fun qnt_tr'(Q,q) (tm as _ $ Abs(_,_,_)) =
    abs_list_tr'(Const(q,Adummy), strip_qnt_vars Q tm, strip_qnt_body Q tm);

val mixfix =
[   (*Propositions*)
Mixfix("(_)", Aform --> Aprop, "Trueprop", [0], 5),
    (*Connectives*)
Infixl("=", [Aterm,Aterm]--->Aform, 50),
Mixfix("~_", Aform-->Aform, "Not", [40],40),
Infixr("&", [Aform,Aform]--->Aform, 35),
Infixr("|", [Aform,Aform]--->Aform, 30),
Infixr("-->", [Aform,Aform]--->Aform, 25),
Infixr("<->", [Aform,Aform]--->Aform, 25),
Mixfix("(3ALL _./ _)", [id_list,Aform]--->Aform, " ALL", [], 10),
Mixfix("(3EX _./ _)",  [id_list,Aform]--->Aform, " EX", [], 10)
];
val ext = {logical_types = [Aterm,Aform], mixfix=mixfix,
parse_translation =
[(" ALL", qnt_tr "Forall"), (" EX", qnt_tr "Exists")],
print_translation =
[("Forall", qnt_tr'("Forall", " ALL")),
 ("Exists", qnt_tr'("Exists", " EX")) ]};

val const_decs = constants mixfix @
[ (["True","False"], Aform),
  (["Forall","Exists"], (Aterm-->Aform)-->Aform)
];

val syn = Syntax.extend pure ext;

fun read a = Syntax.read syn Any a;
fun prin t = Syntax.print_top_level syn t;

end;
end;
