(*  Title: 	HOL/tarski
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Tarski's theorem

Should fix take an object-function rather than a meta-function??
*)


(*** Theory containing constructs for Tarski's theorem ***)

goals_limit := 2;

val tarski_const_decs = 
  [ (["monotone"], 	[Atype,Atype,Aterm-->Aterm]--->Aform),
    (["fix"], 		[Atype,Aterm-->Aterm]--->Aterm),
    (["compose"],	[Atype,Atype,Atype,Aterm,Aterm]--->Aterm),
    (["image"],		[Atype,Atype,Aterm-->Aterm,Aterm]--->Aterm),
    (["idrel"],		Atype-->Aterm),
    (["idtranclose_fun"],	[Atype,Aterm,Aterm]--->Aterm),
    (["idtranclose"],	[Atype,Aterm]--->Aterm),
    (["tranclose"],	[Atype,Aterm]--->Aterm) ];


val tarski_thy = extend_theory HOL_Rule.thy  "tarski" 
    ([], tarski_const_decs)
  [ ("monotone_def", 
     "monotone(A,B,f) == ALL u: A->bool. ALL v: A->bool.\
\		subset(A, u, v) --> subset(B, f(u), f(v))"),

    ("fix_def", 
     "fix(A,f) == inter(A, {| u:A->bool. subset(A, f(u), u) |})"),

    ("compose_def", 
     "compose(A,B,C,S,R) == \
\  	{| u: A*C. EX x:A. EX y:B. EX z:C. \
\		<x,y> <: R & <y,z> <: S & (u = <x,z> : A*C) |}"),

    ("image_def",
      "image(A,B,b,S) == {| z:B. EX x:A.  x<:S & (z = b(x) : B) |}"),

    ("idrel_def",
      "idrel(A) == {| z:A*A. EX x:A. z = <x,x> : A*A |}"),

    ("idtranclose_fun_def",  
      "idtranclose_fun(A,R,S) == un(A*A, idrel(A), compose(A,A,A,R,S))"),

    ("idtranclose_def",   (*written eta-expanded to avoid matching problems*)
      "idtranclose(A,R) == fix(A*A, %u.idtranclose_fun(A,R,u))"),

    ("tranclose_def",   
      "tranclose(A,R) == compose(A,A,A, R, idtranclose(A,R))") ];

local val ax = get_axiom tarski_thy
in  val monotone_def = ax"monotone_def"
    and fix_def = ax"fix_def"
    and compose_def = ax"compose_def"
    and image_def = ax"image_def"
    and idrel_def = ax"idrel_def"
    and idtranclose_fun_def = ax"idtranclose_fun_def"
    and idtranclose_def = ax"idtranclose_def"
    and tranclose_def = ax"tranclose_def"
end;


(*** Lemmas for Tarski's theorem ***)

(** Proved in Theory HOL_Rule **)

val asms = goal HOL_Rule.thy
    "[| subset(A,R,S);  subset(A,S,T);  R: A->bool;  S: A->bool |] ==> \
\    subset(A,R,T)";
by (cut_facts_tac asms 1);
by (Class.fast_tac [] 1);
val subset_trans = result();


val asms = goal HOL_Rule.thy
    "[| subset(A,R,S);  subset(A,S,R);  R: A->bool;  S: A->bool |] ==> \
\    R = S : A->bool";
by (cut_facts_tac asms 1);
by (Class.fast_tac [equal_intr] 1);
val subset_antisym = result();


val asms = goal HOL_Rule.thy
  "[| R<:F;  F: (A->bool)->bool;  R: A->bool |] ==> subset(A, inter(A,F), R)";
by (cut_facts_tac asms 1);
by (Class.fast_tac class_type_rls 1);
val inter_subset_member = result();


val asms = goal HOL_Rule.thy
    "(!u.[| u: A->bool;  u<:F |] ==> subset(A, R, u)) ==> [| F: (A->bool)->bool;  R: A->bool |] ==> subset(A, R, inter(A,F))";
by (REPEAT (Class.step_tac asms 1
      ORELSE  (eresolve_tac (reslist(asms,1,subset_elim)) 1)));
val inter_is_glb = result();


(*** We now switch to Theory tarski_thy ***)

(*** This theory and proof ought to involve axioms stating that
     f is monotone, etc., and defining the partial ordering.
     Then interpretations of the theory would "discharge" these axioms. ***)

val asms = goal tarski_thy
    "[| monotone(A,B,f);  subset(A,R,S);  R: A->bool;  S: A->bool |] ==> \
\    subset(B, f(R), f(S))";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [monotone_def]);
by (Pc.fast_tac [] 1);
val mono_elim = result();


(*f is a function variable since fix is really a binding operator.
  BEWARE: conclusion matches any subset(A, R, S)!  with f = u.R  *)
val asms = goal tarski_thy
    "monotone(A,A,f) ==> (!u.u: A->bool ==> f(u): A->bool) ==> [| S: A->bool;  subset(A, f(S), S) |] ==> subset(A, f(fix(A,f)), S)";
by (rewrite_goals_tac [fix_def]);
by (resolve_tac [subset_trans] 1);
by (resolve_tac asms 2);
by (REPEAT_FIRST (ares_tac ([inter_subset_member,cla_intr] 
	@ class_type_rls @ type_rls @ reslist(asms,1,mono_elim) @ asms)));
val tarski_lemma1 = result();


val asms = goal tarski_thy
    "[| monotone(A,A,f);  !u.u: A->bool ==> f(u): A->bool |] ==> \
\    subset(A, f(fix(A,f)), fix(A,f))";
by (rewrite_goals_tac [fix_def]);
by (REPEAT_FIRST (ares_tac ([inter_is_glb] 
			@ class_type_rls @ type_rls @ asms)));
by (fold_tac [fix_def]);
by (resolve_tac [tarski_lemma1] 1);   (*not to be REPEATed!*)
by (Class.best_tac asms);
val tarski_lemma2 = result();


val asms = goal tarski_thy
    "[| monotone(A,A,f);  !u.u: A->bool ==> f(u): A->bool |] ==> \
\    subset(A, fix(A,f), f(fix(A,f)))";
by (rewrite_goals_tac [fix_def]);
by (REPEAT_FIRST (ares_tac ([inter_subset_member,cla_intr] 
	@ class_type_rls @ type_rls @ reslist(asms,1,mono_elim) @ asms)));
by (fold_tac [fix_def]);
by (REPEAT_FIRST (ares_tac ([tarski_lemma2] @ asms)));
val tarski_lemma3 = result();


val asms = goal tarski_thy
    "(!u.u: A->bool ==> f(u): A->bool) ==> fix(A, f) : A -> bool";
by (rewrite_goals_tac [fix_def]);
by (Class.typechk_tac asms);
val fix_type = result();


val asms = goal tarski_thy
    "monotone(A,A,f) ==> (!u.u: A->bool ==> f(u): A->bool) ==> ( fix(A,f) = f(fix(A,f)) : A->bool )";
by (REPEAT_FIRST (ares_tac ([equal_intr,tarski_lemma2,tarski_lemma3,fix_type] 
			@ type_rls @ asms)));
val tarski_theorem = result();


val asms = goal tarski_thy
    "monotone(A,A,f) ==> subset(A, f(S), S) ==> (!u.u: A->bool ==> f(u): A->bool) ==> S: A->bool ==> subset(A, fix(A,f), S)";
by (resolve_tac [tarski_theorem RS subst] 1);
by (REPEAT (ares_tac (asms @ [tarski_lemma1]) 1));
val tarski_lemma4 = result();



(*** Lemmas about subsets ***)

val asms = goal HOL_Rule.thy
    "[| subset(A,T,R); subset(A,T,S); R: A->bool; S: A->bool; T: A->bool |] \
\     ==> subset(A, T, int(A,R,S))";
by (cut_facts_tac asms 1);
by (Class.fast_tac [] 1);
val subset_int_intr = result();


val asms = goal tarski_thy
    "[| monotone(A,A,f);  R: A->bool;  S: A->bool;  \
\       !u.u: A->bool ==> f(u): A->bool |] ==> \
\    subset(A, f(int(A,R,S)), int(A,f(R),f(S)))";
by (REPEAT_FIRST (resolve_tac ([subset_int_intr] @ reslist(asms,1,mono_elim) @ asms) ));
by (Class.best_tac asms);
val subset_int_mono = result();


(** Induction rule for fixedpoints **)

val asms = goal tarski_thy
    "[| a <: fix(A,f);  a: A |] ==> monotone(A,A,f) ==> (!x.[| x: A;  x <: f(int(A, fix(A,f), {|x: A.P(x)|})) |] ==> P(x)) ==> (!u.u: A->bool ==> f(u): A->bool) ==> P(a)";
by (res_inst_tac [("S", "int(A, fix(A,f), {|x: A.P(x)|})", Aterm),
		   ("c", "a", Aterm)]
		subset_elim 1);
by (resolve_tac [tarski_lemma4] 1);
by (REPEAT (ares_tac (asms @ [subset_int_intr]) 1));
by (resolve_tac [subset_trans] 1);
by (resolve_tac (reslist(asms,1,subset_int_mono)) 1);
by (Class.typechk_tac ([fix_type] @ asms));
by (resolve_tac (reslist(asms,1,tarski_theorem RS sym RS subst)) 1);
by (Class.best_tac ([fix_type] @ asms));
val tarski_induction = result();


(*** The identity relation "idrel" ***)

val asms = goal tarski_thy
    "idrel(A): A*A->bool";
by (rewrite_goals_tac [idrel_def]);
by (Class.typechk_tac []);
val idrel_type = result();


val asms = goal tarski_thy
    "a: A ==> <a,a> <: idrel(A)";
by (rewrite_goals_tac [idrel_def]);
by (Class.fast_tac ([refl]@asms) 1);
val idrel_intr = result();


val asms = goal tarski_thy
    "c <: idrel(A) ==> (!x.[| x: A;  c = <x,x> : A*A |] ==> P) ==> c: A*A ==> P";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [idrel_def]);
by (Class.fast_tac asms 1);
val idrel_elim = result();


(*** The operation "image" ***)

val asms = goal tarski_thy
    "(!x.x: A ==> b(x): B) ==> image(A,B,b,S): B->bool";
by (rewrite_goals_tac [image_def]);
by (Class.typechk_tac asms);
val image_type = result();


val asms = goal tarski_thy
    "[| a <: S;  a: A |] ==> (!x.x: A ==> b(x): B) ==> b(a) <: image(A,B,b,S)";
by (rewrite_goals_tac [image_def]);
by (Class.fast_tac ([refl]@asms) 1);
val image_intr = result();


val asms = goal tarski_thy
    "c <: image(A,B,b,S) ==> (!x.[| x: A;  x <: S;  ( c = b(x) : B ) |] ==> P) ==> c: B ==> P";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [image_def]);
by (Class.fast_tac asms 1);
val image_elim = result();


(*** The operation "compose" -- composition of relations ***)

val asms = goal tarski_thy
    "[| R: A*B->bool;  S: B*C->bool |] ==> compose(A,B,C,S,R): A*C->bool";
by (rewrite_goals_tac [compose_def]);
by (Class.typechk_tac asms);
val compose_type = result();


val asms = goal tarski_thy
    "[| <a,b> <: R;  <b,c> <: S;  a: A;  b: B;  c: C |] ==> <a,c> <: compose(A,B,C,S,R)";
by (rewrite_goals_tac [compose_def]);
by (Class.fast_tac ([refl]@asms) 1);
val compose_intr = result();


(*occasionally needed in this form*)
val asms = goal tarski_thy
    "e <: compose(A,B,C,S,R) ==> (!x y z.[| x: A;  y: B;  z: C;  <x,y> <: R;  <y,z> <: S;  e= <x,z> : A*C |] ==> P) ==> e: A*C ==> P";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [compose_def]);
by (Class.fast_tac asms 1);
val compose_elim_raw = result();

(*better version*)
val asms = goal tarski_thy
    "<a,c> <: compose(A,B,C,S,R) ==> (!y.[| y: B;  <a,y> <: R;  <y,c> <: S |] ==> P) ==> [| a: A;  c: C |] ==> P";
by (resolve_tac (reslist(asms,1,compose_elim_raw)) 1);
by (REPEAT (ares_tac (asms@type_rls) 1  
	ORELSE  eresolve_tac [pair_inject,subst] 1));
val compose_elim = result();



(*** Monotonicity of set operations ***)

val asms = goal tarski_thy
    "[| subset(A, R, S);  R: A->bool |] ==> subset(B, image(A,B,b,R), image(A,B,b,S))";
by (rewrite_goals_tac [image_def]);
by (cut_facts_tac asms 1);
by (Class.fast_tac [] 1);
val image_mono = result();


val asms = goal tarski_thy
    "[| subset(A, R', R);  subset(A, S', S);  R': A->bool;  S': A->bool;  R : A->bool;  S : A->bool |] ==> subset(A, un(A,R',S'),un(A,R,S))";
by (cut_facts_tac asms 1);
by (Class.fast_tac [] 1);
val un_mono = result();


val asms = goal tarski_thy
    "[| subset(A*B, R', R);  subset(B*C, S', S);  R': A*B->bool;  S': B*C->bool |] ==> subset(A*C, compose(A,B,C,S',R'), compose(A,B,C,S,R))";
by (rewrite_goals_tac [compose_def]);
by (cut_facts_tac asms 1);
by (Class.fast_tac [] 1);
val compose_mono = result();


val asms = goal tarski_thy
    "R : A->bool ==> subset(A, R, R)";
by (Class.fast_tac asms 1);
val subset_refl = result();


(*** Transitive closure of a relation ***)

(** The functional idtranclose_fun **)

val asms = goal tarski_thy
    "[| R: A*A->bool;  S: A*A->bool |] ==> idtranclose_fun(A,R,S) : A*A->bool";
by (rewrite_goals_tac [idtranclose_fun_def]);
by (Class.typechk_tac (asms@[idrel_type,compose_type]));
val idtranclose_fun_type = result();


val asms = goal tarski_thy
    "R: A*A->bool ==> monotone(A*A, A*A, idtranclose_fun(A,R))";
by (rewrite_goals_tac [monotone_def, idtranclose_fun_def]);
by (REPEAT (ares_tac [all_intr,imp_intr,subset_refl,compose_mono,un_mono] 1
	THEN  Class.typechk_tac (asms@[idrel_type,compose_type])));
val idtranclose_fun_mono = result();


(** The relation idtranclose(A,R) **)

val asms = goal tarski_thy
    "R: A*A->bool ==> idtranclose(A,R) : A*A->bool";
by (rewrite_goals_tac [idtranclose_def]);
by (Class.typechk_tac (asms @ [fix_type, idtranclose_fun_type]));
val idtranclose_type = result();


val idtranclose_type_rls = 
    [idtranclose_type,idtranclose_fun_type,idrel_type,compose_type];


val asms = goal tarski_thy
    "R: A*A->bool ==> \
\    idtranclose(A,R) = idtranclose_fun(A, R, idtranclose(A,R)) : A*A->bool";
by (rewrite_goals_tac [idtranclose_def]);
by (REPEAT (ares_tac (asms @ 
	[tarski_theorem, idtranclose_fun_mono, idtranclose_fun_type]) 1));
val idtranclose_unfold = result();

(*Reflexivity of idtranclose*)
val asms = goal tarski_thy
    "[| R: A*A->bool;  a: A |] ==> <a,a> <: idtranclose(A,R)";
by (resolve_tac [idtranclose_unfold RS subst] 1);
by (rewrite_goals_tac [idtranclose_fun_def]);
by (Class.fast_tac ([idrel_intr] @ idtranclose_type_rls @ asms) 1);
val idtranclose_refl = result();

(*Closure under composition with R*)
val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  <b,c> <: R; \
\       a: A;  b: A;  c: A;  R: A*A->bool |]   ==> <a,c> <: idtranclose(A,R)";
by (resolve_tac [idtranclose_unfold RS subst] 1);
by (rewrite_goals_tac [idtranclose_fun_def]);
by (Class.fast_tac ([compose_intr] @ idtranclose_type_rls @ asms) 1);
val idtranclose_intr2 = result();

(*idtranclose of R contains R*)
val asms = goal tarski_thy
   "[| <a,b> <: R;  a:A;  b:A;  R:A*A->bool |] ==> <a,b> <: idtranclose(A,R)";
by (resolve_tac [idtranclose_intr2] 1);
by (resolve_tac [idtranclose_refl] 1);
by (REPEAT (resolve_tac asms 1));
val idtranclose_intr1 = result();


(** standard induction rule **)

val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  !x y.x: A ==> P(<x,x>); \
\       !x y z.[| x: A;  y: A;  z: A;  P(<x,y>);  \
\ 		  <x,y> <: idtranclose(A,R);  <y,z> <: R |]  ==>  P(<x,z>); \
\       a: A;  b: A;  R: A*A->bool |]   ==>  P(<a,b>)";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [idtranclose_fun_def,idtranclose_def]);
by (eresolve_tac [tarski_induction] 1);
by (eresolve_tac [un_elim] 3);
by (fold_tac [idtranclose_fun_def,idtranclose_def]);
by (REPEAT_FIRST (eresolve_tac [idrel_elim,compose_elim_raw,subst]
	ORELSE' Class.step_tac ([idtranclose_fun_mono]@idtranclose_type_rls)));
by (ALLGOALS (ares_tac asms));
by (REPEAT_SOME assume_tac);
val idtranclose_full_induction = result();


(** nice induction rule **)

fun subst_tac sP i = 
    res_inst_tac [("P",sP, Aterm-->Aform)] subst i;

(*Matches only equality assumptions, to avoid smashing other goals*)
val eq_asm_rl = trivial(Sign.read_cterm HOL_Rule.sign 
			("?a = ?b: ?A",Aprop));

val sym_equal_tac = eresolve_tac [eq_asm_rl, sym];

val triv_equal_tac = resolve_tac [refl]
	    ORELSE'  sym_equal_tac;

fun trans_tac i = resolve_tac [trans] i
     THEN REPEAT_SOME sym_equal_tac;

val equal_tac = triv_equal_tac ORELSE' trans_tac;


val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  P(a);  \
\	!y z.[| y:A;  z:A;  P(y);  <a,y> <: idtranclose(A,R); <y,z> <: R |] \
\	     ==> P(z); \
\	a: A;  b: A;  R: A*A->bool |]   ==> P(b)";
by (subgoal_tac "ALL y:A. (<a,b> = <a,y> : A*A) --> P(y)" 1);
(*now solve first subgoal*)
by (REPEAT_FIRST (ares_tac [refl]    
	ORELSE'  eresolve_tac [all_elim,imp_elim,subst]));
by (Pc.typechk_tac asms);
by (resolve_tac (reslist(asms,1,idtranclose_full_induction)) 1);
(*delicate task of instantiating induction hyp properly*)
by (REPEAT_FIRST (ares_tac [refl,all_intr,imp_intr] 
	ORELSE' subst2_tac "Pair"  
	ORELSE' eresolve_tac [pair_inject,all_elim,imp_elim]));
by (Pc.typechk_tac asms);
by (subst_tac "P" 1);
by (resolve_tac asms 2);
by (equal_tac 1);
by (resolve_tac asms 1);
by (REPEAT_FIRST (IF_PROP assume_tac));
by (REPEAT (assume_tac 1 ORELSE eresolve_tac [sym RS subst] 1));
val idtranclose_induction = result();


(*transitivity of transitive closure (!!!) -- by induction.*)
val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  <b,c> <: idtranclose(A,R);  \
\	a: A;  b: A;  c: A;  R: A*A->bool |]   ==> <a,c> <: idtranclose(A,R)";
by (res_inst_tac [("b","c",Aterm)] idtranclose_induction 1);
by (REPEAT (ares_tac asms 1 ORELSE eresolve_tac [idtranclose_intr2] 1));
val idtranclose_trans = result();


(*elimination of idtranclose -- by induction *)
val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  a: A;  b: A;  R: A*A->bool |] ==> \
\    (a = b : A)  | (EX y:A. <a,y> <: idtranclose(A,R) & <y,b> <: R)";
by (res_inst_tac [("b","b",Aterm)] idtranclose_induction 1);
by (REPEAT (Class.step_tac (asms@[refl]) 1));
val idtranclose_elim_raw = result();


(*elimination of idtranclose -- nice version *)
val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  a: A;  b: A;  R: A*A->bool; \
\	(a = b : A) ==> P;  \
\	!y.[| y: A;  <a,y> <: idtranclose(A,R);  <y,b> <: R |] ==> P |] \
\    ==> P";
by (resolve_tac [idtranclose_elim_raw RS disj_elim] 1);
by (REPEAT (eresolve_tac ([exists_elim,conj_elim]@asms) 1 
	ORELSE  ares_tac asms 1));
val idtranclose_elim = result();


(**** The relation tranclose(A,R) ****)

(** Conversions between tranclose and idtranclose **)

val asms = goal tarski_thy
    "[| <a,b> <: tranclose(A,R);  a: A;  b: A;  R: A*A->bool |] \
\    ==> <a,b> <: idtranclose(A,R)";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [tranclose_def]);
by (REPEAT (eresolve_tac [compose_elim] 1
	ORELSE  Class.step_tac ([idtranclose_intr2]@idtranclose_type_rls) 1));
val idtranclose_intr_tran = result();

val asms = goal tarski_thy
    "[| <a,b> <: idtranclose(A,R);  <b,c> <: R;  \
\	a: A;  b: A;  c: A;  R: A*A->bool |]   ==>  <a,c> <: tranclose(A,R)";
by (rewrite_goals_tac [tranclose_def]);
by (Class.fast_tac ([compose_intr] @ idtranclose_type_rls @ asms) 1);
val tranclose_intr_id = result();


val asms = goal tarski_thy  "R: A*A->bool ==> tranclose(A,R) : A*A->bool";
by (rewrite_goals_tac [tranclose_def]);
by (Class.typechk_tac (asms @ [idtranclose_type,compose_type]));
val tranclose_type = result();


(*tranclose of R contains R*)
val asms = goal tarski_thy
   "[| <a,b> <: R;  a: A;  b: A;  R: A*A->bool |] ==> <a,b> <: tranclose(A,R)";
by (rewrite_goals_tac [tranclose_def]);
by (Class.fast_tac ([compose_intr,idtranclose_refl,
		     idtranclose_type,compose_type] @ asms) 1);
val tranclose_intr1 = result();


(*Closure under composition with R*)
val asms = goal tarski_thy
    "[| <a,b> <: tranclose(A,R);  <b,c> <: R;  \
\	a: A;  b: A;  c: A;  R: A*A->bool |]  ==>  <a,c> <: tranclose(A,R)";
by (rewrite_goals_tac [tranclose_def]);
by (Class.fast_tac ([idtranclose_intr_tran,compose_intr] @
		    idtranclose_type_rls @ asms) 1);
val tranclose_intr2 = result();


(*elimination of tranclose -- basic version*)
val asms = goal tarski_thy
    "[| <a,b> <: tranclose(A,R);  a: A;  b: A;  R: A*A->bool |] ==> \
\    <a,b> <: R | (EX y:A. <a,y> <: tranclose(A,R) & <y,b> <: R)";
by (resolve_tac [compose_elim] 1);
by (resolve_tac (map (rewrite_rule [tranclose_def]) asms) 1);
by (REPEAT (eresolve_tac [idtranclose_elim, subst] 1
            THEN  Class.typechk_tac asms));
by (REPEAT (Class.step_tac ([tranclose_intr_id]@asms) 1));
val tranclose_elim_raw = result();


(*elimination of tranclose -- nice version *)
val asms = goal tarski_thy
    "[| <a,b> <: tranclose(A,R);  a: A;  b: A;  R: A*A->bool; \
\	<a,b> <: R ==> P; \
\	!y.[| y: A;  <a,y> <: tranclose(A,R);  <y,b> <: R |] ==> P |] \
\    ==> P";
by (resolve_tac [tranclose_elim_raw RS disj_elim] 1);
by (REPEAT (eresolve_tac ([exists_elim,conj_elim]@asms) 1 
	ORELSE  ares_tac asms 1));
val tranclose_elim = result();


(*transitivity of tranclose *)
val asms = goal tarski_thy
    "[| <a,b> <: tranclose(A,R);  <b,c> <: tranclose(A,R);  \
\	a: A;  b: A;  c: A;  R: A*A->bool |]  ==>  <a,c> <: tranclose(A,R)";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [tranclose_def]);
by (REPEAT (eresolve_tac [compose_elim] 1));
by (resolve_tac ([compose_intr]) 1);
by (assume_tac 2);
by (resolve_tac ([idtranclose_trans]) 1);
by (assume_tac 2);
by (resolve_tac ([idtranclose_intr2]) 1);
by (REPEAT (assume_tac 1));
val tranclose_trans = result();

