/* SPIM S20 MIPS simulator.
   X interface to SPIM
   Copyright (C) 1990 by James Larus (larus@cs.wisc.edu), James R. Goodman,
   and Alan Yuen-wui Siow.

   SPIM is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 1, or (at your option) any
   later version.

   SPIM is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GNU CC; see the file COPYING.  If not, write to James R.
   Larus, Computer Sciences Department, University of Wisconsin--Madison,
   1210 West Dayton Street, Madison, WI 53706, USA or to the Free
   Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */


/* $Header: /n/fs/vb/egs/src/spim/RCS/xspim.c,v 1.1 92/05/27 16:28:10 egs Exp Locker: egs $
 */

#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xlib.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/TextP.h>
#include <X11/Xaw/Dialog.h>
#include <X11/keysym.h>

#include <stdio.h>
#include <signal.h>
#include <varargs.h>
#include <setjmp.h>
#ifndef DST_NONE
#include <sys/time.h>
#endif

#include "xspim.h"
#include "spim.h"
#include "inst.h"
#include "mem.h"
#include "reg.h"
#include "y.tab.h"


/* Exported functions: */

void MenuSelect ();
void PopConsole ();
void PopTerminal ();
void center_text_at_PC ();
void execute_program ();
void error ();
void initialize_console ();
int main ();
void read_file ();
void read_input ();
void redisplay_data ();
void redisplay_text ();
int run_error ();
void start_program ();


/* Imported functions: */

void Confirm ();
void ContinuePrompt ();
void CreateSubWindows ();
long atol ();
extern int exit ();
void fatal_error ();
mem_addr find_symbol_address ();
FILE *fopen ();
void initialize_world ();
int print_inst_internal ();
int read_aout_file ();
int read_assembly_file ();
int register_name_to_number ();
int run_program ();
mem_addr starting_address ();
char *str_copy ();
int vfprintf ();


/* Local functions: */

#ifdef IO
static void GenInterrupt ();
#endif
static void addtoq ();
static void control_c_seen ();
static void display_data_seg ();
static char *display_values ();
static char *display_insts ();
static void display_registers ();
static void initialize ();
static void printqueue ();
static void showRunning ();
static void syntax ();
static void writeText ();


typedef struct t_node
{
  struct timeval timeout;
  int exception_code;
  struct t_node *next;
}
time_node;


typedef struct _AppResources
{
  String textFont;
  Boolean bare;
  Boolean asmm;
  Boolean trap;
  Boolean quiet;
  char *filename;
  char *ex_filename;
  char *display2;
  Boolean hex_gpr;
  Boolean hex_fpr;
  Boolean hex_fgr;
  char *initial_data_limit;
  char *initial_data_size;
  char *initial_k_data_limit;
  char *initial_k_data_size;
  char *initial_k_text_size;
  char *initial_stack_limit;
  char *initial_stack_size;
  char *initial_text_size;
} AppResources;


static String fallback_resources[] =
{
  "*font:		*-courier-medium-r-normal--12-*-75-*",
  "*Label*font:		*-adobe-helvetica-bold-r-*-*-12-*-75-*",
  "*panel*font:		*-adobe-helvetica-medium-r-*-*-12-*-75-*",
  "*ShapeStyle:		Oval",
  "*dialog*value.translations: #override \\n <Key>Return: Confirm()",
  "*.translations: #override \\n <Ctrl>C: control_c_seen()",
  "*Form*left:		ChainLeft",
  "*Form*right:		ChainLeft",
  "*Form*top:		ChainTop",
  "*Form*bottom:	ChainTop",
  "*terminal1.label:	SPIM Console",
  "*Shell1*iconName:	SPIM Console",
  "*terminal2.label:	SPIM Terminal",
  "*Shell2*iconName:	SPIM Terminal ",
  NULL,
};


XtActionsRec actionTable[] =
{
  {"Confirm", Confirm},
  {"control_c_seen", control_c_seen},
};


static XtResource resources[] =
{
  {XtNfont, XtCFont, XtRString, sizeof (char *),
     XtOffset (AppResources *, textFont), XtRString, NULL},
  {"bare", "Bare", XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, bare), XtRImmediate, False},
  {"asmm",  "Asmm",  XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, asmm), XtRImmediate, False},
  {"trap", "Trap", XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, trap), XtRImmediate, (XtPointer) True},
  {"quiet", "Quiet", XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, quiet), XtRImmediate, False},
  {"filename", "Filename", XtRString, sizeof (char *),
     XtOffset (AppResources *, filename), XtRString, NULL},
  {"ex_filename", "Ex_Filename", XtRString, sizeof (char *),
     XtOffset (AppResources *, ex_filename), XtRString, NULL},
  {"display2", "Display2", XtRString, sizeof (char *),
     XtOffset (AppResources *, display2), XtRString, NULL},
  {"hexGpr", "DisplayHex", XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, hex_gpr), XtRImmediate, (XtPointer) True},
  {"hexFpr", "DisplayHex", XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, hex_fpr), XtRImmediate, False},
  {"hexFgr", "DisplayHex", XtRBoolean, sizeof (Boolean),
     XtOffset (AppResources *, hex_fgr), XtRImmediate, False},

  {"stext", "Stext", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_text_size), XtRString, NULL},
  {"sdata", "Sdata", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_data_size), XtRString, NULL},
  {"ldata", "Ldata", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_data_limit), XtRString, NULL},
  {"sstack", "Sstack", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_stack_size), XtRString, NULL},
  {"lstack", "Lstack", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_stack_limit), XtRString, NULL},
  {"sktext", "Sktext", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_k_text_size), XtRString, NULL},
  {"skdata", "Skdata", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_k_data_size), XtRString, NULL},
  {"lkdata", "Lkdata", XtRString, sizeof (char *),
     XtOffset (AppResources *, initial_k_data_limit), XtRString, NULL}
};


static XrmOptionDescRec options[] =
{
  {"-bare",   "bare", XrmoptionNoArg, "True"},
  {"-asm",    "asmm",  XrmoptionNoArg, "True"},
  {"-trap",   "trap", XrmoptionNoArg, "True"},
  {"-notrap", "trap", XrmoptionNoArg, "False"},
  {"-quiet",  "quiet", XrmoptionNoArg, "True"},
  {"-noquiet","quiet", XrmoptionNoArg, "False"},
  {"-file",   "filename", XrmoptionSepArg, NULL},
  {"-execute","ex_filename", XrmoptionSepArg, NULL},
  {"-d2",     "display2", XrmoptionSepArg, NULL},
  {"-hexgpr", "hexGpr", XrmoptionNoArg, "True"},
  {"-nohexgpr", "hexGpr", XrmoptionNoArg, "False"},
  {"-hexfpr", "hexFpr", XrmoptionNoArg, "True"},
  {"-nohexfpr", "hexFpr", XrmoptionNoArg, "False"},
  {"-hexfgr", "hexFgr", XrmoptionNoArg, "True"},
  {"-nohexfgr", "hexFgr", XrmoptionNoArg, "False"},
  {"-stext", "stext", XrmoptionSepArg, NULL},
  {"-sdata", "sdata", XrmoptionSepArg, NULL},
  {"-ldata", "ldata", XrmoptionSepArg, NULL},
  {"-sstack", "sstack", XrmoptionSepArg, NULL},
  {"-lstack", "lstack", XrmoptionSepArg, NULL},
  {"-sktext", "sktext", XrmoptionSepArg, NULL},
  {"-skdata", "skdata", XrmoptionSepArg, NULL},
  {"-lkdata", "lkdata", XrmoptionSepArg, NULL}
};


#define TICK_WIDTH 10

#define TICK_HEIGHT 10

static char tick_bits[] = {
  0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x80, 0x01, 0xc1, 0x00, 0x63, 0x00,
  0x36, 0x00, 0x1c, 0x00, 0x08, 0x00, 0x00, 0x00};


#define POPQ(q, node)		\
{				\
 if (q != NULL)			\
   {				\
    node = q;			\
    q = q->next;		\
}}


/* Flags to control the way that registers are displayed. */

int print_gpr_hex;		/* Print GPRs in hex/decimal */
int print_fpr_hex;		/* Print FPRs in hex/floating point */
int print_fgr_hex;		/* Print FGRs in hex/floating point */


/* Exported variables: */

XtAppContext app_context;
Dimension button_width;
int load_trap_handler;
Pixmap mark;
Widget message, console, terminal;
jmp_buf spim_top_level_env;	/* For ^C */
XFontStruct *text_font;


/* Local variables: */

static Dimension app_width;
static Dimension button_height;
static Dimension command_height;
static Dimension command_hspace;
static Dimension command_vspace;
static int console_is_visible;
static Dimension display_height;
static char *ex_file_name = NULL;
static char *file_name = NULL;
static time_node *queue;
static Dimension reg_min_height;
static Dimension reg_max_height;
static Dimension segment_height;
static Widget shell1;
static Widget shell2;
static int terminal_is_visible;
static struct itimerval timerval;
static int spim_is_running = 0;
static Widget toplevel;


/* Imported variables: */

Widget register_window, text_window, data_window;
char *xspim_file_name;
mem_addr program_starting_address;
long initial_text_size, initial_data_size, initial_data_limit;
long initial_stack_size, initial_stack_limit, initial_k_text_size;
long initial_k_data_size, initial_k_data_limit;



static void
initialize (app_res)
     AppResources app_res;
{
  if (app_res.bare)
    bare_machine = 1;
  if (app_res.asmm)
    bare_machine = 0;
  if (app_res.trap)
    load_trap_handler = 0;
  else
    load_trap_handler = 1;
  if (app_res.quiet)
    quiet = 1;
  else
    quiet = 0;
  if (app_res.filename)
    file_name = app_res.filename;
  if (app_res.ex_filename)
    ex_file_name = app_res.ex_filename;

  if (app_res.textFont == NULL)
    app_res.textFont = XtNewString ("8x13");
  if (!(text_font = XLoadQueryFont (XtDisplay (toplevel), app_res.textFont)))
    fatal_error ("Cannot open font %s\n", app_res.textFont);

  mark = XCreateBitmapFromData (XtDisplay (toplevel),
				RootWindowOfScreen (XtScreen (toplevel)),
				tick_bits, TICK_WIDTH, TICK_HEIGHT);

  button_height = TEXTHEIGHT * 1.6;
  button_width = TEXTWIDTH * 12;
  app_width = 6 * (button_width + 16);
  if (app_width < TEXTWIDTH * 4 * 22) /* Register display width */
    app_width = TEXTWIDTH * 4 * 22;
  command_hspace = 8;
  command_vspace = 8;
  command_height = (button_height * 2) + (command_vspace * 3) + 2;
  reg_min_height = 19 * TEXTHEIGHT + 4;
  reg_max_height = reg_min_height + 8 * TEXTHEIGHT + 4;
  segment_height = 10 * TEXTHEIGHT + 4;
  display_height = 8 * TEXTHEIGHT + 4;
#ifdef IO
  timerval.it_interval.tv_usec = 0;
  timerval.it_value.tv_usec = 16000;
  signal (SIGALRM, GenInterrupt);
#endif
  print_gpr_hex = app_res.hex_gpr;
  print_fpr_hex = app_res.hex_fpr;
  print_fgr_hex = app_res.hex_fgr;
}


int
main (argc, argv)
     int argc;
     char **argv;
{
  Widget toplevel2, pane1, pane2;
  AppResources app_res;
  Display *display;
  Arg args[10];
  Cardinal n;

  toplevel = XtAppInitialize (&app_context, "Xspim", options,
			      XtNumber (options), &argc, argv,
			      fallback_resources, NULL, ZERO);

  if (argc != 1)
    syntax (app_context, argv[0]);

  XtGetApplicationResources (toplevel, (XtPointer) &app_res, resources,
			     XtNumber (resources), NULL, ZERO);

  if (app_res.display2 == NULL)
    display = XtDisplay (toplevel);
  else
    display = XtOpenDisplay (app_context, app_res.display2, "xspim",
			     "Xspim", NULL, ZERO, &argc, argv);

  toplevel2 = XtAppCreateShell ("xspim","Xspim",applicationShellWidgetClass,
				display, NULL, ZERO);

  XtAppAddActions (app_context, actionTable, XtNumber (actionTable));

  initialize (app_res);

  /* Console window */

  shell1 = XtCreatePopupShell ("Shell1", topLevelShellWidgetClass,
			       toplevel, NULL, ZERO);
  pane1 = XtCreateManagedWidget ("pane1", panedWidgetClass, shell1,
				 NULL, ZERO);
  n = 0;
  XtSetArg (args[n], XtNeditType, XawtextAppend); n++;
  XtSetArg (args[n], XtNscrollVertical, XawtextScrollWhenNeeded); n++;
  XtSetArg (args[n], XtNpreferredPaneSize, TEXTHEIGHT * 24); n++;
  XtSetArg (args[n], XtNwidth, TEXTWIDTH * 80); n++;
  console = XtCreateManagedWidget ("console", asciiTextWidgetClass, pane1,
				   args, n);
  console_out = (int) console;

  /* Terminal window */

  shell2 = XtCreatePopupShell ("Shell2", topLevelShellWidgetClass,
			       toplevel2, NULL, ZERO);
  pane2 = XtCreateManagedWidget ("pane2", panedWidgetClass, shell2,
				 NULL, ZERO);
  terminal = XtCreateManagedWidget ("terminal", asciiTextWidgetClass, pane2,
				    args, n);

  CreateSubWindows (toplevel, app_width, reg_min_height, reg_max_height,
		    command_height, command_hspace, command_vspace,
		    button_height, segment_height, display_height);

  XtRealizeWidget (toplevel);

  if (app_res.initial_text_size != NULL)
    initial_text_size = atoi (app_res.initial_text_size);
  if (app_res.initial_data_size != NULL)
    initial_data_size = atoi (app_res.initial_data_size);
  if (app_res.initial_data_limit != NULL)
    initial_data_limit = atoi (app_res.initial_data_limit);
  if (app_res.initial_stack_size != NULL)
    initial_stack_size = atoi (app_res.initial_stack_size);
  if (app_res.initial_stack_limit != NULL)
    initial_stack_limit = atoi (app_res.initial_stack_limit);
  if (app_res.initial_k_text_size != NULL)
    initial_k_text_size = atoi (app_res.initial_k_text_size);
  if (app_res.initial_k_data_size != NULL)
    initial_k_data_size = atoi (app_res.initial_k_data_size);
  if (app_res.initial_k_data_limit != NULL)
    initial_k_data_limit = atoi (app_res.initial_k_data_limit);
  initialize_world (load_trap_handler);

  if (file_name)
    {
      read_file (file_name, 1);
      xspim_file_name = str_copy (file_name);
    }
  else if (ex_file_name)
    {
      initialize_world (0);	/* Don't have a trap handler loaded. */
      read_file (ex_file_name, 0);
      xspim_file_name = str_copy (ex_file_name);
    }
  else
    {
      redisplay_text ();
      redisplay_data ();
    }

  XtAppMainLoop (app_context);
  return (0);
}


static void
syntax (app_context, call)
     XtAppContext app_context;
     char *call;
{
  XtDestroyApplicationContext (app_context);
  fprintf (stderr, "Usage:\n %s", call);
  fprintf (stderr, "\t[ -bare/-asm ] [ -trap/-notrap ] [ -quiet/noquiet ]\n");
  fprintf (stderr, "\t[ -d2 <display> ] [ -file/-execute <filename> ]\n");
  fprintf (stderr, "\t[ -s<seg> <size>] [ -l<seg> <size>]\n");
  exit (1);
}


static void
control_c_seen ()
{
  write_output (message_out, "\nExecution interrupted\n");
  redisplay_data ();
  ContinuePrompt (1);
  if (spim_is_running)
    longjmp (spim_top_level_env, 1);
}


void
PopConsole (w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  if (console_is_visible)
    {
      console_is_visible = 0;
      XtPopdown (shell1);
    }
  else
    {
      console_is_visible = 1;
      XtPopup (shell1, XtGrabNone);
    }
}


void
PopTerminal (w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  if (terminal_is_visible)
    {
      terminal_is_visible = 0;
      XtPopdown (shell2);
    }
  else
    {
      terminal_is_visible = 1;
      XtPopup (shell2, XtGrabNone);
    }
}


static void
addtoq (q, node)
     time_node *q;
     time_node *node;
{
  time_node *p, *c;
  int found = 0;

  printf ("in addtoq\n");
  p = c = q;
  if (c == NULL)
    {
      printf ("add..null\n");
      q = node;
    }
  else {
    while ((c != NULL) && (!found))
      if ((c->timeout.tv_sec < node->timeout.tv_sec) ||
	  (c->timeout.tv_usec < node->timeout.tv_usec))
	{
	  p = c;
	  c = c->next;
	}
      else
	found = 1;
    if (c == q)
      {
	node->next = c;
	q = node;
      }
    else
      {
	p->next = node;
	node->next = c;
      }
  }
}


static void
printqueue (q)
     time_node *q;
{
  time_node *tnode;

  tnode = q;
  while (tnode != NULL)
    {
      printf ("exception code %2d for time %ld %ld\n",
	      tnode->exception_code, tnode->timeout.tv_sec,
	      tnode->timeout.tv_usec);
      tnode = tnode->next;
    }
}


static void
GenInterrupt (sig, code, scp)
     int sig, code;
     struct sigcontext *scp;
{
  time_node *tnode, *newnode;
  struct timeval current, current2;
  long usec;

  tnode = NULL;
  POPQ (queue, tnode);
  if (tnode != NULL)
    {
      if ((Status_Reg & 0x1))
	{
	  Status_Reg = Status_Reg & 0xffffffc0 | (Status_Reg & 0xf) << 2;
	  Cause = (tnode->exception_code << 2);
	  exception_occurred = 1;
	  EPC = PC;
	}
      if (tnode->exception_code == CLOCK_EXCPT)
	{
	  newnode = (time_node *) malloc (sizeof (time_node));
	  newnode->exception_code = CLOCK_EXCPT;
	  newnode->next = NULL;
	  gettimeofday (&current2, 0);
	  current2.tv_usec = (current2.tv_usec + 16000) % 1000000;
	  if (current2.tv_usec < 16000)
	    current2.tv_sec = current2.tv_sec + 1;
	  newnode->timeout = current2;
	  if (queue == NULL)
	    queue = newnode;
	  else
	    addtoq (queue, newnode);
	}
      if (queue != NULL)
	{
	  gettimeofday (&current, 0);
	  if ((usec = queue->timeout.tv_usec - current.tv_usec) < 0)
	    usec = 1000000 + queue->timeout.tv_usec - current.tv_usec;
	  timerval.it_interval.tv_usec = 0;
	  timerval.it_value.tv_usec = usec;
	  setitimer (ITIMER_REAL, &timerval, 0);
	}

      free (tnode);
    }
}


void
read_file (name, assembly_file)
     char *name;
     int assembly_file;
{
  int error_flag;

  if (*name == '\0')
    error_flag = 1;
  else if (assembly_file)
    error_flag = read_assembly_file (name);
#ifdef mips
  else
    error_flag = read_aout_file (name);
#endif
  if (!error_flag)
    {
      redisplay_text ();
      redisplay_data ();
    }
}


void
start_program (addr)
     mem_addr addr;
{
#ifdef IO
  struct timeval timev;
  time_node *tnode;
#endif

  if (addr == 0)
      addr = starting_address ();
  if(addr == starting_address ())
      init_prog_fds();

  if (addr) {
#ifdef IO
    tnode = (time_node *) malloc (sizeof (time_node));
    tnode->exception_code = CLOCK_EXCPT;
    tnode->next = NULL;
    gettimeofday (&timev, 0);
    timev.ntv_usec = (timev.tv_usec + 16000) % 1000000;
    if (timev.tv_usec < 16000)
      timev.tv_sec = timev.tv_sec + 1;
    tnode->timeout = timev;
    queue = tnode;
    setitimer (ITIMER_REAL, &timerval, 0);
#endif
    execute_program (addr, DEFAULT_RUN_STEPS, 0, 0);
    queue = NULL;
  }
}


void
execute_program (pc, steps, display, cont_bkpt)
     mem_addr pc;
     int steps, display, cont_bkpt;
{
  if (!setjmp (spim_top_level_env))
    {
      spim_is_running = 1;
/*       showRunning (); */
      if (run_program (pc, steps, display, cont_bkpt))
	ContinuePrompt (0);
    }
  spim_is_running = 0;
  redisplay_data ();
  center_text_at_PC ();
}


static void
showRunning ()
{
  Arg args[1];

  XtSetArg (args[0], XtNstring, "Running.....");
  XtSetValues (register_window, args, ONE);
  XawTextUnsetSelection (text_window);
}


/* Redisplay the contents of the registers and, if modified, the data
   and stack segments. */

void
redisplay_data ()
{
  display_registers ();
  display_data_seg ();
}


/* Redisplay the contents of the registers in a wide variety of
   formats. */

static void
display_registers ()
{
  int i;
  String buf, bufp;
  char *grstr, *fpstr, *fgstr;
  char *grfill, *fpfill, *fgfill;
  Arg args [2];
  static char *reg_names[] = {"r0", "at", "v0", "v1", "a0", "a1", "a2", "a3",
			      "t0", "t1", "t2", "t3", "t4", "t5", "t6", "t7",
			      "s0", "s1", "s2", "s3", "s4", "s5", "s6", "s7",
			      "t8", "t9", "k0", "k1", "gp", "sp", "s8", "ra"};

  buf = (String) malloc (4 * K);
  *buf = '\0';
  bufp = buf;

  if (print_gpr_hex)
    grstr = "R%-2d (%2s) = %08x", grfill = "  ";
  else
    grstr = "R%-2d (%2s) = %-10d", grfill = " ";

  if (print_fpr_hex)
    fpstr = "FP%-2d=%08x%08x", fpfill = " ";
  else
    fpstr = "FP%-2d = %-10.4f", fpfill = "  ";

  if (print_fgr_hex)
    fgstr = "FG%-2d     = %08x", fgfill = "   ";
  else
    fgstr = "FG%-2d     = %-10.4f", fgfill = " ";

  sprintf (bufp, " PC     = %08x    ", PC); bufp += strlen (bufp);
  sprintf (bufp, " EPC    = %08x    ", EPC); bufp += strlen (bufp);
  sprintf (bufp, " Cause  = %08x    ", Cause); bufp += strlen (bufp);
  sprintf (bufp, " BadVAddr= %08x\n", BadVAddr); bufp += strlen (bufp);
  sprintf (bufp, " Status = %08x    ", Status_Reg); bufp += strlen (bufp);
  sprintf (bufp, " HI     = %08x    ", HI); bufp += strlen (bufp);
  sprintf (bufp, " LO     = %08x\n", LO); bufp += strlen (bufp);
  sprintf (bufp, "\t\t\t\t General Registers\n"); bufp += strlen (bufp);
  for (i = 0; i < 8; i++)
    {
      sprintf (bufp, grstr, i, reg_names[i], R[i]);
      bufp += strlen (bufp);
      sprintf (bufp, grfill); bufp += strlen (bufp);
      sprintf (bufp, grstr, i+8, reg_names[i+8], R[i+8]);
      bufp += strlen (bufp);
      sprintf (bufp, grfill); bufp += strlen (bufp);
      sprintf (bufp, grstr, i+16, reg_names[i+16], R[i+16]);
      bufp += strlen (bufp);
      sprintf (bufp, grfill); bufp += strlen (bufp);
      sprintf (bufp, grstr, i+24, reg_names[i+24], R[i+24]);
      bufp += strlen (bufp);
      sprintf (bufp, "\n");
      bufp += 1;
    }
  sprintf (bufp, "\t\t\t      Double Floating Point Registers\n");
  bufp += strlen (bufp);
  if (print_fpr_hex)
    for (i = 0; i < 4; i += 1)
      {
	sprintf (bufp, fpstr, 2*i, FGR[i], FGR[i+1]);
	bufp += strlen (bufp);
	sprintf (bufp, fpfill); bufp += strlen (bufp);
	sprintf (bufp, fpstr, 2*i+8, FGR[i+4], FGR[i+4+1]);
	bufp += strlen (bufp);
	sprintf (bufp, fpfill); bufp += strlen (bufp);
	sprintf (bufp, fpstr, 2*i+16, FGR[i+8], FGR[i+8+1]);
	bufp += strlen (bufp);
	sprintf (bufp, fpfill); bufp += strlen (bufp);
	sprintf (bufp, fpstr, 2*i+24, FGR[i+12], FGR[i+12+1]);
	bufp += strlen (bufp);
	sprintf (bufp, "\n");
	bufp += 1;
      }
  else for (i = 0; i < 4; i += 1)
    {
      sprintf (bufp, fpstr, 2*i, FPR[i]);
      bufp += strlen (bufp);
      sprintf (bufp, fpfill); bufp += strlen (bufp);
      sprintf (bufp, fpstr, 2*i+8, FPR[i+4]);
      bufp += strlen (bufp);
      sprintf (bufp, fpfill); bufp += strlen (bufp);
      sprintf (bufp, fpstr, 2*i+16, FPR[i+8]);
      bufp += strlen (bufp);
      sprintf (bufp, fpfill); bufp += strlen (bufp);
      sprintf (bufp, fpstr, 2*i+24, FPR[i+12]);
      bufp += strlen (bufp);
      sprintf (bufp, "\n");
      bufp += 1;
    }
  sprintf (bufp, "\t\t\t      Single Floating Point Registers\n");
  bufp += strlen (bufp);
  for (i = 0; i < 8; i += 1)
    {
      sprintf (bufp, fgstr, i, FGR[i]);
      bufp += strlen (bufp);
      sprintf (bufp, fgfill); bufp += strlen (bufp);
      sprintf (bufp, fgstr, i+8, FGR[i+8]);
      bufp += strlen (bufp);
      sprintf (bufp, fgfill); bufp += strlen (bufp);
      sprintf (bufp, fgstr, i+16, FGR[i+16]);
      bufp += strlen (bufp);
      sprintf (bufp, fgfill); bufp += strlen (bufp);
      sprintf (bufp, fgstr, i+24, FGR[i+24]);
      bufp += strlen (bufp);
      sprintf (bufp, "\n");
      bufp += 1;
    }

  XtSetArg (args[0], XtNstring, buf);
  XtSetArg (args[1], XtNlength, 4*K);
  XtSetValues (register_window, args, TWO);
}


/* Redisplay the text segment and ktext segments. */

void
redisplay_text ()
{
  char *buf;
  int limit, n;
  Arg args [2];

  if (!text_modified)
    return;
  buf = (char *) malloc (16*K);
  *buf = '\0';
  limit = 16*K;
  n = 0;
  buf = display_insts (TEXT_BOT, text_top, buf, &limit, &n);
  sprintf (&buf[n], "\n\tKERNEL\n");
  n += strlen (&buf[n]);
  buf = display_insts (K_TEXT_BOT, k_text_top, buf, &limit, &n);

  XtSetArg (args[0], XtNstring, buf);
  XtSetArg (args[1], XtNlength, limit);
  XtSetValues (text_window, args, TWO);
  free (buf);
  text_modified = 0;
}


/* Write a printable representation of the instructions in memory
   address FROM...TO to buffer BUF, which is of size LIMIT and whose next
   free location is N.  Return the, possible realloc'ed, buffer. */

static char *
display_insts (from, to, buf, limit, n)
     mem_addr from, to;
     char *buf;
     int *limit, *n;
{
  instruction *inst;
  mem_addr i;

  for (i = from; i < to; i += 4)
    {
      READ_MEM_INST (inst, i);
      if (inst != NULL)
	{
	  *n += print_inst_internal (&buf[*n], inst, i);
	  if ((*limit - *n) < 1*K)
	    {
	      *limit = 2 * *limit;
	      if ((buf = (char *) realloc (buf, *limit)) == 0)
		fatal_error ("realloc failed\n");
	    }
	}
    }
  return (buf);
}


/* Center the text window at the instruction at the current PC and
   highlight the instruction. */

void
center_text_at_PC ()
{
  char buf[100];
  XawTextBlock text;
  XawTextPosition start, finish;
  static mem_addr prev_PC = 0;

  if (PC < TEXT_BOT || (PC > text_top && (PC < K_TEXT_BOT || PC > k_text_top)))
    return;

  sprintf (buf, "\n[0x%08x]", PC);

  text.firstPos = 0;
  text.length = strlen (buf);
  text.ptr = buf;
  text.format = FMT8BIT;

  start = XawTextSearch (text_window, prev_PC <= PC ? XawsdRight : XawsdLeft,
			 &text);
  if (start == XawTextSearchError)
    return;

  XawTextSetInsertionPoint (text_window, start);

  sprintf (buf, "\n[0x%08x]", PC + BYTES_PER_WORD);
  finish = XawTextSearch (text_window, XawsdRight, &text);
  if (finish == XawTextSearchError)
    return;

  XawTextSetSelection (text_window, start, finish);
  /* This extra call is necessary to ensure that the selected line ends up
     in the viewable text region.  There appears to be a bug in X11R4 that
     causes the line to end up just below the viewable region. */
  XawTextSetInsertionPoint (text_window, finish + 1);
  XawTextSetInsertionPoint (text_window, start);
  prev_PC = PC;
}


/* Display the contents of the data and stack segments, if they have
   been modified. */

static void
display_data_seg ()
{
  char *buf;
  int limit, n;
  Arg args [2];

  if (!data_modified)
    return;
  buf = (char *) malloc (16*K);
  *buf = '\0';
  limit = 16*K;
  n = 0;

  sprintf (&buf[n], "\n\tDATA\n");
  n += strlen (&buf[n]);
  buf = display_values (DATA_BOT, data_top, buf, &limit, &n);
  sprintf (&buf[n], "\n\tSTACK\n");
  n += strlen (&buf[n]);
  buf = display_values (R[29] - BYTES_PER_WORD, STACK_TOP, buf, &limit, &n);
  sprintf (&buf[n], "\n\tKERNEL DATA\n");
  n += strlen (&buf[n]);
  buf = display_values (K_DATA_BOT, k_data_top, buf, &limit, &n);

  XtSetArg (args[0], XtNstring, buf);
  XtSetArg (args[1], XtNlength, limit);
  XtSetValues (data_window, args, TWO);
  free (buf);
  data_modified = 0;
}


/* Write a printable representation of the data in memory address
   FROM...TO to buffer BUF, which is of size LIMIT and whose next free
   location is N.  Return the, possible realloc'ed, buffer. */

static char *
display_values (from, to, buf, limit, n)
     mem_addr from, to;
     char *buf;
     int *limit, *n;
{
  mem_addr i, j;

  for (i = (from + BYTES_PER_WORD) & 0xfffffffc; i < to; i += BYTES_PER_WORD)
    {
      mem_word val;

      /* Look for a block of 4 or more zero memory words */
      for (j = 0; i + j < to; j += BYTES_PER_WORD)
	{
	  READ_MEM_WORD (val, i + j);
	  if (val != 0)
	    break;
	}
      if (j >= 4 * BYTES_PER_WORD)
	{
	  sprintf (&buf[*n], "[0x%08x]...[0x%08x]	0x00000000\n",
		   i, i +j);
	  *n += strlen (&buf[*n]);
	  if ((*limit - *n) < 1*K)
	    {
	      *limit = 2 * *limit;
	      if ((buf = (char *) realloc (buf, *limit)) == 0)
		fatal_error ("realloc failed\n");
	    }
	  i = i + j;
	  continue;
	}

      /* Otherwise, print the next four words on a single line */
      sprintf (&buf[*n], "[0x%08x]	0x%08x", i, val);
      *n += strlen (&buf[*n]);
      for ( ; i % 16 != 12; )
	{
	  i += BYTES_PER_WORD;
	  READ_MEM_WORD (val, i);
	  sprintf (&buf[*n], "  0x%08x", val);
	  *n += strlen (&buf[*n]);
	}
      sprintf (&buf[*n], "\n");
      *n += 1;
      if ((*limit - *n) < 1*K)
	{
	  *limit = 2 * *limit;
	  if ((buf = (char *) realloc (buf, *limit)) == 0)
	    fatal_error ("realloc failed\n");
	}
    }
  return (buf);
}


/* Simulate the semantics of fgets, not gets, on an x-window. */

void
read_input (str, str_size)
     char *str;
     int str_size;
{
  char buffer[10];
  KeySym key;
  XComposeStatus compose;
  int charcount;
  XEvent event;
  char *ptr;

  ptr = str;
  str_size -= 1;		/* Reserve space for null */

  if (!console_is_visible)
    {
      XtPopup (shell1, XtGrabNone);
      console_is_visible = 1;
    }

  while (1)
    {
      XtAppNextEvent (app_context, &event);
      if (event.type == KeyPress)
	{
	  charcount = XLookupString (&event.xkey, buffer, 10, &key, &compose);
	  if ((key == XK_Return) || (key == XK_KP_Enter))
	    {
	      *ptr++ = '\n';
	      *ptr = '\0';
	      writeText (console, "\n");
	      return;
	    }
	  else if (*buffer == 3) /* ^C */
	    XtDispatchEvent (&event);
	  else
	    {
	      int n = charcount < str_size - 1 ? charcount : str_size - 1;
	      char *old_ptr = ptr;

	      strncpy (ptr, buffer, n);
	      ptr += n;
	      *ptr = '\0';
	      str_size -= n;
	      writeText (console, old_ptr);
	    }
	}
      if (str_size <= 1)
	{
	  *ptr = '\0';
	  return;
	}
    }
}


void
initialize_console ()
{
  XawTextEnableRedisplay (console);
  writeText (console, "\n");
}



/* Print an error message. */

/*VARARGS0*/
void
error (va_alist)		/* Display only */
va_dcl
{
  va_list args;
  char *fmt;
  char io_buffer [IO_BUFFSIZE];

  va_start (args);
  fmt = va_arg (args, char *);
  vsprintf (io_buffer, fmt, args);
  va_end (args);
  if (message != 0)
    writeText (message, io_buffer);
  else
    fprintf (stderr, "%s", io_buffer);
}


/*VARARGS0*/
int
run_error (va_alist)		/* Display and return to top level */
va_dcl
{
  va_list args;
  char *fmt;
  char io_buffer [IO_BUFFSIZE];

  va_start (args);
  fmt = va_arg (args, char *);
  vsprintf (io_buffer, fmt, args);
  va_end (args);
  if (message != 0)
    writeText (message, io_buffer);
  else
    fprintf (stderr, "%s", io_buffer);
  if (spim_is_running)
    longjmp (spim_top_level_env, 1);
  return (0);			/* So it can be used in expressions */
}


/*VARARGS0*/
void
write_output (va_alist)
va_dcl
{
  va_list args;
  Widget w;
  char *fmt;
  char io_buffer [IO_BUFFSIZE];

  va_start (args);
  w = va_arg (args, Widget);
  fmt = va_arg (args, char *);

  if (w == console && !console_is_visible)
    {
      XtPopup (shell1, XtGrabNone);
      console_is_visible = 1;
    }
  else if (w == terminal && !terminal_is_visible)
    {
      XtPopup (shell2, XtGrabNone);
      terminal_is_visible = 1;
    }

  vsprintf (io_buffer, fmt, args);
  va_end (args);

  writeText (w, io_buffer);

  /* Look for keyboard input (such as ^C) */
  while (XtAppPending (app_context))
    {
      XEvent event;

      XtAppNextEvent (app_context, &event);
      XtDispatchEvent (&event);
    }
}


static void
writeText (w, s)
     Widget w;
     char *s;
{
  TextWidget ctx = (TextWidget) w;
  XawTextBlock textblock;
  XawTextPosition ip = XawTextGetInsertionPoint (w);

  if (!s || strlen (s) == 0) return;

  textblock.firstPos = 0;
  textblock.length = strlen (s);
  textblock.ptr = s;
  textblock.format = FMT8BIT;

  XawTextReplace (w, ip, ip, &textblock);
  XawTextSetInsertionPoint (w, XawTextGetInsertionPoint (w) + textblock.length);
}
