/*  -*-c++-*-     tell emacs to use c++ mode
 *  --------------------------------------------------------------------
 *        $Id$
 *
 *   $RCSfile$
 *    $Source$
 *    $Author$
 *  $Revision$
 *      $Date$
 *      $Name$
 *    $Locker$
 *     $State$
 *
 *  --------------------------------------------------------------------
 *       $Log$
 *
 */
/*  -------------------------------------------------------------------- */

/*  --------------------------------------------------------------------
 *    IPv6CP.H
 *
 *    PPP IP v6 control protocol header
 *
 *    Reference:  RFC 1548  --  The Point-to-Point Protocol (PPP)
 *                RFC 1570  --  PPP LCP Extensions
 *
 *                draft-ietf-ipngwg-pppext-ipv6cp-03.txt, 
 *                Haskin + Allen, Bay Networks, Inc., May 1996
 *
 *
 *    Thomas Nagel <T.Nagel@ping.de>
 *    06-Oct-1996 KTN
 *  --------------------------------------------------------------------
 */

/* 
 * REMARKS:
 *
 * 1. Random Interface-Token
 *
 * The external function magic() is - sort of - misused to generate 
 * a random interface token.
 *
 * This has the side effect, that the usual caller of magic(), the LCP
 * protocol, will not get the usual sequence of random magic numbers.
 *
 * I hope this does not do any harm, because - as the name says - these 
 * numbers should be random; so 'stealing' a number in the middle
 * should not be considered dangerous, because this should not reduce
 * their randomness, but instead (perhaps) even increase it.
 *
 * 2. Function inet_ntop()
 * To use this function set the ifdef HAVE_INET_NTOP
 *
 */
 
/*
 * TODO:
 *
 * - Interface Tokens:
 *   + Check wether the local interface token should be configurable
 *   + Implement way to tell the upper layer the interface token
 *   + Check the scope of uniqueness of interface tokens: 
 *     Do they need to be globally unique?
 *     or is it sufficient, if the local and remote tokens don't conflict?
 *     See ifdef TODO_CHECK_ADDR.
 *
 * - IPv6 Header Compression:
 *   + add IP6 Header Compression Protocol, see ifdef's USE_IPV6_COMPR
 *
 * - IPv6 default routes:
 *   + check if, and how to, add default routes through PPP for IPv6
 *
 * - Reaction on Configure-Requests received:
 *   + send NAK/REJ on options needed, but not received.
 */

#ifndef lint
static char rcsid[] = "$Id$";
#endif

#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "pppd.h"
#include "fsm.h"
#include "ipv6cp.h"
#include "pathnames.h"

/* the following macro provides a way to have local symbols static,
 * when compiling the final version, or non-static, to help debugging.
 */
#ifdef  DEBUG
#define STATIC
#else
#define STATIC static
#endif

#define PPP_IPV6CP      0x8057  /* IPv6 Control Protocol */

/* ------------------------------------------------------------------------
 * external vars
 */

extern int ipv6_enabled;

/* ------------------------------------------------------------------------
 * global vars
 */

ipv6cp_options ipv6cp_wantoptions  [NUM_PPP];   /* what we want to request */
ipv6cp_options ipv6cp_gotoptions   [NUM_PPP];   /* what the peer ack'd */
ipv6cp_options ipv6cp_allowoptions [NUM_PPP];   /* what we allow peer to req */
ipv6cp_options ipv6cp_hisoptions   [NUM_PPP];   /* what we ack'd */


fsm	ipv6cp_fsm [NUM_PPP];	/* IPV6CP fsm structure */

/* ------------------------------------------------------------------------
 * local vars
 */

/* ------------------------------------------------------------------------
 * prototypes of external functions
 */

extern u_long   magic           __P((void));

#ifdef TODO_CHECK_ADDR
extern int      auth_ipv6_addr  __P((int unit, struct in6_addr * addr));
#endif

extern int      sif6up          __P((int unit));
extern int      sif6down        __P((int unit));
extern int      sif6addr        __P((int unit, 
				     struct in6_addr * local, 
				     struct in6_addr * peer));
extern int      cif6addr        __P((int unit, 
				     struct in6_addr * local, 
				     struct in6_addr * peer));

#ifdef USE_IPV6_DEFAULTREOUTES
extern int      sif6defaultroute __P((int unit, 
				      struct in6_addr * remote_gateway));
extern int      cif6defaultroute __P((int unit, struct 
				      struct in6_addr * remote_gateway));
#endif

#ifdef HAVE_INET_NTOP
extern char *	inet_ntop	__P((int af, 
				     struct in6_addr * addr, 
				     size_t addrsize, 
				     char*  buffer));
#endif

/* ------------------------------------------------------------------------
 * prototypes of local functions. (CI = Configuration Information)
 */

STATIC void   
ipv6cp_resetci	(fsm *);		/* Reset our Options */

STATIC int    
ipv6cp_cilen	(fsm *);		/* determine length of our Options */

STATIC void   
ipv6cp_addci	(fsm *, u_char *);	/* Add our Options */

STATIC int    
ipv6cp_ackci	(fsm *, u_char *, int); /* Peer ack'ed some Options */

STATIC int
ipv6cp_nakci    (fsm *, u_char *, int); /* Peer nak'ed some Options */

STATIC int
ipv6cp_rejci	(fsm *, u_char *, int);	/* Peer rejected some Options */

STATIC int
ipv6cp_reqci	(fsm *, u_char *, int *,int);/* Peer sent a configure request*/

STATIC void   
ipv6cp_up	(fsm *);		/* We're UP */

STATIC void   
ipv6cp_down	(fsm *);                /* We're DOWN */

STATIC void   
ipv6cp_script	(fsm *, char *);        /* Run an up/down script */

/* ------------------------------------------------------------------------
 * fsm callback structure
 */

STATIC
fsm_callbacks ipv6cp_callbacks =  /* IPCP callback routines */
{                               
    ipv6cp_resetci,             /* Reset our Configuration Information */
    ipv6cp_cilen,               /* Length of our Configuration Information */
    ipv6cp_addci,               /* Add our Configuration Information */
    ipv6cp_ackci,               /* ACK our Configuration Information */
    ipv6cp_nakci,               /* NAK our Configuration Information */
    ipv6cp_rejci,               /* Reject our Configuration Information */
    ipv6cp_reqci,               /* Request peer's Configuration Information */
    ipv6cp_up,                  /* Called when fsm reaches OPENED state */
    ipv6cp_down,                /* Called when fsm leaves OPENED state */
    NULL,                       /* Called when we want the lower layer up */
    NULL,                       /* Called when we want the lower layer down */
    NULL,                       /* Called when Protocol-Reject received */
    NULL,                       /* Retransmission is necessary */
    NULL,                       /* Called to handle protocol-specific codes */
    "IPV6CP"                    /* String name of protocol */
};

/* ------------------------------------------------------------------------
 * ipv6_build_addr
 *
 * local functiont to construct an IPv6 address from an interface token
 *
 *
 * Excerpt from the internet-draft:
 * 
 * Link-local addresses of PPP interfaces have the following format:
 *
 * | 10 bits  |              86 bits               |     32 bits     |
 * +----------+--------------+---------------------+-----------------+
 * |1111111010|              0                     | Interface Token |
 * +----------+--------------+---------------------+-----------------+
 *
 * The most significant 10 bits of the address is the Link-Local prefix
 * FE80::.  86 zero bits pad out the address between the Link-Local
 * prefix and the Interface Token fields.
 *
 *
 * call example:  ipv6_build_addr( go->Token, (u_char *) &go->ipv6addr );
 * or:            ipv6_build_addr( ho->Token, (u_char *) &ho->ipv6addr );
 *
 */


STATIC void
ipv6_build_addr         (u_long token, u_char * addr)
{
  int n;

  /* the pointer must be valid */
  if( ! addr )
    return;

  /* the interface token must not be zero */
  if( ! token )
    return;

  /* prepend the Link-Local prefix FE80::/96 */

  addr[0] = 0xFE;
  addr[1] = 0x80;

  addr[2] = addr[3] = addr[4] = addr[5] = addr[6] = addr[7] =
  addr[8] = addr[9] = addr[10] = addr[11] = 0;

  /* append the interface token */

  addr[12] = (u_char) ( ntohl( token ) >> 24 ) & 0xFF;
  addr[13] = (u_char) ( ntohl( token ) >> 16 ) & 0xFF;
  addr[14] = (u_char) ( ntohl( token ) >>  8 ) & 0xFF;
  addr[15] = (u_char) ( ntohl( token )       ) & 0xFF;
 
}

/* ------------------------------------------------------------------------
 * ipv6cp_init - Initialize IPV6CP.
 */

void  
ipv6cp_init         (int unit)
{
  fsm *f = &ipv6cp_fsm[unit];

  ipv6cp_options *wo = &ipv6cp_wantoptions	[unit];
  ipv6cp_options *ao = &ipv6cp_allowoptions	[unit];

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_init: unit %d", unit ));

  /* initialize the fsm structure */
  f->unit               = unit;
  f->protocol           = PPP_IPV6CP;
  f->callbacks          = &ipv6cp_callbacks;

  /* initialize the want-options and allow-options */
  memset( wo, 0, sizeof( ipv6cp_options ));
  memset( ao, 0, sizeof( ipv6cp_options ));

  wo->fToken  = ao->fToken = TRUE;      /* always allow token negotiation */
  wo->Token   = magic();		/* take a random number */
  
#ifdef USE_IPV6_COMPR
  wo->fComprProt = TRUE;    /* request IPv6 Compression Protocol */
  ao->fComprProt = TRUE;    /* allow   IPv6 Compression Protocol */

  if( wo->fComprProt )
    /* IPv6 Header Compression */
    wo->CompressionProtocol = CI_IPv6_HDRCOMPRESSION; 
  if( ao->fComprProt )
    /* IPv6 Header Compression */
    ao->CompressionProtocol = CI_IPv6_HDRCOMPRESSION; 
#endif

#ifdef USE_IPV6_DEFAULTREOUTES
  /*
   * XXX This controls whether the user may use the defaultroute option.
   */
  ao->default_route = 1;
#endif

  /* call the prototype fsm initialization code */
  fsm_open( &ipv6cp_fsm[unit] );
}


/* ------------------------------------------------------------------------
 * ipcp_open - IPV6CP is allowed to come up.
 */

void
ipv6cp_open	(int unit)
{
  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_open: unit %d", unit ));

  /* call the prototype fsm open code */
  fsm_open( &ipv6cp_fsm[unit] );
}

/* ------------------------------------------------------------------------
 * ipv6cp_close - Close IPV6CP.
 */

void  
ipv6cp_close        (int unit)
{
  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_close: unit %d", unit ));

  /* call the prototype fsm closure code */
  fsm_close( &ipv6cp_fsm[unit] );
}


/* ------------------------------------------------------------------------
 * ipv6cp_lowerup - The lower layer is up.
 */

void  
ipv6cp_lowerup      (int unit)
{
  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_lowerup: unit %d", unit ));

  /* call the prototype fsm code */
  fsm_lowerup( &ipv6cp_fsm[unit], ipv6_enabled);
}


/* ------------------------------------------------------------------------
 * ipv6cp_lowerdown - The lower layer is down.
 */

void  
ipv6cp_lowerdown    (int unit)
{
  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_lowerdown: unit %d", unit ));

  /* call the prototype fsm code */
  fsm_lowerdown( &ipv6cp_fsm[unit] );
}

/* ------------------------------------------------------------------------
 * ipv6cp_up - IPV6CP has come UP.
 *
 * Configure the IP network interface appropriately and bring it up.
 */

STATIC void  
ipv6cp_up           (fsm *f)
{
  ipv6cp_options *ho = &ipv6cp_hisoptions	[f->unit];
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];
  ipv6cp_options *wo = &ipv6cp_wantoptions	[f->unit];
  int unit = f->unit, pid;

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_up: unit %d", unit ));

  /*
   * We must have an interface token for both ends of the link.
   */
  if( !go->fToken || !go->Token )
  {
    syslog(LOG_ERR, "ipv6: No local interface token for unit %d", unit );
    ipv6cp_close( f->unit );
    return;
  }
  if( !ho->fToken || !ho->Token )
  {
    syslog(LOG_ERR, "ipv6: No remote interface token for unit %d", unit );
    ipv6cp_close( f->unit );
    return;
  }        

  /*
   * Convert the interface tokens to ip v6 addresses
   */

  ipv6_build_addr( go->Token, (u_char *) &go->ipv6addr );
  ipv6_build_addr( ho->Token, (u_char *) &ho->ipv6addr );


#ifdef TODO_CHECK_ADDR
  /*
   * Check that the peer is allowed to use the ipv6 address he wants.
   */
  if( ! auth_ipv6_addr( unit, &ho->ipv6addr ) )
  {
    static char buf[81];
    syslog(LOG_ERR, "Peer is not authorized to use ip v6 address %s",
           inet_ntop( AF_INET6, &ho->ipv6addr, sizeof(struct in6_addr), buf) );
           
    ipv6cp_close( unit );
    return;
  }
#endif
   
  /* set IPv6 address and mask of connection */
  if( ! sif6addr( unit, &go->ipv6addr, &ho->ipv6addr ) )
  {
    IPV6CPDEBUG((LOG_WARNING, "sifaddr failed, unit %d", unit ));
    ipcp_close(f->unit);
    return;
  }
   
   
#ifdef USE_IPV6_COMPR
  /*
   * start IPv6 Header Compression
   */
  if( go->fComprProt )
  {
  }
#endif

  /* start upper level: bring the interface up for IP v6 */
  if( ! sif6up( unit ) )
  {
    IPV6CPDEBUG((LOG_WARNING, "sifup failed, unit %d", unit ));
    ipv6cp_close(unit);
    return;
  }


#ifdef USE_IPV6_DEFAULTREOUTES
  /* assign a default route through the interface if required */
  go->default_route = 0;
  if( wo->default_route )
  {
    if( sif6defaultroute( unit, &ho->ipv6addr ) )
    {
      go->default_route = 1;
    }
  }
#endif

  /* Remark: in contrast to IPv4, proxy ARP is not possible in IPv6,
   * because the generated IPv6 address is a net-local address.
   */

  /*
   * Execute the ipv6-up script, like this:
   * /etc/ppp/ipv6-up interface tty speed 
   *   local-IPv6-address remote-IPv6-address
   */
   
  ipv6cp_script( f, _PATH_IPV6UP );

}

/* ------------------------------------------------------------------------
 * ipv6cp_down - IPV6CP has gone DOWN.
 *
 * Alert other protocols.
 */

STATIC void  
ipv6cp_down         (fsm *f)
{
  ipv6cp_options *ho = &ipv6cp_hisoptions	[f->unit];
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];
  int unit = f->unit;

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_down: unit %d", unit ));

#ifdef USE_IPV6_COMPR
  /* stop IPv6 Header Compression */
  if( go->fComprProt )
  {
  }
#endif

  /* Remark: in contrast to IPv4, proxy ARP is not possible in IPv6,
   * because the generated IPv6 address is a net-local address.
   */

#ifdef USE_IPV6_DEFAULTREOUTES
  /* clear default route through the interface, if added */
  if( go->default_route )
  {
    cif6defaultroute( unit, &ho->ipv6addr );
  }
#endif
      
  /* stop upper level: bring the interface down for IP v6 */
  sif6down( unit );

  /* clear IPv6 addresses of connection */
  cif6addr( unit, &go->ipv6addr, &ho->ipv6addr );

  /* Execute the ipv6-down script */
  ipv6cp_script(f, _PATH_IPV6DOWN);
}


/* ------------------------------------------------------------------------
 * ipcp_script - Execute a script with arguments
 * interface-name tty-name speed local-IPv6-address remote-IPv6-address.
 */
 
STATIC void
ipv6cp_script     (fsm * f, char * script)
{
  char strspeed[32], strlocal[41], strremote[41];
  char *argv[8];

  sprintf(strspeed, "%d", baud_rate);

  strlocal[0] = strremote[0] = '\0';
#ifdef HAVE_INET_NTOP
  {
    char buf[81];
    strcpy(strlocal,
	   inet_ntop( AF_INET6, &ipv6cp_gotoptions[f->unit].ipv6addr,
		      sizeof(struct in6_addr), buf) );
    strcpy(strremote,
	   inet_ntop( AF_INET6, &ipv6cp_hisoptions[f->unit].ipv6addr,
		      sizeof(struct in6_addr), buf) );
  }
#else
  {
  }
#endif

  argv[0] = script;
  argv[1] = ifname;
  argv[2] = devnam;
  argv[3] = strspeed;
  argv[4] = strlocal;
  argv[5] = strremote;
  argv[6] = ipparam;
  argv[7] = NULL;
  
  run_program(script, argv, 0);
}

/* ------------------------------------------------------------------------
 * ipv6cp_input - Input IPV6CP packet.
 */

void  
ipv6cp_input        (int unit, u_char *p, int len)
{
  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_input: unit %d, %d byte", unit, len ));

  /* call the prototype fsm input code */
  fsm_input( &ipv6cp_fsm[unit], p, len);
}


/* ------------------------------------------------------------------------
 * ipv6cp_protrej - A Protocol-Reject was received for IPV6CP.
 *
 * close IPV6CP, then pretend the lower layer went down, so we shut up.
 */

void  
ipv6cp_protrej      (int unit)
{
  IPV6CPDEBUG(( LOG_ERR, "ipv6cp_protrej: unit %d!", unit ));
  /* pretend the lower layer is down */
  ipv6cp_close( unit );
  fsm_lowerdown( &ipv6cp_fsm[unit] );
}


/* ------------------------------------------------------------------------
 * ipv6cp_resetci - Reset our CI.
 */

STATIC void  
ipv6cp_resetci      (fsm* f)
{
  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_resetci: unit %d", f->unit ));

  /* copy out wantoptions to our gotoptions */
  memcpy( &ipv6cp_gotoptions[f->unit],
          &ipv6cp_wantoptions[f->unit],
          sizeof( ipv6cp_options ) );
}

/* ------------------------------------------------------------------------
 * ipv6cp_cilen - Return length of our CI.
 */

STATIC int   
ipv6cp_cilen        (fsm *f)
{
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];
  int len;

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_cilen: unit %d", f->unit ));

  len = 0;
  if( go->fToken ) len += 6;

#ifdef USE_IPV6_COMPR
  if( go->fComprProt )
    len += xxx;
#endif

  return len;
}


/* ------------------------------------------------------------------------
 * ipv6cp_addci - Add our wanted options to a packet.
 *
 * ATTENTION: 
 * the data returned here must have exactly the same size that 
 * ipv6cp_cilen() returns. Otherwise data will be either truncated,
 * or undefined.
 */

STATIC void  
ipv6cp_addci        (fsm *f, u_char *ucp)
{
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_addci: unit %d", f->unit ));

  if( go->fToken )
  {
    u_char  b, b1,b2,b3,b4;
    u_long d;
    
    *ucp++ = CI_INTERFACE_TOKEN;
    b = (u_char) (sizeof(long) +2);
    *ucp++ = b;
    d = htonl( go->Token );
    *ucp++ = (u_char) ( (d >> 24) & 0xff );
    *ucp++ = (u_char) ( (d >> 16) & 0xff );
    *ucp++ = (u_char) ( (d >>  8) & 0xff );
    *ucp++ = (u_char) (  d        & 0xff );

    b1 = (u_char) ((go->Token >> 24) & 0xFF);
    b2 = (u_char) ((go->Token >> 16) & 0xFF);
    b3 = (u_char) ((go->Token >>  8) & 0xFF);
    b4 = (u_char) ((go->Token      ) & 0xFF);
    IPV6CPDEBUG(( LOG_INFO, 
		  "ipv6cp_addci: unit %d: own Token %02x-%02x-%02x-%02x", 
                  f->unit, b1,b2,b3,b4 ));
  }

#ifdef USE_IPV6_COMPR
  if( go->fComprProt )
  {
    *ucp++ = CI_COMPRESSION;
    b = (u_char) 4;
    *ucp++ = b;
    *ucp++ = (u_char) ((go->CompressionProtocol >> 8) & 0xFF);
    *ucp++ = (u_char) ( go->CompressionProtocol       & 0xFF);
  }
#endif

}


/* ------------------------------------------------------------------------
 * ipv6cp_ackci - The peer sent an ACK on out Configure-Request.
 *
 * Returns:
 *      0 - Ack was bad.
 *      1 - Ack was good.
 */

STATIC int   
ipv6cp_ackci        (fsm *f, u_char *p, int len)
{
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];

  u_short olen, otype;
  u_long  dTmp;
  u_char  b, b1,b2,b3,b4;

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_ackci: unit %d", f->unit ));

  while( len > 0 )
  {
    if( len < 2 ) goto bad;

    otype = *p++;
    olen  = *p++;
    len -= 2;

    switch( otype )
    {
    case CI_INTERFACE_TOKEN:
      if( olen != (2+sizeof(long)) )
      {
        goto bad;
      }
      else
      {
        b = *p++; dTmp  = ((u_long) b) << 24;
        b = *p++; dTmp |= ((u_long) b) << 16;
        b = *p++; dTmp |= ((u_long) b) <<  8;
        b = *p++; dTmp |= ((u_long) b);
        len -= 4;

        dTmp = ntohl( dTmp );

        if( go->Token == 0 )
        {
          /* peer assigned us an interface token
           * this should not happen in an ACK !!!
           * (we accept it anyway)
	   */

          go->Token = dTmp;

          b1 = (u_char) ((go->Token >> 24) & 0xFF);
          b2 = (u_char) ((go->Token >> 16) & 0xFF);
          b3 = (u_char) ((go->Token >>  8) & 0xFF);
          b4 = (u_char) ((go->Token      ) & 0xFF);
          IPV6CPDEBUG(( LOG_WARNING,
                        "ipv6cp_ackci: unit %d: "
			"assigned Token %02x-%02x-%02x-%02x", 
                        f->unit, b1,b2,b3,b4 ));
        }
        else
        if( go->Token != dTmp )
        {
          /* peer acknowledged a different address than we told him
           * this should not happen in an ACK !!!
	   */
          goto bad;
        }
      }
      break;

#ifdef USE_IPV6_COMPR
    case CI_COMPRESSION:
      goto bad;
      break;
#endif    

    default:
      goto bad;
      break;
    }
  }

  /*
   * If there are any remaining CIs, then this packet is bad.
   */
  if (len != 0)
    goto bad;

  IPV6CPDEBUG((LOG_INFO, "ipv6cp_ackci: received Ack"));
  return (1);

bad:
  IPV6CPDEBUG((LOG_WARNING, "ipv6cp_ackci: received bad Ack!"));
  return (0);
}

/* ------------------------------------------------------------------------
 * ipv6cp_nakci - NAK some of our CIs.
 *
 * Returns:
 *      0 - Nak was bad.
 *      1 - Nak was good.
 */

STATIC int
ipv6cp_nakci        (fsm *f, u_char *p, int len)
{
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];

  u_short  olen, otype;
  u_long dTmp;
  u_char  b, b1,b2,b3,b4;

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_nakci: unit %d", f->unit ));

  while( len > 0 )
  {
    if( len < 2 ) goto bad;

    otype = *p++;
    olen  = *p++;
    len -= 2;

    switch( otype )
    {
    case CI_INTERFACE_TOKEN:
      if( olen != (2+sizeof(long)) )
      {
        goto bad;
      }
      else
      {
        b = *p++; dTmp  = ((u_long) b) << 24;
        b = *p++; dTmp |= ((u_long) b) << 16;
        b = *p++; dTmp |= ((u_long) b) <<  8;
        b = *p++; dTmp |= ((u_long) b);
        len -= 4;

        dTmp = ntohl( dTmp );
        if( go->Token == 0 )
        {
          /* peer wants to assign us an address */
          go->Token = dTmp;

          b1 = (u_char) ((go->Token >> 24) & 0xFF);
          b2 = (u_char) ((go->Token >> 16) & 0xFF);
          b3 = (u_char) ((go->Token >>  8) & 0xFF);
          b4 = (u_char) ((go->Token      ) & 0xFF);
          IPV6CPDEBUG(( LOG_INFO, 
			"ipv6cp_nakci: unit %d: "
			"assigned Token %02x-%02x-%02x-%02x", 
			f->unit, b1,b2,b3,b4 ));
        }
        else
        if( go->Token != dTmp )
        {
          /* peer wants to assign us a different address 
	   * than we want to use
	   */
          goto bad;
        }
      }
      break;

#ifdef USE_IPV6_COMPR
    case CI_COMPRESSION:
      goto bad;
      break;
#endif    

    default:
      goto bad;
      break;
    }
  }

  /*
   * If there are any remaining CIs, then this packet is bad.
   */
  if (len == 0)
    return 1;
bad:
  IPV6CPDEBUG((LOG_INFO, "ipv6cp_nakci: received bad Nak!"));
  return 0;
}

/* ------------------------------------------------------------------------
 * ipv6cp_rejci - Reject some of our CIs.
 */

STATIC int
ipv6cp_rejci        (fsm *f, u_char *p, int len)
{
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];
  u_short  olen, otype;

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_rejci: unit %d", f->unit ));

  while( len > 0 )
  {
    if( len < 2 ) goto bad;

    otype = *p++;
    olen  = *p++;
    len -= 2;

    switch( otype )
    {
    case CI_INTERFACE_TOKEN:
      /* interface token negotiation failed. */
      IPV6CPDEBUG(( LOG_WARNING, "ipv6cp_rejci: Token rejected" ));

      go->fToken = FALSE;
      
      p += (olen-2);
      len -= (olen -2);
      break;

#ifdef USE_IPV6_COMPR
    case CI_COMPRESSION:
      p += (olen-2);
      len -= (olen-2);
      break;
#endif
    
    default:
      IPV6CPDEBUG(( LOG_WARNING, 
		    "ipv6cp_rejci: unknown option %d rejected", otype ));
      p += (olen-2);
      len -= (olen-2);
      break;
    }
  }

  /*
   * If there are any remaining CIs, then this packet is bad.
   */
  if (len == 0)
    return 1;
bad:
  IPV6CPDEBUG((LOG_INFO, "ipv6cp_rejci: received bad Rej!"));
  return 0;
}

/* ------------------------------------------------------------------------
 * ipv6cp_reqci - Check the peer's requested CIs and send appropriate response.
 *
 * Returns: CONFACK, CONFNAK or CONFREJ and input packet modified
 * appropriately.
 *
 * If reject_if_disagree is non-zero, doesn't return
 * CONFNAK; returns CONFREJ if it can't return CONFACK.
 */

STATIC int
ipv6cp_reqci	(fsm *f, u_char *inp, int *len, int reject_if_disagree)
{
  ipv6cp_options *ho = &ipv6cp_hisoptions	[f->unit];
  ipv6cp_options *go = &ipv6cp_gotoptions	[f->unit];
  ipv6cp_options *ao = &ipv6cp_allowoptions	[f->unit];

  u_char *cip;			/* Pointer to Current CI */
  u_short cilen, citype;	/* Parsed len, type */
  int rc = CONFACK;		/* Final packet return code */
  int orc;			/* Individual option return code */
  u_char *p = inp;              /* Pointer to next char to parse */
  u_char *ucp = inp;            /* Pointer to current output char */
  int l = *len;			/* Length left */
  u_long dTmp, citoken;         /* Parsed token values */
  u_char  b1, b2, b3, b4;	/* bytes in a token */

  IPV6CPDEBUG(( LOG_INFO, "ipv6cp_reqci: unit %d", f->unit ));

  /*
   * Reset all his options.
   */
  ho->fToken      = 0;
  ho->Token       = 0;
  ho->fComprProt  = 0;
#ifdef USE_IPV6_COMPR
#endif
    
  /*
   * Process all his options.
   */
  while (l)
  {
    orc = CONFACK;			/* Assume success */
    cip = p;                            /* Remember begining of CI */

    if (l < 2)                          /* Not enough data for CI header */
    {
      IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: bad CI length: L < 2"));
      orc = CONFREJ;			/* Reject bad CI */
      cilen = l;                        /* Reject till end of packet */
      l = 0;                            /* Don't loop again */
      goto endswitch;
    }

    if(p[1] < 2)                        /*  CI length too small or */
    {
      IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: bad CI length: cilen < 2"));
      orc = CONFREJ;			/* Reject bad CI */
      cilen = l;                        /* Reject till end of packet */
      l = 0;                            /* Don't loop again */
      goto endswitch;
    }

    if(p[1] > l)                       /*  CI length too big? */
    {
      IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: bad CI length: cilen > L"));
      orc = CONFREJ;			/* Reject bad CI */
      cilen = l;                        /* Reject till end of packet */
      l = 0;                            /* Don't loop again */
      goto endswitch;
    }

    citype = *p++;			/* Parse CI type */
    cilen  = *p++;			/* Parse CI length */
    l -= cilen;				/* Adjust remaining length */
    cilen -= 2;				/* Adjust cilen to just data */

    switch (citype)			/* Check CI type */
    {

    case CI_INTERFACE_TOKEN:
      if(! ao->fToken )
      {
        IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: TOKEN not allowed => REJ "));
        p += cilen;     /* Skip rest of CI */
        orc = CONFREJ;  /* Reject CI */
        break;
      }
        
      if(cilen != sizeof(long))         /* Check CI length */
      {
        IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: TOKEN cilen != 4 => REJ "));
        p += cilen;     /* Skip rest of CI */
        orc = CONFREJ;  /* Reject CI */
        break;
      }
        
      /*
       * If he has no address, or if we have his address but
       * disagree about it, then NAK it with our idea.
       * In particular, if we don't know his address, but he does,
       * then accept it.
       */

      /* Parse his token */
      b1 = *p++; dTmp  = ((u_long) b1) << 24;
      b2 = *p++; dTmp |= ((u_long) b2) << 16;
      b3 = *p++; dTmp |= ((u_long) b3) <<  8;
      b4 = *p++; dTmp |= ((u_long) b4);
      citoken = ntohl( dTmp );

      if(citoken == 0)          /* peer wants us to assign him a token */
      {
        IPV6CPDEBUG((LOG_WARNING, 
		     "ipv6cp_reqci: TOKEN is 0 => assign address"));
        if( reject_if_disagree )
          orc = CONFREJ;
        else
          orc = CONFNAK;
        p -= sizeof(long);
        /* generate a random token */
        ao->Token = magic();
        /* insert the token we want to assign */
        dTmp = htonl( ao->Token );
        *p++ = (u_char) ((dTmp >> 24) & 0xFF);
        *p++ = (u_char) ((dTmp >> 16) & 0xFF);
        *p++ = (u_char) ((dTmp >>  8) & 0xFF);
        *p++ = (u_char) ((dTmp      ) & 0xFF);
        break;
      }
      else
      if(citoken == go->Token)  /* peer wants to use a conflicting token */
      {
        IPV6CPDEBUG((LOG_WARNING,
		     "ipv6cp_reqci: TOKEN conflict => assign address"));
        if( reject_if_disagree )
          orc = CONFREJ;
        else
          orc = CONFNAK;
        p -= sizeof(long);
        /* generate a random token */
        ao->Token = magic();
        /* insert the token we want to assign */
        dTmp = htonl( ao->Token );
        *p++ = (u_char) ((dTmp >> 24) & 0xFF);
        *p++ = (u_char) ((dTmp >> 16) & 0xFF);
        *p++ = (u_char) ((dTmp >>  8) & 0xFF);
        *p++ = (u_char) ((dTmp      ) & 0xFF);
        break;
      }
      
      IPV6CPDEBUG(( LOG_INFO, 
		    "ipv6cp_reqci: unit %d: Token %02x-%02x-%02x-%02x", 
                    f->unit, b1,b2,b3,b4 ));
      ho->fToken = TRUE;
      ho->Token  = citoken;
      break;
        
    case CI_COMPRESSION:
#ifndef USE_IPV6_COMPR
      IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: CI_COMPR not impl. => REJ "));
      p += cilen;
      orc = CONFREJ;
#else
#endif
      break;

    default:
      IPV6CPDEBUG((LOG_WARNING, "ipv6cp_reqci: unknown citype %d => REJ ", citype ));
      p += cilen;
      orc = CONFREJ;
      break;
    }
    cilen += 2;				/* Adjust cilen whole CI */

endswitch:

    if (orc == CONFACK &&               /* Good CI */
        rc != CONFACK)                  /*  but prior CI wasnt? */
      continue;                         /* Don't send this one */

    if (orc == CONFNAK)                 /* Nak this CI? */
    {
      if (rc == CONFREJ)                /* Rejecting prior CI? */
        continue;                       /* Don't send this one */
      if (rc == CONFACK)                /* Ack'd all prior CIs? */
      {
        rc = CONFNAK;                   /* Not anymore... */
        ucp = inp;                      /* Backup */
      }
    }

    if (orc == CONFREJ &&               /* Reject this CI */
        rc != CONFREJ)                  /*  but no prior ones? */
    {
      rc = CONFREJ;
      ucp = inp;                        /* Backup */
    }

    if (ucp != cip)			/* Need to move CI? */
      memcpy(ucp, cip, (size_t)cilen);	/* Move it */

    ucp += cilen;			/* Update output pointer */
  }
  
  /*
   * XXX If we wanted to send additional NAKs (for unsent CIs), the
   * code would go here.  This must be done with care since it might
   * require a longer packet than we received.
   */

  *len = ucp - inp;                     /* Compute output length */
  
  IPV6CPDEBUG(( LOG_INFO, 
              "ipv6cp: returning Configure-%s",
              rc == CONFACK ? "ACK" :
              rc == CONFNAK ? "NAK" : "Reject" ));

  return (rc);                          /* Return final code */
}


/* ------------------------------------------------------------------------
 * ipv6cp_printpkt - print the contents of an IPv6CP packet.
 */

char *ipv6cp_codenames[] =
{
    "ConfReq", "ConfAck", "ConfNak", "ConfRej",
    "TermReq", "TermAck", "CodeRej"
};

int
ipv6cp_printpkt(p, plen, printer, arg)
    u_char *p;
    int plen;
    void (*printer)();
    void *arg;
{
    int code, id, len, olen;
    u_char *pstart, *optend;
    u_short cishort;
    u_int32_t cilong;

    if (plen < HEADERLEN)
        return 0;
    pstart = p;
    GETCHAR(code, p);
    GETCHAR(id, p);
    GETSHORT(len, p);
    if (len < HEADERLEN || len > plen)
        return 0;

    if (code >= 1 && code <= sizeof(ipv6cp_codenames) / sizeof(char *))
        printer(arg, " %s", ipv6cp_codenames[code-1]);
    else
        printer(arg, " code=0x%x", code);
    printer(arg, " id=0x%x", id);
    len -= HEADERLEN;
    switch (code) {
    case CONFREQ:
    case CONFACK:
    case CONFNAK:
    case CONFREJ:
        /* print option list */
        while (len >= 2) {
            GETCHAR(code, p);
            GETCHAR(olen, p);
            p -= 2;
            if (olen < 2 || olen > len) {
                break;
            }
            printer(arg, " <");
            len -= olen;
            optend = p + olen;
            switch (code)
            {
            case CI_INTERFACE_TOKEN:
                if( olen == 6 )
                {
                    p += 2;
                    GETLONG(cilong, p);
                    printer(arg, "token %x", ntohl(cilong));
                }
                break;

            case CI_COMPRESSION:
                if( olen >= 4 )
                {
                    p += 2;
                    GETSHORT(cishort, p);
                    printer(arg, "compress %x", htons(cishort));
                }
                break;
            }
            while (p < optend)
            {
                GETCHAR(code, p);
                printer(arg, " %.2x", code);
            }
            printer(arg, ">");
        }
        break;
    }

    /* print the rest of the bytes in the packet */
    for (; len > 0; --len)
    {
        GETCHAR(code, p);
        printer(arg, " %.2x", code);
    }

    return p - pstart;
}

/* ------------------------------------------------------------------------
 * end of file
 * ------------------------------------------------------------------------ */

