/***********************************************************************/
/* COMM2.C - Commands E-J                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\comm2.c 1.4 1993/09/01 16:25:38 MH Interim MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*-------------------------- external data ----------------------------*/
extern LINE *next_line,*curr_line;
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
extern char current_file;         /* pointer to current file */
extern WINDOW *foot,*error_window;
extern bool error_on_screen;
extern char *temp_cmd;
extern char *cmd_rec;
extern unsigned short cmd_rec_len;
extern char *pre_rec;
extern unsigned short pre_rec_len;
extern char *rec;
extern unsigned short rec_len;
extern char mode_insert;        /* defines insert mode toggle */
extern char in_profile;    /* indicates if processing profile */
extern char in_macro;
extern char *tempfilename;
extern unsigned short file_start;
extern char curr_path[MAX_FILE_NAME+1] ;
extern char sp_path[MAX_FILE_NAME+1] ;
extern char sp_fname[MAX_FILE_NAME+1] ;
/*man-start*********************************************************************
COMMAND
     emsg - display error message

SYNTAX
     EMSG [message]

DESCRIPTION
     The EMSG command displays an error message on the error line.
     This command is usually issued from a macro file.

COMPATIBILITY
     XEDIT: Does not support [mmmnnns text] option
     KEDIT: Compatible

SEE ALSO
     EMSG

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Emsg(char *params)
#else
int Emsg(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Emsg");
#endif
 display_error(0,params);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     enter - execute a command

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The ENTER command executes the command currently displayed on the
     command line, if the cursor is currently displayed there.
     If the key associated with ENTER is pressed while in the main or
     prefix window, then the cursor will move to the first column of the
     next line. If the mode is currently in 'insert', then a new line
     is added and the cursor placed at the first column of the new line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Enter(char *params)
#else
int Enter(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short x,y;
 short rc=RC_OK;
 register int i;
 static char first=TRUE;
 static char number_of_commands=0;
 WINDOW *w;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Enter");
#endif
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_COMMAND:
         for (i=0;i<cmd_rec_len;i++)
             temp_cmd[i] = cmd_rec[i];
         temp_cmd[cmd_rec_len] = '\0';
         strtrunc(temp_cmd);
         add_command(temp_cmd);
         rc = command_line(temp_cmd,COMMAND_ONLY_FALSE);
         break;
    case WINDOW_PREFIX:
         Down_arrow((char *)"");
         getyx(CURRENT_WINDOW,y,x);
         wmove(CURRENT_WINDOW,y,0);
         break;
    case WINDOW_MAIN:
                                       /* check for insert mode or not */
         if (mode_insert)
           {
            post_process_line(CURRENT_VIEW->focus_line);
            insert_new_line((char *)"",0,1,get_true_line(),FALSE,FALSE);
           }
         else
           {
            Down_arrow((char *)"");
            getyx(CURRENT_WINDOW,y,x);
            wmove(CURRENT_WINDOW,y,0);
           }
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     expand - expand tab characters to spaces

SYNTAX
     EXPand [target]

DESCRIPTION
     The EXPAND command converts all tab characters to spaces depending
     on the size of a tab determined by [SET] TABS command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Expand(char *params)
#else
int Expand(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define EXP_PARAMS  1
 char *word[EXP_PARAMS+1];
 char parm[EXP_PARAMS];
 register int i;
 unsigned short num_params;
 long num_lines,true_line;
 unsigned short x,y;
 int direction;
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Expand");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,EXP_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (char *)"1";
    }
    if (equal((char *)"all",word[0],3))
       true_line = 1L;
    else
       true_line = get_true_line();

 if ((num_lines = valid_target(word[0],true_line)) == TARGET_ERROR)
   {
    display_error(4,(char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_lines == TARGET_NOT_FOUND)
   {
    display_error(17,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_TARGET_NOT_FOUND);
   }
/*---------------------------------------------------------------------*/
/* Determine in which direction we are working.                        */
/*---------------------------------------------------------------------*/
 if (num_lines < 0)
   {
    direction = DIRECTION_BACKWARD;
    num_lines = num_lines * (-1L);
   }
 else
   direction = DIRECTION_FORWARD;
/*---------------------------------------------------------------------*/
/* Check from which window the command was issued and make adjustments */
/* as required.                                                        */
/* Commands issued from the command window relate to the current line, */
/* commands issued from either the prefix or main window relate to the */
/* focus line.                                                         */
/*---------------------------------------------------------------------*/
 if (equal((char *)"all",word[0],3))
    ;
 else
    if (CURRENT_VIEW->current_window == WINDOW_COMMAND
    || in_profile)
      {
       if (TOF(true_line))
         {
          true_line++;
          if (!valid_integer(word[0]))
             num_lines--;
         }
       if (BOF(true_line))
         {
          true_line--;
          if (!valid_integer(word[0]))
             num_lines--;
         }
      }
    else
      {
       if (TOF(true_line) || BOF(true_line))
         {
          display_error(38,(char *)"");
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_ENVIRON);
         }
       post_process_line(true_line);
      }
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the true_line.                    */
/* This is the first line to change.                                   */
/*---------------------------------------------------------------------*/
#ifdef USE_VOID
 curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,true_line);
#else
 curr = lll_find(CURRENT_FILE->first_line,true_line);
#endif
/*---------------------------------------------------------------------*/
/* Convert all tabs in the current line to spaces.                     */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_lines;i++)
    {
     add_to_recovery_list(curr->line,curr->length);
     tabs_convert(curr,TRUE);
     if (direction == DIRECTION_FORWARD)
        curr = curr->next;
     else
        curr = curr->prev;
    }
/*---------------------------------------------------------------------*/
/* If we are in the profile, then don't do any more processing.        */
/*---------------------------------------------------------------------*/
 if (in_profile)
   {
    pre_process_line(CURRENT_VIEW->focus_line);
#ifdef TRACE
    trace_return();
#endif
    if (CURRENT_TOF || CURRENT_BOF)
       return(RC_TOF_EOF_REACHED);
    else
       return(RC_OK);
   }
 pre_process_line(CURRENT_VIEW->focus_line);
 show_page();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    getyx(CURRENT_WINDOW,y,x);
    y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
    wmove(CURRENT_WINDOW,y,x);
   }

#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     extract - obtain setting information from THE

SYNTAX
     EXTract /item/[...]

DESCRIPTION
     The EXTRACT command is used to relay information about settings
     within the from within a REXX macro. EXTRACT is only valid within
     a REXX macro.

COMPATIBILITY
     XEDIT: Only allows '/' as delimiter.
     KEDIT: Only allows '/' as delimiter.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Extract(char *params)
#else
int Extract(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i;
 int rc,itemno,num_items=0,len;
 char *orig_params;
 int pos;
 char *args;
 bool invalid_item = FALSE;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Extract");
#endif
 if (!in_macro)
   {
    display_error(53,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
#if !defined(NOREXX)
/*---------------------------------------------------------------------*/
/* Check that the first character of parameters is /.                  */
/*---------------------------------------------------------------------*/
 if (*(params) != '/')
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 params++;                               /* throw away first delimiter */
 orig_params = params;
 len = strlen(params);
/*---------------------------------------------------------------------*/
/* Check that we have an item to extract...                            */
/*---------------------------------------------------------------------*/
 if (len == 0)
    invalid_item = TRUE;
 else
    if (len == 1 && (*(params) == '/'))
       invalid_item = TRUE;
 if (invalid_item)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Allow for no trailing '/'...                                        */
/*---------------------------------------------------------------------*/
 if ((*(params+len-1) == '/'))
    num_items = 0;
 else
    num_items = 1;
/*---------------------------------------------------------------------*/
/* Replace all \ with nul character to give us seperate strings.       */
/*---------------------------------------------------------------------*/
 for (i=0;i<len;i++)
   {
    if (*(params+i) == '/')
      {
       *(params+i) = '\0';
       num_items++;
      }
   }
/*---------------------------------------------------------------------*/
/* For each item, extract its variables...                             */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_items;i++)
   {
/*---------------------------------------------------------------------*/
/* First check if the item has any arguments with it.                  */
/*---------------------------------------------------------------------*/
    pos = strzeq(params,' ');
    if (pos == (-1))
       args = "";
    else
      {
       *(params+pos) = '\0';
       args = strtrunc(params+pos+1);
      }
/*---------------------------------------------------------------------*/
/* Find the item in the list of valid extract options...               */
/*---------------------------------------------------------------------*/
    if ((itemno = find_item(params,QUERY_EXTRACT)) == (-1))
       {
        display_error(1,params);
#ifdef TRACE
        trace_return();
#endif
        return(RC_INVALID_OPERAND);
       }
/*---------------------------------------------------------------------*/
/* Get the current settings for the valid item...                      */
/*---------------------------------------------------------------------*/
    itemno = get_item_values(itemno,args,QUERY_EXTRACT,0L,NULL,0L);
/*---------------------------------------------------------------------*/
/* If the arguments to the item are invalid, return with an error.     */
/*---------------------------------------------------------------------*/
    if (itemno == EXTRACT_ARG_ERROR)
       {
#ifdef TRACE
        trace_return();
#endif
        return(RC_INVALID_OPERAND);
       }
/*---------------------------------------------------------------------*/
/* If the REXX variables have already been set, don't try to set them. */
/*---------------------------------------------------------------------*/
    if (itemno != EXTRACT_VARIABLES_SET)
      {
       rc = set_extract_variables(params);
       if (rc == RC_SYSTEM_ERROR)
          break;
      }
    params += strlen(params)+1;
   }
#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     ffile - force file the current file to disk

SYNTAX
     FFile  [filename]

DESCRIPTION
     The FFILE command writes the current file to disk to the current
     file name or to the supplied filename.
     Unlike the FILE command, if the optional filename exists, this
     command will overwrite the file.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     With no parameters, the current file is written.

SEE ALSO
     file,save,ssave

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Ffile(char *params)
#else
int Ffile(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Ffile");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
 if ((rc = save_file(CURRENT_FILE,params,YES,0L,0L,NO,0,max_line_length)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of FFiling, remove the .aus file...   */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->autosave > 0)
    rc = remove_aus_file(CURRENT_FILE);
 free_view_memory();
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     file - file the current file to disk

SYNTAX
     FILE  [filename]

DESCRIPTION
     The FILE command writes the current file to disk to the current
     file name or to the supplied filename.
     Unlike the FFILE command, if the optional filename exists, this
     command will not overwrite the file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     With no parameters, the current file is written.

SEE ALSO
     ffile,save,ssave

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int File(char *params)
#else
int File(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   File");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* If we are filing the current file with the same name AND the number */
/* of alterations is zero, then quit the file.                         */
/*---------------------------------------------------------------------*/
 if (CURRENT_FILE->save_alt == 0 && strcmp(params,"") == 0)
    Quit("");
 else
   {
    if ((rc = save_file(CURRENT_FILE,params,NO,0L,0L,NO,0,max_line_length)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
/*---------------------------------------------------------------------*/
/* If autosave is on at the time of Filing, remove the .aus file...    */
/*---------------------------------------------------------------------*/
    if (CURRENT_FILE->autosave > 0)
       rc = remove_aus_file(CURRENT_FILE);
    free_view_memory();
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     fillbox - fill the marked box block with a character

SYNTAX
     FILLbox [c]

DESCRIPTION
     The FILLBOX command fills the marked block with the specified
     character. If no parameters are supplied and the command is run
     from the command line, then the box will be filled with spaces.
     If the command is not run from the command line, the user is
     prompted for a character to fill the box.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Fillbox(char *params)
#else
int Fillbox(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 char chr;
 int len_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Fillbox");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Validate the marked block.                  .                       */
/*---------------------------------------------------------------------*/
 if (marked_block(TRUE) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* This function not valid for line blocks.                            */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW->mark_type == M_LINE)
   {
    display_error(47,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Check if hex on in effect and translate hex char if required...     */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->hex == ON)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(char *)"");
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen(params);
/*---------------------------------------------------------------------*/
/* Whew, now do something...                                           */
/*---------------------------------------------------------------------*/
 if (len_params > 1)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (len_params == 0)
    chr = ' ';
 else
    chr = *(params);
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND
 && len_params != 1) 
   {
    display_error(0,(char *)"Enter fill character...");
    touchwin(error_window);
    doupdate();
    chr = (char)my_getch(CURRENT_WINDOW);
    error_on_screen = FALSE;
    touchwin(foot);
   }
 box_operations(BOX_F,SOURCE_BLOCK,TRUE,chr);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     forward - scroll forwards a number of screens

SYNTAX
     FOrward [n]

DESCRIPTION
     The FORWARD command scrolls the file contents forwards the number
     of screens specified.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Does not support HALF or Lines options.

DEFAULT
     With no parameters, 1 screen is scrolled.

SEE ALSO
     backward,top

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Forward(char *params)
#else
int Forward(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define FOR_PARAMS  1
 char *word[FOR_PARAMS+1];
 char parm[FOR_PARAMS];
 register int i;
 unsigned short num_params;
 long num_pages;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Forward");
#endif
 num_params = param_split(params,word,FOR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (char *)"1";
   }
 if (num_params != 1)
   {
    display_error(1,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (strcmp(word[0],"*") == 0)
   {
    CURRENT_VIEW->current_line = CURRENT_FILE->number_lines+1;
    post_process_line(CURRENT_VIEW->focus_line);
    CURRENT_VIEW->focus_line = CURRENT_FILE->number_lines+1;
    pre_process_line(CURRENT_VIEW->focus_line);
    if (!in_profile)
      {
       if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
          getyx(PREVIOUS_WINDOW,y,x);
       else
          getyx(CURRENT_WINDOW,y,x);
       show_page();
       y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
       if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
          wmove(PREVIOUS_WINDOW,y,x);
       else
          wmove(CURRENT_WINDOW,y,x);
      }
#ifdef TRACE
    trace_return();
#endif
    if (CURRENT_TOF || CURRENT_BOF)
       return(RC_TOF_EOF_REACHED);
    else
       return(RC_OK);
   }
 if ((num_pages = atol(word[0])) == 0L)
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }

 CURRENT_VIEW->current_line =
    min(CURRENT_VIEW->current_line + (CURRENT_SCREEN.rows-1)*num_pages,
        CURRENT_FILE->number_lines+1);

 post_process_line(CURRENT_VIEW->focus_line);
 CURRENT_VIEW->focus_line = calculate_focus_line(CURRENT_VIEW->current_row,
                                                CURRENT_VIEW->focus_line,
                                                CURRENT_VIEW->current_line);
 pre_process_line(CURRENT_VIEW->focus_line);
 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    if (CURRENT_TOF || CURRENT_BOF)
       return(RC_TOF_EOF_REACHED);
    else
       return(RC_OK);
   }
 getyx(CURRENT_WINDOW,y,x);
 show_page();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
    wmove(CURRENT_WINDOW,y,x);
   }

#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     get - read another file into current file

SYNTAX
     GET [fileid]

DESCRIPTION
     The GET command reads a file into the current file, inserting
     lines after the current line.
     When no fileid is supplied the temporary file is used.

COMPATIBILITY
     XEDIT: Does not support optional firstrec - numrecs
     KEDIT: Does not support optional fromlines - forlines.

SEE ALSO
     put,putd

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Get(char *params)
#else
int Get(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 char *filename;
 FILE *fp;
 LINE *curr;
 long old_number_lines,true_line;
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Get");
#endif
 if (strcmp(params,"") == 0)   /* no fileid supplied */
    filename = tempfilename;
 else
   {
    if ((rc = splitpath(strtrans(params,OSLASH,ISLASH))) != RC_OK)
      {
       display_error(10,params);
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    strcpy(temp_cmd,sp_path);
    strcat(temp_cmd,sp_fname);
    filename = temp_cmd;
   }

 post_process_line(CURRENT_VIEW->focus_line);

 if ((fp = fopen(filename,"r")) == NULL)
   {
    display_error(9,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_ACCESS_DENIED);
   }
 true_line = get_true_line();
#ifdef USE_VOID
 curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,true_line);
#else
 curr = lll_find(CURRENT_FILE->first_line,true_line);
#endif
 if (curr->next == NULL)   /* on bottom of file */
    curr = curr->prev;
 old_number_lines = CURRENT_FILE->number_lines;
 if ((curr = read_file(fp,curr,filename)) == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_ACCESS_DENIED);
   }

 fclose(fp);
 pre_process_line(CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Fix the positioning of the marked block (if there is one and it is  */
/* in the current view).                                               */
/*---------------------------------------------------------------------*/
 if (!in_profile)
   {
    adjust_marked_lines(TRUE,true_line,CURRENT_FILE->number_lines - old_number_lines);
    adjust_pending_prefix(CURRENT_VIEW,TRUE,true_line,CURRENT_FILE->number_lines - old_number_lines);
   }
/*---------------------------------------------------------------------*/
/* Increment the number of lines counter for the current file and the  */
/* number of alterations.                                              */
/*---------------------------------------------------------------------*/
 if ((rc = increment_alt(CURRENT_FILE)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 if (!in_profile)
    show_page();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     help - display help for the editor

SYNTAX
     HELP

DESCRIPTION
     The HELP command displays help for the editor.
     Uses THE_HELP environment variable to point to the help file.

COMPATIBILITY
     XEDIT: Similar in concept.
     KEDIT: Similar in concept.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Help(char *params)
#else
int Help(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
extern char the_help_file[MAX_FILE_NAME+1];
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Help");
#endif
 if (!file_exists(the_help_file))
   {
    display_error(23,(char *)the_help_file);
#ifdef TRACE
    trace_return();
#endif
    return(RC_FILE_NOT_FOUND);
   }
 Xedit(the_help_file);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     input - insert the command line contents into the file

SYNTAX
     Input [line contents]

DESCRIPTION
     The INPUT command inserts the remainder of the command line into the
     file after the current line.

COMPATIBILITY
     XEDIT: Does not provide full input mode option.
     KEDIT: Does not provide full input mode option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Input(char *params)
#else
int Input(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 LINE *curr;
 unsigned long li;
 int len_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Input");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
 if (CURRENT_VIEW->hex == ON)
   {
    if ((len_params = convert_hex_strings(params)) == (-1))
      {
       display_error(32,(char *)"");
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
 else
   len_params = strlen(params);
 insert_new_line(params,len_params,1L,get_true_line(),TRUE,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     join - join a line with the line following

SYNTAX
     Join [ALigned]

DESCRIPTION
     The JOIN command makes one line out of the focus line and the
     line following.
     If aligned is specified, any leading spaces in the following line
     are ignored. If aligned is not specified, all characters, including
     spaces are added.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Join(char *params)
#else
int Join(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc;
 bool aligned;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm2.c:   Join");
#endif
/*---------------------------------------------------------------------*/
/* Check here for parameter value of 'Aligned'.                        */
/*---------------------------------------------------------------------*/
 if (equal((char *)"aligned",params,2))
    aligned = ALIGNED_TRUE;
 else
    if (strcmp(params,"") == 0)
       aligned = ALIGNED_FALSE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_ENVIRON);
      }
 rc = execute_split_join(SPLIT_FALSE,aligned);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
