/***********************************************************************/
/* COMM3.C - Commands K-O                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\comm3.c 1.4 1993/09/01 16:25:40 MH Interim MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*-------------------------- external data ----------------------------*/
extern LINE *next_line,*curr_line;
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
extern char display_screens;                      /* number of screens */
extern char current_file;         /* pointer to current file */
extern WINDOW *foot,*error_window;
extern bool error_on_screen;
extern char *rec;
extern unsigned short rec_len;
extern char mode_insert;        /* defines insert mode toggle */
extern char in_profile;    /* indicates if processing profile */

extern char *temp_cmd;
extern char *last_target;
/*man-start*********************************************************************
COMMAND
     left_arrow - move the cursor left one column

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The left_arrow command moves the cursor left one column in the
     current window. Scrolling of the main window horizontally, occurs
     if the cursor is at the left-most column and the left-most column
     is not the first column of the line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of CURSOR LEFT

SEE ALSO
     Right_arrow

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Left_arrow(char *params)
#else
int Left_arrow(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool CMDARROWSTABLRx;
/*--------------------------- local data ------------------------------*/
 unsigned short x,y;
 short col,old_col;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Left_arrow");
#endif
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* For all windows, if we are not at left column, move 1 pos to left.  */
/*---------------------------------------------------------------------*/
 if (x > 0)
   {
    wmove(CURRENT_WINDOW,y,x-1);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* For all windows, determine if CMDARROWSTABLRx is set for tabbing or */
/* scrolling and act accordingly.                                      */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->current_window)
   {
    case WINDOW_MAIN:
         if (CMDARROWSTABLRx)
           {
            if (CURRENT_VIEW->prefix)
              {
               Tabpre("");
               wmove(CURRENT_WINDOW,y,getmaxx(CURRENT_WINDOW)-1);
              }
            else
               wmove(CURRENT_WINDOW,y,getmaxx(CURRENT_WINDOW)-1);
           }
         else
           {
            if (CURRENT_VIEW->verify_start != CURRENT_VIEW->verify_col)
              {
               old_col = (CURRENT_VIEW->verify_col-1);
               x = CURRENT_SCREEN.cols / 2;
               col = max((short)CURRENT_VIEW->verify_start,
                         (short)(CURRENT_VIEW->verify_col-1) - (short)x);
               CURRENT_VIEW->verify_col = col;
               show_page();
               x = old_col - (CURRENT_VIEW->verify_col-1);
               wmove(CURRENT_WINDOW,y,x-1);
              }
           }
         break;
    case WINDOW_PREFIX:
         if (CMDARROWSTABLRx)
            if (CURRENT_VIEW->prefix)
              {
               Tabpre("");
               wmove(CURRENT_WINDOW,y,getmaxx(CURRENT_WINDOW)-1);
              }
         break;
    case WINDOW_COMMAND:
         if (CMDARROWSTABLRx)
            wmove(CURRENT_WINDOW,y,getmaxx(CURRENT_WINDOW)-1);
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     locate - make line identified by target become the current line

SYNTAX
     [Locate] target

DESCRIPTION
     The LOCATE command looks for the next or previous occurrence of the
     specified target.  If no parameter is supplied, LOCATE
     searches for the string that was used as the last string target, if
     such a string target exists.

COMPATIBILITY
     XEDIT: Does not support [subcommand] option.
     KEDIT: Compatible. (See bleow)
     Does not support not,and,or combinations of string targets.
     ie ~,& and | not supported.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Locate(char *params)
#else
int Locate(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short x,y;
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Locate");
#endif
/*---------------------------------------------------------------------*/
/* If no parameter is specified, use the last_target. If that doesn't  */
/* exist, error.                                                       */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)
   {
    if (strcmp(last_target,"") == 0)
      {
       display_error(39,(char *)"");
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    rc = command_line(last_target,COMMAND_ONLY_FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 rc = command_line(params,COMMAND_ONLY_FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     lowercase - change uppercase characters to lowercase

SYNTAX
     LOWercase [target]

DESCRIPTION
     The LOWERCASE command changes all uppercase characters in the target
     to lowercase. All other characters remain untouched.

COMPATIBILITY
     XEDIT: Equivalent of LOWERCAS command.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Lowercase(char *params)
#else
int Lowercase(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Lowercase");
#endif
 rc = execute_change_case(params,CASE_LOWER);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     macro - execute a macro command file

SYNTAX
     MACRO filename [arguments]

DESCRIPTION
     The MACRO command executes the contents of the specified file
     as command line commands.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.
     Only those systems that support REXX, support argument passing
     to the macro file.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Macro(char *params)
#else
int Macro(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Macro");
#endif
 rc = execute_macro(params,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     mark - mark a portion of text

SYNTAX
     MARK Line|Box|Stream

DESCRIPTION
     The MARK command marks a portion of text for later processing
     by a COPY, MOVE or DELETE command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Does not implement Stream.

STATUS
     Complete...LINE and BOX options are anyway.
**man-end**********************************************************************/
#ifdef PROTO
int Mark(char *params)
#else
int Mark(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
 long true_line;
 unsigned short y,x;
 unsigned short real_col;
 bool type_param=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Mark");
#endif
 if (equal(params,(char *)"line",1))
    type_param = M_LINE;
 if (equal(params,(char *)"box",1))
    type_param = M_BOX;
 if (type_param == 0)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
    true_line = CURRENT_VIEW->current_line;
 else
    true_line = CURRENT_VIEW->focus_line;
/*---------------------------------------------------------------------*/
/* If we are on 'Top of File' or 'Bottom of File' lines, error.        */
/*---------------------------------------------------------------------*/
 if (TOF(true_line) || BOF(true_line))
   {
    display_error(38,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 if (MARK_VIEW != (VIEW_DETAILS *)NULL
 &&  MARK_VIEW != CURRENT_VIEW)
   {
    MARK_VIEW->mark_start_line = MARK_VIEW->mark_end_line = (-1L);
    MARK_VIEW->mark_start_col = MARK_VIEW->mark_end_col = (-1);
   }
 MARK_VIEW = CURRENT_VIEW;
 getyx(CURRENT_WINDOW,y,x);
 CURRENT_VIEW->mark_type = type_param;
/*---------------------------------------------------------------------*/
/* Set the new values for top and bottom lines marked.                 */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->mark_start_line == (-1L)
 &&  CURRENT_VIEW->mark_end_line == (-1L))
    CURRENT_VIEW->mark_start_line = CURRENT_VIEW->mark_end_line = true_line;
 else
   {
    if (true_line > CURRENT_VIEW->mark_end_line)
       CURRENT_VIEW->mark_end_line = true_line;
    if (true_line < CURRENT_VIEW->mark_start_line)
       CURRENT_VIEW->mark_start_line = true_line;
    if (true_line < CURRENT_VIEW->mark_end_line
    &&  true_line > CURRENT_VIEW->mark_start_line)
      {
       if (true_line-CURRENT_VIEW->mark_end_line >
           CURRENT_VIEW->mark_start_line-true_line)
          CURRENT_VIEW->mark_end_line = true_line;
       else
          CURRENT_VIEW->mark_start_line = true_line;
      }
   }
 if (type_param == M_LINE)
   {
    CURRENT_VIEW->mark_start_col = (-1);
    CURRENT_VIEW->mark_end_col = (-1);
   }
 if (type_param == M_BOX)
   {
    real_col = x + CURRENT_VIEW->verify_col;
    if (CURRENT_VIEW->mark_start_col == (-1)
    &&  CURRENT_VIEW->mark_end_col == (-1))
       CURRENT_VIEW->mark_start_col = CURRENT_VIEW->mark_end_col = real_col;
    else
      {
       if ((real_col) > CURRENT_VIEW->mark_end_col)
          CURRENT_VIEW->mark_end_col = (real_col);
       if ((real_col) < CURRENT_VIEW->mark_start_col)
          CURRENT_VIEW->mark_start_col = (real_col);
       if ((real_col) < CURRENT_VIEW->mark_end_col
       &&  (real_col) > CURRENT_VIEW->mark_start_col)
         {
          if ((real_col)-CURRENT_VIEW->mark_end_col >
              CURRENT_VIEW->mark_start_col-(real_col))
             CURRENT_VIEW->mark_end_col = (real_col);
          else
             CURRENT_VIEW->mark_start_col = (real_col);
         }
      }
   }
 show_page();
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     modify - alter a setting

SYNTAX
     MODify set-command

DESCRIPTION
     The MODIFY command displays the current setting of a set command
     on the command line to make changing that setting easy.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET,QUERY

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Modify(char *params)
#else
int Modify(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[18];
/*--------------------------- local data ------------------------------*/
 register int i;
 int itemno;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Modify");
#endif
 if ((itemno = find_item(params,QUERY_MODIFY)) == (-1))
    {
     display_error(1,params);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }

 itemno = get_item_values(itemno,"",QUERY_MODIFY,0L,NULL,0L);
 strcpy(temp_cmd,"set ");
 for (i=0;i<itemno+1;i++)
   {
    strcat(temp_cmd,item_values[i].value);
    strcat(temp_cmd," ");
   }
 Cmsg(temp_cmd);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     move - move a portion of text

SYNTAX
     MOVE target1 target2
      or
     MOVE BLOCK [RESET]

DESCRIPTION
     The MOVE command copies the contents of the marked block to the
     current cursor position and deletes the characters/lines from the
     original position of the marked block.

COMPATIBILITY
     The first form of the MOVE command has not yet been implemented.
     XEDIT: Not implemented.
     KEDIT: Adds extra functionality with [RESET] option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Move(char *params)
#else
int Move(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define MOV_PARAMS 2
 char *word[MOV_PARAMS+1];
 unsigned short num_params;
 LINE *curr,*old_curr,*save_old_curr;
 register int i;
 unsigned short y,x;
 bool same_file;
 long true_line;
 char reset_block=0;
 char copy_command,delete_command;
 int rc;
 long start_line,end_line,num_lines,dest_line;
 VIEW_DETAILS *old_mark_view;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Move");
#endif
 num_params = param_split(params,word,MOV_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
    {
     display_error(3,(char *)"");
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (num_params > 2)
    {
     display_error(2,(char *)"");
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Test for valid parameters...                                        */
/*---------------------------------------------------------------------*/
 if (num_params == 1
 &&  equal((char *)"block",word[0],5))
    reset_block = SOURCE_BLOCK;
 if (num_params == 2
 &&  equal((char *)"block",word[0],5)
 &&  equal((char *)"reset",word[1],5))
    reset_block = SOURCE_BLOCK_RESET;
 if (reset_block == 0)
    {
     display_error(1,params);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Validate marked block, can be in any view.                          */
/*---------------------------------------------------------------------*/
 if (marked_block(FALSE) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* If the cursor is in the marked block...error.                       */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW == CURRENT_VIEW)
   {
    getyx(CURRENT_WINDOW_MAIN,y,x);
    switch(MARK_VIEW->mark_type)
      {
       case M_LINE:
                   if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                   &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line))
                     {
                      display_error(44,(char *)"for line move");
#ifdef TRACE
                      trace_return();
#endif
                      return(RC_INVALID_ENVIRON);
                     }
                   break;
       case M_BOX:
                   if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                   &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
                   &&  (x + CURRENT_VIEW->verify_start >= MARK_VIEW->mark_start_col)
                   &&  (x + CURRENT_VIEW->verify_start <= MARK_VIEW->mark_end_col))
                     {
                      display_error(50,(char *)"");
#ifdef TRACE
                      trace_return();
#endif
                      return(RC_INVALID_ENVIRON);
                     }
                   break;
       default:
                   break;
      }
   }
/*---------------------------------------------------------------------*/
/* If block is a box, call its function.                               */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW->mark_type == M_BOX)
   {
    box_operations(BOX_M,reset_block,FALSE,' ');/* don't reset and don't overlay */
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Determine the target line. If on the command line, target is current*/
/* line, else target line is focus line.                               */
/*---------------------------------------------------------------------*/
 true_line = get_true_line();
/*---------------------------------------------------------------------*/
/* If the true  line is the bottom of file line, subtract 1 from it.   */
/*---------------------------------------------------------------------*/
 if (BOF(true_line))
    true_line--;

 post_process_line(CURRENT_VIEW->focus_line);
 start_line = MARK_VIEW->mark_start_line;
 end_line = MARK_VIEW->mark_end_line;
 num_lines = end_line - start_line + 1L;
 dest_line = true_line;
 old_mark_view = MARK_VIEW;
 if (MARK_VIEW == CURRENT_VIEW)
   {
    copy_command = COMMAND_MOVE_COPY_SAME;
    delete_command = COMMAND_MOVE_DELETE_SAME;
   }
 else
   {
    copy_command = COMMAND_MOVE_COPY_DIFF;
    delete_command = COMMAND_MOVE_DELETE_DIFF;
   }

 rc = rearrange_line_blocks(copy_command,(char)reset_block,start_line,
                            end_line,dest_line,1,MARK_VIEW,CURRENT_VIEW);
 if (old_mark_view == CURRENT_VIEW)
    if (dest_line < start_line)
      {
       start_line += num_lines;
       end_line += num_lines;
       dest_line += num_lines;
      }

 post_process_line(CURRENT_VIEW->focus_line);
 rc = rearrange_line_blocks(delete_command,(char)reset_block,start_line,
                            end_line,start_line,1,old_mark_view,old_mark_view);

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     msg - display message

SYNTAX
     MSG [message]

DESCRIPTION
     The MSG command displays an error message on the error line.
     This command is usually issued from a macro file.
     This is similar to EMSG, but MSG does not sound the bell if SET
     BELL is on.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     EMSG

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Msg(char *params)
#else
int Msg(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool BEEPx;
/*--------------------------- local data ------------------------------*/
 bool save_beep=BEEPx;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Msg");
#endif
 BEEPx = FALSE;
 display_error(0,params);
 BEEPx = save_beep;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     next - move forward in the file a number of lines

SYNTAX
     Next [target]

DESCRIPTION
     The NEXT command moves the current_line forwards the number of
     lines specified by the target. Negative targets move backwards
     through the file.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     Up

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Next(char *params)
#else
int Next(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define NEX_PARAMS  1
 char *word[NEX_PARAMS+1];
 unsigned short num_params;
 long num_lines;
 unsigned short y,x;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Next");
#endif
 num_params = param_split(params,word,NEX_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (char *)"1";
   }
 if (num_params != 1)
   {
    display_error(1,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (equal((char *)"all",word[0],3))         /* all is not a valid target here */
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if ((num_lines = valid_target(word[0],get_true_line())) == TARGET_ERROR)
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 post_process_line(CURRENT_VIEW->focus_line);
 if (in_profile || CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    CURRENT_VIEW->current_line += num_lines;
    CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
    pre_process_line(CURRENT_VIEW->focus_line);
    if (!in_profile)
       show_page();
   }
 else
   {
    getyx(CURRENT_WINDOW,y,x);
    CURRENT_VIEW->focus_line += num_lines;
    pre_process_line(CURRENT_VIEW->focus_line);
    if (num_lines + y <= 0
    ||  num_lines + y >= CURRENT_SCREEN.rows)
      {
       CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
       y = CURRENT_VIEW->current_row;
      }
    else
       y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
    show_page();
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 if (CURRENT_TOF || CURRENT_BOF)
    return(RC_TOF_EOF_REACHED);
 else
    return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     nextwindow - switch focus of editing session to other window

SYNTAX
     NEXTWindow

DESCRIPTION
     The NEXTWINDOW command moves the focus of the editing session to
     the other window (if more than one window is currently displayed)
     or to the next file in the ring.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     Edit,Screen

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Nextwindow(char *params)
#else
int Nextwindow(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Nextwindow");
#endif
 if (strcmp(params,"") != 0)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (display_screens == 1)
   {
    rc = Xedit("");
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 post_process_line(CURRENT_VIEW->focus_line);
 current_screen = (current_screen == 0) ? 1 : 0;
 CURRENT_VIEW = CURRENT_SCREEN.screen_view;
 pre_process_line(CURRENT_VIEW->focus_line);

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     nomsg - execute a command suppressing any messages 

SYNTAX
     NOMSG command [parameters]

DESCRIPTION
     The NOMSG command executes the supplied command but suppresses
     messages that may normally be displayed as a result of the command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Nomsg(char *params)
#else
int Nomsg(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 bool save_msgmode=CURRENT_VIEW->message_mode;
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Nomsg");
#endif
 CURRENT_VIEW->message_mode = FALSE;
#if 0
 if (allocate_temp_space(strlen(params),TEMP_TEMP_CMD) != RC_OK)
    {
     display_error(30,(char *)"");
#ifdef TRACE
     trace_return();
#endif
     return(RC_OUT_OF_MEMORY);
    }
 strcpy(temp_cmd,params);
 rc = command_line(temp_cmd,COMMAND_ONLY_FALSE);
#else
 rc = command_line(params,COMMAND_ONLY_FALSE);
#endif
 CURRENT_VIEW->message_mode = save_msgmode;

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     nop - no operation command

SYNTAX
     NOP

DESCRIPTION
     The NOP command doesn't do anything. It is used as a means of
     setting a key to do nothing.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     Define.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Nop(char *params)
#else
int Nop(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Nop");
#endif

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     os - execute an operating system command

SYNTAX
     OS|DOS|! [command]

DESCRIPTION
     The OS command executes the supplied os command or runs an
     interactive shell if no command is supplied.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of DOS command.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Os(char *params)
#else
int Os(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Os");
#endif
/*---------------------------------------------------------------------*/
/* Execute the supplied parameters as OS commands. Run with output     */
/* displayed and pause before redrawing the windows.                   */
/*---------------------------------------------------------------------*/
 rc = execute_os_command(params,FALSE,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     osnowait - execute an operating system command - no wait

SYNTAX
     OSNowait|DOSNowait command

DESCRIPTION
     The OSNOWAIT command executes the supplied os command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of DOSNOWAIT command.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Osnowait(char *params)
#else
int Osnowait(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Osnowait");
#endif
/*---------------------------------------------------------------------*/
/* Execute the supplied parameters as OS commands. Run with output     */
/* displayed but no pause before redrawing the windows.                */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)                     /* no params....error */
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 rc = execute_os_command(params,FALSE,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     osquiet - execute an operating system command quietly

SYNTAX
     OSQuiet|DOSQuiet command

DESCRIPTION
     The OSQUIET command executes the supplied os command as quietly
     as possible.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Equivalent of DOSQUIET command.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Osquiet(char *params)
#else
int Osquiet(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Osquiet");
#endif
/*---------------------------------------------------------------------*/
/* Execute the supplied parameters as OS commands. Run with no output  */
/* displayed and no pause before redrawing the windows.                */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") == 0)                     /* no params....error */
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 rc = execute_os_command(params,TRUE,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     overlaybox - overlay marked box block on current position

SYNTAX
     OVERLAYBox

DESCRIPTION
     The OVERLAYBOX copies the contents of the marked box block over the
     characters at the current cursor position or to column 1 of the
     current line if issued from the command line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     move block, copy block

STATUS
     Incomplete. Bug with overlaying text on top of itself.
**man-end**********************************************************************/
#ifdef PROTO
int Overlaybox(char *params)
#else
int Overlaybox(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm3.c:   Overlaybox");
#endif
/*---------------------------------------------------------------------*/
/* Ensure there are no parameters.                                     */
/*---------------------------------------------------------------------*/
 if (strcmp(params,"") != 0)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate marked block, can be in any view.                          */
/*---------------------------------------------------------------------*/
 if (marked_block(FALSE) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
/*---------------------------------------------------------------------*/
/* Only allow command for box blocks.                                  */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW->mark_type != M_BOX)
   {
    display_error(47,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_ENVIRON);
   }
 box_operations(BOX_C,SOURCE_BLOCK,TRUE,' ');  /* no reset, overlay */

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
