/*
 * Copyright (c) 1990, 1991, 1992, 1994, 1996, 1997, 1998
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static const char copyright[] =
    "@(#) Copyright (c) 1990, 1991, 1992, 1994, 1996, 1997, 1998\n\
The Regents of the University of California.  All rights reserved.\n";
static const char rcsid[] =
    "@(#) $Header: dexpire.c,v 1.105 98/01/24 14:05:50 leres Exp $ (LBL)";
#endif

/*
 * dexpire - dynamic expire for netnews
 *
 * Most (if not all) other expire programs force the news administrator
 * to predict how many days worth of news articles they have room for.
 * This approach is fundamentally incorrect.
 *
 * A better way is to specify the relative importance (or priority) of
 * newsgroups and have the expire program delete just enough articles
 * in the desired proportion to free the amount of disk space required
 * until the next time the expire program runs. This is how dexpire
 * works.
 *
 * Dexpire breaks all newsgroups into "classes." Each class has a
 * priority associated with it. High priority groups are kept for
 * a longer period than low priority groups.
 *
 * The "standard" class is the highest priority class. Articles in
 * newsgroups in the standard class are kept the longest. The standard
 * class is also used to determine how long lower priority classes
 * are kept.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_LIBUTIL_H
#include <libutil.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

/* Yuck! */
#if HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#if HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#if HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#if HAVE_NDIR_H
#include <ndir.h>
#endif
#endif

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include <err.h>

#include "dexpire.h"
#include "disk.h"
#include "file.h"
#ifndef HAVE_SETPROCTITLE
#include "setproctitle.h"
#endif
#include "util.h"
#include "patchlevel.h"

/* Filesystem node */
struct fsystem {
	dev_t dev;		/* device number */
	struct class *class;	/* class list */
	struct class *standardclass;
	char *subdir;		/* spool subdir name, e.g. "alt" */
	char *dir;		/* spool full pathname name */
	char *tfeedname;	/* temporary file for feedback data */
	char *tlowmark;		/* temporary file for lowmark data */
	long free_kb;		/* current number of free Kbytes */
	long used_kb;		/* current number of used Kbytes */
	long want_kb;		/* desired number of free Kbytes */
	long need_kb;		/* needed number of free Kbytes */
	long deleted;		/* number of articles deleted */
	long freed_blk;		/* number of 512 bytes blocks freed */
	time_t starttime;	/* time at start of dexpire run */
	time_t standardtime;	/* "standard" timestamp */
	time_t deltatime;	/* delta of current and standard timestamps */
	time_t dtime;		/* delta added to standardtime in loop */
	time_t to;		/* timestamp of oldest article */
	pid_t pid;		/* if not 0, pid of subprocess */
	struct fsystem *next;	/* forward link */
};

/* Class node */
struct class {
	u_int pri;		/* priority of this class */
	long deleted;		/* number of articles deleted */
	long freed_blk;		/* number of 512 bytes blocks freed */
	time_t minimum;		/* minimum time to keep articles */
	time_t cutoff;		/* timestamp of cutoff */
	struct group *group;	/* group list */
	struct class *next;	/* forward link */
};

/* Expire rule list */
struct rule {
	char *ngn;		/* newsgroup name, e.g. "alt/slack" */
	u_int pri;		/* priority of this rule */
	time_t minimum;		/* minimum time to keep articles */
	char mod;		/* moderation character */
	char chud[3];
	struct rule *next;	/* forward link */
};

/* Newsgroup node */
struct group {
	char *ngn;		/* newsgroup name, e.g. alt/slack */
	dev_t dev;		/* device number */
	char mod;		/* moderation character */
	char uplowmark;		/* true if "no" differs from active file */
	char chud[2];
	u_long no;		/* number of oldest article */
	u_long nn;		/* number of newest article */
	time_t to;		/* timestamp of oldest article */
	struct group *next;	/* forward link */
};

/* Private data */
static char *active_file = ACTIVE_FILE;
static char *dexplist = DEXPLIST;
static char *dfeedback = DFEEDBACK;
static char *ndfeedback = NDFEEDBACK;
static char *dlowmark = DLOWMARK;
static char default_spool_dir[] = SPOOL_DIR;
static char *spool_dir = default_spool_dir;

static time_t currenttime;	/* current timestamp */

/* We always have at least the top level filesystem */
static struct fsystem topfsystem;

/* List of all filesystems */
static struct fsystem *fsystemlist = &topfsystem;

/* List of rules (read from dexplist) */
static struct rule *rulelist;

/* List of groups (read from active file) */
static struct group *grouplist;

/* Forwards */
static void assigngroups(void);
static int checkfsystem(struct fsystem *);
static dev_t devnumber(char *, char *, const char *);
static int dexpire(struct fsystem *);
static int expireone(struct fsystem *, struct group *, struct class *);
static struct fsystem *findfsystem(dev_t, char *);
int main(int, char **);
static void newdtime(struct fsystem *);
static int newgroup(char *, u_long, u_long, char);
int nextrule(char *, char, int, int);
static int parseactive(void);
static int parseexplist(void);
static void procstart(register struct fsystem *);
static char *proctmpname(const char *, const char *);
static int procwait(void);
static void reportfsystem(struct fsystem *);
static u_long findlowmark(const char *);
static void statgroups(void);
__dead void usage(void) __attribute__((volatile));

/* Public data */
#ifdef HAVE__PROGNAME
extern char *__progname;
#else
char *__progname = "?";
#endif

int nflag;			/* non-destructive mode */
int feedback;			/* update the feedback file */
int lowmark;			/* update the lowmark file */
int debug;			/* debugging modes */
int verbose;			/* verbose information to stdout */
int force;			/* override saftey measures */

int nprocs;			/* number of subprocesses running */

/* External data */
extern char *optarg;
extern int optind;
extern int opterr;

extern char version[];

int
main(int argc, char **argv)
{
	register int op, status;
	register char *cp;
	register struct fsystem *fsp;
	register struct group *gl;
	register long n;
	register FILE *fin, *fout;
	char buf[512];

#ifndef HAVE__PROGNAME
	if ((cp = strrchr(argv[0], '/')) != NULL)
		__progname = cp + 1;
	else
		__progname = argv[0];
	__progname = savestr(__progname);
#endif

#ifndef HAVE_SETPROCTITLE
	initproctitle(argc, argv);
#endif

	/* Force line buffering on stdout */
#ifdef HAVE_SETLINEBUF
	setlinebuf(stdout);
#else
	setvbuf(stdout, NULL, _IOLBF, 0);
#endif

	/* Setup top level subdir */
	fsystemlist->dir = spool_dir;
	fsystemlist->subdir = "";
	fsystemlist->want_kb = DEFAULT_WANT_KB;

	/* Process arguments */
	status = 0;
	while ((op = getopt(argc, argv, "Fdlnuva:c:f:s:")) != EOF)
		switch (op) {

		case 'F':
			/* undocumented */
			++force;
			break;

		case 'a':
			active_file = optarg;
			break;

		case 'c':
			dexplist = optarg;
			break;

		case 'd':
			/* undocumented */
			++debug;
			break;

		case 'f':
			(void)strcpy(buf, optarg);
			cp = strchr(buf, ':');
			if (cp == NULL) {
				cp = optarg;
				n = atol(cp);
				buf[0] = '\0';
			} else {
				*cp++ = '\0';
				n = atol(cp);
			}
			if (n <= 0)
				errx(1, "\"%s\" invalid argument to -f",
				    optarg);
			/* Trailing 'm' means Mbytes */
			while (isdigit(*cp))
				++cp;
			if (*cp == 'm' || *cp == 'M')
				n *= 1024;

			fsp = findfsystem(0, buf);
			fsp->want_kb = n;
			break;

		case 'l':
			++lowmark;
			break;

		case 'n':
			++nflag;
			break;

		case 's':
			if (spool_dir != default_spool_dir)
				errx(1, "Must specify -s before -f");
			/* Make a copy: children can't read parent's args */
			spool_dir = savestr(optarg);
			fsystemlist->dir = spool_dir;
			break;

		case 'u':
			++feedback;
			break;

		case 'v':
			++verbose;
			break;

		default:
			usage();
		}

	if (optind != argc)
		usage();

	/* Fetch current time which is used in various calculations */
	currenttime = time(0);
	if (verbose) {
		msg("Version %s, patchlevel %d", version, PATCHLEVEL);
		msg("Current time: %s", fmtdate(currenttime));
	}

	/* Get dev number for toplevel filesystem (may not be a symlink) */
	fsystemlist->dev = devnumber(spool_dir, NULL, "spool_dir");

	/* Read in the dexplist config file */
	if (parseexplist())
		exit(1);

	/* Parse the active file */
	if (parseactive())
		exit(1);

	/* Get the per newsgroup info */
	statgroups();

	/* Assign newsgroups to specific filesystems */
	assigngroups();

	/* Delete articles, one process per filesystem */
	for (fsp = fsystemlist; fsp != NULL; fsp = fsp->next)
		procstart(fsp);

	/* Wait for all children to finish */
	while (nprocs > 0)
		status |= procwait();

	if (nflag) {
		if (verbose > 1) {
			if (feedback)
				msg("Skipping feedback file update");
			if (lowmark)
				msg("Skipping lowmark file update");
		}
		exit(status);
	}

	/* Update feedback file */
	if (feedback) {
		if (verbose > 1)
			msg("Updating feedback file");
		if ((fout = fopen(ndfeedback, "w")) == NULL)
			err(1, "fopen(%s, w)", ndfeedback);
		for (fsp = fsystemlist; fsp != NULL; fsp = fsp->next) {
			if ((fin = fopen(fsp->tfeedname, "r")) == NULL)
				err(1, "fopen(%s, r)", fsp->tfeedname);

			/* Copy temporary feedback file */
			while (fgets(buf, sizeof(buf), fin) != NULL)
				if (fputs(buf, fout) == EOF)
					err(1, "fputs %s", fsp->tfeedname);
			(void)fclose(fin);
			if (unlink(fsp->tfeedname) < 0)
				err(1, "unlink %s", fsp->tfeedname);
		}

		if (ferror(fout))
			err(1, "ferror(): %s:", ndfeedback);
		(void)fclose(fout);
		if (rename(ndfeedback, dfeedback) < 0 &&
		    unlink(dfeedback) < 0 &&
		    rename(ndfeedback, dfeedback) < 0)
			err(1, "rename(): %s -> %s", ndfeedback, dfeedback);
	}

	/* Create lowmark file */
	if (lowmark) {
		if (verbose > 1)
			msg("Creating lowmark file");
		if ((fout = fopen(dlowmark, "w")) == NULL)
			err(1, "fopen(%s, w)", dlowmark);

		for (fsp = fsystemlist; fsp != NULL; fsp = fsp->next) {
			if ((fin = fopen(fsp->tlowmark, "r")) == NULL)
				err(1, "fopen(%s, r)", fsp->tlowmark);

			/* Copy temporary lowmark file */
			while (fgets(buf, sizeof(buf), fin) != NULL)
				if (fputs(buf, fout) == EOF)
					err(1, "fputs %s", fsp->tlowmark);
			(void)fclose(fin);
			if (unlink(fsp->tlowmark) < 0)
				err(1, "unlink %s", fsp->tlowmark);
		}

		/* Also output lowmark info for groups with no articles */
		for (gl = grouplist; gl != NULL; gl = gl->next)
			if (gl->uplowmark) {
				(void)strcpy(buf, gl->ngn);
				for (cp = buf; *cp != '\0'; ++cp)
					if (*cp == '/')
						*cp = '.';
				fprintf(fout, "%s %ld\n", buf, gl->no);
			}

		if (ferror(fout))
			err(1, "ferror(): %s:", dlowmark);
		(void)fclose(fout);
	}

	exit(status);
}

/*
 * Find the spool filesystem with the specified dev number
 * (or specified subdir if the dev number is zero)
 */
static struct fsystem *
findfsystem(register dev_t dev, register char *subdir)
{
	register struct fsystem *fsp, *lastfsp;
	register char *cp;
	register dev_t tdev;
	char dir[512], tdir[512], link[512];

	lastfsp = fsystemlist;
	for (fsp = fsystemlist; fsp != NULL; fsp = fsp->next) {
		if ((dev != 0 && dev == fsp->dev) ||
		    strcmp(subdir, fsp->subdir) == 0)
			return (fsp);
		lastfsp = fsp;
	}

	/* Append a new entry to end of list */
	fsp = (struct fsystem *)mymalloc(sizeof(*fsp), "fsystem");
	memset((char *)fsp, 0, sizeof(*fsp));

	/* Find the mount point */
	(void)sprintf(dir, "%s/%s", spool_dir, subdir);
	tdev = devnumber(dir, link, "findfsystem");
	/* Copy back possible symlink */
	if (link[0] != '\0')
		(void)strcpy(dir, link);
	if (dev != 0 && dev != tdev)
		err(1, "findfsystem dev mismatch for %s (%d != %d)",
		    dir, (int)dev, (int)tdev);
	fsp->dev = tdev;
	(void)strcpy(tdir, dir);
	while ((cp = strrchr(tdir, '/')) != NULL) {
		if (cp == tdir) {
			/* Oops, we're at the root */
			(void)strcpy(dir, "/");
			break;
		}
		*cp = '\0';
		dev = devnumber(tdir, link, "findfsystem 2");
		if (dev != fsp->dev)
			break;
		/* Update directory */
		if (link[0] == '\0')
			(void)strcpy(dir, tdir);
		else
			(void)strcpy(dir, link);
	}

	if (strcmp(dir, spool_dir) == 0) {
		fsp->dir = spool_dir;
		fsp->subdir = "";
	} else {
		fsp->dir = savestr(dir);
		cp = strrchr(fsp->dir, '/');
		fsp->subdir = cp + 1;
	}
	fsp->want_kb = DEFAULT_WANT_KB;
	lastfsp->next = fsp;
	return (fsp);
}

/*
 * Return the device number of the specified file. If "link" is not NULL,
 * then allow symlinks and update "link" if the file is a symlink
 */
static dev_t
devnumber(register char *file, register char *link, register const char *what)
{
	register int cc;
	struct stat sbuf;
	char temp[512];

	if (link != NULL)
		*link = '\0';
	for (;;) {
		if (lstat(file, &sbuf) < 0)
			err(1, "%s lstat %s", what, file);
		if (S_ISLNK(sbuf.st_mode)) {
			if (link == NULL)
				err(1, "%s %s is a symlink", what, file);
			if ((cc = readlink(file, link, sizeof(temp) - 1)) < 0)
				err(1, "%s readlink %s", what, file);
			link[cc] = '\0';
			(void)strcpy(temp, link);
			file = temp;
			continue;
		}
		return (sbuf.st_dev);
	}
}

static int
parseexplist(void)
{
	register FILE *f;

	/* Read in the dexplist config file */
	if ((f = fopen(dexplist, "r")) == NULL) {
		warn("fopen(%s, r)", dexplist);
		return (1);
	}

	/* Build internal list of rules */
	if (!readexplist(f, nextrule)) {
		(void)fclose(f);
		return (1);
	}
	(void)fclose(f);
	return (0);
}

static int
parseactive(void)
{
	register FILE *f;

	if ((f = fopen(active_file, "r")) == NULL) {
		warn("fopen(%s, r)", active_file);
		return (1);
	}
	if (!readactive(f, newgroup)) {
		(void)fclose(f);
		return (1);
	}
	if (grouplist == NULL) {
		msg("No groups found");
		return (1);
	}
	(void)fclose(f);
	return (0);
}

/* Handle the next rule */
int
nextrule(register char *group, register char mod,
    register int pri, register int days)
{
	register struct rule *rp;
	register time_t minimum;
	static struct rule *lastrule = NULL;

	if (pri < 0)
		errx(1, "priority for group %s is negative (%d)", group, pri);
	if (days < 0)
		errx(1, "minimum for group %s is negative (%d)", group, days);
	minimum = days * 24 * 60 * 60;

	/* Save new rule */
	rp = (struct rule *)mymalloc(sizeof(*rp), "rule");
	memset((char *)rp, 0, sizeof(*rp));
	if (rulelist == NULL)
		rulelist = rp;
	else
		lastrule->next = rp;
	lastrule = rp;
	rp->ngn = savestr(group);
	rp->mod = mod;
	rp->pri = pri;
	rp->minimum = minimum;

	return (1);
}

#define GROUPSIZE 512		/* size to malloc when more space is needed */

/* Handle the next newsgroup from the active file */
static int
newgroup(register char *group, register u_long nn, register u_long no,
    register char mod)
{
	register struct group *gl;
	static struct group *groupptr = NULL;
	static u_int groupsize = 0;

	/* Build a new group entry */
	if (groupsize == 0) {
		groupsize = GROUPSIZE;
		gl = (struct group *)mymalloc(groupsize * sizeof(*gl), "group");
		memset((char *)gl, 0, groupsize * sizeof(*gl));
		groupptr = gl;
	}
	gl = groupptr++;
	--groupsize;

	gl->ngn = savestr(group);
	gl->no = no;
	gl->nn = nn;
	gl->mod = mod;
	gl->next = grouplist;
	grouplist = gl;

	return (1);
}

/* Fetch initial per-group timestamp info */
static void
statgroups(void)
{
	register struct group *gl;
	register time_t start, t;
	register u_long n, no, deleted;
	register int numstats, numgroups, numnotfound, numscandirs;
	register int didscandir, dontscandir;
	register char *art;
	register dev_t dev;
	char dir[512];
	struct stat sbuf;

	/* Get initial timestamp so we can report how long this took */
	start = time(0);
	numstats = 0;
	numgroups = 0;
	deleted = 0;
	numscandirs = 0;

	/* Determine timestamp of oldest article in each group */
	for (gl = grouplist; gl != NULL; gl = gl->next) {
		/*
		 * Change our working directory. Although this seems
		 * inefficient, it's about as expensive as checking to
		 * see if the newsgroup directory exists and can
		 * greatly reduce overhead if the third field of the
		 * active file isn't up to date; when that happens,
		 * we use readdir() to find the low article number.
		 * Opening and reading the current working directory
		 * is much faster because the inode is cached in
		 * the process header.
		 */
		(void)sprintf(dir, "%s/%s", spool_dir, gl->ngn);
		if (chdir(dir) < 0) {
			gl->to = currenttime;
			continue;
		}

		t = 0;
		dev = 0;
		art = "?";
		++numgroups;
		no = 0;
		didscandir = 0;
		dontscandir = 0;
		numnotfound = 0;
		n = gl->no;
		while (n <= gl->nn) {
			/* Construct article filename */
			art = long2str(n);
			++n;

			/* Skip if no file */
			++numstats;
			if (lstat(art, &sbuf) < 0) {
				/*
				 * If we are not finding articles,
				 * the lowmark in the active file
				 * is probably out of date for this
				 * newsgroup. This can happen if
				 * the system crashes, dexpire is
				 * killed before the lowmark info
				 * is updated or when somebody
				 * manually deletes some articles.
				 * In this case, use readdir() to
				 * determine the lowest article
				 * number and then resume looking
				 * for a timestamp.
				 */
				++numnotfound;
				if (numnotfound > 3 &&
				    !didscandir && !dontscandir) {
					if (verbose > 1)
						msg("Switching to readdir()"
						" for %s", gl->ngn);
					didscandir = 1;
					++numscandirs;
					n = findlowmark(".");
					if (n == 0) {
						no = gl->nn + 1;
						break;
					}
				}
				continue;
			}

			/*
			 * Avoid the readdir() code once we find
			 * an article (or symlink).
			 */
			dontscandir = 1;

			/* If a symlink, stat the actual article */
			if (S_ISLNK(sbuf.st_mode)) {
				/* Get the device number for this group */
				if (dev == 0) {
					++numstats;
					dev = devnumber(".", NULL,
					    "statgroups");
				}
				++numstats;
				if (stat(art, &sbuf) < 0) {
					/* Delete dangling symlink */
					if (nflag || unlink(art) >= 0)
						++deleted;
					continue;
				}
			}

			/* Number of oldest article (symlink or otherwise) */
			if (no == 0)
				no = n - 1;

			/* If we got a timestamp, we're done */
			t = sbuf.st_mtime;
			if (t != 0) {
				if (dev == 0)
					dev = sbuf.st_dev;
				break;
			}
		}

		/* Complain if from the first year of Unix history */
		if (t > 0 && t < 365 * 24 * 60 * 60)
			msg("Warning: %s/%s is old: %s",
			    gl->ngn, art, fmtdate(t));

		gl->dev = dev;
		if (gl->no != no && no != 0) {
			gl->uplowmark = 1;
			gl->no = no;
		}
		if (t == 0)
			gl->to = currenttime;
		else
			gl->to = t;
	}

	if (verbose) {
		msg("(Took %s to check %d file%s in %d group%s)",
		    fmtdelta(time(0) - start),
		    numstats, PLURAL(numstats),
		    numgroups, PLURAL(numgroups));
		if (numscandirs > 0)
			msg("(Manually scanned %d newsgroup%s)",
			    numscandirs, PLURAL(numscandirs));
		if (deleted > 0)
			msg("(%seleted %ld dangling symlink%s)",
			    nflag ? "Would have d" : "D",
			    deleted, PLURAL(deleted));
	}
}

/* Assign groups to specific filesystems */
static void
assigngroups(void)
{
	register struct fsystem *fsp, *fsp2;
	register struct group *gl, *glnext, *skipgl;
	register struct class *cl, *cl2;
	register struct rule *rp;
	register char *cp, *cp2;

	/* Spin through all groups */
	fsp = fsystemlist;
	skipgl = NULL;
	for (gl = grouplist; gl != NULL; gl = glnext) {
		glnext = gl->next;

		/* Skip groups without dev's, they have no articles */
		if (gl->dev == 0) {
			if (verbose > 2)
				msg("Skipping %s (no articles)", gl->ngn);
			gl->next = skipgl;
			skipgl = gl;
			continue;
		}

		/* First find the rule that applies to this group */
		for (rp = rulelist; rp != NULL; rp = rp->next) {
			/* Check moderated flag */
			if (rp->mod != gl->mod &&
			    rp->mod != 'x' &&
			    (rp->mod != 'u' || gl->mod != 'y'))
				continue;

			/* Check for a group match */
			cp = gl->ngn;
			cp2 = rp->ngn;
			while (*cp != '\0' && *cp == *cp2) {
				++cp;
				++cp2;
			}
			if (*cp2 == '\0' && (*cp == '\0' || *cp == '/'))
				break;

			/* "all" always matches */
			if (strcmp(rp->ngn, "all") == 0)
				break;
		}

		/* Gotta have a rule */
		if (rp == NULL) {
			msg("Warning: no default rule, discarding %s", gl->ngn);
			gl->next = skipgl;
			skipgl = gl;
			continue;
		}

		/* Ingore groups with priority 0 */
		if (rp->pri == 0) {
			if (verbose > 1)
				msg("Skipping %s (pri == 0)", gl->ngn);
			gl->next = skipgl;
			skipgl = gl;
			continue;
		}

		/* Find the filesystem this group is on */
		if (fsp->dev != gl->dev)
			fsp = findfsystem(gl->dev, gl->ngn);

		/* Look for existing class with the same priority and minimum */
		cl2 = NULL;
		for (cl = fsp->class; cl != NULL; cl = cl->next) {
			if (cl->pri == rp->pri && cl->minimum == rp->minimum)
				break;
			if (cl->pri < rp->pri ||
			    (cl->pri == rp->pri && cl->minimum < rp->minimum))
				cl2 = cl;
		}

		/* New class; insert in ascending order */
		if (cl == NULL) {
			cl = (struct class *)mymalloc(sizeof(*cl), "class");
			memset((char *)cl, 0, sizeof(*cl));
			if (cl2 != NULL) {
				cl->next = cl2->next;
				cl2->next = cl;
			} else {
				cl->next = fsp->class;
				fsp->class = cl;
			}
			cl->pri = rp->pri;
			cl->minimum = rp->minimum;
		}

		/* Find the "standard" (e.g. highest) class */
		for (cl2 = fsp->class; cl2 != NULL; cl2 = cl2->next)
			fsp->standardclass = cl2;

		/* Insert group at front of class list */
		gl->next = cl->group;
		cl->group = gl;
	}
	grouplist = skipgl;

	/* Now that we're done adding filesystems, look for duplicates */
	for (fsp = fsystemlist; fsp != NULL; fsp = fsp->next)
		for (fsp2 = fsp->next; fsp2 != NULL; fsp2 = fsp2->next)
			if (fsp2->dev == fsp->dev)
				err(1, "dup devs (%u \"%s\" vs. %u \"%s\")",
				    (u_int32_t)fsp2->dev, fsp2->subdir,
				    (u_int32_t)fsp->dev, fsp->subdir);
}

/* Calculate new dtime */
static void
newdtime(register struct fsystem *fsp)
{
	register long freed, need, used;
	double r;
	char buf[64];

	freed = BLK2KB(fsp->freed_blk);
	need = fsp->need_kb - freed;
	if (verbose > 1)
		msg("newdtime(): Need to free %ld Kbyte%s",
		    need, PLURAL(need));

	/* Calculate new dtime based on file system usage */
	used = fsp->used_kb - freed;
	if (used == 0)
		used = 1;
	r = ((double)need) / ((double)used);
	if (verbose > 1)
		msg("Need to used ratio is %.2f", r);
	fsp->dtime = (int)(fsp->deltatime * r * 0.25);
	if (verbose > 1)
		(void)strcpy(buf, fmtdelta(fsp->dtime));
	if (fsp->dtime < MIN_DTIME) {
		fsp->dtime = MIN_DTIME;
		if (verbose > 1)
			msg("New dtime is too short (%s), using %s",
			    buf, fmtdelta(fsp->dtime));
	} else if (fsp->dtime > MAX_DTIME) {
		fsp->dtime = MAX_DTIME;
		if (verbose > 1)
			msg("New dtime is too long (%s), using %s",
			    buf, fmtdelta(fsp->dtime));
	} else if (verbose > 1)
		msg("New dtime is %s", buf);
}

/* Make a pass over one newsgroup. Returns true if enough space was freed */
static int
expireone(register struct fsystem *fsp, register struct group *gl,
    register struct class *cl)
{
	register u_long n, deleted;
	register int status, blocks, pct;
	register time_t t;
	register char *art;
	char file[512];
	struct stat sbuf;

	/* Make sure there's something to do */
	if (gl->no > gl->nn) {
		if (verbose > 2)
			msg("No articles in %s", gl->ngn);
		return (0);
	}

	/* Cached article time might tell us there's nothing to do */
	if (gl->to >= cl->cutoff) {
		if (verbose > 2)
			msg("Cached time skip of %s", gl->ngn);
		return (0);
	}

	/* Change working directory to speed up directory lookups */
	(void)sprintf(file, "%s/%s", spool_dir, gl->ngn);
	if (chdir(file) < 0) {
		/* Skip next time and save the unsuccessful chdir() */
		gl->no = gl->nn + 1;
		gl->uplowmark = 1;
		gl->to = currenttime;
		msg("No directory %s", gl->ngn);
		return (0);
	}

	/* Nuke 'em, Dano! */
	if (verbose > 1)
		msg("Running %s", gl->ngn);

	status = 0;
	t = 0;
	deleted = 0;
	for (n = gl->no; n <= gl->nn; ++n) {
		/* Construct article filename */
		art = long2str(n);

		/* Skip if no file */
		t = 0;
		if (lstat(art, &sbuf) < 0)
			continue;

		blocks = sbuf.st_blocks;
		if (S_ISLNK(sbuf.st_mode)) {
			/* Symlinks use 0 or 2 blocks */
			if (blocks > 2)
				blocks = 2;

			/* If actual article does not exist, remove symlink */
			if (stat(art, &sbuf) < 0) {
				if (nflag || unlink(art) >= 0) {
					++deleted;
					fsp->freed_blk += blocks;
					cl->freed_blk += blocks;
				}
				/* Don't bother checking to see if we're done */
				continue;
			}
		}

		/* Complain if from the first year of Unix history */
		t = sbuf.st_mtime;
		if (t < 365 * 24 * 60 * 60)
			msg("Warning: %s/%s is old: %s",
			    gl->ngn, art, fmtdate(t));

		/* If not old enough, we're done with this group */
		if (t >= cl->cutoff)
			break;

		/* Only chalk up the article if it was the last link */
		if (nflag || unlink(art) >= 0) {
			if (sbuf.st_nlink <= 1) {
				++deleted;
				fsp->freed_blk += blocks;
				cl->freed_blk += blocks;
			}
		}

		if (BLK2KB(fsp->freed_blk) >= fsp->need_kb) {
			if (verbose > 1)
				msg("We're done!");
			status = 1;
			break;
		}
	}

	fsp->deleted += deleted;
	cl->deleted += deleted;
	if (verbose > 1)
		msg(" deleted %ld article%s (wanted to delete %ld)",
		    deleted, PLURAL(deleted), n - gl->no + 1);

	/* Process title hack */
	pct = (100 * BLK2KB(fsp->freed_blk)) / fsp->need_kb;
	if (pct > 100)
		pct = 100;
	if (*fsp->subdir == '\0')
		setproctitle("%d%% done", pct);
	else
		setproctitle("%d%% done (%s)", pct, fsp->subdir);

	/* Update oldest article and its timestamp */
	if (gl->no != n && n != 0) {
		gl->uplowmark = 1;
		gl->no = n;
	}
	if (t != 0)
		gl->to = t;
	else
		gl->to = currenttime;

	return (status);
}

/* Prepare filesystem data structures; returns desired exit() status */
static int
checkfsystem(register struct fsystem *fsp)
{
	register struct class *cl;
	register struct group *gl;
	long inodes, ifree;
	register double r;
	register time_t t, d, to;

	if (verbose) {
		msg("Spool directory is %s", fsp->dir);
		msg(" Want %ld Kbyte%s free",
		    fsp->want_kb, PLURAL(fsp->want_kb));
	}

	/* Update various disk parameters */
	if (disk_usage(fsp->dir, &fsp->used_kb, &fsp->free_kb,
	    &inodes, &ifree) < 0)
		err(1, "disk_usage(): %s", fsp->dir);

	/* Calculate how much space we need */
	fsp->need_kb = fsp->want_kb - fsp->free_kb;
	if (verbose)
		msg(" Have %ld Kbyte%s free",
		    fsp->free_kb, PLURAL(fsp->free_kb));

	/* Skip this one if we already have enough free space */
	if (fsp->need_kb <= 0) {
		msg(" Nothing to do!");
		return (0);
	}

	if (verbose)
		msg(" Need to free %ld Kbyte%s",
		    fsp->need_kb, PLURAL(fsp->need_kb));

	/* Complain if asked to free too much */
	r = ((double)fsp->want_kb) / (((double) fsp->free_kb) +
	    ((double)fsp->used_kb));
	if (r >= MAX_FREE && !nflag && !force) {
		warnx(" Cannot free %d%% of the filesystem (%d%% max)",
		    (int)(100.0 * r), (int)(100.0 * MAX_FREE));
		warnx(" (use -F to override or -n if testing)");
		fsp->need_kb = 0;
		return (1);
	}

	if (verbose && inodes > 0)
		msg(" (Spool directory averages %.1f Kbytes/inode)",
		    ((double)fsp->used_kb) / ((double)inodes));

	/* Determine standard time */
	fsp->standardtime = currenttime;
	fsp->to = currenttime;
	for (cl = fsp->class; cl != NULL; cl = cl->next) {
		/* Update oldest class times */
		to = currenttime;
		for (gl = cl->group; gl != NULL; gl = gl->next)
			if (to > gl->to) {
				to = gl->to;
				/* Update oldest filesystem times */
				if (fsp->to > to)
					fsp->to = to;
			}


		/* Safety check */
		if (to == 0) {
			msg(" Oldest %3d%%: has zero time!",
			    (100 * cl->pri) / fsp->standardclass->pri);
			continue;
		}

		/* Determine class delta time */
		d = currenttime - to;
		if (verbose)
			msg(" Oldest %3d%%: %s (delta %s)",
			    (100 * cl->pri) / fsp->standardclass->pri,
			    fmtdate(to), fmtdelta(d));

		/* Increase class delta by class ratio */
		d = (d * fsp->standardclass->pri) / cl->pri;

		/* Convert to absolute time */
		t = currenttime - d;
		if (verbose > 1)
			msg(" Expanded     %s (delta %s)",
			    fmtdate(t), fmtdelta(d));

		if (fsp->standardtime > t)
			fsp->standardtime = t;
	}
	return (0);
}

/*
 * Returns the number of the lowest article file in the directory
 * (or zero if none were found).
 */
static u_long
findlowmark(register const char *dir)
{
	char *cp;
	register DIR *dp;
	register struct dirent *ep;
	register u_long n, low;

	low = 0;
	if ((dp = opendir(dir)) != NULL) {
		while ((ep = readdir(dp)) != NULL) {
			n = strtol(ep->d_name, &cp, 10);
			if (*cp != '\0')
				continue;
			if (low > n || low == 0)
				low = n;
		}
		(void)closedir(dp);
	}

	return (low);
}

static char *
proctmpname(register const char *base, register const char *subdir)
{
	register char *cp;
	static char file[512];

	(void)strcpy(file, base);
	cp = file + strlen(file);
	if (*subdir != '\0') {
		*cp++ = '-';
		(void)strcpy(cp, subdir);
		for (; *cp != '\0'; ++cp)
			if (*cp == '/')
				*cp = '.';
	}
	(void)strcpy(cp, ".tmp");
	return (file);
}

/* Start a subprocess to work on one filesystem */
static void
procstart(register struct fsystem *fsp)
{
	register pid_t pid;
	register int status;
	register char *cp;
	register struct class *cl;
	register struct group *gl;
	register FILE *f;
	char ngn[512];
	static char progname[32];

	/* Setup temporary feedback and lowmark filename before forking */
	if (feedback)
		fsp->tfeedname = savestr(proctmpname(dfeedback, fsp->subdir));
	if (lowmark)
		fsp->tlowmark = savestr(proctmpname(dlowmark, fsp->subdir));

	fflush(stdout);
	fflush(stderr);
	pid = fork();
	if (pid < 0)
		err(1, "procstart %s", fsp->dir);

	/* Parent */
	if (pid != 0) {
		fsp->pid = pid;
		++nprocs;
		return;
	}

	/* Child */
	(void)sprintf(progname, "%s[%ld]", __progname, (long)getpid());
	__progname = progname;

	/* Fetch filesystem info */
	status = checkfsystem(fsp);
	if (fsp->need_kb > 0) {
		/* Expire the articles */
		msg("Expiring %s", fsp->dir);
		status |= dexpire(fsp);
		if (verbose)
			reportfsystem(fsp);
	}

	/* Update the feedback file, if necessary */
	if (feedback && !nflag) {
		if ((f = fopen(fsp->tfeedname, "w")) == NULL)
			err(1, "fopen(%s, w)", fsp->tfeedname);
		for (cl = fsp->class; cl != NULL; cl = cl->next)
			for (gl = cl->group; gl != NULL; gl = gl->next)
				(void)fprintf(f, "%d\t%s\n",
				    (int32_t)gl->to, gl->ngn);
		if (ferror(f))
			err(1, "ferror(): %s", fsp->tfeedname);
		(void)fclose(f);
	}

	/* Update the lowmark file, if necessary */
	if (lowmark) {
		if ((f = fopen(fsp->tlowmark, "w")) == NULL)
			err(1, "fopen(%s, w)", fsp->tlowmark);
		for (cl = fsp->class; cl != NULL; cl = cl->next)
			for (gl = cl->group; gl != NULL; gl = gl->next)
				if (gl->uplowmark) {
					(void)strcpy(ngn, gl->ngn);
					for (cp = ngn; *cp != '\0'; ++cp)
						if (*cp == '/')
							*cp = '.';
					fprintf(f, "%s %ld\n", ngn, gl->no);
				}
		if (ferror(f))
			err(1, "ferror(): %s", fsp->tlowmark);
		(void)fclose(f);
	}

	msg("Finished %s", fsp->dir);
	fflush(stdout);
	fflush(stderr);
	exit(status);
}

/* Wait for subprocesses to exit; return their exist status and update nprocs */
static int
procwait(void)
{
	register pid_t pid;
	register struct fsystem *fsp;
	DECLWAITSTATUS status;

	while ((pid = wait(&status)) != 0) {
		if (pid == -1) {
			if (errno == ECHILD)
				break;
			if (errno == EINTR)
				continue;
			err(1, "wait");
		}
		for (fsp = fsystemlist; fsp != NULL; fsp = fsp->next) {
			if (fsp->pid == pid) {
				--nprocs;
				fsp->pid = 0;
				return (WEXITSTATUS(status));
			}
		}
		msg("reaped unknown process %ld", (long)pid);
	}
	return (0);
}

/* Single filesystem article deletion routine; returns desired exit() status */
static int
dexpire(register struct fsystem *fsp)
{
	register struct class *cl;
	register struct group *gl;
	register time_t d;
	register u_long m, n, olddeleted;
	register int pass;
	register long oldfreed_blk;
	int done;

	/*
	 * We expire the lowest priority classes first.
	 * After each pass, we lower the "standard" time and run again.
	 */
	fsp->starttime = time(0);
	fsp->deltatime = currenttime - fsp->standardtime;
	pass = 0;
	done = 0;
	while (!done) {
		/* Internal book keeping */
		olddeleted = fsp->deleted;
		oldfreed_blk = fsp->freed_blk;
		++pass;

		/* Update dtime */
		newdtime(fsp);

		/* Increase standard time */
		fsp->standardtime += fsp->dtime;
		fsp->deltatime = currenttime - fsp->standardtime;
		if (fsp->deltatime <= 0) {
			msg("Ran out of articles! (dtime is %s)",
			    fmtdelta(fsp->dtime));
			return (1);
		}
		if (verbose > 1)
			msg("Delta time: %s", fmtdelta(fsp->deltatime));

		/* Work from the lowest class up */
		for (cl = fsp->class; !done && cl != NULL; cl = cl->next) {
			/* Calculate cutoff for this class */
			d = (time_t)(((double)fsp->deltatime *
			    (double)cl->pri) / (double)fsp->standardclass->pri);

			/* Skip entire class if we're reached the minimum */
			if (cl->minimum >= d) {
				if (verbose > 1)
					msg("Skip   %3d%%: %s (%s)",
					    (100 * cl->pri) /
						fsp->standardclass->pri,
					    fmtdate(currenttime - cl->minimum),
					    fmtdelta(cl->minimum));
				continue;
			}

			/* Update class cutoff time */
			cl->cutoff = currenttime - d;

			if (verbose > 1)
				msg("Cutoff %3d%%: %s (%s)",
				    (100 * cl->pri) / fsp->standardclass->pri,
				    fmtdate(cl->cutoff), fmtdelta(d));

			/* Run through the groups */
			for (gl = cl->group; !done && gl != NULL; gl = gl->next)
				done = expireone(fsp, gl, cl);
		}

		m = fsp->deleted - olddeleted;
		if (verbose > 1) {
			n = BLK2KB(fsp->freed_blk - oldfreed_blk);
			msg("Pass %d %sdeleted %ld article%s (%ld Kbyte%s)",
			    pass, nflag ? "would have " : "",
			    m, PLURAL(m), n, PLURAL(n));
		}
	}
	return (0);
}

/* Report per-filesystem statistics after expiring articles */
static void
reportfsystem(register struct fsystem *fsp)
{
	register struct class *cl;
	register struct group *gl;
	register int i;
	register time_t d, to;
	register char *cp;

	/* Report final class times */
	for (cl = fsp->class; cl != NULL; cl = cl->next)
		msg("Final  %3d%%: %s (delta %s)",
		    (100 * cl->pri) / fsp->standardclass->pri,
		    fmtdate(cl->cutoff), fmtdelta(currenttime - cl->cutoff));

	/* Report per-class deletion statistics */
	for (cl = fsp->class; cl != NULL; cl = cl->next) {
		msg("Class %3d%% %seleted %ld article%s (%ld Kbyte%s)",
		    (100 * cl->pri) / fsp->standardclass->pri,
		    nflag ? "Would have d" : "D",
		    cl->deleted, PLURAL(cl->deleted),
		    BLK2KB(cl->freed_blk), PLURAL(BLK2KB(cl->freed_blk)));

		if (cl->minimum) {
			i = cl->minimum / (24 * 60 * 60);
			msg(" (%d day%s)", i, PLURAL(i));
		}
	}

	msg("%seleted %ld article%s total (%ld Kbyte%s), took %s",
	    nflag ? "Would have d" : "D",
	    fsp->deleted, PLURAL(fsp->deleted),
	    BLK2KB(fsp->freed_blk), PLURAL(BLK2KB(fsp->freed_blk)),
	    fmtdelta(time(0) - fsp->starttime));

	/* Determine new oldest article time */
	to = currenttime;
	for (cl = fsp->class; cl != NULL; cl = cl->next)
		for (gl = cl->group; gl != NULL; gl = gl->next)
			if (to > gl->to)
				to = gl->to;

	d = currenttime - fsp->to;
	msg("Old oldest %s (%s)", fmtdate(fsp->to), fmtdelta(d));
	d = currenttime - to;
	msg("New oldest %s (%s)", fmtdate(to), fmtdelta(d));

	d = fsp->to - to;
	cp = "gained";
	if (d <= 0) {
		d = -d;
		cp = "lost";
	}
	msg("Article delta %s %s", cp, fmtdelta(d));
}

__dead void
usage(void)
{

	(void)fprintf(stderr, "Version %s\n", version);
	(void)fprintf(stderr,
"usage: %s [-lnuv] [-a active] [-c dexplist] [-s spool_dir] [-f Kbytes]\n",
	    __progname);
	exit(1);
}
