/*
 * Copyright (c) 1991, 1992, 1996, 1997
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static const char copyright[] =
    "@(#) Copyright (c) 1991, 1992, 1996, 1997\n\
The Regents of the University of California.  All rights reserved.\n";
static const char rcsid[] =
    "@(#) $Header: runexp.c,v 1.18 97/06/23 03:05:04 leres Exp $ (LBL)";
#endif

/*
 * runexp - process explicit expires
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/timeb.h>

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include <err.h>

#include "dexpire.h"
#include "util.h"

/* Converts 512 byte blocks to Kbytes */
#define BLK2KB(b) ((b) / 2)

/* Private data */
static char *expires = EXPIRES;
static char *nexpires = NEXPIRES;
static char *spool_dir = SPOOL_DIR;

static time_t currenttime;	/* current timestamp */
#ifdef CNEWS
static struct timeb currenttimeb;
#endif

/* Public data */
#ifdef HAVE__PROGNAME
extern char *__progname;
#else
char *__progname;
#endif

/* Forwards */
static time_t getnumdate(char *);
int runexp(FILE *, FILE *);
int main(int, char **);

int debug = 0;			/* debugging modes */
int verbose = 0;		/* verbose information to stdout */
int nflag = 0;			/* non-destructive mode */

/* Statistics */
int bad = 0;
int ndeleted = 0;		/* number of articles deleted */
int emitted = 0;		/* number of lines written to new file */
static u_int freed_blk = 0;	/* number of 512 bytes blocks freed */

/* External data */

extern char *optarg;
extern int optind;
extern int opterr;

int
main(int argc, char **argv)
{
	register int op;
#ifndef HAVE__PROGNAME
	register char *cp;
#endif
	register FILE *fin, *fout;
	int status;
	char *usage = "usage: %s [-dnv] [expires]";

#ifndef HAVE__PROGNAME
	if ((cp = strrchr(argv[0], '/')) != NULL)
		__progname = cp + 1;
	else
		__progname = argv[0];
#endif

	/* Process arguments */
	while ((op = getopt(argc, argv, "dnv")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'n':
			++nflag;
			break;

		case 'v':
			++verbose;
			break;

		default:
			errx(1, usage, __progname);
		}

	/* Optional expires file */
	if (optind < argc) {
		expires = argv[optind];
		++optind;
	}

	if (optind != argc)
		errx(1, usage, __progname);

	/* Fetch current time (used in various calculations) */
#ifdef CNEWS
	ftime(&currenttimeb);
	currenttime = currenttimeb.time;
#else
	currenttime = time(0);
#endif

	/* Report various times */
	if (verbose)
		msg("Current time: %s", fmtdate(currenttime));

	/* Open the expires file */
	if ((fin = fopen(expires, "r")) == NULL)
		err(1, "fopen(): %s", expires);

	/* Create the new expires file */
	(void) unlink(nexpires);
	if ((fout = fopen(nexpires, "w")) == NULL)
		err(1, "fopen(): %s", nexpires);

	/* Process the expires file */
	status = runexp(fin, fout);

	/* Check for errors and close the files */
	if (ferror(fin)) {
		warn("fopen(): %s", expires);
		status |= 1;
	}
	if (ferror(fout)) {
		warn("fopen(): %s", nexpires);
		status |= 1;
	}

	(void)fclose(fin);
	(void)fclose(fout);

	/* Rename the explicit expires file */
	if (status == 0 && !nflag &&
	    rename(nexpires, expires) < 0 &&
	    unlink(expires) < 0 &&
	    rename(nexpires, expires) < 0) {
		warn("rename(): %s -> %s", nexpires, expires);
		status |= 1;
	}

	if (verbose) {
		msg("%d bad", bad);
		msg("%d emitted", emitted);

		msg("%seleted %d article%s (%d Kbyte%s), took %s",
		    nflag ? "Would have d" : "D",
		    ndeleted, PLURAL(ndeleted),
		    BLK2KB(freed_blk), PLURAL(BLK2KB(freed_blk)),
		    fmtdelta(time(0) - currenttime));
	}

	exit(status);
}

int
runexp(register FILE *fin, register FILE *fout)
{
	register char *cp, *cp2;
	register int n;
	register time_t t;
	register char *art;
	char line[1024];
	char stamp[128];
	struct stat sbuf;
	char *p = "expires file";

	n = 1;

	while (fgets(line, sizeof(line), fin)) {
		/* Step over message id */
		cp = strchr(line, '\t');
		if (cp == NULL) {
			warnx("%s missing message id on line %d", p, n);
			++bad;
			/* Eat this expires line */
			continue;
		}

		/* Find arrival/expires seperator */
		cp = strchr(cp, '~');
		if (cp == NULL) {
			warnx("%s missing expires date on line %d", p, n);
			++bad;
			/* Eat this expires line */
			continue;
		}

		/* Step over '~' */
		++cp;

		/* Look for explicit expires time and date */
		/* Better have an explicit timestamp */
		if (*cp == '-') {
			warnx(
			    "%s missing explicit expires timestamp on line %d",
			    p, n);
			++bad;
			/* Eat this expires line */
			continue;
		}

		/* Find newsgroups while copying timestamp out */
		cp2 = stamp;
		while (*cp != '\t' && *cp != '\0')
			*cp2++ = *cp++;
		*cp2 = '\0';

		/* Better be some newsgroups */
		if (*cp == '\0') {
			warnx("%s missing newsgroups on line %d", p, n);
			++bad;
			/* Eat this expires line */
			continue;
		}

		/* Step over tab */
		++cp;

		/* Terminate possible INN style date ("arrive~expire~posted") */
		cp2 = strchr(stamp, '~');
		if (cp2)
			*cp2 = '\0';

		/* Parse expires timestamp; try for numeric date first (INN) */
		t = getnumdate(stamp);
#ifdef CNEWS
		/* Try to parse timestamp using C News routine */
		if (t < 0)
			t = getdate(stamp, &currenttimeb);
#endif
#ifdef BNEWS
		/* Try to parse timestamp using B News routine */
		/* XXX how the heck do we do that? */
#endif
		if (t < 0) {
			/* Extract message id */
			cp2 = strchr(line, '\t');
			if (cp2)
				*cp2 = '\0';
			warnx("Bad date \"%s\" in %s on line %d",
			    stamp, line, n);
			++bad;
			/* Eat this expires line */
			continue;
		}

#ifdef notdef
		/* Complain if from the first year of Unix history */
		if (t < 365 * 24 * 60 * 60)
			warnx("Warning: art is old: %s", fmtdate(t));
#endif

		/* Keep this entry if it hasn't expired yet */
		if (t > currenttime)
			goto emit;

		/* Loop through newsgroup(s) */
		while (*cp != '\0') {
			/* Construct article pathname */
			art = artpath(spool_dir, cp);

			/* Step over this article */
			while (*cp != '\0' && !isspace(*cp))
				++cp;
			while (isspace(*cp))
				++cp;

			if (stat(art, &sbuf) < 0) {
				/* It's ok if it's already gone */
				if (errno != ENOENT) {
					warn("stat(): %s", art);
					/* Eat this expires line */
					continue;
				}
				if (lstat(art, &sbuf) < 0) {
					/* Ok if there is no symlink */
					if (errno != ENOENT)
						warn("lstat(): %s", art);
					/* Eat this expires line */
					continue;
				}
				/*
				 * If stat() fails but lstat() works it
				 * pretty much has to be a symlink...
				 */
				if (!S_ISLNK(sbuf.st_mode))
					errx(1, "stat/lstat \"can't happen\"");
				/* fall through and delete dangling symlink  */
			}
			if (nflag) {
				if (debug > 1)
					msg("wanted to unlink(%s)", art);
			} else if (unlink(art) < 0) {
				warn("unlink(): %s", art);
				/* Eat this expires line */
				continue;
			}
			if (sbuf.st_nlink <= 1) {
				++ndeleted;
				freed_blk += sbuf.st_blocks;
			}
		}

		/* Eat this expires line */
		continue;

emit:
		/* Output current expires line */
		fputs(line, fout);
		n++;
	}
	emitted = n - 1;
	return (0);
}

static time_t
getnumdate(register char *date)
{
	register int n;
	register time_t t;
	register char *cp;

	/* Must be all digits and not too long */
	n = 0;
	for (cp = date; isdigit(*cp); ++cp)
		++n;

	if (n > 10 || *cp != '\0')
		return ((time_t)-1);

	t = (time_t)atol(date);

	/* Not a numeric date if from the first year of Unix history */
	if (t < 365 * 24 * 60 * 60)
		return ((time_t)-1);

	return (t);
}
