:
#! /bin/sh
#
# This program compares the files in a subtree to a master list
# and mails a listing of new files, deleted files, and files in
# which any of the protection mode, number of links, user, group,
# size, or time of last modification has changed to a set of users.
# Specific files and pathnames may be exempted from this audit
# procedure.
#
# Options:
#
# -1	send output to standard output; do not mail it to the user
# -a<name1>,<name2>,...
#	Copies of the audit report will be mailed to <name1>,
#	<name2>, ...
# -C	Checksum all files in the audit.
# -c<file>
#	Checksum the files named in <file>.
# -d	Only audit the top-level directories; do not recurse.
# -e<file>
#	Take the environment of the remote host from <file>.
# -f<file>
#	Audit the tree with root <file>; the default is /usr/unsupported.
# -g	Generate the file against which future audits are to be compared;
#	do not do an audit.
# -ga	After the audit is done, replace the master file with the results
#	of the file system scan.
# -h<host>
#	Run the audit for host <host>.  A listing will be generated
#	remotely, moved to the local host, and the analysis done locally.
# -i<file>
#	Ignore audit lines with patterns in <file>.  This is done by
#	using egrep to match lines whose first field does not match
#	the patterns.
# -m<file>
#	Compare the results of the audit with the list in <file>.  If run
#	with -g, the results of the audit are saved in this file.
# -n<type>
#	If any of the file systems to be scanned are mounted on a volume
#	of type <type>, do not perform the scan.
# -o<type>
#	If any of the file systems to be scanned are not mounted on a volume
#	of type <type>, do not perform the scan.  This is the inverse of -n.
# -p<file>
#	Same as -m but the comparison is done using patterns.  This file
#	is never overwritten.
# -r<file>
#	Do not notify the auditors of any changes, additions, or deletions
#	to the files listed in <file>; if <file> has not been modified
#	for the past LIMIT days, delete it
# -sb	Confine the audit to block special device files only
# -sc	Confine the audit to character special device files only
# -sg	Confine the audit to setgid files only
# -su	Confine the audit to setuid files only
# -sG	Confine the audit to non-directory setgid files only (necessary
#	hack for Suns running SunOS 4)
# -t<seconds>
#	Time the file listing command out after <seconds> seconds if there
#	has been no output for that long.
# -u	With this option, every user who owns a file that has been added,
#	deleted, or changed will be mailed a message describing the change.
#	This message looks like an audit report.  If a file's ownership
#	changes, both the new and old owner will be told about it.
# -v	Debug mode; print shell input lines as they are read
# -x	Debug mode; print commands and their arguments as they are executed.
# -z<file>
#	The intermediate master file name.  This is used mainly in running
#	remote audits as a recovery mechanism.
# 
# Matt Bishop
# Research Institute for Advanced Computer Science
# NASA Ames Research Center
# Moffett Field, CA  94035
#
# mab@riacs.edu, ...!{decvax!decwrl,ihnp4!ames}!riacs!mab
#
# (c) Copyright 1986 by Matt Bishop and the Research Institute for
#			Advanced Computer Science
#
# version number
VERSION="RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)"
#
#	system and shell variables
PATH=/usr/ucb:/bin:/usr/bin:%%DESTBIN%%	# execution path
export PATH				# for subprograms
#	auditing parameters
FILESYS=%%FILESYS%%		# file system to be examined
AUDITOR=%%AUDITOR%%		# who to send reports to
TMPDIR=%%TMPDIR%%		# directory for temporary files
BINDIR=%%DESTBIN%%		# directory where programs and info are kept
THRESH=%%THRESH%%		# threshhold to question audit result
MINCT=%%MINCT%%			# ignore threshhold if fewer than this
				#   many files are involved
DEBUG=				# assume no debugging
#	auditing subprograms
COMMER=$BINDIR/auditcomm	# comm(1) with wildcards
LISTER=$BINDIR/lstat		# lists the file information for the audit
TYPER=$BINDIR/ftype		# prints files of specific type
GREPER=$BINDIR/auditgrep	# greps for strings, not patterns
MAILER=$BINDIR/auditmail	# builds mail message and sends it
MERGER=$BINDIR/auditmerge	# merge old, new master file
MATCHER=$BINDIR/auditpnm	# pattern matcher
NETSEND=/bin/cat		# send over network
NETREC=/bin/cat			# read from network
PREFER=$BINDIR/auditpref	# strips prefixes etc.
TIMER=$BINDIR/timeout		# times out file lister after this long
#	ignore and master files
LISTDIR=%%LISTDIR%%		# directory for master lists
CHKSUM=$LISTDIR/audit.chk	# compare against this audit listing
MASTER=$LISTDIR/audit.lst	# compare against this audit listing
PATTRN=$LISTDIR/audit.pat	# compare against this audit listing
EXCEPT=$LISTDIR/audit.ign	# ignore these files/directories
TMPIGN=$LISTDIR/audit.rej	# ignore these files/directories for now
#	shell variables for options
CHECKSPEC=no			# no checksum file specified
CHECKSUMALL=no			# checksum all audited files
ESTATUS=0			# exit code (1 if output or letter sent)
GENAUDIT=yes			# do an audit
GENMASTER=no			# generate master list
GENUSRRPT=no			# mail a report to each user too
LOCALHOST=`hostname | sed 's/\..*//'`	# get current host name
HOST="$LOCALHOST"			# audit the current host
LIMIT=%%LIMIT%%			# timeout for Reject files
ONEOUTPUT=no			# output into a letter
OPTLIST=			# list of options (for user information)
RECURSE=yes			# audit the subtree
SENTMAIL=no			# yes if changes mailed to AUDITOR
TYPEOPTS= 			# types of files (see ftype(1))
TIMEOUT=%%TIMEOUT%%		# timeout period
#	remote audit goodies
R_INFO=audit.env		# file containing remote host environment
R_AUDITLS=$BINDIR/auditls	# auditls(8) program
R_CAT=/bin/cat			# cat(1) program
R_CHECKSUM=			# checksum program
R_DELETE=/bin/rm		# rm(1) program
R_ECHO=/bin/echo		# echo(1) program
R_EDIT=/bin/sed			# sed(1) program
R_EXEC=%%REMEXEC%%		# rsh(1) program
R_FIND=$BINDIR/auditscan	# file system scanning program
R_LSTAT=$BINDIR/lstat		# lstat(1) program
R_MOUNT=/etc/mount		# mount(8) program
R_PPND=$BINDIR/auditppnd	# auditppnd(1) program
R_RECV=/bin/cat			# remote read program
R_SEND=/bin/cat			# remote send program
R_TEST=%%TEST%%			# test(1) program
R_USER=				# who to run as remotely
#	temorary file names
TMPX=$TMPDIR/Au$$0		# general-purpose temporary file
TMPY=$TMPDIR/Au$$1		# general-purpose temporary file
OLD=$TMPDIR/Au$$2		# lines in MASTER and not in new audit
NEW=$TMPDIR/Au$$3		# lines not in MASTER but in new audit
OLDNM=$TMPDIR/Au$$4		# list of file names in OLD
NEWNM=$TMPDIR/Au$$5		# list of file names in NEW
NEWFILES=$TMPDIR/Au$$6		# audit info for files created since MASTER
OLDFILES=$TMPDIR/Au$$7		# audit info for files deleted since MASTER
CHANGES=$TMPDIR/Au$$8		# audit info for files changed since MASTER
PATCHANGES=$TMPDIR/Au$$8a
UOLD=$TMPDIR/Au$$9		# like OLDFILES but for one user
UNEW=$TMPDIR/Au$$a		# like NEWFILES but for one user
UCHG=$TMPDIR/Au$$b		# like CHANGES but for one user
NMAS=$TMPDIR/Au$$c		# new master file
OMIT=$TMPDIR/Au$$d		# sed commands to omit named files
SKIPTYPE=$TMPDIR/Au$$e		# list of volume types to skip
DOTYPE=$TMPDIR/Au$$f		# list of volume types to do only
TMPZ=$TMPDIR/Au$$g		# general-purpose temporary file
				# list of temp file names
FILELIST="${TMPX}* ${TMPY}* ${TMPZ}* $OLD $NEW $OLDNM $NEWNM $NEWFILES $OLDFILES $CHANGES $PATCHANGES $UOLD $UNEW $UCHG $OMIT $SKIPTYPE $DOTYPE"
#
# trap some signals
#
trap "rm -f $FILELIST $NMAS; exit 2" 1 2 3 15
#
#===================== STEP 1.  PARSE THE COMMAND LINE =====================
#
FLAG=				# what comes next
for i
do
	#
	# if you are expecting an argument, pick it up
	#
	if test -n "$FLAG"
	then
		case $FLAG in
		a)	AUDITOR=`echo $i | tr ',' ' '`;;	# audit names
		m)	CHKSUM=$i;;	# checksum file name expected
		e)	R_INFO=$i;;	# remote environment file name
		f)	FILESYS=$i;;	# file system expected
		h)	HOST=$i;;	# host name expected
		i)	EXCEPT=$i;;	# ignore file name expected
		m)	MASTER=$i;;	# master file name expected
		p)	PATTRN=$i;;	# pattern master file name expected
		r)	TMPIGN=$i;;	# temp ignore file name expected
		t)	TIMEOUT=$i;;	# time out expected
		n)	echo $i | tr ',' '\012' >> $SKIPTYPE;;	# skip type
		o)	echo $i | tr ',' '\012' >> $DOTYPE;;	# do type
		z)	NMAS=$i;;	# intermediate master file name expected
		esac
		FLAG=
		continue
	fi
	#
	# flag
	#
	case $i in
	-1)				# output to standard output
		ONEOUTPUT=yes;;
	-a*)				# auditors
		AUD=`expr "$i" : '-a\(.*\)'`
		if test -z "$AUD"
		then
			FLAG=a
		else
			AUDITOR="$AUD"
		fi;;
	-C)				# checksum all audited files
		CHECKSUMALL=yes;;
	-c*)				# checksum file name
		CHECKSPEC=yes
		CHK=`expr "$i" : '-c\(.*\)'`
		if test -z "$MAS"
		then
			FLAG=c
		else
			CHKSUM=$CHK
		fi;;
	-d)				# do not recurse down the tree
		RECURSE=no;
		OPTLIST="$OPTLIST,nonrecursive";;
	-e*)				# remote environment file name
		ENV=`expr "$i" : '-e\(.*\)'`
		if test -z "$ENV"
		then
			FLAG=e
		else
			R_INFO=$ENV
		fi;;
	-f*)				# file system
		FIL=`expr "$i" : '-f\(.*\)'`
		if test -z "$FIL"
		then
			FLAG=f
		else
			FILESYS=$FIL
		fi;;
	-g)				# unconditionally generate master file
		GENMASTER=yes;
		GENAUDIT=no;
		OPTLIST="$OPTLIST,update-master-list";;
	-ga)				# replace master file with audit result
		GENMASTER=yes;
		OPTLIST="$OPTLIST,update-master-list";;
	-h*)				# run audit for this host
		HST=`expr "$i" : '-h\(.*\)'`
		if test -z "$HST"
		then
			FLAG=h
		else
			HOST=$HST
		fi;;
	-i*)				# ignore file name
		IGN=`expr "$i" : '-i\(.*\)'`
		if test -z "$IGN"
		then
			FLAG=i
		else
			EXCEPT=$IGN
		fi;;
	-m*)				# master file name
		MAS=`expr "$i" : '-m\(.*\)'`
		if test -z "$MAS"
		then
			FLAG=m
		else
			MASTER=$MAS
		fi;;
	-n*)				# skip volume type
		X=`expr "$i" : '-n\(.*\)'`
		if test -z "$X"
		then
			FLAG=n
		else
			echo $X | tr ',' '\012' >> $SKIPTYPE
		fi;;
	-o*)				# do only volume type
		X=`expr "$i" : '-o\(.*\)'`
		if test -z "$X"
		then
			FLAG=o
		else
			echo $X | tr ',' '\012' >> $DOTYPE
		fi;;
	-p*)				# pattern master file name
		PAT=`expr "$i" : '-p\(.*\)'`
		if test -z "$PAT"
		then
			FLAG=p
		else
			PATTRN=$PAT
		fi;;
	-r*)				# name of temporary ignore file
		TIM=`expr "$i" : '-r\(.*\)'`
		if test -z "$TIM"
		then
			FLAG=r
		else
			TMPIGN=$TIM
		fi;;
	-sG)				# audit setgid non-directory files only
		TYPEOPTS=${TYPEOPTS}G;
		OPTLIST="$OPTLIST,setgid-nondirectories";;
	-sb)				# audit block special device files
		TYPEOPTS=${TYPEOPTS}b;
		OPTLIST="$OPTLIST,block-devices";;
	-sc)				# audit character special device files
		TYPEOPTS=${TYPEOPTS}c;
		OPTLIST="$OPTLIST,character-devices";;
	-sg)				# audit setgid files only
		TYPEOPTS=${TYPEOPTS}g;
		OPTLIST="$OPTLIST,setgid";;
	-su)				# audit setuid non-directory files only
		TYPEOPTS=${TYPEOPTS}u;
		OPTLIST="$OPTLIST,setuid";;
	-t*)				# time out after this
		TIM=`expr "$i" : '-t\(.*\)'`
		if test -z "$TIM"
		then
			FLAG=t
		else
			TIMEOUT=$TIM
		fi;;
	-u)				# notify appropriate users of problems
		GENUSRRPT=yes;
		OPTLIST="$OPTLIST,notify-users";;
	-v) 				# turn on verbose mode
		DEBUG="${DEBUG}v";
		set -$DEBUG;;
	-x) 				# turn on debugging mode
		DEBUG="${DEBUG}x";
		set -$DEBUG;;
	-z*)				# intermediate master file name
		TMAS=`expr "$i" : '-z\(.*\)'`
		if test -z "$TMAS"
		then
			FLAG=z
		else
			NMAS=$TMAS
		fi;;
	*)				# unknown option
		echo "$0: unknown option $i" 1>&2
		exit 1;;
	esac
done
#
# set up the debug option completely
#
if test "X$DEBUG" != X
then
	DEBUG="-$DEBUG"
fi
#
# set up the type option completely
#
if test -n "$TYPEOPTS"
then
	TYPEOPTS="-s$TYPEOPTS"
fi
#
# be sure all expected arguments are there
#
if test -n "$FLAG"
then
	case $FLAG in
	a)	BANNER="expecting list of people to mail audit to";;
	c)	BANNER="expecting file listing files to checksum";;
	e)	BANNER="expecting file of remote program names";;
	f)	BANNER="expecting name of root of tree to be audited";;
	h)	BANNER="expecting name of host to audit";;
	i)	BANNER="expecting file of pathnames to ignore";;
	m)	BANNER="expecting file to compare audit against";;
	n)	BANNER="expecting type of volume not to audit file systems on";;
	o)	BANNER="expecting type of volume to audit file systems on";;
	p)	BANNER="expecting pattern file to compare audit against";;
	r)	BANNER="expecting file of pathnames to ignore temporarily";;
	t)	BANNER="expecting time out interval compare audit against";;
	z)	BANNER="expecting name of new master file";;
	esac
	echo "$0: $BANNER" 1>&2
	exit 1

fi
#
# if a checksum file is named and all files are to be
# checksummed, report this and zap the checksum file
#
if test "$CHECKSUMALL" = yes -a "$CHECKSPEC" = yes
then
	echo "$0: checksum file specified when all to be checksummed" 1>&2
	echo "    ... ignoring checksum file" 1>&2
	CHKSUM=
	ESTATUS=1
fi
#
# if no audit is to be done, a completely new master list is to be
#	generated, so we can ignore any existing master list.
# if an audit is to be done, check the status of the master list
#	if the master list exists and is not readable, exit with an error
#	if the master list is to be replaced and there is no master list,
#		give an error message and generate a new master list
#	otherwise, if the master list does not exist, exit with an error
#
if test \( "$GENAUDIT" = yes \) -a \( ! -r "$MASTER" \)
then
	if test \( -f "$MASTER" \) -o \( -d "$MASTER" \)
	then
		echo "$0: master file $MASTER is unreadable" 1>&2
		exit 1
	elif test "$GENMASTER" = yes
	then
		echo "$0: changing -ga option to -g" 1>&2
		echo "    ... generating $MASTER" 1>&2
		GENAUDIT=no
		ESTATUS=1
	else
		echo "$0: master file $MASTER does not exist" 1>&2
		exit 1
	fi
fi
#
# this pretties up the options list by deleting any leading comma
#
if test -n "$OPTLIST"
then
	X=`echo $OPTLIST | sed 's/^,//'`
else
	X=none
fi
OPTLIST="$X"
#
#===================== STEP 2.  HOST NAME ANALYSIS =====================
#
# Check the file "$R_INFO" for some variables:
#	R_AUDITLS	like auditls(8)	look for ^auditls
#	R_CAT		line cat(1)	look for ^cat
#	R_CHECKSUM	like sum(1)	look for ^checksum
#	R_DELETE	like rm(1)	look for ^delete
#	R_ECHO		like echo(1)	look for ^echo
#	R_EDIT		like sed(1)	look for ^edit
#	R_EXEC		like rsh(1)	look for ^rsh
#	R_FIND		like find(1)	look for ^scan
#	R_LSTAT		like lstat(1)	look for ^lstat
#	R_MOUNT		like mount(8)	look for ^mount
#	R_PPND		like auditppnd(1)	look for ^ppnd
#	R_RECV		rec from net	look for ^locdec
#	R_SEND		send over net	look for ^locenc
#	R_TEST		like test(1)	look for ^test
#	R_USER		who to run as	look for ^user
#
# load the remote environment
#
if test -r "$R_INFO"
then
				# get auditls(8) command
	TMP=`grep -i '^auditls[	 ]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_AUDITLS="$TMP"
	fi
				# get cat(1) command
	TMP=`grep -i '^cat[	 ]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_CAT="$TMP"
	fi
				# get checksum command
	TMP=`grep -i '^checksum[ 	]' $R_INFO | \
					awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_CHECKSUM="$TMP"
	fi
				# get rm(1) command
	TMP=`grep -i '^delete[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_DELETE="$TMP"
	fi
				# get echo(1) command
	TMP=`grep -i '^echo[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_ECHO="$TMP"
	fi
				# get sed(1) command
	TMP=`grep -i '^edit[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_EDIT="$TMP"
	fi
				# get rsh(1) command
	TMP=`grep -i '^rsh[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_EXEC="$TMP"
	fi
				# get find(1) command
	TMP=`grep -i '^scan[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_FIND="$TMP"
	fi
				# get mount(8) command
	TMP=`grep -i '^mount[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_MOUNT="$TMP"
	fi
				# get auditppnd(1) command
	TMP=`grep -i '^ppnd[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_PPND="$TMP"
	fi
				# get netsend command
	TMP="`grep -i '^locenc[	 ]' $R_INFO | awk '-F	' ' { print $2 } '`"
	if test -n "$TMP"
	then
		R_SEND=`echo $TMP | sed -e "s/\$HOST/$HOST/g" \
					-e "s/\$LOCALHOST/$LOCALHOST/g"`
	fi
				# get netreceive command
	TMP=`grep -i '^locdec[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_RECV=`echo $TMP | sed -e "s/\$HOST/$HOST/g" \
					-e "s/\$LOCALHOST/$LOCALHOST/g"`
	fi
				# get test(1) command
	TMP=`grep -i '^test[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_TEST="$TMP"
	fi
				# get user name
	TMP=`grep -i '^user[ 	]' $R_INFO | awk '-F	' ' { print $2 } '`
	if test -n "$TMP"
	then
		R_USER="$TMP"
	fi
fi
#
# if this is not the current host, build the HOSTCMD
#
if test "$LOCALHOST" = "$HOST"
then
	HOSTCMD=
elif test -n "$R_USER"
then
	HOSTCMD="$R_EXEC $HOST -l $R_USER"
else
	HOSTCMD="$R_EXEC $HOST"
fi
#
# check that the file is not on the wrong type of volume
#
TYPE=
if test -r $SKIPTYPE
then
	TYPE=`$HOSTCMD $R_MOUNT | gettype "$FILESYS"`
	if test -n "$TYPE" && fgrep -x "$TYPE" $SKIPTYPE > /dev/null
	then
		rm -f $FILELIST
		exit 0
	fi
fi
if test -r $DOTYPE
then
	if test -z "$TYPE"
	then
		TYPE=`$HOSTCMD $R_MOUNT | gettype "$FILESYS"`
	fi
	if test -n "$TYPE"
	then
		fgrep -x "$TYPE" $DOTYPE > /dev/null
		if test $? -ne 0
		then
			rm -f $FILELIST
			exit 0
		fi
	fi
fi
#
# now add the debugging option (the R_MOUNT is usually
# a binary, so doing this above won't work ...)
#
if test "X$DEBUG" != X
then
	HOSTCMD="$HOSTCMD sh $DEBUG"
fi
#
#=============== STEP 3.  LOG THE FILE SYSTEM INFORMATION ===============
#
# use AUDITLS to do the logging
#
( if test -r $SKIPTYPE
  then
	sed 's/^/FINL -n/' < $SKIPTYPE
  fi
  if test -r $DOTYPE
  then
	sed 's/^/FINL -o/' < $DOTYPE
  fi
  if test "$CHECKSUMALL" = yes
  then
	echo 'CALL'
  elif test -n "$CHKSUM" -a -r "$CHKSUM"
  then
  	echo 'CHKF' "$CHKSUM"
	sed 's/^/CHKL /' < $CHKSUM
  fi
  if test -s "$EXCEPT" -a -r "$EXCEPT"
  then
  	echo 'IGNF' "$EXCEPT"
	awk ' { print $1 } ' $EXCEPT | edprot '{+?}' | sed 's/^/IGNL /'
	echo '^\.$/' | edprot '{+?}' | sed 's/^/IGNL /'
  fi
  if test $RECURSE = yes
  then
	echo 'RECU' "$RECURSE"
  fi
  echo 'FSYS' "$FILESYS"
  echo 'TEST' "$R_TEST"
  echo 'FIND' "$R_FIND"
  echo 'DELF' "$R_DELETE"
  echo 'STRE' "$R_EDIT"
  echo 'TOPT' "$TYPEOPTS"
  echo 'CHKP' "$R_CHECKSUM"
  echo 'QUIT'
) | $R_SEND |\
    $TIMER -t$TIMEOUT -o$NMAS $HOSTCMD $R_AUDITLS $LOCALHOST $R_TEST $R_ECHO|\
		$R_RECV > $TMPZ
STATUS="`grep '^OUTP ERRC' $TMPZ`"
if expr "$STATUS" : 'OUTP ERRC [0-9][0-9]*' > /dev/null
then
	STATUS=`expr "$STATUS" : 'OUTP ERRC \([0-9][0-9]*\)'`
else
	STATUS=255
fi
grep '^OUTP ' $TMPZ | sed 's/^OUTP //' > $TMPX
grep -v '^OUTP ' $TMPZ | grep -v '^OUTP ERRC ' | > $TMPY
#
# if this got timed out or bombed, bomb
#
if test $STATUS -ne 0
then
	echo "$0: could not list file system $FILESYS on $HOST" 1>&2
	if test -s "$TMPY"
	then
		BLANKS="`echo $0 | sed 's/[^ ]/ /g'`"
		echo "$BLANKS  due to the following errors:" 1>&2
		cat $TMPY 1>&2
		echo " " 1>&2
	fi
	rm -f $FILELIST $TMPY
	exit 1
fi
if test -s "$TMPY"
then
	BLANKS="`echo $0 | sed 's/[^ ]/ /g'`"
	echo "$0: the following errors occurred when scanning" 1>&2
	echo "$BLANKS  the file system $FILESYS on $HOST" 1>&2
	echo "$BLANKS  but the listing appeared to complete successfully:" 1>&2
	cat $TMPY 1>&2
	echo " " 1>&2
	rm -f $TMPY
	ESTATUS=1
fi
#
# now sort them into a log file
#
sed '/^\.\.	/d' < $TMPX | sort | sed 's,^\.\/,,' > $NMAS
#
# see if there is a list of exceptions defined and if so, arrange to
# eliminate them from the master file.  note that if you name a directory
# all subdirectories and the files in them are also eliminated (useful for
# spool directories); also, if anything is to be ignored, the directory .
# is also ignored.
#
#if test -s "$EXCEPT"
#then
#	awk ' { print $1 } ' $EXCEPT | edprot +\?\| | sed 's/$/	/' > $OMIT
#	echo '\.	' >> $OMIT
#	egrep -v -f $OMIT < $NMAS > $TMPX
#	if test $? -eq 2
#	then
#		echo "	This error means one of the regular expressions " 1>&2
#		echo "	in the ignore file $EXCEPT" 1>&2
#		echo "	is too long, or you have too many patterns in " 1>&2
#		echo "	that file.  Try to consolidate patterns and " 1>&2
#		echo "	shorten the longest one(s)." 1>&2
#	fi
#	mv $TMPX $NMAS
#	rm -f $OMIT
#fi
#
# check for a potential problem
# if the listing command failed but generated some output, it is possible
# that the $NMAS file will not be null but will report most of the files
# deleted; so, if $NMAS reports $THRESH% or less of the files in $MASTER,
# DO NOT delete the master list or continue the audit but report this
# AT ONCE to the auditor who can then delete $MASTER and regenerate a new
# one if need be
#
if test ! -r $MASTER
then
	OLDCOUNT=0
else
	OLDCOUNT=`wc -l $MASTER | awk ' { print $1 } '`
fi
NEWCOUNT=`wc -l $NMAS | awk ' { print $1 } '`
NEWCOUNT=`expr $THRESH \* $NEWCOUNT`
if test \( $GENMASTER = no -o $GENAUDIT = yes \) -a \( $THRESH -gt 0 \) -a \
	\( $NEWCOUNT -le $OLDCOUNT \) -a \( $OLDCOUNT -ge $MINCT \)
then
	echo ' ' 1>&2
	echo ' ********** WARNING ********** ' 1>&2
	echo " There is a potential problem with the file system $FILESYS" 1>&2
	echo " on $HOST -- the audit showed that file system has $THRESH%" 1>&2
	echo " of the files it had when the master file was made. Either" 1>&2
	echo " the audit failed or most files on that file system have" 1>&2
	echo " been deleted.  Check to be sure it is not the latter," 1>&2
	echo " and if the master file must be regenerated, delete the" 1>&2
	echo " current one and replace it with $NMAS." 1>&2
	if test "$GENMASTER" = yes
	then
		echo " +++ NOTE: THE MASTER FILES HAVE NOT BEEN UPDATED +++" 1>&2
	fi
	echo ' ***************************** ' 1>&2
	echo ' ' 1>&2
	rm -f $FILELIST
	exit 1
fi
#
#========== STEP 4.  ELIMINATE FILES TO BE IGNORED TEMPORARILY ===========
#
#
# now do the reject (temporary ignore) files.  these are tricky because they
# contain full path names and do not allow metacharacters, so we have a fair
# bit of finageling to do to extract the proper names and fix them up so sed
# just matches on them as strings and not patterns
#
PFILESYS=`echo $FILESYS | edprot \"/\ \	`
#
# if the reject file is too old, trash it
#
if test -r $TMPIGN
then
	find $TMPIGN -mtime $LIMIT -exec rm -f {} \;
fi
#
# if not, process it (good luck with this one!)
#
if test -r $TMPIGN
then
	sed -n "s/^$PFILESYS\///p"  < $TMPIGN |\
		sed -n 'p
			/\/$/s,/$,,p' | edprot / |
				sed 's,/$,/[^	]*,' |\
		 			sed 's,^.*$,/^&	/d,' > $OMIT
	echo '/\.	/d' >> $OMIT
fi
#
#===================== STEP 5.  ANALYZE THE OUTPUT =====================
#
# analyze and compare to master list
# we know it exists at this point
#
if test "$GENAUDIT" = yes
then
	#
	# figure out which files have changed, been added, or deleted
	#
	$COMMER -23w $MASTER $NMAS > $OLD
	if test -s "$OMIT"
	then
		sed -f $OMIT < $OLD >$TMPX
		if test $? -ne 0
		then
			echo "	This error means one of the file names " 1>&2
			echo "	in the reject file $TMPIGN" 1>&2
			echo "	is too long, or you have too many names" 1>&2
			echo "	in that file.  Just delete the file and" 1>&2
			echo "	ignore the change messages." 1>&2
			> $OMIT
		else
			mv $TMPX $OLD
		fi
	fi
	$COMMER -13w $MASTER $NMAS > $NEW
	if test -s "$OMIT"
	then
		sed -f $OMIT < $NEW >$TMPX
		if test $? -eq 2
		then
			echo "	This error means one of the file names " 1>&2
			echo "	in the reject file $TMPIGN" 1>&2
			echo "	is too long, or you have too many names" 1>&2
			echo "	in that file.  Just delete the file and" 1>&2
			echo "	ignore the change messages." 1>&2
		else
			mv $TMPX $NEW
		fi
	fi
	#
	# now get the lists of file names ONLY
	#
	awk '{ print $1 }' < $OLD > $OLDNM
	awk '{ print $1 }' < $NEW > $NEWNM
	#
	# report on new files
	#
	> $PATCHANGES
	$COMMER -13w $OLDNM $NEWNM > $TMPX
	if test -s "$TMPX"
	then
		$GREPER $NEW < $TMPX > $NEWFILES
		if test -s "$NEWFILES" -a -n "$PATTRN" -a -s "$PATTRN"
		then
			$MATCHER "(add) " "$NEWFILES" "$PATTRN" "$PATCHANGES" > $TMPX
			mv $TMPX $NEWFILES
		fi
	fi
	#
	# report on old files
	#
	$COMMER -23w $OLDNM $NEWNM > $TMPX
	if test -s "$TMPX"
	then
		$GREPER $OLD < $TMPX > $OLDFILES
		if test -s "$OLDFILES" -a -n "$PATTRN" -a -s "$PATTRN"
		then
			$MATCHER "(del) " "$OLDFILES" "$PATTRN" "$PATCHANGES" > $TMPX
			mv $TMPX $OLDFILES
		fi
	fi
	#
	# report on changed files
	#
	$COMMER -12w $OLDNM $NEWNM > $TMPX
	if test -s "$TMPX"
	then
		$GREPER $OLD "(old)" $NEW "(new)" < $TMPX > $CHANGES
		if test -s "$CHANGES" -a -n "$PATTRN" -a -s "$PATTRN"
		then
			$MATCHER -2.5 "" "$CHANGES" "$PATTRN" "$PATCHANGES" > $TMPX
			mv $TMPX $CHANGES
		fi
	fi
	#
	# now send it to the audit list in a neat format
	# note we put a little header up there, and we
	# do so in such a way that nothing is mailed if
	# there is nothing to report
	#
	( echo $HOST
	  echo $FILESYS
	  echo $AUDITOR
	  echo $OPTLIST
	  echo $ONEOUTPUT
	) | $MAILER "$OLDFILES" "$NEWFILES" "$CHANGES" "$PATCHANGES" $DEBUG
	if test $? -eq 1
	then
		SENTMAIL=yes
	fi
	#
	# now prepare files by user names
	# first, make a list of nonempty files
	#
	LIST=
	if test -n "$OLDFILES" -a -s "$OLDFILES"
	then
		LIST="$LIST$OLDFILES "
	fi
	if test -n "$NEWFILES" -a -s "$NEWFILES"
	then
		LIST="$LIST$NEWFILES "
	fi
	#
	# notify users if requested and something has changed
	#
	if test "$GENUSRRPT" = yes -a \( -n "$LIST" -o -s "$CHANGED" \)
	then
		#
		# reset the option list to label this as a user copy
		#
		if test -z "$OPTLIST"
		then
			OPTLIST=user-copy
		else
			OPTLIST="user-copy,$OPTLIST"
		fi
		#
		# omit empty files
		#
		#
		# get a list of users
		#
		rm -f $TMPX
		if test -n "$LIST"
		then
			awk ' { print $5 } ' < $LIST | sort | uniq > $TMPX
		fi
		if test -n "$CHANGES" -a -s "$CHANGES"
		then
			awk ' { print $6 } ' < $CHANGES | sort | uniq >> $TMPX
		fi
		#
		# we have to do some fancy stuff to ensure the args to "for"
		# are not too long
		#
		split -100 $TMPX $TMPY
		for j in ${TMPY}*
		do
			for i in `cat $j`
			do
				if test -n "$OLDFILES" -a -s "$OLDFILES"
				then
					echo $i | $GREPER -f5 $OLDFILES > $UOLD
				fi
				if test -n "$NEWFILES" -a -s "$NEWFILES"
				then
					echo $i | $GREPER -f5 $NEWFILES > $UNEW
				fi
				if test -n "$CHANGES" -a -s "$CHANGES"
				then
					echo $i | $GREPER -f5 $CHANGES > $UCHG
				fi
				$MAILER "$HOST" "$FILESYS" "$i" "$OPTLIST" "$UOLD" "$UNEW" "$UCHG" $DEBUG
			done
		done
	fi
fi
#
#===================== STEP 6.  SET UP A NEW MASTER FILE =====================
#
# if you are to generate the master list, just move the results to
# the master file
#
if test "$GENMASTER" = yes
then
	if test -r $MASTER
	then
		$MERGER $MASTER $NMAS > $TMPX
		mv $TMPX $MASTER
	else
		mv $NMAS $MASTER
	fi
fi
#
#===================== STEP 7.  CLEAN UP AND EXIT =====================
#
# delete all intermediate files
#
rm -f $FILELIST $NMAS
#
# all done
#
if test "$SENTMAIL" = yes
then
	exit 1
fi
exit $ESTATUS
