/*
 * This program is a public-domain version of comm(1) with a difference
 * 	auditcomm [options] oldfile newfile
 * prints on the standard output 3 columns:
 *	column 1 contains lines only in oldfile
 *	column 2 contains lines only in newfile
 *	column 3 contains lines in both oldfile and newfile
 * The filename '-' means standard input.  Options are:
 *	-1	suppress column 1
 *	-2	suppress column 2
 *	-3	suppress column 3
 * or any combination thereto (note: -123 is a no-op.  The standard program
 * comm(1) actually does all the work and just never prints anything; this
 * one is bright enough to recognize all three columns are to be suppressed,
 * so it simply goes away.
 *   Note that both files are assumed sorted in increasing ASCII collating
 * sequence.
 *   The fun begins when the non-standard option 'w' is given.  This option
 * changes the line comparison function as follows: with 'w', each line is
 * a sequence of fields separated by tabs (\t).  If the field of a line in
 * oldfile is the wildcard WC (currently *), the corresponding field in
 * newfile may contain anything or nothing; they match.
 *
 * Matt Bishop
 * Department of Mathematics and Computer Science
 * Dartmouth College
 * Hanover, NH  03755
 *
 * Matt.Bishop@dartmouth.edu, ...!decvax!dartvax!Matt.Bishop
 */
#include <stdio.h>

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

/*
 * leaders
 */
char *lb1 = "";		/* for column 1 */
char *lb2 = "\t";	/* for column 2 */
char *lb3 = "\t";	/* for column 3 */

/*
 * which columns to print
 */
int showone = 1;	/* print column 1 */
int showtwo = 1;	/* print column 2 */
int showthree = 1;	/* print column 3 */

/*
 * which file do we get a line from
 */
int getnew = 1;		/* get a line from the new file */
int getold = 1;		/* get a line from the old file */

/*
 * buffers for lines from file
 */
char newbuf[BUFSIZ];	/* buffer for line from new file */
char oldbuf[BUFSIZ];	/* buffer for line from old file */

/*
 * pointers to files
 */
FILE *nfp;		/* pointer to new file */
FILE *ofp;		/* pointer to old file */

/*
 * forward references to input functions
 */
char *mgets();		/* like fgets but deletes trailing newline */

/*
 * forward references to string comparison functions
 * allowing wildcard matches
 */
extern int wstrcmp();	/* allows wildcard matches */
extern int strcmp();	/* standard function */

/*
 * used to make comparisons of lines
 */
int (*compar)() = strcmp;

main(argc, argv)
int argc;
char **argv;
{
	register int i;		/* used to walk argument list */
	register char *p;	/* used to walk option argument */
	char *oldfile;		/* name of old file */
	char *newfile;		/* name of new file */

	/*
	 * process the options
	 */
	for (i = 1; i < argc; i++){
		if (*argv[i] != '-')
			break;
		for(p = &argv[i][1]; *p; p++){
			if (*p == 'w')
				compar = wstrcmp;
			else if (*p == '1')
				showone = 0;
			else if (*p == '2')
				showtwo = 0;
			else if (*p == '3')
				showthree = 0;
			else if (*p == '\0')
				break;
			else
				usage(argv[0]);
		}
	}

	/*
	 * reset the labels as appropriate
	 */
	if ((!showone && !showtwo)
		|| (!showtwo && !showthree)
			|| (!showthree && !showone))
		lb2 = lb3 = lb1;
	else if (!showone){
		lb3 = lb2; lb2 = lb1;
	}
	else if (!showtwo){
		lb3 = lb2;
	}
		
	/*
	 * get the old and new file names,
	 * reporting any usage problems
	 */
	if (i == argc)	usage(argv[0]);
	oldfile = argv[i++];
	if (i == argc)	usage(argv[0]);
	newfile = argv[i++];
	if (i != argc)	usage(argv[0]);

	/*
	 * open the old file if it isn't stdin
	 */
	if (strcmp(oldfile, "-") == 0)
		ofp = stdin;
	else if ((ofp = fopen(oldfile, "r")) == NULL){
		perror(oldfile);
		exit(1);
	}
	/*
	 * open the new file if it isn't stdin
	 */
	if (strcmp(newfile, "-") == 0)
		nfp = stdin;
	else if ((nfp = fopen(newfile, "r")) == NULL){
		perror(newfile);
		exit(1);
	}

	/*
	 * hack for -123 (a no-op); don't want to waste CPU power here ...
	 */
	if (!showone && !showtwo && !showthree)
		exit(0);

	/*
	 * the analysis loop
	 * get any required new lines up here
	 */
	while(getnext()){
		/*
		 * compare the lines
		 */
		if ((i = (*compar)(oldbuf, newbuf)) == 0){
			/*
			 * the same; if appropriate, print one
			 * and say you want the next line from each file
			 */
			if (showthree)
				(void) printf("%s%s\n", lb3, oldbuf);
			getnew = getold = 1;
			continue;
		}
		else if (i > 0){
			/*
			 * line in new file, not in old file
			 */
			if (showtwo)
				(void) printf("%s%s\n", lb2, newbuf);
			/*
			 * since new file line was dumped, just get
			 * next one of that file
			 */
			getnew = 1;
			getold = 0;
			continue;
		}
		/*
		 * line in old file, not in new file
		 */
		if (showone)
			(void) printf("%s%s\n", lb1, oldbuf);
		/*
		 * since old file line was dumped, just get
		 * next one of that file
		 */
		getold = 1;
		getnew = 0;
	}

	/*
	 * someone hit EOF
	 * if new file, and are showing lines in old file
	 * not in new file, dump the rest of the old file
	 */
	if (showone){
		if (!getold)
			(void) printf("%s%s\n", lb1, oldbuf);
		while(mgets(oldbuf, BUFSIZ, ofp) != NULL)
			(void) printf("%s%s\n", lb1, oldbuf);
	}
	/*
	 * if old file, and are showing lines in new file
	 * not in old file, dump the rest of the new file
	 */
	if (showtwo){
		if (!getnew)
			(void) printf("%s%s\n", lb1, newbuf);
		while(mgets(newbuf, BUFSIZ, nfp) != NULL)
			(void) printf("%s%s\n", lb2, newbuf);
	}
	/*
	 * close the files
	 * and quit with successful error code
	 */
	(void) fclose(ofp);
	(void) fclose(nfp);
	exit(0);
}

/*
 * get the next line from the appropriate file(s)
 * 1 on success
 * 0 if ANYONE returns EOF
 */
getnext()
{
	/*
	 * get line from old file
	 */
	if (getold && mgets(oldbuf, BUFSIZ, ofp) == NULL)
		return(0);
	getold = 0;
	/*
	 * get line from new file
	 */
	if (getnew && mgets(newbuf, BUFSIZ, nfp) == NULL)
		return(0);
	getnew = 0;
	return(1);
}

/*
 * instruct user how to use this program
 */
usage(progname)
char *progname;		/* argument 0 is program name */
{
	(void) fprintf(stderr, "Usage: %s [ -[123] ] file1 file2\n", progname);
	exit(1);
}

/*
 * just like fgets, but deletes trailing newline
 */
char *mgets(buf, n, fp)
char *buf;				/* buffer for input */
int n;					/* size of buffer */
FILE *fp;				/* file toget inp[ut from */
{
	register int c;		/* input character */
	register char *b;	/* used to put chars into buffer */

	/*
	 * load up the buffer
	 */
	b = buf;
	while((c = getc(fp)) != EOF && c != '\n'){
		*b++ = c;
		/*
		 * check for buffer overflow
		 */
		if (b >= &buf[n-1]){
			/*
			 * got it -- terminate buffer and return
			 */
			*b = '\0';
			return(buf);
		}
	}
	/*
	 * end buffer and return, handling EOF properly
	 */
	*b = '\0';
	return((b == buf && c == EOF) ? NULL : buf);
}

/*
 * this acts like strcmp with one MAJOR difference
 * if there are tabs on the line, the tabs delimit fields
 * the wildcard * in any field of s1 matches whatever s2 has in that
 * field, including nothing
 * other than that, it's just strcmp
 */
wstrcmp(s1, s2)
char *s1, *s2;		/* lines to be compared */
{
	register char *o = s1;		/* walk string 1 (old file) */
	register char *n = s2;		/* walk string 2 (new file) */

	/*
	 * just a long lazy loop ...
	 */
	do{
		if (*o == '*' &&
			    (o[1] == '\t' || o[1] == '\n' || o[1] == '\0')){
			/*
			 * skip s2's field (matches wildcard)
			 */
			while(*n && *n != '\t' && *n != '\n')
				n++;
			o++;
			continue;
		}
		else{
			/*
			 * go to end of field, comparing the two strings
			 * (no wildcards)
			 */
			while(*o && *o != '\t' && *n && *o == *n)
				o++, n++;
		}
		/*
		 * if end of field, go on to next one
		 * if the two chars don't match, done
		 */
		if (*o == '\t' && *n == '\t'){
			o++;
			n++;
		}
		else if (*o != *n)
			break;

	} while(*o && *n);

	/*
	 * return the difference
	 */
	return(*o - *n);
}

