/*
 * This program takes a file with lines of the form
 * .	0755	2	mab	root	512	Oct 15, 1986 at 16:04:19
 * ./Audit	0755	1	mab	root	6385	Oct 15, 1986 at 15:58:32
 * ./Audit.8	0644	1	mab	root	1466	Oct 15, 1986 at 15:44:16
 * and formats them so the columns line up.  It uses blanks so you can run
 * awk or some such postprocessor on the output, too.
 *
 * Matt Bishop
 * Research Institute for Advanced Computer Science
 * NASA Ames Research Center
 * Moffett Field, CA  94035
 *
 * mab@riacs.edu, ...!{decvax!decwrl,ihnp4!ames}!riacs!mab
 *
 * (c) Copyright 1986 by Matt Bishop and the Research Institute for
 *			Advanced Computer Science
 */
#include <stdio.h>

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

#define abs(x)	(((x)>0)?(x):(-(x)))		/* think real hard */

/*
 * this stuff controls the lining up of the columns
 * the elements of "indices" are the widths of the columns
 * (these get set automatically, so we just use "1" or "-1"
 * to indicate field justification -- "-1" is left, "1" is right)
 * title1 and title2 hold the titles and separator lines
 * for each column of the output
 */
#define	NFIELDS		sizeof(indices)/sizeof(int)
int indices[] = {		/* field widths */
	-1,				/* file name */
	 1,				/* file type */
	 1,				/* protection mode */
	 1,				/* number of links */
	-1,				/* user name/number */
	-1,				/* group name/number */
	 1,				/* size in bytes */
	 1,				/* checksum (if present) */
	-1,				/* date */
};
char *title1[] = {		/* titles */
	"file name",			/* file name */
	"type",				/* file type */
	"mode",				/* protection mode */
	"links",			/* number of links */
	"user",				/* user name/number */
	"group",			/* group name/number */
	"size",				/* size in bytes */
	"checksum",			/* checksum (if present) */
	"date",				/* date */
};
char *title2[] = {		/* separators */
	"---------",			/* file name */
	"----",				/* file type */
	"----",				/* protection mode */
	"-----",			/* number of links */
	"----",				/* user name/number */
	"-----",			/* group name/number */
	"----",				/* size in bytes */
	"--------",			/* checksum (if present) */
	"----",				/* date */
};

/*
 * forward reference
 */
char *index();

main(argc, argv)
int argc;
char **argv;
{
	FILE *fp;		/* pointer to file */
	register int i;		/* counter in a for loop */
	register char *p, *q;	/* used to split line into fields */
	int maxlen;		/* length of the string just read in */
	char buf[BUFSIZ];	/* buffer for reading lines */

	/*
	 * exactly ONE argument, and only one argument!
	 */
	if (argc != 2){
		fprintf(stderr, "Usage: %s file1 ...\n", argv[0]);
		exit(1);
	}

	/*
	 * open the file
	 */
	if ((fp = fopen(argv[1], "r")) == NULL){
		perror(argv[1]);
		exit(1);
	}

	/*
	 * scan the file and set up the width array (indices)
	 */
	while(fgets(buf, BUFSIZ, fp) != NULL){
		/*
		 * delete a trailing newline
		 */
		if ((q = index(buf, '\n')) != NULL)
			*q = '\0';
		/*
		 * start everything at the beginning of the line
		 */
		q = p = buf;
		/*
		 * loop until you hit all the fields
		 */
		for(i = 0; i < NFIELDS && q != NULL; i++){
			/*
			 * grab the next field (ended by a tab or NUL)
			 * and get its length
			 */
			if ((q = index(p, '\t')) != NULL)
				*q++ = '\0';
			maxlen = strlen(p);
			/*
			 * if the field is wider than what's there now,
			 * replace what's there now by the new value
			 * note magnitude matters in comparison,
			 * but you have to keep the same sign ...
			 */
			if (abs(indices[i]) < maxlen)
				indices[i] = (indices[i] < 0) ? -maxlen : maxlen;
			/*
			 * iterate
			 */
			p = q;
		}
	}

	/*
	 * print the titles
	 */
	prtitle(title1);
	prtitle(title2);

	/*
	 * at this point, indices[] has the appropriate widths
	 * so go to the beginning of the file (that's why you
	 * can't do this to a pipe) and reread it
	 */
	rewind(fp);

	/*
	 * now format the input file
	 */
	while(fgets(buf, BUFSIZ, fp) != NULL){
		/*
		 * delete a trailing newline
		 */
		if ((q = index(buf, '\n')) != NULL)
			*q = '\0';
		/*
		 * start everything at the beginning of the line
		 */
		q = p = buf;
		/*
		 * loop until you hit all the fields
		 */
		for(i = 0; i < NFIELDS && q != NULL; i++){
			/*
			 * grab the next field (ended by a tab or NUL)
			 * and get its length
			 */
			if ((q = index(p, '\t')) != NULL)
				*q++ = '\0';
			/*
			 * print the field with the appropriate width
			 */
			printf("%*s", indices[i], p);
			/*
			 * if another field follows, print the separator
			 * (two blanks) and reset p; if not, drop a newline
			 */
			if (q != NULL){
				putchar(' ');
				putchar(' ');
				p = q;
			}
			else
				putchar('\n');
		}
	}

	/*
	 * close the input file
	 */
	(void) fclose(fp);

	/*
	 * all done!
	 */
	exit(0);
}

/*
 * find the leftmost occurrence of char c in string s; NULL if none
 */
char *index(s, c)
register char *s;		/* string to be scanned */
register char c;		/* character to be scanned for */
{
	/*
	 * just walk the string
	 */
	while(*s && *s != c)
		s++;
	/*
	 * return a pointer to it
	 */
	return(*s ? s : NULL);
}

/*
 * print titles, properly centered
 */
prtitle(title)
char *title[];			/* array of titles */
{
	register int i;			/* counter in a for loop */
	register int excess;		/* used to center title */

	/*
	 * now print the titles
	 * try to center them in the given fields
	 */
	for(i = 0; i < NFIELDS; i++){
		/*
		 * if the title is wider than the space
		 * widen the space
		 */
		if (abs(indices[i]) <= strlen(title[i])){
			printf("%-s", title[i]);
			excess = strlen(title[i]);
			indices[i] = (indices[i] < 0) ? -excess : excess;
		}
		else{
			/*
			 * otherwise, center it as best you can
			 */
			excess = abs(indices[i]) - strlen(title[i]);
			if ((excess%2) > 0)
				excess++;
			printf("%*c%-s", excess/2, ' ', title[i]);
			/*
			 * if no more titles, don't bother padding the right
			 */
			if (i != NFIELDS-1)
				printf("%*c",
				    abs(indices[i])-strlen(title[i])-excess/2,
									   ' ');
		}
		/*
		 * if there's another title coming up,
		 * put out the field separator
		 */
		if (i != NFIELDS-1){
			putchar(' ');
			putchar(' ');
		}
	}

	/*
	 * go to the next line
	 */
	putchar('\n');
}
