/*
 * file tree scanner
 *
 * options
 * -C <file>
 *	Pretend the file used in the next -c option is named <file>
 * -c <file>
 *	Checksum files (directories) with names in <file>.
 * -d <num>
 *	Scan at most <num> levels of the tree
 * -I <file>
 *	Pretend the file used in the next -i option is named <file>
 * -i <file>
 *	Ignore files (directories) matching the patterns in <file>.
 *	Each line contains one ed(1)-style pattern, with the following
 *	modification: if the pattern ends in '/' (which would follow
 *	any trailing '$' for end of string), then the subtrees beneath
 *	directories matching the pattern are scanned but no information
 *	about the directories is printed; otherwise, the scan ends at the
 *	directory.
 * -l <stat list>
 *	What characteristics of the file to print
 *	a	time of last access
 *	c	time of creation (inode change)
 *	d	device number (not on Crays)
 *	f	file name
 *	g	group
 *	i	inode number
 *	m	time of last modification
 *	n	number of (hard) links
 *	p	protection mode
 *	s	size (of file), major/minor numbers (of devices)
 *	t	type of file:
 *		-	regular file
 *		b	block special (device)
 *		c	character special (device)
 *		d	directory
 *		f	named FIFO
 *		l	symbolic link
 *		s	socket
 *		?	unknown
 *	u	user
 *	x	checksum
 * -n <fstype>
 *	Only scan, and descend from, a directory or file if it is not mounted
 *	on a volume of type <fstype>
 * -o <fstype>
 *	Only scan, and descend from, a directory or file if it is mounted on
 *	a volume of type <fstype>
 * -s <code>
 *	Only report information on files with the specified characteristics:
 *	G	setgid and not a directory
 *	b	block special device
 *	c	character special device
 *	g	setgid
 *	u	setuid
 * -v<format>
 *	Print the exit status code using the fprintf format in <format>.
 *	Printing is done to the standard output, and a newline is appended.
 *	C escapes are NOT recognized.  Note there cannot be a space between
 *	the format string and the v.
 * -x	Checksum all files.
 *
 * ALGORITHM:
 *
 * 0. figure out which type of file system directory is mounted on
 * 1. enter directory
 * 2. read next entry in directory
 *	a. if it's on an uninteresting type of FS, go to 2.
 *	b. if it matches an ignore pattern, go to 2
 *	c. print it
 *	d. if it's a directory, go to 0
 *	when done, quit
 */
#include <stdio.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

/*
 * useful macros
 */
#define MAXDEPTH 	100		/* maximum depth for file tree */
#define	MAXPATH		(10*BUFSIZ)	/* maximum allowed file name length */
#define GOODSTAT	0		/* stat of file succeeded */
#define BADSTAT		1		/* stat of file failed */

/*
 * useful macros for specific machines
 */
#ifdef SYSV
#	define index strchr
#	define u_short	ushort		/* different types */
#endif
#ifndef S_IFLNK
#	define lstat stat		/* no symbolic links */
#endif

/*
 * program flags and options
 */
char *progname = "auditscan";	/* name of program as invoked */
int npats = 0;			/* number of ignore entries */
struct s_ign {
	char *regex;			/* pattern of file name */
	ushort stopmode;		/* S_IFDIR if to continue */
} pats[BUFSIZ];			/* array of ignore patterns, info */
char *chks[BUFSIZ];		/* list of files to checksum */
int nchks = 0;			/* number of files to checksum */
char modes[BUFSIZ] = "";	/* types of files to list */
char stflags[BUFSIZ] = "";	/* how to list file info */
int verbose = 0;		/* don't print exit status code */
char *verbfmt = "exit status %d";	/* format to use to print exit code */
int doxflag = 0;		/* do checksumming */

/*
 * library functions and system variables
 */
char *getenv();
struct group *getgrgid();
struct passwd *getpwuid();
char *index();
char *malloc();			/* allocate memory */
FILE *popen();
char *strcat();			/* append string */
char *strcpy();			/* copy string */
extern int errno;		/* error number from system call */

/*
 ****************
 * main module
 *	process arguments and invoke file tree scanner
 ****************
 */
main(argc, argv)
int argc;
char **argv;
{
	register int i;			/* counter in a for loop */
	char nxtflag = '\0';		/* next option to process */
	char buf[MAXPATH];		/* buffer for file name */
	int depth = MAXDEPTH;		/* how far down to go */

	/*
	 * process the options
	 */
	progname = argv[0];
	for(i = 1; i < argc; i++){
		/*
		 * next arg is arg to option
		 */
		if (nxtflag){
			switch(nxtflag){
			case 'C': setchkname(argv[i]);		break;
			case 'c': loadchk(argv[i]);		break;
			case 'd': depth = atoi(argv[i]);	break;
			case 'I': setignname(argv[i]);		break;
			case 'i': loadpats(argv[i]);		break;
			case 'l': (void) strcat(stflags, argv[i]); break;
			case 'o': mountset(argv[i], 1);		break;
			case 'n': mountset(argv[i], 0);		break;
			case 's': (void) strcat(modes, argv[i]); break;
			default:
				fprintf(stderr, "%s: unknown option %c\n",
					progname, nxtflag);
				mexit(1);
			}
			nxtflag = '\0';
		}
		/*
		 * next arg is option
		 */
		else if (argv[i][0] == '-'){
			if (argv[i][1] == 'x'){
				doxflag = 1;
				continue;
			}
			if (argv[i][1] == 'v'){
				verbose = 1;
				if (argv[i][2] != '\0')
					verbfmt = &argv[i][2];
				continue;
			}
			if (argv[i][2] == '\0' && argv[i][1] != 'v'){
				nxtflag = argv[i][1];
				continue;
			}
			switch(argv[i][1]){
			case 'C': setchkname(&argv[i][2]);	break;
			case 'c': loadchk(&argv[i][2]);		break;
			case 'd': depth = atoi(&argv[i][2]);	break;
			case 'I': setignname(&argv[i][2]);	break;
			case 'i': loadpats(&argv[i][2]);	break;
			case 'l': (void) strcat(stflags, &argv[i][2]); break;
			case 'o': mountset(&argv[i][2], 1);	break;
			case 'n': mountset(&argv[i][2], 0);	break;
			case 's': (void) strcat(modes, &argv[i][2]);
								break;
			default:
				fprintf(stderr, "%s: unknown option %s\n",
							  progname, argv[i]);
				mexit(1);
			}
		}
		else{
			/*
			 * arg is file name -- be sure it can be accessed!
			 */
			if (fexist(argv[i]) < 0){
				perror(argv[i]);
				mexit(1);
			}
			(void) strcpy(buf, argv[i]);
			dofind(buf, depth);
		}
	}

	/*
	 * options left over
	 */
	if (nxtflag)
		fprintf(stderr, "%s: missing argument to -%c\n",
				progname, nxtflag);
	/*
	 * bye!
	 */
	mexit(0);
}

/*
 * a printing exit command
 */
mexit(status)
int status;		/* exit status */
{
	/*
	 * announce termination code
	 */
	if (verbose){
		printf(verbfmt, status);
		putchar('\n');
	}
	/*
	 * bye-ya!
	 */
	exit(status);
	/* NOTREACHED */
}

/*
 * do the per-file processing
 */
check(path, stbuf, type)
char *path;			/* path name of file */
struct stat *stbuf;		/* file's status info */
int type;			/* GOODSTAT if stbuf meaningful */
{
	register int i;			/* counter in a for loop */
	int putout = 1;			/* 1 if file data is to be printed */
	int terrno;			/* temp to hold error number */
	register char *s;		/* used to walk list of s flags */
	int cmp;			/* result of comparison */
	int odoxflag;			/* temp holder for doxflag */

	/*
	 * if bad status and the file isn't here, return;
	 * otherwise, try to continue
	 */
	if (type == BADSTAT){
		if (errno == ENOENT)
			return(-2);
		else
			terrno = errno;
	}

	/*
	 * first, see if it should be ignored
	 */
	if (npats > 0)
		for(i = 0; i < npats; i++)
			if (match(path, pats[i].regex) == 1){
				/*
				 * we DON'T end if the matched thing is . or ..
				 * (possible only as the top of the tree);
				 * we just don't print it
				 */
				if (path[0] == '.' &&
					(path[1] == '\0' || (path[1] == '.' &&
						path[2] == '\0')))
					return(0);
				/*
				 * if / is not at end of pattern, this
				 * ends the tree traversal on the branch
				 */
				if ((pats[i].stopmode&stbuf->st_mode) == 0)
					return(-2);
				/*
				 * print nothing but continue traversal
				 */
				return(0);
			}
	
	/*
	 * if stat is bad here, report it and return
	 */
	if (type == BADSTAT){
		errno = terrno;
		perror(path);
		return(-2);
	}
	/*
	 * check mount device type
	 */
	if (!mountcheck(stbuf))
		return(-2);

	/*
	 * if an 's' option is given,
	 * print the name ONLY if it is true
	 */
	if (modes[0] != '\0'){
		putout = 0;
		for(s = modes; *s && !putout; s++){
			switch(*s){
			case 'G':	/* print if setgid, not directory */
				if ((stbuf->st_mode&S_ISGID) == S_ISGID &&
					  (stbuf->st_mode&S_IFDIR) != S_IFDIR)
					putout = 1;
				break;
			case 'b':	/* print if block device */
				if ((stbuf->st_mode&S_IFMT) == S_IFBLK)
					putout = 1;
				break;
			case 'c':	/* print if character device */
				if ((stbuf->st_mode&S_IFMT) == S_IFCHR)
					putout = 1;
				break;
			case 'g':	/* print if setgid */
				if ((stbuf->st_mode&S_ISGID) == S_ISGID)
					putout = 1;
				break;
			case 'u':	/* print if setuid */
				if ((stbuf->st_mode&S_ISUID) == S_ISUID)
					putout = 1;
				break;
			}
		}
	}
	/*
	 * print the name or information
	 */
	if (putout){
		if (stflags[0] == '\0')
			printf("%s\n", path);
		else{
			odoxflag = doxflag;
			if (!doxflag){
				for(i = 0; i < nchks; i++){
					cmp = strcmp(chks[i], path);
					if (cmp == 0)
						doxflag = 1;
					if (cmp >= 0)
						break;
				}
			}
			nstat(path, stbuf);
			doxflag = odoxflag;
		}
	}
	return(0);
}

/*
 * starts the file tree walk
 */
dofind(s, d)
char *s;			/*  file name */
int d;				/* depth */
{
	/*
	 * if checking mount types, be sure the info
	 * is initialized
	 */
	mountfix();
	/*
	 * go!
	 */
	ftscan(s, d);
}

/*
 * save a string in core
 */
char *strsave(s)
char *s;		/* string */
{
	char *p;	/* temp pointer for space */

	/*
	 * allocate the room
	 */
	if ((p = malloc((unsigned) (strlen(s)+1))) == NULL){
		fprintf(stderr, "%s: ran out of room (strsave)\n", progname);
		mexit(1);
	}
	/*
	 * save the string and return the new location
	 */
	(void) strcpy(p, s);
	return(p);
}

/*
 * test for the existance of the file
 * this is done at the top (command-line) level only!!!
 */
#ifdef sun
#	include <unistd.h>
#else
#	include <sys/file.h>
#endif
int fexist(path)
char *path;			/* root */
{
	return(access(path, F_OK));
}

/*
 ****************
 * file system scanning module
 *	invokes check on each file as it goes down the tree
 ****************
 */
#if defined(sun) || defined(CRAY)
#	include <dirent.h>
#else
#	ifdef BSD4
#		define direct dirent	/* everyone else is backwards ... */
#		include <sys/dir.h>
#	else
#		include <sys/dir.h>
		/*
		 * no directory package -- fake it
		 */
#		define DIR FILE		/* directory type */
static struct dirent {
	char d_name[BUFSIZ];		/* file name */
	ino_t d_ino;			/* inode number (0 = deallocated) */
} rval;			/* directory entry */
		/* open directory */
#		define opendir(name)	fopen(name, "r")
		/* close directory */
#		define closedir(fp)	fclose(fp);
		/* read directory */
struct dirent *readdir(fp)
DIR *fp;		/* pointer to directory */
{
	struct direct dent;	/* entry in directory */
	/* get the entry */
	if (fread(&dent, sizeof(struct direct), 1, fp) != 1)
		return(NULL);
	/* return the info */
	(void) strncpy(rval.d_name, dent.d_name, DIRSIZ);
	rval.d_name[DIRSIZ] = '\0';
	rval.d_ino = dent.d_ino;
	return(&rval);
}
#	endif
#endif

/*
 * the tree walker
 */
int ftscan(path, depth)
char *path;			/* root */
int depth;			/* depth to go to */
{
	struct stat stbuf;	/* stat info of path */
	int type;		/* did stat succeed or not? */
	DIR *dirp;		/* pointer to directory */
	struct dirent *dp;	/* pointer to directory entry */
	char *eps;		/* just beyond end, to add / */
	int fix = 0;		/* 1 if / added */

	/*
	 * figure out what this object is
	 */
	type = GOODSTAT;
	if (lstat(path, &stbuf) < 0)
		type = BADSTAT;
	/*
	 * call the function on this object
	 * if it returns < 0, ignore this branch
	 */
	if (check(path, &stbuf, type) < 0)
		return;
	/*
	 * if it's not a directory, we're at the leaf -- return
	 */
	if (--depth < 0 || (stbuf.st_mode&S_IFMT) != S_IFDIR)
		return;

	/*
	 * in a directory -- read the entries
	 * scan each one
	 */
	if ((dirp = opendir(path)) == NULL){
		perror(path);
		return;
	}
	/*
	 * be sure there's a trailing '/'
	 */
	eps = &path[strlen(path)];
	if (fix = (eps[-1] != '/'))
		*eps++ = '/';
	/*
	 * now get each entry and work on it
	 */
	while((dp = readdir(dirp)) != NULL){
		/*
		 * ignore . or ..
		 */
		if (dp->d_ino == 0 || strcmp(dp->d_name, ".") == 0 ||
				strcmp(dp->d_name, "..") == 0)
			continue;
		/*
		 * get the full path name and scan it
		 */
		(void) strcpy(eps, dp->d_name);
		ftscan(path, depth);
	}
	/*
	 * done -- close the directory
	 * and clobber any added trailing '/'
	 */
	closedir(dirp);
	if (fix)
		*--eps = '\0';
}


#ifdef NFS
/*
 *******
 * the mounting stuff
 *******
 */
#include <mntent.h>
#ifndef MOUNTED
#	define MOUNTED MNTTAB
#endif

/*
 * the structure
 */
struct mnttype {
	char *dir;		/* virtual volume name */
	char *type;		/* what kind of mount */
	dev_t devno;		/* associated device number */
	short toscan;		/* 1 if ok to scan, 0 if not */
} onfs[BUFSIZ];
int nonfs = 0;		/* number in the system */
int mntnotin = 1;	/* 0 when mount info loaded */
int oset = 0;		/* 1 if any -o option given */

/*
 * sort the mounts by device number
 */
fssort(x, y)
register struct mnttype *x, *y;
{
	return(x->devno - y->devno);
}

/*
 * initialize the mount information
 */
mountup()
{
	FILE *fp;		/* file with info on mounted systems */
	register struct mntent *p;	/* points to info about mount */
	struct stat stbuf;	/* status info (dev number) of volume */
	union {
		struct mnttype *x;
		char *c;
	} z;			/* used to align types */

	/*
	 * open the mount file
	 */
	mntnotin = 0;
	if ((fp = setmntent(MOUNTED, "r")) == NULL){
		fprintf(stderr, "%s: can't open %s\n", progname, MOUNTED);
		mexit(1);
	}
	/*
	 * read the info
	 */
	while((p = getmntent(fp)) != NULL){
		/*
		 * can't get info? ignore it!
		 */
		if (stat(p->mnt_dir, &stbuf) < 0){
			perror(p->mnt_dir);
			continue;
		}
		/*
		 * set things up
		 */
		onfs[nonfs].dir = strsave(p->mnt_dir);
		onfs[nonfs].type = strsave(p->mnt_type);
		onfs[nonfs].devno = stbuf.st_dev;
		onfs[nonfs++].toscan = -1;
	}
	/*
	 * close the mount file to conserve descriptors
	 */
	(void) endmntent(fp);
	/*
	 * sort in increasing order of device number
	 */
	z.x = onfs;
	qsort(z.c, nonfs, sizeof(struct mnttype), fssort);
}

/*
 * set all volumes of a certain type to a given value
 * used to mark all nfs-ed volumes scannable (for example)
 */
mountset(type, val)
char *type;		/* type to mark */
int val;		/* how to mark it */
{
	register int i;

	/*
	 * be sure it's loaded
	 */
	if (mntnotin)
		mountup();
		
	/*
	 * usual dumb way ...
	 */
	for(i = 0; i < nonfs; i++)
		if (strcmp(onfs[i].type, type) == 0)
			onfs[i].toscan = val;
}

/*
 * once loaded this resets the unset ones
 * if ANY are -o (ie, restrict to these file systems) they
 *	have been marked as 1; all non-1s go to 0
 * if NONE are -o and ANY are -n (ie, do everything but these)
 *	they have been marked as 0; all non-0s go to 1
 */
mountfix()
{
	register int i;		/* counter in a for loop */

	/*
	 * see if -o is set anywhere
	 */
	for(i = 0; i < nonfs; i++)
		if (onfs[i].toscan == 1){
			oset = 1;
			break;
		}
	/*
	 * reset appropriately
	 */
	for(i = 0; i < nonfs; i++)
		if (oset)
			onfs[i].toscan = (onfs[i].toscan == 1);
		else
			onfs[i].toscan = (onfs[i].toscan != 0);
}

/*
 * return value of toscan for volume on which file is mounted;
 * return inverse of oset if not on any; this has the effect that
 * if -o given, !oset is 0 so scanning should NOT be done (and it's
 * not on a -o file system); but if -n and no -o, !oset is 1 so
 * scanning should be done (and it's not on a -n file system)
 */
mountcheck(stbuf)
struct stat *stbuf;
{
	register int i;	/* counter in a for loop */

	/*
	 * check to see if it's on a volume to be scanned
	 */
	for(i = 0; i < nonfs; i++)
		if (stbuf->st_dev <= onfs[i].devno)
			break;
	/*
	 * if so, indicate whether or not to scan it
	 */
	if (stbuf->st_dev == onfs[i].devno)
		return(onfs[i].toscan);
	/*
	 * otherwise, default action
	 */
	return(!oset);
}
#else
/* ARGSUSED */	mountset(x, y)	{ }
/* ARGSUSED */	mountcheck(x)	{ return(1); }
		mountfix()	{ }
#endif


/*****
 * load checksum file(s)
 *****
 */
char *cname = NULL;	/* file name containing list */

/*
 * set the checksum file name
 */
setchkname(s)
char *s;
{
	cname = s;
}

chsort(x, y)
char **x, **y;
{
	return(strcmp(*x, *y));
}

/*
 * load the checksum names
 */
loadchk(s)
char *s;
{
	FILE *fp;		/* points to checksum file */
	char buf[BUFSIZ];	/* input buffer */
	union {
		char **cp;
		char *c;
	} z;			/* used to align types */

	/*
	 * save checksum file name
	 */
	if (cname == NULL)
		cname = s;
	/*
	 * open it
	 */
	if ((fp = freopen(s, "r", stdin)) == NULL){
		perror(cname);
		return;
	}
	/*
	 * read it
	 */
	while (gets(buf) != NULL){
		if (*buf == '#')
			continue;
		chks[nchks++] = strsave(buf);
	}
	/*
	 * close it
	 */
	(void) fclose(fp);
	cname = NULL;
	/*
	 * sort it
	 */
	z.cp = chks;
	qsort(z.c, nchks, sizeof(char *), chsort);
}

/*****
 * pattern matching
 *****
 */
#ifdef SYSV
#	define	INIT		register char *sp = instring;
#	define	GETC()		(*sp++)
#	define	PEEKC()		(*sp)
#	define	UNGETC(c)	(--sp)
#	define	RETURN(c)	return;
#	define	ERROR(c)	regerr(c)
#	include <regexp.h>
#endif

char *fname = NULL;	/* file name containing patterns */
int lineno;		/* line number */

/*
 * set the ignore file name
 */
setignname(s)
char *s;
{
	fname = s;
}

/*
 * load the patterns
 */
loadpats(s)
char *s;
{
	FILE *fp;		/* points to pattern file */
	char buf[BUFSIZ];	/* input buffer */

	/*
	 * save pattern file name
	 */
	if (fname == NULL)
		fname = s;
	/*
	 * open it
	 */
	if ((fp = freopen(s, "r", stdin)) == NULL){
		perror(fname);
		return;
	}
	/*
	 * read it, and check for errors
	 */
	for(lineno = 1; gets(buf) != NULL; lineno++){
		if (*buf == '#')
			continue;
		if (match("test string", buf) < 0)
			fprintf(stderr,
				"%s: syntax error in ignore file %s line %d\n",
						progname, fname, lineno);
		else{
			if (buf[strlen(buf)-1] == '/'){
				pats[npats].stopmode = S_IFDIR;
				buf[strlen(buf)-1] = '\0';
			}
			pats[npats++].regex = strsave(buf);
		}
	}
	/*
	 * close it
	 */
	(void) fclose(fp);
	fname = NULL;
}

int match(str, pat)
char *str;
char *pat;
{
#ifdef BSD4
	char *p;		/* points to returned error message */
	char *re_comp();	/* pattern compiler */
	int re_exec();		/* pattern checker */

	/*
	 * compile the pattern; croak on error
	 */
	if ((p = re_comp(pat)) != NULL){
		fprintf(stderr, "%s: %s in %s at line %d\n",
		 			progname, p, fname, lineno);
		mexit(1);
	}
	/*
	 * try to match
	 */
	return (re_exec(str));
#endif
#ifdef SYSV
	char patbuf[10*BUFSIZ];	/* buffer for compiled pattern */

	/*
	 * compile the pattern
	 */
	(void) compile(pat, patbuf, &patbuf[10*BUFSIZ], '\0');
	/*
	 * match it
	 */
	return(step(str, patbuf) != 0);
#endif
}

#ifdef SYSV
regerr(n)
int n;
{
	fprintf(stderr, "%s: ", progname);
	switch(n){
	case 11:
		fprintf(stderr, "range endpoint too large");
		break;
	case 16:
		fprintf(stderr, "bad number");
		break;
	case 25:
		fprintf(stderr, "\\digit out of range");
		break;
	case 36:
		fprintf(stderr, "illegal or missing delimiter");
		break;
	case 41:
		fprintf(stderr, "(internal error) no remembered search string");
		break;
	case 42:
		fprintf(stderr, "\\( \\) imbalance");
		break;
	case 43:
		fprintf(stderr, "too many \\(");
		break;
	case 44:
		fprintf(stderr, "more than 2 numbers given in \\{ \\}");
		break;
	case 45:
		fprintf(stderr, "} expected after \\");
		break;
	case 46:
		fprintf(stderr, "first number exceeds second in \\{ \\}");
		break;
	case 49:
		fprintf(stderr, "[ ] imbalance");
		break;
	case 50:
		fprintf(stderr, "regular expression too long");
		break;
	default:
		fprintf(stderr, "unknown regular expression error #%d", n);
		break;
	}
	fprintf(stderr, " in %s at line %d\n", fname, lineno);
	mexit(1);
}
#endif

/*
 * obtain information about a file from its inode
 */
/*
 * some systems use different types for times ...
 * and different function names ...
 */
#ifdef BSD4
#	define TIME_T time_t
#	include <sys/time.h>
#endif
#ifdef SYSV
#	define TIME_T long
#	include <sys/sysmacros.h>
#	include <time.h>
#endif

/*
 * the default checksum environment variable and program
 */
#define CHKSUMVAR	"CHECKSUMMER"		/* environment variable */
#ifdef BSD4
#	define CHKSUMCMD	"/usr/bin/sum %s"	/* default program */
#endif
#ifdef SYSV
#	define CHKSUMCMD	"/bin/sum %s"	/* default program */
#endif

/*
 * names of the months
 */
static char *mos[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
};

/*
 * translate the time into something intelligible
 */
char *gettime(tim)
TIME_T tim;
{
	struct tm *tick;	/* time as a (parseable) structure */
	static char buf[100];	/* bufer for ASCII version of time */

	/*
	 * convert the internal time to something
	 * a bit easier to deal with
	 */
	tick = localtime(&tim);

	/*
	 * print it in human-intelligible format
	 */
	(void) sprintf(buf, "%s %2d, %04d at %02d:%02d:%02d",
		mos[tick->tm_mon], tick->tm_mday, tick->tm_year+1900,
		tick->tm_hour, tick->tm_min, tick->tm_sec);

	/*
	 * return it
	 */
	return(buf);
}
/*
 * this does the file stat and prints out the information as requested
 */
nstat(file, stbuf)
char *file;			/* name of file */
struct stat *stbuf;		/* stat buffer for file */
{
#ifdef S_IFLNK
	register int n;		/* number of chars in name */
#endif
	struct stat lbuf;	/* stat buffer for symbolic link */
	char buf[BUFSIZ];	/* name pointed to */

	/*
	 * print the information
	 */
	prstat(file, stbuf);
#ifdef S_IFLNK
	/*
	 * print the requested information
	 * if a symbolic link, follow it
	 */
	if ((stbuf->st_mode&S_IFMT) == S_IFLNK){
		char mbuf[BUFSIZ];		/* buffer for name(s) */
		if ((n = readlink(file, buf, BUFSIZ)) >= 0){
			buf[n] = '\0';
			(void) sprintf(mbuf, "%s->%s", file, buf);
			if (stat(file, &lbuf) < 0)
				printf("%s (???)\n", mbuf);
			else
				prstat(mbuf, &lbuf);
		}
	}
#endif
}

/*
 * this does the file stat and prints out the information as requested
 */
prstat(file, stbuf)
char *file;			/* name of file */
struct stat *stbuf;		/* stat buffer for file */
{
	register char *cp;		/* used to check for bogus names */
	register char *s;		/* used to walk list of options */
	struct passwd *p;		/* used to map UIDs to user names */
	struct group *g;		/* used to map GIDs to group names */
	char buf[BUFSIZ];		/* for checksum command */
	FILE *fp;			/* for checksum subprocess */

	/*
	 * print the requested information in the order asked for
	 */
	for(s = stflags; *s; s++){
		switch(*s){
		case 'a':		/* time of last access (read) */
			printf("%s", gettime(stbuf->st_atime));
			break;
		case 'c':		/* time inode last changed */
			printf("%s", gettime(stbuf->st_ctime));
			break;
		case 'd':		/* device number */
#ifndef CRAY
			printf("%d", stbuf->st_dev);
#endif CRAY
			break;
		case 'f':		/* file name */
			printf("%s", file);
			break;
		case 'g':		/* group name */
			/*
			 * if you can get the group name,
			 * print that; if not, the group
			 * number will have to do ...
			 */
			if ((g = getgrgid(stbuf->st_gid)) == NULL)
				printf("%d", stbuf->st_gid);
			else if ((cp = index(g->gr_name, ',')) != NULL){
				*cp = '\0';
				printf("%s...", g->gr_name);
			}
			else
				printf("%s", g->gr_name);
			break;
		case 'i':		/* inode number */
			printf("%d", stbuf->st_ino);
			break;
		case 'm':		/* time of last modification */
			printf("%s", gettime(stbuf->st_mtime));
			break;
		case 'n':		/* number of links */
			printf("%d", stbuf->st_nlink);
			break;
		case 'p':		/* protection mode */
			printf("%04o", stbuf->st_mode&07777);
			break;
		case 's':		/* size of file/device numbers */
			switch(stbuf->st_mode&S_IFMT){
			case S_IFCHR:		/* device file; print   */
			case S_IFBLK:		/* major, minor numbers */
				printf("%d,%d", major(stbuf->st_rdev),
						       minor(stbuf->st_rdev));
				break;
			default:		/* other; print size */
				printf("%d", stbuf->st_size);
				break;
			}
			break;
		case 't':		/* type of file */
			switch(stbuf->st_mode&S_IFMT){
			case S_IFDIR:		/* directory */
				putchar('d');
				break;
			case S_IFCHR:		/* character special */
				putchar('c');
				break;
			case S_IFBLK:		/* block special */
				putchar('b');
				break;
			case S_IFREG:		/* regular file */
				putchar('-');
				break;
#ifdef S_IFLNK
			case S_IFLNK:		/* symbolic link */
				putchar('l');
				break;
#endif
#ifdef S_IFSOCK
			case S_IFSOCK:		/* socket */
				putchar('s');
				break;
#endif
#ifdef S_IFIFO
			case S_IFIFO:		/* fifo */
				putchar('f');
				break;
#endif
			default:		/* unknown */
				putchar('?');
				break;
			}
			break;
		case 'u':
			/*
			 * if you can get the user name,
			 * print that; if not, the user
			 * number will have to do ...
			 */
			if ((p = getpwuid(stbuf->st_uid)) == NULL)
				printf("%d", stbuf->st_uid);
			else if ((cp = index(p->pw_name, ',')) != NULL){
				*s = '\0';
				printf("%s...", p->pw_name);
			}
			else
				printf("%s", p->pw_name);
			break;
		case 'x':			/* checksum the file */
			/*
			 * don't compute it if you don't have to
			 */
			if (!doxflag){
				printf("*skipped*");
				break;
			}
			/*
			 * get the checksum program's name
			 */
			if ((cp = getenv(CHKSUMVAR)) == NULL)
				cp = CHKSUMCMD;
			(void) sprintf(buf, cp, file);
			/*
			 * checksum the sucker
			 */
			if ((fp = popen(buf, "r")) != NULL){
				if (fgets(buf, BUFSIZ, fp) != NULL){
					if ((cp = index(buf, '\n')) != NULL)
						*cp = '\0';
					printf("%s", buf);
				}
				else
					printf("*error*");
				(void) pclose(fp);
			}
			else
				printf("*error*");
			break;
		}
		/*
		 * if another field follows, separate the two with a tab
		 */
		if (s[1])
			putchar('\t');
	}
	/*
	 * end the line
	 */
	putchar('\n');
}

