:
#! /bin/sh
#
# This just passes the appropriate command line down to runaudit1.
# It breaks out any auditor and host options specified on the command
# line; the former so it knows to whom to send error messages, the
# latter so it knows where to look for the List file.
#
# Options:
#
# -1	just send one output letter
# -a<name1>,<name2>,...
#	Copies of the audit report will be mailed to <name1>, <name2>, ...
# -c	Checksum all audited files
# -h<host>
#	Run the audit for host <host>.  A listing will be generated
#	remotely, moved to the local host, and the analysis done locally.
# -l<file>
#	Use <file> rather than the default List file.
# -v	Debug mode; print shell input lines as they are read
# -x	Debug mode; print commands and their arguments as they are executed
# -<anything else>
#	Sent as an option to runaudit1
# 
# Matt Bishop
# Research Institute for Advanced Computer Science
# NASA Ames Research Center
# Moffett Field, CA  94035
#
# mab@riacs.edu, ...!{decvax!decwrl,ihnp4!ames}!riacs!mab
#
# (c) Copyright 1986 by Matt Bishop and the Research Institute for
#			Advanced Computer Science
#
# version number
VERSION="RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)"
#
#	system and shell variables
PATH=/usr/ucb:/bin:/usr/bin:%%DESTBIN%%	# execution path
export PATH				# for subprograms
#	relevant programs
MAILER=%%MAILER%%		# program to send mail
SUBJECT=%%SUBJECT%%		# yes if mailer supports command-line subject
NOTIFY=%%AUDITOR%%		# who gets error messages
INTERAUDIT=runaudit1		# audit program
#	relevant files
ERRORS=%%TMPDIR%%/Ae$$		# base of temp files for errors
OUTPUT=%%TMPDIR%%/Ao$$		# output file
#	setup for host
HOST=`hostname | sed 's/\..*//'` # name of the current host
HOSTROOT=%%LISTDIR%%		# all host directories in this directory
HOSTMAP=$HOSTROOT/Equiv		# file to normalize host names
#	options to be passed to audit program
BACKGROUND=no			# background the audits
ONEOUTPUT=no			# many mail letters okay
OPTIONS=			# options to be passed to audit program
DEBUG=				# in debug mode errors go to output!
#	default list of directories to be audited
BINLIST="/bin /usr/bin"		# list of directories to be audited
#
# on interrupt, do some clean up and quit
#
trap "rm -f $ERRORS; exit 1" 1 2 3 15
#
# parse the command line
#
FLAG=				# option waiting for argument
LIST=
LISTF=
COMMAND=$0
CARRY=nothing
for I
do
	COMMAND="$COMMAND $I"
	#
	# if you are expecting an argument, pick it up
	#
	if test -n "$FLAG"
	then
		case $FLAG in
		a)	NOTIFY="$NOTIFY $I"; OPTIONS="$OPTIONS$I";;
		c)	OPTIONS="$OPTIONS$I";;
		h)	HOST=$I; OPTIONS="$OPTIONS$I";;
		l)	LISTF="$LISTF $I";;
		esac
		FLAG=
		continue
	fi
	case $I in
	'&')	BACKGROUND=yes;;
	-1)	OPTIONS="$OPTIONS $I"
		ONEOUTPUT=yes;;
	-c|-d|-g|-ga|-q|-s[Gbcgu]|-u)
		OPTIONS="$OPTIONS $I";;
	-[nort]*)
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-[nor]\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=c
		fi;;
	-a*)				# new list of auditors
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-a\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=a
		else
			NOTIFY=$CARRY
		fi;;
	-h*)				# new host name
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-h\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=h
		else
			HOST="$CARRY"
		fi;;
	-l*)				# list file names
		CARRY=`expr "$I" : '-l\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=l
		else
			LISTF="$LISTF $CARRY"
		fi;;
	-v|-x|-vx|-xv)			# debug mode
		OPTIONS="$OPTIONS $I";
		X=`expr "$I" : '-\(.*\)'`;
		DEBUG="${DEBUG}$X";
		set -$DEBUG;;
	-*)				# bad option -- barf
		echo $0: unknown option \"$i\" >&2
		exit 1;;
	*)				# new file system -- tack it on
		LIST="$LIST $I";;
	esac;
done
#
# set up the debugging option
#
if test "X$DEBUG" != X
then
	DEBUG="-$DEBUG"
fi
#
# canonicalize the host name
#
if test -n "$HOST"
then
	X=
	if test -r $HOSTMAP
	then
		X=`grep "^$HOST[ 	]" $HOSTMAP | awk '{ print $2 }'`
	fi
	if test -n "$X"
	then
		NUMFIELDS=`echo "$X" | awk -F, '{ print NF }'`
		if test "$NUMFIELDS" -eq 0
		then
			HOST="$X"
		elif test "$NUMFIELDS" -eq 1
		then
			HOST=`echo "$X" | awk -F, '{ print $1 }'`
		elif test "$NUMFIELDS" -eq 2
		then
			HOST=`echo "$X" | awk -F, '{ print $2 }'`
		elif test "$NUMFIELDS" -ge 3
		then
			HOST=`echo "$X" | awk -F, '{ print $3 }'`
		fi
	fi
fi
#
# set up the proper files
#
if test -z "$LISTF"
then
	DIRLIST=$HOSTROOT/$HOST/List
else
	for I in $LISTF
	do
		if expr "$I" : '/.*' > /dev/null
		then
			DIRLIST="$DIRLIST $I"
		else
			DIRLIST="$DIRLIST $HOSTROOT/$HOST/$I"
		fi
	done
fi
#
# for various reasons we treat each of the three cases separately:
#
# CASE 1.  Things named on the command line
#
if test "$LIST"
then
	NUMBER=0
	for DIR in $LIST
	do
		TMPOUT=$OUTPUT$NUMBER
		TMPERR=$ERRORS$NUMBER
		NUMBER=`expr $NUMBER + 1`
		X=`echo $DIR | sed 's;/;;g'`
		if test "X$DEBUG" != X
		then
			$INTERAUDIT $OPTIONS $DIR > $TMPOUT
		elif test "$BACKGROUND" = yes
		then
			$INTERAUDIT $OPTIONS $DIR > $TMPOUT 2> $TMPERR &
		else
			$INTERAUDIT $OPTIONS $DIR > $TMPOUT 2> $TMPERR
		fi
	done
#
# CASE 2.  Things listed in a file.  Each line has the directory first, followed
#		by a list of options
#		
elif test -n "$LISTF" -o -r $DIRLIST
then
	NUMBER=0
	for I in $DIRLIST
	do
		sed '/^#/d' $DIRLIST |\
		(\
		while read DIR OPTS ;\
		do\
			TMPOUT=$OUTPUT$NUMBER
			TMPERR=$ERRORS$NUMBER
			NUMBER=`expr $NUMBER + 1`
			X=`echo $DIR | sed 's;/;;g'` ;\
			if test "X$DEBUG" != X
			then
				$INTERAUDIT $OPTIONS $OPTS $DIR > $TMPOUT
			elif test "$BACKGROUND" = yes
			then
				$INTERAUDIT $OPTIONS $OPTS $DIR 2> $TMPERR > $TMPOUT &
			else
				$INTERAUDIT $OPTIONS $OPTS $DIR 2> $TMPERR > $TMPOUT
			fi
		done\
		)
	done
#
# CASE 3.  Things named in the environment variable BINLIST
#
elif test "$BINLIST"
then
	NUMBER=0
	for DIR in $BINLIST
	do
		TMPOUT=$OUTPUT$NUMBER
		TMPERR=$ERRORS$NUMBER
		NUMBER=`expr $NUMBER + 1`
		X=`echo $DIR | sed 's;/;;g'`
		if test "X$DEBUG" != X
		then
			$INTERAUDIT $OPTIONS $DIR > $TMPOUT
		elif test "$BACKGROUND" = yes
		then
			$INTERAUDIT $OPTIONS $DIR > $TMPOUT 2> $TMPERR &
		else
			$INTERAUDIT $OPTIONS $DIR > $TMPOUT 2> $TMPERR
		fi
	done
fi
#
# wait for all children to finish
#
wait
> $ERRORS
for i in $ERRORS*
do
	if test -s $i
	then
		cat $i >> $ERRORS
	fi
done
> $OUTPUT
for i in $OUTPUT*
do
	if test -s $i
	then
		cat $i >> $OUTPUT
	fi
done
#
# if there are errors, mail them to whoever
#
if test -s $ERRORS
then
	echo "$VERSION" > ${ERRORS}0
	echo " " >> ${ERRORS}0
	echo "command: $COMMAND" >> ${ERRORS}0
	echo "host: `hostname`" >> ${ERRORS}0
	echo " " >> ${ERRORS}0
	cat $ERRORS >> ${ERRORS}0
	if test "$SUBJECT" = "yes"
	then
		$MAILER -s "errors from runaudit command" $NOTIFY < ${ERRORS}0
	else
		( echo ' ';\
		  echo '** errors from runaudit command **';\
		  echo ' ';\
		  cat ${ERRORS}0 ; ) | $MAILER $NOTIFY
	fi
fi
#
# if there is any output, mail them to whoever
#
if test -s $OUTPUT
then
	echo "$VERSION" > ${ERRORS}0
	echo " " >> ${ERRORS}0
	echo "command: $COMMAND" >> ${ERRORS}0
	echo "host: `hostname`" >> ${ERRORS}0
	echo " " >> ${ERRORS}0
	cat $OUTPUT >> ${ERRORS}0
	if test "$SUBJECT" = "yes"
	then
		$MAILER -s "output from runaudit command" $NOTIFY < ${ERRORS}0
	else
		( echo ' ';\
		  echo '** output from runaudit command **';\
		  echo ' ';\
		  cat ${ERRORS}0 ; ) | $MAILER $NOTIFY
	fi
fi
#
# delete any error or output file
#
rm -f ${ERRORS}* ${OUTPUT}*
#
# all done
#
exit 0
