/*
 * timeout -- time out a process after a given time if there is no
 *	      change in a named file
 *
 * -o file	if there is a change to file, do NOT time out
 * -s n		signal number to zap descendents with at timeout
 * -t n		time out in n seconds; if n == 0, don't time out
 *
 * Matt Bishop
 * Research Institute for Advanced Computer Science
 * NASA Ames Research Center
 * Moffett Field, CA  94035
 *
 * mab@riacs.edu, ...!{decvax!decwrl,ihnp4!ames}!riacs!mab
 *
 * (c) Copyright 1986 by Matt Bishop and the Research Institute for
 *			Advanced Computer Science
 */
/*
 * If no output to file in num seconds,
 * kill the process.
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#ifdef BSD4
#include <sys/wait.h>
#endif

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

/*
 * macros for consistency
 */
#define	DEFTIMEOUT	3600	/* default time out is 1 hour */
#define	EX_TIMEDOUT	100	/* subprocess timed out */
#define	EX_BADOPT	101	/* invalid option */
#define	EX_CANTRUN	102	/* can't run command */
#define	EX_FILEGONE	103	/* ofile disappeared */

/*
 * This handles the exit status code of a process;
 * TWAIT is the type of the argument to wait(2),
 * EXITCODE(x) is the exit status of the process with pid x, and
 * TERMCODE(x) is the signal which terminated the process with pid x.
 * EXITCODE(x) is meaningless if TERMCODE(x) is nonzero.
 */
#ifdef BSD4
typedef union wait TWAIT;		/* type returned by wait(2) */
#define EXITCODE(x)	((x).w_retcode)	/* exit code of process x */
#define TERMCODE(x)	((x).w_termsig)	/* signal that terminated process x */
#endif
#ifdef SYSV
typedef int TWAIT;			/* type returned by wait(2) */
#define EXITCODE(x)	(((x)>>8)&0xff)	/* exit code of process x */
#define TERMCODE(x)	((x)&0xff)	/* signal that terminated process x */
#endif

/*
 * This handles the child process termination; since the child might
 * itself spawn children, we set the child's process group to be that
 * of this process.  Then signalling the process group gets all the
 * descendents since the process group ID is inherited.
 */
#ifdef BSD4
#define SETPG(x)	(void) setpgrp(x, x)	/* set proc group ID to x */
#define KILLPG(x,sig)	(void) killpg(x, sig)	/* signal a process group */
#endif
#ifdef SYSV
#define SETPG(x)	(void) setpgrp()	/* set proc group ID to x */
#define KILLPG(x,sig)	(void) kill(-(x), sig)	/* signal a process group */
#endif

/*
 * The last modification time of the output file, if any, is checked;
 * since the type of a time varies, this parameterizes it
 */
typedef long TTIME;			/* type of time */

/*
 * global variables
 */
char *progname = "timeout";		/* program name */
int timeout = DEFTIMEOUT;		/* time out interval */
char *ofile = NULL;			/* output file name */
TTIME modtime;				/* time ofile was last modified */
char **cmdargs;				/* list of args making up command */
char command[10 * BUFSIZ];		/* command as a string */
int compid = 0;				/* command's (child's) PID */
int termin = SIGTERM;			/* signal to zap descendents with */
int (*sigstat[NSIG])();			/* original signal settings */

/*
 * After timeout seconds, an alarm signal is sent to this process.
 * This routine traps that signal, checks for any activity, and if
 * there is none kills the command.  Any other signal is passed on
 * to the child, and if none (or if the child terminates) causes
 * the parent to be so signalled.
 */
int sigtrap(), wakeup();

main(argc, argv)
int argc;
char **argv;
{
	register int i;			/* counter in a for loop */
	TWAIT status;			/* status of command */
	int w;				/* return value of wait(2) */

	/*
	 * process the argument list
	 */
	progname = argv[0];
	for(i = 1; i < argc && *argv[i] == '-'; i++){
		switch(argv[i][1]){
		case 'o':			/* get output file name */
			if (argv[i][2] == '\0')
				usage("no file name after -o");
			else
				ofile = &argv[i][2];
			break;
		case 's':			/* get signal number */
			if (argv[i][2] == '\0')
				usage("no signal number after -s");
			else if ((termin = atoi(&argv[i][2])) < 1 ||
							termin >= NSIG)
				usage("invalid signal number");
			break;
		case 't':			/* get timeout interval */
			if (argv[i][2] == '\0')
				usage("no timeout after -t");
			else if ((timeout = atoi(&argv[i][2])) < 0)
				usage("invalid timeout");
			break;
		default:
			usage("bad option");
		}
	}

	/*
	 * the rest is the command
	 */
	if (argv[i] == NULL)
		usage("no command");
	cmdargs = &argv[i];
	(void) strcpy(command, cmdargs[0]);
	for(i = 1; cmdargs[i] != NULL; i++){
		(void) strcat(command, " ");
		(void) strcat(command, cmdargs[i]);
	}

	/*
	 * if there is an ofile, its last mod time
	 * is the current time
	 */
	if (ofile != NULL)
		modtime = time(0L);

	/*
	 * execute the command
	 */
	runcomm(cmdargs);

	/*
	 * set up the signal traps
	 */
	for(i = 0; i < NSIG; i++)
		/* next line screws up on SunOS 4.0; TURKEYS FROM TURLOCK!!!*/
		/*if ((sigstat[i] = signal(i, sigtrap)) == SIG_IGN) */
			(void) signal(i, SIG_IGN);
	if (timeout > 0){
		(void) signal(SIGALRM, wakeup);
		alarm(timeout);
	}

	/*
	 * wait to see what happens
	 */
	while((w = wait(&status)) != -1 && w != compid);

	/*
	 * done; release the signal trap
	 */
	for(i = 0; i < NSIG; i++)
		(void) signal(i, sigstat[i]);
	
	/*
	 * if killed by a signal, do the same
	 */
	if (TERMCODE(status) != 0){
fprintf(stderr, "%s: subprocess terminated by signal %d\n", progname, TERMCODE(status));
		kill(getpid(), TERMCODE(status));
	}

	/*
	 * return the child's exit status
	 */
	exit(EXITCODE(status));
}

/*
 * poor usage
 */
usage(s)
char *s;
{
	fprintf(stderr, "%s: %s\n", progname, s);
	exit(EX_BADOPT);
}

/*
 * handle any kind of signal other than an alarm
 */
sigtrap(signo)
int signo;
{
	/*
	 * ignore any future occurrances of this signal
	 */
	(void) signal(signo, SIG_IGN);

	/*
	 * signal the child if it exists;
	 * if not, follow the default action
	 */
	if (compid == 0)
		kill(getpid(), signo);
	else{
		(void) signal(signo, sigstat[signo]);
		kill(compid, signo);
	}

	/*
	 * if still around, reset the signal trap
	 */
	(void) signal(signo, sigtrap);
}

/*
 * handle an alarm signal
 */
wakeup(signo)
int signo;
{
	struct stat stbuf;		/* for ststing ofile */

	/*
	 * ignore any future occurrances of this signal
	 */
	(void) signal(signo, SIG_IGN);

	/*
	 * if no output file being scanned, kill the process
	 */
	if (ofile == NULL){
		fprintf(stderr, "%s: command \"%s\" timed out after %d seconds\n",
				progname, command, timeout);
		KILLPG(compid, termin);
		exit(EX_TIMEDOUT);
	}

	/*
	 * child not yet dead -- see if the output
	 * file has been modified lately
	 */
	if (stat(ofile, &stbuf) < 0){
		perror(ofile);
		exit(EX_FILEGONE);
	}

	/*
	 * nope -- time the sucker out
	 */
	if (stbuf.st_mtime == modtime){
		fprintf(stderr, "%s: command \"%s\" timed out after %d seconds\n",
				progname, command, timeout);
		KILLPG(compid, termin);
		exit(EX_TIMEDOUT);
	}

	/*
	 * yup -- update timeout time
	 */
	modtime = stbuf.st_mtime;

	/*
	 * reset signal
	 */
	(void) signal(signo, wakeup);

	/*
	 * reset timer
	 */
	alarm(timeout);
}

runcomm(argv)
char **argv;
{
	/*
	 * fork and start the child process
	 */
	if ((compid = fork()) == 0){
		/*
		 * set the process group ID here so the parent
		 * can shoot ALL descendents
		 */
		SETPG(getpid());
		/*
		 * execl it; use /bin/sh to find the right program
		 */
		(void) execvp(argv[0], argv);
	
		/*
		 * if you get down here, something's REALLY wrong!
		 */
		perror(argv[0]);
		_exit(EX_CANTRUN);
	}
}

