/* Mapping tables for GBK handling.
   Copyright (C) 1999, 2000, 2001, 2002, 2004, 2011 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Sean Chen <seanc@turbolinux.com.cn>, 1999.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.  */

/*	This file is added for supporting gb18030-2000 code.
	by Wang ShouHua & Wu Jian
	Redflag Linux company
	www.redflag-linux.com
*/

/* The GB18030 standard was re-released on Nov. 2000 and many code values
 * were changed. The reimplementation of this module is based on the info
 * from Markus Scherer's site:
 * http://oss.software.ibm.com/cvs/icu/~checkout~/icu/source/tools/makeconv/gb18030/gb18030.html
 *
 *   Original - Yong Li <rigel863@yahoo.com>  Jan. 28, 2001
 */

/*   Dec 20 2001 Yu Shao <yshao@redhat.com>
 *   Revision, fixing bugs, adding missing entries.
 */

#include <dlfcn.h>
#include <gconv.h>
#include <stdint.h>
#include <wchar.h>

/* The tables in this file are generated from the charmap.
   In a first step, the charmap is converted to a simple table format:

      ./tst-table-charmap.sh GB18030 < ../localedata/charmaps/GB18030 \
      > GB18030.table
*/

/* Table for GB18030 -> UCS-4, containing the two-byte characters only.
   Generated through

   clisp tab.lisp < GB18030.table

   where tab.lisp is:
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   (locally (declare (compile))
     (let ((count 0))
       (loop
	 (let ((line (read-line nil nil nil)))
	   (unless line (return))
	   (when (eql (position #\Tab line) 6)
	     (let ((gbk (parse-integer line :start 2 :end 6 :radix 16))
		   (ucs (parse-integer line :start 9 :end 13 :radix 16)))
	       (when (= (mod count 4) 0)
		 (format t " "))
	       (format t " ~([0x~4,'0X] = 0x~4,'0X~),"
			 (let ((ch1 (ldb (byte 8 8) gbk))
			       (ch2 (ldb (byte 8 0) gbk)))
			   (+ (* (- ch1 #x81) 192) (- ch2 #x40)))
			 ucs)
	       (incf count)
	       (when (= (mod count 4) 0)
		 (format t "~%"))))))
       (format t "~%")))
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

static const uint16_t __twobyte_to_ucs[] =
{
  [0x0000] = 0x4e02, [0x0001] = 0x4e04, [0x0002] = 0x4e05, [0x0003] = 0x4e06,
  [0x0004] = 0x4e0f, [0x0005] = 0x4e12, [0x0006] = 0x4e17, [0x0007] = 0x4e1f,
  [0x0008] = 0x4e20, [0x0009] = 0x4e21, [0x000a] = 0x4e23, [0x000b] = 0x4e26,
  [0x000c] = 0x4e29, [0x000d] = 0x4e2e, [0x000e] = 0x4e2f, [0x000f] = 0x4e31,
  [0x0010] = 0x4e33, [0x0011] = 0x4e35, [0x0012] = 0x4e37, [0x0013] = 0x4e3c,
  [0x0014] = 0x4e40, [0x0015] = 0x4e41, [0x0016] = 0x4e42, [0x0017] = 0x4e44,
  [0x0018] = 0x4e46, [0x0019] = 0x4e4a, [0x001a] = 0x4e51, [0x001b] = 0x4e55,
  [0x001c] = 0x4e57, [0x001d] = 0x4e5a, [0x001e] = 0x4e5b, [0x001f] = 0x4e62,
  [0x0020] = 0x4e63, [0x0021] = 0x4e64, [0x0022] = 0x4e65, [0x0023] = 0x4e67,
  [0x0024] = 0x4e68, [0x0025] = 0x4e6a, [0x0026] = 0x4e6b, [0x0027] = 0x4e6c,
  [0x0028] = 0x4e6d, [0x0029] = 0x4e6e, [0x002a] = 0x4e6f, [0x002b] = 0x4e72,
  [0x002c] = 0x4e74, [0x002d] = 0x4e75, [0x002e] = 0x4e76, [0x002f] = 0x4e77,
  [0x0030] = 0x4e78, [0x0031] = 0x4e79, [0x0032] = 0x4e7a, [0x0033] = 0x4e7b,
  [0x0034] = 0x4e7c, [0x0035] = 0x4e7d, [0x0036] = 0x4e7f, [0x0037] = 0x4e80,
  [0x0038] = 0x4e81, [0x0039] = 0x4e82, [0x003a] = 0x4e83, [0x003b] = 0x4e84,
  [0x003c] = 0x4e85, [0x003d] = 0x4e87, [0x003e] = 0x4e8a, [0x0040] = 0x4e90,
  [0x0041] = 0x4e96, [0x0042] = 0x4e97, [0x0043] = 0x4e99, [0x0044] = 0x4e9c,
  [0x0045] = 0x4e9d, [0x0046] = 0x4e9e, [0x0047] = 0x4ea3, [0x0048] = 0x4eaa,
  [0x0049] = 0x4eaf, [0x004a] = 0x4eb0, [0x004b] = 0x4eb1, [0x004c] = 0x4eb4,
  [0x004d] = 0x4eb6, [0x004e] = 0x4eb7, [0x004f] = 0x4eb8, [0x0050] = 0x4eb9,
  [0x0051] = 0x4ebc, [0x0052] = 0x4ebd, [0x0053] = 0x4ebe, [0x0054] = 0x4ec8,
  [0x0055] = 0x4ecc, [0x0056] = 0x4ecf, [0x0057] = 0x4ed0, [0x0058] = 0x4ed2,
  [0x0059] = 0x4eda, [0x005a] = 0x4edb, [0x005b] = 0x4edc, [0x005c] = 0x4ee0,
  [0x005d] = 0x4ee2, [0x005e] = 0x4ee6, [0x005f] = 0x4ee7, [0x0060] = 0x4ee9,
  [0x0061] = 0x4eed, [0x0062] = 0x4eee, [0x0063] = 0x4eef, [0x0064] = 0x4ef1,
  [0x0065] = 0x4ef4, [0x0066] = 0x4ef8, [0x0067] = 0x4ef9, [0x0068] = 0x4efa,
  [0x0069] = 0x4efc, [0x006a] = 0x4efe, [0x006b] = 0x4f00, [0x006c] = 0x4f02,
  [0x006d] = 0x4f03, [0x006e] = 0x4f04, [0x006f] = 0x4f05, [0x0070] = 0x4f06,
  [0x0071] = 0x4f07, [0x0072] = 0x4f08, [0x0073] = 0x4f0b, [0x0074] = 0x4f0c,
  [0x0075] = 0x4f12, [0x0076] = 0x4f13, [0x0077] = 0x4f14, [0x0078] = 0x4f15,
  [0x0079] = 0x4f16, [0x007a] = 0x4f1c, [0x007b] = 0x4f1d, [0x007c] = 0x4f21,
  [0x007d] = 0x4f23, [0x007e] = 0x4f28, [0x007f] = 0x4f29, [0x0080] = 0x4f2c,
  [0x0081] = 0x4f2d, [0x0082] = 0x4f2e, [0x0083] = 0x4f31, [0x0084] = 0x4f33,
  [0x0085] = 0x4f35, [0x0086] = 0x4f37, [0x0087] = 0x4f39, [0x0088] = 0x4f3b,
  [0x0089] = 0x4f3e, [0x008a] = 0x4f3f, [0x008b] = 0x4f40, [0x008c] = 0x4f41,
  [0x008d] = 0x4f42, [0x008e] = 0x4f44, [0x008f] = 0x4f45, [0x0090] = 0x4f47,
  [0x0091] = 0x4f48, [0x0092] = 0x4f49, [0x0093] = 0x4f4a, [0x0094] = 0x4f4b,
  [0x0095] = 0x4f4c, [0x0096] = 0x4f52, [0x0097] = 0x4f54, [0x0098] = 0x4f56,
  [0x0099] = 0x4f61, [0x009a] = 0x4f62, [0x009b] = 0x4f66, [0x009c] = 0x4f68,
  [0x009d] = 0x4f6a, [0x009e] = 0x4f6b, [0x009f] = 0x4f6d, [0x00a0] = 0x4f6e,
  [0x00a1] = 0x4f71, [0x00a2] = 0x4f72, [0x00a3] = 0x4f75, [0x00a4] = 0x4f77,
  [0x00a5] = 0x4f78, [0x00a6] = 0x4f79, [0x00a7] = 0x4f7a, [0x00a8] = 0x4f7d,
  [0x00a9] = 0x4f80, [0x00aa] = 0x4f81, [0x00ab] = 0x4f82, [0x00ac] = 0x4f85,
  [0x00ad] = 0x4f86, [0x00ae] = 0x4f87, [0x00af] = 0x4f8a, [0x00b0] = 0x4f8c,
  [0x00b1] = 0x4f8e, [0x00b2] = 0x4f90, [0x00b3] = 0x4f92, [0x00b4] = 0x4f93,
  [0x00b5] = 0x4f95, [0x00b6] = 0x4f96, [0x00b7] = 0x4f98, [0x00b8] = 0x4f99,
  [0x00b9] = 0x4f9a, [0x00ba] = 0x4f9c, [0x00bb] = 0x4f9e, [0x00bc] = 0x4f9f,
  [0x00bd] = 0x4fa1, [0x00be] = 0x4fa2, [0x00c0] = 0x4fa4, [0x00c1] = 0x4fab,
  [0x00c2] = 0x4fad, [0x00c3] = 0x4fb0, [0x00c4] = 0x4fb1, [0x00c5] = 0x4fb2,
  [0x00c6] = 0x4fb3, [0x00c7] = 0x4fb4, [0x00c8] = 0x4fb6, [0x00c9] = 0x4fb7,
  [0x00ca] = 0x4fb8, [0x00cb] = 0x4fb9, [0x00cc] = 0x4fba, [0x00cd] = 0x4fbb,
  [0x00ce] = 0x4fbc, [0x00cf] = 0x4fbd, [0x00d0] = 0x4fbe, [0x00d1] = 0x4fc0,
  [0x00d2] = 0x4fc1, [0x00d3] = 0x4fc2, [0x00d4] = 0x4fc6, [0x00d5] = 0x4fc7,
  [0x00d6] = 0x4fc8, [0x00d7] = 0x4fc9, [0x00d8] = 0x4fcb, [0x00d9] = 0x4fcc,
  [0x00da] = 0x4fcd, [0x00db] = 0x4fd2, [0x00dc] = 0x4fd3, [0x00dd] = 0x4fd4,
  [0x00de] = 0x4fd5, [0x00df] = 0x4fd6, [0x00e0] = 0x4fd9, [0x00e1] = 0x4fdb,
  [0x00e2] = 0x4fe0, [0x00e3] = 0x4fe2, [0x00e4] = 0x4fe4, [0x00e5] = 0x4fe5,
  [0x00e6] = 0x4fe7, [0x00e7] = 0x4feb, [0x00e8] = 0x4fec, [0x00e9] = 0x4ff0,
  [0x00ea] = 0x4ff2, [0x00eb] = 0x4ff4, [0x00ec] = 0x4ff5, [0x00ed] = 0x4ff6,
  [0x00ee] = 0x4ff7, [0x00ef] = 0x4ff9, [0x00f0] = 0x4ffb, [0x00f1] = 0x4ffc,
  [0x00f2] = 0x4ffd, [0x00f3] = 0x4fff, [0x00f4] = 0x5000, [0x00f5] = 0x5001,
  [0x00f6] = 0x5002, [0x00f7] = 0x5003, [0x00f8] = 0x5004, [0x00f9] = 0x5005,
  [0x00fa] = 0x5006, [0x00fb] = 0x5007, [0x00fc] = 0x5008, [0x00fd] = 0x5009,
  [0x00fe] = 0x500a, [0x0100] = 0x500b, [0x0101] = 0x500e, [0x0102] = 0x5010,
  [0x0103] = 0x5011, [0x0104] = 0x5013, [0x0105] = 0x5015, [0x0106] = 0x5016,
  [0x0107] = 0x5017, [0x0108] = 0x501b, [0x0109] = 0x501d, [0x010a] = 0x501e,
  [0x010b] = 0x5020, [0x010c] = 0x5022, [0x010d] = 0x5023, [0x010e] = 0x5024,
  [0x010f] = 0x5027, [0x0110] = 0x502b, [0x0111] = 0x502f, [0x0112] = 0x5030,
  [0x0113] = 0x5031, [0x0114] = 0x5032, [0x0115] = 0x5033, [0x0116] = 0x5034,
  [0x0117] = 0x5035, [0x0118] = 0x5036, [0x0119] = 0x5037, [0x011a] = 0x5038,
  [0x011b] = 0x5039, [0x011c] = 0x503b, [0x011d] = 0x503d, [0x011e] = 0x503f,
  [0x011f] = 0x5040, [0x0120] = 0x5041, [0x0121] = 0x5042, [0x0122] = 0x5044,
  [0x0123] = 0x5045, [0x0124] = 0x5046, [0x0125] = 0x5049, [0x0126] = 0x504a,
  [0x0127] = 0x504b, [0x0128] = 0x504d, [0x0129] = 0x5050, [0x012a] = 0x5051,
  [0x012b] = 0x5052, [0x012c] = 0x5053, [0x012d] = 0x5054, [0x012e] = 0x5056,
  [0x012f] = 0x5057, [0x0130] = 0x5058, [0x0131] = 0x5059, [0x0132] = 0x505b,
  [0x0133] = 0x505d, [0x0134] = 0x505e, [0x0135] = 0x505f, [0x0136] = 0x5060,
  [0x0137] = 0x5061, [0x0138] = 0x5062, [0x0139] = 0x5063, [0x013a] = 0x5064,
  [0x013b] = 0x5066, [0x013c] = 0x5067, [0x013d] = 0x5068, [0x013e] = 0x5069,
  [0x013f] = 0x506a, [0x0140] = 0x506b, [0x0141] = 0x506d, [0x0142] = 0x506e,
  [0x0143] = 0x506f, [0x0144] = 0x5070, [0x0145] = 0x5071, [0x0146] = 0x5072,
  [0x0147] = 0x5073, [0x0148] = 0x5074, [0x0149] = 0x5075, [0x014a] = 0x5078,
  [0x014b] = 0x5079, [0x014c] = 0x507a, [0x014d] = 0x507c, [0x014e] = 0x507d,
  [0x014f] = 0x5081, [0x0150] = 0x5082, [0x0151] = 0x5083, [0x0152] = 0x5084,
  [0x0153] = 0x5086, [0x0154] = 0x5087, [0x0155] = 0x5089, [0x0156] = 0x508a,
  [0x0157] = 0x508b, [0x0158] = 0x508c, [0x0159] = 0x508e, [0x015a] = 0x508f,
  [0x015b] = 0x5090, [0x015c] = 0x5091, [0x015d] = 0x5092, [0x015e] = 0x5093,
  [0x015f] = 0x5094, [0x0160] = 0x5095, [0x0161] = 0x5096, [0x0162] = 0x5097,
  [0x0163] = 0x5098, [0x0164] = 0x5099, [0x0165] = 0x509a, [0x0166] = 0x509b,
  [0x0167] = 0x509c, [0x0168] = 0x509d, [0x0169] = 0x509e, [0x016a] = 0x509f,
  [0x016b] = 0x50a0, [0x016c] = 0x50a1, [0x016d] = 0x50a2, [0x016e] = 0x50a4,
  [0x016f] = 0x50a6, [0x0170] = 0x50aa, [0x0171] = 0x50ab, [0x0172] = 0x50ad,
  [0x0173] = 0x50ae, [0x0174] = 0x50af, [0x0175] = 0x50b0, [0x0176] = 0x50b1,
  [0x0177] = 0x50b3, [0x0178] = 0x50b4, [0x0179] = 0x50b5, [0x017a] = 0x50b6,
  [0x017b] = 0x50b7, [0x017c] = 0x50b8, [0x017d] = 0x50b9, [0x017e] = 0x50bc,
  [0x0180] = 0x50bd, [0x0181] = 0x50be, [0x0182] = 0x50bf, [0x0183] = 0x50c0,
  [0x0184] = 0x50c1, [0x0185] = 0x50c2, [0x0186] = 0x50c3, [0x0187] = 0x50c4,
  [0x0188] = 0x50c5, [0x0189] = 0x50c6, [0x018a] = 0x50c7, [0x018b] = 0x50c8,
  [0x018c] = 0x50c9, [0x018d] = 0x50ca, [0x018e] = 0x50cb, [0x018f] = 0x50cc,
  [0x0190] = 0x50cd, [0x0191] = 0x50ce, [0x0192] = 0x50d0, [0x0193] = 0x50d1,
  [0x0194] = 0x50d2, [0x0195] = 0x50d3, [0x0196] = 0x50d4, [0x0197] = 0x50d5,
  [0x0198] = 0x50d7, [0x0199] = 0x50d8, [0x019a] = 0x50d9, [0x019b] = 0x50db,
  [0x019c] = 0x50dc, [0x019d] = 0x50dd, [0x019e] = 0x50de, [0x019f] = 0x50df,
  [0x01a0] = 0x50e0, [0x01a1] = 0x50e1, [0x01a2] = 0x50e2, [0x01a3] = 0x50e3,
  [0x01a4] = 0x50e4, [0x01a5] = 0x50e5, [0x01a6] = 0x50e8, [0x01a7] = 0x50e9,
  [0x01a8] = 0x50ea, [0x01a9] = 0x50eb, [0x01aa] = 0x50ef, [0x01ab] = 0x50f0,
  [0x01ac] = 0x50f1, [0x01ad] = 0x50f2, [0x01ae] = 0x50f4, [0x01af] = 0x50f6,
  [0x01b0] = 0x50f7, [0x01b1] = 0x50f8, [0x01b2] = 0x50f9, [0x01b3] = 0x50fa,
  [0x01b4] = 0x50fc, [0x01b5] = 0x50fd, [0x01b6] = 0x50fe, [0x01b7] = 0x50ff,
  [0x01b8] = 0x5100, [0x01b9] = 0x5101, [0x01ba] = 0x5102, [0x01bb] = 0x5103,
  [0x01bc] = 0x5104, [0x01bd] = 0x5105, [0x01be] = 0x5108, [0x01c0] = 0x5109,
  [0x01c1] = 0x510a, [0x01c2] = 0x510c, [0x01c3] = 0x510d, [0x01c4] = 0x510e,
  [0x01c5] = 0x510f, [0x01c6] = 0x5110, [0x01c7] = 0x5111, [0x01c8] = 0x5113,
  [0x01c9] = 0x5114, [0x01ca] = 0x5115, [0x01cb] = 0x5116, [0x01cc] = 0x5117,
  [0x01cd] = 0x5118, [0x01ce] = 0x5119, [0x01cf] = 0x511a, [0x01d0] = 0x511b,
  [0x01d1] = 0x511c, [0x01d2] = 0x511d, [0x01d3] = 0x511e, [0x01d4] = 0x511f,
  [0x01d5] = 0x5120, [0x01d6] = 0x5122, [0x01d7] = 0x5123, [0x01d8] = 0x5124,
  [0x01d9] = 0x5125, [0x01da] = 0x5126, [0x01db] = 0x5127, [0x01dc] = 0x5128,
  [0x01dd] = 0x5129, [0x01de] = 0x512a, [0x01df] = 0x512b, [0x01e0] = 0x512c,
  [0x01e1] = 0x512d, [0x01e2] = 0x512e, [0x01e3] = 0x512f, [0x01e4] = 0x5130,
  [0x01e5] = 0x5131, [0x01e6] = 0x5132, [0x01e7] = 0x5133, [0x01e8] = 0x5134,
  [0x01e9] = 0x5135, [0x01ea] = 0x5136, [0x01eb] = 0x5137, [0x01ec] = 0x5138,
  [0x01ed] = 0x5139, [0x01ee] = 0x513a, [0x01ef] = 0x513b, [0x01f0] = 0x513c,
  [0x01f1] = 0x513d, [0x01f2] = 0x513e, [0x01f3] = 0x5142, [0x01f4] = 0x5147,
  [0x01f5] = 0x514a, [0x01f6] = 0x514c, [0x01f7] = 0x514e, [0x01f8] = 0x514f,
  [0x01f9] = 0x5150, [0x01fa] = 0x5152, [0x01fb] = 0x5153, [0x01fc] = 0x5157,
  [0x01fd] = 0x5158, [0x01fe] = 0x5159, [0x01ff] = 0x515b, [0x0200] = 0x515d,
  [0x0201] = 0x515e, [0x0202] = 0x515f, [0x0203] = 0x5160, [0x0204] = 0x5161,
  [0x0205] = 0x5163, [0x0206] = 0x5164, [0x0207] = 0x5166, [0x0208] = 0x5167,
  [0x0209] = 0x5169, [0x020a] = 0x516a, [0x020b] = 0x516f, [0x020c] = 0x5172,
  [0x020d] = 0x517a, [0x020e] = 0x517e, [0x020f] = 0x517f, [0x0210] = 0x5183,
  [0x0211] = 0x5184, [0x0212] = 0x5186, [0x0213] = 0x5187, [0x0214] = 0x518a,
  [0x0215] = 0x518b, [0x0216] = 0x518e, [0x0217] = 0x518f, [0x0218] = 0x5190,
  [0x0219] = 0x5191, [0x021a] = 0x5193, [0x021b] = 0x5194, [0x021c] = 0x5198,
  [0x021d] = 0x519a, [0x021e] = 0x519d, [0x021f] = 0x519e, [0x0220] = 0x519f,
  [0x0221] = 0x51a1, [0x0222] = 0x51a3, [0x0223] = 0x51a6, [0x0224] = 0x51a7,
  [0x0225] = 0x51a8, [0x0226] = 0x51a9, [0x0227] = 0x51aa, [0x0228] = 0x51ad,
  [0x0229] = 0x51ae, [0x022a] = 0x51b4, [0x022b] = 0x51b8, [0x022c] = 0x51b9,
  [0x022d] = 0x51ba, [0x022e] = 0x51be, [0x022f] = 0x51bf, [0x0230] = 0x51c1,
  [0x0231] = 0x51c2, [0x0232] = 0x51c3, [0x0233] = 0x51c5, [0x0234] = 0x51c8,
  [0x0235] = 0x51ca, [0x0236] = 0x51cd, [0x0237] = 0x51ce, [0x0238] = 0x51d0,
  [0x0239] = 0x51d2, [0x023a] = 0x51d3, [0x023b] = 0x51d4, [0x023c] = 0x51d5,
  [0x023d] = 0x51d6, [0x023e] = 0x51d7, [0x0240] = 0x51d8, [0x0241] = 0x51d9,
  [0x0242] = 0x51da, [0x0243] = 0x51dc, [0x0244] = 0x51de, [0x0245] = 0x51df,
  [0x0246] = 0x51e2, [0x0247] = 0x51e3, [0x0248] = 0x51e5, [0x0249] = 0x51e6,
  [0x024a] = 0x51e7, [0x024b] = 0x51e8, [0x024c] = 0x51e9, [0x024d] = 0x51ea,
  [0x024e] = 0x51ec, [0x024f] = 0x51ee, [0x0250] = 0x51f1, [0x0251] = 0x51f2,
  [0x0252] = 0x51f4, [0x0253] = 0x51f7, [0x0254] = 0x51fe, [0x0255] = 0x5204,
  [0x0256] = 0x5205, [0x0257] = 0x5209, [0x0258] = 0x520b, [0x0259] = 0x520c,
  [0x025a] = 0x520f, [0x025b] = 0x5210, [0x025c] = 0x5213, [0x025d] = 0x5214,
  [0x025e] = 0x5215, [0x025f] = 0x521c, [0x0260] = 0x521e, [0x0261] = 0x521f,
  [0x0262] = 0x5221, [0x0263] = 0x5222, [0x0264] = 0x5223, [0x0265] = 0x5225,
  [0x0266] = 0x5226, [0x0267] = 0x5227, [0x0268] = 0x522a, [0x0269] = 0x522c,
  [0x026a] = 0x522f, [0x026b] = 0x5231, [0x026c] = 0x5232, [0x026d] = 0x5234,
  [0x026e] = 0x5235, [0x026f] = 0x523c, [0x0270] = 0x523e, [0x0271] = 0x5244,
  [0x0272] = 0x5245, [0x0273] = 0x5246, [0x0274] = 0x5247, [0x0275] = 0x5248,
  [0x0276] = 0x5249, [0x0277] = 0x524b, [0x0278] = 0x524e, [0x0279] = 0x524f,
  [0x027a] = 0x5252, [0x027b] = 0x5253, [0x027c] = 0x5255, [0x027d] = 0x5257,
  [0x027e] = 0x5258, [0x0280] = 0x5259, [0x0281] = 0x525a, [0x0282] = 0x525b,
  [0x0283] = 0x525d, [0x0284] = 0x525f, [0x0285] = 0x5260, [0x0286] = 0x5262,
  [0x0287] = 0x5263, [0x0288] = 0x5264, [0x0289] = 0x5266, [0x028a] = 0x5268,
  [0x028b] = 0x526b, [0x028c] = 0x526c, [0x028d] = 0x526d, [0x028e] = 0x526e,
  [0x028f] = 0x5270, [0x0290] = 0x5271, [0x0291] = 0x5273, [0x0292] = 0x5274,
  [0x0293] = 0x5275, [0x0294] = 0x5276, [0x0295] = 0x5277, [0x0296] = 0x5278,
  [0x0297] = 0x5279, [0x0298] = 0x527a, [0x0299] = 0x527b, [0x029a] = 0x527c,
  [0x029b] = 0x527e, [0x029c] = 0x5280, [0x029d] = 0x5283, [0x029e] = 0x5284,
  [0x029f] = 0x5285, [0x02a0] = 0x5286, [0x02a1] = 0x5287, [0x02a2] = 0x5289,
  [0x02a3] = 0x528a, [0x02a4] = 0x528b, [0x02a5] = 0x528c, [0x02a6] = 0x528d,
  [0x02a7] = 0x528e, [0x02a8] = 0x528f, [0x02a9] = 0x5291, [0x02aa] = 0x5292,
  [0x02ab] = 0x5294, [0x02ac] = 0x5295, [0x02ad] = 0x5296, [0x02ae] = 0x5297,
  [0x02af] = 0x5298, [0x02b0] = 0x5299, [0x02b1] = 0x529a, [0x02b2] = 0x529c,
  [0x02b3] = 0x52a4, [0x02b4] = 0x52a5, [0x02b5] = 0x52a6, [0x02b6] = 0x52a7,
  [0x02b7] = 0x52ae, [0x02b8] = 0x52af, [0x02b9] = 0x52b0, [0x02ba] = 0x52b4,
  [0x02bb] = 0x52b5, [0x02bc] = 0x52b6, [0x02bd] = 0x52b7, [0x02be] = 0x52b8,
  [0x02bf] = 0x52b9, [0x02c0] = 0x52ba, [0x02c1] = 0x52bb, [0x02c2] = 0x52bc,
  [0x02c3] = 0x52bd, [0x02c4] = 0x52c0, [0x02c5] = 0x52c1, [0x02c6] = 0x52c2,
  [0x02c7] = 0x52c4, [0x02c8] = 0x52c5, [0x02c9] = 0x52c6, [0x02ca] = 0x52c8,
  [0x02cb] = 0x52ca, [0x02cc] = 0x52cc, [0x02cd] = 0x52cd, [0x02ce] = 0x52ce,
  [0x02cf] = 0x52cf, [0x02d0] = 0x52d1, [0x02d1] = 0x52d3, [0x02d2] = 0x52d4,
  [0x02d3] = 0x52d5, [0x02d4] = 0x52d7, [0x02d5] = 0x52d9, [0x02d6] = 0x52da,
  [0x02d7] = 0x52db, [0x02d8] = 0x52dc, [0x02d9] = 0x52dd, [0x02da] = 0x52de,
  [0x02db] = 0x52e0, [0x02dc] = 0x52e1, [0x02dd] = 0x52e2, [0x02de] = 0x52e3,
  [0x02df] = 0x52e5, [0x02e0] = 0x52e6, [0x02e1] = 0x52e7, [0x02e2] = 0x52e8,
  [0x02e3] = 0x52e9, [0x02e4] = 0x52ea, [0x02e5] = 0x52eb, [0x02e6] = 0x52ec,
  [0x02e7] = 0x52ed, [0x02e8] = 0x52ee, [0x02e9] = 0x52ef, [0x02ea] = 0x52f1,
  [0x02eb] = 0x52f2, [0x02ec] = 0x52f3, [0x02ed] = 0x52f4, [0x02ee] = 0x52f5,
  [0x02ef] = 0x52f6, [0x02f0] = 0x52f7, [0x02f1] = 0x52f8, [0x02f2] = 0x52fb,
  [0x02f3] = 0x52fc, [0x02f4] = 0x52fd, [0x02f5] = 0x5301, [0x02f6] = 0x5302,
  [0x02f7] = 0x5303, [0x02f8] = 0x5304, [0x02f9] = 0x5307, [0x02fa] = 0x5309,
  [0x02fb] = 0x530a, [0x02fc] = 0x530b, [0x02fd] = 0x530c, [0x02fe] = 0x530e,
  [0x0300] = 0x5311, [0x0301] = 0x5312, [0x0302] = 0x5313, [0x0303] = 0x5314,
  [0x0304] = 0x5318, [0x0305] = 0x531b, [0x0306] = 0x531c, [0x0307] = 0x531e,
  [0x0308] = 0x531f, [0x0309] = 0x5322, [0x030a] = 0x5324, [0x030b] = 0x5325,
  [0x030c] = 0x5327, [0x030d] = 0x5328, [0x030e] = 0x5329, [0x030f] = 0x532b,
  [0x0310] = 0x532c, [0x0311] = 0x532d, [0x0312] = 0x532f, [0x0313] = 0x5330,
  [0x0314] = 0x5331, [0x0315] = 0x5332, [0x0316] = 0x5333, [0x0317] = 0x5334,
  [0x0318] = 0x5335, [0x0319] = 0x5336, [0x031a] = 0x5337, [0x031b] = 0x5338,
  [0x031c] = 0x533c, [0x031d] = 0x533d, [0x031e] = 0x5340, [0x031f] = 0x5342,
  [0x0320] = 0x5344, [0x0321] = 0x5346, [0x0322] = 0x534b, [0x0323] = 0x534c,
  [0x0324] = 0x534d, [0x0325] = 0x5350, [0x0326] = 0x5354, [0x0327] = 0x5358,
  [0x0328] = 0x5359, [0x0329] = 0x535b, [0x032a] = 0x535d, [0x032b] = 0x5365,
  [0x032c] = 0x5368, [0x032d] = 0x536a, [0x032e] = 0x536c, [0x032f] = 0x536d,
  [0x0330] = 0x5372, [0x0331] = 0x5376, [0x0332] = 0x5379, [0x0333] = 0x537b,
  [0x0334] = 0x537c, [0x0335] = 0x537d, [0x0336] = 0x537e, [0x0337] = 0x5380,
  [0x0338] = 0x5381, [0x0339] = 0x5383, [0x033a] = 0x5387, [0x033b] = 0x5388,
  [0x033c] = 0x538a, [0x033d] = 0x538e, [0x033e] = 0x538f, [0x0340] = 0x5390,
  [0x0341] = 0x5391, [0x0342] = 0x5392, [0x0343] = 0x5393, [0x0344] = 0x5394,
  [0x0345] = 0x5396, [0x0346] = 0x5397, [0x0347] = 0x5399, [0x0348] = 0x539b,
  [0x0349] = 0x539c, [0x034a] = 0x539e, [0x034b] = 0x53a0, [0x034c] = 0x53a1,
  [0x034d] = 0x53a4, [0x034e] = 0x53a7, [0x034f] = 0x53aa, [0x0350] = 0x53ab,
  [0x0351] = 0x53ac, [0x0352] = 0x53ad, [0x0353] = 0x53af, [0x0354] = 0x53b0,
  [0x0355] = 0x53b1, [0x0356] = 0x53b2, [0x0357] = 0x53b3, [0x0358] = 0x53b4,
  [0x0359] = 0x53b5, [0x035a] = 0x53b7, [0x035b] = 0x53b8, [0x035c] = 0x53b9,
  [0x035d] = 0x53ba, [0x035e] = 0x53bc, [0x035f] = 0x53bd, [0x0360] = 0x53be,
  [0x0361] = 0x53c0, [0x0362] = 0x53c3, [0x0363] = 0x53c4, [0x0364] = 0x53c5,
  [0x0365] = 0x53c6, [0x0366] = 0x53c7, [0x0367] = 0x53ce, [0x0368] = 0x53cf,
  [0x0369] = 0x53d0, [0x036a] = 0x53d2, [0x036b] = 0x53d3, [0x036c] = 0x53d5,
  [0x036d] = 0x53da, [0x036e] = 0x53dc, [0x036f] = 0x53dd, [0x0370] = 0x53de,
  [0x0371] = 0x53e1, [0x0372] = 0x53e2, [0x0373] = 0x53e7, [0x0374] = 0x53f4,
  [0x0375] = 0x53fa, [0x0376] = 0x53fe, [0x0377] = 0x53ff, [0x0378] = 0x5400,
  [0x0379] = 0x5402, [0x037a] = 0x5405, [0x037b] = 0x5407, [0x037c] = 0x540b,
  [0x037d] = 0x5414, [0x037e] = 0x5418, [0x037f] = 0x5419, [0x0380] = 0x541a,
  [0x0381] = 0x541c, [0x0382] = 0x5422, [0x0383] = 0x5424, [0x0384] = 0x5425,
  [0x0385] = 0x542a, [0x0386] = 0x5430, [0x0387] = 0x5433, [0x0388] = 0x5436,
  [0x0389] = 0x5437, [0x038a] = 0x543a, [0x038b] = 0x543d, [0x038c] = 0x543f,
  [0x038d] = 0x5441, [0x038e] = 0x5442, [0x038f] = 0x5444, [0x0390] = 0x5445,
  [0x0391] = 0x5447, [0x0392] = 0x5449, [0x0393] = 0x544c, [0x0394] = 0x544d,
  [0x0395] = 0x544e, [0x0396] = 0x544f, [0x0397] = 0x5451, [0x0398] = 0x545a,
  [0x0399] = 0x545d, [0x039a] = 0x545e, [0x039b] = 0x545f, [0x039c] = 0x5460,
  [0x039d] = 0x5461, [0x039e] = 0x5463, [0x039f] = 0x5465, [0x03a0] = 0x5467,
  [0x03a1] = 0x5469, [0x03a2] = 0x546a, [0x03a3] = 0x546b, [0x03a4] = 0x546c,
  [0x03a5] = 0x546d, [0x03a6] = 0x546e, [0x03a7] = 0x546f, [0x03a8] = 0x5470,
  [0x03a9] = 0x5474, [0x03aa] = 0x5479, [0x03ab] = 0x547a, [0x03ac] = 0x547e,
  [0x03ad] = 0x547f, [0x03ae] = 0x5481, [0x03af] = 0x5483, [0x03b0] = 0x5485,
  [0x03b1] = 0x5487, [0x03b2] = 0x5488, [0x03b3] = 0x5489, [0x03b4] = 0x548a,
  [0x03b5] = 0x548d, [0x03b6] = 0x5491, [0x03b7] = 0x5493, [0x03b8] = 0x5497,
  [0x03b9] = 0x5498, [0x03ba] = 0x549c, [0x03bb] = 0x549e, [0x03bc] = 0x549f,
  [0x03bd] = 0x54a0, [0x03be] = 0x54a1, [0x03c0] = 0x54a2, [0x03c1] = 0x54a5,
  [0x03c2] = 0x54ae, [0x03c3] = 0x54b0, [0x03c4] = 0x54b2, [0x03c5] = 0x54b5,
  [0x03c6] = 0x54b6, [0x03c7] = 0x54b7, [0x03c8] = 0x54b9, [0x03c9] = 0x54ba,
  [0x03ca] = 0x54bc, [0x03cb] = 0x54be, [0x03cc] = 0x54c3, [0x03cd] = 0x54c5,
  [0x03ce] = 0x54ca, [0x03cf] = 0x54cb, [0x03d0] = 0x54d6, [0x03d1] = 0x54d8,
  [0x03d2] = 0x54db, [0x03d3] = 0x54e0, [0x03d4] = 0x54e1, [0x03d5] = 0x54e2,
  [0x03d6] = 0x54e3, [0x03d7] = 0x54e4, [0x03d8] = 0x54eb, [0x03d9] = 0x54ec,
  [0x03da] = 0x54ef, [0x03db] = 0x54f0, [0x03dc] = 0x54f1, [0x03dd] = 0x54f4,
  [0x03de] = 0x54f5, [0x03df] = 0x54f6, [0x03e0] = 0x54f7, [0x03e1] = 0x54f8,
  [0x03e2] = 0x54f9, [0x03e3] = 0x54fb, [0x03e4] = 0x54fe, [0x03e5] = 0x5500,
  [0x03e6] = 0x5502, [0x03e7] = 0x5503, [0x03e8] = 0x5504, [0x03e9] = 0x5505,
  [0x03ea] = 0x5508, [0x03eb] = 0x550a, [0x03ec] = 0x550b, [0x03ed] = 0x550c,
  [0x03ee] = 0x550d, [0x03ef] = 0x550e, [0x03f0] = 0x5512, [0x03f1] = 0x5513,
  [0x03f2] = 0x5515, [0x03f3] = 0x5516, [0x03f4] = 0x5517, [0x03f5] = 0x5518,
  [0x03f6] = 0x5519, [0x03f7] = 0x551a, [0x03f8] = 0x551c, [0x03f9] = 0x551d,
  [0x03fa] = 0x551e, [0x03fb] = 0x551f, [0x03fc] = 0x5521, [0x03fd] = 0x5525,
  [0x03fe] = 0x5526, [0x0400] = 0x5528, [0x0401] = 0x5529, [0x0402] = 0x552b,
  [0x0403] = 0x552d, [0x0404] = 0x5532, [0x0405] = 0x5534, [0x0406] = 0x5535,
  [0x0407] = 0x5536, [0x0408] = 0x5538, [0x0409] = 0x5539, [0x040a] = 0x553a,
  [0x040b] = 0x553b, [0x040c] = 0x553d, [0x040d] = 0x5540, [0x040e] = 0x5542,
  [0x040f] = 0x5545, [0x0410] = 0x5547, [0x0411] = 0x5548, [0x0412] = 0x554b,
  [0x0413] = 0x554c, [0x0414] = 0x554d, [0x0415] = 0x554e, [0x0416] = 0x554f,
  [0x0417] = 0x5551, [0x0418] = 0x5552, [0x0419] = 0x5553, [0x041a] = 0x5554,
  [0x041b] = 0x5557, [0x041c] = 0x5558, [0x041d] = 0x5559, [0x041e] = 0x555a,
  [0x041f] = 0x555b, [0x0420] = 0x555d, [0x0421] = 0x555e, [0x0422] = 0x555f,
  [0x0423] = 0x5560, [0x0424] = 0x5562, [0x0425] = 0x5563, [0x0426] = 0x5568,
  [0x0427] = 0x5569, [0x0428] = 0x556b, [0x0429] = 0x556f, [0x042a] = 0x5570,
  [0x042b] = 0x5571, [0x042c] = 0x5572, [0x042d] = 0x5573, [0x042e] = 0x5574,
  [0x042f] = 0x5579, [0x0430] = 0x557a, [0x0431] = 0x557d, [0x0432] = 0x557f,
  [0x0433] = 0x5585, [0x0434] = 0x5586, [0x0435] = 0x558c, [0x0436] = 0x558d,
  [0x0437] = 0x558e, [0x0438] = 0x5590, [0x0439] = 0x5592, [0x043a] = 0x5593,
  [0x043b] = 0x5595, [0x043c] = 0x5596, [0x043d] = 0x5597, [0x043e] = 0x559a,
  [0x043f] = 0x559b, [0x0440] = 0x559e, [0x0441] = 0x55a0, [0x0442] = 0x55a1,
  [0x0443] = 0x55a2, [0x0444] = 0x55a3, [0x0445] = 0x55a4, [0x0446] = 0x55a5,
  [0x0447] = 0x55a6, [0x0448] = 0x55a8, [0x0449] = 0x55a9, [0x044a] = 0x55aa,
  [0x044b] = 0x55ab, [0x044c] = 0x55ac, [0x044d] = 0x55ad, [0x044e] = 0x55ae,
  [0x044f] = 0x55af, [0x0450] = 0x55b0, [0x0451] = 0x55b2, [0x0452] = 0x55b4,
  [0x0453] = 0x55b6, [0x0454] = 0x55b8, [0x0455] = 0x55ba, [0x0456] = 0x55bc,
  [0x0457] = 0x55bf, [0x0458] = 0x55c0, [0x0459] = 0x55c1, [0x045a] = 0x55c2,
  [0x045b] = 0x55c3, [0x045c] = 0x55c6, [0x045d] = 0x55c7, [0x045e] = 0x55c8,
  [0x045f] = 0x55ca, [0x0460] = 0x55cb, [0x0461] = 0x55ce, [0x0462] = 0x55cf,
  [0x0463] = 0x55d0, [0x0464] = 0x55d5, [0x0465] = 0x55d7, [0x0466] = 0x55d8,
  [0x0467] = 0x55d9, [0x0468] = 0x55da, [0x0469] = 0x55db, [0x046a] = 0x55de,
  [0x046b] = 0x55e0, [0x046c] = 0x55e2, [0x046d] = 0x55e7, [0x046e] = 0x55e9,
  [0x046f] = 0x55ed, [0x0470] = 0x55ee, [0x0471] = 0x55f0, [0x0472] = 0x55f1,
  [0x0473] = 0x55f4, [0x0474] = 0x55f6, [0x0475] = 0x55f8, [0x0476] = 0x55f9,
  [0x0477] = 0x55fa, [0x0478] = 0x55fb, [0x0479] = 0x55fc, [0x047a] = 0x55ff,
  [0x047b] = 0x5602, [0x047c] = 0x5603, [0x047d] = 0x5604, [0x047e] = 0x5605,
  [0x0480] = 0x5606, [0x0481] = 0x5607, [0x0482] = 0x560a, [0x0483] = 0x560b,
  [0x0484] = 0x560d, [0x0485] = 0x5610, [0x0486] = 0x5611, [0x0487] = 0x5612,
  [0x0488] = 0x5613, [0x0489] = 0x5614, [0x048a] = 0x5615, [0x048b] = 0x5616,
  [0x048c] = 0x5617, [0x048d] = 0x5619, [0x048e] = 0x561a, [0x048f] = 0x561c,
  [0x0490] = 0x561d, [0x0491] = 0x5620, [0x0492] = 0x5621, [0x0493] = 0x5622,
  [0x0494] = 0x5625, [0x0495] = 0x5626, [0x0496] = 0x5628, [0x0497] = 0x5629,
  [0x0498] = 0x562a, [0x0499] = 0x562b, [0x049a] = 0x562e, [0x049b] = 0x562f,
  [0x049c] = 0x5630, [0x049d] = 0x5633, [0x049e] = 0x5635, [0x049f] = 0x5637,
  [0x04a0] = 0x5638, [0x04a1] = 0x563a, [0x04a2] = 0x563c, [0x04a3] = 0x563d,
  [0x04a4] = 0x563e, [0x04a5] = 0x5640, [0x04a6] = 0x5641, [0x04a7] = 0x5642,
  [0x04a8] = 0x5643, [0x04a9] = 0x5644, [0x04aa] = 0x5645, [0x04ab] = 0x5646,
  [0x04ac] = 0x5647, [0x04ad] = 0x5648, [0x04ae] = 0x5649, [0x04af] = 0x564a,
  [0x04b0] = 0x564b, [0x04b1] = 0x564f, [0x04b2] = 0x5650, [0x04b3] = 0x5651,
  [0x04b4] = 0x5652, [0x04b5] = 0x5653, [0x04b6] = 0x5655, [0x04b7] = 0x5656,
  [0x04b8] = 0x565a, [0x04b9] = 0x565b, [0x04ba] = 0x565d, [0x04bb] = 0x565e,
  [0x04bc] = 0x565f, [0x04bd] = 0x5660, [0x04be] = 0x5661, [0x04c0] = 0x5663,
  [0x04c1] = 0x5665, [0x04c2] = 0x5666, [0x04c3] = 0x5667, [0x04c4] = 0x566d,
  [0x04c5] = 0x566e, [0x04c6] = 0x566f, [0x04c7] = 0x5670, [0x04c8] = 0x5672,
  [0x04c9] = 0x5673, [0x04ca] = 0x5674, [0x04cb] = 0x5675, [0x04cc] = 0x5677,
  [0x04cd] = 0x5678, [0x04ce] = 0x5679, [0x04cf] = 0x567a, [0x04d0] = 0x567d,
  [0x04d1] = 0x567e, [0x04d2] = 0x567f, [0x04d3] = 0x5680, [0x04d4] = 0x5681,
  [0x04d5] = 0x5682, [0x04d6] = 0x5683, [0x04d7] = 0x5684, [0x04d8] = 0x5687,
  [0x04d9] = 0x5688, [0x04da] = 0x5689, [0x04db] = 0x568a, [0x04dc] = 0x568b,
  [0x04dd] = 0x568c, [0x04de] = 0x568d, [0x04df] = 0x5690, [0x04e0] = 0x5691,
  [0x04e1] = 0x5692, [0x04e2] = 0x5694, [0x04e3] = 0x5695, [0x04e4] = 0x5696,
  [0x04e5] = 0x5697, [0x04e6] = 0x5698, [0x04e7] = 0x5699, [0x04e8] = 0x569a,
  [0x04e9] = 0x569b, [0x04ea] = 0x569c, [0x04eb] = 0x569d, [0x04ec] = 0x569e,
  [0x04ed] = 0x569f, [0x04ee] = 0x56a0, [0x04ef] = 0x56a1, [0x04f0] = 0x56a2,
  [0x04f1] = 0x56a4, [0x04f2] = 0x56a5, [0x04f3] = 0x56a6, [0x04f4] = 0x56a7,
  [0x04f5] = 0x56a8, [0x04f6] = 0x56a9, [0x04f7] = 0x56aa, [0x04f8] = 0x56ab,
  [0x04f9] = 0x56ac, [0x04fa] = 0x56ad, [0x04fb] = 0x56ae, [0x04fc] = 0x56b0,
  [0x04fd] = 0x56b1, [0x04fe] = 0x56b2, [0x04ff] = 0x56b3, [0x0500] = 0x56b4,
  [0x0501] = 0x56b5, [0x0502] = 0x56b6, [0x0503] = 0x56b8, [0x0504] = 0x56b9,
  [0x0505] = 0x56ba, [0x0506] = 0x56bb, [0x0507] = 0x56bd, [0x0508] = 0x56be,
  [0x0509] = 0x56bf, [0x050a] = 0x56c0, [0x050b] = 0x56c1, [0x050c] = 0x56c2,
  [0x050d] = 0x56c3, [0x050e] = 0x56c4, [0x050f] = 0x56c5, [0x0510] = 0x56c6,
  [0x0511] = 0x56c7, [0x0512] = 0x56c8, [0x0513] = 0x56c9, [0x0514] = 0x56cb,
  [0x0515] = 0x56cc, [0x0516] = 0x56cd, [0x0517] = 0x56ce, [0x0518] = 0x56cf,
  [0x0519] = 0x56d0, [0x051a] = 0x56d1, [0x051b] = 0x56d2, [0x051c] = 0x56d3,
  [0x051d] = 0x56d5, [0x051e] = 0x56d6, [0x051f] = 0x56d8, [0x0520] = 0x56d9,
  [0x0521] = 0x56dc, [0x0522] = 0x56e3, [0x0523] = 0x56e5, [0x0524] = 0x56e6,
  [0x0525] = 0x56e7, [0x0526] = 0x56e8, [0x0527] = 0x56e9, [0x0528] = 0x56ea,
  [0x0529] = 0x56ec, [0x052a] = 0x56ee, [0x052b] = 0x56ef, [0x052c] = 0x56f2,
  [0x052d] = 0x56f3, [0x052e] = 0x56f6, [0x052f] = 0x56f7, [0x0530] = 0x56f8,
  [0x0531] = 0x56fb, [0x0532] = 0x56fc, [0x0533] = 0x5700, [0x0534] = 0x5701,
  [0x0535] = 0x5702, [0x0536] = 0x5705, [0x0537] = 0x5707, [0x0538] = 0x570b,
  [0x0539] = 0x570c, [0x053a] = 0x570d, [0x053b] = 0x570e, [0x053c] = 0x570f,
  [0x053d] = 0x5710, [0x053e] = 0x5711, [0x0540] = 0x5712, [0x0541] = 0x5713,
  [0x0542] = 0x5714, [0x0543] = 0x5715, [0x0544] = 0x5716, [0x0545] = 0x5717,
  [0x0546] = 0x5718, [0x0547] = 0x5719, [0x0548] = 0x571a, [0x0549] = 0x571b,
  [0x054a] = 0x571d, [0x054b] = 0x571e, [0x054c] = 0x5720, [0x054d] = 0x5721,
  [0x054e] = 0x5722, [0x054f] = 0x5724, [0x0550] = 0x5725, [0x0551] = 0x5726,
  [0x0552] = 0x5727, [0x0553] = 0x572b, [0x0554] = 0x5731, [0x0555] = 0x5732,
  [0x0556] = 0x5734, [0x0557] = 0x5735, [0x0558] = 0x5736, [0x0559] = 0x5737,
  [0x055a] = 0x5738, [0x055b] = 0x573c, [0x055c] = 0x573d, [0x055d] = 0x573f,
  [0x055e] = 0x5741, [0x055f] = 0x5743, [0x0560] = 0x5744, [0x0561] = 0x5745,
  [0x0562] = 0x5746, [0x0563] = 0x5748, [0x0564] = 0x5749, [0x0565] = 0x574b,
  [0x0566] = 0x5752, [0x0567] = 0x5753, [0x0568] = 0x5754, [0x0569] = 0x5755,
  [0x056a] = 0x5756, [0x056b] = 0x5758, [0x056c] = 0x5759, [0x056d] = 0x5762,
  [0x056e] = 0x5763, [0x056f] = 0x5765, [0x0570] = 0x5767, [0x0571] = 0x576c,
  [0x0572] = 0x576e, [0x0573] = 0x5770, [0x0574] = 0x5771, [0x0575] = 0x5772,
  [0x0576] = 0x5774, [0x0577] = 0x5775, [0x0578] = 0x5778, [0x0579] = 0x5779,
  [0x057a] = 0x577a, [0x057b] = 0x577d, [0x057c] = 0x577e, [0x057d] = 0x577f,
  [0x057e] = 0x5780, [0x0580] = 0x5781, [0x0581] = 0x5787, [0x0582] = 0x5788,
  [0x0583] = 0x5789, [0x0584] = 0x578a, [0x0585] = 0x578d, [0x0586] = 0x578e,
  [0x0587] = 0x578f, [0x0588] = 0x5790, [0x0589] = 0x5791, [0x058a] = 0x5794,
  [0x058b] = 0x5795, [0x058c] = 0x5796, [0x058d] = 0x5797, [0x058e] = 0x5798,
  [0x058f] = 0x5799, [0x0590] = 0x579a, [0x0591] = 0x579c, [0x0592] = 0x579d,
  [0x0593] = 0x579e, [0x0594] = 0x579f, [0x0595] = 0x57a5, [0x0596] = 0x57a8,
  [0x0597] = 0x57aa, [0x0598] = 0x57ac, [0x0599] = 0x57af, [0x059a] = 0x57b0,
  [0x059b] = 0x57b1, [0x059c] = 0x57b3, [0x059d] = 0x57b5, [0x059e] = 0x57b6,
  [0x059f] = 0x57b7, [0x05a0] = 0x57b9, [0x05a1] = 0x57ba, [0x05a2] = 0x57bb,
  [0x05a3] = 0x57bc, [0x05a4] = 0x57bd, [0x05a5] = 0x57be, [0x05a6] = 0x57bf,
  [0x05a7] = 0x57c0, [0x05a8] = 0x57c1, [0x05a9] = 0x57c4, [0x05aa] = 0x57c5,
  [0x05ab] = 0x57c6, [0x05ac] = 0x57c7, [0x05ad] = 0x57c8, [0x05ae] = 0x57c9,
  [0x05af] = 0x57ca, [0x05b0] = 0x57cc, [0x05b1] = 0x57cd, [0x05b2] = 0x57d0,
  [0x05b3] = 0x57d1, [0x05b4] = 0x57d3, [0x05b5] = 0x57d6, [0x05b6] = 0x57d7,
  [0x05b7] = 0x57db, [0x05b8] = 0x57dc, [0x05b9] = 0x57de, [0x05ba] = 0x57e1,
  [0x05bb] = 0x57e2, [0x05bc] = 0x57e3, [0x05bd] = 0x57e5, [0x05be] = 0x57e6,
  [0x05bf] = 0x57e7, [0x05c0] = 0x57e8, [0x05c1] = 0x57e9, [0x05c2] = 0x57ea,
  [0x05c3] = 0x57eb, [0x05c4] = 0x57ec, [0x05c5] = 0x57ee, [0x05c6] = 0x57f0,
  [0x05c7] = 0x57f1, [0x05c8] = 0x57f2, [0x05c9] = 0x57f3, [0x05ca] = 0x57f5,
  [0x05cb] = 0x57f6, [0x05cc] = 0x57f7, [0x05cd] = 0x57fb, [0x05ce] = 0x57fc,
  [0x05cf] = 0x57fe, [0x05d0] = 0x57ff, [0x05d1] = 0x5801, [0x05d2] = 0x5803,
  [0x05d3] = 0x5804, [0x05d4] = 0x5805, [0x05d5] = 0x5808, [0x05d6] = 0x5809,
  [0x05d7] = 0x580a, [0x05d8] = 0x580c, [0x05d9] = 0x580e, [0x05da] = 0x580f,
  [0x05db] = 0x5810, [0x05dc] = 0x5812, [0x05dd] = 0x5813, [0x05de] = 0x5814,
  [0x05df] = 0x5816, [0x05e0] = 0x5817, [0x05e1] = 0x5818, [0x05e2] = 0x581a,
  [0x05e3] = 0x581b, [0x05e4] = 0x581c, [0x05e5] = 0x581d, [0x05e6] = 0x581f,
  [0x05e7] = 0x5822, [0x05e8] = 0x5823, [0x05e9] = 0x5825, [0x05ea] = 0x5826,
  [0x05eb] = 0x5827, [0x05ec] = 0x5828, [0x05ed] = 0x5829, [0x05ee] = 0x582b,
  [0x05ef] = 0x582c, [0x05f0] = 0x582d, [0x05f1] = 0x582e, [0x05f2] = 0x582f,
  [0x05f3] = 0x5831, [0x05f4] = 0x5832, [0x05f5] = 0x5833, [0x05f6] = 0x5834,
  [0x05f7] = 0x5836, [0x05f8] = 0x5837, [0x05f9] = 0x5838, [0x05fa] = 0x5839,
  [0x05fb] = 0x583a, [0x05fc] = 0x583b, [0x05fd] = 0x583c, [0x05fe] = 0x583d,
  [0x0600] = 0x583e, [0x0601] = 0x583f, [0x0602] = 0x5840, [0x0603] = 0x5841,
  [0x0604] = 0x5842, [0x0605] = 0x5843, [0x0606] = 0x5845, [0x0607] = 0x5846,
  [0x0608] = 0x5847, [0x0609] = 0x5848, [0x060a] = 0x5849, [0x060b] = 0x584a,
  [0x060c] = 0x584b, [0x060d] = 0x584e, [0x060e] = 0x584f, [0x060f] = 0x5850,
  [0x0610] = 0x5852, [0x0611] = 0x5853, [0x0612] = 0x5855, [0x0613] = 0x5856,
  [0x0614] = 0x5857, [0x0615] = 0x5859, [0x0616] = 0x585a, [0x0617] = 0x585b,
  [0x0618] = 0x585c, [0x0619] = 0x585d, [0x061a] = 0x585f, [0x061b] = 0x5860,
  [0x061c] = 0x5861, [0x061d] = 0x5862, [0x061e] = 0x5863, [0x061f] = 0x5864,
  [0x0620] = 0x5866, [0x0621] = 0x5867, [0x0622] = 0x5868, [0x0623] = 0x5869,
  [0x0624] = 0x586a, [0x0625] = 0x586d, [0x0626] = 0x586e, [0x0627] = 0x586f,
  [0x0628] = 0x5870, [0x0629] = 0x5871, [0x062a] = 0x5872, [0x062b] = 0x5873,
  [0x062c] = 0x5874, [0x062d] = 0x5875, [0x062e] = 0x5876, [0x062f] = 0x5877,
  [0x0630] = 0x5878, [0x0631] = 0x5879, [0x0632] = 0x587a, [0x0633] = 0x587b,
  [0x0634] = 0x587c, [0x0635] = 0x587d, [0x0636] = 0x587f, [0x0637] = 0x5882,
  [0x0638] = 0x5884, [0x0639] = 0x5886, [0x063a] = 0x5887, [0x063b] = 0x5888,
  [0x063c] = 0x588a, [0x063d] = 0x588b, [0x063e] = 0x588c, [0x0640] = 0x588d,
  [0x0641] = 0x588e, [0x0642] = 0x588f, [0x0643] = 0x5890, [0x0644] = 0x5891,
  [0x0645] = 0x5894, [0x0646] = 0x5895, [0x0647] = 0x5896, [0x0648] = 0x5897,
  [0x0649] = 0x5898, [0x064a] = 0x589b, [0x064b] = 0x589c, [0x064c] = 0x589d,
  [0x064d] = 0x58a0, [0x064e] = 0x58a1, [0x064f] = 0x58a2, [0x0650] = 0x58a3,
  [0x0651] = 0x58a4, [0x0652] = 0x58a5, [0x0653] = 0x58a6, [0x0654] = 0x58a7,
  [0x0655] = 0x58aa, [0x0656] = 0x58ab, [0x0657] = 0x58ac, [0x0658] = 0x58ad,
  [0x0659] = 0x58ae, [0x065a] = 0x58af, [0x065b] = 0x58b0, [0x065c] = 0x58b1,
  [0x065d] = 0x58b2, [0x065e] = 0x58b3, [0x065f] = 0x58b4, [0x0660] = 0x58b5,
  [0x0661] = 0x58b6, [0x0662] = 0x58b7, [0x0663] = 0x58b8, [0x0664] = 0x58b9,
  [0x0665] = 0x58ba, [0x0666] = 0x58bb, [0x0667] = 0x58bd, [0x0668] = 0x58be,
  [0x0669] = 0x58bf, [0x066a] = 0x58c0, [0x066b] = 0x58c2, [0x066c] = 0x58c3,
  [0x066d] = 0x58c4, [0x066e] = 0x58c6, [0x066f] = 0x58c7, [0x0670] = 0x58c8,
  [0x0671] = 0x58c9, [0x0672] = 0x58ca, [0x0673] = 0x58cb, [0x0674] = 0x58cc,
  [0x0675] = 0x58cd, [0x0676] = 0x58ce, [0x0677] = 0x58cf, [0x0678] = 0x58d0,
  [0x0679] = 0x58d2, [0x067a] = 0x58d3, [0x067b] = 0x58d4, [0x067c] = 0x58d6,
  [0x067d] = 0x58d7, [0x067e] = 0x58d8, [0x067f] = 0x58d9, [0x0680] = 0x58da,
  [0x0681] = 0x58db, [0x0682] = 0x58dc, [0x0683] = 0x58dd, [0x0684] = 0x58de,
  [0x0685] = 0x58df, [0x0686] = 0x58e0, [0x0687] = 0x58e1, [0x0688] = 0x58e2,
  [0x0689] = 0x58e3, [0x068a] = 0x58e5, [0x068b] = 0x58e6, [0x068c] = 0x58e7,
  [0x068d] = 0x58e8, [0x068e] = 0x58e9, [0x068f] = 0x58ea, [0x0690] = 0x58ed,
  [0x0691] = 0x58ef, [0x0692] = 0x58f1, [0x0693] = 0x58f2, [0x0694] = 0x58f4,
  [0x0695] = 0x58f5, [0x0696] = 0x58f7, [0x0697] = 0x58f8, [0x0698] = 0x58fa,
  [0x0699] = 0x58fb, [0x069a] = 0x58fc, [0x069b] = 0x58fd, [0x069c] = 0x58fe,
  [0x069d] = 0x58ff, [0x069e] = 0x5900, [0x069f] = 0x5901, [0x06a0] = 0x5903,
  [0x06a1] = 0x5905, [0x06a2] = 0x5906, [0x06a3] = 0x5908, [0x06a4] = 0x5909,
  [0x06a5] = 0x590a, [0x06a6] = 0x590b, [0x06a7] = 0x590c, [0x06a8] = 0x590e,
  [0x06a9] = 0x5910, [0x06aa] = 0x5911, [0x06ab] = 0x5912, [0x06ac] = 0x5913,
  [0x06ad] = 0x5917, [0x06ae] = 0x5918, [0x06af] = 0x591b, [0x06b0] = 0x591d,
  [0x06b1] = 0x591e, [0x06b2] = 0x5920, [0x06b3] = 0x5921, [0x06b4] = 0x5922,
  [0x06b5] = 0x5923, [0x06b6] = 0x5926, [0x06b7] = 0x5928, [0x06b8] = 0x592c,
  [0x06b9] = 0x5930, [0x06ba] = 0x5932, [0x06bb] = 0x5933, [0x06bc] = 0x5935,
  [0x06bd] = 0x5936, [0x06be] = 0x593b, [0x06c0] = 0x593d, [0x06c1] = 0x593e,
  [0x06c2] = 0x593f, [0x06c3] = 0x5940, [0x06c4] = 0x5943, [0x06c5] = 0x5945,
  [0x06c6] = 0x5946, [0x06c7] = 0x594a, [0x06c8] = 0x594c, [0x06c9] = 0x594d,
  [0x06ca] = 0x5950, [0x06cb] = 0x5952, [0x06cc] = 0x5953, [0x06cd] = 0x5959,
  [0x06ce] = 0x595b, [0x06cf] = 0x595c, [0x06d0] = 0x595d, [0x06d1] = 0x595e,
  [0x06d2] = 0x595f, [0x06d3] = 0x5961, [0x06d4] = 0x5963, [0x06d5] = 0x5964,
  [0x06d6] = 0x5966, [0x06d7] = 0x5967, [0x06d8] = 0x5968, [0x06d9] = 0x5969,
  [0x06da] = 0x596a, [0x06db] = 0x596b, [0x06dc] = 0x596c, [0x06dd] = 0x596d,
  [0x06de] = 0x596e, [0x06df] = 0x596f, [0x06e0] = 0x5970, [0x06e1] = 0x5971,
  [0x06e2] = 0x5972, [0x06e3] = 0x5975, [0x06e4] = 0x5977, [0x06e5] = 0x597a,
  [0x06e6] = 0x597b, [0x06e7] = 0x597c, [0x06e8] = 0x597e, [0x06e9] = 0x597f,
  [0x06ea] = 0x5980, [0x06eb] = 0x5985, [0x06ec] = 0x5989, [0x06ed] = 0x598b,
  [0x06ee] = 0x598c, [0x06ef] = 0x598e, [0x06f0] = 0x598f, [0x06f1] = 0x5990,
  [0x06f2] = 0x5991, [0x06f3] = 0x5994, [0x06f4] = 0x5995, [0x06f5] = 0x5998,
  [0x06f6] = 0x599a, [0x06f7] = 0x599b, [0x06f8] = 0x599c, [0x06f9] = 0x599d,
  [0x06fa] = 0x599f, [0x06fb] = 0x59a0, [0x06fc] = 0x59a1, [0x06fd] = 0x59a2,
  [0x06fe] = 0x59a6, [0x0700] = 0x59a7, [0x0701] = 0x59ac, [0x0702] = 0x59ad,
  [0x0703] = 0x59b0, [0x0704] = 0x59b1, [0x0705] = 0x59b3, [0x0706] = 0x59b4,
  [0x0707] = 0x59b5, [0x0708] = 0x59b6, [0x0709] = 0x59b7, [0x070a] = 0x59b8,
  [0x070b] = 0x59ba, [0x070c] = 0x59bc, [0x070d] = 0x59bd, [0x070e] = 0x59bf,
  [0x070f] = 0x59c0, [0x0710] = 0x59c1, [0x0711] = 0x59c2, [0x0712] = 0x59c3,
  [0x0713] = 0x59c4, [0x0714] = 0x59c5, [0x0715] = 0x59c7, [0x0716] = 0x59c8,
  [0x0717] = 0x59c9, [0x0718] = 0x59cc, [0x0719] = 0x59cd, [0x071a] = 0x59ce,
  [0x071b] = 0x59cf, [0x071c] = 0x59d5, [0x071d] = 0x59d6, [0x071e] = 0x59d9,
  [0x071f] = 0x59db, [0x0720] = 0x59de, [0x0721] = 0x59df, [0x0722] = 0x59e0,
  [0x0723] = 0x59e1, [0x0724] = 0x59e2, [0x0725] = 0x59e4, [0x0726] = 0x59e6,
  [0x0727] = 0x59e7, [0x0728] = 0x59e9, [0x0729] = 0x59ea, [0x072a] = 0x59eb,
  [0x072b] = 0x59ed, [0x072c] = 0x59ee, [0x072d] = 0x59ef, [0x072e] = 0x59f0,
  [0x072f] = 0x59f1, [0x0730] = 0x59f2, [0x0731] = 0x59f3, [0x0732] = 0x59f4,
  [0x0733] = 0x59f5, [0x0734] = 0x59f6, [0x0735] = 0x59f7, [0x0736] = 0x59f8,
  [0x0737] = 0x59fa, [0x0738] = 0x59fc, [0x0739] = 0x59fd, [0x073a] = 0x59fe,
  [0x073b] = 0x5a00, [0x073c] = 0x5a02, [0x073d] = 0x5a0a, [0x073e] = 0x5a0b,
  [0x073f] = 0x5a0d, [0x0740] = 0x5a0e, [0x0741] = 0x5a0f, [0x0742] = 0x5a10,
  [0x0743] = 0x5a12, [0x0744] = 0x5a14, [0x0745] = 0x5a15, [0x0746] = 0x5a16,
  [0x0747] = 0x5a17, [0x0748] = 0x5a19, [0x0749] = 0x5a1a, [0x074a] = 0x5a1b,
  [0x074b] = 0x5a1d, [0x074c] = 0x5a1e, [0x074d] = 0x5a21, [0x074e] = 0x5a22,
  [0x074f] = 0x5a24, [0x0750] = 0x5a26, [0x0751] = 0x5a27, [0x0752] = 0x5a28,
  [0x0753] = 0x5a2a, [0x0754] = 0x5a2b, [0x0755] = 0x5a2c, [0x0756] = 0x5a2d,
  [0x0757] = 0x5a2e, [0x0758] = 0x5a2f, [0x0759] = 0x5a30, [0x075a] = 0x5a33,
  [0x075b] = 0x5a35, [0x075c] = 0x5a37, [0x075d] = 0x5a38, [0x075e] = 0x5a39,
  [0x075f] = 0x5a3a, [0x0760] = 0x5a3b, [0x0761] = 0x5a3d, [0x0762] = 0x5a3e,
  [0x0763] = 0x5a3f, [0x0764] = 0x5a41, [0x0765] = 0x5a42, [0x0766] = 0x5a43,
  [0x0767] = 0x5a44, [0x0768] = 0x5a45, [0x0769] = 0x5a47, [0x076a] = 0x5a48,
  [0x076b] = 0x5a4b, [0x076c] = 0x5a4c, [0x076d] = 0x5a4d, [0x076e] = 0x5a4e,
  [0x076f] = 0x5a4f, [0x0770] = 0x5a50, [0x0771] = 0x5a51, [0x0772] = 0x5a52,
  [0x0773] = 0x5a53, [0x0774] = 0x5a54, [0x0775] = 0x5a56, [0x0776] = 0x5a57,
  [0x0777] = 0x5a58, [0x0778] = 0x5a59, [0x0779] = 0x5a5b, [0x077a] = 0x5a5c,
  [0x077b] = 0x5a5d, [0x077c] = 0x5a5e, [0x077d] = 0x5a5f, [0x077e] = 0x5a60,
  [0x0780] = 0x5a61, [0x0781] = 0x5a63, [0x0782] = 0x5a64, [0x0783] = 0x5a65,
  [0x0784] = 0x5a66, [0x0785] = 0x5a68, [0x0786] = 0x5a69, [0x0787] = 0x5a6b,
  [0x0788] = 0x5a6c, [0x0789] = 0x5a6d, [0x078a] = 0x5a6e, [0x078b] = 0x5a6f,
  [0x078c] = 0x5a70, [0x078d] = 0x5a71, [0x078e] = 0x5a72, [0x078f] = 0x5a73,
  [0x0790] = 0x5a78, [0x0791] = 0x5a79, [0x0792] = 0x5a7b, [0x0793] = 0x5a7c,
  [0x0794] = 0x5a7d, [0x0795] = 0x5a7e, [0x0796] = 0x5a80, [0x0797] = 0x5a81,
  [0x0798] = 0x5a82, [0x0799] = 0x5a83, [0x079a] = 0x5a84, [0x079b] = 0x5a85,
  [0x079c] = 0x5a86, [0x079d] = 0x5a87, [0x079e] = 0x5a88, [0x079f] = 0x5a89,
  [0x07a0] = 0x5a8a, [0x07a1] = 0x5a8b, [0x07a2] = 0x5a8c, [0x07a3] = 0x5a8d,
  [0x07a4] = 0x5a8e, [0x07a5] = 0x5a8f, [0x07a6] = 0x5a90, [0x07a7] = 0x5a91,
  [0x07a8] = 0x5a93, [0x07a9] = 0x5a94, [0x07aa] = 0x5a95, [0x07ab] = 0x5a96,
  [0x07ac] = 0x5a97, [0x07ad] = 0x5a98, [0x07ae] = 0x5a99, [0x07af] = 0x5a9c,
  [0x07b0] = 0x5a9d, [0x07b1] = 0x5a9e, [0x07b2] = 0x5a9f, [0x07b3] = 0x5aa0,
  [0x07b4] = 0x5aa1, [0x07b5] = 0x5aa2, [0x07b6] = 0x5aa3, [0x07b7] = 0x5aa4,
  [0x07b8] = 0x5aa5, [0x07b9] = 0x5aa6, [0x07ba] = 0x5aa7, [0x07bb] = 0x5aa8,
  [0x07bc] = 0x5aa9, [0x07bd] = 0x5aab, [0x07be] = 0x5aac, [0x07c0] = 0x5aad,
  [0x07c1] = 0x5aae, [0x07c2] = 0x5aaf, [0x07c3] = 0x5ab0, [0x07c4] = 0x5ab1,
  [0x07c5] = 0x5ab4, [0x07c6] = 0x5ab6, [0x07c7] = 0x5ab7, [0x07c8] = 0x5ab9,
  [0x07c9] = 0x5aba, [0x07ca] = 0x5abb, [0x07cb] = 0x5abc, [0x07cc] = 0x5abd,
  [0x07cd] = 0x5abf, [0x07ce] = 0x5ac0, [0x07cf] = 0x5ac3, [0x07d0] = 0x5ac4,
  [0x07d1] = 0x5ac5, [0x07d2] = 0x5ac6, [0x07d3] = 0x5ac7, [0x07d4] = 0x5ac8,
  [0x07d5] = 0x5aca, [0x07d6] = 0x5acb, [0x07d7] = 0x5acd, [0x07d8] = 0x5ace,
  [0x07d9] = 0x5acf, [0x07da] = 0x5ad0, [0x07db] = 0x5ad1, [0x07dc] = 0x5ad3,
  [0x07dd] = 0x5ad5, [0x07de] = 0x5ad7, [0x07df] = 0x5ad9, [0x07e0] = 0x5ada,
  [0x07e1] = 0x5adb, [0x07e2] = 0x5add, [0x07e3] = 0x5ade, [0x07e4] = 0x5adf,
  [0x07e5] = 0x5ae2, [0x07e6] = 0x5ae4, [0x07e7] = 0x5ae5, [0x07e8] = 0x5ae7,
  [0x07e9] = 0x5ae8, [0x07ea] = 0x5aea, [0x07eb] = 0x5aec, [0x07ec] = 0x5aed,
  [0x07ed] = 0x5aee, [0x07ee] = 0x5aef, [0x07ef] = 0x5af0, [0x07f0] = 0x5af2,
  [0x07f1] = 0x5af3, [0x07f2] = 0x5af4, [0x07f3] = 0x5af5, [0x07f4] = 0x5af6,
  [0x07f5] = 0x5af7, [0x07f6] = 0x5af8, [0x07f7] = 0x5af9, [0x07f8] = 0x5afa,
  [0x07f9] = 0x5afb, [0x07fa] = 0x5afc, [0x07fb] = 0x5afd, [0x07fc] = 0x5afe,
  [0x07fd] = 0x5aff, [0x07fe] = 0x5b00, [0x07ff] = 0x5b01, [0x0800] = 0x5b02,
  [0x0801] = 0x5b03, [0x0802] = 0x5b04, [0x0803] = 0x5b05, [0x0804] = 0x5b06,
  [0x0805] = 0x5b07, [0x0806] = 0x5b08, [0x0807] = 0x5b0a, [0x0808] = 0x5b0b,
  [0x0809] = 0x5b0c, [0x080a] = 0x5b0d, [0x080b] = 0x5b0e, [0x080c] = 0x5b0f,
  [0x080d] = 0x5b10, [0x080e] = 0x5b11, [0x080f] = 0x5b12, [0x0810] = 0x5b13,
  [0x0811] = 0x5b14, [0x0812] = 0x5b15, [0x0813] = 0x5b18, [0x0814] = 0x5b19,
  [0x0815] = 0x5b1a, [0x0816] = 0x5b1b, [0x0817] = 0x5b1c, [0x0818] = 0x5b1d,
  [0x0819] = 0x5b1e, [0x081a] = 0x5b1f, [0x081b] = 0x5b20, [0x081c] = 0x5b21,
  [0x081d] = 0x5b22, [0x081e] = 0x5b23, [0x081f] = 0x5b24, [0x0820] = 0x5b25,
  [0x0821] = 0x5b26, [0x0822] = 0x5b27, [0x0823] = 0x5b28, [0x0824] = 0x5b29,
  [0x0825] = 0x5b2a, [0x0826] = 0x5b2b, [0x0827] = 0x5b2c, [0x0828] = 0x5b2d,
  [0x0829] = 0x5b2e, [0x082a] = 0x5b2f, [0x082b] = 0x5b30, [0x082c] = 0x5b31,
  [0x082d] = 0x5b33, [0x082e] = 0x5b35, [0x082f] = 0x5b36, [0x0830] = 0x5b38,
  [0x0831] = 0x5b39, [0x0832] = 0x5b3a, [0x0833] = 0x5b3b, [0x0834] = 0x5b3c,
  [0x0835] = 0x5b3d, [0x0836] = 0x5b3e, [0x0837] = 0x5b3f, [0x0838] = 0x5b41,
  [0x0839] = 0x5b42, [0x083a] = 0x5b43, [0x083b] = 0x5b44, [0x083c] = 0x5b45,
  [0x083d] = 0x5b46, [0x083e] = 0x5b47, [0x0840] = 0x5b48, [0x0841] = 0x5b49,
  [0x0842] = 0x5b4a, [0x0843] = 0x5b4b, [0x0844] = 0x5b4c, [0x0845] = 0x5b4d,
  [0x0846] = 0x5b4e, [0x0847] = 0x5b4f, [0x0848] = 0x5b52, [0x0849] = 0x5b56,
  [0x084a] = 0x5b5e, [0x084b] = 0x5b60, [0x084c] = 0x5b61, [0x084d] = 0x5b67,
  [0x084e] = 0x5b68, [0x084f] = 0x5b6b, [0x0850] = 0x5b6d, [0x0851] = 0x5b6e,
  [0x0852] = 0x5b6f, [0x0853] = 0x5b72, [0x0854] = 0x5b74, [0x0855] = 0x5b76,
  [0x0856] = 0x5b77, [0x0857] = 0x5b78, [0x0858] = 0x5b79, [0x0859] = 0x5b7b,
  [0x085a] = 0x5b7c, [0x085b] = 0x5b7e, [0x085c] = 0x5b7f, [0x085d] = 0x5b82,
  [0x085e] = 0x5b86, [0x085f] = 0x5b8a, [0x0860] = 0x5b8d, [0x0861] = 0x5b8e,
  [0x0862] = 0x5b90, [0x0863] = 0x5b91, [0x0864] = 0x5b92, [0x0865] = 0x5b94,
  [0x0866] = 0x5b96, [0x0867] = 0x5b9f, [0x0868] = 0x5ba7, [0x0869] = 0x5ba8,
  [0x086a] = 0x5ba9, [0x086b] = 0x5bac, [0x086c] = 0x5bad, [0x086d] = 0x5bae,
  [0x086e] = 0x5baf, [0x086f] = 0x5bb1, [0x0870] = 0x5bb2, [0x0871] = 0x5bb7,
  [0x0872] = 0x5bba, [0x0873] = 0x5bbb, [0x0874] = 0x5bbc, [0x0875] = 0x5bc0,
  [0x0876] = 0x5bc1, [0x0877] = 0x5bc3, [0x0878] = 0x5bc8, [0x0879] = 0x5bc9,
  [0x087a] = 0x5bca, [0x087b] = 0x5bcb, [0x087c] = 0x5bcd, [0x087d] = 0x5bce,
  [0x087e] = 0x5bcf, [0x0880] = 0x5bd1, [0x0881] = 0x5bd4, [0x0882] = 0x5bd5,
  [0x0883] = 0x5bd6, [0x0884] = 0x5bd7, [0x0885] = 0x5bd8, [0x0886] = 0x5bd9,
  [0x0887] = 0x5bda, [0x0888] = 0x5bdb, [0x0889] = 0x5bdc, [0x088a] = 0x5be0,
  [0x088b] = 0x5be2, [0x088c] = 0x5be3, [0x088d] = 0x5be6, [0x088e] = 0x5be7,
  [0x088f] = 0x5be9, [0x0890] = 0x5bea, [0x0891] = 0x5beb, [0x0892] = 0x5bec,
  [0x0893] = 0x5bed, [0x0894] = 0x5bef, [0x0895] = 0x5bf1, [0x0896] = 0x5bf2,
  [0x0897] = 0x5bf3, [0x0898] = 0x5bf4, [0x0899] = 0x5bf5, [0x089a] = 0x5bf6,
  [0x089b] = 0x5bf7, [0x089c] = 0x5bfd, [0x089d] = 0x5bfe, [0x089e] = 0x5c00,
  [0x089f] = 0x5c02, [0x08a0] = 0x5c03, [0x08a1] = 0x5c05, [0x08a2] = 0x5c07,
  [0x08a3] = 0x5c08, [0x08a4] = 0x5c0b, [0x08a5] = 0x5c0c, [0x08a6] = 0x5c0d,
  [0x08a7] = 0x5c0e, [0x08a8] = 0x5c10, [0x08a9] = 0x5c12, [0x08aa] = 0x5c13,
  [0x08ab] = 0x5c17, [0x08ac] = 0x5c19, [0x08ad] = 0x5c1b, [0x08ae] = 0x5c1e,
  [0x08af] = 0x5c1f, [0x08b0] = 0x5c20, [0x08b1] = 0x5c21, [0x08b2] = 0x5c23,
  [0x08b3] = 0x5c26, [0x08b4] = 0x5c28, [0x08b5] = 0x5c29, [0x08b6] = 0x5c2a,
  [0x08b7] = 0x5c2b, [0x08b8] = 0x5c2d, [0x08b9] = 0x5c2e, [0x08ba] = 0x5c2f,
  [0x08bb] = 0x5c30, [0x08bc] = 0x5c32, [0x08bd] = 0x5c33, [0x08be] = 0x5c35,
  [0x08bf] = 0x5c36, [0x08c0] = 0x5c37, [0x08c1] = 0x5c43, [0x08c2] = 0x5c44,
  [0x08c3] = 0x5c46, [0x08c4] = 0x5c47, [0x08c5] = 0x5c4c, [0x08c6] = 0x5c4d,
  [0x08c7] = 0x5c52, [0x08c8] = 0x5c53, [0x08c9] = 0x5c54, [0x08ca] = 0x5c56,
  [0x08cb] = 0x5c57, [0x08cc] = 0x5c58, [0x08cd] = 0x5c5a, [0x08ce] = 0x5c5b,
  [0x08cf] = 0x5c5c, [0x08d0] = 0x5c5d, [0x08d1] = 0x5c5f, [0x08d2] = 0x5c62,
  [0x08d3] = 0x5c64, [0x08d4] = 0x5c67, [0x08d5] = 0x5c68, [0x08d6] = 0x5c69,
  [0x08d7] = 0x5c6a, [0x08d8] = 0x5c6b, [0x08d9] = 0x5c6c, [0x08da] = 0x5c6d,
  [0x08db] = 0x5c70, [0x08dc] = 0x5c72, [0x08dd] = 0x5c73, [0x08de] = 0x5c74,
  [0x08df] = 0x5c75, [0x08e0] = 0x5c76, [0x08e1] = 0x5c77, [0x08e2] = 0x5c78,
  [0x08e3] = 0x5c7b, [0x08e4] = 0x5c7c, [0x08e5] = 0x5c7d, [0x08e6] = 0x5c7e,
  [0x08e7] = 0x5c80, [0x08e8] = 0x5c83, [0x08e9] = 0x5c84, [0x08ea] = 0x5c85,
  [0x08eb] = 0x5c86, [0x08ec] = 0x5c87, [0x08ed] = 0x5c89, [0x08ee] = 0x5c8a,
  [0x08ef] = 0x5c8b, [0x08f0] = 0x5c8e, [0x08f1] = 0x5c8f, [0x08f2] = 0x5c92,
  [0x08f3] = 0x5c93, [0x08f4] = 0x5c95, [0x08f5] = 0x5c9d, [0x08f6] = 0x5c9e,
  [0x08f7] = 0x5c9f, [0x08f8] = 0x5ca0, [0x08f9] = 0x5ca1, [0x08fa] = 0x5ca4,
  [0x08fb] = 0x5ca5, [0x08fc] = 0x5ca6, [0x08fd] = 0x5ca7, [0x08fe] = 0x5ca8,
  [0x0900] = 0x5caa, [0x0901] = 0x5cae, [0x0902] = 0x5caf, [0x0903] = 0x5cb0,
  [0x0904] = 0x5cb2, [0x0905] = 0x5cb4, [0x0906] = 0x5cb6, [0x0907] = 0x5cb9,
  [0x0908] = 0x5cba, [0x0909] = 0x5cbb, [0x090a] = 0x5cbc, [0x090b] = 0x5cbe,
  [0x090c] = 0x5cc0, [0x090d] = 0x5cc2, [0x090e] = 0x5cc3, [0x090f] = 0x5cc5,
  [0x0910] = 0x5cc6, [0x0911] = 0x5cc7, [0x0912] = 0x5cc8, [0x0913] = 0x5cc9,
  [0x0914] = 0x5cca, [0x0915] = 0x5ccc, [0x0916] = 0x5ccd, [0x0917] = 0x5cce,
  [0x0918] = 0x5ccf, [0x0919] = 0x5cd0, [0x091a] = 0x5cd1, [0x091b] = 0x5cd3,
  [0x091c] = 0x5cd4, [0x091d] = 0x5cd5, [0x091e] = 0x5cd6, [0x091f] = 0x5cd7,
  [0x0920] = 0x5cd8, [0x0921] = 0x5cda, [0x0922] = 0x5cdb, [0x0923] = 0x5cdc,
  [0x0924] = 0x5cdd, [0x0925] = 0x5cde, [0x0926] = 0x5cdf, [0x0927] = 0x5ce0,
  [0x0928] = 0x5ce2, [0x0929] = 0x5ce3, [0x092a] = 0x5ce7, [0x092b] = 0x5ce9,
  [0x092c] = 0x5ceb, [0x092d] = 0x5cec, [0x092e] = 0x5cee, [0x092f] = 0x5cef,
  [0x0930] = 0x5cf1, [0x0931] = 0x5cf2, [0x0932] = 0x5cf3, [0x0933] = 0x5cf4,
  [0x0934] = 0x5cf5, [0x0935] = 0x5cf6, [0x0936] = 0x5cf7, [0x0937] = 0x5cf8,
  [0x0938] = 0x5cf9, [0x0939] = 0x5cfa, [0x093a] = 0x5cfc, [0x093b] = 0x5cfd,
  [0x093c] = 0x5cfe, [0x093d] = 0x5cff, [0x093e] = 0x5d00, [0x0940] = 0x5d01,
  [0x0941] = 0x5d04, [0x0942] = 0x5d05, [0x0943] = 0x5d08, [0x0944] = 0x5d09,
  [0x0945] = 0x5d0a, [0x0946] = 0x5d0b, [0x0947] = 0x5d0c, [0x0948] = 0x5d0d,
  [0x0949] = 0x5d0f, [0x094a] = 0x5d10, [0x094b] = 0x5d11, [0x094c] = 0x5d12,
  [0x094d] = 0x5d13, [0x094e] = 0x5d15, [0x094f] = 0x5d17, [0x0950] = 0x5d18,
  [0x0951] = 0x5d19, [0x0952] = 0x5d1a, [0x0953] = 0x5d1c, [0x0954] = 0x5d1d,
  [0x0955] = 0x5d1f, [0x0956] = 0x5d20, [0x0957] = 0x5d21, [0x0958] = 0x5d22,
  [0x0959] = 0x5d23, [0x095a] = 0x5d25, [0x095b] = 0x5d28, [0x095c] = 0x5d2a,
  [0x095d] = 0x5d2b, [0x095e] = 0x5d2c, [0x095f] = 0x5d2f, [0x0960] = 0x5d30,
  [0x0961] = 0x5d31, [0x0962] = 0x5d32, [0x0963] = 0x5d33, [0x0964] = 0x5d35,
  [0x0965] = 0x5d36, [0x0966] = 0x5d37, [0x0967] = 0x5d38, [0x0968] = 0x5d39,
  [0x0969] = 0x5d3a, [0x096a] = 0x5d3b, [0x096b] = 0x5d3c, [0x096c] = 0x5d3f,
  [0x096d] = 0x5d40, [0x096e] = 0x5d41, [0x096f] = 0x5d42, [0x0970] = 0x5d43,
  [0x0971] = 0x5d44, [0x0972] = 0x5d45, [0x0973] = 0x5d46, [0x0974] = 0x5d48,
  [0x0975] = 0x5d49, [0x0976] = 0x5d4d, [0x0977] = 0x5d4e, [0x0978] = 0x5d4f,
  [0x0979] = 0x5d50, [0x097a] = 0x5d51, [0x097b] = 0x5d52, [0x097c] = 0x5d53,
  [0x097d] = 0x5d54, [0x097e] = 0x5d55, [0x097f] = 0x5d56, [0x0980] = 0x5d57,
  [0x0981] = 0x5d59, [0x0982] = 0x5d5a, [0x0983] = 0x5d5c, [0x0984] = 0x5d5e,
  [0x0985] = 0x5d5f, [0x0986] = 0x5d60, [0x0987] = 0x5d61, [0x0988] = 0x5d62,
  [0x0989] = 0x5d63, [0x098a] = 0x5d64, [0x098b] = 0x5d65, [0x098c] = 0x5d66,
  [0x098d] = 0x5d67, [0x098e] = 0x5d68, [0x098f] = 0x5d6a, [0x0990] = 0x5d6d,
  [0x0991] = 0x5d6e, [0x0992] = 0x5d70, [0x0993] = 0x5d71, [0x0994] = 0x5d72,
  [0x0995] = 0x5d73, [0x0996] = 0x5d75, [0x0997] = 0x5d76, [0x0998] = 0x5d77,
  [0x0999] = 0x5d78, [0x099a] = 0x5d79, [0x099b] = 0x5d7a, [0x099c] = 0x5d7b,
  [0x099d] = 0x5d7c, [0x099e] = 0x5d7d, [0x099f] = 0x5d7e, [0x09a0] = 0x5d7f,
  [0x09a1] = 0x5d80, [0x09a2] = 0x5d81, [0x09a3] = 0x5d83, [0x09a4] = 0x5d84,
  [0x09a5] = 0x5d85, [0x09a6] = 0x5d86, [0x09a7] = 0x5d87, [0x09a8] = 0x5d88,
  [0x09a9] = 0x5d89, [0x09aa] = 0x5d8a, [0x09ab] = 0x5d8b, [0x09ac] = 0x5d8c,
  [0x09ad] = 0x5d8d, [0x09ae] = 0x5d8e, [0x09af] = 0x5d8f, [0x09b0] = 0x5d90,
  [0x09b1] = 0x5d91, [0x09b2] = 0x5d92, [0x09b3] = 0x5d93, [0x09b4] = 0x5d94,
  [0x09b5] = 0x5d95, [0x09b6] = 0x5d96, [0x09b7] = 0x5d97, [0x09b8] = 0x5d98,
  [0x09b9] = 0x5d9a, [0x09ba] = 0x5d9b, [0x09bb] = 0x5d9c, [0x09bc] = 0x5d9e,
  [0x09bd] = 0x5d9f, [0x09be] = 0x5da0, [0x09c0] = 0x5da1, [0x09c1] = 0x5da2,
  [0x09c2] = 0x5da3, [0x09c3] = 0x5da4, [0x09c4] = 0x5da5, [0x09c5] = 0x5da6,
  [0x09c6] = 0x5da7, [0x09c7] = 0x5da8, [0x09c8] = 0x5da9, [0x09c9] = 0x5daa,
  [0x09ca] = 0x5dab, [0x09cb] = 0x5dac, [0x09cc] = 0x5dad, [0x09cd] = 0x5dae,
  [0x09ce] = 0x5daf, [0x09cf] = 0x5db0, [0x09d0] = 0x5db1, [0x09d1] = 0x5db2,
  [0x09d2] = 0x5db3, [0x09d3] = 0x5db4, [0x09d4] = 0x5db5, [0x09d5] = 0x5db6,
  [0x09d6] = 0x5db8, [0x09d7] = 0x5db9, [0x09d8] = 0x5dba, [0x09d9] = 0x5dbb,
  [0x09da] = 0x5dbc, [0x09db] = 0x5dbd, [0x09dc] = 0x5dbe, [0x09dd] = 0x5dbf,
  [0x09de] = 0x5dc0, [0x09df] = 0x5dc1, [0x09e0] = 0x5dc2, [0x09e1] = 0x5dc3,
  [0x09e2] = 0x5dc4, [0x09e3] = 0x5dc6, [0x09e4] = 0x5dc7, [0x09e5] = 0x5dc8,
  [0x09e6] = 0x5dc9, [0x09e7] = 0x5dca, [0x09e8] = 0x5dcb, [0x09e9] = 0x5dcc,
  [0x09ea] = 0x5dce, [0x09eb] = 0x5dcf, [0x09ec] = 0x5dd0, [0x09ed] = 0x5dd1,
  [0x09ee] = 0x5dd2, [0x09ef] = 0x5dd3, [0x09f0] = 0x5dd4, [0x09f1] = 0x5dd5,
  [0x09f2] = 0x5dd6, [0x09f3] = 0x5dd7, [0x09f4] = 0x5dd8, [0x09f5] = 0x5dd9,
  [0x09f6] = 0x5dda, [0x09f7] = 0x5ddc, [0x09f8] = 0x5ddf, [0x09f9] = 0x5de0,
  [0x09fa] = 0x5de3, [0x09fb] = 0x5de4, [0x09fc] = 0x5dea, [0x09fd] = 0x5dec,
  [0x09fe] = 0x5ded, [0x0a00] = 0x5df0, [0x0a01] = 0x5df5, [0x0a02] = 0x5df6,
  [0x0a03] = 0x5df8, [0x0a04] = 0x5df9, [0x0a05] = 0x5dfa, [0x0a06] = 0x5dfb,
  [0x0a07] = 0x5dfc, [0x0a08] = 0x5dff, [0x0a09] = 0x5e00, [0x0a0a] = 0x5e04,
  [0x0a0b] = 0x5e07, [0x0a0c] = 0x5e09, [0x0a0d] = 0x5e0a, [0x0a0e] = 0x5e0b,
  [0x0a0f] = 0x5e0d, [0x0a10] = 0x5e0e, [0x0a11] = 0x5e12, [0x0a12] = 0x5e13,
  [0x0a13] = 0x5e17, [0x0a14] = 0x5e1e, [0x0a15] = 0x5e1f, [0x0a16] = 0x5e20,
  [0x0a17] = 0x5e21, [0x0a18] = 0x5e22, [0x0a19] = 0x5e23, [0x0a1a] = 0x5e24,
  [0x0a1b] = 0x5e25, [0x0a1c] = 0x5e28, [0x0a1d] = 0x5e29, [0x0a1e] = 0x5e2a,
  [0x0a1f] = 0x5e2b, [0x0a20] = 0x5e2c, [0x0a21] = 0x5e2f, [0x0a22] = 0x5e30,
  [0x0a23] = 0x5e32, [0x0a24] = 0x5e33, [0x0a25] = 0x5e34, [0x0a26] = 0x5e35,
  [0x0a27] = 0x5e36, [0x0a28] = 0x5e39, [0x0a29] = 0x5e3a, [0x0a2a] = 0x5e3e,
  [0x0a2b] = 0x5e3f, [0x0a2c] = 0x5e40, [0x0a2d] = 0x5e41, [0x0a2e] = 0x5e43,
  [0x0a2f] = 0x5e46, [0x0a30] = 0x5e47, [0x0a31] = 0x5e48, [0x0a32] = 0x5e49,
  [0x0a33] = 0x5e4a, [0x0a34] = 0x5e4b, [0x0a35] = 0x5e4d, [0x0a36] = 0x5e4e,
  [0x0a37] = 0x5e4f, [0x0a38] = 0x5e50, [0x0a39] = 0x5e51, [0x0a3a] = 0x5e52,
  [0x0a3b] = 0x5e53, [0x0a3c] = 0x5e56, [0x0a3d] = 0x5e57, [0x0a3e] = 0x5e58,
  [0x0a3f] = 0x5e59, [0x0a40] = 0x5e5a, [0x0a41] = 0x5e5c, [0x0a42] = 0x5e5d,
  [0x0a43] = 0x5e5f, [0x0a44] = 0x5e60, [0x0a45] = 0x5e63, [0x0a46] = 0x5e64,
  [0x0a47] = 0x5e65, [0x0a48] = 0x5e66, [0x0a49] = 0x5e67, [0x0a4a] = 0x5e68,
  [0x0a4b] = 0x5e69, [0x0a4c] = 0x5e6a, [0x0a4d] = 0x5e6b, [0x0a4e] = 0x5e6c,
  [0x0a4f] = 0x5e6d, [0x0a50] = 0x5e6e, [0x0a51] = 0x5e6f, [0x0a52] = 0x5e70,
  [0x0a53] = 0x5e71, [0x0a54] = 0x5e75, [0x0a55] = 0x5e77, [0x0a56] = 0x5e79,
  [0x0a57] = 0x5e7e, [0x0a58] = 0x5e81, [0x0a59] = 0x5e82, [0x0a5a] = 0x5e83,
  [0x0a5b] = 0x5e85, [0x0a5c] = 0x5e88, [0x0a5d] = 0x5e89, [0x0a5e] = 0x5e8c,
  [0x0a5f] = 0x5e8d, [0x0a60] = 0x5e8e, [0x0a61] = 0x5e92, [0x0a62] = 0x5e98,
  [0x0a63] = 0x5e9b, [0x0a64] = 0x5e9d, [0x0a65] = 0x5ea1, [0x0a66] = 0x5ea2,
  [0x0a67] = 0x5ea3, [0x0a68] = 0x5ea4, [0x0a69] = 0x5ea8, [0x0a6a] = 0x5ea9,
  [0x0a6b] = 0x5eaa, [0x0a6c] = 0x5eab, [0x0a6d] = 0x5eac, [0x0a6e] = 0x5eae,
  [0x0a6f] = 0x5eaf, [0x0a70] = 0x5eb0, [0x0a71] = 0x5eb1, [0x0a72] = 0x5eb2,
  [0x0a73] = 0x5eb4, [0x0a74] = 0x5eba, [0x0a75] = 0x5ebb, [0x0a76] = 0x5ebc,
  [0x0a77] = 0x5ebd, [0x0a78] = 0x5ebf, [0x0a79] = 0x5ec0, [0x0a7a] = 0x5ec1,
  [0x0a7b] = 0x5ec2, [0x0a7c] = 0x5ec3, [0x0a7d] = 0x5ec4, [0x0a7e] = 0x5ec5,
  [0x0a80] = 0x5ec6, [0x0a81] = 0x5ec7, [0x0a82] = 0x5ec8, [0x0a83] = 0x5ecb,
  [0x0a84] = 0x5ecc, [0x0a85] = 0x5ecd, [0x0a86] = 0x5ece, [0x0a87] = 0x5ecf,
  [0x0a88] = 0x5ed0, [0x0a89] = 0x5ed4, [0x0a8a] = 0x5ed5, [0x0a8b] = 0x5ed7,
  [0x0a8c] = 0x5ed8, [0x0a8d] = 0x5ed9, [0x0a8e] = 0x5eda, [0x0a8f] = 0x5edc,
  [0x0a90] = 0x5edd, [0x0a91] = 0x5ede, [0x0a92] = 0x5edf, [0x0a93] = 0x5ee0,
  [0x0a94] = 0x5ee1, [0x0a95] = 0x5ee2, [0x0a96] = 0x5ee3, [0x0a97] = 0x5ee4,
  [0x0a98] = 0x5ee5, [0x0a99] = 0x5ee6, [0x0a9a] = 0x5ee7, [0x0a9b] = 0x5ee9,
  [0x0a9c] = 0x5eeb, [0x0a9d] = 0x5eec, [0x0a9e] = 0x5eed, [0x0a9f] = 0x5eee,
  [0x0aa0] = 0x5eef, [0x0aa1] = 0x5ef0, [0x0aa2] = 0x5ef1, [0x0aa3] = 0x5ef2,
  [0x0aa4] = 0x5ef3, [0x0aa5] = 0x5ef5, [0x0aa6] = 0x5ef8, [0x0aa7] = 0x5ef9,
  [0x0aa8] = 0x5efb, [0x0aa9] = 0x5efc, [0x0aaa] = 0x5efd, [0x0aab] = 0x5f05,
  [0x0aac] = 0x5f06, [0x0aad] = 0x5f07, [0x0aae] = 0x5f09, [0x0aaf] = 0x5f0c,
  [0x0ab0] = 0x5f0d, [0x0ab1] = 0x5f0e, [0x0ab2] = 0x5f10, [0x0ab3] = 0x5f12,
  [0x0ab4] = 0x5f14, [0x0ab5] = 0x5f16, [0x0ab6] = 0x5f19, [0x0ab7] = 0x5f1a,
  [0x0ab8] = 0x5f1c, [0x0ab9] = 0x5f1d, [0x0aba] = 0x5f1e, [0x0abb] = 0x5f21,
  [0x0abc] = 0x5f22, [0x0abd] = 0x5f23, [0x0abe] = 0x5f24, [0x0ac0] = 0x5f28,
  [0x0ac1] = 0x5f2b, [0x0ac2] = 0x5f2c, [0x0ac3] = 0x5f2e, [0x0ac4] = 0x5f30,
  [0x0ac5] = 0x5f32, [0x0ac6] = 0x5f33, [0x0ac7] = 0x5f34, [0x0ac8] = 0x5f35,
  [0x0ac9] = 0x5f36, [0x0aca] = 0x5f37, [0x0acb] = 0x5f38, [0x0acc] = 0x5f3b,
  [0x0acd] = 0x5f3d, [0x0ace] = 0x5f3e, [0x0acf] = 0x5f3f, [0x0ad0] = 0x5f41,
  [0x0ad1] = 0x5f42, [0x0ad2] = 0x5f43, [0x0ad3] = 0x5f44, [0x0ad4] = 0x5f45,
  [0x0ad5] = 0x5f46, [0x0ad6] = 0x5f47, [0x0ad7] = 0x5f48, [0x0ad8] = 0x5f49,
  [0x0ad9] = 0x5f4a, [0x0ada] = 0x5f4b, [0x0adb] = 0x5f4c, [0x0adc] = 0x5f4d,
  [0x0add] = 0x5f4e, [0x0ade] = 0x5f4f, [0x0adf] = 0x5f51, [0x0ae0] = 0x5f54,
  [0x0ae1] = 0x5f59, [0x0ae2] = 0x5f5a, [0x0ae3] = 0x5f5b, [0x0ae4] = 0x5f5c,
  [0x0ae5] = 0x5f5e, [0x0ae6] = 0x5f5f, [0x0ae7] = 0x5f60, [0x0ae8] = 0x5f63,
  [0x0ae9] = 0x5f65, [0x0aea] = 0x5f67, [0x0aeb] = 0x5f68, [0x0aec] = 0x5f6b,
  [0x0aed] = 0x5f6e, [0x0aee] = 0x5f6f, [0x0aef] = 0x5f72, [0x0af0] = 0x5f74,
  [0x0af1] = 0x5f75, [0x0af2] = 0x5f76, [0x0af3] = 0x5f78, [0x0af4] = 0x5f7a,
  [0x0af5] = 0x5f7d, [0x0af6] = 0x5f7e, [0x0af7] = 0x5f7f, [0x0af8] = 0x5f83,
  [0x0af9] = 0x5f86, [0x0afa] = 0x5f8d, [0x0afb] = 0x5f8e, [0x0afc] = 0x5f8f,
  [0x0afd] = 0x5f91, [0x0afe] = 0x5f93, [0x0aff] = 0x5f94, [0x0b00] = 0x5f96,
  [0x0b01] = 0x5f9a, [0x0b02] = 0x5f9b, [0x0b03] = 0x5f9d, [0x0b04] = 0x5f9e,
  [0x0b05] = 0x5f9f, [0x0b06] = 0x5fa0, [0x0b07] = 0x5fa2, [0x0b08] = 0x5fa3,
  [0x0b09] = 0x5fa4, [0x0b0a] = 0x5fa5, [0x0b0b] = 0x5fa6, [0x0b0c] = 0x5fa7,
  [0x0b0d] = 0x5fa9, [0x0b0e] = 0x5fab, [0x0b0f] = 0x5fac, [0x0b10] = 0x5faf,
  [0x0b11] = 0x5fb0, [0x0b12] = 0x5fb1, [0x0b13] = 0x5fb2, [0x0b14] = 0x5fb3,
  [0x0b15] = 0x5fb4, [0x0b16] = 0x5fb6, [0x0b17] = 0x5fb8, [0x0b18] = 0x5fb9,
  [0x0b19] = 0x5fba, [0x0b1a] = 0x5fbb, [0x0b1b] = 0x5fbe, [0x0b1c] = 0x5fbf,
  [0x0b1d] = 0x5fc0, [0x0b1e] = 0x5fc1, [0x0b1f] = 0x5fc2, [0x0b20] = 0x5fc7,
  [0x0b21] = 0x5fc8, [0x0b22] = 0x5fca, [0x0b23] = 0x5fcb, [0x0b24] = 0x5fce,
  [0x0b25] = 0x5fd3, [0x0b26] = 0x5fd4, [0x0b27] = 0x5fd5, [0x0b28] = 0x5fda,
  [0x0b29] = 0x5fdb, [0x0b2a] = 0x5fdc, [0x0b2b] = 0x5fde, [0x0b2c] = 0x5fdf,
  [0x0b2d] = 0x5fe2, [0x0b2e] = 0x5fe3, [0x0b2f] = 0x5fe5, [0x0b30] = 0x5fe6,
  [0x0b31] = 0x5fe8, [0x0b32] = 0x5fe9, [0x0b33] = 0x5fec, [0x0b34] = 0x5fef,
  [0x0b35] = 0x5ff0, [0x0b36] = 0x5ff2, [0x0b37] = 0x5ff3, [0x0b38] = 0x5ff4,
  [0x0b39] = 0x5ff6, [0x0b3a] = 0x5ff7, [0x0b3b] = 0x5ff9, [0x0b3c] = 0x5ffa,
  [0x0b3d] = 0x5ffc, [0x0b3e] = 0x6007, [0x0b40] = 0x6008, [0x0b41] = 0x6009,
  [0x0b42] = 0x600b, [0x0b43] = 0x600c, [0x0b44] = 0x6010, [0x0b45] = 0x6011,
  [0x0b46] = 0x6013, [0x0b47] = 0x6017, [0x0b48] = 0x6018, [0x0b49] = 0x601a,
  [0x0b4a] = 0x601e, [0x0b4b] = 0x601f, [0x0b4c] = 0x6022, [0x0b4d] = 0x6023,
  [0x0b4e] = 0x6024, [0x0b4f] = 0x602c, [0x0b50] = 0x602d, [0x0b51] = 0x602e,
  [0x0b52] = 0x6030, [0x0b53] = 0x6031, [0x0b54] = 0x6032, [0x0b55] = 0x6033,
  [0x0b56] = 0x6034, [0x0b57] = 0x6036, [0x0b58] = 0x6037, [0x0b59] = 0x6038,
  [0x0b5a] = 0x6039, [0x0b5b] = 0x603a, [0x0b5c] = 0x603d, [0x0b5d] = 0x603e,
  [0x0b5e] = 0x6040, [0x0b5f] = 0x6044, [0x0b60] = 0x6045, [0x0b61] = 0x6046,
  [0x0b62] = 0x6047, [0x0b63] = 0x6048, [0x0b64] = 0x6049, [0x0b65] = 0x604a,
  [0x0b66] = 0x604c, [0x0b67] = 0x604e, [0x0b68] = 0x604f, [0x0b69] = 0x6051,
  [0x0b6a] = 0x6053, [0x0b6b] = 0x6054, [0x0b6c] = 0x6056, [0x0b6d] = 0x6057,
  [0x0b6e] = 0x6058, [0x0b6f] = 0x605b, [0x0b70] = 0x605c, [0x0b71] = 0x605e,
  [0x0b72] = 0x605f, [0x0b73] = 0x6060, [0x0b74] = 0x6061, [0x0b75] = 0x6065,
  [0x0b76] = 0x6066, [0x0b77] = 0x606e, [0x0b78] = 0x6071, [0x0b79] = 0x6072,
  [0x0b7a] = 0x6074, [0x0b7b] = 0x6075, [0x0b7c] = 0x6077, [0x0b7d] = 0x607e,
  [0x0b7e] = 0x6080, [0x0b80] = 0x6081, [0x0b81] = 0x6082, [0x0b82] = 0x6085,
  [0x0b83] = 0x6086, [0x0b84] = 0x6087, [0x0b85] = 0x6088, [0x0b86] = 0x608a,
  [0x0b87] = 0x608b, [0x0b88] = 0x608e, [0x0b89] = 0x608f, [0x0b8a] = 0x6090,
  [0x0b8b] = 0x6091, [0x0b8c] = 0x6093, [0x0b8d] = 0x6095, [0x0b8e] = 0x6097,
  [0x0b8f] = 0x6098, [0x0b90] = 0x6099, [0x0b91] = 0x609c, [0x0b92] = 0x609e,
  [0x0b93] = 0x60a1, [0x0b94] = 0x60a2, [0x0b95] = 0x60a4, [0x0b96] = 0x60a5,
  [0x0b97] = 0x60a7, [0x0b98] = 0x60a9, [0x0b99] = 0x60aa, [0x0b9a] = 0x60ae,
  [0x0b9b] = 0x60b0, [0x0b9c] = 0x60b3, [0x0b9d] = 0x60b5, [0x0b9e] = 0x60b6,
  [0x0b9f] = 0x60b7, [0x0ba0] = 0x60b9, [0x0ba1] = 0x60ba, [0x0ba2] = 0x60bd,
  [0x0ba3] = 0x60be, [0x0ba4] = 0x60bf, [0x0ba5] = 0x60c0, [0x0ba6] = 0x60c1,
  [0x0ba7] = 0x60c2, [0x0ba8] = 0x60c3, [0x0ba9] = 0x60c4, [0x0baa] = 0x60c7,
  [0x0bab] = 0x60c8, [0x0bac] = 0x60c9, [0x0bad] = 0x60cc, [0x0bae] = 0x60cd,
  [0x0baf] = 0x60ce, [0x0bb0] = 0x60cf, [0x0bb1] = 0x60d0, [0x0bb2] = 0x60d2,
  [0x0bb3] = 0x60d3, [0x0bb4] = 0x60d4, [0x0bb5] = 0x60d6, [0x0bb6] = 0x60d7,
  [0x0bb7] = 0x60d9, [0x0bb8] = 0x60db, [0x0bb9] = 0x60de, [0x0bba] = 0x60e1,
  [0x0bbb] = 0x60e2, [0x0bbc] = 0x60e3, [0x0bbd] = 0x60e4, [0x0bbe] = 0x60e5,
  [0x0bbf] = 0x60ea, [0x0bc0] = 0x60f1, [0x0bc1] = 0x60f2, [0x0bc2] = 0x60f5,
  [0x0bc3] = 0x60f7, [0x0bc4] = 0x60f8, [0x0bc5] = 0x60fb, [0x0bc6] = 0x60fc,
  [0x0bc7] = 0x60fd, [0x0bc8] = 0x60fe, [0x0bc9] = 0x60ff, [0x0bca] = 0x6102,
  [0x0bcb] = 0x6103, [0x0bcc] = 0x6104, [0x0bcd] = 0x6105, [0x0bce] = 0x6107,
  [0x0bcf] = 0x610a, [0x0bd0] = 0x610b, [0x0bd1] = 0x610c, [0x0bd2] = 0x6110,
  [0x0bd3] = 0x6111, [0x0bd4] = 0x6112, [0x0bd5] = 0x6113, [0x0bd6] = 0x6114,
  [0x0bd7] = 0x6116, [0x0bd8] = 0x6117, [0x0bd9] = 0x6118, [0x0bda] = 0x6119,
  [0x0bdb] = 0x611b, [0x0bdc] = 0x611c, [0x0bdd] = 0x611d, [0x0bde] = 0x611e,
  [0x0bdf] = 0x6121, [0x0be0] = 0x6122, [0x0be1] = 0x6125, [0x0be2] = 0x6128,
  [0x0be3] = 0x6129, [0x0be4] = 0x612a, [0x0be5] = 0x612c, [0x0be6] = 0x612d,
  [0x0be7] = 0x612e, [0x0be8] = 0x612f, [0x0be9] = 0x6130, [0x0bea] = 0x6131,
  [0x0beb] = 0x6132, [0x0bec] = 0x6133, [0x0bed] = 0x6134, [0x0bee] = 0x6135,
  [0x0bef] = 0x6136, [0x0bf0] = 0x6137, [0x0bf1] = 0x6138, [0x0bf2] = 0x6139,
  [0x0bf3] = 0x613a, [0x0bf4] = 0x613b, [0x0bf5] = 0x613c, [0x0bf6] = 0x613d,
  [0x0bf7] = 0x613e, [0x0bf8] = 0x6140, [0x0bf9] = 0x6141, [0x0bfa] = 0x6142,
  [0x0bfb] = 0x6143, [0x0bfc] = 0x6144, [0x0bfd] = 0x6145, [0x0bfe] = 0x6146,
  [0x0c00] = 0x6147, [0x0c01] = 0x6149, [0x0c02] = 0x614b, [0x0c03] = 0x614d,
  [0x0c04] = 0x614f, [0x0c05] = 0x6150, [0x0c06] = 0x6152, [0x0c07] = 0x6153,
  [0x0c08] = 0x6154, [0x0c09] = 0x6156, [0x0c0a] = 0x6157, [0x0c0b] = 0x6158,
  [0x0c0c] = 0x6159, [0x0c0d] = 0x615a, [0x0c0e] = 0x615b, [0x0c0f] = 0x615c,
  [0x0c10] = 0x615e, [0x0c11] = 0x615f, [0x0c12] = 0x6160, [0x0c13] = 0x6161,
  [0x0c14] = 0x6163, [0x0c15] = 0x6164, [0x0c16] = 0x6165, [0x0c17] = 0x6166,
  [0x0c18] = 0x6169, [0x0c19] = 0x616a, [0x0c1a] = 0x616b, [0x0c1b] = 0x616c,
  [0x0c1c] = 0x616d, [0x0c1d] = 0x616e, [0x0c1e] = 0x616f, [0x0c1f] = 0x6171,
  [0x0c20] = 0x6172, [0x0c21] = 0x6173, [0x0c22] = 0x6174, [0x0c23] = 0x6176,
  [0x0c24] = 0x6178, [0x0c25] = 0x6179, [0x0c26] = 0x617a, [0x0c27] = 0x617b,
  [0x0c28] = 0x617c, [0x0c29] = 0x617d, [0x0c2a] = 0x617e, [0x0c2b] = 0x617f,
  [0x0c2c] = 0x6180, [0x0c2d] = 0x6181, [0x0c2e] = 0x6182, [0x0c2f] = 0x6183,
  [0x0c30] = 0x6184, [0x0c31] = 0x6185, [0x0c32] = 0x6186, [0x0c33] = 0x6187,
  [0x0c34] = 0x6188, [0x0c35] = 0x6189, [0x0c36] = 0x618a, [0x0c37] = 0x618c,
  [0x0c38] = 0x618d, [0x0c39] = 0x618f, [0x0c3a] = 0x6190, [0x0c3b] = 0x6191,
  [0x0c3c] = 0x6192, [0x0c3d] = 0x6193, [0x0c3e] = 0x6195, [0x0c40] = 0x6196,
  [0x0c41] = 0x6197, [0x0c42] = 0x6198, [0x0c43] = 0x6199, [0x0c44] = 0x619a,
  [0x0c45] = 0x619b, [0x0c46] = 0x619c, [0x0c47] = 0x619e, [0x0c48] = 0x619f,
  [0x0c49] = 0x61a0, [0x0c4a] = 0x61a1, [0x0c4b] = 0x61a2, [0x0c4c] = 0x61a3,
  [0x0c4d] = 0x61a4, [0x0c4e] = 0x61a5, [0x0c4f] = 0x61a6, [0x0c50] = 0x61aa,
  [0x0c51] = 0x61ab, [0x0c52] = 0x61ad, [0x0c53] = 0x61ae, [0x0c54] = 0x61af,
  [0x0c55] = 0x61b0, [0x0c56] = 0x61b1, [0x0c57] = 0x61b2, [0x0c58] = 0x61b3,
  [0x0c59] = 0x61b4, [0x0c5a] = 0x61b5, [0x0c5b] = 0x61b6, [0x0c5c] = 0x61b8,
  [0x0c5d] = 0x61b9, [0x0c5e] = 0x61ba, [0x0c5f] = 0x61bb, [0x0c60] = 0x61bc,
  [0x0c61] = 0x61bd, [0x0c62] = 0x61bf, [0x0c63] = 0x61c0, [0x0c64] = 0x61c1,
  [0x0c65] = 0x61c3, [0x0c66] = 0x61c4, [0x0c67] = 0x61c5, [0x0c68] = 0x61c6,
  [0x0c69] = 0x61c7, [0x0c6a] = 0x61c9, [0x0c6b] = 0x61cc, [0x0c6c] = 0x61cd,
  [0x0c6d] = 0x61ce, [0x0c6e] = 0x61cf, [0x0c6f] = 0x61d0, [0x0c70] = 0x61d3,
  [0x0c71] = 0x61d5, [0x0c72] = 0x61d6, [0x0c73] = 0x61d7, [0x0c74] = 0x61d8,
  [0x0c75] = 0x61d9, [0x0c76] = 0x61da, [0x0c77] = 0x61db, [0x0c78] = 0x61dc,
  [0x0c79] = 0x61dd, [0x0c7a] = 0x61de, [0x0c7b] = 0x61df, [0x0c7c] = 0x61e0,
  [0x0c7d] = 0x61e1, [0x0c7e] = 0x61e2, [0x0c7f] = 0x61e3, [0x0c80] = 0x61e4,
  [0x0c81] = 0x61e5, [0x0c82] = 0x61e7, [0x0c83] = 0x61e8, [0x0c84] = 0x61e9,
  [0x0c85] = 0x61ea, [0x0c86] = 0x61eb, [0x0c87] = 0x61ec, [0x0c88] = 0x61ed,
  [0x0c89] = 0x61ee, [0x0c8a] = 0x61ef, [0x0c8b] = 0x61f0, [0x0c8c] = 0x61f1,
  [0x0c8d] = 0x61f2, [0x0c8e] = 0x61f3, [0x0c8f] = 0x61f4, [0x0c90] = 0x61f6,
  [0x0c91] = 0x61f7, [0x0c92] = 0x61f8, [0x0c93] = 0x61f9, [0x0c94] = 0x61fa,
  [0x0c95] = 0x61fb, [0x0c96] = 0x61fc, [0x0c97] = 0x61fd, [0x0c98] = 0x61fe,
  [0x0c99] = 0x6200, [0x0c9a] = 0x6201, [0x0c9b] = 0x6202, [0x0c9c] = 0x6203,
  [0x0c9d] = 0x6204, [0x0c9e] = 0x6205, [0x0c9f] = 0x6207, [0x0ca0] = 0x6209,
  [0x0ca1] = 0x6213, [0x0ca2] = 0x6214, [0x0ca3] = 0x6219, [0x0ca4] = 0x621c,
  [0x0ca5] = 0x621d, [0x0ca6] = 0x621e, [0x0ca7] = 0x6220, [0x0ca8] = 0x6223,
  [0x0ca9] = 0x6226, [0x0caa] = 0x6227, [0x0cab] = 0x6228, [0x0cac] = 0x6229,
  [0x0cad] = 0x622b, [0x0cae] = 0x622d, [0x0caf] = 0x622f, [0x0cb0] = 0x6230,
  [0x0cb1] = 0x6231, [0x0cb2] = 0x6232, [0x0cb3] = 0x6235, [0x0cb4] = 0x6236,
  [0x0cb5] = 0x6238, [0x0cb6] = 0x6239, [0x0cb7] = 0x623a, [0x0cb8] = 0x623b,
  [0x0cb9] = 0x623c, [0x0cba] = 0x6242, [0x0cbb] = 0x6244, [0x0cbc] = 0x6245,
  [0x0cbd] = 0x6246, [0x0cbe] = 0x624a, [0x0cc0] = 0x624f, [0x0cc1] = 0x6250,
  [0x0cc2] = 0x6255, [0x0cc3] = 0x6256, [0x0cc4] = 0x6257, [0x0cc5] = 0x6259,
  [0x0cc6] = 0x625a, [0x0cc7] = 0x625c, [0x0cc8] = 0x625d, [0x0cc9] = 0x625e,
  [0x0cca] = 0x625f, [0x0ccb] = 0x6260, [0x0ccc] = 0x6261, [0x0ccd] = 0x6262,
  [0x0cce] = 0x6264, [0x0ccf] = 0x6265, [0x0cd0] = 0x6268, [0x0cd1] = 0x6271,
  [0x0cd2] = 0x6272, [0x0cd3] = 0x6274, [0x0cd4] = 0x6275, [0x0cd5] = 0x6277,
  [0x0cd6] = 0x6278, [0x0cd7] = 0x627a, [0x0cd8] = 0x627b, [0x0cd9] = 0x627d,
  [0x0cda] = 0x6281, [0x0cdb] = 0x6282, [0x0cdc] = 0x6283, [0x0cdd] = 0x6285,
  [0x0cde] = 0x6286, [0x0cdf] = 0x6287, [0x0ce0] = 0x6288, [0x0ce1] = 0x628b,
  [0x0ce2] = 0x628c, [0x0ce3] = 0x628d, [0x0ce4] = 0x628e, [0x0ce5] = 0x628f,
  [0x0ce6] = 0x6290, [0x0ce7] = 0x6294, [0x0ce8] = 0x6299, [0x0ce9] = 0x629c,
  [0x0cea] = 0x629d, [0x0ceb] = 0x629e, [0x0cec] = 0x62a3, [0x0ced] = 0x62a6,
  [0x0cee] = 0x62a7, [0x0cef] = 0x62a9, [0x0cf0] = 0x62aa, [0x0cf1] = 0x62ad,
  [0x0cf2] = 0x62ae, [0x0cf3] = 0x62af, [0x0cf4] = 0x62b0, [0x0cf5] = 0x62b2,
  [0x0cf6] = 0x62b3, [0x0cf7] = 0x62b4, [0x0cf8] = 0x62b6, [0x0cf9] = 0x62b7,
  [0x0cfa] = 0x62b8, [0x0cfb] = 0x62ba, [0x0cfc] = 0x62be, [0x0cfd] = 0x62c0,
  [0x0cfe] = 0x62c1, [0x0d00] = 0x62c3, [0x0d01] = 0x62cb, [0x0d02] = 0x62cf,
  [0x0d03] = 0x62d1, [0x0d04] = 0x62d5, [0x0d05] = 0x62dd, [0x0d06] = 0x62de,
  [0x0d07] = 0x62e0, [0x0d08] = 0x62e1, [0x0d09] = 0x62e4, [0x0d0a] = 0x62ea,
  [0x0d0b] = 0x62eb, [0x0d0c] = 0x62f0, [0x0d0d] = 0x62f2, [0x0d0e] = 0x62f5,
  [0x0d0f] = 0x62f8, [0x0d10] = 0x62f9, [0x0d11] = 0x62fa, [0x0d12] = 0x62fb,
  [0x0d13] = 0x6300, [0x0d14] = 0x6303, [0x0d15] = 0x6304, [0x0d16] = 0x6305,
  [0x0d17] = 0x6306, [0x0d18] = 0x630a, [0x0d19] = 0x630b, [0x0d1a] = 0x630c,
  [0x0d1b] = 0x630d, [0x0d1c] = 0x630f, [0x0d1d] = 0x6310, [0x0d1e] = 0x6312,
  [0x0d1f] = 0x6313, [0x0d20] = 0x6314, [0x0d21] = 0x6315, [0x0d22] = 0x6317,
  [0x0d23] = 0x6318, [0x0d24] = 0x6319, [0x0d25] = 0x631c, [0x0d26] = 0x6326,
  [0x0d27] = 0x6327, [0x0d28] = 0x6329, [0x0d29] = 0x632c, [0x0d2a] = 0x632d,
  [0x0d2b] = 0x632e, [0x0d2c] = 0x6330, [0x0d2d] = 0x6331, [0x0d2e] = 0x6333,
  [0x0d2f] = 0x6334, [0x0d30] = 0x6335, [0x0d31] = 0x6336, [0x0d32] = 0x6337,
  [0x0d33] = 0x6338, [0x0d34] = 0x633b, [0x0d35] = 0x633c, [0x0d36] = 0x633e,
  [0x0d37] = 0x633f, [0x0d38] = 0x6340, [0x0d39] = 0x6341, [0x0d3a] = 0x6344,
  [0x0d3b] = 0x6347, [0x0d3c] = 0x6348, [0x0d3d] = 0x634a, [0x0d3e] = 0x6351,
  [0x0d3f] = 0x6352, [0x0d40] = 0x6353, [0x0d41] = 0x6354, [0x0d42] = 0x6356,
  [0x0d43] = 0x6357, [0x0d44] = 0x6358, [0x0d45] = 0x6359, [0x0d46] = 0x635a,
  [0x0d47] = 0x635b, [0x0d48] = 0x635c, [0x0d49] = 0x635d, [0x0d4a] = 0x6360,
  [0x0d4b] = 0x6364, [0x0d4c] = 0x6365, [0x0d4d] = 0x6366, [0x0d4e] = 0x6368,
  [0x0d4f] = 0x636a, [0x0d50] = 0x636b, [0x0d51] = 0x636c, [0x0d52] = 0x636f,
  [0x0d53] = 0x6370, [0x0d54] = 0x6372, [0x0d55] = 0x6373, [0x0d56] = 0x6374,
  [0x0d57] = 0x6375, [0x0d58] = 0x6378, [0x0d59] = 0x6379, [0x0d5a] = 0x637c,
  [0x0d5b] = 0x637d, [0x0d5c] = 0x637e, [0x0d5d] = 0x637f, [0x0d5e] = 0x6381,
  [0x0d5f] = 0x6383, [0x0d60] = 0x6384, [0x0d61] = 0x6385, [0x0d62] = 0x6386,
  [0x0d63] = 0x638b, [0x0d64] = 0x638d, [0x0d65] = 0x6391, [0x0d66] = 0x6393,
  [0x0d67] = 0x6394, [0x0d68] = 0x6395, [0x0d69] = 0x6397, [0x0d6a] = 0x6399,
  [0x0d6b] = 0x639a, [0x0d6c] = 0x639b, [0x0d6d] = 0x639c, [0x0d6e] = 0x639d,
  [0x0d6f] = 0x639e, [0x0d70] = 0x639f, [0x0d71] = 0x63a1, [0x0d72] = 0x63a4,
  [0x0d73] = 0x63a6, [0x0d74] = 0x63ab, [0x0d75] = 0x63af, [0x0d76] = 0x63b1,
  [0x0d77] = 0x63b2, [0x0d78] = 0x63b5, [0x0d79] = 0x63b6, [0x0d7a] = 0x63b9,
  [0x0d7b] = 0x63bb, [0x0d7c] = 0x63bd, [0x0d7d] = 0x63bf, [0x0d7e] = 0x63c0,
  [0x0d80] = 0x63c1, [0x0d81] = 0x63c2, [0x0d82] = 0x63c3, [0x0d83] = 0x63c5,
  [0x0d84] = 0x63c7, [0x0d85] = 0x63c8, [0x0d86] = 0x63ca, [0x0d87] = 0x63cb,
  [0x0d88] = 0x63cc, [0x0d89] = 0x63d1, [0x0d8a] = 0x63d3, [0x0d8b] = 0x63d4,
  [0x0d8c] = 0x63d5, [0x0d8d] = 0x63d7, [0x0d8e] = 0x63d8, [0x0d8f] = 0x63d9,
  [0x0d90] = 0x63da, [0x0d91] = 0x63db, [0x0d92] = 0x63dc, [0x0d93] = 0x63dd,
  [0x0d94] = 0x63df, [0x0d95] = 0x63e2, [0x0d96] = 0x63e4, [0x0d97] = 0x63e5,
  [0x0d98] = 0x63e6, [0x0d99] = 0x63e7, [0x0d9a] = 0x63e8, [0x0d9b] = 0x63eb,
  [0x0d9c] = 0x63ec, [0x0d9d] = 0x63ee, [0x0d9e] = 0x63ef, [0x0d9f] = 0x63f0,
  [0x0da0] = 0x63f1, [0x0da1] = 0x63f3, [0x0da2] = 0x63f5, [0x0da3] = 0x63f7,
  [0x0da4] = 0x63f9, [0x0da5] = 0x63fa, [0x0da6] = 0x63fb, [0x0da7] = 0x63fc,
  [0x0da8] = 0x63fe, [0x0da9] = 0x6403, [0x0daa] = 0x6404, [0x0dab] = 0x6406,
  [0x0dac] = 0x6407, [0x0dad] = 0x6408, [0x0dae] = 0x6409, [0x0daf] = 0x640a,
  [0x0db0] = 0x640d, [0x0db1] = 0x640e, [0x0db2] = 0x6411, [0x0db3] = 0x6412,
  [0x0db4] = 0x6415, [0x0db5] = 0x6416, [0x0db6] = 0x6417, [0x0db7] = 0x6418,
  [0x0db8] = 0x6419, [0x0db9] = 0x641a, [0x0dba] = 0x641d, [0x0dbb] = 0x641f,
  [0x0dbc] = 0x6422, [0x0dbd] = 0x6423, [0x0dbe] = 0x6424, [0x0dc0] = 0x6425,
  [0x0dc1] = 0x6427, [0x0dc2] = 0x6428, [0x0dc3] = 0x6429, [0x0dc4] = 0x642b,
  [0x0dc5] = 0x642e, [0x0dc6] = 0x642f, [0x0dc7] = 0x6430, [0x0dc8] = 0x6431,
  [0x0dc9] = 0x6432, [0x0dca] = 0x6433, [0x0dcb] = 0x6435, [0x0dcc] = 0x6436,
  [0x0dcd] = 0x6437, [0x0dce] = 0x6438, [0x0dcf] = 0x6439, [0x0dd0] = 0x643b,
  [0x0dd1] = 0x643c, [0x0dd2] = 0x643e, [0x0dd3] = 0x6440, [0x0dd4] = 0x6442,
  [0x0dd5] = 0x6443, [0x0dd6] = 0x6449, [0x0dd7] = 0x644b, [0x0dd8] = 0x644c,
  [0x0dd9] = 0x644d, [0x0dda] = 0x644e, [0x0ddb] = 0x644f, [0x0ddc] = 0x6450,
  [0x0ddd] = 0x6451, [0x0dde] = 0x6453, [0x0ddf] = 0x6455, [0x0de0] = 0x6456,
  [0x0de1] = 0x6457, [0x0de2] = 0x6459, [0x0de3] = 0x645a, [0x0de4] = 0x645b,
  [0x0de5] = 0x645c, [0x0de6] = 0x645d, [0x0de7] = 0x645f, [0x0de8] = 0x6460,
  [0x0de9] = 0x6461, [0x0dea] = 0x6462, [0x0deb] = 0x6463, [0x0dec] = 0x6464,
  [0x0ded] = 0x6465, [0x0dee] = 0x6466, [0x0def] = 0x6468, [0x0df0] = 0x646a,
  [0x0df1] = 0x646b, [0x0df2] = 0x646c, [0x0df3] = 0x646e, [0x0df4] = 0x646f,
  [0x0df5] = 0x6470, [0x0df6] = 0x6471, [0x0df7] = 0x6472, [0x0df8] = 0x6473,
  [0x0df9] = 0x6474, [0x0dfa] = 0x6475, [0x0dfb] = 0x6476, [0x0dfc] = 0x6477,
  [0x0dfd] = 0x647b, [0x0dfe] = 0x647c, [0x0dff] = 0x647d, [0x0e00] = 0x647e,
  [0x0e01] = 0x647f, [0x0e02] = 0x6480, [0x0e03] = 0x6481, [0x0e04] = 0x6483,
  [0x0e05] = 0x6486, [0x0e06] = 0x6488, [0x0e07] = 0x6489, [0x0e08] = 0x648a,
  [0x0e09] = 0x648b, [0x0e0a] = 0x648c, [0x0e0b] = 0x648d, [0x0e0c] = 0x648e,
  [0x0e0d] = 0x648f, [0x0e0e] = 0x6490, [0x0e0f] = 0x6493, [0x0e10] = 0x6494,
  [0x0e11] = 0x6497, [0x0e12] = 0x6498, [0x0e13] = 0x649a, [0x0e14] = 0x649b,
  [0x0e15] = 0x649c, [0x0e16] = 0x649d, [0x0e17] = 0x649f, [0x0e18] = 0x64a0,
  [0x0e19] = 0x64a1, [0x0e1a] = 0x64a2, [0x0e1b] = 0x64a3, [0x0e1c] = 0x64a5,
  [0x0e1d] = 0x64a6, [0x0e1e] = 0x64a7, [0x0e1f] = 0x64a8, [0x0e20] = 0x64aa,
  [0x0e21] = 0x64ab, [0x0e22] = 0x64af, [0x0e23] = 0x64b1, [0x0e24] = 0x64b2,
  [0x0e25] = 0x64b3, [0x0e26] = 0x64b4, [0x0e27] = 0x64b6, [0x0e28] = 0x64b9,
  [0x0e29] = 0x64bb, [0x0e2a] = 0x64bd, [0x0e2b] = 0x64be, [0x0e2c] = 0x64bf,
  [0x0e2d] = 0x64c1, [0x0e2e] = 0x64c3, [0x0e2f] = 0x64c4, [0x0e30] = 0x64c6,
  [0x0e31] = 0x64c7, [0x0e32] = 0x64c8, [0x0e33] = 0x64c9, [0x0e34] = 0x64ca,
  [0x0e35] = 0x64cb, [0x0e36] = 0x64cc, [0x0e37] = 0x64cf, [0x0e38] = 0x64d1,
  [0x0e39] = 0x64d3, [0x0e3a] = 0x64d4, [0x0e3b] = 0x64d5, [0x0e3c] = 0x64d6,
  [0x0e3d] = 0x64d9, [0x0e3e] = 0x64da, [0x0e40] = 0x64db, [0x0e41] = 0x64dc,
  [0x0e42] = 0x64dd, [0x0e43] = 0x64df, [0x0e44] = 0x64e0, [0x0e45] = 0x64e1,
  [0x0e46] = 0x64e3, [0x0e47] = 0x64e5, [0x0e48] = 0x64e7, [0x0e49] = 0x64e8,
  [0x0e4a] = 0x64e9, [0x0e4b] = 0x64ea, [0x0e4c] = 0x64eb, [0x0e4d] = 0x64ec,
  [0x0e4e] = 0x64ed, [0x0e4f] = 0x64ee, [0x0e50] = 0x64ef, [0x0e51] = 0x64f0,
  [0x0e52] = 0x64f1, [0x0e53] = 0x64f2, [0x0e54] = 0x64f3, [0x0e55] = 0x64f4,
  [0x0e56] = 0x64f5, [0x0e57] = 0x64f6, [0x0e58] = 0x64f7, [0x0e59] = 0x64f8,
  [0x0e5a] = 0x64f9, [0x0e5b] = 0x64fa, [0x0e5c] = 0x64fb, [0x0e5d] = 0x64fc,
  [0x0e5e] = 0x64fd, [0x0e5f] = 0x64fe, [0x0e60] = 0x64ff, [0x0e61] = 0x6501,
  [0x0e62] = 0x6502, [0x0e63] = 0x6503, [0x0e64] = 0x6504, [0x0e65] = 0x6505,
  [0x0e66] = 0x6506, [0x0e67] = 0x6507, [0x0e68] = 0x6508, [0x0e69] = 0x650a,
  [0x0e6a] = 0x650b, [0x0e6b] = 0x650c, [0x0e6c] = 0x650d, [0x0e6d] = 0x650e,
  [0x0e6e] = 0x650f, [0x0e6f] = 0x6510, [0x0e70] = 0x6511, [0x0e71] = 0x6513,
  [0x0e72] = 0x6514, [0x0e73] = 0x6515, [0x0e74] = 0x6516, [0x0e75] = 0x6517,
  [0x0e76] = 0x6519, [0x0e77] = 0x651a, [0x0e78] = 0x651b, [0x0e79] = 0x651c,
  [0x0e7a] = 0x651d, [0x0e7b] = 0x651e, [0x0e7c] = 0x651f, [0x0e7d] = 0x6520,
  [0x0e7e] = 0x6521, [0x0e80] = 0x6522, [0x0e81] = 0x6523, [0x0e82] = 0x6524,
  [0x0e83] = 0x6526, [0x0e84] = 0x6527, [0x0e85] = 0x6528, [0x0e86] = 0x6529,
  [0x0e87] = 0x652a, [0x0e88] = 0x652c, [0x0e89] = 0x652d, [0x0e8a] = 0x6530,
  [0x0e8b] = 0x6531, [0x0e8c] = 0x6532, [0x0e8d] = 0x6533, [0x0e8e] = 0x6537,
  [0x0e8f] = 0x653a, [0x0e90] = 0x653c, [0x0e91] = 0x653d, [0x0e92] = 0x6540,
  [0x0e93] = 0x6541, [0x0e94] = 0x6542, [0x0e95] = 0x6543, [0x0e96] = 0x6544,
  [0x0e97] = 0x6546, [0x0e98] = 0x6547, [0x0e99] = 0x654a, [0x0e9a] = 0x654b,
  [0x0e9b] = 0x654d, [0x0e9c] = 0x654e, [0x0e9d] = 0x6550, [0x0e9e] = 0x6552,
  [0x0e9f] = 0x6553, [0x0ea0] = 0x6554, [0x0ea1] = 0x6557, [0x0ea2] = 0x6558,
  [0x0ea3] = 0x655a, [0x0ea4] = 0x655c, [0x0ea5] = 0x655f, [0x0ea6] = 0x6560,
  [0x0ea7] = 0x6561, [0x0ea8] = 0x6564, [0x0ea9] = 0x6565, [0x0eaa] = 0x6567,
  [0x0eab] = 0x6568, [0x0eac] = 0x6569, [0x0ead] = 0x656a, [0x0eae] = 0x656d,
  [0x0eaf] = 0x656e, [0x0eb0] = 0x656f, [0x0eb1] = 0x6571, [0x0eb2] = 0x6573,
  [0x0eb3] = 0x6575, [0x0eb4] = 0x6576, [0x0eb5] = 0x6578, [0x0eb6] = 0x6579,
  [0x0eb7] = 0x657a, [0x0eb8] = 0x657b, [0x0eb9] = 0x657c, [0x0eba] = 0x657d,
  [0x0ebb] = 0x657e, [0x0ebc] = 0x657f, [0x0ebd] = 0x6580, [0x0ebe] = 0x6581,
  [0x0ebf] = 0x6582, [0x0ec0] = 0x6583, [0x0ec1] = 0x6584, [0x0ec2] = 0x6585,
  [0x0ec3] = 0x6586, [0x0ec4] = 0x6588, [0x0ec5] = 0x6589, [0x0ec6] = 0x658a,
  [0x0ec7] = 0x658d, [0x0ec8] = 0x658e, [0x0ec9] = 0x658f, [0x0eca] = 0x6592,
  [0x0ecb] = 0x6594, [0x0ecc] = 0x6595, [0x0ecd] = 0x6596, [0x0ece] = 0x6598,
  [0x0ecf] = 0x659a, [0x0ed0] = 0x659d, [0x0ed1] = 0x659e, [0x0ed2] = 0x65a0,
  [0x0ed3] = 0x65a2, [0x0ed4] = 0x65a3, [0x0ed5] = 0x65a6, [0x0ed6] = 0x65a8,
  [0x0ed7] = 0x65aa, [0x0ed8] = 0x65ac, [0x0ed9] = 0x65ae, [0x0eda] = 0x65b1,
  [0x0edb] = 0x65b2, [0x0edc] = 0x65b3, [0x0edd] = 0x65b4, [0x0ede] = 0x65b5,
  [0x0edf] = 0x65b6, [0x0ee0] = 0x65b7, [0x0ee1] = 0x65b8, [0x0ee2] = 0x65ba,
  [0x0ee3] = 0x65bb, [0x0ee4] = 0x65be, [0x0ee5] = 0x65bf, [0x0ee6] = 0x65c0,
  [0x0ee7] = 0x65c2, [0x0ee8] = 0x65c7, [0x0ee9] = 0x65c8, [0x0eea] = 0x65c9,
  [0x0eeb] = 0x65ca, [0x0eec] = 0x65cd, [0x0eed] = 0x65d0, [0x0eee] = 0x65d1,
  [0x0eef] = 0x65d3, [0x0ef0] = 0x65d4, [0x0ef1] = 0x65d5, [0x0ef2] = 0x65d8,
  [0x0ef3] = 0x65d9, [0x0ef4] = 0x65da, [0x0ef5] = 0x65db, [0x0ef6] = 0x65dc,
  [0x0ef7] = 0x65dd, [0x0ef8] = 0x65de, [0x0ef9] = 0x65df, [0x0efa] = 0x65e1,
  [0x0efb] = 0x65e3, [0x0efc] = 0x65e4, [0x0efd] = 0x65ea, [0x0efe] = 0x65eb,
  [0x0f00] = 0x65f2, [0x0f01] = 0x65f3, [0x0f02] = 0x65f4, [0x0f03] = 0x65f5,
  [0x0f04] = 0x65f8, [0x0f05] = 0x65f9, [0x0f06] = 0x65fb, [0x0f07] = 0x65fc,
  [0x0f08] = 0x65fd, [0x0f09] = 0x65fe, [0x0f0a] = 0x65ff, [0x0f0b] = 0x6601,
  [0x0f0c] = 0x6604, [0x0f0d] = 0x6605, [0x0f0e] = 0x6607, [0x0f0f] = 0x6608,
  [0x0f10] = 0x6609, [0x0f11] = 0x660b, [0x0f12] = 0x660d, [0x0f13] = 0x6610,
  [0x0f14] = 0x6611, [0x0f15] = 0x6612, [0x0f16] = 0x6616, [0x0f17] = 0x6617,
  [0x0f18] = 0x6618, [0x0f19] = 0x661a, [0x0f1a] = 0x661b, [0x0f1b] = 0x661c,
  [0x0f1c] = 0x661e, [0x0f1d] = 0x6621, [0x0f1e] = 0x6622, [0x0f1f] = 0x6623,
  [0x0f20] = 0x6624, [0x0f21] = 0x6626, [0x0f22] = 0x6629, [0x0f23] = 0x662a,
  [0x0f24] = 0x662b, [0x0f25] = 0x662c, [0x0f26] = 0x662e, [0x0f27] = 0x6630,
  [0x0f28] = 0x6632, [0x0f29] = 0x6633, [0x0f2a] = 0x6637, [0x0f2b] = 0x6638,
  [0x0f2c] = 0x6639, [0x0f2d] = 0x663a, [0x0f2e] = 0x663b, [0x0f2f] = 0x663d,
  [0x0f30] = 0x663f, [0x0f31] = 0x6640, [0x0f32] = 0x6642, [0x0f33] = 0x6644,
  [0x0f34] = 0x6645, [0x0f35] = 0x6646, [0x0f36] = 0x6647, [0x0f37] = 0x6648,
  [0x0f38] = 0x6649, [0x0f39] = 0x664a, [0x0f3a] = 0x664d, [0x0f3b] = 0x664e,
  [0x0f3c] = 0x6650, [0x0f3d] = 0x6651, [0x0f3e] = 0x6658, [0x0f40] = 0x6659,
  [0x0f41] = 0x665b, [0x0f42] = 0x665c, [0x0f43] = 0x665d, [0x0f44] = 0x665e,
  [0x0f45] = 0x6660, [0x0f46] = 0x6662, [0x0f47] = 0x6663, [0x0f48] = 0x6665,
  [0x0f49] = 0x6667, [0x0f4a] = 0x6669, [0x0f4b] = 0x666a, [0x0f4c] = 0x666b,
  [0x0f4d] = 0x666c, [0x0f4e] = 0x666d, [0x0f4f] = 0x6671, [0x0f50] = 0x6672,
  [0x0f51] = 0x6673, [0x0f52] = 0x6675, [0x0f53] = 0x6678, [0x0f54] = 0x6679,
  [0x0f55] = 0x667b, [0x0f56] = 0x667c, [0x0f57] = 0x667d, [0x0f58] = 0x667f,
  [0x0f59] = 0x6680, [0x0f5a] = 0x6681, [0x0f5b] = 0x6683, [0x0f5c] = 0x6685,
  [0x0f5d] = 0x6686, [0x0f5e] = 0x6688, [0x0f5f] = 0x6689, [0x0f60] = 0x668a,
  [0x0f61] = 0x668b, [0x0f62] = 0x668d, [0x0f63] = 0x668e, [0x0f64] = 0x668f,
  [0x0f65] = 0x6690, [0x0f66] = 0x6692, [0x0f67] = 0x6693, [0x0f68] = 0x6694,
  [0x0f69] = 0x6695, [0x0f6a] = 0x6698, [0x0f6b] = 0x6699, [0x0f6c] = 0x669a,
  [0x0f6d] = 0x669b, [0x0f6e] = 0x669c, [0x0f6f] = 0x669e, [0x0f70] = 0x669f,
  [0x0f71] = 0x66a0, [0x0f72] = 0x66a1, [0x0f73] = 0x66a2, [0x0f74] = 0x66a3,
  [0x0f75] = 0x66a4, [0x0f76] = 0x66a5, [0x0f77] = 0x66a6, [0x0f78] = 0x66a9,
  [0x0f79] = 0x66aa, [0x0f7a] = 0x66ab, [0x0f7b] = 0x66ac, [0x0f7c] = 0x66ad,
  [0x0f7d] = 0x66af, [0x0f7e] = 0x66b0, [0x0f7f] = 0x66b1, [0x0f80] = 0x66b2,
  [0x0f81] = 0x66b3, [0x0f82] = 0x66b5, [0x0f83] = 0x66b6, [0x0f84] = 0x66b7,
  [0x0f85] = 0x66b8, [0x0f86] = 0x66ba, [0x0f87] = 0x66bb, [0x0f88] = 0x66bc,
  [0x0f89] = 0x66bd, [0x0f8a] = 0x66bf, [0x0f8b] = 0x66c0, [0x0f8c] = 0x66c1,
  [0x0f8d] = 0x66c2, [0x0f8e] = 0x66c3, [0x0f8f] = 0x66c4, [0x0f90] = 0x66c5,
  [0x0f91] = 0x66c6, [0x0f92] = 0x66c7, [0x0f93] = 0x66c8, [0x0f94] = 0x66c9,
  [0x0f95] = 0x66ca, [0x0f96] = 0x66cb, [0x0f97] = 0x66cc, [0x0f98] = 0x66cd,
  [0x0f99] = 0x66ce, [0x0f9a] = 0x66cf, [0x0f9b] = 0x66d0, [0x0f9c] = 0x66d1,
  [0x0f9d] = 0x66d2, [0x0f9e] = 0x66d3, [0x0f9f] = 0x66d4, [0x0fa0] = 0x66d5,
  [0x0fa1] = 0x66d6, [0x0fa2] = 0x66d7, [0x0fa3] = 0x66d8, [0x0fa4] = 0x66da,
  [0x0fa5] = 0x66de, [0x0fa6] = 0x66df, [0x0fa7] = 0x66e0, [0x0fa8] = 0x66e1,
  [0x0fa9] = 0x66e2, [0x0faa] = 0x66e3, [0x0fab] = 0x66e4, [0x0fac] = 0x66e5,
  [0x0fad] = 0x66e7, [0x0fae] = 0x66e8, [0x0faf] = 0x66ea, [0x0fb0] = 0x66eb,
  [0x0fb1] = 0x66ec, [0x0fb2] = 0x66ed, [0x0fb3] = 0x66ee, [0x0fb4] = 0x66ef,
  [0x0fb5] = 0x66f1, [0x0fb6] = 0x66f5, [0x0fb7] = 0x66f6, [0x0fb8] = 0x66f8,
  [0x0fb9] = 0x66fa, [0x0fba] = 0x66fb, [0x0fbb] = 0x66fd, [0x0fbc] = 0x6701,
  [0x0fbd] = 0x6702, [0x0fbe] = 0x6703, [0x0fc0] = 0x6704, [0x0fc1] = 0x6705,
  [0x0fc2] = 0x6706, [0x0fc3] = 0x6707, [0x0fc4] = 0x670c, [0x0fc5] = 0x670e,
  [0x0fc6] = 0x670f, [0x0fc7] = 0x6711, [0x0fc8] = 0x6712, [0x0fc9] = 0x6713,
  [0x0fca] = 0x6716, [0x0fcb] = 0x6718, [0x0fcc] = 0x6719, [0x0fcd] = 0x671a,
  [0x0fce] = 0x671c, [0x0fcf] = 0x671e, [0x0fd0] = 0x6720, [0x0fd1] = 0x6721,
  [0x0fd2] = 0x6722, [0x0fd3] = 0x6723, [0x0fd4] = 0x6724, [0x0fd5] = 0x6725,
  [0x0fd6] = 0x6727, [0x0fd7] = 0x6729, [0x0fd8] = 0x672e, [0x0fd9] = 0x6730,
  [0x0fda] = 0x6732, [0x0fdb] = 0x6733, [0x0fdc] = 0x6736, [0x0fdd] = 0x6737,
  [0x0fde] = 0x6738, [0x0fdf] = 0x6739, [0x0fe0] = 0x673b, [0x0fe1] = 0x673c,
  [0x0fe2] = 0x673e, [0x0fe3] = 0x673f, [0x0fe4] = 0x6741, [0x0fe5] = 0x6744,
  [0x0fe6] = 0x6745, [0x0fe7] = 0x6747, [0x0fe8] = 0x674a, [0x0fe9] = 0x674b,
  [0x0fea] = 0x674d, [0x0feb] = 0x6752, [0x0fec] = 0x6754, [0x0fed] = 0x6755,
  [0x0fee] = 0x6757, [0x0fef] = 0x6758, [0x0ff0] = 0x6759, [0x0ff1] = 0x675a,
  [0x0ff2] = 0x675b, [0x0ff3] = 0x675d, [0x0ff4] = 0x6762, [0x0ff5] = 0x6763,
  [0x0ff6] = 0x6764, [0x0ff7] = 0x6766, [0x0ff8] = 0x6767, [0x0ff9] = 0x676b,
  [0x0ffa] = 0x676c, [0x0ffb] = 0x676e, [0x0ffc] = 0x6771, [0x0ffd] = 0x6774,
  [0x0ffe] = 0x6776, [0x1000] = 0x6778, [0x1001] = 0x6779, [0x1002] = 0x677a,
  [0x1003] = 0x677b, [0x1004] = 0x677d, [0x1005] = 0x6780, [0x1006] = 0x6782,
  [0x1007] = 0x6783, [0x1008] = 0x6785, [0x1009] = 0x6786, [0x100a] = 0x6788,
  [0x100b] = 0x678a, [0x100c] = 0x678c, [0x100d] = 0x678d, [0x100e] = 0x678e,
  [0x100f] = 0x678f, [0x1010] = 0x6791, [0x1011] = 0x6792, [0x1012] = 0x6793,
  [0x1013] = 0x6794, [0x1014] = 0x6796, [0x1015] = 0x6799, [0x1016] = 0x679b,
  [0x1017] = 0x679f, [0x1018] = 0x67a0, [0x1019] = 0x67a1, [0x101a] = 0x67a4,
  [0x101b] = 0x67a6, [0x101c] = 0x67a9, [0x101d] = 0x67ac, [0x101e] = 0x67ae,
  [0x101f] = 0x67b1, [0x1020] = 0x67b2, [0x1021] = 0x67b4, [0x1022] = 0x67b9,
  [0x1023] = 0x67ba, [0x1024] = 0x67bb, [0x1025] = 0x67bc, [0x1026] = 0x67bd,
  [0x1027] = 0x67be, [0x1028] = 0x67bf, [0x1029] = 0x67c0, [0x102a] = 0x67c2,
  [0x102b] = 0x67c5, [0x102c] = 0x67c6, [0x102d] = 0x67c7, [0x102e] = 0x67c8,
  [0x102f] = 0x67c9, [0x1030] = 0x67ca, [0x1031] = 0x67cb, [0x1032] = 0x67cc,
  [0x1033] = 0x67cd, [0x1034] = 0x67ce, [0x1035] = 0x67d5, [0x1036] = 0x67d6,
  [0x1037] = 0x67d7, [0x1038] = 0x67db, [0x1039] = 0x67df, [0x103a] = 0x67e1,
  [0x103b] = 0x67e3, [0x103c] = 0x67e4, [0x103d] = 0x67e6, [0x103e] = 0x67e7,
  [0x103f] = 0x67e8, [0x1040] = 0x67ea, [0x1041] = 0x67eb, [0x1042] = 0x67ed,
  [0x1043] = 0x67ee, [0x1044] = 0x67f2, [0x1045] = 0x67f5, [0x1046] = 0x67f6,
  [0x1047] = 0x67f7, [0x1048] = 0x67f8, [0x1049] = 0x67f9, [0x104a] = 0x67fa,
  [0x104b] = 0x67fb, [0x104c] = 0x67fc, [0x104d] = 0x67fe, [0x104e] = 0x6801,
  [0x104f] = 0x6802, [0x1050] = 0x6803, [0x1051] = 0x6804, [0x1052] = 0x6806,
  [0x1053] = 0x680d, [0x1054] = 0x6810, [0x1055] = 0x6812, [0x1056] = 0x6814,
  [0x1057] = 0x6815, [0x1058] = 0x6818, [0x1059] = 0x6819, [0x105a] = 0x681a,
  [0x105b] = 0x681b, [0x105c] = 0x681c, [0x105d] = 0x681e, [0x105e] = 0x681f,
  [0x105f] = 0x6820, [0x1060] = 0x6822, [0x1061] = 0x6823, [0x1062] = 0x6824,
  [0x1063] = 0x6825, [0x1064] = 0x6826, [0x1065] = 0x6827, [0x1066] = 0x6828,
  [0x1067] = 0x682b, [0x1068] = 0x682c, [0x1069] = 0x682d, [0x106a] = 0x682e,
  [0x106b] = 0x682f, [0x106c] = 0x6830, [0x106d] = 0x6831, [0x106e] = 0x6834,
  [0x106f] = 0x6835, [0x1070] = 0x6836, [0x1071] = 0x683a, [0x1072] = 0x683b,
  [0x1073] = 0x683f, [0x1074] = 0x6847, [0x1075] = 0x684b, [0x1076] = 0x684d,
  [0x1077] = 0x684f, [0x1078] = 0x6852, [0x1079] = 0x6856, [0x107a] = 0x6857,
  [0x107b] = 0x6858, [0x107c] = 0x6859, [0x107d] = 0x685a, [0x107e] = 0x685b,
  [0x1080] = 0x685c, [0x1081] = 0x685d, [0x1082] = 0x685e, [0x1083] = 0x685f,
  [0x1084] = 0x686a, [0x1085] = 0x686c, [0x1086] = 0x686d, [0x1087] = 0x686e,
  [0x1088] = 0x686f, [0x1089] = 0x6870, [0x108a] = 0x6871, [0x108b] = 0x6872,
  [0x108c] = 0x6873, [0x108d] = 0x6875, [0x108e] = 0x6878, [0x108f] = 0x6879,
  [0x1090] = 0x687a, [0x1091] = 0x687b, [0x1092] = 0x687c, [0x1093] = 0x687d,
  [0x1094] = 0x687e, [0x1095] = 0x687f, [0x1096] = 0x6880, [0x1097] = 0x6882,
  [0x1098] = 0x6884, [0x1099] = 0x6887, [0x109a] = 0x6888, [0x109b] = 0x6889,
  [0x109c] = 0x688a, [0x109d] = 0x688b, [0x109e] = 0x688c, [0x109f] = 0x688d,
  [0x10a0] = 0x688e, [0x10a1] = 0x6890, [0x10a2] = 0x6891, [0x10a3] = 0x6892,
  [0x10a4] = 0x6894, [0x10a5] = 0x6895, [0x10a6] = 0x6896, [0x10a7] = 0x6898,
  [0x10a8] = 0x6899, [0x10a9] = 0x689a, [0x10aa] = 0x689b, [0x10ab] = 0x689c,
  [0x10ac] = 0x689d, [0x10ad] = 0x689e, [0x10ae] = 0x689f, [0x10af] = 0x68a0,
  [0x10b0] = 0x68a1, [0x10b1] = 0x68a3, [0x10b2] = 0x68a4, [0x10b3] = 0x68a5,
  [0x10b4] = 0x68a9, [0x10b5] = 0x68aa, [0x10b6] = 0x68ab, [0x10b7] = 0x68ac,
  [0x10b8] = 0x68ae, [0x10b9] = 0x68b1, [0x10ba] = 0x68b2, [0x10bb] = 0x68b4,
  [0x10bc] = 0x68b6, [0x10bd] = 0x68b7, [0x10be] = 0x68b8, [0x10c0] = 0x68b9,
  [0x10c1] = 0x68ba, [0x10c2] = 0x68bb, [0x10c3] = 0x68bc, [0x10c4] = 0x68bd,
  [0x10c5] = 0x68be, [0x10c6] = 0x68bf, [0x10c7] = 0x68c1, [0x10c8] = 0x68c3,
  [0x10c9] = 0x68c4, [0x10ca] = 0x68c5, [0x10cb] = 0x68c6, [0x10cc] = 0x68c7,
  [0x10cd] = 0x68c8, [0x10ce] = 0x68ca, [0x10cf] = 0x68cc, [0x10d0] = 0x68ce,
  [0x10d1] = 0x68cf, [0x10d2] = 0x68d0, [0x10d3] = 0x68d1, [0x10d4] = 0x68d3,
  [0x10d5] = 0x68d4, [0x10d6] = 0x68d6, [0x10d7] = 0x68d7, [0x10d8] = 0x68d9,
  [0x10d9] = 0x68db, [0x10da] = 0x68dc, [0x10db] = 0x68dd, [0x10dc] = 0x68de,
  [0x10dd] = 0x68df, [0x10de] = 0x68e1, [0x10df] = 0x68e2, [0x10e0] = 0x68e4,
  [0x10e1] = 0x68e5, [0x10e2] = 0x68e6, [0x10e3] = 0x68e7, [0x10e4] = 0x68e8,
  [0x10e5] = 0x68e9, [0x10e6] = 0x68ea, [0x10e7] = 0x68eb, [0x10e8] = 0x68ec,
  [0x10e9] = 0x68ed, [0x10ea] = 0x68ef, [0x10eb] = 0x68f2, [0x10ec] = 0x68f3,
  [0x10ed] = 0x68f4, [0x10ee] = 0x68f6, [0x10ef] = 0x68f7, [0x10f0] = 0x68f8,
  [0x10f1] = 0x68fb, [0x10f2] = 0x68fd, [0x10f3] = 0x68fe, [0x10f4] = 0x68ff,
  [0x10f5] = 0x6900, [0x10f6] = 0x6902, [0x10f7] = 0x6903, [0x10f8] = 0x6904,
  [0x10f9] = 0x6906, [0x10fa] = 0x6907, [0x10fb] = 0x6908, [0x10fc] = 0x6909,
  [0x10fd] = 0x690a, [0x10fe] = 0x690c, [0x10ff] = 0x690f, [0x1100] = 0x6911,
  [0x1101] = 0x6913, [0x1102] = 0x6914, [0x1103] = 0x6915, [0x1104] = 0x6916,
  [0x1105] = 0x6917, [0x1106] = 0x6918, [0x1107] = 0x6919, [0x1108] = 0x691a,
  [0x1109] = 0x691b, [0x110a] = 0x691c, [0x110b] = 0x691d, [0x110c] = 0x691e,
  [0x110d] = 0x6921, [0x110e] = 0x6922, [0x110f] = 0x6923, [0x1110] = 0x6925,
  [0x1111] = 0x6926, [0x1112] = 0x6927, [0x1113] = 0x6928, [0x1114] = 0x6929,
  [0x1115] = 0x692a, [0x1116] = 0x692b, [0x1117] = 0x692c, [0x1118] = 0x692e,
  [0x1119] = 0x692f, [0x111a] = 0x6931, [0x111b] = 0x6932, [0x111c] = 0x6933,
  [0x111d] = 0x6935, [0x111e] = 0x6936, [0x111f] = 0x6937, [0x1120] = 0x6938,
  [0x1121] = 0x693a, [0x1122] = 0x693b, [0x1123] = 0x693c, [0x1124] = 0x693e,
  [0x1125] = 0x6940, [0x1126] = 0x6941, [0x1127] = 0x6943, [0x1128] = 0x6944,
  [0x1129] = 0x6945, [0x112a] = 0x6946, [0x112b] = 0x6947, [0x112c] = 0x6948,
  [0x112d] = 0x6949, [0x112e] = 0x694a, [0x112f] = 0x694b, [0x1130] = 0x694c,
  [0x1131] = 0x694d, [0x1132] = 0x694e, [0x1133] = 0x694f, [0x1134] = 0x6950,
  [0x1135] = 0x6951, [0x1136] = 0x6952, [0x1137] = 0x6953, [0x1138] = 0x6955,
  [0x1139] = 0x6956, [0x113a] = 0x6958, [0x113b] = 0x6959, [0x113c] = 0x695b,
  [0x113d] = 0x695c, [0x113e] = 0x695f, [0x1140] = 0x6961, [0x1141] = 0x6962,
  [0x1142] = 0x6964, [0x1143] = 0x6965, [0x1144] = 0x6967, [0x1145] = 0x6968,
  [0x1146] = 0x6969, [0x1147] = 0x696a, [0x1148] = 0x696c, [0x1149] = 0x696d,
  [0x114a] = 0x696f, [0x114b] = 0x6970, [0x114c] = 0x6972, [0x114d] = 0x6973,
  [0x114e] = 0x6974, [0x114f] = 0x6975, [0x1150] = 0x6976, [0x1151] = 0x697a,
  [0x1152] = 0x697b, [0x1153] = 0x697d, [0x1154] = 0x697e, [0x1155] = 0x697f,
  [0x1156] = 0x6981, [0x1157] = 0x6983, [0x1158] = 0x6985, [0x1159] = 0x698a,
  [0x115a] = 0x698b, [0x115b] = 0x698c, [0x115c] = 0x698e, [0x115d] = 0x698f,
  [0x115e] = 0x6990, [0x115f] = 0x6991, [0x1160] = 0x6992, [0x1161] = 0x6993,
  [0x1162] = 0x6996, [0x1163] = 0x6997, [0x1164] = 0x6999, [0x1165] = 0x699a,
  [0x1166] = 0x699d, [0x1167] = 0x699e, [0x1168] = 0x699f, [0x1169] = 0x69a0,
  [0x116a] = 0x69a1, [0x116b] = 0x69a2, [0x116c] = 0x69a3, [0x116d] = 0x69a4,
  [0x116e] = 0x69a5, [0x116f] = 0x69a6, [0x1170] = 0x69a9, [0x1171] = 0x69aa,
  [0x1172] = 0x69ac, [0x1173] = 0x69ae, [0x1174] = 0x69af, [0x1175] = 0x69b0,
  [0x1176] = 0x69b2, [0x1177] = 0x69b3, [0x1178] = 0x69b5, [0x1179] = 0x69b6,
  [0x117a] = 0x69b8, [0x117b] = 0x69b9, [0x117c] = 0x69ba, [0x117d] = 0x69bc,
  [0x117e] = 0x69bd, [0x1180] = 0x69be, [0x1181] = 0x69bf, [0x1182] = 0x69c0,
  [0x1183] = 0x69c2, [0x1184] = 0x69c3, [0x1185] = 0x69c4, [0x1186] = 0x69c5,
  [0x1187] = 0x69c6, [0x1188] = 0x69c7, [0x1189] = 0x69c8, [0x118a] = 0x69c9,
  [0x118b] = 0x69cb, [0x118c] = 0x69cd, [0x118d] = 0x69cf, [0x118e] = 0x69d1,
  [0x118f] = 0x69d2, [0x1190] = 0x69d3, [0x1191] = 0x69d5, [0x1192] = 0x69d6,
  [0x1193] = 0x69d7, [0x1194] = 0x69d8, [0x1195] = 0x69d9, [0x1196] = 0x69da,
  [0x1197] = 0x69dc, [0x1198] = 0x69dd, [0x1199] = 0x69de, [0x119a] = 0x69e1,
  [0x119b] = 0x69e2, [0x119c] = 0x69e3, [0x119d] = 0x69e4, [0x119e] = 0x69e5,
  [0x119f] = 0x69e6, [0x11a0] = 0x69e7, [0x11a1] = 0x69e8, [0x11a2] = 0x69e9,
  [0x11a3] = 0x69ea, [0x11a4] = 0x69eb, [0x11a5] = 0x69ec, [0x11a6] = 0x69ee,
  [0x11a7] = 0x69ef, [0x11a8] = 0x69f0, [0x11a9] = 0x69f1, [0x11aa] = 0x69f3,
  [0x11ab] = 0x69f4, [0x11ac] = 0x69f5, [0x11ad] = 0x69f6, [0x11ae] = 0x69f7,
  [0x11af] = 0x69f8, [0x11b0] = 0x69f9, [0x11b1] = 0x69fa, [0x11b2] = 0x69fb,
  [0x11b3] = 0x69fc, [0x11b4] = 0x69fe, [0x11b5] = 0x6a00, [0x11b6] = 0x6a01,
  [0x11b7] = 0x6a02, [0x11b8] = 0x6a03, [0x11b9] = 0x6a04, [0x11ba] = 0x6a05,
  [0x11bb] = 0x6a06, [0x11bc] = 0x6a07, [0x11bd] = 0x6a08, [0x11be] = 0x6a09,
  [0x11bf] = 0x6a0b, [0x11c0] = 0x6a0c, [0x11c1] = 0x6a0d, [0x11c2] = 0x6a0e,
  [0x11c3] = 0x6a0f, [0x11c4] = 0x6a10, [0x11c5] = 0x6a11, [0x11c6] = 0x6a12,
  [0x11c7] = 0x6a13, [0x11c8] = 0x6a14, [0x11c9] = 0x6a15, [0x11ca] = 0x6a16,
  [0x11cb] = 0x6a19, [0x11cc] = 0x6a1a, [0x11cd] = 0x6a1b, [0x11ce] = 0x6a1c,
  [0x11cf] = 0x6a1d, [0x11d0] = 0x6a1e, [0x11d1] = 0x6a20, [0x11d2] = 0x6a22,
  [0x11d3] = 0x6a23, [0x11d4] = 0x6a24, [0x11d5] = 0x6a25, [0x11d6] = 0x6a26,
  [0x11d7] = 0x6a27, [0x11d8] = 0x6a29, [0x11d9] = 0x6a2b, [0x11da] = 0x6a2c,
  [0x11db] = 0x6a2d, [0x11dc] = 0x6a2e, [0x11dd] = 0x6a30, [0x11de] = 0x6a32,
  [0x11df] = 0x6a33, [0x11e0] = 0x6a34, [0x11e1] = 0x6a36, [0x11e2] = 0x6a37,
  [0x11e3] = 0x6a38, [0x11e4] = 0x6a39, [0x11e5] = 0x6a3a, [0x11e6] = 0x6a3b,
  [0x11e7] = 0x6a3c, [0x11e8] = 0x6a3f, [0x11e9] = 0x6a40, [0x11ea] = 0x6a41,
  [0x11eb] = 0x6a42, [0x11ec] = 0x6a43, [0x11ed] = 0x6a45, [0x11ee] = 0x6a46,
  [0x11ef] = 0x6a48, [0x11f0] = 0x6a49, [0x11f1] = 0x6a4a, [0x11f2] = 0x6a4b,
  [0x11f3] = 0x6a4c, [0x11f4] = 0x6a4d, [0x11f5] = 0x6a4e, [0x11f6] = 0x6a4f,
  [0x11f7] = 0x6a51, [0x11f8] = 0x6a52, [0x11f9] = 0x6a53, [0x11fa] = 0x6a54,
  [0x11fb] = 0x6a55, [0x11fc] = 0x6a56, [0x11fd] = 0x6a57, [0x11fe] = 0x6a5a,
  [0x1200] = 0x6a5c, [0x1201] = 0x6a5d, [0x1202] = 0x6a5e, [0x1203] = 0x6a5f,
  [0x1204] = 0x6a60, [0x1205] = 0x6a62, [0x1206] = 0x6a63, [0x1207] = 0x6a64,
  [0x1208] = 0x6a66, [0x1209] = 0x6a67, [0x120a] = 0x6a68, [0x120b] = 0x6a69,
  [0x120c] = 0x6a6a, [0x120d] = 0x6a6b, [0x120e] = 0x6a6c, [0x120f] = 0x6a6d,
  [0x1210] = 0x6a6e, [0x1211] = 0x6a6f, [0x1212] = 0x6a70, [0x1213] = 0x6a72,
  [0x1214] = 0x6a73, [0x1215] = 0x6a74, [0x1216] = 0x6a75, [0x1217] = 0x6a76,
  [0x1218] = 0x6a77, [0x1219] = 0x6a78, [0x121a] = 0x6a7a, [0x121b] = 0x6a7b,
  [0x121c] = 0x6a7d, [0x121d] = 0x6a7e, [0x121e] = 0x6a7f, [0x121f] = 0x6a81,
  [0x1220] = 0x6a82, [0x1221] = 0x6a83, [0x1222] = 0x6a85, [0x1223] = 0x6a86,
  [0x1224] = 0x6a87, [0x1225] = 0x6a88, [0x1226] = 0x6a89, [0x1227] = 0x6a8a,
  [0x1228] = 0x6a8b, [0x1229] = 0x6a8c, [0x122a] = 0x6a8d, [0x122b] = 0x6a8f,
  [0x122c] = 0x6a92, [0x122d] = 0x6a93, [0x122e] = 0x6a94, [0x122f] = 0x6a95,
  [0x1230] = 0x6a96, [0x1231] = 0x6a98, [0x1232] = 0x6a99, [0x1233] = 0x6a9a,
  [0x1234] = 0x6a9b, [0x1235] = 0x6a9c, [0x1236] = 0x6a9d, [0x1237] = 0x6a9e,
  [0x1238] = 0x6a9f, [0x1239] = 0x6aa1, [0x123a] = 0x6aa2, [0x123b] = 0x6aa3,
  [0x123c] = 0x6aa4, [0x123d] = 0x6aa5, [0x123e] = 0x6aa6, [0x1240] = 0x6aa7,
  [0x1241] = 0x6aa8, [0x1242] = 0x6aaa, [0x1243] = 0x6aad, [0x1244] = 0x6aae,
  [0x1245] = 0x6aaf, [0x1246] = 0x6ab0, [0x1247] = 0x6ab1, [0x1248] = 0x6ab2,
  [0x1249] = 0x6ab3, [0x124a] = 0x6ab4, [0x124b] = 0x6ab5, [0x124c] = 0x6ab6,
  [0x124d] = 0x6ab7, [0x124e] = 0x6ab8, [0x124f] = 0x6ab9, [0x1250] = 0x6aba,
  [0x1251] = 0x6abb, [0x1252] = 0x6abc, [0x1253] = 0x6abd, [0x1254] = 0x6abe,
  [0x1255] = 0x6abf, [0x1256] = 0x6ac0, [0x1257] = 0x6ac1, [0x1258] = 0x6ac2,
  [0x1259] = 0x6ac3, [0x125a] = 0x6ac4, [0x125b] = 0x6ac5, [0x125c] = 0x6ac6,
  [0x125d] = 0x6ac7, [0x125e] = 0x6ac8, [0x125f] = 0x6ac9, [0x1260] = 0x6aca,
  [0x1261] = 0x6acb, [0x1262] = 0x6acc, [0x1263] = 0x6acd, [0x1264] = 0x6ace,
  [0x1265] = 0x6acf, [0x1266] = 0x6ad0, [0x1267] = 0x6ad1, [0x1268] = 0x6ad2,
  [0x1269] = 0x6ad3, [0x126a] = 0x6ad4, [0x126b] = 0x6ad5, [0x126c] = 0x6ad6,
  [0x126d] = 0x6ad7, [0x126e] = 0x6ad8, [0x126f] = 0x6ad9, [0x1270] = 0x6ada,
  [0x1271] = 0x6adb, [0x1272] = 0x6adc, [0x1273] = 0x6add, [0x1274] = 0x6ade,
  [0x1275] = 0x6adf, [0x1276] = 0x6ae0, [0x1277] = 0x6ae1, [0x1278] = 0x6ae2,
  [0x1279] = 0x6ae3, [0x127a] = 0x6ae4, [0x127b] = 0x6ae5, [0x127c] = 0x6ae6,
  [0x127d] = 0x6ae7, [0x127e] = 0x6ae8, [0x127f] = 0x6ae9, [0x1280] = 0x6aea,
  [0x1281] = 0x6aeb, [0x1282] = 0x6aec, [0x1283] = 0x6aed, [0x1284] = 0x6aee,
  [0x1285] = 0x6aef, [0x1286] = 0x6af0, [0x1287] = 0x6af1, [0x1288] = 0x6af2,
  [0x1289] = 0x6af3, [0x128a] = 0x6af4, [0x128b] = 0x6af5, [0x128c] = 0x6af6,
  [0x128d] = 0x6af7, [0x128e] = 0x6af8, [0x128f] = 0x6af9, [0x1290] = 0x6afa,
  [0x1291] = 0x6afb, [0x1292] = 0x6afc, [0x1293] = 0x6afd, [0x1294] = 0x6afe,
  [0x1295] = 0x6aff, [0x1296] = 0x6b00, [0x1297] = 0x6b01, [0x1298] = 0x6b02,
  [0x1299] = 0x6b03, [0x129a] = 0x6b04, [0x129b] = 0x6b05, [0x129c] = 0x6b06,
  [0x129d] = 0x6b07, [0x129e] = 0x6b08, [0x129f] = 0x6b09, [0x12a0] = 0x6b0a,
  [0x12a1] = 0x6b0b, [0x12a2] = 0x6b0c, [0x12a3] = 0x6b0d, [0x12a4] = 0x6b0e,
  [0x12a5] = 0x6b0f, [0x12a6] = 0x6b10, [0x12a7] = 0x6b11, [0x12a8] = 0x6b12,
  [0x12a9] = 0x6b13, [0x12aa] = 0x6b14, [0x12ab] = 0x6b15, [0x12ac] = 0x6b16,
  [0x12ad] = 0x6b17, [0x12ae] = 0x6b18, [0x12af] = 0x6b19, [0x12b0] = 0x6b1a,
  [0x12b1] = 0x6b1b, [0x12b2] = 0x6b1c, [0x12b3] = 0x6b1d, [0x12b4] = 0x6b1e,
  [0x12b5] = 0x6b1f, [0x12b6] = 0x6b25, [0x12b7] = 0x6b26, [0x12b8] = 0x6b28,
  [0x12b9] = 0x6b29, [0x12ba] = 0x6b2a, [0x12bb] = 0x6b2b, [0x12bc] = 0x6b2c,
  [0x12bd] = 0x6b2d, [0x12be] = 0x6b2e, [0x12c0] = 0x6b2f, [0x12c1] = 0x6b30,
  [0x12c2] = 0x6b31, [0x12c3] = 0x6b33, [0x12c4] = 0x6b34, [0x12c5] = 0x6b35,
  [0x12c6] = 0x6b36, [0x12c7] = 0x6b38, [0x12c8] = 0x6b3b, [0x12c9] = 0x6b3c,
  [0x12ca] = 0x6b3d, [0x12cb] = 0x6b3f, [0x12cc] = 0x6b40, [0x12cd] = 0x6b41,
  [0x12ce] = 0x6b42, [0x12cf] = 0x6b44, [0x12d0] = 0x6b45, [0x12d1] = 0x6b48,
  [0x12d2] = 0x6b4a, [0x12d3] = 0x6b4b, [0x12d4] = 0x6b4d, [0x12d5] = 0x6b4e,
  [0x12d6] = 0x6b4f, [0x12d7] = 0x6b50, [0x12d8] = 0x6b51, [0x12d9] = 0x6b52,
  [0x12da] = 0x6b53, [0x12db] = 0x6b54, [0x12dc] = 0x6b55, [0x12dd] = 0x6b56,
  [0x12de] = 0x6b57, [0x12df] = 0x6b58, [0x12e0] = 0x6b5a, [0x12e1] = 0x6b5b,
  [0x12e2] = 0x6b5c, [0x12e3] = 0x6b5d, [0x12e4] = 0x6b5e, [0x12e5] = 0x6b5f,
  [0x12e6] = 0x6b60, [0x12e7] = 0x6b61, [0x12e8] = 0x6b68, [0x12e9] = 0x6b69,
  [0x12ea] = 0x6b6b, [0x12eb] = 0x6b6c, [0x12ec] = 0x6b6d, [0x12ed] = 0x6b6e,
  [0x12ee] = 0x6b6f, [0x12ef] = 0x6b70, [0x12f0] = 0x6b71, [0x12f1] = 0x6b72,
  [0x12f2] = 0x6b73, [0x12f3] = 0x6b74, [0x12f4] = 0x6b75, [0x12f5] = 0x6b76,
  [0x12f6] = 0x6b77, [0x12f7] = 0x6b78, [0x12f8] = 0x6b7a, [0x12f9] = 0x6b7d,
  [0x12fa] = 0x6b7e, [0x12fb] = 0x6b7f, [0x12fc] = 0x6b80, [0x12fd] = 0x6b85,
  [0x12fe] = 0x6b88, [0x1300] = 0x6b8c, [0x1301] = 0x6b8e, [0x1302] = 0x6b8f,
  [0x1303] = 0x6b90, [0x1304] = 0x6b91, [0x1305] = 0x6b94, [0x1306] = 0x6b95,
  [0x1307] = 0x6b97, [0x1308] = 0x6b98, [0x1309] = 0x6b99, [0x130a] = 0x6b9c,
  [0x130b] = 0x6b9d, [0x130c] = 0x6b9e, [0x130d] = 0x6b9f, [0x130e] = 0x6ba0,
  [0x130f] = 0x6ba2, [0x1310] = 0x6ba3, [0x1311] = 0x6ba4, [0x1312] = 0x6ba5,
  [0x1313] = 0x6ba6, [0x1314] = 0x6ba7, [0x1315] = 0x6ba8, [0x1316] = 0x6ba9,
  [0x1317] = 0x6bab, [0x1318] = 0x6bac, [0x1319] = 0x6bad, [0x131a] = 0x6bae,
  [0x131b] = 0x6baf, [0x131c] = 0x6bb0, [0x131d] = 0x6bb1, [0x131e] = 0x6bb2,
  [0x131f] = 0x6bb6, [0x1320] = 0x6bb8, [0x1321] = 0x6bb9, [0x1322] = 0x6bba,
  [0x1323] = 0x6bbb, [0x1324] = 0x6bbc, [0x1325] = 0x6bbd, [0x1326] = 0x6bbe,
  [0x1327] = 0x6bc0, [0x1328] = 0x6bc3, [0x1329] = 0x6bc4, [0x132a] = 0x6bc6,
  [0x132b] = 0x6bc7, [0x132c] = 0x6bc8, [0x132d] = 0x6bc9, [0x132e] = 0x6bca,
  [0x132f] = 0x6bcc, [0x1330] = 0x6bce, [0x1331] = 0x6bd0, [0x1332] = 0x6bd1,
  [0x1333] = 0x6bd8, [0x1334] = 0x6bda, [0x1335] = 0x6bdc, [0x1336] = 0x6bdd,
  [0x1337] = 0x6bde, [0x1338] = 0x6bdf, [0x1339] = 0x6be0, [0x133a] = 0x6be2,
  [0x133b] = 0x6be3, [0x133c] = 0x6be4, [0x133d] = 0x6be5, [0x133e] = 0x6be6,
  [0x133f] = 0x6be7, [0x1340] = 0x6be8, [0x1341] = 0x6be9, [0x1342] = 0x6bec,
  [0x1343] = 0x6bed, [0x1344] = 0x6bee, [0x1345] = 0x6bf0, [0x1346] = 0x6bf1,
  [0x1347] = 0x6bf2, [0x1348] = 0x6bf4, [0x1349] = 0x6bf6, [0x134a] = 0x6bf7,
  [0x134b] = 0x6bf8, [0x134c] = 0x6bfa, [0x134d] = 0x6bfb, [0x134e] = 0x6bfc,
  [0x134f] = 0x6bfe, [0x1350] = 0x6bff, [0x1351] = 0x6c00, [0x1352] = 0x6c01,
  [0x1353] = 0x6c02, [0x1354] = 0x6c03, [0x1355] = 0x6c04, [0x1356] = 0x6c08,
  [0x1357] = 0x6c09, [0x1358] = 0x6c0a, [0x1359] = 0x6c0b, [0x135a] = 0x6c0c,
  [0x135b] = 0x6c0e, [0x135c] = 0x6c12, [0x135d] = 0x6c17, [0x135e] = 0x6c1c,
  [0x135f] = 0x6c1d, [0x1360] = 0x6c1e, [0x1361] = 0x6c20, [0x1362] = 0x6c23,
  [0x1363] = 0x6c25, [0x1364] = 0x6c2b, [0x1365] = 0x6c2c, [0x1366] = 0x6c2d,
  [0x1367] = 0x6c31, [0x1368] = 0x6c33, [0x1369] = 0x6c36, [0x136a] = 0x6c37,
  [0x136b] = 0x6c39, [0x136c] = 0x6c3a, [0x136d] = 0x6c3b, [0x136e] = 0x6c3c,
  [0x136f] = 0x6c3e, [0x1370] = 0x6c3f, [0x1371] = 0x6c43, [0x1372] = 0x6c44,
  [0x1373] = 0x6c45, [0x1374] = 0x6c48, [0x1375] = 0x6c4b, [0x1376] = 0x6c4c,
  [0x1377] = 0x6c4d, [0x1378] = 0x6c4e, [0x1379] = 0x6c4f, [0x137a] = 0x6c51,
  [0x137b] = 0x6c52, [0x137c] = 0x6c53, [0x137d] = 0x6c56, [0x137e] = 0x6c58,
  [0x1380] = 0x6c59, [0x1381] = 0x6c5a, [0x1382] = 0x6c62, [0x1383] = 0x6c63,
  [0x1384] = 0x6c65, [0x1385] = 0x6c66, [0x1386] = 0x6c67, [0x1387] = 0x6c6b,
  [0x1388] = 0x6c6c, [0x1389] = 0x6c6d, [0x138a] = 0x6c6e, [0x138b] = 0x6c6f,
  [0x138c] = 0x6c71, [0x138d] = 0x6c73, [0x138e] = 0x6c75, [0x138f] = 0x6c77,
  [0x1390] = 0x6c78, [0x1391] = 0x6c7a, [0x1392] = 0x6c7b, [0x1393] = 0x6c7c,
  [0x1394] = 0x6c7f, [0x1395] = 0x6c80, [0x1396] = 0x6c84, [0x1397] = 0x6c87,
  [0x1398] = 0x6c8a, [0x1399] = 0x6c8b, [0x139a] = 0x6c8d, [0x139b] = 0x6c8e,
  [0x139c] = 0x6c91, [0x139d] = 0x6c92, [0x139e] = 0x6c95, [0x139f] = 0x6c96,
  [0x13a0] = 0x6c97, [0x13a1] = 0x6c98, [0x13a2] = 0x6c9a, [0x13a3] = 0x6c9c,
  [0x13a4] = 0x6c9d, [0x13a5] = 0x6c9e, [0x13a6] = 0x6ca0, [0x13a7] = 0x6ca2,
  [0x13a8] = 0x6ca8, [0x13a9] = 0x6cac, [0x13aa] = 0x6caf, [0x13ab] = 0x6cb0,
  [0x13ac] = 0x6cb4, [0x13ad] = 0x6cb5, [0x13ae] = 0x6cb6, [0x13af] = 0x6cb7,
  [0x13b0] = 0x6cba, [0x13b1] = 0x6cc0, [0x13b2] = 0x6cc1, [0x13b3] = 0x6cc2,
  [0x13b4] = 0x6cc3, [0x13b5] = 0x6cc6, [0x13b6] = 0x6cc7, [0x13b7] = 0x6cc8,
  [0x13b8] = 0x6ccb, [0x13b9] = 0x6ccd, [0x13ba] = 0x6cce, [0x13bb] = 0x6ccf,
  [0x13bc] = 0x6cd1, [0x13bd] = 0x6cd2, [0x13be] = 0x6cd8, [0x13c0] = 0x6cd9,
  [0x13c1] = 0x6cda, [0x13c2] = 0x6cdc, [0x13c3] = 0x6cdd, [0x13c4] = 0x6cdf,
  [0x13c5] = 0x6ce4, [0x13c6] = 0x6ce6, [0x13c7] = 0x6ce7, [0x13c8] = 0x6ce9,
  [0x13c9] = 0x6cec, [0x13ca] = 0x6ced, [0x13cb] = 0x6cf2, [0x13cc] = 0x6cf4,
  [0x13cd] = 0x6cf9, [0x13ce] = 0x6cff, [0x13cf] = 0x6d00, [0x13d0] = 0x6d02,
  [0x13d1] = 0x6d03, [0x13d2] = 0x6d05, [0x13d3] = 0x6d06, [0x13d4] = 0x6d08,
  [0x13d5] = 0x6d09, [0x13d6] = 0x6d0a, [0x13d7] = 0x6d0d, [0x13d8] = 0x6d0f,
  [0x13d9] = 0x6d10, [0x13da] = 0x6d11, [0x13db] = 0x6d13, [0x13dc] = 0x6d14,
  [0x13dd] = 0x6d15, [0x13de] = 0x6d16, [0x13df] = 0x6d18, [0x13e0] = 0x6d1c,
  [0x13e1] = 0x6d1d, [0x13e2] = 0x6d1f, [0x13e3] = 0x6d20, [0x13e4] = 0x6d21,
  [0x13e5] = 0x6d22, [0x13e6] = 0x6d23, [0x13e7] = 0x6d24, [0x13e8] = 0x6d26,
  [0x13e9] = 0x6d28, [0x13ea] = 0x6d29, [0x13eb] = 0x6d2c, [0x13ec] = 0x6d2d,
  [0x13ed] = 0x6d2f, [0x13ee] = 0x6d30, [0x13ef] = 0x6d34, [0x13f0] = 0x6d36,
  [0x13f1] = 0x6d37, [0x13f2] = 0x6d38, [0x13f3] = 0x6d3a, [0x13f4] = 0x6d3f,
  [0x13f5] = 0x6d40, [0x13f6] = 0x6d42, [0x13f7] = 0x6d44, [0x13f8] = 0x6d49,
  [0x13f9] = 0x6d4c, [0x13fa] = 0x6d50, [0x13fb] = 0x6d55, [0x13fc] = 0x6d56,
  [0x13fd] = 0x6d57, [0x13fe] = 0x6d58, [0x13ff] = 0x6d5b, [0x1400] = 0x6d5d,
  [0x1401] = 0x6d5f, [0x1402] = 0x6d61, [0x1403] = 0x6d62, [0x1404] = 0x6d64,
  [0x1405] = 0x6d65, [0x1406] = 0x6d67, [0x1407] = 0x6d68, [0x1408] = 0x6d6b,
  [0x1409] = 0x6d6c, [0x140a] = 0x6d6d, [0x140b] = 0x6d70, [0x140c] = 0x6d71,
  [0x140d] = 0x6d72, [0x140e] = 0x6d73, [0x140f] = 0x6d75, [0x1410] = 0x6d76,
  [0x1411] = 0x6d79, [0x1412] = 0x6d7a, [0x1413] = 0x6d7b, [0x1414] = 0x6d7d,
  [0x1415] = 0x6d7e, [0x1416] = 0x6d7f, [0x1417] = 0x6d80, [0x1418] = 0x6d81,
  [0x1419] = 0x6d83, [0x141a] = 0x6d84, [0x141b] = 0x6d86, [0x141c] = 0x6d87,
  [0x141d] = 0x6d8a, [0x141e] = 0x6d8b, [0x141f] = 0x6d8d, [0x1420] = 0x6d8f,
  [0x1421] = 0x6d90, [0x1422] = 0x6d92, [0x1423] = 0x6d96, [0x1424] = 0x6d97,
  [0x1425] = 0x6d98, [0x1426] = 0x6d99, [0x1427] = 0x6d9a, [0x1428] = 0x6d9c,
  [0x1429] = 0x6da2, [0x142a] = 0x6da5, [0x142b] = 0x6dac, [0x142c] = 0x6dad,
  [0x142d] = 0x6db0, [0x142e] = 0x6db1, [0x142f] = 0x6db3, [0x1430] = 0x6db4,
  [0x1431] = 0x6db6, [0x1432] = 0x6db7, [0x1433] = 0x6db9, [0x1434] = 0x6dba,
  [0x1435] = 0x6dbb, [0x1436] = 0x6dbc, [0x1437] = 0x6dbd, [0x1438] = 0x6dbe,
  [0x1439] = 0x6dc1, [0x143a] = 0x6dc2, [0x143b] = 0x6dc3, [0x143c] = 0x6dc8,
  [0x143d] = 0x6dc9, [0x143e] = 0x6dca, [0x1440] = 0x6dcd, [0x1441] = 0x6dce,
  [0x1442] = 0x6dcf, [0x1443] = 0x6dd0, [0x1444] = 0x6dd2, [0x1445] = 0x6dd3,
  [0x1446] = 0x6dd4, [0x1447] = 0x6dd5, [0x1448] = 0x6dd7, [0x1449] = 0x6dda,
  [0x144a] = 0x6ddb, [0x144b] = 0x6ddc, [0x144c] = 0x6ddf, [0x144d] = 0x6de2,
  [0x144e] = 0x6de3, [0x144f] = 0x6de5, [0x1450] = 0x6de7, [0x1451] = 0x6de8,
  [0x1452] = 0x6de9, [0x1453] = 0x6dea, [0x1454] = 0x6ded, [0x1455] = 0x6def,
  [0x1456] = 0x6df0, [0x1457] = 0x6df2, [0x1458] = 0x6df4, [0x1459] = 0x6df5,
  [0x145a] = 0x6df6, [0x145b] = 0x6df8, [0x145c] = 0x6dfa, [0x145d] = 0x6dfd,
  [0x145e] = 0x6dfe, [0x145f] = 0x6dff, [0x1460] = 0x6e00, [0x1461] = 0x6e01,
  [0x1462] = 0x6e02, [0x1463] = 0x6e03, [0x1464] = 0x6e04, [0x1465] = 0x6e06,
  [0x1466] = 0x6e07, [0x1467] = 0x6e08, [0x1468] = 0x6e09, [0x1469] = 0x6e0b,
  [0x146a] = 0x6e0f, [0x146b] = 0x6e12, [0x146c] = 0x6e13, [0x146d] = 0x6e15,
  [0x146e] = 0x6e18, [0x146f] = 0x6e19, [0x1470] = 0x6e1b, [0x1471] = 0x6e1c,
  [0x1472] = 0x6e1e, [0x1473] = 0x6e1f, [0x1474] = 0x6e22, [0x1475] = 0x6e26,
  [0x1476] = 0x6e27, [0x1477] = 0x6e28, [0x1478] = 0x6e2a, [0x1479] = 0x6e2c,
  [0x147a] = 0x6e2e, [0x147b] = 0x6e30, [0x147c] = 0x6e31, [0x147d] = 0x6e33,
  [0x147e] = 0x6e35, [0x1480] = 0x6e36, [0x1481] = 0x6e37, [0x1482] = 0x6e39,
  [0x1483] = 0x6e3b, [0x1484] = 0x6e3c, [0x1485] = 0x6e3d, [0x1486] = 0x6e3e,
  [0x1487] = 0x6e3f, [0x1488] = 0x6e40, [0x1489] = 0x6e41, [0x148a] = 0x6e42,
  [0x148b] = 0x6e45, [0x148c] = 0x6e46, [0x148d] = 0x6e47, [0x148e] = 0x6e48,
  [0x148f] = 0x6e49, [0x1490] = 0x6e4a, [0x1491] = 0x6e4b, [0x1492] = 0x6e4c,
  [0x1493] = 0x6e4f, [0x1494] = 0x6e50, [0x1495] = 0x6e51, [0x1496] = 0x6e52,
  [0x1497] = 0x6e55, [0x1498] = 0x6e57, [0x1499] = 0x6e59, [0x149a] = 0x6e5a,
  [0x149b] = 0x6e5c, [0x149c] = 0x6e5d, [0x149d] = 0x6e5e, [0x149e] = 0x6e60,
  [0x149f] = 0x6e61, [0x14a0] = 0x6e62, [0x14a1] = 0x6e63, [0x14a2] = 0x6e64,
  [0x14a3] = 0x6e65, [0x14a4] = 0x6e66, [0x14a5] = 0x6e67, [0x14a6] = 0x6e68,
  [0x14a7] = 0x6e69, [0x14a8] = 0x6e6a, [0x14a9] = 0x6e6c, [0x14aa] = 0x6e6d,
  [0x14ab] = 0x6e6f, [0x14ac] = 0x6e70, [0x14ad] = 0x6e71, [0x14ae] = 0x6e72,
  [0x14af] = 0x6e73, [0x14b0] = 0x6e74, [0x14b1] = 0x6e75, [0x14b2] = 0x6e76,
  [0x14b3] = 0x6e77, [0x14b4] = 0x6e78, [0x14b5] = 0x6e79, [0x14b6] = 0x6e7a,
  [0x14b7] = 0x6e7b, [0x14b8] = 0x6e7c, [0x14b9] = 0x6e7d, [0x14ba] = 0x6e80,
  [0x14bb] = 0x6e81, [0x14bc] = 0x6e82, [0x14bd] = 0x6e84, [0x14be] = 0x6e87,
  [0x14bf] = 0x6e88, [0x14c0] = 0x6e8a, [0x14c1] = 0x6e8b, [0x14c2] = 0x6e8c,
  [0x14c3] = 0x6e8d, [0x14c4] = 0x6e8e, [0x14c5] = 0x6e91, [0x14c6] = 0x6e92,
  [0x14c7] = 0x6e93, [0x14c8] = 0x6e94, [0x14c9] = 0x6e95, [0x14ca] = 0x6e96,
  [0x14cb] = 0x6e97, [0x14cc] = 0x6e99, [0x14cd] = 0x6e9a, [0x14ce] = 0x6e9b,
  [0x14cf] = 0x6e9d, [0x14d0] = 0x6e9e, [0x14d1] = 0x6ea0, [0x14d2] = 0x6ea1,
  [0x14d3] = 0x6ea3, [0x14d4] = 0x6ea4, [0x14d5] = 0x6ea6, [0x14d6] = 0x6ea8,
  [0x14d7] = 0x6ea9, [0x14d8] = 0x6eab, [0x14d9] = 0x6eac, [0x14da] = 0x6ead,
  [0x14db] = 0x6eae, [0x14dc] = 0x6eb0, [0x14dd] = 0x6eb3, [0x14de] = 0x6eb5,
  [0x14df] = 0x6eb8, [0x14e0] = 0x6eb9, [0x14e1] = 0x6ebc, [0x14e2] = 0x6ebe,
  [0x14e3] = 0x6ebf, [0x14e4] = 0x6ec0, [0x14e5] = 0x6ec3, [0x14e6] = 0x6ec4,
  [0x14e7] = 0x6ec5, [0x14e8] = 0x6ec6, [0x14e9] = 0x6ec8, [0x14ea] = 0x6ec9,
  [0x14eb] = 0x6eca, [0x14ec] = 0x6ecc, [0x14ed] = 0x6ecd, [0x14ee] = 0x6ece,
  [0x14ef] = 0x6ed0, [0x14f0] = 0x6ed2, [0x14f1] = 0x6ed6, [0x14f2] = 0x6ed8,
  [0x14f3] = 0x6ed9, [0x14f4] = 0x6edb, [0x14f5] = 0x6edc, [0x14f6] = 0x6edd,
  [0x14f7] = 0x6ee3, [0x14f8] = 0x6ee7, [0x14f9] = 0x6eea, [0x14fa] = 0x6eeb,
  [0x14fb] = 0x6eec, [0x14fc] = 0x6eed, [0x14fd] = 0x6eee, [0x14fe] = 0x6eef,
  [0x1500] = 0x6ef0, [0x1501] = 0x6ef1, [0x1502] = 0x6ef2, [0x1503] = 0x6ef3,
  [0x1504] = 0x6ef5, [0x1505] = 0x6ef6, [0x1506] = 0x6ef7, [0x1507] = 0x6ef8,
  [0x1508] = 0x6efa, [0x1509] = 0x6efb, [0x150a] = 0x6efc, [0x150b] = 0x6efd,
  [0x150c] = 0x6efe, [0x150d] = 0x6eff, [0x150e] = 0x6f00, [0x150f] = 0x6f01,
  [0x1510] = 0x6f03, [0x1511] = 0x6f04, [0x1512] = 0x6f05, [0x1513] = 0x6f07,
  [0x1514] = 0x6f08, [0x1515] = 0x6f0a, [0x1516] = 0x6f0b, [0x1517] = 0x6f0c,
  [0x1518] = 0x6f0d, [0x1519] = 0x6f0e, [0x151a] = 0x6f10, [0x151b] = 0x6f11,
  [0x151c] = 0x6f12, [0x151d] = 0x6f16, [0x151e] = 0x6f17, [0x151f] = 0x6f18,
  [0x1520] = 0x6f19, [0x1521] = 0x6f1a, [0x1522] = 0x6f1b, [0x1523] = 0x6f1c,
  [0x1524] = 0x6f1d, [0x1525] = 0x6f1e, [0x1526] = 0x6f1f, [0x1527] = 0x6f21,
  [0x1528] = 0x6f22, [0x1529] = 0x6f23, [0x152a] = 0x6f25, [0x152b] = 0x6f26,
  [0x152c] = 0x6f27, [0x152d] = 0x6f28, [0x152e] = 0x6f2c, [0x152f] = 0x6f2e,
  [0x1530] = 0x6f30, [0x1531] = 0x6f32, [0x1532] = 0x6f34, [0x1533] = 0x6f35,
  [0x1534] = 0x6f37, [0x1535] = 0x6f38, [0x1536] = 0x6f39, [0x1537] = 0x6f3a,
  [0x1538] = 0x6f3b, [0x1539] = 0x6f3c, [0x153a] = 0x6f3d, [0x153b] = 0x6f3f,
  [0x153c] = 0x6f40, [0x153d] = 0x6f41, [0x153e] = 0x6f42, [0x1540] = 0x6f43,
  [0x1541] = 0x6f44, [0x1542] = 0x6f45, [0x1543] = 0x6f48, [0x1544] = 0x6f49,
  [0x1545] = 0x6f4a, [0x1546] = 0x6f4c, [0x1547] = 0x6f4e, [0x1548] = 0x6f4f,
  [0x1549] = 0x6f50, [0x154a] = 0x6f51, [0x154b] = 0x6f52, [0x154c] = 0x6f53,
  [0x154d] = 0x6f54, [0x154e] = 0x6f55, [0x154f] = 0x6f56, [0x1550] = 0x6f57,
  [0x1551] = 0x6f59, [0x1552] = 0x6f5a, [0x1553] = 0x6f5b, [0x1554] = 0x6f5d,
  [0x1555] = 0x6f5f, [0x1556] = 0x6f60, [0x1557] = 0x6f61, [0x1558] = 0x6f63,
  [0x1559] = 0x6f64, [0x155a] = 0x6f65, [0x155b] = 0x6f67, [0x155c] = 0x6f68,
  [0x155d] = 0x6f69, [0x155e] = 0x6f6a, [0x155f] = 0x6f6b, [0x1560] = 0x6f6c,
  [0x1561] = 0x6f6f, [0x1562] = 0x6f70, [0x1563] = 0x6f71, [0x1564] = 0x6f73,
  [0x1565] = 0x6f75, [0x1566] = 0x6f76, [0x1567] = 0x6f77, [0x1568] = 0x6f79,
  [0x1569] = 0x6f7b, [0x156a] = 0x6f7d, [0x156b] = 0x6f7e, [0x156c] = 0x6f7f,
  [0x156d] = 0x6f80, [0x156e] = 0x6f81, [0x156f] = 0x6f82, [0x1570] = 0x6f83,
  [0x1571] = 0x6f85, [0x1572] = 0x6f86, [0x1573] = 0x6f87, [0x1574] = 0x6f8a,
  [0x1575] = 0x6f8b, [0x1576] = 0x6f8f, [0x1577] = 0x6f90, [0x1578] = 0x6f91,
  [0x1579] = 0x6f92, [0x157a] = 0x6f93, [0x157b] = 0x6f94, [0x157c] = 0x6f95,
  [0x157d] = 0x6f96, [0x157e] = 0x6f97, [0x157f] = 0x6f98, [0x1580] = 0x6f99,
  [0x1581] = 0x6f9a, [0x1582] = 0x6f9b, [0x1583] = 0x6f9d, [0x1584] = 0x6f9e,
  [0x1585] = 0x6f9f, [0x1586] = 0x6fa0, [0x1587] = 0x6fa2, [0x1588] = 0x6fa3,
  [0x1589] = 0x6fa4, [0x158a] = 0x6fa5, [0x158b] = 0x6fa6, [0x158c] = 0x6fa8,
  [0x158d] = 0x6fa9, [0x158e] = 0x6faa, [0x158f] = 0x6fab, [0x1590] = 0x6fac,
  [0x1591] = 0x6fad, [0x1592] = 0x6fae, [0x1593] = 0x6faf, [0x1594] = 0x6fb0,
  [0x1595] = 0x6fb1, [0x1596] = 0x6fb2, [0x1597] = 0x6fb4, [0x1598] = 0x6fb5,
  [0x1599] = 0x6fb7, [0x159a] = 0x6fb8, [0x159b] = 0x6fba, [0x159c] = 0x6fbb,
  [0x159d] = 0x6fbc, [0x159e] = 0x6fbd, [0x159f] = 0x6fbe, [0x15a0] = 0x6fbf,
  [0x15a1] = 0x6fc1, [0x15a2] = 0x6fc3, [0x15a3] = 0x6fc4, [0x15a4] = 0x6fc5,
  [0x15a5] = 0x6fc6, [0x15a6] = 0x6fc7, [0x15a7] = 0x6fc8, [0x15a8] = 0x6fca,
  [0x15a9] = 0x6fcb, [0x15aa] = 0x6fcc, [0x15ab] = 0x6fcd, [0x15ac] = 0x6fce,
  [0x15ad] = 0x6fcf, [0x15ae] = 0x6fd0, [0x15af] = 0x6fd3, [0x15b0] = 0x6fd4,
  [0x15b1] = 0x6fd5, [0x15b2] = 0x6fd6, [0x15b3] = 0x6fd7, [0x15b4] = 0x6fd8,
  [0x15b5] = 0x6fd9, [0x15b6] = 0x6fda, [0x15b7] = 0x6fdb, [0x15b8] = 0x6fdc,
  [0x15b9] = 0x6fdd, [0x15ba] = 0x6fdf, [0x15bb] = 0x6fe2, [0x15bc] = 0x6fe3,
  [0x15bd] = 0x6fe4, [0x15be] = 0x6fe5, [0x15c0] = 0x6fe6, [0x15c1] = 0x6fe7,
  [0x15c2] = 0x6fe8, [0x15c3] = 0x6fe9, [0x15c4] = 0x6fea, [0x15c5] = 0x6feb,
  [0x15c6] = 0x6fec, [0x15c7] = 0x6fed, [0x15c8] = 0x6ff0, [0x15c9] = 0x6ff1,
  [0x15ca] = 0x6ff2, [0x15cb] = 0x6ff3, [0x15cc] = 0x6ff4, [0x15cd] = 0x6ff5,
  [0x15ce] = 0x6ff6, [0x15cf] = 0x6ff7, [0x15d0] = 0x6ff8, [0x15d1] = 0x6ff9,
  [0x15d2] = 0x6ffa, [0x15d3] = 0x6ffb, [0x15d4] = 0x6ffc, [0x15d5] = 0x6ffd,
  [0x15d6] = 0x6ffe, [0x15d7] = 0x6fff, [0x15d8] = 0x7000, [0x15d9] = 0x7001,
  [0x15da] = 0x7002, [0x15db] = 0x7003, [0x15dc] = 0x7004, [0x15dd] = 0x7005,
  [0x15de] = 0x7006, [0x15df] = 0x7007, [0x15e0] = 0x7008, [0x15e1] = 0x7009,
  [0x15e2] = 0x700a, [0x15e3] = 0x700b, [0x15e4] = 0x700c, [0x15e5] = 0x700d,
  [0x15e6] = 0x700e, [0x15e7] = 0x700f, [0x15e8] = 0x7010, [0x15e9] = 0x7012,
  [0x15ea] = 0x7013, [0x15eb] = 0x7014, [0x15ec] = 0x7015, [0x15ed] = 0x7016,
  [0x15ee] = 0x7017, [0x15ef] = 0x7018, [0x15f0] = 0x7019, [0x15f1] = 0x701c,
  [0x15f2] = 0x701d, [0x15f3] = 0x701e, [0x15f4] = 0x701f, [0x15f5] = 0x7020,
  [0x15f6] = 0x7021, [0x15f7] = 0x7022, [0x15f8] = 0x7024, [0x15f9] = 0x7025,
  [0x15fa] = 0x7026, [0x15fb] = 0x7027, [0x15fc] = 0x7028, [0x15fd] = 0x7029,
  [0x15fe] = 0x702a, [0x1600] = 0x702b, [0x1601] = 0x702c, [0x1602] = 0x702d,
  [0x1603] = 0x702e, [0x1604] = 0x702f, [0x1605] = 0x7030, [0x1606] = 0x7031,
  [0x1607] = 0x7032, [0x1608] = 0x7033, [0x1609] = 0x7034, [0x160a] = 0x7036,
  [0x160b] = 0x7037, [0x160c] = 0x7038, [0x160d] = 0x703a, [0x160e] = 0x703b,
  [0x160f] = 0x703c, [0x1610] = 0x703d, [0x1611] = 0x703e, [0x1612] = 0x703f,
  [0x1613] = 0x7040, [0x1614] = 0x7041, [0x1615] = 0x7042, [0x1616] = 0x7043,
  [0x1617] = 0x7044, [0x1618] = 0x7045, [0x1619] = 0x7046, [0x161a] = 0x7047,
  [0x161b] = 0x7048, [0x161c] = 0x7049, [0x161d] = 0x704a, [0x161e] = 0x704b,
  [0x161f] = 0x704d, [0x1620] = 0x704e, [0x1621] = 0x7050, [0x1622] = 0x7051,
  [0x1623] = 0x7052, [0x1624] = 0x7053, [0x1625] = 0x7054, [0x1626] = 0x7055,
  [0x1627] = 0x7056, [0x1628] = 0x7057, [0x1629] = 0x7058, [0x162a] = 0x7059,
  [0x162b] = 0x705a, [0x162c] = 0x705b, [0x162d] = 0x705c, [0x162e] = 0x705d,
  [0x162f] = 0x705f, [0x1630] = 0x7060, [0x1631] = 0x7061, [0x1632] = 0x7062,
  [0x1633] = 0x7063, [0x1634] = 0x7064, [0x1635] = 0x7065, [0x1636] = 0x7066,
  [0x1637] = 0x7067, [0x1638] = 0x7068, [0x1639] = 0x7069, [0x163a] = 0x706a,
  [0x163b] = 0x706e, [0x163c] = 0x7071, [0x163d] = 0x7072, [0x163e] = 0x7073,
  [0x163f] = 0x7074, [0x1640] = 0x7077, [0x1641] = 0x7079, [0x1642] = 0x707a,
  [0x1643] = 0x707b, [0x1644] = 0x707d, [0x1645] = 0x7081, [0x1646] = 0x7082,
  [0x1647] = 0x7083, [0x1648] = 0x7084, [0x1649] = 0x7086, [0x164a] = 0x7087,
  [0x164b] = 0x7088, [0x164c] = 0x708b, [0x164d] = 0x708c, [0x164e] = 0x708d,
  [0x164f] = 0x708f, [0x1650] = 0x7090, [0x1651] = 0x7091, [0x1652] = 0x7093,
  [0x1653] = 0x7097, [0x1654] = 0x7098, [0x1655] = 0x709a, [0x1656] = 0x709b,
  [0x1657] = 0x709e, [0x1658] = 0x709f, [0x1659] = 0x70a0, [0x165a] = 0x70a1,
  [0x165b] = 0x70a2, [0x165c] = 0x70a3, [0x165d] = 0x70a4, [0x165e] = 0x70a5,
  [0x165f] = 0x70a6, [0x1660] = 0x70a7, [0x1661] = 0x70a8, [0x1662] = 0x70a9,
  [0x1663] = 0x70aa, [0x1664] = 0x70b0, [0x1665] = 0x70b2, [0x1666] = 0x70b4,
  [0x1667] = 0x70b5, [0x1668] = 0x70b6, [0x1669] = 0x70ba, [0x166a] = 0x70be,
  [0x166b] = 0x70bf, [0x166c] = 0x70c4, [0x166d] = 0x70c5, [0x166e] = 0x70c6,
  [0x166f] = 0x70c7, [0x1670] = 0x70c9, [0x1671] = 0x70cb, [0x1672] = 0x70cc,
  [0x1673] = 0x70cd, [0x1674] = 0x70ce, [0x1675] = 0x70cf, [0x1676] = 0x70d0,
  [0x1677] = 0x70d1, [0x1678] = 0x70d2, [0x1679] = 0x70d3, [0x167a] = 0x70d4,
  [0x167b] = 0x70d5, [0x167c] = 0x70d6, [0x167d] = 0x70d7, [0x167e] = 0x70da,
  [0x1680] = 0x70dc, [0x1681] = 0x70dd, [0x1682] = 0x70de, [0x1683] = 0x70e0,
  [0x1684] = 0x70e1, [0x1685] = 0x70e2, [0x1686] = 0x70e3, [0x1687] = 0x70e5,
  [0x1688] = 0x70ea, [0x1689] = 0x70ee, [0x168a] = 0x70f0, [0x168b] = 0x70f1,
  [0x168c] = 0x70f2, [0x168d] = 0x70f3, [0x168e] = 0x70f4, [0x168f] = 0x70f5,
  [0x1690] = 0x70f6, [0x1691] = 0x70f8, [0x1692] = 0x70fa, [0x1693] = 0x70fb,
  [0x1694] = 0x70fc, [0x1695] = 0x70fe, [0x1696] = 0x70ff, [0x1697] = 0x7100,
  [0x1698] = 0x7101, [0x1699] = 0x7102, [0x169a] = 0x7103, [0x169b] = 0x7104,
  [0x169c] = 0x7105, [0x169d] = 0x7106, [0x169e] = 0x7107, [0x169f] = 0x7108,
  [0x16a0] = 0x710b, [0x16a1] = 0x710c, [0x16a2] = 0x710d, [0x16a3] = 0x710e,
  [0x16a4] = 0x710f, [0x16a5] = 0x7111, [0x16a6] = 0x7112, [0x16a7] = 0x7114,
  [0x16a8] = 0x7117, [0x16a9] = 0x711b, [0x16aa] = 0x711c, [0x16ab] = 0x711d,
  [0x16ac] = 0x711e, [0x16ad] = 0x711f, [0x16ae] = 0x7120, [0x16af] = 0x7121,
  [0x16b0] = 0x7122, [0x16b1] = 0x7123, [0x16b2] = 0x7124, [0x16b3] = 0x7125,
  [0x16b4] = 0x7127, [0x16b5] = 0x7128, [0x16b6] = 0x7129, [0x16b7] = 0x712a,
  [0x16b8] = 0x712b, [0x16b9] = 0x712c, [0x16ba] = 0x712d, [0x16bb] = 0x712e,
  [0x16bc] = 0x7132, [0x16bd] = 0x7133, [0x16be] = 0x7134, [0x16c0] = 0x7135,
  [0x16c1] = 0x7137, [0x16c2] = 0x7138, [0x16c3] = 0x7139, [0x16c4] = 0x713a,
  [0x16c5] = 0x713b, [0x16c6] = 0x713c, [0x16c7] = 0x713d, [0x16c8] = 0x713e,
  [0x16c9] = 0x713f, [0x16ca] = 0x7140, [0x16cb] = 0x7141, [0x16cc] = 0x7142,
  [0x16cd] = 0x7143, [0x16ce] = 0x7144, [0x16cf] = 0x7146, [0x16d0] = 0x7147,
  [0x16d1] = 0x7148, [0x16d2] = 0x7149, [0x16d3] = 0x714b, [0x16d4] = 0x714d,
  [0x16d5] = 0x714f, [0x16d6] = 0x7150, [0x16d7] = 0x7151, [0x16d8] = 0x7152,
  [0x16d9] = 0x7153, [0x16da] = 0x7154, [0x16db] = 0x7155, [0x16dc] = 0x7156,
  [0x16dd] = 0x7157, [0x16de] = 0x7158, [0x16df] = 0x7159, [0x16e0] = 0x715a,
  [0x16e1] = 0x715b, [0x16e2] = 0x715d, [0x16e3] = 0x715f, [0x16e4] = 0x7160,
  [0x16e5] = 0x7161, [0x16e6] = 0x7162, [0x16e7] = 0x7163, [0x16e8] = 0x7165,
  [0x16e9] = 0x7169, [0x16ea] = 0x716a, [0x16eb] = 0x716b, [0x16ec] = 0x716c,
  [0x16ed] = 0x716d, [0x16ee] = 0x716f, [0x16ef] = 0x7170, [0x16f0] = 0x7171,
  [0x16f1] = 0x7174, [0x16f2] = 0x7175, [0x16f3] = 0x7176, [0x16f4] = 0x7177,
  [0x16f5] = 0x7179, [0x16f6] = 0x717b, [0x16f7] = 0x717c, [0x16f8] = 0x717e,
  [0x16f9] = 0x717f, [0x16fa] = 0x7180, [0x16fb] = 0x7181, [0x16fc] = 0x7182,
  [0x16fd] = 0x7183, [0x16fe] = 0x7185, [0x16ff] = 0x7186, [0x1700] = 0x7187,
  [0x1701] = 0x7188, [0x1702] = 0x7189, [0x1703] = 0x718b, [0x1704] = 0x718c,
  [0x1705] = 0x718d, [0x1706] = 0x718e, [0x1707] = 0x7190, [0x1708] = 0x7191,
  [0x1709] = 0x7192, [0x170a] = 0x7193, [0x170b] = 0x7195, [0x170c] = 0x7196,
  [0x170d] = 0x7197, [0x170e] = 0x719a, [0x170f] = 0x719b, [0x1710] = 0x719c,
  [0x1711] = 0x719d, [0x1712] = 0x719e, [0x1713] = 0x71a1, [0x1714] = 0x71a2,
  [0x1715] = 0x71a3, [0x1716] = 0x71a4, [0x1717] = 0x71a5, [0x1718] = 0x71a6,
  [0x1719] = 0x71a7, [0x171a] = 0x71a9, [0x171b] = 0x71aa, [0x171c] = 0x71ab,
  [0x171d] = 0x71ad, [0x171e] = 0x71ae, [0x171f] = 0x71af, [0x1720] = 0x71b0,
  [0x1721] = 0x71b1, [0x1722] = 0x71b2, [0x1723] = 0x71b4, [0x1724] = 0x71b6,
  [0x1725] = 0x71b7, [0x1726] = 0x71b8, [0x1727] = 0x71ba, [0x1728] = 0x71bb,
  [0x1729] = 0x71bc, [0x172a] = 0x71bd, [0x172b] = 0x71be, [0x172c] = 0x71bf,
  [0x172d] = 0x71c0, [0x172e] = 0x71c1, [0x172f] = 0x71c2, [0x1730] = 0x71c4,
  [0x1731] = 0x71c5, [0x1732] = 0x71c6, [0x1733] = 0x71c7, [0x1734] = 0x71c8,
  [0x1735] = 0x71c9, [0x1736] = 0x71ca, [0x1737] = 0x71cb, [0x1738] = 0x71cc,
  [0x1739] = 0x71cd, [0x173a] = 0x71cf, [0x173b] = 0x71d0, [0x173c] = 0x71d1,
  [0x173d] = 0x71d2, [0x173e] = 0x71d3, [0x1740] = 0x71d6, [0x1741] = 0x71d7,
  [0x1742] = 0x71d8, [0x1743] = 0x71d9, [0x1744] = 0x71da, [0x1745] = 0x71db,
  [0x1746] = 0x71dc, [0x1747] = 0x71dd, [0x1748] = 0x71de, [0x1749] = 0x71df,
  [0x174a] = 0x71e1, [0x174b] = 0x71e2, [0x174c] = 0x71e3, [0x174d] = 0x71e4,
  [0x174e] = 0x71e6, [0x174f] = 0x71e8, [0x1750] = 0x71e9, [0x1751] = 0x71ea,
  [0x1752] = 0x71eb, [0x1753] = 0x71ec, [0x1754] = 0x71ed, [0x1755] = 0x71ef,
  [0x1756] = 0x71f0, [0x1757] = 0x71f1, [0x1758] = 0x71f2, [0x1759] = 0x71f3,
  [0x175a] = 0x71f4, [0x175b] = 0x71f5, [0x175c] = 0x71f6, [0x175d] = 0x71f7,
  [0x175e] = 0x71f8, [0x175f] = 0x71fa, [0x1760] = 0x71fb, [0x1761] = 0x71fc,
  [0x1762] = 0x71fd, [0x1763] = 0x71fe, [0x1764] = 0x71ff, [0x1765] = 0x7200,
  [0x1766] = 0x7201, [0x1767] = 0x7202, [0x1768] = 0x7203, [0x1769] = 0x7204,
  [0x176a] = 0x7205, [0x176b] = 0x7207, [0x176c] = 0x7208, [0x176d] = 0x7209,
  [0x176e] = 0x720a, [0x176f] = 0x720b, [0x1770] = 0x720c, [0x1771] = 0x720d,
  [0x1772] = 0x720e, [0x1773] = 0x720f, [0x1774] = 0x7210, [0x1775] = 0x7211,
  [0x1776] = 0x7212, [0x1777] = 0x7213, [0x1778] = 0x7214, [0x1779] = 0x7215,
  [0x177a] = 0x7216, [0x177b] = 0x7217, [0x177c] = 0x7218, [0x177d] = 0x7219,
  [0x177e] = 0x721a, [0x1780] = 0x721b, [0x1781] = 0x721c, [0x1782] = 0x721e,
  [0x1783] = 0x721f, [0x1784] = 0x7220, [0x1785] = 0x7221, [0x1786] = 0x7222,
  [0x1787] = 0x7223, [0x1788] = 0x7224, [0x1789] = 0x7225, [0x178a] = 0x7226,
  [0x178b] = 0x7227, [0x178c] = 0x7229, [0x178d] = 0x722b, [0x178e] = 0x722d,
  [0x178f] = 0x722e, [0x1790] = 0x722f, [0x1791] = 0x7232, [0x1792] = 0x7233,
  [0x1793] = 0x7234, [0x1794] = 0x723a, [0x1795] = 0x723c, [0x1796] = 0x723e,
  [0x1797] = 0x7240, [0x1798] = 0x7241, [0x1799] = 0x7242, [0x179a] = 0x7243,
  [0x179b] = 0x7244, [0x179c] = 0x7245, [0x179d] = 0x7246, [0x179e] = 0x7249,
  [0x179f] = 0x724a, [0x17a0] = 0x724b, [0x17a1] = 0x724e, [0x17a2] = 0x724f,
  [0x17a3] = 0x7250, [0x17a4] = 0x7251, [0x17a5] = 0x7253, [0x17a6] = 0x7254,
  [0x17a7] = 0x7255, [0x17a8] = 0x7257, [0x17a9] = 0x7258, [0x17aa] = 0x725a,
  [0x17ab] = 0x725c, [0x17ac] = 0x725e, [0x17ad] = 0x7260, [0x17ae] = 0x7263,
  [0x17af] = 0x7264, [0x17b0] = 0x7265, [0x17b1] = 0x7268, [0x17b2] = 0x726a,
  [0x17b3] = 0x726b, [0x17b4] = 0x726c, [0x17b5] = 0x726d, [0x17b6] = 0x7270,
  [0x17b7] = 0x7271, [0x17b8] = 0x7273, [0x17b9] = 0x7274, [0x17ba] = 0x7276,
  [0x17bb] = 0x7277, [0x17bc] = 0x7278, [0x17bd] = 0x727b, [0x17be] = 0x727c,
  [0x17bf] = 0x727d, [0x17c0] = 0x7282, [0x17c1] = 0x7283, [0x17c2] = 0x7285,
  [0x17c3] = 0x7286, [0x17c4] = 0x7287, [0x17c5] = 0x7288, [0x17c6] = 0x7289,
  [0x17c7] = 0x728c, [0x17c8] = 0x728e, [0x17c9] = 0x7290, [0x17ca] = 0x7291,
  [0x17cb] = 0x7293, [0x17cc] = 0x7294, [0x17cd] = 0x7295, [0x17ce] = 0x7296,
  [0x17cf] = 0x7297, [0x17d0] = 0x7298, [0x17d1] = 0x7299, [0x17d2] = 0x729a,
  [0x17d3] = 0x729b, [0x17d4] = 0x729c, [0x17d5] = 0x729d, [0x17d6] = 0x729e,
  [0x17d7] = 0x72a0, [0x17d8] = 0x72a1, [0x17d9] = 0x72a2, [0x17da] = 0x72a3,
  [0x17db] = 0x72a4, [0x17dc] = 0x72a5, [0x17dd] = 0x72a6, [0x17de] = 0x72a7,
  [0x17df] = 0x72a8, [0x17e0] = 0x72a9, [0x17e1] = 0x72aa, [0x17e2] = 0x72ab,
  [0x17e3] = 0x72ae, [0x17e4] = 0x72b1, [0x17e5] = 0x72b2, [0x17e6] = 0x72b3,
  [0x17e7] = 0x72b5, [0x17e8] = 0x72ba, [0x17e9] = 0x72bb, [0x17ea] = 0x72bc,
  [0x17eb] = 0x72bd, [0x17ec] = 0x72be, [0x17ed] = 0x72bf, [0x17ee] = 0x72c0,
  [0x17ef] = 0x72c5, [0x17f0] = 0x72c6, [0x17f1] = 0x72c7, [0x17f2] = 0x72c9,
  [0x17f3] = 0x72ca, [0x17f4] = 0x72cb, [0x17f5] = 0x72cc, [0x17f6] = 0x72cf,
  [0x17f7] = 0x72d1, [0x17f8] = 0x72d3, [0x17f9] = 0x72d4, [0x17fa] = 0x72d5,
  [0x17fb] = 0x72d6, [0x17fc] = 0x72d8, [0x17fd] = 0x72da, [0x17fe] = 0x72db,
  [0x1800] = 0xe4c6, [0x1801] = 0xe4c7, [0x1802] = 0xe4c8, [0x1803] = 0xe4c9,
  [0x1804] = 0xe4ca, [0x1805] = 0xe4cb, [0x1806] = 0xe4cc, [0x1807] = 0xe4cd,
  [0x1808] = 0xe4ce, [0x1809] = 0xe4cf, [0x180a] = 0xe4d0, [0x180b] = 0xe4d1,
  [0x180c] = 0xe4d2, [0x180d] = 0xe4d3, [0x180e] = 0xe4d4, [0x180f] = 0xe4d5,
  [0x1810] = 0xe4d6, [0x1811] = 0xe4d7, [0x1812] = 0xe4d8, [0x1813] = 0xe4d9,
  [0x1814] = 0xe4da, [0x1815] = 0xe4db, [0x1816] = 0xe4dc, [0x1817] = 0xe4dd,
  [0x1818] = 0xe4de, [0x1819] = 0xe4df, [0x181a] = 0xe4e0, [0x181b] = 0xe4e1,
  [0x181c] = 0xe4e2, [0x181d] = 0xe4e3, [0x181e] = 0xe4e4, [0x181f] = 0xe4e5,
  [0x1820] = 0xe4e6, [0x1821] = 0xe4e7, [0x1822] = 0xe4e8, [0x1823] = 0xe4e9,
  [0x1824] = 0xe4ea, [0x1825] = 0xe4eb, [0x1826] = 0xe4ec, [0x1827] = 0xe4ed,
  [0x1828] = 0xe4ee, [0x1829] = 0xe4ef, [0x182a] = 0xe4f0, [0x182b] = 0xe4f1,
  [0x182c] = 0xe4f2, [0x182d] = 0xe4f3, [0x182e] = 0xe4f4, [0x182f] = 0xe4f5,
  [0x1830] = 0xe4f6, [0x1831] = 0xe4f7, [0x1832] = 0xe4f8, [0x1833] = 0xe4f9,
  [0x1834] = 0xe4fa, [0x1835] = 0xe4fb, [0x1836] = 0xe4fc, [0x1837] = 0xe4fd,
  [0x1838] = 0xe4fe, [0x1839] = 0xe4ff, [0x183a] = 0xe500, [0x183b] = 0xe501,
  [0x183c] = 0xe502, [0x183d] = 0xe503, [0x183e] = 0xe504, [0x1840] = 0xe505,
  [0x1841] = 0xe506, [0x1842] = 0xe507, [0x1843] = 0xe508, [0x1844] = 0xe509,
  [0x1845] = 0xe50a, [0x1846] = 0xe50b, [0x1847] = 0xe50c, [0x1848] = 0xe50d,
  [0x1849] = 0xe50e, [0x184a] = 0xe50f, [0x184b] = 0xe510, [0x184c] = 0xe511,
  [0x184d] = 0xe512, [0x184e] = 0xe513, [0x184f] = 0xe514, [0x1850] = 0xe515,
  [0x1851] = 0xe516, [0x1852] = 0xe517, [0x1853] = 0xe518, [0x1854] = 0xe519,
  [0x1855] = 0xe51a, [0x1856] = 0xe51b, [0x1857] = 0xe51c, [0x1858] = 0xe51d,
  [0x1859] = 0xe51e, [0x185a] = 0xe51f, [0x185b] = 0xe520, [0x185c] = 0xe521,
  [0x185d] = 0xe522, [0x185e] = 0xe523, [0x185f] = 0xe524, [0x1860] = 0xe525,
  [0x1861] = 0x3000, [0x1862] = 0x3001, [0x1863] = 0x3002, [0x1864] = 0x00b7,
  [0x1865] = 0x02c9, [0x1866] = 0x02c7, [0x1867] = 0x00a8, [0x1868] = 0x3003,
  [0x1869] = 0x3005, [0x186a] = 0x2014, [0x186b] = 0xff5e, [0x186c] = 0x2016,
  [0x186d] = 0x2026, [0x186e] = 0x2018, [0x186f] = 0x2019, [0x1870] = 0x201c,
  [0x1871] = 0x201d, [0x1872] = 0x3014, [0x1873] = 0x3015, [0x1874] = 0x3008,
  [0x1875] = 0x3009, [0x1876] = 0x300a, [0x1877] = 0x300b, [0x1878] = 0x300c,
  [0x1879] = 0x300d, [0x187a] = 0x300e, [0x187b] = 0x300f, [0x187c] = 0x3016,
  [0x187d] = 0x3017, [0x187e] = 0x3010, [0x187f] = 0x3011, [0x1880] = 0x00b1,
  [0x1881] = 0x00d7, [0x1882] = 0x00f7, [0x1883] = 0x2236, [0x1884] = 0x2227,
  [0x1885] = 0x2228, [0x1886] = 0x2211, [0x1887] = 0x220f, [0x1888] = 0x222a,
  [0x1889] = 0x2229, [0x188a] = 0x2208, [0x188b] = 0x2237, [0x188c] = 0x221a,
  [0x188d] = 0x22a5, [0x188e] = 0x2225, [0x188f] = 0x2220, [0x1890] = 0x2312,
  [0x1891] = 0x2299, [0x1892] = 0x222b, [0x1893] = 0x222e, [0x1894] = 0x2261,
  [0x1895] = 0x224c, [0x1896] = 0x2248, [0x1897] = 0x223d, [0x1898] = 0x221d,
  [0x1899] = 0x2260, [0x189a] = 0x226e, [0x189b] = 0x226f, [0x189c] = 0x2264,
  [0x189d] = 0x2265, [0x189e] = 0x221e, [0x189f] = 0x2235, [0x18a0] = 0x2234,
  [0x18a1] = 0x2642, [0x18a2] = 0x2640, [0x18a3] = 0x00b0, [0x18a4] = 0x2032,
  [0x18a5] = 0x2033, [0x18a6] = 0x2103, [0x18a7] = 0xff04, [0x18a8] = 0x00a4,
  [0x18a9] = 0xffe0, [0x18aa] = 0xffe1, [0x18ab] = 0x2030, [0x18ac] = 0x00a7,
  [0x18ad] = 0x2116, [0x18ae] = 0x2606, [0x18af] = 0x2605, [0x18b0] = 0x25cb,
  [0x18b1] = 0x25cf, [0x18b2] = 0x25ce, [0x18b3] = 0x25c7, [0x18b4] = 0x25c6,
  [0x18b5] = 0x25a1, [0x18b6] = 0x25a0, [0x18b7] = 0x25b3, [0x18b8] = 0x25b2,
  [0x18b9] = 0x203b, [0x18ba] = 0x2192, [0x18bb] = 0x2190, [0x18bc] = 0x2191,
  [0x18bd] = 0x2193, [0x18be] = 0x3013, [0x18c0] = 0xe526, [0x18c1] = 0xe527,
  [0x18c2] = 0xe528, [0x18c3] = 0xe529, [0x18c4] = 0xe52a, [0x18c5] = 0xe52b,
  [0x18c6] = 0xe52c, [0x18c7] = 0xe52d, [0x18c8] = 0xe52e, [0x18c9] = 0xe52f,
  [0x18ca] = 0xe530, [0x18cb] = 0xe531, [0x18cc] = 0xe532, [0x18cd] = 0xe533,
  [0x18ce] = 0xe534, [0x18cf] = 0xe535, [0x18d0] = 0xe536, [0x18d1] = 0xe537,
  [0x18d2] = 0xe538, [0x18d3] = 0xe539, [0x18d4] = 0xe53a, [0x18d5] = 0xe53b,
  [0x18d6] = 0xe53c, [0x18d7] = 0xe53d, [0x18d8] = 0xe53e, [0x18d9] = 0xe53f,
  [0x18da] = 0xe540, [0x18db] = 0xe541, [0x18dc] = 0xe542, [0x18dd] = 0xe543,
  [0x18de] = 0xe544, [0x18df] = 0xe545, [0x18e0] = 0xe546, [0x18e1] = 0xe547,
  [0x18e2] = 0xe548, [0x18e3] = 0xe549, [0x18e4] = 0xe54a, [0x18e5] = 0xe54b,
  [0x18e6] = 0xe54c, [0x18e7] = 0xe54d, [0x18e8] = 0xe54e, [0x18e9] = 0xe54f,
  [0x18ea] = 0xe550, [0x18eb] = 0xe551, [0x18ec] = 0xe552, [0x18ed] = 0xe553,
  [0x18ee] = 0xe554, [0x18ef] = 0xe555, [0x18f0] = 0xe556, [0x18f1] = 0xe557,
  [0x18f2] = 0xe558, [0x18f3] = 0xe559, [0x18f4] = 0xe55a, [0x18f5] = 0xe55b,
  [0x18f6] = 0xe55c, [0x18f7] = 0xe55d, [0x18f8] = 0xe55e, [0x18f9] = 0xe55f,
  [0x18fa] = 0xe560, [0x18fb] = 0xe561, [0x18fc] = 0xe562, [0x18fd] = 0xe563,
  [0x18fe] = 0xe564, [0x1900] = 0xe565, [0x1901] = 0xe566, [0x1902] = 0xe567,
  [0x1903] = 0xe568, [0x1904] = 0xe569, [0x1905] = 0xe56a, [0x1906] = 0xe56b,
  [0x1907] = 0xe56c, [0x1908] = 0xe56d, [0x1909] = 0xe56e, [0x190a] = 0xe56f,
  [0x190b] = 0xe570, [0x190c] = 0xe571, [0x190d] = 0xe572, [0x190e] = 0xe573,
  [0x190f] = 0xe574, [0x1910] = 0xe575, [0x1911] = 0xe576, [0x1912] = 0xe577,
  [0x1913] = 0xe578, [0x1914] = 0xe579, [0x1915] = 0xe57a, [0x1916] = 0xe57b,
  [0x1917] = 0xe57c, [0x1918] = 0xe57d, [0x1919] = 0xe57e, [0x191a] = 0xe57f,
  [0x191b] = 0xe580, [0x191c] = 0xe581, [0x191d] = 0xe582, [0x191e] = 0xe583,
  [0x191f] = 0xe584, [0x1920] = 0xe585, [0x1921] = 0x2170, [0x1922] = 0x2171,
  [0x1923] = 0x2172, [0x1924] = 0x2173, [0x1925] = 0x2174, [0x1926] = 0x2175,
  [0x1927] = 0x2176, [0x1928] = 0x2177, [0x1929] = 0x2178, [0x192a] = 0x2179,
  [0x192b] = 0xe766, [0x192c] = 0xe767, [0x192d] = 0xe768, [0x192e] = 0xe769,
  [0x192f] = 0xe76a, [0x1930] = 0xe76b, [0x1931] = 0x2488, [0x1932] = 0x2489,
  [0x1933] = 0x248a, [0x1934] = 0x248b, [0x1935] = 0x248c, [0x1936] = 0x248d,
  [0x1937] = 0x248e, [0x1938] = 0x248f, [0x1939] = 0x2490, [0x193a] = 0x2491,
  [0x193b] = 0x2492, [0x193c] = 0x2493, [0x193d] = 0x2494, [0x193e] = 0x2495,
  [0x193f] = 0x2496, [0x1940] = 0x2497, [0x1941] = 0x2498, [0x1942] = 0x2499,
  [0x1943] = 0x249a, [0x1944] = 0x249b, [0x1945] = 0x2474, [0x1946] = 0x2475,
  [0x1947] = 0x2476, [0x1948] = 0x2477, [0x1949] = 0x2478, [0x194a] = 0x2479,
  [0x194b] = 0x247a, [0x194c] = 0x247b, [0x194d] = 0x247c, [0x194e] = 0x247d,
  [0x194f] = 0x247e, [0x1950] = 0x247f, [0x1951] = 0x2480, [0x1952] = 0x2481,
  [0x1953] = 0x2482, [0x1954] = 0x2483, [0x1955] = 0x2484, [0x1956] = 0x2485,
  [0x1957] = 0x2486, [0x1958] = 0x2487, [0x1959] = 0x2460, [0x195a] = 0x2461,
  [0x195b] = 0x2462, [0x195c] = 0x2463, [0x195d] = 0x2464, [0x195e] = 0x2465,
  [0x195f] = 0x2466, [0x1960] = 0x2467, [0x1961] = 0x2468, [0x1962] = 0x2469,
  [0x1963] = 0x20ac, [0x1964] = 0xe76d, [0x1965] = 0x3220, [0x1966] = 0x3221,
  [0x1967] = 0x3222, [0x1968] = 0x3223, [0x1969] = 0x3224, [0x196a] = 0x3225,
  [0x196b] = 0x3226, [0x196c] = 0x3227, [0x196d] = 0x3228, [0x196e] = 0x3229,
  [0x196f] = 0xe76e, [0x1970] = 0xe76f, [0x1971] = 0x2160, [0x1972] = 0x2161,
  [0x1973] = 0x2162, [0x1974] = 0x2163, [0x1975] = 0x2164, [0x1976] = 0x2165,
  [0x1977] = 0x2166, [0x1978] = 0x2167, [0x1979] = 0x2168, [0x197a] = 0x2169,
  [0x197b] = 0x216a, [0x197c] = 0x216b, [0x197d] = 0xe770, [0x197e] = 0xe771,
  [0x1980] = 0xe586, [0x1981] = 0xe587, [0x1982] = 0xe588, [0x1983] = 0xe589,
  [0x1984] = 0xe58a, [0x1985] = 0xe58b, [0x1986] = 0xe58c, [0x1987] = 0xe58d,
  [0x1988] = 0xe58e, [0x1989] = 0xe58f, [0x198a] = 0xe590, [0x198b] = 0xe591,
  [0x198c] = 0xe592, [0x198d] = 0xe593, [0x198e] = 0xe594, [0x198f] = 0xe595,
  [0x1990] = 0xe596, [0x1991] = 0xe597, [0x1992] = 0xe598, [0x1993] = 0xe599,
  [0x1994] = 0xe59a, [0x1995] = 0xe59b, [0x1996] = 0xe59c, [0x1997] = 0xe59d,
  [0x1998] = 0xe59e, [0x1999] = 0xe59f, [0x199a] = 0xe5a0, [0x199b] = 0xe5a1,
  [0x199c] = 0xe5a2, [0x199d] = 0xe5a3, [0x199e] = 0xe5a4, [0x199f] = 0xe5a5,
  [0x19a0] = 0xe5a6, [0x19a1] = 0xe5a7, [0x19a2] = 0xe5a8, [0x19a3] = 0xe5a9,
  [0x19a4] = 0xe5aa, [0x19a5] = 0xe5ab, [0x19a6] = 0xe5ac, [0x19a7] = 0xe5ad,
  [0x19a8] = 0xe5ae, [0x19a9] = 0xe5af, [0x19aa] = 0xe5b0, [0x19ab] = 0xe5b1,
  [0x19ac] = 0xe5b2, [0x19ad] = 0xe5b3, [0x19ae] = 0xe5b4, [0x19af] = 0xe5b5,
  [0x19b0] = 0xe5b6, [0x19b1] = 0xe5b7, [0x19b2] = 0xe5b8, [0x19b3] = 0xe5b9,
  [0x19b4] = 0xe5ba, [0x19b5] = 0xe5bb, [0x19b6] = 0xe5bc, [0x19b7] = 0xe5bd,
  [0x19b8] = 0xe5be, [0x19b9] = 0xe5bf, [0x19ba] = 0xe5c0, [0x19bb] = 0xe5c1,
  [0x19bc] = 0xe5c2, [0x19bd] = 0xe5c3, [0x19be] = 0xe5c4, [0x19c0] = 0xe5c5,
  [0x19c1] = 0xe5c6, [0x19c2] = 0xe5c7, [0x19c3] = 0xe5c8, [0x19c4] = 0xe5c9,
  [0x19c5] = 0xe5ca, [0x19c6] = 0xe5cb, [0x19c7] = 0xe5cc, [0x19c8] = 0xe5cd,
  [0x19c9] = 0xe5ce, [0x19ca] = 0xe5cf, [0x19cb] = 0xe5d0, [0x19cc] = 0xe5d1,
  [0x19cd] = 0xe5d2, [0x19ce] = 0xe5d3, [0x19cf] = 0xe5d4, [0x19d0] = 0xe5d5,
  [0x19d1] = 0xe5d6, [0x19d2] = 0xe5d7, [0x19d3] = 0xe5d8, [0x19d4] = 0xe5d9,
  [0x19d5] = 0xe5da, [0x19d6] = 0xe5db, [0x19d7] = 0xe5dc, [0x19d8] = 0xe5dd,
  [0x19d9] = 0xe5de, [0x19da] = 0xe5df, [0x19db] = 0xe5e0, [0x19dc] = 0xe5e1,
  [0x19dd] = 0xe5e2, [0x19de] = 0xe5e3, [0x19df] = 0xe5e4, [0x19e0] = 0xe5e5,
  [0x19e1] = 0xff01, [0x19e2] = 0xff02, [0x19e3] = 0xff03, [0x19e4] = 0xffe5,
  [0x19e5] = 0xff05, [0x19e6] = 0xff06, [0x19e7] = 0xff07, [0x19e8] = 0xff08,
  [0x19e9] = 0xff09, [0x19ea] = 0xff0a, [0x19eb] = 0xff0b, [0x19ec] = 0xff0c,
  [0x19ed] = 0xff0d, [0x19ee] = 0xff0e, [0x19ef] = 0xff0f, [0x19f0] = 0xff10,
  [0x19f1] = 0xff11, [0x19f2] = 0xff12, [0x19f3] = 0xff13, [0x19f4] = 0xff14,
  [0x19f5] = 0xff15, [0x19f6] = 0xff16, [0x19f7] = 0xff17, [0x19f8] = 0xff18,
  [0x19f9] = 0xff19, [0x19fa] = 0xff1a, [0x19fb] = 0xff1b, [0x19fc] = 0xff1c,
  [0x19fd] = 0xff1d, [0x19fe] = 0xff1e, [0x19ff] = 0xff1f, [0x1a00] = 0xff20,
  [0x1a01] = 0xff21, [0x1a02] = 0xff22, [0x1a03] = 0xff23, [0x1a04] = 0xff24,
  [0x1a05] = 0xff25, [0x1a06] = 0xff26, [0x1a07] = 0xff27, [0x1a08] = 0xff28,
  [0x1a09] = 0xff29, [0x1a0a] = 0xff2a, [0x1a0b] = 0xff2b, [0x1a0c] = 0xff2c,
  [0x1a0d] = 0xff2d, [0x1a0e] = 0xff2e, [0x1a0f] = 0xff2f, [0x1a10] = 0xff30,
  [0x1a11] = 0xff31, [0x1a12] = 0xff32, [0x1a13] = 0xff33, [0x1a14] = 0xff34,
  [0x1a15] = 0xff35, [0x1a16] = 0xff36, [0x1a17] = 0xff37, [0x1a18] = 0xff38,
  [0x1a19] = 0xff39, [0x1a1a] = 0xff3a, [0x1a1b] = 0xff3b, [0x1a1c] = 0xff3c,
  [0x1a1d] = 0xff3d, [0x1a1e] = 0xff3e, [0x1a1f] = 0xff3f, [0x1a20] = 0xff40,
  [0x1a21] = 0xff41, [0x1a22] = 0xff42, [0x1a23] = 0xff43, [0x1a24] = 0xff44,
  [0x1a25] = 0xff45, [0x1a26] = 0xff46, [0x1a27] = 0xff47, [0x1a28] = 0xff48,
  [0x1a29] = 0xff49, [0x1a2a] = 0xff4a, [0x1a2b] = 0xff4b, [0x1a2c] = 0xff4c,
  [0x1a2d] = 0xff4d, [0x1a2e] = 0xff4e, [0x1a2f] = 0xff4f, [0x1a30] = 0xff50,
  [0x1a31] = 0xff51, [0x1a32] = 0xff52, [0x1a33] = 0xff53, [0x1a34] = 0xff54,
  [0x1a35] = 0xff55, [0x1a36] = 0xff56, [0x1a37] = 0xff57, [0x1a38] = 0xff58,
  [0x1a39] = 0xff59, [0x1a3a] = 0xff5a, [0x1a3b] = 0xff5b, [0x1a3c] = 0xff5c,
  [0x1a3d] = 0xff5d, [0x1a3e] = 0xffe3, [0x1a40] = 0xe5e6, [0x1a41] = 0xe5e7,
  [0x1a42] = 0xe5e8, [0x1a43] = 0xe5e9, [0x1a44] = 0xe5ea, [0x1a45] = 0xe5eb,
  [0x1a46] = 0xe5ec, [0x1a47] = 0xe5ed, [0x1a48] = 0xe5ee, [0x1a49] = 0xe5ef,
  [0x1a4a] = 0xe5f0, [0x1a4b] = 0xe5f1, [0x1a4c] = 0xe5f2, [0x1a4d] = 0xe5f3,
  [0x1a4e] = 0xe5f4, [0x1a4f] = 0xe5f5, [0x1a50] = 0xe5f6, [0x1a51] = 0xe5f7,
  [0x1a52] = 0xe5f8, [0x1a53] = 0xe5f9, [0x1a54] = 0xe5fa, [0x1a55] = 0xe5fb,
  [0x1a56] = 0xe5fc, [0x1a57] = 0xe5fd, [0x1a58] = 0xe5fe, [0x1a59] = 0xe5ff,
  [0x1a5a] = 0xe600, [0x1a5b] = 0xe601, [0x1a5c] = 0xe602, [0x1a5d] = 0xe603,
  [0x1a5e] = 0xe604, [0x1a5f] = 0xe605, [0x1a60] = 0xe606, [0x1a61] = 0xe607,
  [0x1a62] = 0xe608, [0x1a63] = 0xe609, [0x1a64] = 0xe60a, [0x1a65] = 0xe60b,
  [0x1a66] = 0xe60c, [0x1a67] = 0xe60d, [0x1a68] = 0xe60e, [0x1a69] = 0xe60f,
  [0x1a6a] = 0xe610, [0x1a6b] = 0xe611, [0x1a6c] = 0xe612, [0x1a6d] = 0xe613,
  [0x1a6e] = 0xe614, [0x1a6f] = 0xe615, [0x1a70] = 0xe616, [0x1a71] = 0xe617,
  [0x1a72] = 0xe618, [0x1a73] = 0xe619, [0x1a74] = 0xe61a, [0x1a75] = 0xe61b,
  [0x1a76] = 0xe61c, [0x1a77] = 0xe61d, [0x1a78] = 0xe61e, [0x1a79] = 0xe61f,
  [0x1a7a] = 0xe620, [0x1a7b] = 0xe621, [0x1a7c] = 0xe622, [0x1a7d] = 0xe623,
  [0x1a7e] = 0xe624, [0x1a80] = 0xe625, [0x1a81] = 0xe626, [0x1a82] = 0xe627,
  [0x1a83] = 0xe628, [0x1a84] = 0xe629, [0x1a85] = 0xe62a, [0x1a86] = 0xe62b,
  [0x1a87] = 0xe62c, [0x1a88] = 0xe62d, [0x1a89] = 0xe62e, [0x1a8a] = 0xe62f,
  [0x1a8b] = 0xe630, [0x1a8c] = 0xe631, [0x1a8d] = 0xe632, [0x1a8e] = 0xe633,
  [0x1a8f] = 0xe634, [0x1a90] = 0xe635, [0x1a91] = 0xe636, [0x1a92] = 0xe637,
  [0x1a93] = 0xe638, [0x1a94] = 0xe639, [0x1a95] = 0xe63a, [0x1a96] = 0xe63b,
  [0x1a97] = 0xe63c, [0x1a98] = 0xe63d, [0x1a99] = 0xe63e, [0x1a9a] = 0xe63f,
  [0x1a9b] = 0xe640, [0x1a9c] = 0xe641, [0x1a9d] = 0xe642, [0x1a9e] = 0xe643,
  [0x1a9f] = 0xe644, [0x1aa0] = 0xe645, [0x1aa1] = 0x3041, [0x1aa2] = 0x3042,
  [0x1aa3] = 0x3043, [0x1aa4] = 0x3044, [0x1aa5] = 0x3045, [0x1aa6] = 0x3046,
  [0x1aa7] = 0x3047, [0x1aa8] = 0x3048, [0x1aa9] = 0x3049, [0x1aaa] = 0x304a,
  [0x1aab] = 0x304b, [0x1aac] = 0x304c, [0x1aad] = 0x304d, [0x1aae] = 0x304e,
  [0x1aaf] = 0x304f, [0x1ab0] = 0x3050, [0x1ab1] = 0x3051, [0x1ab2] = 0x3052,
  [0x1ab3] = 0x3053, [0x1ab4] = 0x3054, [0x1ab5] = 0x3055, [0x1ab6] = 0x3056,
  [0x1ab7] = 0x3057, [0x1ab8] = 0x3058, [0x1ab9] = 0x3059, [0x1aba] = 0x305a,
  [0x1abb] = 0x305b, [0x1abc] = 0x305c, [0x1abd] = 0x305d, [0x1abe] = 0x305e,
  [0x1abf] = 0x305f, [0x1ac0] = 0x3060, [0x1ac1] = 0x3061, [0x1ac2] = 0x3062,
  [0x1ac3] = 0x3063, [0x1ac4] = 0x3064, [0x1ac5] = 0x3065, [0x1ac6] = 0x3066,
  [0x1ac7] = 0x3067, [0x1ac8] = 0x3068, [0x1ac9] = 0x3069, [0x1aca] = 0x306a,
  [0x1acb] = 0x306b, [0x1acc] = 0x306c, [0x1acd] = 0x306d, [0x1ace] = 0x306e,
  [0x1acf] = 0x306f, [0x1ad0] = 0x3070, [0x1ad1] = 0x3071, [0x1ad2] = 0x3072,
  [0x1ad3] = 0x3073, [0x1ad4] = 0x3074, [0x1ad5] = 0x3075, [0x1ad6] = 0x3076,
  [0x1ad7] = 0x3077, [0x1ad8] = 0x3078, [0x1ad9] = 0x3079, [0x1ada] = 0x307a,
  [0x1adb] = 0x307b, [0x1adc] = 0x307c, [0x1add] = 0x307d, [0x1ade] = 0x307e,
  [0x1adf] = 0x307f, [0x1ae0] = 0x3080, [0x1ae1] = 0x3081, [0x1ae2] = 0x3082,
  [0x1ae3] = 0x3083, [0x1ae4] = 0x3084, [0x1ae5] = 0x3085, [0x1ae6] = 0x3086,
  [0x1ae7] = 0x3087, [0x1ae8] = 0x3088, [0x1ae9] = 0x3089, [0x1aea] = 0x308a,
  [0x1aeb] = 0x308b, [0x1aec] = 0x308c, [0x1aed] = 0x308d, [0x1aee] = 0x308e,
  [0x1aef] = 0x308f, [0x1af0] = 0x3090, [0x1af1] = 0x3091, [0x1af2] = 0x3092,
  [0x1af3] = 0x3093, [0x1af4] = 0xe772, [0x1af5] = 0xe773, [0x1af6] = 0xe774,
  [0x1af7] = 0xe775, [0x1af8] = 0xe776, [0x1af9] = 0xe777, [0x1afa] = 0xe778,
  [0x1afb] = 0xe779, [0x1afc] = 0xe77a, [0x1afd] = 0xe77b, [0x1afe] = 0xe77c,
  [0x1b00] = 0xe646, [0x1b01] = 0xe647, [0x1b02] = 0xe648, [0x1b03] = 0xe649,
  [0x1b04] = 0xe64a, [0x1b05] = 0xe64b, [0x1b06] = 0xe64c, [0x1b07] = 0xe64d,
  [0x1b08] = 0xe64e, [0x1b09] = 0xe64f, [0x1b0a] = 0xe650, [0x1b0b] = 0xe651,
  [0x1b0c] = 0xe652, [0x1b0d] = 0xe653, [0x1b0e] = 0xe654, [0x1b0f] = 0xe655,
  [0x1b10] = 0xe656, [0x1b11] = 0xe657, [0x1b12] = 0xe658, [0x1b13] = 0xe659,
  [0x1b14] = 0xe65a, [0x1b15] = 0xe65b, [0x1b16] = 0xe65c, [0x1b17] = 0xe65d,
  [0x1b18] = 0xe65e, [0x1b19] = 0xe65f, [0x1b1a] = 0xe660, [0x1b1b] = 0xe661,
  [0x1b1c] = 0xe662, [0x1b1d] = 0xe663, [0x1b1e] = 0xe664, [0x1b1f] = 0xe665,
  [0x1b20] = 0xe666, [0x1b21] = 0xe667, [0x1b22] = 0xe668, [0x1b23] = 0xe669,
  [0x1b24] = 0xe66a, [0x1b25] = 0xe66b, [0x1b26] = 0xe66c, [0x1b27] = 0xe66d,
  [0x1b28] = 0xe66e, [0x1b29] = 0xe66f, [0x1b2a] = 0xe670, [0x1b2b] = 0xe671,
  [0x1b2c] = 0xe672, [0x1b2d] = 0xe673, [0x1b2e] = 0xe674, [0x1b2f] = 0xe675,
  [0x1b30] = 0xe676, [0x1b31] = 0xe677, [0x1b32] = 0xe678, [0x1b33] = 0xe679,
  [0x1b34] = 0xe67a, [0x1b35] = 0xe67b, [0x1b36] = 0xe67c, [0x1b37] = 0xe67d,
  [0x1b38] = 0xe67e, [0x1b39] = 0xe67f, [0x1b3a] = 0xe680, [0x1b3b] = 0xe681,
  [0x1b3c] = 0xe682, [0x1b3d] = 0xe683, [0x1b3e] = 0xe684, [0x1b40] = 0xe685,
  [0x1b41] = 0xe686, [0x1b42] = 0xe687, [0x1b43] = 0xe688, [0x1b44] = 0xe689,
  [0x1b45] = 0xe68a, [0x1b46] = 0xe68b, [0x1b47] = 0xe68c, [0x1b48] = 0xe68d,
  [0x1b49] = 0xe68e, [0x1b4a] = 0xe68f, [0x1b4b] = 0xe690, [0x1b4c] = 0xe691,
  [0x1b4d] = 0xe692, [0x1b4e] = 0xe693, [0x1b4f] = 0xe694, [0x1b50] = 0xe695,
  [0x1b51] = 0xe696, [0x1b52] = 0xe697, [0x1b53] = 0xe698, [0x1b54] = 0xe699,
  [0x1b55] = 0xe69a, [0x1b56] = 0xe69b, [0x1b57] = 0xe69c, [0x1b58] = 0xe69d,
  [0x1b59] = 0xe69e, [0x1b5a] = 0xe69f, [0x1b5b] = 0xe6a0, [0x1b5c] = 0xe6a1,
  [0x1b5d] = 0xe6a2, [0x1b5e] = 0xe6a3, [0x1b5f] = 0xe6a4, [0x1b60] = 0xe6a5,
  [0x1b61] = 0x30a1, [0x1b62] = 0x30a2, [0x1b63] = 0x30a3, [0x1b64] = 0x30a4,
  [0x1b65] = 0x30a5, [0x1b66] = 0x30a6, [0x1b67] = 0x30a7, [0x1b68] = 0x30a8,
  [0x1b69] = 0x30a9, [0x1b6a] = 0x30aa, [0x1b6b] = 0x30ab, [0x1b6c] = 0x30ac,
  [0x1b6d] = 0x30ad, [0x1b6e] = 0x30ae, [0x1b6f] = 0x30af, [0x1b70] = 0x30b0,
  [0x1b71] = 0x30b1, [0x1b72] = 0x30b2, [0x1b73] = 0x30b3, [0x1b74] = 0x30b4,
  [0x1b75] = 0x30b5, [0x1b76] = 0x30b6, [0x1b77] = 0x30b7, [0x1b78] = 0x30b8,
  [0x1b79] = 0x30b9, [0x1b7a] = 0x30ba, [0x1b7b] = 0x30bb, [0x1b7c] = 0x30bc,
  [0x1b7d] = 0x30bd, [0x1b7e] = 0x30be, [0x1b7f] = 0x30bf, [0x1b80] = 0x30c0,
  [0x1b81] = 0x30c1, [0x1b82] = 0x30c2, [0x1b83] = 0x30c3, [0x1b84] = 0x30c4,
  [0x1b85] = 0x30c5, [0x1b86] = 0x30c6, [0x1b87] = 0x30c7, [0x1b88] = 0x30c8,
  [0x1b89] = 0x30c9, [0x1b8a] = 0x30ca, [0x1b8b] = 0x30cb, [0x1b8c] = 0x30cc,
  [0x1b8d] = 0x30cd, [0x1b8e] = 0x30ce, [0x1b8f] = 0x30cf, [0x1b90] = 0x30d0,
  [0x1b91] = 0x30d1, [0x1b92] = 0x30d2, [0x1b93] = 0x30d3, [0x1b94] = 0x30d4,
  [0x1b95] = 0x30d5, [0x1b96] = 0x30d6, [0x1b97] = 0x30d7, [0x1b98] = 0x30d8,
  [0x1b99] = 0x30d9, [0x1b9a] = 0x30da, [0x1b9b] = 0x30db, [0x1b9c] = 0x30dc,
  [0x1b9d] = 0x30dd, [0x1b9e] = 0x30de, [0x1b9f] = 0x30df, [0x1ba0] = 0x30e0,
  [0x1ba1] = 0x30e1, [0x1ba2] = 0x30e2, [0x1ba3] = 0x30e3, [0x1ba4] = 0x30e4,
  [0x1ba5] = 0x30e5, [0x1ba6] = 0x30e6, [0x1ba7] = 0x30e7, [0x1ba8] = 0x30e8,
  [0x1ba9] = 0x30e9, [0x1baa] = 0x30ea, [0x1bab] = 0x30eb, [0x1bac] = 0x30ec,
  [0x1bad] = 0x30ed, [0x1bae] = 0x30ee, [0x1baf] = 0x30ef, [0x1bb0] = 0x30f0,
  [0x1bb1] = 0x30f1, [0x1bb2] = 0x30f2, [0x1bb3] = 0x30f3, [0x1bb4] = 0x30f4,
  [0x1bb5] = 0x30f5, [0x1bb6] = 0x30f6, [0x1bb7] = 0xe77d, [0x1bb8] = 0xe77e,
  [0x1bb9] = 0xe77f, [0x1bba] = 0xe780, [0x1bbb] = 0xe781, [0x1bbc] = 0xe782,
  [0x1bbd] = 0xe783, [0x1bbe] = 0xe784, [0x1bc0] = 0xe6a6, [0x1bc1] = 0xe6a7,
  [0x1bc2] = 0xe6a8, [0x1bc3] = 0xe6a9, [0x1bc4] = 0xe6aa, [0x1bc5] = 0xe6ab,
  [0x1bc6] = 0xe6ac, [0x1bc7] = 0xe6ad, [0x1bc8] = 0xe6ae, [0x1bc9] = 0xe6af,
  [0x1bca] = 0xe6b0, [0x1bcb] = 0xe6b1, [0x1bcc] = 0xe6b2, [0x1bcd] = 0xe6b3,
  [0x1bce] = 0xe6b4, [0x1bcf] = 0xe6b5, [0x1bd0] = 0xe6b6, [0x1bd1] = 0xe6b7,
  [0x1bd2] = 0xe6b8, [0x1bd3] = 0xe6b9, [0x1bd4] = 0xe6ba, [0x1bd5] = 0xe6bb,
  [0x1bd6] = 0xe6bc, [0x1bd7] = 0xe6bd, [0x1bd8] = 0xe6be, [0x1bd9] = 0xe6bf,
  [0x1bda] = 0xe6c0, [0x1bdb] = 0xe6c1, [0x1bdc] = 0xe6c2, [0x1bdd] = 0xe6c3,
  [0x1bde] = 0xe6c4, [0x1bdf] = 0xe6c5, [0x1be0] = 0xe6c6, [0x1be1] = 0xe6c7,
  [0x1be2] = 0xe6c8, [0x1be3] = 0xe6c9, [0x1be4] = 0xe6ca, [0x1be5] = 0xe6cb,
  [0x1be6] = 0xe6cc, [0x1be7] = 0xe6cd, [0x1be8] = 0xe6ce, [0x1be9] = 0xe6cf,
  [0x1bea] = 0xe6d0, [0x1beb] = 0xe6d1, [0x1bec] = 0xe6d2, [0x1bed] = 0xe6d3,
  [0x1bee] = 0xe6d4, [0x1bef] = 0xe6d5, [0x1bf0] = 0xe6d6, [0x1bf1] = 0xe6d7,
  [0x1bf2] = 0xe6d8, [0x1bf3] = 0xe6d9, [0x1bf4] = 0xe6da, [0x1bf5] = 0xe6db,
  [0x1bf6] = 0xe6dc, [0x1bf7] = 0xe6dd, [0x1bf8] = 0xe6de, [0x1bf9] = 0xe6df,
  [0x1bfa] = 0xe6e0, [0x1bfb] = 0xe6e1, [0x1bfc] = 0xe6e2, [0x1bfd] = 0xe6e3,
  [0x1bfe] = 0xe6e4, [0x1c00] = 0xe6e5, [0x1c01] = 0xe6e6, [0x1c02] = 0xe6e7,
  [0x1c03] = 0xe6e8, [0x1c04] = 0xe6e9, [0x1c05] = 0xe6ea, [0x1c06] = 0xe6eb,
  [0x1c07] = 0xe6ec, [0x1c08] = 0xe6ed, [0x1c09] = 0xe6ee, [0x1c0a] = 0xe6ef,
  [0x1c0b] = 0xe6f0, [0x1c0c] = 0xe6f1, [0x1c0d] = 0xe6f2, [0x1c0e] = 0xe6f3,
  [0x1c0f] = 0xe6f4, [0x1c10] = 0xe6f5, [0x1c11] = 0xe6f6, [0x1c12] = 0xe6f7,
  [0x1c13] = 0xe6f8, [0x1c14] = 0xe6f9, [0x1c15] = 0xe6fa, [0x1c16] = 0xe6fb,
  [0x1c17] = 0xe6fc, [0x1c18] = 0xe6fd, [0x1c19] = 0xe6fe, [0x1c1a] = 0xe6ff,
  [0x1c1b] = 0xe700, [0x1c1c] = 0xe701, [0x1c1d] = 0xe702, [0x1c1e] = 0xe703,
  [0x1c1f] = 0xe704, [0x1c20] = 0xe705, [0x1c21] = 0x0391, [0x1c22] = 0x0392,
  [0x1c23] = 0x0393, [0x1c24] = 0x0394, [0x1c25] = 0x0395, [0x1c26] = 0x0396,
  [0x1c27] = 0x0397, [0x1c28] = 0x0398, [0x1c29] = 0x0399, [0x1c2a] = 0x039a,
  [0x1c2b] = 0x039b, [0x1c2c] = 0x039c, [0x1c2d] = 0x039d, [0x1c2e] = 0x039e,
  [0x1c2f] = 0x039f, [0x1c30] = 0x03a0, [0x1c31] = 0x03a1, [0x1c32] = 0x03a3,
  [0x1c33] = 0x03a4, [0x1c34] = 0x03a5, [0x1c35] = 0x03a6, [0x1c36] = 0x03a7,
  [0x1c37] = 0x03a8, [0x1c38] = 0x03a9, [0x1c39] = 0xe785, [0x1c3a] = 0xe786,
  [0x1c3b] = 0xe787, [0x1c3c] = 0xe788, [0x1c3d] = 0xe789, [0x1c3e] = 0xe78a,
  [0x1c3f] = 0xe78b, [0x1c40] = 0xe78c, [0x1c41] = 0x03b1, [0x1c42] = 0x03b2,
  [0x1c43] = 0x03b3, [0x1c44] = 0x03b4, [0x1c45] = 0x03b5, [0x1c46] = 0x03b6,
  [0x1c47] = 0x03b7, [0x1c48] = 0x03b8, [0x1c49] = 0x03b9, [0x1c4a] = 0x03ba,
  [0x1c4b] = 0x03bb, [0x1c4c] = 0x03bc, [0x1c4d] = 0x03bd, [0x1c4e] = 0x03be,
  [0x1c4f] = 0x03bf, [0x1c50] = 0x03c0, [0x1c51] = 0x03c1, [0x1c52] = 0x03c3,
  [0x1c53] = 0x03c4, [0x1c54] = 0x03c5, [0x1c55] = 0x03c6, [0x1c56] = 0x03c7,
  [0x1c57] = 0x03c8, [0x1c58] = 0x03c9, [0x1c59] = 0xe78d, [0x1c5a] = 0xe78e,
  [0x1c5b] = 0xe78f, [0x1c5c] = 0xe790, [0x1c5d] = 0xe791, [0x1c5e] = 0xe792,
  [0x1c5f] = 0xe793, [0x1c60] = 0xfe35, [0x1c61] = 0xfe36, [0x1c62] = 0xfe39,
  [0x1c63] = 0xfe3a, [0x1c64] = 0xfe3f, [0x1c65] = 0xfe40, [0x1c66] = 0xfe3d,
  [0x1c67] = 0xfe3e, [0x1c68] = 0xfe41, [0x1c69] = 0xfe42, [0x1c6a] = 0xfe43,
  [0x1c6b] = 0xfe44, [0x1c6c] = 0xe794, [0x1c6d] = 0xe795, [0x1c6e] = 0xfe3b,
  [0x1c6f] = 0xfe3c, [0x1c70] = 0xfe37, [0x1c71] = 0xfe38, [0x1c72] = 0xfe31,
  [0x1c73] = 0xe796, [0x1c74] = 0xfe33, [0x1c75] = 0xfe34, [0x1c76] = 0xe797,
  [0x1c77] = 0xe798, [0x1c78] = 0xe799, [0x1c79] = 0xe79a, [0x1c7a] = 0xe79b,
  [0x1c7b] = 0xe79c, [0x1c7c] = 0xe79d, [0x1c7d] = 0xe79e, [0x1c7e] = 0xe79f,
  [0x1c80] = 0xe706, [0x1c81] = 0xe707, [0x1c82] = 0xe708, [0x1c83] = 0xe709,
  [0x1c84] = 0xe70a, [0x1c85] = 0xe70b, [0x1c86] = 0xe70c, [0x1c87] = 0xe70d,
  [0x1c88] = 0xe70e, [0x1c89] = 0xe70f, [0x1c8a] = 0xe710, [0x1c8b] = 0xe711,
  [0x1c8c] = 0xe712, [0x1c8d] = 0xe713, [0x1c8e] = 0xe714, [0x1c8f] = 0xe715,
  [0x1c90] = 0xe716, [0x1c91] = 0xe717, [0x1c92] = 0xe718, [0x1c93] = 0xe719,
  [0x1c94] = 0xe71a, [0x1c95] = 0xe71b, [0x1c96] = 0xe71c, [0x1c97] = 0xe71d,
  [0x1c98] = 0xe71e, [0x1c99] = 0xe71f, [0x1c9a] = 0xe720, [0x1c9b] = 0xe721,
  [0x1c9c] = 0xe722, [0x1c9d] = 0xe723, [0x1c9e] = 0xe724, [0x1c9f] = 0xe725,
  [0x1ca0] = 0xe726, [0x1ca1] = 0xe727, [0x1ca2] = 0xe728, [0x1ca3] = 0xe729,
  [0x1ca4] = 0xe72a, [0x1ca5] = 0xe72b, [0x1ca6] = 0xe72c, [0x1ca7] = 0xe72d,
  [0x1ca8] = 0xe72e, [0x1ca9] = 0xe72f, [0x1caa] = 0xe730, [0x1cab] = 0xe731,
  [0x1cac] = 0xe732, [0x1cad] = 0xe733, [0x1cae] = 0xe734, [0x1caf] = 0xe735,
  [0x1cb0] = 0xe736, [0x1cb1] = 0xe737, [0x1cb2] = 0xe738, [0x1cb3] = 0xe739,
  [0x1cb4] = 0xe73a, [0x1cb5] = 0xe73b, [0x1cb6] = 0xe73c, [0x1cb7] = 0xe73d,
  [0x1cb8] = 0xe73e, [0x1cb9] = 0xe73f, [0x1cba] = 0xe740, [0x1cbb] = 0xe741,
  [0x1cbc] = 0xe742, [0x1cbd] = 0xe743, [0x1cbe] = 0xe744, [0x1cc0] = 0xe745,
  [0x1cc1] = 0xe746, [0x1cc2] = 0xe747, [0x1cc3] = 0xe748, [0x1cc4] = 0xe749,
  [0x1cc5] = 0xe74a, [0x1cc6] = 0xe74b, [0x1cc7] = 0xe74c, [0x1cc8] = 0xe74d,
  [0x1cc9] = 0xe74e, [0x1cca] = 0xe74f, [0x1ccb] = 0xe750, [0x1ccc] = 0xe751,
  [0x1ccd] = 0xe752, [0x1cce] = 0xe753, [0x1ccf] = 0xe754, [0x1cd0] = 0xe755,
  [0x1cd1] = 0xe756, [0x1cd2] = 0xe757, [0x1cd3] = 0xe758, [0x1cd4] = 0xe759,
  [0x1cd5] = 0xe75a, [0x1cd6] = 0xe75b, [0x1cd7] = 0xe75c, [0x1cd8] = 0xe75d,
  [0x1cd9] = 0xe75e, [0x1cda] = 0xe75f, [0x1cdb] = 0xe760, [0x1cdc] = 0xe761,
  [0x1cdd] = 0xe762, [0x1cde] = 0xe763, [0x1cdf] = 0xe764, [0x1ce0] = 0xe765,
  [0x1ce1] = 0x0410, [0x1ce2] = 0x0411, [0x1ce3] = 0x0412, [0x1ce4] = 0x0413,
  [0x1ce5] = 0x0414, [0x1ce6] = 0x0415, [0x1ce7] = 0x0401, [0x1ce8] = 0x0416,
  [0x1ce9] = 0x0417, [0x1cea] = 0x0418, [0x1ceb] = 0x0419, [0x1cec] = 0x041a,
  [0x1ced] = 0x041b, [0x1cee] = 0x041c, [0x1cef] = 0x041d, [0x1cf0] = 0x041e,
  [0x1cf1] = 0x041f, [0x1cf2] = 0x0420, [0x1cf3] = 0x0421, [0x1cf4] = 0x0422,
  [0x1cf5] = 0x0423, [0x1cf6] = 0x0424, [0x1cf7] = 0x0425, [0x1cf8] = 0x0426,
  [0x1cf9] = 0x0427, [0x1cfa] = 0x0428, [0x1cfb] = 0x0429, [0x1cfc] = 0x042a,
  [0x1cfd] = 0x042b, [0x1cfe] = 0x042c, [0x1cff] = 0x042d, [0x1d00] = 0x042e,
  [0x1d01] = 0x042f, [0x1d02] = 0xe7a0, [0x1d03] = 0xe7a1, [0x1d04] = 0xe7a2,
  [0x1d05] = 0xe7a3, [0x1d06] = 0xe7a4, [0x1d07] = 0xe7a5, [0x1d08] = 0xe7a6,
  [0x1d09] = 0xe7a7, [0x1d0a] = 0xe7a8, [0x1d0b] = 0xe7a9, [0x1d0c] = 0xe7aa,
  [0x1d0d] = 0xe7ab, [0x1d0e] = 0xe7ac, [0x1d0f] = 0xe7ad, [0x1d10] = 0xe7ae,
  [0x1d11] = 0x0430, [0x1d12] = 0x0431, [0x1d13] = 0x0432, [0x1d14] = 0x0433,
  [0x1d15] = 0x0434, [0x1d16] = 0x0435, [0x1d17] = 0x0451, [0x1d18] = 0x0436,
  [0x1d19] = 0x0437, [0x1d1a] = 0x0438, [0x1d1b] = 0x0439, [0x1d1c] = 0x043a,
  [0x1d1d] = 0x043b, [0x1d1e] = 0x043c, [0x1d1f] = 0x043d, [0x1d20] = 0x043e,
  [0x1d21] = 0x043f, [0x1d22] = 0x0440, [0x1d23] = 0x0441, [0x1d24] = 0x0442,
  [0x1d25] = 0x0443, [0x1d26] = 0x0444, [0x1d27] = 0x0445, [0x1d28] = 0x0446,
  [0x1d29] = 0x0447, [0x1d2a] = 0x0448, [0x1d2b] = 0x0449, [0x1d2c] = 0x044a,
  [0x1d2d] = 0x044b, [0x1d2e] = 0x044c, [0x1d2f] = 0x044d, [0x1d30] = 0x044e,
  [0x1d31] = 0x044f, [0x1d32] = 0xe7af, [0x1d33] = 0xe7b0, [0x1d34] = 0xe7b1,
  [0x1d35] = 0xe7b2, [0x1d36] = 0xe7b3, [0x1d37] = 0xe7b4, [0x1d38] = 0xe7b5,
  [0x1d39] = 0xe7b6, [0x1d3a] = 0xe7b7, [0x1d3b] = 0xe7b8, [0x1d3c] = 0xe7b9,
  [0x1d3d] = 0xe7ba, [0x1d3e] = 0xe7bb, [0x1d40] = 0x02ca, [0x1d41] = 0x02cb,
  [0x1d42] = 0x02d9, [0x1d43] = 0x2013, [0x1d44] = 0x2015, [0x1d45] = 0x2025,
  [0x1d46] = 0x2035, [0x1d47] = 0x2105, [0x1d48] = 0x2109, [0x1d49] = 0x2196,
  [0x1d4a] = 0x2197, [0x1d4b] = 0x2198, [0x1d4c] = 0x2199, [0x1d4d] = 0x2215,
  [0x1d4e] = 0x221f, [0x1d4f] = 0x2223, [0x1d50] = 0x2252, [0x1d51] = 0x2266,
  [0x1d52] = 0x2267, [0x1d53] = 0x22bf, [0x1d54] = 0x2550, [0x1d55] = 0x2551,
  [0x1d56] = 0x2552, [0x1d57] = 0x2553, [0x1d58] = 0x2554, [0x1d59] = 0x2555,
  [0x1d5a] = 0x2556, [0x1d5b] = 0x2557, [0x1d5c] = 0x2558, [0x1d5d] = 0x2559,
  [0x1d5e] = 0x255a, [0x1d5f] = 0x255b, [0x1d60] = 0x255c, [0x1d61] = 0x255d,
  [0x1d62] = 0x255e, [0x1d63] = 0x255f, [0x1d64] = 0x2560, [0x1d65] = 0x2561,
  [0x1d66] = 0x2562, [0x1d67] = 0x2563, [0x1d68] = 0x2564, [0x1d69] = 0x2565,
  [0x1d6a] = 0x2566, [0x1d6b] = 0x2567, [0x1d6c] = 0x2568, [0x1d6d] = 0x2569,
  [0x1d6e] = 0x256a, [0x1d6f] = 0x256b, [0x1d70] = 0x256c, [0x1d71] = 0x256d,
  [0x1d72] = 0x256e, [0x1d73] = 0x256f, [0x1d74] = 0x2570, [0x1d75] = 0x2571,
  [0x1d76] = 0x2572, [0x1d77] = 0x2573, [0x1d78] = 0x2581, [0x1d79] = 0x2582,
  [0x1d7a] = 0x2583, [0x1d7b] = 0x2584, [0x1d7c] = 0x2585, [0x1d7d] = 0x2586,
  [0x1d7e] = 0x2587, [0x1d80] = 0x2588, [0x1d81] = 0x2589, [0x1d82] = 0x258a,
  [0x1d83] = 0x258b, [0x1d84] = 0x258c, [0x1d85] = 0x258d, [0x1d86] = 0x258e,
  [0x1d87] = 0x258f, [0x1d88] = 0x2593, [0x1d89] = 0x2594, [0x1d8a] = 0x2595,
  [0x1d8b] = 0x25bc, [0x1d8c] = 0x25bd, [0x1d8d] = 0x25e2, [0x1d8e] = 0x25e3,
  [0x1d8f] = 0x25e4, [0x1d90] = 0x25e5, [0x1d91] = 0x2609, [0x1d92] = 0x2295,
  [0x1d93] = 0x3012, [0x1d94] = 0x301d, [0x1d95] = 0x301e, [0x1d96] = 0xe7bc,
  [0x1d97] = 0xe7bd, [0x1d98] = 0xe7be, [0x1d99] = 0xe7bf, [0x1d9a] = 0xe7c0,
  [0x1d9b] = 0xe7c1, [0x1d9c] = 0xe7c2, [0x1d9d] = 0xe7c3, [0x1d9e] = 0xe7c4,
  [0x1d9f] = 0xe7c5, [0x1da0] = 0xe7c6, [0x1da1] = 0x0101, [0x1da2] = 0x00e1,
  [0x1da3] = 0x01ce, [0x1da4] = 0x00e0, [0x1da5] = 0x0113, [0x1da6] = 0x00e9,
  [0x1da7] = 0x011b, [0x1da8] = 0x00e8, [0x1da9] = 0x012b, [0x1daa] = 0x00ed,
  [0x1dab] = 0x01d0, [0x1dac] = 0x00ec, [0x1dad] = 0x014d, [0x1dae] = 0x00f3,
  [0x1daf] = 0x01d2, [0x1db0] = 0x00f2, [0x1db1] = 0x016b, [0x1db2] = 0x00fa,
  [0x1db3] = 0x01d4, [0x1db4] = 0x00f9, [0x1db5] = 0x01d6, [0x1db6] = 0x01d8,
  [0x1db7] = 0x01da, [0x1db8] = 0x01dc, [0x1db9] = 0x00fc, [0x1dba] = 0x00ea,
  [0x1dbb] = 0x0251, [0x1dbc] = 0x1e3f, [0x1dbd] = 0x0144, [0x1dbe] = 0x0148,
  [0x1dbf] = 0x01f9, [0x1dc0] = 0x0261, [0x1dc1] = 0xe7c9, [0x1dc2] = 0xe7ca,
  [0x1dc3] = 0xe7cb, [0x1dc4] = 0xe7cc, [0x1dc5] = 0x3105, [0x1dc6] = 0x3106,
  [0x1dc7] = 0x3107, [0x1dc8] = 0x3108, [0x1dc9] = 0x3109, [0x1dca] = 0x310a,
  [0x1dcb] = 0x310b, [0x1dcc] = 0x310c, [0x1dcd] = 0x310d, [0x1dce] = 0x310e,
  [0x1dcf] = 0x310f, [0x1dd0] = 0x3110, [0x1dd1] = 0x3111, [0x1dd2] = 0x3112,
  [0x1dd3] = 0x3113, [0x1dd4] = 0x3114, [0x1dd5] = 0x3115, [0x1dd6] = 0x3116,
  [0x1dd7] = 0x3117, [0x1dd8] = 0x3118, [0x1dd9] = 0x3119, [0x1dda] = 0x311a,
  [0x1ddb] = 0x311b, [0x1ddc] = 0x311c, [0x1ddd] = 0x311d, [0x1dde] = 0x311e,
  [0x1ddf] = 0x311f, [0x1de0] = 0x3120, [0x1de1] = 0x3121, [0x1de2] = 0x3122,
  [0x1de3] = 0x3123, [0x1de4] = 0x3124, [0x1de5] = 0x3125, [0x1de6] = 0x3126,
  [0x1de7] = 0x3127, [0x1de8] = 0x3128, [0x1de9] = 0x3129, [0x1dea] = 0xe7cd,
  [0x1deb] = 0xe7ce, [0x1dec] = 0xe7cf, [0x1ded] = 0xe7d0, [0x1dee] = 0xe7d1,
  [0x1def] = 0xe7d2, [0x1df0] = 0xe7d3, [0x1df1] = 0xe7d4, [0x1df2] = 0xe7d5,
  [0x1df3] = 0xe7d6, [0x1df4] = 0xe7d7, [0x1df5] = 0xe7d8, [0x1df6] = 0xe7d9,
  [0x1df7] = 0xe7da, [0x1df8] = 0xe7db, [0x1df9] = 0xe7dc, [0x1dfa] = 0xe7dd,
  [0x1dfb] = 0xe7de, [0x1dfc] = 0xe7df, [0x1dfd] = 0xe7e0, [0x1dfe] = 0xe7e1,
  [0x1e00] = 0x3021, [0x1e01] = 0x3022, [0x1e02] = 0x3023, [0x1e03] = 0x3024,
  [0x1e04] = 0x3025, [0x1e05] = 0x3026, [0x1e06] = 0x3027, [0x1e07] = 0x3028,
  [0x1e08] = 0x3029, [0x1e09] = 0x32a3, [0x1e0a] = 0x338e, [0x1e0b] = 0x338f,
  [0x1e0c] = 0x339c, [0x1e0d] = 0x339d, [0x1e0e] = 0x339e, [0x1e0f] = 0x33a1,
  [0x1e10] = 0x33c4, [0x1e11] = 0x33ce, [0x1e12] = 0x33d1, [0x1e13] = 0x33d2,
  [0x1e14] = 0x33d5, [0x1e15] = 0xfe30, [0x1e16] = 0xffe2, [0x1e17] = 0xffe4,
  [0x1e18] = 0xe7e2, [0x1e19] = 0x2121, [0x1e1a] = 0x3231, [0x1e1b] = 0xe7e3,
  [0x1e1c] = 0x2010, [0x1e1d] = 0xe7e4, [0x1e1e] = 0xe7e5, [0x1e1f] = 0xe7e6,
  [0x1e20] = 0x30fc, [0x1e21] = 0x309b, [0x1e22] = 0x309c, [0x1e23] = 0x30fd,
  [0x1e24] = 0x30fe, [0x1e25] = 0x3006, [0x1e26] = 0x309d, [0x1e27] = 0x309e,
  [0x1e28] = 0xfe49, [0x1e29] = 0xfe4a, [0x1e2a] = 0xfe4b, [0x1e2b] = 0xfe4c,
  [0x1e2c] = 0xfe4d, [0x1e2d] = 0xfe4e, [0x1e2e] = 0xfe4f, [0x1e2f] = 0xfe50,
  [0x1e30] = 0xfe51, [0x1e31] = 0xfe52, [0x1e32] = 0xfe54, [0x1e33] = 0xfe55,
  [0x1e34] = 0xfe56, [0x1e35] = 0xfe57, [0x1e36] = 0xfe59, [0x1e37] = 0xfe5a,
  [0x1e38] = 0xfe5b, [0x1e39] = 0xfe5c, [0x1e3a] = 0xfe5d, [0x1e3b] = 0xfe5e,
  [0x1e3c] = 0xfe5f, [0x1e3d] = 0xfe60, [0x1e3e] = 0xfe61, [0x1e40] = 0xfe62,
  [0x1e41] = 0xfe63, [0x1e42] = 0xfe64, [0x1e43] = 0xfe65, [0x1e44] = 0xfe66,
  [0x1e45] = 0xfe68, [0x1e46] = 0xfe69, [0x1e47] = 0xfe6a, [0x1e48] = 0xfe6b,
  [0x1e49] = 0x303e, [0x1e4a] = 0x2ff0, [0x1e4b] = 0x2ff1, [0x1e4c] = 0x2ff2,
  [0x1e4d] = 0x2ff3, [0x1e4e] = 0x2ff4, [0x1e4f] = 0x2ff5, [0x1e50] = 0x2ff6,
  [0x1e51] = 0x2ff7, [0x1e52] = 0x2ff8, [0x1e53] = 0x2ff9, [0x1e54] = 0x2ffa,
  [0x1e55] = 0x2ffb, [0x1e56] = 0x3007, [0x1e57] = 0xe7f4, [0x1e58] = 0xe7f5,
  [0x1e59] = 0xe7f6, [0x1e5a] = 0xe7f7, [0x1e5b] = 0xe7f8, [0x1e5c] = 0xe7f9,
  [0x1e5d] = 0xe7fa, [0x1e5e] = 0xe7fb, [0x1e5f] = 0xe7fc, [0x1e60] = 0xe7fd,
  [0x1e61] = 0xe7fe, [0x1e62] = 0xe7ff, [0x1e63] = 0xe800, [0x1e64] = 0x2500,
  [0x1e65] = 0x2501, [0x1e66] = 0x2502, [0x1e67] = 0x2503, [0x1e68] = 0x2504,
  [0x1e69] = 0x2505, [0x1e6a] = 0x2506, [0x1e6b] = 0x2507, [0x1e6c] = 0x2508,
  [0x1e6d] = 0x2509, [0x1e6e] = 0x250a, [0x1e6f] = 0x250b, [0x1e70] = 0x250c,
  [0x1e71] = 0x250d, [0x1e72] = 0x250e, [0x1e73] = 0x250f, [0x1e74] = 0x2510,
  [0x1e75] = 0x2511, [0x1e76] = 0x2512, [0x1e77] = 0x2513, [0x1e78] = 0x2514,
  [0x1e79] = 0x2515, [0x1e7a] = 0x2516, [0x1e7b] = 0x2517, [0x1e7c] = 0x2518,
  [0x1e7d] = 0x2519, [0x1e7e] = 0x251a, [0x1e7f] = 0x251b, [0x1e80] = 0x251c,
  [0x1e81] = 0x251d, [0x1e82] = 0x251e, [0x1e83] = 0x251f, [0x1e84] = 0x2520,
  [0x1e85] = 0x2521, [0x1e86] = 0x2522, [0x1e87] = 0x2523, [0x1e88] = 0x2524,
  [0x1e89] = 0x2525, [0x1e8a] = 0x2526, [0x1e8b] = 0x2527, [0x1e8c] = 0x2528,
  [0x1e8d] = 0x2529, [0x1e8e] = 0x252a, [0x1e8f] = 0x252b, [0x1e90] = 0x252c,
  [0x1e91] = 0x252d, [0x1e92] = 0x252e, [0x1e93] = 0x252f, [0x1e94] = 0x2530,
  [0x1e95] = 0x2531, [0x1e96] = 0x2532, [0x1e97] = 0x2533, [0x1e98] = 0x2534,
  [0x1e99] = 0x2535, [0x1e9a] = 0x2536, [0x1e9b] = 0x2537, [0x1e9c] = 0x2538,
  [0x1e9d] = 0x2539, [0x1e9e] = 0x253a, [0x1e9f] = 0x253b, [0x1ea0] = 0x253c,
  [0x1ea1] = 0x253d, [0x1ea2] = 0x253e, [0x1ea3] = 0x253f, [0x1ea4] = 0x2540,
  [0x1ea5] = 0x2541, [0x1ea6] = 0x2542, [0x1ea7] = 0x2543, [0x1ea8] = 0x2544,
  [0x1ea9] = 0x2545, [0x1eaa] = 0x2546, [0x1eab] = 0x2547, [0x1eac] = 0x2548,
  [0x1ead] = 0x2549, [0x1eae] = 0x254a, [0x1eaf] = 0x254b, [0x1eb0] = 0xe801,
  [0x1eb1] = 0xe802, [0x1eb2] = 0xe803, [0x1eb3] = 0xe804, [0x1eb4] = 0xe805,
  [0x1eb5] = 0xe806, [0x1eb6] = 0xe807, [0x1eb7] = 0xe808, [0x1eb8] = 0xe809,
  [0x1eb9] = 0xe80a, [0x1eba] = 0xe80b, [0x1ebb] = 0xe80c, [0x1ebc] = 0xe80d,
  [0x1ebd] = 0xe80e, [0x1ebe] = 0xe80f, [0x1ec0] = 0x72dc, [0x1ec1] = 0x72dd,
  [0x1ec2] = 0x72df, [0x1ec3] = 0x72e2, [0x1ec4] = 0x72e3, [0x1ec5] = 0x72e4,
  [0x1ec6] = 0x72e5, [0x1ec7] = 0x72e6, [0x1ec8] = 0x72e7, [0x1ec9] = 0x72ea,
  [0x1eca] = 0x72eb, [0x1ecb] = 0x72f5, [0x1ecc] = 0x72f6, [0x1ecd] = 0x72f9,
  [0x1ece] = 0x72fd, [0x1ecf] = 0x72fe, [0x1ed0] = 0x72ff, [0x1ed1] = 0x7300,
  [0x1ed2] = 0x7302, [0x1ed3] = 0x7304, [0x1ed4] = 0x7305, [0x1ed5] = 0x7306,
  [0x1ed6] = 0x7307, [0x1ed7] = 0x7308, [0x1ed8] = 0x7309, [0x1ed9] = 0x730b,
  [0x1eda] = 0x730c, [0x1edb] = 0x730d, [0x1edc] = 0x730f, [0x1edd] = 0x7310,
  [0x1ede] = 0x7311, [0x1edf] = 0x7312, [0x1ee0] = 0x7314, [0x1ee1] = 0x7318,
  [0x1ee2] = 0x7319, [0x1ee3] = 0x731a, [0x1ee4] = 0x731f, [0x1ee5] = 0x7320,
  [0x1ee6] = 0x7323, [0x1ee7] = 0x7324, [0x1ee8] = 0x7326, [0x1ee9] = 0x7327,
  [0x1eea] = 0x7328, [0x1eeb] = 0x732d, [0x1eec] = 0x732f, [0x1eed] = 0x7330,
  [0x1eee] = 0x7332, [0x1eef] = 0x7333, [0x1ef0] = 0x7335, [0x1ef1] = 0x7336,
  [0x1ef2] = 0x733a, [0x1ef3] = 0x733b, [0x1ef4] = 0x733c, [0x1ef5] = 0x733d,
  [0x1ef6] = 0x7340, [0x1ef7] = 0x7341, [0x1ef8] = 0x7342, [0x1ef9] = 0x7343,
  [0x1efa] = 0x7344, [0x1efb] = 0x7345, [0x1efc] = 0x7346, [0x1efd] = 0x7347,
  [0x1efe] = 0x7348, [0x1f00] = 0x7349, [0x1f01] = 0x734a, [0x1f02] = 0x734b,
  [0x1f03] = 0x734c, [0x1f04] = 0x734e, [0x1f05] = 0x734f, [0x1f06] = 0x7351,
  [0x1f07] = 0x7353, [0x1f08] = 0x7354, [0x1f09] = 0x7355, [0x1f0a] = 0x7356,
  [0x1f0b] = 0x7358, [0x1f0c] = 0x7359, [0x1f0d] = 0x735a, [0x1f0e] = 0x735b,
  [0x1f0f] = 0x735c, [0x1f10] = 0x735d, [0x1f11] = 0x735e, [0x1f12] = 0x735f,
  [0x1f13] = 0x7361, [0x1f14] = 0x7362, [0x1f15] = 0x7363, [0x1f16] = 0x7364,
  [0x1f17] = 0x7365, [0x1f18] = 0x7366, [0x1f19] = 0x7367, [0x1f1a] = 0x7368,
  [0x1f1b] = 0x7369, [0x1f1c] = 0x736a, [0x1f1d] = 0x736b, [0x1f1e] = 0x736e,
  [0x1f1f] = 0x7370, [0x1f20] = 0x7371, [0x1f21] = 0xe000, [0x1f22] = 0xe001,
  [0x1f23] = 0xe002, [0x1f24] = 0xe003, [0x1f25] = 0xe004, [0x1f26] = 0xe005,
  [0x1f27] = 0xe006, [0x1f28] = 0xe007, [0x1f29] = 0xe008, [0x1f2a] = 0xe009,
  [0x1f2b] = 0xe00a, [0x1f2c] = 0xe00b, [0x1f2d] = 0xe00c, [0x1f2e] = 0xe00d,
  [0x1f2f] = 0xe00e, [0x1f30] = 0xe00f, [0x1f31] = 0xe010, [0x1f32] = 0xe011,
  [0x1f33] = 0xe012, [0x1f34] = 0xe013, [0x1f35] = 0xe014, [0x1f36] = 0xe015,
  [0x1f37] = 0xe016, [0x1f38] = 0xe017, [0x1f39] = 0xe018, [0x1f3a] = 0xe019,
  [0x1f3b] = 0xe01a, [0x1f3c] = 0xe01b, [0x1f3d] = 0xe01c, [0x1f3e] = 0xe01d,
  [0x1f3f] = 0xe01e, [0x1f40] = 0xe01f, [0x1f41] = 0xe020, [0x1f42] = 0xe021,
  [0x1f43] = 0xe022, [0x1f44] = 0xe023, [0x1f45] = 0xe024, [0x1f46] = 0xe025,
  [0x1f47] = 0xe026, [0x1f48] = 0xe027, [0x1f49] = 0xe028, [0x1f4a] = 0xe029,
  [0x1f4b] = 0xe02a, [0x1f4c] = 0xe02b, [0x1f4d] = 0xe02c, [0x1f4e] = 0xe02d,
  [0x1f4f] = 0xe02e, [0x1f50] = 0xe02f, [0x1f51] = 0xe030, [0x1f52] = 0xe031,
  [0x1f53] = 0xe032, [0x1f54] = 0xe033, [0x1f55] = 0xe034, [0x1f56] = 0xe035,
  [0x1f57] = 0xe036, [0x1f58] = 0xe037, [0x1f59] = 0xe038, [0x1f5a] = 0xe039,
  [0x1f5b] = 0xe03a, [0x1f5c] = 0xe03b, [0x1f5d] = 0xe03c, [0x1f5e] = 0xe03d,
  [0x1f5f] = 0xe03e, [0x1f60] = 0xe03f, [0x1f61] = 0xe040, [0x1f62] = 0xe041,
  [0x1f63] = 0xe042, [0x1f64] = 0xe043, [0x1f65] = 0xe044, [0x1f66] = 0xe045,
  [0x1f67] = 0xe046, [0x1f68] = 0xe047, [0x1f69] = 0xe048, [0x1f6a] = 0xe049,
  [0x1f6b] = 0xe04a, [0x1f6c] = 0xe04b, [0x1f6d] = 0xe04c, [0x1f6e] = 0xe04d,
  [0x1f6f] = 0xe04e, [0x1f70] = 0xe04f, [0x1f71] = 0xe050, [0x1f72] = 0xe051,
  [0x1f73] = 0xe052, [0x1f74] = 0xe053, [0x1f75] = 0xe054, [0x1f76] = 0xe055,
  [0x1f77] = 0xe056, [0x1f78] = 0xe057, [0x1f79] = 0xe058, [0x1f7a] = 0xe059,
  [0x1f7b] = 0xe05a, [0x1f7c] = 0xe05b, [0x1f7d] = 0xe05c, [0x1f7e] = 0xe05d,
  [0x1f80] = 0x7372, [0x1f81] = 0x7373, [0x1f82] = 0x7374, [0x1f83] = 0x7375,
  [0x1f84] = 0x7376, [0x1f85] = 0x7377, [0x1f86] = 0x7378, [0x1f87] = 0x7379,
  [0x1f88] = 0x737a, [0x1f89] = 0x737b, [0x1f8a] = 0x737c, [0x1f8b] = 0x737d,
  [0x1f8c] = 0x737f, [0x1f8d] = 0x7380, [0x1f8e] = 0x7381, [0x1f8f] = 0x7382,
  [0x1f90] = 0x7383, [0x1f91] = 0x7385, [0x1f92] = 0x7386, [0x1f93] = 0x7388,
  [0x1f94] = 0x738a, [0x1f95] = 0x738c, [0x1f96] = 0x738d, [0x1f97] = 0x738f,
  [0x1f98] = 0x7390, [0x1f99] = 0x7392, [0x1f9a] = 0x7393, [0x1f9b] = 0x7394,
  [0x1f9c] = 0x7395, [0x1f9d] = 0x7397, [0x1f9e] = 0x7398, [0x1f9f] = 0x7399,
  [0x1fa0] = 0x739a, [0x1fa1] = 0x739c, [0x1fa2] = 0x739d, [0x1fa3] = 0x739e,
  [0x1fa4] = 0x73a0, [0x1fa5] = 0x73a1, [0x1fa6] = 0x73a3, [0x1fa7] = 0x73a4,
  [0x1fa8] = 0x73a5, [0x1fa9] = 0x73a6, [0x1faa] = 0x73a7, [0x1fab] = 0x73a8,
  [0x1fac] = 0x73aa, [0x1fad] = 0x73ac, [0x1fae] = 0x73ad, [0x1faf] = 0x73b1,
  [0x1fb0] = 0x73b4, [0x1fb1] = 0x73b5, [0x1fb2] = 0x73b6, [0x1fb3] = 0x73b8,
  [0x1fb4] = 0x73b9, [0x1fb5] = 0x73bc, [0x1fb6] = 0x73bd, [0x1fb7] = 0x73be,
  [0x1fb8] = 0x73bf, [0x1fb9] = 0x73c1, [0x1fba] = 0x73c3, [0x1fbb] = 0x73c4,
  [0x1fbc] = 0x73c5, [0x1fbd] = 0x73c6, [0x1fbe] = 0x73c7, [0x1fc0] = 0x73cb,
  [0x1fc1] = 0x73cc, [0x1fc2] = 0x73ce, [0x1fc3] = 0x73d2, [0x1fc4] = 0x73d3,
  [0x1fc5] = 0x73d4, [0x1fc6] = 0x73d5, [0x1fc7] = 0x73d6, [0x1fc8] = 0x73d7,
  [0x1fc9] = 0x73d8, [0x1fca] = 0x73da, [0x1fcb] = 0x73db, [0x1fcc] = 0x73dc,
  [0x1fcd] = 0x73dd, [0x1fce] = 0x73df, [0x1fcf] = 0x73e1, [0x1fd0] = 0x73e2,
  [0x1fd1] = 0x73e3, [0x1fd2] = 0x73e4, [0x1fd3] = 0x73e6, [0x1fd4] = 0x73e8,
  [0x1fd5] = 0x73ea, [0x1fd6] = 0x73eb, [0x1fd7] = 0x73ec, [0x1fd8] = 0x73ee,
  [0x1fd9] = 0x73ef, [0x1fda] = 0x73f0, [0x1fdb] = 0x73f1, [0x1fdc] = 0x73f3,
  [0x1fdd] = 0x73f4, [0x1fde] = 0x73f5, [0x1fdf] = 0x73f6, [0x1fe0] = 0x73f7,
  [0x1fe1] = 0xe05e, [0x1fe2] = 0xe05f, [0x1fe3] = 0xe060, [0x1fe4] = 0xe061,
  [0x1fe5] = 0xe062, [0x1fe6] = 0xe063, [0x1fe7] = 0xe064, [0x1fe8] = 0xe065,
  [0x1fe9] = 0xe066, [0x1fea] = 0xe067, [0x1feb] = 0xe068, [0x1fec] = 0xe069,
  [0x1fed] = 0xe06a, [0x1fee] = 0xe06b, [0x1fef] = 0xe06c, [0x1ff0] = 0xe06d,
  [0x1ff1] = 0xe06e, [0x1ff2] = 0xe06f, [0x1ff3] = 0xe070, [0x1ff4] = 0xe071,
  [0x1ff5] = 0xe072, [0x1ff6] = 0xe073, [0x1ff7] = 0xe074, [0x1ff8] = 0xe075,
  [0x1ff9] = 0xe076, [0x1ffa] = 0xe077, [0x1ffb] = 0xe078, [0x1ffc] = 0xe079,
  [0x1ffd] = 0xe07a, [0x1ffe] = 0xe07b, [0x1fff] = 0xe07c, [0x2000] = 0xe07d,
  [0x2001] = 0xe07e, [0x2002] = 0xe07f, [0x2003] = 0xe080, [0x2004] = 0xe081,
  [0x2005] = 0xe082, [0x2006] = 0xe083, [0x2007] = 0xe084, [0x2008] = 0xe085,
  [0x2009] = 0xe086, [0x200a] = 0xe087, [0x200b] = 0xe088, [0x200c] = 0xe089,
  [0x200d] = 0xe08a, [0x200e] = 0xe08b, [0x200f] = 0xe08c, [0x2010] = 0xe08d,
  [0x2011] = 0xe08e, [0x2012] = 0xe08f, [0x2013] = 0xe090, [0x2014] = 0xe091,
  [0x2015] = 0xe092, [0x2016] = 0xe093, [0x2017] = 0xe094, [0x2018] = 0xe095,
  [0x2019] = 0xe096, [0x201a] = 0xe097, [0x201b] = 0xe098, [0x201c] = 0xe099,
  [0x201d] = 0xe09a, [0x201e] = 0xe09b, [0x201f] = 0xe09c, [0x2020] = 0xe09d,
  [0x2021] = 0xe09e, [0x2022] = 0xe09f, [0x2023] = 0xe0a0, [0x2024] = 0xe0a1,
  [0x2025] = 0xe0a2, [0x2026] = 0xe0a3, [0x2027] = 0xe0a4, [0x2028] = 0xe0a5,
  [0x2029] = 0xe0a6, [0x202a] = 0xe0a7, [0x202b] = 0xe0a8, [0x202c] = 0xe0a9,
  [0x202d] = 0xe0aa, [0x202e] = 0xe0ab, [0x202f] = 0xe0ac, [0x2030] = 0xe0ad,
  [0x2031] = 0xe0ae, [0x2032] = 0xe0af, [0x2033] = 0xe0b0, [0x2034] = 0xe0b1,
  [0x2035] = 0xe0b2, [0x2036] = 0xe0b3, [0x2037] = 0xe0b4, [0x2038] = 0xe0b5,
  [0x2039] = 0xe0b6, [0x203a] = 0xe0b7, [0x203b] = 0xe0b8, [0x203c] = 0xe0b9,
  [0x203d] = 0xe0ba, [0x203e] = 0xe0bb, [0x2040] = 0x73f8, [0x2041] = 0x73f9,
  [0x2042] = 0x73fa, [0x2043] = 0x73fb, [0x2044] = 0x73fc, [0x2045] = 0x73fd,
  [0x2046] = 0x73fe, [0x2047] = 0x73ff, [0x2048] = 0x7400, [0x2049] = 0x7401,
  [0x204a] = 0x7402, [0x204b] = 0x7404, [0x204c] = 0x7407, [0x204d] = 0x7408,
  [0x204e] = 0x740b, [0x204f] = 0x740c, [0x2050] = 0x740d, [0x2051] = 0x740e,
  [0x2052] = 0x7411, [0x2053] = 0x7412, [0x2054] = 0x7413, [0x2055] = 0x7414,
  [0x2056] = 0x7415, [0x2057] = 0x7416, [0x2058] = 0x7417, [0x2059] = 0x7418,
  [0x205a] = 0x7419, [0x205b] = 0x741c, [0x205c] = 0x741d, [0x205d] = 0x741e,
  [0x205e] = 0x741f, [0x205f] = 0x7420, [0x2060] = 0x7421, [0x2061] = 0x7423,
  [0x2062] = 0x7424, [0x2063] = 0x7427, [0x2064] = 0x7429, [0x2065] = 0x742b,
  [0x2066] = 0x742d, [0x2067] = 0x742f, [0x2068] = 0x7431, [0x2069] = 0x7432,
  [0x206a] = 0x7437, [0x206b] = 0x7438, [0x206c] = 0x7439, [0x206d] = 0x743a,
  [0x206e] = 0x743b, [0x206f] = 0x743d, [0x2070] = 0x743e, [0x2071] = 0x743f,
  [0x2072] = 0x7440, [0x2073] = 0x7442, [0x2074] = 0x7443, [0x2075] = 0x7444,
  [0x2076] = 0x7445, [0x2077] = 0x7446, [0x2078] = 0x7447, [0x2079] = 0x7448,
  [0x207a] = 0x7449, [0x207b] = 0x744a, [0x207c] = 0x744b, [0x207d] = 0x744c,
  [0x207e] = 0x744d, [0x2080] = 0x744e, [0x2081] = 0x744f, [0x2082] = 0x7450,
  [0x2083] = 0x7451, [0x2084] = 0x7452, [0x2085] = 0x7453, [0x2086] = 0x7454,
  [0x2087] = 0x7456, [0x2088] = 0x7458, [0x2089] = 0x745d, [0x208a] = 0x7460,
  [0x208b] = 0x7461, [0x208c] = 0x7462, [0x208d] = 0x7463, [0x208e] = 0x7464,
  [0x208f] = 0x7465, [0x2090] = 0x7466, [0x2091] = 0x7467, [0x2092] = 0x7468,
  [0x2093] = 0x7469, [0x2094] = 0x746a, [0x2095] = 0x746b, [0x2096] = 0x746c,
  [0x2097] = 0x746e, [0x2098] = 0x746f, [0x2099] = 0x7471, [0x209a] = 0x7472,
  [0x209b] = 0x7473, [0x209c] = 0x7474, [0x209d] = 0x7475, [0x209e] = 0x7478,
  [0x209f] = 0x7479, [0x20a0] = 0x747a, [0x20a1] = 0xe0bc, [0x20a2] = 0xe0bd,
  [0x20a3] = 0xe0be, [0x20a4] = 0xe0bf, [0x20a5] = 0xe0c0, [0x20a6] = 0xe0c1,
  [0x20a7] = 0xe0c2, [0x20a8] = 0xe0c3, [0x20a9] = 0xe0c4, [0x20aa] = 0xe0c5,
  [0x20ab] = 0xe0c6, [0x20ac] = 0xe0c7, [0x20ad] = 0xe0c8, [0x20ae] = 0xe0c9,
  [0x20af] = 0xe0ca, [0x20b0] = 0xe0cb, [0x20b1] = 0xe0cc, [0x20b2] = 0xe0cd,
  [0x20b3] = 0xe0ce, [0x20b4] = 0xe0cf, [0x20b5] = 0xe0d0, [0x20b6] = 0xe0d1,
  [0x20b7] = 0xe0d2, [0x20b8] = 0xe0d3, [0x20b9] = 0xe0d4, [0x20ba] = 0xe0d5,
  [0x20bb] = 0xe0d6, [0x20bc] = 0xe0d7, [0x20bd] = 0xe0d8, [0x20be] = 0xe0d9,
  [0x20bf] = 0xe0da, [0x20c0] = 0xe0db, [0x20c1] = 0xe0dc, [0x20c2] = 0xe0dd,
  [0x20c3] = 0xe0de, [0x20c4] = 0xe0df, [0x20c5] = 0xe0e0, [0x20c6] = 0xe0e1,
  [0x20c7] = 0xe0e2, [0x20c8] = 0xe0e3, [0x20c9] = 0xe0e4, [0x20ca] = 0xe0e5,
  [0x20cb] = 0xe0e6, [0x20cc] = 0xe0e7, [0x20cd] = 0xe0e8, [0x20ce] = 0xe0e9,
  [0x20cf] = 0xe0ea, [0x20d0] = 0xe0eb, [0x20d1] = 0xe0ec, [0x20d2] = 0xe0ed,
  [0x20d3] = 0xe0ee, [0x20d4] = 0xe0ef, [0x20d5] = 0xe0f0, [0x20d6] = 0xe0f1,
  [0x20d7] = 0xe0f2, [0x20d8] = 0xe0f3, [0x20d9] = 0xe0f4, [0x20da] = 0xe0f5,
  [0x20db] = 0xe0f6, [0x20dc] = 0xe0f7, [0x20dd] = 0xe0f8, [0x20de] = 0xe0f9,
  [0x20df] = 0xe0fa, [0x20e0] = 0xe0fb, [0x20e1] = 0xe0fc, [0x20e2] = 0xe0fd,
  [0x20e3] = 0xe0fe, [0x20e4] = 0xe0ff, [0x20e5] = 0xe100, [0x20e6] = 0xe101,
  [0x20e7] = 0xe102, [0x20e8] = 0xe103, [0x20e9] = 0xe104, [0x20ea] = 0xe105,
  [0x20eb] = 0xe106, [0x20ec] = 0xe107, [0x20ed] = 0xe108, [0x20ee] = 0xe109,
  [0x20ef] = 0xe10a, [0x20f0] = 0xe10b, [0x20f1] = 0xe10c, [0x20f2] = 0xe10d,
  [0x20f3] = 0xe10e, [0x20f4] = 0xe10f, [0x20f5] = 0xe110, [0x20f6] = 0xe111,
  [0x20f7] = 0xe112, [0x20f8] = 0xe113, [0x20f9] = 0xe114, [0x20fa] = 0xe115,
  [0x20fb] = 0xe116, [0x20fc] = 0xe117, [0x20fd] = 0xe118, [0x20fe] = 0xe119,
  [0x2100] = 0x747b, [0x2101] = 0x747c, [0x2102] = 0x747d, [0x2103] = 0x747f,
  [0x2104] = 0x7482, [0x2105] = 0x7484, [0x2106] = 0x7485, [0x2107] = 0x7486,
  [0x2108] = 0x7488, [0x2109] = 0x7489, [0x210a] = 0x748a, [0x210b] = 0x748c,
  [0x210c] = 0x748d, [0x210d] = 0x748f, [0x210e] = 0x7491, [0x210f] = 0x7492,
  [0x2110] = 0x7493, [0x2111] = 0x7494, [0x2112] = 0x7495, [0x2113] = 0x7496,
  [0x2114] = 0x7497, [0x2115] = 0x7498, [0x2116] = 0x7499, [0x2117] = 0x749a,
  [0x2118] = 0x749b, [0x2119] = 0x749d, [0x211a] = 0x749f, [0x211b] = 0x74a0,
  [0x211c] = 0x74a1, [0x211d] = 0x74a2, [0x211e] = 0x74a3, [0x211f] = 0x74a4,
  [0x2120] = 0x74a5, [0x2121] = 0x74a6, [0x2122] = 0x74aa, [0x2123] = 0x74ab,
  [0x2124] = 0x74ac, [0x2125] = 0x74ad, [0x2126] = 0x74ae, [0x2127] = 0x74af,
  [0x2128] = 0x74b0, [0x2129] = 0x74b1, [0x212a] = 0x74b2, [0x212b] = 0x74b3,
  [0x212c] = 0x74b4, [0x212d] = 0x74b5, [0x212e] = 0x74b6, [0x212f] = 0x74b7,
  [0x2130] = 0x74b8, [0x2131] = 0x74b9, [0x2132] = 0x74bb, [0x2133] = 0x74bc,
  [0x2134] = 0x74bd, [0x2135] = 0x74be, [0x2136] = 0x74bf, [0x2137] = 0x74c0,
  [0x2138] = 0x74c1, [0x2139] = 0x74c2, [0x213a] = 0x74c3, [0x213b] = 0x74c4,
  [0x213c] = 0x74c5, [0x213d] = 0x74c6, [0x213e] = 0x74c7, [0x2140] = 0x74c8,
  [0x2141] = 0x74c9, [0x2142] = 0x74ca, [0x2143] = 0x74cb, [0x2144] = 0x74cc,
  [0x2145] = 0x74cd, [0x2146] = 0x74ce, [0x2147] = 0x74cf, [0x2148] = 0x74d0,
  [0x2149] = 0x74d1, [0x214a] = 0x74d3, [0x214b] = 0x74d4, [0x214c] = 0x74d5,
  [0x214d] = 0x74d6, [0x214e] = 0x74d7, [0x214f] = 0x74d8, [0x2150] = 0x74d9,
  [0x2151] = 0x74da, [0x2152] = 0x74db, [0x2153] = 0x74dd, [0x2154] = 0x74df,
  [0x2155] = 0x74e1, [0x2156] = 0x74e5, [0x2157] = 0x74e7, [0x2158] = 0x74e8,
  [0x2159] = 0x74e9, [0x215a] = 0x74ea, [0x215b] = 0x74eb, [0x215c] = 0x74ec,
  [0x215d] = 0x74ed, [0x215e] = 0x74f0, [0x215f] = 0x74f1, [0x2160] = 0x74f2,
  [0x2161] = 0xe11a, [0x2162] = 0xe11b, [0x2163] = 0xe11c, [0x2164] = 0xe11d,
  [0x2165] = 0xe11e, [0x2166] = 0xe11f, [0x2167] = 0xe120, [0x2168] = 0xe121,
  [0x2169] = 0xe122, [0x216a] = 0xe123, [0x216b] = 0xe124, [0x216c] = 0xe125,
  [0x216d] = 0xe126, [0x216e] = 0xe127, [0x216f] = 0xe128, [0x2170] = 0xe129,
  [0x2171] = 0xe12a, [0x2172] = 0xe12b, [0x2173] = 0xe12c, [0x2174] = 0xe12d,
  [0x2175] = 0xe12e, [0x2176] = 0xe12f, [0x2177] = 0xe130, [0x2178] = 0xe131,
  [0x2179] = 0xe132, [0x217a] = 0xe133, [0x217b] = 0xe134, [0x217c] = 0xe135,
  [0x217d] = 0xe136, [0x217e] = 0xe137, [0x217f] = 0xe138, [0x2180] = 0xe139,
  [0x2181] = 0xe13a, [0x2182] = 0xe13b, [0x2183] = 0xe13c, [0x2184] = 0xe13d,
  [0x2185] = 0xe13e, [0x2186] = 0xe13f, [0x2187] = 0xe140, [0x2188] = 0xe141,
  [0x2189] = 0xe142, [0x218a] = 0xe143, [0x218b] = 0xe144, [0x218c] = 0xe145,
  [0x218d] = 0xe146, [0x218e] = 0xe147, [0x218f] = 0xe148, [0x2190] = 0xe149,
  [0x2191] = 0xe14a, [0x2192] = 0xe14b, [0x2193] = 0xe14c, [0x2194] = 0xe14d,
  [0x2195] = 0xe14e, [0x2196] = 0xe14f, [0x2197] = 0xe150, [0x2198] = 0xe151,
  [0x2199] = 0xe152, [0x219a] = 0xe153, [0x219b] = 0xe154, [0x219c] = 0xe155,
  [0x219d] = 0xe156, [0x219e] = 0xe157, [0x219f] = 0xe158, [0x21a0] = 0xe159,
  [0x21a1] = 0xe15a, [0x21a2] = 0xe15b, [0x21a3] = 0xe15c, [0x21a4] = 0xe15d,
  [0x21a5] = 0xe15e, [0x21a6] = 0xe15f, [0x21a7] = 0xe160, [0x21a8] = 0xe161,
  [0x21a9] = 0xe162, [0x21aa] = 0xe163, [0x21ab] = 0xe164, [0x21ac] = 0xe165,
  [0x21ad] = 0xe166, [0x21ae] = 0xe167, [0x21af] = 0xe168, [0x21b0] = 0xe169,
  [0x21b1] = 0xe16a, [0x21b2] = 0xe16b, [0x21b3] = 0xe16c, [0x21b4] = 0xe16d,
  [0x21b5] = 0xe16e, [0x21b6] = 0xe16f, [0x21b7] = 0xe170, [0x21b8] = 0xe171,
  [0x21b9] = 0xe172, [0x21ba] = 0xe173, [0x21bb] = 0xe174, [0x21bc] = 0xe175,
  [0x21bd] = 0xe176, [0x21be] = 0xe177, [0x21c0] = 0x74f3, [0x21c1] = 0x74f5,
  [0x21c2] = 0x74f8, [0x21c3] = 0x74f9, [0x21c4] = 0x74fa, [0x21c5] = 0x74fb,
  [0x21c6] = 0x74fc, [0x21c7] = 0x74fd, [0x21c8] = 0x74fe, [0x21c9] = 0x7500,
  [0x21ca] = 0x7501, [0x21cb] = 0x7502, [0x21cc] = 0x7503, [0x21cd] = 0x7505,
  [0x21ce] = 0x7506, [0x21cf] = 0x7507, [0x21d0] = 0x7508, [0x21d1] = 0x7509,
  [0x21d2] = 0x750a, [0x21d3] = 0x750b, [0x21d4] = 0x750c, [0x21d5] = 0x750e,
  [0x21d6] = 0x7510, [0x21d7] = 0x7512, [0x21d8] = 0x7514, [0x21d9] = 0x7515,
  [0x21da] = 0x7516, [0x21db] = 0x7517, [0x21dc] = 0x751b, [0x21dd] = 0x751d,
  [0x21de] = 0x751e, [0x21df] = 0x7520, [0x21e0] = 0x7521, [0x21e1] = 0x7522,
  [0x21e2] = 0x7523, [0x21e3] = 0x7524, [0x21e4] = 0x7526, [0x21e5] = 0x7527,
  [0x21e6] = 0x752a, [0x21e7] = 0x752e, [0x21e8] = 0x7534, [0x21e9] = 0x7536,
  [0x21ea] = 0x7539, [0x21eb] = 0x753c, [0x21ec] = 0x753d, [0x21ed] = 0x753f,
  [0x21ee] = 0x7541, [0x21ef] = 0x7542, [0x21f0] = 0x7543, [0x21f1] = 0x7544,
  [0x21f2] = 0x7546, [0x21f3] = 0x7547, [0x21f4] = 0x7549, [0x21f5] = 0x754a,
  [0x21f6] = 0x754d, [0x21f7] = 0x7550, [0x21f8] = 0x7551, [0x21f9] = 0x7552,
  [0x21fa] = 0x7553, [0x21fb] = 0x7555, [0x21fc] = 0x7556, [0x21fd] = 0x7557,
  [0x21fe] = 0x7558, [0x2200] = 0x755d, [0x2201] = 0x755e, [0x2202] = 0x755f,
  [0x2203] = 0x7560, [0x2204] = 0x7561, [0x2205] = 0x7562, [0x2206] = 0x7563,
  [0x2207] = 0x7564, [0x2208] = 0x7567, [0x2209] = 0x7568, [0x220a] = 0x7569,
  [0x220b] = 0x756b, [0x220c] = 0x756c, [0x220d] = 0x756d, [0x220e] = 0x756e,
  [0x220f] = 0x756f, [0x2210] = 0x7570, [0x2211] = 0x7571, [0x2212] = 0x7573,
  [0x2213] = 0x7575, [0x2214] = 0x7576, [0x2215] = 0x7577, [0x2216] = 0x757a,
  [0x2217] = 0x757b, [0x2218] = 0x757c, [0x2219] = 0x757d, [0x221a] = 0x757e,
  [0x221b] = 0x7580, [0x221c] = 0x7581, [0x221d] = 0x7582, [0x221e] = 0x7584,
  [0x221f] = 0x7585, [0x2220] = 0x7587, [0x2221] = 0xe178, [0x2222] = 0xe179,
  [0x2223] = 0xe17a, [0x2224] = 0xe17b, [0x2225] = 0xe17c, [0x2226] = 0xe17d,
  [0x2227] = 0xe17e, [0x2228] = 0xe17f, [0x2229] = 0xe180, [0x222a] = 0xe181,
  [0x222b] = 0xe182, [0x222c] = 0xe183, [0x222d] = 0xe184, [0x222e] = 0xe185,
  [0x222f] = 0xe186, [0x2230] = 0xe187, [0x2231] = 0xe188, [0x2232] = 0xe189,
  [0x2233] = 0xe18a, [0x2234] = 0xe18b, [0x2235] = 0xe18c, [0x2236] = 0xe18d,
  [0x2237] = 0xe18e, [0x2238] = 0xe18f, [0x2239] = 0xe190, [0x223a] = 0xe191,
  [0x223b] = 0xe192, [0x223c] = 0xe193, [0x223d] = 0xe194, [0x223e] = 0xe195,
  [0x223f] = 0xe196, [0x2240] = 0xe197, [0x2241] = 0xe198, [0x2242] = 0xe199,
  [0x2243] = 0xe19a, [0x2244] = 0xe19b, [0x2245] = 0xe19c, [0x2246] = 0xe19d,
  [0x2247] = 0xe19e, [0x2248] = 0xe19f, [0x2249] = 0xe1a0, [0x224a] = 0xe1a1,
  [0x224b] = 0xe1a2, [0x224c] = 0xe1a3, [0x224d] = 0xe1a4, [0x224e] = 0xe1a5,
  [0x224f] = 0xe1a6, [0x2250] = 0xe1a7, [0x2251] = 0xe1a8, [0x2252] = 0xe1a9,
  [0x2253] = 0xe1aa, [0x2254] = 0xe1ab, [0x2255] = 0xe1ac, [0x2256] = 0xe1ad,
  [0x2257] = 0xe1ae, [0x2258] = 0xe1af, [0x2259] = 0xe1b0, [0x225a] = 0xe1b1,
  [0x225b] = 0xe1b2, [0x225c] = 0xe1b3, [0x225d] = 0xe1b4, [0x225e] = 0xe1b5,
  [0x225f] = 0xe1b6, [0x2260] = 0xe1b7, [0x2261] = 0xe1b8, [0x2262] = 0xe1b9,
  [0x2263] = 0xe1ba, [0x2264] = 0xe1bb, [0x2265] = 0xe1bc, [0x2266] = 0xe1bd,
  [0x2267] = 0xe1be, [0x2268] = 0xe1bf, [0x2269] = 0xe1c0, [0x226a] = 0xe1c1,
  [0x226b] = 0xe1c2, [0x226c] = 0xe1c3, [0x226d] = 0xe1c4, [0x226e] = 0xe1c5,
  [0x226f] = 0xe1c6, [0x2270] = 0xe1c7, [0x2271] = 0xe1c8, [0x2272] = 0xe1c9,
  [0x2273] = 0xe1ca, [0x2274] = 0xe1cb, [0x2275] = 0xe1cc, [0x2276] = 0xe1cd,
  [0x2277] = 0xe1ce, [0x2278] = 0xe1cf, [0x2279] = 0xe1d0, [0x227a] = 0xe1d1,
  [0x227b] = 0xe1d2, [0x227c] = 0xe1d3, [0x227d] = 0xe1d4, [0x227e] = 0xe1d5,
  [0x2280] = 0x7588, [0x2281] = 0x7589, [0x2282] = 0x758a, [0x2283] = 0x758c,
  [0x2284] = 0x758d, [0x2285] = 0x758e, [0x2286] = 0x7590, [0x2287] = 0x7593,
  [0x2288] = 0x7595, [0x2289] = 0x7598, [0x228a] = 0x759b, [0x228b] = 0x759c,
  [0x228c] = 0x759e, [0x228d] = 0x75a2, [0x228e] = 0x75a6, [0x228f] = 0x75a7,
  [0x2290] = 0x75a8, [0x2291] = 0x75a9, [0x2292] = 0x75aa, [0x2293] = 0x75ad,
  [0x2294] = 0x75b6, [0x2295] = 0x75b7, [0x2296] = 0x75ba, [0x2297] = 0x75bb,
  [0x2298] = 0x75bf, [0x2299] = 0x75c0, [0x229a] = 0x75c1, [0x229b] = 0x75c6,
  [0x229c] = 0x75cb, [0x229d] = 0x75cc, [0x229e] = 0x75ce, [0x229f] = 0x75cf,
  [0x22a0] = 0x75d0, [0x22a1] = 0x75d1, [0x22a2] = 0x75d3, [0x22a3] = 0x75d7,
  [0x22a4] = 0x75d9, [0x22a5] = 0x75da, [0x22a6] = 0x75dc, [0x22a7] = 0x75dd,
  [0x22a8] = 0x75df, [0x22a9] = 0x75e0, [0x22aa] = 0x75e1, [0x22ab] = 0x75e5,
  [0x22ac] = 0x75e9, [0x22ad] = 0x75ec, [0x22ae] = 0x75ed, [0x22af] = 0x75ee,
  [0x22b0] = 0x75ef, [0x22b1] = 0x75f2, [0x22b2] = 0x75f3, [0x22b3] = 0x75f5,
  [0x22b4] = 0x75f6, [0x22b5] = 0x75f7, [0x22b6] = 0x75f8, [0x22b7] = 0x75fa,
  [0x22b8] = 0x75fb, [0x22b9] = 0x75fd, [0x22ba] = 0x75fe, [0x22bb] = 0x7602,
  [0x22bc] = 0x7604, [0x22bd] = 0x7606, [0x22be] = 0x7607, [0x22c0] = 0x7608,
  [0x22c1] = 0x7609, [0x22c2] = 0x760b, [0x22c3] = 0x760d, [0x22c4] = 0x760e,
  [0x22c5] = 0x760f, [0x22c6] = 0x7611, [0x22c7] = 0x7612, [0x22c8] = 0x7613,
  [0x22c9] = 0x7614, [0x22ca] = 0x7616, [0x22cb] = 0x761a, [0x22cc] = 0x761c,
  [0x22cd] = 0x761d, [0x22ce] = 0x761e, [0x22cf] = 0x7621, [0x22d0] = 0x7623,
  [0x22d1] = 0x7627, [0x22d2] = 0x7628, [0x22d3] = 0x762c, [0x22d4] = 0x762e,
  [0x22d5] = 0x762f, [0x22d6] = 0x7631, [0x22d7] = 0x7632, [0x22d8] = 0x7636,
  [0x22d9] = 0x7637, [0x22da] = 0x7639, [0x22db] = 0x763a, [0x22dc] = 0x763b,
  [0x22dd] = 0x763d, [0x22de] = 0x7641, [0x22df] = 0x7642, [0x22e0] = 0x7644,
  [0x22e1] = 0xe1d6, [0x22e2] = 0xe1d7, [0x22e3] = 0xe1d8, [0x22e4] = 0xe1d9,
  [0x22e5] = 0xe1da, [0x22e6] = 0xe1db, [0x22e7] = 0xe1dc, [0x22e8] = 0xe1dd,
  [0x22e9] = 0xe1de, [0x22ea] = 0xe1df, [0x22eb] = 0xe1e0, [0x22ec] = 0xe1e1,
  [0x22ed] = 0xe1e2, [0x22ee] = 0xe1e3, [0x22ef] = 0xe1e4, [0x22f0] = 0xe1e5,
  [0x22f1] = 0xe1e6, [0x22f2] = 0xe1e7, [0x22f3] = 0xe1e8, [0x22f4] = 0xe1e9,
  [0x22f5] = 0xe1ea, [0x22f6] = 0xe1eb, [0x22f7] = 0xe1ec, [0x22f8] = 0xe1ed,
  [0x22f9] = 0xe1ee, [0x22fa] = 0xe1ef, [0x22fb] = 0xe1f0, [0x22fc] = 0xe1f1,
  [0x22fd] = 0xe1f2, [0x22fe] = 0xe1f3, [0x22ff] = 0xe1f4, [0x2300] = 0xe1f5,
  [0x2301] = 0xe1f6, [0x2302] = 0xe1f7, [0x2303] = 0xe1f8, [0x2304] = 0xe1f9,
  [0x2305] = 0xe1fa, [0x2306] = 0xe1fb, [0x2307] = 0xe1fc, [0x2308] = 0xe1fd,
  [0x2309] = 0xe1fe, [0x230a] = 0xe1ff, [0x230b] = 0xe200, [0x230c] = 0xe201,
  [0x230d] = 0xe202, [0x230e] = 0xe203, [0x230f] = 0xe204, [0x2310] = 0xe205,
  [0x2311] = 0xe206, [0x2312] = 0xe207, [0x2313] = 0xe208, [0x2314] = 0xe209,
  [0x2315] = 0xe20a, [0x2316] = 0xe20b, [0x2317] = 0xe20c, [0x2318] = 0xe20d,
  [0x2319] = 0xe20e, [0x231a] = 0xe20f, [0x231b] = 0xe210, [0x231c] = 0xe211,
  [0x231d] = 0xe212, [0x231e] = 0xe213, [0x231f] = 0xe214, [0x2320] = 0xe215,
  [0x2321] = 0xe216, [0x2322] = 0xe217, [0x2323] = 0xe218, [0x2324] = 0xe219,
  [0x2325] = 0xe21a, [0x2326] = 0xe21b, [0x2327] = 0xe21c, [0x2328] = 0xe21d,
  [0x2329] = 0xe21e, [0x232a] = 0xe21f, [0x232b] = 0xe220, [0x232c] = 0xe221,
  [0x232d] = 0xe222, [0x232e] = 0xe223, [0x232f] = 0xe224, [0x2330] = 0xe225,
  [0x2331] = 0xe226, [0x2332] = 0xe227, [0x2333] = 0xe228, [0x2334] = 0xe229,
  [0x2335] = 0xe22a, [0x2336] = 0xe22b, [0x2337] = 0xe22c, [0x2338] = 0xe22d,
  [0x2339] = 0xe22e, [0x233a] = 0xe22f, [0x233b] = 0xe230, [0x233c] = 0xe231,
  [0x233d] = 0xe232, [0x233e] = 0xe233, [0x2340] = 0x7645, [0x2341] = 0x7646,
  [0x2342] = 0x7647, [0x2343] = 0x7648, [0x2344] = 0x7649, [0x2345] = 0x764a,
  [0x2346] = 0x764b, [0x2347] = 0x764e, [0x2348] = 0x764f, [0x2349] = 0x7650,
  [0x234a] = 0x7651, [0x234b] = 0x7652, [0x234c] = 0x7653, [0x234d] = 0x7655,
  [0x234e] = 0x7657, [0x234f] = 0x7658, [0x2350] = 0x7659, [0x2351] = 0x765a,
  [0x2352] = 0x765b, [0x2353] = 0x765d, [0x2354] = 0x765f, [0x2355] = 0x7660,
  [0x2356] = 0x7661, [0x2357] = 0x7662, [0x2358] = 0x7664, [0x2359] = 0x7665,
  [0x235a] = 0x7666, [0x235b] = 0x7667, [0x235c] = 0x7668, [0x235d] = 0x7669,
  [0x235e] = 0x766a, [0x235f] = 0x766c, [0x2360] = 0x766d, [0x2361] = 0x766e,
  [0x2362] = 0x7670, [0x2363] = 0x7671, [0x2364] = 0x7672, [0x2365] = 0x7673,
  [0x2366] = 0x7674, [0x2367] = 0x7675, [0x2368] = 0x7676, [0x2369] = 0x7677,
  [0x236a] = 0x7679, [0x236b] = 0x767a, [0x236c] = 0x767c, [0x236d] = 0x767f,
  [0x236e] = 0x7680, [0x236f] = 0x7681, [0x2370] = 0x7683, [0x2371] = 0x7685,
  [0x2372] = 0x7689, [0x2373] = 0x768a, [0x2374] = 0x768c, [0x2375] = 0x768d,
  [0x2376] = 0x768f, [0x2377] = 0x7690, [0x2378] = 0x7692, [0x2379] = 0x7694,
  [0x237a] = 0x7695, [0x237b] = 0x7697, [0x237c] = 0x7698, [0x237d] = 0x769a,
  [0x237e] = 0x769b, [0x2380] = 0x769c, [0x2381] = 0x769d, [0x2382] = 0x769e,
  [0x2383] = 0x769f, [0x2384] = 0x76a0, [0x2385] = 0x76a1, [0x2386] = 0x76a2,
  [0x2387] = 0x76a3, [0x2388] = 0x76a5, [0x2389] = 0x76a6, [0x238a] = 0x76a7,
  [0x238b] = 0x76a8, [0x238c] = 0x76a9, [0x238d] = 0x76aa, [0x238e] = 0x76ab,
  [0x238f] = 0x76ac, [0x2390] = 0x76ad, [0x2391] = 0x76af, [0x2392] = 0x76b0,
  [0x2393] = 0x76b3, [0x2394] = 0x76b5, [0x2395] = 0x76b6, [0x2396] = 0x76b7,
  [0x2397] = 0x76b8, [0x2398] = 0x76b9, [0x2399] = 0x76ba, [0x239a] = 0x76bb,
  [0x239b] = 0x76bc, [0x239c] = 0x76bd, [0x239d] = 0x76be, [0x239e] = 0x76c0,
  [0x239f] = 0x76c1, [0x23a0] = 0x76c3, [0x23a1] = 0x554a, [0x23a2] = 0x963f,
  [0x23a3] = 0x57c3, [0x23a4] = 0x6328, [0x23a5] = 0x54ce, [0x23a6] = 0x5509,
  [0x23a7] = 0x54c0, [0x23a8] = 0x7691, [0x23a9] = 0x764c, [0x23aa] = 0x853c,
  [0x23ab] = 0x77ee, [0x23ac] = 0x827e, [0x23ad] = 0x788d, [0x23ae] = 0x7231,
  [0x23af] = 0x9698, [0x23b0] = 0x978d, [0x23b1] = 0x6c28, [0x23b2] = 0x5b89,
  [0x23b3] = 0x4ffa, [0x23b4] = 0x6309, [0x23b5] = 0x6697, [0x23b6] = 0x5cb8,
  [0x23b7] = 0x80fa, [0x23b8] = 0x6848, [0x23b9] = 0x80ae, [0x23ba] = 0x6602,
  [0x23bb] = 0x76ce, [0x23bc] = 0x51f9, [0x23bd] = 0x6556, [0x23be] = 0x71ac,
  [0x23bf] = 0x7ff1, [0x23c0] = 0x8884, [0x23c1] = 0x50b2, [0x23c2] = 0x5965,
  [0x23c3] = 0x61ca, [0x23c4] = 0x6fb3, [0x23c5] = 0x82ad, [0x23c6] = 0x634c,
  [0x23c7] = 0x6252, [0x23c8] = 0x53ed, [0x23c9] = 0x5427, [0x23ca] = 0x7b06,
  [0x23cb] = 0x516b, [0x23cc] = 0x75a4, [0x23cd] = 0x5df4, [0x23ce] = 0x62d4,
  [0x23cf] = 0x8dcb, [0x23d0] = 0x9776, [0x23d1] = 0x628a, [0x23d2] = 0x8019,
  [0x23d3] = 0x575d, [0x23d4] = 0x9738, [0x23d5] = 0x7f62, [0x23d6] = 0x7238,
  [0x23d7] = 0x767d, [0x23d8] = 0x67cf, [0x23d9] = 0x767e, [0x23da] = 0x6446,
  [0x23db] = 0x4f70, [0x23dc] = 0x8d25, [0x23dd] = 0x62dc, [0x23de] = 0x7a17,
  [0x23df] = 0x6591, [0x23e0] = 0x73ed, [0x23e1] = 0x642c, [0x23e2] = 0x6273,
  [0x23e3] = 0x822c, [0x23e4] = 0x9881, [0x23e5] = 0x677f, [0x23e6] = 0x7248,
  [0x23e7] = 0x626e, [0x23e8] = 0x62cc, [0x23e9] = 0x4f34, [0x23ea] = 0x74e3,
  [0x23eb] = 0x534a, [0x23ec] = 0x529e, [0x23ed] = 0x7eca, [0x23ee] = 0x90a6,
  [0x23ef] = 0x5e2e, [0x23f0] = 0x6886, [0x23f1] = 0x699c, [0x23f2] = 0x8180,
  [0x23f3] = 0x7ed1, [0x23f4] = 0x68d2, [0x23f5] = 0x78c5, [0x23f6] = 0x868c,
  [0x23f7] = 0x9551, [0x23f8] = 0x508d, [0x23f9] = 0x8c24, [0x23fa] = 0x82de,
  [0x23fb] = 0x80de, [0x23fc] = 0x5305, [0x23fd] = 0x8912, [0x23fe] = 0x5265,
  [0x2400] = 0x76c4, [0x2401] = 0x76c7, [0x2402] = 0x76c9, [0x2403] = 0x76cb,
  [0x2404] = 0x76cc, [0x2405] = 0x76d3, [0x2406] = 0x76d5, [0x2407] = 0x76d9,
  [0x2408] = 0x76da, [0x2409] = 0x76dc, [0x240a] = 0x76dd, [0x240b] = 0x76de,
  [0x240c] = 0x76e0, [0x240d] = 0x76e1, [0x240e] = 0x76e2, [0x240f] = 0x76e3,
  [0x2410] = 0x76e4, [0x2411] = 0x76e6, [0x2412] = 0x76e7, [0x2413] = 0x76e8,
  [0x2414] = 0x76e9, [0x2415] = 0x76ea, [0x2416] = 0x76eb, [0x2417] = 0x76ec,
  [0x2418] = 0x76ed, [0x2419] = 0x76f0, [0x241a] = 0x76f3, [0x241b] = 0x76f5,
  [0x241c] = 0x76f6, [0x241d] = 0x76f7, [0x241e] = 0x76fa, [0x241f] = 0x76fb,
  [0x2420] = 0x76fd, [0x2421] = 0x76ff, [0x2422] = 0x7700, [0x2423] = 0x7702,
  [0x2424] = 0x7703, [0x2425] = 0x7705, [0x2426] = 0x7706, [0x2427] = 0x770a,
  [0x2428] = 0x770c, [0x2429] = 0x770e, [0x242a] = 0x770f, [0x242b] = 0x7710,
  [0x242c] = 0x7711, [0x242d] = 0x7712, [0x242e] = 0x7713, [0x242f] = 0x7714,
  [0x2430] = 0x7715, [0x2431] = 0x7716, [0x2432] = 0x7717, [0x2433] = 0x7718,
  [0x2434] = 0x771b, [0x2435] = 0x771c, [0x2436] = 0x771d, [0x2437] = 0x771e,
  [0x2438] = 0x7721, [0x2439] = 0x7723, [0x243a] = 0x7724, [0x243b] = 0x7725,
  [0x243c] = 0x7727, [0x243d] = 0x772a, [0x243e] = 0x772b, [0x2440] = 0x772c,
  [0x2441] = 0x772e, [0x2442] = 0x7730, [0x2443] = 0x7731, [0x2444] = 0x7732,
  [0x2445] = 0x7733, [0x2446] = 0x7734, [0x2447] = 0x7739, [0x2448] = 0x773b,
  [0x2449] = 0x773d, [0x244a] = 0x773e, [0x244b] = 0x773f, [0x244c] = 0x7742,
  [0x244d] = 0x7744, [0x244e] = 0x7745, [0x244f] = 0x7746, [0x2450] = 0x7748,
  [0x2451] = 0x7749, [0x2452] = 0x774a, [0x2453] = 0x774b, [0x2454] = 0x774c,
  [0x2455] = 0x774d, [0x2456] = 0x774e, [0x2457] = 0x774f, [0x2458] = 0x7752,
  [0x2459] = 0x7753, [0x245a] = 0x7754, [0x245b] = 0x7755, [0x245c] = 0x7756,
  [0x245d] = 0x7757, [0x245e] = 0x7758, [0x245f] = 0x7759, [0x2460] = 0x775c,
  [0x2461] = 0x8584, [0x2462] = 0x96f9, [0x2463] = 0x4fdd, [0x2464] = 0x5821,
  [0x2465] = 0x9971, [0x2466] = 0x5b9d, [0x2467] = 0x62b1, [0x2468] = 0x62a5,
  [0x2469] = 0x66b4, [0x246a] = 0x8c79, [0x246b] = 0x9c8d, [0x246c] = 0x7206,
  [0x246d] = 0x676f, [0x246e] = 0x7891, [0x246f] = 0x60b2, [0x2470] = 0x5351,
  [0x2471] = 0x5317, [0x2472] = 0x8f88, [0x2473] = 0x80cc, [0x2474] = 0x8d1d,
  [0x2475] = 0x94a1, [0x2476] = 0x500d, [0x2477] = 0x72c8, [0x2478] = 0x5907,
  [0x2479] = 0x60eb, [0x247a] = 0x7119, [0x247b] = 0x88ab, [0x247c] = 0x5954,
  [0x247d] = 0x82ef, [0x247e] = 0x672c, [0x247f] = 0x7b28, [0x2480] = 0x5d29,
  [0x2481] = 0x7ef7, [0x2482] = 0x752d, [0x2483] = 0x6cf5, [0x2484] = 0x8e66,
  [0x2485] = 0x8ff8, [0x2486] = 0x903c, [0x2487] = 0x9f3b, [0x2488] = 0x6bd4,
  [0x2489] = 0x9119, [0x248a] = 0x7b14, [0x248b] = 0x5f7c, [0x248c] = 0x78a7,
  [0x248d] = 0x84d6, [0x248e] = 0x853d, [0x248f] = 0x6bd5, [0x2490] = 0x6bd9,
  [0x2491] = 0x6bd6, [0x2492] = 0x5e01, [0x2493] = 0x5e87, [0x2494] = 0x75f9,
  [0x2495] = 0x95ed, [0x2496] = 0x655d, [0x2497] = 0x5f0a, [0x2498] = 0x5fc5,
  [0x2499] = 0x8f9f, [0x249a] = 0x58c1, [0x249b] = 0x81c2, [0x249c] = 0x907f,
  [0x249d] = 0x965b, [0x249e] = 0x97ad, [0x249f] = 0x8fb9, [0x24a0] = 0x7f16,
  [0x24a1] = 0x8d2c, [0x24a2] = 0x6241, [0x24a3] = 0x4fbf, [0x24a4] = 0x53d8,
  [0x24a5] = 0x535e, [0x24a6] = 0x8fa8, [0x24a7] = 0x8fa9, [0x24a8] = 0x8fab,
  [0x24a9] = 0x904d, [0x24aa] = 0x6807, [0x24ab] = 0x5f6a, [0x24ac] = 0x8198,
  [0x24ad] = 0x8868, [0x24ae] = 0x9cd6, [0x24af] = 0x618b, [0x24b0] = 0x522b,
  [0x24b1] = 0x762a, [0x24b2] = 0x5f6c, [0x24b3] = 0x658c, [0x24b4] = 0x6fd2,
  [0x24b5] = 0x6ee8, [0x24b6] = 0x5bbe, [0x24b7] = 0x6448, [0x24b8] = 0x5175,
  [0x24b9] = 0x51b0, [0x24ba] = 0x67c4, [0x24bb] = 0x4e19, [0x24bc] = 0x79c9,
  [0x24bd] = 0x997c, [0x24be] = 0x70b3, [0x24c0] = 0x775d, [0x24c1] = 0x775e,
  [0x24c2] = 0x775f, [0x24c3] = 0x7760, [0x24c4] = 0x7764, [0x24c5] = 0x7767,
  [0x24c6] = 0x7769, [0x24c7] = 0x776a, [0x24c8] = 0x776d, [0x24c9] = 0x776e,
  [0x24ca] = 0x776f, [0x24cb] = 0x7770, [0x24cc] = 0x7771, [0x24cd] = 0x7772,
  [0x24ce] = 0x7773, [0x24cf] = 0x7774, [0x24d0] = 0x7775, [0x24d1] = 0x7776,
  [0x24d2] = 0x7777, [0x24d3] = 0x7778, [0x24d4] = 0x777a, [0x24d5] = 0x777b,
  [0x24d6] = 0x777c, [0x24d7] = 0x7781, [0x24d8] = 0x7782, [0x24d9] = 0x7783,
  [0x24da] = 0x7786, [0x24db] = 0x7787, [0x24dc] = 0x7788, [0x24dd] = 0x7789,
  [0x24de] = 0x778a, [0x24df] = 0x778b, [0x24e0] = 0x778f, [0x24e1] = 0x7790,
  [0x24e2] = 0x7793, [0x24e3] = 0x7794, [0x24e4] = 0x7795, [0x24e5] = 0x7796,
  [0x24e6] = 0x7797, [0x24e7] = 0x7798, [0x24e8] = 0x7799, [0x24e9] = 0x779a,
  [0x24ea] = 0x779b, [0x24eb] = 0x779c, [0x24ec] = 0x779d, [0x24ed] = 0x779e,
  [0x24ee] = 0x77a1, [0x24ef] = 0x77a3, [0x24f0] = 0x77a4, [0x24f1] = 0x77a6,
  [0x24f2] = 0x77a8, [0x24f3] = 0x77ab, [0x24f4] = 0x77ad, [0x24f5] = 0x77ae,
  [0x24f6] = 0x77af, [0x24f7] = 0x77b1, [0x24f8] = 0x77b2, [0x24f9] = 0x77b4,
  [0x24fa] = 0x77b6, [0x24fb] = 0x77b7, [0x24fc] = 0x77b8, [0x24fd] = 0x77b9,
  [0x24fe] = 0x77ba, [0x2500] = 0x77bc, [0x2501] = 0x77be, [0x2502] = 0x77c0,
  [0x2503] = 0x77c1, [0x2504] = 0x77c2, [0x2505] = 0x77c3, [0x2506] = 0x77c4,
  [0x2507] = 0x77c5, [0x2508] = 0x77c6, [0x2509] = 0x77c7, [0x250a] = 0x77c8,
  [0x250b] = 0x77c9, [0x250c] = 0x77ca, [0x250d] = 0x77cb, [0x250e] = 0x77cc,
  [0x250f] = 0x77ce, [0x2510] = 0x77cf, [0x2511] = 0x77d0, [0x2512] = 0x77d1,
  [0x2513] = 0x77d2, [0x2514] = 0x77d3, [0x2515] = 0x77d4, [0x2516] = 0x77d5,
  [0x2517] = 0x77d6, [0x2518] = 0x77d8, [0x2519] = 0x77d9, [0x251a] = 0x77da,
  [0x251b] = 0x77dd, [0x251c] = 0x77de, [0x251d] = 0x77df, [0x251e] = 0x77e0,
  [0x251f] = 0x77e1, [0x2520] = 0x77e4, [0x2521] = 0x75c5, [0x2522] = 0x5e76,
  [0x2523] = 0x73bb, [0x2524] = 0x83e0, [0x2525] = 0x64ad, [0x2526] = 0x62e8,
  [0x2527] = 0x94b5, [0x2528] = 0x6ce2, [0x2529] = 0x535a, [0x252a] = 0x52c3,
  [0x252b] = 0x640f, [0x252c] = 0x94c2, [0x252d] = 0x7b94, [0x252e] = 0x4f2f,
  [0x252f] = 0x5e1b, [0x2530] = 0x8236, [0x2531] = 0x8116, [0x2532] = 0x818a,
  [0x2533] = 0x6e24, [0x2534] = 0x6cca, [0x2535] = 0x9a73, [0x2536] = 0x6355,
  [0x2537] = 0x535c, [0x2538] = 0x54fa, [0x2539] = 0x8865, [0x253a] = 0x57e0,
  [0x253b] = 0x4e0d, [0x253c] = 0x5e03, [0x253d] = 0x6b65, [0x253e] = 0x7c3f,
  [0x253f] = 0x90e8, [0x2540] = 0x6016, [0x2541] = 0x64e6, [0x2542] = 0x731c,
  [0x2543] = 0x88c1, [0x2544] = 0x6750, [0x2545] = 0x624d, [0x2546] = 0x8d22,
  [0x2547] = 0x776c, [0x2548] = 0x8e29, [0x2549] = 0x91c7, [0x254a] = 0x5f69,
  [0x254b] = 0x83dc, [0x254c] = 0x8521, [0x254d] = 0x9910, [0x254e] = 0x53c2,
  [0x254f] = 0x8695, [0x2550] = 0x6b8b, [0x2551] = 0x60ed, [0x2552] = 0x60e8,
  [0x2553] = 0x707f, [0x2554] = 0x82cd, [0x2555] = 0x8231, [0x2556] = 0x4ed3,
  [0x2557] = 0x6ca7, [0x2558] = 0x85cf, [0x2559] = 0x64cd, [0x255a] = 0x7cd9,
  [0x255b] = 0x69fd, [0x255c] = 0x66f9, [0x255d] = 0x8349, [0x255e] = 0x5395,
  [0x255f] = 0x7b56, [0x2560] = 0x4fa7, [0x2561] = 0x518c, [0x2562] = 0x6d4b,
  [0x2563] = 0x5c42, [0x2564] = 0x8e6d, [0x2565] = 0x63d2, [0x2566] = 0x53c9,
  [0x2567] = 0x832c, [0x2568] = 0x8336, [0x2569] = 0x67e5, [0x256a] = 0x78b4,
  [0x256b] = 0x643d, [0x256c] = 0x5bdf, [0x256d] = 0x5c94, [0x256e] = 0x5dee,
  [0x256f] = 0x8be7, [0x2570] = 0x62c6, [0x2571] = 0x67f4, [0x2572] = 0x8c7a,
  [0x2573] = 0x6400, [0x2574] = 0x63ba, [0x2575] = 0x8749, [0x2576] = 0x998b,
  [0x2577] = 0x8c17, [0x2578] = 0x7f20, [0x2579] = 0x94f2, [0x257a] = 0x4ea7,
  [0x257b] = 0x9610, [0x257c] = 0x98a4, [0x257d] = 0x660c, [0x257e] = 0x7316,
  [0x2580] = 0x77e6, [0x2581] = 0x77e8, [0x2582] = 0x77ea, [0x2583] = 0x77ef,
  [0x2584] = 0x77f0, [0x2585] = 0x77f1, [0x2586] = 0x77f2, [0x2587] = 0x77f4,
  [0x2588] = 0x77f5, [0x2589] = 0x77f7, [0x258a] = 0x77f9, [0x258b] = 0x77fa,
  [0x258c] = 0x77fb, [0x258d] = 0x77fc, [0x258e] = 0x7803, [0x258f] = 0x7804,
  [0x2590] = 0x7805, [0x2591] = 0x7806, [0x2592] = 0x7807, [0x2593] = 0x7808,
  [0x2594] = 0x780a, [0x2595] = 0x780b, [0x2596] = 0x780e, [0x2597] = 0x780f,
  [0x2598] = 0x7810, [0x2599] = 0x7813, [0x259a] = 0x7815, [0x259b] = 0x7819,
  [0x259c] = 0x781b, [0x259d] = 0x781e, [0x259e] = 0x7820, [0x259f] = 0x7821,
  [0x25a0] = 0x7822, [0x25a1] = 0x7824, [0x25a2] = 0x7828, [0x25a3] = 0x782a,
  [0x25a4] = 0x782b, [0x25a5] = 0x782e, [0x25a6] = 0x782f, [0x25a7] = 0x7831,
  [0x25a8] = 0x7832, [0x25a9] = 0x7833, [0x25aa] = 0x7835, [0x25ab] = 0x7836,
  [0x25ac] = 0x783d, [0x25ad] = 0x783f, [0x25ae] = 0x7841, [0x25af] = 0x7842,
  [0x25b0] = 0x7843, [0x25b1] = 0x7844, [0x25b2] = 0x7846, [0x25b3] = 0x7848,
  [0x25b4] = 0x7849, [0x25b5] = 0x784a, [0x25b6] = 0x784b, [0x25b7] = 0x784d,
  [0x25b8] = 0x784f, [0x25b9] = 0x7851, [0x25ba] = 0x7853, [0x25bb] = 0x7854,
  [0x25bc] = 0x7858, [0x25bd] = 0x7859, [0x25be] = 0x785a, [0x25c0] = 0x785b,
  [0x25c1] = 0x785c, [0x25c2] = 0x785e, [0x25c3] = 0x785f, [0x25c4] = 0x7860,
  [0x25c5] = 0x7861, [0x25c6] = 0x7862, [0x25c7] = 0x7863, [0x25c8] = 0x7864,
  [0x25c9] = 0x7865, [0x25ca] = 0x7866, [0x25cb] = 0x7867, [0x25cc] = 0x7868,
  [0x25cd] = 0x7869, [0x25ce] = 0x786f, [0x25cf] = 0x7870, [0x25d0] = 0x7871,
  [0x25d1] = 0x7872, [0x25d2] = 0x7873, [0x25d3] = 0x7874, [0x25d4] = 0x7875,
  [0x25d5] = 0x7876, [0x25d6] = 0x7878, [0x25d7] = 0x7879, [0x25d8] = 0x787a,
  [0x25d9] = 0x787b, [0x25da] = 0x787d, [0x25db] = 0x787e, [0x25dc] = 0x787f,
  [0x25dd] = 0x7880, [0x25de] = 0x7881, [0x25df] = 0x7882, [0x25e0] = 0x7883,
  [0x25e1] = 0x573a, [0x25e2] = 0x5c1d, [0x25e3] = 0x5e38, [0x25e4] = 0x957f,
  [0x25e5] = 0x507f, [0x25e6] = 0x80a0, [0x25e7] = 0x5382, [0x25e8] = 0x655e,
  [0x25e9] = 0x7545, [0x25ea] = 0x5531, [0x25eb] = 0x5021, [0x25ec] = 0x8d85,
  [0x25ed] = 0x6284, [0x25ee] = 0x949e, [0x25ef] = 0x671d, [0x25f0] = 0x5632,
  [0x25f1] = 0x6f6e, [0x25f2] = 0x5de2, [0x25f3] = 0x5435, [0x25f4] = 0x7092,
  [0x25f5] = 0x8f66, [0x25f6] = 0x626f, [0x25f7] = 0x64a4, [0x25f8] = 0x63a3,
  [0x25f9] = 0x5f7b, [0x25fa] = 0x6f88, [0x25fb] = 0x90f4, [0x25fc] = 0x81e3,
  [0x25fd] = 0x8fb0, [0x25fe] = 0x5c18, [0x25ff] = 0x6668, [0x2600] = 0x5ff1,
  [0x2601] = 0x6c89, [0x2602] = 0x9648, [0x2603] = 0x8d81, [0x2604] = 0x886c,
  [0x2605] = 0x6491, [0x2606] = 0x79f0, [0x2607] = 0x57ce, [0x2608] = 0x6a59,
  [0x2609] = 0x6210, [0x260a] = 0x5448, [0x260b] = 0x4e58, [0x260c] = 0x7a0b,
  [0x260d] = 0x60e9, [0x260e] = 0x6f84, [0x260f] = 0x8bda, [0x2610] = 0x627f,
  [0x2611] = 0x901e, [0x2612] = 0x9a8b, [0x2613] = 0x79e4, [0x2614] = 0x5403,
  [0x2615] = 0x75f4, [0x2616] = 0x6301, [0x2617] = 0x5319, [0x2618] = 0x6c60,
  [0x2619] = 0x8fdf, [0x261a] = 0x5f1b, [0x261b] = 0x9a70, [0x261c] = 0x803b,
  [0x261d] = 0x9f7f, [0x261e] = 0x4f88, [0x261f] = 0x5c3a, [0x2620] = 0x8d64,
  [0x2621] = 0x7fc5, [0x2622] = 0x65a5, [0x2623] = 0x70bd, [0x2624] = 0x5145,
  [0x2625] = 0x51b2, [0x2626] = 0x866b, [0x2627] = 0x5d07, [0x2628] = 0x5ba0,
  [0x2629] = 0x62bd, [0x262a] = 0x916c, [0x262b] = 0x7574, [0x262c] = 0x8e0c,
  [0x262d] = 0x7a20, [0x262e] = 0x6101, [0x262f] = 0x7b79, [0x2630] = 0x4ec7,
  [0x2631] = 0x7ef8, [0x2632] = 0x7785, [0x2633] = 0x4e11, [0x2634] = 0x81ed,
  [0x2635] = 0x521d, [0x2636] = 0x51fa, [0x2637] = 0x6a71, [0x2638] = 0x53a8,
  [0x2639] = 0x8e87, [0x263a] = 0x9504, [0x263b] = 0x96cf, [0x263c] = 0x6ec1,
  [0x263d] = 0x9664, [0x263e] = 0x695a, [0x2640] = 0x7884, [0x2641] = 0x7885,
  [0x2642] = 0x7886, [0x2643] = 0x7888, [0x2644] = 0x788a, [0x2645] = 0x788b,
  [0x2646] = 0x788f, [0x2647] = 0x7890, [0x2648] = 0x7892, [0x2649] = 0x7894,
  [0x264a] = 0x7895, [0x264b] = 0x7896, [0x264c] = 0x7899, [0x264d] = 0x789d,
  [0x264e] = 0x789e, [0x264f] = 0x78a0, [0x2650] = 0x78a2, [0x2651] = 0x78a4,
  [0x2652] = 0x78a6, [0x2653] = 0x78a8, [0x2654] = 0x78a9, [0x2655] = 0x78aa,
  [0x2656] = 0x78ab, [0x2657] = 0x78ac, [0x2658] = 0x78ad, [0x2659] = 0x78ae,
  [0x265a] = 0x78af, [0x265b] = 0x78b5, [0x265c] = 0x78b6, [0x265d] = 0x78b7,
  [0x265e] = 0x78b8, [0x265f] = 0x78ba, [0x2660] = 0x78bb, [0x2661] = 0x78bc,
  [0x2662] = 0x78bd, [0x2663] = 0x78bf, [0x2664] = 0x78c0, [0x2665] = 0x78c2,
  [0x2666] = 0x78c3, [0x2667] = 0x78c4, [0x2668] = 0x78c6, [0x2669] = 0x78c7,
  [0x266a] = 0x78c8, [0x266b] = 0x78cc, [0x266c] = 0x78cd, [0x266d] = 0x78ce,
  [0x266e] = 0x78cf, [0x266f] = 0x78d1, [0x2670] = 0x78d2, [0x2671] = 0x78d3,
  [0x2672] = 0x78d6, [0x2673] = 0x78d7, [0x2674] = 0x78d8, [0x2675] = 0x78da,
  [0x2676] = 0x78db, [0x2677] = 0x78dc, [0x2678] = 0x78dd, [0x2679] = 0x78de,
  [0x267a] = 0x78df, [0x267b] = 0x78e0, [0x267c] = 0x78e1, [0x267d] = 0x78e2,
  [0x267e] = 0x78e3, [0x2680] = 0x78e4, [0x2681] = 0x78e5, [0x2682] = 0x78e6,
  [0x2683] = 0x78e7, [0x2684] = 0x78e9, [0x2685] = 0x78ea, [0x2686] = 0x78eb,
  [0x2687] = 0x78ed, [0x2688] = 0x78ee, [0x2689] = 0x78ef, [0x268a] = 0x78f0,
  [0x268b] = 0x78f1, [0x268c] = 0x78f3, [0x268d] = 0x78f5, [0x268e] = 0x78f6,
  [0x268f] = 0x78f8, [0x2690] = 0x78f9, [0x2691] = 0x78fb, [0x2692] = 0x78fc,
  [0x2693] = 0x78fd, [0x2694] = 0x78fe, [0x2695] = 0x78ff, [0x2696] = 0x7900,
  [0x2697] = 0x7902, [0x2698] = 0x7903, [0x2699] = 0x7904, [0x269a] = 0x7906,
  [0x269b] = 0x7907, [0x269c] = 0x7908, [0x269d] = 0x7909, [0x269e] = 0x790a,
  [0x269f] = 0x790b, [0x26a0] = 0x790c, [0x26a1] = 0x7840, [0x26a2] = 0x50a8,
  [0x26a3] = 0x77d7, [0x26a4] = 0x6410, [0x26a5] = 0x89e6, [0x26a6] = 0x5904,
  [0x26a7] = 0x63e3, [0x26a8] = 0x5ddd, [0x26a9] = 0x7a7f, [0x26aa] = 0x693d,
  [0x26ab] = 0x4f20, [0x26ac] = 0x8239, [0x26ad] = 0x5598, [0x26ae] = 0x4e32,
  [0x26af] = 0x75ae, [0x26b0] = 0x7a97, [0x26b1] = 0x5e62, [0x26b2] = 0x5e8a,
  [0x26b3] = 0x95ef, [0x26b4] = 0x521b, [0x26b5] = 0x5439, [0x26b6] = 0x708a,
  [0x26b7] = 0x6376, [0x26b8] = 0x9524, [0x26b9] = 0x5782, [0x26ba] = 0x6625,
  [0x26bb] = 0x693f, [0x26bc] = 0x9187, [0x26bd] = 0x5507, [0x26be] = 0x6df3,
  [0x26bf] = 0x7eaf, [0x26c0] = 0x8822, [0x26c1] = 0x6233, [0x26c2] = 0x7ef0,
  [0x26c3] = 0x75b5, [0x26c4] = 0x8328, [0x26c5] = 0x78c1, [0x26c6] = 0x96cc,
  [0x26c7] = 0x8f9e, [0x26c8] = 0x6148, [0x26c9] = 0x74f7, [0x26ca] = 0x8bcd,
  [0x26cb] = 0x6b64, [0x26cc] = 0x523a, [0x26cd] = 0x8d50, [0x26ce] = 0x6b21,
  [0x26cf] = 0x806a, [0x26d0] = 0x8471, [0x26d1] = 0x56f1, [0x26d2] = 0x5306,
  [0x26d3] = 0x4ece, [0x26d4] = 0x4e1b, [0x26d5] = 0x51d1, [0x26d6] = 0x7c97,
  [0x26d7] = 0x918b, [0x26d8] = 0x7c07, [0x26d9] = 0x4fc3, [0x26da] = 0x8e7f,
  [0x26db] = 0x7be1, [0x26dc] = 0x7a9c, [0x26dd] = 0x6467, [0x26de] = 0x5d14,
  [0x26df] = 0x50ac, [0x26e0] = 0x8106, [0x26e1] = 0x7601, [0x26e2] = 0x7cb9,
  [0x26e3] = 0x6dec, [0x26e4] = 0x7fe0, [0x26e5] = 0x6751, [0x26e6] = 0x5b58,
  [0x26e7] = 0x5bf8, [0x26e8] = 0x78cb, [0x26e9] = 0x64ae, [0x26ea] = 0x6413,
  [0x26eb] = 0x63aa, [0x26ec] = 0x632b, [0x26ed] = 0x9519, [0x26ee] = 0x642d,
  [0x26ef] = 0x8fbe, [0x26f0] = 0x7b54, [0x26f1] = 0x7629, [0x26f2] = 0x6253,
  [0x26f3] = 0x5927, [0x26f4] = 0x5446, [0x26f5] = 0x6b79, [0x26f6] = 0x50a3,
  [0x26f7] = 0x6234, [0x26f8] = 0x5e26, [0x26f9] = 0x6b86, [0x26fa] = 0x4ee3,
  [0x26fb] = 0x8d37, [0x26fc] = 0x888b, [0x26fd] = 0x5f85, [0x26fe] = 0x902e,
  [0x2700] = 0x790d, [0x2701] = 0x790e, [0x2702] = 0x790f, [0x2703] = 0x7910,
  [0x2704] = 0x7911, [0x2705] = 0x7912, [0x2706] = 0x7914, [0x2707] = 0x7915,
  [0x2708] = 0x7916, [0x2709] = 0x7917, [0x270a] = 0x7918, [0x270b] = 0x7919,
  [0x270c] = 0x791a, [0x270d] = 0x791b, [0x270e] = 0x791c, [0x270f] = 0x791d,
  [0x2710] = 0x791f, [0x2711] = 0x7920, [0x2712] = 0x7921, [0x2713] = 0x7922,
  [0x2714] = 0x7923, [0x2715] = 0x7925, [0x2716] = 0x7926, [0x2717] = 0x7927,
  [0x2718] = 0x7928, [0x2719] = 0x7929, [0x271a] = 0x792a, [0x271b] = 0x792b,
  [0x271c] = 0x792c, [0x271d] = 0x792d, [0x271e] = 0x792e, [0x271f] = 0x792f,
  [0x2720] = 0x7930, [0x2721] = 0x7931, [0x2722] = 0x7932, [0x2723] = 0x7933,
  [0x2724] = 0x7935, [0x2725] = 0x7936, [0x2726] = 0x7937, [0x2727] = 0x7938,
  [0x2728] = 0x7939, [0x2729] = 0x793d, [0x272a] = 0x793f, [0x272b] = 0x7942,
  [0x272c] = 0x7943, [0x272d] = 0x7944, [0x272e] = 0x7945, [0x272f] = 0x7947,
  [0x2730] = 0x794a, [0x2731] = 0x794b, [0x2732] = 0x794c, [0x2733] = 0x794d,
  [0x2734] = 0x794e, [0x2735] = 0x794f, [0x2736] = 0x7950, [0x2737] = 0x7951,
  [0x2738] = 0x7952, [0x2739] = 0x7954, [0x273a] = 0x7955, [0x273b] = 0x7958,
  [0x273c] = 0x7959, [0x273d] = 0x7961, [0x273e] = 0x7963, [0x2740] = 0x7964,
  [0x2741] = 0x7966, [0x2742] = 0x7969, [0x2743] = 0x796a, [0x2744] = 0x796b,
  [0x2745] = 0x796c, [0x2746] = 0x796e, [0x2747] = 0x7970, [0x2748] = 0x7971,
  [0x2749] = 0x7972, [0x274a] = 0x7973, [0x274b] = 0x7974, [0x274c] = 0x7975,
  [0x274d] = 0x7976, [0x274e] = 0x7979, [0x274f] = 0x797b, [0x2750] = 0x797c,
  [0x2751] = 0x797d, [0x2752] = 0x797e, [0x2753] = 0x797f, [0x2754] = 0x7982,
  [0x2755] = 0x7983, [0x2756] = 0x7986, [0x2757] = 0x7987, [0x2758] = 0x7988,
  [0x2759] = 0x7989, [0x275a] = 0x798b, [0x275b] = 0x798c, [0x275c] = 0x798d,
  [0x275d] = 0x798e, [0x275e] = 0x7990, [0x275f] = 0x7991, [0x2760] = 0x7992,
  [0x2761] = 0x6020, [0x2762] = 0x803d, [0x2763] = 0x62c5, [0x2764] = 0x4e39,
  [0x2765] = 0x5355, [0x2766] = 0x90f8, [0x2767] = 0x63b8, [0x2768] = 0x80c6,
  [0x2769] = 0x65e6, [0x276a] = 0x6c2e, [0x276b] = 0x4f46, [0x276c] = 0x60ee,
  [0x276d] = 0x6de1, [0x276e] = 0x8bde, [0x276f] = 0x5f39, [0x2770] = 0x86cb,
  [0x2771] = 0x5f53, [0x2772] = 0x6321, [0x2773] = 0x515a, [0x2774] = 0x8361,
  [0x2775] = 0x6863, [0x2776] = 0x5200, [0x2777] = 0x6363, [0x2778] = 0x8e48,
  [0x2779] = 0x5012, [0x277a] = 0x5c9b, [0x277b] = 0x7977, [0x277c] = 0x5bfc,
  [0x277d] = 0x5230, [0x277e] = 0x7a3b, [0x277f] = 0x60bc, [0x2780] = 0x9053,
  [0x2781] = 0x76d7, [0x2782] = 0x5fb7, [0x2783] = 0x5f97, [0x2784] = 0x7684,
  [0x2785] = 0x8e6c, [0x2786] = 0x706f, [0x2787] = 0x767b, [0x2788] = 0x7b49,
  [0x2789] = 0x77aa, [0x278a] = 0x51f3, [0x278b] = 0x9093, [0x278c] = 0x5824,
  [0x278d] = 0x4f4e, [0x278e] = 0x6ef4, [0x278f] = 0x8fea, [0x2790] = 0x654c,
  [0x2791] = 0x7b1b, [0x2792] = 0x72c4, [0x2793] = 0x6da4, [0x2794] = 0x7fdf,
  [0x2795] = 0x5ae1, [0x2796] = 0x62b5, [0x2797] = 0x5e95, [0x2798] = 0x5730,
  [0x2799] = 0x8482, [0x279a] = 0x7b2c, [0x279b] = 0x5e1d, [0x279c] = 0x5f1f,
  [0x279d] = 0x9012, [0x279e] = 0x7f14, [0x279f] = 0x98a0, [0x27a0] = 0x6382,
  [0x27a1] = 0x6ec7, [0x27a2] = 0x7898, [0x27a3] = 0x70b9, [0x27a4] = 0x5178,
  [0x27a5] = 0x975b, [0x27a6] = 0x57ab, [0x27a7] = 0x7535, [0x27a8] = 0x4f43,
  [0x27a9] = 0x7538, [0x27aa] = 0x5e97, [0x27ab] = 0x60e6, [0x27ac] = 0x5960,
  [0x27ad] = 0x6dc0, [0x27ae] = 0x6bbf, [0x27af] = 0x7889, [0x27b0] = 0x53fc,
  [0x27b1] = 0x96d5, [0x27b2] = 0x51cb, [0x27b3] = 0x5201, [0x27b4] = 0x6389,
  [0x27b5] = 0x540a, [0x27b6] = 0x9493, [0x27b7] = 0x8c03, [0x27b8] = 0x8dcc,
  [0x27b9] = 0x7239, [0x27ba] = 0x789f, [0x27bb] = 0x8776, [0x27bc] = 0x8fed,
  [0x27bd] = 0x8c0d, [0x27be] = 0x53e0, [0x27c0] = 0x7993, [0x27c1] = 0x7994,
  [0x27c2] = 0x7995, [0x27c3] = 0x7996, [0x27c4] = 0x7997, [0x27c5] = 0x7998,
  [0x27c6] = 0x7999, [0x27c7] = 0x799b, [0x27c8] = 0x799c, [0x27c9] = 0x799d,
  [0x27ca] = 0x799e, [0x27cb] = 0x799f, [0x27cc] = 0x79a0, [0x27cd] = 0x79a1,
  [0x27ce] = 0x79a2, [0x27cf] = 0x79a3, [0x27d0] = 0x79a4, [0x27d1] = 0x79a5,
  [0x27d2] = 0x79a6, [0x27d3] = 0x79a8, [0x27d4] = 0x79a9, [0x27d5] = 0x79aa,
  [0x27d6] = 0x79ab, [0x27d7] = 0x79ac, [0x27d8] = 0x79ad, [0x27d9] = 0x79ae,
  [0x27da] = 0x79af, [0x27db] = 0x79b0, [0x27dc] = 0x79b1, [0x27dd] = 0x79b2,
  [0x27de] = 0x79b4, [0x27df] = 0x79b5, [0x27e0] = 0x79b6, [0x27e1] = 0x79b7,
  [0x27e2] = 0x79b8, [0x27e3] = 0x79bc, [0x27e4] = 0x79bf, [0x27e5] = 0x79c2,
  [0x27e6] = 0x79c4, [0x27e7] = 0x79c5, [0x27e8] = 0x79c7, [0x27e9] = 0x79c8,
  [0x27ea] = 0x79ca, [0x27eb] = 0x79cc, [0x27ec] = 0x79ce, [0x27ed] = 0x79cf,
  [0x27ee] = 0x79d0, [0x27ef] = 0x79d3, [0x27f0] = 0x79d4, [0x27f1] = 0x79d6,
  [0x27f2] = 0x79d7, [0x27f3] = 0x79d9, [0x27f4] = 0x79da, [0x27f5] = 0x79db,
  [0x27f6] = 0x79dc, [0x27f7] = 0x79dd, [0x27f8] = 0x79de, [0x27f9] = 0x79e0,
  [0x27fa] = 0x79e1, [0x27fb] = 0x79e2, [0x27fc] = 0x79e5, [0x27fd] = 0x79e8,
  [0x27fe] = 0x79ea, [0x2800] = 0x79ec, [0x2801] = 0x79ee, [0x2802] = 0x79f1,
  [0x2803] = 0x79f2, [0x2804] = 0x79f3, [0x2805] = 0x79f4, [0x2806] = 0x79f5,
  [0x2807] = 0x79f6, [0x2808] = 0x79f7, [0x2809] = 0x79f9, [0x280a] = 0x79fa,
  [0x280b] = 0x79fc, [0x280c] = 0x79fe, [0x280d] = 0x79ff, [0x280e] = 0x7a01,
  [0x280f] = 0x7a04, [0x2810] = 0x7a05, [0x2811] = 0x7a07, [0x2812] = 0x7a08,
  [0x2813] = 0x7a09, [0x2814] = 0x7a0a, [0x2815] = 0x7a0c, [0x2816] = 0x7a0f,
  [0x2817] = 0x7a10, [0x2818] = 0x7a11, [0x2819] = 0x7a12, [0x281a] = 0x7a13,
  [0x281b] = 0x7a15, [0x281c] = 0x7a16, [0x281d] = 0x7a18, [0x281e] = 0x7a19,
  [0x281f] = 0x7a1b, [0x2820] = 0x7a1c, [0x2821] = 0x4e01, [0x2822] = 0x76ef,
  [0x2823] = 0x53ee, [0x2824] = 0x9489, [0x2825] = 0x9876, [0x2826] = 0x9f0e,
  [0x2827] = 0x952d, [0x2828] = 0x5b9a, [0x2829] = 0x8ba2, [0x282a] = 0x4e22,
  [0x282b] = 0x4e1c, [0x282c] = 0x51ac, [0x282d] = 0x8463, [0x282e] = 0x61c2,
  [0x282f] = 0x52a8, [0x2830] = 0x680b, [0x2831] = 0x4f97, [0x2832] = 0x606b,
  [0x2833] = 0x51bb, [0x2834] = 0x6d1e, [0x2835] = 0x515c, [0x2836] = 0x6296,
  [0x2837] = 0x6597, [0x2838] = 0x9661, [0x2839] = 0x8c46, [0x283a] = 0x9017,
  [0x283b] = 0x75d8, [0x283c] = 0x90fd, [0x283d] = 0x7763, [0x283e] = 0x6bd2,
  [0x283f] = 0x728a, [0x2840] = 0x72ec, [0x2841] = 0x8bfb, [0x2842] = 0x5835,
  [0x2843] = 0x7779, [0x2844] = 0x8d4c, [0x2845] = 0x675c, [0x2846] = 0x9540,
  [0x2847] = 0x809a, [0x2848] = 0x5ea6, [0x2849] = 0x6e21, [0x284a] = 0x5992,
  [0x284b] = 0x7aef, [0x284c] = 0x77ed, [0x284d] = 0x953b, [0x284e] = 0x6bb5,
  [0x284f] = 0x65ad, [0x2850] = 0x7f0e, [0x2851] = 0x5806, [0x2852] = 0x5151,
  [0x2853] = 0x961f, [0x2854] = 0x5bf9, [0x2855] = 0x58a9, [0x2856] = 0x5428,
  [0x2857] = 0x8e72, [0x2858] = 0x6566, [0x2859] = 0x987f, [0x285a] = 0x56e4,
  [0x285b] = 0x949d, [0x285c] = 0x76fe, [0x285d] = 0x9041, [0x285e] = 0x6387,
  [0x285f] = 0x54c6, [0x2860] = 0x591a, [0x2861] = 0x593a, [0x2862] = 0x579b,
  [0x2863] = 0x8eb2, [0x2864] = 0x6735, [0x2865] = 0x8dfa, [0x2866] = 0x8235,
  [0x2867] = 0x5241, [0x2868] = 0x60f0, [0x2869] = 0x5815, [0x286a] = 0x86fe,
  [0x286b] = 0x5ce8, [0x286c] = 0x9e45, [0x286d] = 0x4fc4, [0x286e] = 0x989d,
  [0x286f] = 0x8bb9, [0x2870] = 0x5a25, [0x2871] = 0x6076, [0x2872] = 0x5384,
  [0x2873] = 0x627c, [0x2874] = 0x904f, [0x2875] = 0x9102, [0x2876] = 0x997f,
  [0x2877] = 0x6069, [0x2878] = 0x800c, [0x2879] = 0x513f, [0x287a] = 0x8033,
  [0x287b] = 0x5c14, [0x287c] = 0x9975, [0x287d] = 0x6d31, [0x287e] = 0x4e8c,
  [0x2880] = 0x7a1d, [0x2881] = 0x7a1f, [0x2882] = 0x7a21, [0x2883] = 0x7a22,
  [0x2884] = 0x7a24, [0x2885] = 0x7a25, [0x2886] = 0x7a26, [0x2887] = 0x7a27,
  [0x2888] = 0x7a28, [0x2889] = 0x7a29, [0x288a] = 0x7a2a, [0x288b] = 0x7a2b,
  [0x288c] = 0x7a2c, [0x288d] = 0x7a2d, [0x288e] = 0x7a2e, [0x288f] = 0x7a2f,
  [0x2890] = 0x7a30, [0x2891] = 0x7a31, [0x2892] = 0x7a32, [0x2893] = 0x7a34,
  [0x2894] = 0x7a35, [0x2895] = 0x7a36, [0x2896] = 0x7a38, [0x2897] = 0x7a3a,
  [0x2898] = 0x7a3e, [0x2899] = 0x7a40, [0x289a] = 0x7a41, [0x289b] = 0x7a42,
  [0x289c] = 0x7a43, [0x289d] = 0x7a44, [0x289e] = 0x7a45, [0x289f] = 0x7a47,
  [0x28a0] = 0x7a48, [0x28a1] = 0x7a49, [0x28a2] = 0x7a4a, [0x28a3] = 0x7a4b,
  [0x28a4] = 0x7a4c, [0x28a5] = 0x7a4d, [0x28a6] = 0x7a4e, [0x28a7] = 0x7a4f,
  [0x28a8] = 0x7a50, [0x28a9] = 0x7a52, [0x28aa] = 0x7a53, [0x28ab] = 0x7a54,
  [0x28ac] = 0x7a55, [0x28ad] = 0x7a56, [0x28ae] = 0x7a58, [0x28af] = 0x7a59,
  [0x28b0] = 0x7a5a, [0x28b1] = 0x7a5b, [0x28b2] = 0x7a5c, [0x28b3] = 0x7a5d,
  [0x28b4] = 0x7a5e, [0x28b5] = 0x7a5f, [0x28b6] = 0x7a60, [0x28b7] = 0x7a61,
  [0x28b8] = 0x7a62, [0x28b9] = 0x7a63, [0x28ba] = 0x7a64, [0x28bb] = 0x7a65,
  [0x28bc] = 0x7a66, [0x28bd] = 0x7a67, [0x28be] = 0x7a68, [0x28c0] = 0x7a69,
  [0x28c1] = 0x7a6a, [0x28c2] = 0x7a6b, [0x28c3] = 0x7a6c, [0x28c4] = 0x7a6d,
  [0x28c5] = 0x7a6e, [0x28c6] = 0x7a6f, [0x28c7] = 0x7a71, [0x28c8] = 0x7a72,
  [0x28c9] = 0x7a73, [0x28ca] = 0x7a75, [0x28cb] = 0x7a7b, [0x28cc] = 0x7a7c,
  [0x28cd] = 0x7a7d, [0x28ce] = 0x7a7e, [0x28cf] = 0x7a82, [0x28d0] = 0x7a85,
  [0x28d1] = 0x7a87, [0x28d2] = 0x7a89, [0x28d3] = 0x7a8a, [0x28d4] = 0x7a8b,
  [0x28d5] = 0x7a8c, [0x28d6] = 0x7a8e, [0x28d7] = 0x7a8f, [0x28d8] = 0x7a90,
  [0x28d9] = 0x7a93, [0x28da] = 0x7a94, [0x28db] = 0x7a99, [0x28dc] = 0x7a9a,
  [0x28dd] = 0x7a9b, [0x28de] = 0x7a9e, [0x28df] = 0x7aa1, [0x28e0] = 0x7aa2,
  [0x28e1] = 0x8d30, [0x28e2] = 0x53d1, [0x28e3] = 0x7f5a, [0x28e4] = 0x7b4f,
  [0x28e5] = 0x4f10, [0x28e6] = 0x4e4f, [0x28e7] = 0x9600, [0x28e8] = 0x6cd5,
  [0x28e9] = 0x73d0, [0x28ea] = 0x85e9, [0x28eb] = 0x5e06, [0x28ec] = 0x756a,
  [0x28ed] = 0x7ffb, [0x28ee] = 0x6a0a, [0x28ef] = 0x77fe, [0x28f0] = 0x9492,
  [0x28f1] = 0x7e41, [0x28f2] = 0x51e1, [0x28f3] = 0x70e6, [0x28f4] = 0x53cd,
  [0x28f5] = 0x8fd4, [0x28f6] = 0x8303, [0x28f7] = 0x8d29, [0x28f8] = 0x72af,
  [0x28f9] = 0x996d, [0x28fa] = 0x6cdb, [0x28fb] = 0x574a, [0x28fc] = 0x82b3,
  [0x28fd] = 0x65b9, [0x28fe] = 0x80aa, [0x28ff] = 0x623f, [0x2900] = 0x9632,
  [0x2901] = 0x59a8, [0x2902] = 0x4eff, [0x2903] = 0x8bbf, [0x2904] = 0x7eba,
  [0x2905] = 0x653e, [0x2906] = 0x83f2, [0x2907] = 0x975e, [0x2908] = 0x5561,
  [0x2909] = 0x98de, [0x290a] = 0x80a5, [0x290b] = 0x532a, [0x290c] = 0x8bfd,
  [0x290d] = 0x5420, [0x290e] = 0x80ba, [0x290f] = 0x5e9f, [0x2910] = 0x6cb8,
  [0x2911] = 0x8d39, [0x2912] = 0x82ac, [0x2913] = 0x915a, [0x2914] = 0x5429,
  [0x2915] = 0x6c1b, [0x2916] = 0x5206, [0x2917] = 0x7eb7, [0x2918] = 0x575f,
  [0x2919] = 0x711a, [0x291a] = 0x6c7e, [0x291b] = 0x7c89, [0x291c] = 0x594b,
  [0x291d] = 0x4efd, [0x291e] = 0x5fff, [0x291f] = 0x6124, [0x2920] = 0x7caa,
  [0x2921] = 0x4e30, [0x2922] = 0x5c01, [0x2923] = 0x67ab, [0x2924] = 0x8702,
  [0x2925] = 0x5cf0, [0x2926] = 0x950b, [0x2927] = 0x98ce, [0x2928] = 0x75af,
  [0x2929] = 0x70fd, [0x292a] = 0x9022, [0x292b] = 0x51af, [0x292c] = 0x7f1d,
  [0x292d] = 0x8bbd, [0x292e] = 0x5949, [0x292f] = 0x51e4, [0x2930] = 0x4f5b,
  [0x2931] = 0x5426, [0x2932] = 0x592b, [0x2933] = 0x6577, [0x2934] = 0x80a4,
  [0x2935] = 0x5b75, [0x2936] = 0x6276, [0x2937] = 0x62c2, [0x2938] = 0x8f90,
  [0x2939] = 0x5e45, [0x293a] = 0x6c1f, [0x293b] = 0x7b26, [0x293c] = 0x4f0f,
  [0x293d] = 0x4fd8, [0x293e] = 0x670d, [0x2940] = 0x7aa3, [0x2941] = 0x7aa4,
  [0x2942] = 0x7aa7, [0x2943] = 0x7aa9, [0x2944] = 0x7aaa, [0x2945] = 0x7aab,
  [0x2946] = 0x7aae, [0x2947] = 0x7aaf, [0x2948] = 0x7ab0, [0x2949] = 0x7ab1,
  [0x294a] = 0x7ab2, [0x294b] = 0x7ab4, [0x294c] = 0x7ab5, [0x294d] = 0x7ab6,
  [0x294e] = 0x7ab7, [0x294f] = 0x7ab8, [0x2950] = 0x7ab9, [0x2951] = 0x7aba,
  [0x2952] = 0x7abb, [0x2953] = 0x7abc, [0x2954] = 0x7abd, [0x2955] = 0x7abe,
  [0x2956] = 0x7ac0, [0x2957] = 0x7ac1, [0x2958] = 0x7ac2, [0x2959] = 0x7ac3,
  [0x295a] = 0x7ac4, [0x295b] = 0x7ac5, [0x295c] = 0x7ac6, [0x295d] = 0x7ac7,
  [0x295e] = 0x7ac8, [0x295f] = 0x7ac9, [0x2960] = 0x7aca, [0x2961] = 0x7acc,
  [0x2962] = 0x7acd, [0x2963] = 0x7ace, [0x2964] = 0x7acf, [0x2965] = 0x7ad0,
  [0x2966] = 0x7ad1, [0x2967] = 0x7ad2, [0x2968] = 0x7ad3, [0x2969] = 0x7ad4,
  [0x296a] = 0x7ad5, [0x296b] = 0x7ad7, [0x296c] = 0x7ad8, [0x296d] = 0x7ada,
  [0x296e] = 0x7adb, [0x296f] = 0x7adc, [0x2970] = 0x7add, [0x2971] = 0x7ae1,
  [0x2972] = 0x7ae2, [0x2973] = 0x7ae4, [0x2974] = 0x7ae7, [0x2975] = 0x7ae8,
  [0x2976] = 0x7ae9, [0x2977] = 0x7aea, [0x2978] = 0x7aeb, [0x2979] = 0x7aec,
  [0x297a] = 0x7aee, [0x297b] = 0x7af0, [0x297c] = 0x7af1, [0x297d] = 0x7af2,
  [0x297e] = 0x7af3, [0x2980] = 0x7af4, [0x2981] = 0x7af5, [0x2982] = 0x7af6,
  [0x2983] = 0x7af7, [0x2984] = 0x7af8, [0x2985] = 0x7afb, [0x2986] = 0x7afc,
  [0x2987] = 0x7afe, [0x2988] = 0x7b00, [0x2989] = 0x7b01, [0x298a] = 0x7b02,
  [0x298b] = 0x7b05, [0x298c] = 0x7b07, [0x298d] = 0x7b09, [0x298e] = 0x7b0c,
  [0x298f] = 0x7b0d, [0x2990] = 0x7b0e, [0x2991] = 0x7b10, [0x2992] = 0x7b12,
  [0x2993] = 0x7b13, [0x2994] = 0x7b16, [0x2995] = 0x7b17, [0x2996] = 0x7b18,
  [0x2997] = 0x7b1a, [0x2998] = 0x7b1c, [0x2999] = 0x7b1d, [0x299a] = 0x7b1f,
  [0x299b] = 0x7b21, [0x299c] = 0x7b22, [0x299d] = 0x7b23, [0x299e] = 0x7b27,
  [0x299f] = 0x7b29, [0x29a0] = 0x7b2d, [0x29a1] = 0x6d6e, [0x29a2] = 0x6daa,
  [0x29a3] = 0x798f, [0x29a4] = 0x88b1, [0x29a5] = 0x5f17, [0x29a6] = 0x752b,
  [0x29a7] = 0x629a, [0x29a8] = 0x8f85, [0x29a9] = 0x4fef, [0x29aa] = 0x91dc,
  [0x29ab] = 0x65a7, [0x29ac] = 0x812f, [0x29ad] = 0x8151, [0x29ae] = 0x5e9c,
  [0x29af] = 0x8150, [0x29b0] = 0x8d74, [0x29b1] = 0x526f, [0x29b2] = 0x8986,
  [0x29b3] = 0x8d4b, [0x29b4] = 0x590d, [0x29b5] = 0x5085, [0x29b6] = 0x4ed8,
  [0x29b7] = 0x961c, [0x29b8] = 0x7236, [0x29b9] = 0x8179, [0x29ba] = 0x8d1f,
  [0x29bb] = 0x5bcc, [0x29bc] = 0x8ba3, [0x29bd] = 0x9644, [0x29be] = 0x5987,
  [0x29bf] = 0x7f1a, [0x29c0] = 0x5490, [0x29c1] = 0x5676, [0x29c2] = 0x560e,
  [0x29c3] = 0x8be5, [0x29c4] = 0x6539, [0x29c5] = 0x6982, [0x29c6] = 0x9499,
  [0x29c7] = 0x76d6, [0x29c8] = 0x6e89, [0x29c9] = 0x5e72, [0x29ca] = 0x7518,
  [0x29cb] = 0x6746, [0x29cc] = 0x67d1, [0x29cd] = 0x7aff, [0x29ce] = 0x809d,
  [0x29cf] = 0x8d76, [0x29d0] = 0x611f, [0x29d1] = 0x79c6, [0x29d2] = 0x6562,
  [0x29d3] = 0x8d63, [0x29d4] = 0x5188, [0x29d5] = 0x521a, [0x29d6] = 0x94a2,
  [0x29d7] = 0x7f38, [0x29d8] = 0x809b, [0x29d9] = 0x7eb2, [0x29da] = 0x5c97,
  [0x29db] = 0x6e2f, [0x29dc] = 0x6760, [0x29dd] = 0x7bd9, [0x29de] = 0x768b,
  [0x29df] = 0x9ad8, [0x29e0] = 0x818f, [0x29e1] = 0x7f94, [0x29e2] = 0x7cd5,
  [0x29e3] = 0x641e, [0x29e4] = 0x9550, [0x29e5] = 0x7a3f, [0x29e6] = 0x544a,
  [0x29e7] = 0x54e5, [0x29e8] = 0x6b4c, [0x29e9] = 0x6401, [0x29ea] = 0x6208,
  [0x29eb] = 0x9e3d, [0x29ec] = 0x80f3, [0x29ed] = 0x7599, [0x29ee] = 0x5272,
  [0x29ef] = 0x9769, [0x29f0] = 0x845b, [0x29f1] = 0x683c, [0x29f2] = 0x86e4,
  [0x29f3] = 0x9601, [0x29f4] = 0x9694, [0x29f5] = 0x94ec, [0x29f6] = 0x4e2a,
  [0x29f7] = 0x5404, [0x29f8] = 0x7ed9, [0x29f9] = 0x6839, [0x29fa] = 0x8ddf,
  [0x29fb] = 0x8015, [0x29fc] = 0x66f4, [0x29fd] = 0x5e9a, [0x29fe] = 0x7fb9,
  [0x2a00] = 0x7b2f, [0x2a01] = 0x7b30, [0x2a02] = 0x7b32, [0x2a03] = 0x7b34,
  [0x2a04] = 0x7b35, [0x2a05] = 0x7b36, [0x2a06] = 0x7b37, [0x2a07] = 0x7b39,
  [0x2a08] = 0x7b3b, [0x2a09] = 0x7b3d, [0x2a0a] = 0x7b3f, [0x2a0b] = 0x7b40,
  [0x2a0c] = 0x7b41, [0x2a0d] = 0x7b42, [0x2a0e] = 0x7b43, [0x2a0f] = 0x7b44,
  [0x2a10] = 0x7b46, [0x2a11] = 0x7b48, [0x2a12] = 0x7b4a, [0x2a13] = 0x7b4d,
  [0x2a14] = 0x7b4e, [0x2a15] = 0x7b53, [0x2a16] = 0x7b55, [0x2a17] = 0x7b57,
  [0x2a18] = 0x7b59, [0x2a19] = 0x7b5c, [0x2a1a] = 0x7b5e, [0x2a1b] = 0x7b5f,
  [0x2a1c] = 0x7b61, [0x2a1d] = 0x7b63, [0x2a1e] = 0x7b64, [0x2a1f] = 0x7b65,
  [0x2a20] = 0x7b66, [0x2a21] = 0x7b67, [0x2a22] = 0x7b68, [0x2a23] = 0x7b69,
  [0x2a24] = 0x7b6a, [0x2a25] = 0x7b6b, [0x2a26] = 0x7b6c, [0x2a27] = 0x7b6d,
  [0x2a28] = 0x7b6f, [0x2a29] = 0x7b70, [0x2a2a] = 0x7b73, [0x2a2b] = 0x7b74,
  [0x2a2c] = 0x7b76, [0x2a2d] = 0x7b78, [0x2a2e] = 0x7b7a, [0x2a2f] = 0x7b7c,
  [0x2a30] = 0x7b7d, [0x2a31] = 0x7b7f, [0x2a32] = 0x7b81, [0x2a33] = 0x7b82,
  [0x2a34] = 0x7b83, [0x2a35] = 0x7b84, [0x2a36] = 0x7b86, [0x2a37] = 0x7b87,
  [0x2a38] = 0x7b88, [0x2a39] = 0x7b89, [0x2a3a] = 0x7b8a, [0x2a3b] = 0x7b8b,
  [0x2a3c] = 0x7b8c, [0x2a3d] = 0x7b8e, [0x2a3e] = 0x7b8f, [0x2a40] = 0x7b91,
  [0x2a41] = 0x7b92, [0x2a42] = 0x7b93, [0x2a43] = 0x7b96, [0x2a44] = 0x7b98,
  [0x2a45] = 0x7b99, [0x2a46] = 0x7b9a, [0x2a47] = 0x7b9b, [0x2a48] = 0x7b9e,
  [0x2a49] = 0x7b9f, [0x2a4a] = 0x7ba0, [0x2a4b] = 0x7ba3, [0x2a4c] = 0x7ba4,
  [0x2a4d] = 0x7ba5, [0x2a4e] = 0x7bae, [0x2a4f] = 0x7baf, [0x2a50] = 0x7bb0,
  [0x2a51] = 0x7bb2, [0x2a52] = 0x7bb3, [0x2a53] = 0x7bb5, [0x2a54] = 0x7bb6,
  [0x2a55] = 0x7bb7, [0x2a56] = 0x7bb9, [0x2a57] = 0x7bba, [0x2a58] = 0x7bbb,
  [0x2a59] = 0x7bbc, [0x2a5a] = 0x7bbd, [0x2a5b] = 0x7bbe, [0x2a5c] = 0x7bbf,
  [0x2a5d] = 0x7bc0, [0x2a5e] = 0x7bc2, [0x2a5f] = 0x7bc3, [0x2a60] = 0x7bc4,
  [0x2a61] = 0x57c2, [0x2a62] = 0x803f, [0x2a63] = 0x6897, [0x2a64] = 0x5de5,
  [0x2a65] = 0x653b, [0x2a66] = 0x529f, [0x2a67] = 0x606d, [0x2a68] = 0x9f9a,
  [0x2a69] = 0x4f9b, [0x2a6a] = 0x8eac, [0x2a6b] = 0x516c, [0x2a6c] = 0x5bab,
  [0x2a6d] = 0x5f13, [0x2a6e] = 0x5de9, [0x2a6f] = 0x6c5e, [0x2a70] = 0x62f1,
  [0x2a71] = 0x8d21, [0x2a72] = 0x5171, [0x2a73] = 0x94a9, [0x2a74] = 0x52fe,
  [0x2a75] = 0x6c9f, [0x2a76] = 0x82df, [0x2a77] = 0x72d7, [0x2a78] = 0x57a2,
  [0x2a79] = 0x6784, [0x2a7a] = 0x8d2d, [0x2a7b] = 0x591f, [0x2a7c] = 0x8f9c,
  [0x2a7d] = 0x83c7, [0x2a7e] = 0x5495, [0x2a7f] = 0x7b8d, [0x2a80] = 0x4f30,
  [0x2a81] = 0x6cbd, [0x2a82] = 0x5b64, [0x2a83] = 0x59d1, [0x2a84] = 0x9f13,
  [0x2a85] = 0x53e4, [0x2a86] = 0x86ca, [0x2a87] = 0x9aa8, [0x2a88] = 0x8c37,
  [0x2a89] = 0x80a1, [0x2a8a] = 0x6545, [0x2a8b] = 0x987e, [0x2a8c] = 0x56fa,
  [0x2a8d] = 0x96c7, [0x2a8e] = 0x522e, [0x2a8f] = 0x74dc, [0x2a90] = 0x5250,
  [0x2a91] = 0x5be1, [0x2a92] = 0x6302, [0x2a93] = 0x8902, [0x2a94] = 0x4e56,
  [0x2a95] = 0x62d0, [0x2a96] = 0x602a, [0x2a97] = 0x68fa, [0x2a98] = 0x5173,
  [0x2a99] = 0x5b98, [0x2a9a] = 0x51a0, [0x2a9b] = 0x89c2, [0x2a9c] = 0x7ba1,
  [0x2a9d] = 0x9986, [0x2a9e] = 0x7f50, [0x2a9f] = 0x60ef, [0x2aa0] = 0x704c,
  [0x2aa1] = 0x8d2f, [0x2aa2] = 0x5149, [0x2aa3] = 0x5e7f, [0x2aa4] = 0x901b,
  [0x2aa5] = 0x7470, [0x2aa6] = 0x89c4, [0x2aa7] = 0x572d, [0x2aa8] = 0x7845,
  [0x2aa9] = 0x5f52, [0x2aaa] = 0x9f9f, [0x2aab] = 0x95fa, [0x2aac] = 0x8f68,
  [0x2aad] = 0x9b3c, [0x2aae] = 0x8be1, [0x2aaf] = 0x7678, [0x2ab0] = 0x6842,
  [0x2ab1] = 0x67dc, [0x2ab2] = 0x8dea, [0x2ab3] = 0x8d35, [0x2ab4] = 0x523d,
  [0x2ab5] = 0x8f8a, [0x2ab6] = 0x6eda, [0x2ab7] = 0x68cd, [0x2ab8] = 0x9505,
  [0x2ab9] = 0x90ed, [0x2aba] = 0x56fd, [0x2abb] = 0x679c, [0x2abc] = 0x88f9,
  [0x2abd] = 0x8fc7, [0x2abe] = 0x54c8, [0x2ac0] = 0x7bc5, [0x2ac1] = 0x7bc8,
  [0x2ac2] = 0x7bc9, [0x2ac3] = 0x7bca, [0x2ac4] = 0x7bcb, [0x2ac5] = 0x7bcd,
  [0x2ac6] = 0x7bce, [0x2ac7] = 0x7bcf, [0x2ac8] = 0x7bd0, [0x2ac9] = 0x7bd2,
  [0x2aca] = 0x7bd4, [0x2acb] = 0x7bd5, [0x2acc] = 0x7bd6, [0x2acd] = 0x7bd7,
  [0x2ace] = 0x7bd8, [0x2acf] = 0x7bdb, [0x2ad0] = 0x7bdc, [0x2ad1] = 0x7bde,
  [0x2ad2] = 0x7bdf, [0x2ad3] = 0x7be0, [0x2ad4] = 0x7be2, [0x2ad5] = 0x7be3,
  [0x2ad6] = 0x7be4, [0x2ad7] = 0x7be7, [0x2ad8] = 0x7be8, [0x2ad9] = 0x7be9,
  [0x2ada] = 0x7beb, [0x2adb] = 0x7bec, [0x2adc] = 0x7bed, [0x2add] = 0x7bef,
  [0x2ade] = 0x7bf0, [0x2adf] = 0x7bf2, [0x2ae0] = 0x7bf3, [0x2ae1] = 0x7bf4,
  [0x2ae2] = 0x7bf5, [0x2ae3] = 0x7bf6, [0x2ae4] = 0x7bf8, [0x2ae5] = 0x7bf9,
  [0x2ae6] = 0x7bfa, [0x2ae7] = 0x7bfb, [0x2ae8] = 0x7bfd, [0x2ae9] = 0x7bff,
  [0x2aea] = 0x7c00, [0x2aeb] = 0x7c01, [0x2aec] = 0x7c02, [0x2aed] = 0x7c03,
  [0x2aee] = 0x7c04, [0x2aef] = 0x7c05, [0x2af0] = 0x7c06, [0x2af1] = 0x7c08,
  [0x2af2] = 0x7c09, [0x2af3] = 0x7c0a, [0x2af4] = 0x7c0d, [0x2af5] = 0x7c0e,
  [0x2af6] = 0x7c10, [0x2af7] = 0x7c11, [0x2af8] = 0x7c12, [0x2af9] = 0x7c13,
  [0x2afa] = 0x7c14, [0x2afb] = 0x7c15, [0x2afc] = 0x7c17, [0x2afd] = 0x7c18,
  [0x2afe] = 0x7c19, [0x2b00] = 0x7c1a, [0x2b01] = 0x7c1b, [0x2b02] = 0x7c1c,
  [0x2b03] = 0x7c1d, [0x2b04] = 0x7c1e, [0x2b05] = 0x7c20, [0x2b06] = 0x7c21,
  [0x2b07] = 0x7c22, [0x2b08] = 0x7c23, [0x2b09] = 0x7c24, [0x2b0a] = 0x7c25,
  [0x2b0b] = 0x7c28, [0x2b0c] = 0x7c29, [0x2b0d] = 0x7c2b, [0x2b0e] = 0x7c2c,
  [0x2b0f] = 0x7c2d, [0x2b10] = 0x7c2e, [0x2b11] = 0x7c2f, [0x2b12] = 0x7c30,
  [0x2b13] = 0x7c31, [0x2b14] = 0x7c32, [0x2b15] = 0x7c33, [0x2b16] = 0x7c34,
  [0x2b17] = 0x7c35, [0x2b18] = 0x7c36, [0x2b19] = 0x7c37, [0x2b1a] = 0x7c39,
  [0x2b1b] = 0x7c3a, [0x2b1c] = 0x7c3b, [0x2b1d] = 0x7c3c, [0x2b1e] = 0x7c3d,
  [0x2b1f] = 0x7c3e, [0x2b20] = 0x7c42, [0x2b21] = 0x9ab8, [0x2b22] = 0x5b69,
  [0x2b23] = 0x6d77, [0x2b24] = 0x6c26, [0x2b25] = 0x4ea5, [0x2b26] = 0x5bb3,
  [0x2b27] = 0x9a87, [0x2b28] = 0x9163, [0x2b29] = 0x61a8, [0x2b2a] = 0x90af,
  [0x2b2b] = 0x97e9, [0x2b2c] = 0x542b, [0x2b2d] = 0x6db5, [0x2b2e] = 0x5bd2,
  [0x2b2f] = 0x51fd, [0x2b30] = 0x558a, [0x2b31] = 0x7f55, [0x2b32] = 0x7ff0,
  [0x2b33] = 0x64bc, [0x2b34] = 0x634d, [0x2b35] = 0x65f1, [0x2b36] = 0x61be,
  [0x2b37] = 0x608d, [0x2b38] = 0x710a, [0x2b39] = 0x6c57, [0x2b3a] = 0x6c49,
  [0x2b3b] = 0x592f, [0x2b3c] = 0x676d, [0x2b3d] = 0x822a, [0x2b3e] = 0x58d5,
  [0x2b3f] = 0x568e, [0x2b40] = 0x8c6a, [0x2b41] = 0x6beb, [0x2b42] = 0x90dd,
  [0x2b43] = 0x597d, [0x2b44] = 0x8017, [0x2b45] = 0x53f7, [0x2b46] = 0x6d69,
  [0x2b47] = 0x5475, [0x2b48] = 0x559d, [0x2b49] = 0x8377, [0x2b4a] = 0x83cf,
  [0x2b4b] = 0x6838, [0x2b4c] = 0x79be, [0x2b4d] = 0x548c, [0x2b4e] = 0x4f55,
  [0x2b4f] = 0x5408, [0x2b50] = 0x76d2, [0x2b51] = 0x8c89, [0x2b52] = 0x9602,
  [0x2b53] = 0x6cb3, [0x2b54] = 0x6db8, [0x2b55] = 0x8d6b, [0x2b56] = 0x8910,
  [0x2b57] = 0x9e64, [0x2b58] = 0x8d3a, [0x2b59] = 0x563f, [0x2b5a] = 0x9ed1,
  [0x2b5b] = 0x75d5, [0x2b5c] = 0x5f88, [0x2b5d] = 0x72e0, [0x2b5e] = 0x6068,
  [0x2b5f] = 0x54fc, [0x2b60] = 0x4ea8, [0x2b61] = 0x6a2a, [0x2b62] = 0x8861,
  [0x2b63] = 0x6052, [0x2b64] = 0x8f70, [0x2b65] = 0x54c4, [0x2b66] = 0x70d8,
  [0x2b67] = 0x8679, [0x2b68] = 0x9e3f, [0x2b69] = 0x6d2a, [0x2b6a] = 0x5b8f,
  [0x2b6b] = 0x5f18, [0x2b6c] = 0x7ea2, [0x2b6d] = 0x5589, [0x2b6e] = 0x4faf,
  [0x2b6f] = 0x7334, [0x2b70] = 0x543c, [0x2b71] = 0x539a, [0x2b72] = 0x5019,
  [0x2b73] = 0x540e, [0x2b74] = 0x547c, [0x2b75] = 0x4e4e, [0x2b76] = 0x5ffd,
  [0x2b77] = 0x745a, [0x2b78] = 0x58f6, [0x2b79] = 0x846b, [0x2b7a] = 0x80e1,
  [0x2b7b] = 0x8774, [0x2b7c] = 0x72d0, [0x2b7d] = 0x7cca, [0x2b7e] = 0x6e56,
  [0x2b80] = 0x7c43, [0x2b81] = 0x7c44, [0x2b82] = 0x7c45, [0x2b83] = 0x7c46,
  [0x2b84] = 0x7c47, [0x2b85] = 0x7c48, [0x2b86] = 0x7c49, [0x2b87] = 0x7c4a,
  [0x2b88] = 0x7c4b, [0x2b89] = 0x7c4c, [0x2b8a] = 0x7c4e, [0x2b8b] = 0x7c4f,
  [0x2b8c] = 0x7c50, [0x2b8d] = 0x7c51, [0x2b8e] = 0x7c52, [0x2b8f] = 0x7c53,
  [0x2b90] = 0x7c54, [0x2b91] = 0x7c55, [0x2b92] = 0x7c56, [0x2b93] = 0x7c57,
  [0x2b94] = 0x7c58, [0x2b95] = 0x7c59, [0x2b96] = 0x7c5a, [0x2b97] = 0x7c5b,
  [0x2b98] = 0x7c5c, [0x2b99] = 0x7c5d, [0x2b9a] = 0x7c5e, [0x2b9b] = 0x7c5f,
  [0x2b9c] = 0x7c60, [0x2b9d] = 0x7c61, [0x2b9e] = 0x7c62, [0x2b9f] = 0x7c63,
  [0x2ba0] = 0x7c64, [0x2ba1] = 0x7c65, [0x2ba2] = 0x7c66, [0x2ba3] = 0x7c67,
  [0x2ba4] = 0x7c68, [0x2ba5] = 0x7c69, [0x2ba6] = 0x7c6a, [0x2ba7] = 0x7c6b,
  [0x2ba8] = 0x7c6c, [0x2ba9] = 0x7c6d, [0x2baa] = 0x7c6e, [0x2bab] = 0x7c6f,
  [0x2bac] = 0x7c70, [0x2bad] = 0x7c71, [0x2bae] = 0x7c72, [0x2baf] = 0x7c75,
  [0x2bb0] = 0x7c76, [0x2bb1] = 0x7c77, [0x2bb2] = 0x7c78, [0x2bb3] = 0x7c79,
  [0x2bb4] = 0x7c7a, [0x2bb5] = 0x7c7e, [0x2bb6] = 0x7c7f, [0x2bb7] = 0x7c80,
  [0x2bb8] = 0x7c81, [0x2bb9] = 0x7c82, [0x2bba] = 0x7c83, [0x2bbb] = 0x7c84,
  [0x2bbc] = 0x7c85, [0x2bbd] = 0x7c86, [0x2bbe] = 0x7c87, [0x2bc0] = 0x7c88,
  [0x2bc1] = 0x7c8a, [0x2bc2] = 0x7c8b, [0x2bc3] = 0x7c8c, [0x2bc4] = 0x7c8d,
  [0x2bc5] = 0x7c8e, [0x2bc6] = 0x7c8f, [0x2bc7] = 0x7c90, [0x2bc8] = 0x7c93,
  [0x2bc9] = 0x7c94, [0x2bca] = 0x7c96, [0x2bcb] = 0x7c99, [0x2bcc] = 0x7c9a,
  [0x2bcd] = 0x7c9b, [0x2bce] = 0x7ca0, [0x2bcf] = 0x7ca1, [0x2bd0] = 0x7ca3,
  [0x2bd1] = 0x7ca6, [0x2bd2] = 0x7ca7, [0x2bd3] = 0x7ca8, [0x2bd4] = 0x7ca9,
  [0x2bd5] = 0x7cab, [0x2bd6] = 0x7cac, [0x2bd7] = 0x7cad, [0x2bd8] = 0x7caf,
  [0x2bd9] = 0x7cb0, [0x2bda] = 0x7cb4, [0x2bdb] = 0x7cb5, [0x2bdc] = 0x7cb6,
  [0x2bdd] = 0x7cb7, [0x2bde] = 0x7cb8, [0x2bdf] = 0x7cba, [0x2be0] = 0x7cbb,
  [0x2be1] = 0x5f27, [0x2be2] = 0x864e, [0x2be3] = 0x552c, [0x2be4] = 0x62a4,
  [0x2be5] = 0x4e92, [0x2be6] = 0x6caa, [0x2be7] = 0x6237, [0x2be8] = 0x82b1,
  [0x2be9] = 0x54d7, [0x2bea] = 0x534e, [0x2beb] = 0x733e, [0x2bec] = 0x6ed1,
  [0x2bed] = 0x753b, [0x2bee] = 0x5212, [0x2bef] = 0x5316, [0x2bf0] = 0x8bdd,
  [0x2bf1] = 0x69d0, [0x2bf2] = 0x5f8a, [0x2bf3] = 0x6000, [0x2bf4] = 0x6dee,
  [0x2bf5] = 0x574f, [0x2bf6] = 0x6b22, [0x2bf7] = 0x73af, [0x2bf8] = 0x6853,
  [0x2bf9] = 0x8fd8, [0x2bfa] = 0x7f13, [0x2bfb] = 0x6362, [0x2bfc] = 0x60a3,
  [0x2bfd] = 0x5524, [0x2bfe] = 0x75ea, [0x2bff] = 0x8c62, [0x2c00] = 0x7115,
  [0x2c01] = 0x6da3, [0x2c02] = 0x5ba6, [0x2c03] = 0x5e7b, [0x2c04] = 0x8352,
  [0x2c05] = 0x614c, [0x2c06] = 0x9ec4, [0x2c07] = 0x78fa, [0x2c08] = 0x8757,
  [0x2c09] = 0x7c27, [0x2c0a] = 0x7687, [0x2c0b] = 0x51f0, [0x2c0c] = 0x60f6,
  [0x2c0d] = 0x714c, [0x2c0e] = 0x6643, [0x2c0f] = 0x5e4c, [0x2c10] = 0x604d,
  [0x2c11] = 0x8c0e, [0x2c12] = 0x7070, [0x2c13] = 0x6325, [0x2c14] = 0x8f89,
  [0x2c15] = 0x5fbd, [0x2c16] = 0x6062, [0x2c17] = 0x86d4, [0x2c18] = 0x56de,
  [0x2c19] = 0x6bc1, [0x2c1a] = 0x6094, [0x2c1b] = 0x6167, [0x2c1c] = 0x5349,
  [0x2c1d] = 0x60e0, [0x2c1e] = 0x6666, [0x2c1f] = 0x8d3f, [0x2c20] = 0x79fd,
  [0x2c21] = 0x4f1a, [0x2c22] = 0x70e9, [0x2c23] = 0x6c47, [0x2c24] = 0x8bb3,
  [0x2c25] = 0x8bf2, [0x2c26] = 0x7ed8, [0x2c27] = 0x8364, [0x2c28] = 0x660f,
  [0x2c29] = 0x5a5a, [0x2c2a] = 0x9b42, [0x2c2b] = 0x6d51, [0x2c2c] = 0x6df7,
  [0x2c2d] = 0x8c41, [0x2c2e] = 0x6d3b, [0x2c2f] = 0x4f19, [0x2c30] = 0x706b,
  [0x2c31] = 0x83b7, [0x2c32] = 0x6216, [0x2c33] = 0x60d1, [0x2c34] = 0x970d,
  [0x2c35] = 0x8d27, [0x2c36] = 0x7978, [0x2c37] = 0x51fb, [0x2c38] = 0x573e,
  [0x2c39] = 0x57fa, [0x2c3a] = 0x673a, [0x2c3b] = 0x7578, [0x2c3c] = 0x7a3d,
  [0x2c3d] = 0x79ef, [0x2c3e] = 0x7b95, [0x2c40] = 0x7cbf, [0x2c41] = 0x7cc0,
  [0x2c42] = 0x7cc2, [0x2c43] = 0x7cc3, [0x2c44] = 0x7cc4, [0x2c45] = 0x7cc6,
  [0x2c46] = 0x7cc9, [0x2c47] = 0x7ccb, [0x2c48] = 0x7cce, [0x2c49] = 0x7ccf,
  [0x2c4a] = 0x7cd0, [0x2c4b] = 0x7cd1, [0x2c4c] = 0x7cd2, [0x2c4d] = 0x7cd3,
  [0x2c4e] = 0x7cd4, [0x2c4f] = 0x7cd8, [0x2c50] = 0x7cda, [0x2c51] = 0x7cdb,
  [0x2c52] = 0x7cdd, [0x2c53] = 0x7cde, [0x2c54] = 0x7ce1, [0x2c55] = 0x7ce2,
  [0x2c56] = 0x7ce3, [0x2c57] = 0x7ce4, [0x2c58] = 0x7ce5, [0x2c59] = 0x7ce6,
  [0x2c5a] = 0x7ce7, [0x2c5b] = 0x7ce9, [0x2c5c] = 0x7cea, [0x2c5d] = 0x7ceb,
  [0x2c5e] = 0x7cec, [0x2c5f] = 0x7ced, [0x2c60] = 0x7cee, [0x2c61] = 0x7cf0,
  [0x2c62] = 0x7cf1, [0x2c63] = 0x7cf2, [0x2c64] = 0x7cf3, [0x2c65] = 0x7cf4,
  [0x2c66] = 0x7cf5, [0x2c67] = 0x7cf6, [0x2c68] = 0x7cf7, [0x2c69] = 0x7cf9,
  [0x2c6a] = 0x7cfa, [0x2c6b] = 0x7cfc, [0x2c6c] = 0x7cfd, [0x2c6d] = 0x7cfe,
  [0x2c6e] = 0x7cff, [0x2c6f] = 0x7d00, [0x2c70] = 0x7d01, [0x2c71] = 0x7d02,
  [0x2c72] = 0x7d03, [0x2c73] = 0x7d04, [0x2c74] = 0x7d05, [0x2c75] = 0x7d06,
  [0x2c76] = 0x7d07, [0x2c77] = 0x7d08, [0x2c78] = 0x7d09, [0x2c79] = 0x7d0b,
  [0x2c7a] = 0x7d0c, [0x2c7b] = 0x7d0d, [0x2c7c] = 0x7d0e, [0x2c7d] = 0x7d0f,
  [0x2c7e] = 0x7d10, [0x2c80] = 0x7d11, [0x2c81] = 0x7d12, [0x2c82] = 0x7d13,
  [0x2c83] = 0x7d14, [0x2c84] = 0x7d15, [0x2c85] = 0x7d16, [0x2c86] = 0x7d17,
  [0x2c87] = 0x7d18, [0x2c88] = 0x7d19, [0x2c89] = 0x7d1a, [0x2c8a] = 0x7d1b,
  [0x2c8b] = 0x7d1c, [0x2c8c] = 0x7d1d, [0x2c8d] = 0x7d1e, [0x2c8e] = 0x7d1f,
  [0x2c8f] = 0x7d21, [0x2c90] = 0x7d23, [0x2c91] = 0x7d24, [0x2c92] = 0x7d25,
  [0x2c93] = 0x7d26, [0x2c94] = 0x7d28, [0x2c95] = 0x7d29, [0x2c96] = 0x7d2a,
  [0x2c97] = 0x7d2c, [0x2c98] = 0x7d2d, [0x2c99] = 0x7d2e, [0x2c9a] = 0x7d30,
  [0x2c9b] = 0x7d31, [0x2c9c] = 0x7d32, [0x2c9d] = 0x7d33, [0x2c9e] = 0x7d34,
  [0x2c9f] = 0x7d35, [0x2ca0] = 0x7d36, [0x2ca1] = 0x808c, [0x2ca2] = 0x9965,
  [0x2ca3] = 0x8ff9, [0x2ca4] = 0x6fc0, [0x2ca5] = 0x8ba5, [0x2ca6] = 0x9e21,
  [0x2ca7] = 0x59ec, [0x2ca8] = 0x7ee9, [0x2ca9] = 0x7f09, [0x2caa] = 0x5409,
  [0x2cab] = 0x6781, [0x2cac] = 0x68d8, [0x2cad] = 0x8f91, [0x2cae] = 0x7c4d,
  [0x2caf] = 0x96c6, [0x2cb0] = 0x53ca, [0x2cb1] = 0x6025, [0x2cb2] = 0x75be,
  [0x2cb3] = 0x6c72, [0x2cb4] = 0x5373, [0x2cb5] = 0x5ac9, [0x2cb6] = 0x7ea7,
  [0x2cb7] = 0x6324, [0x2cb8] = 0x51e0, [0x2cb9] = 0x810a, [0x2cba] = 0x5df1,
  [0x2cbb] = 0x84df, [0x2cbc] = 0x6280, [0x2cbd] = 0x5180, [0x2cbe] = 0x5b63,
  [0x2cbf] = 0x4f0e, [0x2cc0] = 0x796d, [0x2cc1] = 0x5242, [0x2cc2] = 0x60b8,
  [0x2cc3] = 0x6d4e, [0x2cc4] = 0x5bc4, [0x2cc5] = 0x5bc2, [0x2cc6] = 0x8ba1,
  [0x2cc7] = 0x8bb0, [0x2cc8] = 0x65e2, [0x2cc9] = 0x5fcc, [0x2cca] = 0x9645,
  [0x2ccb] = 0x5993, [0x2ccc] = 0x7ee7, [0x2ccd] = 0x7eaa, [0x2cce] = 0x5609,
  [0x2ccf] = 0x67b7, [0x2cd0] = 0x5939, [0x2cd1] = 0x4f73, [0x2cd2] = 0x5bb6,
  [0x2cd3] = 0x52a0, [0x2cd4] = 0x835a, [0x2cd5] = 0x988a, [0x2cd6] = 0x8d3e,
  [0x2cd7] = 0x7532, [0x2cd8] = 0x94be, [0x2cd9] = 0x5047, [0x2cda] = 0x7a3c,
  [0x2cdb] = 0x4ef7, [0x2cdc] = 0x67b6, [0x2cdd] = 0x9a7e, [0x2cde] = 0x5ac1,
  [0x2cdf] = 0x6b7c, [0x2ce0] = 0x76d1, [0x2ce1] = 0x575a, [0x2ce2] = 0x5c16,
  [0x2ce3] = 0x7b3a, [0x2ce4] = 0x95f4, [0x2ce5] = 0x714e, [0x2ce6] = 0x517c,
  [0x2ce7] = 0x80a9, [0x2ce8] = 0x8270, [0x2ce9] = 0x5978, [0x2cea] = 0x7f04,
  [0x2ceb] = 0x8327, [0x2cec] = 0x68c0, [0x2ced] = 0x67ec, [0x2cee] = 0x78b1,
  [0x2cef] = 0x7877, [0x2cf0] = 0x62e3, [0x2cf1] = 0x6361, [0x2cf2] = 0x7b80,
  [0x2cf3] = 0x4fed, [0x2cf4] = 0x526a, [0x2cf5] = 0x51cf, [0x2cf6] = 0x8350,
  [0x2cf7] = 0x69db, [0x2cf8] = 0x9274, [0x2cf9] = 0x8df5, [0x2cfa] = 0x8d31,
  [0x2cfb] = 0x89c1, [0x2cfc] = 0x952e, [0x2cfd] = 0x7bad, [0x2cfe] = 0x4ef6,
  [0x2d00] = 0x7d37, [0x2d01] = 0x7d38, [0x2d02] = 0x7d39, [0x2d03] = 0x7d3a,
  [0x2d04] = 0x7d3b, [0x2d05] = 0x7d3c, [0x2d06] = 0x7d3d, [0x2d07] = 0x7d3e,
  [0x2d08] = 0x7d3f, [0x2d09] = 0x7d40, [0x2d0a] = 0x7d41, [0x2d0b] = 0x7d42,
  [0x2d0c] = 0x7d43, [0x2d0d] = 0x7d44, [0x2d0e] = 0x7d45, [0x2d0f] = 0x7d46,
  [0x2d10] = 0x7d47, [0x2d11] = 0x7d48, [0x2d12] = 0x7d49, [0x2d13] = 0x7d4a,
  [0x2d14] = 0x7d4b, [0x2d15] = 0x7d4c, [0x2d16] = 0x7d4d, [0x2d17] = 0x7d4e,
  [0x2d18] = 0x7d4f, [0x2d19] = 0x7d50, [0x2d1a] = 0x7d51, [0x2d1b] = 0x7d52,
  [0x2d1c] = 0x7d53, [0x2d1d] = 0x7d54, [0x2d1e] = 0x7d55, [0x2d1f] = 0x7d56,
  [0x2d20] = 0x7d57, [0x2d21] = 0x7d58, [0x2d22] = 0x7d59, [0x2d23] = 0x7d5a,
  [0x2d24] = 0x7d5b, [0x2d25] = 0x7d5c, [0x2d26] = 0x7d5d, [0x2d27] = 0x7d5e,
  [0x2d28] = 0x7d5f, [0x2d29] = 0x7d60, [0x2d2a] = 0x7d61, [0x2d2b] = 0x7d62,
  [0x2d2c] = 0x7d63, [0x2d2d] = 0x7d64, [0x2d2e] = 0x7d65, [0x2d2f] = 0x7d66,
  [0x2d30] = 0x7d67, [0x2d31] = 0x7d68, [0x2d32] = 0x7d69, [0x2d33] = 0x7d6a,
  [0x2d34] = 0x7d6b, [0x2d35] = 0x7d6c, [0x2d36] = 0x7d6d, [0x2d37] = 0x7d6f,
  [0x2d38] = 0x7d70, [0x2d39] = 0x7d71, [0x2d3a] = 0x7d72, [0x2d3b] = 0x7d73,
  [0x2d3c] = 0x7d74, [0x2d3d] = 0x7d75, [0x2d3e] = 0x7d76, [0x2d40] = 0x7d78,
  [0x2d41] = 0x7d79, [0x2d42] = 0x7d7a, [0x2d43] = 0x7d7b, [0x2d44] = 0x7d7c,
  [0x2d45] = 0x7d7d, [0x2d46] = 0x7d7e, [0x2d47] = 0x7d7f, [0x2d48] = 0x7d80,
  [0x2d49] = 0x7d81, [0x2d4a] = 0x7d82, [0x2d4b] = 0x7d83, [0x2d4c] = 0x7d84,
  [0x2d4d] = 0x7d85, [0x2d4e] = 0x7d86, [0x2d4f] = 0x7d87, [0x2d50] = 0x7d88,
  [0x2d51] = 0x7d89, [0x2d52] = 0x7d8a, [0x2d53] = 0x7d8b, [0x2d54] = 0x7d8c,
  [0x2d55] = 0x7d8d, [0x2d56] = 0x7d8e, [0x2d57] = 0x7d8f, [0x2d58] = 0x7d90,
  [0x2d59] = 0x7d91, [0x2d5a] = 0x7d92, [0x2d5b] = 0x7d93, [0x2d5c] = 0x7d94,
  [0x2d5d] = 0x7d95, [0x2d5e] = 0x7d96, [0x2d5f] = 0x7d97, [0x2d60] = 0x7d98,
  [0x2d61] = 0x5065, [0x2d62] = 0x8230, [0x2d63] = 0x5251, [0x2d64] = 0x996f,
  [0x2d65] = 0x6e10, [0x2d66] = 0x6e85, [0x2d67] = 0x6da7, [0x2d68] = 0x5efa,
  [0x2d69] = 0x50f5, [0x2d6a] = 0x59dc, [0x2d6b] = 0x5c06, [0x2d6c] = 0x6d46,
  [0x2d6d] = 0x6c5f, [0x2d6e] = 0x7586, [0x2d6f] = 0x848b, [0x2d70] = 0x6868,
  [0x2d71] = 0x5956, [0x2d72] = 0x8bb2, [0x2d73] = 0x5320, [0x2d74] = 0x9171,
  [0x2d75] = 0x964d, [0x2d76] = 0x8549, [0x2d77] = 0x6912, [0x2d78] = 0x7901,
  [0x2d79] = 0x7126, [0x2d7a] = 0x80f6, [0x2d7b] = 0x4ea4, [0x2d7c] = 0x90ca,
  [0x2d7d] = 0x6d47, [0x2d7e] = 0x9a84, [0x2d7f] = 0x5a07, [0x2d80] = 0x56bc,
  [0x2d81] = 0x6405, [0x2d82] = 0x94f0, [0x2d83] = 0x77eb, [0x2d84] = 0x4fa5,
  [0x2d85] = 0x811a, [0x2d86] = 0x72e1, [0x2d87] = 0x89d2, [0x2d88] = 0x997a,
  [0x2d89] = 0x7f34, [0x2d8a] = 0x7ede, [0x2d8b] = 0x527f, [0x2d8c] = 0x6559,
  [0x2d8d] = 0x9175, [0x2d8e] = 0x8f7f, [0x2d8f] = 0x8f83, [0x2d90] = 0x53eb,
  [0x2d91] = 0x7a96, [0x2d92] = 0x63ed, [0x2d93] = 0x63a5, [0x2d94] = 0x7686,
  [0x2d95] = 0x79f8, [0x2d96] = 0x8857, [0x2d97] = 0x9636, [0x2d98] = 0x622a,
  [0x2d99] = 0x52ab, [0x2d9a] = 0x8282, [0x2d9b] = 0x6854, [0x2d9c] = 0x6770,
  [0x2d9d] = 0x6377, [0x2d9e] = 0x776b, [0x2d9f] = 0x7aed, [0x2da0] = 0x6d01,
  [0x2da1] = 0x7ed3, [0x2da2] = 0x89e3, [0x2da3] = 0x59d0, [0x2da4] = 0x6212,
  [0x2da5] = 0x85c9, [0x2da6] = 0x82a5, [0x2da7] = 0x754c, [0x2da8] = 0x501f,
  [0x2da9] = 0x4ecb, [0x2daa] = 0x75a5, [0x2dab] = 0x8beb, [0x2dac] = 0x5c4a,
  [0x2dad] = 0x5dfe, [0x2dae] = 0x7b4b, [0x2daf] = 0x65a4, [0x2db0] = 0x91d1,
  [0x2db1] = 0x4eca, [0x2db2] = 0x6d25, [0x2db3] = 0x895f, [0x2db4] = 0x7d27,
  [0x2db5] = 0x9526, [0x2db6] = 0x4ec5, [0x2db7] = 0x8c28, [0x2db8] = 0x8fdb,
  [0x2db9] = 0x9773, [0x2dba] = 0x664b, [0x2dbb] = 0x7981, [0x2dbc] = 0x8fd1,
  [0x2dbd] = 0x70ec, [0x2dbe] = 0x6d78, [0x2dc0] = 0x7d99, [0x2dc1] = 0x7d9a,
  [0x2dc2] = 0x7d9b, [0x2dc3] = 0x7d9c, [0x2dc4] = 0x7d9d, [0x2dc5] = 0x7d9e,
  [0x2dc6] = 0x7d9f, [0x2dc7] = 0x7da0, [0x2dc8] = 0x7da1, [0x2dc9] = 0x7da2,
  [0x2dca] = 0x7da3, [0x2dcb] = 0x7da4, [0x2dcc] = 0x7da5, [0x2dcd] = 0x7da7,
  [0x2dce] = 0x7da8, [0x2dcf] = 0x7da9, [0x2dd0] = 0x7daa, [0x2dd1] = 0x7dab,
  [0x2dd2] = 0x7dac, [0x2dd3] = 0x7dad, [0x2dd4] = 0x7daf, [0x2dd5] = 0x7db0,
  [0x2dd6] = 0x7db1, [0x2dd7] = 0x7db2, [0x2dd8] = 0x7db3, [0x2dd9] = 0x7db4,
  [0x2dda] = 0x7db5, [0x2ddb] = 0x7db6, [0x2ddc] = 0x7db7, [0x2ddd] = 0x7db8,
  [0x2dde] = 0x7db9, [0x2ddf] = 0x7dba, [0x2de0] = 0x7dbb, [0x2de1] = 0x7dbc,
  [0x2de2] = 0x7dbd, [0x2de3] = 0x7dbe, [0x2de4] = 0x7dbf, [0x2de5] = 0x7dc0,
  [0x2de6] = 0x7dc1, [0x2de7] = 0x7dc2, [0x2de8] = 0x7dc3, [0x2de9] = 0x7dc4,
  [0x2dea] = 0x7dc5, [0x2deb] = 0x7dc6, [0x2dec] = 0x7dc7, [0x2ded] = 0x7dc8,
  [0x2dee] = 0x7dc9, [0x2def] = 0x7dca, [0x2df0] = 0x7dcb, [0x2df1] = 0x7dcc,
  [0x2df2] = 0x7dcd, [0x2df3] = 0x7dce, [0x2df4] = 0x7dcf, [0x2df5] = 0x7dd0,
  [0x2df6] = 0x7dd1, [0x2df7] = 0x7dd2, [0x2df8] = 0x7dd3, [0x2df9] = 0x7dd4,
  [0x2dfa] = 0x7dd5, [0x2dfb] = 0x7dd6, [0x2dfc] = 0x7dd7, [0x2dfd] = 0x7dd8,
  [0x2dfe] = 0x7dd9, [0x2e00] = 0x7dda, [0x2e01] = 0x7ddb, [0x2e02] = 0x7ddc,
  [0x2e03] = 0x7ddd, [0x2e04] = 0x7dde, [0x2e05] = 0x7ddf, [0x2e06] = 0x7de0,
  [0x2e07] = 0x7de1, [0x2e08] = 0x7de2, [0x2e09] = 0x7de3, [0x2e0a] = 0x7de4,
  [0x2e0b] = 0x7de5, [0x2e0c] = 0x7de6, [0x2e0d] = 0x7de7, [0x2e0e] = 0x7de8,
  [0x2e0f] = 0x7de9, [0x2e10] = 0x7dea, [0x2e11] = 0x7deb, [0x2e12] = 0x7dec,
  [0x2e13] = 0x7ded, [0x2e14] = 0x7dee, [0x2e15] = 0x7def, [0x2e16] = 0x7df0,
  [0x2e17] = 0x7df1, [0x2e18] = 0x7df2, [0x2e19] = 0x7df3, [0x2e1a] = 0x7df4,
  [0x2e1b] = 0x7df5, [0x2e1c] = 0x7df6, [0x2e1d] = 0x7df7, [0x2e1e] = 0x7df8,
  [0x2e1f] = 0x7df9, [0x2e20] = 0x7dfa, [0x2e21] = 0x5c3d, [0x2e22] = 0x52b2,
  [0x2e23] = 0x8346, [0x2e24] = 0x5162, [0x2e25] = 0x830e, [0x2e26] = 0x775b,
  [0x2e27] = 0x6676, [0x2e28] = 0x9cb8, [0x2e29] = 0x4eac, [0x2e2a] = 0x60ca,
  [0x2e2b] = 0x7cbe, [0x2e2c] = 0x7cb3, [0x2e2d] = 0x7ecf, [0x2e2e] = 0x4e95,
  [0x2e2f] = 0x8b66, [0x2e30] = 0x666f, [0x2e31] = 0x9888, [0x2e32] = 0x9759,
  [0x2e33] = 0x5883, [0x2e34] = 0x656c, [0x2e35] = 0x955c, [0x2e36] = 0x5f84,
  [0x2e37] = 0x75c9, [0x2e38] = 0x9756, [0x2e39] = 0x7adf, [0x2e3a] = 0x7ade,
  [0x2e3b] = 0x51c0, [0x2e3c] = 0x70af, [0x2e3d] = 0x7a98, [0x2e3e] = 0x63ea,
  [0x2e3f] = 0x7a76, [0x2e40] = 0x7ea0, [0x2e41] = 0x7396, [0x2e42] = 0x97ed,
  [0x2e43] = 0x4e45, [0x2e44] = 0x7078, [0x2e45] = 0x4e5d, [0x2e46] = 0x9152,
  [0x2e47] = 0x53a9, [0x2e48] = 0x6551, [0x2e49] = 0x65e7, [0x2e4a] = 0x81fc,
  [0x2e4b] = 0x8205, [0x2e4c] = 0x548e, [0x2e4d] = 0x5c31, [0x2e4e] = 0x759a,
  [0x2e4f] = 0x97a0, [0x2e50] = 0x62d8, [0x2e51] = 0x72d9, [0x2e52] = 0x75bd,
  [0x2e53] = 0x5c45, [0x2e54] = 0x9a79, [0x2e55] = 0x83ca, [0x2e56] = 0x5c40,
  [0x2e57] = 0x5480, [0x2e58] = 0x77e9, [0x2e59] = 0x4e3e, [0x2e5a] = 0x6cae,
  [0x2e5b] = 0x805a, [0x2e5c] = 0x62d2, [0x2e5d] = 0x636e, [0x2e5e] = 0x5de8,
  [0x2e5f] = 0x5177, [0x2e60] = 0x8ddd, [0x2e61] = 0x8e1e, [0x2e62] = 0x952f,
  [0x2e63] = 0x4ff1, [0x2e64] = 0x53e5, [0x2e65] = 0x60e7, [0x2e66] = 0x70ac,
  [0x2e67] = 0x5267, [0x2e68] = 0x6350, [0x2e69] = 0x9e43, [0x2e6a] = 0x5a1f,
  [0x2e6b] = 0x5026, [0x2e6c] = 0x7737, [0x2e6d] = 0x5377, [0x2e6e] = 0x7ee2,
  [0x2e6f] = 0x6485, [0x2e70] = 0x652b, [0x2e71] = 0x6289, [0x2e72] = 0x6398,
  [0x2e73] = 0x5014, [0x2e74] = 0x7235, [0x2e75] = 0x89c9, [0x2e76] = 0x51b3,
  [0x2e77] = 0x8bc0, [0x2e78] = 0x7edd, [0x2e79] = 0x5747, [0x2e7a] = 0x83cc,
  [0x2e7b] = 0x94a7, [0x2e7c] = 0x519b, [0x2e7d] = 0x541b, [0x2e7e] = 0x5cfb,
  [0x2e80] = 0x7dfb, [0x2e81] = 0x7dfc, [0x2e82] = 0x7dfd, [0x2e83] = 0x7dfe,
  [0x2e84] = 0x7dff, [0x2e85] = 0x7e00, [0x2e86] = 0x7e01, [0x2e87] = 0x7e02,
  [0x2e88] = 0x7e03, [0x2e89] = 0x7e04, [0x2e8a] = 0x7e05, [0x2e8b] = 0x7e06,
  [0x2e8c] = 0x7e07, [0x2e8d] = 0x7e08, [0x2e8e] = 0x7e09, [0x2e8f] = 0x7e0a,
  [0x2e90] = 0x7e0b, [0x2e91] = 0x7e0c, [0x2e92] = 0x7e0d, [0x2e93] = 0x7e0e,
  [0x2e94] = 0x7e0f, [0x2e95] = 0x7e10, [0x2e96] = 0x7e11, [0x2e97] = 0x7e12,
  [0x2e98] = 0x7e13, [0x2e99] = 0x7e14, [0x2e9a] = 0x7e15, [0x2e9b] = 0x7e16,
  [0x2e9c] = 0x7e17, [0x2e9d] = 0x7e18, [0x2e9e] = 0x7e19, [0x2e9f] = 0x7e1a,
  [0x2ea0] = 0x7e1b, [0x2ea1] = 0x7e1c, [0x2ea2] = 0x7e1d, [0x2ea3] = 0x7e1e,
  [0x2ea4] = 0x7e1f, [0x2ea5] = 0x7e20, [0x2ea6] = 0x7e21, [0x2ea7] = 0x7e22,
  [0x2ea8] = 0x7e23, [0x2ea9] = 0x7e24, [0x2eaa] = 0x7e25, [0x2eab] = 0x7e26,
  [0x2eac] = 0x7e27, [0x2ead] = 0x7e28, [0x2eae] = 0x7e29, [0x2eaf] = 0x7e2a,
  [0x2eb0] = 0x7e2b, [0x2eb1] = 0x7e2c, [0x2eb2] = 0x7e2d, [0x2eb3] = 0x7e2e,
  [0x2eb4] = 0x7e2f, [0x2eb5] = 0x7e30, [0x2eb6] = 0x7e31, [0x2eb7] = 0x7e32,
  [0x2eb8] = 0x7e33, [0x2eb9] = 0x7e34, [0x2eba] = 0x7e35, [0x2ebb] = 0x7e36,
  [0x2ebc] = 0x7e37, [0x2ebd] = 0x7e38, [0x2ebe] = 0x7e39, [0x2ec0] = 0x7e3a,
  [0x2ec1] = 0x7e3c, [0x2ec2] = 0x7e3d, [0x2ec3] = 0x7e3e, [0x2ec4] = 0x7e3f,
  [0x2ec5] = 0x7e40, [0x2ec6] = 0x7e42, [0x2ec7] = 0x7e43, [0x2ec8] = 0x7e44,
  [0x2ec9] = 0x7e45, [0x2eca] = 0x7e46, [0x2ecb] = 0x7e48, [0x2ecc] = 0x7e49,
  [0x2ecd] = 0x7e4a, [0x2ece] = 0x7e4b, [0x2ecf] = 0x7e4c, [0x2ed0] = 0x7e4d,
  [0x2ed1] = 0x7e4e, [0x2ed2] = 0x7e4f, [0x2ed3] = 0x7e50, [0x2ed4] = 0x7e51,
  [0x2ed5] = 0x7e52, [0x2ed6] = 0x7e53, [0x2ed7] = 0x7e54, [0x2ed8] = 0x7e55,
  [0x2ed9] = 0x7e56, [0x2eda] = 0x7e57, [0x2edb] = 0x7e58, [0x2edc] = 0x7e59,
  [0x2edd] = 0x7e5a, [0x2ede] = 0x7e5b, [0x2edf] = 0x7e5c, [0x2ee0] = 0x7e5d,
  [0x2ee1] = 0x4fca, [0x2ee2] = 0x7ae3, [0x2ee3] = 0x6d5a, [0x2ee4] = 0x90e1,
  [0x2ee5] = 0x9a8f, [0x2ee6] = 0x5580, [0x2ee7] = 0x5496, [0x2ee8] = 0x5361,
  [0x2ee9] = 0x54af, [0x2eea] = 0x5f00, [0x2eeb] = 0x63e9, [0x2eec] = 0x6977,
  [0x2eed] = 0x51ef, [0x2eee] = 0x6168, [0x2eef] = 0x520a, [0x2ef0] = 0x582a,
  [0x2ef1] = 0x52d8, [0x2ef2] = 0x574e, [0x2ef3] = 0x780d, [0x2ef4] = 0x770b,
  [0x2ef5] = 0x5eb7, [0x2ef6] = 0x6177, [0x2ef7] = 0x7ce0, [0x2ef8] = 0x625b,
  [0x2ef9] = 0x6297, [0x2efa] = 0x4ea2, [0x2efb] = 0x7095, [0x2efc] = 0x8003,
  [0x2efd] = 0x62f7, [0x2efe] = 0x70e4, [0x2eff] = 0x9760, [0x2f00] = 0x5777,
  [0x2f01] = 0x82db, [0x2f02] = 0x67ef, [0x2f03] = 0x68f5, [0x2f04] = 0x78d5,
  [0x2f05] = 0x9897, [0x2f06] = 0x79d1, [0x2f07] = 0x58f3, [0x2f08] = 0x54b3,
  [0x2f09] = 0x53ef, [0x2f0a] = 0x6e34, [0x2f0b] = 0x514b, [0x2f0c] = 0x523b,
  [0x2f0d] = 0x5ba2, [0x2f0e] = 0x8bfe, [0x2f0f] = 0x80af, [0x2f10] = 0x5543,
  [0x2f11] = 0x57a6, [0x2f12] = 0x6073, [0x2f13] = 0x5751, [0x2f14] = 0x542d,
  [0x2f15] = 0x7a7a, [0x2f16] = 0x6050, [0x2f17] = 0x5b54, [0x2f18] = 0x63a7,
  [0x2f19] = 0x62a0, [0x2f1a] = 0x53e3, [0x2f1b] = 0x6263, [0x2f1c] = 0x5bc7,
  [0x2f1d] = 0x67af, [0x2f1e] = 0x54ed, [0x2f1f] = 0x7a9f, [0x2f20] = 0x82e6,
  [0x2f21] = 0x9177, [0x2f22] = 0x5e93, [0x2f23] = 0x88e4, [0x2f24] = 0x5938,
  [0x2f25] = 0x57ae, [0x2f26] = 0x630e, [0x2f27] = 0x8de8, [0x2f28] = 0x80ef,
  [0x2f29] = 0x5757, [0x2f2a] = 0x7b77, [0x2f2b] = 0x4fa9, [0x2f2c] = 0x5feb,
  [0x2f2d] = 0x5bbd, [0x2f2e] = 0x6b3e, [0x2f2f] = 0x5321, [0x2f30] = 0x7b50,
  [0x2f31] = 0x72c2, [0x2f32] = 0x6846, [0x2f33] = 0x77ff, [0x2f34] = 0x7736,
  [0x2f35] = 0x65f7, [0x2f36] = 0x51b5, [0x2f37] = 0x4e8f, [0x2f38] = 0x76d4,
  [0x2f39] = 0x5cbf, [0x2f3a] = 0x7aa5, [0x2f3b] = 0x8475, [0x2f3c] = 0x594e,
  [0x2f3d] = 0x9b41, [0x2f3e] = 0x5080, [0x2f40] = 0x7e5e, [0x2f41] = 0x7e5f,
  [0x2f42] = 0x7e60, [0x2f43] = 0x7e61, [0x2f44] = 0x7e62, [0x2f45] = 0x7e63,
  [0x2f46] = 0x7e64, [0x2f47] = 0x7e65, [0x2f48] = 0x7e66, [0x2f49] = 0x7e67,
  [0x2f4a] = 0x7e68, [0x2f4b] = 0x7e69, [0x2f4c] = 0x7e6a, [0x2f4d] = 0x7e6b,
  [0x2f4e] = 0x7e6c, [0x2f4f] = 0x7e6d, [0x2f50] = 0x7e6e, [0x2f51] = 0x7e6f,
  [0x2f52] = 0x7e70, [0x2f53] = 0x7e71, [0x2f54] = 0x7e72, [0x2f55] = 0x7e73,
  [0x2f56] = 0x7e74, [0x2f57] = 0x7e75, [0x2f58] = 0x7e76, [0x2f59] = 0x7e77,
  [0x2f5a] = 0x7e78, [0x2f5b] = 0x7e79, [0x2f5c] = 0x7e7a, [0x2f5d] = 0x7e7b,
  [0x2f5e] = 0x7e7c, [0x2f5f] = 0x7e7d, [0x2f60] = 0x7e7e, [0x2f61] = 0x7e7f,
  [0x2f62] = 0x7e80, [0x2f63] = 0x7e81, [0x2f64] = 0x7e83, [0x2f65] = 0x7e84,
  [0x2f66] = 0x7e85, [0x2f67] = 0x7e86, [0x2f68] = 0x7e87, [0x2f69] = 0x7e88,
  [0x2f6a] = 0x7e89, [0x2f6b] = 0x7e8a, [0x2f6c] = 0x7e8b, [0x2f6d] = 0x7e8c,
  [0x2f6e] = 0x7e8d, [0x2f6f] = 0x7e8e, [0x2f70] = 0x7e8f, [0x2f71] = 0x7e90,
  [0x2f72] = 0x7e91, [0x2f73] = 0x7e92, [0x2f74] = 0x7e93, [0x2f75] = 0x7e94,
  [0x2f76] = 0x7e95, [0x2f77] = 0x7e96, [0x2f78] = 0x7e97, [0x2f79] = 0x7e98,
  [0x2f7a] = 0x7e99, [0x2f7b] = 0x7e9a, [0x2f7c] = 0x7e9c, [0x2f7d] = 0x7e9d,
  [0x2f7e] = 0x7e9e, [0x2f80] = 0x7eae, [0x2f81] = 0x7eb4, [0x2f82] = 0x7ebb,
  [0x2f83] = 0x7ebc, [0x2f84] = 0x7ed6, [0x2f85] = 0x7ee4, [0x2f86] = 0x7eec,
  [0x2f87] = 0x7ef9, [0x2f88] = 0x7f0a, [0x2f89] = 0x7f10, [0x2f8a] = 0x7f1e,
  [0x2f8b] = 0x7f37, [0x2f8c] = 0x7f39, [0x2f8d] = 0x7f3b, [0x2f8e] = 0x7f3c,
  [0x2f8f] = 0x7f3d, [0x2f90] = 0x7f3e, [0x2f91] = 0x7f3f, [0x2f92] = 0x7f40,
  [0x2f93] = 0x7f41, [0x2f94] = 0x7f43, [0x2f95] = 0x7f46, [0x2f96] = 0x7f47,
  [0x2f97] = 0x7f48, [0x2f98] = 0x7f49, [0x2f99] = 0x7f4a, [0x2f9a] = 0x7f4b,
  [0x2f9b] = 0x7f4c, [0x2f9c] = 0x7f4d, [0x2f9d] = 0x7f4e, [0x2f9e] = 0x7f4f,
  [0x2f9f] = 0x7f52, [0x2fa0] = 0x7f53, [0x2fa1] = 0x9988, [0x2fa2] = 0x6127,
  [0x2fa3] = 0x6e83, [0x2fa4] = 0x5764, [0x2fa5] = 0x6606, [0x2fa6] = 0x6346,
  [0x2fa7] = 0x56f0, [0x2fa8] = 0x62ec, [0x2fa9] = 0x6269, [0x2faa] = 0x5ed3,
  [0x2fab] = 0x9614, [0x2fac] = 0x5783, [0x2fad] = 0x62c9, [0x2fae] = 0x5587,
  [0x2faf] = 0x8721, [0x2fb0] = 0x814a, [0x2fb1] = 0x8fa3, [0x2fb2] = 0x5566,
  [0x2fb3] = 0x83b1, [0x2fb4] = 0x6765, [0x2fb5] = 0x8d56, [0x2fb6] = 0x84dd,
  [0x2fb7] = 0x5a6a, [0x2fb8] = 0x680f, [0x2fb9] = 0x62e6, [0x2fba] = 0x7bee,
  [0x2fbb] = 0x9611, [0x2fbc] = 0x5170, [0x2fbd] = 0x6f9c, [0x2fbe] = 0x8c30,
  [0x2fbf] = 0x63fd, [0x2fc0] = 0x89c8, [0x2fc1] = 0x61d2, [0x2fc2] = 0x7f06,
  [0x2fc3] = 0x70c2, [0x2fc4] = 0x6ee5, [0x2fc5] = 0x7405, [0x2fc6] = 0x6994,
  [0x2fc7] = 0x72fc, [0x2fc8] = 0x5eca, [0x2fc9] = 0x90ce, [0x2fca] = 0x6717,
  [0x2fcb] = 0x6d6a, [0x2fcc] = 0x635e, [0x2fcd] = 0x52b3, [0x2fce] = 0x7262,
  [0x2fcf] = 0x8001, [0x2fd0] = 0x4f6c, [0x2fd1] = 0x59e5, [0x2fd2] = 0x916a,
  [0x2fd3] = 0x70d9, [0x2fd4] = 0x6d9d, [0x2fd5] = 0x52d2, [0x2fd6] = 0x4e50,
  [0x2fd7] = 0x96f7, [0x2fd8] = 0x956d, [0x2fd9] = 0x857e, [0x2fda] = 0x78ca,
  [0x2fdb] = 0x7d2f, [0x2fdc] = 0x5121, [0x2fdd] = 0x5792, [0x2fde] = 0x64c2,
  [0x2fdf] = 0x808b, [0x2fe0] = 0x7c7b, [0x2fe1] = 0x6cea, [0x2fe2] = 0x68f1,
  [0x2fe3] = 0x695e, [0x2fe4] = 0x51b7, [0x2fe5] = 0x5398, [0x2fe6] = 0x68a8,
  [0x2fe7] = 0x7281, [0x2fe8] = 0x9ece, [0x2fe9] = 0x7bf1, [0x2fea] = 0x72f8,
  [0x2feb] = 0x79bb, [0x2fec] = 0x6f13, [0x2fed] = 0x7406, [0x2fee] = 0x674e,
  [0x2fef] = 0x91cc, [0x2ff0] = 0x9ca4, [0x2ff1] = 0x793c, [0x2ff2] = 0x8389,
  [0x2ff3] = 0x8354, [0x2ff4] = 0x540f, [0x2ff5] = 0x6817, [0x2ff6] = 0x4e3d,
  [0x2ff7] = 0x5389, [0x2ff8] = 0x52b1, [0x2ff9] = 0x783e, [0x2ffa] = 0x5386,
  [0x2ffb] = 0x5229, [0x2ffc] = 0x5088, [0x2ffd] = 0x4f8b, [0x2ffe] = 0x4fd0,
  [0x3000] = 0x7f56, [0x3001] = 0x7f59, [0x3002] = 0x7f5b, [0x3003] = 0x7f5c,
  [0x3004] = 0x7f5d, [0x3005] = 0x7f5e, [0x3006] = 0x7f60, [0x3007] = 0x7f63,
  [0x3008] = 0x7f64, [0x3009] = 0x7f65, [0x300a] = 0x7f66, [0x300b] = 0x7f67,
  [0x300c] = 0x7f6b, [0x300d] = 0x7f6c, [0x300e] = 0x7f6d, [0x300f] = 0x7f6f,
  [0x3010] = 0x7f70, [0x3011] = 0x7f73, [0x3012] = 0x7f75, [0x3013] = 0x7f76,
  [0x3014] = 0x7f77, [0x3015] = 0x7f78, [0x3016] = 0x7f7a, [0x3017] = 0x7f7b,
  [0x3018] = 0x7f7c, [0x3019] = 0x7f7d, [0x301a] = 0x7f7f, [0x301b] = 0x7f80,
  [0x301c] = 0x7f82, [0x301d] = 0x7f83, [0x301e] = 0x7f84, [0x301f] = 0x7f85,
  [0x3020] = 0x7f86, [0x3021] = 0x7f87, [0x3022] = 0x7f88, [0x3023] = 0x7f89,
  [0x3024] = 0x7f8b, [0x3025] = 0x7f8d, [0x3026] = 0x7f8f, [0x3027] = 0x7f90,
  [0x3028] = 0x7f91, [0x3029] = 0x7f92, [0x302a] = 0x7f93, [0x302b] = 0x7f95,
  [0x302c] = 0x7f96, [0x302d] = 0x7f97, [0x302e] = 0x7f98, [0x302f] = 0x7f99,
  [0x3030] = 0x7f9b, [0x3031] = 0x7f9c, [0x3032] = 0x7fa0, [0x3033] = 0x7fa2,
  [0x3034] = 0x7fa3, [0x3035] = 0x7fa5, [0x3036] = 0x7fa6, [0x3037] = 0x7fa8,
  [0x3038] = 0x7fa9, [0x3039] = 0x7faa, [0x303a] = 0x7fab, [0x303b] = 0x7fac,
  [0x303c] = 0x7fad, [0x303d] = 0x7fae, [0x303e] = 0x7fb1, [0x3040] = 0x7fb3,
  [0x3041] = 0x7fb4, [0x3042] = 0x7fb5, [0x3043] = 0x7fb6, [0x3044] = 0x7fb7,
  [0x3045] = 0x7fba, [0x3046] = 0x7fbb, [0x3047] = 0x7fbe, [0x3048] = 0x7fc0,
  [0x3049] = 0x7fc2, [0x304a] = 0x7fc3, [0x304b] = 0x7fc4, [0x304c] = 0x7fc6,
  [0x304d] = 0x7fc7, [0x304e] = 0x7fc8, [0x304f] = 0x7fc9, [0x3050] = 0x7fcb,
  [0x3051] = 0x7fcd, [0x3052] = 0x7fcf, [0x3053] = 0x7fd0, [0x3054] = 0x7fd1,
  [0x3055] = 0x7fd2, [0x3056] = 0x7fd3, [0x3057] = 0x7fd6, [0x3058] = 0x7fd7,
  [0x3059] = 0x7fd9, [0x305a] = 0x7fda, [0x305b] = 0x7fdb, [0x305c] = 0x7fdc,
  [0x305d] = 0x7fdd, [0x305e] = 0x7fde, [0x305f] = 0x7fe2, [0x3060] = 0x7fe3,
  [0x3061] = 0x75e2, [0x3062] = 0x7acb, [0x3063] = 0x7c92, [0x3064] = 0x6ca5,
  [0x3065] = 0x96b6, [0x3066] = 0x529b, [0x3067] = 0x7483, [0x3068] = 0x54e9,
  [0x3069] = 0x4fe9, [0x306a] = 0x8054, [0x306b] = 0x83b2, [0x306c] = 0x8fde,
  [0x306d] = 0x9570, [0x306e] = 0x5ec9, [0x306f] = 0x601c, [0x3070] = 0x6d9f,
  [0x3071] = 0x5e18, [0x3072] = 0x655b, [0x3073] = 0x8138, [0x3074] = 0x94fe,
  [0x3075] = 0x604b, [0x3076] = 0x70bc, [0x3077] = 0x7ec3, [0x3078] = 0x7cae,
  [0x3079] = 0x51c9, [0x307a] = 0x6881, [0x307b] = 0x7cb1, [0x307c] = 0x826f,
  [0x307d] = 0x4e24, [0x307e] = 0x8f86, [0x307f] = 0x91cf, [0x3080] = 0x667e,
  [0x3081] = 0x4eae, [0x3082] = 0x8c05, [0x3083] = 0x64a9, [0x3084] = 0x804a,
  [0x3085] = 0x50da, [0x3086] = 0x7597, [0x3087] = 0x71ce, [0x3088] = 0x5be5,
  [0x3089] = 0x8fbd, [0x308a] = 0x6f66, [0x308b] = 0x4e86, [0x308c] = 0x6482,
  [0x308d] = 0x9563, [0x308e] = 0x5ed6, [0x308f] = 0x6599, [0x3090] = 0x5217,
  [0x3091] = 0x88c2, [0x3092] = 0x70c8, [0x3093] = 0x52a3, [0x3094] = 0x730e,
  [0x3095] = 0x7433, [0x3096] = 0x6797, [0x3097] = 0x78f7, [0x3098] = 0x9716,
  [0x3099] = 0x4e34, [0x309a] = 0x90bb, [0x309b] = 0x9cde, [0x309c] = 0x6dcb,
  [0x309d] = 0x51db, [0x309e] = 0x8d41, [0x309f] = 0x541d, [0x30a0] = 0x62ce,
  [0x30a1] = 0x73b2, [0x30a2] = 0x83f1, [0x30a3] = 0x96f6, [0x30a4] = 0x9f84,
  [0x30a5] = 0x94c3, [0x30a6] = 0x4f36, [0x30a7] = 0x7f9a, [0x30a8] = 0x51cc,
  [0x30a9] = 0x7075, [0x30aa] = 0x9675, [0x30ab] = 0x5cad, [0x30ac] = 0x9886,
  [0x30ad] = 0x53e6, [0x30ae] = 0x4ee4, [0x30af] = 0x6e9c, [0x30b0] = 0x7409,
  [0x30b1] = 0x69b4, [0x30b2] = 0x786b, [0x30b3] = 0x998f, [0x30b4] = 0x7559,
  [0x30b5] = 0x5218, [0x30b6] = 0x7624, [0x30b7] = 0x6d41, [0x30b8] = 0x67f3,
  [0x30b9] = 0x516d, [0x30ba] = 0x9f99, [0x30bb] = 0x804b, [0x30bc] = 0x5499,
  [0x30bd] = 0x7b3c, [0x30be] = 0x7abf, [0x30c0] = 0x7fe4, [0x30c1] = 0x7fe7,
  [0x30c2] = 0x7fe8, [0x30c3] = 0x7fea, [0x30c4] = 0x7feb, [0x30c5] = 0x7fec,
  [0x30c6] = 0x7fed, [0x30c7] = 0x7fef, [0x30c8] = 0x7ff2, [0x30c9] = 0x7ff4,
  [0x30ca] = 0x7ff5, [0x30cb] = 0x7ff6, [0x30cc] = 0x7ff7, [0x30cd] = 0x7ff8,
  [0x30ce] = 0x7ff9, [0x30cf] = 0x7ffa, [0x30d0] = 0x7ffd, [0x30d1] = 0x7ffe,
  [0x30d2] = 0x7fff, [0x30d3] = 0x8002, [0x30d4] = 0x8007, [0x30d5] = 0x8008,
  [0x30d6] = 0x8009, [0x30d7] = 0x800a, [0x30d8] = 0x800e, [0x30d9] = 0x800f,
  [0x30da] = 0x8011, [0x30db] = 0x8013, [0x30dc] = 0x801a, [0x30dd] = 0x801b,
  [0x30de] = 0x801d, [0x30df] = 0x801e, [0x30e0] = 0x801f, [0x30e1] = 0x8021,
  [0x30e2] = 0x8023, [0x30e3] = 0x8024, [0x30e4] = 0x802b, [0x30e5] = 0x802c,
  [0x30e6] = 0x802d, [0x30e7] = 0x802e, [0x30e8] = 0x802f, [0x30e9] = 0x8030,
  [0x30ea] = 0x8032, [0x30eb] = 0x8034, [0x30ec] = 0x8039, [0x30ed] = 0x803a,
  [0x30ee] = 0x803c, [0x30ef] = 0x803e, [0x30f0] = 0x8040, [0x30f1] = 0x8041,
  [0x30f2] = 0x8044, [0x30f3] = 0x8045, [0x30f4] = 0x8047, [0x30f5] = 0x8048,
  [0x30f6] = 0x8049, [0x30f7] = 0x804e, [0x30f8] = 0x804f, [0x30f9] = 0x8050,
  [0x30fa] = 0x8051, [0x30fb] = 0x8053, [0x30fc] = 0x8055, [0x30fd] = 0x8056,
  [0x30fe] = 0x8057, [0x3100] = 0x8059, [0x3101] = 0x805b, [0x3102] = 0x805c,
  [0x3103] = 0x805d, [0x3104] = 0x805e, [0x3105] = 0x805f, [0x3106] = 0x8060,
  [0x3107] = 0x8061, [0x3108] = 0x8062, [0x3109] = 0x8063, [0x310a] = 0x8064,
  [0x310b] = 0x8065, [0x310c] = 0x8066, [0x310d] = 0x8067, [0x310e] = 0x8068,
  [0x310f] = 0x806b, [0x3110] = 0x806c, [0x3111] = 0x806d, [0x3112] = 0x806e,
  [0x3113] = 0x806f, [0x3114] = 0x8070, [0x3115] = 0x8072, [0x3116] = 0x8073,
  [0x3117] = 0x8074, [0x3118] = 0x8075, [0x3119] = 0x8076, [0x311a] = 0x8077,
  [0x311b] = 0x8078, [0x311c] = 0x8079, [0x311d] = 0x807a, [0x311e] = 0x807b,
  [0x311f] = 0x807c, [0x3120] = 0x807d, [0x3121] = 0x9686, [0x3122] = 0x5784,
  [0x3123] = 0x62e2, [0x3124] = 0x9647, [0x3125] = 0x697c, [0x3126] = 0x5a04,
  [0x3127] = 0x6402, [0x3128] = 0x7bd3, [0x3129] = 0x6f0f, [0x312a] = 0x964b,
  [0x312b] = 0x82a6, [0x312c] = 0x5362, [0x312d] = 0x9885, [0x312e] = 0x5e90,
  [0x312f] = 0x7089, [0x3130] = 0x63b3, [0x3131] = 0x5364, [0x3132] = 0x864f,
  [0x3133] = 0x9c81, [0x3134] = 0x9e93, [0x3135] = 0x788c, [0x3136] = 0x9732,
  [0x3137] = 0x8def, [0x3138] = 0x8d42, [0x3139] = 0x9e7f, [0x313a] = 0x6f5e,
  [0x313b] = 0x7984, [0x313c] = 0x5f55, [0x313d] = 0x9646, [0x313e] = 0x622e,
  [0x313f] = 0x9a74, [0x3140] = 0x5415, [0x3141] = 0x94dd, [0x3142] = 0x4fa3,
  [0x3143] = 0x65c5, [0x3144] = 0x5c65, [0x3145] = 0x5c61, [0x3146] = 0x7f15,
  [0x3147] = 0x8651, [0x3148] = 0x6c2f, [0x3149] = 0x5f8b, [0x314a] = 0x7387,
  [0x314b] = 0x6ee4, [0x314c] = 0x7eff, [0x314d] = 0x5ce6, [0x314e] = 0x631b,
  [0x314f] = 0x5b6a, [0x3150] = 0x6ee6, [0x3151] = 0x5375, [0x3152] = 0x4e71,
  [0x3153] = 0x63a0, [0x3154] = 0x7565, [0x3155] = 0x62a1, [0x3156] = 0x8f6e,
  [0x3157] = 0x4f26, [0x3158] = 0x4ed1, [0x3159] = 0x6ca6, [0x315a] = 0x7eb6,
  [0x315b] = 0x8bba, [0x315c] = 0x841d, [0x315d] = 0x87ba, [0x315e] = 0x7f57,
  [0x315f] = 0x903b, [0x3160] = 0x9523, [0x3161] = 0x7ba9, [0x3162] = 0x9aa1,
  [0x3163] = 0x88f8, [0x3164] = 0x843d, [0x3165] = 0x6d1b, [0x3166] = 0x9a86,
  [0x3167] = 0x7edc, [0x3168] = 0x5988, [0x3169] = 0x9ebb, [0x316a] = 0x739b,
  [0x316b] = 0x7801, [0x316c] = 0x8682, [0x316d] = 0x9a6c, [0x316e] = 0x9a82,
  [0x316f] = 0x561b, [0x3170] = 0x5417, [0x3171] = 0x57cb, [0x3172] = 0x4e70,
  [0x3173] = 0x9ea6, [0x3174] = 0x5356, [0x3175] = 0x8fc8, [0x3176] = 0x8109,
  [0x3177] = 0x7792, [0x3178] = 0x9992, [0x3179] = 0x86ee, [0x317a] = 0x6ee1,
  [0x317b] = 0x8513, [0x317c] = 0x66fc, [0x317d] = 0x6162, [0x317e] = 0x6f2b,
  [0x3180] = 0x807e, [0x3181] = 0x8081, [0x3182] = 0x8082, [0x3183] = 0x8085,
  [0x3184] = 0x8088, [0x3185] = 0x808a, [0x3186] = 0x808d, [0x3187] = 0x808e,
  [0x3188] = 0x808f, [0x3189] = 0x8090, [0x318a] = 0x8091, [0x318b] = 0x8092,
  [0x318c] = 0x8094, [0x318d] = 0x8095, [0x318e] = 0x8097, [0x318f] = 0x8099,
  [0x3190] = 0x809e, [0x3191] = 0x80a3, [0x3192] = 0x80a6, [0x3193] = 0x80a7,
  [0x3194] = 0x80a8, [0x3195] = 0x80ac, [0x3196] = 0x80b0, [0x3197] = 0x80b3,
  [0x3198] = 0x80b5, [0x3199] = 0x80b6, [0x319a] = 0x80b8, [0x319b] = 0x80b9,
  [0x319c] = 0x80bb, [0x319d] = 0x80c5, [0x319e] = 0x80c7, [0x319f] = 0x80c8,
  [0x31a0] = 0x80c9, [0x31a1] = 0x80ca, [0x31a2] = 0x80cb, [0x31a3] = 0x80cf,
  [0x31a4] = 0x80d0, [0x31a5] = 0x80d1, [0x31a6] = 0x80d2, [0x31a7] = 0x80d3,
  [0x31a8] = 0x80d4, [0x31a9] = 0x80d5, [0x31aa] = 0x80d8, [0x31ab] = 0x80df,
  [0x31ac] = 0x80e0, [0x31ad] = 0x80e2, [0x31ae] = 0x80e3, [0x31af] = 0x80e6,
  [0x31b0] = 0x80ee, [0x31b1] = 0x80f5, [0x31b2] = 0x80f7, [0x31b3] = 0x80f9,
  [0x31b4] = 0x80fb, [0x31b5] = 0x80fe, [0x31b6] = 0x80ff, [0x31b7] = 0x8100,
  [0x31b8] = 0x8101, [0x31b9] = 0x8103, [0x31ba] = 0x8104, [0x31bb] = 0x8105,
  [0x31bc] = 0x8107, [0x31bd] = 0x8108, [0x31be] = 0x810b, [0x31c0] = 0x810c,
  [0x31c1] = 0x8115, [0x31c2] = 0x8117, [0x31c3] = 0x8119, [0x31c4] = 0x811b,
  [0x31c5] = 0x811c, [0x31c6] = 0x811d, [0x31c7] = 0x811f, [0x31c8] = 0x8120,
  [0x31c9] = 0x8121, [0x31ca] = 0x8122, [0x31cb] = 0x8123, [0x31cc] = 0x8124,
  [0x31cd] = 0x8125, [0x31ce] = 0x8126, [0x31cf] = 0x8127, [0x31d0] = 0x8128,
  [0x31d1] = 0x8129, [0x31d2] = 0x812a, [0x31d3] = 0x812b, [0x31d4] = 0x812d,
  [0x31d5] = 0x812e, [0x31d6] = 0x8130, [0x31d7] = 0x8133, [0x31d8] = 0x8134,
  [0x31d9] = 0x8135, [0x31da] = 0x8137, [0x31db] = 0x8139, [0x31dc] = 0x813a,
  [0x31dd] = 0x813b, [0x31de] = 0x813c, [0x31df] = 0x813d, [0x31e0] = 0x813f,
  [0x31e1] = 0x8c29, [0x31e2] = 0x8292, [0x31e3] = 0x832b, [0x31e4] = 0x76f2,
  [0x31e5] = 0x6c13, [0x31e6] = 0x5fd9, [0x31e7] = 0x83bd, [0x31e8] = 0x732b,
  [0x31e9] = 0x8305, [0x31ea] = 0x951a, [0x31eb] = 0x6bdb, [0x31ec] = 0x77db,
  [0x31ed] = 0x94c6, [0x31ee] = 0x536f, [0x31ef] = 0x8302, [0x31f0] = 0x5192,
  [0x31f1] = 0x5e3d, [0x31f2] = 0x8c8c, [0x31f3] = 0x8d38, [0x31f4] = 0x4e48,
  [0x31f5] = 0x73ab, [0x31f6] = 0x679a, [0x31f7] = 0x6885, [0x31f8] = 0x9176,
  [0x31f9] = 0x9709, [0x31fa] = 0x7164, [0x31fb] = 0x6ca1, [0x31fc] = 0x7709,
  [0x31fd] = 0x5a92, [0x31fe] = 0x9541, [0x31ff] = 0x6bcf, [0x3200] = 0x7f8e,
  [0x3201] = 0x6627, [0x3202] = 0x5bd0, [0x3203] = 0x59b9, [0x3204] = 0x5a9a,
  [0x3205] = 0x95e8, [0x3206] = 0x95f7, [0x3207] = 0x4eec, [0x3208] = 0x840c,
  [0x3209] = 0x8499, [0x320a] = 0x6aac, [0x320b] = 0x76df, [0x320c] = 0x9530,
  [0x320d] = 0x731b, [0x320e] = 0x68a6, [0x320f] = 0x5b5f, [0x3210] = 0x772f,
  [0x3211] = 0x919a, [0x3212] = 0x9761, [0x3213] = 0x7cdc, [0x3214] = 0x8ff7,
  [0x3215] = 0x8c1c, [0x3216] = 0x5f25, [0x3217] = 0x7c73, [0x3218] = 0x79d8,
  [0x3219] = 0x89c5, [0x321a] = 0x6ccc, [0x321b] = 0x871c, [0x321c] = 0x5bc6,
  [0x321d] = 0x5e42, [0x321e] = 0x68c9, [0x321f] = 0x7720, [0x3220] = 0x7ef5,
  [0x3221] = 0x5195, [0x3222] = 0x514d, [0x3223] = 0x52c9, [0x3224] = 0x5a29,
  [0x3225] = 0x7f05, [0x3226] = 0x9762, [0x3227] = 0x82d7, [0x3228] = 0x63cf,
  [0x3229] = 0x7784, [0x322a] = 0x85d0, [0x322b] = 0x79d2, [0x322c] = 0x6e3a,
  [0x322d] = 0x5e99, [0x322e] = 0x5999, [0x322f] = 0x8511, [0x3230] = 0x706d,
  [0x3231] = 0x6c11, [0x3232] = 0x62bf, [0x3233] = 0x76bf, [0x3234] = 0x654f,
  [0x3235] = 0x60af, [0x3236] = 0x95fd, [0x3237] = 0x660e, [0x3238] = 0x879f,
  [0x3239] = 0x9e23, [0x323a] = 0x94ed, [0x323b] = 0x540d, [0x323c] = 0x547d,
  [0x323d] = 0x8c2c, [0x323e] = 0x6478, [0x3240] = 0x8140, [0x3241] = 0x8141,
  [0x3242] = 0x8142, [0x3243] = 0x8143, [0x3244] = 0x8144, [0x3245] = 0x8145,
  [0x3246] = 0x8147, [0x3247] = 0x8149, [0x3248] = 0x814d, [0x3249] = 0x814e,
  [0x324a] = 0x814f, [0x324b] = 0x8152, [0x324c] = 0x8156, [0x324d] = 0x8157,
  [0x324e] = 0x8158, [0x324f] = 0x815b, [0x3250] = 0x815c, [0x3251] = 0x815d,
  [0x3252] = 0x815e, [0x3253] = 0x815f, [0x3254] = 0x8161, [0x3255] = 0x8162,
  [0x3256] = 0x8163, [0x3257] = 0x8164, [0x3258] = 0x8166, [0x3259] = 0x8168,
  [0x325a] = 0x816a, [0x325b] = 0x816b, [0x325c] = 0x816c, [0x325d] = 0x816f,
  [0x325e] = 0x8172, [0x325f] = 0x8173, [0x3260] = 0x8175, [0x3261] = 0x8176,
  [0x3262] = 0x8177, [0x3263] = 0x8178, [0x3264] = 0x8181, [0x3265] = 0x8183,
  [0x3266] = 0x8184, [0x3267] = 0x8185, [0x3268] = 0x8186, [0x3269] = 0x8187,
  [0x326a] = 0x8189, [0x326b] = 0x818b, [0x326c] = 0x818c, [0x326d] = 0x818d,
  [0x326e] = 0x818e, [0x326f] = 0x8190, [0x3270] = 0x8192, [0x3271] = 0x8193,
  [0x3272] = 0x8194, [0x3273] = 0x8195, [0x3274] = 0x8196, [0x3275] = 0x8197,
  [0x3276] = 0x8199, [0x3277] = 0x819a, [0x3278] = 0x819e, [0x3279] = 0x819f,
  [0x327a] = 0x81a0, [0x327b] = 0x81a1, [0x327c] = 0x81a2, [0x327d] = 0x81a4,
  [0x327e] = 0x81a5, [0x3280] = 0x81a7, [0x3281] = 0x81a9, [0x3282] = 0x81ab,
  [0x3283] = 0x81ac, [0x3284] = 0x81ad, [0x3285] = 0x81ae, [0x3286] = 0x81af,
  [0x3287] = 0x81b0, [0x3288] = 0x81b1, [0x3289] = 0x81b2, [0x328a] = 0x81b4,
  [0x328b] = 0x81b5, [0x328c] = 0x81b6, [0x328d] = 0x81b7, [0x328e] = 0x81b8,
  [0x328f] = 0x81b9, [0x3290] = 0x81bc, [0x3291] = 0x81bd, [0x3292] = 0x81be,
  [0x3293] = 0x81bf, [0x3294] = 0x81c4, [0x3295] = 0x81c5, [0x3296] = 0x81c7,
  [0x3297] = 0x81c8, [0x3298] = 0x81c9, [0x3299] = 0x81cb, [0x329a] = 0x81cd,
  [0x329b] = 0x81ce, [0x329c] = 0x81cf, [0x329d] = 0x81d0, [0x329e] = 0x81d1,
  [0x329f] = 0x81d2, [0x32a0] = 0x81d3, [0x32a1] = 0x6479, [0x32a2] = 0x8611,
  [0x32a3] = 0x6a21, [0x32a4] = 0x819c, [0x32a5] = 0x78e8, [0x32a6] = 0x6469,
  [0x32a7] = 0x9b54, [0x32a8] = 0x62b9, [0x32a9] = 0x672b, [0x32aa] = 0x83ab,
  [0x32ab] = 0x58a8, [0x32ac] = 0x9ed8, [0x32ad] = 0x6cab, [0x32ae] = 0x6f20,
  [0x32af] = 0x5bde, [0x32b0] = 0x964c, [0x32b1] = 0x8c0b, [0x32b2] = 0x725f,
  [0x32b3] = 0x67d0, [0x32b4] = 0x62c7, [0x32b5] = 0x7261, [0x32b6] = 0x4ea9,
  [0x32b7] = 0x59c6, [0x32b8] = 0x6bcd, [0x32b9] = 0x5893, [0x32ba] = 0x66ae,
  [0x32bb] = 0x5e55, [0x32bc] = 0x52df, [0x32bd] = 0x6155, [0x32be] = 0x6728,
  [0x32bf] = 0x76ee, [0x32c0] = 0x7766, [0x32c1] = 0x7267, [0x32c2] = 0x7a46,
  [0x32c3] = 0x62ff, [0x32c4] = 0x54ea, [0x32c5] = 0x5450, [0x32c6] = 0x94a0,
  [0x32c7] = 0x90a3, [0x32c8] = 0x5a1c, [0x32c9] = 0x7eb3, [0x32ca] = 0x6c16,
  [0x32cb] = 0x4e43, [0x32cc] = 0x5976, [0x32cd] = 0x8010, [0x32ce] = 0x5948,
  [0x32cf] = 0x5357, [0x32d0] = 0x7537, [0x32d1] = 0x96be, [0x32d2] = 0x56ca,
  [0x32d3] = 0x6320, [0x32d4] = 0x8111, [0x32d5] = 0x607c, [0x32d6] = 0x95f9,
  [0x32d7] = 0x6dd6, [0x32d8] = 0x5462, [0x32d9] = 0x9981, [0x32da] = 0x5185,
  [0x32db] = 0x5ae9, [0x32dc] = 0x80fd, [0x32dd] = 0x59ae, [0x32de] = 0x9713,
  [0x32df] = 0x502a, [0x32e0] = 0x6ce5, [0x32e1] = 0x5c3c, [0x32e2] = 0x62df,
  [0x32e3] = 0x4f60, [0x32e4] = 0x533f, [0x32e5] = 0x817b, [0x32e6] = 0x9006,
  [0x32e7] = 0x6eba, [0x32e8] = 0x852b, [0x32e9] = 0x62c8, [0x32ea] = 0x5e74,
  [0x32eb] = 0x78be, [0x32ec] = 0x64b5, [0x32ed] = 0x637b, [0x32ee] = 0x5ff5,
  [0x32ef] = 0x5a18, [0x32f0] = 0x917f, [0x32f1] = 0x9e1f, [0x32f2] = 0x5c3f,
  [0x32f3] = 0x634f, [0x32f4] = 0x8042, [0x32f5] = 0x5b7d, [0x32f6] = 0x556e,
  [0x32f7] = 0x954a, [0x32f8] = 0x954d, [0x32f9] = 0x6d85, [0x32fa] = 0x60a8,
  [0x32fb] = 0x67e0, [0x32fc] = 0x72de, [0x32fd] = 0x51dd, [0x32fe] = 0x5b81,
  [0x3300] = 0x81d4, [0x3301] = 0x81d5, [0x3302] = 0x81d6, [0x3303] = 0x81d7,
  [0x3304] = 0x81d8, [0x3305] = 0x81d9, [0x3306] = 0x81da, [0x3307] = 0x81db,
  [0x3308] = 0x81dc, [0x3309] = 0x81dd, [0x330a] = 0x81de, [0x330b] = 0x81df,
  [0x330c] = 0x81e0, [0x330d] = 0x81e1, [0x330e] = 0x81e2, [0x330f] = 0x81e4,
  [0x3310] = 0x81e5, [0x3311] = 0x81e6, [0x3312] = 0x81e8, [0x3313] = 0x81e9,
  [0x3314] = 0x81eb, [0x3315] = 0x81ee, [0x3316] = 0x81ef, [0x3317] = 0x81f0,
  [0x3318] = 0x81f1, [0x3319] = 0x81f2, [0x331a] = 0x81f5, [0x331b] = 0x81f6,
  [0x331c] = 0x81f7, [0x331d] = 0x81f8, [0x331e] = 0x81f9, [0x331f] = 0x81fa,
  [0x3320] = 0x81fd, [0x3321] = 0x81ff, [0x3322] = 0x8203, [0x3323] = 0x8207,
  [0x3324] = 0x8208, [0x3325] = 0x8209, [0x3326] = 0x820a, [0x3327] = 0x820b,
  [0x3328] = 0x820e, [0x3329] = 0x820f, [0x332a] = 0x8211, [0x332b] = 0x8213,
  [0x332c] = 0x8215, [0x332d] = 0x8216, [0x332e] = 0x8217, [0x332f] = 0x8218,
  [0x3330] = 0x8219, [0x3331] = 0x821a, [0x3332] = 0x821d, [0x3333] = 0x8220,
  [0x3334] = 0x8224, [0x3335] = 0x8225, [0x3336] = 0x8226, [0x3337] = 0x8227,
  [0x3338] = 0x8229, [0x3339] = 0x822e, [0x333a] = 0x8232, [0x333b] = 0x823a,
  [0x333c] = 0x823c, [0x333d] = 0x823d, [0x333e] = 0x823f, [0x3340] = 0x8240,
  [0x3341] = 0x8241, [0x3342] = 0x8242, [0x3343] = 0x8243, [0x3344] = 0x8245,
  [0x3345] = 0x8246, [0x3346] = 0x8248, [0x3347] = 0x824a, [0x3348] = 0x824c,
  [0x3349] = 0x824d, [0x334a] = 0x824e, [0x334b] = 0x8250, [0x334c] = 0x8251,
  [0x334d] = 0x8252, [0x334e] = 0x8253, [0x334f] = 0x8254, [0x3350] = 0x8255,
  [0x3351] = 0x8256, [0x3352] = 0x8257, [0x3353] = 0x8259, [0x3354] = 0x825b,
  [0x3355] = 0x825c, [0x3356] = 0x825d, [0x3357] = 0x825e, [0x3358] = 0x8260,
  [0x3359] = 0x8261, [0x335a] = 0x8262, [0x335b] = 0x8263, [0x335c] = 0x8264,
  [0x335d] = 0x8265, [0x335e] = 0x8266, [0x335f] = 0x8267, [0x3360] = 0x8269,
  [0x3361] = 0x62e7, [0x3362] = 0x6cde, [0x3363] = 0x725b, [0x3364] = 0x626d,
  [0x3365] = 0x94ae, [0x3366] = 0x7ebd, [0x3367] = 0x8113, [0x3368] = 0x6d53,
  [0x3369] = 0x519c, [0x336a] = 0x5f04, [0x336b] = 0x5974, [0x336c] = 0x52aa,
  [0x336d] = 0x6012, [0x336e] = 0x5973, [0x336f] = 0x6696, [0x3370] = 0x8650,
  [0x3371] = 0x759f, [0x3372] = 0x632a, [0x3373] = 0x61e6, [0x3374] = 0x7cef,
  [0x3375] = 0x8bfa, [0x3376] = 0x54e6, [0x3377] = 0x6b27, [0x3378] = 0x9e25,
  [0x3379] = 0x6bb4, [0x337a] = 0x85d5, [0x337b] = 0x5455, [0x337c] = 0x5076,
  [0x337d] = 0x6ca4, [0x337e] = 0x556a, [0x337f] = 0x8db4, [0x3380] = 0x722c,
  [0x3381] = 0x5e15, [0x3382] = 0x6015, [0x3383] = 0x7436, [0x3384] = 0x62cd,
  [0x3385] = 0x6392, [0x3386] = 0x724c, [0x3387] = 0x5f98, [0x3388] = 0x6e43,
  [0x3389] = 0x6d3e, [0x338a] = 0x6500, [0x338b] = 0x6f58, [0x338c] = 0x76d8,
  [0x338d] = 0x78d0, [0x338e] = 0x76fc, [0x338f] = 0x7554, [0x3390] = 0x5224,
  [0x3391] = 0x53db, [0x3392] = 0x4e53, [0x3393] = 0x5e9e, [0x3394] = 0x65c1,
  [0x3395] = 0x802a, [0x3396] = 0x80d6, [0x3397] = 0x629b, [0x3398] = 0x5486,
  [0x3399] = 0x5228, [0x339a] = 0x70ae, [0x339b] = 0x888d, [0x339c] = 0x8dd1,
  [0x339d] = 0x6ce1, [0x339e] = 0x5478, [0x339f] = 0x80da, [0x33a0] = 0x57f9,
  [0x33a1] = 0x88f4, [0x33a2] = 0x8d54, [0x33a3] = 0x966a, [0x33a4] = 0x914d,
  [0x33a5] = 0x4f69, [0x33a6] = 0x6c9b, [0x33a7] = 0x55b7, [0x33a8] = 0x76c6,
  [0x33a9] = 0x7830, [0x33aa] = 0x62a8, [0x33ab] = 0x70f9, [0x33ac] = 0x6f8e,
  [0x33ad] = 0x5f6d, [0x33ae] = 0x84ec, [0x33af] = 0x68da, [0x33b0] = 0x787c,
  [0x33b1] = 0x7bf7, [0x33b2] = 0x81a8, [0x33b3] = 0x670b, [0x33b4] = 0x9e4f,
  [0x33b5] = 0x6367, [0x33b6] = 0x78b0, [0x33b7] = 0x576f, [0x33b8] = 0x7812,
  [0x33b9] = 0x9739, [0x33ba] = 0x6279, [0x33bb] = 0x62ab, [0x33bc] = 0x5288,
  [0x33bd] = 0x7435, [0x33be] = 0x6bd7, [0x33c0] = 0x826a, [0x33c1] = 0x826b,
  [0x33c2] = 0x826c, [0x33c3] = 0x826d, [0x33c4] = 0x8271, [0x33c5] = 0x8275,
  [0x33c6] = 0x8276, [0x33c7] = 0x8277, [0x33c8] = 0x8278, [0x33c9] = 0x827b,
  [0x33ca] = 0x827c, [0x33cb] = 0x8280, [0x33cc] = 0x8281, [0x33cd] = 0x8283,
  [0x33ce] = 0x8285, [0x33cf] = 0x8286, [0x33d0] = 0x8287, [0x33d1] = 0x8289,
  [0x33d2] = 0x828c, [0x33d3] = 0x8290, [0x33d4] = 0x8293, [0x33d5] = 0x8294,
  [0x33d6] = 0x8295, [0x33d7] = 0x8296, [0x33d8] = 0x829a, [0x33d9] = 0x829b,
  [0x33da] = 0x829e, [0x33db] = 0x82a0, [0x33dc] = 0x82a2, [0x33dd] = 0x82a3,
  [0x33de] = 0x82a7, [0x33df] = 0x82b2, [0x33e0] = 0x82b5, [0x33e1] = 0x82b6,
  [0x33e2] = 0x82ba, [0x33e3] = 0x82bb, [0x33e4] = 0x82bc, [0x33e5] = 0x82bf,
  [0x33e6] = 0x82c0, [0x33e7] = 0x82c2, [0x33e8] = 0x82c3, [0x33e9] = 0x82c5,
  [0x33ea] = 0x82c6, [0x33eb] = 0x82c9, [0x33ec] = 0x82d0, [0x33ed] = 0x82d6,
  [0x33ee] = 0x82d9, [0x33ef] = 0x82da, [0x33f0] = 0x82dd, [0x33f1] = 0x82e2,
  [0x33f2] = 0x82e7, [0x33f3] = 0x82e8, [0x33f4] = 0x82e9, [0x33f5] = 0x82ea,
  [0x33f6] = 0x82ec, [0x33f7] = 0x82ed, [0x33f8] = 0x82ee, [0x33f9] = 0x82f0,
  [0x33fa] = 0x82f2, [0x33fb] = 0x82f3, [0x33fc] = 0x82f5, [0x33fd] = 0x82f6,
  [0x33fe] = 0x82f8, [0x3400] = 0x82fa, [0x3401] = 0x82fc, [0x3402] = 0x82fd,
  [0x3403] = 0x82fe, [0x3404] = 0x82ff, [0x3405] = 0x8300, [0x3406] = 0x830a,
  [0x3407] = 0x830b, [0x3408] = 0x830d, [0x3409] = 0x8310, [0x340a] = 0x8312,
  [0x340b] = 0x8313, [0x340c] = 0x8316, [0x340d] = 0x8318, [0x340e] = 0x8319,
  [0x340f] = 0x831d, [0x3410] = 0x831e, [0x3411] = 0x831f, [0x3412] = 0x8320,
  [0x3413] = 0x8321, [0x3414] = 0x8322, [0x3415] = 0x8323, [0x3416] = 0x8324,
  [0x3417] = 0x8325, [0x3418] = 0x8326, [0x3419] = 0x8329, [0x341a] = 0x832a,
  [0x341b] = 0x832e, [0x341c] = 0x8330, [0x341d] = 0x8332, [0x341e] = 0x8337,
  [0x341f] = 0x833b, [0x3420] = 0x833d, [0x3421] = 0x5564, [0x3422] = 0x813e,
  [0x3423] = 0x75b2, [0x3424] = 0x76ae, [0x3425] = 0x5339, [0x3426] = 0x75de,
  [0x3427] = 0x50fb, [0x3428] = 0x5c41, [0x3429] = 0x8b6c, [0x342a] = 0x7bc7,
  [0x342b] = 0x504f, [0x342c] = 0x7247, [0x342d] = 0x9a97, [0x342e] = 0x98d8,
  [0x342f] = 0x6f02, [0x3430] = 0x74e2, [0x3431] = 0x7968, [0x3432] = 0x6487,
  [0x3433] = 0x77a5, [0x3434] = 0x62fc, [0x3435] = 0x9891, [0x3436] = 0x8d2b,
  [0x3437] = 0x54c1, [0x3438] = 0x8058, [0x3439] = 0x4e52, [0x343a] = 0x576a,
  [0x343b] = 0x82f9, [0x343c] = 0x840d, [0x343d] = 0x5e73, [0x343e] = 0x51ed,
  [0x343f] = 0x74f6, [0x3440] = 0x8bc4, [0x3441] = 0x5c4f, [0x3442] = 0x5761,
  [0x3443] = 0x6cfc, [0x3444] = 0x9887, [0x3445] = 0x5a46, [0x3446] = 0x7834,
  [0x3447] = 0x9b44, [0x3448] = 0x8feb, [0x3449] = 0x7c95, [0x344a] = 0x5256,
  [0x344b] = 0x6251, [0x344c] = 0x94fa, [0x344d] = 0x4ec6, [0x344e] = 0x8386,
  [0x344f] = 0x8461, [0x3450] = 0x83e9, [0x3451] = 0x84b2, [0x3452] = 0x57d4,
  [0x3453] = 0x6734, [0x3454] = 0x5703, [0x3455] = 0x666e, [0x3456] = 0x6d66,
  [0x3457] = 0x8c31, [0x3458] = 0x66dd, [0x3459] = 0x7011, [0x345a] = 0x671f,
  [0x345b] = 0x6b3a, [0x345c] = 0x6816, [0x345d] = 0x621a, [0x345e] = 0x59bb,
  [0x345f] = 0x4e03, [0x3460] = 0x51c4, [0x3461] = 0x6f06, [0x3462] = 0x67d2,
  [0x3463] = 0x6c8f, [0x3464] = 0x5176, [0x3465] = 0x68cb, [0x3466] = 0x5947,
  [0x3467] = 0x6b67, [0x3468] = 0x7566, [0x3469] = 0x5d0e, [0x346a] = 0x8110,
  [0x346b] = 0x9f50, [0x346c] = 0x65d7, [0x346d] = 0x7948, [0x346e] = 0x7941,
  [0x346f] = 0x9a91, [0x3470] = 0x8d77, [0x3471] = 0x5c82, [0x3472] = 0x4e5e,
  [0x3473] = 0x4f01, [0x3474] = 0x542f, [0x3475] = 0x5951, [0x3476] = 0x780c,
  [0x3477] = 0x5668, [0x3478] = 0x6c14, [0x3479] = 0x8fc4, [0x347a] = 0x5f03,
  [0x347b] = 0x6c7d, [0x347c] = 0x6ce3, [0x347d] = 0x8bab, [0x347e] = 0x6390,
  [0x3480] = 0x833e, [0x3481] = 0x833f, [0x3482] = 0x8341, [0x3483] = 0x8342,
  [0x3484] = 0x8344, [0x3485] = 0x8345, [0x3486] = 0x8348, [0x3487] = 0x834a,
  [0x3488] = 0x834b, [0x3489] = 0x834c, [0x348a] = 0x834d, [0x348b] = 0x834e,
  [0x348c] = 0x8353, [0x348d] = 0x8355, [0x348e] = 0x8356, [0x348f] = 0x8357,
  [0x3490] = 0x8358, [0x3491] = 0x8359, [0x3492] = 0x835d, [0x3493] = 0x8362,
  [0x3494] = 0x8370, [0x3495] = 0x8371, [0x3496] = 0x8372, [0x3497] = 0x8373,
  [0x3498] = 0x8374, [0x3499] = 0x8375, [0x349a] = 0x8376, [0x349b] = 0x8379,
  [0x349c] = 0x837a, [0x349d] = 0x837e, [0x349e] = 0x837f, [0x349f] = 0x8380,
  [0x34a0] = 0x8381, [0x34a1] = 0x8382, [0x34a2] = 0x8383, [0x34a3] = 0x8384,
  [0x34a4] = 0x8387, [0x34a5] = 0x8388, [0x34a6] = 0x838a, [0x34a7] = 0x838b,
  [0x34a8] = 0x838c, [0x34a9] = 0x838d, [0x34aa] = 0x838f, [0x34ab] = 0x8390,
  [0x34ac] = 0x8391, [0x34ad] = 0x8394, [0x34ae] = 0x8395, [0x34af] = 0x8396,
  [0x34b0] = 0x8397, [0x34b1] = 0x8399, [0x34b2] = 0x839a, [0x34b3] = 0x839d,
  [0x34b4] = 0x839f, [0x34b5] = 0x83a1, [0x34b6] = 0x83a2, [0x34b7] = 0x83a3,
  [0x34b8] = 0x83a4, [0x34b9] = 0x83a5, [0x34ba] = 0x83a6, [0x34bb] = 0x83a7,
  [0x34bc] = 0x83ac, [0x34bd] = 0x83ad, [0x34be] = 0x83ae, [0x34c0] = 0x83af,
  [0x34c1] = 0x83b5, [0x34c2] = 0x83bb, [0x34c3] = 0x83be, [0x34c4] = 0x83bf,
  [0x34c5] = 0x83c2, [0x34c6] = 0x83c3, [0x34c7] = 0x83c4, [0x34c8] = 0x83c6,
  [0x34c9] = 0x83c8, [0x34ca] = 0x83c9, [0x34cb] = 0x83cb, [0x34cc] = 0x83cd,
  [0x34cd] = 0x83ce, [0x34ce] = 0x83d0, [0x34cf] = 0x83d1, [0x34d0] = 0x83d2,
  [0x34d1] = 0x83d3, [0x34d2] = 0x83d5, [0x34d3] = 0x83d7, [0x34d4] = 0x83d9,
  [0x34d5] = 0x83da, [0x34d6] = 0x83db, [0x34d7] = 0x83de, [0x34d8] = 0x83e2,
  [0x34d9] = 0x83e3, [0x34da] = 0x83e4, [0x34db] = 0x83e6, [0x34dc] = 0x83e7,
  [0x34dd] = 0x83e8, [0x34de] = 0x83eb, [0x34df] = 0x83ec, [0x34e0] = 0x83ed,
  [0x34e1] = 0x6070, [0x34e2] = 0x6d3d, [0x34e3] = 0x7275, [0x34e4] = 0x6266,
  [0x34e5] = 0x948e, [0x34e6] = 0x94c5, [0x34e7] = 0x5343, [0x34e8] = 0x8fc1,
  [0x34e9] = 0x7b7e, [0x34ea] = 0x4edf, [0x34eb] = 0x8c26, [0x34ec] = 0x4e7e,
  [0x34ed] = 0x9ed4, [0x34ee] = 0x94b1, [0x34ef] = 0x94b3, [0x34f0] = 0x524d,
  [0x34f1] = 0x6f5c, [0x34f2] = 0x9063, [0x34f3] = 0x6d45, [0x34f4] = 0x8c34,
  [0x34f5] = 0x5811, [0x34f6] = 0x5d4c, [0x34f7] = 0x6b20, [0x34f8] = 0x6b49,
  [0x34f9] = 0x67aa, [0x34fa] = 0x545b, [0x34fb] = 0x8154, [0x34fc] = 0x7f8c,
  [0x34fd] = 0x5899, [0x34fe] = 0x8537, [0x34ff] = 0x5f3a, [0x3500] = 0x62a2,
  [0x3501] = 0x6a47, [0x3502] = 0x9539, [0x3503] = 0x6572, [0x3504] = 0x6084,
  [0x3505] = 0x6865, [0x3506] = 0x77a7, [0x3507] = 0x4e54, [0x3508] = 0x4fa8,
  [0x3509] = 0x5de7, [0x350a] = 0x9798, [0x350b] = 0x64ac, [0x350c] = 0x7fd8,
  [0x350d] = 0x5ced, [0x350e] = 0x4fcf, [0x350f] = 0x7a8d, [0x3510] = 0x5207,
  [0x3511] = 0x8304, [0x3512] = 0x4e14, [0x3513] = 0x602f, [0x3514] = 0x7a83,
  [0x3515] = 0x94a6, [0x3516] = 0x4fb5, [0x3517] = 0x4eb2, [0x3518] = 0x79e6,
  [0x3519] = 0x7434, [0x351a] = 0x52e4, [0x351b] = 0x82b9, [0x351c] = 0x64d2,
  [0x351d] = 0x79bd, [0x351e] = 0x5bdd, [0x351f] = 0x6c81, [0x3520] = 0x9752,
  [0x3521] = 0x8f7b, [0x3522] = 0x6c22, [0x3523] = 0x503e, [0x3524] = 0x537f,
  [0x3525] = 0x6e05, [0x3526] = 0x64ce, [0x3527] = 0x6674, [0x3528] = 0x6c30,
  [0x3529] = 0x60c5, [0x352a] = 0x9877, [0x352b] = 0x8bf7, [0x352c] = 0x5e86,
  [0x352d] = 0x743c, [0x352e] = 0x7a77, [0x352f] = 0x79cb, [0x3530] = 0x4e18,
  [0x3531] = 0x90b1, [0x3532] = 0x7403, [0x3533] = 0x6c42, [0x3534] = 0x56da,
  [0x3535] = 0x914b, [0x3536] = 0x6cc5, [0x3537] = 0x8d8b, [0x3538] = 0x533a,
  [0x3539] = 0x86c6, [0x353a] = 0x66f2, [0x353b] = 0x8eaf, [0x353c] = 0x5c48,
  [0x353d] = 0x9a71, [0x353e] = 0x6e20, [0x3540] = 0x83ee, [0x3541] = 0x83ef,
  [0x3542] = 0x83f3, [0x3543] = 0x83f4, [0x3544] = 0x83f5, [0x3545] = 0x83f6,
  [0x3546] = 0x83f7, [0x3547] = 0x83fa, [0x3548] = 0x83fb, [0x3549] = 0x83fc,
  [0x354a] = 0x83fe, [0x354b] = 0x83ff, [0x354c] = 0x8400, [0x354d] = 0x8402,
  [0x354e] = 0x8405, [0x354f] = 0x8407, [0x3550] = 0x8408, [0x3551] = 0x8409,
  [0x3552] = 0x840a, [0x3553] = 0x8410, [0x3554] = 0x8412, [0x3555] = 0x8413,
  [0x3556] = 0x8414, [0x3557] = 0x8415, [0x3558] = 0x8416, [0x3559] = 0x8417,
  [0x355a] = 0x8419, [0x355b] = 0x841a, [0x355c] = 0x841b, [0x355d] = 0x841e,
  [0x355e] = 0x841f, [0x355f] = 0x8420, [0x3560] = 0x8421, [0x3561] = 0x8422,
  [0x3562] = 0x8423, [0x3563] = 0x8429, [0x3564] = 0x842a, [0x3565] = 0x842b,
  [0x3566] = 0x842c, [0x3567] = 0x842d, [0x3568] = 0x842e, [0x3569] = 0x842f,
  [0x356a] = 0x8430, [0x356b] = 0x8432, [0x356c] = 0x8433, [0x356d] = 0x8434,
  [0x356e] = 0x8435, [0x356f] = 0x8436, [0x3570] = 0x8437, [0x3571] = 0x8439,
  [0x3572] = 0x843a, [0x3573] = 0x843b, [0x3574] = 0x843e, [0x3575] = 0x843f,
  [0x3576] = 0x8440, [0x3577] = 0x8441, [0x3578] = 0x8442, [0x3579] = 0x8443,
  [0x357a] = 0x8444, [0x357b] = 0x8445, [0x357c] = 0x8447, [0x357d] = 0x8448,
  [0x357e] = 0x8449, [0x3580] = 0x844a, [0x3581] = 0x844b, [0x3582] = 0x844c,
  [0x3583] = 0x844d, [0x3584] = 0x844e, [0x3585] = 0x844f, [0x3586] = 0x8450,
  [0x3587] = 0x8452, [0x3588] = 0x8453, [0x3589] = 0x8454, [0x358a] = 0x8455,
  [0x358b] = 0x8456, [0x358c] = 0x8458, [0x358d] = 0x845d, [0x358e] = 0x845e,
  [0x358f] = 0x845f, [0x3590] = 0x8460, [0x3591] = 0x8462, [0x3592] = 0x8464,
  [0x3593] = 0x8465, [0x3594] = 0x8466, [0x3595] = 0x8467, [0x3596] = 0x8468,
  [0x3597] = 0x846a, [0x3598] = 0x846e, [0x3599] = 0x846f, [0x359a] = 0x8470,
  [0x359b] = 0x8472, [0x359c] = 0x8474, [0x359d] = 0x8477, [0x359e] = 0x8479,
  [0x359f] = 0x847b, [0x35a0] = 0x847c, [0x35a1] = 0x53d6, [0x35a2] = 0x5a36,
  [0x35a3] = 0x9f8b, [0x35a4] = 0x8da3, [0x35a5] = 0x53bb, [0x35a6] = 0x5708,
  [0x35a7] = 0x98a7, [0x35a8] = 0x6743, [0x35a9] = 0x919b, [0x35aa] = 0x6cc9,
  [0x35ab] = 0x5168, [0x35ac] = 0x75ca, [0x35ad] = 0x62f3, [0x35ae] = 0x72ac,
  [0x35af] = 0x5238, [0x35b0] = 0x529d, [0x35b1] = 0x7f3a, [0x35b2] = 0x7094,
  [0x35b3] = 0x7638, [0x35b4] = 0x5374, [0x35b5] = 0x9e4a, [0x35b6] = 0x69b7,
  [0x35b7] = 0x786e, [0x35b8] = 0x96c0, [0x35b9] = 0x88d9, [0x35ba] = 0x7fa4,
  [0x35bb] = 0x7136, [0x35bc] = 0x71c3, [0x35bd] = 0x5189, [0x35be] = 0x67d3,
  [0x35bf] = 0x74e4, [0x35c0] = 0x58e4, [0x35c1] = 0x6518, [0x35c2] = 0x56b7,
  [0x35c3] = 0x8ba9, [0x35c4] = 0x9976, [0x35c5] = 0x6270, [0x35c6] = 0x7ed5,
  [0x35c7] = 0x60f9, [0x35c8] = 0x70ed, [0x35c9] = 0x58ec, [0x35ca] = 0x4ec1,
  [0x35cb] = 0x4eba, [0x35cc] = 0x5fcd, [0x35cd] = 0x97e7, [0x35ce] = 0x4efb,
  [0x35cf] = 0x8ba4, [0x35d0] = 0x5203, [0x35d1] = 0x598a, [0x35d2] = 0x7eab,
  [0x35d3] = 0x6254, [0x35d4] = 0x4ecd, [0x35d5] = 0x65e5, [0x35d6] = 0x620e,
  [0x35d7] = 0x8338, [0x35d8] = 0x84c9, [0x35d9] = 0x8363, [0x35da] = 0x878d,
  [0x35db] = 0x7194, [0x35dc] = 0x6eb6, [0x35dd] = 0x5bb9, [0x35de] = 0x7ed2,
  [0x35df] = 0x5197, [0x35e0] = 0x63c9, [0x35e1] = 0x67d4, [0x35e2] = 0x8089,
  [0x35e3] = 0x8339, [0x35e4] = 0x8815, [0x35e5] = 0x5112, [0x35e6] = 0x5b7a,
  [0x35e7] = 0x5982, [0x35e8] = 0x8fb1, [0x35e9] = 0x4e73, [0x35ea] = 0x6c5d,
  [0x35eb] = 0x5165, [0x35ec] = 0x8925, [0x35ed] = 0x8f6f, [0x35ee] = 0x962e,
  [0x35ef] = 0x854a, [0x35f0] = 0x745e, [0x35f1] = 0x9510, [0x35f2] = 0x95f0,
  [0x35f3] = 0x6da6, [0x35f4] = 0x82e5, [0x35f5] = 0x5f31, [0x35f6] = 0x6492,
  [0x35f7] = 0x6d12, [0x35f8] = 0x8428, [0x35f9] = 0x816e, [0x35fa] = 0x9cc3,
  [0x35fb] = 0x585e, [0x35fc] = 0x8d5b, [0x35fd] = 0x4e09, [0x35fe] = 0x53c1,
  [0x3600] = 0x847d, [0x3601] = 0x847e, [0x3602] = 0x847f, [0x3603] = 0x8480,
  [0x3604] = 0x8481, [0x3605] = 0x8483, [0x3606] = 0x8484, [0x3607] = 0x8485,
  [0x3608] = 0x8486, [0x3609] = 0x848a, [0x360a] = 0x848d, [0x360b] = 0x848f,
  [0x360c] = 0x8490, [0x360d] = 0x8491, [0x360e] = 0x8492, [0x360f] = 0x8493,
  [0x3610] = 0x8494, [0x3611] = 0x8495, [0x3612] = 0x8496, [0x3613] = 0x8498,
  [0x3614] = 0x849a, [0x3615] = 0x849b, [0x3616] = 0x849d, [0x3617] = 0x849e,
  [0x3618] = 0x849f, [0x3619] = 0x84a0, [0x361a] = 0x84a2, [0x361b] = 0x84a3,
  [0x361c] = 0x84a4, [0x361d] = 0x84a5, [0x361e] = 0x84a6, [0x361f] = 0x84a7,
  [0x3620] = 0x84a8, [0x3621] = 0x84a9, [0x3622] = 0x84aa, [0x3623] = 0x84ab,
  [0x3624] = 0x84ac, [0x3625] = 0x84ad, [0x3626] = 0x84ae, [0x3627] = 0x84b0,
  [0x3628] = 0x84b1, [0x3629] = 0x84b3, [0x362a] = 0x84b5, [0x362b] = 0x84b6,
  [0x362c] = 0x84b7, [0x362d] = 0x84bb, [0x362e] = 0x84bc, [0x362f] = 0x84be,
  [0x3630] = 0x84c0, [0x3631] = 0x84c2, [0x3632] = 0x84c3, [0x3633] = 0x84c5,
  [0x3634] = 0x84c6, [0x3635] = 0x84c7, [0x3636] = 0x84c8, [0x3637] = 0x84cb,
  [0x3638] = 0x84cc, [0x3639] = 0x84ce, [0x363a] = 0x84cf, [0x363b] = 0x84d2,
  [0x363c] = 0x84d4, [0x363d] = 0x84d5, [0x363e] = 0x84d7, [0x3640] = 0x84d8,
  [0x3641] = 0x84d9, [0x3642] = 0x84da, [0x3643] = 0x84db, [0x3644] = 0x84dc,
  [0x3645] = 0x84de, [0x3646] = 0x84e1, [0x3647] = 0x84e2, [0x3648] = 0x84e4,
  [0x3649] = 0x84e7, [0x364a] = 0x84e8, [0x364b] = 0x84e9, [0x364c] = 0x84ea,
  [0x364d] = 0x84eb, [0x364e] = 0x84ed, [0x364f] = 0x84ee, [0x3650] = 0x84ef,
  [0x3651] = 0x84f1, [0x3652] = 0x84f2, [0x3653] = 0x84f3, [0x3654] = 0x84f4,
  [0x3655] = 0x84f5, [0x3656] = 0x84f6, [0x3657] = 0x84f7, [0x3658] = 0x84f8,
  [0x3659] = 0x84f9, [0x365a] = 0x84fa, [0x365b] = 0x84fb, [0x365c] = 0x84fd,
  [0x365d] = 0x84fe, [0x365e] = 0x8500, [0x365f] = 0x8501, [0x3660] = 0x8502,
  [0x3661] = 0x4f1e, [0x3662] = 0x6563, [0x3663] = 0x6851, [0x3664] = 0x55d3,
  [0x3665] = 0x4e27, [0x3666] = 0x6414, [0x3667] = 0x9a9a, [0x3668] = 0x626b,
  [0x3669] = 0x5ac2, [0x366a] = 0x745f, [0x366b] = 0x8272, [0x366c] = 0x6da9,
  [0x366d] = 0x68ee, [0x366e] = 0x50e7, [0x366f] = 0x838e, [0x3670] = 0x7802,
  [0x3671] = 0x6740, [0x3672] = 0x5239, [0x3673] = 0x6c99, [0x3674] = 0x7eb1,
  [0x3675] = 0x50bb, [0x3676] = 0x5565, [0x3677] = 0x715e, [0x3678] = 0x7b5b,
  [0x3679] = 0x6652, [0x367a] = 0x73ca, [0x367b] = 0x82eb, [0x367c] = 0x6749,
  [0x367d] = 0x5c71, [0x367e] = 0x5220, [0x367f] = 0x717d, [0x3680] = 0x886b,
  [0x3681] = 0x95ea, [0x3682] = 0x9655, [0x3683] = 0x64c5, [0x3684] = 0x8d61,
  [0x3685] = 0x81b3, [0x3686] = 0x5584, [0x3687] = 0x6c55, [0x3688] = 0x6247,
  [0x3689] = 0x7f2e, [0x368a] = 0x5892, [0x368b] = 0x4f24, [0x368c] = 0x5546,
  [0x368d] = 0x8d4f, [0x368e] = 0x664c, [0x368f] = 0x4e0a, [0x3690] = 0x5c1a,
  [0x3691] = 0x88f3, [0x3692] = 0x68a2, [0x3693] = 0x634e, [0x3694] = 0x7a0d,
  [0x3695] = 0x70e7, [0x3696] = 0x828d, [0x3697] = 0x52fa, [0x3698] = 0x97f6,
  [0x3699] = 0x5c11, [0x369a] = 0x54e8, [0x369b] = 0x90b5, [0x369c] = 0x7ecd,
  [0x369d] = 0x5962, [0x369e] = 0x8d4a, [0x369f] = 0x86c7, [0x36a0] = 0x820c,
  [0x36a1] = 0x820d, [0x36a2] = 0x8d66, [0x36a3] = 0x6444, [0x36a4] = 0x5c04,
  [0x36a5] = 0x6151, [0x36a6] = 0x6d89, [0x36a7] = 0x793e, [0x36a8] = 0x8bbe,
  [0x36a9] = 0x7837, [0x36aa] = 0x7533, [0x36ab] = 0x547b, [0x36ac] = 0x4f38,
  [0x36ad] = 0x8eab, [0x36ae] = 0x6df1, [0x36af] = 0x5a20, [0x36b0] = 0x7ec5,
  [0x36b1] = 0x795e, [0x36b2] = 0x6c88, [0x36b3] = 0x5ba1, [0x36b4] = 0x5a76,
  [0x36b5] = 0x751a, [0x36b6] = 0x80be, [0x36b7] = 0x614e, [0x36b8] = 0x6e17,
  [0x36b9] = 0x58f0, [0x36ba] = 0x751f, [0x36bb] = 0x7525, [0x36bc] = 0x7272,
  [0x36bd] = 0x5347, [0x36be] = 0x7ef3, [0x36c0] = 0x8503, [0x36c1] = 0x8504,
  [0x36c2] = 0x8505, [0x36c3] = 0x8506, [0x36c4] = 0x8507, [0x36c5] = 0x8508,
  [0x36c6] = 0x8509, [0x36c7] = 0x850a, [0x36c8] = 0x850b, [0x36c9] = 0x850d,
  [0x36ca] = 0x850e, [0x36cb] = 0x850f, [0x36cc] = 0x8510, [0x36cd] = 0x8512,
  [0x36ce] = 0x8514, [0x36cf] = 0x8515, [0x36d0] = 0x8516, [0x36d1] = 0x8518,
  [0x36d2] = 0x8519, [0x36d3] = 0x851b, [0x36d4] = 0x851c, [0x36d5] = 0x851d,
  [0x36d6] = 0x851e, [0x36d7] = 0x8520, [0x36d8] = 0x8522, [0x36d9] = 0x8523,
  [0x36da] = 0x8524, [0x36db] = 0x8525, [0x36dc] = 0x8526, [0x36dd] = 0x8527,
  [0x36de] = 0x8528, [0x36df] = 0x8529, [0x36e0] = 0x852a, [0x36e1] = 0x852d,
  [0x36e2] = 0x852e, [0x36e3] = 0x852f, [0x36e4] = 0x8530, [0x36e5] = 0x8531,
  [0x36e6] = 0x8532, [0x36e7] = 0x8533, [0x36e8] = 0x8534, [0x36e9] = 0x8535,
  [0x36ea] = 0x8536, [0x36eb] = 0x853e, [0x36ec] = 0x853f, [0x36ed] = 0x8540,
  [0x36ee] = 0x8541, [0x36ef] = 0x8542, [0x36f0] = 0x8544, [0x36f1] = 0x8545,
  [0x36f2] = 0x8546, [0x36f3] = 0x8547, [0x36f4] = 0x854b, [0x36f5] = 0x854c,
  [0x36f6] = 0x854d, [0x36f7] = 0x854e, [0x36f8] = 0x854f, [0x36f9] = 0x8550,
  [0x36fa] = 0x8551, [0x36fb] = 0x8552, [0x36fc] = 0x8553, [0x36fd] = 0x8554,
  [0x36fe] = 0x8555, [0x3700] = 0x8557, [0x3701] = 0x8558, [0x3702] = 0x855a,
  [0x3703] = 0x855b, [0x3704] = 0x855c, [0x3705] = 0x855d, [0x3706] = 0x855f,
  [0x3707] = 0x8560, [0x3708] = 0x8561, [0x3709] = 0x8562, [0x370a] = 0x8563,
  [0x370b] = 0x8565, [0x370c] = 0x8566, [0x370d] = 0x8567, [0x370e] = 0x8569,
  [0x370f] = 0x856a, [0x3710] = 0x856b, [0x3711] = 0x856c, [0x3712] = 0x856d,
  [0x3713] = 0x856e, [0x3714] = 0x856f, [0x3715] = 0x8570, [0x3716] = 0x8571,
  [0x3717] = 0x8573, [0x3718] = 0x8575, [0x3719] = 0x8576, [0x371a] = 0x8577,
  [0x371b] = 0x8578, [0x371c] = 0x857c, [0x371d] = 0x857d, [0x371e] = 0x857f,
  [0x371f] = 0x8580, [0x3720] = 0x8581, [0x3721] = 0x7701, [0x3722] = 0x76db,
  [0x3723] = 0x5269, [0x3724] = 0x80dc, [0x3725] = 0x5723, [0x3726] = 0x5e08,
  [0x3727] = 0x5931, [0x3728] = 0x72ee, [0x3729] = 0x65bd, [0x372a] = 0x6e7f,
  [0x372b] = 0x8bd7, [0x372c] = 0x5c38, [0x372d] = 0x8671, [0x372e] = 0x5341,
  [0x372f] = 0x77f3, [0x3730] = 0x62fe, [0x3731] = 0x65f6, [0x3732] = 0x4ec0,
  [0x3733] = 0x98df, [0x3734] = 0x8680, [0x3735] = 0x5b9e, [0x3736] = 0x8bc6,
  [0x3737] = 0x53f2, [0x3738] = 0x77e2, [0x3739] = 0x4f7f, [0x373a] = 0x5c4e,
  [0x373b] = 0x9a76, [0x373c] = 0x59cb, [0x373d] = 0x5f0f, [0x373e] = 0x793a,
  [0x373f] = 0x58eb, [0x3740] = 0x4e16, [0x3741] = 0x67ff, [0x3742] = 0x4e8b,
  [0x3743] = 0x62ed, [0x3744] = 0x8a93, [0x3745] = 0x901d, [0x3746] = 0x52bf,
  [0x3747] = 0x662f, [0x3748] = 0x55dc, [0x3749] = 0x566c, [0x374a] = 0x9002,
  [0x374b] = 0x4ed5, [0x374c] = 0x4f8d, [0x374d] = 0x91ca, [0x374e] = 0x9970,
  [0x374f] = 0x6c0f, [0x3750] = 0x5e02, [0x3751] = 0x6043, [0x3752] = 0x5ba4,
  [0x3753] = 0x89c6, [0x3754] = 0x8bd5, [0x3755] = 0x6536, [0x3756] = 0x624b,
  [0x3757] = 0x9996, [0x3758] = 0x5b88, [0x3759] = 0x5bff, [0x375a] = 0x6388,
  [0x375b] = 0x552e, [0x375c] = 0x53d7, [0x375d] = 0x7626, [0x375e] = 0x517d,
  [0x375f] = 0x852c, [0x3760] = 0x67a2, [0x3761] = 0x68b3, [0x3762] = 0x6b8a,
  [0x3763] = 0x6292, [0x3764] = 0x8f93, [0x3765] = 0x53d4, [0x3766] = 0x8212,
  [0x3767] = 0x6dd1, [0x3768] = 0x758f, [0x3769] = 0x4e66, [0x376a] = 0x8d4e,
  [0x376b] = 0x5b70, [0x376c] = 0x719f, [0x376d] = 0x85af, [0x376e] = 0x6691,
  [0x376f] = 0x66d9, [0x3770] = 0x7f72, [0x3771] = 0x8700, [0x3772] = 0x9ecd,
  [0x3773] = 0x9f20, [0x3774] = 0x5c5e, [0x3775] = 0x672f, [0x3776] = 0x8ff0,
  [0x3777] = 0x6811, [0x3778] = 0x675f, [0x3779] = 0x620d, [0x377a] = 0x7ad6,
  [0x377b] = 0x5885, [0x377c] = 0x5eb6, [0x377d] = 0x6570, [0x377e] = 0x6f31,
  [0x3780] = 0x8582, [0x3781] = 0x8583, [0x3782] = 0x8586, [0x3783] = 0x8588,
  [0x3784] = 0x8589, [0x3785] = 0x858a, [0x3786] = 0x858b, [0x3787] = 0x858c,
  [0x3788] = 0x858d, [0x3789] = 0x858e, [0x378a] = 0x8590, [0x378b] = 0x8591,
  [0x378c] = 0x8592, [0x378d] = 0x8593, [0x378e] = 0x8594, [0x378f] = 0x8595,
  [0x3790] = 0x8596, [0x3791] = 0x8597, [0x3792] = 0x8598, [0x3793] = 0x8599,
  [0x3794] = 0x859a, [0x3795] = 0x859d, [0x3796] = 0x859e, [0x3797] = 0x859f,
  [0x3798] = 0x85a0, [0x3799] = 0x85a1, [0x379a] = 0x85a2, [0x379b] = 0x85a3,
  [0x379c] = 0x85a5, [0x379d] = 0x85a6, [0x379e] = 0x85a7, [0x379f] = 0x85a9,
  [0x37a0] = 0x85ab, [0x37a1] = 0x85ac, [0x37a2] = 0x85ad, [0x37a3] = 0x85b1,
  [0x37a4] = 0x85b2, [0x37a5] = 0x85b3, [0x37a6] = 0x85b4, [0x37a7] = 0x85b5,
  [0x37a8] = 0x85b6, [0x37a9] = 0x85b8, [0x37aa] = 0x85ba, [0x37ab] = 0x85bb,
  [0x37ac] = 0x85bc, [0x37ad] = 0x85bd, [0x37ae] = 0x85be, [0x37af] = 0x85bf,
  [0x37b0] = 0x85c0, [0x37b1] = 0x85c2, [0x37b2] = 0x85c3, [0x37b3] = 0x85c4,
  [0x37b4] = 0x85c5, [0x37b5] = 0x85c6, [0x37b6] = 0x85c7, [0x37b7] = 0x85c8,
  [0x37b8] = 0x85ca, [0x37b9] = 0x85cb, [0x37ba] = 0x85cc, [0x37bb] = 0x85cd,
  [0x37bc] = 0x85ce, [0x37bd] = 0x85d1, [0x37be] = 0x85d2, [0x37c0] = 0x85d4,
  [0x37c1] = 0x85d6, [0x37c2] = 0x85d7, [0x37c3] = 0x85d8, [0x37c4] = 0x85d9,
  [0x37c5] = 0x85da, [0x37c6] = 0x85db, [0x37c7] = 0x85dd, [0x37c8] = 0x85de,
  [0x37c9] = 0x85df, [0x37ca] = 0x85e0, [0x37cb] = 0x85e1, [0x37cc] = 0x85e2,
  [0x37cd] = 0x85e3, [0x37ce] = 0x85e5, [0x37cf] = 0x85e6, [0x37d0] = 0x85e7,
  [0x37d1] = 0x85e8, [0x37d2] = 0x85ea, [0x37d3] = 0x85eb, [0x37d4] = 0x85ec,
  [0x37d5] = 0x85ed, [0x37d6] = 0x85ee, [0x37d7] = 0x85ef, [0x37d8] = 0x85f0,
  [0x37d9] = 0x85f1, [0x37da] = 0x85f2, [0x37db] = 0x85f3, [0x37dc] = 0x85f4,
  [0x37dd] = 0x85f5, [0x37de] = 0x85f6, [0x37df] = 0x85f7, [0x37e0] = 0x85f8,
  [0x37e1] = 0x6055, [0x37e2] = 0x5237, [0x37e3] = 0x800d, [0x37e4] = 0x6454,
  [0x37e5] = 0x8870, [0x37e6] = 0x7529, [0x37e7] = 0x5e05, [0x37e8] = 0x6813,
  [0x37e9] = 0x62f4, [0x37ea] = 0x971c, [0x37eb] = 0x53cc, [0x37ec] = 0x723d,
  [0x37ed] = 0x8c01, [0x37ee] = 0x6c34, [0x37ef] = 0x7761, [0x37f0] = 0x7a0e,
  [0x37f1] = 0x542e, [0x37f2] = 0x77ac, [0x37f3] = 0x987a, [0x37f4] = 0x821c,
  [0x37f5] = 0x8bf4, [0x37f6] = 0x7855, [0x37f7] = 0x6714, [0x37f8] = 0x70c1,
  [0x37f9] = 0x65af, [0x37fa] = 0x6495, [0x37fb] = 0x5636, [0x37fc] = 0x601d,
  [0x37fd] = 0x79c1, [0x37fe] = 0x53f8, [0x37ff] = 0x4e1d, [0x3800] = 0x6b7b,
  [0x3801] = 0x8086, [0x3802] = 0x5bfa, [0x3803] = 0x55e3, [0x3804] = 0x56db,
  [0x3805] = 0x4f3a, [0x3806] = 0x4f3c, [0x3807] = 0x9972, [0x3808] = 0x5df3,
  [0x3809] = 0x677e, [0x380a] = 0x8038, [0x380b] = 0x6002, [0x380c] = 0x9882,
  [0x380d] = 0x9001, [0x380e] = 0x5b8b, [0x380f] = 0x8bbc, [0x3810] = 0x8bf5,
  [0x3811] = 0x641c, [0x3812] = 0x8258, [0x3813] = 0x64de, [0x3814] = 0x55fd,
  [0x3815] = 0x82cf, [0x3816] = 0x9165, [0x3817] = 0x4fd7, [0x3818] = 0x7d20,
  [0x3819] = 0x901f, [0x381a] = 0x7c9f, [0x381b] = 0x50f3, [0x381c] = 0x5851,
  [0x381d] = 0x6eaf, [0x381e] = 0x5bbf, [0x381f] = 0x8bc9, [0x3820] = 0x8083,
  [0x3821] = 0x9178, [0x3822] = 0x849c, [0x3823] = 0x7b97, [0x3824] = 0x867d,
  [0x3825] = 0x968b, [0x3826] = 0x968f, [0x3827] = 0x7ee5, [0x3828] = 0x9ad3,
  [0x3829] = 0x788e, [0x382a] = 0x5c81, [0x382b] = 0x7a57, [0x382c] = 0x9042,
  [0x382d] = 0x96a7, [0x382e] = 0x795f, [0x382f] = 0x5b59, [0x3830] = 0x635f,
  [0x3831] = 0x7b0b, [0x3832] = 0x84d1, [0x3833] = 0x68ad, [0x3834] = 0x5506,
  [0x3835] = 0x7f29, [0x3836] = 0x7410, [0x3837] = 0x7d22, [0x3838] = 0x9501,
  [0x3839] = 0x6240, [0x383a] = 0x584c, [0x383b] = 0x4ed6, [0x383c] = 0x5b83,
  [0x383d] = 0x5979, [0x383e] = 0x5854, [0x3840] = 0x85f9, [0x3841] = 0x85fa,
  [0x3842] = 0x85fc, [0x3843] = 0x85fd, [0x3844] = 0x85fe, [0x3845] = 0x8600,
  [0x3846] = 0x8601, [0x3847] = 0x8602, [0x3848] = 0x8603, [0x3849] = 0x8604,
  [0x384a] = 0x8606, [0x384b] = 0x8607, [0x384c] = 0x8608, [0x384d] = 0x8609,
  [0x384e] = 0x860a, [0x384f] = 0x860b, [0x3850] = 0x860c, [0x3851] = 0x860d,
  [0x3852] = 0x860e, [0x3853] = 0x860f, [0x3854] = 0x8610, [0x3855] = 0x8612,
  [0x3856] = 0x8613, [0x3857] = 0x8614, [0x3858] = 0x8615, [0x3859] = 0x8617,
  [0x385a] = 0x8618, [0x385b] = 0x8619, [0x385c] = 0x861a, [0x385d] = 0x861b,
  [0x385e] = 0x861c, [0x385f] = 0x861d, [0x3860] = 0x861e, [0x3861] = 0x861f,
  [0x3862] = 0x8620, [0x3863] = 0x8621, [0x3864] = 0x8622, [0x3865] = 0x8623,
  [0x3866] = 0x8624, [0x3867] = 0x8625, [0x3868] = 0x8626, [0x3869] = 0x8628,
  [0x386a] = 0x862a, [0x386b] = 0x862b, [0x386c] = 0x862c, [0x386d] = 0x862d,
  [0x386e] = 0x862e, [0x386f] = 0x862f, [0x3870] = 0x8630, [0x3871] = 0x8631,
  [0x3872] = 0x8632, [0x3873] = 0x8633, [0x3874] = 0x8634, [0x3875] = 0x8635,
  [0x3876] = 0x8636, [0x3877] = 0x8637, [0x3878] = 0x8639, [0x3879] = 0x863a,
  [0x387a] = 0x863b, [0x387b] = 0x863d, [0x387c] = 0x863e, [0x387d] = 0x863f,
  [0x387e] = 0x8640, [0x3880] = 0x8641, [0x3881] = 0x8642, [0x3882] = 0x8643,
  [0x3883] = 0x8644, [0x3884] = 0x8645, [0x3885] = 0x8646, [0x3886] = 0x8647,
  [0x3887] = 0x8648, [0x3888] = 0x8649, [0x3889] = 0x864a, [0x388a] = 0x864b,
  [0x388b] = 0x864c, [0x388c] = 0x8652, [0x388d] = 0x8653, [0x388e] = 0x8655,
  [0x388f] = 0x8656, [0x3890] = 0x8657, [0x3891] = 0x8658, [0x3892] = 0x8659,
  [0x3893] = 0x865b, [0x3894] = 0x865c, [0x3895] = 0x865d, [0x3896] = 0x865f,
  [0x3897] = 0x8660, [0x3898] = 0x8661, [0x3899] = 0x8663, [0x389a] = 0x8664,
  [0x389b] = 0x8665, [0x389c] = 0x8666, [0x389d] = 0x8667, [0x389e] = 0x8668,
  [0x389f] = 0x8669, [0x38a0] = 0x866a, [0x38a1] = 0x736d, [0x38a2] = 0x631e,
  [0x38a3] = 0x8e4b, [0x38a4] = 0x8e0f, [0x38a5] = 0x80ce, [0x38a6] = 0x82d4,
  [0x38a7] = 0x62ac, [0x38a8] = 0x53f0, [0x38a9] = 0x6cf0, [0x38aa] = 0x915e,
  [0x38ab] = 0x592a, [0x38ac] = 0x6001, [0x38ad] = 0x6c70, [0x38ae] = 0x574d,
  [0x38af] = 0x644a, [0x38b0] = 0x8d2a, [0x38b1] = 0x762b, [0x38b2] = 0x6ee9,
  [0x38b3] = 0x575b, [0x38b4] = 0x6a80, [0x38b5] = 0x75f0, [0x38b6] = 0x6f6d,
  [0x38b7] = 0x8c2d, [0x38b8] = 0x8c08, [0x38b9] = 0x5766, [0x38ba] = 0x6bef,
  [0x38bb] = 0x8892, [0x38bc] = 0x78b3, [0x38bd] = 0x63a2, [0x38be] = 0x53f9,
  [0x38bf] = 0x70ad, [0x38c0] = 0x6c64, [0x38c1] = 0x5858, [0x38c2] = 0x642a,
  [0x38c3] = 0x5802, [0x38c4] = 0x68e0, [0x38c5] = 0x819b, [0x38c6] = 0x5510,
  [0x38c7] = 0x7cd6, [0x38c8] = 0x5018, [0x38c9] = 0x8eba, [0x38ca] = 0x6dcc,
  [0x38cb] = 0x8d9f, [0x38cc] = 0x70eb, [0x38cd] = 0x638f, [0x38ce] = 0x6d9b,
  [0x38cf] = 0x6ed4, [0x38d0] = 0x7ee6, [0x38d1] = 0x8404, [0x38d2] = 0x6843,
  [0x38d3] = 0x9003, [0x38d4] = 0x6dd8, [0x38d5] = 0x9676, [0x38d6] = 0x8ba8,
  [0x38d7] = 0x5957, [0x38d8] = 0x7279, [0x38d9] = 0x85e4, [0x38da] = 0x817e,
  [0x38db] = 0x75bc, [0x38dc] = 0x8a8a, [0x38dd] = 0x68af, [0x38de] = 0x5254,
  [0x38df] = 0x8e22, [0x38e0] = 0x9511, [0x38e1] = 0x63d0, [0x38e2] = 0x9898,
  [0x38e3] = 0x8e44, [0x38e4] = 0x557c, [0x38e5] = 0x4f53, [0x38e6] = 0x66ff,
  [0x38e7] = 0x568f, [0x38e8] = 0x60d5, [0x38e9] = 0x6d95, [0x38ea] = 0x5243,
  [0x38eb] = 0x5c49, [0x38ec] = 0x5929, [0x38ed] = 0x6dfb, [0x38ee] = 0x586b,
  [0x38ef] = 0x7530, [0x38f0] = 0x751c, [0x38f1] = 0x606c, [0x38f2] = 0x8214,
  [0x38f3] = 0x8146, [0x38f4] = 0x6311, [0x38f5] = 0x6761, [0x38f6] = 0x8fe2,
  [0x38f7] = 0x773a, [0x38f8] = 0x8df3, [0x38f9] = 0x8d34, [0x38fa] = 0x94c1,
  [0x38fb] = 0x5e16, [0x38fc] = 0x5385, [0x38fd] = 0x542c, [0x38fe] = 0x70c3,
  [0x3900] = 0x866d, [0x3901] = 0x866f, [0x3902] = 0x8670, [0x3903] = 0x8672,
  [0x3904] = 0x8673, [0x3905] = 0x8674, [0x3906] = 0x8675, [0x3907] = 0x8676,
  [0x3908] = 0x8677, [0x3909] = 0x8678, [0x390a] = 0x8683, [0x390b] = 0x8684,
  [0x390c] = 0x8685, [0x390d] = 0x8686, [0x390e] = 0x8687, [0x390f] = 0x8688,
  [0x3910] = 0x8689, [0x3911] = 0x868e, [0x3912] = 0x868f, [0x3913] = 0x8690,
  [0x3914] = 0x8691, [0x3915] = 0x8692, [0x3916] = 0x8694, [0x3917] = 0x8696,
  [0x3918] = 0x8697, [0x3919] = 0x8698, [0x391a] = 0x8699, [0x391b] = 0x869a,
  [0x391c] = 0x869b, [0x391d] = 0x869e, [0x391e] = 0x869f, [0x391f] = 0x86a0,
  [0x3920] = 0x86a1, [0x3921] = 0x86a2, [0x3922] = 0x86a5, [0x3923] = 0x86a6,
  [0x3924] = 0x86ab, [0x3925] = 0x86ad, [0x3926] = 0x86ae, [0x3927] = 0x86b2,
  [0x3928] = 0x86b3, [0x3929] = 0x86b7, [0x392a] = 0x86b8, [0x392b] = 0x86b9,
  [0x392c] = 0x86bb, [0x392d] = 0x86bc, [0x392e] = 0x86bd, [0x392f] = 0x86be,
  [0x3930] = 0x86bf, [0x3931] = 0x86c1, [0x3932] = 0x86c2, [0x3933] = 0x86c3,
  [0x3934] = 0x86c5, [0x3935] = 0x86c8, [0x3936] = 0x86cc, [0x3937] = 0x86cd,
  [0x3938] = 0x86d2, [0x3939] = 0x86d3, [0x393a] = 0x86d5, [0x393b] = 0x86d6,
  [0x393c] = 0x86d7, [0x393d] = 0x86da, [0x393e] = 0x86dc, [0x3940] = 0x86dd,
  [0x3941] = 0x86e0, [0x3942] = 0x86e1, [0x3943] = 0x86e2, [0x3944] = 0x86e3,
  [0x3945] = 0x86e5, [0x3946] = 0x86e6, [0x3947] = 0x86e7, [0x3948] = 0x86e8,
  [0x3949] = 0x86ea, [0x394a] = 0x86eb, [0x394b] = 0x86ec, [0x394c] = 0x86ef,
  [0x394d] = 0x86f5, [0x394e] = 0x86f6, [0x394f] = 0x86f7, [0x3950] = 0x86fa,
  [0x3951] = 0x86fb, [0x3952] = 0x86fc, [0x3953] = 0x86fd, [0x3954] = 0x86ff,
  [0x3955] = 0x8701, [0x3956] = 0x8704, [0x3957] = 0x8705, [0x3958] = 0x8706,
  [0x3959] = 0x870b, [0x395a] = 0x870c, [0x395b] = 0x870e, [0x395c] = 0x870f,
  [0x395d] = 0x8710, [0x395e] = 0x8711, [0x395f] = 0x8714, [0x3960] = 0x8716,
  [0x3961] = 0x6c40, [0x3962] = 0x5ef7, [0x3963] = 0x505c, [0x3964] = 0x4ead,
  [0x3965] = 0x5ead, [0x3966] = 0x633a, [0x3967] = 0x8247, [0x3968] = 0x901a,
  [0x3969] = 0x6850, [0x396a] = 0x916e, [0x396b] = 0x77b3, [0x396c] = 0x540c,
  [0x396d] = 0x94dc, [0x396e] = 0x5f64, [0x396f] = 0x7ae5, [0x3970] = 0x6876,
  [0x3971] = 0x6345, [0x3972] = 0x7b52, [0x3973] = 0x7edf, [0x3974] = 0x75db,
  [0x3975] = 0x5077, [0x3976] = 0x6295, [0x3977] = 0x5934, [0x3978] = 0x900f,
  [0x3979] = 0x51f8, [0x397a] = 0x79c3, [0x397b] = 0x7a81, [0x397c] = 0x56fe,
  [0x397d] = 0x5f92, [0x397e] = 0x9014, [0x397f] = 0x6d82, [0x3980] = 0x5c60,
  [0x3981] = 0x571f, [0x3982] = 0x5410, [0x3983] = 0x5154, [0x3984] = 0x6e4d,
  [0x3985] = 0x56e2, [0x3986] = 0x63a8, [0x3987] = 0x9893, [0x3988] = 0x817f,
  [0x3989] = 0x8715, [0x398a] = 0x892a, [0x398b] = 0x9000, [0x398c] = 0x541e,
  [0x398d] = 0x5c6f, [0x398e] = 0x81c0, [0x398f] = 0x62d6, [0x3990] = 0x6258,
  [0x3991] = 0x8131, [0x3992] = 0x9e35, [0x3993] = 0x9640, [0x3994] = 0x9a6e,
  [0x3995] = 0x9a7c, [0x3996] = 0x692d, [0x3997] = 0x59a5, [0x3998] = 0x62d3,
  [0x3999] = 0x553e, [0x399a] = 0x6316, [0x399b] = 0x54c7, [0x399c] = 0x86d9,
  [0x399d] = 0x6d3c, [0x399e] = 0x5a03, [0x399f] = 0x74e6, [0x39a0] = 0x889c,
  [0x39a1] = 0x6b6a, [0x39a2] = 0x5916, [0x39a3] = 0x8c4c, [0x39a4] = 0x5f2f,
  [0x39a5] = 0x6e7e, [0x39a6] = 0x73a9, [0x39a7] = 0x987d, [0x39a8] = 0x4e38,
  [0x39a9] = 0x70f7, [0x39aa] = 0x5b8c, [0x39ab] = 0x7897, [0x39ac] = 0x633d,
  [0x39ad] = 0x665a, [0x39ae] = 0x7696, [0x39af] = 0x60cb, [0x39b0] = 0x5b9b,
  [0x39b1] = 0x5a49, [0x39b2] = 0x4e07, [0x39b3] = 0x8155, [0x39b4] = 0x6c6a,
  [0x39b5] = 0x738b, [0x39b6] = 0x4ea1, [0x39b7] = 0x6789, [0x39b8] = 0x7f51,
  [0x39b9] = 0x5f80, [0x39ba] = 0x65fa, [0x39bb] = 0x671b, [0x39bc] = 0x5fd8,
  [0x39bd] = 0x5984, [0x39be] = 0x5a01, [0x39c0] = 0x8719, [0x39c1] = 0x871b,
  [0x39c2] = 0x871d, [0x39c3] = 0x871f, [0x39c4] = 0x8720, [0x39c5] = 0x8724,
  [0x39c6] = 0x8726, [0x39c7] = 0x8727, [0x39c8] = 0x8728, [0x39c9] = 0x872a,
  [0x39ca] = 0x872b, [0x39cb] = 0x872c, [0x39cc] = 0x872d, [0x39cd] = 0x872f,
  [0x39ce] = 0x8730, [0x39cf] = 0x8732, [0x39d0] = 0x8733, [0x39d1] = 0x8735,
  [0x39d2] = 0x8736, [0x39d3] = 0x8738, [0x39d4] = 0x8739, [0x39d5] = 0x873a,
  [0x39d6] = 0x873c, [0x39d7] = 0x873d, [0x39d8] = 0x8740, [0x39d9] = 0x8741,
  [0x39da] = 0x8742, [0x39db] = 0x8743, [0x39dc] = 0x8744, [0x39dd] = 0x8745,
  [0x39de] = 0x8746, [0x39df] = 0x874a, [0x39e0] = 0x874b, [0x39e1] = 0x874d,
  [0x39e2] = 0x874f, [0x39e3] = 0x8750, [0x39e4] = 0x8751, [0x39e5] = 0x8752,
  [0x39e6] = 0x8754, [0x39e7] = 0x8755, [0x39e8] = 0x8756, [0x39e9] = 0x8758,
  [0x39ea] = 0x875a, [0x39eb] = 0x875b, [0x39ec] = 0x875c, [0x39ed] = 0x875d,
  [0x39ee] = 0x875e, [0x39ef] = 0x875f, [0x39f0] = 0x8761, [0x39f1] = 0x8762,
  [0x39f2] = 0x8766, [0x39f3] = 0x8767, [0x39f4] = 0x8768, [0x39f5] = 0x8769,
  [0x39f6] = 0x876a, [0x39f7] = 0x876b, [0x39f8] = 0x876c, [0x39f9] = 0x876d,
  [0x39fa] = 0x876f, [0x39fb] = 0x8771, [0x39fc] = 0x8772, [0x39fd] = 0x8773,
  [0x39fe] = 0x8775, [0x3a00] = 0x8777, [0x3a01] = 0x8778, [0x3a02] = 0x8779,
  [0x3a03] = 0x877a, [0x3a04] = 0x877f, [0x3a05] = 0x8780, [0x3a06] = 0x8781,
  [0x3a07] = 0x8784, [0x3a08] = 0x8786, [0x3a09] = 0x8787, [0x3a0a] = 0x8789,
  [0x3a0b] = 0x878a, [0x3a0c] = 0x878c, [0x3a0d] = 0x878e, [0x3a0e] = 0x878f,
  [0x3a0f] = 0x8790, [0x3a10] = 0x8791, [0x3a11] = 0x8792, [0x3a12] = 0x8794,
  [0x3a13] = 0x8795, [0x3a14] = 0x8796, [0x3a15] = 0x8798, [0x3a16] = 0x8799,
  [0x3a17] = 0x879a, [0x3a18] = 0x879b, [0x3a19] = 0x879c, [0x3a1a] = 0x879d,
  [0x3a1b] = 0x879e, [0x3a1c] = 0x87a0, [0x3a1d] = 0x87a1, [0x3a1e] = 0x87a2,
  [0x3a1f] = 0x87a3, [0x3a20] = 0x87a4, [0x3a21] = 0x5dcd, [0x3a22] = 0x5fae,
  [0x3a23] = 0x5371, [0x3a24] = 0x97e6, [0x3a25] = 0x8fdd, [0x3a26] = 0x6845,
  [0x3a27] = 0x56f4, [0x3a28] = 0x552f, [0x3a29] = 0x60df, [0x3a2a] = 0x4e3a,
  [0x3a2b] = 0x6f4d, [0x3a2c] = 0x7ef4, [0x3a2d] = 0x82c7, [0x3a2e] = 0x840e,
  [0x3a2f] = 0x59d4, [0x3a30] = 0x4f1f, [0x3a31] = 0x4f2a, [0x3a32] = 0x5c3e,
  [0x3a33] = 0x7eac, [0x3a34] = 0x672a, [0x3a35] = 0x851a, [0x3a36] = 0x5473,
  [0x3a37] = 0x754f, [0x3a38] = 0x80c3, [0x3a39] = 0x5582, [0x3a3a] = 0x9b4f,
  [0x3a3b] = 0x4f4d, [0x3a3c] = 0x6e2d, [0x3a3d] = 0x8c13, [0x3a3e] = 0x5c09,
  [0x3a3f] = 0x6170, [0x3a40] = 0x536b, [0x3a41] = 0x761f, [0x3a42] = 0x6e29,
  [0x3a43] = 0x868a, [0x3a44] = 0x6587, [0x3a45] = 0x95fb, [0x3a46] = 0x7eb9,
  [0x3a47] = 0x543b, [0x3a48] = 0x7a33, [0x3a49] = 0x7d0a, [0x3a4a] = 0x95ee,
  [0x3a4b] = 0x55e1, [0x3a4c] = 0x7fc1, [0x3a4d] = 0x74ee, [0x3a4e] = 0x631d,
  [0x3a4f] = 0x8717, [0x3a50] = 0x6da1, [0x3a51] = 0x7a9d, [0x3a52] = 0x6211,
  [0x3a53] = 0x65a1, [0x3a54] = 0x5367, [0x3a55] = 0x63e1, [0x3a56] = 0x6c83,
  [0x3a57] = 0x5deb, [0x3a58] = 0x545c, [0x3a59] = 0x94a8, [0x3a5a] = 0x4e4c,
  [0x3a5b] = 0x6c61, [0x3a5c] = 0x8bec, [0x3a5d] = 0x5c4b, [0x3a5e] = 0x65e0,
  [0x3a5f] = 0x829c, [0x3a60] = 0x68a7, [0x3a61] = 0x543e, [0x3a62] = 0x5434,
  [0x3a63] = 0x6bcb, [0x3a64] = 0x6b66, [0x3a65] = 0x4e94, [0x3a66] = 0x6342,
  [0x3a67] = 0x5348, [0x3a68] = 0x821e, [0x3a69] = 0x4f0d, [0x3a6a] = 0x4fae,
  [0x3a6b] = 0x575e, [0x3a6c] = 0x620a, [0x3a6d] = 0x96fe, [0x3a6e] = 0x6664,
  [0x3a6f] = 0x7269, [0x3a70] = 0x52ff, [0x3a71] = 0x52a1, [0x3a72] = 0x609f,
  [0x3a73] = 0x8bef, [0x3a74] = 0x6614, [0x3a75] = 0x7199, [0x3a76] = 0x6790,
  [0x3a77] = 0x897f, [0x3a78] = 0x7852, [0x3a79] = 0x77fd, [0x3a7a] = 0x6670,
  [0x3a7b] = 0x563b, [0x3a7c] = 0x5438, [0x3a7d] = 0x9521, [0x3a7e] = 0x727a,
  [0x3a80] = 0x87a5, [0x3a81] = 0x87a6, [0x3a82] = 0x87a7, [0x3a83] = 0x87a9,
  [0x3a84] = 0x87aa, [0x3a85] = 0x87ae, [0x3a86] = 0x87b0, [0x3a87] = 0x87b1,
  [0x3a88] = 0x87b2, [0x3a89] = 0x87b4, [0x3a8a] = 0x87b6, [0x3a8b] = 0x87b7,
  [0x3a8c] = 0x87b8, [0x3a8d] = 0x87b9, [0x3a8e] = 0x87bb, [0x3a8f] = 0x87bc,
  [0x3a90] = 0x87be, [0x3a91] = 0x87bf, [0x3a92] = 0x87c1, [0x3a93] = 0x87c2,
  [0x3a94] = 0x87c3, [0x3a95] = 0x87c4, [0x3a96] = 0x87c5, [0x3a97] = 0x87c7,
  [0x3a98] = 0x87c8, [0x3a99] = 0x87c9, [0x3a9a] = 0x87cc, [0x3a9b] = 0x87cd,
  [0x3a9c] = 0x87ce, [0x3a9d] = 0x87cf, [0x3a9e] = 0x87d0, [0x3a9f] = 0x87d4,
  [0x3aa0] = 0x87d5, [0x3aa1] = 0x87d6, [0x3aa2] = 0x87d7, [0x3aa3] = 0x87d8,
  [0x3aa4] = 0x87d9, [0x3aa5] = 0x87da, [0x3aa6] = 0x87dc, [0x3aa7] = 0x87dd,
  [0x3aa8] = 0x87de, [0x3aa9] = 0x87df, [0x3aaa] = 0x87e1, [0x3aab] = 0x87e2,
  [0x3aac] = 0x87e3, [0x3aad] = 0x87e4, [0x3aae] = 0x87e6, [0x3aaf] = 0x87e7,
  [0x3ab0] = 0x87e8, [0x3ab1] = 0x87e9, [0x3ab2] = 0x87eb, [0x3ab3] = 0x87ec,
  [0x3ab4] = 0x87ed, [0x3ab5] = 0x87ef, [0x3ab6] = 0x87f0, [0x3ab7] = 0x87f1,
  [0x3ab8] = 0x87f2, [0x3ab9] = 0x87f3, [0x3aba] = 0x87f4, [0x3abb] = 0x87f5,
  [0x3abc] = 0x87f6, [0x3abd] = 0x87f7, [0x3abe] = 0x87f8, [0x3ac0] = 0x87fa,
  [0x3ac1] = 0x87fb, [0x3ac2] = 0x87fc, [0x3ac3] = 0x87fd, [0x3ac4] = 0x87ff,
  [0x3ac5] = 0x8800, [0x3ac6] = 0x8801, [0x3ac7] = 0x8802, [0x3ac8] = 0x8804,
  [0x3ac9] = 0x8805, [0x3aca] = 0x8806, [0x3acb] = 0x8807, [0x3acc] = 0x8808,
  [0x3acd] = 0x8809, [0x3ace] = 0x880b, [0x3acf] = 0x880c, [0x3ad0] = 0x880d,
  [0x3ad1] = 0x880e, [0x3ad2] = 0x880f, [0x3ad3] = 0x8810, [0x3ad4] = 0x8811,
  [0x3ad5] = 0x8812, [0x3ad6] = 0x8814, [0x3ad7] = 0x8817, [0x3ad8] = 0x8818,
  [0x3ad9] = 0x8819, [0x3ada] = 0x881a, [0x3adb] = 0x881c, [0x3adc] = 0x881d,
  [0x3add] = 0x881e, [0x3ade] = 0x881f, [0x3adf] = 0x8820, [0x3ae0] = 0x8823,
  [0x3ae1] = 0x7a00, [0x3ae2] = 0x606f, [0x3ae3] = 0x5e0c, [0x3ae4] = 0x6089,
  [0x3ae5] = 0x819d, [0x3ae6] = 0x5915, [0x3ae7] = 0x60dc, [0x3ae8] = 0x7184,
  [0x3ae9] = 0x70ef, [0x3aea] = 0x6eaa, [0x3aeb] = 0x6c50, [0x3aec] = 0x7280,
  [0x3aed] = 0x6a84, [0x3aee] = 0x88ad, [0x3aef] = 0x5e2d, [0x3af0] = 0x4e60,
  [0x3af1] = 0x5ab3, [0x3af2] = 0x559c, [0x3af3] = 0x94e3, [0x3af4] = 0x6d17,
  [0x3af5] = 0x7cfb, [0x3af6] = 0x9699, [0x3af7] = 0x620f, [0x3af8] = 0x7ec6,
  [0x3af9] = 0x778e, [0x3afa] = 0x867e, [0x3afb] = 0x5323, [0x3afc] = 0x971e,
  [0x3afd] = 0x8f96, [0x3afe] = 0x6687, [0x3aff] = 0x5ce1, [0x3b00] = 0x4fa0,
  [0x3b01] = 0x72ed, [0x3b02] = 0x4e0b, [0x3b03] = 0x53a6, [0x3b04] = 0x590f,
  [0x3b05] = 0x5413, [0x3b06] = 0x6380, [0x3b07] = 0x9528, [0x3b08] = 0x5148,
  [0x3b09] = 0x4ed9, [0x3b0a] = 0x9c9c, [0x3b0b] = 0x7ea4, [0x3b0c] = 0x54b8,
  [0x3b0d] = 0x8d24, [0x3b0e] = 0x8854, [0x3b0f] = 0x8237, [0x3b10] = 0x95f2,
  [0x3b11] = 0x6d8e, [0x3b12] = 0x5f26, [0x3b13] = 0x5acc, [0x3b14] = 0x663e,
  [0x3b15] = 0x9669, [0x3b16] = 0x73b0, [0x3b17] = 0x732e, [0x3b18] = 0x53bf,
  [0x3b19] = 0x817a, [0x3b1a] = 0x9985, [0x3b1b] = 0x7fa1, [0x3b1c] = 0x5baa,
  [0x3b1d] = 0x9677, [0x3b1e] = 0x9650, [0x3b1f] = 0x7ebf, [0x3b20] = 0x76f8,
  [0x3b21] = 0x53a2, [0x3b22] = 0x9576, [0x3b23] = 0x9999, [0x3b24] = 0x7bb1,
  [0x3b25] = 0x8944, [0x3b26] = 0x6e58, [0x3b27] = 0x4e61, [0x3b28] = 0x7fd4,
  [0x3b29] = 0x7965, [0x3b2a] = 0x8be6, [0x3b2b] = 0x60f3, [0x3b2c] = 0x54cd,
  [0x3b2d] = 0x4eab, [0x3b2e] = 0x9879, [0x3b2f] = 0x5df7, [0x3b30] = 0x6a61,
  [0x3b31] = 0x50cf, [0x3b32] = 0x5411, [0x3b33] = 0x8c61, [0x3b34] = 0x8427,
  [0x3b35] = 0x785d, [0x3b36] = 0x9704, [0x3b37] = 0x524a, [0x3b38] = 0x54ee,
  [0x3b39] = 0x56a3, [0x3b3a] = 0x9500, [0x3b3b] = 0x6d88, [0x3b3c] = 0x5bb5,
  [0x3b3d] = 0x6dc6, [0x3b3e] = 0x6653, [0x3b40] = 0x8824, [0x3b41] = 0x8825,
  [0x3b42] = 0x8826, [0x3b43] = 0x8827, [0x3b44] = 0x8828, [0x3b45] = 0x8829,
  [0x3b46] = 0x882a, [0x3b47] = 0x882b, [0x3b48] = 0x882c, [0x3b49] = 0x882d,
  [0x3b4a] = 0x882e, [0x3b4b] = 0x882f, [0x3b4c] = 0x8830, [0x3b4d] = 0x8831,
  [0x3b4e] = 0x8833, [0x3b4f] = 0x8834, [0x3b50] = 0x8835, [0x3b51] = 0x8836,
  [0x3b52] = 0x8837, [0x3b53] = 0x8838, [0x3b54] = 0x883a, [0x3b55] = 0x883b,
  [0x3b56] = 0x883d, [0x3b57] = 0x883e, [0x3b58] = 0x883f, [0x3b59] = 0x8841,
  [0x3b5a] = 0x8842, [0x3b5b] = 0x8843, [0x3b5c] = 0x8846, [0x3b5d] = 0x8847,
  [0x3b5e] = 0x8848, [0x3b5f] = 0x8849, [0x3b60] = 0x884a, [0x3b61] = 0x884b,
  [0x3b62] = 0x884e, [0x3b63] = 0x884f, [0x3b64] = 0x8850, [0x3b65] = 0x8851,
  [0x3b66] = 0x8852, [0x3b67] = 0x8853, [0x3b68] = 0x8855, [0x3b69] = 0x8856,
  [0x3b6a] = 0x8858, [0x3b6b] = 0x885a, [0x3b6c] = 0x885b, [0x3b6d] = 0x885c,
  [0x3b6e] = 0x885d, [0x3b6f] = 0x885e, [0x3b70] = 0x885f, [0x3b71] = 0x8860,
  [0x3b72] = 0x8866, [0x3b73] = 0x8867, [0x3b74] = 0x886a, [0x3b75] = 0x886d,
  [0x3b76] = 0x886f, [0x3b77] = 0x8871, [0x3b78] = 0x8873, [0x3b79] = 0x8874,
  [0x3b7a] = 0x8875, [0x3b7b] = 0x8876, [0x3b7c] = 0x8878, [0x3b7d] = 0x8879,
  [0x3b7e] = 0x887a, [0x3b80] = 0x887b, [0x3b81] = 0x887c, [0x3b82] = 0x8880,
  [0x3b83] = 0x8883, [0x3b84] = 0x8886, [0x3b85] = 0x8887, [0x3b86] = 0x8889,
  [0x3b87] = 0x888a, [0x3b88] = 0x888c, [0x3b89] = 0x888e, [0x3b8a] = 0x888f,
  [0x3b8b] = 0x8890, [0x3b8c] = 0x8891, [0x3b8d] = 0x8893, [0x3b8e] = 0x8894,
  [0x3b8f] = 0x8895, [0x3b90] = 0x8897, [0x3b91] = 0x8898, [0x3b92] = 0x8899,
  [0x3b93] = 0x889a, [0x3b94] = 0x889b, [0x3b95] = 0x889d, [0x3b96] = 0x889e,
  [0x3b97] = 0x889f, [0x3b98] = 0x88a0, [0x3b99] = 0x88a1, [0x3b9a] = 0x88a3,
  [0x3b9b] = 0x88a5, [0x3b9c] = 0x88a6, [0x3b9d] = 0x88a7, [0x3b9e] = 0x88a8,
  [0x3b9f] = 0x88a9, [0x3ba0] = 0x88aa, [0x3ba1] = 0x5c0f, [0x3ba2] = 0x5b5d,
  [0x3ba3] = 0x6821, [0x3ba4] = 0x8096, [0x3ba5] = 0x5578, [0x3ba6] = 0x7b11,
  [0x3ba7] = 0x6548, [0x3ba8] = 0x6954, [0x3ba9] = 0x4e9b, [0x3baa] = 0x6b47,
  [0x3bab] = 0x874e, [0x3bac] = 0x978b, [0x3bad] = 0x534f, [0x3bae] = 0x631f,
  [0x3baf] = 0x643a, [0x3bb0] = 0x90aa, [0x3bb1] = 0x659c, [0x3bb2] = 0x80c1,
  [0x3bb3] = 0x8c10, [0x3bb4] = 0x5199, [0x3bb5] = 0x68b0, [0x3bb6] = 0x5378,
  [0x3bb7] = 0x87f9, [0x3bb8] = 0x61c8, [0x3bb9] = 0x6cc4, [0x3bba] = 0x6cfb,
  [0x3bbb] = 0x8c22, [0x3bbc] = 0x5c51, [0x3bbd] = 0x85aa, [0x3bbe] = 0x82af,
  [0x3bbf] = 0x950c, [0x3bc0] = 0x6b23, [0x3bc1] = 0x8f9b, [0x3bc2] = 0x65b0,
  [0x3bc3] = 0x5ffb, [0x3bc4] = 0x5fc3, [0x3bc5] = 0x4fe1, [0x3bc6] = 0x8845,
  [0x3bc7] = 0x661f, [0x3bc8] = 0x8165, [0x3bc9] = 0x7329, [0x3bca] = 0x60fa,
  [0x3bcb] = 0x5174, [0x3bcc] = 0x5211, [0x3bcd] = 0x578b, [0x3bce] = 0x5f62,
  [0x3bcf] = 0x90a2, [0x3bd0] = 0x884c, [0x3bd1] = 0x9192, [0x3bd2] = 0x5e78,
  [0x3bd3] = 0x674f, [0x3bd4] = 0x6027, [0x3bd5] = 0x59d3, [0x3bd6] = 0x5144,
  [0x3bd7] = 0x51f6, [0x3bd8] = 0x80f8, [0x3bd9] = 0x5308, [0x3bda] = 0x6c79,
  [0x3bdb] = 0x96c4, [0x3bdc] = 0x718a, [0x3bdd] = 0x4f11, [0x3bde] = 0x4fee,
  [0x3bdf] = 0x7f9e, [0x3be0] = 0x673d, [0x3be1] = 0x55c5, [0x3be2] = 0x9508,
  [0x3be3] = 0x79c0, [0x3be4] = 0x8896, [0x3be5] = 0x7ee3, [0x3be6] = 0x589f,
  [0x3be7] = 0x620c, [0x3be8] = 0x9700, [0x3be9] = 0x865a, [0x3bea] = 0x5618,
  [0x3beb] = 0x987b, [0x3bec] = 0x5f90, [0x3bed] = 0x8bb8, [0x3bee] = 0x84c4,
  [0x3bef] = 0x9157, [0x3bf0] = 0x53d9, [0x3bf1] = 0x65ed, [0x3bf2] = 0x5e8f,
  [0x3bf3] = 0x755c, [0x3bf4] = 0x6064, [0x3bf5] = 0x7d6e, [0x3bf6] = 0x5a7f,
  [0x3bf7] = 0x7eea, [0x3bf8] = 0x7eed, [0x3bf9] = 0x8f69, [0x3bfa] = 0x55a7,
  [0x3bfb] = 0x5ba3, [0x3bfc] = 0x60ac, [0x3bfd] = 0x65cb, [0x3bfe] = 0x7384,
  [0x3c00] = 0x88ac, [0x3c01] = 0x88ae, [0x3c02] = 0x88af, [0x3c03] = 0x88b0,
  [0x3c04] = 0x88b2, [0x3c05] = 0x88b3, [0x3c06] = 0x88b4, [0x3c07] = 0x88b5,
  [0x3c08] = 0x88b6, [0x3c09] = 0x88b8, [0x3c0a] = 0x88b9, [0x3c0b] = 0x88ba,
  [0x3c0c] = 0x88bb, [0x3c0d] = 0x88bd, [0x3c0e] = 0x88be, [0x3c0f] = 0x88bf,
  [0x3c10] = 0x88c0, [0x3c11] = 0x88c3, [0x3c12] = 0x88c4, [0x3c13] = 0x88c7,
  [0x3c14] = 0x88c8, [0x3c15] = 0x88ca, [0x3c16] = 0x88cb, [0x3c17] = 0x88cc,
  [0x3c18] = 0x88cd, [0x3c19] = 0x88cf, [0x3c1a] = 0x88d0, [0x3c1b] = 0x88d1,
  [0x3c1c] = 0x88d3, [0x3c1d] = 0x88d6, [0x3c1e] = 0x88d7, [0x3c1f] = 0x88da,
  [0x3c20] = 0x88db, [0x3c21] = 0x88dc, [0x3c22] = 0x88dd, [0x3c23] = 0x88de,
  [0x3c24] = 0x88e0, [0x3c25] = 0x88e1, [0x3c26] = 0x88e6, [0x3c27] = 0x88e7,
  [0x3c28] = 0x88e9, [0x3c29] = 0x88ea, [0x3c2a] = 0x88eb, [0x3c2b] = 0x88ec,
  [0x3c2c] = 0x88ed, [0x3c2d] = 0x88ee, [0x3c2e] = 0x88ef, [0x3c2f] = 0x88f2,
  [0x3c30] = 0x88f5, [0x3c31] = 0x88f6, [0x3c32] = 0x88f7, [0x3c33] = 0x88fa,
  [0x3c34] = 0x88fb, [0x3c35] = 0x88fd, [0x3c36] = 0x88ff, [0x3c37] = 0x8900,
  [0x3c38] = 0x8901, [0x3c39] = 0x8903, [0x3c3a] = 0x8904, [0x3c3b] = 0x8905,
  [0x3c3c] = 0x8906, [0x3c3d] = 0x8907, [0x3c3e] = 0x8908, [0x3c40] = 0x8909,
  [0x3c41] = 0x890b, [0x3c42] = 0x890c, [0x3c43] = 0x890d, [0x3c44] = 0x890e,
  [0x3c45] = 0x890f, [0x3c46] = 0x8911, [0x3c47] = 0x8914, [0x3c48] = 0x8915,
  [0x3c49] = 0x8916, [0x3c4a] = 0x8917, [0x3c4b] = 0x8918, [0x3c4c] = 0x891c,
  [0x3c4d] = 0x891d, [0x3c4e] = 0x891e, [0x3c4f] = 0x891f, [0x3c50] = 0x8920,
  [0x3c51] = 0x8922, [0x3c52] = 0x8923, [0x3c53] = 0x8924, [0x3c54] = 0x8926,
  [0x3c55] = 0x8927, [0x3c56] = 0x8928, [0x3c57] = 0x8929, [0x3c58] = 0x892c,
  [0x3c59] = 0x892d, [0x3c5a] = 0x892e, [0x3c5b] = 0x892f, [0x3c5c] = 0x8931,
  [0x3c5d] = 0x8932, [0x3c5e] = 0x8933, [0x3c5f] = 0x8935, [0x3c60] = 0x8937,
  [0x3c61] = 0x9009, [0x3c62] = 0x7663, [0x3c63] = 0x7729, [0x3c64] = 0x7eda,
  [0x3c65] = 0x9774, [0x3c66] = 0x859b, [0x3c67] = 0x5b66, [0x3c68] = 0x7a74,
  [0x3c69] = 0x96ea, [0x3c6a] = 0x8840, [0x3c6b] = 0x52cb, [0x3c6c] = 0x718f,
  [0x3c6d] = 0x5faa, [0x3c6e] = 0x65ec, [0x3c6f] = 0x8be2, [0x3c70] = 0x5bfb,
  [0x3c71] = 0x9a6f, [0x3c72] = 0x5de1, [0x3c73] = 0x6b89, [0x3c74] = 0x6c5b,
  [0x3c75] = 0x8bad, [0x3c76] = 0x8baf, [0x3c77] = 0x900a, [0x3c78] = 0x8fc5,
  [0x3c79] = 0x538b, [0x3c7a] = 0x62bc, [0x3c7b] = 0x9e26, [0x3c7c] = 0x9e2d,
  [0x3c7d] = 0x5440, [0x3c7e] = 0x4e2b, [0x3c7f] = 0x82bd, [0x3c80] = 0x7259,
  [0x3c81] = 0x869c, [0x3c82] = 0x5d16, [0x3c83] = 0x8859, [0x3c84] = 0x6daf,
  [0x3c85] = 0x96c5, [0x3c86] = 0x54d1, [0x3c87] = 0x4e9a, [0x3c88] = 0x8bb6,
  [0x3c89] = 0x7109, [0x3c8a] = 0x54bd, [0x3c8b] = 0x9609, [0x3c8c] = 0x70df,
  [0x3c8d] = 0x6df9, [0x3c8e] = 0x76d0, [0x3c8f] = 0x4e25, [0x3c90] = 0x7814,
  [0x3c91] = 0x8712, [0x3c92] = 0x5ca9, [0x3c93] = 0x5ef6, [0x3c94] = 0x8a00,
  [0x3c95] = 0x989c, [0x3c96] = 0x960e, [0x3c97] = 0x708e, [0x3c98] = 0x6cbf,
  [0x3c99] = 0x5944, [0x3c9a] = 0x63a9, [0x3c9b] = 0x773c, [0x3c9c] = 0x884d,
  [0x3c9d] = 0x6f14, [0x3c9e] = 0x8273, [0x3c9f] = 0x5830, [0x3ca0] = 0x71d5,
  [0x3ca1] = 0x538c, [0x3ca2] = 0x781a, [0x3ca3] = 0x96c1, [0x3ca4] = 0x5501,
  [0x3ca5] = 0x5f66, [0x3ca6] = 0x7130, [0x3ca7] = 0x5bb4, [0x3ca8] = 0x8c1a,
  [0x3ca9] = 0x9a8c, [0x3caa] = 0x6b83, [0x3cab] = 0x592e, [0x3cac] = 0x9e2f,
  [0x3cad] = 0x79e7, [0x3cae] = 0x6768, [0x3caf] = 0x626c, [0x3cb0] = 0x4f6f,
  [0x3cb1] = 0x75a1, [0x3cb2] = 0x7f8a, [0x3cb3] = 0x6d0b, [0x3cb4] = 0x9633,
  [0x3cb5] = 0x6c27, [0x3cb6] = 0x4ef0, [0x3cb7] = 0x75d2, [0x3cb8] = 0x517b,
  [0x3cb9] = 0x6837, [0x3cba] = 0x6f3e, [0x3cbb] = 0x9080, [0x3cbc] = 0x8170,
  [0x3cbd] = 0x5996, [0x3cbe] = 0x7476, [0x3cc0] = 0x8938, [0x3cc1] = 0x8939,
  [0x3cc2] = 0x893a, [0x3cc3] = 0x893b, [0x3cc4] = 0x893c, [0x3cc5] = 0x893d,
  [0x3cc6] = 0x893e, [0x3cc7] = 0x893f, [0x3cc8] = 0x8940, [0x3cc9] = 0x8942,
  [0x3cca] = 0x8943, [0x3ccb] = 0x8945, [0x3ccc] = 0x8946, [0x3ccd] = 0x8947,
  [0x3cce] = 0x8948, [0x3ccf] = 0x8949, [0x3cd0] = 0x894a, [0x3cd1] = 0x894b,
  [0x3cd2] = 0x894c, [0x3cd3] = 0x894d, [0x3cd4] = 0x894e, [0x3cd5] = 0x894f,
  [0x3cd6] = 0x8950, [0x3cd7] = 0x8951, [0x3cd8] = 0x8952, [0x3cd9] = 0x8953,
  [0x3cda] = 0x8954, [0x3cdb] = 0x8955, [0x3cdc] = 0x8956, [0x3cdd] = 0x8957,
  [0x3cde] = 0x8958, [0x3cdf] = 0x8959, [0x3ce0] = 0x895a, [0x3ce1] = 0x895b,
  [0x3ce2] = 0x895c, [0x3ce3] = 0x895d, [0x3ce4] = 0x8960, [0x3ce5] = 0x8961,
  [0x3ce6] = 0x8962, [0x3ce7] = 0x8963, [0x3ce8] = 0x8964, [0x3ce9] = 0x8965,
  [0x3cea] = 0x8967, [0x3ceb] = 0x8968, [0x3cec] = 0x8969, [0x3ced] = 0x896a,
  [0x3cee] = 0x896b, [0x3cef] = 0x896c, [0x3cf0] = 0x896d, [0x3cf1] = 0x896e,
  [0x3cf2] = 0x896f, [0x3cf3] = 0x8970, [0x3cf4] = 0x8971, [0x3cf5] = 0x8972,
  [0x3cf6] = 0x8973, [0x3cf7] = 0x8974, [0x3cf8] = 0x8975, [0x3cf9] = 0x8976,
  [0x3cfa] = 0x8977, [0x3cfb] = 0x8978, [0x3cfc] = 0x8979, [0x3cfd] = 0x897a,
  [0x3cfe] = 0x897c, [0x3d00] = 0x897d, [0x3d01] = 0x897e, [0x3d02] = 0x8980,
  [0x3d03] = 0x8982, [0x3d04] = 0x8984, [0x3d05] = 0x8985, [0x3d06] = 0x8987,
  [0x3d07] = 0x8988, [0x3d08] = 0x8989, [0x3d09] = 0x898a, [0x3d0a] = 0x898b,
  [0x3d0b] = 0x898c, [0x3d0c] = 0x898d, [0x3d0d] = 0x898e, [0x3d0e] = 0x898f,
  [0x3d0f] = 0x8990, [0x3d10] = 0x8991, [0x3d11] = 0x8992, [0x3d12] = 0x8993,
  [0x3d13] = 0x8994, [0x3d14] = 0x8995, [0x3d15] = 0x8996, [0x3d16] = 0x8997,
  [0x3d17] = 0x8998, [0x3d18] = 0x8999, [0x3d19] = 0x899a, [0x3d1a] = 0x899b,
  [0x3d1b] = 0x899c, [0x3d1c] = 0x899d, [0x3d1d] = 0x899e, [0x3d1e] = 0x899f,
  [0x3d1f] = 0x89a0, [0x3d20] = 0x89a1, [0x3d21] = 0x6447, [0x3d22] = 0x5c27,
  [0x3d23] = 0x9065, [0x3d24] = 0x7a91, [0x3d25] = 0x8c23, [0x3d26] = 0x59da,
  [0x3d27] = 0x54ac, [0x3d28] = 0x8200, [0x3d29] = 0x836f, [0x3d2a] = 0x8981,
  [0x3d2b] = 0x8000, [0x3d2c] = 0x6930, [0x3d2d] = 0x564e, [0x3d2e] = 0x8036,
  [0x3d2f] = 0x7237, [0x3d30] = 0x91ce, [0x3d31] = 0x51b6, [0x3d32] = 0x4e5f,
  [0x3d33] = 0x9875, [0x3d34] = 0x6396, [0x3d35] = 0x4e1a, [0x3d36] = 0x53f6,
  [0x3d37] = 0x66f3, [0x3d38] = 0x814b, [0x3d39] = 0x591c, [0x3d3a] = 0x6db2,
  [0x3d3b] = 0x4e00, [0x3d3c] = 0x58f9, [0x3d3d] = 0x533b, [0x3d3e] = 0x63d6,
  [0x3d3f] = 0x94f1, [0x3d40] = 0x4f9d, [0x3d41] = 0x4f0a, [0x3d42] = 0x8863,
  [0x3d43] = 0x9890, [0x3d44] = 0x5937, [0x3d45] = 0x9057, [0x3d46] = 0x79fb,
  [0x3d47] = 0x4eea, [0x3d48] = 0x80f0, [0x3d49] = 0x7591, [0x3d4a] = 0x6c82,
  [0x3d4b] = 0x5b9c, [0x3d4c] = 0x59e8, [0x3d4d] = 0x5f5d, [0x3d4e] = 0x6905,
  [0x3d4f] = 0x8681, [0x3d50] = 0x501a, [0x3d51] = 0x5df2, [0x3d52] = 0x4e59,
  [0x3d53] = 0x77e3, [0x3d54] = 0x4ee5, [0x3d55] = 0x827a, [0x3d56] = 0x6291,
  [0x3d57] = 0x6613, [0x3d58] = 0x9091, [0x3d59] = 0x5c79, [0x3d5a] = 0x4ebf,
  [0x3d5b] = 0x5f79, [0x3d5c] = 0x81c6, [0x3d5d] = 0x9038, [0x3d5e] = 0x8084,
  [0x3d5f] = 0x75ab, [0x3d60] = 0x4ea6, [0x3d61] = 0x88d4, [0x3d62] = 0x610f,
  [0x3d63] = 0x6bc5, [0x3d64] = 0x5fc6, [0x3d65] = 0x4e49, [0x3d66] = 0x76ca,
  [0x3d67] = 0x6ea2, [0x3d68] = 0x8be3, [0x3d69] = 0x8bae, [0x3d6a] = 0x8c0a,
  [0x3d6b] = 0x8bd1, [0x3d6c] = 0x5f02, [0x3d6d] = 0x7ffc, [0x3d6e] = 0x7fcc,
  [0x3d6f] = 0x7ece, [0x3d70] = 0x8335, [0x3d71] = 0x836b, [0x3d72] = 0x56e0,
  [0x3d73] = 0x6bb7, [0x3d74] = 0x97f3, [0x3d75] = 0x9634, [0x3d76] = 0x59fb,
  [0x3d77] = 0x541f, [0x3d78] = 0x94f6, [0x3d79] = 0x6deb, [0x3d7a] = 0x5bc5,
  [0x3d7b] = 0x996e, [0x3d7c] = 0x5c39, [0x3d7d] = 0x5f15, [0x3d7e] = 0x9690,
  [0x3d80] = 0x89a2, [0x3d81] = 0x89a3, [0x3d82] = 0x89a4, [0x3d83] = 0x89a5,
  [0x3d84] = 0x89a6, [0x3d85] = 0x89a7, [0x3d86] = 0x89a8, [0x3d87] = 0x89a9,
  [0x3d88] = 0x89aa, [0x3d89] = 0x89ab, [0x3d8a] = 0x89ac, [0x3d8b] = 0x89ad,
  [0x3d8c] = 0x89ae, [0x3d8d] = 0x89af, [0x3d8e] = 0x89b0, [0x3d8f] = 0x89b1,
  [0x3d90] = 0x89b2, [0x3d91] = 0x89b3, [0x3d92] = 0x89b4, [0x3d93] = 0x89b5,
  [0x3d94] = 0x89b6, [0x3d95] = 0x89b7, [0x3d96] = 0x89b8, [0x3d97] = 0x89b9,
  [0x3d98] = 0x89ba, [0x3d99] = 0x89bb, [0x3d9a] = 0x89bc, [0x3d9b] = 0x89bd,
  [0x3d9c] = 0x89be, [0x3d9d] = 0x89bf, [0x3d9e] = 0x89c0, [0x3d9f] = 0x89c3,
  [0x3da0] = 0x89cd, [0x3da1] = 0x89d3, [0x3da2] = 0x89d4, [0x3da3] = 0x89d5,
  [0x3da4] = 0x89d7, [0x3da5] = 0x89d8, [0x3da6] = 0x89d9, [0x3da7] = 0x89db,
  [0x3da8] = 0x89dd, [0x3da9] = 0x89df, [0x3daa] = 0x89e0, [0x3dab] = 0x89e1,
  [0x3dac] = 0x89e2, [0x3dad] = 0x89e4, [0x3dae] = 0x89e7, [0x3daf] = 0x89e8,
  [0x3db0] = 0x89e9, [0x3db1] = 0x89ea, [0x3db2] = 0x89ec, [0x3db3] = 0x89ed,
  [0x3db4] = 0x89ee, [0x3db5] = 0x89f0, [0x3db6] = 0x89f1, [0x3db7] = 0x89f2,
  [0x3db8] = 0x89f4, [0x3db9] = 0x89f5, [0x3dba] = 0x89f6, [0x3dbb] = 0x89f7,
  [0x3dbc] = 0x89f8, [0x3dbd] = 0x89f9, [0x3dbe] = 0x89fa, [0x3dc0] = 0x89fb,
  [0x3dc1] = 0x89fc, [0x3dc2] = 0x89fd, [0x3dc3] = 0x89fe, [0x3dc4] = 0x89ff,
  [0x3dc5] = 0x8a01, [0x3dc6] = 0x8a02, [0x3dc7] = 0x8a03, [0x3dc8] = 0x8a04,
  [0x3dc9] = 0x8a05, [0x3dca] = 0x8a06, [0x3dcb] = 0x8a08, [0x3dcc] = 0x8a09,
  [0x3dcd] = 0x8a0a, [0x3dce] = 0x8a0b, [0x3dcf] = 0x8a0c, [0x3dd0] = 0x8a0d,
  [0x3dd1] = 0x8a0e, [0x3dd2] = 0x8a0f, [0x3dd3] = 0x8a10, [0x3dd4] = 0x8a11,
  [0x3dd5] = 0x8a12, [0x3dd6] = 0x8a13, [0x3dd7] = 0x8a14, [0x3dd8] = 0x8a15,
  [0x3dd9] = 0x8a16, [0x3dda] = 0x8a17, [0x3ddb] = 0x8a18, [0x3ddc] = 0x8a19,
  [0x3ddd] = 0x8a1a, [0x3dde] = 0x8a1b, [0x3ddf] = 0x8a1c, [0x3de0] = 0x8a1d,
  [0x3de1] = 0x5370, [0x3de2] = 0x82f1, [0x3de3] = 0x6a31, [0x3de4] = 0x5a74,
  [0x3de5] = 0x9e70, [0x3de6] = 0x5e94, [0x3de7] = 0x7f28, [0x3de8] = 0x83b9,
  [0x3de9] = 0x8424, [0x3dea] = 0x8425, [0x3deb] = 0x8367, [0x3dec] = 0x8747,
  [0x3ded] = 0x8fce, [0x3dee] = 0x8d62, [0x3def] = 0x76c8, [0x3df0] = 0x5f71,
  [0x3df1] = 0x9896, [0x3df2] = 0x786c, [0x3df3] = 0x6620, [0x3df4] = 0x54df,
  [0x3df5] = 0x62e5, [0x3df6] = 0x4f63, [0x3df7] = 0x81c3, [0x3df8] = 0x75c8,
  [0x3df9] = 0x5eb8, [0x3dfa] = 0x96cd, [0x3dfb] = 0x8e0a, [0x3dfc] = 0x86f9,
  [0x3dfd] = 0x548f, [0x3dfe] = 0x6cf3, [0x3dff] = 0x6d8c, [0x3e00] = 0x6c38,
  [0x3e01] = 0x607f, [0x3e02] = 0x52c7, [0x3e03] = 0x7528, [0x3e04] = 0x5e7d,
  [0x3e05] = 0x4f18, [0x3e06] = 0x60a0, [0x3e07] = 0x5fe7, [0x3e08] = 0x5c24,
  [0x3e09] = 0x7531, [0x3e0a] = 0x90ae, [0x3e0b] = 0x94c0, [0x3e0c] = 0x72b9,
  [0x3e0d] = 0x6cb9, [0x3e0e] = 0x6e38, [0x3e0f] = 0x9149, [0x3e10] = 0x6709,
  [0x3e11] = 0x53cb, [0x3e12] = 0x53f3, [0x3e13] = 0x4f51, [0x3e14] = 0x91c9,
  [0x3e15] = 0x8bf1, [0x3e16] = 0x53c8, [0x3e17] = 0x5e7c, [0x3e18] = 0x8fc2,
  [0x3e19] = 0x6de4, [0x3e1a] = 0x4e8e, [0x3e1b] = 0x76c2, [0x3e1c] = 0x6986,
  [0x3e1d] = 0x865e, [0x3e1e] = 0x611a, [0x3e1f] = 0x8206, [0x3e20] = 0x4f59,
  [0x3e21] = 0x4fde, [0x3e22] = 0x903e, [0x3e23] = 0x9c7c, [0x3e24] = 0x6109,
  [0x3e25] = 0x6e1d, [0x3e26] = 0x6e14, [0x3e27] = 0x9685, [0x3e28] = 0x4e88,
  [0x3e29] = 0x5a31, [0x3e2a] = 0x96e8, [0x3e2b] = 0x4e0e, [0x3e2c] = 0x5c7f,
  [0x3e2d] = 0x79b9, [0x3e2e] = 0x5b87, [0x3e2f] = 0x8bed, [0x3e30] = 0x7fbd,
  [0x3e31] = 0x7389, [0x3e32] = 0x57df, [0x3e33] = 0x828b, [0x3e34] = 0x90c1,
  [0x3e35] = 0x5401, [0x3e36] = 0x9047, [0x3e37] = 0x55bb, [0x3e38] = 0x5cea,
  [0x3e39] = 0x5fa1, [0x3e3a] = 0x6108, [0x3e3b] = 0x6b32, [0x3e3c] = 0x72f1,
  [0x3e3d] = 0x80b2, [0x3e3e] = 0x8a89, [0x3e40] = 0x8a1e, [0x3e41] = 0x8a1f,
  [0x3e42] = 0x8a20, [0x3e43] = 0x8a21, [0x3e44] = 0x8a22, [0x3e45] = 0x8a23,
  [0x3e46] = 0x8a24, [0x3e47] = 0x8a25, [0x3e48] = 0x8a26, [0x3e49] = 0x8a27,
  [0x3e4a] = 0x8a28, [0x3e4b] = 0x8a29, [0x3e4c] = 0x8a2a, [0x3e4d] = 0x8a2b,
  [0x3e4e] = 0x8a2c, [0x3e4f] = 0x8a2d, [0x3e50] = 0x8a2e, [0x3e51] = 0x8a2f,
  [0x3e52] = 0x8a30, [0x3e53] = 0x8a31, [0x3e54] = 0x8a32, [0x3e55] = 0x8a33,
  [0x3e56] = 0x8a34, [0x3e57] = 0x8a35, [0x3e58] = 0x8a36, [0x3e59] = 0x8a37,
  [0x3e5a] = 0x8a38, [0x3e5b] = 0x8a39, [0x3e5c] = 0x8a3a, [0x3e5d] = 0x8a3b,
  [0x3e5e] = 0x8a3c, [0x3e5f] = 0x8a3d, [0x3e60] = 0x8a3f, [0x3e61] = 0x8a40,
  [0x3e62] = 0x8a41, [0x3e63] = 0x8a42, [0x3e64] = 0x8a43, [0x3e65] = 0x8a44,
  [0x3e66] = 0x8a45, [0x3e67] = 0x8a46, [0x3e68] = 0x8a47, [0x3e69] = 0x8a49,
  [0x3e6a] = 0x8a4a, [0x3e6b] = 0x8a4b, [0x3e6c] = 0x8a4c, [0x3e6d] = 0x8a4d,
  [0x3e6e] = 0x8a4e, [0x3e6f] = 0x8a4f, [0x3e70] = 0x8a50, [0x3e71] = 0x8a51,
  [0x3e72] = 0x8a52, [0x3e73] = 0x8a53, [0x3e74] = 0x8a54, [0x3e75] = 0x8a55,
  [0x3e76] = 0x8a56, [0x3e77] = 0x8a57, [0x3e78] = 0x8a58, [0x3e79] = 0x8a59,
  [0x3e7a] = 0x8a5a, [0x3e7b] = 0x8a5b, [0x3e7c] = 0x8a5c, [0x3e7d] = 0x8a5d,
  [0x3e7e] = 0x8a5e, [0x3e80] = 0x8a5f, [0x3e81] = 0x8a60, [0x3e82] = 0x8a61,
  [0x3e83] = 0x8a62, [0x3e84] = 0x8a63, [0x3e85] = 0x8a64, [0x3e86] = 0x8a65,
  [0x3e87] = 0x8a66, [0x3e88] = 0x8a67, [0x3e89] = 0x8a68, [0x3e8a] = 0x8a69,
  [0x3e8b] = 0x8a6a, [0x3e8c] = 0x8a6b, [0x3e8d] = 0x8a6c, [0x3e8e] = 0x8a6d,
  [0x3e8f] = 0x8a6e, [0x3e90] = 0x8a6f, [0x3e91] = 0x8a70, [0x3e92] = 0x8a71,
  [0x3e93] = 0x8a72, [0x3e94] = 0x8a73, [0x3e95] = 0x8a74, [0x3e96] = 0x8a75,
  [0x3e97] = 0x8a76, [0x3e98] = 0x8a77, [0x3e99] = 0x8a78, [0x3e9a] = 0x8a7a,
  [0x3e9b] = 0x8a7b, [0x3e9c] = 0x8a7c, [0x3e9d] = 0x8a7d, [0x3e9e] = 0x8a7e,
  [0x3e9f] = 0x8a7f, [0x3ea0] = 0x8a80, [0x3ea1] = 0x6d74, [0x3ea2] = 0x5bd3,
  [0x3ea3] = 0x88d5, [0x3ea4] = 0x9884, [0x3ea5] = 0x8c6b, [0x3ea6] = 0x9a6d,
  [0x3ea7] = 0x9e33, [0x3ea8] = 0x6e0a, [0x3ea9] = 0x51a4, [0x3eaa] = 0x5143,
  [0x3eab] = 0x57a3, [0x3eac] = 0x8881, [0x3ead] = 0x539f, [0x3eae] = 0x63f4,
  [0x3eaf] = 0x8f95, [0x3eb0] = 0x56ed, [0x3eb1] = 0x5458, [0x3eb2] = 0x5706,
  [0x3eb3] = 0x733f, [0x3eb4] = 0x6e90, [0x3eb5] = 0x7f18, [0x3eb6] = 0x8fdc,
  [0x3eb7] = 0x82d1, [0x3eb8] = 0x613f, [0x3eb9] = 0x6028, [0x3eba] = 0x9662,
  [0x3ebb] = 0x66f0, [0x3ebc] = 0x7ea6, [0x3ebd] = 0x8d8a, [0x3ebe] = 0x8dc3,
  [0x3ebf] = 0x94a5, [0x3ec0] = 0x5cb3, [0x3ec1] = 0x7ca4, [0x3ec2] = 0x6708,
  [0x3ec3] = 0x60a6, [0x3ec4] = 0x9605, [0x3ec5] = 0x8018, [0x3ec6] = 0x4e91,
  [0x3ec7] = 0x90e7, [0x3ec8] = 0x5300, [0x3ec9] = 0x9668, [0x3eca] = 0x5141,
  [0x3ecb] = 0x8fd0, [0x3ecc] = 0x8574, [0x3ecd] = 0x915d, [0x3ece] = 0x6655,
  [0x3ecf] = 0x97f5, [0x3ed0] = 0x5b55, [0x3ed1] = 0x531d, [0x3ed2] = 0x7838,
  [0x3ed3] = 0x6742, [0x3ed4] = 0x683d, [0x3ed5] = 0x54c9, [0x3ed6] = 0x707e,
  [0x3ed7] = 0x5bb0, [0x3ed8] = 0x8f7d, [0x3ed9] = 0x518d, [0x3eda] = 0x5728,
  [0x3edb] = 0x54b1, [0x3edc] = 0x6512, [0x3edd] = 0x6682, [0x3ede] = 0x8d5e,
  [0x3edf] = 0x8d43, [0x3ee0] = 0x810f, [0x3ee1] = 0x846c, [0x3ee2] = 0x906d,
  [0x3ee3] = 0x7cdf, [0x3ee4] = 0x51ff, [0x3ee5] = 0x85fb, [0x3ee6] = 0x67a3,
  [0x3ee7] = 0x65e9, [0x3ee8] = 0x6fa1, [0x3ee9] = 0x86a4, [0x3eea] = 0x8e81,
  [0x3eeb] = 0x566a, [0x3eec] = 0x9020, [0x3eed] = 0x7682, [0x3eee] = 0x7076,
  [0x3eef] = 0x71e5, [0x3ef0] = 0x8d23, [0x3ef1] = 0x62e9, [0x3ef2] = 0x5219,
  [0x3ef3] = 0x6cfd, [0x3ef4] = 0x8d3c, [0x3ef5] = 0x600e, [0x3ef6] = 0x589e,
  [0x3ef7] = 0x618e, [0x3ef8] = 0x66fe, [0x3ef9] = 0x8d60, [0x3efa] = 0x624e,
  [0x3efb] = 0x55b3, [0x3efc] = 0x6e23, [0x3efd] = 0x672d, [0x3efe] = 0x8f67,
  [0x3f00] = 0x8a81, [0x3f01] = 0x8a82, [0x3f02] = 0x8a83, [0x3f03] = 0x8a84,
  [0x3f04] = 0x8a85, [0x3f05] = 0x8a86, [0x3f06] = 0x8a87, [0x3f07] = 0x8a88,
  [0x3f08] = 0x8a8b, [0x3f09] = 0x8a8c, [0x3f0a] = 0x8a8d, [0x3f0b] = 0x8a8e,
  [0x3f0c] = 0x8a8f, [0x3f0d] = 0x8a90, [0x3f0e] = 0x8a91, [0x3f0f] = 0x8a92,
  [0x3f10] = 0x8a94, [0x3f11] = 0x8a95, [0x3f12] = 0x8a96, [0x3f13] = 0x8a97,
  [0x3f14] = 0x8a98, [0x3f15] = 0x8a99, [0x3f16] = 0x8a9a, [0x3f17] = 0x8a9b,
  [0x3f18] = 0x8a9c, [0x3f19] = 0x8a9d, [0x3f1a] = 0x8a9e, [0x3f1b] = 0x8a9f,
  [0x3f1c] = 0x8aa0, [0x3f1d] = 0x8aa1, [0x3f1e] = 0x8aa2, [0x3f1f] = 0x8aa3,
  [0x3f20] = 0x8aa4, [0x3f21] = 0x8aa5, [0x3f22] = 0x8aa6, [0x3f23] = 0x8aa7,
  [0x3f24] = 0x8aa8, [0x3f25] = 0x8aa9, [0x3f26] = 0x8aaa, [0x3f27] = 0x8aab,
  [0x3f28] = 0x8aac, [0x3f29] = 0x8aad, [0x3f2a] = 0x8aae, [0x3f2b] = 0x8aaf,
  [0x3f2c] = 0x8ab0, [0x3f2d] = 0x8ab1, [0x3f2e] = 0x8ab2, [0x3f2f] = 0x8ab3,
  [0x3f30] = 0x8ab4, [0x3f31] = 0x8ab5, [0x3f32] = 0x8ab6, [0x3f33] = 0x8ab7,
  [0x3f34] = 0x8ab8, [0x3f35] = 0x8ab9, [0x3f36] = 0x8aba, [0x3f37] = 0x8abb,
  [0x3f38] = 0x8abc, [0x3f39] = 0x8abd, [0x3f3a] = 0x8abe, [0x3f3b] = 0x8abf,
  [0x3f3c] = 0x8ac0, [0x3f3d] = 0x8ac1, [0x3f3e] = 0x8ac2, [0x3f40] = 0x8ac3,
  [0x3f41] = 0x8ac4, [0x3f42] = 0x8ac5, [0x3f43] = 0x8ac6, [0x3f44] = 0x8ac7,
  [0x3f45] = 0x8ac8, [0x3f46] = 0x8ac9, [0x3f47] = 0x8aca, [0x3f48] = 0x8acb,
  [0x3f49] = 0x8acc, [0x3f4a] = 0x8acd, [0x3f4b] = 0x8ace, [0x3f4c] = 0x8acf,
  [0x3f4d] = 0x8ad0, [0x3f4e] = 0x8ad1, [0x3f4f] = 0x8ad2, [0x3f50] = 0x8ad3,
  [0x3f51] = 0x8ad4, [0x3f52] = 0x8ad5, [0x3f53] = 0x8ad6, [0x3f54] = 0x8ad7,
  [0x3f55] = 0x8ad8, [0x3f56] = 0x8ad9, [0x3f57] = 0x8ada, [0x3f58] = 0x8adb,
  [0x3f59] = 0x8adc, [0x3f5a] = 0x8add, [0x3f5b] = 0x8ade, [0x3f5c] = 0x8adf,
  [0x3f5d] = 0x8ae0, [0x3f5e] = 0x8ae1, [0x3f5f] = 0x8ae2, [0x3f60] = 0x8ae3,
  [0x3f61] = 0x94e1, [0x3f62] = 0x95f8, [0x3f63] = 0x7728, [0x3f64] = 0x6805,
  [0x3f65] = 0x69a8, [0x3f66] = 0x548b, [0x3f67] = 0x4e4d, [0x3f68] = 0x70b8,
  [0x3f69] = 0x8bc8, [0x3f6a] = 0x6458, [0x3f6b] = 0x658b, [0x3f6c] = 0x5b85,
  [0x3f6d] = 0x7a84, [0x3f6e] = 0x503a, [0x3f6f] = 0x5be8, [0x3f70] = 0x77bb,
  [0x3f71] = 0x6be1, [0x3f72] = 0x8a79, [0x3f73] = 0x7c98, [0x3f74] = 0x6cbe,
  [0x3f75] = 0x76cf, [0x3f76] = 0x65a9, [0x3f77] = 0x8f97, [0x3f78] = 0x5d2d,
  [0x3f79] = 0x5c55, [0x3f7a] = 0x8638, [0x3f7b] = 0x6808, [0x3f7c] = 0x5360,
  [0x3f7d] = 0x6218, [0x3f7e] = 0x7ad9, [0x3f7f] = 0x6e5b, [0x3f80] = 0x7efd,
  [0x3f81] = 0x6a1f, [0x3f82] = 0x7ae0, [0x3f83] = 0x5f70, [0x3f84] = 0x6f33,
  [0x3f85] = 0x5f20, [0x3f86] = 0x638c, [0x3f87] = 0x6da8, [0x3f88] = 0x6756,
  [0x3f89] = 0x4e08, [0x3f8a] = 0x5e10, [0x3f8b] = 0x8d26, [0x3f8c] = 0x4ed7,
  [0x3f8d] = 0x80c0, [0x3f8e] = 0x7634, [0x3f8f] = 0x969c, [0x3f90] = 0x62db,
  [0x3f91] = 0x662d, [0x3f92] = 0x627e, [0x3f93] = 0x6cbc, [0x3f94] = 0x8d75,
  [0x3f95] = 0x7167, [0x3f96] = 0x7f69, [0x3f97] = 0x5146, [0x3f98] = 0x8087,
  [0x3f99] = 0x53ec, [0x3f9a] = 0x906e, [0x3f9b] = 0x6298, [0x3f9c] = 0x54f2,
  [0x3f9d] = 0x86f0, [0x3f9e] = 0x8f99, [0x3f9f] = 0x8005, [0x3fa0] = 0x9517,
  [0x3fa1] = 0x8517, [0x3fa2] = 0x8fd9, [0x3fa3] = 0x6d59, [0x3fa4] = 0x73cd,
  [0x3fa5] = 0x659f, [0x3fa6] = 0x771f, [0x3fa7] = 0x7504, [0x3fa8] = 0x7827,
  [0x3fa9] = 0x81fb, [0x3faa] = 0x8d1e, [0x3fab] = 0x9488, [0x3fac] = 0x4fa6,
  [0x3fad] = 0x6795, [0x3fae] = 0x75b9, [0x3faf] = 0x8bca, [0x3fb0] = 0x9707,
  [0x3fb1] = 0x632f, [0x3fb2] = 0x9547, [0x3fb3] = 0x9635, [0x3fb4] = 0x84b8,
  [0x3fb5] = 0x6323, [0x3fb6] = 0x7741, [0x3fb7] = 0x5f81, [0x3fb8] = 0x72f0,
  [0x3fb9] = 0x4e89, [0x3fba] = 0x6014, [0x3fbb] = 0x6574, [0x3fbc] = 0x62ef,
  [0x3fbd] = 0x6b63, [0x3fbe] = 0x653f, [0x3fc0] = 0x8ae4, [0x3fc1] = 0x8ae5,
  [0x3fc2] = 0x8ae6, [0x3fc3] = 0x8ae7, [0x3fc4] = 0x8ae8, [0x3fc5] = 0x8ae9,
  [0x3fc6] = 0x8aea, [0x3fc7] = 0x8aeb, [0x3fc8] = 0x8aec, [0x3fc9] = 0x8aed,
  [0x3fca] = 0x8aee, [0x3fcb] = 0x8aef, [0x3fcc] = 0x8af0, [0x3fcd] = 0x8af1,
  [0x3fce] = 0x8af2, [0x3fcf] = 0x8af3, [0x3fd0] = 0x8af4, [0x3fd1] = 0x8af5,
  [0x3fd2] = 0x8af6, [0x3fd3] = 0x8af7, [0x3fd4] = 0x8af8, [0x3fd5] = 0x8af9,
  [0x3fd6] = 0x8afa, [0x3fd7] = 0x8afb, [0x3fd8] = 0x8afc, [0x3fd9] = 0x8afd,
  [0x3fda] = 0x8afe, [0x3fdb] = 0x8aff, [0x3fdc] = 0x8b00, [0x3fdd] = 0x8b01,
  [0x3fde] = 0x8b02, [0x3fdf] = 0x8b03, [0x3fe0] = 0x8b04, [0x3fe1] = 0x8b05,
  [0x3fe2] = 0x8b06, [0x3fe3] = 0x8b08, [0x3fe4] = 0x8b09, [0x3fe5] = 0x8b0a,
  [0x3fe6] = 0x8b0b, [0x3fe7] = 0x8b0c, [0x3fe8] = 0x8b0d, [0x3fe9] = 0x8b0e,
  [0x3fea] = 0x8b0f, [0x3feb] = 0x8b10, [0x3fec] = 0x8b11, [0x3fed] = 0x8b12,
  [0x3fee] = 0x8b13, [0x3fef] = 0x8b14, [0x3ff0] = 0x8b15, [0x3ff1] = 0x8b16,
  [0x3ff2] = 0x8b17, [0x3ff3] = 0x8b18, [0x3ff4] = 0x8b19, [0x3ff5] = 0x8b1a,
  [0x3ff6] = 0x8b1b, [0x3ff7] = 0x8b1c, [0x3ff8] = 0x8b1d, [0x3ff9] = 0x8b1e,
  [0x3ffa] = 0x8b1f, [0x3ffb] = 0x8b20, [0x3ffc] = 0x8b21, [0x3ffd] = 0x8b22,
  [0x3ffe] = 0x8b23, [0x4000] = 0x8b24, [0x4001] = 0x8b25, [0x4002] = 0x8b27,
  [0x4003] = 0x8b28, [0x4004] = 0x8b29, [0x4005] = 0x8b2a, [0x4006] = 0x8b2b,
  [0x4007] = 0x8b2c, [0x4008] = 0x8b2d, [0x4009] = 0x8b2e, [0x400a] = 0x8b2f,
  [0x400b] = 0x8b30, [0x400c] = 0x8b31, [0x400d] = 0x8b32, [0x400e] = 0x8b33,
  [0x400f] = 0x8b34, [0x4010] = 0x8b35, [0x4011] = 0x8b36, [0x4012] = 0x8b37,
  [0x4013] = 0x8b38, [0x4014] = 0x8b39, [0x4015] = 0x8b3a, [0x4016] = 0x8b3b,
  [0x4017] = 0x8b3c, [0x4018] = 0x8b3d, [0x4019] = 0x8b3e, [0x401a] = 0x8b3f,
  [0x401b] = 0x8b40, [0x401c] = 0x8b41, [0x401d] = 0x8b42, [0x401e] = 0x8b43,
  [0x401f] = 0x8b44, [0x4020] = 0x8b45, [0x4021] = 0x5e27, [0x4022] = 0x75c7,
  [0x4023] = 0x90d1, [0x4024] = 0x8bc1, [0x4025] = 0x829d, [0x4026] = 0x679d,
  [0x4027] = 0x652f, [0x4028] = 0x5431, [0x4029] = 0x8718, [0x402a] = 0x77e5,
  [0x402b] = 0x80a2, [0x402c] = 0x8102, [0x402d] = 0x6c41, [0x402e] = 0x4e4b,
  [0x402f] = 0x7ec7, [0x4030] = 0x804c, [0x4031] = 0x76f4, [0x4032] = 0x690d,
  [0x4033] = 0x6b96, [0x4034] = 0x6267, [0x4035] = 0x503c, [0x4036] = 0x4f84,
  [0x4037] = 0x5740, [0x4038] = 0x6307, [0x4039] = 0x6b62, [0x403a] = 0x8dbe,
  [0x403b] = 0x53ea, [0x403c] = 0x65e8, [0x403d] = 0x7eb8, [0x403e] = 0x5fd7,
  [0x403f] = 0x631a, [0x4040] = 0x63b7, [0x4041] = 0x81f3, [0x4042] = 0x81f4,
  [0x4043] = 0x7f6e, [0x4044] = 0x5e1c, [0x4045] = 0x5cd9, [0x4046] = 0x5236,
  [0x4047] = 0x667a, [0x4048] = 0x79e9, [0x4049] = 0x7a1a, [0x404a] = 0x8d28,
  [0x404b] = 0x7099, [0x404c] = 0x75d4, [0x404d] = 0x6ede, [0x404e] = 0x6cbb,
  [0x404f] = 0x7a92, [0x4050] = 0x4e2d, [0x4051] = 0x76c5, [0x4052] = 0x5fe0,
  [0x4053] = 0x949f, [0x4054] = 0x8877, [0x4055] = 0x7ec8, [0x4056] = 0x79cd,
  [0x4057] = 0x80bf, [0x4058] = 0x91cd, [0x4059] = 0x4ef2, [0x405a] = 0x4f17,
  [0x405b] = 0x821f, [0x405c] = 0x5468, [0x405d] = 0x5dde, [0x405e] = 0x6d32,
  [0x405f] = 0x8bcc, [0x4060] = 0x7ca5, [0x4061] = 0x8f74, [0x4062] = 0x8098,
  [0x4063] = 0x5e1a, [0x4064] = 0x5492, [0x4065] = 0x76b1, [0x4066] = 0x5b99,
  [0x4067] = 0x663c, [0x4068] = 0x9aa4, [0x4069] = 0x73e0, [0x406a] = 0x682a,
  [0x406b] = 0x86db, [0x406c] = 0x6731, [0x406d] = 0x732a, [0x406e] = 0x8bf8,
  [0x406f] = 0x8bdb, [0x4070] = 0x9010, [0x4071] = 0x7af9, [0x4072] = 0x70db,
  [0x4073] = 0x716e, [0x4074] = 0x62c4, [0x4075] = 0x77a9, [0x4076] = 0x5631,
  [0x4077] = 0x4e3b, [0x4078] = 0x8457, [0x4079] = 0x67f1, [0x407a] = 0x52a9,
  [0x407b] = 0x86c0, [0x407c] = 0x8d2e, [0x407d] = 0x94f8, [0x407e] = 0x7b51,
  [0x4080] = 0x8b46, [0x4081] = 0x8b47, [0x4082] = 0x8b48, [0x4083] = 0x8b49,
  [0x4084] = 0x8b4a, [0x4085] = 0x8b4b, [0x4086] = 0x8b4c, [0x4087] = 0x8b4d,
  [0x4088] = 0x8b4e, [0x4089] = 0x8b4f, [0x408a] = 0x8b50, [0x408b] = 0x8b51,
  [0x408c] = 0x8b52, [0x408d] = 0x8b53, [0x408e] = 0x8b54, [0x408f] = 0x8b55,
  [0x4090] = 0x8b56, [0x4091] = 0x8b57, [0x4092] = 0x8b58, [0x4093] = 0x8b59,
  [0x4094] = 0x8b5a, [0x4095] = 0x8b5b, [0x4096] = 0x8b5c, [0x4097] = 0x8b5d,
  [0x4098] = 0x8b5e, [0x4099] = 0x8b5f, [0x409a] = 0x8b60, [0x409b] = 0x8b61,
  [0x409c] = 0x8b62, [0x409d] = 0x8b63, [0x409e] = 0x8b64, [0x409f] = 0x8b65,
  [0x40a0] = 0x8b67, [0x40a1] = 0x8b68, [0x40a2] = 0x8b69, [0x40a3] = 0x8b6a,
  [0x40a4] = 0x8b6b, [0x40a5] = 0x8b6d, [0x40a6] = 0x8b6e, [0x40a7] = 0x8b6f,
  [0x40a8] = 0x8b70, [0x40a9] = 0x8b71, [0x40aa] = 0x8b72, [0x40ab] = 0x8b73,
  [0x40ac] = 0x8b74, [0x40ad] = 0x8b75, [0x40ae] = 0x8b76, [0x40af] = 0x8b77,
  [0x40b0] = 0x8b78, [0x40b1] = 0x8b79, [0x40b2] = 0x8b7a, [0x40b3] = 0x8b7b,
  [0x40b4] = 0x8b7c, [0x40b5] = 0x8b7d, [0x40b6] = 0x8b7e, [0x40b7] = 0x8b7f,
  [0x40b8] = 0x8b80, [0x40b9] = 0x8b81, [0x40ba] = 0x8b82, [0x40bb] = 0x8b83,
  [0x40bc] = 0x8b84, [0x40bd] = 0x8b85, [0x40be] = 0x8b86, [0x40c0] = 0x8b87,
  [0x40c1] = 0x8b88, [0x40c2] = 0x8b89, [0x40c3] = 0x8b8a, [0x40c4] = 0x8b8b,
  [0x40c5] = 0x8b8c, [0x40c6] = 0x8b8d, [0x40c7] = 0x8b8e, [0x40c8] = 0x8b8f,
  [0x40c9] = 0x8b90, [0x40ca] = 0x8b91, [0x40cb] = 0x8b92, [0x40cc] = 0x8b93,
  [0x40cd] = 0x8b94, [0x40ce] = 0x8b95, [0x40cf] = 0x8b96, [0x40d0] = 0x8b97,
  [0x40d1] = 0x8b98, [0x40d2] = 0x8b99, [0x40d3] = 0x8b9a, [0x40d4] = 0x8b9b,
  [0x40d5] = 0x8b9c, [0x40d6] = 0x8b9d, [0x40d7] = 0x8b9e, [0x40d8] = 0x8b9f,
  [0x40d9] = 0x8bac, [0x40da] = 0x8bb1, [0x40db] = 0x8bbb, [0x40dc] = 0x8bc7,
  [0x40dd] = 0x8bd0, [0x40de] = 0x8bea, [0x40df] = 0x8c09, [0x40e0] = 0x8c1e,
  [0x40e1] = 0x4f4f, [0x40e2] = 0x6ce8, [0x40e3] = 0x795d, [0x40e4] = 0x9a7b,
  [0x40e5] = 0x6293, [0x40e6] = 0x722a, [0x40e7] = 0x62fd, [0x40e8] = 0x4e13,
  [0x40e9] = 0x7816, [0x40ea] = 0x8f6c, [0x40eb] = 0x64b0, [0x40ec] = 0x8d5a,
  [0x40ed] = 0x7bc6, [0x40ee] = 0x6869, [0x40ef] = 0x5e84, [0x40f0] = 0x88c5,
  [0x40f1] = 0x5986, [0x40f2] = 0x649e, [0x40f3] = 0x58ee, [0x40f4] = 0x72b6,
  [0x40f5] = 0x690e, [0x40f6] = 0x9525, [0x40f7] = 0x8ffd, [0x40f8] = 0x8d58,
  [0x40f9] = 0x5760, [0x40fa] = 0x7f00, [0x40fb] = 0x8c06, [0x40fc] = 0x51c6,
  [0x40fd] = 0x6349, [0x40fe] = 0x62d9, [0x40ff] = 0x5353, [0x4100] = 0x684c,
  [0x4101] = 0x7422, [0x4102] = 0x8301, [0x4103] = 0x914c, [0x4104] = 0x5544,
  [0x4105] = 0x7740, [0x4106] = 0x707c, [0x4107] = 0x6d4a, [0x4108] = 0x5179,
  [0x4109] = 0x54a8, [0x410a] = 0x8d44, [0x410b] = 0x59ff, [0x410c] = 0x6ecb,
  [0x410d] = 0x6dc4, [0x410e] = 0x5b5c, [0x410f] = 0x7d2b, [0x4110] = 0x4ed4,
  [0x4111] = 0x7c7d, [0x4112] = 0x6ed3, [0x4113] = 0x5b50, [0x4114] = 0x81ea,
  [0x4115] = 0x6e0d, [0x4116] = 0x5b57, [0x4117] = 0x9b03, [0x4118] = 0x68d5,
  [0x4119] = 0x8e2a, [0x411a] = 0x5b97, [0x411b] = 0x7efc, [0x411c] = 0x603b,
  [0x411d] = 0x7eb5, [0x411e] = 0x90b9, [0x411f] = 0x8d70, [0x4120] = 0x594f,
  [0x4121] = 0x63cd, [0x4122] = 0x79df, [0x4123] = 0x8db3, [0x4124] = 0x5352,
  [0x4125] = 0x65cf, [0x4126] = 0x7956, [0x4127] = 0x8bc5, [0x4128] = 0x963b,
  [0x4129] = 0x7ec4, [0x412a] = 0x94bb, [0x412b] = 0x7e82, [0x412c] = 0x5634,
  [0x412d] = 0x9189, [0x412e] = 0x6700, [0x412f] = 0x7f6a, [0x4130] = 0x5c0a,
  [0x4131] = 0x9075, [0x4132] = 0x6628, [0x4133] = 0x5de6, [0x4134] = 0x4f50,
  [0x4135] = 0x67de, [0x4136] = 0x505a, [0x4137] = 0x4f5c, [0x4138] = 0x5750,
  [0x4139] = 0x5ea7, [0x413a] = 0xe810, [0x413b] = 0xe811, [0x413c] = 0xe812,
  [0x413d] = 0xe813, [0x413e] = 0xe814, [0x4140] = 0x8c38, [0x4141] = 0x8c39,
  [0x4142] = 0x8c3a, [0x4143] = 0x8c3b, [0x4144] = 0x8c3c, [0x4145] = 0x8c3d,
  [0x4146] = 0x8c3e, [0x4147] = 0x8c3f, [0x4148] = 0x8c40, [0x4149] = 0x8c42,
  [0x414a] = 0x8c43, [0x414b] = 0x8c44, [0x414c] = 0x8c45, [0x414d] = 0x8c48,
  [0x414e] = 0x8c4a, [0x414f] = 0x8c4b, [0x4150] = 0x8c4d, [0x4151] = 0x8c4e,
  [0x4152] = 0x8c4f, [0x4153] = 0x8c50, [0x4154] = 0x8c51, [0x4155] = 0x8c52,
  [0x4156] = 0x8c53, [0x4157] = 0x8c54, [0x4158] = 0x8c56, [0x4159] = 0x8c57,
  [0x415a] = 0x8c58, [0x415b] = 0x8c59, [0x415c] = 0x8c5b, [0x415d] = 0x8c5c,
  [0x415e] = 0x8c5d, [0x415f] = 0x8c5e, [0x4160] = 0x8c5f, [0x4161] = 0x8c60,
  [0x4162] = 0x8c63, [0x4163] = 0x8c64, [0x4164] = 0x8c65, [0x4165] = 0x8c66,
  [0x4166] = 0x8c67, [0x4167] = 0x8c68, [0x4168] = 0x8c69, [0x4169] = 0x8c6c,
  [0x416a] = 0x8c6d, [0x416b] = 0x8c6e, [0x416c] = 0x8c6f, [0x416d] = 0x8c70,
  [0x416e] = 0x8c71, [0x416f] = 0x8c72, [0x4170] = 0x8c74, [0x4171] = 0x8c75,
  [0x4172] = 0x8c76, [0x4173] = 0x8c77, [0x4174] = 0x8c7b, [0x4175] = 0x8c7c,
  [0x4176] = 0x8c7d, [0x4177] = 0x8c7e, [0x4178] = 0x8c7f, [0x4179] = 0x8c80,
  [0x417a] = 0x8c81, [0x417b] = 0x8c83, [0x417c] = 0x8c84, [0x417d] = 0x8c86,
  [0x417e] = 0x8c87, [0x4180] = 0x8c88, [0x4181] = 0x8c8b, [0x4182] = 0x8c8d,
  [0x4183] = 0x8c8e, [0x4184] = 0x8c8f, [0x4185] = 0x8c90, [0x4186] = 0x8c91,
  [0x4187] = 0x8c92, [0x4188] = 0x8c93, [0x4189] = 0x8c95, [0x418a] = 0x8c96,
  [0x418b] = 0x8c97, [0x418c] = 0x8c99, [0x418d] = 0x8c9a, [0x418e] = 0x8c9b,
  [0x418f] = 0x8c9c, [0x4190] = 0x8c9d, [0x4191] = 0x8c9e, [0x4192] = 0x8c9f,
  [0x4193] = 0x8ca0, [0x4194] = 0x8ca1, [0x4195] = 0x8ca2, [0x4196] = 0x8ca3,
  [0x4197] = 0x8ca4, [0x4198] = 0x8ca5, [0x4199] = 0x8ca6, [0x419a] = 0x8ca7,
  [0x419b] = 0x8ca8, [0x419c] = 0x8ca9, [0x419d] = 0x8caa, [0x419e] = 0x8cab,
  [0x419f] = 0x8cac, [0x41a0] = 0x8cad, [0x41a1] = 0x4e8d, [0x41a2] = 0x4e0c,
  [0x41a3] = 0x5140, [0x41a4] = 0x4e10, [0x41a5] = 0x5eff, [0x41a6] = 0x5345,
  [0x41a7] = 0x4e15, [0x41a8] = 0x4e98, [0x41a9] = 0x4e1e, [0x41aa] = 0x9b32,
  [0x41ab] = 0x5b6c, [0x41ac] = 0x5669, [0x41ad] = 0x4e28, [0x41ae] = 0x79ba,
  [0x41af] = 0x4e3f, [0x41b0] = 0x5315, [0x41b1] = 0x4e47, [0x41b2] = 0x592d,
  [0x41b3] = 0x723b, [0x41b4] = 0x536e, [0x41b5] = 0x6c10, [0x41b6] = 0x56df,
  [0x41b7] = 0x80e4, [0x41b8] = 0x9997, [0x41b9] = 0x6bd3, [0x41ba] = 0x777e,
  [0x41bb] = 0x9f17, [0x41bc] = 0x4e36, [0x41bd] = 0x4e9f, [0x41be] = 0x9f10,
  [0x41bf] = 0x4e5c, [0x41c0] = 0x4e69, [0x41c1] = 0x4e93, [0x41c2] = 0x8288,
  [0x41c3] = 0x5b5b, [0x41c4] = 0x556c, [0x41c5] = 0x560f, [0x41c6] = 0x4ec4,
  [0x41c7] = 0x538d, [0x41c8] = 0x539d, [0x41c9] = 0x53a3, [0x41ca] = 0x53a5,
  [0x41cb] = 0x53ae, [0x41cc] = 0x9765, [0x41cd] = 0x8d5d, [0x41ce] = 0x531a,
  [0x41cf] = 0x53f5, [0x41d0] = 0x5326, [0x41d1] = 0x532e, [0x41d2] = 0x533e,
  [0x41d3] = 0x8d5c, [0x41d4] = 0x5366, [0x41d5] = 0x5363, [0x41d6] = 0x5202,
  [0x41d7] = 0x5208, [0x41d8] = 0x520e, [0x41d9] = 0x522d, [0x41da] = 0x5233,
  [0x41db] = 0x523f, [0x41dc] = 0x5240, [0x41dd] = 0x524c, [0x41de] = 0x525e,
  [0x41df] = 0x5261, [0x41e0] = 0x525c, [0x41e1] = 0x84af, [0x41e2] = 0x527d,
  [0x41e3] = 0x5282, [0x41e4] = 0x5281, [0x41e5] = 0x5290, [0x41e6] = 0x5293,
  [0x41e7] = 0x5182, [0x41e8] = 0x7f54, [0x41e9] = 0x4ebb, [0x41ea] = 0x4ec3,
  [0x41eb] = 0x4ec9, [0x41ec] = 0x4ec2, [0x41ed] = 0x4ee8, [0x41ee] = 0x4ee1,
  [0x41ef] = 0x4eeb, [0x41f0] = 0x4ede, [0x41f1] = 0x4f1b, [0x41f2] = 0x4ef3,
  [0x41f3] = 0x4f22, [0x41f4] = 0x4f64, [0x41f5] = 0x4ef5, [0x41f6] = 0x4f25,
  [0x41f7] = 0x4f27, [0x41f8] = 0x4f09, [0x41f9] = 0x4f2b, [0x41fa] = 0x4f5e,
  [0x41fb] = 0x4f67, [0x41fc] = 0x6538, [0x41fd] = 0x4f5a, [0x41fe] = 0x4f5d,
  [0x4200] = 0x8cae, [0x4201] = 0x8caf, [0x4202] = 0x8cb0, [0x4203] = 0x8cb1,
  [0x4204] = 0x8cb2, [0x4205] = 0x8cb3, [0x4206] = 0x8cb4, [0x4207] = 0x8cb5,
  [0x4208] = 0x8cb6, [0x4209] = 0x8cb7, [0x420a] = 0x8cb8, [0x420b] = 0x8cb9,
  [0x420c] = 0x8cba, [0x420d] = 0x8cbb, [0x420e] = 0x8cbc, [0x420f] = 0x8cbd,
  [0x4210] = 0x8cbe, [0x4211] = 0x8cbf, [0x4212] = 0x8cc0, [0x4213] = 0x8cc1,
  [0x4214] = 0x8cc2, [0x4215] = 0x8cc3, [0x4216] = 0x8cc4, [0x4217] = 0x8cc5,
  [0x4218] = 0x8cc6, [0x4219] = 0x8cc7, [0x421a] = 0x8cc8, [0x421b] = 0x8cc9,
  [0x421c] = 0x8cca, [0x421d] = 0x8ccb, [0x421e] = 0x8ccc, [0x421f] = 0x8ccd,
  [0x4220] = 0x8cce, [0x4221] = 0x8ccf, [0x4222] = 0x8cd0, [0x4223] = 0x8cd1,
  [0x4224] = 0x8cd2, [0x4225] = 0x8cd3, [0x4226] = 0x8cd4, [0x4227] = 0x8cd5,
  [0x4228] = 0x8cd6, [0x4229] = 0x8cd7, [0x422a] = 0x8cd8, [0x422b] = 0x8cd9,
  [0x422c] = 0x8cda, [0x422d] = 0x8cdb, [0x422e] = 0x8cdc, [0x422f] = 0x8cdd,
  [0x4230] = 0x8cde, [0x4231] = 0x8cdf, [0x4232] = 0x8ce0, [0x4233] = 0x8ce1,
  [0x4234] = 0x8ce2, [0x4235] = 0x8ce3, [0x4236] = 0x8ce4, [0x4237] = 0x8ce5,
  [0x4238] = 0x8ce6, [0x4239] = 0x8ce7, [0x423a] = 0x8ce8, [0x423b] = 0x8ce9,
  [0x423c] = 0x8cea, [0x423d] = 0x8ceb, [0x423e] = 0x8cec, [0x4240] = 0x8ced,
  [0x4241] = 0x8cee, [0x4242] = 0x8cef, [0x4243] = 0x8cf0, [0x4244] = 0x8cf1,
  [0x4245] = 0x8cf2, [0x4246] = 0x8cf3, [0x4247] = 0x8cf4, [0x4248] = 0x8cf5,
  [0x4249] = 0x8cf6, [0x424a] = 0x8cf7, [0x424b] = 0x8cf8, [0x424c] = 0x8cf9,
  [0x424d] = 0x8cfa, [0x424e] = 0x8cfb, [0x424f] = 0x8cfc, [0x4250] = 0x8cfd,
  [0x4251] = 0x8cfe, [0x4252] = 0x8cff, [0x4253] = 0x8d00, [0x4254] = 0x8d01,
  [0x4255] = 0x8d02, [0x4256] = 0x8d03, [0x4257] = 0x8d04, [0x4258] = 0x8d05,
  [0x4259] = 0x8d06, [0x425a] = 0x8d07, [0x425b] = 0x8d08, [0x425c] = 0x8d09,
  [0x425d] = 0x8d0a, [0x425e] = 0x8d0b, [0x425f] = 0x8d0c, [0x4260] = 0x8d0d,
  [0x4261] = 0x4f5f, [0x4262] = 0x4f57, [0x4263] = 0x4f32, [0x4264] = 0x4f3d,
  [0x4265] = 0x4f76, [0x4266] = 0x4f74, [0x4267] = 0x4f91, [0x4268] = 0x4f89,
  [0x4269] = 0x4f83, [0x426a] = 0x4f8f, [0x426b] = 0x4f7e, [0x426c] = 0x4f7b,
  [0x426d] = 0x4faa, [0x426e] = 0x4f7c, [0x426f] = 0x4fac, [0x4270] = 0x4f94,
  [0x4271] = 0x4fe6, [0x4272] = 0x4fe8, [0x4273] = 0x4fea, [0x4274] = 0x4fc5,
  [0x4275] = 0x4fda, [0x4276] = 0x4fe3, [0x4277] = 0x4fdc, [0x4278] = 0x4fd1,
  [0x4279] = 0x4fdf, [0x427a] = 0x4ff8, [0x427b] = 0x5029, [0x427c] = 0x504c,
  [0x427d] = 0x4ff3, [0x427e] = 0x502c, [0x427f] = 0x500f, [0x4280] = 0x502e,
  [0x4281] = 0x502d, [0x4282] = 0x4ffe, [0x4283] = 0x501c, [0x4284] = 0x500c,
  [0x4285] = 0x5025, [0x4286] = 0x5028, [0x4287] = 0x507e, [0x4288] = 0x5043,
  [0x4289] = 0x5055, [0x428a] = 0x5048, [0x428b] = 0x504e, [0x428c] = 0x506c,
  [0x428d] = 0x507b, [0x428e] = 0x50a5, [0x428f] = 0x50a7, [0x4290] = 0x50a9,
  [0x4291] = 0x50ba, [0x4292] = 0x50d6, [0x4293] = 0x5106, [0x4294] = 0x50ed,
  [0x4295] = 0x50ec, [0x4296] = 0x50e6, [0x4297] = 0x50ee, [0x4298] = 0x5107,
  [0x4299] = 0x510b, [0x429a] = 0x4edd, [0x429b] = 0x6c3d, [0x429c] = 0x4f58,
  [0x429d] = 0x4f65, [0x429e] = 0x4fce, [0x429f] = 0x9fa0, [0x42a0] = 0x6c46,
  [0x42a1] = 0x7c74, [0x42a2] = 0x516e, [0x42a3] = 0x5dfd, [0x42a4] = 0x9ec9,
  [0x42a5] = 0x9998, [0x42a6] = 0x5181, [0x42a7] = 0x5914, [0x42a8] = 0x52f9,
  [0x42a9] = 0x530d, [0x42aa] = 0x8a07, [0x42ab] = 0x5310, [0x42ac] = 0x51eb,
  [0x42ad] = 0x5919, [0x42ae] = 0x5155, [0x42af] = 0x4ea0, [0x42b0] = 0x5156,
  [0x42b1] = 0x4eb3, [0x42b2] = 0x886e, [0x42b3] = 0x88a4, [0x42b4] = 0x4eb5,
  [0x42b5] = 0x8114, [0x42b6] = 0x88d2, [0x42b7] = 0x7980, [0x42b8] = 0x5b34,
  [0x42b9] = 0x8803, [0x42ba] = 0x7fb8, [0x42bb] = 0x51ab, [0x42bc] = 0x51b1,
  [0x42bd] = 0x51bd, [0x42be] = 0x51bc, [0x42c0] = 0x8d0e, [0x42c1] = 0x8d0f,
  [0x42c2] = 0x8d10, [0x42c3] = 0x8d11, [0x42c4] = 0x8d12, [0x42c5] = 0x8d13,
  [0x42c6] = 0x8d14, [0x42c7] = 0x8d15, [0x42c8] = 0x8d16, [0x42c9] = 0x8d17,
  [0x42ca] = 0x8d18, [0x42cb] = 0x8d19, [0x42cc] = 0x8d1a, [0x42cd] = 0x8d1b,
  [0x42ce] = 0x8d1c, [0x42cf] = 0x8d20, [0x42d0] = 0x8d51, [0x42d1] = 0x8d52,
  [0x42d2] = 0x8d57, [0x42d3] = 0x8d5f, [0x42d4] = 0x8d65, [0x42d5] = 0x8d68,
  [0x42d6] = 0x8d69, [0x42d7] = 0x8d6a, [0x42d8] = 0x8d6c, [0x42d9] = 0x8d6e,
  [0x42da] = 0x8d6f, [0x42db] = 0x8d71, [0x42dc] = 0x8d72, [0x42dd] = 0x8d78,
  [0x42de] = 0x8d79, [0x42df] = 0x8d7a, [0x42e0] = 0x8d7b, [0x42e1] = 0x8d7c,
  [0x42e2] = 0x8d7d, [0x42e3] = 0x8d7e, [0x42e4] = 0x8d7f, [0x42e5] = 0x8d80,
  [0x42e6] = 0x8d82, [0x42e7] = 0x8d83, [0x42e8] = 0x8d86, [0x42e9] = 0x8d87,
  [0x42ea] = 0x8d88, [0x42eb] = 0x8d89, [0x42ec] = 0x8d8c, [0x42ed] = 0x8d8d,
  [0x42ee] = 0x8d8e, [0x42ef] = 0x8d8f, [0x42f0] = 0x8d90, [0x42f1] = 0x8d92,
  [0x42f2] = 0x8d93, [0x42f3] = 0x8d95, [0x42f4] = 0x8d96, [0x42f5] = 0x8d97,
  [0x42f6] = 0x8d98, [0x42f7] = 0x8d99, [0x42f8] = 0x8d9a, [0x42f9] = 0x8d9b,
  [0x42fa] = 0x8d9c, [0x42fb] = 0x8d9d, [0x42fc] = 0x8d9e, [0x42fd] = 0x8da0,
  [0x42fe] = 0x8da1, [0x4300] = 0x8da2, [0x4301] = 0x8da4, [0x4302] = 0x8da5,
  [0x4303] = 0x8da6, [0x4304] = 0x8da7, [0x4305] = 0x8da8, [0x4306] = 0x8da9,
  [0x4307] = 0x8daa, [0x4308] = 0x8dab, [0x4309] = 0x8dac, [0x430a] = 0x8dad,
  [0x430b] = 0x8dae, [0x430c] = 0x8daf, [0x430d] = 0x8db0, [0x430e] = 0x8db2,
  [0x430f] = 0x8db6, [0x4310] = 0x8db7, [0x4311] = 0x8db9, [0x4312] = 0x8dbb,
  [0x4313] = 0x8dbd, [0x4314] = 0x8dc0, [0x4315] = 0x8dc1, [0x4316] = 0x8dc2,
  [0x4317] = 0x8dc5, [0x4318] = 0x8dc7, [0x4319] = 0x8dc8, [0x431a] = 0x8dc9,
  [0x431b] = 0x8dca, [0x431c] = 0x8dcd, [0x431d] = 0x8dd0, [0x431e] = 0x8dd2,
  [0x431f] = 0x8dd3, [0x4320] = 0x8dd4, [0x4321] = 0x51c7, [0x4322] = 0x5196,
  [0x4323] = 0x51a2, [0x4324] = 0x51a5, [0x4325] = 0x8ba0, [0x4326] = 0x8ba6,
  [0x4327] = 0x8ba7, [0x4328] = 0x8baa, [0x4329] = 0x8bb4, [0x432a] = 0x8bb5,
  [0x432b] = 0x8bb7, [0x432c] = 0x8bc2, [0x432d] = 0x8bc3, [0x432e] = 0x8bcb,
  [0x432f] = 0x8bcf, [0x4330] = 0x8bce, [0x4331] = 0x8bd2, [0x4332] = 0x8bd3,
  [0x4333] = 0x8bd4, [0x4334] = 0x8bd6, [0x4335] = 0x8bd8, [0x4336] = 0x8bd9,
  [0x4337] = 0x8bdc, [0x4338] = 0x8bdf, [0x4339] = 0x8be0, [0x433a] = 0x8be4,
  [0x433b] = 0x8be8, [0x433c] = 0x8be9, [0x433d] = 0x8bee, [0x433e] = 0x8bf0,
  [0x433f] = 0x8bf3, [0x4340] = 0x8bf6, [0x4341] = 0x8bf9, [0x4342] = 0x8bfc,
  [0x4343] = 0x8bff, [0x4344] = 0x8c00, [0x4345] = 0x8c02, [0x4346] = 0x8c04,
  [0x4347] = 0x8c07, [0x4348] = 0x8c0c, [0x4349] = 0x8c0f, [0x434a] = 0x8c11,
  [0x434b] = 0x8c12, [0x434c] = 0x8c14, [0x434d] = 0x8c15, [0x434e] = 0x8c16,
  [0x434f] = 0x8c19, [0x4350] = 0x8c1b, [0x4351] = 0x8c18, [0x4352] = 0x8c1d,
  [0x4353] = 0x8c1f, [0x4354] = 0x8c20, [0x4355] = 0x8c21, [0x4356] = 0x8c25,
  [0x4357] = 0x8c27, [0x4358] = 0x8c2a, [0x4359] = 0x8c2b, [0x435a] = 0x8c2e,
  [0x435b] = 0x8c2f, [0x435c] = 0x8c32, [0x435d] = 0x8c33, [0x435e] = 0x8c35,
  [0x435f] = 0x8c36, [0x4360] = 0x5369, [0x4361] = 0x537a, [0x4362] = 0x961d,
  [0x4363] = 0x9622, [0x4364] = 0x9621, [0x4365] = 0x9631, [0x4366] = 0x962a,
  [0x4367] = 0x963d, [0x4368] = 0x963c, [0x4369] = 0x9642, [0x436a] = 0x9649,
  [0x436b] = 0x9654, [0x436c] = 0x965f, [0x436d] = 0x9667, [0x436e] = 0x966c,
  [0x436f] = 0x9672, [0x4370] = 0x9674, [0x4371] = 0x9688, [0x4372] = 0x968d,
  [0x4373] = 0x9697, [0x4374] = 0x96b0, [0x4375] = 0x9097, [0x4376] = 0x909b,
  [0x4377] = 0x909d, [0x4378] = 0x9099, [0x4379] = 0x90ac, [0x437a] = 0x90a1,
  [0x437b] = 0x90b4, [0x437c] = 0x90b3, [0x437d] = 0x90b6, [0x437e] = 0x90ba,
  [0x4380] = 0x8dd5, [0x4381] = 0x8dd8, [0x4382] = 0x8dd9, [0x4383] = 0x8ddc,
  [0x4384] = 0x8de0, [0x4385] = 0x8de1, [0x4386] = 0x8de2, [0x4387] = 0x8de5,
  [0x4388] = 0x8de6, [0x4389] = 0x8de7, [0x438a] = 0x8de9, [0x438b] = 0x8ded,
  [0x438c] = 0x8dee, [0x438d] = 0x8df0, [0x438e] = 0x8df1, [0x438f] = 0x8df2,
  [0x4390] = 0x8df4, [0x4391] = 0x8df6, [0x4392] = 0x8dfc, [0x4393] = 0x8dfe,
  [0x4394] = 0x8dff, [0x4395] = 0x8e00, [0x4396] = 0x8e01, [0x4397] = 0x8e02,
  [0x4398] = 0x8e03, [0x4399] = 0x8e04, [0x439a] = 0x8e06, [0x439b] = 0x8e07,
  [0x439c] = 0x8e08, [0x439d] = 0x8e0b, [0x439e] = 0x8e0d, [0x439f] = 0x8e0e,
  [0x43a0] = 0x8e10, [0x43a1] = 0x8e11, [0x43a2] = 0x8e12, [0x43a3] = 0x8e13,
  [0x43a4] = 0x8e15, [0x43a5] = 0x8e16, [0x43a6] = 0x8e17, [0x43a7] = 0x8e18,
  [0x43a8] = 0x8e19, [0x43a9] = 0x8e1a, [0x43aa] = 0x8e1b, [0x43ab] = 0x8e1c,
  [0x43ac] = 0x8e20, [0x43ad] = 0x8e21, [0x43ae] = 0x8e24, [0x43af] = 0x8e25,
  [0x43b0] = 0x8e26, [0x43b1] = 0x8e27, [0x43b2] = 0x8e28, [0x43b3] = 0x8e2b,
  [0x43b4] = 0x8e2d, [0x43b5] = 0x8e30, [0x43b6] = 0x8e32, [0x43b7] = 0x8e33,
  [0x43b8] = 0x8e34, [0x43b9] = 0x8e36, [0x43ba] = 0x8e37, [0x43bb] = 0x8e38,
  [0x43bc] = 0x8e3b, [0x43bd] = 0x8e3c, [0x43be] = 0x8e3e, [0x43c0] = 0x8e3f,
  [0x43c1] = 0x8e43, [0x43c2] = 0x8e45, [0x43c3] = 0x8e46, [0x43c4] = 0x8e4c,
  [0x43c5] = 0x8e4d, [0x43c6] = 0x8e4e, [0x43c7] = 0x8e4f, [0x43c8] = 0x8e50,
  [0x43c9] = 0x8e53, [0x43ca] = 0x8e54, [0x43cb] = 0x8e55, [0x43cc] = 0x8e56,
  [0x43cd] = 0x8e57, [0x43ce] = 0x8e58, [0x43cf] = 0x8e5a, [0x43d0] = 0x8e5b,
  [0x43d1] = 0x8e5c, [0x43d2] = 0x8e5d, [0x43d3] = 0x8e5e, [0x43d4] = 0x8e5f,
  [0x43d5] = 0x8e60, [0x43d6] = 0x8e61, [0x43d7] = 0x8e62, [0x43d8] = 0x8e63,
  [0x43d9] = 0x8e64, [0x43da] = 0x8e65, [0x43db] = 0x8e67, [0x43dc] = 0x8e68,
  [0x43dd] = 0x8e6a, [0x43de] = 0x8e6b, [0x43df] = 0x8e6e, [0x43e0] = 0x8e71,
  [0x43e1] = 0x90b8, [0x43e2] = 0x90b0, [0x43e3] = 0x90cf, [0x43e4] = 0x90c5,
  [0x43e5] = 0x90be, [0x43e6] = 0x90d0, [0x43e7] = 0x90c4, [0x43e8] = 0x90c7,
  [0x43e9] = 0x90d3, [0x43ea] = 0x90e6, [0x43eb] = 0x90e2, [0x43ec] = 0x90dc,
  [0x43ed] = 0x90d7, [0x43ee] = 0x90db, [0x43ef] = 0x90eb, [0x43f0] = 0x90ef,
  [0x43f1] = 0x90fe, [0x43f2] = 0x9104, [0x43f3] = 0x9122, [0x43f4] = 0x911e,
  [0x43f5] = 0x9123, [0x43f6] = 0x9131, [0x43f7] = 0x912f, [0x43f8] = 0x9139,
  [0x43f9] = 0x9143, [0x43fa] = 0x9146, [0x43fb] = 0x520d, [0x43fc] = 0x5942,
  [0x43fd] = 0x52a2, [0x43fe] = 0x52ac, [0x43ff] = 0x52ad, [0x4400] = 0x52be,
  [0x4401] = 0x54ff, [0x4402] = 0x52d0, [0x4403] = 0x52d6, [0x4404] = 0x52f0,
  [0x4405] = 0x53df, [0x4406] = 0x71ee, [0x4407] = 0x77cd, [0x4408] = 0x5ef4,
  [0x4409] = 0x51f5, [0x440a] = 0x51fc, [0x440b] = 0x9b2f, [0x440c] = 0x53b6,
  [0x440d] = 0x5f01, [0x440e] = 0x755a, [0x440f] = 0x5def, [0x4410] = 0x574c,
  [0x4411] = 0x57a9, [0x4412] = 0x57a1, [0x4413] = 0x587e, [0x4414] = 0x58bc,
  [0x4415] = 0x58c5, [0x4416] = 0x58d1, [0x4417] = 0x5729, [0x4418] = 0x572c,
  [0x4419] = 0x572a, [0x441a] = 0x5733, [0x441b] = 0x5739, [0x441c] = 0x572e,
  [0x441d] = 0x572f, [0x441e] = 0x575c, [0x441f] = 0x573b, [0x4420] = 0x5742,
  [0x4421] = 0x5769, [0x4422] = 0x5785, [0x4423] = 0x576b, [0x4424] = 0x5786,
  [0x4425] = 0x577c, [0x4426] = 0x577b, [0x4427] = 0x5768, [0x4428] = 0x576d,
  [0x4429] = 0x5776, [0x442a] = 0x5773, [0x442b] = 0x57ad, [0x442c] = 0x57a4,
  [0x442d] = 0x578c, [0x442e] = 0x57b2, [0x442f] = 0x57cf, [0x4430] = 0x57a7,
  [0x4431] = 0x57b4, [0x4432] = 0x5793, [0x4433] = 0x57a0, [0x4434] = 0x57d5,
  [0x4435] = 0x57d8, [0x4436] = 0x57da, [0x4437] = 0x57d9, [0x4438] = 0x57d2,
  [0x4439] = 0x57b8, [0x443a] = 0x57f4, [0x443b] = 0x57ef, [0x443c] = 0x57f8,
  [0x443d] = 0x57e4, [0x443e] = 0x57dd, [0x4440] = 0x8e73, [0x4441] = 0x8e75,
  [0x4442] = 0x8e77, [0x4443] = 0x8e78, [0x4444] = 0x8e79, [0x4445] = 0x8e7a,
  [0x4446] = 0x8e7b, [0x4447] = 0x8e7d, [0x4448] = 0x8e7e, [0x4449] = 0x8e80,
  [0x444a] = 0x8e82, [0x444b] = 0x8e83, [0x444c] = 0x8e84, [0x444d] = 0x8e86,
  [0x444e] = 0x8e88, [0x444f] = 0x8e89, [0x4450] = 0x8e8a, [0x4451] = 0x8e8b,
  [0x4452] = 0x8e8c, [0x4453] = 0x8e8d, [0x4454] = 0x8e8e, [0x4455] = 0x8e91,
  [0x4456] = 0x8e92, [0x4457] = 0x8e93, [0x4458] = 0x8e95, [0x4459] = 0x8e96,
  [0x445a] = 0x8e97, [0x445b] = 0x8e98, [0x445c] = 0x8e99, [0x445d] = 0x8e9a,
  [0x445e] = 0x8e9b, [0x445f] = 0x8e9d, [0x4460] = 0x8e9f, [0x4461] = 0x8ea0,
  [0x4462] = 0x8ea1, [0x4463] = 0x8ea2, [0x4464] = 0x8ea3, [0x4465] = 0x8ea4,
  [0x4466] = 0x8ea5, [0x4467] = 0x8ea6, [0x4468] = 0x8ea7, [0x4469] = 0x8ea8,
  [0x446a] = 0x8ea9, [0x446b] = 0x8eaa, [0x446c] = 0x8ead, [0x446d] = 0x8eae,
  [0x446e] = 0x8eb0, [0x446f] = 0x8eb1, [0x4470] = 0x8eb3, [0x4471] = 0x8eb4,
  [0x4472] = 0x8eb5, [0x4473] = 0x8eb6, [0x4474] = 0x8eb7, [0x4475] = 0x8eb8,
  [0x4476] = 0x8eb9, [0x4477] = 0x8ebb, [0x4478] = 0x8ebc, [0x4479] = 0x8ebd,
  [0x447a] = 0x8ebe, [0x447b] = 0x8ebf, [0x447c] = 0x8ec0, [0x447d] = 0x8ec1,
  [0x447e] = 0x8ec2, [0x4480] = 0x8ec3, [0x4481] = 0x8ec4, [0x4482] = 0x8ec5,
  [0x4483] = 0x8ec6, [0x4484] = 0x8ec7, [0x4485] = 0x8ec8, [0x4486] = 0x8ec9,
  [0x4487] = 0x8eca, [0x4488] = 0x8ecb, [0x4489] = 0x8ecc, [0x448a] = 0x8ecd,
  [0x448b] = 0x8ecf, [0x448c] = 0x8ed0, [0x448d] = 0x8ed1, [0x448e] = 0x8ed2,
  [0x448f] = 0x8ed3, [0x4490] = 0x8ed4, [0x4491] = 0x8ed5, [0x4492] = 0x8ed6,
  [0x4493] = 0x8ed7, [0x4494] = 0x8ed8, [0x4495] = 0x8ed9, [0x4496] = 0x8eda,
  [0x4497] = 0x8edb, [0x4498] = 0x8edc, [0x4499] = 0x8edd, [0x449a] = 0x8ede,
  [0x449b] = 0x8edf, [0x449c] = 0x8ee0, [0x449d] = 0x8ee1, [0x449e] = 0x8ee2,
  [0x449f] = 0x8ee3, [0x44a0] = 0x8ee4, [0x44a1] = 0x580b, [0x44a2] = 0x580d,
  [0x44a3] = 0x57fd, [0x44a4] = 0x57ed, [0x44a5] = 0x5800, [0x44a6] = 0x581e,
  [0x44a7] = 0x5819, [0x44a8] = 0x5844, [0x44a9] = 0x5820, [0x44aa] = 0x5865,
  [0x44ab] = 0x586c, [0x44ac] = 0x5881, [0x44ad] = 0x5889, [0x44ae] = 0x589a,
  [0x44af] = 0x5880, [0x44b0] = 0x99a8, [0x44b1] = 0x9f19, [0x44b2] = 0x61ff,
  [0x44b3] = 0x8279, [0x44b4] = 0x827d, [0x44b5] = 0x827f, [0x44b6] = 0x828f,
  [0x44b7] = 0x828a, [0x44b8] = 0x82a8, [0x44b9] = 0x8284, [0x44ba] = 0x828e,
  [0x44bb] = 0x8291, [0x44bc] = 0x8297, [0x44bd] = 0x8299, [0x44be] = 0x82ab,
  [0x44bf] = 0x82b8, [0x44c0] = 0x82be, [0x44c1] = 0x82b0, [0x44c2] = 0x82c8,
  [0x44c3] = 0x82ca, [0x44c4] = 0x82e3, [0x44c5] = 0x8298, [0x44c6] = 0x82b7,
  [0x44c7] = 0x82ae, [0x44c8] = 0x82cb, [0x44c9] = 0x82cc, [0x44ca] = 0x82c1,
  [0x44cb] = 0x82a9, [0x44cc] = 0x82b4, [0x44cd] = 0x82a1, [0x44ce] = 0x82aa,
  [0x44cf] = 0x829f, [0x44d0] = 0x82c4, [0x44d1] = 0x82ce, [0x44d2] = 0x82a4,
  [0x44d3] = 0x82e1, [0x44d4] = 0x8309, [0x44d5] = 0x82f7, [0x44d6] = 0x82e4,
  [0x44d7] = 0x830f, [0x44d8] = 0x8307, [0x44d9] = 0x82dc, [0x44da] = 0x82f4,
  [0x44db] = 0x82d2, [0x44dc] = 0x82d8, [0x44dd] = 0x830c, [0x44de] = 0x82fb,
  [0x44df] = 0x82d3, [0x44e0] = 0x8311, [0x44e1] = 0x831a, [0x44e2] = 0x8306,
  [0x44e3] = 0x8314, [0x44e4] = 0x8315, [0x44e5] = 0x82e0, [0x44e6] = 0x82d5,
  [0x44e7] = 0x831c, [0x44e8] = 0x8351, [0x44e9] = 0x835b, [0x44ea] = 0x835c,
  [0x44eb] = 0x8308, [0x44ec] = 0x8392, [0x44ed] = 0x833c, [0x44ee] = 0x8334,
  [0x44ef] = 0x8331, [0x44f0] = 0x839b, [0x44f1] = 0x835e, [0x44f2] = 0x832f,
  [0x44f3] = 0x834f, [0x44f4] = 0x8347, [0x44f5] = 0x8343, [0x44f6] = 0x835f,
  [0x44f7] = 0x8340, [0x44f8] = 0x8317, [0x44f9] = 0x8360, [0x44fa] = 0x832d,
  [0x44fb] = 0x833a, [0x44fc] = 0x8333, [0x44fd] = 0x8366, [0x44fe] = 0x8365,
  [0x4500] = 0x8ee5, [0x4501] = 0x8ee6, [0x4502] = 0x8ee7, [0x4503] = 0x8ee8,
  [0x4504] = 0x8ee9, [0x4505] = 0x8eea, [0x4506] = 0x8eeb, [0x4507] = 0x8eec,
  [0x4508] = 0x8eed, [0x4509] = 0x8eee, [0x450a] = 0x8eef, [0x450b] = 0x8ef0,
  [0x450c] = 0x8ef1, [0x450d] = 0x8ef2, [0x450e] = 0x8ef3, [0x450f] = 0x8ef4,
  [0x4510] = 0x8ef5, [0x4511] = 0x8ef6, [0x4512] = 0x8ef7, [0x4513] = 0x8ef8,
  [0x4514] = 0x8ef9, [0x4515] = 0x8efa, [0x4516] = 0x8efb, [0x4517] = 0x8efc,
  [0x4518] = 0x8efd, [0x4519] = 0x8efe, [0x451a] = 0x8eff, [0x451b] = 0x8f00,
  [0x451c] = 0x8f01, [0x451d] = 0x8f02, [0x451e] = 0x8f03, [0x451f] = 0x8f04,
  [0x4520] = 0x8f05, [0x4521] = 0x8f06, [0x4522] = 0x8f07, [0x4523] = 0x8f08,
  [0x4524] = 0x8f09, [0x4525] = 0x8f0a, [0x4526] = 0x8f0b, [0x4527] = 0x8f0c,
  [0x4528] = 0x8f0d, [0x4529] = 0x8f0e, [0x452a] = 0x8f0f, [0x452b] = 0x8f10,
  [0x452c] = 0x8f11, [0x452d] = 0x8f12, [0x452e] = 0x8f13, [0x452f] = 0x8f14,
  [0x4530] = 0x8f15, [0x4531] = 0x8f16, [0x4532] = 0x8f17, [0x4533] = 0x8f18,
  [0x4534] = 0x8f19, [0x4535] = 0x8f1a, [0x4536] = 0x8f1b, [0x4537] = 0x8f1c,
  [0x4538] = 0x8f1d, [0x4539] = 0x8f1e, [0x453a] = 0x8f1f, [0x453b] = 0x8f20,
  [0x453c] = 0x8f21, [0x453d] = 0x8f22, [0x453e] = 0x8f23, [0x4540] = 0x8f24,
  [0x4541] = 0x8f25, [0x4542] = 0x8f26, [0x4543] = 0x8f27, [0x4544] = 0x8f28,
  [0x4545] = 0x8f29, [0x4546] = 0x8f2a, [0x4547] = 0x8f2b, [0x4548] = 0x8f2c,
  [0x4549] = 0x8f2d, [0x454a] = 0x8f2e, [0x454b] = 0x8f2f, [0x454c] = 0x8f30,
  [0x454d] = 0x8f31, [0x454e] = 0x8f32, [0x454f] = 0x8f33, [0x4550] = 0x8f34,
  [0x4551] = 0x8f35, [0x4552] = 0x8f36, [0x4553] = 0x8f37, [0x4554] = 0x8f38,
  [0x4555] = 0x8f39, [0x4556] = 0x8f3a, [0x4557] = 0x8f3b, [0x4558] = 0x8f3c,
  [0x4559] = 0x8f3d, [0x455a] = 0x8f3e, [0x455b] = 0x8f3f, [0x455c] = 0x8f40,
  [0x455d] = 0x8f41, [0x455e] = 0x8f42, [0x455f] = 0x8f43, [0x4560] = 0x8f44,
  [0x4561] = 0x8368, [0x4562] = 0x831b, [0x4563] = 0x8369, [0x4564] = 0x836c,
  [0x4565] = 0x836a, [0x4566] = 0x836d, [0x4567] = 0x836e, [0x4568] = 0x83b0,
  [0x4569] = 0x8378, [0x456a] = 0x83b3, [0x456b] = 0x83b4, [0x456c] = 0x83a0,
  [0x456d] = 0x83aa, [0x456e] = 0x8393, [0x456f] = 0x839c, [0x4570] = 0x8385,
  [0x4571] = 0x837c, [0x4572] = 0x83b6, [0x4573] = 0x83a9, [0x4574] = 0x837d,
  [0x4575] = 0x83b8, [0x4576] = 0x837b, [0x4577] = 0x8398, [0x4578] = 0x839e,
  [0x4579] = 0x83a8, [0x457a] = 0x83ba, [0x457b] = 0x83bc, [0x457c] = 0x83c1,
  [0x457d] = 0x8401, [0x457e] = 0x83e5, [0x457f] = 0x83d8, [0x4580] = 0x5807,
  [0x4581] = 0x8418, [0x4582] = 0x840b, [0x4583] = 0x83dd, [0x4584] = 0x83fd,
  [0x4585] = 0x83d6, [0x4586] = 0x841c, [0x4587] = 0x8438, [0x4588] = 0x8411,
  [0x4589] = 0x8406, [0x458a] = 0x83d4, [0x458b] = 0x83df, [0x458c] = 0x840f,
  [0x458d] = 0x8403, [0x458e] = 0x83f8, [0x458f] = 0x83f9, [0x4590] = 0x83ea,
  [0x4591] = 0x83c5, [0x4592] = 0x83c0, [0x4593] = 0x8426, [0x4594] = 0x83f0,
  [0x4595] = 0x83e1, [0x4596] = 0x845c, [0x4597] = 0x8451, [0x4598] = 0x845a,
  [0x4599] = 0x8459, [0x459a] = 0x8473, [0x459b] = 0x8487, [0x459c] = 0x8488,
  [0x459d] = 0x847a, [0x459e] = 0x8489, [0x459f] = 0x8478, [0x45a0] = 0x843c,
  [0x45a1] = 0x8446, [0x45a2] = 0x8469, [0x45a3] = 0x8476, [0x45a4] = 0x848c,
  [0x45a5] = 0x848e, [0x45a6] = 0x8431, [0x45a7] = 0x846d, [0x45a8] = 0x84c1,
  [0x45a9] = 0x84cd, [0x45aa] = 0x84d0, [0x45ab] = 0x84e6, [0x45ac] = 0x84bd,
  [0x45ad] = 0x84d3, [0x45ae] = 0x84ca, [0x45af] = 0x84bf, [0x45b0] = 0x84ba,
  [0x45b1] = 0x84e0, [0x45b2] = 0x84a1, [0x45b3] = 0x84b9, [0x45b4] = 0x84b4,
  [0x45b5] = 0x8497, [0x45b6] = 0x84e5, [0x45b7] = 0x84e3, [0x45b8] = 0x850c,
  [0x45b9] = 0x750d, [0x45ba] = 0x8538, [0x45bb] = 0x84f0, [0x45bc] = 0x8539,
  [0x45bd] = 0x851f, [0x45be] = 0x853a, [0x45c0] = 0x8f45, [0x45c1] = 0x8f46,
  [0x45c2] = 0x8f47, [0x45c3] = 0x8f48, [0x45c4] = 0x8f49, [0x45c5] = 0x8f4a,
  [0x45c6] = 0x8f4b, [0x45c7] = 0x8f4c, [0x45c8] = 0x8f4d, [0x45c9] = 0x8f4e,
  [0x45ca] = 0x8f4f, [0x45cb] = 0x8f50, [0x45cc] = 0x8f51, [0x45cd] = 0x8f52,
  [0x45ce] = 0x8f53, [0x45cf] = 0x8f54, [0x45d0] = 0x8f55, [0x45d1] = 0x8f56,
  [0x45d2] = 0x8f57, [0x45d3] = 0x8f58, [0x45d4] = 0x8f59, [0x45d5] = 0x8f5a,
  [0x45d6] = 0x8f5b, [0x45d7] = 0x8f5c, [0x45d8] = 0x8f5d, [0x45d9] = 0x8f5e,
  [0x45da] = 0x8f5f, [0x45db] = 0x8f60, [0x45dc] = 0x8f61, [0x45dd] = 0x8f62,
  [0x45de] = 0x8f63, [0x45df] = 0x8f64, [0x45e0] = 0x8f65, [0x45e1] = 0x8f6a,
  [0x45e2] = 0x8f80, [0x45e3] = 0x8f8c, [0x45e4] = 0x8f92, [0x45e5] = 0x8f9d,
  [0x45e6] = 0x8fa0, [0x45e7] = 0x8fa1, [0x45e8] = 0x8fa2, [0x45e9] = 0x8fa4,
  [0x45ea] = 0x8fa5, [0x45eb] = 0x8fa6, [0x45ec] = 0x8fa7, [0x45ed] = 0x8faa,
  [0x45ee] = 0x8fac, [0x45ef] = 0x8fad, [0x45f0] = 0x8fae, [0x45f1] = 0x8faf,
  [0x45f2] = 0x8fb2, [0x45f3] = 0x8fb3, [0x45f4] = 0x8fb4, [0x45f5] = 0x8fb5,
  [0x45f6] = 0x8fb7, [0x45f7] = 0x8fb8, [0x45f8] = 0x8fba, [0x45f9] = 0x8fbb,
  [0x45fa] = 0x8fbc, [0x45fb] = 0x8fbf, [0x45fc] = 0x8fc0, [0x45fd] = 0x8fc3,
  [0x45fe] = 0x8fc6, [0x4600] = 0x8fc9, [0x4601] = 0x8fca, [0x4602] = 0x8fcb,
  [0x4603] = 0x8fcc, [0x4604] = 0x8fcd, [0x4605] = 0x8fcf, [0x4606] = 0x8fd2,
  [0x4607] = 0x8fd6, [0x4608] = 0x8fd7, [0x4609] = 0x8fda, [0x460a] = 0x8fe0,
  [0x460b] = 0x8fe1, [0x460c] = 0x8fe3, [0x460d] = 0x8fe7, [0x460e] = 0x8fec,
  [0x460f] = 0x8fef, [0x4610] = 0x8ff1, [0x4611] = 0x8ff2, [0x4612] = 0x8ff4,
  [0x4613] = 0x8ff5, [0x4614] = 0x8ff6, [0x4615] = 0x8ffa, [0x4616] = 0x8ffb,
  [0x4617] = 0x8ffc, [0x4618] = 0x8ffe, [0x4619] = 0x8fff, [0x461a] = 0x9007,
  [0x461b] = 0x9008, [0x461c] = 0x900c, [0x461d] = 0x900e, [0x461e] = 0x9013,
  [0x461f] = 0x9015, [0x4620] = 0x9018, [0x4621] = 0x8556, [0x4622] = 0x853b,
  [0x4623] = 0x84ff, [0x4624] = 0x84fc, [0x4625] = 0x8559, [0x4626] = 0x8548,
  [0x4627] = 0x8568, [0x4628] = 0x8564, [0x4629] = 0x855e, [0x462a] = 0x857a,
  [0x462b] = 0x77a2, [0x462c] = 0x8543, [0x462d] = 0x8572, [0x462e] = 0x857b,
  [0x462f] = 0x85a4, [0x4630] = 0x85a8, [0x4631] = 0x8587, [0x4632] = 0x858f,
  [0x4633] = 0x8579, [0x4634] = 0x85ae, [0x4635] = 0x859c, [0x4636] = 0x8585,
  [0x4637] = 0x85b9, [0x4638] = 0x85b7, [0x4639] = 0x85b0, [0x463a] = 0x85d3,
  [0x463b] = 0x85c1, [0x463c] = 0x85dc, [0x463d] = 0x85ff, [0x463e] = 0x8627,
  [0x463f] = 0x8605, [0x4640] = 0x8629, [0x4641] = 0x8616, [0x4642] = 0x863c,
  [0x4643] = 0x5efe, [0x4644] = 0x5f08, [0x4645] = 0x593c, [0x4646] = 0x5941,
  [0x4647] = 0x8037, [0x4648] = 0x5955, [0x4649] = 0x595a, [0x464a] = 0x5958,
  [0x464b] = 0x530f, [0x464c] = 0x5c22, [0x464d] = 0x5c25, [0x464e] = 0x5c2c,
  [0x464f] = 0x5c34, [0x4650] = 0x624c, [0x4651] = 0x626a, [0x4652] = 0x629f,
  [0x4653] = 0x62bb, [0x4654] = 0x62ca, [0x4655] = 0x62da, [0x4656] = 0x62d7,
  [0x4657] = 0x62ee, [0x4658] = 0x6322, [0x4659] = 0x62f6, [0x465a] = 0x6339,
  [0x465b] = 0x634b, [0x465c] = 0x6343, [0x465d] = 0x63ad, [0x465e] = 0x63f6,
  [0x465f] = 0x6371, [0x4660] = 0x637a, [0x4661] = 0x638e, [0x4662] = 0x63b4,
  [0x4663] = 0x636d, [0x4664] = 0x63ac, [0x4665] = 0x638a, [0x4666] = 0x6369,
  [0x4667] = 0x63ae, [0x4668] = 0x63bc, [0x4669] = 0x63f2, [0x466a] = 0x63f8,
  [0x466b] = 0x63e0, [0x466c] = 0x63ff, [0x466d] = 0x63c4, [0x466e] = 0x63de,
  [0x466f] = 0x63ce, [0x4670] = 0x6452, [0x4671] = 0x63c6, [0x4672] = 0x63be,
  [0x4673] = 0x6445, [0x4674] = 0x6441, [0x4675] = 0x640b, [0x4676] = 0x641b,
  [0x4677] = 0x6420, [0x4678] = 0x640c, [0x4679] = 0x6426, [0x467a] = 0x6421,
  [0x467b] = 0x645e, [0x467c] = 0x6484, [0x467d] = 0x646d, [0x467e] = 0x6496,
  [0x4680] = 0x9019, [0x4681] = 0x901c, [0x4682] = 0x9023, [0x4683] = 0x9024,
  [0x4684] = 0x9025, [0x4685] = 0x9027, [0x4686] = 0x9028, [0x4687] = 0x9029,
  [0x4688] = 0x902a, [0x4689] = 0x902b, [0x468a] = 0x902c, [0x468b] = 0x9030,
  [0x468c] = 0x9031, [0x468d] = 0x9032, [0x468e] = 0x9033, [0x468f] = 0x9034,
  [0x4690] = 0x9037, [0x4691] = 0x9039, [0x4692] = 0x903a, [0x4693] = 0x903d,
  [0x4694] = 0x903f, [0x4695] = 0x9040, [0x4696] = 0x9043, [0x4697] = 0x9045,
  [0x4698] = 0x9046, [0x4699] = 0x9048, [0x469a] = 0x9049, [0x469b] = 0x904a,
  [0x469c] = 0x904b, [0x469d] = 0x904c, [0x469e] = 0x904e, [0x469f] = 0x9054,
  [0x46a0] = 0x9055, [0x46a1] = 0x9056, [0x46a2] = 0x9059, [0x46a3] = 0x905a,
  [0x46a4] = 0x905c, [0x46a5] = 0x905d, [0x46a6] = 0x905e, [0x46a7] = 0x905f,
  [0x46a8] = 0x9060, [0x46a9] = 0x9061, [0x46aa] = 0x9064, [0x46ab] = 0x9066,
  [0x46ac] = 0x9067, [0x46ad] = 0x9069, [0x46ae] = 0x906a, [0x46af] = 0x906b,
  [0x46b0] = 0x906c, [0x46b1] = 0x906f, [0x46b2] = 0x9070, [0x46b3] = 0x9071,
  [0x46b4] = 0x9072, [0x46b5] = 0x9073, [0x46b6] = 0x9076, [0x46b7] = 0x9077,
  [0x46b8] = 0x9078, [0x46b9] = 0x9079, [0x46ba] = 0x907a, [0x46bb] = 0x907b,
  [0x46bc] = 0x907c, [0x46bd] = 0x907e, [0x46be] = 0x9081, [0x46c0] = 0x9084,
  [0x46c1] = 0x9085, [0x46c2] = 0x9086, [0x46c3] = 0x9087, [0x46c4] = 0x9089,
  [0x46c5] = 0x908a, [0x46c6] = 0x908c, [0x46c7] = 0x908d, [0x46c8] = 0x908e,
  [0x46c9] = 0x908f, [0x46ca] = 0x9090, [0x46cb] = 0x9092, [0x46cc] = 0x9094,
  [0x46cd] = 0x9096, [0x46ce] = 0x9098, [0x46cf] = 0x909a, [0x46d0] = 0x909c,
  [0x46d1] = 0x909e, [0x46d2] = 0x909f, [0x46d3] = 0x90a0, [0x46d4] = 0x90a4,
  [0x46d5] = 0x90a5, [0x46d6] = 0x90a7, [0x46d7] = 0x90a8, [0x46d8] = 0x90a9,
  [0x46d9] = 0x90ab, [0x46da] = 0x90ad, [0x46db] = 0x90b2, [0x46dc] = 0x90b7,
  [0x46dd] = 0x90bc, [0x46de] = 0x90bd, [0x46df] = 0x90bf, [0x46e0] = 0x90c0,
  [0x46e1] = 0x647a, [0x46e2] = 0x64b7, [0x46e3] = 0x64b8, [0x46e4] = 0x6499,
  [0x46e5] = 0x64ba, [0x46e6] = 0x64c0, [0x46e7] = 0x64d0, [0x46e8] = 0x64d7,
  [0x46e9] = 0x64e4, [0x46ea] = 0x64e2, [0x46eb] = 0x6509, [0x46ec] = 0x6525,
  [0x46ed] = 0x652e, [0x46ee] = 0x5f0b, [0x46ef] = 0x5fd2, [0x46f0] = 0x7519,
  [0x46f1] = 0x5f11, [0x46f2] = 0x535f, [0x46f3] = 0x53f1, [0x46f4] = 0x53fd,
  [0x46f5] = 0x53e9, [0x46f6] = 0x53e8, [0x46f7] = 0x53fb, [0x46f8] = 0x5412,
  [0x46f9] = 0x5416, [0x46fa] = 0x5406, [0x46fb] = 0x544b, [0x46fc] = 0x5452,
  [0x46fd] = 0x5453, [0x46fe] = 0x5454, [0x46ff] = 0x5456, [0x4700] = 0x5443,
  [0x4701] = 0x5421, [0x4702] = 0x5457, [0x4703] = 0x5459, [0x4704] = 0x5423,
  [0x4705] = 0x5432, [0x4706] = 0x5482, [0x4707] = 0x5494, [0x4708] = 0x5477,
  [0x4709] = 0x5471, [0x470a] = 0x5464, [0x470b] = 0x549a, [0x470c] = 0x549b,
  [0x470d] = 0x5484, [0x470e] = 0x5476, [0x470f] = 0x5466, [0x4710] = 0x549d,
  [0x4711] = 0x54d0, [0x4712] = 0x54ad, [0x4713] = 0x54c2, [0x4714] = 0x54b4,
  [0x4715] = 0x54d2, [0x4716] = 0x54a7, [0x4717] = 0x54a6, [0x4718] = 0x54d3,
  [0x4719] = 0x54d4, [0x471a] = 0x5472, [0x471b] = 0x54a3, [0x471c] = 0x54d5,
  [0x471d] = 0x54bb, [0x471e] = 0x54bf, [0x471f] = 0x54cc, [0x4720] = 0x54d9,
  [0x4721] = 0x54da, [0x4722] = 0x54dc, [0x4723] = 0x54a9, [0x4724] = 0x54aa,
  [0x4725] = 0x54a4, [0x4726] = 0x54dd, [0x4727] = 0x54cf, [0x4728] = 0x54de,
  [0x4729] = 0x551b, [0x472a] = 0x54e7, [0x472b] = 0x5520, [0x472c] = 0x54fd,
  [0x472d] = 0x5514, [0x472e] = 0x54f3, [0x472f] = 0x5522, [0x4730] = 0x5523,
  [0x4731] = 0x550f, [0x4732] = 0x5511, [0x4733] = 0x5527, [0x4734] = 0x552a,
  [0x4735] = 0x5567, [0x4736] = 0x558f, [0x4737] = 0x55b5, [0x4738] = 0x5549,
  [0x4739] = 0x556d, [0x473a] = 0x5541, [0x473b] = 0x5555, [0x473c] = 0x553f,
  [0x473d] = 0x5550, [0x473e] = 0x553c, [0x4740] = 0x90c2, [0x4741] = 0x90c3,
  [0x4742] = 0x90c6, [0x4743] = 0x90c8, [0x4744] = 0x90c9, [0x4745] = 0x90cb,
  [0x4746] = 0x90cc, [0x4747] = 0x90cd, [0x4748] = 0x90d2, [0x4749] = 0x90d4,
  [0x474a] = 0x90d5, [0x474b] = 0x90d6, [0x474c] = 0x90d8, [0x474d] = 0x90d9,
  [0x474e] = 0x90da, [0x474f] = 0x90de, [0x4750] = 0x90df, [0x4751] = 0x90e0,
  [0x4752] = 0x90e3, [0x4753] = 0x90e4, [0x4754] = 0x90e5, [0x4755] = 0x90e9,
  [0x4756] = 0x90ea, [0x4757] = 0x90ec, [0x4758] = 0x90ee, [0x4759] = 0x90f0,
  [0x475a] = 0x90f1, [0x475b] = 0x90f2, [0x475c] = 0x90f3, [0x475d] = 0x90f5,
  [0x475e] = 0x90f6, [0x475f] = 0x90f7, [0x4760] = 0x90f9, [0x4761] = 0x90fa,
  [0x4762] = 0x90fb, [0x4763] = 0x90fc, [0x4764] = 0x90ff, [0x4765] = 0x9100,
  [0x4766] = 0x9101, [0x4767] = 0x9103, [0x4768] = 0x9105, [0x4769] = 0x9106,
  [0x476a] = 0x9107, [0x476b] = 0x9108, [0x476c] = 0x9109, [0x476d] = 0x910a,
  [0x476e] = 0x910b, [0x476f] = 0x910c, [0x4770] = 0x910d, [0x4771] = 0x910e,
  [0x4772] = 0x910f, [0x4773] = 0x9110, [0x4774] = 0x9111, [0x4775] = 0x9112,
  [0x4776] = 0x9113, [0x4777] = 0x9114, [0x4778] = 0x9115, [0x4779] = 0x9116,
  [0x477a] = 0x9117, [0x477b] = 0x9118, [0x477c] = 0x911a, [0x477d] = 0x911b,
  [0x477e] = 0x911c, [0x4780] = 0x911d, [0x4781] = 0x911f, [0x4782] = 0x9120,
  [0x4783] = 0x9121, [0x4784] = 0x9124, [0x4785] = 0x9125, [0x4786] = 0x9126,
  [0x4787] = 0x9127, [0x4788] = 0x9128, [0x4789] = 0x9129, [0x478a] = 0x912a,
  [0x478b] = 0x912b, [0x478c] = 0x912c, [0x478d] = 0x912d, [0x478e] = 0x912e,
  [0x478f] = 0x9130, [0x4790] = 0x9132, [0x4791] = 0x9133, [0x4792] = 0x9134,
  [0x4793] = 0x9135, [0x4794] = 0x9136, [0x4795] = 0x9137, [0x4796] = 0x9138,
  [0x4797] = 0x913a, [0x4798] = 0x913b, [0x4799] = 0x913c, [0x479a] = 0x913d,
  [0x479b] = 0x913e, [0x479c] = 0x913f, [0x479d] = 0x9140, [0x479e] = 0x9141,
  [0x479f] = 0x9142, [0x47a0] = 0x9144, [0x47a1] = 0x5537, [0x47a2] = 0x5556,
  [0x47a3] = 0x5575, [0x47a4] = 0x5576, [0x47a5] = 0x5577, [0x47a6] = 0x5533,
  [0x47a7] = 0x5530, [0x47a8] = 0x555c, [0x47a9] = 0x558b, [0x47aa] = 0x55d2,
  [0x47ab] = 0x5583, [0x47ac] = 0x55b1, [0x47ad] = 0x55b9, [0x47ae] = 0x5588,
  [0x47af] = 0x5581, [0x47b0] = 0x559f, [0x47b1] = 0x557e, [0x47b2] = 0x55d6,
  [0x47b3] = 0x5591, [0x47b4] = 0x557b, [0x47b5] = 0x55df, [0x47b6] = 0x55bd,
  [0x47b7] = 0x55be, [0x47b8] = 0x5594, [0x47b9] = 0x5599, [0x47ba] = 0x55ea,
  [0x47bb] = 0x55f7, [0x47bc] = 0x55c9, [0x47bd] = 0x561f, [0x47be] = 0x55d1,
  [0x47bf] = 0x55eb, [0x47c0] = 0x55ec, [0x47c1] = 0x55d4, [0x47c2] = 0x55e6,
  [0x47c3] = 0x55dd, [0x47c4] = 0x55c4, [0x47c5] = 0x55ef, [0x47c6] = 0x55e5,
  [0x47c7] = 0x55f2, [0x47c8] = 0x55f3, [0x47c9] = 0x55cc, [0x47ca] = 0x55cd,
  [0x47cb] = 0x55e8, [0x47cc] = 0x55f5, [0x47cd] = 0x55e4, [0x47ce] = 0x8f94,
  [0x47cf] = 0x561e, [0x47d0] = 0x5608, [0x47d1] = 0x560c, [0x47d2] = 0x5601,
  [0x47d3] = 0x5624, [0x47d4] = 0x5623, [0x47d5] = 0x55fe, [0x47d6] = 0x5600,
  [0x47d7] = 0x5627, [0x47d8] = 0x562d, [0x47d9] = 0x5658, [0x47da] = 0x5639,
  [0x47db] = 0x5657, [0x47dc] = 0x562c, [0x47dd] = 0x564d, [0x47de] = 0x5662,
  [0x47df] = 0x5659, [0x47e0] = 0x565c, [0x47e1] = 0x564c, [0x47e2] = 0x5654,
  [0x47e3] = 0x5686, [0x47e4] = 0x5664, [0x47e5] = 0x5671, [0x47e6] = 0x566b,
  [0x47e7] = 0x567b, [0x47e8] = 0x567c, [0x47e9] = 0x5685, [0x47ea] = 0x5693,
  [0x47eb] = 0x56af, [0x47ec] = 0x56d4, [0x47ed] = 0x56d7, [0x47ee] = 0x56dd,
  [0x47ef] = 0x56e1, [0x47f0] = 0x56f5, [0x47f1] = 0x56eb, [0x47f2] = 0x56f9,
  [0x47f3] = 0x56ff, [0x47f4] = 0x5704, [0x47f5] = 0x570a, [0x47f6] = 0x5709,
  [0x47f7] = 0x571c, [0x47f8] = 0x5e0f, [0x47f9] = 0x5e19, [0x47fa] = 0x5e14,
  [0x47fb] = 0x5e11, [0x47fc] = 0x5e31, [0x47fd] = 0x5e3b, [0x47fe] = 0x5e3c,
  [0x4800] = 0x9145, [0x4801] = 0x9147, [0x4802] = 0x9148, [0x4803] = 0x9151,
  [0x4804] = 0x9153, [0x4805] = 0x9154, [0x4806] = 0x9155, [0x4807] = 0x9156,
  [0x4808] = 0x9158, [0x4809] = 0x9159, [0x480a] = 0x915b, [0x480b] = 0x915c,
  [0x480c] = 0x915f, [0x480d] = 0x9160, [0x480e] = 0x9166, [0x480f] = 0x9167,
  [0x4810] = 0x9168, [0x4811] = 0x916b, [0x4812] = 0x916d, [0x4813] = 0x9173,
  [0x4814] = 0x917a, [0x4815] = 0x917b, [0x4816] = 0x917c, [0x4817] = 0x9180,
  [0x4818] = 0x9181, [0x4819] = 0x9182, [0x481a] = 0x9183, [0x481b] = 0x9184,
  [0x481c] = 0x9186, [0x481d] = 0x9188, [0x481e] = 0x918a, [0x481f] = 0x918e,
  [0x4820] = 0x918f, [0x4821] = 0x9193, [0x4822] = 0x9194, [0x4823] = 0x9195,
  [0x4824] = 0x9196, [0x4825] = 0x9197, [0x4826] = 0x9198, [0x4827] = 0x9199,
  [0x4828] = 0x919c, [0x4829] = 0x919d, [0x482a] = 0x919e, [0x482b] = 0x919f,
  [0x482c] = 0x91a0, [0x482d] = 0x91a1, [0x482e] = 0x91a4, [0x482f] = 0x91a5,
  [0x4830] = 0x91a6, [0x4831] = 0x91a7, [0x4832] = 0x91a8, [0x4833] = 0x91a9,
  [0x4834] = 0x91ab, [0x4835] = 0x91ac, [0x4836] = 0x91b0, [0x4837] = 0x91b1,
  [0x4838] = 0x91b2, [0x4839] = 0x91b3, [0x483a] = 0x91b6, [0x483b] = 0x91b7,
  [0x483c] = 0x91b8, [0x483d] = 0x91b9, [0x483e] = 0x91bb, [0x4840] = 0x91bc,
  [0x4841] = 0x91bd, [0x4842] = 0x91be, [0x4843] = 0x91bf, [0x4844] = 0x91c0,
  [0x4845] = 0x91c1, [0x4846] = 0x91c2, [0x4847] = 0x91c3, [0x4848] = 0x91c4,
  [0x4849] = 0x91c5, [0x484a] = 0x91c6, [0x484b] = 0x91c8, [0x484c] = 0x91cb,
  [0x484d] = 0x91d0, [0x484e] = 0x91d2, [0x484f] = 0x91d3, [0x4850] = 0x91d4,
  [0x4851] = 0x91d5, [0x4852] = 0x91d6, [0x4853] = 0x91d7, [0x4854] = 0x91d8,
  [0x4855] = 0x91d9, [0x4856] = 0x91da, [0x4857] = 0x91db, [0x4858] = 0x91dd,
  [0x4859] = 0x91de, [0x485a] = 0x91df, [0x485b] = 0x91e0, [0x485c] = 0x91e1,
  [0x485d] = 0x91e2, [0x485e] = 0x91e3, [0x485f] = 0x91e4, [0x4860] = 0x91e5,
  [0x4861] = 0x5e37, [0x4862] = 0x5e44, [0x4863] = 0x5e54, [0x4864] = 0x5e5b,
  [0x4865] = 0x5e5e, [0x4866] = 0x5e61, [0x4867] = 0x5c8c, [0x4868] = 0x5c7a,
  [0x4869] = 0x5c8d, [0x486a] = 0x5c90, [0x486b] = 0x5c96, [0x486c] = 0x5c88,
  [0x486d] = 0x5c98, [0x486e] = 0x5c99, [0x486f] = 0x5c91, [0x4870] = 0x5c9a,
  [0x4871] = 0x5c9c, [0x4872] = 0x5cb5, [0x4873] = 0x5ca2, [0x4874] = 0x5cbd,
  [0x4875] = 0x5cac, [0x4876] = 0x5cab, [0x4877] = 0x5cb1, [0x4878] = 0x5ca3,
  [0x4879] = 0x5cc1, [0x487a] = 0x5cb7, [0x487b] = 0x5cc4, [0x487c] = 0x5cd2,
  [0x487d] = 0x5ce4, [0x487e] = 0x5ccb, [0x487f] = 0x5ce5, [0x4880] = 0x5d02,
  [0x4881] = 0x5d03, [0x4882] = 0x5d27, [0x4883] = 0x5d26, [0x4884] = 0x5d2e,
  [0x4885] = 0x5d24, [0x4886] = 0x5d1e, [0x4887] = 0x5d06, [0x4888] = 0x5d1b,
  [0x4889] = 0x5d58, [0x488a] = 0x5d3e, [0x488b] = 0x5d34, [0x488c] = 0x5d3d,
  [0x488d] = 0x5d6c, [0x488e] = 0x5d5b, [0x488f] = 0x5d6f, [0x4890] = 0x5d5d,
  [0x4891] = 0x5d6b, [0x4892] = 0x5d4b, [0x4893] = 0x5d4a, [0x4894] = 0x5d69,
  [0x4895] = 0x5d74, [0x4896] = 0x5d82, [0x4897] = 0x5d99, [0x4898] = 0x5d9d,
  [0x4899] = 0x8c73, [0x489a] = 0x5db7, [0x489b] = 0x5dc5, [0x489c] = 0x5f73,
  [0x489d] = 0x5f77, [0x489e] = 0x5f82, [0x489f] = 0x5f87, [0x48a0] = 0x5f89,
  [0x48a1] = 0x5f8c, [0x48a2] = 0x5f95, [0x48a3] = 0x5f99, [0x48a4] = 0x5f9c,
  [0x48a5] = 0x5fa8, [0x48a6] = 0x5fad, [0x48a7] = 0x5fb5, [0x48a8] = 0x5fbc,
  [0x48a9] = 0x8862, [0x48aa] = 0x5f61, [0x48ab] = 0x72ad, [0x48ac] = 0x72b0,
  [0x48ad] = 0x72b4, [0x48ae] = 0x72b7, [0x48af] = 0x72b8, [0x48b0] = 0x72c3,
  [0x48b1] = 0x72c1, [0x48b2] = 0x72ce, [0x48b3] = 0x72cd, [0x48b4] = 0x72d2,
  [0x48b5] = 0x72e8, [0x48b6] = 0x72ef, [0x48b7] = 0x72e9, [0x48b8] = 0x72f2,
  [0x48b9] = 0x72f4, [0x48ba] = 0x72f7, [0x48bb] = 0x7301, [0x48bc] = 0x72f3,
  [0x48bd] = 0x7303, [0x48be] = 0x72fa, [0x48c0] = 0x91e6, [0x48c1] = 0x91e7,
  [0x48c2] = 0x91e8, [0x48c3] = 0x91e9, [0x48c4] = 0x91ea, [0x48c5] = 0x91eb,
  [0x48c6] = 0x91ec, [0x48c7] = 0x91ed, [0x48c8] = 0x91ee, [0x48c9] = 0x91ef,
  [0x48ca] = 0x91f0, [0x48cb] = 0x91f1, [0x48cc] = 0x91f2, [0x48cd] = 0x91f3,
  [0x48ce] = 0x91f4, [0x48cf] = 0x91f5, [0x48d0] = 0x91f6, [0x48d1] = 0x91f7,
  [0x48d2] = 0x91f8, [0x48d3] = 0x91f9, [0x48d4] = 0x91fa, [0x48d5] = 0x91fb,
  [0x48d6] = 0x91fc, [0x48d7] = 0x91fd, [0x48d8] = 0x91fe, [0x48d9] = 0x91ff,
  [0x48da] = 0x9200, [0x48db] = 0x9201, [0x48dc] = 0x9202, [0x48dd] = 0x9203,
  [0x48de] = 0x9204, [0x48df] = 0x9205, [0x48e0] = 0x9206, [0x48e1] = 0x9207,
  [0x48e2] = 0x9208, [0x48e3] = 0x9209, [0x48e4] = 0x920a, [0x48e5] = 0x920b,
  [0x48e6] = 0x920c, [0x48e7] = 0x920d, [0x48e8] = 0x920e, [0x48e9] = 0x920f,
  [0x48ea] = 0x9210, [0x48eb] = 0x9211, [0x48ec] = 0x9212, [0x48ed] = 0x9213,
  [0x48ee] = 0x9214, [0x48ef] = 0x9215, [0x48f0] = 0x9216, [0x48f1] = 0x9217,
  [0x48f2] = 0x9218, [0x48f3] = 0x9219, [0x48f4] = 0x921a, [0x48f5] = 0x921b,
  [0x48f6] = 0x921c, [0x48f7] = 0x921d, [0x48f8] = 0x921e, [0x48f9] = 0x921f,
  [0x48fa] = 0x9220, [0x48fb] = 0x9221, [0x48fc] = 0x9222, [0x48fd] = 0x9223,
  [0x48fe] = 0x9224, [0x4900] = 0x9225, [0x4901] = 0x9226, [0x4902] = 0x9227,
  [0x4903] = 0x9228, [0x4904] = 0x9229, [0x4905] = 0x922a, [0x4906] = 0x922b,
  [0x4907] = 0x922c, [0x4908] = 0x922d, [0x4909] = 0x922e, [0x490a] = 0x922f,
  [0x490b] = 0x9230, [0x490c] = 0x9231, [0x490d] = 0x9232, [0x490e] = 0x9233,
  [0x490f] = 0x9234, [0x4910] = 0x9235, [0x4911] = 0x9236, [0x4912] = 0x9237,
  [0x4913] = 0x9238, [0x4914] = 0x9239, [0x4915] = 0x923a, [0x4916] = 0x923b,
  [0x4917] = 0x923c, [0x4918] = 0x923d, [0x4919] = 0x923e, [0x491a] = 0x923f,
  [0x491b] = 0x9240, [0x491c] = 0x9241, [0x491d] = 0x9242, [0x491e] = 0x9243,
  [0x491f] = 0x9244, [0x4920] = 0x9245, [0x4921] = 0x72fb, [0x4922] = 0x7317,
  [0x4923] = 0x7313, [0x4924] = 0x7321, [0x4925] = 0x730a, [0x4926] = 0x731e,
  [0x4927] = 0x731d, [0x4928] = 0x7315, [0x4929] = 0x7322, [0x492a] = 0x7339,
  [0x492b] = 0x7325, [0x492c] = 0x732c, [0x492d] = 0x7338, [0x492e] = 0x7331,
  [0x492f] = 0x7350, [0x4930] = 0x734d, [0x4931] = 0x7357, [0x4932] = 0x7360,
  [0x4933] = 0x736c, [0x4934] = 0x736f, [0x4935] = 0x737e, [0x4936] = 0x821b,
  [0x4937] = 0x5925, [0x4938] = 0x98e7, [0x4939] = 0x5924, [0x493a] = 0x5902,
  [0x493b] = 0x9963, [0x493c] = 0x9967, [0x493d] = 0x9968, [0x493e] = 0x9969,
  [0x493f] = 0x996a, [0x4940] = 0x996b, [0x4941] = 0x996c, [0x4942] = 0x9974,
  [0x4943] = 0x9977, [0x4944] = 0x997d, [0x4945] = 0x9980, [0x4946] = 0x9984,
  [0x4947] = 0x9987, [0x4948] = 0x998a, [0x4949] = 0x998d, [0x494a] = 0x9990,
  [0x494b] = 0x9991, [0x494c] = 0x9993, [0x494d] = 0x9994, [0x494e] = 0x9995,
  [0x494f] = 0x5e80, [0x4950] = 0x5e91, [0x4951] = 0x5e8b, [0x4952] = 0x5e96,
  [0x4953] = 0x5ea5, [0x4954] = 0x5ea0, [0x4955] = 0x5eb9, [0x4956] = 0x5eb5,
  [0x4957] = 0x5ebe, [0x4958] = 0x5eb3, [0x4959] = 0x8d53, [0x495a] = 0x5ed2,
  [0x495b] = 0x5ed1, [0x495c] = 0x5edb, [0x495d] = 0x5ee8, [0x495e] = 0x5eea,
  [0x495f] = 0x81ba, [0x4960] = 0x5fc4, [0x4961] = 0x5fc9, [0x4962] = 0x5fd6,
  [0x4963] = 0x5fcf, [0x4964] = 0x6003, [0x4965] = 0x5fee, [0x4966] = 0x6004,
  [0x4967] = 0x5fe1, [0x4968] = 0x5fe4, [0x4969] = 0x5ffe, [0x496a] = 0x6005,
  [0x496b] = 0x6006, [0x496c] = 0x5fea, [0x496d] = 0x5fed, [0x496e] = 0x5ff8,
  [0x496f] = 0x6019, [0x4970] = 0x6035, [0x4971] = 0x6026, [0x4972] = 0x601b,
  [0x4973] = 0x600f, [0x4974] = 0x600d, [0x4975] = 0x6029, [0x4976] = 0x602b,
  [0x4977] = 0x600a, [0x4978] = 0x603f, [0x4979] = 0x6021, [0x497a] = 0x6078,
  [0x497b] = 0x6079, [0x497c] = 0x607b, [0x497d] = 0x607a, [0x497e] = 0x6042,
  [0x4980] = 0x9246, [0x4981] = 0x9247, [0x4982] = 0x9248, [0x4983] = 0x9249,
  [0x4984] = 0x924a, [0x4985] = 0x924b, [0x4986] = 0x924c, [0x4987] = 0x924d,
  [0x4988] = 0x924e, [0x4989] = 0x924f, [0x498a] = 0x9250, [0x498b] = 0x9251,
  [0x498c] = 0x9252, [0x498d] = 0x9253, [0x498e] = 0x9254, [0x498f] = 0x9255,
  [0x4990] = 0x9256, [0x4991] = 0x9257, [0x4992] = 0x9258, [0x4993] = 0x9259,
  [0x4994] = 0x925a, [0x4995] = 0x925b, [0x4996] = 0x925c, [0x4997] = 0x925d,
  [0x4998] = 0x925e, [0x4999] = 0x925f, [0x499a] = 0x9260, [0x499b] = 0x9261,
  [0x499c] = 0x9262, [0x499d] = 0x9263, [0x499e] = 0x9264, [0x499f] = 0x9265,
  [0x49a0] = 0x9266, [0x49a1] = 0x9267, [0x49a2] = 0x9268, [0x49a3] = 0x9269,
  [0x49a4] = 0x926a, [0x49a5] = 0x926b, [0x49a6] = 0x926c, [0x49a7] = 0x926d,
  [0x49a8] = 0x926e, [0x49a9] = 0x926f, [0x49aa] = 0x9270, [0x49ab] = 0x9271,
  [0x49ac] = 0x9272, [0x49ad] = 0x9273, [0x49ae] = 0x9275, [0x49af] = 0x9276,
  [0x49b0] = 0x9277, [0x49b1] = 0x9278, [0x49b2] = 0x9279, [0x49b3] = 0x927a,
  [0x49b4] = 0x927b, [0x49b5] = 0x927c, [0x49b6] = 0x927d, [0x49b7] = 0x927e,
  [0x49b8] = 0x927f, [0x49b9] = 0x9280, [0x49ba] = 0x9281, [0x49bb] = 0x9282,
  [0x49bc] = 0x9283, [0x49bd] = 0x9284, [0x49be] = 0x9285, [0x49c0] = 0x9286,
  [0x49c1] = 0x9287, [0x49c2] = 0x9288, [0x49c3] = 0x9289, [0x49c4] = 0x928a,
  [0x49c5] = 0x928b, [0x49c6] = 0x928c, [0x49c7] = 0x928d, [0x49c8] = 0x928f,
  [0x49c9] = 0x9290, [0x49ca] = 0x9291, [0x49cb] = 0x9292, [0x49cc] = 0x9293,
  [0x49cd] = 0x9294, [0x49ce] = 0x9295, [0x49cf] = 0x9296, [0x49d0] = 0x9297,
  [0x49d1] = 0x9298, [0x49d2] = 0x9299, [0x49d3] = 0x929a, [0x49d4] = 0x929b,
  [0x49d5] = 0x929c, [0x49d6] = 0x929d, [0x49d7] = 0x929e, [0x49d8] = 0x929f,
  [0x49d9] = 0x92a0, [0x49da] = 0x92a1, [0x49db] = 0x92a2, [0x49dc] = 0x92a3,
  [0x49dd] = 0x92a4, [0x49de] = 0x92a5, [0x49df] = 0x92a6, [0x49e0] = 0x92a7,
  [0x49e1] = 0x606a, [0x49e2] = 0x607d, [0x49e3] = 0x6096, [0x49e4] = 0x609a,
  [0x49e5] = 0x60ad, [0x49e6] = 0x609d, [0x49e7] = 0x6083, [0x49e8] = 0x6092,
  [0x49e9] = 0x608c, [0x49ea] = 0x609b, [0x49eb] = 0x60ec, [0x49ec] = 0x60bb,
  [0x49ed] = 0x60b1, [0x49ee] = 0x60dd, [0x49ef] = 0x60d8, [0x49f0] = 0x60c6,
  [0x49f1] = 0x60da, [0x49f2] = 0x60b4, [0x49f3] = 0x6120, [0x49f4] = 0x6126,
  [0x49f5] = 0x6115, [0x49f6] = 0x6123, [0x49f7] = 0x60f4, [0x49f8] = 0x6100,
  [0x49f9] = 0x610e, [0x49fa] = 0x612b, [0x49fb] = 0x614a, [0x49fc] = 0x6175,
  [0x49fd] = 0x61ac, [0x49fe] = 0x6194, [0x49ff] = 0x61a7, [0x4a00] = 0x61b7,
  [0x4a01] = 0x61d4, [0x4a02] = 0x61f5, [0x4a03] = 0x5fdd, [0x4a04] = 0x96b3,
  [0x4a05] = 0x95e9, [0x4a06] = 0x95eb, [0x4a07] = 0x95f1, [0x4a08] = 0x95f3,
  [0x4a09] = 0x95f5, [0x4a0a] = 0x95f6, [0x4a0b] = 0x95fc, [0x4a0c] = 0x95fe,
  [0x4a0d] = 0x9603, [0x4a0e] = 0x9604, [0x4a0f] = 0x9606, [0x4a10] = 0x9608,
  [0x4a11] = 0x960a, [0x4a12] = 0x960b, [0x4a13] = 0x960c, [0x4a14] = 0x960d,
  [0x4a15] = 0x960f, [0x4a16] = 0x9612, [0x4a17] = 0x9615, [0x4a18] = 0x9616,
  [0x4a19] = 0x9617, [0x4a1a] = 0x9619, [0x4a1b] = 0x961a, [0x4a1c] = 0x4e2c,
  [0x4a1d] = 0x723f, [0x4a1e] = 0x6215, [0x4a1f] = 0x6c35, [0x4a20] = 0x6c54,
  [0x4a21] = 0x6c5c, [0x4a22] = 0x6c4a, [0x4a23] = 0x6ca3, [0x4a24] = 0x6c85,
  [0x4a25] = 0x6c90, [0x4a26] = 0x6c94, [0x4a27] = 0x6c8c, [0x4a28] = 0x6c68,
  [0x4a29] = 0x6c69, [0x4a2a] = 0x6c74, [0x4a2b] = 0x6c76, [0x4a2c] = 0x6c86,
  [0x4a2d] = 0x6ca9, [0x4a2e] = 0x6cd0, [0x4a2f] = 0x6cd4, [0x4a30] = 0x6cad,
  [0x4a31] = 0x6cf7, [0x4a32] = 0x6cf8, [0x4a33] = 0x6cf1, [0x4a34] = 0x6cd7,
  [0x4a35] = 0x6cb2, [0x4a36] = 0x6ce0, [0x4a37] = 0x6cd6, [0x4a38] = 0x6cfa,
  [0x4a39] = 0x6ceb, [0x4a3a] = 0x6cee, [0x4a3b] = 0x6cb1, [0x4a3c] = 0x6cd3,
  [0x4a3d] = 0x6cef, [0x4a3e] = 0x6cfe, [0x4a40] = 0x92a8, [0x4a41] = 0x92a9,
  [0x4a42] = 0x92aa, [0x4a43] = 0x92ab, [0x4a44] = 0x92ac, [0x4a45] = 0x92ad,
  [0x4a46] = 0x92af, [0x4a47] = 0x92b0, [0x4a48] = 0x92b1, [0x4a49] = 0x92b2,
  [0x4a4a] = 0x92b3, [0x4a4b] = 0x92b4, [0x4a4c] = 0x92b5, [0x4a4d] = 0x92b6,
  [0x4a4e] = 0x92b7, [0x4a4f] = 0x92b8, [0x4a50] = 0x92b9, [0x4a51] = 0x92ba,
  [0x4a52] = 0x92bb, [0x4a53] = 0x92bc, [0x4a54] = 0x92bd, [0x4a55] = 0x92be,
  [0x4a56] = 0x92bf, [0x4a57] = 0x92c0, [0x4a58] = 0x92c1, [0x4a59] = 0x92c2,
  [0x4a5a] = 0x92c3, [0x4a5b] = 0x92c4, [0x4a5c] = 0x92c5, [0x4a5d] = 0x92c6,
  [0x4a5e] = 0x92c7, [0x4a5f] = 0x92c9, [0x4a60] = 0x92ca, [0x4a61] = 0x92cb,
  [0x4a62] = 0x92cc, [0x4a63] = 0x92cd, [0x4a64] = 0x92ce, [0x4a65] = 0x92cf,
  [0x4a66] = 0x92d0, [0x4a67] = 0x92d1, [0x4a68] = 0x92d2, [0x4a69] = 0x92d3,
  [0x4a6a] = 0x92d4, [0x4a6b] = 0x92d5, [0x4a6c] = 0x92d6, [0x4a6d] = 0x92d7,
  [0x4a6e] = 0x92d8, [0x4a6f] = 0x92d9, [0x4a70] = 0x92da, [0x4a71] = 0x92db,
  [0x4a72] = 0x92dc, [0x4a73] = 0x92dd, [0x4a74] = 0x92de, [0x4a75] = 0x92df,
  [0x4a76] = 0x92e0, [0x4a77] = 0x92e1, [0x4a78] = 0x92e2, [0x4a79] = 0x92e3,
  [0x4a7a] = 0x92e4, [0x4a7b] = 0x92e5, [0x4a7c] = 0x92e6, [0x4a7d] = 0x92e7,
  [0x4a7e] = 0x92e8, [0x4a80] = 0x92e9, [0x4a81] = 0x92ea, [0x4a82] = 0x92eb,
  [0x4a83] = 0x92ec, [0x4a84] = 0x92ed, [0x4a85] = 0x92ee, [0x4a86] = 0x92ef,
  [0x4a87] = 0x92f0, [0x4a88] = 0x92f1, [0x4a89] = 0x92f2, [0x4a8a] = 0x92f3,
  [0x4a8b] = 0x92f4, [0x4a8c] = 0x92f5, [0x4a8d] = 0x92f6, [0x4a8e] = 0x92f7,
  [0x4a8f] = 0x92f8, [0x4a90] = 0x92f9, [0x4a91] = 0x92fa, [0x4a92] = 0x92fb,
  [0x4a93] = 0x92fc, [0x4a94] = 0x92fd, [0x4a95] = 0x92fe, [0x4a96] = 0x92ff,
  [0x4a97] = 0x9300, [0x4a98] = 0x9301, [0x4a99] = 0x9302, [0x4a9a] = 0x9303,
  [0x4a9b] = 0x9304, [0x4a9c] = 0x9305, [0x4a9d] = 0x9306, [0x4a9e] = 0x9307,
  [0x4a9f] = 0x9308, [0x4aa0] = 0x9309, [0x4aa1] = 0x6d39, [0x4aa2] = 0x6d27,
  [0x4aa3] = 0x6d0c, [0x4aa4] = 0x6d43, [0x4aa5] = 0x6d48, [0x4aa6] = 0x6d07,
  [0x4aa7] = 0x6d04, [0x4aa8] = 0x6d19, [0x4aa9] = 0x6d0e, [0x4aaa] = 0x6d2b,
  [0x4aab] = 0x6d4d, [0x4aac] = 0x6d2e, [0x4aad] = 0x6d35, [0x4aae] = 0x6d1a,
  [0x4aaf] = 0x6d4f, [0x4ab0] = 0x6d52, [0x4ab1] = 0x6d54, [0x4ab2] = 0x6d33,
  [0x4ab3] = 0x6d91, [0x4ab4] = 0x6d6f, [0x4ab5] = 0x6d9e, [0x4ab6] = 0x6da0,
  [0x4ab7] = 0x6d5e, [0x4ab8] = 0x6d93, [0x4ab9] = 0x6d94, [0x4aba] = 0x6d5c,
  [0x4abb] = 0x6d60, [0x4abc] = 0x6d7c, [0x4abd] = 0x6d63, [0x4abe] = 0x6e1a,
  [0x4abf] = 0x6dc7, [0x4ac0] = 0x6dc5, [0x4ac1] = 0x6dde, [0x4ac2] = 0x6e0e,
  [0x4ac3] = 0x6dbf, [0x4ac4] = 0x6de0, [0x4ac5] = 0x6e11, [0x4ac6] = 0x6de6,
  [0x4ac7] = 0x6ddd, [0x4ac8] = 0x6dd9, [0x4ac9] = 0x6e16, [0x4aca] = 0x6dab,
  [0x4acb] = 0x6e0c, [0x4acc] = 0x6dae, [0x4acd] = 0x6e2b, [0x4ace] = 0x6e6e,
  [0x4acf] = 0x6e4e, [0x4ad0] = 0x6e6b, [0x4ad1] = 0x6eb2, [0x4ad2] = 0x6e5f,
  [0x4ad3] = 0x6e86, [0x4ad4] = 0x6e53, [0x4ad5] = 0x6e54, [0x4ad6] = 0x6e32,
  [0x4ad7] = 0x6e25, [0x4ad8] = 0x6e44, [0x4ad9] = 0x6edf, [0x4ada] = 0x6eb1,
  [0x4adb] = 0x6e98, [0x4adc] = 0x6ee0, [0x4add] = 0x6f2d, [0x4ade] = 0x6ee2,
  [0x4adf] = 0x6ea5, [0x4ae0] = 0x6ea7, [0x4ae1] = 0x6ebd, [0x4ae2] = 0x6ebb,
  [0x4ae3] = 0x6eb7, [0x4ae4] = 0x6ed7, [0x4ae5] = 0x6eb4, [0x4ae6] = 0x6ecf,
  [0x4ae7] = 0x6e8f, [0x4ae8] = 0x6ec2, [0x4ae9] = 0x6e9f, [0x4aea] = 0x6f62,
  [0x4aeb] = 0x6f46, [0x4aec] = 0x6f47, [0x4aed] = 0x6f24, [0x4aee] = 0x6f15,
  [0x4aef] = 0x6ef9, [0x4af0] = 0x6f2f, [0x4af1] = 0x6f36, [0x4af2] = 0x6f4b,
  [0x4af3] = 0x6f74, [0x4af4] = 0x6f2a, [0x4af5] = 0x6f09, [0x4af6] = 0x6f29,
  [0x4af7] = 0x6f89, [0x4af8] = 0x6f8d, [0x4af9] = 0x6f8c, [0x4afa] = 0x6f78,
  [0x4afb] = 0x6f72, [0x4afc] = 0x6f7c, [0x4afd] = 0x6f7a, [0x4afe] = 0x6fd1,
  [0x4b00] = 0x930a, [0x4b01] = 0x930b, [0x4b02] = 0x930c, [0x4b03] = 0x930d,
  [0x4b04] = 0x930e, [0x4b05] = 0x930f, [0x4b06] = 0x9310, [0x4b07] = 0x9311,
  [0x4b08] = 0x9312, [0x4b09] = 0x9313, [0x4b0a] = 0x9314, [0x4b0b] = 0x9315,
  [0x4b0c] = 0x9316, [0x4b0d] = 0x9317, [0x4b0e] = 0x9318, [0x4b0f] = 0x9319,
  [0x4b10] = 0x931a, [0x4b11] = 0x931b, [0x4b12] = 0x931c, [0x4b13] = 0x931d,
  [0x4b14] = 0x931e, [0x4b15] = 0x931f, [0x4b16] = 0x9320, [0x4b17] = 0x9321,
  [0x4b18] = 0x9322, [0x4b19] = 0x9323, [0x4b1a] = 0x9324, [0x4b1b] = 0x9325,
  [0x4b1c] = 0x9326, [0x4b1d] = 0x9327, [0x4b1e] = 0x9328, [0x4b1f] = 0x9329,
  [0x4b20] = 0x932a, [0x4b21] = 0x932b, [0x4b22] = 0x932c, [0x4b23] = 0x932d,
  [0x4b24] = 0x932e, [0x4b25] = 0x932f, [0x4b26] = 0x9330, [0x4b27] = 0x9331,
  [0x4b28] = 0x9332, [0x4b29] = 0x9333, [0x4b2a] = 0x9334, [0x4b2b] = 0x9335,
  [0x4b2c] = 0x9336, [0x4b2d] = 0x9337, [0x4b2e] = 0x9338, [0x4b2f] = 0x9339,
  [0x4b30] = 0x933a, [0x4b31] = 0x933b, [0x4b32] = 0x933c, [0x4b33] = 0x933d,
  [0x4b34] = 0x933f, [0x4b35] = 0x9340, [0x4b36] = 0x9341, [0x4b37] = 0x9342,
  [0x4b38] = 0x9343, [0x4b39] = 0x9344, [0x4b3a] = 0x9345, [0x4b3b] = 0x9346,
  [0x4b3c] = 0x9347, [0x4b3d] = 0x9348, [0x4b3e] = 0x9349, [0x4b40] = 0x934a,
  [0x4b41] = 0x934b, [0x4b42] = 0x934c, [0x4b43] = 0x934d, [0x4b44] = 0x934e,
  [0x4b45] = 0x934f, [0x4b46] = 0x9350, [0x4b47] = 0x9351, [0x4b48] = 0x9352,
  [0x4b49] = 0x9353, [0x4b4a] = 0x9354, [0x4b4b] = 0x9355, [0x4b4c] = 0x9356,
  [0x4b4d] = 0x9357, [0x4b4e] = 0x9358, [0x4b4f] = 0x9359, [0x4b50] = 0x935a,
  [0x4b51] = 0x935b, [0x4b52] = 0x935c, [0x4b53] = 0x935d, [0x4b54] = 0x935e,
  [0x4b55] = 0x935f, [0x4b56] = 0x9360, [0x4b57] = 0x9361, [0x4b58] = 0x9362,
  [0x4b59] = 0x9363, [0x4b5a] = 0x9364, [0x4b5b] = 0x9365, [0x4b5c] = 0x9366,
  [0x4b5d] = 0x9367, [0x4b5e] = 0x9368, [0x4b5f] = 0x9369, [0x4b60] = 0x936b,
  [0x4b61] = 0x6fc9, [0x4b62] = 0x6fa7, [0x4b63] = 0x6fb9, [0x4b64] = 0x6fb6,
  [0x4b65] = 0x6fc2, [0x4b66] = 0x6fe1, [0x4b67] = 0x6fee, [0x4b68] = 0x6fde,
  [0x4b69] = 0x6fe0, [0x4b6a] = 0x6fef, [0x4b6b] = 0x701a, [0x4b6c] = 0x7023,
  [0x4b6d] = 0x701b, [0x4b6e] = 0x7039, [0x4b6f] = 0x7035, [0x4b70] = 0x704f,
  [0x4b71] = 0x705e, [0x4b72] = 0x5b80, [0x4b73] = 0x5b84, [0x4b74] = 0x5b95,
  [0x4b75] = 0x5b93, [0x4b76] = 0x5ba5, [0x4b77] = 0x5bb8, [0x4b78] = 0x752f,
  [0x4b79] = 0x9a9e, [0x4b7a] = 0x6434, [0x4b7b] = 0x5be4, [0x4b7c] = 0x5bee,
  [0x4b7d] = 0x8930, [0x4b7e] = 0x5bf0, [0x4b7f] = 0x8e47, [0x4b80] = 0x8b07,
  [0x4b81] = 0x8fb6, [0x4b82] = 0x8fd3, [0x4b83] = 0x8fd5, [0x4b84] = 0x8fe5,
  [0x4b85] = 0x8fee, [0x4b86] = 0x8fe4, [0x4b87] = 0x8fe9, [0x4b88] = 0x8fe6,
  [0x4b89] = 0x8ff3, [0x4b8a] = 0x8fe8, [0x4b8b] = 0x9005, [0x4b8c] = 0x9004,
  [0x4b8d] = 0x900b, [0x4b8e] = 0x9026, [0x4b8f] = 0x9011, [0x4b90] = 0x900d,
  [0x4b91] = 0x9016, [0x4b92] = 0x9021, [0x4b93] = 0x9035, [0x4b94] = 0x9036,
  [0x4b95] = 0x902d, [0x4b96] = 0x902f, [0x4b97] = 0x9044, [0x4b98] = 0x9051,
  [0x4b99] = 0x9052, [0x4b9a] = 0x9050, [0x4b9b] = 0x9068, [0x4b9c] = 0x9058,
  [0x4b9d] = 0x9062, [0x4b9e] = 0x905b, [0x4b9f] = 0x66b9, [0x4ba0] = 0x9074,
  [0x4ba1] = 0x907d, [0x4ba2] = 0x9082, [0x4ba3] = 0x9088, [0x4ba4] = 0x9083,
  [0x4ba5] = 0x908b, [0x4ba6] = 0x5f50, [0x4ba7] = 0x5f57, [0x4ba8] = 0x5f56,
  [0x4ba9] = 0x5f58, [0x4baa] = 0x5c3b, [0x4bab] = 0x54ab, [0x4bac] = 0x5c50,
  [0x4bad] = 0x5c59, [0x4bae] = 0x5b71, [0x4baf] = 0x5c63, [0x4bb0] = 0x5c66,
  [0x4bb1] = 0x7fbc, [0x4bb2] = 0x5f2a, [0x4bb3] = 0x5f29, [0x4bb4] = 0x5f2d,
  [0x4bb5] = 0x8274, [0x4bb6] = 0x5f3c, [0x4bb7] = 0x9b3b, [0x4bb8] = 0x5c6e,
  [0x4bb9] = 0x5981, [0x4bba] = 0x5983, [0x4bbb] = 0x598d, [0x4bbc] = 0x59a9,
  [0x4bbd] = 0x59aa, [0x4bbe] = 0x59a3, [0x4bc0] = 0x936c, [0x4bc1] = 0x936d,
  [0x4bc2] = 0x936e, [0x4bc3] = 0x936f, [0x4bc4] = 0x9370, [0x4bc5] = 0x9371,
  [0x4bc6] = 0x9372, [0x4bc7] = 0x9373, [0x4bc8] = 0x9374, [0x4bc9] = 0x9375,
  [0x4bca] = 0x9376, [0x4bcb] = 0x9377, [0x4bcc] = 0x9378, [0x4bcd] = 0x9379,
  [0x4bce] = 0x937a, [0x4bcf] = 0x937b, [0x4bd0] = 0x937c, [0x4bd1] = 0x937d,
  [0x4bd2] = 0x937e, [0x4bd3] = 0x937f, [0x4bd4] = 0x9380, [0x4bd5] = 0x9381,
  [0x4bd6] = 0x9382, [0x4bd7] = 0x9383, [0x4bd8] = 0x9384, [0x4bd9] = 0x9385,
  [0x4bda] = 0x9386, [0x4bdb] = 0x9387, [0x4bdc] = 0x9388, [0x4bdd] = 0x9389,
  [0x4bde] = 0x938a, [0x4bdf] = 0x938b, [0x4be0] = 0x938c, [0x4be1] = 0x938d,
  [0x4be2] = 0x938e, [0x4be3] = 0x9390, [0x4be4] = 0x9391, [0x4be5] = 0x9392,
  [0x4be6] = 0x9393, [0x4be7] = 0x9394, [0x4be8] = 0x9395, [0x4be9] = 0x9396,
  [0x4bea] = 0x9397, [0x4beb] = 0x9398, [0x4bec] = 0x9399, [0x4bed] = 0x939a,
  [0x4bee] = 0x939b, [0x4bef] = 0x939c, [0x4bf0] = 0x939d, [0x4bf1] = 0x939e,
  [0x4bf2] = 0x939f, [0x4bf3] = 0x93a0, [0x4bf4] = 0x93a1, [0x4bf5] = 0x93a2,
  [0x4bf6] = 0x93a3, [0x4bf7] = 0x93a4, [0x4bf8] = 0x93a5, [0x4bf9] = 0x93a6,
  [0x4bfa] = 0x93a7, [0x4bfb] = 0x93a8, [0x4bfc] = 0x93a9, [0x4bfd] = 0x93aa,
  [0x4bfe] = 0x93ab, [0x4c00] = 0x93ac, [0x4c01] = 0x93ad, [0x4c02] = 0x93ae,
  [0x4c03] = 0x93af, [0x4c04] = 0x93b0, [0x4c05] = 0x93b1, [0x4c06] = 0x93b2,
  [0x4c07] = 0x93b3, [0x4c08] = 0x93b4, [0x4c09] = 0x93b5, [0x4c0a] = 0x93b6,
  [0x4c0b] = 0x93b7, [0x4c0c] = 0x93b8, [0x4c0d] = 0x93b9, [0x4c0e] = 0x93ba,
  [0x4c0f] = 0x93bb, [0x4c10] = 0x93bc, [0x4c11] = 0x93bd, [0x4c12] = 0x93be,
  [0x4c13] = 0x93bf, [0x4c14] = 0x93c0, [0x4c15] = 0x93c1, [0x4c16] = 0x93c2,
  [0x4c17] = 0x93c3, [0x4c18] = 0x93c4, [0x4c19] = 0x93c5, [0x4c1a] = 0x93c6,
  [0x4c1b] = 0x93c7, [0x4c1c] = 0x93c8, [0x4c1d] = 0x93c9, [0x4c1e] = 0x93cb,
  [0x4c1f] = 0x93cc, [0x4c20] = 0x93cd, [0x4c21] = 0x5997, [0x4c22] = 0x59ca,
  [0x4c23] = 0x59ab, [0x4c24] = 0x599e, [0x4c25] = 0x59a4, [0x4c26] = 0x59d2,
  [0x4c27] = 0x59b2, [0x4c28] = 0x59af, [0x4c29] = 0x59d7, [0x4c2a] = 0x59be,
  [0x4c2b] = 0x5a05, [0x4c2c] = 0x5a06, [0x4c2d] = 0x59dd, [0x4c2e] = 0x5a08,
  [0x4c2f] = 0x59e3, [0x4c30] = 0x59d8, [0x4c31] = 0x59f9, [0x4c32] = 0x5a0c,
  [0x4c33] = 0x5a09, [0x4c34] = 0x5a32, [0x4c35] = 0x5a34, [0x4c36] = 0x5a11,
  [0x4c37] = 0x5a23, [0x4c38] = 0x5a13, [0x4c39] = 0x5a40, [0x4c3a] = 0x5a67,
  [0x4c3b] = 0x5a4a, [0x4c3c] = 0x5a55, [0x4c3d] = 0x5a3c, [0x4c3e] = 0x5a62,
  [0x4c3f] = 0x5a75, [0x4c40] = 0x80ec, [0x4c41] = 0x5aaa, [0x4c42] = 0x5a9b,
  [0x4c43] = 0x5a77, [0x4c44] = 0x5a7a, [0x4c45] = 0x5abe, [0x4c46] = 0x5aeb,
  [0x4c47] = 0x5ab2, [0x4c48] = 0x5ad2, [0x4c49] = 0x5ad4, [0x4c4a] = 0x5ab8,
  [0x4c4b] = 0x5ae0, [0x4c4c] = 0x5ae3, [0x4c4d] = 0x5af1, [0x4c4e] = 0x5ad6,
  [0x4c4f] = 0x5ae6, [0x4c50] = 0x5ad8, [0x4c51] = 0x5adc, [0x4c52] = 0x5b09,
  [0x4c53] = 0x5b17, [0x4c54] = 0x5b16, [0x4c55] = 0x5b32, [0x4c56] = 0x5b37,
  [0x4c57] = 0x5b40, [0x4c58] = 0x5c15, [0x4c59] = 0x5c1c, [0x4c5a] = 0x5b5a,
  [0x4c5b] = 0x5b65, [0x4c5c] = 0x5b73, [0x4c5d] = 0x5b51, [0x4c5e] = 0x5b53,
  [0x4c5f] = 0x5b62, [0x4c60] = 0x9a75, [0x4c61] = 0x9a77, [0x4c62] = 0x9a78,
  [0x4c63] = 0x9a7a, [0x4c64] = 0x9a7f, [0x4c65] = 0x9a7d, [0x4c66] = 0x9a80,
  [0x4c67] = 0x9a81, [0x4c68] = 0x9a85, [0x4c69] = 0x9a88, [0x4c6a] = 0x9a8a,
  [0x4c6b] = 0x9a90, [0x4c6c] = 0x9a92, [0x4c6d] = 0x9a93, [0x4c6e] = 0x9a96,
  [0x4c6f] = 0x9a98, [0x4c70] = 0x9a9b, [0x4c71] = 0x9a9c, [0x4c72] = 0x9a9d,
  [0x4c73] = 0x9a9f, [0x4c74] = 0x9aa0, [0x4c75] = 0x9aa2, [0x4c76] = 0x9aa3,
  [0x4c77] = 0x9aa5, [0x4c78] = 0x9aa7, [0x4c79] = 0x7e9f, [0x4c7a] = 0x7ea1,
  [0x4c7b] = 0x7ea3, [0x4c7c] = 0x7ea5, [0x4c7d] = 0x7ea8, [0x4c7e] = 0x7ea9,
  [0x4c80] = 0x93ce, [0x4c81] = 0x93cf, [0x4c82] = 0x93d0, [0x4c83] = 0x93d1,
  [0x4c84] = 0x93d2, [0x4c85] = 0x93d3, [0x4c86] = 0x93d4, [0x4c87] = 0x93d5,
  [0x4c88] = 0x93d7, [0x4c89] = 0x93d8, [0x4c8a] = 0x93d9, [0x4c8b] = 0x93da,
  [0x4c8c] = 0x93db, [0x4c8d] = 0x93dc, [0x4c8e] = 0x93dd, [0x4c8f] = 0x93de,
  [0x4c90] = 0x93df, [0x4c91] = 0x93e0, [0x4c92] = 0x93e1, [0x4c93] = 0x93e2,
  [0x4c94] = 0x93e3, [0x4c95] = 0x93e4, [0x4c96] = 0x93e5, [0x4c97] = 0x93e6,
  [0x4c98] = 0x93e7, [0x4c99] = 0x93e8, [0x4c9a] = 0x93e9, [0x4c9b] = 0x93ea,
  [0x4c9c] = 0x93eb, [0x4c9d] = 0x93ec, [0x4c9e] = 0x93ed, [0x4c9f] = 0x93ee,
  [0x4ca0] = 0x93ef, [0x4ca1] = 0x93f0, [0x4ca2] = 0x93f1, [0x4ca3] = 0x93f2,
  [0x4ca4] = 0x93f3, [0x4ca5] = 0x93f4, [0x4ca6] = 0x93f5, [0x4ca7] = 0x93f6,
  [0x4ca8] = 0x93f7, [0x4ca9] = 0x93f8, [0x4caa] = 0x93f9, [0x4cab] = 0x93fa,
  [0x4cac] = 0x93fb, [0x4cad] = 0x93fc, [0x4cae] = 0x93fd, [0x4caf] = 0x93fe,
  [0x4cb0] = 0x93ff, [0x4cb1] = 0x9400, [0x4cb2] = 0x9401, [0x4cb3] = 0x9402,
  [0x4cb4] = 0x9403, [0x4cb5] = 0x9404, [0x4cb6] = 0x9405, [0x4cb7] = 0x9406,
  [0x4cb8] = 0x9407, [0x4cb9] = 0x9408, [0x4cba] = 0x9409, [0x4cbb] = 0x940a,
  [0x4cbc] = 0x940b, [0x4cbd] = 0x940c, [0x4cbe] = 0x940d, [0x4cc0] = 0x940e,
  [0x4cc1] = 0x940f, [0x4cc2] = 0x9410, [0x4cc3] = 0x9411, [0x4cc4] = 0x9412,
  [0x4cc5] = 0x9413, [0x4cc6] = 0x9414, [0x4cc7] = 0x9415, [0x4cc8] = 0x9416,
  [0x4cc9] = 0x9417, [0x4cca] = 0x9418, [0x4ccb] = 0x9419, [0x4ccc] = 0x941a,
  [0x4ccd] = 0x941b, [0x4cce] = 0x941c, [0x4ccf] = 0x941d, [0x4cd0] = 0x941e,
  [0x4cd1] = 0x941f, [0x4cd2] = 0x9420, [0x4cd3] = 0x9421, [0x4cd4] = 0x9422,
  [0x4cd5] = 0x9423, [0x4cd6] = 0x9424, [0x4cd7] = 0x9425, [0x4cd8] = 0x9426,
  [0x4cd9] = 0x9427, [0x4cda] = 0x9428, [0x4cdb] = 0x9429, [0x4cdc] = 0x942a,
  [0x4cdd] = 0x942b, [0x4cde] = 0x942c, [0x4cdf] = 0x942d, [0x4ce0] = 0x942e,
  [0x4ce1] = 0x7ead, [0x4ce2] = 0x7eb0, [0x4ce3] = 0x7ebe, [0x4ce4] = 0x7ec0,
  [0x4ce5] = 0x7ec1, [0x4ce6] = 0x7ec2, [0x4ce7] = 0x7ec9, [0x4ce8] = 0x7ecb,
  [0x4ce9] = 0x7ecc, [0x4cea] = 0x7ed0, [0x4ceb] = 0x7ed4, [0x4cec] = 0x7ed7,
  [0x4ced] = 0x7edb, [0x4cee] = 0x7ee0, [0x4cef] = 0x7ee1, [0x4cf0] = 0x7ee8,
  [0x4cf1] = 0x7eeb, [0x4cf2] = 0x7eee, [0x4cf3] = 0x7eef, [0x4cf4] = 0x7ef1,
  [0x4cf5] = 0x7ef2, [0x4cf6] = 0x7f0d, [0x4cf7] = 0x7ef6, [0x4cf8] = 0x7efa,
  [0x4cf9] = 0x7efb, [0x4cfa] = 0x7efe, [0x4cfb] = 0x7f01, [0x4cfc] = 0x7f02,
  [0x4cfd] = 0x7f03, [0x4cfe] = 0x7f07, [0x4cff] = 0x7f08, [0x4d00] = 0x7f0b,
  [0x4d01] = 0x7f0c, [0x4d02] = 0x7f0f, [0x4d03] = 0x7f11, [0x4d04] = 0x7f12,
  [0x4d05] = 0x7f17, [0x4d06] = 0x7f19, [0x4d07] = 0x7f1c, [0x4d08] = 0x7f1b,
  [0x4d09] = 0x7f1f, [0x4d0a] = 0x7f21, [0x4d0b] = 0x7f22, [0x4d0c] = 0x7f23,
  [0x4d0d] = 0x7f24, [0x4d0e] = 0x7f25, [0x4d0f] = 0x7f26, [0x4d10] = 0x7f27,
  [0x4d11] = 0x7f2a, [0x4d12] = 0x7f2b, [0x4d13] = 0x7f2c, [0x4d14] = 0x7f2d,
  [0x4d15] = 0x7f2f, [0x4d16] = 0x7f30, [0x4d17] = 0x7f31, [0x4d18] = 0x7f32,
  [0x4d19] = 0x7f33, [0x4d1a] = 0x7f35, [0x4d1b] = 0x5e7a, [0x4d1c] = 0x757f,
  [0x4d1d] = 0x5ddb, [0x4d1e] = 0x753e, [0x4d1f] = 0x9095, [0x4d20] = 0x738e,
  [0x4d21] = 0x7391, [0x4d22] = 0x73ae, [0x4d23] = 0x73a2, [0x4d24] = 0x739f,
  [0x4d25] = 0x73cf, [0x4d26] = 0x73c2, [0x4d27] = 0x73d1, [0x4d28] = 0x73b7,
  [0x4d29] = 0x73b3, [0x4d2a] = 0x73c0, [0x4d2b] = 0x73c9, [0x4d2c] = 0x73c8,
  [0x4d2d] = 0x73e5, [0x4d2e] = 0x73d9, [0x4d2f] = 0x987c, [0x4d30] = 0x740a,
  [0x4d31] = 0x73e9, [0x4d32] = 0x73e7, [0x4d33] = 0x73de, [0x4d34] = 0x73ba,
  [0x4d35] = 0x73f2, [0x4d36] = 0x740f, [0x4d37] = 0x742a, [0x4d38] = 0x745b,
  [0x4d39] = 0x7426, [0x4d3a] = 0x7425, [0x4d3b] = 0x7428, [0x4d3c] = 0x7430,
  [0x4d3d] = 0x742e, [0x4d3e] = 0x742c, [0x4d40] = 0x942f, [0x4d41] = 0x9430,
  [0x4d42] = 0x9431, [0x4d43] = 0x9432, [0x4d44] = 0x9433, [0x4d45] = 0x9434,
  [0x4d46] = 0x9435, [0x4d47] = 0x9436, [0x4d48] = 0x9437, [0x4d49] = 0x9438,
  [0x4d4a] = 0x9439, [0x4d4b] = 0x943a, [0x4d4c] = 0x943b, [0x4d4d] = 0x943c,
  [0x4d4e] = 0x943d, [0x4d4f] = 0x943f, [0x4d50] = 0x9440, [0x4d51] = 0x9441,
  [0x4d52] = 0x9442, [0x4d53] = 0x9443, [0x4d54] = 0x9444, [0x4d55] = 0x9445,
  [0x4d56] = 0x9446, [0x4d57] = 0x9447, [0x4d58] = 0x9448, [0x4d59] = 0x9449,
  [0x4d5a] = 0x944a, [0x4d5b] = 0x944b, [0x4d5c] = 0x944c, [0x4d5d] = 0x944d,
  [0x4d5e] = 0x944e, [0x4d5f] = 0x944f, [0x4d60] = 0x9450, [0x4d61] = 0x9451,
  [0x4d62] = 0x9452, [0x4d63] = 0x9453, [0x4d64] = 0x9454, [0x4d65] = 0x9455,
  [0x4d66] = 0x9456, [0x4d67] = 0x9457, [0x4d68] = 0x9458, [0x4d69] = 0x9459,
  [0x4d6a] = 0x945a, [0x4d6b] = 0x945b, [0x4d6c] = 0x945c, [0x4d6d] = 0x945d,
  [0x4d6e] = 0x945e, [0x4d6f] = 0x945f, [0x4d70] = 0x9460, [0x4d71] = 0x9461,
  [0x4d72] = 0x9462, [0x4d73] = 0x9463, [0x4d74] = 0x9464, [0x4d75] = 0x9465,
  [0x4d76] = 0x9466, [0x4d77] = 0x9467, [0x4d78] = 0x9468, [0x4d79] = 0x9469,
  [0x4d7a] = 0x946a, [0x4d7b] = 0x946c, [0x4d7c] = 0x946d, [0x4d7d] = 0x946e,
  [0x4d7e] = 0x946f, [0x4d80] = 0x9470, [0x4d81] = 0x9471, [0x4d82] = 0x9472,
  [0x4d83] = 0x9473, [0x4d84] = 0x9474, [0x4d85] = 0x9475, [0x4d86] = 0x9476,
  [0x4d87] = 0x9477, [0x4d88] = 0x9478, [0x4d89] = 0x9479, [0x4d8a] = 0x947a,
  [0x4d8b] = 0x947b, [0x4d8c] = 0x947c, [0x4d8d] = 0x947d, [0x4d8e] = 0x947e,
  [0x4d8f] = 0x947f, [0x4d90] = 0x9480, [0x4d91] = 0x9481, [0x4d92] = 0x9482,
  [0x4d93] = 0x9483, [0x4d94] = 0x9484, [0x4d95] = 0x9491, [0x4d96] = 0x9496,
  [0x4d97] = 0x9498, [0x4d98] = 0x94c7, [0x4d99] = 0x94cf, [0x4d9a] = 0x94d3,
  [0x4d9b] = 0x94d4, [0x4d9c] = 0x94da, [0x4d9d] = 0x94e6, [0x4d9e] = 0x94fb,
  [0x4d9f] = 0x951c, [0x4da0] = 0x9520, [0x4da1] = 0x741b, [0x4da2] = 0x741a,
  [0x4da3] = 0x7441, [0x4da4] = 0x745c, [0x4da5] = 0x7457, [0x4da6] = 0x7455,
  [0x4da7] = 0x7459, [0x4da8] = 0x7477, [0x4da9] = 0x746d, [0x4daa] = 0x747e,
  [0x4dab] = 0x749c, [0x4dac] = 0x748e, [0x4dad] = 0x7480, [0x4dae] = 0x7481,
  [0x4daf] = 0x7487, [0x4db0] = 0x748b, [0x4db1] = 0x749e, [0x4db2] = 0x74a8,
  [0x4db3] = 0x74a9, [0x4db4] = 0x7490, [0x4db5] = 0x74a7, [0x4db6] = 0x74d2,
  [0x4db7] = 0x74ba, [0x4db8] = 0x97ea, [0x4db9] = 0x97eb, [0x4dba] = 0x97ec,
  [0x4dbb] = 0x674c, [0x4dbc] = 0x6753, [0x4dbd] = 0x675e, [0x4dbe] = 0x6748,
  [0x4dbf] = 0x6769, [0x4dc0] = 0x67a5, [0x4dc1] = 0x6787, [0x4dc2] = 0x676a,
  [0x4dc3] = 0x6773, [0x4dc4] = 0x6798, [0x4dc5] = 0x67a7, [0x4dc6] = 0x6775,
  [0x4dc7] = 0x67a8, [0x4dc8] = 0x679e, [0x4dc9] = 0x67ad, [0x4dca] = 0x678b,
  [0x4dcb] = 0x6777, [0x4dcc] = 0x677c, [0x4dcd] = 0x67f0, [0x4dce] = 0x6809,
  [0x4dcf] = 0x67d8, [0x4dd0] = 0x680a, [0x4dd1] = 0x67e9, [0x4dd2] = 0x67b0,
  [0x4dd3] = 0x680c, [0x4dd4] = 0x67d9, [0x4dd5] = 0x67b5, [0x4dd6] = 0x67da,
  [0x4dd7] = 0x67b3, [0x4dd8] = 0x67dd, [0x4dd9] = 0x6800, [0x4dda] = 0x67c3,
  [0x4ddb] = 0x67b8, [0x4ddc] = 0x67e2, [0x4ddd] = 0x680e, [0x4dde] = 0x67c1,
  [0x4ddf] = 0x67fd, [0x4de0] = 0x6832, [0x4de1] = 0x6833, [0x4de2] = 0x6860,
  [0x4de3] = 0x6861, [0x4de4] = 0x684e, [0x4de5] = 0x6862, [0x4de6] = 0x6844,
  [0x4de7] = 0x6864, [0x4de8] = 0x6883, [0x4de9] = 0x681d, [0x4dea] = 0x6855,
  [0x4deb] = 0x6866, [0x4dec] = 0x6841, [0x4ded] = 0x6867, [0x4dee] = 0x6840,
  [0x4def] = 0x683e, [0x4df0] = 0x684a, [0x4df1] = 0x6849, [0x4df2] = 0x6829,
  [0x4df3] = 0x68b5, [0x4df4] = 0x688f, [0x4df5] = 0x6874, [0x4df6] = 0x6877,
  [0x4df7] = 0x6893, [0x4df8] = 0x686b, [0x4df9] = 0x68c2, [0x4dfa] = 0x696e,
  [0x4dfb] = 0x68fc, [0x4dfc] = 0x691f, [0x4dfd] = 0x6920, [0x4dfe] = 0x68f9,
  [0x4e00] = 0x9527, [0x4e01] = 0x9533, [0x4e02] = 0x953d, [0x4e03] = 0x9543,
  [0x4e04] = 0x9548, [0x4e05] = 0x954b, [0x4e06] = 0x9555, [0x4e07] = 0x955a,
  [0x4e08] = 0x9560, [0x4e09] = 0x956e, [0x4e0a] = 0x9574, [0x4e0b] = 0x9575,
  [0x4e0c] = 0x9577, [0x4e0d] = 0x9578, [0x4e0e] = 0x9579, [0x4e0f] = 0x957a,
  [0x4e10] = 0x957b, [0x4e11] = 0x957c, [0x4e12] = 0x957d, [0x4e13] = 0x957e,
  [0x4e14] = 0x9580, [0x4e15] = 0x9581, [0x4e16] = 0x9582, [0x4e17] = 0x9583,
  [0x4e18] = 0x9584, [0x4e19] = 0x9585, [0x4e1a] = 0x9586, [0x4e1b] = 0x9587,
  [0x4e1c] = 0x9588, [0x4e1d] = 0x9589, [0x4e1e] = 0x958a, [0x4e1f] = 0x958b,
  [0x4e20] = 0x958c, [0x4e21] = 0x958d, [0x4e22] = 0x958e, [0x4e23] = 0x958f,
  [0x4e24] = 0x9590, [0x4e25] = 0x9591, [0x4e26] = 0x9592, [0x4e27] = 0x9593,
  [0x4e28] = 0x9594, [0x4e29] = 0x9595, [0x4e2a] = 0x9596, [0x4e2b] = 0x9597,
  [0x4e2c] = 0x9598, [0x4e2d] = 0x9599, [0x4e2e] = 0x959a, [0x4e2f] = 0x959b,
  [0x4e30] = 0x959c, [0x4e31] = 0x959d, [0x4e32] = 0x959e, [0x4e33] = 0x959f,
  [0x4e34] = 0x95a0, [0x4e35] = 0x95a1, [0x4e36] = 0x95a2, [0x4e37] = 0x95a3,
  [0x4e38] = 0x95a4, [0x4e39] = 0x95a5, [0x4e3a] = 0x95a6, [0x4e3b] = 0x95a7,
  [0x4e3c] = 0x95a8, [0x4e3d] = 0x95a9, [0x4e3e] = 0x95aa, [0x4e40] = 0x95ab,
  [0x4e41] = 0x95ac, [0x4e42] = 0x95ad, [0x4e43] = 0x95ae, [0x4e44] = 0x95af,
  [0x4e45] = 0x95b0, [0x4e46] = 0x95b1, [0x4e47] = 0x95b2, [0x4e48] = 0x95b3,
  [0x4e49] = 0x95b4, [0x4e4a] = 0x95b5, [0x4e4b] = 0x95b6, [0x4e4c] = 0x95b7,
  [0x4e4d] = 0x95b8, [0x4e4e] = 0x95b9, [0x4e4f] = 0x95ba, [0x4e50] = 0x95bb,
  [0x4e51] = 0x95bc, [0x4e52] = 0x95bd, [0x4e53] = 0x95be, [0x4e54] = 0x95bf,
  [0x4e55] = 0x95c0, [0x4e56] = 0x95c1, [0x4e57] = 0x95c2, [0x4e58] = 0x95c3,
  [0x4e59] = 0x95c4, [0x4e5a] = 0x95c5, [0x4e5b] = 0x95c6, [0x4e5c] = 0x95c7,
  [0x4e5d] = 0x95c8, [0x4e5e] = 0x95c9, [0x4e5f] = 0x95ca, [0x4e60] = 0x95cb,
  [0x4e61] = 0x6924, [0x4e62] = 0x68f0, [0x4e63] = 0x690b, [0x4e64] = 0x6901,
  [0x4e65] = 0x6957, [0x4e66] = 0x68e3, [0x4e67] = 0x6910, [0x4e68] = 0x6971,
  [0x4e69] = 0x6939, [0x4e6a] = 0x6960, [0x4e6b] = 0x6942, [0x4e6c] = 0x695d,
  [0x4e6d] = 0x6984, [0x4e6e] = 0x696b, [0x4e6f] = 0x6980, [0x4e70] = 0x6998,
  [0x4e71] = 0x6978, [0x4e72] = 0x6934, [0x4e73] = 0x69cc, [0x4e74] = 0x6987,
  [0x4e75] = 0x6988, [0x4e76] = 0x69ce, [0x4e77] = 0x6989, [0x4e78] = 0x6966,
  [0x4e79] = 0x6963, [0x4e7a] = 0x6979, [0x4e7b] = 0x699b, [0x4e7c] = 0x69a7,
  [0x4e7d] = 0x69bb, [0x4e7e] = 0x69ab, [0x4e7f] = 0x69ad, [0x4e80] = 0x69d4,
  [0x4e81] = 0x69b1, [0x4e82] = 0x69c1, [0x4e83] = 0x69ca, [0x4e84] = 0x69df,
  [0x4e85] = 0x6995, [0x4e86] = 0x69e0, [0x4e87] = 0x698d, [0x4e88] = 0x69ff,
  [0x4e89] = 0x6a2f, [0x4e8a] = 0x69ed, [0x4e8b] = 0x6a17, [0x4e8c] = 0x6a18,
  [0x4e8d] = 0x6a65, [0x4e8e] = 0x69f2, [0x4e8f] = 0x6a44, [0x4e90] = 0x6a3e,
  [0x4e91] = 0x6aa0, [0x4e92] = 0x6a50, [0x4e93] = 0x6a5b, [0x4e94] = 0x6a35,
  [0x4e95] = 0x6a8e, [0x4e96] = 0x6a79, [0x4e97] = 0x6a3d, [0x4e98] = 0x6a28,
  [0x4e99] = 0x6a58, [0x4e9a] = 0x6a7c, [0x4e9b] = 0x6a91, [0x4e9c] = 0x6a90,
  [0x4e9d] = 0x6aa9, [0x4e9e] = 0x6a97, [0x4e9f] = 0x6aab, [0x4ea0] = 0x7337,
  [0x4ea1] = 0x7352, [0x4ea2] = 0x6b81, [0x4ea3] = 0x6b82, [0x4ea4] = 0x6b87,
  [0x4ea5] = 0x6b84, [0x4ea6] = 0x6b92, [0x4ea7] = 0x6b93, [0x4ea8] = 0x6b8d,
  [0x4ea9] = 0x6b9a, [0x4eaa] = 0x6b9b, [0x4eab] = 0x6ba1, [0x4eac] = 0x6baa,
  [0x4ead] = 0x8f6b, [0x4eae] = 0x8f6d, [0x4eaf] = 0x8f71, [0x4eb0] = 0x8f72,
  [0x4eb1] = 0x8f73, [0x4eb2] = 0x8f75, [0x4eb3] = 0x8f76, [0x4eb4] = 0x8f78,
  [0x4eb5] = 0x8f77, [0x4eb6] = 0x8f79, [0x4eb7] = 0x8f7a, [0x4eb8] = 0x8f7c,
  [0x4eb9] = 0x8f7e, [0x4eba] = 0x8f81, [0x4ebb] = 0x8f82, [0x4ebc] = 0x8f84,
  [0x4ebd] = 0x8f87, [0x4ebe] = 0x8f8b, [0x4ec0] = 0x95cc, [0x4ec1] = 0x95cd,
  [0x4ec2] = 0x95ce, [0x4ec3] = 0x95cf, [0x4ec4] = 0x95d0, [0x4ec5] = 0x95d1,
  [0x4ec6] = 0x95d2, [0x4ec7] = 0x95d3, [0x4ec8] = 0x95d4, [0x4ec9] = 0x95d5,
  [0x4eca] = 0x95d6, [0x4ecb] = 0x95d7, [0x4ecc] = 0x95d8, [0x4ecd] = 0x95d9,
  [0x4ece] = 0x95da, [0x4ecf] = 0x95db, [0x4ed0] = 0x95dc, [0x4ed1] = 0x95dd,
  [0x4ed2] = 0x95de, [0x4ed3] = 0x95df, [0x4ed4] = 0x95e0, [0x4ed5] = 0x95e1,
  [0x4ed6] = 0x95e2, [0x4ed7] = 0x95e3, [0x4ed8] = 0x95e4, [0x4ed9] = 0x95e5,
  [0x4eda] = 0x95e6, [0x4edb] = 0x95e7, [0x4edc] = 0x95ec, [0x4edd] = 0x95ff,
  [0x4ede] = 0x9607, [0x4edf] = 0x9613, [0x4ee0] = 0x9618, [0x4ee1] = 0x961b,
  [0x4ee2] = 0x961e, [0x4ee3] = 0x9620, [0x4ee4] = 0x9623, [0x4ee5] = 0x9624,
  [0x4ee6] = 0x9625, [0x4ee7] = 0x9626, [0x4ee8] = 0x9627, [0x4ee9] = 0x9628,
  [0x4eea] = 0x9629, [0x4eeb] = 0x962b, [0x4eec] = 0x962c, [0x4eed] = 0x962d,
  [0x4eee] = 0x962f, [0x4eef] = 0x9630, [0x4ef0] = 0x9637, [0x4ef1] = 0x9638,
  [0x4ef2] = 0x9639, [0x4ef3] = 0x963a, [0x4ef4] = 0x963e, [0x4ef5] = 0x9641,
  [0x4ef6] = 0x9643, [0x4ef7] = 0x964a, [0x4ef8] = 0x964e, [0x4ef9] = 0x964f,
  [0x4efa] = 0x9651, [0x4efb] = 0x9652, [0x4efc] = 0x9653, [0x4efd] = 0x9656,
  [0x4efe] = 0x9657, [0x4f00] = 0x9658, [0x4f01] = 0x9659, [0x4f02] = 0x965a,
  [0x4f03] = 0x965c, [0x4f04] = 0x965d, [0x4f05] = 0x965e, [0x4f06] = 0x9660,
  [0x4f07] = 0x9663, [0x4f08] = 0x9665, [0x4f09] = 0x9666, [0x4f0a] = 0x966b,
  [0x4f0b] = 0x966d, [0x4f0c] = 0x966e, [0x4f0d] = 0x966f, [0x4f0e] = 0x9670,
  [0x4f0f] = 0x9671, [0x4f10] = 0x9673, [0x4f11] = 0x9678, [0x4f12] = 0x9679,
  [0x4f13] = 0x967a, [0x4f14] = 0x967b, [0x4f15] = 0x967c, [0x4f16] = 0x967d,
  [0x4f17] = 0x967e, [0x4f18] = 0x967f, [0x4f19] = 0x9680, [0x4f1a] = 0x9681,
  [0x4f1b] = 0x9682, [0x4f1c] = 0x9683, [0x4f1d] = 0x9684, [0x4f1e] = 0x9687,
  [0x4f1f] = 0x9689, [0x4f20] = 0x968a, [0x4f21] = 0x8f8d, [0x4f22] = 0x8f8e,
  [0x4f23] = 0x8f8f, [0x4f24] = 0x8f98, [0x4f25] = 0x8f9a, [0x4f26] = 0x8ece,
  [0x4f27] = 0x620b, [0x4f28] = 0x6217, [0x4f29] = 0x621b, [0x4f2a] = 0x621f,
  [0x4f2b] = 0x6222, [0x4f2c] = 0x6221, [0x4f2d] = 0x6225, [0x4f2e] = 0x6224,
  [0x4f2f] = 0x622c, [0x4f30] = 0x81e7, [0x4f31] = 0x74ef, [0x4f32] = 0x74f4,
  [0x4f33] = 0x74ff, [0x4f34] = 0x750f, [0x4f35] = 0x7511, [0x4f36] = 0x7513,
  [0x4f37] = 0x6534, [0x4f38] = 0x65ee, [0x4f39] = 0x65ef, [0x4f3a] = 0x65f0,
  [0x4f3b] = 0x660a, [0x4f3c] = 0x6619, [0x4f3d] = 0x6772, [0x4f3e] = 0x6603,
  [0x4f3f] = 0x6615, [0x4f40] = 0x6600, [0x4f41] = 0x7085, [0x4f42] = 0x66f7,
  [0x4f43] = 0x661d, [0x4f44] = 0x6634, [0x4f45] = 0x6631, [0x4f46] = 0x6636,
  [0x4f47] = 0x6635, [0x4f48] = 0x8006, [0x4f49] = 0x665f, [0x4f4a] = 0x6654,
  [0x4f4b] = 0x6641, [0x4f4c] = 0x664f, [0x4f4d] = 0x6656, [0x4f4e] = 0x6661,
  [0x4f4f] = 0x6657, [0x4f50] = 0x6677, [0x4f51] = 0x6684, [0x4f52] = 0x668c,
  [0x4f53] = 0x66a7, [0x4f54] = 0x669d, [0x4f55] = 0x66be, [0x4f56] = 0x66db,
  [0x4f57] = 0x66dc, [0x4f58] = 0x66e6, [0x4f59] = 0x66e9, [0x4f5a] = 0x8d32,
  [0x4f5b] = 0x8d33, [0x4f5c] = 0x8d36, [0x4f5d] = 0x8d3b, [0x4f5e] = 0x8d3d,
  [0x4f5f] = 0x8d40, [0x4f60] = 0x8d45, [0x4f61] = 0x8d46, [0x4f62] = 0x8d48,
  [0x4f63] = 0x8d49, [0x4f64] = 0x8d47, [0x4f65] = 0x8d4d, [0x4f66] = 0x8d55,
  [0x4f67] = 0x8d59, [0x4f68] = 0x89c7, [0x4f69] = 0x89ca, [0x4f6a] = 0x89cb,
  [0x4f6b] = 0x89cc, [0x4f6c] = 0x89ce, [0x4f6d] = 0x89cf, [0x4f6e] = 0x89d0,
  [0x4f6f] = 0x89d1, [0x4f70] = 0x726e, [0x4f71] = 0x729f, [0x4f72] = 0x725d,
  [0x4f73] = 0x7266, [0x4f74] = 0x726f, [0x4f75] = 0x727e, [0x4f76] = 0x727f,
  [0x4f77] = 0x7284, [0x4f78] = 0x728b, [0x4f79] = 0x728d, [0x4f7a] = 0x728f,
  [0x4f7b] = 0x7292, [0x4f7c] = 0x6308, [0x4f7d] = 0x6332, [0x4f7e] = 0x63b0,
  [0x4f80] = 0x968c, [0x4f81] = 0x968e, [0x4f82] = 0x9691, [0x4f83] = 0x9692,
  [0x4f84] = 0x9693, [0x4f85] = 0x9695, [0x4f86] = 0x9696, [0x4f87] = 0x969a,
  [0x4f88] = 0x969b, [0x4f89] = 0x969d, [0x4f8a] = 0x969e, [0x4f8b] = 0x969f,
  [0x4f8c] = 0x96a0, [0x4f8d] = 0x96a1, [0x4f8e] = 0x96a2, [0x4f8f] = 0x96a3,
  [0x4f90] = 0x96a4, [0x4f91] = 0x96a5, [0x4f92] = 0x96a6, [0x4f93] = 0x96a8,
  [0x4f94] = 0x96a9, [0x4f95] = 0x96aa, [0x4f96] = 0x96ab, [0x4f97] = 0x96ac,
  [0x4f98] = 0x96ad, [0x4f99] = 0x96ae, [0x4f9a] = 0x96af, [0x4f9b] = 0x96b1,
  [0x4f9c] = 0x96b2, [0x4f9d] = 0x96b4, [0x4f9e] = 0x96b5, [0x4f9f] = 0x96b7,
  [0x4fa0] = 0x96b8, [0x4fa1] = 0x96ba, [0x4fa2] = 0x96bb, [0x4fa3] = 0x96bf,
  [0x4fa4] = 0x96c2, [0x4fa5] = 0x96c3, [0x4fa6] = 0x96c8, [0x4fa7] = 0x96ca,
  [0x4fa8] = 0x96cb, [0x4fa9] = 0x96d0, [0x4faa] = 0x96d1, [0x4fab] = 0x96d3,
  [0x4fac] = 0x96d4, [0x4fad] = 0x96d6, [0x4fae] = 0x96d7, [0x4faf] = 0x96d8,
  [0x4fb0] = 0x96d9, [0x4fb1] = 0x96da, [0x4fb2] = 0x96db, [0x4fb3] = 0x96dc,
  [0x4fb4] = 0x96dd, [0x4fb5] = 0x96de, [0x4fb6] = 0x96df, [0x4fb7] = 0x96e1,
  [0x4fb8] = 0x96e2, [0x4fb9] = 0x96e3, [0x4fba] = 0x96e4, [0x4fbb] = 0x96e5,
  [0x4fbc] = 0x96e6, [0x4fbd] = 0x96e7, [0x4fbe] = 0x96eb, [0x4fc0] = 0x96ec,
  [0x4fc1] = 0x96ed, [0x4fc2] = 0x96ee, [0x4fc3] = 0x96f0, [0x4fc4] = 0x96f1,
  [0x4fc5] = 0x96f2, [0x4fc6] = 0x96f4, [0x4fc7] = 0x96f5, [0x4fc8] = 0x96f8,
  [0x4fc9] = 0x96fa, [0x4fca] = 0x96fb, [0x4fcb] = 0x96fc, [0x4fcc] = 0x96fd,
  [0x4fcd] = 0x96ff, [0x4fce] = 0x9702, [0x4fcf] = 0x9703, [0x4fd0] = 0x9705,
  [0x4fd1] = 0x970a, [0x4fd2] = 0x970b, [0x4fd3] = 0x970c, [0x4fd4] = 0x9710,
  [0x4fd5] = 0x9711, [0x4fd6] = 0x9712, [0x4fd7] = 0x9714, [0x4fd8] = 0x9715,
  [0x4fd9] = 0x9717, [0x4fda] = 0x9718, [0x4fdb] = 0x9719, [0x4fdc] = 0x971a,
  [0x4fdd] = 0x971b, [0x4fde] = 0x971d, [0x4fdf] = 0x971f, [0x4fe0] = 0x9720,
  [0x4fe1] = 0x643f, [0x4fe2] = 0x64d8, [0x4fe3] = 0x8004, [0x4fe4] = 0x6bea,
  [0x4fe5] = 0x6bf3, [0x4fe6] = 0x6bfd, [0x4fe7] = 0x6bf5, [0x4fe8] = 0x6bf9,
  [0x4fe9] = 0x6c05, [0x4fea] = 0x6c07, [0x4feb] = 0x6c06, [0x4fec] = 0x6c0d,
  [0x4fed] = 0x6c15, [0x4fee] = 0x6c18, [0x4fef] = 0x6c19, [0x4ff0] = 0x6c1a,
  [0x4ff1] = 0x6c21, [0x4ff2] = 0x6c29, [0x4ff3] = 0x6c24, [0x4ff4] = 0x6c2a,
  [0x4ff5] = 0x6c32, [0x4ff6] = 0x6535, [0x4ff7] = 0x6555, [0x4ff8] = 0x656b,
  [0x4ff9] = 0x724d, [0x4ffa] = 0x7252, [0x4ffb] = 0x7256, [0x4ffc] = 0x7230,
  [0x4ffd] = 0x8662, [0x4ffe] = 0x5216, [0x4fff] = 0x809f, [0x5000] = 0x809c,
  [0x5001] = 0x8093, [0x5002] = 0x80bc, [0x5003] = 0x670a, [0x5004] = 0x80bd,
  [0x5005] = 0x80b1, [0x5006] = 0x80ab, [0x5007] = 0x80ad, [0x5008] = 0x80b4,
  [0x5009] = 0x80b7, [0x500a] = 0x80e7, [0x500b] = 0x80e8, [0x500c] = 0x80e9,
  [0x500d] = 0x80ea, [0x500e] = 0x80db, [0x500f] = 0x80c2, [0x5010] = 0x80c4,
  [0x5011] = 0x80d9, [0x5012] = 0x80cd, [0x5013] = 0x80d7, [0x5014] = 0x6710,
  [0x5015] = 0x80dd, [0x5016] = 0x80eb, [0x5017] = 0x80f1, [0x5018] = 0x80f4,
  [0x5019] = 0x80ed, [0x501a] = 0x810d, [0x501b] = 0x810e, [0x501c] = 0x80f2,
  [0x501d] = 0x80fc, [0x501e] = 0x6715, [0x501f] = 0x8112, [0x5020] = 0x8c5a,
  [0x5021] = 0x8136, [0x5022] = 0x811e, [0x5023] = 0x812c, [0x5024] = 0x8118,
  [0x5025] = 0x8132, [0x5026] = 0x8148, [0x5027] = 0x814c, [0x5028] = 0x8153,
  [0x5029] = 0x8174, [0x502a] = 0x8159, [0x502b] = 0x815a, [0x502c] = 0x8171,
  [0x502d] = 0x8160, [0x502e] = 0x8169, [0x502f] = 0x817c, [0x5030] = 0x817d,
  [0x5031] = 0x816d, [0x5032] = 0x8167, [0x5033] = 0x584d, [0x5034] = 0x5ab5,
  [0x5035] = 0x8188, [0x5036] = 0x8182, [0x5037] = 0x8191, [0x5038] = 0x6ed5,
  [0x5039] = 0x81a3, [0x503a] = 0x81aa, [0x503b] = 0x81cc, [0x503c] = 0x6726,
  [0x503d] = 0x81ca, [0x503e] = 0x81bb, [0x5040] = 0x9721, [0x5041] = 0x9722,
  [0x5042] = 0x9723, [0x5043] = 0x9724, [0x5044] = 0x9725, [0x5045] = 0x9726,
  [0x5046] = 0x9727, [0x5047] = 0x9728, [0x5048] = 0x9729, [0x5049] = 0x972b,
  [0x504a] = 0x972c, [0x504b] = 0x972e, [0x504c] = 0x972f, [0x504d] = 0x9731,
  [0x504e] = 0x9733, [0x504f] = 0x9734, [0x5050] = 0x9735, [0x5051] = 0x9736,
  [0x5052] = 0x9737, [0x5053] = 0x973a, [0x5054] = 0x973b, [0x5055] = 0x973c,
  [0x5056] = 0x973d, [0x5057] = 0x973f, [0x5058] = 0x9740, [0x5059] = 0x9741,
  [0x505a] = 0x9742, [0x505b] = 0x9743, [0x505c] = 0x9744, [0x505d] = 0x9745,
  [0x505e] = 0x9746, [0x505f] = 0x9747, [0x5060] = 0x9748, [0x5061] = 0x9749,
  [0x5062] = 0x974a, [0x5063] = 0x974b, [0x5064] = 0x974c, [0x5065] = 0x974d,
  [0x5066] = 0x974e, [0x5067] = 0x974f, [0x5068] = 0x9750, [0x5069] = 0x9751,
  [0x506a] = 0x9754, [0x506b] = 0x9755, [0x506c] = 0x9757, [0x506d] = 0x9758,
  [0x506e] = 0x975a, [0x506f] = 0x975c, [0x5070] = 0x975d, [0x5071] = 0x975f,
  [0x5072] = 0x9763, [0x5073] = 0x9764, [0x5074] = 0x9766, [0x5075] = 0x9767,
  [0x5076] = 0x9768, [0x5077] = 0x976a, [0x5078] = 0x976b, [0x5079] = 0x976c,
  [0x507a] = 0x976d, [0x507b] = 0x976e, [0x507c] = 0x976f, [0x507d] = 0x9770,
  [0x507e] = 0x9771, [0x5080] = 0x9772, [0x5081] = 0x9775, [0x5082] = 0x9777,
  [0x5083] = 0x9778, [0x5084] = 0x9779, [0x5085] = 0x977a, [0x5086] = 0x977b,
  [0x5087] = 0x977d, [0x5088] = 0x977e, [0x5089] = 0x977f, [0x508a] = 0x9780,
  [0x508b] = 0x9781, [0x508c] = 0x9782, [0x508d] = 0x9783, [0x508e] = 0x9784,
  [0x508f] = 0x9786, [0x5090] = 0x9787, [0x5091] = 0x9788, [0x5092] = 0x9789,
  [0x5093] = 0x978a, [0x5094] = 0x978c, [0x5095] = 0x978e, [0x5096] = 0x978f,
  [0x5097] = 0x9790, [0x5098] = 0x9793, [0x5099] = 0x9795, [0x509a] = 0x9796,
  [0x509b] = 0x9797, [0x509c] = 0x9799, [0x509d] = 0x979a, [0x509e] = 0x979b,
  [0x509f] = 0x979c, [0x50a0] = 0x979d, [0x50a1] = 0x81c1, [0x50a2] = 0x81a6,
  [0x50a3] = 0x6b24, [0x50a4] = 0x6b37, [0x50a5] = 0x6b39, [0x50a6] = 0x6b43,
  [0x50a7] = 0x6b46, [0x50a8] = 0x6b59, [0x50a9] = 0x98d1, [0x50aa] = 0x98d2,
  [0x50ab] = 0x98d3, [0x50ac] = 0x98d5, [0x50ad] = 0x98d9, [0x50ae] = 0x98da,
  [0x50af] = 0x6bb3, [0x50b0] = 0x5f40, [0x50b1] = 0x6bc2, [0x50b2] = 0x89f3,
  [0x50b3] = 0x6590, [0x50b4] = 0x9f51, [0x50b5] = 0x6593, [0x50b6] = 0x65bc,
  [0x50b7] = 0x65c6, [0x50b8] = 0x65c4, [0x50b9] = 0x65c3, [0x50ba] = 0x65cc,
  [0x50bb] = 0x65ce, [0x50bc] = 0x65d2, [0x50bd] = 0x65d6, [0x50be] = 0x7080,
  [0x50bf] = 0x709c, [0x50c0] = 0x7096, [0x50c1] = 0x709d, [0x50c2] = 0x70bb,
  [0x50c3] = 0x70c0, [0x50c4] = 0x70b7, [0x50c5] = 0x70ab, [0x50c6] = 0x70b1,
  [0x50c7] = 0x70e8, [0x50c8] = 0x70ca, [0x50c9] = 0x7110, [0x50ca] = 0x7113,
  [0x50cb] = 0x7116, [0x50cc] = 0x712f, [0x50cd] = 0x7131, [0x50ce] = 0x7173,
  [0x50cf] = 0x715c, [0x50d0] = 0x7168, [0x50d1] = 0x7145, [0x50d2] = 0x7172,
  [0x50d3] = 0x714a, [0x50d4] = 0x7178, [0x50d5] = 0x717a, [0x50d6] = 0x7198,
  [0x50d7] = 0x71b3, [0x50d8] = 0x71b5, [0x50d9] = 0x71a8, [0x50da] = 0x71a0,
  [0x50db] = 0x71e0, [0x50dc] = 0x71d4, [0x50dd] = 0x71e7, [0x50de] = 0x71f9,
  [0x50df] = 0x721d, [0x50e0] = 0x7228, [0x50e1] = 0x706c, [0x50e2] = 0x7118,
  [0x50e3] = 0x7166, [0x50e4] = 0x71b9, [0x50e5] = 0x623e, [0x50e6] = 0x623d,
  [0x50e7] = 0x6243, [0x50e8] = 0x6248, [0x50e9] = 0x6249, [0x50ea] = 0x793b,
  [0x50eb] = 0x7940, [0x50ec] = 0x7946, [0x50ed] = 0x7949, [0x50ee] = 0x795b,
  [0x50ef] = 0x795c, [0x50f0] = 0x7953, [0x50f1] = 0x795a, [0x50f2] = 0x7962,
  [0x50f3] = 0x7957, [0x50f4] = 0x7960, [0x50f5] = 0x796f, [0x50f6] = 0x7967,
  [0x50f7] = 0x797a, [0x50f8] = 0x7985, [0x50f9] = 0x798a, [0x50fa] = 0x799a,
  [0x50fb] = 0x79a7, [0x50fc] = 0x79b3, [0x50fd] = 0x5fd1, [0x50fe] = 0x5fd0,
  [0x5100] = 0x979e, [0x5101] = 0x979f, [0x5102] = 0x97a1, [0x5103] = 0x97a2,
  [0x5104] = 0x97a4, [0x5105] = 0x97a5, [0x5106] = 0x97a6, [0x5107] = 0x97a7,
  [0x5108] = 0x97a8, [0x5109] = 0x97a9, [0x510a] = 0x97aa, [0x510b] = 0x97ac,
  [0x510c] = 0x97ae, [0x510d] = 0x97b0, [0x510e] = 0x97b1, [0x510f] = 0x97b3,
  [0x5110] = 0x97b5, [0x5111] = 0x97b6, [0x5112] = 0x97b7, [0x5113] = 0x97b8,
  [0x5114] = 0x97b9, [0x5115] = 0x97ba, [0x5116] = 0x97bb, [0x5117] = 0x97bc,
  [0x5118] = 0x97bd, [0x5119] = 0x97be, [0x511a] = 0x97bf, [0x511b] = 0x97c0,
  [0x511c] = 0x97c1, [0x511d] = 0x97c2, [0x511e] = 0x97c3, [0x511f] = 0x97c4,
  [0x5120] = 0x97c5, [0x5121] = 0x97c6, [0x5122] = 0x97c7, [0x5123] = 0x97c8,
  [0x5124] = 0x97c9, [0x5125] = 0x97ca, [0x5126] = 0x97cb, [0x5127] = 0x97cc,
  [0x5128] = 0x97cd, [0x5129] = 0x97ce, [0x512a] = 0x97cf, [0x512b] = 0x97d0,
  [0x512c] = 0x97d1, [0x512d] = 0x97d2, [0x512e] = 0x97d3, [0x512f] = 0x97d4,
  [0x5130] = 0x97d5, [0x5131] = 0x97d6, [0x5132] = 0x97d7, [0x5133] = 0x97d8,
  [0x5134] = 0x97d9, [0x5135] = 0x97da, [0x5136] = 0x97db, [0x5137] = 0x97dc,
  [0x5138] = 0x97dd, [0x5139] = 0x97de, [0x513a] = 0x97df, [0x513b] = 0x97e0,
  [0x513c] = 0x97e1, [0x513d] = 0x97e2, [0x513e] = 0x97e3, [0x5140] = 0x97e4,
  [0x5141] = 0x97e5, [0x5142] = 0x97e8, [0x5143] = 0x97ee, [0x5144] = 0x97ef,
  [0x5145] = 0x97f0, [0x5146] = 0x97f1, [0x5147] = 0x97f2, [0x5148] = 0x97f4,
  [0x5149] = 0x97f7, [0x514a] = 0x97f8, [0x514b] = 0x97f9, [0x514c] = 0x97fa,
  [0x514d] = 0x97fb, [0x514e] = 0x97fc, [0x514f] = 0x97fd, [0x5150] = 0x97fe,
  [0x5151] = 0x97ff, [0x5152] = 0x9800, [0x5153] = 0x9801, [0x5154] = 0x9802,
  [0x5155] = 0x9803, [0x5156] = 0x9804, [0x5157] = 0x9805, [0x5158] = 0x9806,
  [0x5159] = 0x9807, [0x515a] = 0x9808, [0x515b] = 0x9809, [0x515c] = 0x980a,
  [0x515d] = 0x980b, [0x515e] = 0x980c, [0x515f] = 0x980d, [0x5160] = 0x980e,
  [0x5161] = 0x603c, [0x5162] = 0x605d, [0x5163] = 0x605a, [0x5164] = 0x6067,
  [0x5165] = 0x6041, [0x5166] = 0x6059, [0x5167] = 0x6063, [0x5168] = 0x60ab,
  [0x5169] = 0x6106, [0x516a] = 0x610d, [0x516b] = 0x615d, [0x516c] = 0x61a9,
  [0x516d] = 0x619d, [0x516e] = 0x61cb, [0x516f] = 0x61d1, [0x5170] = 0x6206,
  [0x5171] = 0x8080, [0x5172] = 0x807f, [0x5173] = 0x6c93, [0x5174] = 0x6cf6,
  [0x5175] = 0x6dfc, [0x5176] = 0x77f6, [0x5177] = 0x77f8, [0x5178] = 0x7800,
  [0x5179] = 0x7809, [0x517a] = 0x7817, [0x517b] = 0x7818, [0x517c] = 0x7811,
  [0x517d] = 0x65ab, [0x517e] = 0x782d, [0x517f] = 0x781c, [0x5180] = 0x781d,
  [0x5181] = 0x7839, [0x5182] = 0x783a, [0x5183] = 0x783b, [0x5184] = 0x781f,
  [0x5185] = 0x783c, [0x5186] = 0x7825, [0x5187] = 0x782c, [0x5188] = 0x7823,
  [0x5189] = 0x7829, [0x518a] = 0x784e, [0x518b] = 0x786d, [0x518c] = 0x7856,
  [0x518d] = 0x7857, [0x518e] = 0x7826, [0x518f] = 0x7850, [0x5190] = 0x7847,
  [0x5191] = 0x784c, [0x5192] = 0x786a, [0x5193] = 0x789b, [0x5194] = 0x7893,
  [0x5195] = 0x789a, [0x5196] = 0x7887, [0x5197] = 0x789c, [0x5198] = 0x78a1,
  [0x5199] = 0x78a3, [0x519a] = 0x78b2, [0x519b] = 0x78b9, [0x519c] = 0x78a5,
  [0x519d] = 0x78d4, [0x519e] = 0x78d9, [0x519f] = 0x78c9, [0x51a0] = 0x78ec,
  [0x51a1] = 0x78f2, [0x51a2] = 0x7905, [0x51a3] = 0x78f4, [0x51a4] = 0x7913,
  [0x51a5] = 0x7924, [0x51a6] = 0x791e, [0x51a7] = 0x7934, [0x51a8] = 0x9f9b,
  [0x51a9] = 0x9ef9, [0x51aa] = 0x9efb, [0x51ab] = 0x9efc, [0x51ac] = 0x76f1,
  [0x51ad] = 0x7704, [0x51ae] = 0x770d, [0x51af] = 0x76f9, [0x51b0] = 0x7707,
  [0x51b1] = 0x7708, [0x51b2] = 0x771a, [0x51b3] = 0x7722, [0x51b4] = 0x7719,
  [0x51b5] = 0x772d, [0x51b6] = 0x7726, [0x51b7] = 0x7735, [0x51b8] = 0x7738,
  [0x51b9] = 0x7750, [0x51ba] = 0x7751, [0x51bb] = 0x7747, [0x51bc] = 0x7743,
  [0x51bd] = 0x775a, [0x51be] = 0x7768, [0x51c0] = 0x980f, [0x51c1] = 0x9810,
  [0x51c2] = 0x9811, [0x51c3] = 0x9812, [0x51c4] = 0x9813, [0x51c5] = 0x9814,
  [0x51c6] = 0x9815, [0x51c7] = 0x9816, [0x51c8] = 0x9817, [0x51c9] = 0x9818,
  [0x51ca] = 0x9819, [0x51cb] = 0x981a, [0x51cc] = 0x981b, [0x51cd] = 0x981c,
  [0x51ce] = 0x981d, [0x51cf] = 0x981e, [0x51d0] = 0x981f, [0x51d1] = 0x9820,
  [0x51d2] = 0x9821, [0x51d3] = 0x9822, [0x51d4] = 0x9823, [0x51d5] = 0x9824,
  [0x51d6] = 0x9825, [0x51d7] = 0x9826, [0x51d8] = 0x9827, [0x51d9] = 0x9828,
  [0x51da] = 0x9829, [0x51db] = 0x982a, [0x51dc] = 0x982b, [0x51dd] = 0x982c,
  [0x51de] = 0x982d, [0x51df] = 0x982e, [0x51e0] = 0x982f, [0x51e1] = 0x9830,
  [0x51e2] = 0x9831, [0x51e3] = 0x9832, [0x51e4] = 0x9833, [0x51e5] = 0x9834,
  [0x51e6] = 0x9835, [0x51e7] = 0x9836, [0x51e8] = 0x9837, [0x51e9] = 0x9838,
  [0x51ea] = 0x9839, [0x51eb] = 0x983a, [0x51ec] = 0x983b, [0x51ed] = 0x983c,
  [0x51ee] = 0x983d, [0x51ef] = 0x983e, [0x51f0] = 0x983f, [0x51f1] = 0x9840,
  [0x51f2] = 0x9841, [0x51f3] = 0x9842, [0x51f4] = 0x9843, [0x51f5] = 0x9844,
  [0x51f6] = 0x9845, [0x51f7] = 0x9846, [0x51f8] = 0x9847, [0x51f9] = 0x9848,
  [0x51fa] = 0x9849, [0x51fb] = 0x984a, [0x51fc] = 0x984b, [0x51fd] = 0x984c,
  [0x51fe] = 0x984d, [0x5200] = 0x984e, [0x5201] = 0x984f, [0x5202] = 0x9850,
  [0x5203] = 0x9851, [0x5204] = 0x9852, [0x5205] = 0x9853, [0x5206] = 0x9854,
  [0x5207] = 0x9855, [0x5208] = 0x9856, [0x5209] = 0x9857, [0x520a] = 0x9858,
  [0x520b] = 0x9859, [0x520c] = 0x985a, [0x520d] = 0x985b, [0x520e] = 0x985c,
  [0x520f] = 0x985d, [0x5210] = 0x985e, [0x5211] = 0x985f, [0x5212] = 0x9860,
  [0x5213] = 0x9861, [0x5214] = 0x9862, [0x5215] = 0x9863, [0x5216] = 0x9864,
  [0x5217] = 0x9865, [0x5218] = 0x9866, [0x5219] = 0x9867, [0x521a] = 0x9868,
  [0x521b] = 0x9869, [0x521c] = 0x986a, [0x521d] = 0x986b, [0x521e] = 0x986c,
  [0x521f] = 0x986d, [0x5220] = 0x986e, [0x5221] = 0x7762, [0x5222] = 0x7765,
  [0x5223] = 0x777f, [0x5224] = 0x778d, [0x5225] = 0x777d, [0x5226] = 0x7780,
  [0x5227] = 0x778c, [0x5228] = 0x7791, [0x5229] = 0x779f, [0x522a] = 0x77a0,
  [0x522b] = 0x77b0, [0x522c] = 0x77b5, [0x522d] = 0x77bd, [0x522e] = 0x753a,
  [0x522f] = 0x7540, [0x5230] = 0x754e, [0x5231] = 0x754b, [0x5232] = 0x7548,
  [0x5233] = 0x755b, [0x5234] = 0x7572, [0x5235] = 0x7579, [0x5236] = 0x7583,
  [0x5237] = 0x7f58, [0x5238] = 0x7f61, [0x5239] = 0x7f5f, [0x523a] = 0x8a48,
  [0x523b] = 0x7f68, [0x523c] = 0x7f74, [0x523d] = 0x7f71, [0x523e] = 0x7f79,
  [0x523f] = 0x7f81, [0x5240] = 0x7f7e, [0x5241] = 0x76cd, [0x5242] = 0x76e5,
  [0x5243] = 0x8832, [0x5244] = 0x9485, [0x5245] = 0x9486, [0x5246] = 0x9487,
  [0x5247] = 0x948b, [0x5248] = 0x948a, [0x5249] = 0x948c, [0x524a] = 0x948d,
  [0x524b] = 0x948f, [0x524c] = 0x9490, [0x524d] = 0x9494, [0x524e] = 0x9497,
  [0x524f] = 0x9495, [0x5250] = 0x949a, [0x5251] = 0x949b, [0x5252] = 0x949c,
  [0x5253] = 0x94a3, [0x5254] = 0x94a4, [0x5255] = 0x94ab, [0x5256] = 0x94aa,
  [0x5257] = 0x94ad, [0x5258] = 0x94ac, [0x5259] = 0x94af, [0x525a] = 0x94b0,
  [0x525b] = 0x94b2, [0x525c] = 0x94b4, [0x525d] = 0x94b6, [0x525e] = 0x94b7,
  [0x525f] = 0x94b8, [0x5260] = 0x94b9, [0x5261] = 0x94ba, [0x5262] = 0x94bc,
  [0x5263] = 0x94bd, [0x5264] = 0x94bf, [0x5265] = 0x94c4, [0x5266] = 0x94c8,
  [0x5267] = 0x94c9, [0x5268] = 0x94ca, [0x5269] = 0x94cb, [0x526a] = 0x94cc,
  [0x526b] = 0x94cd, [0x526c] = 0x94ce, [0x526d] = 0x94d0, [0x526e] = 0x94d1,
  [0x526f] = 0x94d2, [0x5270] = 0x94d5, [0x5271] = 0x94d6, [0x5272] = 0x94d7,
  [0x5273] = 0x94d9, [0x5274] = 0x94d8, [0x5275] = 0x94db, [0x5276] = 0x94de,
  [0x5277] = 0x94df, [0x5278] = 0x94e0, [0x5279] = 0x94e2, [0x527a] = 0x94e4,
  [0x527b] = 0x94e5, [0x527c] = 0x94e7, [0x527d] = 0x94e8, [0x527e] = 0x94ea,
  [0x5280] = 0x986f, [0x5281] = 0x9870, [0x5282] = 0x9871, [0x5283] = 0x9872,
  [0x5284] = 0x9873, [0x5285] = 0x9874, [0x5286] = 0x988b, [0x5287] = 0x988e,
  [0x5288] = 0x9892, [0x5289] = 0x9895, [0x528a] = 0x9899, [0x528b] = 0x98a3,
  [0x528c] = 0x98a8, [0x528d] = 0x98a9, [0x528e] = 0x98aa, [0x528f] = 0x98ab,
  [0x5290] = 0x98ac, [0x5291] = 0x98ad, [0x5292] = 0x98ae, [0x5293] = 0x98af,
  [0x5294] = 0x98b0, [0x5295] = 0x98b1, [0x5296] = 0x98b2, [0x5297] = 0x98b3,
  [0x5298] = 0x98b4, [0x5299] = 0x98b5, [0x529a] = 0x98b6, [0x529b] = 0x98b7,
  [0x529c] = 0x98b8, [0x529d] = 0x98b9, [0x529e] = 0x98ba, [0x529f] = 0x98bb,
  [0x52a0] = 0x98bc, [0x52a1] = 0x98bd, [0x52a2] = 0x98be, [0x52a3] = 0x98bf,
  [0x52a4] = 0x98c0, [0x52a5] = 0x98c1, [0x52a6] = 0x98c2, [0x52a7] = 0x98c3,
  [0x52a8] = 0x98c4, [0x52a9] = 0x98c5, [0x52aa] = 0x98c6, [0x52ab] = 0x98c7,
  [0x52ac] = 0x98c8, [0x52ad] = 0x98c9, [0x52ae] = 0x98ca, [0x52af] = 0x98cb,
  [0x52b0] = 0x98cc, [0x52b1] = 0x98cd, [0x52b2] = 0x98cf, [0x52b3] = 0x98d0,
  [0x52b4] = 0x98d4, [0x52b5] = 0x98d6, [0x52b6] = 0x98d7, [0x52b7] = 0x98db,
  [0x52b8] = 0x98dc, [0x52b9] = 0x98dd, [0x52ba] = 0x98e0, [0x52bb] = 0x98e1,
  [0x52bc] = 0x98e2, [0x52bd] = 0x98e3, [0x52be] = 0x98e4, [0x52c0] = 0x98e5,
  [0x52c1] = 0x98e6, [0x52c2] = 0x98e9, [0x52c3] = 0x98ea, [0x52c4] = 0x98eb,
  [0x52c5] = 0x98ec, [0x52c6] = 0x98ed, [0x52c7] = 0x98ee, [0x52c8] = 0x98ef,
  [0x52c9] = 0x98f0, [0x52ca] = 0x98f1, [0x52cb] = 0x98f2, [0x52cc] = 0x98f3,
  [0x52cd] = 0x98f4, [0x52ce] = 0x98f5, [0x52cf] = 0x98f6, [0x52d0] = 0x98f7,
  [0x52d1] = 0x98f8, [0x52d2] = 0x98f9, [0x52d3] = 0x98fa, [0x52d4] = 0x98fb,
  [0x52d5] = 0x98fc, [0x52d6] = 0x98fd, [0x52d7] = 0x98fe, [0x52d8] = 0x98ff,
  [0x52d9] = 0x9900, [0x52da] = 0x9901, [0x52db] = 0x9902, [0x52dc] = 0x9903,
  [0x52dd] = 0x9904, [0x52de] = 0x9905, [0x52df] = 0x9906, [0x52e0] = 0x9907,
  [0x52e1] = 0x94e9, [0x52e2] = 0x94eb, [0x52e3] = 0x94ee, [0x52e4] = 0x94ef,
  [0x52e5] = 0x94f3, [0x52e6] = 0x94f4, [0x52e7] = 0x94f5, [0x52e8] = 0x94f7,
  [0x52e9] = 0x94f9, [0x52ea] = 0x94fc, [0x52eb] = 0x94fd, [0x52ec] = 0x94ff,
  [0x52ed] = 0x9503, [0x52ee] = 0x9502, [0x52ef] = 0x9506, [0x52f0] = 0x9507,
  [0x52f1] = 0x9509, [0x52f2] = 0x950a, [0x52f3] = 0x950d, [0x52f4] = 0x950e,
  [0x52f5] = 0x950f, [0x52f6] = 0x9512, [0x52f7] = 0x9513, [0x52f8] = 0x9514,
  [0x52f9] = 0x9515, [0x52fa] = 0x9516, [0x52fb] = 0x9518, [0x52fc] = 0x951b,
  [0x52fd] = 0x951d, [0x52fe] = 0x951e, [0x52ff] = 0x951f, [0x5300] = 0x9522,
  [0x5301] = 0x952a, [0x5302] = 0x952b, [0x5303] = 0x9529, [0x5304] = 0x952c,
  [0x5305] = 0x9531, [0x5306] = 0x9532, [0x5307] = 0x9534, [0x5308] = 0x9536,
  [0x5309] = 0x9537, [0x530a] = 0x9538, [0x530b] = 0x953c, [0x530c] = 0x953e,
  [0x530d] = 0x953f, [0x530e] = 0x9542, [0x530f] = 0x9535, [0x5310] = 0x9544,
  [0x5311] = 0x9545, [0x5312] = 0x9546, [0x5313] = 0x9549, [0x5314] = 0x954c,
  [0x5315] = 0x954e, [0x5316] = 0x954f, [0x5317] = 0x9552, [0x5318] = 0x9553,
  [0x5319] = 0x9554, [0x531a] = 0x9556, [0x531b] = 0x9557, [0x531c] = 0x9558,
  [0x531d] = 0x9559, [0x531e] = 0x955b, [0x531f] = 0x955e, [0x5320] = 0x955f,
  [0x5321] = 0x955d, [0x5322] = 0x9561, [0x5323] = 0x9562, [0x5324] = 0x9564,
  [0x5325] = 0x9565, [0x5326] = 0x9566, [0x5327] = 0x9567, [0x5328] = 0x9568,
  [0x5329] = 0x9569, [0x532a] = 0x956a, [0x532b] = 0x956b, [0x532c] = 0x956c,
  [0x532d] = 0x956f, [0x532e] = 0x9571, [0x532f] = 0x9572, [0x5330] = 0x9573,
  [0x5331] = 0x953a, [0x5332] = 0x77e7, [0x5333] = 0x77ec, [0x5334] = 0x96c9,
  [0x5335] = 0x79d5, [0x5336] = 0x79ed, [0x5337] = 0x79e3, [0x5338] = 0x79eb,
  [0x5339] = 0x7a06, [0x533a] = 0x5d47, [0x533b] = 0x7a03, [0x533c] = 0x7a02,
  [0x533d] = 0x7a1e, [0x533e] = 0x7a14, [0x5340] = 0x9908, [0x5341] = 0x9909,
  [0x5342] = 0x990a, [0x5343] = 0x990b, [0x5344] = 0x990c, [0x5345] = 0x990e,
  [0x5346] = 0x990f, [0x5347] = 0x9911, [0x5348] = 0x9912, [0x5349] = 0x9913,
  [0x534a] = 0x9914, [0x534b] = 0x9915, [0x534c] = 0x9916, [0x534d] = 0x9917,
  [0x534e] = 0x9918, [0x534f] = 0x9919, [0x5350] = 0x991a, [0x5351] = 0x991b,
  [0x5352] = 0x991c, [0x5353] = 0x991d, [0x5354] = 0x991e, [0x5355] = 0x991f,
  [0x5356] = 0x9920, [0x5357] = 0x9921, [0x5358] = 0x9922, [0x5359] = 0x9923,
  [0x535a] = 0x9924, [0x535b] = 0x9925, [0x535c] = 0x9926, [0x535d] = 0x9927,
  [0x535e] = 0x9928, [0x535f] = 0x9929, [0x5360] = 0x992a, [0x5361] = 0x992b,
  [0x5362] = 0x992c, [0x5363] = 0x992d, [0x5364] = 0x992f, [0x5365] = 0x9930,
  [0x5366] = 0x9931, [0x5367] = 0x9932, [0x5368] = 0x9933, [0x5369] = 0x9934,
  [0x536a] = 0x9935, [0x536b] = 0x9936, [0x536c] = 0x9937, [0x536d] = 0x9938,
  [0x536e] = 0x9939, [0x536f] = 0x993a, [0x5370] = 0x993b, [0x5371] = 0x993c,
  [0x5372] = 0x993d, [0x5373] = 0x993e, [0x5374] = 0x993f, [0x5375] = 0x9940,
  [0x5376] = 0x9941, [0x5377] = 0x9942, [0x5378] = 0x9943, [0x5379] = 0x9944,
  [0x537a] = 0x9945, [0x537b] = 0x9946, [0x537c] = 0x9947, [0x537d] = 0x9948,
  [0x537e] = 0x9949, [0x5380] = 0x994a, [0x5381] = 0x994b, [0x5382] = 0x994c,
  [0x5383] = 0x994d, [0x5384] = 0x994e, [0x5385] = 0x994f, [0x5386] = 0x9950,
  [0x5387] = 0x9951, [0x5388] = 0x9952, [0x5389] = 0x9953, [0x538a] = 0x9956,
  [0x538b] = 0x9957, [0x538c] = 0x9958, [0x538d] = 0x9959, [0x538e] = 0x995a,
  [0x538f] = 0x995b, [0x5390] = 0x995c, [0x5391] = 0x995d, [0x5392] = 0x995e,
  [0x5393] = 0x995f, [0x5394] = 0x9960, [0x5395] = 0x9961, [0x5396] = 0x9962,
  [0x5397] = 0x9964, [0x5398] = 0x9966, [0x5399] = 0x9973, [0x539a] = 0x9978,
  [0x539b] = 0x9979, [0x539c] = 0x997b, [0x539d] = 0x997e, [0x539e] = 0x9982,
  [0x539f] = 0x9983, [0x53a0] = 0x9989, [0x53a1] = 0x7a39, [0x53a2] = 0x7a37,
  [0x53a3] = 0x7a51, [0x53a4] = 0x9ecf, [0x53a5] = 0x99a5, [0x53a6] = 0x7a70,
  [0x53a7] = 0x7688, [0x53a8] = 0x768e, [0x53a9] = 0x7693, [0x53aa] = 0x7699,
  [0x53ab] = 0x76a4, [0x53ac] = 0x74de, [0x53ad] = 0x74e0, [0x53ae] = 0x752c,
  [0x53af] = 0x9e20, [0x53b0] = 0x9e22, [0x53b1] = 0x9e28, [0x53b2] = 0x9e29,
  [0x53b3] = 0x9e2a, [0x53b4] = 0x9e2b, [0x53b5] = 0x9e2c, [0x53b6] = 0x9e32,
  [0x53b7] = 0x9e31, [0x53b8] = 0x9e36, [0x53b9] = 0x9e38, [0x53ba] = 0x9e37,
  [0x53bb] = 0x9e39, [0x53bc] = 0x9e3a, [0x53bd] = 0x9e3e, [0x53be] = 0x9e41,
  [0x53bf] = 0x9e42, [0x53c0] = 0x9e44, [0x53c1] = 0x9e46, [0x53c2] = 0x9e47,
  [0x53c3] = 0x9e48, [0x53c4] = 0x9e49, [0x53c5] = 0x9e4b, [0x53c6] = 0x9e4c,
  [0x53c7] = 0x9e4e, [0x53c8] = 0x9e51, [0x53c9] = 0x9e55, [0x53ca] = 0x9e57,
  [0x53cb] = 0x9e5a, [0x53cc] = 0x9e5b, [0x53cd] = 0x9e5c, [0x53ce] = 0x9e5e,
  [0x53cf] = 0x9e63, [0x53d0] = 0x9e66, [0x53d1] = 0x9e67, [0x53d2] = 0x9e68,
  [0x53d3] = 0x9e69, [0x53d4] = 0x9e6a, [0x53d5] = 0x9e6b, [0x53d6] = 0x9e6c,
  [0x53d7] = 0x9e71, [0x53d8] = 0x9e6d, [0x53d9] = 0x9e73, [0x53da] = 0x7592,
  [0x53db] = 0x7594, [0x53dc] = 0x7596, [0x53dd] = 0x75a0, [0x53de] = 0x759d,
  [0x53df] = 0x75ac, [0x53e0] = 0x75a3, [0x53e1] = 0x75b3, [0x53e2] = 0x75b4,
  [0x53e3] = 0x75b8, [0x53e4] = 0x75c4, [0x53e5] = 0x75b1, [0x53e6] = 0x75b0,
  [0x53e7] = 0x75c3, [0x53e8] = 0x75c2, [0x53e9] = 0x75d6, [0x53ea] = 0x75cd,
  [0x53eb] = 0x75e3, [0x53ec] = 0x75e8, [0x53ed] = 0x75e6, [0x53ee] = 0x75e4,
  [0x53ef] = 0x75eb, [0x53f0] = 0x75e7, [0x53f1] = 0x7603, [0x53f2] = 0x75f1,
  [0x53f3] = 0x75fc, [0x53f4] = 0x75ff, [0x53f5] = 0x7610, [0x53f6] = 0x7600,
  [0x53f7] = 0x7605, [0x53f8] = 0x760c, [0x53f9] = 0x7617, [0x53fa] = 0x760a,
  [0x53fb] = 0x7625, [0x53fc] = 0x7618, [0x53fd] = 0x7615, [0x53fe] = 0x7619,
  [0x5400] = 0x998c, [0x5401] = 0x998e, [0x5402] = 0x999a, [0x5403] = 0x999b,
  [0x5404] = 0x999c, [0x5405] = 0x999d, [0x5406] = 0x999e, [0x5407] = 0x999f,
  [0x5408] = 0x99a0, [0x5409] = 0x99a1, [0x540a] = 0x99a2, [0x540b] = 0x99a3,
  [0x540c] = 0x99a4, [0x540d] = 0x99a6, [0x540e] = 0x99a7, [0x540f] = 0x99a9,
  [0x5410] = 0x99aa, [0x5411] = 0x99ab, [0x5412] = 0x99ac, [0x5413] = 0x99ad,
  [0x5414] = 0x99ae, [0x5415] = 0x99af, [0x5416] = 0x99b0, [0x5417] = 0x99b1,
  [0x5418] = 0x99b2, [0x5419] = 0x99b3, [0x541a] = 0x99b4, [0x541b] = 0x99b5,
  [0x541c] = 0x99b6, [0x541d] = 0x99b7, [0x541e] = 0x99b8, [0x541f] = 0x99b9,
  [0x5420] = 0x99ba, [0x5421] = 0x99bb, [0x5422] = 0x99bc, [0x5423] = 0x99bd,
  [0x5424] = 0x99be, [0x5425] = 0x99bf, [0x5426] = 0x99c0, [0x5427] = 0x99c1,
  [0x5428] = 0x99c2, [0x5429] = 0x99c3, [0x542a] = 0x99c4, [0x542b] = 0x99c5,
  [0x542c] = 0x99c6, [0x542d] = 0x99c7, [0x542e] = 0x99c8, [0x542f] = 0x99c9,
  [0x5430] = 0x99ca, [0x5431] = 0x99cb, [0x5432] = 0x99cc, [0x5433] = 0x99cd,
  [0x5434] = 0x99ce, [0x5435] = 0x99cf, [0x5436] = 0x99d0, [0x5437] = 0x99d1,
  [0x5438] = 0x99d2, [0x5439] = 0x99d3, [0x543a] = 0x99d4, [0x543b] = 0x99d5,
  [0x543c] = 0x99d6, [0x543d] = 0x99d7, [0x543e] = 0x99d8, [0x5440] = 0x99d9,
  [0x5441] = 0x99da, [0x5442] = 0x99db, [0x5443] = 0x99dc, [0x5444] = 0x99dd,
  [0x5445] = 0x99de, [0x5446] = 0x99df, [0x5447] = 0x99e0, [0x5448] = 0x99e1,
  [0x5449] = 0x99e2, [0x544a] = 0x99e3, [0x544b] = 0x99e4, [0x544c] = 0x99e5,
  [0x544d] = 0x99e6, [0x544e] = 0x99e7, [0x544f] = 0x99e8, [0x5450] = 0x99e9,
  [0x5451] = 0x99ea, [0x5452] = 0x99eb, [0x5453] = 0x99ec, [0x5454] = 0x99ed,
  [0x5455] = 0x99ee, [0x5456] = 0x99ef, [0x5457] = 0x99f0, [0x5458] = 0x99f1,
  [0x5459] = 0x99f2, [0x545a] = 0x99f3, [0x545b] = 0x99f4, [0x545c] = 0x99f5,
  [0x545d] = 0x99f6, [0x545e] = 0x99f7, [0x545f] = 0x99f8, [0x5460] = 0x99f9,
  [0x5461] = 0x761b, [0x5462] = 0x763c, [0x5463] = 0x7622, [0x5464] = 0x7620,
  [0x5465] = 0x7640, [0x5466] = 0x762d, [0x5467] = 0x7630, [0x5468] = 0x763f,
  [0x5469] = 0x7635, [0x546a] = 0x7643, [0x546b] = 0x763e, [0x546c] = 0x7633,
  [0x546d] = 0x764d, [0x546e] = 0x765e, [0x546f] = 0x7654, [0x5470] = 0x765c,
  [0x5471] = 0x7656, [0x5472] = 0x766b, [0x5473] = 0x766f, [0x5474] = 0x7fca,
  [0x5475] = 0x7ae6, [0x5476] = 0x7a78, [0x5477] = 0x7a79, [0x5478] = 0x7a80,
  [0x5479] = 0x7a86, [0x547a] = 0x7a88, [0x547b] = 0x7a95, [0x547c] = 0x7aa6,
  [0x547d] = 0x7aa0, [0x547e] = 0x7aac, [0x547f] = 0x7aa8, [0x5480] = 0x7aad,
  [0x5481] = 0x7ab3, [0x5482] = 0x8864, [0x5483] = 0x8869, [0x5484] = 0x8872,
  [0x5485] = 0x887d, [0x5486] = 0x887f, [0x5487] = 0x8882, [0x5488] = 0x88a2,
  [0x5489] = 0x88c6, [0x548a] = 0x88b7, [0x548b] = 0x88bc, [0x548c] = 0x88c9,
  [0x548d] = 0x88e2, [0x548e] = 0x88ce, [0x548f] = 0x88e3, [0x5490] = 0x88e5,
  [0x5491] = 0x88f1, [0x5492] = 0x891a, [0x5493] = 0x88fc, [0x5494] = 0x88e8,
  [0x5495] = 0x88fe, [0x5496] = 0x88f0, [0x5497] = 0x8921, [0x5498] = 0x8919,
  [0x5499] = 0x8913, [0x549a] = 0x891b, [0x549b] = 0x890a, [0x549c] = 0x8934,
  [0x549d] = 0x892b, [0x549e] = 0x8936, [0x549f] = 0x8941, [0x54a0] = 0x8966,
  [0x54a1] = 0x897b, [0x54a2] = 0x758b, [0x54a3] = 0x80e5, [0x54a4] = 0x76b2,
  [0x54a5] = 0x76b4, [0x54a6] = 0x77dc, [0x54a7] = 0x8012, [0x54a8] = 0x8014,
  [0x54a9] = 0x8016, [0x54aa] = 0x801c, [0x54ab] = 0x8020, [0x54ac] = 0x8022,
  [0x54ad] = 0x8025, [0x54ae] = 0x8026, [0x54af] = 0x8027, [0x54b0] = 0x8029,
  [0x54b1] = 0x8028, [0x54b2] = 0x8031, [0x54b3] = 0x800b, [0x54b4] = 0x8035,
  [0x54b5] = 0x8043, [0x54b6] = 0x8046, [0x54b7] = 0x804d, [0x54b8] = 0x8052,
  [0x54b9] = 0x8069, [0x54ba] = 0x8071, [0x54bb] = 0x8983, [0x54bc] = 0x9878,
  [0x54bd] = 0x9880, [0x54be] = 0x9883, [0x54c0] = 0x99fa, [0x54c1] = 0x99fb,
  [0x54c2] = 0x99fc, [0x54c3] = 0x99fd, [0x54c4] = 0x99fe, [0x54c5] = 0x99ff,
  [0x54c6] = 0x9a00, [0x54c7] = 0x9a01, [0x54c8] = 0x9a02, [0x54c9] = 0x9a03,
  [0x54ca] = 0x9a04, [0x54cb] = 0x9a05, [0x54cc] = 0x9a06, [0x54cd] = 0x9a07,
  [0x54ce] = 0x9a08, [0x54cf] = 0x9a09, [0x54d0] = 0x9a0a, [0x54d1] = 0x9a0b,
  [0x54d2] = 0x9a0c, [0x54d3] = 0x9a0d, [0x54d4] = 0x9a0e, [0x54d5] = 0x9a0f,
  [0x54d6] = 0x9a10, [0x54d7] = 0x9a11, [0x54d8] = 0x9a12, [0x54d9] = 0x9a13,
  [0x54da] = 0x9a14, [0x54db] = 0x9a15, [0x54dc] = 0x9a16, [0x54dd] = 0x9a17,
  [0x54de] = 0x9a18, [0x54df] = 0x9a19, [0x54e0] = 0x9a1a, [0x54e1] = 0x9a1b,
  [0x54e2] = 0x9a1c, [0x54e3] = 0x9a1d, [0x54e4] = 0x9a1e, [0x54e5] = 0x9a1f,
  [0x54e6] = 0x9a20, [0x54e7] = 0x9a21, [0x54e8] = 0x9a22, [0x54e9] = 0x9a23,
  [0x54ea] = 0x9a24, [0x54eb] = 0x9a25, [0x54ec] = 0x9a26, [0x54ed] = 0x9a27,
  [0x54ee] = 0x9a28, [0x54ef] = 0x9a29, [0x54f0] = 0x9a2a, [0x54f1] = 0x9a2b,
  [0x54f2] = 0x9a2c, [0x54f3] = 0x9a2d, [0x54f4] = 0x9a2e, [0x54f5] = 0x9a2f,
  [0x54f6] = 0x9a30, [0x54f7] = 0x9a31, [0x54f8] = 0x9a32, [0x54f9] = 0x9a33,
  [0x54fa] = 0x9a34, [0x54fb] = 0x9a35, [0x54fc] = 0x9a36, [0x54fd] = 0x9a37,
  [0x54fe] = 0x9a38, [0x5500] = 0x9a39, [0x5501] = 0x9a3a, [0x5502] = 0x9a3b,
  [0x5503] = 0x9a3c, [0x5504] = 0x9a3d, [0x5505] = 0x9a3e, [0x5506] = 0x9a3f,
  [0x5507] = 0x9a40, [0x5508] = 0x9a41, [0x5509] = 0x9a42, [0x550a] = 0x9a43,
  [0x550b] = 0x9a44, [0x550c] = 0x9a45, [0x550d] = 0x9a46, [0x550e] = 0x9a47,
  [0x550f] = 0x9a48, [0x5510] = 0x9a49, [0x5511] = 0x9a4a, [0x5512] = 0x9a4b,
  [0x5513] = 0x9a4c, [0x5514] = 0x9a4d, [0x5515] = 0x9a4e, [0x5516] = 0x9a4f,
  [0x5517] = 0x9a50, [0x5518] = 0x9a51, [0x5519] = 0x9a52, [0x551a] = 0x9a53,
  [0x551b] = 0x9a54, [0x551c] = 0x9a55, [0x551d] = 0x9a56, [0x551e] = 0x9a57,
  [0x551f] = 0x9a58, [0x5520] = 0x9a59, [0x5521] = 0x9889, [0x5522] = 0x988c,
  [0x5523] = 0x988d, [0x5524] = 0x988f, [0x5525] = 0x9894, [0x5526] = 0x989a,
  [0x5527] = 0x989b, [0x5528] = 0x989e, [0x5529] = 0x989f, [0x552a] = 0x98a1,
  [0x552b] = 0x98a2, [0x552c] = 0x98a5, [0x552d] = 0x98a6, [0x552e] = 0x864d,
  [0x552f] = 0x8654, [0x5530] = 0x866c, [0x5531] = 0x866e, [0x5532] = 0x867f,
  [0x5533] = 0x867a, [0x5534] = 0x867c, [0x5535] = 0x867b, [0x5536] = 0x86a8,
  [0x5537] = 0x868d, [0x5538] = 0x868b, [0x5539] = 0x86ac, [0x553a] = 0x869d,
  [0x553b] = 0x86a7, [0x553c] = 0x86a3, [0x553d] = 0x86aa, [0x553e] = 0x8693,
  [0x553f] = 0x86a9, [0x5540] = 0x86b6, [0x5541] = 0x86c4, [0x5542] = 0x86b5,
  [0x5543] = 0x86ce, [0x5544] = 0x86b0, [0x5545] = 0x86ba, [0x5546] = 0x86b1,
  [0x5547] = 0x86af, [0x5548] = 0x86c9, [0x5549] = 0x86cf, [0x554a] = 0x86b4,
  [0x554b] = 0x86e9, [0x554c] = 0x86f1, [0x554d] = 0x86f2, [0x554e] = 0x86ed,
  [0x554f] = 0x86f3, [0x5550] = 0x86d0, [0x5551] = 0x8713, [0x5552] = 0x86de,
  [0x5553] = 0x86f4, [0x5554] = 0x86df, [0x5555] = 0x86d8, [0x5556] = 0x86d1,
  [0x5557] = 0x8703, [0x5558] = 0x8707, [0x5559] = 0x86f8, [0x555a] = 0x8708,
  [0x555b] = 0x870a, [0x555c] = 0x870d, [0x555d] = 0x8709, [0x555e] = 0x8723,
  [0x555f] = 0x873b, [0x5560] = 0x871e, [0x5561] = 0x8725, [0x5562] = 0x872e,
  [0x5563] = 0x871a, [0x5564] = 0x873e, [0x5565] = 0x8748, [0x5566] = 0x8734,
  [0x5567] = 0x8731, [0x5568] = 0x8729, [0x5569] = 0x8737, [0x556a] = 0x873f,
  [0x556b] = 0x8782, [0x556c] = 0x8722, [0x556d] = 0x877d, [0x556e] = 0x877e,
  [0x556f] = 0x877b, [0x5570] = 0x8760, [0x5571] = 0x8770, [0x5572] = 0x874c,
  [0x5573] = 0x876e, [0x5574] = 0x878b, [0x5575] = 0x8753, [0x5576] = 0x8763,
  [0x5577] = 0x877c, [0x5578] = 0x8764, [0x5579] = 0x8759, [0x557a] = 0x8765,
  [0x557b] = 0x8793, [0x557c] = 0x87af, [0x557d] = 0x87a8, [0x557e] = 0x87d2,
  [0x5580] = 0x9a5a, [0x5581] = 0x9a5b, [0x5582] = 0x9a5c, [0x5583] = 0x9a5d,
  [0x5584] = 0x9a5e, [0x5585] = 0x9a5f, [0x5586] = 0x9a60, [0x5587] = 0x9a61,
  [0x5588] = 0x9a62, [0x5589] = 0x9a63, [0x558a] = 0x9a64, [0x558b] = 0x9a65,
  [0x558c] = 0x9a66, [0x558d] = 0x9a67, [0x558e] = 0x9a68, [0x558f] = 0x9a69,
  [0x5590] = 0x9a6a, [0x5591] = 0x9a6b, [0x5592] = 0x9a72, [0x5593] = 0x9a83,
  [0x5594] = 0x9a89, [0x5595] = 0x9a8d, [0x5596] = 0x9a8e, [0x5597] = 0x9a94,
  [0x5598] = 0x9a95, [0x5599] = 0x9a99, [0x559a] = 0x9aa6, [0x559b] = 0x9aa9,
  [0x559c] = 0x9aaa, [0x559d] = 0x9aab, [0x559e] = 0x9aac, [0x559f] = 0x9aad,
  [0x55a0] = 0x9aae, [0x55a1] = 0x9aaf, [0x55a2] = 0x9ab2, [0x55a3] = 0x9ab3,
  [0x55a4] = 0x9ab4, [0x55a5] = 0x9ab5, [0x55a6] = 0x9ab9, [0x55a7] = 0x9abb,
  [0x55a8] = 0x9abd, [0x55a9] = 0x9abe, [0x55aa] = 0x9abf, [0x55ab] = 0x9ac3,
  [0x55ac] = 0x9ac4, [0x55ad] = 0x9ac6, [0x55ae] = 0x9ac7, [0x55af] = 0x9ac8,
  [0x55b0] = 0x9ac9, [0x55b1] = 0x9aca, [0x55b2] = 0x9acd, [0x55b3] = 0x9ace,
  [0x55b4] = 0x9acf, [0x55b5] = 0x9ad0, [0x55b6] = 0x9ad2, [0x55b7] = 0x9ad4,
  [0x55b8] = 0x9ad5, [0x55b9] = 0x9ad6, [0x55ba] = 0x9ad7, [0x55bb] = 0x9ad9,
  [0x55bc] = 0x9ada, [0x55bd] = 0x9adb, [0x55be] = 0x9adc, [0x55c0] = 0x9add,
  [0x55c1] = 0x9ade, [0x55c2] = 0x9ae0, [0x55c3] = 0x9ae2, [0x55c4] = 0x9ae3,
  [0x55c5] = 0x9ae4, [0x55c6] = 0x9ae5, [0x55c7] = 0x9ae7, [0x55c8] = 0x9ae8,
  [0x55c9] = 0x9ae9, [0x55ca] = 0x9aea, [0x55cb] = 0x9aec, [0x55cc] = 0x9aee,
  [0x55cd] = 0x9af0, [0x55ce] = 0x9af1, [0x55cf] = 0x9af2, [0x55d0] = 0x9af3,
  [0x55d1] = 0x9af4, [0x55d2] = 0x9af5, [0x55d3] = 0x9af6, [0x55d4] = 0x9af7,
  [0x55d5] = 0x9af8, [0x55d6] = 0x9afa, [0x55d7] = 0x9afc, [0x55d8] = 0x9afd,
  [0x55d9] = 0x9afe, [0x55da] = 0x9aff, [0x55db] = 0x9b00, [0x55dc] = 0x9b01,
  [0x55dd] = 0x9b02, [0x55de] = 0x9b04, [0x55df] = 0x9b05, [0x55e0] = 0x9b06,
  [0x55e1] = 0x87c6, [0x55e2] = 0x8788, [0x55e3] = 0x8785, [0x55e4] = 0x87ad,
  [0x55e5] = 0x8797, [0x55e6] = 0x8783, [0x55e7] = 0x87ab, [0x55e8] = 0x87e5,
  [0x55e9] = 0x87ac, [0x55ea] = 0x87b5, [0x55eb] = 0x87b3, [0x55ec] = 0x87cb,
  [0x55ed] = 0x87d3, [0x55ee] = 0x87bd, [0x55ef] = 0x87d1, [0x55f0] = 0x87c0,
  [0x55f1] = 0x87ca, [0x55f2] = 0x87db, [0x55f3] = 0x87ea, [0x55f4] = 0x87e0,
  [0x55f5] = 0x87ee, [0x55f6] = 0x8816, [0x55f7] = 0x8813, [0x55f8] = 0x87fe,
  [0x55f9] = 0x880a, [0x55fa] = 0x881b, [0x55fb] = 0x8821, [0x55fc] = 0x8839,
  [0x55fd] = 0x883c, [0x55fe] = 0x7f36, [0x55ff] = 0x7f42, [0x5600] = 0x7f44,
  [0x5601] = 0x7f45, [0x5602] = 0x8210, [0x5603] = 0x7afa, [0x5604] = 0x7afd,
  [0x5605] = 0x7b08, [0x5606] = 0x7b03, [0x5607] = 0x7b04, [0x5608] = 0x7b15,
  [0x5609] = 0x7b0a, [0x560a] = 0x7b2b, [0x560b] = 0x7b0f, [0x560c] = 0x7b47,
  [0x560d] = 0x7b38, [0x560e] = 0x7b2a, [0x560f] = 0x7b19, [0x5610] = 0x7b2e,
  [0x5611] = 0x7b31, [0x5612] = 0x7b20, [0x5613] = 0x7b25, [0x5614] = 0x7b24,
  [0x5615] = 0x7b33, [0x5616] = 0x7b3e, [0x5617] = 0x7b1e, [0x5618] = 0x7b58,
  [0x5619] = 0x7b5a, [0x561a] = 0x7b45, [0x561b] = 0x7b75, [0x561c] = 0x7b4c,
  [0x561d] = 0x7b5d, [0x561e] = 0x7b60, [0x561f] = 0x7b6e, [0x5620] = 0x7b7b,
  [0x5621] = 0x7b62, [0x5622] = 0x7b72, [0x5623] = 0x7b71, [0x5624] = 0x7b90,
  [0x5625] = 0x7ba6, [0x5626] = 0x7ba7, [0x5627] = 0x7bb8, [0x5628] = 0x7bac,
  [0x5629] = 0x7b9d, [0x562a] = 0x7ba8, [0x562b] = 0x7b85, [0x562c] = 0x7baa,
  [0x562d] = 0x7b9c, [0x562e] = 0x7ba2, [0x562f] = 0x7bab, [0x5630] = 0x7bb4,
  [0x5631] = 0x7bd1, [0x5632] = 0x7bc1, [0x5633] = 0x7bcc, [0x5634] = 0x7bdd,
  [0x5635] = 0x7bda, [0x5636] = 0x7be5, [0x5637] = 0x7be6, [0x5638] = 0x7bea,
  [0x5639] = 0x7c0c, [0x563a] = 0x7bfe, [0x563b] = 0x7bfc, [0x563c] = 0x7c0f,
  [0x563d] = 0x7c16, [0x563e] = 0x7c0b, [0x5640] = 0x9b07, [0x5641] = 0x9b09,
  [0x5642] = 0x9b0a, [0x5643] = 0x9b0b, [0x5644] = 0x9b0c, [0x5645] = 0x9b0d,
  [0x5646] = 0x9b0e, [0x5647] = 0x9b10, [0x5648] = 0x9b11, [0x5649] = 0x9b12,
  [0x564a] = 0x9b14, [0x564b] = 0x9b15, [0x564c] = 0x9b16, [0x564d] = 0x9b17,
  [0x564e] = 0x9b18, [0x564f] = 0x9b19, [0x5650] = 0x9b1a, [0x5651] = 0x9b1b,
  [0x5652] = 0x9b1c, [0x5653] = 0x9b1d, [0x5654] = 0x9b1e, [0x5655] = 0x9b20,
  [0x5656] = 0x9b21, [0x5657] = 0x9b22, [0x5658] = 0x9b24, [0x5659] = 0x9b25,
  [0x565a] = 0x9b26, [0x565b] = 0x9b27, [0x565c] = 0x9b28, [0x565d] = 0x9b29,
  [0x565e] = 0x9b2a, [0x565f] = 0x9b2b, [0x5660] = 0x9b2c, [0x5661] = 0x9b2d,
  [0x5662] = 0x9b2e, [0x5663] = 0x9b30, [0x5664] = 0x9b31, [0x5665] = 0x9b33,
  [0x5666] = 0x9b34, [0x5667] = 0x9b35, [0x5668] = 0x9b36, [0x5669] = 0x9b37,
  [0x566a] = 0x9b38, [0x566b] = 0x9b39, [0x566c] = 0x9b3a, [0x566d] = 0x9b3d,
  [0x566e] = 0x9b3e, [0x566f] = 0x9b3f, [0x5670] = 0x9b40, [0x5671] = 0x9b46,
  [0x5672] = 0x9b4a, [0x5673] = 0x9b4b, [0x5674] = 0x9b4c, [0x5675] = 0x9b4e,
  [0x5676] = 0x9b50, [0x5677] = 0x9b52, [0x5678] = 0x9b53, [0x5679] = 0x9b55,
  [0x567a] = 0x9b56, [0x567b] = 0x9b57, [0x567c] = 0x9b58, [0x567d] = 0x9b59,
  [0x567e] = 0x9b5a, [0x5680] = 0x9b5b, [0x5681] = 0x9b5c, [0x5682] = 0x9b5d,
  [0x5683] = 0x9b5e, [0x5684] = 0x9b5f, [0x5685] = 0x9b60, [0x5686] = 0x9b61,
  [0x5687] = 0x9b62, [0x5688] = 0x9b63, [0x5689] = 0x9b64, [0x568a] = 0x9b65,
  [0x568b] = 0x9b66, [0x568c] = 0x9b67, [0x568d] = 0x9b68, [0x568e] = 0x9b69,
  [0x568f] = 0x9b6a, [0x5690] = 0x9b6b, [0x5691] = 0x9b6c, [0x5692] = 0x9b6d,
  [0x5693] = 0x9b6e, [0x5694] = 0x9b6f, [0x5695] = 0x9b70, [0x5696] = 0x9b71,
  [0x5697] = 0x9b72, [0x5698] = 0x9b73, [0x5699] = 0x9b74, [0x569a] = 0x9b75,
  [0x569b] = 0x9b76, [0x569c] = 0x9b77, [0x569d] = 0x9b78, [0x569e] = 0x9b79,
  [0x569f] = 0x9b7a, [0x56a0] = 0x9b7b, [0x56a1] = 0x7c1f, [0x56a2] = 0x7c2a,
  [0x56a3] = 0x7c26, [0x56a4] = 0x7c38, [0x56a5] = 0x7c41, [0x56a6] = 0x7c40,
  [0x56a7] = 0x81fe, [0x56a8] = 0x8201, [0x56a9] = 0x8202, [0x56aa] = 0x8204,
  [0x56ab] = 0x81ec, [0x56ac] = 0x8844, [0x56ad] = 0x8221, [0x56ae] = 0x8222,
  [0x56af] = 0x8223, [0x56b0] = 0x822d, [0x56b1] = 0x822f, [0x56b2] = 0x8228,
  [0x56b3] = 0x822b, [0x56b4] = 0x8238, [0x56b5] = 0x823b, [0x56b6] = 0x8233,
  [0x56b7] = 0x8234, [0x56b8] = 0x823e, [0x56b9] = 0x8244, [0x56ba] = 0x8249,
  [0x56bb] = 0x824b, [0x56bc] = 0x824f, [0x56bd] = 0x825a, [0x56be] = 0x825f,
  [0x56bf] = 0x8268, [0x56c0] = 0x887e, [0x56c1] = 0x8885, [0x56c2] = 0x8888,
  [0x56c3] = 0x88d8, [0x56c4] = 0x88df, [0x56c5] = 0x895e, [0x56c6] = 0x7f9d,
  [0x56c7] = 0x7f9f, [0x56c8] = 0x7fa7, [0x56c9] = 0x7faf, [0x56ca] = 0x7fb0,
  [0x56cb] = 0x7fb2, [0x56cc] = 0x7c7c, [0x56cd] = 0x6549, [0x56ce] = 0x7c91,
  [0x56cf] = 0x7c9d, [0x56d0] = 0x7c9c, [0x56d1] = 0x7c9e, [0x56d2] = 0x7ca2,
  [0x56d3] = 0x7cb2, [0x56d4] = 0x7cbc, [0x56d5] = 0x7cbd, [0x56d6] = 0x7cc1,
  [0x56d7] = 0x7cc7, [0x56d8] = 0x7ccc, [0x56d9] = 0x7ccd, [0x56da] = 0x7cc8,
  [0x56db] = 0x7cc5, [0x56dc] = 0x7cd7, [0x56dd] = 0x7ce8, [0x56de] = 0x826e,
  [0x56df] = 0x66a8, [0x56e0] = 0x7fbf, [0x56e1] = 0x7fce, [0x56e2] = 0x7fd5,
  [0x56e3] = 0x7fe5, [0x56e4] = 0x7fe1, [0x56e5] = 0x7fe6, [0x56e6] = 0x7fe9,
  [0x56e7] = 0x7fee, [0x56e8] = 0x7ff3, [0x56e9] = 0x7cf8, [0x56ea] = 0x7d77,
  [0x56eb] = 0x7da6, [0x56ec] = 0x7dae, [0x56ed] = 0x7e47, [0x56ee] = 0x7e9b,
  [0x56ef] = 0x9eb8, [0x56f0] = 0x9eb4, [0x56f1] = 0x8d73, [0x56f2] = 0x8d84,
  [0x56f3] = 0x8d94, [0x56f4] = 0x8d91, [0x56f5] = 0x8db1, [0x56f6] = 0x8d67,
  [0x56f7] = 0x8d6d, [0x56f8] = 0x8c47, [0x56f9] = 0x8c49, [0x56fa] = 0x914a,
  [0x56fb] = 0x9150, [0x56fc] = 0x914e, [0x56fd] = 0x914f, [0x56fe] = 0x9164,
  [0x5700] = 0x9b7c, [0x5701] = 0x9b7d, [0x5702] = 0x9b7e, [0x5703] = 0x9b7f,
  [0x5704] = 0x9b80, [0x5705] = 0x9b81, [0x5706] = 0x9b82, [0x5707] = 0x9b83,
  [0x5708] = 0x9b84, [0x5709] = 0x9b85, [0x570a] = 0x9b86, [0x570b] = 0x9b87,
  [0x570c] = 0x9b88, [0x570d] = 0x9b89, [0x570e] = 0x9b8a, [0x570f] = 0x9b8b,
  [0x5710] = 0x9b8c, [0x5711] = 0x9b8d, [0x5712] = 0x9b8e, [0x5713] = 0x9b8f,
  [0x5714] = 0x9b90, [0x5715] = 0x9b91, [0x5716] = 0x9b92, [0x5717] = 0x9b93,
  [0x5718] = 0x9b94, [0x5719] = 0x9b95, [0x571a] = 0x9b96, [0x571b] = 0x9b97,
  [0x571c] = 0x9b98, [0x571d] = 0x9b99, [0x571e] = 0x9b9a, [0x571f] = 0x9b9b,
  [0x5720] = 0x9b9c, [0x5721] = 0x9b9d, [0x5722] = 0x9b9e, [0x5723] = 0x9b9f,
  [0x5724] = 0x9ba0, [0x5725] = 0x9ba1, [0x5726] = 0x9ba2, [0x5727] = 0x9ba3,
  [0x5728] = 0x9ba4, [0x5729] = 0x9ba5, [0x572a] = 0x9ba6, [0x572b] = 0x9ba7,
  [0x572c] = 0x9ba8, [0x572d] = 0x9ba9, [0x572e] = 0x9baa, [0x572f] = 0x9bab,
  [0x5730] = 0x9bac, [0x5731] = 0x9bad, [0x5732] = 0x9bae, [0x5733] = 0x9baf,
  [0x5734] = 0x9bb0, [0x5735] = 0x9bb1, [0x5736] = 0x9bb2, [0x5737] = 0x9bb3,
  [0x5738] = 0x9bb4, [0x5739] = 0x9bb5, [0x573a] = 0x9bb6, [0x573b] = 0x9bb7,
  [0x573c] = 0x9bb8, [0x573d] = 0x9bb9, [0x573e] = 0x9bba, [0x5740] = 0x9bbb,
  [0x5741] = 0x9bbc, [0x5742] = 0x9bbd, [0x5743] = 0x9bbe, [0x5744] = 0x9bbf,
  [0x5745] = 0x9bc0, [0x5746] = 0x9bc1, [0x5747] = 0x9bc2, [0x5748] = 0x9bc3,
  [0x5749] = 0x9bc4, [0x574a] = 0x9bc5, [0x574b] = 0x9bc6, [0x574c] = 0x9bc7,
  [0x574d] = 0x9bc8, [0x574e] = 0x9bc9, [0x574f] = 0x9bca, [0x5750] = 0x9bcb,
  [0x5751] = 0x9bcc, [0x5752] = 0x9bcd, [0x5753] = 0x9bce, [0x5754] = 0x9bcf,
  [0x5755] = 0x9bd0, [0x5756] = 0x9bd1, [0x5757] = 0x9bd2, [0x5758] = 0x9bd3,
  [0x5759] = 0x9bd4, [0x575a] = 0x9bd5, [0x575b] = 0x9bd6, [0x575c] = 0x9bd7,
  [0x575d] = 0x9bd8, [0x575e] = 0x9bd9, [0x575f] = 0x9bda, [0x5760] = 0x9bdb,
  [0x5761] = 0x9162, [0x5762] = 0x9161, [0x5763] = 0x9170, [0x5764] = 0x9169,
  [0x5765] = 0x916f, [0x5766] = 0x917d, [0x5767] = 0x917e, [0x5768] = 0x9172,
  [0x5769] = 0x9174, [0x576a] = 0x9179, [0x576b] = 0x918c, [0x576c] = 0x9185,
  [0x576d] = 0x9190, [0x576e] = 0x918d, [0x576f] = 0x9191, [0x5770] = 0x91a2,
  [0x5771] = 0x91a3, [0x5772] = 0x91aa, [0x5773] = 0x91ad, [0x5774] = 0x91ae,
  [0x5775] = 0x91af, [0x5776] = 0x91b5, [0x5777] = 0x91b4, [0x5778] = 0x91ba,
  [0x5779] = 0x8c55, [0x577a] = 0x9e7e, [0x577b] = 0x8db8, [0x577c] = 0x8deb,
  [0x577d] = 0x8e05, [0x577e] = 0x8e59, [0x577f] = 0x8e69, [0x5780] = 0x8db5,
  [0x5781] = 0x8dbf, [0x5782] = 0x8dbc, [0x5783] = 0x8dba, [0x5784] = 0x8dc4,
  [0x5785] = 0x8dd6, [0x5786] = 0x8dd7, [0x5787] = 0x8dda, [0x5788] = 0x8dde,
  [0x5789] = 0x8dce, [0x578a] = 0x8dcf, [0x578b] = 0x8ddb, [0x578c] = 0x8dc6,
  [0x578d] = 0x8dec, [0x578e] = 0x8df7, [0x578f] = 0x8df8, [0x5790] = 0x8de3,
  [0x5791] = 0x8df9, [0x5792] = 0x8dfb, [0x5793] = 0x8de4, [0x5794] = 0x8e09,
  [0x5795] = 0x8dfd, [0x5796] = 0x8e14, [0x5797] = 0x8e1d, [0x5798] = 0x8e1f,
  [0x5799] = 0x8e2c, [0x579a] = 0x8e2e, [0x579b] = 0x8e23, [0x579c] = 0x8e2f,
  [0x579d] = 0x8e3a, [0x579e] = 0x8e40, [0x579f] = 0x8e39, [0x57a0] = 0x8e35,
  [0x57a1] = 0x8e3d, [0x57a2] = 0x8e31, [0x57a3] = 0x8e49, [0x57a4] = 0x8e41,
  [0x57a5] = 0x8e42, [0x57a6] = 0x8e51, [0x57a7] = 0x8e52, [0x57a8] = 0x8e4a,
  [0x57a9] = 0x8e70, [0x57aa] = 0x8e76, [0x57ab] = 0x8e7c, [0x57ac] = 0x8e6f,
  [0x57ad] = 0x8e74, [0x57ae] = 0x8e85, [0x57af] = 0x8e8f, [0x57b0] = 0x8e94,
  [0x57b1] = 0x8e90, [0x57b2] = 0x8e9c, [0x57b3] = 0x8e9e, [0x57b4] = 0x8c78,
  [0x57b5] = 0x8c82, [0x57b6] = 0x8c8a, [0x57b7] = 0x8c85, [0x57b8] = 0x8c98,
  [0x57b9] = 0x8c94, [0x57ba] = 0x659b, [0x57bb] = 0x89d6, [0x57bc] = 0x89de,
  [0x57bd] = 0x89da, [0x57be] = 0x89dc, [0x57c0] = 0x9bdc, [0x57c1] = 0x9bdd,
  [0x57c2] = 0x9bde, [0x57c3] = 0x9bdf, [0x57c4] = 0x9be0, [0x57c5] = 0x9be1,
  [0x57c6] = 0x9be2, [0x57c7] = 0x9be3, [0x57c8] = 0x9be4, [0x57c9] = 0x9be5,
  [0x57ca] = 0x9be6, [0x57cb] = 0x9be7, [0x57cc] = 0x9be8, [0x57cd] = 0x9be9,
  [0x57ce] = 0x9bea, [0x57cf] = 0x9beb, [0x57d0] = 0x9bec, [0x57d1] = 0x9bed,
  [0x57d2] = 0x9bee, [0x57d3] = 0x9bef, [0x57d4] = 0x9bf0, [0x57d5] = 0x9bf1,
  [0x57d6] = 0x9bf2, [0x57d7] = 0x9bf3, [0x57d8] = 0x9bf4, [0x57d9] = 0x9bf5,
  [0x57da] = 0x9bf6, [0x57db] = 0x9bf7, [0x57dc] = 0x9bf8, [0x57dd] = 0x9bf9,
  [0x57de] = 0x9bfa, [0x57df] = 0x9bfb, [0x57e0] = 0x9bfc, [0x57e1] = 0x9bfd,
  [0x57e2] = 0x9bfe, [0x57e3] = 0x9bff, [0x57e4] = 0x9c00, [0x57e5] = 0x9c01,
  [0x57e6] = 0x9c02, [0x57e7] = 0x9c03, [0x57e8] = 0x9c04, [0x57e9] = 0x9c05,
  [0x57ea] = 0x9c06, [0x57eb] = 0x9c07, [0x57ec] = 0x9c08, [0x57ed] = 0x9c09,
  [0x57ee] = 0x9c0a, [0x57ef] = 0x9c0b, [0x57f0] = 0x9c0c, [0x57f1] = 0x9c0d,
  [0x57f2] = 0x9c0e, [0x57f3] = 0x9c0f, [0x57f4] = 0x9c10, [0x57f5] = 0x9c11,
  [0x57f6] = 0x9c12, [0x57f7] = 0x9c13, [0x57f8] = 0x9c14, [0x57f9] = 0x9c15,
  [0x57fa] = 0x9c16, [0x57fb] = 0x9c17, [0x57fc] = 0x9c18, [0x57fd] = 0x9c19,
  [0x57fe] = 0x9c1a, [0x5800] = 0x9c1b, [0x5801] = 0x9c1c, [0x5802] = 0x9c1d,
  [0x5803] = 0x9c1e, [0x5804] = 0x9c1f, [0x5805] = 0x9c20, [0x5806] = 0x9c21,
  [0x5807] = 0x9c22, [0x5808] = 0x9c23, [0x5809] = 0x9c24, [0x580a] = 0x9c25,
  [0x580b] = 0x9c26, [0x580c] = 0x9c27, [0x580d] = 0x9c28, [0x580e] = 0x9c29,
  [0x580f] = 0x9c2a, [0x5810] = 0x9c2b, [0x5811] = 0x9c2c, [0x5812] = 0x9c2d,
  [0x5813] = 0x9c2e, [0x5814] = 0x9c2f, [0x5815] = 0x9c30, [0x5816] = 0x9c31,
  [0x5817] = 0x9c32, [0x5818] = 0x9c33, [0x5819] = 0x9c34, [0x581a] = 0x9c35,
  [0x581b] = 0x9c36, [0x581c] = 0x9c37, [0x581d] = 0x9c38, [0x581e] = 0x9c39,
  [0x581f] = 0x9c3a, [0x5820] = 0x9c3b, [0x5821] = 0x89e5, [0x5822] = 0x89eb,
  [0x5823] = 0x89ef, [0x5824] = 0x8a3e, [0x5825] = 0x8b26, [0x5826] = 0x9753,
  [0x5827] = 0x96e9, [0x5828] = 0x96f3, [0x5829] = 0x96ef, [0x582a] = 0x9706,
  [0x582b] = 0x9701, [0x582c] = 0x9708, [0x582d] = 0x970f, [0x582e] = 0x970e,
  [0x582f] = 0x972a, [0x5830] = 0x972d, [0x5831] = 0x9730, [0x5832] = 0x973e,
  [0x5833] = 0x9f80, [0x5834] = 0x9f83, [0x5835] = 0x9f85, [0x5836] = 0x9f86,
  [0x5837] = 0x9f87, [0x5838] = 0x9f88, [0x5839] = 0x9f89, [0x583a] = 0x9f8a,
  [0x583b] = 0x9f8c, [0x583c] = 0x9efe, [0x583d] = 0x9f0b, [0x583e] = 0x9f0d,
  [0x583f] = 0x96b9, [0x5840] = 0x96bc, [0x5841] = 0x96bd, [0x5842] = 0x96ce,
  [0x5843] = 0x96d2, [0x5844] = 0x77bf, [0x5845] = 0x96e0, [0x5846] = 0x928e,
  [0x5847] = 0x92ae, [0x5848] = 0x92c8, [0x5849] = 0x933e, [0x584a] = 0x936a,
  [0x584b] = 0x93ca, [0x584c] = 0x938f, [0x584d] = 0x943e, [0x584e] = 0x946b,
  [0x584f] = 0x9c7f, [0x5850] = 0x9c82, [0x5851] = 0x9c85, [0x5852] = 0x9c86,
  [0x5853] = 0x9c87, [0x5854] = 0x9c88, [0x5855] = 0x7a23, [0x5856] = 0x9c8b,
  [0x5857] = 0x9c8e, [0x5858] = 0x9c90, [0x5859] = 0x9c91, [0x585a] = 0x9c92,
  [0x585b] = 0x9c94, [0x585c] = 0x9c95, [0x585d] = 0x9c9a, [0x585e] = 0x9c9b,
  [0x585f] = 0x9c9e, [0x5860] = 0x9c9f, [0x5861] = 0x9ca0, [0x5862] = 0x9ca1,
  [0x5863] = 0x9ca2, [0x5864] = 0x9ca3, [0x5865] = 0x9ca5, [0x5866] = 0x9ca6,
  [0x5867] = 0x9ca7, [0x5868] = 0x9ca8, [0x5869] = 0x9ca9, [0x586a] = 0x9cab,
  [0x586b] = 0x9cad, [0x586c] = 0x9cae, [0x586d] = 0x9cb0, [0x586e] = 0x9cb1,
  [0x586f] = 0x9cb2, [0x5870] = 0x9cb3, [0x5871] = 0x9cb4, [0x5872] = 0x9cb5,
  [0x5873] = 0x9cb6, [0x5874] = 0x9cb7, [0x5875] = 0x9cba, [0x5876] = 0x9cbb,
  [0x5877] = 0x9cbc, [0x5878] = 0x9cbd, [0x5879] = 0x9cc4, [0x587a] = 0x9cc5,
  [0x587b] = 0x9cc6, [0x587c] = 0x9cc7, [0x587d] = 0x9cca, [0x587e] = 0x9ccb,
  [0x5880] = 0x9c3c, [0x5881] = 0x9c3d, [0x5882] = 0x9c3e, [0x5883] = 0x9c3f,
  [0x5884] = 0x9c40, [0x5885] = 0x9c41, [0x5886] = 0x9c42, [0x5887] = 0x9c43,
  [0x5888] = 0x9c44, [0x5889] = 0x9c45, [0x588a] = 0x9c46, [0x588b] = 0x9c47,
  [0x588c] = 0x9c48, [0x588d] = 0x9c49, [0x588e] = 0x9c4a, [0x588f] = 0x9c4b,
  [0x5890] = 0x9c4c, [0x5891] = 0x9c4d, [0x5892] = 0x9c4e, [0x5893] = 0x9c4f,
  [0x5894] = 0x9c50, [0x5895] = 0x9c51, [0x5896] = 0x9c52, [0x5897] = 0x9c53,
  [0x5898] = 0x9c54, [0x5899] = 0x9c55, [0x589a] = 0x9c56, [0x589b] = 0x9c57,
  [0x589c] = 0x9c58, [0x589d] = 0x9c59, [0x589e] = 0x9c5a, [0x589f] = 0x9c5b,
  [0x58a0] = 0x9c5c, [0x58a1] = 0x9c5d, [0x58a2] = 0x9c5e, [0x58a3] = 0x9c5f,
  [0x58a4] = 0x9c60, [0x58a5] = 0x9c61, [0x58a6] = 0x9c62, [0x58a7] = 0x9c63,
  [0x58a8] = 0x9c64, [0x58a9] = 0x9c65, [0x58aa] = 0x9c66, [0x58ab] = 0x9c67,
  [0x58ac] = 0x9c68, [0x58ad] = 0x9c69, [0x58ae] = 0x9c6a, [0x58af] = 0x9c6b,
  [0x58b0] = 0x9c6c, [0x58b1] = 0x9c6d, [0x58b2] = 0x9c6e, [0x58b3] = 0x9c6f,
  [0x58b4] = 0x9c70, [0x58b5] = 0x9c71, [0x58b6] = 0x9c72, [0x58b7] = 0x9c73,
  [0x58b8] = 0x9c74, [0x58b9] = 0x9c75, [0x58ba] = 0x9c76, [0x58bb] = 0x9c77,
  [0x58bc] = 0x9c78, [0x58bd] = 0x9c79, [0x58be] = 0x9c7a, [0x58c0] = 0x9c7b,
  [0x58c1] = 0x9c7d, [0x58c2] = 0x9c7e, [0x58c3] = 0x9c80, [0x58c4] = 0x9c83,
  [0x58c5] = 0x9c84, [0x58c6] = 0x9c89, [0x58c7] = 0x9c8a, [0x58c8] = 0x9c8c,
  [0x58c9] = 0x9c8f, [0x58ca] = 0x9c93, [0x58cb] = 0x9c96, [0x58cc] = 0x9c97,
  [0x58cd] = 0x9c98, [0x58ce] = 0x9c99, [0x58cf] = 0x9c9d, [0x58d0] = 0x9caa,
  [0x58d1] = 0x9cac, [0x58d2] = 0x9caf, [0x58d3] = 0x9cb9, [0x58d4] = 0x9cbe,
  [0x58d5] = 0x9cbf, [0x58d6] = 0x9cc0, [0x58d7] = 0x9cc1, [0x58d8] = 0x9cc2,
  [0x58d9] = 0x9cc8, [0x58da] = 0x9cc9, [0x58db] = 0x9cd1, [0x58dc] = 0x9cd2,
  [0x58dd] = 0x9cda, [0x58de] = 0x9cdb, [0x58df] = 0x9ce0, [0x58e0] = 0x9ce1,
  [0x58e1] = 0x9ccc, [0x58e2] = 0x9ccd, [0x58e3] = 0x9cce, [0x58e4] = 0x9ccf,
  [0x58e5] = 0x9cd0, [0x58e6] = 0x9cd3, [0x58e7] = 0x9cd4, [0x58e8] = 0x9cd5,
  [0x58e9] = 0x9cd7, [0x58ea] = 0x9cd8, [0x58eb] = 0x9cd9, [0x58ec] = 0x9cdc,
  [0x58ed] = 0x9cdd, [0x58ee] = 0x9cdf, [0x58ef] = 0x9ce2, [0x58f0] = 0x977c,
  [0x58f1] = 0x9785, [0x58f2] = 0x9791, [0x58f3] = 0x9792, [0x58f4] = 0x9794,
  [0x58f5] = 0x97af, [0x58f6] = 0x97ab, [0x58f7] = 0x97a3, [0x58f8] = 0x97b2,
  [0x58f9] = 0x97b4, [0x58fa] = 0x9ab1, [0x58fb] = 0x9ab0, [0x58fc] = 0x9ab7,
  [0x58fd] = 0x9e58, [0x58fe] = 0x9ab6, [0x58ff] = 0x9aba, [0x5900] = 0x9abc,
  [0x5901] = 0x9ac1, [0x5902] = 0x9ac0, [0x5903] = 0x9ac5, [0x5904] = 0x9ac2,
  [0x5905] = 0x9acb, [0x5906] = 0x9acc, [0x5907] = 0x9ad1, [0x5908] = 0x9b45,
  [0x5909] = 0x9b43, [0x590a] = 0x9b47, [0x590b] = 0x9b49, [0x590c] = 0x9b48,
  [0x590d] = 0x9b4d, [0x590e] = 0x9b51, [0x590f] = 0x98e8, [0x5910] = 0x990d,
  [0x5911] = 0x992e, [0x5912] = 0x9955, [0x5913] = 0x9954, [0x5914] = 0x9adf,
  [0x5915] = 0x9ae1, [0x5916] = 0x9ae6, [0x5917] = 0x9aef, [0x5918] = 0x9aeb,
  [0x5919] = 0x9afb, [0x591a] = 0x9aed, [0x591b] = 0x9af9, [0x591c] = 0x9b08,
  [0x591d] = 0x9b0f, [0x591e] = 0x9b13, [0x591f] = 0x9b1f, [0x5920] = 0x9b23,
  [0x5921] = 0x9ebd, [0x5922] = 0x9ebe, [0x5923] = 0x7e3b, [0x5924] = 0x9e82,
  [0x5925] = 0x9e87, [0x5926] = 0x9e88, [0x5927] = 0x9e8b, [0x5928] = 0x9e92,
  [0x5929] = 0x93d6, [0x592a] = 0x9e9d, [0x592b] = 0x9e9f, [0x592c] = 0x9edb,
  [0x592d] = 0x9edc, [0x592e] = 0x9edd, [0x592f] = 0x9ee0, [0x5930] = 0x9edf,
  [0x5931] = 0x9ee2, [0x5932] = 0x9ee9, [0x5933] = 0x9ee7, [0x5934] = 0x9ee5,
  [0x5935] = 0x9eea, [0x5936] = 0x9eef, [0x5937] = 0x9f22, [0x5938] = 0x9f2c,
  [0x5939] = 0x9f2f, [0x593a] = 0x9f39, [0x593b] = 0x9f37, [0x593c] = 0x9f3d,
  [0x593d] = 0x9f3e, [0x593e] = 0x9f44, [0x5940] = 0x9ce3, [0x5941] = 0x9ce4,
  [0x5942] = 0x9ce5, [0x5943] = 0x9ce6, [0x5944] = 0x9ce7, [0x5945] = 0x9ce8,
  [0x5946] = 0x9ce9, [0x5947] = 0x9cea, [0x5948] = 0x9ceb, [0x5949] = 0x9cec,
  [0x594a] = 0x9ced, [0x594b] = 0x9cee, [0x594c] = 0x9cef, [0x594d] = 0x9cf0,
  [0x594e] = 0x9cf1, [0x594f] = 0x9cf2, [0x5950] = 0x9cf3, [0x5951] = 0x9cf4,
  [0x5952] = 0x9cf5, [0x5953] = 0x9cf6, [0x5954] = 0x9cf7, [0x5955] = 0x9cf8,
  [0x5956] = 0x9cf9, [0x5957] = 0x9cfa, [0x5958] = 0x9cfb, [0x5959] = 0x9cfc,
  [0x595a] = 0x9cfd, [0x595b] = 0x9cfe, [0x595c] = 0x9cff, [0x595d] = 0x9d00,
  [0x595e] = 0x9d01, [0x595f] = 0x9d02, [0x5960] = 0x9d03, [0x5961] = 0x9d04,
  [0x5962] = 0x9d05, [0x5963] = 0x9d06, [0x5964] = 0x9d07, [0x5965] = 0x9d08,
  [0x5966] = 0x9d09, [0x5967] = 0x9d0a, [0x5968] = 0x9d0b, [0x5969] = 0x9d0c,
  [0x596a] = 0x9d0d, [0x596b] = 0x9d0e, [0x596c] = 0x9d0f, [0x596d] = 0x9d10,
  [0x596e] = 0x9d11, [0x596f] = 0x9d12, [0x5970] = 0x9d13, [0x5971] = 0x9d14,
  [0x5972] = 0x9d15, [0x5973] = 0x9d16, [0x5974] = 0x9d17, [0x5975] = 0x9d18,
  [0x5976] = 0x9d19, [0x5977] = 0x9d1a, [0x5978] = 0x9d1b, [0x5979] = 0x9d1c,
  [0x597a] = 0x9d1d, [0x597b] = 0x9d1e, [0x597c] = 0x9d1f, [0x597d] = 0x9d20,
  [0x597e] = 0x9d21, [0x5980] = 0x9d22, [0x5981] = 0x9d23, [0x5982] = 0x9d24,
  [0x5983] = 0x9d25, [0x5984] = 0x9d26, [0x5985] = 0x9d27, [0x5986] = 0x9d28,
  [0x5987] = 0x9d29, [0x5988] = 0x9d2a, [0x5989] = 0x9d2b, [0x598a] = 0x9d2c,
  [0x598b] = 0x9d2d, [0x598c] = 0x9d2e, [0x598d] = 0x9d2f, [0x598e] = 0x9d30,
  [0x598f] = 0x9d31, [0x5990] = 0x9d32, [0x5991] = 0x9d33, [0x5992] = 0x9d34,
  [0x5993] = 0x9d35, [0x5994] = 0x9d36, [0x5995] = 0x9d37, [0x5996] = 0x9d38,
  [0x5997] = 0x9d39, [0x5998] = 0x9d3a, [0x5999] = 0x9d3b, [0x599a] = 0x9d3c,
  [0x599b] = 0x9d3d, [0x599c] = 0x9d3e, [0x599d] = 0x9d3f, [0x599e] = 0x9d40,
  [0x599f] = 0x9d41, [0x59a0] = 0x9d42, [0x59a1] = 0xe234, [0x59a2] = 0xe235,
  [0x59a3] = 0xe236, [0x59a4] = 0xe237, [0x59a5] = 0xe238, [0x59a6] = 0xe239,
  [0x59a7] = 0xe23a, [0x59a8] = 0xe23b, [0x59a9] = 0xe23c, [0x59aa] = 0xe23d,
  [0x59ab] = 0xe23e, [0x59ac] = 0xe23f, [0x59ad] = 0xe240, [0x59ae] = 0xe241,
  [0x59af] = 0xe242, [0x59b0] = 0xe243, [0x59b1] = 0xe244, [0x59b2] = 0xe245,
  [0x59b3] = 0xe246, [0x59b4] = 0xe247, [0x59b5] = 0xe248, [0x59b6] = 0xe249,
  [0x59b7] = 0xe24a, [0x59b8] = 0xe24b, [0x59b9] = 0xe24c, [0x59ba] = 0xe24d,
  [0x59bb] = 0xe24e, [0x59bc] = 0xe24f, [0x59bd] = 0xe250, [0x59be] = 0xe251,
  [0x59bf] = 0xe252, [0x59c0] = 0xe253, [0x59c1] = 0xe254, [0x59c2] = 0xe255,
  [0x59c3] = 0xe256, [0x59c4] = 0xe257, [0x59c5] = 0xe258, [0x59c6] = 0xe259,
  [0x59c7] = 0xe25a, [0x59c8] = 0xe25b, [0x59c9] = 0xe25c, [0x59ca] = 0xe25d,
  [0x59cb] = 0xe25e, [0x59cc] = 0xe25f, [0x59cd] = 0xe260, [0x59ce] = 0xe261,
  [0x59cf] = 0xe262, [0x59d0] = 0xe263, [0x59d1] = 0xe264, [0x59d2] = 0xe265,
  [0x59d3] = 0xe266, [0x59d4] = 0xe267, [0x59d5] = 0xe268, [0x59d6] = 0xe269,
  [0x59d7] = 0xe26a, [0x59d8] = 0xe26b, [0x59d9] = 0xe26c, [0x59da] = 0xe26d,
  [0x59db] = 0xe26e, [0x59dc] = 0xe26f, [0x59dd] = 0xe270, [0x59de] = 0xe271,
  [0x59df] = 0xe272, [0x59e0] = 0xe273, [0x59e1] = 0xe274, [0x59e2] = 0xe275,
  [0x59e3] = 0xe276, [0x59e4] = 0xe277, [0x59e5] = 0xe278, [0x59e6] = 0xe279,
  [0x59e7] = 0xe27a, [0x59e8] = 0xe27b, [0x59e9] = 0xe27c, [0x59ea] = 0xe27d,
  [0x59eb] = 0xe27e, [0x59ec] = 0xe27f, [0x59ed] = 0xe280, [0x59ee] = 0xe281,
  [0x59ef] = 0xe282, [0x59f0] = 0xe283, [0x59f1] = 0xe284, [0x59f2] = 0xe285,
  [0x59f3] = 0xe286, [0x59f4] = 0xe287, [0x59f5] = 0xe288, [0x59f6] = 0xe289,
  [0x59f7] = 0xe28a, [0x59f8] = 0xe28b, [0x59f9] = 0xe28c, [0x59fa] = 0xe28d,
  [0x59fb] = 0xe28e, [0x59fc] = 0xe28f, [0x59fd] = 0xe290, [0x59fe] = 0xe291,
  [0x5a00] = 0x9d43, [0x5a01] = 0x9d44, [0x5a02] = 0x9d45, [0x5a03] = 0x9d46,
  [0x5a04] = 0x9d47, [0x5a05] = 0x9d48, [0x5a06] = 0x9d49, [0x5a07] = 0x9d4a,
  [0x5a08] = 0x9d4b, [0x5a09] = 0x9d4c, [0x5a0a] = 0x9d4d, [0x5a0b] = 0x9d4e,
  [0x5a0c] = 0x9d4f, [0x5a0d] = 0x9d50, [0x5a0e] = 0x9d51, [0x5a0f] = 0x9d52,
  [0x5a10] = 0x9d53, [0x5a11] = 0x9d54, [0x5a12] = 0x9d55, [0x5a13] = 0x9d56,
  [0x5a14] = 0x9d57, [0x5a15] = 0x9d58, [0x5a16] = 0x9d59, [0x5a17] = 0x9d5a,
  [0x5a18] = 0x9d5b, [0x5a19] = 0x9d5c, [0x5a1a] = 0x9d5d, [0x5a1b] = 0x9d5e,
  [0x5a1c] = 0x9d5f, [0x5a1d] = 0x9d60, [0x5a1e] = 0x9d61, [0x5a1f] = 0x9d62,
  [0x5a20] = 0x9d63, [0x5a21] = 0x9d64, [0x5a22] = 0x9d65, [0x5a23] = 0x9d66,
  [0x5a24] = 0x9d67, [0x5a25] = 0x9d68, [0x5a26] = 0x9d69, [0x5a27] = 0x9d6a,
  [0x5a28] = 0x9d6b, [0x5a29] = 0x9d6c, [0x5a2a] = 0x9d6d, [0x5a2b] = 0x9d6e,
  [0x5a2c] = 0x9d6f, [0x5a2d] = 0x9d70, [0x5a2e] = 0x9d71, [0x5a2f] = 0x9d72,
  [0x5a30] = 0x9d73, [0x5a31] = 0x9d74, [0x5a32] = 0x9d75, [0x5a33] = 0x9d76,
  [0x5a34] = 0x9d77, [0x5a35] = 0x9d78, [0x5a36] = 0x9d79, [0x5a37] = 0x9d7a,
  [0x5a38] = 0x9d7b, [0x5a39] = 0x9d7c, [0x5a3a] = 0x9d7d, [0x5a3b] = 0x9d7e,
  [0x5a3c] = 0x9d7f, [0x5a3d] = 0x9d80, [0x5a3e] = 0x9d81, [0x5a40] = 0x9d82,
  [0x5a41] = 0x9d83, [0x5a42] = 0x9d84, [0x5a43] = 0x9d85, [0x5a44] = 0x9d86,
  [0x5a45] = 0x9d87, [0x5a46] = 0x9d88, [0x5a47] = 0x9d89, [0x5a48] = 0x9d8a,
  [0x5a49] = 0x9d8b, [0x5a4a] = 0x9d8c, [0x5a4b] = 0x9d8d, [0x5a4c] = 0x9d8e,
  [0x5a4d] = 0x9d8f, [0x5a4e] = 0x9d90, [0x5a4f] = 0x9d91, [0x5a50] = 0x9d92,
  [0x5a51] = 0x9d93, [0x5a52] = 0x9d94, [0x5a53] = 0x9d95, [0x5a54] = 0x9d96,
  [0x5a55] = 0x9d97, [0x5a56] = 0x9d98, [0x5a57] = 0x9d99, [0x5a58] = 0x9d9a,
  [0x5a59] = 0x9d9b, [0x5a5a] = 0x9d9c, [0x5a5b] = 0x9d9d, [0x5a5c] = 0x9d9e,
  [0x5a5d] = 0x9d9f, [0x5a5e] = 0x9da0, [0x5a5f] = 0x9da1, [0x5a60] = 0x9da2,
  [0x5a61] = 0xe292, [0x5a62] = 0xe293, [0x5a63] = 0xe294, [0x5a64] = 0xe295,
  [0x5a65] = 0xe296, [0x5a66] = 0xe297, [0x5a67] = 0xe298, [0x5a68] = 0xe299,
  [0x5a69] = 0xe29a, [0x5a6a] = 0xe29b, [0x5a6b] = 0xe29c, [0x5a6c] = 0xe29d,
  [0x5a6d] = 0xe29e, [0x5a6e] = 0xe29f, [0x5a6f] = 0xe2a0, [0x5a70] = 0xe2a1,
  [0x5a71] = 0xe2a2, [0x5a72] = 0xe2a3, [0x5a73] = 0xe2a4, [0x5a74] = 0xe2a5,
  [0x5a75] = 0xe2a6, [0x5a76] = 0xe2a7, [0x5a77] = 0xe2a8, [0x5a78] = 0xe2a9,
  [0x5a79] = 0xe2aa, [0x5a7a] = 0xe2ab, [0x5a7b] = 0xe2ac, [0x5a7c] = 0xe2ad,
  [0x5a7d] = 0xe2ae, [0x5a7e] = 0xe2af, [0x5a7f] = 0xe2b0, [0x5a80] = 0xe2b1,
  [0x5a81] = 0xe2b2, [0x5a82] = 0xe2b3, [0x5a83] = 0xe2b4, [0x5a84] = 0xe2b5,
  [0x5a85] = 0xe2b6, [0x5a86] = 0xe2b7, [0x5a87] = 0xe2b8, [0x5a88] = 0xe2b9,
  [0x5a89] = 0xe2ba, [0x5a8a] = 0xe2bb, [0x5a8b] = 0xe2bc, [0x5a8c] = 0xe2bd,
  [0x5a8d] = 0xe2be, [0x5a8e] = 0xe2bf, [0x5a8f] = 0xe2c0, [0x5a90] = 0xe2c1,
  [0x5a91] = 0xe2c2, [0x5a92] = 0xe2c3, [0x5a93] = 0xe2c4, [0x5a94] = 0xe2c5,
  [0x5a95] = 0xe2c6, [0x5a96] = 0xe2c7, [0x5a97] = 0xe2c8, [0x5a98] = 0xe2c9,
  [0x5a99] = 0xe2ca, [0x5a9a] = 0xe2cb, [0x5a9b] = 0xe2cc, [0x5a9c] = 0xe2cd,
  [0x5a9d] = 0xe2ce, [0x5a9e] = 0xe2cf, [0x5a9f] = 0xe2d0, [0x5aa0] = 0xe2d1,
  [0x5aa1] = 0xe2d2, [0x5aa2] = 0xe2d3, [0x5aa3] = 0xe2d4, [0x5aa4] = 0xe2d5,
  [0x5aa5] = 0xe2d6, [0x5aa6] = 0xe2d7, [0x5aa7] = 0xe2d8, [0x5aa8] = 0xe2d9,
  [0x5aa9] = 0xe2da, [0x5aaa] = 0xe2db, [0x5aab] = 0xe2dc, [0x5aac] = 0xe2dd,
  [0x5aad] = 0xe2de, [0x5aae] = 0xe2df, [0x5aaf] = 0xe2e0, [0x5ab0] = 0xe2e1,
  [0x5ab1] = 0xe2e2, [0x5ab2] = 0xe2e3, [0x5ab3] = 0xe2e4, [0x5ab4] = 0xe2e5,
  [0x5ab5] = 0xe2e6, [0x5ab6] = 0xe2e7, [0x5ab7] = 0xe2e8, [0x5ab8] = 0xe2e9,
  [0x5ab9] = 0xe2ea, [0x5aba] = 0xe2eb, [0x5abb] = 0xe2ec, [0x5abc] = 0xe2ed,
  [0x5abd] = 0xe2ee, [0x5abe] = 0xe2ef, [0x5ac0] = 0x9da3, [0x5ac1] = 0x9da4,
  [0x5ac2] = 0x9da5, [0x5ac3] = 0x9da6, [0x5ac4] = 0x9da7, [0x5ac5] = 0x9da8,
  [0x5ac6] = 0x9da9, [0x5ac7] = 0x9daa, [0x5ac8] = 0x9dab, [0x5ac9] = 0x9dac,
  [0x5aca] = 0x9dad, [0x5acb] = 0x9dae, [0x5acc] = 0x9daf, [0x5acd] = 0x9db0,
  [0x5ace] = 0x9db1, [0x5acf] = 0x9db2, [0x5ad0] = 0x9db3, [0x5ad1] = 0x9db4,
  [0x5ad2] = 0x9db5, [0x5ad3] = 0x9db6, [0x5ad4] = 0x9db7, [0x5ad5] = 0x9db8,
  [0x5ad6] = 0x9db9, [0x5ad7] = 0x9dba, [0x5ad8] = 0x9dbb, [0x5ad9] = 0x9dbc,
  [0x5ada] = 0x9dbd, [0x5adb] = 0x9dbe, [0x5adc] = 0x9dbf, [0x5add] = 0x9dc0,
  [0x5ade] = 0x9dc1, [0x5adf] = 0x9dc2, [0x5ae0] = 0x9dc3, [0x5ae1] = 0x9dc4,
  [0x5ae2] = 0x9dc5, [0x5ae3] = 0x9dc6, [0x5ae4] = 0x9dc7, [0x5ae5] = 0x9dc8,
  [0x5ae6] = 0x9dc9, [0x5ae7] = 0x9dca, [0x5ae8] = 0x9dcb, [0x5ae9] = 0x9dcc,
  [0x5aea] = 0x9dcd, [0x5aeb] = 0x9dce, [0x5aec] = 0x9dcf, [0x5aed] = 0x9dd0,
  [0x5aee] = 0x9dd1, [0x5aef] = 0x9dd2, [0x5af0] = 0x9dd3, [0x5af1] = 0x9dd4,
  [0x5af2] = 0x9dd5, [0x5af3] = 0x9dd6, [0x5af4] = 0x9dd7, [0x5af5] = 0x9dd8,
  [0x5af6] = 0x9dd9, [0x5af7] = 0x9dda, [0x5af8] = 0x9ddb, [0x5af9] = 0x9ddc,
  [0x5afa] = 0x9ddd, [0x5afb] = 0x9dde, [0x5afc] = 0x9ddf, [0x5afd] = 0x9de0,
  [0x5afe] = 0x9de1, [0x5b00] = 0x9de2, [0x5b01] = 0x9de3, [0x5b02] = 0x9de4,
  [0x5b03] = 0x9de5, [0x5b04] = 0x9de6, [0x5b05] = 0x9de7, [0x5b06] = 0x9de8,
  [0x5b07] = 0x9de9, [0x5b08] = 0x9dea, [0x5b09] = 0x9deb, [0x5b0a] = 0x9dec,
  [0x5b0b] = 0x9ded, [0x5b0c] = 0x9dee, [0x5b0d] = 0x9def, [0x5b0e] = 0x9df0,
  [0x5b0f] = 0x9df1, [0x5b10] = 0x9df2, [0x5b11] = 0x9df3, [0x5b12] = 0x9df4,
  [0x5b13] = 0x9df5, [0x5b14] = 0x9df6, [0x5b15] = 0x9df7, [0x5b16] = 0x9df8,
  [0x5b17] = 0x9df9, [0x5b18] = 0x9dfa, [0x5b19] = 0x9dfb, [0x5b1a] = 0x9dfc,
  [0x5b1b] = 0x9dfd, [0x5b1c] = 0x9dfe, [0x5b1d] = 0x9dff, [0x5b1e] = 0x9e00,
  [0x5b1f] = 0x9e01, [0x5b20] = 0x9e02, [0x5b21] = 0xe2f0, [0x5b22] = 0xe2f1,
  [0x5b23] = 0xe2f2, [0x5b24] = 0xe2f3, [0x5b25] = 0xe2f4, [0x5b26] = 0xe2f5,
  [0x5b27] = 0xe2f6, [0x5b28] = 0xe2f7, [0x5b29] = 0xe2f8, [0x5b2a] = 0xe2f9,
  [0x5b2b] = 0xe2fa, [0x5b2c] = 0xe2fb, [0x5b2d] = 0xe2fc, [0x5b2e] = 0xe2fd,
  [0x5b2f] = 0xe2fe, [0x5b30] = 0xe2ff, [0x5b31] = 0xe300, [0x5b32] = 0xe301,
  [0x5b33] = 0xe302, [0x5b34] = 0xe303, [0x5b35] = 0xe304, [0x5b36] = 0xe305,
  [0x5b37] = 0xe306, [0x5b38] = 0xe307, [0x5b39] = 0xe308, [0x5b3a] = 0xe309,
  [0x5b3b] = 0xe30a, [0x5b3c] = 0xe30b, [0x5b3d] = 0xe30c, [0x5b3e] = 0xe30d,
  [0x5b3f] = 0xe30e, [0x5b40] = 0xe30f, [0x5b41] = 0xe310, [0x5b42] = 0xe311,
  [0x5b43] = 0xe312, [0x5b44] = 0xe313, [0x5b45] = 0xe314, [0x5b46] = 0xe315,
  [0x5b47] = 0xe316, [0x5b48] = 0xe317, [0x5b49] = 0xe318, [0x5b4a] = 0xe319,
  [0x5b4b] = 0xe31a, [0x5b4c] = 0xe31b, [0x5b4d] = 0xe31c, [0x5b4e] = 0xe31d,
  [0x5b4f] = 0xe31e, [0x5b50] = 0xe31f, [0x5b51] = 0xe320, [0x5b52] = 0xe321,
  [0x5b53] = 0xe322, [0x5b54] = 0xe323, [0x5b55] = 0xe324, [0x5b56] = 0xe325,
  [0x5b57] = 0xe326, [0x5b58] = 0xe327, [0x5b59] = 0xe328, [0x5b5a] = 0xe329,
  [0x5b5b] = 0xe32a, [0x5b5c] = 0xe32b, [0x5b5d] = 0xe32c, [0x5b5e] = 0xe32d,
  [0x5b5f] = 0xe32e, [0x5b60] = 0xe32f, [0x5b61] = 0xe330, [0x5b62] = 0xe331,
  [0x5b63] = 0xe332, [0x5b64] = 0xe333, [0x5b65] = 0xe334, [0x5b66] = 0xe335,
  [0x5b67] = 0xe336, [0x5b68] = 0xe337, [0x5b69] = 0xe338, [0x5b6a] = 0xe339,
  [0x5b6b] = 0xe33a, [0x5b6c] = 0xe33b, [0x5b6d] = 0xe33c, [0x5b6e] = 0xe33d,
  [0x5b6f] = 0xe33e, [0x5b70] = 0xe33f, [0x5b71] = 0xe340, [0x5b72] = 0xe341,
  [0x5b73] = 0xe342, [0x5b74] = 0xe343, [0x5b75] = 0xe344, [0x5b76] = 0xe345,
  [0x5b77] = 0xe346, [0x5b78] = 0xe347, [0x5b79] = 0xe348, [0x5b7a] = 0xe349,
  [0x5b7b] = 0xe34a, [0x5b7c] = 0xe34b, [0x5b7d] = 0xe34c, [0x5b7e] = 0xe34d,
  [0x5b80] = 0x9e03, [0x5b81] = 0x9e04, [0x5b82] = 0x9e05, [0x5b83] = 0x9e06,
  [0x5b84] = 0x9e07, [0x5b85] = 0x9e08, [0x5b86] = 0x9e09, [0x5b87] = 0x9e0a,
  [0x5b88] = 0x9e0b, [0x5b89] = 0x9e0c, [0x5b8a] = 0x9e0d, [0x5b8b] = 0x9e0e,
  [0x5b8c] = 0x9e0f, [0x5b8d] = 0x9e10, [0x5b8e] = 0x9e11, [0x5b8f] = 0x9e12,
  [0x5b90] = 0x9e13, [0x5b91] = 0x9e14, [0x5b92] = 0x9e15, [0x5b93] = 0x9e16,
  [0x5b94] = 0x9e17, [0x5b95] = 0x9e18, [0x5b96] = 0x9e19, [0x5b97] = 0x9e1a,
  [0x5b98] = 0x9e1b, [0x5b99] = 0x9e1c, [0x5b9a] = 0x9e1d, [0x5b9b] = 0x9e1e,
  [0x5b9c] = 0x9e24, [0x5b9d] = 0x9e27, [0x5b9e] = 0x9e2e, [0x5b9f] = 0x9e30,
  [0x5ba0] = 0x9e34, [0x5ba1] = 0x9e3b, [0x5ba2] = 0x9e3c, [0x5ba3] = 0x9e40,
  [0x5ba4] = 0x9e4d, [0x5ba5] = 0x9e50, [0x5ba6] = 0x9e52, [0x5ba7] = 0x9e53,
  [0x5ba8] = 0x9e54, [0x5ba9] = 0x9e56, [0x5baa] = 0x9e59, [0x5bab] = 0x9e5d,
  [0x5bac] = 0x9e5f, [0x5bad] = 0x9e60, [0x5bae] = 0x9e61, [0x5baf] = 0x9e62,
  [0x5bb0] = 0x9e65, [0x5bb1] = 0x9e6e, [0x5bb2] = 0x9e6f, [0x5bb3] = 0x9e72,
  [0x5bb4] = 0x9e74, [0x5bb5] = 0x9e75, [0x5bb6] = 0x9e76, [0x5bb7] = 0x9e77,
  [0x5bb8] = 0x9e78, [0x5bb9] = 0x9e79, [0x5bba] = 0x9e7a, [0x5bbb] = 0x9e7b,
  [0x5bbc] = 0x9e7c, [0x5bbd] = 0x9e7d, [0x5bbe] = 0x9e80, [0x5bc0] = 0x9e81,
  [0x5bc1] = 0x9e83, [0x5bc2] = 0x9e84, [0x5bc3] = 0x9e85, [0x5bc4] = 0x9e86,
  [0x5bc5] = 0x9e89, [0x5bc6] = 0x9e8a, [0x5bc7] = 0x9e8c, [0x5bc8] = 0x9e8d,
  [0x5bc9] = 0x9e8e, [0x5bca] = 0x9e8f, [0x5bcb] = 0x9e90, [0x5bcc] = 0x9e91,
  [0x5bcd] = 0x9e94, [0x5bce] = 0x9e95, [0x5bcf] = 0x9e96, [0x5bd0] = 0x9e97,
  [0x5bd1] = 0x9e98, [0x5bd2] = 0x9e99, [0x5bd3] = 0x9e9a, [0x5bd4] = 0x9e9b,
  [0x5bd5] = 0x9e9c, [0x5bd6] = 0x9e9e, [0x5bd7] = 0x9ea0, [0x5bd8] = 0x9ea1,
  [0x5bd9] = 0x9ea2, [0x5bda] = 0x9ea3, [0x5bdb] = 0x9ea4, [0x5bdc] = 0x9ea5,
  [0x5bdd] = 0x9ea7, [0x5bde] = 0x9ea8, [0x5bdf] = 0x9ea9, [0x5be0] = 0x9eaa,
  [0x5be1] = 0xe34e, [0x5be2] = 0xe34f, [0x5be3] = 0xe350, [0x5be4] = 0xe351,
  [0x5be5] = 0xe352, [0x5be6] = 0xe353, [0x5be7] = 0xe354, [0x5be8] = 0xe355,
  [0x5be9] = 0xe356, [0x5bea] = 0xe357, [0x5beb] = 0xe358, [0x5bec] = 0xe359,
  [0x5bed] = 0xe35a, [0x5bee] = 0xe35b, [0x5bef] = 0xe35c, [0x5bf0] = 0xe35d,
  [0x5bf1] = 0xe35e, [0x5bf2] = 0xe35f, [0x5bf3] = 0xe360, [0x5bf4] = 0xe361,
  [0x5bf5] = 0xe362, [0x5bf6] = 0xe363, [0x5bf7] = 0xe364, [0x5bf8] = 0xe365,
  [0x5bf9] = 0xe366, [0x5bfa] = 0xe367, [0x5bfb] = 0xe368, [0x5bfc] = 0xe369,
  [0x5bfd] = 0xe36a, [0x5bfe] = 0xe36b, [0x5bff] = 0xe36c, [0x5c00] = 0xe36d,
  [0x5c01] = 0xe36e, [0x5c02] = 0xe36f, [0x5c03] = 0xe370, [0x5c04] = 0xe371,
  [0x5c05] = 0xe372, [0x5c06] = 0xe373, [0x5c07] = 0xe374, [0x5c08] = 0xe375,
  [0x5c09] = 0xe376, [0x5c0a] = 0xe377, [0x5c0b] = 0xe378, [0x5c0c] = 0xe379,
  [0x5c0d] = 0xe37a, [0x5c0e] = 0xe37b, [0x5c0f] = 0xe37c, [0x5c10] = 0xe37d,
  [0x5c11] = 0xe37e, [0x5c12] = 0xe37f, [0x5c13] = 0xe380, [0x5c14] = 0xe381,
  [0x5c15] = 0xe382, [0x5c16] = 0xe383, [0x5c17] = 0xe384, [0x5c18] = 0xe385,
  [0x5c19] = 0xe386, [0x5c1a] = 0xe387, [0x5c1b] = 0xe388, [0x5c1c] = 0xe389,
  [0x5c1d] = 0xe38a, [0x5c1e] = 0xe38b, [0x5c1f] = 0xe38c, [0x5c20] = 0xe38d,
  [0x5c21] = 0xe38e, [0x5c22] = 0xe38f, [0x5c23] = 0xe390, [0x5c24] = 0xe391,
  [0x5c25] = 0xe392, [0x5c26] = 0xe393, [0x5c27] = 0xe394, [0x5c28] = 0xe395,
  [0x5c29] = 0xe396, [0x5c2a] = 0xe397, [0x5c2b] = 0xe398, [0x5c2c] = 0xe399,
  [0x5c2d] = 0xe39a, [0x5c2e] = 0xe39b, [0x5c2f] = 0xe39c, [0x5c30] = 0xe39d,
  [0x5c31] = 0xe39e, [0x5c32] = 0xe39f, [0x5c33] = 0xe3a0, [0x5c34] = 0xe3a1,
  [0x5c35] = 0xe3a2, [0x5c36] = 0xe3a3, [0x5c37] = 0xe3a4, [0x5c38] = 0xe3a5,
  [0x5c39] = 0xe3a6, [0x5c3a] = 0xe3a7, [0x5c3b] = 0xe3a8, [0x5c3c] = 0xe3a9,
  [0x5c3d] = 0xe3aa, [0x5c3e] = 0xe3ab, [0x5c40] = 0x9eab, [0x5c41] = 0x9eac,
  [0x5c42] = 0x9ead, [0x5c43] = 0x9eae, [0x5c44] = 0x9eaf, [0x5c45] = 0x9eb0,
  [0x5c46] = 0x9eb1, [0x5c47] = 0x9eb2, [0x5c48] = 0x9eb3, [0x5c49] = 0x9eb5,
  [0x5c4a] = 0x9eb6, [0x5c4b] = 0x9eb7, [0x5c4c] = 0x9eb9, [0x5c4d] = 0x9eba,
  [0x5c4e] = 0x9ebc, [0x5c4f] = 0x9ebf, [0x5c50] = 0x9ec0, [0x5c51] = 0x9ec1,
  [0x5c52] = 0x9ec2, [0x5c53] = 0x9ec3, [0x5c54] = 0x9ec5, [0x5c55] = 0x9ec6,
  [0x5c56] = 0x9ec7, [0x5c57] = 0x9ec8, [0x5c58] = 0x9eca, [0x5c59] = 0x9ecb,
  [0x5c5a] = 0x9ecc, [0x5c5b] = 0x9ed0, [0x5c5c] = 0x9ed2, [0x5c5d] = 0x9ed3,
  [0x5c5e] = 0x9ed5, [0x5c5f] = 0x9ed6, [0x5c60] = 0x9ed7, [0x5c61] = 0x9ed9,
  [0x5c62] = 0x9eda, [0x5c63] = 0x9ede, [0x5c64] = 0x9ee1, [0x5c65] = 0x9ee3,
  [0x5c66] = 0x9ee4, [0x5c67] = 0x9ee6, [0x5c68] = 0x9ee8, [0x5c69] = 0x9eeb,
  [0x5c6a] = 0x9eec, [0x5c6b] = 0x9eed, [0x5c6c] = 0x9eee, [0x5c6d] = 0x9ef0,
  [0x5c6e] = 0x9ef1, [0x5c6f] = 0x9ef2, [0x5c70] = 0x9ef3, [0x5c71] = 0x9ef4,
  [0x5c72] = 0x9ef5, [0x5c73] = 0x9ef6, [0x5c74] = 0x9ef7, [0x5c75] = 0x9ef8,
  [0x5c76] = 0x9efa, [0x5c77] = 0x9efd, [0x5c78] = 0x9eff, [0x5c79] = 0x9f00,
  [0x5c7a] = 0x9f01, [0x5c7b] = 0x9f02, [0x5c7c] = 0x9f03, [0x5c7d] = 0x9f04,
  [0x5c7e] = 0x9f05, [0x5c80] = 0x9f06, [0x5c81] = 0x9f07, [0x5c82] = 0x9f08,
  [0x5c83] = 0x9f09, [0x5c84] = 0x9f0a, [0x5c85] = 0x9f0c, [0x5c86] = 0x9f0f,
  [0x5c87] = 0x9f11, [0x5c88] = 0x9f12, [0x5c89] = 0x9f14, [0x5c8a] = 0x9f15,
  [0x5c8b] = 0x9f16, [0x5c8c] = 0x9f18, [0x5c8d] = 0x9f1a, [0x5c8e] = 0x9f1b,
  [0x5c8f] = 0x9f1c, [0x5c90] = 0x9f1d, [0x5c91] = 0x9f1e, [0x5c92] = 0x9f1f,
  [0x5c93] = 0x9f21, [0x5c94] = 0x9f23, [0x5c95] = 0x9f24, [0x5c96] = 0x9f25,
  [0x5c97] = 0x9f26, [0x5c98] = 0x9f27, [0x5c99] = 0x9f28, [0x5c9a] = 0x9f29,
  [0x5c9b] = 0x9f2a, [0x5c9c] = 0x9f2b, [0x5c9d] = 0x9f2d, [0x5c9e] = 0x9f2e,
  [0x5c9f] = 0x9f30, [0x5ca0] = 0x9f31, [0x5ca1] = 0xe3ac, [0x5ca2] = 0xe3ad,
  [0x5ca3] = 0xe3ae, [0x5ca4] = 0xe3af, [0x5ca5] = 0xe3b0, [0x5ca6] = 0xe3b1,
  [0x5ca7] = 0xe3b2, [0x5ca8] = 0xe3b3, [0x5ca9] = 0xe3b4, [0x5caa] = 0xe3b5,
  [0x5cab] = 0xe3b6, [0x5cac] = 0xe3b7, [0x5cad] = 0xe3b8, [0x5cae] = 0xe3b9,
  [0x5caf] = 0xe3ba, [0x5cb0] = 0xe3bb, [0x5cb1] = 0xe3bc, [0x5cb2] = 0xe3bd,
  [0x5cb3] = 0xe3be, [0x5cb4] = 0xe3bf, [0x5cb5] = 0xe3c0, [0x5cb6] = 0xe3c1,
  [0x5cb7] = 0xe3c2, [0x5cb8] = 0xe3c3, [0x5cb9] = 0xe3c4, [0x5cba] = 0xe3c5,
  [0x5cbb] = 0xe3c6, [0x5cbc] = 0xe3c7, [0x5cbd] = 0xe3c8, [0x5cbe] = 0xe3c9,
  [0x5cbf] = 0xe3ca, [0x5cc0] = 0xe3cb, [0x5cc1] = 0xe3cc, [0x5cc2] = 0xe3cd,
  [0x5cc3] = 0xe3ce, [0x5cc4] = 0xe3cf, [0x5cc5] = 0xe3d0, [0x5cc6] = 0xe3d1,
  [0x5cc7] = 0xe3d2, [0x5cc8] = 0xe3d3, [0x5cc9] = 0xe3d4, [0x5cca] = 0xe3d5,
  [0x5ccb] = 0xe3d6, [0x5ccc] = 0xe3d7, [0x5ccd] = 0xe3d8, [0x5cce] = 0xe3d9,
  [0x5ccf] = 0xe3da, [0x5cd0] = 0xe3db, [0x5cd1] = 0xe3dc, [0x5cd2] = 0xe3dd,
  [0x5cd3] = 0xe3de, [0x5cd4] = 0xe3df, [0x5cd5] = 0xe3e0, [0x5cd6] = 0xe3e1,
  [0x5cd7] = 0xe3e2, [0x5cd8] = 0xe3e3, [0x5cd9] = 0xe3e4, [0x5cda] = 0xe3e5,
  [0x5cdb] = 0xe3e6, [0x5cdc] = 0xe3e7, [0x5cdd] = 0xe3e8, [0x5cde] = 0xe3e9,
  [0x5cdf] = 0xe3ea, [0x5ce0] = 0xe3eb, [0x5ce1] = 0xe3ec, [0x5ce2] = 0xe3ed,
  [0x5ce3] = 0xe3ee, [0x5ce4] = 0xe3ef, [0x5ce5] = 0xe3f0, [0x5ce6] = 0xe3f1,
  [0x5ce7] = 0xe3f2, [0x5ce8] = 0xe3f3, [0x5ce9] = 0xe3f4, [0x5cea] = 0xe3f5,
  [0x5ceb] = 0xe3f6, [0x5cec] = 0xe3f7, [0x5ced] = 0xe3f8, [0x5cee] = 0xe3f9,
  [0x5cef] = 0xe3fa, [0x5cf0] = 0xe3fb, [0x5cf1] = 0xe3fc, [0x5cf2] = 0xe3fd,
  [0x5cf3] = 0xe3fe, [0x5cf4] = 0xe3ff, [0x5cf5] = 0xe400, [0x5cf6] = 0xe401,
  [0x5cf7] = 0xe402, [0x5cf8] = 0xe403, [0x5cf9] = 0xe404, [0x5cfa] = 0xe405,
  [0x5cfb] = 0xe406, [0x5cfc] = 0xe407, [0x5cfd] = 0xe408, [0x5cfe] = 0xe409,
  [0x5d00] = 0x9f32, [0x5d01] = 0x9f33, [0x5d02] = 0x9f34, [0x5d03] = 0x9f35,
  [0x5d04] = 0x9f36, [0x5d05] = 0x9f38, [0x5d06] = 0x9f3a, [0x5d07] = 0x9f3c,
  [0x5d08] = 0x9f3f, [0x5d09] = 0x9f40, [0x5d0a] = 0x9f41, [0x5d0b] = 0x9f42,
  [0x5d0c] = 0x9f43, [0x5d0d] = 0x9f45, [0x5d0e] = 0x9f46, [0x5d0f] = 0x9f47,
  [0x5d10] = 0x9f48, [0x5d11] = 0x9f49, [0x5d12] = 0x9f4a, [0x5d13] = 0x9f4b,
  [0x5d14] = 0x9f4c, [0x5d15] = 0x9f4d, [0x5d16] = 0x9f4e, [0x5d17] = 0x9f4f,
  [0x5d18] = 0x9f52, [0x5d19] = 0x9f53, [0x5d1a] = 0x9f54, [0x5d1b] = 0x9f55,
  [0x5d1c] = 0x9f56, [0x5d1d] = 0x9f57, [0x5d1e] = 0x9f58, [0x5d1f] = 0x9f59,
  [0x5d20] = 0x9f5a, [0x5d21] = 0x9f5b, [0x5d22] = 0x9f5c, [0x5d23] = 0x9f5d,
  [0x5d24] = 0x9f5e, [0x5d25] = 0x9f5f, [0x5d26] = 0x9f60, [0x5d27] = 0x9f61,
  [0x5d28] = 0x9f62, [0x5d29] = 0x9f63, [0x5d2a] = 0x9f64, [0x5d2b] = 0x9f65,
  [0x5d2c] = 0x9f66, [0x5d2d] = 0x9f67, [0x5d2e] = 0x9f68, [0x5d2f] = 0x9f69,
  [0x5d30] = 0x9f6a, [0x5d31] = 0x9f6b, [0x5d32] = 0x9f6c, [0x5d33] = 0x9f6d,
  [0x5d34] = 0x9f6e, [0x5d35] = 0x9f6f, [0x5d36] = 0x9f70, [0x5d37] = 0x9f71,
  [0x5d38] = 0x9f72, [0x5d39] = 0x9f73, [0x5d3a] = 0x9f74, [0x5d3b] = 0x9f75,
  [0x5d3c] = 0x9f76, [0x5d3d] = 0x9f77, [0x5d3e] = 0x9f78, [0x5d40] = 0x9f79,
  [0x5d41] = 0x9f7a, [0x5d42] = 0x9f7b, [0x5d43] = 0x9f7c, [0x5d44] = 0x9f7d,
  [0x5d45] = 0x9f7e, [0x5d46] = 0x9f81, [0x5d47] = 0x9f82, [0x5d48] = 0x9f8d,
  [0x5d49] = 0x9f8e, [0x5d4a] = 0x9f8f, [0x5d4b] = 0x9f90, [0x5d4c] = 0x9f91,
  [0x5d4d] = 0x9f92, [0x5d4e] = 0x9f93, [0x5d4f] = 0x9f94, [0x5d50] = 0x9f95,
  [0x5d51] = 0x9f96, [0x5d52] = 0x9f97, [0x5d53] = 0x9f98, [0x5d54] = 0x9f9c,
  [0x5d55] = 0x9f9d, [0x5d56] = 0x9f9e, [0x5d57] = 0x9fa1, [0x5d58] = 0x9fa2,
  [0x5d59] = 0x9fa3, [0x5d5a] = 0x9fa4, [0x5d5b] = 0x9fa5, [0x5d5c] = 0xf92c,
  [0x5d5d] = 0xf979, [0x5d5e] = 0xf995, [0x5d5f] = 0xf9e7, [0x5d60] = 0xf9f1,
  [0x5d61] = 0xe40a, [0x5d62] = 0xe40b, [0x5d63] = 0xe40c, [0x5d64] = 0xe40d,
  [0x5d65] = 0xe40e, [0x5d66] = 0xe40f, [0x5d67] = 0xe410, [0x5d68] = 0xe411,
  [0x5d69] = 0xe412, [0x5d6a] = 0xe413, [0x5d6b] = 0xe414, [0x5d6c] = 0xe415,
  [0x5d6d] = 0xe416, [0x5d6e] = 0xe417, [0x5d6f] = 0xe418, [0x5d70] = 0xe419,
  [0x5d71] = 0xe41a, [0x5d72] = 0xe41b, [0x5d73] = 0xe41c, [0x5d74] = 0xe41d,
  [0x5d75] = 0xe41e, [0x5d76] = 0xe41f, [0x5d77] = 0xe420, [0x5d78] = 0xe421,
  [0x5d79] = 0xe422, [0x5d7a] = 0xe423, [0x5d7b] = 0xe424, [0x5d7c] = 0xe425,
  [0x5d7d] = 0xe426, [0x5d7e] = 0xe427, [0x5d7f] = 0xe428, [0x5d80] = 0xe429,
  [0x5d81] = 0xe42a, [0x5d82] = 0xe42b, [0x5d83] = 0xe42c, [0x5d84] = 0xe42d,
  [0x5d85] = 0xe42e, [0x5d86] = 0xe42f, [0x5d87] = 0xe430, [0x5d88] = 0xe431,
  [0x5d89] = 0xe432, [0x5d8a] = 0xe433, [0x5d8b] = 0xe434, [0x5d8c] = 0xe435,
  [0x5d8d] = 0xe436, [0x5d8e] = 0xe437, [0x5d8f] = 0xe438, [0x5d90] = 0xe439,
  [0x5d91] = 0xe43a, [0x5d92] = 0xe43b, [0x5d93] = 0xe43c, [0x5d94] = 0xe43d,
  [0x5d95] = 0xe43e, [0x5d96] = 0xe43f, [0x5d97] = 0xe440, [0x5d98] = 0xe441,
  [0x5d99] = 0xe442, [0x5d9a] = 0xe443, [0x5d9b] = 0xe444, [0x5d9c] = 0xe445,
  [0x5d9d] = 0xe446, [0x5d9e] = 0xe447, [0x5d9f] = 0xe448, [0x5da0] = 0xe449,
  [0x5da1] = 0xe44a, [0x5da2] = 0xe44b, [0x5da3] = 0xe44c, [0x5da4] = 0xe44d,
  [0x5da5] = 0xe44e, [0x5da6] = 0xe44f, [0x5da7] = 0xe450, [0x5da8] = 0xe451,
  [0x5da9] = 0xe452, [0x5daa] = 0xe453, [0x5dab] = 0xe454, [0x5dac] = 0xe455,
  [0x5dad] = 0xe456, [0x5dae] = 0xe457, [0x5daf] = 0xe458, [0x5db0] = 0xe459,
  [0x5db1] = 0xe45a, [0x5db2] = 0xe45b, [0x5db3] = 0xe45c, [0x5db4] = 0xe45d,
  [0x5db5] = 0xe45e, [0x5db6] = 0xe45f, [0x5db7] = 0xe460, [0x5db8] = 0xe461,
  [0x5db9] = 0xe462, [0x5dba] = 0xe463, [0x5dbb] = 0xe464, [0x5dbc] = 0xe465,
  [0x5dbd] = 0xe466, [0x5dbe] = 0xe467, [0x5dc0] = 0xfa0c, [0x5dc1] = 0xfa0d,
  [0x5dc2] = 0xfa0e, [0x5dc3] = 0xfa0f, [0x5dc4] = 0xfa11, [0x5dc5] = 0xfa13,
  [0x5dc6] = 0xfa14, [0x5dc7] = 0xfa18, [0x5dc8] = 0xfa1f, [0x5dc9] = 0xfa20,
  [0x5dca] = 0xfa21, [0x5dcb] = 0xfa23, [0x5dcc] = 0xfa24, [0x5dcd] = 0xfa27,
  [0x5dce] = 0xfa28, [0x5dcf] = 0xfa29, [0x5dd0] = 0x2e81, [0x5dd1] = 0xe816,
  [0x5dd2] = 0xe817, [0x5dd3] = 0xe818, [0x5dd4] = 0x2e84, [0x5dd5] = 0x3473,
  [0x5dd6] = 0x3447, [0x5dd7] = 0x2e88, [0x5dd8] = 0x2e8b, [0x5dd9] = 0xe81e,
  [0x5dda] = 0x359e, [0x5ddb] = 0x361a, [0x5ddc] = 0x360e, [0x5ddd] = 0x2e8c,
  [0x5dde] = 0x2e97, [0x5ddf] = 0x396e, [0x5de0] = 0x3918, [0x5de1] = 0xe826,
  [0x5de2] = 0x39cf, [0x5de3] = 0x39df, [0x5de4] = 0x3a73, [0x5de5] = 0x39d0,
  [0x5de6] = 0xe82b, [0x5de7] = 0xe82c, [0x5de8] = 0x3b4e, [0x5de9] = 0x3c6e,
  [0x5dea] = 0x3ce0, [0x5deb] = 0x2ea7, [0x5dec] = 0xe831, [0x5ded] = 0xe832,
  [0x5dee] = 0x2eaa, [0x5def] = 0x4056, [0x5df0] = 0x415f, [0x5df1] = 0x2eae,
  [0x5df2] = 0x4337, [0x5df3] = 0x2eb3, [0x5df4] = 0x2eb6, [0x5df5] = 0x2eb7,
  [0x5df6] = 0xe83b, [0x5df7] = 0x43b1, [0x5df8] = 0x43ac, [0x5df9] = 0x2ebb,
  [0x5dfa] = 0x43dd, [0x5dfb] = 0x44d6, [0x5dfc] = 0x4661, [0x5dfd] = 0x464c,
  [0x5dfe] = 0xe843, [0x5e00] = 0x4723, [0x5e01] = 0x4729, [0x5e02] = 0x477c,
  [0x5e03] = 0x478d, [0x5e04] = 0x2eca, [0x5e05] = 0x4947, [0x5e06] = 0x497a,
  [0x5e07] = 0x497d, [0x5e08] = 0x4982, [0x5e09] = 0x4983, [0x5e0a] = 0x4985,
  [0x5e0b] = 0x4986, [0x5e0c] = 0x499f, [0x5e0d] = 0x499b, [0x5e0e] = 0x49b7,
  [0x5e0f] = 0x49b6, [0x5e10] = 0xe854, [0x5e11] = 0xe855, [0x5e12] = 0x4ca3,
  [0x5e13] = 0x4c9f, [0x5e14] = 0x4ca0, [0x5e15] = 0x4ca1, [0x5e16] = 0x4c77,
  [0x5e17] = 0x4ca2, [0x5e18] = 0x4d13, [0x5e19] = 0x4d14, [0x5e1a] = 0x4d15,
  [0x5e1b] = 0x4d16, [0x5e1c] = 0x4d17, [0x5e1d] = 0x4d18, [0x5e1e] = 0x4d19,
  [0x5e1f] = 0x4dae, [0x5e20] = 0xe864, [0x5e21] = 0xe468, [0x5e22] = 0xe469,
  [0x5e23] = 0xe46a, [0x5e24] = 0xe46b, [0x5e25] = 0xe46c, [0x5e26] = 0xe46d,
  [0x5e27] = 0xe46e, [0x5e28] = 0xe46f, [0x5e29] = 0xe470, [0x5e2a] = 0xe471,
  [0x5e2b] = 0xe472, [0x5e2c] = 0xe473, [0x5e2d] = 0xe474, [0x5e2e] = 0xe475,
  [0x5e2f] = 0xe476, [0x5e30] = 0xe477, [0x5e31] = 0xe478, [0x5e32] = 0xe479,
  [0x5e33] = 0xe47a, [0x5e34] = 0xe47b, [0x5e35] = 0xe47c, [0x5e36] = 0xe47d,
  [0x5e37] = 0xe47e, [0x5e38] = 0xe47f, [0x5e39] = 0xe480, [0x5e3a] = 0xe481,
  [0x5e3b] = 0xe482, [0x5e3c] = 0xe483, [0x5e3d] = 0xe484, [0x5e3e] = 0xe485,
  [0x5e3f] = 0xe486, [0x5e40] = 0xe487, [0x5e41] = 0xe488, [0x5e42] = 0xe489,
  [0x5e43] = 0xe48a, [0x5e44] = 0xe48b, [0x5e45] = 0xe48c, [0x5e46] = 0xe48d,
  [0x5e47] = 0xe48e, [0x5e48] = 0xe48f, [0x5e49] = 0xe490, [0x5e4a] = 0xe491,
  [0x5e4b] = 0xe492, [0x5e4c] = 0xe493, [0x5e4d] = 0xe494, [0x5e4e] = 0xe495,
  [0x5e4f] = 0xe496, [0x5e50] = 0xe497, [0x5e51] = 0xe498, [0x5e52] = 0xe499,
  [0x5e53] = 0xe49a, [0x5e54] = 0xe49b, [0x5e55] = 0xe49c, [0x5e56] = 0xe49d,
  [0x5e57] = 0xe49e, [0x5e58] = 0xe49f, [0x5e59] = 0xe4a0, [0x5e5a] = 0xe4a1,
  [0x5e5b] = 0xe4a2, [0x5e5c] = 0xe4a3, [0x5e5d] = 0xe4a4, [0x5e5e] = 0xe4a5,
  [0x5e5f] = 0xe4a6, [0x5e60] = 0xe4a7, [0x5e61] = 0xe4a8, [0x5e62] = 0xe4a9,
  [0x5e63] = 0xe4aa, [0x5e64] = 0xe4ab, [0x5e65] = 0xe4ac, [0x5e66] = 0xe4ad,
  [0x5e67] = 0xe4ae, [0x5e68] = 0xe4af, [0x5e69] = 0xe4b0, [0x5e6a] = 0xe4b1,
  [0x5e6b] = 0xe4b2, [0x5e6c] = 0xe4b3, [0x5e6d] = 0xe4b4, [0x5e6e] = 0xe4b5,
  [0x5e6f] = 0xe4b6, [0x5e70] = 0xe4b7, [0x5e71] = 0xe4b8, [0x5e72] = 0xe4b9,
  [0x5e73] = 0xe4ba, [0x5e74] = 0xe4bb, [0x5e75] = 0xe4bc, [0x5e76] = 0xe4bd,
  [0x5e77] = 0xe4be, [0x5e78] = 0xe4bf, [0x5e79] = 0xe4c0, [0x5e7a] = 0xe4c1,
  [0x5e7b] = 0xe4c2, [0x5e7c] = 0xe4c3, [0x5e7d] = 0xe4c4, [0x5e7e] = 0xe4c5
};

/* Table for GB18030 -> UCS-4, containing the four-byte characters only,
   excluding four linear ranges.  Generated through

   clisp tab.lisp < GB18030.table

   where tab.lisp is:
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   (locally (declare (compile))
     (let ((count 0))
       (loop
	 (let ((line (read-line nil nil nil)))
	   (unless line (return))
	   (when (eql (position #\Tab line) 10)
	     (let* ((gbk (parse-integer line :start 2 :end 10 :radix 16))
		    (ucs (parse-integer line :start 13 :radix 16))
		    (ch1 (ldb (byte 8 24) gbk))
		    (ch2 (ldb (byte 8 16) gbk))
		    (ch3 (ldb (byte 8 8) gbk))
		    (ch4 (ldb (byte 8 0) gbk))
		    (linear (+ (* (+ (* (+ (* (- ch1 #x81) 10)
					   (- ch2 #x30))
					126)
				     (- ch3 #x81))
				  10)
			       (- ch4 #x30)))
		    ; There are five ranges. The gap lengths are:
		    ;   (- #x1d21 #x334) = 6637,
		    ;   (- #x2c41 #x2403) = 2110,
		    ;   (- #x82bd #x4a63) = 14426,
		    ;   (- #x93d5 #x830e) = 4295.
		    (idx (cond ((<= 0 linear #x333) linear)
			       ((<= #x1d21 linear #x2402) (- linear 6637))
			       ((<= #x2c41 linear #x4a62) (- linear 6637 2110))
			       ((<= #x82bd linear #x830d) (- linear 6637 2110 14426))
			       ((<= #x93d5 linear #x99de) (- linear 6637 2110 14426 4295))
			       (t nil))))
	       (when idx
		 (when (= (mod count 4) 0)
		   (format t " "))
		 (format t " ~([0x~4,'0X] = 0x~4,'0X~)," idx ucs)
		 (incf count)
		 (when (= (mod count 4) 0)
		   (format t "~%")))))))
       (format t "~%")))
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

static const uint16_t __fourbyte_to_ucs[39419-11172-4295] =
{
  [0x0000] = 0x0080, [0x0001] = 0x0081, [0x0002] = 0x0082, [0x0003] = 0x0083,
  [0x0004] = 0x0084, [0x0005] = 0x0085, [0x0006] = 0x0086, [0x0007] = 0x0087,
  [0x0008] = 0x0088, [0x0009] = 0x0089, [0x000a] = 0x008a, [0x000b] = 0x008b,
  [0x000c] = 0x008c, [0x000d] = 0x008d, [0x000e] = 0x008e, [0x000f] = 0x008f,
  [0x0010] = 0x0090, [0x0011] = 0x0091, [0x0012] = 0x0092, [0x0013] = 0x0093,
  [0x0014] = 0x0094, [0x0015] = 0x0095, [0x0016] = 0x0096, [0x0017] = 0x0097,
  [0x0018] = 0x0098, [0x0019] = 0x0099, [0x001a] = 0x009a, [0x001b] = 0x009b,
  [0x001c] = 0x009c, [0x001d] = 0x009d, [0x001e] = 0x009e, [0x001f] = 0x009f,
  [0x0020] = 0x00a0, [0x0021] = 0x00a1, [0x0022] = 0x00a2, [0x0023] = 0x00a3,
  [0x0024] = 0x00a5, [0x0025] = 0x00a6, [0x0026] = 0x00a9, [0x0027] = 0x00aa,
  [0x0028] = 0x00ab, [0x0029] = 0x00ac, [0x002a] = 0x00ad, [0x002b] = 0x00ae,
  [0x002c] = 0x00af, [0x002d] = 0x00b2, [0x002e] = 0x00b3, [0x002f] = 0x00b4,
  [0x0030] = 0x00b5, [0x0031] = 0x00b6, [0x0032] = 0x00b8, [0x0033] = 0x00b9,
  [0x0034] = 0x00ba, [0x0035] = 0x00bb, [0x0036] = 0x00bc, [0x0037] = 0x00bd,
  [0x0038] = 0x00be, [0x0039] = 0x00bf, [0x003a] = 0x00c0, [0x003b] = 0x00c1,
  [0x003c] = 0x00c2, [0x003d] = 0x00c3, [0x003e] = 0x00c4, [0x003f] = 0x00c5,
  [0x0040] = 0x00c6, [0x0041] = 0x00c7, [0x0042] = 0x00c8, [0x0043] = 0x00c9,
  [0x0044] = 0x00ca, [0x0045] = 0x00cb, [0x0046] = 0x00cc, [0x0047] = 0x00cd,
  [0x0048] = 0x00ce, [0x0049] = 0x00cf, [0x004a] = 0x00d0, [0x004b] = 0x00d1,
  [0x004c] = 0x00d2, [0x004d] = 0x00d3, [0x004e] = 0x00d4, [0x004f] = 0x00d5,
  [0x0050] = 0x00d6, [0x0051] = 0x00d8, [0x0052] = 0x00d9, [0x0053] = 0x00da,
  [0x0054] = 0x00db, [0x0055] = 0x00dc, [0x0056] = 0x00dd, [0x0057] = 0x00de,
  [0x0058] = 0x00df, [0x0059] = 0x00e2, [0x005a] = 0x00e3, [0x005b] = 0x00e4,
  [0x005c] = 0x00e5, [0x005d] = 0x00e6, [0x005e] = 0x00e7, [0x005f] = 0x00eb,
  [0x0060] = 0x00ee, [0x0061] = 0x00ef, [0x0062] = 0x00f0, [0x0063] = 0x00f1,
  [0x0064] = 0x00f4, [0x0065] = 0x00f5, [0x0066] = 0x00f6, [0x0067] = 0x00f8,
  [0x0068] = 0x00fb, [0x0069] = 0x00fd, [0x006a] = 0x00fe, [0x006b] = 0x00ff,
  [0x006c] = 0x0100, [0x006d] = 0x0102, [0x006e] = 0x0103, [0x006f] = 0x0104,
  [0x0070] = 0x0105, [0x0071] = 0x0106, [0x0072] = 0x0107, [0x0073] = 0x0108,
  [0x0074] = 0x0109, [0x0075] = 0x010a, [0x0076] = 0x010b, [0x0077] = 0x010c,
  [0x0078] = 0x010d, [0x0079] = 0x010e, [0x007a] = 0x010f, [0x007b] = 0x0110,
  [0x007c] = 0x0111, [0x007d] = 0x0112, [0x007e] = 0x0114, [0x007f] = 0x0115,
  [0x0080] = 0x0116, [0x0081] = 0x0117, [0x0082] = 0x0118, [0x0083] = 0x0119,
  [0x0084] = 0x011a, [0x0085] = 0x011c, [0x0086] = 0x011d, [0x0087] = 0x011e,
  [0x0088] = 0x011f, [0x0089] = 0x0120, [0x008a] = 0x0121, [0x008b] = 0x0122,
  [0x008c] = 0x0123, [0x008d] = 0x0124, [0x008e] = 0x0125, [0x008f] = 0x0126,
  [0x0090] = 0x0127, [0x0091] = 0x0128, [0x0092] = 0x0129, [0x0093] = 0x012a,
  [0x0094] = 0x012c, [0x0095] = 0x012d, [0x0096] = 0x012e, [0x0097] = 0x012f,
  [0x0098] = 0x0130, [0x0099] = 0x0131, [0x009a] = 0x0132, [0x009b] = 0x0133,
  [0x009c] = 0x0134, [0x009d] = 0x0135, [0x009e] = 0x0136, [0x009f] = 0x0137,
  [0x00a0] = 0x0138, [0x00a1] = 0x0139, [0x00a2] = 0x013a, [0x00a3] = 0x013b,
  [0x00a4] = 0x013c, [0x00a5] = 0x013d, [0x00a6] = 0x013e, [0x00a7] = 0x013f,
  [0x00a8] = 0x0140, [0x00a9] = 0x0141, [0x00aa] = 0x0142, [0x00ab] = 0x0143,
  [0x00ac] = 0x0145, [0x00ad] = 0x0146, [0x00ae] = 0x0147, [0x00af] = 0x0149,
  [0x00b0] = 0x014a, [0x00b1] = 0x014b, [0x00b2] = 0x014c, [0x00b3] = 0x014e,
  [0x00b4] = 0x014f, [0x00b5] = 0x0150, [0x00b6] = 0x0151, [0x00b7] = 0x0152,
  [0x00b8] = 0x0153, [0x00b9] = 0x0154, [0x00ba] = 0x0155, [0x00bb] = 0x0156,
  [0x00bc] = 0x0157, [0x00bd] = 0x0158, [0x00be] = 0x0159, [0x00bf] = 0x015a,
  [0x00c0] = 0x015b, [0x00c1] = 0x015c, [0x00c2] = 0x015d, [0x00c3] = 0x015e,
  [0x00c4] = 0x015f, [0x00c5] = 0x0160, [0x00c6] = 0x0161, [0x00c7] = 0x0162,
  [0x00c8] = 0x0163, [0x00c9] = 0x0164, [0x00ca] = 0x0165, [0x00cb] = 0x0166,
  [0x00cc] = 0x0167, [0x00cd] = 0x0168, [0x00ce] = 0x0169, [0x00cf] = 0x016a,
  [0x00d0] = 0x016c, [0x00d1] = 0x016d, [0x00d2] = 0x016e, [0x00d3] = 0x016f,
  [0x00d4] = 0x0170, [0x00d5] = 0x0171, [0x00d6] = 0x0172, [0x00d7] = 0x0173,
  [0x00d8] = 0x0174, [0x00d9] = 0x0175, [0x00da] = 0x0176, [0x00db] = 0x0177,
  [0x00dc] = 0x0178, [0x00dd] = 0x0179, [0x00de] = 0x017a, [0x00df] = 0x017b,
  [0x00e0] = 0x017c, [0x00e1] = 0x017d, [0x00e2] = 0x017e, [0x00e3] = 0x017f,
  [0x00e4] = 0x0180, [0x00e5] = 0x0181, [0x00e6] = 0x0182, [0x00e7] = 0x0183,
  [0x00e8] = 0x0184, [0x00e9] = 0x0185, [0x00ea] = 0x0186, [0x00eb] = 0x0187,
  [0x00ec] = 0x0188, [0x00ed] = 0x0189, [0x00ee] = 0x018a, [0x00ef] = 0x018b,
  [0x00f0] = 0x018c, [0x00f1] = 0x018d, [0x00f2] = 0x018e, [0x00f3] = 0x018f,
  [0x00f4] = 0x0190, [0x00f5] = 0x0191, [0x00f6] = 0x0192, [0x00f7] = 0x0193,
  [0x00f8] = 0x0194, [0x00f9] = 0x0195, [0x00fa] = 0x0196, [0x00fb] = 0x0197,
  [0x00fc] = 0x0198, [0x00fd] = 0x0199, [0x00fe] = 0x019a, [0x00ff] = 0x019b,
  [0x0100] = 0x019c, [0x0101] = 0x019d, [0x0102] = 0x019e, [0x0103] = 0x019f,
  [0x0104] = 0x01a0, [0x0105] = 0x01a1, [0x0106] = 0x01a2, [0x0107] = 0x01a3,
  [0x0108] = 0x01a4, [0x0109] = 0x01a5, [0x010a] = 0x01a6, [0x010b] = 0x01a7,
  [0x010c] = 0x01a8, [0x010d] = 0x01a9, [0x010e] = 0x01aa, [0x010f] = 0x01ab,
  [0x0110] = 0x01ac, [0x0111] = 0x01ad, [0x0112] = 0x01ae, [0x0113] = 0x01af,
  [0x0114] = 0x01b0, [0x0115] = 0x01b1, [0x0116] = 0x01b2, [0x0117] = 0x01b3,
  [0x0118] = 0x01b4, [0x0119] = 0x01b5, [0x011a] = 0x01b6, [0x011b] = 0x01b7,
  [0x011c] = 0x01b8, [0x011d] = 0x01b9, [0x011e] = 0x01ba, [0x011f] = 0x01bb,
  [0x0120] = 0x01bc, [0x0121] = 0x01bd, [0x0122] = 0x01be, [0x0123] = 0x01bf,
  [0x0124] = 0x01c0, [0x0125] = 0x01c1, [0x0126] = 0x01c2, [0x0127] = 0x01c3,
  [0x0128] = 0x01c4, [0x0129] = 0x01c5, [0x012a] = 0x01c6, [0x012b] = 0x01c7,
  [0x012c] = 0x01c8, [0x012d] = 0x01c9, [0x012e] = 0x01ca, [0x012f] = 0x01cb,
  [0x0130] = 0x01cc, [0x0131] = 0x01cd, [0x0132] = 0x01cf, [0x0133] = 0x01d1,
  [0x0134] = 0x01d3, [0x0135] = 0x01d5, [0x0136] = 0x01d7, [0x0137] = 0x01d9,
  [0x0138] = 0x01db, [0x0139] = 0x01dd, [0x013a] = 0x01de, [0x013b] = 0x01df,
  [0x013c] = 0x01e0, [0x013d] = 0x01e1, [0x013e] = 0x01e2, [0x013f] = 0x01e3,
  [0x0140] = 0x01e4, [0x0141] = 0x01e5, [0x0142] = 0x01e6, [0x0143] = 0x01e7,
  [0x0144] = 0x01e8, [0x0145] = 0x01e9, [0x0146] = 0x01ea, [0x0147] = 0x01eb,
  [0x0148] = 0x01ec, [0x0149] = 0x01ed, [0x014a] = 0x01ee, [0x014b] = 0x01ef,
  [0x014c] = 0x01f0, [0x014d] = 0x01f1, [0x014e] = 0x01f2, [0x014f] = 0x01f3,
  [0x0150] = 0x01f4, [0x0151] = 0x01f5, [0x0152] = 0x01f6, [0x0153] = 0x01f7,
  [0x0154] = 0x01f8, [0x0155] = 0x01fa, [0x0156] = 0x01fb, [0x0157] = 0x01fc,
  [0x0158] = 0x01fd, [0x0159] = 0x01fe, [0x015a] = 0x01ff, [0x015b] = 0x0200,
  [0x015c] = 0x0201, [0x015d] = 0x0202, [0x015e] = 0x0203, [0x015f] = 0x0204,
  [0x0160] = 0x0205, [0x0161] = 0x0206, [0x0162] = 0x0207, [0x0163] = 0x0208,
  [0x0164] = 0x0209, [0x0165] = 0x020a, [0x0166] = 0x020b, [0x0167] = 0x020c,
  [0x0168] = 0x020d, [0x0169] = 0x020e, [0x016a] = 0x020f, [0x016b] = 0x0210,
  [0x016c] = 0x0211, [0x016d] = 0x0212, [0x016e] = 0x0213, [0x016f] = 0x0214,
  [0x0170] = 0x0215, [0x0171] = 0x0216, [0x0172] = 0x0217, [0x0173] = 0x0218,
  [0x0174] = 0x0219, [0x0175] = 0x021a, [0x0176] = 0x021b, [0x0177] = 0x021c,
  [0x0178] = 0x021d, [0x0179] = 0x021e, [0x017a] = 0x021f, [0x017b] = 0x0220,
  [0x017c] = 0x0221, [0x017d] = 0x0222, [0x017e] = 0x0223, [0x017f] = 0x0224,
  [0x0180] = 0x0225, [0x0181] = 0x0226, [0x0182] = 0x0227, [0x0183] = 0x0228,
  [0x0184] = 0x0229, [0x0185] = 0x022a, [0x0186] = 0x022b, [0x0187] = 0x022c,
  [0x0188] = 0x022d, [0x0189] = 0x022e, [0x018a] = 0x022f, [0x018b] = 0x0230,
  [0x018c] = 0x0231, [0x018d] = 0x0232, [0x018e] = 0x0233, [0x018f] = 0x0234,
  [0x0190] = 0x0235, [0x0191] = 0x0236, [0x0192] = 0x0237, [0x0193] = 0x0238,
  [0x0194] = 0x0239, [0x0195] = 0x023a, [0x0196] = 0x023b, [0x0197] = 0x023c,
  [0x0198] = 0x023d, [0x0199] = 0x023e, [0x019a] = 0x023f, [0x019b] = 0x0240,
  [0x019c] = 0x0241, [0x019d] = 0x0242, [0x019e] = 0x0243, [0x019f] = 0x0244,
  [0x01a0] = 0x0245, [0x01a1] = 0x0246, [0x01a2] = 0x0247, [0x01a3] = 0x0248,
  [0x01a4] = 0x0249, [0x01a5] = 0x024a, [0x01a6] = 0x024b, [0x01a7] = 0x024c,
  [0x01a8] = 0x024d, [0x01a9] = 0x024e, [0x01aa] = 0x024f, [0x01ab] = 0x0250,
  [0x01ac] = 0x0252, [0x01ad] = 0x0253, [0x01ae] = 0x0254, [0x01af] = 0x0255,
  [0x01b0] = 0x0256, [0x01b1] = 0x0257, [0x01b2] = 0x0258, [0x01b3] = 0x0259,
  [0x01b4] = 0x025a, [0x01b5] = 0x025b, [0x01b6] = 0x025c, [0x01b7] = 0x025d,
  [0x01b8] = 0x025e, [0x01b9] = 0x025f, [0x01ba] = 0x0260, [0x01bb] = 0x0262,
  [0x01bc] = 0x0263, [0x01bd] = 0x0264, [0x01be] = 0x0265, [0x01bf] = 0x0266,
  [0x01c0] = 0x0267, [0x01c1] = 0x0268, [0x01c2] = 0x0269, [0x01c3] = 0x026a,
  [0x01c4] = 0x026b, [0x01c5] = 0x026c, [0x01c6] = 0x026d, [0x01c7] = 0x026e,
  [0x01c8] = 0x026f, [0x01c9] = 0x0270, [0x01ca] = 0x0271, [0x01cb] = 0x0272,
  [0x01cc] = 0x0273, [0x01cd] = 0x0274, [0x01ce] = 0x0275, [0x01cf] = 0x0276,
  [0x01d0] = 0x0277, [0x01d1] = 0x0278, [0x01d2] = 0x0279, [0x01d3] = 0x027a,
  [0x01d4] = 0x027b, [0x01d5] = 0x027c, [0x01d6] = 0x027d, [0x01d7] = 0x027e,
  [0x01d8] = 0x027f, [0x01d9] = 0x0280, [0x01da] = 0x0281, [0x01db] = 0x0282,
  [0x01dc] = 0x0283, [0x01dd] = 0x0284, [0x01de] = 0x0285, [0x01df] = 0x0286,
  [0x01e0] = 0x0287, [0x01e1] = 0x0288, [0x01e2] = 0x0289, [0x01e3] = 0x028a,
  [0x01e4] = 0x028b, [0x01e5] = 0x028c, [0x01e6] = 0x028d, [0x01e7] = 0x028e,
  [0x01e8] = 0x028f, [0x01e9] = 0x0290, [0x01ea] = 0x0291, [0x01eb] = 0x0292,
  [0x01ec] = 0x0293, [0x01ed] = 0x0294, [0x01ee] = 0x0295, [0x01ef] = 0x0296,
  [0x01f0] = 0x0297, [0x01f1] = 0x0298, [0x01f2] = 0x0299, [0x01f3] = 0x029a,
  [0x01f4] = 0x029b, [0x01f5] = 0x029c, [0x01f6] = 0x029d, [0x01f7] = 0x029e,
  [0x01f8] = 0x029f, [0x01f9] = 0x02a0, [0x01fa] = 0x02a1, [0x01fb] = 0x02a2,
  [0x01fc] = 0x02a3, [0x01fd] = 0x02a4, [0x01fe] = 0x02a5, [0x01ff] = 0x02a6,
  [0x0200] = 0x02a7, [0x0201] = 0x02a8, [0x0202] = 0x02a9, [0x0203] = 0x02aa,
  [0x0204] = 0x02ab, [0x0205] = 0x02ac, [0x0206] = 0x02ad, [0x0207] = 0x02ae,
  [0x0208] = 0x02af, [0x0209] = 0x02b0, [0x020a] = 0x02b1, [0x020b] = 0x02b2,
  [0x020c] = 0x02b3, [0x020d] = 0x02b4, [0x020e] = 0x02b5, [0x020f] = 0x02b6,
  [0x0210] = 0x02b7, [0x0211] = 0x02b8, [0x0212] = 0x02b9, [0x0213] = 0x02ba,
  [0x0214] = 0x02bb, [0x0215] = 0x02bc, [0x0216] = 0x02bd, [0x0217] = 0x02be,
  [0x0218] = 0x02bf, [0x0219] = 0x02c0, [0x021a] = 0x02c1, [0x021b] = 0x02c2,
  [0x021c] = 0x02c3, [0x021d] = 0x02c4, [0x021e] = 0x02c5, [0x021f] = 0x02c6,
  [0x0220] = 0x02c8, [0x0221] = 0x02cc, [0x0222] = 0x02cd, [0x0223] = 0x02ce,
  [0x0224] = 0x02cf, [0x0225] = 0x02d0, [0x0226] = 0x02d1, [0x0227] = 0x02d2,
  [0x0228] = 0x02d3, [0x0229] = 0x02d4, [0x022a] = 0x02d5, [0x022b] = 0x02d6,
  [0x022c] = 0x02d7, [0x022d] = 0x02d8, [0x022e] = 0x02da, [0x022f] = 0x02db,
  [0x0230] = 0x02dc, [0x0231] = 0x02dd, [0x0232] = 0x02de, [0x0233] = 0x02df,
  [0x0234] = 0x02e0, [0x0235] = 0x02e1, [0x0236] = 0x02e2, [0x0237] = 0x02e3,
  [0x0238] = 0x02e4, [0x0239] = 0x02e5, [0x023a] = 0x02e6, [0x023b] = 0x02e7,
  [0x023c] = 0x02e8, [0x023d] = 0x02e9, [0x023e] = 0x02ea, [0x023f] = 0x02eb,
  [0x0240] = 0x02ec, [0x0241] = 0x02ed, [0x0242] = 0x02ee, [0x0243] = 0x02ef,
  [0x0244] = 0x02f0, [0x0245] = 0x02f1, [0x0246] = 0x02f2, [0x0247] = 0x02f3,
  [0x0248] = 0x02f4, [0x0249] = 0x02f5, [0x024a] = 0x02f6, [0x024b] = 0x02f7,
  [0x024c] = 0x02f8, [0x024d] = 0x02f9, [0x024e] = 0x02fa, [0x024f] = 0x02fb,
  [0x0250] = 0x02fc, [0x0251] = 0x02fd, [0x0252] = 0x02fe, [0x0253] = 0x02ff,
  [0x0254] = 0x0300, [0x0255] = 0x0301, [0x0256] = 0x0302, [0x0257] = 0x0303,
  [0x0258] = 0x0304, [0x0259] = 0x0305, [0x025a] = 0x0306, [0x025b] = 0x0307,
  [0x025c] = 0x0308, [0x025d] = 0x0309, [0x025e] = 0x030a, [0x025f] = 0x030b,
  [0x0260] = 0x030c, [0x0261] = 0x030d, [0x0262] = 0x030e, [0x0263] = 0x030f,
  [0x0264] = 0x0310, [0x0265] = 0x0311, [0x0266] = 0x0312, [0x0267] = 0x0313,
  [0x0268] = 0x0314, [0x0269] = 0x0315, [0x026a] = 0x0316, [0x026b] = 0x0317,
  [0x026c] = 0x0318, [0x026d] = 0x0319, [0x026e] = 0x031a, [0x026f] = 0x031b,
  [0x0270] = 0x031c, [0x0271] = 0x031d, [0x0272] = 0x031e, [0x0273] = 0x031f,
  [0x0274] = 0x0320, [0x0275] = 0x0321, [0x0276] = 0x0322, [0x0277] = 0x0323,
  [0x0278] = 0x0324, [0x0279] = 0x0325, [0x027a] = 0x0326, [0x027b] = 0x0327,
  [0x027c] = 0x0328, [0x027d] = 0x0329, [0x027e] = 0x032a, [0x027f] = 0x032b,
  [0x0280] = 0x032c, [0x0281] = 0x032d, [0x0282] = 0x032e, [0x0283] = 0x032f,
  [0x0284] = 0x0330, [0x0285] = 0x0331, [0x0286] = 0x0332, [0x0287] = 0x0333,
  [0x0288] = 0x0334, [0x0289] = 0x0335, [0x028a] = 0x0336, [0x028b] = 0x0337,
  [0x028c] = 0x0338, [0x028d] = 0x0339, [0x028e] = 0x033a, [0x028f] = 0x033b,
  [0x0290] = 0x033c, [0x0291] = 0x033d, [0x0292] = 0x033e, [0x0293] = 0x033f,
  [0x0294] = 0x0340, [0x0295] = 0x0341, [0x0296] = 0x0342, [0x0297] = 0x0343,
  [0x0298] = 0x0344, [0x0299] = 0x0345, [0x029a] = 0x0346, [0x029b] = 0x0347,
  [0x029c] = 0x0348, [0x029d] = 0x0349, [0x029e] = 0x034a, [0x029f] = 0x034b,
  [0x02a0] = 0x034c, [0x02a1] = 0x034d, [0x02a2] = 0x034e, [0x02a3] = 0x034f,
  [0x02a4] = 0x0350, [0x02a5] = 0x0351, [0x02a6] = 0x0352, [0x02a7] = 0x0353,
  [0x02a8] = 0x0354, [0x02a9] = 0x0355, [0x02aa] = 0x0356, [0x02ab] = 0x0357,
  [0x02ac] = 0x0358, [0x02ad] = 0x0359, [0x02ae] = 0x035a, [0x02af] = 0x035b,
  [0x02b0] = 0x035c, [0x02b1] = 0x035d, [0x02b2] = 0x035e, [0x02b3] = 0x035f,
  [0x02b4] = 0x0360, [0x02b5] = 0x0361, [0x02b6] = 0x0362, [0x02b7] = 0x0363,
  [0x02b8] = 0x0364, [0x02b9] = 0x0365, [0x02ba] = 0x0366, [0x02bb] = 0x0367,
  [0x02bc] = 0x0368, [0x02bd] = 0x0369, [0x02be] = 0x036a, [0x02bf] = 0x036b,
  [0x02c0] = 0x036c, [0x02c1] = 0x036d, [0x02c2] = 0x036e, [0x02c3] = 0x036f,
  [0x02c4] = 0x0370, [0x02c5] = 0x0371, [0x02c6] = 0x0372, [0x02c7] = 0x0373,
  [0x02c8] = 0x0374, [0x02c9] = 0x0375, [0x02ca] = 0x0376, [0x02cb] = 0x0377,
  [0x02ce] = 0x037a, [0x02cf] = 0x037b, [0x02d0] = 0x037c, [0x02d1] = 0x037d,
  [0x02d2] = 0x037e, [0x02d8] = 0x0384, [0x02d9] = 0x0385, [0x02da] = 0x0386,
  [0x02db] = 0x0387, [0x02dc] = 0x0388, [0x02dd] = 0x0389, [0x02de] = 0x038a,
  [0x02e0] = 0x038c, [0x02e2] = 0x038e, [0x02e3] = 0x038f, [0x02e4] = 0x0390,
  [0x02e6] = 0x03aa, [0x02e7] = 0x03ab, [0x02e8] = 0x03ac, [0x02e9] = 0x03ad,
  [0x02ea] = 0x03ae, [0x02eb] = 0x03af, [0x02ec] = 0x03b0, [0x02ed] = 0x03c2,
  [0x02ee] = 0x03ca, [0x02ef] = 0x03cb, [0x02f0] = 0x03cc, [0x02f1] = 0x03cd,
  [0x02f2] = 0x03ce, [0x02f4] = 0x03d0, [0x02f5] = 0x03d1, [0x02f6] = 0x03d2,
  [0x02f7] = 0x03d3, [0x02f8] = 0x03d4, [0x02f9] = 0x03d5, [0x02fa] = 0x03d6,
  [0x02fb] = 0x03d7, [0x02fc] = 0x03d8, [0x02fd] = 0x03d9, [0x02fe] = 0x03da,
  [0x02ff] = 0x03db, [0x0300] = 0x03dc, [0x0301] = 0x03dd, [0x0302] = 0x03de,
  [0x0303] = 0x03df, [0x0304] = 0x03e0, [0x0305] = 0x03e1, [0x0306] = 0x03e2,
  [0x0307] = 0x03e3, [0x0308] = 0x03e4, [0x0309] = 0x03e5, [0x030a] = 0x03e6,
  [0x030b] = 0x03e7, [0x030c] = 0x03e8, [0x030d] = 0x03e9, [0x030e] = 0x03ea,
  [0x030f] = 0x03eb, [0x0310] = 0x03ec, [0x0311] = 0x03ed, [0x0312] = 0x03ee,
  [0x0313] = 0x03ef, [0x0314] = 0x03f0, [0x0315] = 0x03f1, [0x0316] = 0x03f2,
  [0x0317] = 0x03f3, [0x0318] = 0x03f4, [0x0319] = 0x03f5, [0x031a] = 0x03f6,
  [0x031b] = 0x03f7, [0x031c] = 0x03f8, [0x031d] = 0x03f9, [0x031e] = 0x03fa,
  [0x031f] = 0x03fb, [0x0320] = 0x03fc, [0x0321] = 0x03fd, [0x0322] = 0x03fe,
  [0x0323] = 0x03ff, [0x0324] = 0x0400, [0x0325] = 0x0402, [0x0326] = 0x0403,
  [0x0327] = 0x0404, [0x0328] = 0x0405, [0x0329] = 0x0406, [0x032a] = 0x0407,
  [0x032b] = 0x0408, [0x032c] = 0x0409, [0x032d] = 0x040a, [0x032e] = 0x040b,
  [0x032f] = 0x040c, [0x0330] = 0x040d, [0x0331] = 0x040e, [0x0332] = 0x040f,
  [0x0333] = 0x0450, [0x0334] = 0xe7c7, [0x0505] = 0x2011, [0x0506] = 0x2012,
  [0x0507] = 0x2017, [0x0508] = 0x201a, [0x0509] = 0x201b, [0x050a] = 0x201e,
  [0x050b] = 0x201f, [0x050c] = 0x2020, [0x050d] = 0x2021, [0x050e] = 0x2022,
  [0x050f] = 0x2023, [0x0510] = 0x2024, [0x0511] = 0x2027, [0x0512] = 0x2028,
  [0x0513] = 0x2029, [0x0514] = 0x202a, [0x0515] = 0x202b, [0x0516] = 0x202c,
  [0x0517] = 0x202d, [0x0518] = 0x202e, [0x0519] = 0x202f, [0x051a] = 0x2031,
  [0x051b] = 0x2034, [0x051c] = 0x2036, [0x051d] = 0x2037, [0x051e] = 0x2038,
  [0x051f] = 0x2039, [0x0520] = 0x203a, [0x0521] = 0x203c, [0x0522] = 0x203d,
  [0x0523] = 0x203e, [0x0524] = 0x203f, [0x0525] = 0x2040, [0x0526] = 0x2041,
  [0x0527] = 0x2042, [0x0528] = 0x2043, [0x0529] = 0x2044, [0x052a] = 0x2045,
  [0x052b] = 0x2046, [0x052c] = 0x2047, [0x052d] = 0x2048, [0x052e] = 0x2049,
  [0x052f] = 0x204a, [0x0530] = 0x204b, [0x0531] = 0x204c, [0x0532] = 0x204d,
  [0x0533] = 0x204e, [0x0534] = 0x204f, [0x0535] = 0x2050, [0x0536] = 0x2051,
  [0x0537] = 0x2052, [0x0538] = 0x2053, [0x0539] = 0x2054, [0x053a] = 0x2055,
  [0x053b] = 0x2056, [0x053c] = 0x2057, [0x053d] = 0x2058, [0x053e] = 0x2059,
  [0x053f] = 0x205a, [0x0540] = 0x205b, [0x0541] = 0x205c, [0x0542] = 0x205d,
  [0x0543] = 0x205e, [0x0544] = 0x205f, [0x0545] = 0x2060, [0x0546] = 0x2061,
  [0x0547] = 0x2062, [0x0548] = 0x2063, [0x0549] = 0x2064, [0x054f] = 0x206a,
  [0x0550] = 0x206b, [0x0551] = 0x206c, [0x0552] = 0x206d, [0x0553] = 0x206e,
  [0x0554] = 0x206f, [0x0555] = 0x2070, [0x0556] = 0x2071, [0x0559] = 0x2074,
  [0x055a] = 0x2075, [0x055b] = 0x2076, [0x055c] = 0x2077, [0x055d] = 0x2078,
  [0x055e] = 0x2079, [0x055f] = 0x207a, [0x0560] = 0x207b, [0x0561] = 0x207c,
  [0x0562] = 0x207d, [0x0563] = 0x207e, [0x0564] = 0x207f, [0x0565] = 0x2080,
  [0x0566] = 0x2081, [0x0567] = 0x2082, [0x0568] = 0x2083, [0x0569] = 0x2084,
  [0x056a] = 0x2085, [0x056b] = 0x2086, [0x056c] = 0x2087, [0x056d] = 0x2088,
  [0x056e] = 0x2089, [0x056f] = 0x208a, [0x0570] = 0x208b, [0x0571] = 0x208c,
  [0x0572] = 0x208d, [0x0573] = 0x208e, [0x0575] = 0x2090, [0x0576] = 0x2091,
  [0x0577] = 0x2092, [0x0578] = 0x2093, [0x0579] = 0x2094, [0x057a] = 0x2095,
  [0x057b] = 0x2096, [0x057c] = 0x2097, [0x057d] = 0x2098, [0x057e] = 0x2099,
  [0x057f] = 0x209a, [0x0580] = 0x209b, [0x0581] = 0x209c, [0x0585] = 0x20a0,
  [0x0586] = 0x20a1, [0x0587] = 0x20a2, [0x0588] = 0x20a3, [0x0589] = 0x20a4,
  [0x058a] = 0x20a5, [0x058b] = 0x20a6, [0x058c] = 0x20a7, [0x058d] = 0x20a8,
  [0x058e] = 0x20a9, [0x058f] = 0x20aa, [0x0590] = 0x20ab, [0x0591] = 0x20ad,
  [0x0592] = 0x20ae, [0x0593] = 0x20af, [0x0594] = 0x20b0, [0x0595] = 0x20b1,
  [0x0596] = 0x20b2, [0x0597] = 0x20b3, [0x0598] = 0x20b4, [0x0599] = 0x20b5,
  [0x059a] = 0x20b6, [0x059b] = 0x20b7, [0x059c] = 0x20b8, [0x059d] = 0x20b9,
  [0x05b4] = 0x20d0, [0x05b5] = 0x20d1, [0x05b6] = 0x20d2, [0x05b7] = 0x20d3,
  [0x05b8] = 0x20d4, [0x05b9] = 0x20d5, [0x05ba] = 0x20d6, [0x05bb] = 0x20d7,
  [0x05bc] = 0x20d8, [0x05bd] = 0x20d9, [0x05be] = 0x20da, [0x05bf] = 0x20db,
  [0x05c0] = 0x20dc, [0x05c1] = 0x20dd, [0x05c2] = 0x20de, [0x05c3] = 0x20df,
  [0x05c4] = 0x20e0, [0x05c5] = 0x20e1, [0x05c6] = 0x20e2, [0x05c7] = 0x20e3,
  [0x05c8] = 0x20e4, [0x05c9] = 0x20e5, [0x05ca] = 0x20e6, [0x05cb] = 0x20e7,
  [0x05cc] = 0x20e8, [0x05cd] = 0x20e9, [0x05ce] = 0x20ea, [0x05cf] = 0x20eb,
  [0x05d0] = 0x20ec, [0x05d1] = 0x20ed, [0x05d2] = 0x20ee, [0x05d3] = 0x20ef,
  [0x05d4] = 0x20f0, [0x05e4] = 0x2100, [0x05e5] = 0x2101, [0x05e6] = 0x2102,
  [0x05e7] = 0x2104, [0x05e8] = 0x2106, [0x05e9] = 0x2107, [0x05ea] = 0x2108,
  [0x05eb] = 0x210a, [0x05ec] = 0x210b, [0x05ed] = 0x210c, [0x05ee] = 0x210d,
  [0x05ef] = 0x210e, [0x05f0] = 0x210f, [0x05f1] = 0x2110, [0x05f2] = 0x2111,
  [0x05f3] = 0x2112, [0x05f4] = 0x2113, [0x05f5] = 0x2114, [0x05f6] = 0x2115,
  [0x05f7] = 0x2117, [0x05f8] = 0x2118, [0x05f9] = 0x2119, [0x05fa] = 0x211a,
  [0x05fb] = 0x211b, [0x05fc] = 0x211c, [0x05fd] = 0x211d, [0x05fe] = 0x211e,
  [0x05ff] = 0x211f, [0x0600] = 0x2120, [0x0601] = 0x2122, [0x0602] = 0x2123,
  [0x0603] = 0x2124, [0x0604] = 0x2125, [0x0605] = 0x2126, [0x0606] = 0x2127,
  [0x0607] = 0x2128, [0x0608] = 0x2129, [0x0609] = 0x212a, [0x060a] = 0x212b,
  [0x060b] = 0x212c, [0x060c] = 0x212d, [0x060d] = 0x212e, [0x060e] = 0x212f,
  [0x060f] = 0x2130, [0x0610] = 0x2131, [0x0611] = 0x2132, [0x0612] = 0x2133,
  [0x0613] = 0x2134, [0x0614] = 0x2135, [0x0615] = 0x2136, [0x0616] = 0x2137,
  [0x0617] = 0x2138, [0x0618] = 0x2139, [0x0619] = 0x213a, [0x061a] = 0x213b,
  [0x061b] = 0x213c, [0x061c] = 0x213d, [0x061d] = 0x213e, [0x061e] = 0x213f,
  [0x061f] = 0x2140, [0x0620] = 0x2141, [0x0621] = 0x2142, [0x0622] = 0x2143,
  [0x0623] = 0x2144, [0x0624] = 0x2145, [0x0625] = 0x2146, [0x0626] = 0x2147,
  [0x0627] = 0x2148, [0x0628] = 0x2149, [0x0629] = 0x214a, [0x062a] = 0x214b,
  [0x062b] = 0x214c, [0x062c] = 0x214d, [0x062d] = 0x214e, [0x062e] = 0x214f,
  [0x062f] = 0x2150, [0x0630] = 0x2151, [0x0631] = 0x2152, [0x0632] = 0x2153,
  [0x0633] = 0x2154, [0x0634] = 0x2155, [0x0635] = 0x2156, [0x0636] = 0x2157,
  [0x0637] = 0x2158, [0x0638] = 0x2159, [0x0639] = 0x215a, [0x063a] = 0x215b,
  [0x063b] = 0x215c, [0x063c] = 0x215d, [0x063d] = 0x215e, [0x063e] = 0x215f,
  [0x063f] = 0x216c, [0x0640] = 0x216d, [0x0641] = 0x216e, [0x0642] = 0x216f,
  [0x0643] = 0x217a, [0x0644] = 0x217b, [0x0645] = 0x217c, [0x0646] = 0x217d,
  [0x0647] = 0x217e, [0x0648] = 0x217f, [0x0649] = 0x2180, [0x064a] = 0x2181,
  [0x064b] = 0x2182, [0x064c] = 0x2183, [0x064d] = 0x2184, [0x064e] = 0x2185,
  [0x064f] = 0x2186, [0x0650] = 0x2187, [0x0651] = 0x2188, [0x0652] = 0x2189,
  [0x0659] = 0x2194, [0x065a] = 0x2195, [0x065b] = 0x219a, [0x065c] = 0x219b,
  [0x065d] = 0x219c, [0x065e] = 0x219d, [0x065f] = 0x219e, [0x0660] = 0x219f,
  [0x0661] = 0x21a0, [0x0662] = 0x21a1, [0x0663] = 0x21a2, [0x0664] = 0x21a3,
  [0x0665] = 0x21a4, [0x0666] = 0x21a5, [0x0667] = 0x21a6, [0x0668] = 0x21a7,
  [0x0669] = 0x21a8, [0x066a] = 0x21a9, [0x066b] = 0x21aa, [0x066c] = 0x21ab,
  [0x066d] = 0x21ac, [0x066e] = 0x21ad, [0x066f] = 0x21ae, [0x0670] = 0x21af,
  [0x0671] = 0x21b0, [0x0672] = 0x21b1, [0x0673] = 0x21b2, [0x0674] = 0x21b3,
  [0x0675] = 0x21b4, [0x0676] = 0x21b5, [0x0677] = 0x21b6, [0x0678] = 0x21b7,
  [0x0679] = 0x21b8, [0x067a] = 0x21b9, [0x067b] = 0x21ba, [0x067c] = 0x21bb,
  [0x067d] = 0x21bc, [0x067e] = 0x21bd, [0x067f] = 0x21be, [0x0680] = 0x21bf,
  [0x0681] = 0x21c0, [0x0682] = 0x21c1, [0x0683] = 0x21c2, [0x0684] = 0x21c3,
  [0x0685] = 0x21c4, [0x0686] = 0x21c5, [0x0687] = 0x21c6, [0x0688] = 0x21c7,
  [0x0689] = 0x21c8, [0x068a] = 0x21c9, [0x068b] = 0x21ca, [0x068c] = 0x21cb,
  [0x068d] = 0x21cc, [0x068e] = 0x21cd, [0x068f] = 0x21ce, [0x0690] = 0x21cf,
  [0x0691] = 0x21d0, [0x0692] = 0x21d1, [0x0693] = 0x21d2, [0x0694] = 0x21d3,
  [0x0695] = 0x21d4, [0x0696] = 0x21d5, [0x0697] = 0x21d6, [0x0698] = 0x21d7,
  [0x0699] = 0x21d8, [0x069a] = 0x21d9, [0x069b] = 0x21da, [0x069c] = 0x21db,
  [0x069d] = 0x21dc, [0x069e] = 0x21dd, [0x069f] = 0x21de, [0x06a0] = 0x21df,
  [0x06a1] = 0x21e0, [0x06a2] = 0x21e1, [0x06a3] = 0x21e2, [0x06a4] = 0x21e3,
  [0x06a5] = 0x21e4, [0x06a6] = 0x21e5, [0x06a7] = 0x21e6, [0x06a8] = 0x21e7,
  [0x06a9] = 0x21e8, [0x06aa] = 0x21e9, [0x06ab] = 0x21ea, [0x06ac] = 0x21eb,
  [0x06ad] = 0x21ec, [0x06ae] = 0x21ed, [0x06af] = 0x21ee, [0x06b0] = 0x21ef,
  [0x06b1] = 0x21f0, [0x06b2] = 0x21f1, [0x06b3] = 0x21f2, [0x06b4] = 0x21f3,
  [0x06b5] = 0x21f4, [0x06b6] = 0x21f5, [0x06b7] = 0x21f6, [0x06b8] = 0x21f7,
  [0x06b9] = 0x21f8, [0x06ba] = 0x21f9, [0x06bb] = 0x21fa, [0x06bc] = 0x21fb,
  [0x06bd] = 0x21fc, [0x06be] = 0x21fd, [0x06bf] = 0x21fe, [0x06c0] = 0x21ff,
  [0x06c1] = 0x2200, [0x06c2] = 0x2201, [0x06c3] = 0x2202, [0x06c4] = 0x2203,
  [0x06c5] = 0x2204, [0x06c6] = 0x2205, [0x06c7] = 0x2206, [0x06c8] = 0x2207,
  [0x06c9] = 0x2209, [0x06ca] = 0x220a, [0x06cb] = 0x220b, [0x06cc] = 0x220c,
  [0x06cd] = 0x220d, [0x06ce] = 0x220e, [0x06cf] = 0x2210, [0x06d0] = 0x2212,
  [0x06d1] = 0x2213, [0x06d2] = 0x2214, [0x06d3] = 0x2216, [0x06d4] = 0x2217,
  [0x06d5] = 0x2218, [0x06d6] = 0x2219, [0x06d7] = 0x221b, [0x06d8] = 0x221c,
  [0x06d9] = 0x2221, [0x06da] = 0x2222, [0x06db] = 0x2224, [0x06dc] = 0x2226,
  [0x06dd] = 0x222c, [0x06de] = 0x222d, [0x06df] = 0x222f, [0x06e0] = 0x2230,
  [0x06e1] = 0x2231, [0x06e2] = 0x2232, [0x06e3] = 0x2233, [0x06e4] = 0x2238,
  [0x06e5] = 0x2239, [0x06e6] = 0x223a, [0x06e7] = 0x223b, [0x06e8] = 0x223c,
  [0x06e9] = 0x223e, [0x06ea] = 0x223f, [0x06eb] = 0x2240, [0x06ec] = 0x2241,
  [0x06ed] = 0x2242, [0x06ee] = 0x2243, [0x06ef] = 0x2244, [0x06f0] = 0x2245,
  [0x06f1] = 0x2246, [0x06f2] = 0x2247, [0x06f3] = 0x2249, [0x06f4] = 0x224a,
  [0x06f5] = 0x224b, [0x06f6] = 0x224d, [0x06f7] = 0x224e, [0x06f8] = 0x224f,
  [0x06f9] = 0x2250, [0x06fa] = 0x2251, [0x06fb] = 0x2253, [0x06fc] = 0x2254,
  [0x06fd] = 0x2255, [0x06fe] = 0x2256, [0x06ff] = 0x2257, [0x0700] = 0x2258,
  [0x0701] = 0x2259, [0x0702] = 0x225a, [0x0703] = 0x225b, [0x0704] = 0x225c,
  [0x0705] = 0x225d, [0x0706] = 0x225e, [0x0707] = 0x225f, [0x0708] = 0x2262,
  [0x0709] = 0x2263, [0x070a] = 0x2268, [0x070b] = 0x2269, [0x070c] = 0x226a,
  [0x070d] = 0x226b, [0x070e] = 0x226c, [0x070f] = 0x226d, [0x0710] = 0x2270,
  [0x0711] = 0x2271, [0x0712] = 0x2272, [0x0713] = 0x2273, [0x0714] = 0x2274,
  [0x0715] = 0x2275, [0x0716] = 0x2276, [0x0717] = 0x2277, [0x0718] = 0x2278,
  [0x0719] = 0x2279, [0x071a] = 0x227a, [0x071b] = 0x227b, [0x071c] = 0x227c,
  [0x071d] = 0x227d, [0x071e] = 0x227e, [0x071f] = 0x227f, [0x0720] = 0x2280,
  [0x0721] = 0x2281, [0x0722] = 0x2282, [0x0723] = 0x2283, [0x0724] = 0x2284,
  [0x0725] = 0x2285, [0x0726] = 0x2286, [0x0727] = 0x2287, [0x0728] = 0x2288,
  [0x0729] = 0x2289, [0x072a] = 0x228a, [0x072b] = 0x228b, [0x072c] = 0x228c,
  [0x072d] = 0x228d, [0x072e] = 0x228e, [0x072f] = 0x228f, [0x0730] = 0x2290,
  [0x0731] = 0x2291, [0x0732] = 0x2292, [0x0733] = 0x2293, [0x0734] = 0x2294,
  [0x0735] = 0x2296, [0x0736] = 0x2297, [0x0737] = 0x2298, [0x0738] = 0x229a,
  [0x0739] = 0x229b, [0x073a] = 0x229c, [0x073b] = 0x229d, [0x073c] = 0x229e,
  [0x073d] = 0x229f, [0x073e] = 0x22a0, [0x073f] = 0x22a1, [0x0740] = 0x22a2,
  [0x0741] = 0x22a3, [0x0742] = 0x22a4, [0x0743] = 0x22a6, [0x0744] = 0x22a7,
  [0x0745] = 0x22a8, [0x0746] = 0x22a9, [0x0747] = 0x22aa, [0x0748] = 0x22ab,
  [0x0749] = 0x22ac, [0x074a] = 0x22ad, [0x074b] = 0x22ae, [0x074c] = 0x22af,
  [0x074d] = 0x22b0, [0x074e] = 0x22b1, [0x074f] = 0x22b2, [0x0750] = 0x22b3,
  [0x0751] = 0x22b4, [0x0752] = 0x22b5, [0x0753] = 0x22b6, [0x0754] = 0x22b7,
  [0x0755] = 0x22b8, [0x0756] = 0x22b9, [0x0757] = 0x22ba, [0x0758] = 0x22bb,
  [0x0759] = 0x22bc, [0x075a] = 0x22bd, [0x075b] = 0x22be, [0x075c] = 0x22c0,
  [0x075d] = 0x22c1, [0x075e] = 0x22c2, [0x075f] = 0x22c3, [0x0760] = 0x22c4,
  [0x0761] = 0x22c5, [0x0762] = 0x22c6, [0x0763] = 0x22c7, [0x0764] = 0x22c8,
  [0x0765] = 0x22c9, [0x0766] = 0x22ca, [0x0767] = 0x22cb, [0x0768] = 0x22cc,
  [0x0769] = 0x22cd, [0x076a] = 0x22ce, [0x076b] = 0x22cf, [0x076c] = 0x22d0,
  [0x076d] = 0x22d1, [0x076e] = 0x22d2, [0x076f] = 0x22d3, [0x0770] = 0x22d4,
  [0x0771] = 0x22d5, [0x0772] = 0x22d6, [0x0773] = 0x22d7, [0x0774] = 0x22d8,
  [0x0775] = 0x22d9, [0x0776] = 0x22da, [0x0777] = 0x22db, [0x0778] = 0x22dc,
  [0x0779] = 0x22dd, [0x077a] = 0x22de, [0x077b] = 0x22df, [0x077c] = 0x22e0,
  [0x077d] = 0x22e1, [0x077e] = 0x22e2, [0x077f] = 0x22e3, [0x0780] = 0x22e4,
  [0x0781] = 0x22e5, [0x0782] = 0x22e6, [0x0783] = 0x22e7, [0x0784] = 0x22e8,
  [0x0785] = 0x22e9, [0x0786] = 0x22ea, [0x0787] = 0x22eb, [0x0788] = 0x22ec,
  [0x0789] = 0x22ed, [0x078a] = 0x22ee, [0x078b] = 0x22ef, [0x078c] = 0x22f0,
  [0x078d] = 0x22f1, [0x078e] = 0x22f2, [0x078f] = 0x22f3, [0x0790] = 0x22f4,
  [0x0791] = 0x22f5, [0x0792] = 0x22f6, [0x0793] = 0x22f7, [0x0794] = 0x22f8,
  [0x0795] = 0x22f9, [0x0796] = 0x22fa, [0x0797] = 0x22fb, [0x0798] = 0x22fc,
  [0x0799] = 0x22fd, [0x079a] = 0x22fe, [0x079b] = 0x22ff, [0x079c] = 0x2300,
  [0x079d] = 0x2301, [0x079e] = 0x2302, [0x079f] = 0x2303, [0x07a0] = 0x2304,
  [0x07a1] = 0x2305, [0x07a2] = 0x2306, [0x07a3] = 0x2307, [0x07a4] = 0x2308,
  [0x07a5] = 0x2309, [0x07a6] = 0x230a, [0x07a7] = 0x230b, [0x07a8] = 0x230c,
  [0x07a9] = 0x230d, [0x07aa] = 0x230e, [0x07ab] = 0x230f, [0x07ac] = 0x2310,
  [0x07ad] = 0x2311, [0x07ae] = 0x2313, [0x07af] = 0x2314, [0x07b0] = 0x2315,
  [0x07b1] = 0x2316, [0x07b2] = 0x2317, [0x07b3] = 0x2318, [0x07b4] = 0x2319,
  [0x07b5] = 0x231a, [0x07b6] = 0x231b, [0x07b7] = 0x231c, [0x07b8] = 0x231d,
  [0x07b9] = 0x231e, [0x07ba] = 0x231f, [0x07bb] = 0x2320, [0x07bc] = 0x2321,
  [0x07bd] = 0x2322, [0x07be] = 0x2323, [0x07bf] = 0x2324, [0x07c0] = 0x2325,
  [0x07c1] = 0x2326, [0x07c2] = 0x2327, [0x07c3] = 0x2328, [0x07c4] = 0x2329,
  [0x07c5] = 0x232a, [0x07c6] = 0x232b, [0x07c7] = 0x232c, [0x07c8] = 0x232d,
  [0x07c9] = 0x232e, [0x07ca] = 0x232f, [0x07cb] = 0x2330, [0x07cc] = 0x2331,
  [0x07cd] = 0x2332, [0x07ce] = 0x2333, [0x07cf] = 0x2334, [0x07d0] = 0x2335,
  [0x07d1] = 0x2336, [0x07d2] = 0x2337, [0x07d3] = 0x2338, [0x07d4] = 0x2339,
  [0x07d5] = 0x233a, [0x07d6] = 0x233b, [0x07d7] = 0x233c, [0x07d8] = 0x233d,
  [0x07d9] = 0x233e, [0x07da] = 0x233f, [0x07db] = 0x2340, [0x07dc] = 0x2341,
  [0x07dd] = 0x2342, [0x07de] = 0x2343, [0x07df] = 0x2344, [0x07e0] = 0x2345,
  [0x07e1] = 0x2346, [0x07e2] = 0x2347, [0x07e3] = 0x2348, [0x07e4] = 0x2349,
  [0x07e5] = 0x234a, [0x07e6] = 0x234b, [0x07e7] = 0x234c, [0x07e8] = 0x234d,
  [0x07e9] = 0x234e, [0x07ea] = 0x234f, [0x07eb] = 0x2350, [0x07ec] = 0x2351,
  [0x07ed] = 0x2352, [0x07ee] = 0x2353, [0x07ef] = 0x2354, [0x07f0] = 0x2355,
  [0x07f1] = 0x2356, [0x07f2] = 0x2357, [0x07f3] = 0x2358, [0x07f4] = 0x2359,
  [0x07f5] = 0x235a, [0x07f6] = 0x235b, [0x07f7] = 0x235c, [0x07f8] = 0x235d,
  [0x07f9] = 0x235e, [0x07fa] = 0x235f, [0x07fb] = 0x2360, [0x07fc] = 0x2361,
  [0x07fd] = 0x2362, [0x07fe] = 0x2363, [0x07ff] = 0x2364, [0x0800] = 0x2365,
  [0x0801] = 0x2366, [0x0802] = 0x2367, [0x0803] = 0x2368, [0x0804] = 0x2369,
  [0x0805] = 0x236a, [0x0806] = 0x236b, [0x0807] = 0x236c, [0x0808] = 0x236d,
  [0x0809] = 0x236e, [0x080a] = 0x236f, [0x080b] = 0x2370, [0x080c] = 0x2371,
  [0x080d] = 0x2372, [0x080e] = 0x2373, [0x080f] = 0x2374, [0x0810] = 0x2375,
  [0x0811] = 0x2376, [0x0812] = 0x2377, [0x0813] = 0x2378, [0x0814] = 0x2379,
  [0x0815] = 0x237a, [0x0816] = 0x237b, [0x0817] = 0x237c, [0x0818] = 0x237d,
  [0x0819] = 0x237e, [0x081a] = 0x237f, [0x081b] = 0x2380, [0x081c] = 0x2381,
  [0x081d] = 0x2382, [0x081e] = 0x2383, [0x081f] = 0x2384, [0x0820] = 0x2385,
  [0x0821] = 0x2386, [0x0822] = 0x2387, [0x0823] = 0x2388, [0x0824] = 0x2389,
  [0x0825] = 0x238a, [0x0826] = 0x238b, [0x0827] = 0x238c, [0x0828] = 0x238d,
  [0x0829] = 0x238e, [0x082a] = 0x238f, [0x082b] = 0x2390, [0x082c] = 0x2391,
  [0x082d] = 0x2392, [0x082e] = 0x2393, [0x082f] = 0x2394, [0x0830] = 0x2395,
  [0x0831] = 0x2396, [0x0832] = 0x2397, [0x0833] = 0x2398, [0x0834] = 0x2399,
  [0x0835] = 0x239a, [0x0836] = 0x239b, [0x0837] = 0x239c, [0x0838] = 0x239d,
  [0x0839] = 0x239e, [0x083a] = 0x239f, [0x083b] = 0x23a0, [0x083c] = 0x23a1,
  [0x083d] = 0x23a2, [0x083e] = 0x23a3, [0x083f] = 0x23a4, [0x0840] = 0x23a5,
  [0x0841] = 0x23a6, [0x0842] = 0x23a7, [0x0843] = 0x23a8, [0x0844] = 0x23a9,
  [0x0845] = 0x23aa, [0x0846] = 0x23ab, [0x0847] = 0x23ac, [0x0848] = 0x23ad,
  [0x0849] = 0x23ae, [0x084a] = 0x23af, [0x084b] = 0x23b0, [0x084c] = 0x23b1,
  [0x084d] = 0x23b2, [0x084e] = 0x23b3, [0x084f] = 0x23b4, [0x0850] = 0x23b5,
  [0x0851] = 0x23b6, [0x0852] = 0x23b7, [0x0853] = 0x23b8, [0x0854] = 0x23b9,
  [0x0855] = 0x23ba, [0x0856] = 0x23bb, [0x0857] = 0x23bc, [0x0858] = 0x23bd,
  [0x0859] = 0x23be, [0x085a] = 0x23bf, [0x085b] = 0x23c0, [0x085c] = 0x23c1,
  [0x085d] = 0x23c2, [0x085e] = 0x23c3, [0x085f] = 0x23c4, [0x0860] = 0x23c5,
  [0x0861] = 0x23c6, [0x0862] = 0x23c7, [0x0863] = 0x23c8, [0x0864] = 0x23c9,
  [0x0865] = 0x23ca, [0x0866] = 0x23cb, [0x0867] = 0x23cc, [0x0868] = 0x23cd,
  [0x0869] = 0x23ce, [0x086a] = 0x23cf, [0x086b] = 0x23d0, [0x086c] = 0x23d1,
  [0x086d] = 0x23d2, [0x086e] = 0x23d3, [0x086f] = 0x23d4, [0x0870] = 0x23d5,
  [0x0871] = 0x23d6, [0x0872] = 0x23d7, [0x0873] = 0x23d8, [0x0874] = 0x23d9,
  [0x0875] = 0x23da, [0x0876] = 0x23db, [0x0877] = 0x23dc, [0x0878] = 0x23dd,
  [0x0879] = 0x23de, [0x087a] = 0x23df, [0x087b] = 0x23e0, [0x087c] = 0x23e1,
  [0x087d] = 0x23e2, [0x087e] = 0x23e3, [0x087f] = 0x23e4, [0x0880] = 0x23e5,
  [0x0881] = 0x23e6, [0x0882] = 0x23e7, [0x0883] = 0x23e8, [0x0884] = 0x23e9,
  [0x0885] = 0x23ea, [0x0886] = 0x23eb, [0x0887] = 0x23ec, [0x0888] = 0x23ed,
  [0x0889] = 0x23ee, [0x088a] = 0x23ef, [0x088b] = 0x23f0, [0x088c] = 0x23f1,
  [0x088d] = 0x23f2, [0x088e] = 0x23f3, [0x089b] = 0x2400, [0x089c] = 0x2401,
  [0x089d] = 0x2402, [0x089e] = 0x2403, [0x089f] = 0x2404, [0x08a0] = 0x2405,
  [0x08a1] = 0x2406, [0x08a2] = 0x2407, [0x08a3] = 0x2408, [0x08a4] = 0x2409,
  [0x08a5] = 0x240a, [0x08a6] = 0x240b, [0x08a7] = 0x240c, [0x08a8] = 0x240d,
  [0x08a9] = 0x240e, [0x08aa] = 0x240f, [0x08ab] = 0x2410, [0x08ac] = 0x2411,
  [0x08ad] = 0x2412, [0x08ae] = 0x2413, [0x08af] = 0x2414, [0x08b0] = 0x2415,
  [0x08b1] = 0x2416, [0x08b2] = 0x2417, [0x08b3] = 0x2418, [0x08b4] = 0x2419,
  [0x08b5] = 0x241a, [0x08b6] = 0x241b, [0x08b7] = 0x241c, [0x08b8] = 0x241d,
  [0x08b9] = 0x241e, [0x08ba] = 0x241f, [0x08bb] = 0x2420, [0x08bc] = 0x2421,
  [0x08bd] = 0x2422, [0x08be] = 0x2423, [0x08bf] = 0x2424, [0x08c0] = 0x2425,
  [0x08c1] = 0x2426, [0x08db] = 0x2440, [0x08dc] = 0x2441, [0x08dd] = 0x2442,
  [0x08de] = 0x2443, [0x08df] = 0x2444, [0x08e0] = 0x2445, [0x08e1] = 0x2446,
  [0x08e2] = 0x2447, [0x08e3] = 0x2448, [0x08e4] = 0x2449, [0x08e5] = 0x244a,
  [0x08fb] = 0x246a, [0x08fc] = 0x246b, [0x08fd] = 0x246c, [0x08fe] = 0x246d,
  [0x08ff] = 0x246e, [0x0900] = 0x246f, [0x0901] = 0x2470, [0x0902] = 0x2471,
  [0x0903] = 0x2472, [0x0904] = 0x2473, [0x0905] = 0x249c, [0x0906] = 0x249d,
  [0x0907] = 0x249e, [0x0908] = 0x249f, [0x0909] = 0x24a0, [0x090a] = 0x24a1,
  [0x090b] = 0x24a2, [0x090c] = 0x24a3, [0x090d] = 0x24a4, [0x090e] = 0x24a5,
  [0x090f] = 0x24a6, [0x0910] = 0x24a7, [0x0911] = 0x24a8, [0x0912] = 0x24a9,
  [0x0913] = 0x24aa, [0x0914] = 0x24ab, [0x0915] = 0x24ac, [0x0916] = 0x24ad,
  [0x0917] = 0x24ae, [0x0918] = 0x24af, [0x0919] = 0x24b0, [0x091a] = 0x24b1,
  [0x091b] = 0x24b2, [0x091c] = 0x24b3, [0x091d] = 0x24b4, [0x091e] = 0x24b5,
  [0x091f] = 0x24b6, [0x0920] = 0x24b7, [0x0921] = 0x24b8, [0x0922] = 0x24b9,
  [0x0923] = 0x24ba, [0x0924] = 0x24bb, [0x0925] = 0x24bc, [0x0926] = 0x24bd,
  [0x0927] = 0x24be, [0x0928] = 0x24bf, [0x0929] = 0x24c0, [0x092a] = 0x24c1,
  [0x092b] = 0x24c2, [0x092c] = 0x24c3, [0x092d] = 0x24c4, [0x092e] = 0x24c5,
  [0x092f] = 0x24c6, [0x0930] = 0x24c7, [0x0931] = 0x24c8, [0x0932] = 0x24c9,
  [0x0933] = 0x24ca, [0x0934] = 0x24cb, [0x0935] = 0x24cc, [0x0936] = 0x24cd,
  [0x0937] = 0x24ce, [0x0938] = 0x24cf, [0x0939] = 0x24d0, [0x093a] = 0x24d1,
  [0x093b] = 0x24d2, [0x093c] = 0x24d3, [0x093d] = 0x24d4, [0x093e] = 0x24d5,
  [0x093f] = 0x24d6, [0x0940] = 0x24d7, [0x0941] = 0x24d8, [0x0942] = 0x24d9,
  [0x0943] = 0x24da, [0x0944] = 0x24db, [0x0945] = 0x24dc, [0x0946] = 0x24dd,
  [0x0947] = 0x24de, [0x0948] = 0x24df, [0x0949] = 0x24e0, [0x094a] = 0x24e1,
  [0x094b] = 0x24e2, [0x094c] = 0x24e3, [0x094d] = 0x24e4, [0x094e] = 0x24e5,
  [0x094f] = 0x24e6, [0x0950] = 0x24e7, [0x0951] = 0x24e8, [0x0952] = 0x24e9,
  [0x0953] = 0x24ea, [0x0954] = 0x24eb, [0x0955] = 0x24ec, [0x0956] = 0x24ed,
  [0x0957] = 0x24ee, [0x0958] = 0x24ef, [0x0959] = 0x24f0, [0x095a] = 0x24f1,
  [0x095b] = 0x24f2, [0x095c] = 0x24f3, [0x095d] = 0x24f4, [0x095e] = 0x24f5,
  [0x095f] = 0x24f6, [0x0960] = 0x24f7, [0x0961] = 0x24f8, [0x0962] = 0x24f9,
  [0x0963] = 0x24fa, [0x0964] = 0x24fb, [0x0965] = 0x24fc, [0x0966] = 0x24fd,
  [0x0967] = 0x24fe, [0x0968] = 0x24ff, [0x0969] = 0x254c, [0x096a] = 0x254d,
  [0x096b] = 0x254e, [0x096c] = 0x254f, [0x096d] = 0x2574, [0x096e] = 0x2575,
  [0x096f] = 0x2576, [0x0970] = 0x2577, [0x0971] = 0x2578, [0x0972] = 0x2579,
  [0x0973] = 0x257a, [0x0974] = 0x257b, [0x0975] = 0x257c, [0x0976] = 0x257d,
  [0x0977] = 0x257e, [0x0978] = 0x257f, [0x0979] = 0x2580, [0x097a] = 0x2590,
  [0x097b] = 0x2591, [0x097c] = 0x2592, [0x097d] = 0x2596, [0x097e] = 0x2597,
  [0x097f] = 0x2598, [0x0980] = 0x2599, [0x0981] = 0x259a, [0x0982] = 0x259b,
  [0x0983] = 0x259c, [0x0984] = 0x259d, [0x0985] = 0x259e, [0x0986] = 0x259f,
  [0x0987] = 0x25a2, [0x0988] = 0x25a3, [0x0989] = 0x25a4, [0x098a] = 0x25a5,
  [0x098b] = 0x25a6, [0x098c] = 0x25a7, [0x098d] = 0x25a8, [0x098e] = 0x25a9,
  [0x098f] = 0x25aa, [0x0990] = 0x25ab, [0x0991] = 0x25ac, [0x0992] = 0x25ad,
  [0x0993] = 0x25ae, [0x0994] = 0x25af, [0x0995] = 0x25b0, [0x0996] = 0x25b1,
  [0x0997] = 0x25b4, [0x0998] = 0x25b5, [0x0999] = 0x25b6, [0x099a] = 0x25b7,
  [0x099b] = 0x25b8, [0x099c] = 0x25b9, [0x099d] = 0x25ba, [0x099e] = 0x25bb,
  [0x099f] = 0x25be, [0x09a0] = 0x25bf, [0x09a1] = 0x25c0, [0x09a2] = 0x25c1,
  [0x09a3] = 0x25c2, [0x09a4] = 0x25c3, [0x09a5] = 0x25c4, [0x09a6] = 0x25c5,
  [0x09a7] = 0x25c8, [0x09a8] = 0x25c9, [0x09a9] = 0x25ca, [0x09aa] = 0x25cc,
  [0x09ab] = 0x25cd, [0x09ac] = 0x25d0, [0x09ad] = 0x25d1, [0x09ae] = 0x25d2,
  [0x09af] = 0x25d3, [0x09b0] = 0x25d4, [0x09b1] = 0x25d5, [0x09b2] = 0x25d6,
  [0x09b3] = 0x25d7, [0x09b4] = 0x25d8, [0x09b5] = 0x25d9, [0x09b6] = 0x25da,
  [0x09b7] = 0x25db, [0x09b8] = 0x25dc, [0x09b9] = 0x25dd, [0x09ba] = 0x25de,
  [0x09bb] = 0x25df, [0x09bc] = 0x25e0, [0x09bd] = 0x25e1, [0x09be] = 0x25e6,
  [0x09bf] = 0x25e7, [0x09c0] = 0x25e8, [0x09c1] = 0x25e9, [0x09c2] = 0x25ea,
  [0x09c3] = 0x25eb, [0x09c4] = 0x25ec, [0x09c5] = 0x25ed, [0x09c6] = 0x25ee,
  [0x09c7] = 0x25ef, [0x09c8] = 0x25f0, [0x09c9] = 0x25f1, [0x09ca] = 0x25f2,
  [0x09cb] = 0x25f3, [0x09cc] = 0x25f4, [0x09cd] = 0x25f5, [0x09ce] = 0x25f6,
  [0x09cf] = 0x25f7, [0x09d0] = 0x25f8, [0x09d1] = 0x25f9, [0x09d2] = 0x25fa,
  [0x09d3] = 0x25fb, [0x09d4] = 0x25fc, [0x09d5] = 0x25fd, [0x09d6] = 0x25fe,
  [0x09d7] = 0x25ff, [0x09d8] = 0x2600, [0x09d9] = 0x2601, [0x09da] = 0x2602,
  [0x09db] = 0x2603, [0x09dc] = 0x2604, [0x09dd] = 0x2607, [0x09de] = 0x2608,
  [0x09df] = 0x260a, [0x09e0] = 0x260b, [0x09e1] = 0x260c, [0x09e2] = 0x260d,
  [0x09e3] = 0x260e, [0x09e4] = 0x260f, [0x09e5] = 0x2610, [0x09e6] = 0x2611,
  [0x09e7] = 0x2612, [0x09e8] = 0x2613, [0x09e9] = 0x2614, [0x09ea] = 0x2615,
  [0x09eb] = 0x2616, [0x09ec] = 0x2617, [0x09ed] = 0x2618, [0x09ee] = 0x2619,
  [0x09ef] = 0x261a, [0x09f0] = 0x261b, [0x09f1] = 0x261c, [0x09f2] = 0x261d,
  [0x09f3] = 0x261e, [0x09f4] = 0x261f, [0x09f5] = 0x2620, [0x09f6] = 0x2621,
  [0x09f7] = 0x2622, [0x09f8] = 0x2623, [0x09f9] = 0x2624, [0x09fa] = 0x2625,
  [0x09fb] = 0x2626, [0x09fc] = 0x2627, [0x09fd] = 0x2628, [0x09fe] = 0x2629,
  [0x09ff] = 0x262a, [0x0a00] = 0x262b, [0x0a01] = 0x262c, [0x0a02] = 0x262d,
  [0x0a03] = 0x262e, [0x0a04] = 0x262f, [0x0a05] = 0x2630, [0x0a06] = 0x2631,
  [0x0a07] = 0x2632, [0x0a08] = 0x2633, [0x0a09] = 0x2634, [0x0a0a] = 0x2635,
  [0x0a0b] = 0x2636, [0x0a0c] = 0x2637, [0x0a0d] = 0x2638, [0x0a0e] = 0x2639,
  [0x0a0f] = 0x263a, [0x0a10] = 0x263b, [0x0a11] = 0x263c, [0x0a12] = 0x263d,
  [0x0a13] = 0x263e, [0x0a14] = 0x263f, [0x0a15] = 0x2641, [0x0a16] = 0x2e82,
  [0x0a17] = 0x2e83, [0x0a18] = 0x2e85, [0x0a19] = 0x2e86, [0x0a1a] = 0x2e87,
  [0x0a1b] = 0x2e89, [0x0a1c] = 0x2e8a, [0x0a1d] = 0x2e8d, [0x0a1e] = 0x2e8e,
  [0x0a1f] = 0x2e8f, [0x0a20] = 0x2e90, [0x0a21] = 0x2e91, [0x0a22] = 0x2e92,
  [0x0a23] = 0x2e93, [0x0a24] = 0x2e94, [0x0a25] = 0x2e95, [0x0a26] = 0x2e96,
  [0x0a27] = 0x2e98, [0x0a28] = 0x2e99, [0x0a2a] = 0x2e9b, [0x0a2b] = 0x2e9c,
  [0x0a2c] = 0x2e9d, [0x0a2d] = 0x2e9e, [0x0a2e] = 0x2e9f, [0x0a2f] = 0x2ea0,
  [0x0a30] = 0x2ea1, [0x0a31] = 0x2ea2, [0x0a32] = 0x2ea3, [0x0a33] = 0x2ea4,
  [0x0a34] = 0x2ea5, [0x0a35] = 0x2ea6, [0x0a36] = 0x2ea8, [0x0a37] = 0x2ea9,
  [0x0a38] = 0x2eab, [0x0a39] = 0x2eac, [0x0a3a] = 0x2ead, [0x0a3b] = 0x2eaf,
  [0x0a3c] = 0x2eb0, [0x0a3d] = 0x2eb1, [0x0a3e] = 0x2eb2, [0x0a3f] = 0x2eb4,
  [0x0a40] = 0x2eb5, [0x0a41] = 0x2eb8, [0x0a42] = 0x2eb9, [0x0a43] = 0x2eba,
  [0x0a44] = 0x2ebc, [0x0a45] = 0x2ebd, [0x0a46] = 0x2ebe, [0x0a47] = 0x2ebf,
  [0x0a48] = 0x2ec0, [0x0a49] = 0x2ec1, [0x0a4a] = 0x2ec2, [0x0a4b] = 0x2ec3,
  [0x0a4c] = 0x2ec4, [0x0a4d] = 0x2ec5, [0x0a4e] = 0x2ec6, [0x0a4f] = 0x2ec7,
  [0x0a50] = 0x2ec8, [0x0a51] = 0x2ec9, [0x0a52] = 0x2ecb, [0x0a53] = 0x2ecc,
  [0x0a54] = 0x2ecd, [0x0a55] = 0x2ece, [0x0a56] = 0x2ecf, [0x0a57] = 0x2ed0,
  [0x0a58] = 0x2ed1, [0x0a59] = 0x2ed2, [0x0a5a] = 0x2ed3, [0x0a5b] = 0x2ed4,
  [0x0a5c] = 0x2ed5, [0x0a5d] = 0x2ed6, [0x0a5e] = 0x2ed7, [0x0a5f] = 0x2ed8,
  [0x0a60] = 0x2ed9, [0x0a61] = 0x2eda, [0x0a62] = 0x2edb, [0x0a63] = 0x2edc,
  [0x0a64] = 0x2edd, [0x0a65] = 0x2ede, [0x0a66] = 0x2edf, [0x0a67] = 0x2ee0,
  [0x0a68] = 0x2ee1, [0x0a69] = 0x2ee2, [0x0a6a] = 0x2ee3, [0x0a6b] = 0x2ee4,
  [0x0a6c] = 0x2ee5, [0x0a6d] = 0x2ee6, [0x0a6e] = 0x2ee7, [0x0a6f] = 0x2ee8,
  [0x0a70] = 0x2ee9, [0x0a71] = 0x2eea, [0x0a72] = 0x2eeb, [0x0a73] = 0x2eec,
  [0x0a74] = 0x2eed, [0x0a75] = 0x2eee, [0x0a76] = 0x2eef, [0x0a77] = 0x2ef0,
  [0x0a78] = 0x2ef1, [0x0a79] = 0x2ef2, [0x0a7a] = 0x2ef3, [0x0a87] = 0x2f00,
  [0x0a88] = 0x2f01, [0x0a89] = 0x2f02, [0x0a8a] = 0x2f03, [0x0a8b] = 0x2f04,
  [0x0a8c] = 0x2f05, [0x0a8d] = 0x2f06, [0x0a8e] = 0x2f07, [0x0a8f] = 0x2f08,
  [0x0a90] = 0x2f09, [0x0a91] = 0x2f0a, [0x0a92] = 0x2f0b, [0x0a93] = 0x2f0c,
  [0x0a94] = 0x2f0d, [0x0a95] = 0x2f0e, [0x0a96] = 0x2f0f, [0x0a97] = 0x2f10,
  [0x0a98] = 0x2f11, [0x0a99] = 0x2f12, [0x0a9a] = 0x2f13, [0x0a9b] = 0x2f14,
  [0x0a9c] = 0x2f15, [0x0a9d] = 0x2f16, [0x0a9e] = 0x2f17, [0x0a9f] = 0x2f18,
  [0x0aa0] = 0x2f19, [0x0aa1] = 0x2f1a, [0x0aa2] = 0x2f1b, [0x0aa3] = 0x2f1c,
  [0x0aa4] = 0x2f1d, [0x0aa5] = 0x2f1e, [0x0aa6] = 0x2f1f, [0x0aa7] = 0x2f20,
  [0x0aa8] = 0x2f21, [0x0aa9] = 0x2f22, [0x0aaa] = 0x2f23, [0x0aab] = 0x2f24,
  [0x0aac] = 0x2f25, [0x0aad] = 0x2f26, [0x0aae] = 0x2f27, [0x0aaf] = 0x2f28,
  [0x0ab0] = 0x2f29, [0x0ab1] = 0x2f2a, [0x0ab2] = 0x2f2b, [0x0ab3] = 0x2f2c,
  [0x0ab4] = 0x2f2d, [0x0ab5] = 0x2f2e, [0x0ab6] = 0x2f2f, [0x0ab7] = 0x2f30,
  [0x0ab8] = 0x2f31, [0x0ab9] = 0x2f32, [0x0aba] = 0x2f33, [0x0abb] = 0x2f34,
  [0x0abc] = 0x2f35, [0x0abd] = 0x2f36, [0x0abe] = 0x2f37, [0x0abf] = 0x2f38,
  [0x0ac0] = 0x2f39, [0x0ac1] = 0x2f3a, [0x0ac2] = 0x2f3b, [0x0ac3] = 0x2f3c,
  [0x0ac4] = 0x2f3d, [0x0ac5] = 0x2f3e, [0x0ac6] = 0x2f3f, [0x0ac7] = 0x2f40,
  [0x0ac8] = 0x2f41, [0x0ac9] = 0x2f42, [0x0aca] = 0x2f43, [0x0acb] = 0x2f44,
  [0x0acc] = 0x2f45, [0x0acd] = 0x2f46, [0x0ace] = 0x2f47, [0x0acf] = 0x2f48,
  [0x0ad0] = 0x2f49, [0x0ad1] = 0x2f4a, [0x0ad2] = 0x2f4b, [0x0ad3] = 0x2f4c,
  [0x0ad4] = 0x2f4d, [0x0ad5] = 0x2f4e, [0x0ad6] = 0x2f4f, [0x0ad7] = 0x2f50,
  [0x0ad8] = 0x2f51, [0x0ad9] = 0x2f52, [0x0ada] = 0x2f53, [0x0adb] = 0x2f54,
  [0x0adc] = 0x2f55, [0x0add] = 0x2f56, [0x0ade] = 0x2f57, [0x0adf] = 0x2f58,
  [0x0ae0] = 0x2f59, [0x0ae1] = 0x2f5a, [0x0ae2] = 0x2f5b, [0x0ae3] = 0x2f5c,
  [0x0ae4] = 0x2f5d, [0x0ae5] = 0x2f5e, [0x0ae6] = 0x2f5f, [0x0ae7] = 0x2f60,
  [0x0ae8] = 0x2f61, [0x0ae9] = 0x2f62, [0x0aea] = 0x2f63, [0x0aeb] = 0x2f64,
  [0x0aec] = 0x2f65, [0x0aed] = 0x2f66, [0x0aee] = 0x2f67, [0x0aef] = 0x2f68,
  [0x0af0] = 0x2f69, [0x0af1] = 0x2f6a, [0x0af2] = 0x2f6b, [0x0af3] = 0x2f6c,
  [0x0af4] = 0x2f6d, [0x0af5] = 0x2f6e, [0x0af6] = 0x2f6f, [0x0af7] = 0x2f70,
  [0x0af8] = 0x2f71, [0x0af9] = 0x2f72, [0x0afa] = 0x2f73, [0x0afb] = 0x2f74,
  [0x0afc] = 0x2f75, [0x0afd] = 0x2f76, [0x0afe] = 0x2f77, [0x0aff] = 0x2f78,
  [0x0b00] = 0x2f79, [0x0b01] = 0x2f7a, [0x0b02] = 0x2f7b, [0x0b03] = 0x2f7c,
  [0x0b04] = 0x2f7d, [0x0b05] = 0x2f7e, [0x0b06] = 0x2f7f, [0x0b07] = 0x2f80,
  [0x0b08] = 0x2f81, [0x0b09] = 0x2f82, [0x0b0a] = 0x2f83, [0x0b0b] = 0x2f84,
  [0x0b0c] = 0x2f85, [0x0b0d] = 0x2f86, [0x0b0e] = 0x2f87, [0x0b0f] = 0x2f88,
  [0x0b10] = 0x2f89, [0x0b11] = 0x2f8a, [0x0b12] = 0x2f8b, [0x0b13] = 0x2f8c,
  [0x0b14] = 0x2f8d, [0x0b15] = 0x2f8e, [0x0b16] = 0x2f8f, [0x0b17] = 0x2f90,
  [0x0b18] = 0x2f91, [0x0b19] = 0x2f92, [0x0b1a] = 0x2f93, [0x0b1b] = 0x2f94,
  [0x0b1c] = 0x2f95, [0x0b1d] = 0x2f96, [0x0b1e] = 0x2f97, [0x0b1f] = 0x2f98,
  [0x0b20] = 0x2f99, [0x0b21] = 0x2f9a, [0x0b22] = 0x2f9b, [0x0b23] = 0x2f9c,
  [0x0b24] = 0x2f9d, [0x0b25] = 0x2f9e, [0x0b26] = 0x2f9f, [0x0b27] = 0x2fa0,
  [0x0b28] = 0x2fa1, [0x0b29] = 0x2fa2, [0x0b2a] = 0x2fa3, [0x0b2b] = 0x2fa4,
  [0x0b2c] = 0x2fa5, [0x0b2d] = 0x2fa6, [0x0b2e] = 0x2fa7, [0x0b2f] = 0x2fa8,
  [0x0b30] = 0x2fa9, [0x0b31] = 0x2faa, [0x0b32] = 0x2fab, [0x0b33] = 0x2fac,
  [0x0b34] = 0x2fad, [0x0b35] = 0x2fae, [0x0b36] = 0x2faf, [0x0b37] = 0x2fb0,
  [0x0b38] = 0x2fb1, [0x0b39] = 0x2fb2, [0x0b3a] = 0x2fb3, [0x0b3b] = 0x2fb4,
  [0x0b3c] = 0x2fb5, [0x0b3d] = 0x2fb6, [0x0b3e] = 0x2fb7, [0x0b3f] = 0x2fb8,
  [0x0b40] = 0x2fb9, [0x0b41] = 0x2fba, [0x0b42] = 0x2fbb, [0x0b43] = 0x2fbc,
  [0x0b44] = 0x2fbd, [0x0b45] = 0x2fbe, [0x0b46] = 0x2fbf, [0x0b47] = 0x2fc0,
  [0x0b48] = 0x2fc1, [0x0b49] = 0x2fc2, [0x0b4a] = 0x2fc3, [0x0b4b] = 0x2fc4,
  [0x0b4c] = 0x2fc5, [0x0b4d] = 0x2fc6, [0x0b4e] = 0x2fc7, [0x0b4f] = 0x2fc8,
  [0x0b50] = 0x2fc9, [0x0b51] = 0x2fca, [0x0b52] = 0x2fcb, [0x0b53] = 0x2fcc,
  [0x0b54] = 0x2fcd, [0x0b55] = 0x2fce, [0x0b56] = 0x2fcf, [0x0b57] = 0x2fd0,
  [0x0b58] = 0x2fd1, [0x0b59] = 0x2fd2, [0x0b5a] = 0x2fd3, [0x0b5b] = 0x2fd4,
  [0x0b5c] = 0x2fd5, [0x0b7b] = 0x3004, [0x0b7c] = 0x3018, [0x0b7d] = 0x3019,
  [0x0b7e] = 0x301a, [0x0b7f] = 0x301b, [0x0b80] = 0x301c, [0x0b81] = 0x301f,
  [0x0b82] = 0x3020, [0x0b83] = 0x302a, [0x0b84] = 0x302b, [0x0b85] = 0x302c,
  [0x0b86] = 0x302d, [0x0b87] = 0x302e, [0x0b88] = 0x302f, [0x0b89] = 0x3030,
  [0x0b8a] = 0x3031, [0x0b8b] = 0x3032, [0x0b8c] = 0x3033, [0x0b8d] = 0x3034,
  [0x0b8e] = 0x3035, [0x0b8f] = 0x3036, [0x0b90] = 0x3037, [0x0b91] = 0x3038,
  [0x0b92] = 0x3039, [0x0b93] = 0x303a, [0x0b94] = 0x303b, [0x0b95] = 0x303c,
  [0x0b96] = 0x303d, [0x0b97] = 0x303f, [0x0b99] = 0x3094, [0x0b9a] = 0x3095,
  [0x0b9b] = 0x3096, [0x0b9e] = 0x3099, [0x0b9f] = 0x309a, [0x0ba0] = 0x309f,
  [0x0ba1] = 0x30a0, [0x0ba2] = 0x30f7, [0x0ba3] = 0x30f8, [0x0ba4] = 0x30f9,
  [0x0ba5] = 0x30fa, [0x0ba6] = 0x30fb, [0x0ba7] = 0x30ff, [0x0bad] = 0x312a,
  [0x0bae] = 0x312b, [0x0baf] = 0x312c, [0x0bb0] = 0x312d, [0x0bb4] = 0x3131,
  [0x0bb5] = 0x3132, [0x0bb6] = 0x3133, [0x0bb7] = 0x3134, [0x0bb8] = 0x3135,
  [0x0bb9] = 0x3136, [0x0bba] = 0x3137, [0x0bbb] = 0x3138, [0x0bbc] = 0x3139,
  [0x0bbd] = 0x313a, [0x0bbe] = 0x313b, [0x0bbf] = 0x313c, [0x0bc0] = 0x313d,
  [0x0bc1] = 0x313e, [0x0bc2] = 0x313f, [0x0bc3] = 0x3140, [0x0bc4] = 0x3141,
  [0x0bc5] = 0x3142, [0x0bc6] = 0x3143, [0x0bc7] = 0x3144, [0x0bc8] = 0x3145,
  [0x0bc9] = 0x3146, [0x0bca] = 0x3147, [0x0bcb] = 0x3148, [0x0bcc] = 0x3149,
  [0x0bcd] = 0x314a, [0x0bce] = 0x314b, [0x0bcf] = 0x314c, [0x0bd0] = 0x314d,
  [0x0bd1] = 0x314e, [0x0bd2] = 0x314f, [0x0bd3] = 0x3150, [0x0bd4] = 0x3151,
  [0x0bd5] = 0x3152, [0x0bd6] = 0x3153, [0x0bd7] = 0x3154, [0x0bd8] = 0x3155,
  [0x0bd9] = 0x3156, [0x0bda] = 0x3157, [0x0bdb] = 0x3158, [0x0bdc] = 0x3159,
  [0x0bdd] = 0x315a, [0x0bde] = 0x315b, [0x0bdf] = 0x315c, [0x0be0] = 0x315d,
  [0x0be1] = 0x315e, [0x0be2] = 0x315f, [0x0be3] = 0x3160, [0x0be4] = 0x3161,
  [0x0be5] = 0x3162, [0x0be6] = 0x3163, [0x0be7] = 0x3164, [0x0be8] = 0x3165,
  [0x0be9] = 0x3166, [0x0bea] = 0x3167, [0x0beb] = 0x3168, [0x0bec] = 0x3169,
  [0x0bed] = 0x316a, [0x0bee] = 0x316b, [0x0bef] = 0x316c, [0x0bf0] = 0x316d,
  [0x0bf1] = 0x316e, [0x0bf2] = 0x316f, [0x0bf3] = 0x3170, [0x0bf4] = 0x3171,
  [0x0bf5] = 0x3172, [0x0bf6] = 0x3173, [0x0bf7] = 0x3174, [0x0bf8] = 0x3175,
  [0x0bf9] = 0x3176, [0x0bfa] = 0x3177, [0x0bfb] = 0x3178, [0x0bfc] = 0x3179,
  [0x0bfd] = 0x317a, [0x0bfe] = 0x317b, [0x0bff] = 0x317c, [0x0c00] = 0x317d,
  [0x0c01] = 0x317e, [0x0c02] = 0x317f, [0x0c03] = 0x3180, [0x0c04] = 0x3181,
  [0x0c05] = 0x3182, [0x0c06] = 0x3183, [0x0c07] = 0x3184, [0x0c08] = 0x3185,
  [0x0c09] = 0x3186, [0x0c0a] = 0x3187, [0x0c0b] = 0x3188, [0x0c0c] = 0x3189,
  [0x0c0d] = 0x318a, [0x0c0e] = 0x318b, [0x0c0f] = 0x318c, [0x0c10] = 0x318d,
  [0x0c11] = 0x318e, [0x0c13] = 0x3190, [0x0c14] = 0x3191, [0x0c15] = 0x3192,
  [0x0c16] = 0x3193, [0x0c17] = 0x3194, [0x0c18] = 0x3195, [0x0c19] = 0x3196,
  [0x0c1a] = 0x3197, [0x0c1b] = 0x3198, [0x0c1c] = 0x3199, [0x0c1d] = 0x319a,
  [0x0c1e] = 0x319b, [0x0c1f] = 0x319c, [0x0c20] = 0x319d, [0x0c21] = 0x319e,
  [0x0c22] = 0x319f, [0x0c23] = 0x31a0, [0x0c24] = 0x31a1, [0x0c25] = 0x31a2,
  [0x0c26] = 0x31a3, [0x0c27] = 0x31a4, [0x0c28] = 0x31a5, [0x0c29] = 0x31a6,
  [0x0c2a] = 0x31a7, [0x0c2b] = 0x31a8, [0x0c2c] = 0x31a9, [0x0c2d] = 0x31aa,
  [0x0c2e] = 0x31ab, [0x0c2f] = 0x31ac, [0x0c30] = 0x31ad, [0x0c31] = 0x31ae,
  [0x0c32] = 0x31af, [0x0c33] = 0x31b0, [0x0c34] = 0x31b1, [0x0c35] = 0x31b2,
  [0x0c36] = 0x31b3, [0x0c37] = 0x31b4, [0x0c38] = 0x31b5, [0x0c39] = 0x31b6,
  [0x0c3a] = 0x31b7, [0x0c3b] = 0x31b8, [0x0c3c] = 0x31b9, [0x0c3d] = 0x31ba,
  [0x0c43] = 0x31c0, [0x0c44] = 0x31c1, [0x0c45] = 0x31c2, [0x0c46] = 0x31c3,
  [0x0c47] = 0x31c4, [0x0c48] = 0x31c5, [0x0c49] = 0x31c6, [0x0c4a] = 0x31c7,
  [0x0c4b] = 0x31c8, [0x0c4c] = 0x31c9, [0x0c4d] = 0x31ca, [0x0c4e] = 0x31cb,
  [0x0c4f] = 0x31cc, [0x0c50] = 0x31cd, [0x0c51] = 0x31ce, [0x0c52] = 0x31cf,
  [0x0c53] = 0x31d0, [0x0c54] = 0x31d1, [0x0c55] = 0x31d2, [0x0c56] = 0x31d3,
  [0x0c57] = 0x31d4, [0x0c58] = 0x31d5, [0x0c59] = 0x31d6, [0x0c5a] = 0x31d7,
  [0x0c5b] = 0x31d8, [0x0c5c] = 0x31d9, [0x0c5d] = 0x31da, [0x0c5e] = 0x31db,
  [0x0c5f] = 0x31dc, [0x0c60] = 0x31dd, [0x0c61] = 0x31de, [0x0c62] = 0x31df,
  [0x0c63] = 0x31e0, [0x0c64] = 0x31e1, [0x0c65] = 0x31e2, [0x0c66] = 0x31e3,
  [0x0c73] = 0x31f0, [0x0c74] = 0x31f1, [0x0c75] = 0x31f2, [0x0c76] = 0x31f3,
  [0x0c77] = 0x31f4, [0x0c78] = 0x31f5, [0x0c79] = 0x31f6, [0x0c7a] = 0x31f7,
  [0x0c7b] = 0x31f8, [0x0c7c] = 0x31f9, [0x0c7d] = 0x31fa, [0x0c7e] = 0x31fb,
  [0x0c7f] = 0x31fc, [0x0c80] = 0x31fd, [0x0c81] = 0x31fe, [0x0c82] = 0x31ff,
  [0x0c83] = 0x3200, [0x0c84] = 0x3201, [0x0c85] = 0x3202, [0x0c86] = 0x3203,
  [0x0c87] = 0x3204, [0x0c88] = 0x3205, [0x0c89] = 0x3206, [0x0c8a] = 0x3207,
  [0x0c8b] = 0x3208, [0x0c8c] = 0x3209, [0x0c8d] = 0x320a, [0x0c8e] = 0x320b,
  [0x0c8f] = 0x320c, [0x0c90] = 0x320d, [0x0c91] = 0x320e, [0x0c92] = 0x320f,
  [0x0c93] = 0x3210, [0x0c94] = 0x3211, [0x0c95] = 0x3212, [0x0c96] = 0x3213,
  [0x0c97] = 0x3214, [0x0c98] = 0x3215, [0x0c99] = 0x3216, [0x0c9a] = 0x3217,
  [0x0c9b] = 0x3218, [0x0c9c] = 0x3219, [0x0c9d] = 0x321a, [0x0c9e] = 0x321b,
  [0x0c9f] = 0x321c, [0x0ca0] = 0x321d, [0x0ca1] = 0x321e, [0x0ca3] = 0x322a,
  [0x0ca4] = 0x322b, [0x0ca5] = 0x322c, [0x0ca6] = 0x322d, [0x0ca7] = 0x322e,
  [0x0ca8] = 0x322f, [0x0ca9] = 0x3230, [0x0caa] = 0x3232, [0x0cab] = 0x3233,
  [0x0cac] = 0x3234, [0x0cad] = 0x3235, [0x0cae] = 0x3236, [0x0caf] = 0x3237,
  [0x0cb0] = 0x3238, [0x0cb1] = 0x3239, [0x0cb2] = 0x323a, [0x0cb3] = 0x323b,
  [0x0cb4] = 0x323c, [0x0cb5] = 0x323d, [0x0cb6] = 0x323e, [0x0cb7] = 0x323f,
  [0x0cb8] = 0x3240, [0x0cb9] = 0x3241, [0x0cba] = 0x3242, [0x0cbb] = 0x3243,
  [0x0cbc] = 0x3244, [0x0cbd] = 0x3245, [0x0cbe] = 0x3246, [0x0cbf] = 0x3247,
  [0x0cc0] = 0x3248, [0x0cc1] = 0x3249, [0x0cc2] = 0x324a, [0x0cc3] = 0x324b,
  [0x0cc4] = 0x324c, [0x0cc5] = 0x324d, [0x0cc6] = 0x324e, [0x0cc7] = 0x324f,
  [0x0cc8] = 0x3250, [0x0cc9] = 0x3251, [0x0cca] = 0x3252, [0x0ccb] = 0x3253,
  [0x0ccc] = 0x3254, [0x0ccd] = 0x3255, [0x0cce] = 0x3256, [0x0ccf] = 0x3257,
  [0x0cd0] = 0x3258, [0x0cd1] = 0x3259, [0x0cd2] = 0x325a, [0x0cd3] = 0x325b,
  [0x0cd4] = 0x325c, [0x0cd5] = 0x325d, [0x0cd6] = 0x325e, [0x0cd7] = 0x325f,
  [0x0cd8] = 0x3260, [0x0cd9] = 0x3261, [0x0cda] = 0x3262, [0x0cdb] = 0x3263,
  [0x0cdc] = 0x3264, [0x0cdd] = 0x3265, [0x0cde] = 0x3266, [0x0cdf] = 0x3267,
  [0x0ce0] = 0x3268, [0x0ce1] = 0x3269, [0x0ce2] = 0x326a, [0x0ce3] = 0x326b,
  [0x0ce4] = 0x326c, [0x0ce5] = 0x326d, [0x0ce6] = 0x326e, [0x0ce7] = 0x326f,
  [0x0ce8] = 0x3270, [0x0ce9] = 0x3271, [0x0cea] = 0x3272, [0x0ceb] = 0x3273,
  [0x0cec] = 0x3274, [0x0ced] = 0x3275, [0x0cee] = 0x3276, [0x0cef] = 0x3277,
  [0x0cf0] = 0x3278, [0x0cf1] = 0x3279, [0x0cf2] = 0x327a, [0x0cf3] = 0x327b,
  [0x0cf4] = 0x327c, [0x0cf5] = 0x327d, [0x0cf6] = 0x327e, [0x0cf7] = 0x327f,
  [0x0cf8] = 0x3280, [0x0cf9] = 0x3281, [0x0cfa] = 0x3282, [0x0cfb] = 0x3283,
  [0x0cfc] = 0x3284, [0x0cfd] = 0x3285, [0x0cfe] = 0x3286, [0x0cff] = 0x3287,
  [0x0d00] = 0x3288, [0x0d01] = 0x3289, [0x0d02] = 0x328a, [0x0d03] = 0x328b,
  [0x0d04] = 0x328c, [0x0d05] = 0x328d, [0x0d06] = 0x328e, [0x0d07] = 0x328f,
  [0x0d08] = 0x3290, [0x0d09] = 0x3291, [0x0d0a] = 0x3292, [0x0d0b] = 0x3293,
  [0x0d0c] = 0x3294, [0x0d0d] = 0x3295, [0x0d0e] = 0x3296, [0x0d0f] = 0x3297,
  [0x0d10] = 0x3298, [0x0d11] = 0x3299, [0x0d12] = 0x329a, [0x0d13] = 0x329b,
  [0x0d14] = 0x329c, [0x0d15] = 0x329d, [0x0d16] = 0x329e, [0x0d17] = 0x329f,
  [0x0d18] = 0x32a0, [0x0d19] = 0x32a1, [0x0d1a] = 0x32a2, [0x0d1b] = 0x32a4,
  [0x0d1c] = 0x32a5, [0x0d1d] = 0x32a6, [0x0d1e] = 0x32a7, [0x0d1f] = 0x32a8,
  [0x0d20] = 0x32a9, [0x0d21] = 0x32aa, [0x0d22] = 0x32ab, [0x0d23] = 0x32ac,
  [0x0d24] = 0x32ad, [0x0d25] = 0x32ae, [0x0d26] = 0x32af, [0x0d27] = 0x32b0,
  [0x0d28] = 0x32b1, [0x0d29] = 0x32b2, [0x0d2a] = 0x32b3, [0x0d2b] = 0x32b4,
  [0x0d2c] = 0x32b5, [0x0d2d] = 0x32b6, [0x0d2e] = 0x32b7, [0x0d2f] = 0x32b8,
  [0x0d30] = 0x32b9, [0x0d31] = 0x32ba, [0x0d32] = 0x32bb, [0x0d33] = 0x32bc,
  [0x0d34] = 0x32bd, [0x0d35] = 0x32be, [0x0d36] = 0x32bf, [0x0d37] = 0x32c0,
  [0x0d38] = 0x32c1, [0x0d39] = 0x32c2, [0x0d3a] = 0x32c3, [0x0d3b] = 0x32c4,
  [0x0d3c] = 0x32c5, [0x0d3d] = 0x32c6, [0x0d3e] = 0x32c7, [0x0d3f] = 0x32c8,
  [0x0d40] = 0x32c9, [0x0d41] = 0x32ca, [0x0d42] = 0x32cb, [0x0d43] = 0x32cc,
  [0x0d44] = 0x32cd, [0x0d45] = 0x32ce, [0x0d46] = 0x32cf, [0x0d47] = 0x32d0,
  [0x0d48] = 0x32d1, [0x0d49] = 0x32d2, [0x0d4a] = 0x32d3, [0x0d4b] = 0x32d4,
  [0x0d4c] = 0x32d5, [0x0d4d] = 0x32d6, [0x0d4e] = 0x32d7, [0x0d4f] = 0x32d8,
  [0x0d50] = 0x32d9, [0x0d51] = 0x32da, [0x0d52] = 0x32db, [0x0d53] = 0x32dc,
  [0x0d54] = 0x32dd, [0x0d55] = 0x32de, [0x0d56] = 0x32df, [0x0d57] = 0x32e0,
  [0x0d58] = 0x32e1, [0x0d59] = 0x32e2, [0x0d5a] = 0x32e3, [0x0d5b] = 0x32e4,
  [0x0d5c] = 0x32e5, [0x0d5d] = 0x32e6, [0x0d5e] = 0x32e7, [0x0d5f] = 0x32e8,
  [0x0d60] = 0x32e9, [0x0d61] = 0x32ea, [0x0d62] = 0x32eb, [0x0d63] = 0x32ec,
  [0x0d64] = 0x32ed, [0x0d65] = 0x32ee, [0x0d66] = 0x32ef, [0x0d67] = 0x32f0,
  [0x0d68] = 0x32f1, [0x0d69] = 0x32f2, [0x0d6a] = 0x32f3, [0x0d6b] = 0x32f4,
  [0x0d6c] = 0x32f5, [0x0d6d] = 0x32f6, [0x0d6e] = 0x32f7, [0x0d6f] = 0x32f8,
  [0x0d70] = 0x32f9, [0x0d71] = 0x32fa, [0x0d72] = 0x32fb, [0x0d73] = 0x32fc,
  [0x0d74] = 0x32fd, [0x0d75] = 0x32fe, [0x0d77] = 0x3300, [0x0d78] = 0x3301,
  [0x0d79] = 0x3302, [0x0d7a] = 0x3303, [0x0d7b] = 0x3304, [0x0d7c] = 0x3305,
  [0x0d7d] = 0x3306, [0x0d7e] = 0x3307, [0x0d7f] = 0x3308, [0x0d80] = 0x3309,
  [0x0d81] = 0x330a, [0x0d82] = 0x330b, [0x0d83] = 0x330c, [0x0d84] = 0x330d,
  [0x0d85] = 0x330e, [0x0d86] = 0x330f, [0x0d87] = 0x3310, [0x0d88] = 0x3311,
  [0x0d89] = 0x3312, [0x0d8a] = 0x3313, [0x0d8b] = 0x3314, [0x0d8c] = 0x3315,
  [0x0d8d] = 0x3316, [0x0d8e] = 0x3317, [0x0d8f] = 0x3318, [0x0d90] = 0x3319,
  [0x0d91] = 0x331a, [0x0d92] = 0x331b, [0x0d93] = 0x331c, [0x0d94] = 0x331d,
  [0x0d95] = 0x331e, [0x0d96] = 0x331f, [0x0d97] = 0x3320, [0x0d98] = 0x3321,
  [0x0d99] = 0x3322, [0x0d9a] = 0x3323, [0x0d9b] = 0x3324, [0x0d9c] = 0x3325,
  [0x0d9d] = 0x3326, [0x0d9e] = 0x3327, [0x0d9f] = 0x3328, [0x0da0] = 0x3329,
  [0x0da1] = 0x332a, [0x0da2] = 0x332b, [0x0da3] = 0x332c, [0x0da4] = 0x332d,
  [0x0da5] = 0x332e, [0x0da6] = 0x332f, [0x0da7] = 0x3330, [0x0da8] = 0x3331,
  [0x0da9] = 0x3332, [0x0daa] = 0x3333, [0x0dab] = 0x3334, [0x0dac] = 0x3335,
  [0x0dad] = 0x3336, [0x0dae] = 0x3337, [0x0daf] = 0x3338, [0x0db0] = 0x3339,
  [0x0db1] = 0x333a, [0x0db2] = 0x333b, [0x0db3] = 0x333c, [0x0db4] = 0x333d,
  [0x0db5] = 0x333e, [0x0db6] = 0x333f, [0x0db7] = 0x3340, [0x0db8] = 0x3341,
  [0x0db9] = 0x3342, [0x0dba] = 0x3343, [0x0dbb] = 0x3344, [0x0dbc] = 0x3345,
  [0x0dbd] = 0x3346, [0x0dbe] = 0x3347, [0x0dbf] = 0x3348, [0x0dc0] = 0x3349,
  [0x0dc1] = 0x334a, [0x0dc2] = 0x334b, [0x0dc3] = 0x334c, [0x0dc4] = 0x334d,
  [0x0dc5] = 0x334e, [0x0dc6] = 0x334f, [0x0dc7] = 0x3350, [0x0dc8] = 0x3351,
  [0x0dc9] = 0x3352, [0x0dca] = 0x3353, [0x0dcb] = 0x3354, [0x0dcc] = 0x3355,
  [0x0dcd] = 0x3356, [0x0dce] = 0x3357, [0x0dcf] = 0x3358, [0x0dd0] = 0x3359,
  [0x0dd1] = 0x335a, [0x0dd2] = 0x335b, [0x0dd3] = 0x335c, [0x0dd4] = 0x335d,
  [0x0dd5] = 0x335e, [0x0dd6] = 0x335f, [0x0dd7] = 0x3360, [0x0dd8] = 0x3361,
  [0x0dd9] = 0x3362, [0x0dda] = 0x3363, [0x0ddb] = 0x3364, [0x0ddc] = 0x3365,
  [0x0ddd] = 0x3366, [0x0dde] = 0x3367, [0x0ddf] = 0x3368, [0x0de0] = 0x3369,
  [0x0de1] = 0x336a, [0x0de2] = 0x336b, [0x0de3] = 0x336c, [0x0de4] = 0x336d,
  [0x0de5] = 0x336e, [0x0de6] = 0x336f, [0x0de7] = 0x3370, [0x0de8] = 0x3371,
  [0x0de9] = 0x3372, [0x0dea] = 0x3373, [0x0deb] = 0x3374, [0x0dec] = 0x3375,
  [0x0ded] = 0x3376, [0x0dee] = 0x3377, [0x0def] = 0x3378, [0x0df0] = 0x3379,
  [0x0df1] = 0x337a, [0x0df2] = 0x337b, [0x0df3] = 0x337c, [0x0df4] = 0x337d,
  [0x0df5] = 0x337e, [0x0df6] = 0x337f, [0x0df7] = 0x3380, [0x0df8] = 0x3381,
  [0x0df9] = 0x3382, [0x0dfa] = 0x3383, [0x0dfb] = 0x3384, [0x0dfc] = 0x3385,
  [0x0dfd] = 0x3386, [0x0dfe] = 0x3387, [0x0dff] = 0x3388, [0x0e00] = 0x3389,
  [0x0e01] = 0x338a, [0x0e02] = 0x338b, [0x0e03] = 0x338c, [0x0e04] = 0x338d,
  [0x0e05] = 0x3390, [0x0e06] = 0x3391, [0x0e07] = 0x3392, [0x0e08] = 0x3393,
  [0x0e09] = 0x3394, [0x0e0a] = 0x3395, [0x0e0b] = 0x3396, [0x0e0c] = 0x3397,
  [0x0e0d] = 0x3398, [0x0e0e] = 0x3399, [0x0e0f] = 0x339a, [0x0e10] = 0x339b,
  [0x0e11] = 0x339f, [0x0e12] = 0x33a0, [0x0e13] = 0x33a2, [0x0e14] = 0x33a3,
  [0x0e15] = 0x33a4, [0x0e16] = 0x33a5, [0x0e17] = 0x33a6, [0x0e18] = 0x33a7,
  [0x0e19] = 0x33a8, [0x0e1a] = 0x33a9, [0x0e1b] = 0x33aa, [0x0e1c] = 0x33ab,
  [0x0e1d] = 0x33ac, [0x0e1e] = 0x33ad, [0x0e1f] = 0x33ae, [0x0e20] = 0x33af,
  [0x0e21] = 0x33b0, [0x0e22] = 0x33b1, [0x0e23] = 0x33b2, [0x0e24] = 0x33b3,
  [0x0e25] = 0x33b4, [0x0e26] = 0x33b5, [0x0e27] = 0x33b6, [0x0e28] = 0x33b7,
  [0x0e29] = 0x33b8, [0x0e2a] = 0x33b9, [0x0e2b] = 0x33ba, [0x0e2c] = 0x33bb,
  [0x0e2d] = 0x33bc, [0x0e2e] = 0x33bd, [0x0e2f] = 0x33be, [0x0e30] = 0x33bf,
  [0x0e31] = 0x33c0, [0x0e32] = 0x33c1, [0x0e33] = 0x33c2, [0x0e34] = 0x33c3,
  [0x0e35] = 0x33c5, [0x0e36] = 0x33c6, [0x0e37] = 0x33c7, [0x0e38] = 0x33c8,
  [0x0e39] = 0x33c9, [0x0e3a] = 0x33ca, [0x0e3b] = 0x33cb, [0x0e3c] = 0x33cc,
  [0x0e3d] = 0x33cd, [0x0e3e] = 0x33cf, [0x0e3f] = 0x33d0, [0x0e40] = 0x33d3,
  [0x0e41] = 0x33d4, [0x0e42] = 0x33d6, [0x0e43] = 0x33d7, [0x0e44] = 0x33d8,
  [0x0e45] = 0x33d9, [0x0e46] = 0x33da, [0x0e47] = 0x33db, [0x0e48] = 0x33dc,
  [0x0e49] = 0x33dd, [0x0e4a] = 0x33de, [0x0e4b] = 0x33df, [0x0e4c] = 0x33e0,
  [0x0e4d] = 0x33e1, [0x0e4e] = 0x33e2, [0x0e4f] = 0x33e3, [0x0e50] = 0x33e4,
  [0x0e51] = 0x33e5, [0x0e52] = 0x33e6, [0x0e53] = 0x33e7, [0x0e54] = 0x33e8,
  [0x0e55] = 0x33e9, [0x0e56] = 0x33ea, [0x0e57] = 0x33eb, [0x0e58] = 0x33ec,
  [0x0e59] = 0x33ed, [0x0e5a] = 0x33ee, [0x0e5b] = 0x33ef, [0x0e5c] = 0x33f0,
  [0x0e5d] = 0x33f1, [0x0e5e] = 0x33f2, [0x0e5f] = 0x33f3, [0x0e60] = 0x33f4,
  [0x0e61] = 0x33f5, [0x0e62] = 0x33f6, [0x0e63] = 0x33f7, [0x0e64] = 0x33f8,
  [0x0e65] = 0x33f9, [0x0e66] = 0x33fa, [0x0e67] = 0x33fb, [0x0e68] = 0x33fc,
  [0x0e69] = 0x33fd, [0x0e6a] = 0x33fe, [0x0e6b] = 0x33ff, [0x0e6c] = 0x3400,
  [0x0e6d] = 0x3401, [0x0e6e] = 0x3402, [0x0e6f] = 0x3403, [0x0e70] = 0x3404,
  [0x0e71] = 0x3405, [0x0e72] = 0x3406, [0x0e73] = 0x3407, [0x0e74] = 0x3408,
  [0x0e75] = 0x3409, [0x0e76] = 0x340a, [0x0e77] = 0x340b, [0x0e78] = 0x340c,
  [0x0e79] = 0x340d, [0x0e7a] = 0x340e, [0x0e7b] = 0x340f, [0x0e7c] = 0x3410,
  [0x0e7d] = 0x3411, [0x0e7e] = 0x3412, [0x0e7f] = 0x3413, [0x0e80] = 0x3414,
  [0x0e81] = 0x3415, [0x0e82] = 0x3416, [0x0e83] = 0x3417, [0x0e84] = 0x3418,
  [0x0e85] = 0x3419, [0x0e86] = 0x341a, [0x0e87] = 0x341b, [0x0e88] = 0x341c,
  [0x0e89] = 0x341d, [0x0e8a] = 0x341e, [0x0e8b] = 0x341f, [0x0e8c] = 0x3420,
  [0x0e8d] = 0x3421, [0x0e8e] = 0x3422, [0x0e8f] = 0x3423, [0x0e90] = 0x3424,
  [0x0e91] = 0x3425, [0x0e92] = 0x3426, [0x0e93] = 0x3427, [0x0e94] = 0x3428,
  [0x0e95] = 0x3429, [0x0e96] = 0x342a, [0x0e97] = 0x342b, [0x0e98] = 0x342c,
  [0x0e99] = 0x342d, [0x0e9a] = 0x342e, [0x0e9b] = 0x342f, [0x0e9c] = 0x3430,
  [0x0e9d] = 0x3431, [0x0e9e] = 0x3432, [0x0e9f] = 0x3433, [0x0ea0] = 0x3434,
  [0x0ea1] = 0x3435, [0x0ea2] = 0x3436, [0x0ea3] = 0x3437, [0x0ea4] = 0x3438,
  [0x0ea5] = 0x3439, [0x0ea6] = 0x343a, [0x0ea7] = 0x343b, [0x0ea8] = 0x343c,
  [0x0ea9] = 0x343d, [0x0eaa] = 0x343e, [0x0eab] = 0x343f, [0x0eac] = 0x3440,
  [0x0ead] = 0x3441, [0x0eae] = 0x3442, [0x0eaf] = 0x3443, [0x0eb0] = 0x3444,
  [0x0eb1] = 0x3445, [0x0eb2] = 0x3446, [0x0eb3] = 0x3448, [0x0eb4] = 0x3449,
  [0x0eb5] = 0x344a, [0x0eb6] = 0x344b, [0x0eb7] = 0x344c, [0x0eb8] = 0x344d,
  [0x0eb9] = 0x344e, [0x0eba] = 0x344f, [0x0ebb] = 0x3450, [0x0ebc] = 0x3451,
  [0x0ebd] = 0x3452, [0x0ebe] = 0x3453, [0x0ebf] = 0x3454, [0x0ec0] = 0x3455,
  [0x0ec1] = 0x3456, [0x0ec2] = 0x3457, [0x0ec3] = 0x3458, [0x0ec4] = 0x3459,
  [0x0ec5] = 0x345a, [0x0ec6] = 0x345b, [0x0ec7] = 0x345c, [0x0ec8] = 0x345d,
  [0x0ec9] = 0x345e, [0x0eca] = 0x345f, [0x0ecb] = 0x3460, [0x0ecc] = 0x3461,
  [0x0ecd] = 0x3462, [0x0ece] = 0x3463, [0x0ecf] = 0x3464, [0x0ed0] = 0x3465,
  [0x0ed1] = 0x3466, [0x0ed2] = 0x3467, [0x0ed3] = 0x3468, [0x0ed4] = 0x3469,
  [0x0ed5] = 0x346a, [0x0ed6] = 0x346b, [0x0ed7] = 0x346c, [0x0ed8] = 0x346d,
  [0x0ed9] = 0x346e, [0x0eda] = 0x346f, [0x0edb] = 0x3470, [0x0edc] = 0x3471,
  [0x0edd] = 0x3472, [0x0ede] = 0x3474, [0x0edf] = 0x3475, [0x0ee0] = 0x3476,
  [0x0ee1] = 0x3477, [0x0ee2] = 0x3478, [0x0ee3] = 0x3479, [0x0ee4] = 0x347a,
  [0x0ee5] = 0x347b, [0x0ee6] = 0x347c, [0x0ee7] = 0x347d, [0x0ee8] = 0x347e,
  [0x0ee9] = 0x347f, [0x0eea] = 0x3480, [0x0eeb] = 0x3481, [0x0eec] = 0x3482,
  [0x0eed] = 0x3483, [0x0eee] = 0x3484, [0x0eef] = 0x3485, [0x0ef0] = 0x3486,
  [0x0ef1] = 0x3487, [0x0ef2] = 0x3488, [0x0ef3] = 0x3489, [0x0ef4] = 0x348a,
  [0x0ef5] = 0x348b, [0x0ef6] = 0x348c, [0x0ef7] = 0x348d, [0x0ef8] = 0x348e,
  [0x0ef9] = 0x348f, [0x0efa] = 0x3490, [0x0efb] = 0x3491, [0x0efc] = 0x3492,
  [0x0efd] = 0x3493, [0x0efe] = 0x3494, [0x0eff] = 0x3495, [0x0f00] = 0x3496,
  [0x0f01] = 0x3497, [0x0f02] = 0x3498, [0x0f03] = 0x3499, [0x0f04] = 0x349a,
  [0x0f05] = 0x349b, [0x0f06] = 0x349c, [0x0f07] = 0x349d, [0x0f08] = 0x349e,
  [0x0f09] = 0x349f, [0x0f0a] = 0x34a0, [0x0f0b] = 0x34a1, [0x0f0c] = 0x34a2,
  [0x0f0d] = 0x34a3, [0x0f0e] = 0x34a4, [0x0f0f] = 0x34a5, [0x0f10] = 0x34a6,
  [0x0f11] = 0x34a7, [0x0f12] = 0x34a8, [0x0f13] = 0x34a9, [0x0f14] = 0x34aa,
  [0x0f15] = 0x34ab, [0x0f16] = 0x34ac, [0x0f17] = 0x34ad, [0x0f18] = 0x34ae,
  [0x0f19] = 0x34af, [0x0f1a] = 0x34b0, [0x0f1b] = 0x34b1, [0x0f1c] = 0x34b2,
  [0x0f1d] = 0x34b3, [0x0f1e] = 0x34b4, [0x0f1f] = 0x34b5, [0x0f20] = 0x34b6,
  [0x0f21] = 0x34b7, [0x0f22] = 0x34b8, [0x0f23] = 0x34b9, [0x0f24] = 0x34ba,
  [0x0f25] = 0x34bb, [0x0f26] = 0x34bc, [0x0f27] = 0x34bd, [0x0f28] = 0x34be,
  [0x0f29] = 0x34bf, [0x0f2a] = 0x34c0, [0x0f2b] = 0x34c1, [0x0f2c] = 0x34c2,
  [0x0f2d] = 0x34c3, [0x0f2e] = 0x34c4, [0x0f2f] = 0x34c5, [0x0f30] = 0x34c6,
  [0x0f31] = 0x34c7, [0x0f32] = 0x34c8, [0x0f33] = 0x34c9, [0x0f34] = 0x34ca,
  [0x0f35] = 0x34cb, [0x0f36] = 0x34cc, [0x0f37] = 0x34cd, [0x0f38] = 0x34ce,
  [0x0f39] = 0x34cf, [0x0f3a] = 0x34d0, [0x0f3b] = 0x34d1, [0x0f3c] = 0x34d2,
  [0x0f3d] = 0x34d3, [0x0f3e] = 0x34d4, [0x0f3f] = 0x34d5, [0x0f40] = 0x34d6,
  [0x0f41] = 0x34d7, [0x0f42] = 0x34d8, [0x0f43] = 0x34d9, [0x0f44] = 0x34da,
  [0x0f45] = 0x34db, [0x0f46] = 0x34dc, [0x0f47] = 0x34dd, [0x0f48] = 0x34de,
  [0x0f49] = 0x34df, [0x0f4a] = 0x34e0, [0x0f4b] = 0x34e1, [0x0f4c] = 0x34e2,
  [0x0f4d] = 0x34e3, [0x0f4e] = 0x34e4, [0x0f4f] = 0x34e5, [0x0f50] = 0x34e6,
  [0x0f51] = 0x34e7, [0x0f52] = 0x34e8, [0x0f53] = 0x34e9, [0x0f54] = 0x34ea,
  [0x0f55] = 0x34eb, [0x0f56] = 0x34ec, [0x0f57] = 0x34ed, [0x0f58] = 0x34ee,
  [0x0f59] = 0x34ef, [0x0f5a] = 0x34f0, [0x0f5b] = 0x34f1, [0x0f5c] = 0x34f2,
  [0x0f5d] = 0x34f3, [0x0f5e] = 0x34f4, [0x0f5f] = 0x34f5, [0x0f60] = 0x34f6,
  [0x0f61] = 0x34f7, [0x0f62] = 0x34f8, [0x0f63] = 0x34f9, [0x0f64] = 0x34fa,
  [0x0f65] = 0x34fb, [0x0f66] = 0x34fc, [0x0f67] = 0x34fd, [0x0f68] = 0x34fe,
  [0x0f69] = 0x34ff, [0x0f6a] = 0x3500, [0x0f6b] = 0x3501, [0x0f6c] = 0x3502,
  [0x0f6d] = 0x3503, [0x0f6e] = 0x3504, [0x0f6f] = 0x3505, [0x0f70] = 0x3506,
  [0x0f71] = 0x3507, [0x0f72] = 0x3508, [0x0f73] = 0x3509, [0x0f74] = 0x350a,
  [0x0f75] = 0x350b, [0x0f76] = 0x350c, [0x0f77] = 0x350d, [0x0f78] = 0x350e,
  [0x0f79] = 0x350f, [0x0f7a] = 0x3510, [0x0f7b] = 0x3511, [0x0f7c] = 0x3512,
  [0x0f7d] = 0x3513, [0x0f7e] = 0x3514, [0x0f7f] = 0x3515, [0x0f80] = 0x3516,
  [0x0f81] = 0x3517, [0x0f82] = 0x3518, [0x0f83] = 0x3519, [0x0f84] = 0x351a,
  [0x0f85] = 0x351b, [0x0f86] = 0x351c, [0x0f87] = 0x351d, [0x0f88] = 0x351e,
  [0x0f89] = 0x351f, [0x0f8a] = 0x3520, [0x0f8b] = 0x3521, [0x0f8c] = 0x3522,
  [0x0f8d] = 0x3523, [0x0f8e] = 0x3524, [0x0f8f] = 0x3525, [0x0f90] = 0x3526,
  [0x0f91] = 0x3527, [0x0f92] = 0x3528, [0x0f93] = 0x3529, [0x0f94] = 0x352a,
  [0x0f95] = 0x352b, [0x0f96] = 0x352c, [0x0f97] = 0x352d, [0x0f98] = 0x352e,
  [0x0f99] = 0x352f, [0x0f9a] = 0x3530, [0x0f9b] = 0x3531, [0x0f9c] = 0x3532,
  [0x0f9d] = 0x3533, [0x0f9e] = 0x3534, [0x0f9f] = 0x3535, [0x0fa0] = 0x3536,
  [0x0fa1] = 0x3537, [0x0fa2] = 0x3538, [0x0fa3] = 0x3539, [0x0fa4] = 0x353a,
  [0x0fa5] = 0x353b, [0x0fa6] = 0x353c, [0x0fa7] = 0x353d, [0x0fa8] = 0x353e,
  [0x0fa9] = 0x353f, [0x0faa] = 0x3540, [0x0fab] = 0x3541, [0x0fac] = 0x3542,
  [0x0fad] = 0x3543, [0x0fae] = 0x3544, [0x0faf] = 0x3545, [0x0fb0] = 0x3546,
  [0x0fb1] = 0x3547, [0x0fb2] = 0x3548, [0x0fb3] = 0x3549, [0x0fb4] = 0x354a,
  [0x0fb5] = 0x354b, [0x0fb6] = 0x354c, [0x0fb7] = 0x354d, [0x0fb8] = 0x354e,
  [0x0fb9] = 0x354f, [0x0fba] = 0x3550, [0x0fbb] = 0x3551, [0x0fbc] = 0x3552,
  [0x0fbd] = 0x3553, [0x0fbe] = 0x3554, [0x0fbf] = 0x3555, [0x0fc0] = 0x3556,
  [0x0fc1] = 0x3557, [0x0fc2] = 0x3558, [0x0fc3] = 0x3559, [0x0fc4] = 0x355a,
  [0x0fc5] = 0x355b, [0x0fc6] = 0x355c, [0x0fc7] = 0x355d, [0x0fc8] = 0x355e,
  [0x0fc9] = 0x355f, [0x0fca] = 0x3560, [0x0fcb] = 0x3561, [0x0fcc] = 0x3562,
  [0x0fcd] = 0x3563, [0x0fce] = 0x3564, [0x0fcf] = 0x3565, [0x0fd0] = 0x3566,
  [0x0fd1] = 0x3567, [0x0fd2] = 0x3568, [0x0fd3] = 0x3569, [0x0fd4] = 0x356a,
  [0x0fd5] = 0x356b, [0x0fd6] = 0x356c, [0x0fd7] = 0x356d, [0x0fd8] = 0x356e,
  [0x0fd9] = 0x356f, [0x0fda] = 0x3570, [0x0fdb] = 0x3571, [0x0fdc] = 0x3572,
  [0x0fdd] = 0x3573, [0x0fde] = 0x3574, [0x0fdf] = 0x3575, [0x0fe0] = 0x3576,
  [0x0fe1] = 0x3577, [0x0fe2] = 0x3578, [0x0fe3] = 0x3579, [0x0fe4] = 0x357a,
  [0x0fe5] = 0x357b, [0x0fe6] = 0x357c, [0x0fe7] = 0x357d, [0x0fe8] = 0x357e,
  [0x0fe9] = 0x357f, [0x0fea] = 0x3580, [0x0feb] = 0x3581, [0x0fec] = 0x3582,
  [0x0fed] = 0x3583, [0x0fee] = 0x3584, [0x0fef] = 0x3585, [0x0ff0] = 0x3586,
  [0x0ff1] = 0x3587, [0x0ff2] = 0x3588, [0x0ff3] = 0x3589, [0x0ff4] = 0x358a,
  [0x0ff5] = 0x358b, [0x0ff6] = 0x358c, [0x0ff7] = 0x358d, [0x0ff8] = 0x358e,
  [0x0ff9] = 0x358f, [0x0ffa] = 0x3590, [0x0ffb] = 0x3591, [0x0ffc] = 0x3592,
  [0x0ffd] = 0x3593, [0x0ffe] = 0x3594, [0x0fff] = 0x3595, [0x1000] = 0x3596,
  [0x1001] = 0x3597, [0x1002] = 0x3598, [0x1003] = 0x3599, [0x1004] = 0x359a,
  [0x1005] = 0x359b, [0x1006] = 0x359c, [0x1007] = 0x359d, [0x1008] = 0x359f,
  [0x1009] = 0x35a0, [0x100a] = 0x35a1, [0x100b] = 0x35a2, [0x100c] = 0x35a3,
  [0x100d] = 0x35a4, [0x100e] = 0x35a5, [0x100f] = 0x35a6, [0x1010] = 0x35a7,
  [0x1011] = 0x35a8, [0x1012] = 0x35a9, [0x1013] = 0x35aa, [0x1014] = 0x35ab,
  [0x1015] = 0x35ac, [0x1016] = 0x35ad, [0x1017] = 0x35ae, [0x1018] = 0x35af,
  [0x1019] = 0x35b0, [0x101a] = 0x35b1, [0x101b] = 0x35b2, [0x101c] = 0x35b3,
  [0x101d] = 0x35b4, [0x101e] = 0x35b5, [0x101f] = 0x35b6, [0x1020] = 0x35b7,
  [0x1021] = 0x35b8, [0x1022] = 0x35b9, [0x1023] = 0x35ba, [0x1024] = 0x35bb,
  [0x1025] = 0x35bc, [0x1026] = 0x35bd, [0x1027] = 0x35be, [0x1028] = 0x35bf,
  [0x1029] = 0x35c0, [0x102a] = 0x35c1, [0x102b] = 0x35c2, [0x102c] = 0x35c3,
  [0x102d] = 0x35c4, [0x102e] = 0x35c5, [0x102f] = 0x35c6, [0x1030] = 0x35c7,
  [0x1031] = 0x35c8, [0x1032] = 0x35c9, [0x1033] = 0x35ca, [0x1034] = 0x35cb,
  [0x1035] = 0x35cc, [0x1036] = 0x35cd, [0x1037] = 0x35ce, [0x1038] = 0x35cf,
  [0x1039] = 0x35d0, [0x103a] = 0x35d1, [0x103b] = 0x35d2, [0x103c] = 0x35d3,
  [0x103d] = 0x35d4, [0x103e] = 0x35d5, [0x103f] = 0x35d6, [0x1040] = 0x35d7,
  [0x1041] = 0x35d8, [0x1042] = 0x35d9, [0x1043] = 0x35da, [0x1044] = 0x35db,
  [0x1045] = 0x35dc, [0x1046] = 0x35dd, [0x1047] = 0x35de, [0x1048] = 0x35df,
  [0x1049] = 0x35e0, [0x104a] = 0x35e1, [0x104b] = 0x35e2, [0x104c] = 0x35e3,
  [0x104d] = 0x35e4, [0x104e] = 0x35e5, [0x104f] = 0x35e6, [0x1050] = 0x35e7,
  [0x1051] = 0x35e8, [0x1052] = 0x35e9, [0x1053] = 0x35ea, [0x1054] = 0x35eb,
  [0x1055] = 0x35ec, [0x1056] = 0x35ed, [0x1057] = 0x35ee, [0x1058] = 0x35ef,
  [0x1059] = 0x35f0, [0x105a] = 0x35f1, [0x105b] = 0x35f2, [0x105c] = 0x35f3,
  [0x105d] = 0x35f4, [0x105e] = 0x35f5, [0x105f] = 0x35f6, [0x1060] = 0x35f7,
  [0x1061] = 0x35f8, [0x1062] = 0x35f9, [0x1063] = 0x35fa, [0x1064] = 0x35fb,
  [0x1065] = 0x35fc, [0x1066] = 0x35fd, [0x1067] = 0x35fe, [0x1068] = 0x35ff,
  [0x1069] = 0x3600, [0x106a] = 0x3601, [0x106b] = 0x3602, [0x106c] = 0x3603,
  [0x106d] = 0x3604, [0x106e] = 0x3605, [0x106f] = 0x3606, [0x1070] = 0x3607,
  [0x1071] = 0x3608, [0x1072] = 0x3609, [0x1073] = 0x360a, [0x1074] = 0x360b,
  [0x1075] = 0x360c, [0x1076] = 0x360d, [0x1077] = 0x360f, [0x1078] = 0x3610,
  [0x1079] = 0x3611, [0x107a] = 0x3612, [0x107b] = 0x3613, [0x107c] = 0x3614,
  [0x107d] = 0x3615, [0x107e] = 0x3616, [0x107f] = 0x3617, [0x1080] = 0x3618,
  [0x1081] = 0x3619, [0x137f] = 0x3919, [0x1380] = 0x391a, [0x1381] = 0x391b,
  [0x1382] = 0x391c, [0x1383] = 0x391d, [0x1384] = 0x391e, [0x1385] = 0x391f,
  [0x1386] = 0x3920, [0x1387] = 0x3921, [0x1388] = 0x3922, [0x1389] = 0x3923,
  [0x138a] = 0x3924, [0x138b] = 0x3925, [0x138c] = 0x3926, [0x138d] = 0x3927,
  [0x138e] = 0x3928, [0x138f] = 0x3929, [0x1390] = 0x392a, [0x1391] = 0x392b,
  [0x1392] = 0x392c, [0x1393] = 0x392d, [0x1394] = 0x392e, [0x1395] = 0x392f,
  [0x1396] = 0x3930, [0x1397] = 0x3931, [0x1398] = 0x3932, [0x1399] = 0x3933,
  [0x139a] = 0x3934, [0x139b] = 0x3935, [0x139c] = 0x3936, [0x139d] = 0x3937,
  [0x139e] = 0x3938, [0x139f] = 0x3939, [0x13a0] = 0x393a, [0x13a1] = 0x393b,
  [0x13a2] = 0x393c, [0x13a3] = 0x393d, [0x13a4] = 0x393e, [0x13a5] = 0x393f,
  [0x13a6] = 0x3940, [0x13a7] = 0x3941, [0x13a8] = 0x3942, [0x13a9] = 0x3943,
  [0x13aa] = 0x3944, [0x13ab] = 0x3945, [0x13ac] = 0x3946, [0x13ad] = 0x3947,
  [0x13ae] = 0x3948, [0x13af] = 0x3949, [0x13b0] = 0x394a, [0x13b1] = 0x394b,
  [0x13b2] = 0x394c, [0x13b3] = 0x394d, [0x13b4] = 0x394e, [0x13b5] = 0x394f,
  [0x13b6] = 0x3950, [0x13b7] = 0x3951, [0x13b8] = 0x3952, [0x13b9] = 0x3953,
  [0x13ba] = 0x3954, [0x13bb] = 0x3955, [0x13bc] = 0x3956, [0x13bd] = 0x3957,
  [0x13be] = 0x3958, [0x13bf] = 0x3959, [0x13c0] = 0x395a, [0x13c1] = 0x395b,
  [0x13c2] = 0x395c, [0x13c3] = 0x395d, [0x13c4] = 0x395e, [0x13c5] = 0x395f,
  [0x13c6] = 0x3960, [0x13c7] = 0x3961, [0x13c8] = 0x3962, [0x13c9] = 0x3963,
  [0x13ca] = 0x3964, [0x13cb] = 0x3965, [0x13cc] = 0x3966, [0x13cd] = 0x3967,
  [0x13ce] = 0x3968, [0x13cf] = 0x3969, [0x13d0] = 0x396a, [0x13d1] = 0x396b,
  [0x13d2] = 0x396c, [0x13d3] = 0x396d, [0x13d4] = 0x396f, [0x13d5] = 0x3970,
  [0x13d6] = 0x3971, [0x13d7] = 0x3972, [0x13d8] = 0x3973, [0x13d9] = 0x3974,
  [0x13da] = 0x3975, [0x13db] = 0x3976, [0x13dc] = 0x3977, [0x13dd] = 0x3978,
  [0x13de] = 0x3979, [0x13df] = 0x397a, [0x13e0] = 0x397b, [0x13e1] = 0x397c,
  [0x13e2] = 0x397d, [0x13e3] = 0x397e, [0x13e4] = 0x397f, [0x13e5] = 0x3980,
  [0x13e6] = 0x3981, [0x13e7] = 0x3982, [0x13e8] = 0x3983, [0x13e9] = 0x3984,
  [0x13ea] = 0x3985, [0x13eb] = 0x3986, [0x13ec] = 0x3987, [0x13ed] = 0x3988,
  [0x13ee] = 0x3989, [0x13ef] = 0x398a, [0x13f0] = 0x398b, [0x13f1] = 0x398c,
  [0x13f2] = 0x398d, [0x13f3] = 0x398e, [0x13f4] = 0x398f, [0x13f5] = 0x3990,
  [0x13f6] = 0x3991, [0x13f7] = 0x3992, [0x13f8] = 0x3993, [0x13f9] = 0x3994,
  [0x13fa] = 0x3995, [0x13fb] = 0x3996, [0x13fc] = 0x3997, [0x13fd] = 0x3998,
  [0x13fe] = 0x3999, [0x13ff] = 0x399a, [0x1400] = 0x399b, [0x1401] = 0x399c,
  [0x1402] = 0x399d, [0x1403] = 0x399e, [0x1404] = 0x399f, [0x1405] = 0x39a0,
  [0x1406] = 0x39a1, [0x1407] = 0x39a2, [0x1408] = 0x39a3, [0x1409] = 0x39a4,
  [0x140a] = 0x39a5, [0x140b] = 0x39a6, [0x140c] = 0x39a7, [0x140d] = 0x39a8,
  [0x140e] = 0x39a9, [0x140f] = 0x39aa, [0x1410] = 0x39ab, [0x1411] = 0x39ac,
  [0x1412] = 0x39ad, [0x1413] = 0x39ae, [0x1414] = 0x39af, [0x1415] = 0x39b0,
  [0x1416] = 0x39b1, [0x1417] = 0x39b2, [0x1418] = 0x39b3, [0x1419] = 0x39b4,
  [0x141a] = 0x39b5, [0x141b] = 0x39b6, [0x141c] = 0x39b7, [0x141d] = 0x39b8,
  [0x141e] = 0x39b9, [0x141f] = 0x39ba, [0x1420] = 0x39bb, [0x1421] = 0x39bc,
  [0x1422] = 0x39bd, [0x1423] = 0x39be, [0x1424] = 0x39bf, [0x1425] = 0x39c0,
  [0x1426] = 0x39c1, [0x1427] = 0x39c2, [0x1428] = 0x39c3, [0x1429] = 0x39c4,
  [0x142a] = 0x39c5, [0x142b] = 0x39c6, [0x142c] = 0x39c7, [0x142d] = 0x39c8,
  [0x142e] = 0x39c9, [0x142f] = 0x39ca, [0x1430] = 0x39cb, [0x1431] = 0x39cc,
  [0x1432] = 0x39cd, [0x1433] = 0x39ce, [0x1434] = 0x39d1, [0x1435] = 0x39d2,
  [0x1436] = 0x39d3, [0x1437] = 0x39d4, [0x1438] = 0x39d5, [0x1439] = 0x39d6,
  [0x143a] = 0x39d7, [0x143b] = 0x39d8, [0x143c] = 0x39d9, [0x143d] = 0x39da,
  [0x143e] = 0x39db, [0x143f] = 0x39dc, [0x1440] = 0x39dd, [0x1441] = 0x39de,
  [0x1442] = 0x39e0, [0x1443] = 0x39e1, [0x1444] = 0x39e2, [0x1445] = 0x39e3,
  [0x1446] = 0x39e4, [0x1447] = 0x39e5, [0x1448] = 0x39e6, [0x1449] = 0x39e7,
  [0x144a] = 0x39e8, [0x144b] = 0x39e9, [0x144c] = 0x39ea, [0x144d] = 0x39eb,
  [0x144e] = 0x39ec, [0x144f] = 0x39ed, [0x1450] = 0x39ee, [0x1451] = 0x39ef,
  [0x1452] = 0x39f0, [0x1453] = 0x39f1, [0x1454] = 0x39f2, [0x1455] = 0x39f3,
  [0x1456] = 0x39f4, [0x1457] = 0x39f5, [0x1458] = 0x39f6, [0x1459] = 0x39f7,
  [0x145a] = 0x39f8, [0x145b] = 0x39f9, [0x145c] = 0x39fa, [0x145d] = 0x39fb,
  [0x145e] = 0x39fc, [0x145f] = 0x39fd, [0x1460] = 0x39fe, [0x1461] = 0x39ff,
  [0x1462] = 0x3a00, [0x1463] = 0x3a01, [0x1464] = 0x3a02, [0x1465] = 0x3a03,
  [0x1466] = 0x3a04, [0x1467] = 0x3a05, [0x1468] = 0x3a06, [0x1469] = 0x3a07,
  [0x146a] = 0x3a08, [0x146b] = 0x3a09, [0x146c] = 0x3a0a, [0x146d] = 0x3a0b,
  [0x146e] = 0x3a0c, [0x146f] = 0x3a0d, [0x1470] = 0x3a0e, [0x1471] = 0x3a0f,
  [0x1472] = 0x3a10, [0x1473] = 0x3a11, [0x1474] = 0x3a12, [0x1475] = 0x3a13,
  [0x1476] = 0x3a14, [0x1477] = 0x3a15, [0x1478] = 0x3a16, [0x1479] = 0x3a17,
  [0x147a] = 0x3a18, [0x147b] = 0x3a19, [0x147c] = 0x3a1a, [0x147d] = 0x3a1b,
  [0x147e] = 0x3a1c, [0x147f] = 0x3a1d, [0x1480] = 0x3a1e, [0x1481] = 0x3a1f,
  [0x1482] = 0x3a20, [0x1483] = 0x3a21, [0x1484] = 0x3a22, [0x1485] = 0x3a23,
  [0x1486] = 0x3a24, [0x1487] = 0x3a25, [0x1488] = 0x3a26, [0x1489] = 0x3a27,
  [0x148a] = 0x3a28, [0x148b] = 0x3a29, [0x148c] = 0x3a2a, [0x148d] = 0x3a2b,
  [0x148e] = 0x3a2c, [0x148f] = 0x3a2d, [0x1490] = 0x3a2e, [0x1491] = 0x3a2f,
  [0x1492] = 0x3a30, [0x1493] = 0x3a31, [0x1494] = 0x3a32, [0x1495] = 0x3a33,
  [0x1496] = 0x3a34, [0x1497] = 0x3a35, [0x1498] = 0x3a36, [0x1499] = 0x3a37,
  [0x149a] = 0x3a38, [0x149b] = 0x3a39, [0x149c] = 0x3a3a, [0x149d] = 0x3a3b,
  [0x149e] = 0x3a3c, [0x149f] = 0x3a3d, [0x14a0] = 0x3a3e, [0x14a1] = 0x3a3f,
  [0x14a2] = 0x3a40, [0x14a3] = 0x3a41, [0x14a4] = 0x3a42, [0x14a5] = 0x3a43,
  [0x14a6] = 0x3a44, [0x14a7] = 0x3a45, [0x14a8] = 0x3a46, [0x14a9] = 0x3a47,
  [0x14aa] = 0x3a48, [0x14ab] = 0x3a49, [0x14ac] = 0x3a4a, [0x14ad] = 0x3a4b,
  [0x14ae] = 0x3a4c, [0x14af] = 0x3a4d, [0x14b0] = 0x3a4e, [0x14b1] = 0x3a4f,
  [0x14b2] = 0x3a50, [0x14b3] = 0x3a51, [0x14b4] = 0x3a52, [0x14b5] = 0x3a53,
  [0x14b6] = 0x3a54, [0x14b7] = 0x3a55, [0x14b8] = 0x3a56, [0x14b9] = 0x3a57,
  [0x14ba] = 0x3a58, [0x14bb] = 0x3a59, [0x14bc] = 0x3a5a, [0x14bd] = 0x3a5b,
  [0x14be] = 0x3a5c, [0x14bf] = 0x3a5d, [0x14c0] = 0x3a5e, [0x14c1] = 0x3a5f,
  [0x14c2] = 0x3a60, [0x14c3] = 0x3a61, [0x14c4] = 0x3a62, [0x14c5] = 0x3a63,
  [0x14c6] = 0x3a64, [0x14c7] = 0x3a65, [0x14c8] = 0x3a66, [0x14c9] = 0x3a67,
  [0x14ca] = 0x3a68, [0x14cb] = 0x3a69, [0x14cc] = 0x3a6a, [0x14cd] = 0x3a6b,
  [0x14ce] = 0x3a6c, [0x14cf] = 0x3a6d, [0x14d0] = 0x3a6e, [0x14d1] = 0x3a6f,
  [0x14d2] = 0x3a70, [0x14d3] = 0x3a71, [0x14d4] = 0x3a72, [0x14d5] = 0x3a74,
  [0x14d6] = 0x3a75, [0x14d7] = 0x3a76, [0x14d8] = 0x3a77, [0x14d9] = 0x3a78,
  [0x14da] = 0x3a79, [0x14db] = 0x3a7a, [0x14dc] = 0x3a7b, [0x14dd] = 0x3a7c,
  [0x14de] = 0x3a7d, [0x14df] = 0x3a7e, [0x14e0] = 0x3a7f, [0x14e1] = 0x3a80,
  [0x14e2] = 0x3a81, [0x14e3] = 0x3a82, [0x14e4] = 0x3a83, [0x14e5] = 0x3a84,
  [0x14e6] = 0x3a85, [0x14e7] = 0x3a86, [0x14e8] = 0x3a87, [0x14e9] = 0x3a88,
  [0x14ea] = 0x3a89, [0x14eb] = 0x3a8a, [0x14ec] = 0x3a8b, [0x14ed] = 0x3a8c,
  [0x14ee] = 0x3a8d, [0x14ef] = 0x3a8e, [0x14f0] = 0x3a8f, [0x14f1] = 0x3a90,
  [0x14f2] = 0x3a91, [0x14f3] = 0x3a92, [0x14f4] = 0x3a93, [0x14f5] = 0x3a94,
  [0x14f6] = 0x3a95, [0x14f7] = 0x3a96, [0x14f8] = 0x3a97, [0x14f9] = 0x3a98,
  [0x14fa] = 0x3a99, [0x14fb] = 0x3a9a, [0x14fc] = 0x3a9b, [0x14fd] = 0x3a9c,
  [0x14fe] = 0x3a9d, [0x14ff] = 0x3a9e, [0x1500] = 0x3a9f, [0x1501] = 0x3aa0,
  [0x1502] = 0x3aa1, [0x1503] = 0x3aa2, [0x1504] = 0x3aa3, [0x1505] = 0x3aa4,
  [0x1506] = 0x3aa5, [0x1507] = 0x3aa6, [0x1508] = 0x3aa7, [0x1509] = 0x3aa8,
  [0x150a] = 0x3aa9, [0x150b] = 0x3aaa, [0x150c] = 0x3aab, [0x150d] = 0x3aac,
  [0x150e] = 0x3aad, [0x150f] = 0x3aae, [0x1510] = 0x3aaf, [0x1511] = 0x3ab0,
  [0x1512] = 0x3ab1, [0x1513] = 0x3ab2, [0x1514] = 0x3ab3, [0x1515] = 0x3ab4,
  [0x1516] = 0x3ab5, [0x1517] = 0x3ab6, [0x1518] = 0x3ab7, [0x1519] = 0x3ab8,
  [0x151a] = 0x3ab9, [0x151b] = 0x3aba, [0x151c] = 0x3abb, [0x151d] = 0x3abc,
  [0x151e] = 0x3abd, [0x151f] = 0x3abe, [0x1520] = 0x3abf, [0x1521] = 0x3ac0,
  [0x1522] = 0x3ac1, [0x1523] = 0x3ac2, [0x1524] = 0x3ac3, [0x1525] = 0x3ac4,
  [0x1526] = 0x3ac5, [0x1527] = 0x3ac6, [0x1528] = 0x3ac7, [0x1529] = 0x3ac8,
  [0x152a] = 0x3ac9, [0x152b] = 0x3aca, [0x152c] = 0x3acb, [0x152d] = 0x3acc,
  [0x152e] = 0x3acd, [0x152f] = 0x3ace, [0x1530] = 0x3acf, [0x1531] = 0x3ad0,
  [0x1532] = 0x3ad1, [0x1533] = 0x3ad2, [0x1534] = 0x3ad3, [0x1535] = 0x3ad4,
  [0x1536] = 0x3ad5, [0x1537] = 0x3ad6, [0x1538] = 0x3ad7, [0x1539] = 0x3ad8,
  [0x153a] = 0x3ad9, [0x153b] = 0x3ada, [0x153c] = 0x3adb, [0x153d] = 0x3adc,
  [0x153e] = 0x3add, [0x153f] = 0x3ade, [0x1540] = 0x3adf, [0x1541] = 0x3ae0,
  [0x1542] = 0x3ae1, [0x1543] = 0x3ae2, [0x1544] = 0x3ae3, [0x1545] = 0x3ae4,
  [0x1546] = 0x3ae5, [0x1547] = 0x3ae6, [0x1548] = 0x3ae7, [0x1549] = 0x3ae8,
  [0x154a] = 0x3ae9, [0x154b] = 0x3aea, [0x154c] = 0x3aeb, [0x154d] = 0x3aec,
  [0x154e] = 0x3aed, [0x154f] = 0x3aee, [0x1550] = 0x3aef, [0x1551] = 0x3af0,
  [0x1552] = 0x3af1, [0x1553] = 0x3af2, [0x1554] = 0x3af3, [0x1555] = 0x3af4,
  [0x1556] = 0x3af5, [0x1557] = 0x3af6, [0x1558] = 0x3af7, [0x1559] = 0x3af8,
  [0x155a] = 0x3af9, [0x155b] = 0x3afa, [0x155c] = 0x3afb, [0x155d] = 0x3afc,
  [0x155e] = 0x3afd, [0x155f] = 0x3afe, [0x1560] = 0x3aff, [0x1561] = 0x3b00,
  [0x1562] = 0x3b01, [0x1563] = 0x3b02, [0x1564] = 0x3b03, [0x1565] = 0x3b04,
  [0x1566] = 0x3b05, [0x1567] = 0x3b06, [0x1568] = 0x3b07, [0x1569] = 0x3b08,
  [0x156a] = 0x3b09, [0x156b] = 0x3b0a, [0x156c] = 0x3b0b, [0x156d] = 0x3b0c,
  [0x156e] = 0x3b0d, [0x156f] = 0x3b0e, [0x1570] = 0x3b0f, [0x1571] = 0x3b10,
  [0x1572] = 0x3b11, [0x1573] = 0x3b12, [0x1574] = 0x3b13, [0x1575] = 0x3b14,
  [0x1576] = 0x3b15, [0x1577] = 0x3b16, [0x1578] = 0x3b17, [0x1579] = 0x3b18,
  [0x157a] = 0x3b19, [0x157b] = 0x3b1a, [0x157c] = 0x3b1b, [0x157d] = 0x3b1c,
  [0x157e] = 0x3b1d, [0x157f] = 0x3b1e, [0x1580] = 0x3b1f, [0x1581] = 0x3b20,
  [0x1582] = 0x3b21, [0x1583] = 0x3b22, [0x1584] = 0x3b23, [0x1585] = 0x3b24,
  [0x1586] = 0x3b25, [0x1587] = 0x3b26, [0x1588] = 0x3b27, [0x1589] = 0x3b28,
  [0x158a] = 0x3b29, [0x158b] = 0x3b2a, [0x158c] = 0x3b2b, [0x158d] = 0x3b2c,
  [0x158e] = 0x3b2d, [0x158f] = 0x3b2e, [0x1590] = 0x3b2f, [0x1591] = 0x3b30,
  [0x1592] = 0x3b31, [0x1593] = 0x3b32, [0x1594] = 0x3b33, [0x1595] = 0x3b34,
  [0x1596] = 0x3b35, [0x1597] = 0x3b36, [0x1598] = 0x3b37, [0x1599] = 0x3b38,
  [0x159a] = 0x3b39, [0x159b] = 0x3b3a, [0x159c] = 0x3b3b, [0x159d] = 0x3b3c,
  [0x159e] = 0x3b3d, [0x159f] = 0x3b3e, [0x15a0] = 0x3b3f, [0x15a1] = 0x3b40,
  [0x15a2] = 0x3b41, [0x15a3] = 0x3b42, [0x15a4] = 0x3b43, [0x15a5] = 0x3b44,
  [0x15a6] = 0x3b45, [0x15a7] = 0x3b46, [0x15a8] = 0x3b47, [0x15a9] = 0x3b48,
  [0x15aa] = 0x3b49, [0x15ab] = 0x3b4a, [0x15ac] = 0x3b4b, [0x15ad] = 0x3b4c,
  [0x15ae] = 0x3b4d, [0x15af] = 0x3b4f, [0x15b0] = 0x3b50, [0x15b1] = 0x3b51,
  [0x15b2] = 0x3b52, [0x15b3] = 0x3b53, [0x15b4] = 0x3b54, [0x15b5] = 0x3b55,
  [0x15b6] = 0x3b56, [0x15b7] = 0x3b57, [0x15b8] = 0x3b58, [0x15b9] = 0x3b59,
  [0x15ba] = 0x3b5a, [0x15bb] = 0x3b5b, [0x15bc] = 0x3b5c, [0x15bd] = 0x3b5d,
  [0x15be] = 0x3b5e, [0x15bf] = 0x3b5f, [0x15c0] = 0x3b60, [0x15c1] = 0x3b61,
  [0x15c2] = 0x3b62, [0x15c3] = 0x3b63, [0x15c4] = 0x3b64, [0x15c5] = 0x3b65,
  [0x15c6] = 0x3b66, [0x15c7] = 0x3b67, [0x15c8] = 0x3b68, [0x15c9] = 0x3b69,
  [0x15ca] = 0x3b6a, [0x15cb] = 0x3b6b, [0x15cc] = 0x3b6c, [0x15cd] = 0x3b6d,
  [0x15ce] = 0x3b6e, [0x15cf] = 0x3b6f, [0x15d0] = 0x3b70, [0x15d1] = 0x3b71,
  [0x15d2] = 0x3b72, [0x15d3] = 0x3b73, [0x15d4] = 0x3b74, [0x15d5] = 0x3b75,
  [0x15d6] = 0x3b76, [0x15d7] = 0x3b77, [0x15d8] = 0x3b78, [0x15d9] = 0x3b79,
  [0x15da] = 0x3b7a, [0x15db] = 0x3b7b, [0x15dc] = 0x3b7c, [0x15dd] = 0x3b7d,
  [0x15de] = 0x3b7e, [0x15df] = 0x3b7f, [0x15e0] = 0x3b80, [0x15e1] = 0x3b81,
  [0x15e2] = 0x3b82, [0x15e3] = 0x3b83, [0x15e4] = 0x3b84, [0x15e5] = 0x3b85,
  [0x15e6] = 0x3b86, [0x15e7] = 0x3b87, [0x15e8] = 0x3b88, [0x15e9] = 0x3b89,
  [0x15ea] = 0x3b8a, [0x15eb] = 0x3b8b, [0x15ec] = 0x3b8c, [0x15ed] = 0x3b8d,
  [0x15ee] = 0x3b8e, [0x15ef] = 0x3b8f, [0x15f0] = 0x3b90, [0x15f1] = 0x3b91,
  [0x15f2] = 0x3b92, [0x15f3] = 0x3b93, [0x15f4] = 0x3b94, [0x15f5] = 0x3b95,
  [0x15f6] = 0x3b96, [0x15f7] = 0x3b97, [0x15f8] = 0x3b98, [0x15f9] = 0x3b99,
  [0x15fa] = 0x3b9a, [0x15fb] = 0x3b9b, [0x15fc] = 0x3b9c, [0x15fd] = 0x3b9d,
  [0x15fe] = 0x3b9e, [0x15ff] = 0x3b9f, [0x1600] = 0x3ba0, [0x1601] = 0x3ba1,
  [0x1602] = 0x3ba2, [0x1603] = 0x3ba3, [0x1604] = 0x3ba4, [0x1605] = 0x3ba5,
  [0x1606] = 0x3ba6, [0x1607] = 0x3ba7, [0x1608] = 0x3ba8, [0x1609] = 0x3ba9,
  [0x160a] = 0x3baa, [0x160b] = 0x3bab, [0x160c] = 0x3bac, [0x160d] = 0x3bad,
  [0x160e] = 0x3bae, [0x160f] = 0x3baf, [0x1610] = 0x3bb0, [0x1611] = 0x3bb1,
  [0x1612] = 0x3bb2, [0x1613] = 0x3bb3, [0x1614] = 0x3bb4, [0x1615] = 0x3bb5,
  [0x1616] = 0x3bb6, [0x1617] = 0x3bb7, [0x1618] = 0x3bb8, [0x1619] = 0x3bb9,
  [0x161a] = 0x3bba, [0x161b] = 0x3bbb, [0x161c] = 0x3bbc, [0x161d] = 0x3bbd,
  [0x161e] = 0x3bbe, [0x161f] = 0x3bbf, [0x1620] = 0x3bc0, [0x1621] = 0x3bc1,
  [0x1622] = 0x3bc2, [0x1623] = 0x3bc3, [0x1624] = 0x3bc4, [0x1625] = 0x3bc5,
  [0x1626] = 0x3bc6, [0x1627] = 0x3bc7, [0x1628] = 0x3bc8, [0x1629] = 0x3bc9,
  [0x162a] = 0x3bca, [0x162b] = 0x3bcb, [0x162c] = 0x3bcc, [0x162d] = 0x3bcd,
  [0x162e] = 0x3bce, [0x162f] = 0x3bcf, [0x1630] = 0x3bd0, [0x1631] = 0x3bd1,
  [0x1632] = 0x3bd2, [0x1633] = 0x3bd3, [0x1634] = 0x3bd4, [0x1635] = 0x3bd5,
  [0x1636] = 0x3bd6, [0x1637] = 0x3bd7, [0x1638] = 0x3bd8, [0x1639] = 0x3bd9,
  [0x163a] = 0x3bda, [0x163b] = 0x3bdb, [0x163c] = 0x3bdc, [0x163d] = 0x3bdd,
  [0x163e] = 0x3bde, [0x163f] = 0x3bdf, [0x1640] = 0x3be0, [0x1641] = 0x3be1,
  [0x1642] = 0x3be2, [0x1643] = 0x3be3, [0x1644] = 0x3be4, [0x1645] = 0x3be5,
  [0x1646] = 0x3be6, [0x1647] = 0x3be7, [0x1648] = 0x3be8, [0x1649] = 0x3be9,
  [0x164a] = 0x3bea, [0x164b] = 0x3beb, [0x164c] = 0x3bec, [0x164d] = 0x3bed,
  [0x164e] = 0x3bee, [0x164f] = 0x3bef, [0x1650] = 0x3bf0, [0x1651] = 0x3bf1,
  [0x1652] = 0x3bf2, [0x1653] = 0x3bf3, [0x1654] = 0x3bf4, [0x1655] = 0x3bf5,
  [0x1656] = 0x3bf6, [0x1657] = 0x3bf7, [0x1658] = 0x3bf8, [0x1659] = 0x3bf9,
  [0x165a] = 0x3bfa, [0x165b] = 0x3bfb, [0x165c] = 0x3bfc, [0x165d] = 0x3bfd,
  [0x165e] = 0x3bfe, [0x165f] = 0x3bff, [0x1660] = 0x3c00, [0x1661] = 0x3c01,
  [0x1662] = 0x3c02, [0x1663] = 0x3c03, [0x1664] = 0x3c04, [0x1665] = 0x3c05,
  [0x1666] = 0x3c06, [0x1667] = 0x3c07, [0x1668] = 0x3c08, [0x1669] = 0x3c09,
  [0x166a] = 0x3c0a, [0x166b] = 0x3c0b, [0x166c] = 0x3c0c, [0x166d] = 0x3c0d,
  [0x166e] = 0x3c0e, [0x166f] = 0x3c0f, [0x1670] = 0x3c10, [0x1671] = 0x3c11,
  [0x1672] = 0x3c12, [0x1673] = 0x3c13, [0x1674] = 0x3c14, [0x1675] = 0x3c15,
  [0x1676] = 0x3c16, [0x1677] = 0x3c17, [0x1678] = 0x3c18, [0x1679] = 0x3c19,
  [0x167a] = 0x3c1a, [0x167b] = 0x3c1b, [0x167c] = 0x3c1c, [0x167d] = 0x3c1d,
  [0x167e] = 0x3c1e, [0x167f] = 0x3c1f, [0x1680] = 0x3c20, [0x1681] = 0x3c21,
  [0x1682] = 0x3c22, [0x1683] = 0x3c23, [0x1684] = 0x3c24, [0x1685] = 0x3c25,
  [0x1686] = 0x3c26, [0x1687] = 0x3c27, [0x1688] = 0x3c28, [0x1689] = 0x3c29,
  [0x168a] = 0x3c2a, [0x168b] = 0x3c2b, [0x168c] = 0x3c2c, [0x168d] = 0x3c2d,
  [0x168e] = 0x3c2e, [0x168f] = 0x3c2f, [0x1690] = 0x3c30, [0x1691] = 0x3c31,
  [0x1692] = 0x3c32, [0x1693] = 0x3c33, [0x1694] = 0x3c34, [0x1695] = 0x3c35,
  [0x1696] = 0x3c36, [0x1697] = 0x3c37, [0x1698] = 0x3c38, [0x1699] = 0x3c39,
  [0x169a] = 0x3c3a, [0x169b] = 0x3c3b, [0x169c] = 0x3c3c, [0x169d] = 0x3c3d,
  [0x169e] = 0x3c3e, [0x169f] = 0x3c3f, [0x16a0] = 0x3c40, [0x16a1] = 0x3c41,
  [0x16a2] = 0x3c42, [0x16a3] = 0x3c43, [0x16a4] = 0x3c44, [0x16a5] = 0x3c45,
  [0x16a6] = 0x3c46, [0x16a7] = 0x3c47, [0x16a8] = 0x3c48, [0x16a9] = 0x3c49,
  [0x16aa] = 0x3c4a, [0x16ab] = 0x3c4b, [0x16ac] = 0x3c4c, [0x16ad] = 0x3c4d,
  [0x16ae] = 0x3c4e, [0x16af] = 0x3c4f, [0x16b0] = 0x3c50, [0x16b1] = 0x3c51,
  [0x16b2] = 0x3c52, [0x16b3] = 0x3c53, [0x16b4] = 0x3c54, [0x16b5] = 0x3c55,
  [0x16b6] = 0x3c56, [0x16b7] = 0x3c57, [0x16b8] = 0x3c58, [0x16b9] = 0x3c59,
  [0x16ba] = 0x3c5a, [0x16bb] = 0x3c5b, [0x16bc] = 0x3c5c, [0x16bd] = 0x3c5d,
  [0x16be] = 0x3c5e, [0x16bf] = 0x3c5f, [0x16c0] = 0x3c60, [0x16c1] = 0x3c61,
  [0x16c2] = 0x3c62, [0x16c3] = 0x3c63, [0x16c4] = 0x3c64, [0x16c5] = 0x3c65,
  [0x16c6] = 0x3c66, [0x16c7] = 0x3c67, [0x16c8] = 0x3c68, [0x16c9] = 0x3c69,
  [0x16ca] = 0x3c6a, [0x16cb] = 0x3c6b, [0x16cc] = 0x3c6c, [0x16cd] = 0x3c6d,
  [0x16ce] = 0x3c6f, [0x16cf] = 0x3c70, [0x16d0] = 0x3c71, [0x16d1] = 0x3c72,
  [0x16d2] = 0x3c73, [0x16d3] = 0x3c74, [0x16d4] = 0x3c75, [0x16d5] = 0x3c76,
  [0x16d6] = 0x3c77, [0x16d7] = 0x3c78, [0x16d8] = 0x3c79, [0x16d9] = 0x3c7a,
  [0x16da] = 0x3c7b, [0x16db] = 0x3c7c, [0x16dc] = 0x3c7d, [0x16dd] = 0x3c7e,
  [0x16de] = 0x3c7f, [0x16df] = 0x3c80, [0x16e0] = 0x3c81, [0x16e1] = 0x3c82,
  [0x16e2] = 0x3c83, [0x16e3] = 0x3c84, [0x16e4] = 0x3c85, [0x16e5] = 0x3c86,
  [0x16e6] = 0x3c87, [0x16e7] = 0x3c88, [0x16e8] = 0x3c89, [0x16e9] = 0x3c8a,
  [0x16ea] = 0x3c8b, [0x16eb] = 0x3c8c, [0x16ec] = 0x3c8d, [0x16ed] = 0x3c8e,
  [0x16ee] = 0x3c8f, [0x16ef] = 0x3c90, [0x16f0] = 0x3c91, [0x16f1] = 0x3c92,
  [0x16f2] = 0x3c93, [0x16f3] = 0x3c94, [0x16f4] = 0x3c95, [0x16f5] = 0x3c96,
  [0x16f6] = 0x3c97, [0x16f7] = 0x3c98, [0x16f8] = 0x3c99, [0x16f9] = 0x3c9a,
  [0x16fa] = 0x3c9b, [0x16fb] = 0x3c9c, [0x16fc] = 0x3c9d, [0x16fd] = 0x3c9e,
  [0x16fe] = 0x3c9f, [0x16ff] = 0x3ca0, [0x1700] = 0x3ca1, [0x1701] = 0x3ca2,
  [0x1702] = 0x3ca3, [0x1703] = 0x3ca4, [0x1704] = 0x3ca5, [0x1705] = 0x3ca6,
  [0x1706] = 0x3ca7, [0x1707] = 0x3ca8, [0x1708] = 0x3ca9, [0x1709] = 0x3caa,
  [0x170a] = 0x3cab, [0x170b] = 0x3cac, [0x170c] = 0x3cad, [0x170d] = 0x3cae,
  [0x170e] = 0x3caf, [0x170f] = 0x3cb0, [0x1710] = 0x3cb1, [0x1711] = 0x3cb2,
  [0x1712] = 0x3cb3, [0x1713] = 0x3cb4, [0x1714] = 0x3cb5, [0x1715] = 0x3cb6,
  [0x1716] = 0x3cb7, [0x1717] = 0x3cb8, [0x1718] = 0x3cb9, [0x1719] = 0x3cba,
  [0x171a] = 0x3cbb, [0x171b] = 0x3cbc, [0x171c] = 0x3cbd, [0x171d] = 0x3cbe,
  [0x171e] = 0x3cbf, [0x171f] = 0x3cc0, [0x1720] = 0x3cc1, [0x1721] = 0x3cc2,
  [0x1722] = 0x3cc3, [0x1723] = 0x3cc4, [0x1724] = 0x3cc5, [0x1725] = 0x3cc6,
  [0x1726] = 0x3cc7, [0x1727] = 0x3cc8, [0x1728] = 0x3cc9, [0x1729] = 0x3cca,
  [0x172a] = 0x3ccb, [0x172b] = 0x3ccc, [0x172c] = 0x3ccd, [0x172d] = 0x3cce,
  [0x172e] = 0x3ccf, [0x172f] = 0x3cd0, [0x1730] = 0x3cd1, [0x1731] = 0x3cd2,
  [0x1732] = 0x3cd3, [0x1733] = 0x3cd4, [0x1734] = 0x3cd5, [0x1735] = 0x3cd6,
  [0x1736] = 0x3cd7, [0x1737] = 0x3cd8, [0x1738] = 0x3cd9, [0x1739] = 0x3cda,
  [0x173a] = 0x3cdb, [0x173b] = 0x3cdc, [0x173c] = 0x3cdd, [0x173d] = 0x3cde,
  [0x173e] = 0x3cdf, [0x1ab4] = 0x4057, [0x1ab5] = 0x4058, [0x1ab6] = 0x4059,
  [0x1ab7] = 0x405a, [0x1ab8] = 0x405b, [0x1ab9] = 0x405c, [0x1aba] = 0x405d,
  [0x1abb] = 0x405e, [0x1abc] = 0x405f, [0x1abd] = 0x4060, [0x1abe] = 0x4061,
  [0x1abf] = 0x4062, [0x1ac0] = 0x4063, [0x1ac1] = 0x4064, [0x1ac2] = 0x4065,
  [0x1ac3] = 0x4066, [0x1ac4] = 0x4067, [0x1ac5] = 0x4068, [0x1ac6] = 0x4069,
  [0x1ac7] = 0x406a, [0x1ac8] = 0x406b, [0x1ac9] = 0x406c, [0x1aca] = 0x406d,
  [0x1acb] = 0x406e, [0x1acc] = 0x406f, [0x1acd] = 0x4070, [0x1ace] = 0x4071,
  [0x1acf] = 0x4072, [0x1ad0] = 0x4073, [0x1ad1] = 0x4074, [0x1ad2] = 0x4075,
  [0x1ad3] = 0x4076, [0x1ad4] = 0x4077, [0x1ad5] = 0x4078, [0x1ad6] = 0x4079,
  [0x1ad7] = 0x407a, [0x1ad8] = 0x407b, [0x1ad9] = 0x407c, [0x1ada] = 0x407d,
  [0x1adb] = 0x407e, [0x1adc] = 0x407f, [0x1add] = 0x4080, [0x1ade] = 0x4081,
  [0x1adf] = 0x4082, [0x1ae0] = 0x4083, [0x1ae1] = 0x4084, [0x1ae2] = 0x4085,
  [0x1ae3] = 0x4086, [0x1ae4] = 0x4087, [0x1ae5] = 0x4088, [0x1ae6] = 0x4089,
  [0x1ae7] = 0x408a, [0x1ae8] = 0x408b, [0x1ae9] = 0x408c, [0x1aea] = 0x408d,
  [0x1aeb] = 0x408e, [0x1aec] = 0x408f, [0x1aed] = 0x4090, [0x1aee] = 0x4091,
  [0x1aef] = 0x4092, [0x1af0] = 0x4093, [0x1af1] = 0x4094, [0x1af2] = 0x4095,
  [0x1af3] = 0x4096, [0x1af4] = 0x4097, [0x1af5] = 0x4098, [0x1af6] = 0x4099,
  [0x1af7] = 0x409a, [0x1af8] = 0x409b, [0x1af9] = 0x409c, [0x1afa] = 0x409d,
  [0x1afb] = 0x409e, [0x1afc] = 0x409f, [0x1afd] = 0x40a0, [0x1afe] = 0x40a1,
  [0x1aff] = 0x40a2, [0x1b00] = 0x40a3, [0x1b01] = 0x40a4, [0x1b02] = 0x40a5,
  [0x1b03] = 0x40a6, [0x1b04] = 0x40a7, [0x1b05] = 0x40a8, [0x1b06] = 0x40a9,
  [0x1b07] = 0x40aa, [0x1b08] = 0x40ab, [0x1b09] = 0x40ac, [0x1b0a] = 0x40ad,
  [0x1b0b] = 0x40ae, [0x1b0c] = 0x40af, [0x1b0d] = 0x40b0, [0x1b0e] = 0x40b1,
  [0x1b0f] = 0x40b2, [0x1b10] = 0x40b3, [0x1b11] = 0x40b4, [0x1b12] = 0x40b5,
  [0x1b13] = 0x40b6, [0x1b14] = 0x40b7, [0x1b15] = 0x40b8, [0x1b16] = 0x40b9,
  [0x1b17] = 0x40ba, [0x1b18] = 0x40bb, [0x1b19] = 0x40bc, [0x1b1a] = 0x40bd,
  [0x1b1b] = 0x40be, [0x1b1c] = 0x40bf, [0x1b1d] = 0x40c0, [0x1b1e] = 0x40c1,
  [0x1b1f] = 0x40c2, [0x1b20] = 0x40c3, [0x1b21] = 0x40c4, [0x1b22] = 0x40c5,
  [0x1b23] = 0x40c6, [0x1b24] = 0x40c7, [0x1b25] = 0x40c8, [0x1b26] = 0x40c9,
  [0x1b27] = 0x40ca, [0x1b28] = 0x40cb, [0x1b29] = 0x40cc, [0x1b2a] = 0x40cd,
  [0x1b2b] = 0x40ce, [0x1b2c] = 0x40cf, [0x1b2d] = 0x40d0, [0x1b2e] = 0x40d1,
  [0x1b2f] = 0x40d2, [0x1b30] = 0x40d3, [0x1b31] = 0x40d4, [0x1b32] = 0x40d5,
  [0x1b33] = 0x40d6, [0x1b34] = 0x40d7, [0x1b35] = 0x40d8, [0x1b36] = 0x40d9,
  [0x1b37] = 0x40da, [0x1b38] = 0x40db, [0x1b39] = 0x40dc, [0x1b3a] = 0x40dd,
  [0x1b3b] = 0x40de, [0x1b3c] = 0x40df, [0x1b3d] = 0x40e0, [0x1b3e] = 0x40e1,
  [0x1b3f] = 0x40e2, [0x1b40] = 0x40e3, [0x1b41] = 0x40e4, [0x1b42] = 0x40e5,
  [0x1b43] = 0x40e6, [0x1b44] = 0x40e7, [0x1b45] = 0x40e8, [0x1b46] = 0x40e9,
  [0x1b47] = 0x40ea, [0x1b48] = 0x40eb, [0x1b49] = 0x40ec, [0x1b4a] = 0x40ed,
  [0x1b4b] = 0x40ee, [0x1b4c] = 0x40ef, [0x1b4d] = 0x40f0, [0x1b4e] = 0x40f1,
  [0x1b4f] = 0x40f2, [0x1b50] = 0x40f3, [0x1b51] = 0x40f4, [0x1b52] = 0x40f5,
  [0x1b53] = 0x40f6, [0x1b54] = 0x40f7, [0x1b55] = 0x40f8, [0x1b56] = 0x40f9,
  [0x1b57] = 0x40fa, [0x1b58] = 0x40fb, [0x1b59] = 0x40fc, [0x1b5a] = 0x40fd,
  [0x1b5b] = 0x40fe, [0x1b5c] = 0x40ff, [0x1b5d] = 0x4100, [0x1b5e] = 0x4101,
  [0x1b5f] = 0x4102, [0x1b60] = 0x4103, [0x1b61] = 0x4104, [0x1b62] = 0x4105,
  [0x1b63] = 0x4106, [0x1b64] = 0x4107, [0x1b65] = 0x4108, [0x1b66] = 0x4109,
  [0x1b67] = 0x410a, [0x1b68] = 0x410b, [0x1b69] = 0x410c, [0x1b6a] = 0x410d,
  [0x1b6b] = 0x410e, [0x1b6c] = 0x410f, [0x1b6d] = 0x4110, [0x1b6e] = 0x4111,
  [0x1b6f] = 0x4112, [0x1b70] = 0x4113, [0x1b71] = 0x4114, [0x1b72] = 0x4115,
  [0x1b73] = 0x4116, [0x1b74] = 0x4117, [0x1b75] = 0x4118, [0x1b76] = 0x4119,
  [0x1b77] = 0x411a, [0x1b78] = 0x411b, [0x1b79] = 0x411c, [0x1b7a] = 0x411d,
  [0x1b7b] = 0x411e, [0x1b7c] = 0x411f, [0x1b7d] = 0x4120, [0x1b7e] = 0x4121,
  [0x1b7f] = 0x4122, [0x1b80] = 0x4123, [0x1b81] = 0x4124, [0x1b82] = 0x4125,
  [0x1b83] = 0x4126, [0x1b84] = 0x4127, [0x1b85] = 0x4128, [0x1b86] = 0x4129,
  [0x1b87] = 0x412a, [0x1b88] = 0x412b, [0x1b89] = 0x412c, [0x1b8a] = 0x412d,
  [0x1b8b] = 0x412e, [0x1b8c] = 0x412f, [0x1b8d] = 0x4130, [0x1b8e] = 0x4131,
  [0x1b8f] = 0x4132, [0x1b90] = 0x4133, [0x1b91] = 0x4134, [0x1b92] = 0x4135,
  [0x1b93] = 0x4136, [0x1b94] = 0x4137, [0x1b95] = 0x4138, [0x1b96] = 0x4139,
  [0x1b97] = 0x413a, [0x1b98] = 0x413b, [0x1b99] = 0x413c, [0x1b9a] = 0x413d,
  [0x1b9b] = 0x413e, [0x1b9c] = 0x413f, [0x1b9d] = 0x4140, [0x1b9e] = 0x4141,
  [0x1b9f] = 0x4142, [0x1ba0] = 0x4143, [0x1ba1] = 0x4144, [0x1ba2] = 0x4145,
  [0x1ba3] = 0x4146, [0x1ba4] = 0x4147, [0x1ba5] = 0x4148, [0x1ba6] = 0x4149,
  [0x1ba7] = 0x414a, [0x1ba8] = 0x414b, [0x1ba9] = 0x414c, [0x1baa] = 0x414d,
  [0x1bab] = 0x414e, [0x1bac] = 0x414f, [0x1bad] = 0x4150, [0x1bae] = 0x4151,
  [0x1baf] = 0x4152, [0x1bb0] = 0x4153, [0x1bb1] = 0x4154, [0x1bb2] = 0x4155,
  [0x1bb3] = 0x4156, [0x1bb4] = 0x4157, [0x1bb5] = 0x4158, [0x1bb6] = 0x4159,
  [0x1bb7] = 0x415a, [0x1bb8] = 0x415b, [0x1bb9] = 0x415c, [0x1bba] = 0x415d,
  [0x1bbb] = 0x415e, [0x1d93] = 0x4338, [0x1d94] = 0x4339, [0x1d95] = 0x433a,
  [0x1d96] = 0x433b, [0x1d97] = 0x433c, [0x1d98] = 0x433d, [0x1d99] = 0x433e,
  [0x1d9a] = 0x433f, [0x1d9b] = 0x4340, [0x1d9c] = 0x4341, [0x1d9d] = 0x4342,
  [0x1d9e] = 0x4343, [0x1d9f] = 0x4344, [0x1da0] = 0x4345, [0x1da1] = 0x4346,
  [0x1da2] = 0x4347, [0x1da3] = 0x4348, [0x1da4] = 0x4349, [0x1da5] = 0x434a,
  [0x1da6] = 0x434b, [0x1da7] = 0x434c, [0x1da8] = 0x434d, [0x1da9] = 0x434e,
  [0x1daa] = 0x434f, [0x1dab] = 0x4350, [0x1dac] = 0x4351, [0x1dad] = 0x4352,
  [0x1dae] = 0x4353, [0x1daf] = 0x4354, [0x1db0] = 0x4355, [0x1db1] = 0x4356,
  [0x1db2] = 0x4357, [0x1db3] = 0x4358, [0x1db4] = 0x4359, [0x1db5] = 0x435a,
  [0x1db6] = 0x435b, [0x1db7] = 0x435c, [0x1db8] = 0x435d, [0x1db9] = 0x435e,
  [0x1dba] = 0x435f, [0x1dbb] = 0x4360, [0x1dbc] = 0x4361, [0x1dbd] = 0x4362,
  [0x1dbe] = 0x4363, [0x1dbf] = 0x4364, [0x1dc0] = 0x4365, [0x1dc1] = 0x4366,
  [0x1dc2] = 0x4367, [0x1dc3] = 0x4368, [0x1dc4] = 0x4369, [0x1dc5] = 0x436a,
  [0x1dc6] = 0x436b, [0x1dc7] = 0x436c, [0x1dc8] = 0x436d, [0x1dc9] = 0x436e,
  [0x1dca] = 0x436f, [0x1dcb] = 0x4370, [0x1dcc] = 0x4371, [0x1dcd] = 0x4372,
  [0x1dce] = 0x4373, [0x1dcf] = 0x4374, [0x1dd0] = 0x4375, [0x1dd1] = 0x4376,
  [0x1dd2] = 0x4377, [0x1dd3] = 0x4378, [0x1dd4] = 0x4379, [0x1dd5] = 0x437a,
  [0x1dd6] = 0x437b, [0x1dd7] = 0x437c, [0x1dd8] = 0x437d, [0x1dd9] = 0x437e,
  [0x1dda] = 0x437f, [0x1ddb] = 0x4380, [0x1ddc] = 0x4381, [0x1ddd] = 0x4382,
  [0x1dde] = 0x4383, [0x1ddf] = 0x4384, [0x1de0] = 0x4385, [0x1de1] = 0x4386,
  [0x1de2] = 0x4387, [0x1de3] = 0x4388, [0x1de4] = 0x4389, [0x1de5] = 0x438a,
  [0x1de6] = 0x438b, [0x1de7] = 0x438c, [0x1de8] = 0x438d, [0x1de9] = 0x438e,
  [0x1dea] = 0x438f, [0x1deb] = 0x4390, [0x1dec] = 0x4391, [0x1ded] = 0x4392,
  [0x1dee] = 0x4393, [0x1def] = 0x4394, [0x1df0] = 0x4395, [0x1df1] = 0x4396,
  [0x1df2] = 0x4397, [0x1df3] = 0x4398, [0x1df4] = 0x4399, [0x1df5] = 0x439a,
  [0x1df6] = 0x439b, [0x1df7] = 0x439c, [0x1df8] = 0x439d, [0x1df9] = 0x439e,
  [0x1dfa] = 0x439f, [0x1dfb] = 0x43a0, [0x1dfc] = 0x43a1, [0x1dfd] = 0x43a2,
  [0x1dfe] = 0x43a3, [0x1dff] = 0x43a4, [0x1e00] = 0x43a5, [0x1e01] = 0x43a6,
  [0x1e02] = 0x43a7, [0x1e03] = 0x43a8, [0x1e04] = 0x43a9, [0x1e05] = 0x43aa,
  [0x1e06] = 0x43ab, [0x1e07] = 0x43ad, [0x1e08] = 0x43ae, [0x1e09] = 0x43af,
  [0x1e0a] = 0x43b0, [0x1e0b] = 0x43b2, [0x1e0c] = 0x43b3, [0x1e0d] = 0x43b4,
  [0x1e0e] = 0x43b5, [0x1e0f] = 0x43b6, [0x1e10] = 0x43b7, [0x1e11] = 0x43b8,
  [0x1e12] = 0x43b9, [0x1e13] = 0x43ba, [0x1e14] = 0x43bb, [0x1e15] = 0x43bc,
  [0x1e16] = 0x43bd, [0x1e17] = 0x43be, [0x1e18] = 0x43bf, [0x1e19] = 0x43c0,
  [0x1e1a] = 0x43c1, [0x1e1b] = 0x43c2, [0x1e1c] = 0x43c3, [0x1e1d] = 0x43c4,
  [0x1e1e] = 0x43c5, [0x1e1f] = 0x43c6, [0x1e20] = 0x43c7, [0x1e21] = 0x43c8,
  [0x1e22] = 0x43c9, [0x1e23] = 0x43ca, [0x1e24] = 0x43cb, [0x1e25] = 0x43cc,
  [0x1e26] = 0x43cd, [0x1e27] = 0x43ce, [0x1e28] = 0x43cf, [0x1e29] = 0x43d0,
  [0x1e2a] = 0x43d1, [0x1e2b] = 0x43d2, [0x1e2c] = 0x43d3, [0x1e2d] = 0x43d4,
  [0x1e2e] = 0x43d5, [0x1e2f] = 0x43d6, [0x1e30] = 0x43d7, [0x1e31] = 0x43d8,
  [0x1e32] = 0x43d9, [0x1e33] = 0x43da, [0x1e34] = 0x43db, [0x1e35] = 0x43dc,
  [0x1e36] = 0x43de, [0x1e37] = 0x43df, [0x1e38] = 0x43e0, [0x1e39] = 0x43e1,
  [0x1e3a] = 0x43e2, [0x1e3b] = 0x43e3, [0x1e3c] = 0x43e4, [0x1e3d] = 0x43e5,
  [0x1e3e] = 0x43e6, [0x1e3f] = 0x43e7, [0x1e40] = 0x43e8, [0x1e41] = 0x43e9,
  [0x1e42] = 0x43ea, [0x1e43] = 0x43eb, [0x1e44] = 0x43ec, [0x1e45] = 0x43ed,
  [0x1e46] = 0x43ee, [0x1e47] = 0x43ef, [0x1e48] = 0x43f0, [0x1e49] = 0x43f1,
  [0x1e4a] = 0x43f2, [0x1e4b] = 0x43f3, [0x1e4c] = 0x43f4, [0x1e4d] = 0x43f5,
  [0x1e4e] = 0x43f6, [0x1e4f] = 0x43f7, [0x1e50] = 0x43f8, [0x1e51] = 0x43f9,
  [0x1e52] = 0x43fa, [0x1e53] = 0x43fb, [0x1e54] = 0x43fc, [0x1e55] = 0x43fd,
  [0x1e56] = 0x43fe, [0x1e57] = 0x43ff, [0x1e58] = 0x4400, [0x1e59] = 0x4401,
  [0x1e5a] = 0x4402, [0x1e5b] = 0x4403, [0x1e5c] = 0x4404, [0x1e5d] = 0x4405,
  [0x1e5e] = 0x4406, [0x1e5f] = 0x4407, [0x1e60] = 0x4408, [0x1e61] = 0x4409,
  [0x1e62] = 0x440a, [0x1e63] = 0x440b, [0x1e64] = 0x440c, [0x1e65] = 0x440d,
  [0x1e66] = 0x440e, [0x1e67] = 0x440f, [0x1e68] = 0x4410, [0x1e69] = 0x4411,
  [0x1e6a] = 0x4412, [0x1e6b] = 0x4413, [0x1e6c] = 0x4414, [0x1e6d] = 0x4415,
  [0x1e6e] = 0x4416, [0x1e6f] = 0x4417, [0x1e70] = 0x4418, [0x1e71] = 0x4419,
  [0x1e72] = 0x441a, [0x1e73] = 0x441b, [0x1e74] = 0x441c, [0x1e75] = 0x441d,
  [0x1e76] = 0x441e, [0x1e77] = 0x441f, [0x1e78] = 0x4420, [0x1e79] = 0x4421,
  [0x1e7a] = 0x4422, [0x1e7b] = 0x4423, [0x1e7c] = 0x4424, [0x1e7d] = 0x4425,
  [0x1e7e] = 0x4426, [0x1e7f] = 0x4427, [0x1e80] = 0x4428, [0x1e81] = 0x4429,
  [0x1e82] = 0x442a, [0x1e83] = 0x442b, [0x1e84] = 0x442c, [0x1e85] = 0x442d,
  [0x1e86] = 0x442e, [0x1e87] = 0x442f, [0x1e88] = 0x4430, [0x1e89] = 0x4431,
  [0x1e8a] = 0x4432, [0x1e8b] = 0x4433, [0x1e8c] = 0x4434, [0x1e8d] = 0x4435,
  [0x1e8e] = 0x4436, [0x1e8f] = 0x4437, [0x1e90] = 0x4438, [0x1e91] = 0x4439,
  [0x1e92] = 0x443a, [0x1e93] = 0x443b, [0x1e94] = 0x443c, [0x1e95] = 0x443d,
  [0x1e96] = 0x443e, [0x1e97] = 0x443f, [0x1e98] = 0x4440, [0x1e99] = 0x4441,
  [0x1e9a] = 0x4442, [0x1e9b] = 0x4443, [0x1e9c] = 0x4444, [0x1e9d] = 0x4445,
  [0x1e9e] = 0x4446, [0x1e9f] = 0x4447, [0x1ea0] = 0x4448, [0x1ea1] = 0x4449,
  [0x1ea2] = 0x444a, [0x1ea3] = 0x444b, [0x1ea4] = 0x444c, [0x1ea5] = 0x444d,
  [0x1ea6] = 0x444e, [0x1ea7] = 0x444f, [0x1ea8] = 0x4450, [0x1ea9] = 0x4451,
  [0x1eaa] = 0x4452, [0x1eab] = 0x4453, [0x1eac] = 0x4454, [0x1ead] = 0x4455,
  [0x1eae] = 0x4456, [0x1eaf] = 0x4457, [0x1eb0] = 0x4458, [0x1eb1] = 0x4459,
  [0x1eb2] = 0x445a, [0x1eb3] = 0x445b, [0x1eb4] = 0x445c, [0x1eb5] = 0x445d,
  [0x1eb6] = 0x445e, [0x1eb7] = 0x445f, [0x1eb8] = 0x4460, [0x1eb9] = 0x4461,
  [0x1eba] = 0x4462, [0x1ebb] = 0x4463, [0x1ebc] = 0x4464, [0x1ebd] = 0x4465,
  [0x1ebe] = 0x4466, [0x1ebf] = 0x4467, [0x1ec0] = 0x4468, [0x1ec1] = 0x4469,
  [0x1ec2] = 0x446a, [0x1ec3] = 0x446b, [0x1ec4] = 0x446c, [0x1ec5] = 0x446d,
  [0x1ec6] = 0x446e, [0x1ec7] = 0x446f, [0x1ec8] = 0x4470, [0x1ec9] = 0x4471,
  [0x1eca] = 0x4472, [0x1ecb] = 0x4473, [0x1ecc] = 0x4474, [0x1ecd] = 0x4475,
  [0x1ece] = 0x4476, [0x1ecf] = 0x4477, [0x1ed0] = 0x4478, [0x1ed1] = 0x4479,
  [0x1ed2] = 0x447a, [0x1ed3] = 0x447b, [0x1ed4] = 0x447c, [0x1ed5] = 0x447d,
  [0x1ed6] = 0x447e, [0x1ed7] = 0x447f, [0x1ed8] = 0x4480, [0x1ed9] = 0x4481,
  [0x1eda] = 0x4482, [0x1edb] = 0x4483, [0x1edc] = 0x4484, [0x1edd] = 0x4485,
  [0x1ede] = 0x4486, [0x1edf] = 0x4487, [0x1ee0] = 0x4488, [0x1ee1] = 0x4489,
  [0x1ee2] = 0x448a, [0x1ee3] = 0x448b, [0x1ee4] = 0x448c, [0x1ee5] = 0x448d,
  [0x1ee6] = 0x448e, [0x1ee7] = 0x448f, [0x1ee8] = 0x4490, [0x1ee9] = 0x4491,
  [0x1eea] = 0x4492, [0x1eeb] = 0x4493, [0x1eec] = 0x4494, [0x1eed] = 0x4495,
  [0x1eee] = 0x4496, [0x1eef] = 0x4497, [0x1ef0] = 0x4498, [0x1ef1] = 0x4499,
  [0x1ef2] = 0x449a, [0x1ef3] = 0x449b, [0x1ef4] = 0x449c, [0x1ef5] = 0x449d,
  [0x1ef6] = 0x449e, [0x1ef7] = 0x449f, [0x1ef8] = 0x44a0, [0x1ef9] = 0x44a1,
  [0x1efa] = 0x44a2, [0x1efb] = 0x44a3, [0x1efc] = 0x44a4, [0x1efd] = 0x44a5,
  [0x1efe] = 0x44a6, [0x1eff] = 0x44a7, [0x1f00] = 0x44a8, [0x1f01] = 0x44a9,
  [0x1f02] = 0x44aa, [0x1f03] = 0x44ab, [0x1f04] = 0x44ac, [0x1f05] = 0x44ad,
  [0x1f06] = 0x44ae, [0x1f07] = 0x44af, [0x1f08] = 0x44b0, [0x1f09] = 0x44b1,
  [0x1f0a] = 0x44b2, [0x1f0b] = 0x44b3, [0x1f0c] = 0x44b4, [0x1f0d] = 0x44b5,
  [0x1f0e] = 0x44b6, [0x1f0f] = 0x44b7, [0x1f10] = 0x44b8, [0x1f11] = 0x44b9,
  [0x1f12] = 0x44ba, [0x1f13] = 0x44bb, [0x1f14] = 0x44bc, [0x1f15] = 0x44bd,
  [0x1f16] = 0x44be, [0x1f17] = 0x44bf, [0x1f18] = 0x44c0, [0x1f19] = 0x44c1,
  [0x1f1a] = 0x44c2, [0x1f1b] = 0x44c3, [0x1f1c] = 0x44c4, [0x1f1d] = 0x44c5,
  [0x1f1e] = 0x44c6, [0x1f1f] = 0x44c7, [0x1f20] = 0x44c8, [0x1f21] = 0x44c9,
  [0x1f22] = 0x44ca, [0x1f23] = 0x44cb, [0x1f24] = 0x44cc, [0x1f25] = 0x44cd,
  [0x1f26] = 0x44ce, [0x1f27] = 0x44cf, [0x1f28] = 0x44d0, [0x1f29] = 0x44d1,
  [0x1f2a] = 0x44d2, [0x1f2b] = 0x44d3, [0x1f2c] = 0x44d4, [0x1f2d] = 0x44d5,
  [0x20a3] = 0x464d, [0x20a4] = 0x464e, [0x20a5] = 0x464f, [0x20a6] = 0x4650,
  [0x20a7] = 0x4651, [0x20a8] = 0x4652, [0x20a9] = 0x4653, [0x20aa] = 0x4654,
  [0x20ab] = 0x4655, [0x20ac] = 0x4656, [0x20ad] = 0x4657, [0x20ae] = 0x4658,
  [0x20af] = 0x4659, [0x20b0] = 0x465a, [0x20b1] = 0x465b, [0x20b2] = 0x465c,
  [0x20b3] = 0x465d, [0x20b4] = 0x465e, [0x20b5] = 0x465f, [0x20b6] = 0x4660,
  [0x20b7] = 0x4662, [0x20b8] = 0x4663, [0x20b9] = 0x4664, [0x20ba] = 0x4665,
  [0x20bb] = 0x4666, [0x20bc] = 0x4667, [0x20bd] = 0x4668, [0x20be] = 0x4669,
  [0x20bf] = 0x466a, [0x20c0] = 0x466b, [0x20c1] = 0x466c, [0x20c2] = 0x466d,
  [0x20c3] = 0x466e, [0x20c4] = 0x466f, [0x20c5] = 0x4670, [0x20c6] = 0x4671,
  [0x20c7] = 0x4672, [0x20c8] = 0x4673, [0x20c9] = 0x4674, [0x20ca] = 0x4675,
  [0x20cb] = 0x4676, [0x20cc] = 0x4677, [0x20cd] = 0x4678, [0x20ce] = 0x4679,
  [0x20cf] = 0x467a, [0x20d0] = 0x467b, [0x20d1] = 0x467c, [0x20d2] = 0x467d,
  [0x20d3] = 0x467e, [0x20d4] = 0x467f, [0x20d5] = 0x4680, [0x20d6] = 0x4681,
  [0x20d7] = 0x4682, [0x20d8] = 0x4683, [0x20d9] = 0x4684, [0x20da] = 0x4685,
  [0x20db] = 0x4686, [0x20dc] = 0x4687, [0x20dd] = 0x4688, [0x20de] = 0x4689,
  [0x20df] = 0x468a, [0x20e0] = 0x468b, [0x20e1] = 0x468c, [0x20e2] = 0x468d,
  [0x20e3] = 0x468e, [0x20e4] = 0x468f, [0x20e5] = 0x4690, [0x20e6] = 0x4691,
  [0x20e7] = 0x4692, [0x20e8] = 0x4693, [0x20e9] = 0x4694, [0x20ea] = 0x4695,
  [0x20eb] = 0x4696, [0x20ec] = 0x4697, [0x20ed] = 0x4698, [0x20ee] = 0x4699,
  [0x20ef] = 0x469a, [0x20f0] = 0x469b, [0x20f1] = 0x469c, [0x20f2] = 0x469d,
  [0x20f3] = 0x469e, [0x20f4] = 0x469f, [0x20f5] = 0x46a0, [0x20f6] = 0x46a1,
  [0x20f7] = 0x46a2, [0x20f8] = 0x46a3, [0x20f9] = 0x46a4, [0x20fa] = 0x46a5,
  [0x20fb] = 0x46a6, [0x20fc] = 0x46a7, [0x20fd] = 0x46a8, [0x20fe] = 0x46a9,
  [0x20ff] = 0x46aa, [0x2100] = 0x46ab, [0x2101] = 0x46ac, [0x2102] = 0x46ad,
  [0x2103] = 0x46ae, [0x2104] = 0x46af, [0x2105] = 0x46b0, [0x2106] = 0x46b1,
  [0x2107] = 0x46b2, [0x2108] = 0x46b3, [0x2109] = 0x46b4, [0x210a] = 0x46b5,
  [0x210b] = 0x46b6, [0x210c] = 0x46b7, [0x210d] = 0x46b8, [0x210e] = 0x46b9,
  [0x210f] = 0x46ba, [0x2110] = 0x46bb, [0x2111] = 0x46bc, [0x2112] = 0x46bd,
  [0x2113] = 0x46be, [0x2114] = 0x46bf, [0x2115] = 0x46c0, [0x2116] = 0x46c1,
  [0x2117] = 0x46c2, [0x2118] = 0x46c3, [0x2119] = 0x46c4, [0x211a] = 0x46c5,
  [0x211b] = 0x46c6, [0x211c] = 0x46c7, [0x211d] = 0x46c8, [0x211e] = 0x46c9,
  [0x211f] = 0x46ca, [0x2120] = 0x46cb, [0x2121] = 0x46cc, [0x2122] = 0x46cd,
  [0x2123] = 0x46ce, [0x2124] = 0x46cf, [0x2125] = 0x46d0, [0x2126] = 0x46d1,
  [0x2127] = 0x46d2, [0x2128] = 0x46d3, [0x2129] = 0x46d4, [0x212a] = 0x46d5,
  [0x212b] = 0x46d6, [0x212c] = 0x46d7, [0x212d] = 0x46d8, [0x212e] = 0x46d9,
  [0x212f] = 0x46da, [0x2130] = 0x46db, [0x2131] = 0x46dc, [0x2132] = 0x46dd,
  [0x2133] = 0x46de, [0x2134] = 0x46df, [0x2135] = 0x46e0, [0x2136] = 0x46e1,
  [0x2137] = 0x46e2, [0x2138] = 0x46e3, [0x2139] = 0x46e4, [0x213a] = 0x46e5,
  [0x213b] = 0x46e6, [0x213c] = 0x46e7, [0x213d] = 0x46e8, [0x213e] = 0x46e9,
  [0x213f] = 0x46ea, [0x2140] = 0x46eb, [0x2141] = 0x46ec, [0x2142] = 0x46ed,
  [0x2143] = 0x46ee, [0x2144] = 0x46ef, [0x2145] = 0x46f0, [0x2146] = 0x46f1,
  [0x2147] = 0x46f2, [0x2148] = 0x46f3, [0x2149] = 0x46f4, [0x214a] = 0x46f5,
  [0x214b] = 0x46f6, [0x214c] = 0x46f7, [0x214d] = 0x46f8, [0x214e] = 0x46f9,
  [0x214f] = 0x46fa, [0x2150] = 0x46fb, [0x2151] = 0x46fc, [0x2152] = 0x46fd,
  [0x2153] = 0x46fe, [0x2154] = 0x46ff, [0x2155] = 0x4700, [0x2156] = 0x4701,
  [0x2157] = 0x4702, [0x2158] = 0x4703, [0x2159] = 0x4704, [0x215a] = 0x4705,
  [0x215b] = 0x4706, [0x215c] = 0x4707, [0x215d] = 0x4708, [0x215e] = 0x4709,
  [0x215f] = 0x470a, [0x2160] = 0x470b, [0x2161] = 0x470c, [0x2162] = 0x470d,
  [0x2163] = 0x470e, [0x2164] = 0x470f, [0x2165] = 0x4710, [0x2166] = 0x4711,
  [0x2167] = 0x4712, [0x2168] = 0x4713, [0x2169] = 0x4714, [0x216a] = 0x4715,
  [0x216b] = 0x4716, [0x216c] = 0x4717, [0x216d] = 0x4718, [0x216e] = 0x4719,
  [0x216f] = 0x471a, [0x2170] = 0x471b, [0x2171] = 0x471c, [0x2172] = 0x471d,
  [0x2173] = 0x471e, [0x2174] = 0x471f, [0x2175] = 0x4720, [0x2176] = 0x4721,
  [0x2177] = 0x4722, [0x2178] = 0x4724, [0x2179] = 0x4725, [0x217a] = 0x4726,
  [0x217b] = 0x4727, [0x217c] = 0x4728, [0x217d] = 0x472a, [0x217e] = 0x472b,
  [0x217f] = 0x472c, [0x2180] = 0x472d, [0x2181] = 0x472e, [0x2182] = 0x472f,
  [0x2183] = 0x4730, [0x2184] = 0x4731, [0x2185] = 0x4732, [0x2186] = 0x4733,
  [0x2187] = 0x4734, [0x2188] = 0x4735, [0x2189] = 0x4736, [0x218a] = 0x4737,
  [0x218b] = 0x4738, [0x218c] = 0x4739, [0x218d] = 0x473a, [0x218e] = 0x473b,
  [0x218f] = 0x473c, [0x2190] = 0x473d, [0x2191] = 0x473e, [0x2192] = 0x473f,
  [0x2193] = 0x4740, [0x2194] = 0x4741, [0x2195] = 0x4742, [0x2196] = 0x4743,
  [0x2197] = 0x4744, [0x2198] = 0x4745, [0x2199] = 0x4746, [0x219a] = 0x4747,
  [0x219b] = 0x4748, [0x219c] = 0x4749, [0x219d] = 0x474a, [0x219e] = 0x474b,
  [0x219f] = 0x474c, [0x21a0] = 0x474d, [0x21a1] = 0x474e, [0x21a2] = 0x474f,
  [0x21a3] = 0x4750, [0x21a4] = 0x4751, [0x21a5] = 0x4752, [0x21a6] = 0x4753,
  [0x21a7] = 0x4754, [0x21a8] = 0x4755, [0x21a9] = 0x4756, [0x21aa] = 0x4757,
  [0x21ab] = 0x4758, [0x21ac] = 0x4759, [0x21ad] = 0x475a, [0x21ae] = 0x475b,
  [0x21af] = 0x475c, [0x21b0] = 0x475d, [0x21b1] = 0x475e, [0x21b2] = 0x475f,
  [0x21b3] = 0x4760, [0x21b4] = 0x4761, [0x21b5] = 0x4762, [0x21b6] = 0x4763,
  [0x21b7] = 0x4764, [0x21b8] = 0x4765, [0x21b9] = 0x4766, [0x21ba] = 0x4767,
  [0x21bb] = 0x4768, [0x21bc] = 0x4769, [0x21bd] = 0x476a, [0x21be] = 0x476b,
  [0x21bf] = 0x476c, [0x21c0] = 0x476d, [0x21c1] = 0x476e, [0x21c2] = 0x476f,
  [0x21c3] = 0x4770, [0x21c4] = 0x4771, [0x21c5] = 0x4772, [0x21c6] = 0x4773,
  [0x21c7] = 0x4774, [0x21c8] = 0x4775, [0x21c9] = 0x4776, [0x21ca] = 0x4777,
  [0x21cb] = 0x4778, [0x21cc] = 0x4779, [0x21cd] = 0x477a, [0x21ce] = 0x477b,
  [0x21cf] = 0x477d, [0x21d0] = 0x477e, [0x21d1] = 0x477f, [0x21d2] = 0x4780,
  [0x21d3] = 0x4781, [0x21d4] = 0x4782, [0x21d5] = 0x4783, [0x21d6] = 0x4784,
  [0x21d7] = 0x4785, [0x21d8] = 0x4786, [0x21d9] = 0x4787, [0x21da] = 0x4788,
  [0x21db] = 0x4789, [0x21dc] = 0x478a, [0x21dd] = 0x478b, [0x21de] = 0x478c,
  [0x2398] = 0x4948, [0x2399] = 0x4949, [0x239a] = 0x494a, [0x239b] = 0x494b,
  [0x239c] = 0x494c, [0x239d] = 0x494d, [0x239e] = 0x494e, [0x239f] = 0x494f,
  [0x23a0] = 0x4950, [0x23a1] = 0x4951, [0x23a2] = 0x4952, [0x23a3] = 0x4953,
  [0x23a4] = 0x4954, [0x23a5] = 0x4955, [0x23a6] = 0x4956, [0x23a7] = 0x4957,
  [0x23a8] = 0x4958, [0x23a9] = 0x4959, [0x23aa] = 0x495a, [0x23ab] = 0x495b,
  [0x23ac] = 0x495c, [0x23ad] = 0x495d, [0x23ae] = 0x495e, [0x23af] = 0x495f,
  [0x23b0] = 0x4960, [0x23b1] = 0x4961, [0x23b2] = 0x4962, [0x23b3] = 0x4963,
  [0x23b4] = 0x4964, [0x23b5] = 0x4965, [0x23b6] = 0x4966, [0x23b7] = 0x4967,
  [0x23b8] = 0x4968, [0x23b9] = 0x4969, [0x23ba] = 0x496a, [0x23bb] = 0x496b,
  [0x23bc] = 0x496c, [0x23bd] = 0x496d, [0x23be] = 0x496e, [0x23bf] = 0x496f,
  [0x23c0] = 0x4970, [0x23c1] = 0x4971, [0x23c2] = 0x4972, [0x23c3] = 0x4973,
  [0x23c4] = 0x4974, [0x23c5] = 0x4975, [0x23c6] = 0x4976, [0x23c7] = 0x4977,
  [0x23c8] = 0x4978, [0x23c9] = 0x4979, [0x23ca] = 0x497b, [0x23cb] = 0x497c,
  [0x23cc] = 0x497e, [0x23cd] = 0x497f, [0x23ce] = 0x4980, [0x23cf] = 0x4981,
  [0x23d0] = 0x4984, [0x23d1] = 0x4987, [0x23d2] = 0x4988, [0x23d3] = 0x4989,
  [0x23d4] = 0x498a, [0x23d5] = 0x498b, [0x23d6] = 0x498c, [0x23d7] = 0x498d,
  [0x23d8] = 0x498e, [0x23d9] = 0x498f, [0x23da] = 0x4990, [0x23db] = 0x4991,
  [0x23dc] = 0x4992, [0x23dd] = 0x4993, [0x23de] = 0x4994, [0x23df] = 0x4995,
  [0x23e0] = 0x4996, [0x23e1] = 0x4997, [0x23e2] = 0x4998, [0x23e3] = 0x4999,
  [0x23e4] = 0x499a, [0x23e5] = 0x499c, [0x23e6] = 0x499d, [0x23e7] = 0x499e,
  [0x23e8] = 0x49a0, [0x23e9] = 0x49a1, [0x23ea] = 0x49a2, [0x23eb] = 0x49a3,
  [0x23ec] = 0x49a4, [0x23ed] = 0x49a5, [0x23ee] = 0x49a6, [0x23ef] = 0x49a7,
  [0x23f0] = 0x49a8, [0x23f1] = 0x49a9, [0x23f2] = 0x49aa, [0x23f3] = 0x49ab,
  [0x23f4] = 0x49ac, [0x23f5] = 0x49ad, [0x23f6] = 0x49ae, [0x23f7] = 0x49af,
  [0x23f8] = 0x49b0, [0x23f9] = 0x49b1, [0x23fa] = 0x49b2, [0x23fb] = 0x49b3,
  [0x23fc] = 0x49b4, [0x23fd] = 0x49b5, [0x26bd] = 0x4c78, [0x26be] = 0x4c79,
  [0x26bf] = 0x4c7a, [0x26c0] = 0x4c7b, [0x26c1] = 0x4c7c, [0x26c2] = 0x4c7d,
  [0x26c3] = 0x4c7e, [0x26c4] = 0x4c7f, [0x26c5] = 0x4c80, [0x26c6] = 0x4c81,
  [0x26c7] = 0x4c82, [0x26c8] = 0x4c83, [0x26c9] = 0x4c84, [0x26ca] = 0x4c85,
  [0x26cb] = 0x4c86, [0x26cc] = 0x4c87, [0x26cd] = 0x4c88, [0x26ce] = 0x4c89,
  [0x26cf] = 0x4c8a, [0x26d0] = 0x4c8b, [0x26d1] = 0x4c8c, [0x26d2] = 0x4c8d,
  [0x26d3] = 0x4c8e, [0x26d4] = 0x4c8f, [0x26d5] = 0x4c90, [0x26d6] = 0x4c91,
  [0x26d7] = 0x4c92, [0x26d8] = 0x4c93, [0x26d9] = 0x4c94, [0x26da] = 0x4c95,
  [0x26db] = 0x4c96, [0x26dc] = 0x4c97, [0x26dd] = 0x4c98, [0x26de] = 0x4c99,
  [0x26df] = 0x4c9a, [0x26e0] = 0x4c9b, [0x26e1] = 0x4c9c, [0x26e2] = 0x4c9d,
  [0x26e3] = 0x4c9e, [0x26e4] = 0x4ca4, [0x26e5] = 0x4ca5, [0x26e6] = 0x4ca6,
  [0x26e7] = 0x4ca7, [0x26e8] = 0x4ca8, [0x26e9] = 0x4ca9, [0x26ea] = 0x4caa,
  [0x26eb] = 0x4cab, [0x26ec] = 0x4cac, [0x26ed] = 0x4cad, [0x26ee] = 0x4cae,
  [0x26ef] = 0x4caf, [0x26f0] = 0x4cb0, [0x26f1] = 0x4cb1, [0x26f2] = 0x4cb2,
  [0x26f3] = 0x4cb3, [0x26f4] = 0x4cb4, [0x26f5] = 0x4cb5, [0x26f6] = 0x4cb6,
  [0x26f7] = 0x4cb7, [0x26f8] = 0x4cb8, [0x26f9] = 0x4cb9, [0x26fa] = 0x4cba,
  [0x26fb] = 0x4cbb, [0x26fc] = 0x4cbc, [0x26fd] = 0x4cbd, [0x26fe] = 0x4cbe,
  [0x26ff] = 0x4cbf, [0x2700] = 0x4cc0, [0x2701] = 0x4cc1, [0x2702] = 0x4cc2,
  [0x2703] = 0x4cc3, [0x2704] = 0x4cc4, [0x2705] = 0x4cc5, [0x2706] = 0x4cc6,
  [0x2707] = 0x4cc7, [0x2708] = 0x4cc8, [0x2709] = 0x4cc9, [0x270a] = 0x4cca,
  [0x270b] = 0x4ccb, [0x270c] = 0x4ccc, [0x270d] = 0x4ccd, [0x270e] = 0x4cce,
  [0x270f] = 0x4ccf, [0x2710] = 0x4cd0, [0x2711] = 0x4cd1, [0x2712] = 0x4cd2,
  [0x2713] = 0x4cd3, [0x2714] = 0x4cd4, [0x2715] = 0x4cd5, [0x2716] = 0x4cd6,
  [0x2717] = 0x4cd7, [0x2718] = 0x4cd8, [0x2719] = 0x4cd9, [0x271a] = 0x4cda,
  [0x271b] = 0x4cdb, [0x271c] = 0x4cdc, [0x271d] = 0x4cdd, [0x271e] = 0x4cde,
  [0x271f] = 0x4cdf, [0x2720] = 0x4ce0, [0x2721] = 0x4ce1, [0x2722] = 0x4ce2,
  [0x2723] = 0x4ce3, [0x2724] = 0x4ce4, [0x2725] = 0x4ce5, [0x2726] = 0x4ce6,
  [0x2727] = 0x4ce7, [0x2728] = 0x4ce8, [0x2729] = 0x4ce9, [0x272a] = 0x4cea,
  [0x272b] = 0x4ceb, [0x272c] = 0x4cec, [0x272d] = 0x4ced, [0x272e] = 0x4cee,
  [0x272f] = 0x4cef, [0x2730] = 0x4cf0, [0x2731] = 0x4cf1, [0x2732] = 0x4cf2,
  [0x2733] = 0x4cf3, [0x2734] = 0x4cf4, [0x2735] = 0x4cf5, [0x2736] = 0x4cf6,
  [0x2737] = 0x4cf7, [0x2738] = 0x4cf8, [0x2739] = 0x4cf9, [0x273a] = 0x4cfa,
  [0x273b] = 0x4cfb, [0x273c] = 0x4cfc, [0x273d] = 0x4cfd, [0x273e] = 0x4cfe,
  [0x273f] = 0x4cff, [0x2740] = 0x4d00, [0x2741] = 0x4d01, [0x2742] = 0x4d02,
  [0x2743] = 0x4d03, [0x2744] = 0x4d04, [0x2745] = 0x4d05, [0x2746] = 0x4d06,
  [0x2747] = 0x4d07, [0x2748] = 0x4d08, [0x2749] = 0x4d09, [0x274a] = 0x4d0a,
  [0x274b] = 0x4d0b, [0x274c] = 0x4d0c, [0x274d] = 0x4d0d, [0x274e] = 0x4d0e,
  [0x274f] = 0x4d0f, [0x2750] = 0x4d10, [0x2751] = 0x4d11, [0x2752] = 0x4d12,
  [0x2753] = 0x4d1a, [0x2754] = 0x4d1b, [0x2755] = 0x4d1c, [0x2756] = 0x4d1d,
  [0x2757] = 0x4d1e, [0x2758] = 0x4d1f, [0x2759] = 0x4d20, [0x275a] = 0x4d21,
  [0x275b] = 0x4d22, [0x275c] = 0x4d23, [0x275d] = 0x4d24, [0x275e] = 0x4d25,
  [0x275f] = 0x4d26, [0x2760] = 0x4d27, [0x2761] = 0x4d28, [0x2762] = 0x4d29,
  [0x2763] = 0x4d2a, [0x2764] = 0x4d2b, [0x2765] = 0x4d2c, [0x2766] = 0x4d2d,
  [0x2767] = 0x4d2e, [0x2768] = 0x4d2f, [0x2769] = 0x4d30, [0x276a] = 0x4d31,
  [0x276b] = 0x4d32, [0x276c] = 0x4d33, [0x276d] = 0x4d34, [0x276e] = 0x4d35,
  [0x276f] = 0x4d36, [0x2770] = 0x4d37, [0x2771] = 0x4d38, [0x2772] = 0x4d39,
  [0x2773] = 0x4d3a, [0x2774] = 0x4d3b, [0x2775] = 0x4d3c, [0x2776] = 0x4d3d,
  [0x2777] = 0x4d3e, [0x2778] = 0x4d3f, [0x2779] = 0x4d40, [0x277a] = 0x4d41,
  [0x277b] = 0x4d42, [0x277c] = 0x4d43, [0x277d] = 0x4d44, [0x277e] = 0x4d45,
  [0x277f] = 0x4d46, [0x2780] = 0x4d47, [0x2781] = 0x4d48, [0x2782] = 0x4d49,
  [0x2783] = 0x4d4a, [0x2784] = 0x4d4b, [0x2785] = 0x4d4c, [0x2786] = 0x4d4d,
  [0x2787] = 0x4d4e, [0x2788] = 0x4d4f, [0x2789] = 0x4d50, [0x278a] = 0x4d51,
  [0x278b] = 0x4d52, [0x278c] = 0x4d53, [0x278d] = 0x4d54, [0x278e] = 0x4d55,
  [0x278f] = 0x4d56, [0x2790] = 0x4d57, [0x2791] = 0x4d58, [0x2792] = 0x4d59,
  [0x2793] = 0x4d5a, [0x2794] = 0x4d5b, [0x2795] = 0x4d5c, [0x2796] = 0x4d5d,
  [0x2797] = 0x4d5e, [0x2798] = 0x4d5f, [0x2799] = 0x4d60, [0x279a] = 0x4d61,
  [0x279b] = 0x4d62, [0x279c] = 0x4d63, [0x279d] = 0x4d64, [0x279e] = 0x4d65,
  [0x279f] = 0x4d66, [0x27a0] = 0x4d67, [0x27a1] = 0x4d68, [0x27a2] = 0x4d69,
  [0x27a3] = 0x4d6a, [0x27a4] = 0x4d6b, [0x27a5] = 0x4d6c, [0x27a6] = 0x4d6d,
  [0x27a7] = 0x4d6e, [0x27a8] = 0x4d6f, [0x27a9] = 0x4d70, [0x27aa] = 0x4d71,
  [0x27ab] = 0x4d72, [0x27ac] = 0x4d73, [0x27ad] = 0x4d74, [0x27ae] = 0x4d75,
  [0x27af] = 0x4d76, [0x27b0] = 0x4d77, [0x27b1] = 0x4d78, [0x27b2] = 0x4d79,
  [0x27b3] = 0x4d7a, [0x27b4] = 0x4d7b, [0x27b5] = 0x4d7c, [0x27b6] = 0x4d7d,
  [0x27b7] = 0x4d7e, [0x27b8] = 0x4d7f, [0x27b9] = 0x4d80, [0x27ba] = 0x4d81,
  [0x27bb] = 0x4d82, [0x27bc] = 0x4d83, [0x27bd] = 0x4d84, [0x27be] = 0x4d85,
  [0x27bf] = 0x4d86, [0x27c0] = 0x4d87, [0x27c1] = 0x4d88, [0x27c2] = 0x4d89,
  [0x27c3] = 0x4d8a, [0x27c4] = 0x4d8b, [0x27c5] = 0x4d8c, [0x27c6] = 0x4d8d,
  [0x27c7] = 0x4d8e, [0x27c8] = 0x4d8f, [0x27c9] = 0x4d90, [0x27ca] = 0x4d91,
  [0x27cb] = 0x4d92, [0x27cc] = 0x4d93, [0x27cd] = 0x4d94, [0x27ce] = 0x4d95,
  [0x27cf] = 0x4d96, [0x27d0] = 0x4d97, [0x27d1] = 0x4d98, [0x27d2] = 0x4d99,
  [0x27d3] = 0x4d9a, [0x27d4] = 0x4d9b, [0x27d5] = 0x4d9c, [0x27d6] = 0x4d9d,
  [0x27d7] = 0x4d9e, [0x27d8] = 0x4d9f, [0x27d9] = 0x4da0, [0x27da] = 0x4da1,
  [0x27db] = 0x4da2, [0x27dc] = 0x4da3, [0x27dd] = 0x4da4, [0x27de] = 0x4da5,
  [0x27df] = 0x4da6, [0x27e0] = 0x4da7, [0x27e1] = 0x4da8, [0x27e2] = 0x4da9,
  [0x27e3] = 0x4daa, [0x27e4] = 0x4dab, [0x27e5] = 0x4dac, [0x27e6] = 0x4dad,
  [0x27e7] = 0x4daf, [0x27e8] = 0x4db0, [0x27e9] = 0x4db1, [0x27ea] = 0x4db2,
  [0x27eb] = 0x4db3, [0x27ec] = 0x4db4, [0x27ed] = 0x4db5, [0x27f8] = 0x4dc0,
  [0x27f9] = 0x4dc1, [0x27fa] = 0x4dc2, [0x27fb] = 0x4dc3, [0x27fc] = 0x4dc4,
  [0x27fd] = 0x4dc5, [0x27fe] = 0x4dc6, [0x27ff] = 0x4dc7, [0x2800] = 0x4dc8,
  [0x2801] = 0x4dc9, [0x2802] = 0x4dca, [0x2803] = 0x4dcb, [0x2804] = 0x4dcc,
  [0x2805] = 0x4dcd, [0x2806] = 0x4dce, [0x2807] = 0x4dcf, [0x2808] = 0x4dd0,
  [0x2809] = 0x4dd1, [0x280a] = 0x4dd2, [0x280b] = 0x4dd3, [0x280c] = 0x4dd4,
  [0x280d] = 0x4dd5, [0x280e] = 0x4dd6, [0x280f] = 0x4dd7, [0x2810] = 0x4dd8,
  [0x2811] = 0x4dd9, [0x2812] = 0x4dda, [0x2813] = 0x4ddb, [0x2814] = 0x4ddc,
  [0x2815] = 0x4ddd, [0x2816] = 0x4dde, [0x2817] = 0x4ddf, [0x2818] = 0x4de0,
  [0x2819] = 0x4de1, [0x281a] = 0x4de2, [0x281b] = 0x4de3, [0x281c] = 0x4de4,
  [0x281d] = 0x4de5, [0x281e] = 0x4de6, [0x281f] = 0x4de7, [0x2820] = 0x4de8,
  [0x2821] = 0x4de9, [0x2822] = 0x4dea, [0x2823] = 0x4deb, [0x2824] = 0x4dec,
  [0x2825] = 0x4ded, [0x2826] = 0x4dee, [0x2827] = 0x4def, [0x2828] = 0x4df0,
  [0x2829] = 0x4df1, [0x282a] = 0x4df2, [0x282b] = 0x4df3, [0x282c] = 0x4df4,
  [0x282d] = 0x4df5, [0x282e] = 0x4df6, [0x282f] = 0x4df7, [0x2830] = 0x4df8,
  [0x2831] = 0x4df9, [0x2832] = 0x4dfa, [0x2833] = 0x4dfb, [0x2834] = 0x4dfc,
  [0x2835] = 0x4dfd, [0x2836] = 0x4dfe, [0x2837] = 0x4dff, [0x2838] = 0xe76c,
  [0x2839] = 0xe7c8, [0x283a] = 0xe7e7, [0x283b] = 0xe7e8, [0x283c] = 0xe7e9,
  [0x283d] = 0xe7ea, [0x283e] = 0xe7eb, [0x283f] = 0xe7ec, [0x2840] = 0xe7ed,
  [0x2841] = 0xe7ee, [0x2842] = 0xe7ef, [0x2843] = 0xe7f0, [0x2844] = 0xe7f1,
  [0x2845] = 0xe7f2, [0x2846] = 0xe7f3, [0x2847] = 0xe815, [0x2848] = 0xe819,
  [0x2849] = 0xe81a, [0x284a] = 0xe81b, [0x284b] = 0xe81c, [0x284c] = 0xe81d,
  [0x284d] = 0xe81f, [0x284e] = 0xe820, [0x284f] = 0xe821, [0x2850] = 0xe822,
  [0x2851] = 0xe823, [0x2852] = 0xe824, [0x2853] = 0xe825, [0x2854] = 0xe827,
  [0x2855] = 0xe828, [0x2856] = 0xe829, [0x2857] = 0xe82a, [0x2858] = 0xe82d,
  [0x2859] = 0xe82e, [0x285a] = 0xe82f, [0x285b] = 0xe830, [0x285c] = 0xe833,
  [0x285d] = 0xe834, [0x285e] = 0xe835, [0x285f] = 0xe836, [0x2860] = 0xe837,
  [0x2861] = 0xe838, [0x2862] = 0xe839, [0x2863] = 0xe83a, [0x2864] = 0xe83c,
  [0x2865] = 0xe83d, [0x2866] = 0xe83e, [0x2867] = 0xe83f, [0x2868] = 0xe840,
  [0x2869] = 0xe841, [0x286a] = 0xe842, [0x286b] = 0xe844, [0x286c] = 0xe845,
  [0x286d] = 0xe846, [0x286e] = 0xe847, [0x286f] = 0xe848, [0x2870] = 0xe849,
  [0x2871] = 0xe84a, [0x2872] = 0xe84b, [0x2873] = 0xe84c, [0x2874] = 0xe84d,
  [0x2875] = 0xe84e, [0x2876] = 0xe84f, [0x2877] = 0xe850, [0x2878] = 0xe851,
  [0x2879] = 0xe852, [0x287a] = 0xe853, [0x287b] = 0xe856, [0x287c] = 0xe857,
  [0x287d] = 0xe858, [0x287e] = 0xe859, [0x287f] = 0xe85a, [0x2880] = 0xe85b,
  [0x2881] = 0xe85c, [0x2882] = 0xe85d, [0x2883] = 0xe85e, [0x2884] = 0xe85f,
  [0x2885] = 0xe860, [0x2886] = 0xe861, [0x2887] = 0xe862, [0x2888] = 0xe863,
  [0x2889] = 0xf92d, [0x288a] = 0xf92e, [0x288b] = 0xf92f, [0x288c] = 0xf930,
  [0x288d] = 0xf931, [0x288e] = 0xf932, [0x288f] = 0xf933, [0x2890] = 0xf934,
  [0x2891] = 0xf935, [0x2892] = 0xf936, [0x2893] = 0xf937, [0x2894] = 0xf938,
  [0x2895] = 0xf939, [0x2896] = 0xf93a, [0x2897] = 0xf93b, [0x2898] = 0xf93c,
  [0x2899] = 0xf93d, [0x289a] = 0xf93e, [0x289b] = 0xf93f, [0x289c] = 0xf940,
  [0x289d] = 0xf941, [0x289e] = 0xf942, [0x289f] = 0xf943, [0x28a0] = 0xf944,
  [0x28a1] = 0xf945, [0x28a2] = 0xf946, [0x28a3] = 0xf947, [0x28a4] = 0xf948,
  [0x28a5] = 0xf949, [0x28a6] = 0xf94a, [0x28a7] = 0xf94b, [0x28a8] = 0xf94c,
  [0x28a9] = 0xf94d, [0x28aa] = 0xf94e, [0x28ab] = 0xf94f, [0x28ac] = 0xf950,
  [0x28ad] = 0xf951, [0x28ae] = 0xf952, [0x28af] = 0xf953, [0x28b0] = 0xf954,
  [0x28b1] = 0xf955, [0x28b2] = 0xf956, [0x28b3] = 0xf957, [0x28b4] = 0xf958,
  [0x28b5] = 0xf959, [0x28b6] = 0xf95a, [0x28b7] = 0xf95b, [0x28b8] = 0xf95c,
  [0x28b9] = 0xf95d, [0x28ba] = 0xf95e, [0x28bb] = 0xf95f, [0x28bc] = 0xf960,
  [0x28bd] = 0xf961, [0x28be] = 0xf962, [0x28bf] = 0xf963, [0x28c0] = 0xf964,
  [0x28c1] = 0xf965, [0x28c2] = 0xf966, [0x28c3] = 0xf967, [0x28c4] = 0xf968,
  [0x28c5] = 0xf969, [0x28c6] = 0xf96a, [0x28c7] = 0xf96b, [0x28c8] = 0xf96c,
  [0x28c9] = 0xf96d, [0x28ca] = 0xf96e, [0x28cb] = 0xf96f, [0x28cc] = 0xf970,
  [0x28cd] = 0xf971, [0x28ce] = 0xf972, [0x28cf] = 0xf973, [0x28d0] = 0xf974,
  [0x28d1] = 0xf975, [0x28d2] = 0xf976, [0x28d3] = 0xf977, [0x28d4] = 0xf978,
  [0x28d5] = 0xf97a, [0x28d6] = 0xf97b, [0x28d7] = 0xf97c, [0x28d8] = 0xf97d,
  [0x28d9] = 0xf97e, [0x28da] = 0xf97f, [0x28db] = 0xf980, [0x28dc] = 0xf981,
  [0x28dd] = 0xf982, [0x28de] = 0xf983, [0x28df] = 0xf984, [0x28e0] = 0xf985,
  [0x28e1] = 0xf986, [0x28e2] = 0xf987, [0x28e3] = 0xf988, [0x28e4] = 0xf989,
  [0x28e5] = 0xf98a, [0x28e6] = 0xf98b, [0x28e7] = 0xf98c, [0x28e8] = 0xf98d,
  [0x28e9] = 0xf98e, [0x28ea] = 0xf98f, [0x28eb] = 0xf990, [0x28ec] = 0xf991,
  [0x28ed] = 0xf992, [0x28ee] = 0xf993, [0x28ef] = 0xf994, [0x28f0] = 0xf996,
  [0x28f1] = 0xf997, [0x28f2] = 0xf998, [0x28f3] = 0xf999, [0x28f4] = 0xf99a,
  [0x28f5] = 0xf99b, [0x28f6] = 0xf99c, [0x28f7] = 0xf99d, [0x28f8] = 0xf99e,
  [0x28f9] = 0xf99f, [0x28fa] = 0xf9a0, [0x28fb] = 0xf9a1, [0x28fc] = 0xf9a2,
  [0x28fd] = 0xf9a3, [0x28fe] = 0xf9a4, [0x28ff] = 0xf9a5, [0x2900] = 0xf9a6,
  [0x2901] = 0xf9a7, [0x2902] = 0xf9a8, [0x2903] = 0xf9a9, [0x2904] = 0xf9aa,
  [0x2905] = 0xf9ab, [0x2906] = 0xf9ac, [0x2907] = 0xf9ad, [0x2908] = 0xf9ae,
  [0x2909] = 0xf9af, [0x290a] = 0xf9b0, [0x290b] = 0xf9b1, [0x290c] = 0xf9b2,
  [0x290d] = 0xf9b3, [0x290e] = 0xf9b4, [0x290f] = 0xf9b5, [0x2910] = 0xf9b6,
  [0x2911] = 0xf9b7, [0x2912] = 0xf9b8, [0x2913] = 0xf9b9, [0x2914] = 0xf9ba,
  [0x2915] = 0xf9bb, [0x2916] = 0xf9bc, [0x2917] = 0xf9bd, [0x2918] = 0xf9be,
  [0x2919] = 0xf9bf, [0x291a] = 0xf9c0, [0x291b] = 0xf9c1, [0x291c] = 0xf9c2,
  [0x291d] = 0xf9c3, [0x291e] = 0xf9c4, [0x291f] = 0xf9c5, [0x2920] = 0xf9c6,
  [0x2921] = 0xf9c7, [0x2922] = 0xf9c8, [0x2923] = 0xf9c9, [0x2924] = 0xf9ca,
  [0x2925] = 0xf9cb, [0x2926] = 0xf9cc, [0x2927] = 0xf9cd, [0x2928] = 0xf9ce,
  [0x2929] = 0xf9cf, [0x292a] = 0xf9d0, [0x292b] = 0xf9d1, [0x292c] = 0xf9d2,
  [0x292d] = 0xf9d3, [0x292e] = 0xf9d4, [0x292f] = 0xf9d5, [0x2930] = 0xf9d6,
  [0x2931] = 0xf9d7, [0x2932] = 0xf9d8, [0x2933] = 0xf9d9, [0x2934] = 0xf9da,
  [0x2935] = 0xf9db, [0x2936] = 0xf9dc, [0x2937] = 0xf9dd, [0x2938] = 0xf9de,
  [0x2939] = 0xf9df, [0x293a] = 0xf9e0, [0x293b] = 0xf9e1, [0x293c] = 0xf9e2,
  [0x293d] = 0xf9e3, [0x293e] = 0xf9e4, [0x293f] = 0xf9e5, [0x2940] = 0xf9e6,
  [0x2941] = 0xf9e8, [0x2942] = 0xf9e9, [0x2943] = 0xf9ea, [0x2944] = 0xf9eb,
  [0x2945] = 0xf9ec, [0x2946] = 0xf9ed, [0x2947] = 0xf9ee, [0x2948] = 0xf9ef,
  [0x2949] = 0xf9f0, [0x294a] = 0xf9f2, [0x294b] = 0xf9f3, [0x294c] = 0xf9f4,
  [0x294d] = 0xf9f5, [0x294e] = 0xf9f6, [0x294f] = 0xf9f7, [0x2950] = 0xf9f8,
  [0x2951] = 0xf9f9, [0x2952] = 0xf9fa, [0x2953] = 0xf9fb, [0x2954] = 0xf9fc,
  [0x2955] = 0xf9fd, [0x2956] = 0xf9fe, [0x2957] = 0xf9ff, [0x2958] = 0xfa00,
  [0x2959] = 0xfa01, [0x295a] = 0xfa02, [0x295b] = 0xfa03, [0x295c] = 0xfa04,
  [0x295d] = 0xfa05, [0x295e] = 0xfa06, [0x295f] = 0xfa07, [0x2960] = 0xfa08,
  [0x2961] = 0xfa09, [0x2962] = 0xfa0a, [0x2963] = 0xfa0b, [0x2964] = 0xfa10,
  [0x2965] = 0xfa12, [0x2966] = 0xfa15, [0x2967] = 0xfa16, [0x2968] = 0xfa17,
  [0x2969] = 0xfa19, [0x296a] = 0xfa1a, [0x296b] = 0xfa1b, [0x296c] = 0xfa1c,
  [0x296d] = 0xfa1d, [0x296e] = 0xfa1e, [0x296f] = 0xfa22, [0x2970] = 0xfa25,
  [0x2971] = 0xfa26, [0x2d78] = 0xfe32, [0x2d79] = 0xfe45, [0x2d7a] = 0xfe46,
  [0x2d7b] = 0xfe47, [0x2d7c] = 0xfe48, [0x2d7e] = 0xfe58, [0x2d84] = 0xfe70,
  [0x2d85] = 0xfe71, [0x2d86] = 0xfe72, [0x2d87] = 0xfe73, [0x2d88] = 0xfe74,
  [0x2d8a] = 0xfe76, [0x2d8b] = 0xfe77, [0x2d8c] = 0xfe78, [0x2d8d] = 0xfe79,
  [0x2d8e] = 0xfe7a, [0x2d8f] = 0xfe7b, [0x2d90] = 0xfe7c, [0x2d91] = 0xfe7d,
  [0x2d92] = 0xfe7e, [0x2d93] = 0xfe7f, [0x2d94] = 0xfe80, [0x2d95] = 0xfe81,
  [0x2d96] = 0xfe82, [0x2d97] = 0xfe83, [0x2d98] = 0xfe84, [0x2d99] = 0xfe85,
  [0x2d9a] = 0xfe86, [0x2d9b] = 0xfe87, [0x2d9c] = 0xfe88, [0x2d9d] = 0xfe89,
  [0x2d9e] = 0xfe8a, [0x2d9f] = 0xfe8b, [0x2da0] = 0xfe8c, [0x2da1] = 0xfe8d,
  [0x2da2] = 0xfe8e, [0x2da3] = 0xfe8f, [0x2da4] = 0xfe90, [0x2da5] = 0xfe91,
  [0x2da6] = 0xfe92, [0x2da7] = 0xfe93, [0x2da8] = 0xfe94, [0x2da9] = 0xfe95,
  [0x2daa] = 0xfe96, [0x2dab] = 0xfe97, [0x2dac] = 0xfe98, [0x2dad] = 0xfe99,
  [0x2dae] = 0xfe9a, [0x2daf] = 0xfe9b, [0x2db0] = 0xfe9c, [0x2db1] = 0xfe9d,
  [0x2db2] = 0xfe9e, [0x2db3] = 0xfe9f, [0x2db4] = 0xfea0, [0x2db5] = 0xfea1,
  [0x2db6] = 0xfea2, [0x2db7] = 0xfea3, [0x2db8] = 0xfea4, [0x2db9] = 0xfea5,
  [0x2dba] = 0xfea6, [0x2dbb] = 0xfea7, [0x2dbc] = 0xfea8, [0x2dbd] = 0xfea9,
  [0x2dbe] = 0xfeaa, [0x2dbf] = 0xfeab, [0x2dc0] = 0xfeac, [0x2dc1] = 0xfead,
  [0x2dc2] = 0xfeae, [0x2dc3] = 0xfeaf, [0x2dc4] = 0xfeb0, [0x2dc5] = 0xfeb1,
  [0x2dc6] = 0xfeb2, [0x2dc7] = 0xfeb3, [0x2dc8] = 0xfeb4, [0x2dc9] = 0xfeb5,
  [0x2dca] = 0xfeb6, [0x2dcb] = 0xfeb7, [0x2dcc] = 0xfeb8, [0x2dcd] = 0xfeb9,
  [0x2dce] = 0xfeba, [0x2dcf] = 0xfebb, [0x2dd0] = 0xfebc, [0x2dd1] = 0xfebd,
  [0x2dd2] = 0xfebe, [0x2dd3] = 0xfebf, [0x2dd4] = 0xfec0, [0x2dd5] = 0xfec1,
  [0x2dd6] = 0xfec2, [0x2dd7] = 0xfec3, [0x2dd8] = 0xfec4, [0x2dd9] = 0xfec5,
  [0x2dda] = 0xfec6, [0x2ddb] = 0xfec7, [0x2ddc] = 0xfec8, [0x2ddd] = 0xfec9,
  [0x2dde] = 0xfeca, [0x2ddf] = 0xfecb, [0x2de0] = 0xfecc, [0x2de1] = 0xfecd,
  [0x2de2] = 0xfece, [0x2de3] = 0xfecf, [0x2de4] = 0xfed0, [0x2de5] = 0xfed1,
  [0x2de6] = 0xfed2, [0x2de7] = 0xfed3, [0x2de8] = 0xfed4, [0x2de9] = 0xfed5,
  [0x2dea] = 0xfed6, [0x2deb] = 0xfed7, [0x2dec] = 0xfed8, [0x2ded] = 0xfed9,
  [0x2dee] = 0xfeda, [0x2def] = 0xfedb, [0x2df0] = 0xfedc, [0x2df1] = 0xfedd,
  [0x2df2] = 0xfede, [0x2df3] = 0xfedf, [0x2df4] = 0xfee0, [0x2df5] = 0xfee1,
  [0x2df6] = 0xfee2, [0x2df7] = 0xfee3, [0x2df8] = 0xfee4, [0x2df9] = 0xfee5,
  [0x2dfa] = 0xfee6, [0x2dfb] = 0xfee7, [0x2dfc] = 0xfee8, [0x2dfd] = 0xfee9,
  [0x2dfe] = 0xfeea, [0x2dff] = 0xfeeb, [0x2e00] = 0xfeec, [0x2e01] = 0xfeed,
  [0x2e02] = 0xfeee, [0x2e03] = 0xfeef, [0x2e04] = 0xfef0, [0x2e05] = 0xfef1,
  [0x2e06] = 0xfef2, [0x2e07] = 0xfef3, [0x2e08] = 0xfef4, [0x2e09] = 0xfef5,
  [0x2e0a] = 0xfef6, [0x2e0b] = 0xfef7, [0x2e0c] = 0xfef8, [0x2e0d] = 0xfef9,
  [0x2e0e] = 0xfefa, [0x2e0f] = 0xfefb, [0x2e10] = 0xfefc, [0x2e13] = 0xfeff,
  [0x2e15] = 0xff5f, [0x2e16] = 0xff60, [0x2e17] = 0xff61, [0x2e18] = 0xff62,
  [0x2e19] = 0xff63, [0x2e1a] = 0xff64, [0x2e1b] = 0xff65, [0x2e1c] = 0xff66,
  [0x2e1d] = 0xff67, [0x2e1e] = 0xff68, [0x2e1f] = 0xff69, [0x2e20] = 0xff6a,
  [0x2e21] = 0xff6b, [0x2e22] = 0xff6c, [0x2e23] = 0xff6d, [0x2e24] = 0xff6e,
  [0x2e25] = 0xff6f, [0x2e26] = 0xff70, [0x2e27] = 0xff71, [0x2e28] = 0xff72,
  [0x2e29] = 0xff73, [0x2e2a] = 0xff74, [0x2e2b] = 0xff75, [0x2e2c] = 0xff76,
  [0x2e2d] = 0xff77, [0x2e2e] = 0xff78, [0x2e2f] = 0xff79, [0x2e30] = 0xff7a,
  [0x2e31] = 0xff7b, [0x2e32] = 0xff7c, [0x2e33] = 0xff7d, [0x2e34] = 0xff7e,
  [0x2e35] = 0xff7f, [0x2e36] = 0xff80, [0x2e37] = 0xff81, [0x2e38] = 0xff82,
  [0x2e39] = 0xff83, [0x2e3a] = 0xff84, [0x2e3b] = 0xff85, [0x2e3c] = 0xff86,
  [0x2e3d] = 0xff87, [0x2e3e] = 0xff88, [0x2e3f] = 0xff89, [0x2e40] = 0xff8a,
  [0x2e41] = 0xff8b, [0x2e42] = 0xff8c, [0x2e43] = 0xff8d, [0x2e44] = 0xff8e,
  [0x2e45] = 0xff8f, [0x2e46] = 0xff90, [0x2e47] = 0xff91, [0x2e48] = 0xff92,
  [0x2e49] = 0xff93, [0x2e4a] = 0xff94, [0x2e4b] = 0xff95, [0x2e4c] = 0xff96,
  [0x2e4d] = 0xff97, [0x2e4e] = 0xff98, [0x2e4f] = 0xff99, [0x2e50] = 0xff9a,
  [0x2e51] = 0xff9b, [0x2e52] = 0xff9c, [0x2e53] = 0xff9d, [0x2e54] = 0xff9e,
  [0x2e55] = 0xff9f, [0x2e56] = 0xffa0, [0x2e57] = 0xffa1, [0x2e58] = 0xffa2,
  [0x2e59] = 0xffa3, [0x2e5a] = 0xffa4, [0x2e5b] = 0xffa5, [0x2e5c] = 0xffa6,
  [0x2e5d] = 0xffa7, [0x2e5e] = 0xffa8, [0x2e5f] = 0xffa9, [0x2e60] = 0xffaa,
  [0x2e61] = 0xffab, [0x2e62] = 0xffac, [0x2e63] = 0xffad, [0x2e64] = 0xffae,
  [0x2e65] = 0xffaf, [0x2e66] = 0xffb0, [0x2e67] = 0xffb1, [0x2e68] = 0xffb2,
  [0x2e69] = 0xffb3, [0x2e6a] = 0xffb4, [0x2e6b] = 0xffb5, [0x2e6c] = 0xffb6,
  [0x2e6d] = 0xffb7, [0x2e6e] = 0xffb8, [0x2e6f] = 0xffb9, [0x2e70] = 0xffba,
  [0x2e71] = 0xffbb, [0x2e72] = 0xffbc, [0x2e73] = 0xffbd, [0x2e74] = 0xffbe,
  [0x2e78] = 0xffc2, [0x2e79] = 0xffc3, [0x2e7a] = 0xffc4, [0x2e7b] = 0xffc5,
  [0x2e7c] = 0xffc6, [0x2e7d] = 0xffc7, [0x2e80] = 0xffca, [0x2e81] = 0xffcb,
  [0x2e82] = 0xffcc, [0x2e83] = 0xffcd, [0x2e84] = 0xffce, [0x2e85] = 0xffcf,
  [0x2e88] = 0xffd2, [0x2e89] = 0xffd3, [0x2e8a] = 0xffd4, [0x2e8b] = 0xffd5,
  [0x2e8c] = 0xffd6, [0x2e8d] = 0xffd7, [0x2e90] = 0xffda, [0x2e91] = 0xffdb,
  [0x2e92] = 0xffdc,
};

/* Table for UCS-4 -> GB18030, for the range U+0080..U+9FBB.
   There are three kinds of two-byte entries (ch1, ch2):
     (ch1, ch2) = (0, 0) means an unmapped character,
     0x20 <= ch1 <= 0x80 means a four-byte character,
     ch1 >= 0x81 means a two-byte character.
   Generated through

   sort -k 2 < GB18030.table | clisp tab.lisp

   where tab.lisp is:
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   (locally (declare (compile))
     (let ((count 0)
	   (range-start #x0080) (range-end #x9FBB))
       (flet ((output-pair (ucs entry)
		(when (= (mod count 3) 0)
		  (format t " "))
		(format t " ~([0x~4,'0X] = \"\\x~2,'0X\\x~2,'0X\"~),"
			; There are two ranges. The gap length is:
			;   (- #xd7a4 #xac00) = 11172.
			(cond ((<= range-start ucs range-end)
			       (- ucs range-start)))
			(ash entry -8) (ldb (byte 8 0) entry))
		(incf count)
		(when (= (mod count 3) 0)
		  (format t "~%"))))
	 (loop
	   (let ((line (read-line nil nil nil)))
	     (unless line (return))
	     (when (eql (position #\Tab line) 6)
	       (let* ((gbk (parse-integer line :start 2 :end 6 :radix 16))
		      (ucs (parse-integer line :start 9 :radix 16)))
		 (when (<= range-start ucs range-end)
		   (assert (>= gbk #x8100))
		   (output-pair ucs gbk))))
	     (when (eql (position #\Tab line) 10)
	       (let* ((gbk (parse-integer line :start 2 :end 10 :radix 16))
		      (ucs (parse-integer line :start 13 :radix 16))
		      (ch1 (ldb (byte 8 24) gbk))
		      (ch2 (ldb (byte 8 16) gbk))
		      (ch3 (ldb (byte 8 8) gbk))
		      (ch4 (ldb (byte 8 0) gbk))
		      (linear (+ (* (+ (* (+ (* (- ch1 #x81) 10)
					     (- ch2 #x30))
					  126)
				       (- ch3 #x81))
				    10)
				 (- ch4 #x30))))
		 (when (<= range-start ucs range-end)
		   (output-pair ucs (+ #x2000 linear))))))))
       (format t "~%")))
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

static const unsigned char __ucs_to_gb18030_tab1[][2] =
{
  [0x0000] = "\x20\x00", [0x0001] = "\x20\x01", [0x0002] = "\x20\x02",
  [0x0003] = "\x20\x03", [0x0004] = "\x20\x04", [0x0005] = "\x20\x05",
  [0x0006] = "\x20\x06", [0x0007] = "\x20\x07", [0x0008] = "\x20\x08",
  [0x0009] = "\x20\x09", [0x000a] = "\x20\x0a", [0x000b] = "\x20\x0b",
  [0x000c] = "\x20\x0c", [0x000d] = "\x20\x0d", [0x000e] = "\x20\x0e",
  [0x000f] = "\x20\x0f", [0x0010] = "\x20\x10", [0x0011] = "\x20\x11",
  [0x0012] = "\x20\x12", [0x0013] = "\x20\x13", [0x0014] = "\x20\x14",
  [0x0015] = "\x20\x15", [0x0016] = "\x20\x16", [0x0017] = "\x20\x17",
  [0x0018] = "\x20\x18", [0x0019] = "\x20\x19", [0x001a] = "\x20\x1a",
  [0x001b] = "\x20\x1b", [0x001c] = "\x20\x1c", [0x001d] = "\x20\x1d",
  [0x001e] = "\x20\x1e", [0x001f] = "\x20\x1f", [0x0020] = "\x20\x20",
  [0x0021] = "\x20\x21", [0x0022] = "\x20\x22", [0x0023] = "\x20\x23",
  [0x0024] = "\xa1\xe8", [0x0025] = "\x20\x24", [0x0026] = "\x20\x25",
  [0x0027] = "\xa1\xec", [0x0028] = "\xa1\xa7", [0x0029] = "\x20\x26",
  [0x002a] = "\x20\x27", [0x002b] = "\x20\x28", [0x002c] = "\x20\x29",
  [0x002d] = "\x20\x2a", [0x002e] = "\x20\x2b", [0x002f] = "\x20\x2c",
  [0x0030] = "\xa1\xe3", [0x0031] = "\xa1\xc0", [0x0032] = "\x20\x2d",
  [0x0033] = "\x20\x2e", [0x0034] = "\x20\x2f", [0x0035] = "\x20\x30",
  [0x0036] = "\x20\x31", [0x0037] = "\xa1\xa4", [0x0038] = "\x20\x32",
  [0x0039] = "\x20\x33", [0x003a] = "\x20\x34", [0x003b] = "\x20\x35",
  [0x003c] = "\x20\x36", [0x003d] = "\x20\x37", [0x003e] = "\x20\x38",
  [0x003f] = "\x20\x39", [0x0040] = "\x20\x3a", [0x0041] = "\x20\x3b",
  [0x0042] = "\x20\x3c", [0x0043] = "\x20\x3d", [0x0044] = "\x20\x3e",
  [0x0045] = "\x20\x3f", [0x0046] = "\x20\x40", [0x0047] = "\x20\x41",
  [0x0048] = "\x20\x42", [0x0049] = "\x20\x43", [0x004a] = "\x20\x44",
  [0x004b] = "\x20\x45", [0x004c] = "\x20\x46", [0x004d] = "\x20\x47",
  [0x004e] = "\x20\x48", [0x004f] = "\x20\x49", [0x0050] = "\x20\x4a",
  [0x0051] = "\x20\x4b", [0x0052] = "\x20\x4c", [0x0053] = "\x20\x4d",
  [0x0054] = "\x20\x4e", [0x0055] = "\x20\x4f", [0x0056] = "\x20\x50",
  [0x0057] = "\xa1\xc1", [0x0058] = "\x20\x51", [0x0059] = "\x20\x52",
  [0x005a] = "\x20\x53", [0x005b] = "\x20\x54", [0x005c] = "\x20\x55",
  [0x005d] = "\x20\x56", [0x005e] = "\x20\x57", [0x005f] = "\x20\x58",
  [0x0060] = "\xa8\xa4", [0x0061] = "\xa8\xa2", [0x0062] = "\x20\x59",
  [0x0063] = "\x20\x5a", [0x0064] = "\x20\x5b", [0x0065] = "\x20\x5c",
  [0x0066] = "\x20\x5d", [0x0067] = "\x20\x5e", [0x0068] = "\xa8\xa8",
  [0x0069] = "\xa8\xa6", [0x006a] = "\xa8\xba", [0x006b] = "\x20\x5f",
  [0x006c] = "\xa8\xac", [0x006d] = "\xa8\xaa", [0x006e] = "\x20\x60",
  [0x006f] = "\x20\x61", [0x0070] = "\x20\x62", [0x0071] = "\x20\x63",
  [0x0072] = "\xa8\xb0", [0x0073] = "\xa8\xae", [0x0074] = "\x20\x64",
  [0x0075] = "\x20\x65", [0x0076] = "\x20\x66", [0x0077] = "\xa1\xc2",
  [0x0078] = "\x20\x67", [0x0079] = "\xa8\xb4", [0x007a] = "\xa8\xb2",
  [0x007b] = "\x20\x68", [0x007c] = "\xa8\xb9", [0x007d] = "\x20\x69",
  [0x007e] = "\x20\x6a", [0x007f] = "\x20\x6b", [0x0080] = "\x20\x6c",
  [0x0081] = "\xa8\xa1", [0x0082] = "\x20\x6d", [0x0083] = "\x20\x6e",
  [0x0084] = "\x20\x6f", [0x0085] = "\x20\x70", [0x0086] = "\x20\x71",
  [0x0087] = "\x20\x72", [0x0088] = "\x20\x73", [0x0089] = "\x20\x74",
  [0x008a] = "\x20\x75", [0x008b] = "\x20\x76", [0x008c] = "\x20\x77",
  [0x008d] = "\x20\x78", [0x008e] = "\x20\x79", [0x008f] = "\x20\x7a",
  [0x0090] = "\x20\x7b", [0x0091] = "\x20\x7c", [0x0092] = "\x20\x7d",
  [0x0093] = "\xa8\xa5", [0x0094] = "\x20\x7e", [0x0095] = "\x20\x7f",
  [0x0096] = "\x20\x80", [0x0097] = "\x20\x81", [0x0098] = "\x20\x82",
  [0x0099] = "\x20\x83", [0x009a] = "\x20\x84", [0x009b] = "\xa8\xa7",
  [0x009c] = "\x20\x85", [0x009d] = "\x20\x86", [0x009e] = "\x20\x87",
  [0x009f] = "\x20\x88", [0x00a0] = "\x20\x89", [0x00a1] = "\x20\x8a",
  [0x00a2] = "\x20\x8b", [0x00a3] = "\x20\x8c", [0x00a4] = "\x20\x8d",
  [0x00a5] = "\x20\x8e", [0x00a6] = "\x20\x8f", [0x00a7] = "\x20\x90",
  [0x00a8] = "\x20\x91", [0x00a9] = "\x20\x92", [0x00aa] = "\x20\x93",
  [0x00ab] = "\xa8\xa9", [0x00ac] = "\x20\x94", [0x00ad] = "\x20\x95",
  [0x00ae] = "\x20\x96", [0x00af] = "\x20\x97", [0x00b0] = "\x20\x98",
  [0x00b1] = "\x20\x99", [0x00b2] = "\x20\x9a", [0x00b3] = "\x20\x9b",
  [0x00b4] = "\x20\x9c", [0x00b5] = "\x20\x9d", [0x00b6] = "\x20\x9e",
  [0x00b7] = "\x20\x9f", [0x00b8] = "\x20\xa0", [0x00b9] = "\x20\xa1",
  [0x00ba] = "\x20\xa2", [0x00bb] = "\x20\xa3", [0x00bc] = "\x20\xa4",
  [0x00bd] = "\x20\xa5", [0x00be] = "\x20\xa6", [0x00bf] = "\x20\xa7",
  [0x00c0] = "\x20\xa8", [0x00c1] = "\x20\xa9", [0x00c2] = "\x20\xaa",
  [0x00c3] = "\x20\xab", [0x00c4] = "\xa8\xbd", [0x00c5] = "\x20\xac",
  [0x00c6] = "\x20\xad", [0x00c7] = "\x20\xae", [0x00c8] = "\xa8\xbe",
  [0x00c9] = "\x20\xaf", [0x00ca] = "\x20\xb0", [0x00cb] = "\x20\xb1",
  [0x00cc] = "\x20\xb2", [0x00cd] = "\xa8\xad", [0x00ce] = "\x20\xb3",
  [0x00cf] = "\x20\xb4", [0x00d0] = "\x20\xb5", [0x00d1] = "\x20\xb6",
  [0x00d2] = "\x20\xb7", [0x00d3] = "\x20\xb8", [0x00d4] = "\x20\xb9",
  [0x00d5] = "\x20\xba", [0x00d6] = "\x20\xbb", [0x00d7] = "\x20\xbc",
  [0x00d8] = "\x20\xbd", [0x00d9] = "\x20\xbe", [0x00da] = "\x20\xbf",
  [0x00db] = "\x20\xc0", [0x00dc] = "\x20\xc1", [0x00dd] = "\x20\xc2",
  [0x00de] = "\x20\xc3", [0x00df] = "\x20\xc4", [0x00e0] = "\x20\xc5",
  [0x00e1] = "\x20\xc6", [0x00e2] = "\x20\xc7", [0x00e3] = "\x20\xc8",
  [0x00e4] = "\x20\xc9", [0x00e5] = "\x20\xca", [0x00e6] = "\x20\xcb",
  [0x00e7] = "\x20\xcc", [0x00e8] = "\x20\xcd", [0x00e9] = "\x20\xce",
  [0x00ea] = "\x20\xcf", [0x00eb] = "\xa8\xb1", [0x00ec] = "\x20\xd0",
  [0x00ed] = "\x20\xd1", [0x00ee] = "\x20\xd2", [0x00ef] = "\x20\xd3",
  [0x00f0] = "\x20\xd4", [0x00f1] = "\x20\xd5", [0x00f2] = "\x20\xd6",
  [0x00f3] = "\x20\xd7", [0x00f4] = "\x20\xd8", [0x00f5] = "\x20\xd9",
  [0x00f6] = "\x20\xda", [0x00f7] = "\x20\xdb", [0x00f8] = "\x20\xdc",
  [0x00f9] = "\x20\xdd", [0x00fa] = "\x20\xde", [0x00fb] = "\x20\xdf",
  [0x00fc] = "\x20\xe0", [0x00fd] = "\x20\xe1", [0x00fe] = "\x20\xe2",
  [0x00ff] = "\x20\xe3", [0x0100] = "\x20\xe4", [0x0101] = "\x20\xe5",
  [0x0102] = "\x20\xe6", [0x0103] = "\x20\xe7", [0x0104] = "\x20\xe8",
  [0x0105] = "\x20\xe9", [0x0106] = "\x20\xea", [0x0107] = "\x20\xeb",
  [0x0108] = "\x20\xec", [0x0109] = "\x20\xed", [0x010a] = "\x20\xee",
  [0x010b] = "\x20\xef", [0x010c] = "\x20\xf0", [0x010d] = "\x20\xf1",
  [0x010e] = "\x20\xf2", [0x010f] = "\x20\xf3", [0x0110] = "\x20\xf4",
  [0x0111] = "\x20\xf5", [0x0112] = "\x20\xf6", [0x0113] = "\x20\xf7",
  [0x0114] = "\x20\xf8", [0x0115] = "\x20\xf9", [0x0116] = "\x20\xfa",
  [0x0117] = "\x20\xfb", [0x0118] = "\x20\xfc", [0x0119] = "\x20\xfd",
  [0x011a] = "\x20\xfe", [0x011b] = "\x20\xff", [0x011c] = "\x21\x00",
  [0x011d] = "\x21\x01", [0x011e] = "\x21\x02", [0x011f] = "\x21\x03",
  [0x0120] = "\x21\x04", [0x0121] = "\x21\x05", [0x0122] = "\x21\x06",
  [0x0123] = "\x21\x07", [0x0124] = "\x21\x08", [0x0125] = "\x21\x09",
  [0x0126] = "\x21\x0a", [0x0127] = "\x21\x0b", [0x0128] = "\x21\x0c",
  [0x0129] = "\x21\x0d", [0x012a] = "\x21\x0e", [0x012b] = "\x21\x0f",
  [0x012c] = "\x21\x10", [0x012d] = "\x21\x11", [0x012e] = "\x21\x12",
  [0x012f] = "\x21\x13", [0x0130] = "\x21\x14", [0x0131] = "\x21\x15",
  [0x0132] = "\x21\x16", [0x0133] = "\x21\x17", [0x0134] = "\x21\x18",
  [0x0135] = "\x21\x19", [0x0136] = "\x21\x1a", [0x0137] = "\x21\x1b",
  [0x0138] = "\x21\x1c", [0x0139] = "\x21\x1d", [0x013a] = "\x21\x1e",
  [0x013b] = "\x21\x1f", [0x013c] = "\x21\x20", [0x013d] = "\x21\x21",
  [0x013e] = "\x21\x22", [0x013f] = "\x21\x23", [0x0140] = "\x21\x24",
  [0x0141] = "\x21\x25", [0x0142] = "\x21\x26", [0x0143] = "\x21\x27",
  [0x0144] = "\x21\x28", [0x0145] = "\x21\x29", [0x0146] = "\x21\x2a",
  [0x0147] = "\x21\x2b", [0x0148] = "\x21\x2c", [0x0149] = "\x21\x2d",
  [0x014a] = "\x21\x2e", [0x014b] = "\x21\x2f", [0x014c] = "\x21\x30",
  [0x014d] = "\x21\x31", [0x014e] = "\xa8\xa3", [0x014f] = "\x21\x32",
  [0x0150] = "\xa8\xab", [0x0151] = "\x21\x33", [0x0152] = "\xa8\xaf",
  [0x0153] = "\x21\x34", [0x0154] = "\xa8\xb3", [0x0155] = "\x21\x35",
  [0x0156] = "\xa8\xb5", [0x0157] = "\x21\x36", [0x0158] = "\xa8\xb6",
  [0x0159] = "\x21\x37", [0x015a] = "\xa8\xb7", [0x015b] = "\x21\x38",
  [0x015c] = "\xa8\xb8", [0x015d] = "\x21\x39", [0x015e] = "\x21\x3a",
  [0x015f] = "\x21\x3b", [0x0160] = "\x21\x3c", [0x0161] = "\x21\x3d",
  [0x0162] = "\x21\x3e", [0x0163] = "\x21\x3f", [0x0164] = "\x21\x40",
  [0x0165] = "\x21\x41", [0x0166] = "\x21\x42", [0x0167] = "\x21\x43",
  [0x0168] = "\x21\x44", [0x0169] = "\x21\x45", [0x016a] = "\x21\x46",
  [0x016b] = "\x21\x47", [0x016c] = "\x21\x48", [0x016d] = "\x21\x49",
  [0x016e] = "\x21\x4a", [0x016f] = "\x21\x4b", [0x0170] = "\x21\x4c",
  [0x0171] = "\x21\x4d", [0x0172] = "\x21\x4e", [0x0173] = "\x21\x4f",
  [0x0174] = "\x21\x50", [0x0175] = "\x21\x51", [0x0176] = "\x21\x52",
  [0x0177] = "\x21\x53", [0x0178] = "\x21\x54", [0x0179] = "\xa8\xbf",
  [0x017a] = "\x21\x55", [0x017b] = "\x21\x56", [0x017c] = "\x21\x57",
  [0x017d] = "\x21\x58", [0x017e] = "\x21\x59", [0x017f] = "\x21\x5a",
  [0x0180] = "\x21\x5b", [0x0181] = "\x21\x5c", [0x0182] = "\x21\x5d",
  [0x0183] = "\x21\x5e", [0x0184] = "\x21\x5f", [0x0185] = "\x21\x60",
  [0x0186] = "\x21\x61", [0x0187] = "\x21\x62", [0x0188] = "\x21\x63",
  [0x0189] = "\x21\x64", [0x018a] = "\x21\x65", [0x018b] = "\x21\x66",
  [0x018c] = "\x21\x67", [0x018d] = "\x21\x68", [0x018e] = "\x21\x69",
  [0x018f] = "\x21\x6a", [0x0190] = "\x21\x6b", [0x0191] = "\x21\x6c",
  [0x0192] = "\x21\x6d", [0x0193] = "\x21\x6e", [0x0194] = "\x21\x6f",
  [0x0195] = "\x21\x70", [0x0196] = "\x21\x71", [0x0197] = "\x21\x72",
  [0x0198] = "\x21\x73", [0x0199] = "\x21\x74", [0x019a] = "\x21\x75",
  [0x019b] = "\x21\x76", [0x019c] = "\x21\x77", [0x019d] = "\x21\x78",
  [0x019e] = "\x21\x79", [0x019f] = "\x21\x7a", [0x01a0] = "\x21\x7b",
  [0x01a1] = "\x21\x7c", [0x01a2] = "\x21\x7d", [0x01a3] = "\x21\x7e",
  [0x01a4] = "\x21\x7f", [0x01a5] = "\x21\x80", [0x01a6] = "\x21\x81",
  [0x01a7] = "\x21\x82", [0x01a8] = "\x21\x83", [0x01a9] = "\x21\x84",
  [0x01aa] = "\x21\x85", [0x01ab] = "\x21\x86", [0x01ac] = "\x21\x87",
  [0x01ad] = "\x21\x88", [0x01ae] = "\x21\x89", [0x01af] = "\x21\x8a",
  [0x01b0] = "\x21\x8b", [0x01b1] = "\x21\x8c", [0x01b2] = "\x21\x8d",
  [0x01b3] = "\x21\x8e", [0x01b4] = "\x21\x8f", [0x01b5] = "\x21\x90",
  [0x01b6] = "\x21\x91", [0x01b7] = "\x21\x92", [0x01b8] = "\x21\x93",
  [0x01b9] = "\x21\x94", [0x01ba] = "\x21\x95", [0x01bb] = "\x21\x96",
  [0x01bc] = "\x21\x97", [0x01bd] = "\x21\x98", [0x01be] = "\x21\x99",
  [0x01bf] = "\x21\x9a", [0x01c0] = "\x21\x9b", [0x01c1] = "\x21\x9c",
  [0x01c2] = "\x21\x9d", [0x01c3] = "\x21\x9e", [0x01c4] = "\x21\x9f",
  [0x01c5] = "\x21\xa0", [0x01c6] = "\x21\xa1", [0x01c7] = "\x21\xa2",
  [0x01c8] = "\x21\xa3", [0x01c9] = "\x21\xa4", [0x01ca] = "\x21\xa5",
  [0x01cb] = "\x21\xa6", [0x01cc] = "\x21\xa7", [0x01cd] = "\x21\xa8",
  [0x01ce] = "\x21\xa9", [0x01cf] = "\x21\xaa", [0x01d0] = "\x21\xab",
  [0x01d1] = "\xa8\xbb", [0x01d2] = "\x21\xac", [0x01d3] = "\x21\xad",
  [0x01d4] = "\x21\xae", [0x01d5] = "\x21\xaf", [0x01d6] = "\x21\xb0",
  [0x01d7] = "\x21\xb1", [0x01d8] = "\x21\xb2", [0x01d9] = "\x21\xb3",
  [0x01da] = "\x21\xb4", [0x01db] = "\x21\xb5", [0x01dc] = "\x21\xb6",
  [0x01dd] = "\x21\xb7", [0x01de] = "\x21\xb8", [0x01df] = "\x21\xb9",
  [0x01e0] = "\x21\xba", [0x01e1] = "\xa8\xc0", [0x01e2] = "\x21\xbb",
  [0x01e3] = "\x21\xbc", [0x01e4] = "\x21\xbd", [0x01e5] = "\x21\xbe",
  [0x01e6] = "\x21\xbf", [0x01e7] = "\x21\xc0", [0x01e8] = "\x21\xc1",
  [0x01e9] = "\x21\xc2", [0x01ea] = "\x21\xc3", [0x01eb] = "\x21\xc4",
  [0x01ec] = "\x21\xc5", [0x01ed] = "\x21\xc6", [0x01ee] = "\x21\xc7",
  [0x01ef] = "\x21\xc8", [0x01f0] = "\x21\xc9", [0x01f1] = "\x21\xca",
  [0x01f2] = "\x21\xcb", [0x01f3] = "\x21\xcc", [0x01f4] = "\x21\xcd",
  [0x01f5] = "\x21\xce", [0x01f6] = "\x21\xcf", [0x01f7] = "\x21\xd0",
  [0x01f8] = "\x21\xd1", [0x01f9] = "\x21\xd2", [0x01fa] = "\x21\xd3",
  [0x01fb] = "\x21\xd4", [0x01fc] = "\x21\xd5", [0x01fd] = "\x21\xd6",
  [0x01fe] = "\x21\xd7", [0x01ff] = "\x21\xd8", [0x0200] = "\x21\xd9",
  [0x0201] = "\x21\xda", [0x0202] = "\x21\xdb", [0x0203] = "\x21\xdc",
  [0x0204] = "\x21\xdd", [0x0205] = "\x21\xde", [0x0206] = "\x21\xdf",
  [0x0207] = "\x21\xe0", [0x0208] = "\x21\xe1", [0x0209] = "\x21\xe2",
  [0x020a] = "\x21\xe3", [0x020b] = "\x21\xe4", [0x020c] = "\x21\xe5",
  [0x020d] = "\x21\xe6", [0x020e] = "\x21\xe7", [0x020f] = "\x21\xe8",
  [0x0210] = "\x21\xe9", [0x0211] = "\x21\xea", [0x0212] = "\x21\xeb",
  [0x0213] = "\x21\xec", [0x0214] = "\x21\xed", [0x0215] = "\x21\xee",
  [0x0216] = "\x21\xef", [0x0217] = "\x21\xf0", [0x0218] = "\x21\xf1",
  [0x0219] = "\x21\xf2", [0x021a] = "\x21\xf3", [0x021b] = "\x21\xf4",
  [0x021c] = "\x21\xf5", [0x021d] = "\x21\xf6", [0x021e] = "\x21\xf7",
  [0x021f] = "\x21\xf8", [0x0220] = "\x21\xf9", [0x0221] = "\x21\xfa",
  [0x0222] = "\x21\xfb", [0x0223] = "\x21\xfc", [0x0224] = "\x21\xfd",
  [0x0225] = "\x21\xfe", [0x0226] = "\x21\xff", [0x0227] = "\x22\x00",
  [0x0228] = "\x22\x01", [0x0229] = "\x22\x02", [0x022a] = "\x22\x03",
  [0x022b] = "\x22\x04", [0x022c] = "\x22\x05", [0x022d] = "\x22\x06",
  [0x022e] = "\x22\x07", [0x022f] = "\x22\x08", [0x0230] = "\x22\x09",
  [0x0231] = "\x22\x0a", [0x0232] = "\x22\x0b", [0x0233] = "\x22\x0c",
  [0x0234] = "\x22\x0d", [0x0235] = "\x22\x0e", [0x0236] = "\x22\x0f",
  [0x0237] = "\x22\x10", [0x0238] = "\x22\x11", [0x0239] = "\x22\x12",
  [0x023a] = "\x22\x13", [0x023b] = "\x22\x14", [0x023c] = "\x22\x15",
  [0x023d] = "\x22\x16", [0x023e] = "\x22\x17", [0x023f] = "\x22\x18",
  [0x0240] = "\x22\x19", [0x0241] = "\x22\x1a", [0x0242] = "\x22\x1b",
  [0x0243] = "\x22\x1c", [0x0244] = "\x22\x1d", [0x0245] = "\x22\x1e",
  [0x0246] = "\x22\x1f", [0x0247] = "\xa1\xa6", [0x0248] = "\x22\x20",
  [0x0249] = "\xa1\xa5", [0x024a] = "\xa8\x40", [0x024b] = "\xa8\x41",
  [0x024c] = "\x22\x21", [0x024d] = "\x22\x22", [0x024e] = "\x22\x23",
  [0x024f] = "\x22\x24", [0x0250] = "\x22\x25", [0x0251] = "\x22\x26",
  [0x0252] = "\x22\x27", [0x0253] = "\x22\x28", [0x0254] = "\x22\x29",
  [0x0255] = "\x22\x2a", [0x0256] = "\x22\x2b", [0x0257] = "\x22\x2c",
  [0x0258] = "\x22\x2d", [0x0259] = "\xa8\x42", [0x025a] = "\x22\x2e",
  [0x025b] = "\x22\x2f", [0x025c] = "\x22\x30", [0x025d] = "\x22\x31",
  [0x025e] = "\x22\x32", [0x025f] = "\x22\x33", [0x0260] = "\x22\x34",
  [0x0261] = "\x22\x35", [0x0262] = "\x22\x36", [0x0263] = "\x22\x37",
  [0x0264] = "\x22\x38", [0x0265] = "\x22\x39", [0x0266] = "\x22\x3a",
  [0x0267] = "\x22\x3b", [0x0268] = "\x22\x3c", [0x0269] = "\x22\x3d",
  [0x026a] = "\x22\x3e", [0x026b] = "\x22\x3f", [0x026c] = "\x22\x40",
  [0x026d] = "\x22\x41", [0x026e] = "\x22\x42", [0x026f] = "\x22\x43",
  [0x0270] = "\x22\x44", [0x0271] = "\x22\x45", [0x0272] = "\x22\x46",
  [0x0273] = "\x22\x47", [0x0274] = "\x22\x48", [0x0275] = "\x22\x49",
  [0x0276] = "\x22\x4a", [0x0277] = "\x22\x4b", [0x0278] = "\x22\x4c",
  [0x0279] = "\x22\x4d", [0x027a] = "\x22\x4e", [0x027b] = "\x22\x4f",
  [0x027c] = "\x22\x50", [0x027d] = "\x22\x51", [0x027e] = "\x22\x52",
  [0x027f] = "\x22\x53", [0x0280] = "\x22\x54", [0x0281] = "\x22\x55",
  [0x0282] = "\x22\x56", [0x0283] = "\x22\x57", [0x0284] = "\x22\x58",
  [0x0285] = "\x22\x59", [0x0286] = "\x22\x5a", [0x0287] = "\x22\x5b",
  [0x0288] = "\x22\x5c", [0x0289] = "\x22\x5d", [0x028a] = "\x22\x5e",
  [0x028b] = "\x22\x5f", [0x028c] = "\x22\x60", [0x028d] = "\x22\x61",
  [0x028e] = "\x22\x62", [0x028f] = "\x22\x63", [0x0290] = "\x22\x64",
  [0x0291] = "\x22\x65", [0x0292] = "\x22\x66", [0x0293] = "\x22\x67",
  [0x0294] = "\x22\x68", [0x0295] = "\x22\x69", [0x0296] = "\x22\x6a",
  [0x0297] = "\x22\x6b", [0x0298] = "\x22\x6c", [0x0299] = "\x22\x6d",
  [0x029a] = "\x22\x6e", [0x029b] = "\x22\x6f", [0x029c] = "\x22\x70",
  [0x029d] = "\x22\x71", [0x029e] = "\x22\x72", [0x029f] = "\x22\x73",
  [0x02a0] = "\x22\x74", [0x02a1] = "\x22\x75", [0x02a2] = "\x22\x76",
  [0x02a3] = "\x22\x77", [0x02a4] = "\x22\x78", [0x02a5] = "\x22\x79",
  [0x02a6] = "\x22\x7a", [0x02a7] = "\x22\x7b", [0x02a8] = "\x22\x7c",
  [0x02a9] = "\x22\x7d", [0x02aa] = "\x22\x7e", [0x02ab] = "\x22\x7f",
  [0x02ac] = "\x22\x80", [0x02ad] = "\x22\x81", [0x02ae] = "\x22\x82",
  [0x02af] = "\x22\x83", [0x02b0] = "\x22\x84", [0x02b1] = "\x22\x85",
  [0x02b2] = "\x22\x86", [0x02b3] = "\x22\x87", [0x02b4] = "\x22\x88",
  [0x02b5] = "\x22\x89", [0x02b6] = "\x22\x8a", [0x02b7] = "\x22\x8b",
  [0x02b8] = "\x22\x8c", [0x02b9] = "\x22\x8d", [0x02ba] = "\x22\x8e",
  [0x02bb] = "\x22\x8f", [0x02bc] = "\x22\x90", [0x02bd] = "\x22\x91",
  [0x02be] = "\x22\x92", [0x02bf] = "\x22\x93", [0x02c0] = "\x22\x94",
  [0x02c1] = "\x22\x95", [0x02c2] = "\x22\x96", [0x02c3] = "\x22\x97",
  [0x02c4] = "\x22\x98", [0x02c5] = "\x22\x99", [0x02c6] = "\x22\x9a",
  [0x02c7] = "\x22\x9b", [0x02c8] = "\x22\x9c", [0x02c9] = "\x22\x9d",
  [0x02ca] = "\x22\x9e", [0x02cb] = "\x22\x9f", [0x02cc] = "\x22\xa0",
  [0x02cd] = "\x22\xa1", [0x02ce] = "\x22\xa2", [0x02cf] = "\x22\xa3",
  [0x02d0] = "\x22\xa4", [0x02d1] = "\x22\xa5", [0x02d2] = "\x22\xa6",
  [0x02d3] = "\x22\xa7", [0x02d4] = "\x22\xa8", [0x02d5] = "\x22\xa9",
  [0x02d6] = "\x22\xaa", [0x02d7] = "\x22\xab", [0x02d8] = "\x22\xac",
  [0x02d9] = "\x22\xad", [0x02da] = "\x22\xae", [0x02db] = "\x22\xaf",
  [0x02dc] = "\x22\xb0", [0x02dd] = "\x22\xb1", [0x02de] = "\x22\xb2",
  [0x02df] = "\x22\xb3", [0x02e0] = "\x22\xb4", [0x02e1] = "\x22\xb5",
  [0x02e2] = "\x22\xb6", [0x02e3] = "\x22\xb7", [0x02e4] = "\x22\xb8",
  [0x02e5] = "\x22\xb9", [0x02e6] = "\x22\xba", [0x02e7] = "\x22\xbb",
  [0x02e8] = "\x22\xbc", [0x02e9] = "\x22\xbd", [0x02ea] = "\x22\xbe",
  [0x02eb] = "\x22\xbf", [0x02ec] = "\x22\xc0", [0x02ed] = "\x22\xc1",
  [0x02ee] = "\x22\xc2", [0x02ef] = "\x22\xc3", [0x02f0] = "\x22\xc4",
  [0x02f1] = "\x22\xc5", [0x02f2] = "\x22\xc6", [0x02f3] = "\x22\xc7",
  [0x02f4] = "\x22\xc8", [0x02f5] = "\x22\xc9", [0x02f6] = "\x22\xca",
  [0x02f7] = "\x22\xcb", [0x02fa] = "\x22\xce", [0x02fb] = "\x22\xcf",
  [0x02fc] = "\x22\xd0", [0x02fd] = "\x22\xd1", [0x02fe] = "\x22\xd2",
  [0x0304] = "\x22\xd8", [0x0305] = "\x22\xd9", [0x0306] = "\x22\xda",
  [0x0307] = "\x22\xdb", [0x0308] = "\x22\xdc", [0x0309] = "\x22\xdd",
  [0x030a] = "\x22\xde", [0x030c] = "\x22\xe0", [0x030e] = "\x22\xe2",
  [0x030f] = "\x22\xe3", [0x0310] = "\x22\xe4", [0x0311] = "\xa6\xa1",
  [0x0312] = "\xa6\xa2", [0x0313] = "\xa6\xa3", [0x0314] = "\xa6\xa4",
  [0x0315] = "\xa6\xa5", [0x0316] = "\xa6\xa6", [0x0317] = "\xa6\xa7",
  [0x0318] = "\xa6\xa8", [0x0319] = "\xa6\xa9", [0x031a] = "\xa6\xaa",
  [0x031b] = "\xa6\xab", [0x031c] = "\xa6\xac", [0x031d] = "\xa6\xad",
  [0x031e] = "\xa6\xae", [0x031f] = "\xa6\xaf", [0x0320] = "\xa6\xb0",
  [0x0321] = "\xa6\xb1", [0x0323] = "\xa6\xb2", [0x0324] = "\xa6\xb3",
  [0x0325] = "\xa6\xb4", [0x0326] = "\xa6\xb5", [0x0327] = "\xa6\xb6",
  [0x0328] = "\xa6\xb7", [0x0329] = "\xa6\xb8", [0x032a] = "\x22\xe6",
  [0x032b] = "\x22\xe7", [0x032c] = "\x22\xe8", [0x032d] = "\x22\xe9",
  [0x032e] = "\x22\xea", [0x032f] = "\x22\xeb", [0x0330] = "\x22\xec",
  [0x0331] = "\xa6\xc1", [0x0332] = "\xa6\xc2", [0x0333] = "\xa6\xc3",
  [0x0334] = "\xa6\xc4", [0x0335] = "\xa6\xc5", [0x0336] = "\xa6\xc6",
  [0x0337] = "\xa6\xc7", [0x0338] = "\xa6\xc8", [0x0339] = "\xa6\xc9",
  [0x033a] = "\xa6\xca", [0x033b] = "\xa6\xcb", [0x033c] = "\xa6\xcc",
  [0x033d] = "\xa6\xcd", [0x033e] = "\xa6\xce", [0x033f] = "\xa6\xcf",
  [0x0340] = "\xa6\xd0", [0x0341] = "\xa6\xd1", [0x0342] = "\x22\xed",
  [0x0343] = "\xa6\xd2", [0x0344] = "\xa6\xd3", [0x0345] = "\xa6\xd4",
  [0x0346] = "\xa6\xd5", [0x0347] = "\xa6\xd6", [0x0348] = "\xa6\xd7",
  [0x0349] = "\xa6\xd8", [0x034a] = "\x22\xee", [0x034b] = "\x22\xef",
  [0x034c] = "\x22\xf0", [0x034d] = "\x22\xf1", [0x034e] = "\x22\xf2",
  [0x0350] = "\x22\xf4", [0x0351] = "\x22\xf5", [0x0352] = "\x22\xf6",
  [0x0353] = "\x22\xf7", [0x0354] = "\x22\xf8", [0x0355] = "\x22\xf9",
  [0x0356] = "\x22\xfa", [0x0357] = "\x22\xfb", [0x0358] = "\x22\xfc",
  [0x0359] = "\x22\xfd", [0x035a] = "\x22\xfe", [0x035b] = "\x22\xff",
  [0x035c] = "\x23\x00", [0x035d] = "\x23\x01", [0x035e] = "\x23\x02",
  [0x035f] = "\x23\x03", [0x0360] = "\x23\x04", [0x0361] = "\x23\x05",
  [0x0362] = "\x23\x06", [0x0363] = "\x23\x07", [0x0364] = "\x23\x08",
  [0x0365] = "\x23\x09", [0x0366] = "\x23\x0a", [0x0367] = "\x23\x0b",
  [0x0368] = "\x23\x0c", [0x0369] = "\x23\x0d", [0x036a] = "\x23\x0e",
  [0x036b] = "\x23\x0f", [0x036c] = "\x23\x10", [0x036d] = "\x23\x11",
  [0x036e] = "\x23\x12", [0x036f] = "\x23\x13", [0x0370] = "\x23\x14",
  [0x0371] = "\x23\x15", [0x0372] = "\x23\x16", [0x0373] = "\x23\x17",
  [0x0374] = "\x23\x18", [0x0375] = "\x23\x19", [0x0376] = "\x23\x1a",
  [0x0377] = "\x23\x1b", [0x0378] = "\x23\x1c", [0x0379] = "\x23\x1d",
  [0x037a] = "\x23\x1e", [0x037b] = "\x23\x1f", [0x037c] = "\x23\x20",
  [0x037d] = "\x23\x21", [0x037e] = "\x23\x22", [0x037f] = "\x23\x23",
  [0x0380] = "\x23\x24", [0x0381] = "\xa7\xa7", [0x0382] = "\x23\x25",
  [0x0383] = "\x23\x26", [0x0384] = "\x23\x27", [0x0385] = "\x23\x28",
  [0x0386] = "\x23\x29", [0x0387] = "\x23\x2a", [0x0388] = "\x23\x2b",
  [0x0389] = "\x23\x2c", [0x038a] = "\x23\x2d", [0x038b] = "\x23\x2e",
  [0x038c] = "\x23\x2f", [0x038d] = "\x23\x30", [0x038e] = "\x23\x31",
  [0x038f] = "\x23\x32", [0x0390] = "\xa7\xa1", [0x0391] = "\xa7\xa2",
  [0x0392] = "\xa7\xa3", [0x0393] = "\xa7\xa4", [0x0394] = "\xa7\xa5",
  [0x0395] = "\xa7\xa6", [0x0396] = "\xa7\xa8", [0x0397] = "\xa7\xa9",
  [0x0398] = "\xa7\xaa", [0x0399] = "\xa7\xab", [0x039a] = "\xa7\xac",
  [0x039b] = "\xa7\xad", [0x039c] = "\xa7\xae", [0x039d] = "\xa7\xaf",
  [0x039e] = "\xa7\xb0", [0x039f] = "\xa7\xb1", [0x03a0] = "\xa7\xb2",
  [0x03a1] = "\xa7\xb3", [0x03a2] = "\xa7\xb4", [0x03a3] = "\xa7\xb5",
  [0x03a4] = "\xa7\xb6", [0x03a5] = "\xa7\xb7", [0x03a6] = "\xa7\xb8",
  [0x03a7] = "\xa7\xb9", [0x03a8] = "\xa7\xba", [0x03a9] = "\xa7\xbb",
  [0x03aa] = "\xa7\xbc", [0x03ab] = "\xa7\xbd", [0x03ac] = "\xa7\xbe",
  [0x03ad] = "\xa7\xbf", [0x03ae] = "\xa7\xc0", [0x03af] = "\xa7\xc1",
  [0x03b0] = "\xa7\xd1", [0x03b1] = "\xa7\xd2", [0x03b2] = "\xa7\xd3",
  [0x03b3] = "\xa7\xd4", [0x03b4] = "\xa7\xd5", [0x03b5] = "\xa7\xd6",
  [0x03b6] = "\xa7\xd8", [0x03b7] = "\xa7\xd9", [0x03b8] = "\xa7\xda",
  [0x03b9] = "\xa7\xdb", [0x03ba] = "\xa7\xdc", [0x03bb] = "\xa7\xdd",
  [0x03bc] = "\xa7\xde", [0x03bd] = "\xa7\xdf", [0x03be] = "\xa7\xe0",
  [0x03bf] = "\xa7\xe1", [0x03c0] = "\xa7\xe2", [0x03c1] = "\xa7\xe3",
  [0x03c2] = "\xa7\xe4", [0x03c3] = "\xa7\xe5", [0x03c4] = "\xa7\xe6",
  [0x03c5] = "\xa7\xe7", [0x03c6] = "\xa7\xe8", [0x03c7] = "\xa7\xe9",
  [0x03c8] = "\xa7\xea", [0x03c9] = "\xa7\xeb", [0x03ca] = "\xa7\xec",
  [0x03cb] = "\xa7\xed", [0x03cc] = "\xa7\xee", [0x03cd] = "\xa7\xef",
  [0x03ce] = "\xa7\xf0", [0x03cf] = "\xa7\xf1", [0x03d0] = "\x23\x33",
  [0x03d1] = "\xa7\xd7", [0x1dbf] = "\xa8\xbc", [0x1f90] = "\xa9\x5c",
  [0x1f91] = "\x3e\xf2", [0x1f92] = "\x3e\xf3", [0x1f93] = "\xa8\x43",
  [0x1f94] = "\xa1\xaa", [0x1f95] = "\xa8\x44", [0x1f96] = "\xa1\xac",
  [0x1f97] = "\x3e\xf4", [0x1f98] = "\xa1\xae", [0x1f99] = "\xa1\xaf",
  [0x1f9a] = "\x3e\xf5", [0x1f9b] = "\x3e\xf6", [0x1f9c] = "\xa1\xb0",
  [0x1f9d] = "\xa1\xb1", [0x1f9e] = "\x3e\xf7", [0x1f9f] = "\x3e\xf8",
  [0x1fa0] = "\x3e\xf9", [0x1fa1] = "\x3e\xfa", [0x1fa2] = "\x3e\xfb",
  [0x1fa3] = "\x3e\xfc", [0x1fa4] = "\x3e\xfd", [0x1fa5] = "\xa8\x45",
  [0x1fa6] = "\xa1\xad", [0x1fa7] = "\x3e\xfe", [0x1fa8] = "\x3e\xff",
  [0x1fa9] = "\x3f\x00", [0x1faa] = "\x3f\x01", [0x1fab] = "\x3f\x02",
  [0x1fac] = "\x3f\x03", [0x1fad] = "\x3f\x04", [0x1fae] = "\x3f\x05",
  [0x1faf] = "\x3f\x06", [0x1fb0] = "\xa1\xeb", [0x1fb1] = "\x3f\x07",
  [0x1fb2] = "\xa1\xe4", [0x1fb3] = "\xa1\xe5", [0x1fb4] = "\x3f\x08",
  [0x1fb5] = "\xa8\x46", [0x1fb6] = "\x3f\x09", [0x1fb7] = "\x3f\x0a",
  [0x1fb8] = "\x3f\x0b", [0x1fb9] = "\x3f\x0c", [0x1fba] = "\x3f\x0d",
  [0x1fbb] = "\xa1\xf9", [0x1fbc] = "\x3f\x0e", [0x1fbd] = "\x3f\x0f",
  [0x1fbe] = "\x3f\x10", [0x1fbf] = "\x3f\x11", [0x1fc0] = "\x3f\x12",
  [0x1fc1] = "\x3f\x13", [0x1fc2] = "\x3f\x14", [0x1fc3] = "\x3f\x15",
  [0x1fc4] = "\x3f\x16", [0x1fc5] = "\x3f\x17", [0x1fc6] = "\x3f\x18",
  [0x1fc7] = "\x3f\x19", [0x1fc8] = "\x3f\x1a", [0x1fc9] = "\x3f\x1b",
  [0x1fca] = "\x3f\x1c", [0x1fcb] = "\x3f\x1d", [0x1fcc] = "\x3f\x1e",
  [0x1fcd] = "\x3f\x1f", [0x1fce] = "\x3f\x20", [0x1fcf] = "\x3f\x21",
  [0x1fd0] = "\x3f\x22", [0x1fd1] = "\x3f\x23", [0x1fd2] = "\x3f\x24",
  [0x1fd3] = "\x3f\x25", [0x1fd4] = "\x3f\x26", [0x1fd5] = "\x3f\x27",
  [0x1fd6] = "\x3f\x28", [0x1fd7] = "\x3f\x29", [0x1fd8] = "\x3f\x2a",
  [0x1fd9] = "\x3f\x2b", [0x1fda] = "\x3f\x2c", [0x1fdb] = "\x3f\x2d",
  [0x1fdc] = "\x3f\x2e", [0x1fdd] = "\x3f\x2f", [0x1fde] = "\x3f\x30",
  [0x1fdf] = "\x3f\x31", [0x1fe0] = "\x3f\x32", [0x1fe1] = "\x3f\x33",
  [0x1fe2] = "\x3f\x34", [0x1fe3] = "\x3f\x35", [0x1fe4] = "\x3f\x36",
  [0x1fea] = "\x3f\x3c", [0x1feb] = "\x3f\x3d", [0x1fec] = "\x3f\x3e",
  [0x1fed] = "\x3f\x3f", [0x1fee] = "\x3f\x40", [0x1fef] = "\x3f\x41",
  [0x1ff0] = "\x3f\x42", [0x1ff1] = "\x3f\x43", [0x1ff4] = "\x3f\x46",
  [0x1ff5] = "\x3f\x47", [0x1ff6] = "\x3f\x48", [0x1ff7] = "\x3f\x49",
  [0x1ff8] = "\x3f\x4a", [0x1ff9] = "\x3f\x4b", [0x1ffa] = "\x3f\x4c",
  [0x1ffb] = "\x3f\x4d", [0x1ffc] = "\x3f\x4e", [0x1ffd] = "\x3f\x4f",
  [0x1ffe] = "\x3f\x50", [0x1fff] = "\x3f\x51", [0x2000] = "\x3f\x52",
  [0x2001] = "\x3f\x53", [0x2002] = "\x3f\x54", [0x2003] = "\x3f\x55",
  [0x2004] = "\x3f\x56", [0x2005] = "\x3f\x57", [0x2006] = "\x3f\x58",
  [0x2007] = "\x3f\x59", [0x2008] = "\x3f\x5a", [0x2009] = "\x3f\x5b",
  [0x200a] = "\x3f\x5c", [0x200b] = "\x3f\x5d", [0x200c] = "\x3f\x5e",
  [0x200d] = "\x3f\x5f", [0x200e] = "\x3f\x60", [0x2010] = "\x3f\x62",
  [0x2011] = "\x3f\x63", [0x2012] = "\x3f\x64", [0x2013] = "\x3f\x65",
  [0x2014] = "\x3f\x66", [0x2015] = "\x3f\x67", [0x2016] = "\x3f\x68",
  [0x2017] = "\x3f\x69", [0x2018] = "\x3f\x6a", [0x2019] = "\x3f\x6b",
  [0x201a] = "\x3f\x6c", [0x201b] = "\x3f\x6d", [0x201c] = "\x3f\x6e",
  [0x2020] = "\x3f\x72", [0x2021] = "\x3f\x73", [0x2022] = "\x3f\x74",
  [0x2023] = "\x3f\x75", [0x2024] = "\x3f\x76", [0x2025] = "\x3f\x77",
  [0x2026] = "\x3f\x78", [0x2027] = "\x3f\x79", [0x2028] = "\x3f\x7a",
  [0x2029] = "\x3f\x7b", [0x202a] = "\x3f\x7c", [0x202b] = "\x3f\x7d",
  [0x202c] = "\xa2\xe3", [0x202d] = "\x3f\x7e", [0x202e] = "\x3f\x7f",
  [0x202f] = "\x3f\x80", [0x2030] = "\x3f\x81", [0x2031] = "\x3f\x82",
  [0x2032] = "\x3f\x83", [0x2033] = "\x3f\x84", [0x2034] = "\x3f\x85",
  [0x2035] = "\x3f\x86", [0x2036] = "\x3f\x87", [0x2037] = "\x3f\x88",
  [0x2038] = "\x3f\x89", [0x2039] = "\x3f\x8a", [0x2050] = "\x3f\xa1",
  [0x2051] = "\x3f\xa2", [0x2052] = "\x3f\xa3", [0x2053] = "\x3f\xa4",
  [0x2054] = "\x3f\xa5", [0x2055] = "\x3f\xa6", [0x2056] = "\x3f\xa7",
  [0x2057] = "\x3f\xa8", [0x2058] = "\x3f\xa9", [0x2059] = "\x3f\xaa",
  [0x205a] = "\x3f\xab", [0x205b] = "\x3f\xac", [0x205c] = "\x3f\xad",
  [0x205d] = "\x3f\xae", [0x205e] = "\x3f\xaf", [0x205f] = "\x3f\xb0",
  [0x2060] = "\x3f\xb1", [0x2061] = "\x3f\xb2", [0x2062] = "\x3f\xb3",
  [0x2063] = "\x3f\xb4", [0x2064] = "\x3f\xb5", [0x2065] = "\x3f\xb6",
  [0x2066] = "\x3f\xb7", [0x2067] = "\x3f\xb8", [0x2068] = "\x3f\xb9",
  [0x2069] = "\x3f\xba", [0x206a] = "\x3f\xbb", [0x206b] = "\x3f\xbc",
  [0x206c] = "\x3f\xbd", [0x206d] = "\x3f\xbe", [0x206e] = "\x3f\xbf",
  [0x206f] = "\x3f\xc0", [0x2070] = "\x3f\xc1", [0x2080] = "\x3f\xd1",
  [0x2081] = "\x3f\xd2", [0x2082] = "\x3f\xd3", [0x2083] = "\xa1\xe6",
  [0x2084] = "\x3f\xd4", [0x2085] = "\xa8\x47", [0x2086] = "\x3f\xd5",
  [0x2087] = "\x3f\xd6", [0x2088] = "\x3f\xd7", [0x2089] = "\xa8\x48",
  [0x208a] = "\x3f\xd8", [0x208b] = "\x3f\xd9", [0x208c] = "\x3f\xda",
  [0x208d] = "\x3f\xdb", [0x208e] = "\x3f\xdc", [0x208f] = "\x3f\xdd",
  [0x2090] = "\x3f\xde", [0x2091] = "\x3f\xdf", [0x2092] = "\x3f\xe0",
  [0x2093] = "\x3f\xe1", [0x2094] = "\x3f\xe2", [0x2095] = "\x3f\xe3",
  [0x2096] = "\xa1\xed", [0x2097] = "\x3f\xe4", [0x2098] = "\x3f\xe5",
  [0x2099] = "\x3f\xe6", [0x209a] = "\x3f\xe7", [0x209b] = "\x3f\xe8",
  [0x209c] = "\x3f\xe9", [0x209d] = "\x3f\xea", [0x209e] = "\x3f\xeb",
  [0x209f] = "\x3f\xec", [0x20a0] = "\x3f\xed", [0x20a1] = "\xa9\x59",
  [0x20a2] = "\x3f\xee", [0x20a3] = "\x3f\xef", [0x20a4] = "\x3f\xf0",
  [0x20a5] = "\x3f\xf1", [0x20a6] = "\x3f\xf2", [0x20a7] = "\x3f\xf3",
  [0x20a8] = "\x3f\xf4", [0x20a9] = "\x3f\xf5", [0x20aa] = "\x3f\xf6",
  [0x20ab] = "\x3f\xf7", [0x20ac] = "\x3f\xf8", [0x20ad] = "\x3f\xf9",
  [0x20ae] = "\x3f\xfa", [0x20af] = "\x3f\xfb", [0x20b0] = "\x3f\xfc",
  [0x20b1] = "\x3f\xfd", [0x20b2] = "\x3f\xfe", [0x20b3] = "\x3f\xff",
  [0x20b4] = "\x40\x00", [0x20b5] = "\x40\x01", [0x20b6] = "\x40\x02",
  [0x20b7] = "\x40\x03", [0x20b8] = "\x40\x04", [0x20b9] = "\x40\x05",
  [0x20ba] = "\x40\x06", [0x20bb] = "\x40\x07", [0x20bc] = "\x40\x08",
  [0x20bd] = "\x40\x09", [0x20be] = "\x40\x0a", [0x20bf] = "\x40\x0b",
  [0x20c0] = "\x40\x0c", [0x20c1] = "\x40\x0d", [0x20c2] = "\x40\x0e",
  [0x20c3] = "\x40\x0f", [0x20c4] = "\x40\x10", [0x20c5] = "\x40\x11",
  [0x20c6] = "\x40\x12", [0x20c7] = "\x40\x13", [0x20c8] = "\x40\x14",
  [0x20c9] = "\x40\x15", [0x20ca] = "\x40\x16", [0x20cb] = "\x40\x17",
  [0x20cc] = "\x40\x18", [0x20cd] = "\x40\x19", [0x20ce] = "\x40\x1a",
  [0x20cf] = "\x40\x1b", [0x20d0] = "\x40\x1c", [0x20d1] = "\x40\x1d",
  [0x20d2] = "\x40\x1e", [0x20d3] = "\x40\x1f", [0x20d4] = "\x40\x20",
  [0x20d5] = "\x40\x21", [0x20d6] = "\x40\x22", [0x20d7] = "\x40\x23",
  [0x20d8] = "\x40\x24", [0x20d9] = "\x40\x25", [0x20da] = "\x40\x26",
  [0x20db] = "\x40\x27", [0x20dc] = "\x40\x28", [0x20dd] = "\x40\x29",
  [0x20de] = "\x40\x2a", [0x20df] = "\x40\x2b", [0x20e0] = "\xa2\xf1",
  [0x20e1] = "\xa2\xf2", [0x20e2] = "\xa2\xf3", [0x20e3] = "\xa2\xf4",
  [0x20e4] = "\xa2\xf5", [0x20e5] = "\xa2\xf6", [0x20e6] = "\xa2\xf7",
  [0x20e7] = "\xa2\xf8", [0x20e8] = "\xa2\xf9", [0x20e9] = "\xa2\xfa",
  [0x20ea] = "\xa2\xfb", [0x20eb] = "\xa2\xfc", [0x20ec] = "\x40\x2c",
  [0x20ed] = "\x40\x2d", [0x20ee] = "\x40\x2e", [0x20ef] = "\x40\x2f",
  [0x20f0] = "\xa2\xa1", [0x20f1] = "\xa2\xa2", [0x20f2] = "\xa2\xa3",
  [0x20f3] = "\xa2\xa4", [0x20f4] = "\xa2\xa5", [0x20f5] = "\xa2\xa6",
  [0x20f6] = "\xa2\xa7", [0x20f7] = "\xa2\xa8", [0x20f8] = "\xa2\xa9",
  [0x20f9] = "\xa2\xaa", [0x20fa] = "\x40\x30", [0x20fb] = "\x40\x31",
  [0x20fc] = "\x40\x32", [0x20fd] = "\x40\x33", [0x20fe] = "\x40\x34",
  [0x20ff] = "\x40\x35", [0x2100] = "\x40\x36", [0x2101] = "\x40\x37",
  [0x2102] = "\x40\x38", [0x2103] = "\x40\x39", [0x2104] = "\x40\x3a",
  [0x2105] = "\x40\x3b", [0x2106] = "\x40\x3c", [0x2107] = "\x40\x3d",
  [0x2108] = "\x40\x3e", [0x2109] = "\x40\x3f", [0x2110] = "\xa1\xfb",
  [0x2111] = "\xa1\xfc", [0x2112] = "\xa1\xfa", [0x2113] = "\xa1\xfd",
  [0x2114] = "\x40\x46", [0x2115] = "\x40\x47", [0x2116] = "\xa8\x49",
  [0x2117] = "\xa8\x4a", [0x2118] = "\xa8\x4b", [0x2119] = "\xa8\x4c",
  [0x211a] = "\x40\x48", [0x211b] = "\x40\x49", [0x211c] = "\x40\x4a",
  [0x211d] = "\x40\x4b", [0x211e] = "\x40\x4c", [0x211f] = "\x40\x4d",
  [0x2120] = "\x40\x4e", [0x2121] = "\x40\x4f", [0x2122] = "\x40\x50",
  [0x2123] = "\x40\x51", [0x2124] = "\x40\x52", [0x2125] = "\x40\x53",
  [0x2126] = "\x40\x54", [0x2127] = "\x40\x55", [0x2128] = "\x40\x56",
  [0x2129] = "\x40\x57", [0x212a] = "\x40\x58", [0x212b] = "\x40\x59",
  [0x212c] = "\x40\x5a", [0x212d] = "\x40\x5b", [0x212e] = "\x40\x5c",
  [0x212f] = "\x40\x5d", [0x2130] = "\x40\x5e", [0x2131] = "\x40\x5f",
  [0x2132] = "\x40\x60", [0x2133] = "\x40\x61", [0x2134] = "\x40\x62",
  [0x2135] = "\x40\x63", [0x2136] = "\x40\x64", [0x2137] = "\x40\x65",
  [0x2138] = "\x40\x66", [0x2139] = "\x40\x67", [0x213a] = "\x40\x68",
  [0x213b] = "\x40\x69", [0x213c] = "\x40\x6a", [0x213d] = "\x40\x6b",
  [0x213e] = "\x40\x6c", [0x213f] = "\x40\x6d", [0x2140] = "\x40\x6e",
  [0x2141] = "\x40\x6f", [0x2142] = "\x40\x70", [0x2143] = "\x40\x71",
  [0x2144] = "\x40\x72", [0x2145] = "\x40\x73", [0x2146] = "\x40\x74",
  [0x2147] = "\x40\x75", [0x2148] = "\x40\x76", [0x2149] = "\x40\x77",
  [0x214a] = "\x40\x78", [0x214b] = "\x40\x79", [0x214c] = "\x40\x7a",
  [0x214d] = "\x40\x7b", [0x214e] = "\x40\x7c", [0x214f] = "\x40\x7d",
  [0x2150] = "\x40\x7e", [0x2151] = "\x40\x7f", [0x2152] = "\x40\x80",
  [0x2153] = "\x40\x81", [0x2154] = "\x40\x82", [0x2155] = "\x40\x83",
  [0x2156] = "\x40\x84", [0x2157] = "\x40\x85", [0x2158] = "\x40\x86",
  [0x2159] = "\x40\x87", [0x215a] = "\x40\x88", [0x215b] = "\x40\x89",
  [0x215c] = "\x40\x8a", [0x215d] = "\x40\x8b", [0x215e] = "\x40\x8c",
  [0x215f] = "\x40\x8d", [0x2160] = "\x40\x8e", [0x2161] = "\x40\x8f",
  [0x2162] = "\x40\x90", [0x2163] = "\x40\x91", [0x2164] = "\x40\x92",
  [0x2165] = "\x40\x93", [0x2166] = "\x40\x94", [0x2167] = "\x40\x95",
  [0x2168] = "\x40\x96", [0x2169] = "\x40\x97", [0x216a] = "\x40\x98",
  [0x216b] = "\x40\x99", [0x216c] = "\x40\x9a", [0x216d] = "\x40\x9b",
  [0x216e] = "\x40\x9c", [0x216f] = "\x40\x9d", [0x2170] = "\x40\x9e",
  [0x2171] = "\x40\x9f", [0x2172] = "\x40\xa0", [0x2173] = "\x40\xa1",
  [0x2174] = "\x40\xa2", [0x2175] = "\x40\xa3", [0x2176] = "\x40\xa4",
  [0x2177] = "\x40\xa5", [0x2178] = "\x40\xa6", [0x2179] = "\x40\xa7",
  [0x217a] = "\x40\xa8", [0x217b] = "\x40\xa9", [0x217c] = "\x40\xaa",
  [0x217d] = "\x40\xab", [0x217e] = "\x40\xac", [0x217f] = "\x40\xad",
  [0x2180] = "\x40\xae", [0x2181] = "\x40\xaf", [0x2182] = "\x40\xb0",
  [0x2183] = "\x40\xb1", [0x2184] = "\x40\xb2", [0x2185] = "\x40\xb3",
  [0x2186] = "\x40\xb4", [0x2187] = "\x40\xb5", [0x2188] = "\xa1\xca",
  [0x2189] = "\x40\xb6", [0x218a] = "\x40\xb7", [0x218b] = "\x40\xb8",
  [0x218c] = "\x40\xb9", [0x218d] = "\x40\xba", [0x218e] = "\x40\xbb",
  [0x218f] = "\xa1\xc7", [0x2190] = "\x40\xbc", [0x2191] = "\xa1\xc6",
  [0x2192] = "\x40\xbd", [0x2193] = "\x40\xbe", [0x2194] = "\x40\xbf",
  [0x2195] = "\xa8\x4d", [0x2196] = "\x40\xc0", [0x2197] = "\x40\xc1",
  [0x2198] = "\x40\xc2", [0x2199] = "\x40\xc3", [0x219a] = "\xa1\xcc",
  [0x219b] = "\x40\xc4", [0x219c] = "\x40\xc5", [0x219d] = "\xa1\xd8",
  [0x219e] = "\xa1\xde", [0x219f] = "\xa8\x4e", [0x21a0] = "\xa1\xcf",
  [0x21a1] = "\x40\xc6", [0x21a2] = "\x40\xc7", [0x21a3] = "\xa8\x4f",
  [0x21a4] = "\x40\xc8", [0x21a5] = "\xa1\xce", [0x21a6] = "\x40\xc9",
  [0x21a7] = "\xa1\xc4", [0x21a8] = "\xa1\xc5", [0x21a9] = "\xa1\xc9",
  [0x21aa] = "\xa1\xc8", [0x21ab] = "\xa1\xd2", [0x21ac] = "\x40\xca",
  [0x21ad] = "\x40\xcb", [0x21ae] = "\xa1\xd3", [0x21af] = "\x40\xcc",
  [0x21b0] = "\x40\xcd", [0x21b1] = "\x40\xce", [0x21b2] = "\x40\xcf",
  [0x21b3] = "\x40\xd0", [0x21b4] = "\xa1\xe0", [0x21b5] = "\xa1\xdf",
  [0x21b6] = "\xa1\xc3", [0x21b7] = "\xa1\xcb", [0x21b8] = "\x40\xd1",
  [0x21b9] = "\x40\xd2", [0x21ba] = "\x40\xd3", [0x21bb] = "\x40\xd4",
  [0x21bc] = "\x40\xd5", [0x21bd] = "\xa1\xd7", [0x21be] = "\x40\xd6",
  [0x21bf] = "\x40\xd7", [0x21c0] = "\x40\xd8", [0x21c1] = "\x40\xd9",
  [0x21c2] = "\x40\xda", [0x21c3] = "\x40\xdb", [0x21c4] = "\x40\xdc",
  [0x21c5] = "\x40\xdd", [0x21c6] = "\x40\xde", [0x21c7] = "\x40\xdf",
  [0x21c8] = "\xa1\xd6", [0x21c9] = "\x40\xe0", [0x21ca] = "\x40\xe1",
  [0x21cb] = "\x40\xe2", [0x21cc] = "\xa1\xd5", [0x21cd] = "\x40\xe3",
  [0x21ce] = "\x40\xe4", [0x21cf] = "\x40\xe5", [0x21d0] = "\x40\xe6",
  [0x21d1] = "\x40\xe7", [0x21d2] = "\xa8\x50", [0x21d3] = "\x40\xe8",
  [0x21d4] = "\x40\xe9", [0x21d5] = "\x40\xea", [0x21d6] = "\x40\xeb",
  [0x21d7] = "\x40\xec", [0x21d8] = "\x40\xed", [0x21d9] = "\x40\xee",
  [0x21da] = "\x40\xef", [0x21db] = "\x40\xf0", [0x21dc] = "\x40\xf1",
  [0x21dd] = "\x40\xf2", [0x21de] = "\x40\xf3", [0x21df] = "\x40\xf4",
  [0x21e0] = "\xa1\xd9", [0x21e1] = "\xa1\xd4", [0x21e2] = "\x40\xf5",
  [0x21e3] = "\x40\xf6", [0x21e4] = "\xa1\xdc", [0x21e5] = "\xa1\xdd",
  [0x21e6] = "\xa8\x51", [0x21e7] = "\xa8\x52", [0x21e8] = "\x40\xf7",
  [0x21e9] = "\x40\xf8", [0x21ea] = "\x40\xf9", [0x21eb] = "\x40\xfa",
  [0x21ec] = "\x40\xfb", [0x21ed] = "\x40\xfc", [0x21ee] = "\xa1\xda",
  [0x21ef] = "\xa1\xdb", [0x21f0] = "\x40\xfd", [0x21f1] = "\x40\xfe",
  [0x21f2] = "\x40\xff", [0x21f3] = "\x41\x00", [0x21f4] = "\x41\x01",
  [0x21f5] = "\x41\x02", [0x21f6] = "\x41\x03", [0x21f7] = "\x41\x04",
  [0x21f8] = "\x41\x05", [0x21f9] = "\x41\x06", [0x21fa] = "\x41\x07",
  [0x21fb] = "\x41\x08", [0x21fc] = "\x41\x09", [0x21fd] = "\x41\x0a",
  [0x21fe] = "\x41\x0b", [0x21ff] = "\x41\x0c", [0x2200] = "\x41\x0d",
  [0x2201] = "\x41\x0e", [0x2202] = "\x41\x0f", [0x2203] = "\x41\x10",
  [0x2204] = "\x41\x11", [0x2205] = "\x41\x12", [0x2206] = "\x41\x13",
  [0x2207] = "\x41\x14", [0x2208] = "\x41\x15", [0x2209] = "\x41\x16",
  [0x220a] = "\x41\x17", [0x220b] = "\x41\x18", [0x220c] = "\x41\x19",
  [0x220d] = "\x41\x1a", [0x220e] = "\x41\x1b", [0x220f] = "\x41\x1c",
  [0x2210] = "\x41\x1d", [0x2211] = "\x41\x1e", [0x2212] = "\x41\x1f",
  [0x2213] = "\x41\x20", [0x2214] = "\x41\x21", [0x2215] = "\xa8\x92",
  [0x2216] = "\x41\x22", [0x2217] = "\x41\x23", [0x2218] = "\x41\x24",
  [0x2219] = "\xa1\xd1", [0x221a] = "\x41\x25", [0x221b] = "\x41\x26",
  [0x221c] = "\x41\x27", [0x221d] = "\x41\x28", [0x221e] = "\x41\x29",
  [0x221f] = "\x41\x2a", [0x2220] = "\x41\x2b", [0x2221] = "\x41\x2c",
  [0x2222] = "\x41\x2d", [0x2223] = "\x41\x2e", [0x2224] = "\x41\x2f",
  [0x2225] = "\xa1\xcd", [0x2226] = "\x41\x30", [0x2227] = "\x41\x31",
  [0x2228] = "\x41\x32", [0x2229] = "\x41\x33", [0x222a] = "\x41\x34",
  [0x222b] = "\x41\x35", [0x222c] = "\x41\x36", [0x222d] = "\x41\x37",
  [0x222e] = "\x41\x38", [0x222f] = "\x41\x39", [0x2230] = "\x41\x3a",
  [0x2231] = "\x41\x3b", [0x2232] = "\x41\x3c", [0x2233] = "\x41\x3d",
  [0x2234] = "\x41\x3e", [0x2235] = "\x41\x3f", [0x2236] = "\x41\x40",
  [0x2237] = "\x41\x41", [0x2238] = "\x41\x42", [0x2239] = "\x41\x43",
  [0x223a] = "\x41\x44", [0x223b] = "\x41\x45", [0x223c] = "\x41\x46",
  [0x223d] = "\x41\x47", [0x223e] = "\x41\x48", [0x223f] = "\xa8\x53",
  [0x2240] = "\x41\x49", [0x2241] = "\x41\x4a", [0x2242] = "\x41\x4b",
  [0x2243] = "\x41\x4c", [0x2244] = "\x41\x4d", [0x2245] = "\x41\x4e",
  [0x2246] = "\x41\x4f", [0x2247] = "\x41\x50", [0x2248] = "\x41\x51",
  [0x2249] = "\x41\x52", [0x224a] = "\x41\x53", [0x224b] = "\x41\x54",
  [0x224c] = "\x41\x55", [0x224d] = "\x41\x56", [0x224e] = "\x41\x57",
  [0x224f] = "\x41\x58", [0x2250] = "\x41\x59", [0x2251] = "\x41\x5a",
  [0x2252] = "\x41\x5b", [0x2253] = "\x41\x5c", [0x2254] = "\x41\x5d",
  [0x2255] = "\x41\x5e", [0x2256] = "\x41\x5f", [0x2257] = "\x41\x60",
  [0x2258] = "\x41\x61", [0x2259] = "\x41\x62", [0x225a] = "\x41\x63",
  [0x225b] = "\x41\x64", [0x225c] = "\x41\x65", [0x225d] = "\x41\x66",
  [0x225e] = "\x41\x67", [0x225f] = "\x41\x68", [0x2260] = "\x41\x69",
  [0x2261] = "\x41\x6a", [0x2262] = "\x41\x6b", [0x2263] = "\x41\x6c",
  [0x2264] = "\x41\x6d", [0x2265] = "\x41\x6e", [0x2266] = "\x41\x6f",
  [0x2267] = "\x41\x70", [0x2268] = "\x41\x71", [0x2269] = "\x41\x72",
  [0x226a] = "\x41\x73", [0x226b] = "\x41\x74", [0x226c] = "\x41\x75",
  [0x226d] = "\x41\x76", [0x226e] = "\x41\x77", [0x226f] = "\x41\x78",
  [0x2270] = "\x41\x79", [0x2271] = "\x41\x7a", [0x2272] = "\x41\x7b",
  [0x2273] = "\x41\x7c", [0x2274] = "\x41\x7d", [0x2275] = "\x41\x7e",
  [0x2276] = "\x41\x7f", [0x2277] = "\x41\x80", [0x2278] = "\x41\x81",
  [0x2279] = "\x41\x82", [0x227a] = "\x41\x83", [0x227b] = "\x41\x84",
  [0x227c] = "\x41\x85", [0x227d] = "\x41\x86", [0x227e] = "\x41\x87",
  [0x227f] = "\x41\x88", [0x2280] = "\x41\x89", [0x2281] = "\x41\x8a",
  [0x2282] = "\x41\x8b", [0x2283] = "\x41\x8c", [0x2284] = "\x41\x8d",
  [0x2285] = "\x41\x8e", [0x2286] = "\x41\x8f", [0x2287] = "\x41\x90",
  [0x2288] = "\x41\x91", [0x2289] = "\x41\x92", [0x228a] = "\x41\x93",
  [0x228b] = "\x41\x94", [0x228c] = "\x41\x95", [0x228d] = "\x41\x96",
  [0x228e] = "\x41\x97", [0x228f] = "\x41\x98", [0x2290] = "\x41\x99",
  [0x2291] = "\x41\x9a", [0x2292] = "\xa1\xd0", [0x2293] = "\x41\x9b",
  [0x2294] = "\x41\x9c", [0x2295] = "\x41\x9d", [0x2296] = "\x41\x9e",
  [0x2297] = "\x41\x9f", [0x2298] = "\x41\xa0", [0x2299] = "\x41\xa1",
  [0x229a] = "\x41\xa2", [0x229b] = "\x41\xa3", [0x229c] = "\x41\xa4",
  [0x229d] = "\x41\xa5", [0x229e] = "\x41\xa6", [0x229f] = "\x41\xa7",
  [0x22a0] = "\x41\xa8", [0x22a1] = "\x41\xa9", [0x22a2] = "\x41\xaa",
  [0x22a3] = "\x41\xab", [0x22a4] = "\x41\xac", [0x22a5] = "\x41\xad",
  [0x22a6] = "\x41\xae", [0x22a7] = "\x41\xaf", [0x22a8] = "\x41\xb0",
  [0x22a9] = "\x41\xb1", [0x22aa] = "\x41\xb2", [0x22ab] = "\x41\xb3",
  [0x22ac] = "\x41\xb4", [0x22ad] = "\x41\xb5", [0x22ae] = "\x41\xb6",
  [0x22af] = "\x41\xb7", [0x22b0] = "\x41\xb8", [0x22b1] = "\x41\xb9",
  [0x22b2] = "\x41\xba", [0x22b3] = "\x41\xbb", [0x22b4] = "\x41\xbc",
  [0x22b5] = "\x41\xbd", [0x22b6] = "\x41\xbe", [0x22b7] = "\x41\xbf",
  [0x22b8] = "\x41\xc0", [0x22b9] = "\x41\xc1", [0x22ba] = "\x41\xc2",
  [0x22bb] = "\x41\xc3", [0x22bc] = "\x41\xc4", [0x22bd] = "\x41\xc5",
  [0x22be] = "\x41\xc6", [0x22bf] = "\x41\xc7", [0x22c0] = "\x41\xc8",
  [0x22c1] = "\x41\xc9", [0x22c2] = "\x41\xca", [0x22c3] = "\x41\xcb",
  [0x22c4] = "\x41\xcc", [0x22c5] = "\x41\xcd", [0x22c6] = "\x41\xce",
  [0x22c7] = "\x41\xcf", [0x22c8] = "\x41\xd0", [0x22c9] = "\x41\xd1",
  [0x22ca] = "\x41\xd2", [0x22cb] = "\x41\xd3", [0x22cc] = "\x41\xd4",
  [0x22cd] = "\x41\xd5", [0x22ce] = "\x41\xd6", [0x22cf] = "\x41\xd7",
  [0x22d0] = "\x41\xd8", [0x22d1] = "\x41\xd9", [0x22d2] = "\x41\xda",
  [0x22d3] = "\x41\xdb", [0x22d4] = "\x41\xdc", [0x22d5] = "\x41\xdd",
  [0x22d6] = "\x41\xde", [0x22d7] = "\x41\xdf", [0x22d8] = "\x41\xe0",
  [0x22d9] = "\x41\xe1", [0x22da] = "\x41\xe2", [0x22db] = "\x41\xe3",
  [0x22dc] = "\x41\xe4", [0x22dd] = "\x41\xe5", [0x22de] = "\x41\xe6",
  [0x22df] = "\x41\xe7", [0x22e0] = "\x41\xe8", [0x22e1] = "\x41\xe9",
  [0x22e2] = "\x41\xea", [0x22e3] = "\x41\xeb", [0x22e4] = "\x41\xec",
  [0x22e5] = "\x41\xed", [0x22e6] = "\x41\xee", [0x22e7] = "\x41\xef",
  [0x22e8] = "\x41\xf0", [0x22e9] = "\x41\xf1", [0x22ea] = "\x41\xf2",
  [0x22eb] = "\x41\xf3", [0x22ec] = "\x41\xf4", [0x22ed] = "\x41\xf5",
  [0x22ee] = "\x41\xf6", [0x22ef] = "\x41\xf7", [0x22f0] = "\x41\xf8",
  [0x22f1] = "\x41\xf9", [0x22f2] = "\x41\xfa", [0x22f3] = "\x41\xfb",
  [0x22f4] = "\x41\xfc", [0x22f5] = "\x41\xfd", [0x22f6] = "\x41\xfe",
  [0x22f7] = "\x41\xff", [0x22f8] = "\x42\x00", [0x22f9] = "\x42\x01",
  [0x22fa] = "\x42\x02", [0x22fb] = "\x42\x03", [0x22fc] = "\x42\x04",
  [0x22fd] = "\x42\x05", [0x22fe] = "\x42\x06", [0x22ff] = "\x42\x07",
  [0x2300] = "\x42\x08", [0x2301] = "\x42\x09", [0x2302] = "\x42\x0a",
  [0x2303] = "\x42\x0b", [0x2304] = "\x42\x0c", [0x2305] = "\x42\x0d",
  [0x2306] = "\x42\x0e", [0x2307] = "\x42\x0f", [0x2308] = "\x42\x10",
  [0x2309] = "\x42\x11", [0x230a] = "\x42\x12", [0x230b] = "\x42\x13",
  [0x230c] = "\x42\x14", [0x230d] = "\x42\x15", [0x230e] = "\x42\x16",
  [0x230f] = "\x42\x17", [0x2310] = "\x42\x18", [0x2311] = "\x42\x19",
  [0x2312] = "\x42\x1a", [0x2313] = "\x42\x1b", [0x2314] = "\x42\x1c",
  [0x2315] = "\x42\x1d", [0x2316] = "\x42\x1e", [0x2317] = "\x42\x1f",
  [0x2318] = "\x42\x20", [0x2319] = "\x42\x21", [0x231a] = "\x42\x22",
  [0x231b] = "\x42\x23", [0x231c] = "\x42\x24", [0x231d] = "\x42\x25",
  [0x231e] = "\x42\x26", [0x231f] = "\x42\x27", [0x2320] = "\x42\x28",
  [0x2321] = "\x42\x29", [0x2322] = "\x42\x2a", [0x2323] = "\x42\x2b",
  [0x2324] = "\x42\x2c", [0x2325] = "\x42\x2d", [0x2326] = "\x42\x2e",
  [0x2327] = "\x42\x2f", [0x2328] = "\x42\x30", [0x2329] = "\x42\x31",
  [0x232a] = "\x42\x32", [0x232b] = "\x42\x33", [0x232c] = "\x42\x34",
  [0x232d] = "\x42\x35", [0x232e] = "\x42\x36", [0x232f] = "\x42\x37",
  [0x2330] = "\x42\x38", [0x2331] = "\x42\x39", [0x2332] = "\x42\x3a",
  [0x2333] = "\x42\x3b", [0x2334] = "\x42\x3c", [0x2335] = "\x42\x3d",
  [0x2336] = "\x42\x3e", [0x2337] = "\x42\x3f", [0x2338] = "\x42\x40",
  [0x2339] = "\x42\x41", [0x233a] = "\x42\x42", [0x233b] = "\x42\x43",
  [0x233c] = "\x42\x44", [0x233d] = "\x42\x45", [0x233e] = "\x42\x46",
  [0x233f] = "\x42\x47", [0x2340] = "\x42\x48", [0x2341] = "\x42\x49",
  [0x2342] = "\x42\x4a", [0x2343] = "\x42\x4b", [0x2344] = "\x42\x4c",
  [0x2345] = "\x42\x4d", [0x2346] = "\x42\x4e", [0x2347] = "\x42\x4f",
  [0x2348] = "\x42\x50", [0x2349] = "\x42\x51", [0x234a] = "\x42\x52",
  [0x234b] = "\x42\x53", [0x234c] = "\x42\x54", [0x234d] = "\x42\x55",
  [0x234e] = "\x42\x56", [0x234f] = "\x42\x57", [0x2350] = "\x42\x58",
  [0x2351] = "\x42\x59", [0x2352] = "\x42\x5a", [0x2353] = "\x42\x5b",
  [0x2354] = "\x42\x5c", [0x2355] = "\x42\x5d", [0x2356] = "\x42\x5e",
  [0x2357] = "\x42\x5f", [0x2358] = "\x42\x60", [0x2359] = "\x42\x61",
  [0x235a] = "\x42\x62", [0x235b] = "\x42\x63", [0x235c] = "\x42\x64",
  [0x235d] = "\x42\x65", [0x235e] = "\x42\x66", [0x235f] = "\x42\x67",
  [0x2360] = "\x42\x68", [0x2361] = "\x42\x69", [0x2362] = "\x42\x6a",
  [0x2363] = "\x42\x6b", [0x2364] = "\x42\x6c", [0x2365] = "\x42\x6d",
  [0x2366] = "\x42\x6e", [0x2367] = "\x42\x6f", [0x2368] = "\x42\x70",
  [0x2369] = "\x42\x71", [0x236a] = "\x42\x72", [0x236b] = "\x42\x73",
  [0x236c] = "\x42\x74", [0x236d] = "\x42\x75", [0x236e] = "\x42\x76",
  [0x236f] = "\x42\x77", [0x2370] = "\x42\x78", [0x2371] = "\x42\x79",
  [0x2372] = "\x42\x7a", [0x2373] = "\x42\x7b", [0x2380] = "\x42\x88",
  [0x2381] = "\x42\x89", [0x2382] = "\x42\x8a", [0x2383] = "\x42\x8b",
  [0x2384] = "\x42\x8c", [0x2385] = "\x42\x8d", [0x2386] = "\x42\x8e",
  [0x2387] = "\x42\x8f", [0x2388] = "\x42\x90", [0x2389] = "\x42\x91",
  [0x238a] = "\x42\x92", [0x238b] = "\x42\x93", [0x238c] = "\x42\x94",
  [0x238d] = "\x42\x95", [0x238e] = "\x42\x96", [0x238f] = "\x42\x97",
  [0x2390] = "\x42\x98", [0x2391] = "\x42\x99", [0x2392] = "\x42\x9a",
  [0x2393] = "\x42\x9b", [0x2394] = "\x42\x9c", [0x2395] = "\x42\x9d",
  [0x2396] = "\x42\x9e", [0x2397] = "\x42\x9f", [0x2398] = "\x42\xa0",
  [0x2399] = "\x42\xa1", [0x239a] = "\x42\xa2", [0x239b] = "\x42\xa3",
  [0x239c] = "\x42\xa4", [0x239d] = "\x42\xa5", [0x239e] = "\x42\xa6",
  [0x239f] = "\x42\xa7", [0x23a0] = "\x42\xa8", [0x23a1] = "\x42\xa9",
  [0x23a2] = "\x42\xaa", [0x23a3] = "\x42\xab", [0x23a4] = "\x42\xac",
  [0x23a5] = "\x42\xad", [0x23a6] = "\x42\xae", [0x23c0] = "\x42\xc8",
  [0x23c1] = "\x42\xc9", [0x23c2] = "\x42\xca", [0x23c3] = "\x42\xcb",
  [0x23c4] = "\x42\xcc", [0x23c5] = "\x42\xcd", [0x23c6] = "\x42\xce",
  [0x23c7] = "\x42\xcf", [0x23c8] = "\x42\xd0", [0x23c9] = "\x42\xd1",
  [0x23ca] = "\x42\xd2", [0x23e0] = "\xa2\xd9", [0x23e1] = "\xa2\xda",
  [0x23e2] = "\xa2\xdb", [0x23e3] = "\xa2\xdc", [0x23e4] = "\xa2\xdd",
  [0x23e5] = "\xa2\xde", [0x23e6] = "\xa2\xdf", [0x23e7] = "\xa2\xe0",
  [0x23e8] = "\xa2\xe1", [0x23e9] = "\xa2\xe2", [0x23ea] = "\x42\xe8",
  [0x23eb] = "\x42\xe9", [0x23ec] = "\x42\xea", [0x23ed] = "\x42\xeb",
  [0x23ee] = "\x42\xec", [0x23ef] = "\x42\xed", [0x23f0] = "\x42\xee",
  [0x23f1] = "\x42\xef", [0x23f2] = "\x42\xf0", [0x23f3] = "\x42\xf1",
  [0x23f4] = "\xa2\xc5", [0x23f5] = "\xa2\xc6", [0x23f6] = "\xa2\xc7",
  [0x23f7] = "\xa2\xc8", [0x23f8] = "\xa2\xc9", [0x23f9] = "\xa2\xca",
  [0x23fa] = "\xa2\xcb", [0x23fb] = "\xa2\xcc", [0x23fc] = "\xa2\xcd",
  [0x23fd] = "\xa2\xce", [0x23fe] = "\xa2\xcf", [0x23ff] = "\xa2\xd0",
  [0x2400] = "\xa2\xd1", [0x2401] = "\xa2\xd2", [0x2402] = "\xa2\xd3",
  [0x2403] = "\xa2\xd4", [0x2404] = "\xa2\xd5", [0x2405] = "\xa2\xd6",
  [0x2406] = "\xa2\xd7", [0x2407] = "\xa2\xd8", [0x2408] = "\xa2\xb1",
  [0x2409] = "\xa2\xb2", [0x240a] = "\xa2\xb3", [0x240b] = "\xa2\xb4",
  [0x240c] = "\xa2\xb5", [0x240d] = "\xa2\xb6", [0x240e] = "\xa2\xb7",
  [0x240f] = "\xa2\xb8", [0x2410] = "\xa2\xb9", [0x2411] = "\xa2\xba",
  [0x2412] = "\xa2\xbb", [0x2413] = "\xa2\xbc", [0x2414] = "\xa2\xbd",
  [0x2415] = "\xa2\xbe", [0x2416] = "\xa2\xbf", [0x2417] = "\xa2\xc0",
  [0x2418] = "\xa2\xc1", [0x2419] = "\xa2\xc2", [0x241a] = "\xa2\xc3",
  [0x241b] = "\xa2\xc4", [0x241c] = "\x42\xf2", [0x241d] = "\x42\xf3",
  [0x241e] = "\x42\xf4", [0x241f] = "\x42\xf5", [0x2420] = "\x42\xf6",
  [0x2421] = "\x42\xf7", [0x2422] = "\x42\xf8", [0x2423] = "\x42\xf9",
  [0x2424] = "\x42\xfa", [0x2425] = "\x42\xfb", [0x2426] = "\x42\xfc",
  [0x2427] = "\x42\xfd", [0x2428] = "\x42\xfe", [0x2429] = "\x42\xff",
  [0x242a] = "\x43\x00", [0x242b] = "\x43\x01", [0x242c] = "\x43\x02",
  [0x242d] = "\x43\x03", [0x242e] = "\x43\x04", [0x242f] = "\x43\x05",
  [0x2430] = "\x43\x06", [0x2431] = "\x43\x07", [0x2432] = "\x43\x08",
  [0x2433] = "\x43\x09", [0x2434] = "\x43\x0a", [0x2435] = "\x43\x0b",
  [0x2436] = "\x43\x0c", [0x2437] = "\x43\x0d", [0x2438] = "\x43\x0e",
  [0x2439] = "\x43\x0f", [0x243a] = "\x43\x10", [0x243b] = "\x43\x11",
  [0x243c] = "\x43\x12", [0x243d] = "\x43\x13", [0x243e] = "\x43\x14",
  [0x243f] = "\x43\x15", [0x2440] = "\x43\x16", [0x2441] = "\x43\x17",
  [0x2442] = "\x43\x18", [0x2443] = "\x43\x19", [0x2444] = "\x43\x1a",
  [0x2445] = "\x43\x1b", [0x2446] = "\x43\x1c", [0x2447] = "\x43\x1d",
  [0x2448] = "\x43\x1e", [0x2449] = "\x43\x1f", [0x244a] = "\x43\x20",
  [0x244b] = "\x43\x21", [0x244c] = "\x43\x22", [0x244d] = "\x43\x23",
  [0x244e] = "\x43\x24", [0x244f] = "\x43\x25", [0x2450] = "\x43\x26",
  [0x2451] = "\x43\x27", [0x2452] = "\x43\x28", [0x2453] = "\x43\x29",
  [0x2454] = "\x43\x2a", [0x2455] = "\x43\x2b", [0x2456] = "\x43\x2c",
  [0x2457] = "\x43\x2d", [0x2458] = "\x43\x2e", [0x2459] = "\x43\x2f",
  [0x245a] = "\x43\x30", [0x245b] = "\x43\x31", [0x245c] = "\x43\x32",
  [0x245d] = "\x43\x33", [0x245e] = "\x43\x34", [0x245f] = "\x43\x35",
  [0x2460] = "\x43\x36", [0x2461] = "\x43\x37", [0x2462] = "\x43\x38",
  [0x2463] = "\x43\x39", [0x2464] = "\x43\x3a", [0x2465] = "\x43\x3b",
  [0x2466] = "\x43\x3c", [0x2467] = "\x43\x3d", [0x2468] = "\x43\x3e",
  [0x2469] = "\x43\x3f", [0x246a] = "\x43\x40", [0x246b] = "\x43\x41",
  [0x246c] = "\x43\x42", [0x246d] = "\x43\x43", [0x246e] = "\x43\x44",
  [0x246f] = "\x43\x45", [0x2470] = "\x43\x46", [0x2471] = "\x43\x47",
  [0x2472] = "\x43\x48", [0x2473] = "\x43\x49", [0x2474] = "\x43\x4a",
  [0x2475] = "\x43\x4b", [0x2476] = "\x43\x4c", [0x2477] = "\x43\x4d",
  [0x2478] = "\x43\x4e", [0x2479] = "\x43\x4f", [0x247a] = "\x43\x50",
  [0x247b] = "\x43\x51", [0x247c] = "\x43\x52", [0x247d] = "\x43\x53",
  [0x247e] = "\x43\x54", [0x247f] = "\x43\x55", [0x2480] = "\xa9\xa4",
  [0x2481] = "\xa9\xa5", [0x2482] = "\xa9\xa6", [0x2483] = "\xa9\xa7",
  [0x2484] = "\xa9\xa8", [0x2485] = "\xa9\xa9", [0x2486] = "\xa9\xaa",
  [0x2487] = "\xa9\xab", [0x2488] = "\xa9\xac", [0x2489] = "\xa9\xad",
  [0x248a] = "\xa9\xae", [0x248b] = "\xa9\xaf", [0x248c] = "\xa9\xb0",
  [0x248d] = "\xa9\xb1", [0x248e] = "\xa9\xb2", [0x248f] = "\xa9\xb3",
  [0x2490] = "\xa9\xb4", [0x2491] = "\xa9\xb5", [0x2492] = "\xa9\xb6",
  [0x2493] = "\xa9\xb7", [0x2494] = "\xa9\xb8", [0x2495] = "\xa9\xb9",
  [0x2496] = "\xa9\xba", [0x2497] = "\xa9\xbb", [0x2498] = "\xa9\xbc",
  [0x2499] = "\xa9\xbd", [0x249a] = "\xa9\xbe", [0x249b] = "\xa9\xbf",
  [0x249c] = "\xa9\xc0", [0x249d] = "\xa9\xc1", [0x249e] = "\xa9\xc2",
  [0x249f] = "\xa9\xc3", [0x24a0] = "\xa9\xc4", [0x24a1] = "\xa9\xc5",
  [0x24a2] = "\xa9\xc6", [0x24a3] = "\xa9\xc7", [0x24a4] = "\xa9\xc8",
  [0x24a5] = "\xa9\xc9", [0x24a6] = "\xa9\xca", [0x24a7] = "\xa9\xcb",
  [0x24a8] = "\xa9\xcc", [0x24a9] = "\xa9\xcd", [0x24aa] = "\xa9\xce",
  [0x24ab] = "\xa9\xcf", [0x24ac] = "\xa9\xd0", [0x24ad] = "\xa9\xd1",
  [0x24ae] = "\xa9\xd2", [0x24af] = "\xa9\xd3", [0x24b0] = "\xa9\xd4",
  [0x24b1] = "\xa9\xd5", [0x24b2] = "\xa9\xd6", [0x24b3] = "\xa9\xd7",
  [0x24b4] = "\xa9\xd8", [0x24b5] = "\xa9\xd9", [0x24b6] = "\xa9\xda",
  [0x24b7] = "\xa9\xdb", [0x24b8] = "\xa9\xdc", [0x24b9] = "\xa9\xdd",
  [0x24ba] = "\xa9\xde", [0x24bb] = "\xa9\xdf", [0x24bc] = "\xa9\xe0",
  [0x24bd] = "\xa9\xe1", [0x24be] = "\xa9\xe2", [0x24bf] = "\xa9\xe3",
  [0x24c0] = "\xa9\xe4", [0x24c1] = "\xa9\xe5", [0x24c2] = "\xa9\xe6",
  [0x24c3] = "\xa9\xe7", [0x24c4] = "\xa9\xe8", [0x24c5] = "\xa9\xe9",
  [0x24c6] = "\xa9\xea", [0x24c7] = "\xa9\xeb", [0x24c8] = "\xa9\xec",
  [0x24c9] = "\xa9\xed", [0x24ca] = "\xa9\xee", [0x24cb] = "\xa9\xef",
  [0x24cc] = "\x43\x56", [0x24cd] = "\x43\x57", [0x24ce] = "\x43\x58",
  [0x24cf] = "\x43\x59", [0x24d0] = "\xa8\x54", [0x24d1] = "\xa8\x55",
  [0x24d2] = "\xa8\x56", [0x24d3] = "\xa8\x57", [0x24d4] = "\xa8\x58",
  [0x24d5] = "\xa8\x59", [0x24d6] = "\xa8\x5a", [0x24d7] = "\xa8\x5b",
  [0x24d8] = "\xa8\x5c", [0x24d9] = "\xa8\x5d", [0x24da] = "\xa8\x5e",
  [0x24db] = "\xa8\x5f", [0x24dc] = "\xa8\x60", [0x24dd] = "\xa8\x61",
  [0x24de] = "\xa8\x62", [0x24df] = "\xa8\x63", [0x24e0] = "\xa8\x64",
  [0x24e1] = "\xa8\x65", [0x24e2] = "\xa8\x66", [0x24e3] = "\xa8\x67",
  [0x24e4] = "\xa8\x68", [0x24e5] = "\xa8\x69", [0x24e6] = "\xa8\x6a",
  [0x24e7] = "\xa8\x6b", [0x24e8] = "\xa8\x6c", [0x24e9] = "\xa8\x6d",
  [0x24ea] = "\xa8\x6e", [0x24eb] = "\xa8\x6f", [0x24ec] = "\xa8\x70",
  [0x24ed] = "\xa8\x71", [0x24ee] = "\xa8\x72", [0x24ef] = "\xa8\x73",
  [0x24f0] = "\xa8\x74", [0x24f1] = "\xa8\x75", [0x24f2] = "\xa8\x76",
  [0x24f3] = "\xa8\x77", [0x24f4] = "\x43\x5a", [0x24f5] = "\x43\x5b",
  [0x24f6] = "\x43\x5c", [0x24f7] = "\x43\x5d", [0x24f8] = "\x43\x5e",
  [0x24f9] = "\x43\x5f", [0x24fa] = "\x43\x60", [0x24fb] = "\x43\x61",
  [0x24fc] = "\x43\x62", [0x24fd] = "\x43\x63", [0x24fe] = "\x43\x64",
  [0x24ff] = "\x43\x65", [0x2500] = "\x43\x66", [0x2501] = "\xa8\x78",
  [0x2502] = "\xa8\x79", [0x2503] = "\xa8\x7a", [0x2504] = "\xa8\x7b",
  [0x2505] = "\xa8\x7c", [0x2506] = "\xa8\x7d", [0x2507] = "\xa8\x7e",
  [0x2508] = "\xa8\x80", [0x2509] = "\xa8\x81", [0x250a] = "\xa8\x82",
  [0x250b] = "\xa8\x83", [0x250c] = "\xa8\x84", [0x250d] = "\xa8\x85",
  [0x250e] = "\xa8\x86", [0x250f] = "\xa8\x87", [0x2510] = "\x43\x67",
  [0x2511] = "\x43\x68", [0x2512] = "\x43\x69", [0x2513] = "\xa8\x88",
  [0x2514] = "\xa8\x89", [0x2515] = "\xa8\x8a", [0x2516] = "\x43\x6a",
  [0x2517] = "\x43\x6b", [0x2518] = "\x43\x6c", [0x2519] = "\x43\x6d",
  [0x251a] = "\x43\x6e", [0x251b] = "\x43\x6f", [0x251c] = "\x43\x70",
  [0x251d] = "\x43\x71", [0x251e] = "\x43\x72", [0x251f] = "\x43\x73",
  [0x2520] = "\xa1\xf6", [0x2521] = "\xa1\xf5", [0x2522] = "\x43\x74",
  [0x2523] = "\x43\x75", [0x2524] = "\x43\x76", [0x2525] = "\x43\x77",
  [0x2526] = "\x43\x78", [0x2527] = "\x43\x79", [0x2528] = "\x43\x7a",
  [0x2529] = "\x43\x7b", [0x252a] = "\x43\x7c", [0x252b] = "\x43\x7d",
  [0x252c] = "\x43\x7e", [0x252d] = "\x43\x7f", [0x252e] = "\x43\x80",
  [0x252f] = "\x43\x81", [0x2530] = "\x43\x82", [0x2531] = "\x43\x83",
  [0x2532] = "\xa1\xf8", [0x2533] = "\xa1\xf7", [0x2534] = "\x43\x84",
  [0x2535] = "\x43\x85", [0x2536] = "\x43\x86", [0x2537] = "\x43\x87",
  [0x2538] = "\x43\x88", [0x2539] = "\x43\x89", [0x253a] = "\x43\x8a",
  [0x253b] = "\x43\x8b", [0x253c] = "\xa8\x8b", [0x253d] = "\xa8\x8c",
  [0x253e] = "\x43\x8c", [0x253f] = "\x43\x8d", [0x2540] = "\x43\x8e",
  [0x2541] = "\x43\x8f", [0x2542] = "\x43\x90", [0x2543] = "\x43\x91",
  [0x2544] = "\x43\x92", [0x2545] = "\x43\x93", [0x2546] = "\xa1\xf4",
  [0x2547] = "\xa1\xf3", [0x2548] = "\x43\x94", [0x2549] = "\x43\x95",
  [0x254a] = "\x43\x96", [0x254b] = "\xa1\xf0", [0x254c] = "\x43\x97",
  [0x254d] = "\x43\x98", [0x254e] = "\xa1\xf2", [0x254f] = "\xa1\xf1",
  [0x2550] = "\x43\x99", [0x2551] = "\x43\x9a", [0x2552] = "\x43\x9b",
  [0x2553] = "\x43\x9c", [0x2554] = "\x43\x9d", [0x2555] = "\x43\x9e",
  [0x2556] = "\x43\x9f", [0x2557] = "\x43\xa0", [0x2558] = "\x43\xa1",
  [0x2559] = "\x43\xa2", [0x255a] = "\x43\xa3", [0x255b] = "\x43\xa4",
  [0x255c] = "\x43\xa5", [0x255d] = "\x43\xa6", [0x255e] = "\x43\xa7",
  [0x255f] = "\x43\xa8", [0x2560] = "\x43\xa9", [0x2561] = "\x43\xaa",
  [0x2562] = "\xa8\x8d", [0x2563] = "\xa8\x8e", [0x2564] = "\xa8\x8f",
  [0x2565] = "\xa8\x90", [0x2566] = "\x43\xab", [0x2567] = "\x43\xac",
  [0x2568] = "\x43\xad", [0x2569] = "\x43\xae", [0x256a] = "\x43\xaf",
  [0x256b] = "\x43\xb0", [0x256c] = "\x43\xb1", [0x256d] = "\x43\xb2",
  [0x256e] = "\x43\xb3", [0x256f] = "\x43\xb4", [0x2570] = "\x43\xb5",
  [0x2571] = "\x43\xb6", [0x2572] = "\x43\xb7", [0x2573] = "\x43\xb8",
  [0x2574] = "\x43\xb9", [0x2575] = "\x43\xba", [0x2576] = "\x43\xbb",
  [0x2577] = "\x43\xbc", [0x2578] = "\x43\xbd", [0x2579] = "\x43\xbe",
  [0x257a] = "\x43\xbf", [0x257b] = "\x43\xc0", [0x257c] = "\x43\xc1",
  [0x257d] = "\x43\xc2", [0x257e] = "\x43\xc3", [0x257f] = "\x43\xc4",
  [0x2580] = "\x43\xc5", [0x2581] = "\x43\xc6", [0x2582] = "\x43\xc7",
  [0x2583] = "\x43\xc8", [0x2584] = "\x43\xc9", [0x2585] = "\xa1\xef",
  [0x2586] = "\xa1\xee", [0x2587] = "\x43\xca", [0x2588] = "\x43\xcb",
  [0x2589] = "\xa8\x91", [0x258a] = "\x43\xcc", [0x258b] = "\x43\xcd",
  [0x258c] = "\x43\xce", [0x258d] = "\x43\xcf", [0x258e] = "\x43\xd0",
  [0x258f] = "\x43\xd1", [0x2590] = "\x43\xd2", [0x2591] = "\x43\xd3",
  [0x2592] = "\x43\xd4", [0x2593] = "\x43\xd5", [0x2594] = "\x43\xd6",
  [0x2595] = "\x43\xd7", [0x2596] = "\x43\xd8", [0x2597] = "\x43\xd9",
  [0x2598] = "\x43\xda", [0x2599] = "\x43\xdb", [0x259a] = "\x43\xdc",
  [0x259b] = "\x43\xdd", [0x259c] = "\x43\xde", [0x259d] = "\x43\xdf",
  [0x259e] = "\x43\xe0", [0x259f] = "\x43\xe1", [0x25a0] = "\x43\xe2",
  [0x25a1] = "\x43\xe3", [0x25a2] = "\x43\xe4", [0x25a3] = "\x43\xe5",
  [0x25a4] = "\x43\xe6", [0x25a5] = "\x43\xe7", [0x25a6] = "\x43\xe8",
  [0x25a7] = "\x43\xe9", [0x25a8] = "\x43\xea", [0x25a9] = "\x43\xeb",
  [0x25aa] = "\x43\xec", [0x25ab] = "\x43\xed", [0x25ac] = "\x43\xee",
  [0x25ad] = "\x43\xef", [0x25ae] = "\x43\xf0", [0x25af] = "\x43\xf1",
  [0x25b0] = "\x43\xf2", [0x25b1] = "\x43\xf3", [0x25b2] = "\x43\xf4",
  [0x25b3] = "\x43\xf5", [0x25b4] = "\x43\xf6", [0x25b5] = "\x43\xf7",
  [0x25b6] = "\x43\xf8", [0x25b7] = "\x43\xf9", [0x25b8] = "\x43\xfa",
  [0x25b9] = "\x43\xfb", [0x25ba] = "\x43\xfc", [0x25bb] = "\x43\xfd",
  [0x25bc] = "\x43\xfe", [0x25bd] = "\x43\xff", [0x25be] = "\x44\x00",
  [0x25bf] = "\x44\x01", [0x25c0] = "\xa1\xe2", [0x25c1] = "\x44\x02",
  [0x25c2] = "\xa1\xe1", [0x2e01] = "\xfe\x50", [0x2e02] = "\x4c\x41",
  [0x2e03] = "\x4c\x42", [0x2e04] = "\xfe\x54", [0x2e05] = "\x4c\x43",
  [0x2e06] = "\x4c\x44", [0x2e07] = "\x4c\x45", [0x2e08] = "\xfe\x57",
  [0x2e09] = "\x4c\x46", [0x2e0a] = "\x4c\x47", [0x2e0b] = "\xfe\x58",
  [0x2e0c] = "\xfe\x5d", [0x2e0d] = "\x4c\x48", [0x2e0e] = "\x4c\x49",
  [0x2e0f] = "\x4c\x4a", [0x2e10] = "\x4c\x4b", [0x2e11] = "\x4c\x4c",
  [0x2e12] = "\x4c\x4d", [0x2e13] = "\x4c\x4e", [0x2e14] = "\x4c\x4f",
  [0x2e15] = "\x4c\x50", [0x2e16] = "\x4c\x51", [0x2e17] = "\xfe\x5e",
  [0x2e18] = "\x4c\x52", [0x2e19] = "\x4c\x53", [0x2e1b] = "\x4c\x55",
  [0x2e1c] = "\x4c\x56", [0x2e1d] = "\x4c\x57", [0x2e1e] = "\x4c\x58",
  [0x2e1f] = "\x4c\x59", [0x2e20] = "\x4c\x5a", [0x2e21] = "\x4c\x5b",
  [0x2e22] = "\x4c\x5c", [0x2e23] = "\x4c\x5d", [0x2e24] = "\x4c\x5e",
  [0x2e25] = "\x4c\x5f", [0x2e26] = "\x4c\x60", [0x2e27] = "\xfe\x6b",
  [0x2e28] = "\x4c\x61", [0x2e29] = "\x4c\x62", [0x2e2a] = "\xfe\x6e",
  [0x2e2b] = "\x4c\x63", [0x2e2c] = "\x4c\x64", [0x2e2d] = "\x4c\x65",
  [0x2e2e] = "\xfe\x71", [0x2e2f] = "\x4c\x66", [0x2e30] = "\x4c\x67",
  [0x2e31] = "\x4c\x68", [0x2e32] = "\x4c\x69", [0x2e33] = "\xfe\x73",
  [0x2e34] = "\x4c\x6a", [0x2e35] = "\x4c\x6b", [0x2e36] = "\xfe\x74",
  [0x2e37] = "\xfe\x75", [0x2e38] = "\x4c\x6c", [0x2e39] = "\x4c\x6d",
  [0x2e3a] = "\x4c\x6e", [0x2e3b] = "\xfe\x79", [0x2e3c] = "\x4c\x6f",
  [0x2e3d] = "\x4c\x70", [0x2e3e] = "\x4c\x71", [0x2e3f] = "\x4c\x72",
  [0x2e40] = "\x4c\x73", [0x2e41] = "\x4c\x74", [0x2e42] = "\x4c\x75",
  [0x2e43] = "\x4c\x76", [0x2e44] = "\x4c\x77", [0x2e45] = "\x4c\x78",
  [0x2e46] = "\x4c\x79", [0x2e47] = "\x4c\x7a", [0x2e48] = "\x4c\x7b",
  [0x2e49] = "\x4c\x7c", [0x2e4a] = "\xfe\x84", [0x2e4b] = "\x4c\x7d",
  [0x2e4c] = "\x4c\x7e", [0x2e4d] = "\x4c\x7f", [0x2e4e] = "\x4c\x80",
  [0x2e4f] = "\x4c\x81", [0x2e50] = "\x4c\x82", [0x2e51] = "\x4c\x83",
  [0x2e52] = "\x4c\x84", [0x2e53] = "\x4c\x85", [0x2e54] = "\x4c\x86",
  [0x2e55] = "\x4c\x87", [0x2e56] = "\x4c\x88", [0x2e57] = "\x4c\x89",
  [0x2e58] = "\x4c\x8a", [0x2e59] = "\x4c\x8b", [0x2e5a] = "\x4c\x8c",
  [0x2e5b] = "\x4c\x8d", [0x2e5c] = "\x4c\x8e", [0x2e5d] = "\x4c\x8f",
  [0x2e5e] = "\x4c\x90", [0x2e5f] = "\x4c\x91", [0x2e60] = "\x4c\x92",
  [0x2e61] = "\x4c\x93", [0x2e62] = "\x4c\x94", [0x2e63] = "\x4c\x95",
  [0x2e64] = "\x4c\x96", [0x2e65] = "\x4c\x97", [0x2e66] = "\x4c\x98",
  [0x2e67] = "\x4c\x99", [0x2e68] = "\x4c\x9a", [0x2e69] = "\x4c\x9b",
  [0x2e6a] = "\x4c\x9c", [0x2e6b] = "\x4c\x9d", [0x2e6c] = "\x4c\x9e",
  [0x2e6d] = "\x4c\x9f", [0x2e6e] = "\x4c\xa0", [0x2e6f] = "\x4c\xa1",
  [0x2e70] = "\x4c\xa2", [0x2e71] = "\x4c\xa3", [0x2e72] = "\x4c\xa4",
  [0x2e73] = "\x4c\xa5", [0x2e80] = "\x4c\xb2", [0x2e81] = "\x4c\xb3",
  [0x2e82] = "\x4c\xb4", [0x2e83] = "\x4c\xb5", [0x2e84] = "\x4c\xb6",
  [0x2e85] = "\x4c\xb7", [0x2e86] = "\x4c\xb8", [0x2e87] = "\x4c\xb9",
  [0x2e88] = "\x4c\xba", [0x2e89] = "\x4c\xbb", [0x2e8a] = "\x4c\xbc",
  [0x2e8b] = "\x4c\xbd", [0x2e8c] = "\x4c\xbe", [0x2e8d] = "\x4c\xbf",
  [0x2e8e] = "\x4c\xc0", [0x2e8f] = "\x4c\xc1", [0x2e90] = "\x4c\xc2",
  [0x2e91] = "\x4c\xc3", [0x2e92] = "\x4c\xc4", [0x2e93] = "\x4c\xc5",
  [0x2e94] = "\x4c\xc6", [0x2e95] = "\x4c\xc7", [0x2e96] = "\x4c\xc8",
  [0x2e97] = "\x4c\xc9", [0x2e98] = "\x4c\xca", [0x2e99] = "\x4c\xcb",
  [0x2e9a] = "\x4c\xcc", [0x2e9b] = "\x4c\xcd", [0x2e9c] = "\x4c\xce",
  [0x2e9d] = "\x4c\xcf", [0x2e9e] = "\x4c\xd0", [0x2e9f] = "\x4c\xd1",
  [0x2ea0] = "\x4c\xd2", [0x2ea1] = "\x4c\xd3", [0x2ea2] = "\x4c\xd4",
  [0x2ea3] = "\x4c\xd5", [0x2ea4] = "\x4c\xd6", [0x2ea5] = "\x4c\xd7",
  [0x2ea6] = "\x4c\xd8", [0x2ea7] = "\x4c\xd9", [0x2ea8] = "\x4c\xda",
  [0x2ea9] = "\x4c\xdb", [0x2eaa] = "\x4c\xdc", [0x2eab] = "\x4c\xdd",
  [0x2eac] = "\x4c\xde", [0x2ead] = "\x4c\xdf", [0x2eae] = "\x4c\xe0",
  [0x2eaf] = "\x4c\xe1", [0x2eb0] = "\x4c\xe2", [0x2eb1] = "\x4c\xe3",
  [0x2eb2] = "\x4c\xe4", [0x2eb3] = "\x4c\xe5", [0x2eb4] = "\x4c\xe6",
  [0x2eb5] = "\x4c\xe7", [0x2eb6] = "\x4c\xe8", [0x2eb7] = "\x4c\xe9",
  [0x2eb8] = "\x4c\xea", [0x2eb9] = "\x4c\xeb", [0x2eba] = "\x4c\xec",
  [0x2ebb] = "\x4c\xed", [0x2ebc] = "\x4c\xee", [0x2ebd] = "\x4c\xef",
  [0x2ebe] = "\x4c\xf0", [0x2ebf] = "\x4c\xf1", [0x2ec0] = "\x4c\xf2",
  [0x2ec1] = "\x4c\xf3", [0x2ec2] = "\x4c\xf4", [0x2ec3] = "\x4c\xf5",
  [0x2ec4] = "\x4c\xf6", [0x2ec5] = "\x4c\xf7", [0x2ec6] = "\x4c\xf8",
  [0x2ec7] = "\x4c\xf9", [0x2ec8] = "\x4c\xfa", [0x2ec9] = "\x4c\xfb",
  [0x2eca] = "\x4c\xfc", [0x2ecb] = "\x4c\xfd", [0x2ecc] = "\x4c\xfe",
  [0x2ecd] = "\x4c\xff", [0x2ece] = "\x4d\x00", [0x2ecf] = "\x4d\x01",
  [0x2ed0] = "\x4d\x02", [0x2ed1] = "\x4d\x03", [0x2ed2] = "\x4d\x04",
  [0x2ed3] = "\x4d\x05", [0x2ed4] = "\x4d\x06", [0x2ed5] = "\x4d\x07",
  [0x2ed6] = "\x4d\x08", [0x2ed7] = "\x4d\x09", [0x2ed8] = "\x4d\x0a",
  [0x2ed9] = "\x4d\x0b", [0x2eda] = "\x4d\x0c", [0x2edb] = "\x4d\x0d",
  [0x2edc] = "\x4d\x0e", [0x2edd] = "\x4d\x0f", [0x2ede] = "\x4d\x10",
  [0x2edf] = "\x4d\x11", [0x2ee0] = "\x4d\x12", [0x2ee1] = "\x4d\x13",
  [0x2ee2] = "\x4d\x14", [0x2ee3] = "\x4d\x15", [0x2ee4] = "\x4d\x16",
  [0x2ee5] = "\x4d\x17", [0x2ee6] = "\x4d\x18", [0x2ee7] = "\x4d\x19",
  [0x2ee8] = "\x4d\x1a", [0x2ee9] = "\x4d\x1b", [0x2eea] = "\x4d\x1c",
  [0x2eeb] = "\x4d\x1d", [0x2eec] = "\x4d\x1e", [0x2eed] = "\x4d\x1f",
  [0x2eee] = "\x4d\x20", [0x2eef] = "\x4d\x21", [0x2ef0] = "\x4d\x22",
  [0x2ef1] = "\x4d\x23", [0x2ef2] = "\x4d\x24", [0x2ef3] = "\x4d\x25",
  [0x2ef4] = "\x4d\x26", [0x2ef5] = "\x4d\x27", [0x2ef6] = "\x4d\x28",
  [0x2ef7] = "\x4d\x29", [0x2ef8] = "\x4d\x2a", [0x2ef9] = "\x4d\x2b",
  [0x2efa] = "\x4d\x2c", [0x2efb] = "\x4d\x2d", [0x2efc] = "\x4d\x2e",
  [0x2efd] = "\x4d\x2f", [0x2efe] = "\x4d\x30", [0x2eff] = "\x4d\x31",
  [0x2f00] = "\x4d\x32", [0x2f01] = "\x4d\x33", [0x2f02] = "\x4d\x34",
  [0x2f03] = "\x4d\x35", [0x2f04] = "\x4d\x36", [0x2f05] = "\x4d\x37",
  [0x2f06] = "\x4d\x38", [0x2f07] = "\x4d\x39", [0x2f08] = "\x4d\x3a",
  [0x2f09] = "\x4d\x3b", [0x2f0a] = "\x4d\x3c", [0x2f0b] = "\x4d\x3d",
  [0x2f0c] = "\x4d\x3e", [0x2f0d] = "\x4d\x3f", [0x2f0e] = "\x4d\x40",
  [0x2f0f] = "\x4d\x41", [0x2f10] = "\x4d\x42", [0x2f11] = "\x4d\x43",
  [0x2f12] = "\x4d\x44", [0x2f13] = "\x4d\x45", [0x2f14] = "\x4d\x46",
  [0x2f15] = "\x4d\x47", [0x2f16] = "\x4d\x48", [0x2f17] = "\x4d\x49",
  [0x2f18] = "\x4d\x4a", [0x2f19] = "\x4d\x4b", [0x2f1a] = "\x4d\x4c",
  [0x2f1b] = "\x4d\x4d", [0x2f1c] = "\x4d\x4e", [0x2f1d] = "\x4d\x4f",
  [0x2f1e] = "\x4d\x50", [0x2f1f] = "\x4d\x51", [0x2f20] = "\x4d\x52",
  [0x2f21] = "\x4d\x53", [0x2f22] = "\x4d\x54", [0x2f23] = "\x4d\x55",
  [0x2f24] = "\x4d\x56", [0x2f25] = "\x4d\x57", [0x2f26] = "\x4d\x58",
  [0x2f27] = "\x4d\x59", [0x2f28] = "\x4d\x5a", [0x2f29] = "\x4d\x5b",
  [0x2f2a] = "\x4d\x5c", [0x2f2b] = "\x4d\x5d", [0x2f2c] = "\x4d\x5e",
  [0x2f2d] = "\x4d\x5f", [0x2f2e] = "\x4d\x60", [0x2f2f] = "\x4d\x61",
  [0x2f30] = "\x4d\x62", [0x2f31] = "\x4d\x63", [0x2f32] = "\x4d\x64",
  [0x2f33] = "\x4d\x65", [0x2f34] = "\x4d\x66", [0x2f35] = "\x4d\x67",
  [0x2f36] = "\x4d\x68", [0x2f37] = "\x4d\x69", [0x2f38] = "\x4d\x6a",
  [0x2f39] = "\x4d\x6b", [0x2f3a] = "\x4d\x6c", [0x2f3b] = "\x4d\x6d",
  [0x2f3c] = "\x4d\x6e", [0x2f3d] = "\x4d\x6f", [0x2f3e] = "\x4d\x70",
  [0x2f3f] = "\x4d\x71", [0x2f40] = "\x4d\x72", [0x2f41] = "\x4d\x73",
  [0x2f42] = "\x4d\x74", [0x2f43] = "\x4d\x75", [0x2f44] = "\x4d\x76",
  [0x2f45] = "\x4d\x77", [0x2f46] = "\x4d\x78", [0x2f47] = "\x4d\x79",
  [0x2f48] = "\x4d\x7a", [0x2f49] = "\x4d\x7b", [0x2f4a] = "\x4d\x7c",
  [0x2f4b] = "\x4d\x7d", [0x2f4c] = "\x4d\x7e", [0x2f4d] = "\x4d\x7f",
  [0x2f4e] = "\x4d\x80", [0x2f4f] = "\x4d\x81", [0x2f50] = "\x4d\x82",
  [0x2f51] = "\x4d\x83", [0x2f52] = "\x4d\x84", [0x2f53] = "\x4d\x85",
  [0x2f54] = "\x4d\x86", [0x2f55] = "\x4d\x87", [0x2f70] = "\xa9\x8a",
  [0x2f71] = "\xa9\x8b", [0x2f72] = "\xa9\x8c", [0x2f73] = "\xa9\x8d",
  [0x2f74] = "\xa9\x8e", [0x2f75] = "\xa9\x8f", [0x2f76] = "\xa9\x90",
  [0x2f77] = "\xa9\x91", [0x2f78] = "\xa9\x92", [0x2f79] = "\xa9\x93",
  [0x2f7a] = "\xa9\x94", [0x2f7b] = "\xa9\x95", [0x2f80] = "\xa1\xa1",
  [0x2f81] = "\xa1\xa2", [0x2f82] = "\xa1\xa3", [0x2f83] = "\xa1\xa8",
  [0x2f84] = "\x4d\xa6", [0x2f85] = "\xa1\xa9", [0x2f86] = "\xa9\x65",
  [0x2f87] = "\xa9\x96", [0x2f88] = "\xa1\xb4", [0x2f89] = "\xa1\xb5",
  [0x2f8a] = "\xa1\xb6", [0x2f8b] = "\xa1\xb7", [0x2f8c] = "\xa1\xb8",
  [0x2f8d] = "\xa1\xb9", [0x2f8e] = "\xa1\xba", [0x2f8f] = "\xa1\xbb",
  [0x2f90] = "\xa1\xbe", [0x2f91] = "\xa1\xbf", [0x2f92] = "\xa8\x93",
  [0x2f93] = "\xa1\xfe", [0x2f94] = "\xa1\xb2", [0x2f95] = "\xa1\xb3",
  [0x2f96] = "\xa1\xbc", [0x2f97] = "\xa1\xbd", [0x2f98] = "\x4d\xa7",
  [0x2f99] = "\x4d\xa8", [0x2f9a] = "\x4d\xa9", [0x2f9b] = "\x4d\xaa",
  [0x2f9c] = "\x4d\xab", [0x2f9d] = "\xa8\x94", [0x2f9e] = "\xa8\x95",
  [0x2f9f] = "\x4d\xac", [0x2fa0] = "\x4d\xad", [0x2fa1] = "\xa9\x40",
  [0x2fa2] = "\xa9\x41", [0x2fa3] = "\xa9\x42", [0x2fa4] = "\xa9\x43",
  [0x2fa5] = "\xa9\x44", [0x2fa6] = "\xa9\x45", [0x2fa7] = "\xa9\x46",
  [0x2fa8] = "\xa9\x47", [0x2fa9] = "\xa9\x48", [0x2faa] = "\x4d\xae",
  [0x2fab] = "\x4d\xaf", [0x2fac] = "\x4d\xb0", [0x2fad] = "\x4d\xb1",
  [0x2fae] = "\x4d\xb2", [0x2faf] = "\x4d\xb3", [0x2fb0] = "\x4d\xb4",
  [0x2fb1] = "\x4d\xb5", [0x2fb2] = "\x4d\xb6", [0x2fb3] = "\x4d\xb7",
  [0x2fb4] = "\x4d\xb8", [0x2fb5] = "\x4d\xb9", [0x2fb6] = "\x4d\xba",
  [0x2fb7] = "\x4d\xbb", [0x2fb8] = "\x4d\xbc", [0x2fb9] = "\x4d\xbd",
  [0x2fba] = "\x4d\xbe", [0x2fbb] = "\x4d\xbf", [0x2fbc] = "\x4d\xc0",
  [0x2fbd] = "\x4d\xc1", [0x2fbe] = "\xa9\x89", [0x2fbf] = "\x4d\xc2",
  [0x2fc1] = "\xa4\xa1", [0x2fc2] = "\xa4\xa2", [0x2fc3] = "\xa4\xa3",
  [0x2fc4] = "\xa4\xa4", [0x2fc5] = "\xa4\xa5", [0x2fc6] = "\xa4\xa6",
  [0x2fc7] = "\xa4\xa7", [0x2fc8] = "\xa4\xa8", [0x2fc9] = "\xa4\xa9",
  [0x2fca] = "\xa4\xaa", [0x2fcb] = "\xa4\xab", [0x2fcc] = "\xa4\xac",
  [0x2fcd] = "\xa4\xad", [0x2fce] = "\xa4\xae", [0x2fcf] = "\xa4\xaf",
  [0x2fd0] = "\xa4\xb0", [0x2fd1] = "\xa4\xb1", [0x2fd2] = "\xa4\xb2",
  [0x2fd3] = "\xa4\xb3", [0x2fd4] = "\xa4\xb4", [0x2fd5] = "\xa4\xb5",
  [0x2fd6] = "\xa4\xb6", [0x2fd7] = "\xa4\xb7", [0x2fd8] = "\xa4\xb8",
  [0x2fd9] = "\xa4\xb9", [0x2fda] = "\xa4\xba", [0x2fdb] = "\xa4\xbb",
  [0x2fdc] = "\xa4\xbc", [0x2fdd] = "\xa4\xbd", [0x2fde] = "\xa4\xbe",
  [0x2fdf] = "\xa4\xbf", [0x2fe0] = "\xa4\xc0", [0x2fe1] = "\xa4\xc1",
  [0x2fe2] = "\xa4\xc2", [0x2fe3] = "\xa4\xc3", [0x2fe4] = "\xa4\xc4",
  [0x2fe5] = "\xa4\xc5", [0x2fe6] = "\xa4\xc6", [0x2fe7] = "\xa4\xc7",
  [0x2fe8] = "\xa4\xc8", [0x2fe9] = "\xa4\xc9", [0x2fea] = "\xa4\xca",
  [0x2feb] = "\xa4\xcb", [0x2fec] = "\xa4\xcc", [0x2fed] = "\xa4\xcd",
  [0x2fee] = "\xa4\xce", [0x2fef] = "\xa4\xcf", [0x2ff0] = "\xa4\xd0",
  [0x2ff1] = "\xa4\xd1", [0x2ff2] = "\xa4\xd2", [0x2ff3] = "\xa4\xd3",
  [0x2ff4] = "\xa4\xd4", [0x2ff5] = "\xa4\xd5", [0x2ff6] = "\xa4\xd6",
  [0x2ff7] = "\xa4\xd7", [0x2ff8] = "\xa4\xd8", [0x2ff9] = "\xa4\xd9",
  [0x2ffa] = "\xa4\xda", [0x2ffb] = "\xa4\xdb", [0x2ffc] = "\xa4\xdc",
  [0x2ffd] = "\xa4\xdd", [0x2ffe] = "\xa4\xde", [0x2fff] = "\xa4\xdf",
  [0x3000] = "\xa4\xe0", [0x3001] = "\xa4\xe1", [0x3002] = "\xa4\xe2",
  [0x3003] = "\xa4\xe3", [0x3004] = "\xa4\xe4", [0x3005] = "\xa4\xe5",
  [0x3006] = "\xa4\xe6", [0x3007] = "\xa4\xe7", [0x3008] = "\xa4\xe8",
  [0x3009] = "\xa4\xe9", [0x300a] = "\xa4\xea", [0x300b] = "\xa4\xeb",
  [0x300c] = "\xa4\xec", [0x300d] = "\xa4\xed", [0x300e] = "\xa4\xee",
  [0x300f] = "\xa4\xef", [0x3010] = "\xa4\xf0", [0x3011] = "\xa4\xf1",
  [0x3012] = "\xa4\xf2", [0x3013] = "\xa4\xf3", [0x3014] = "\x4d\xc4",
  [0x3015] = "\x4d\xc5", [0x3016] = "\x4d\xc6", [0x3019] = "\x4d\xc9",
  [0x301a] = "\x4d\xca", [0x301b] = "\xa9\x61", [0x301c] = "\xa9\x62",
  [0x301d] = "\xa9\x66", [0x301e] = "\xa9\x67", [0x301f] = "\x4d\xcb",
  [0x3020] = "\x4d\xcc", [0x3021] = "\xa5\xa1", [0x3022] = "\xa5\xa2",
  [0x3023] = "\xa5\xa3", [0x3024] = "\xa5\xa4", [0x3025] = "\xa5\xa5",
  [0x3026] = "\xa5\xa6", [0x3027] = "\xa5\xa7", [0x3028] = "\xa5\xa8",
  [0x3029] = "\xa5\xa9", [0x302a] = "\xa5\xaa", [0x302b] = "\xa5\xab",
  [0x302c] = "\xa5\xac", [0x302d] = "\xa5\xad", [0x302e] = "\xa5\xae",
  [0x302f] = "\xa5\xaf", [0x3030] = "\xa5\xb0", [0x3031] = "\xa5\xb1",
  [0x3032] = "\xa5\xb2", [0x3033] = "\xa5\xb3", [0x3034] = "\xa5\xb4",
  [0x3035] = "\xa5\xb5", [0x3036] = "\xa5\xb6", [0x3037] = "\xa5\xb7",
  [0x3038] = "\xa5\xb8", [0x3039] = "\xa5\xb9", [0x303a] = "\xa5\xba",
  [0x303b] = "\xa5\xbb", [0x303c] = "\xa5\xbc", [0x303d] = "\xa5\xbd",
  [0x303e] = "\xa5\xbe", [0x303f] = "\xa5\xbf", [0x3040] = "\xa5\xc0",
  [0x3041] = "\xa5\xc1", [0x3042] = "\xa5\xc2", [0x3043] = "\xa5\xc3",
  [0x3044] = "\xa5\xc4", [0x3045] = "\xa5\xc5", [0x3046] = "\xa5\xc6",
  [0x3047] = "\xa5\xc7", [0x3048] = "\xa5\xc8", [0x3049] = "\xa5\xc9",
  [0x304a] = "\xa5\xca", [0x304b] = "\xa5\xcb", [0x304c] = "\xa5\xcc",
  [0x304d] = "\xa5\xcd", [0x304e] = "\xa5\xce", [0x304f] = "\xa5\xcf",
  [0x3050] = "\xa5\xd0", [0x3051] = "\xa5\xd1", [0x3052] = "\xa5\xd2",
  [0x3053] = "\xa5\xd3", [0x3054] = "\xa5\xd4", [0x3055] = "\xa5\xd5",
  [0x3056] = "\xa5\xd6", [0x3057] = "\xa5\xd7", [0x3058] = "\xa5\xd8",
  [0x3059] = "\xa5\xd9", [0x305a] = "\xa5\xda", [0x305b] = "\xa5\xdb",
  [0x305c] = "\xa5\xdc", [0x305d] = "\xa5\xdd", [0x305e] = "\xa5\xde",
  [0x305f] = "\xa5\xdf", [0x3060] = "\xa5\xe0", [0x3061] = "\xa5\xe1",
  [0x3062] = "\xa5\xe2", [0x3063] = "\xa5\xe3", [0x3064] = "\xa5\xe4",
  [0x3065] = "\xa5\xe5", [0x3066] = "\xa5\xe6", [0x3067] = "\xa5\xe7",
  [0x3068] = "\xa5\xe8", [0x3069] = "\xa5\xe9", [0x306a] = "\xa5\xea",
  [0x306b] = "\xa5\xeb", [0x306c] = "\xa5\xec", [0x306d] = "\xa5\xed",
  [0x306e] = "\xa5\xee", [0x306f] = "\xa5\xef", [0x3070] = "\xa5\xf0",
  [0x3071] = "\xa5\xf1", [0x3072] = "\xa5\xf2", [0x3073] = "\xa5\xf3",
  [0x3074] = "\xa5\xf4", [0x3075] = "\xa5\xf5", [0x3076] = "\xa5\xf6",
  [0x3077] = "\x4d\xcd", [0x3078] = "\x4d\xce", [0x3079] = "\x4d\xcf",
  [0x307a] = "\x4d\xd0", [0x307b] = "\x4d\xd1", [0x307c] = "\xa9\x60",
  [0x307d] = "\xa9\x63", [0x307e] = "\xa9\x64", [0x307f] = "\x4d\xd2",
  [0x3085] = "\xa8\xc5", [0x3086] = "\xa8\xc6", [0x3087] = "\xa8\xc7",
  [0x3088] = "\xa8\xc8", [0x3089] = "\xa8\xc9", [0x308a] = "\xa8\xca",
  [0x308b] = "\xa8\xcb", [0x308c] = "\xa8\xcc", [0x308d] = "\xa8\xcd",
  [0x308e] = "\xa8\xce", [0x308f] = "\xa8\xcf", [0x3090] = "\xa8\xd0",
  [0x3091] = "\xa8\xd1", [0x3092] = "\xa8\xd2", [0x3093] = "\xa8\xd3",
  [0x3094] = "\xa8\xd4", [0x3095] = "\xa8\xd5", [0x3096] = "\xa8\xd6",
  [0x3097] = "\xa8\xd7", [0x3098] = "\xa8\xd8", [0x3099] = "\xa8\xd9",
  [0x309a] = "\xa8\xda", [0x309b] = "\xa8\xdb", [0x309c] = "\xa8\xdc",
  [0x309d] = "\xa8\xdd", [0x309e] = "\xa8\xde", [0x309f] = "\xa8\xdf",
  [0x30a0] = "\xa8\xe0", [0x30a1] = "\xa8\xe1", [0x30a2] = "\xa8\xe2",
  [0x30a3] = "\xa8\xe3", [0x30a4] = "\xa8\xe4", [0x30a5] = "\xa8\xe5",
  [0x30a6] = "\xa8\xe6", [0x30a7] = "\xa8\xe7", [0x30a8] = "\xa8\xe8",
  [0x30a9] = "\xa8\xe9", [0x30aa] = "\x4d\xd8", [0x30ab] = "\x4d\xd9",
  [0x30ac] = "\x4d\xda", [0x30ad] = "\x4d\xdb", [0x30b1] = "\x4d\xdf",
  [0x30b2] = "\x4d\xe0", [0x30b3] = "\x4d\xe1", [0x30b4] = "\x4d\xe2",
  [0x30b5] = "\x4d\xe3", [0x30b6] = "\x4d\xe4", [0x30b7] = "\x4d\xe5",
  [0x30b8] = "\x4d\xe6", [0x30b9] = "\x4d\xe7", [0x30ba] = "\x4d\xe8",
  [0x30bb] = "\x4d\xe9", [0x30bc] = "\x4d\xea", [0x30bd] = "\x4d\xeb",
  [0x30be] = "\x4d\xec", [0x30bf] = "\x4d\xed", [0x30c0] = "\x4d\xee",
  [0x30c1] = "\x4d\xef", [0x30c2] = "\x4d\xf0", [0x30c3] = "\x4d\xf1",
  [0x30c4] = "\x4d\xf2", [0x30c5] = "\x4d\xf3", [0x30c6] = "\x4d\xf4",
  [0x30c7] = "\x4d\xf5", [0x30c8] = "\x4d\xf6", [0x30c9] = "\x4d\xf7",
  [0x30ca] = "\x4d\xf8", [0x30cb] = "\x4d\xf9", [0x30cc] = "\x4d\xfa",
  [0x30cd] = "\x4d\xfb", [0x30ce] = "\x4d\xfc", [0x30cf] = "\x4d\xfd",
  [0x30d0] = "\x4d\xfe", [0x30d1] = "\x4d\xff", [0x30d2] = "\x4e\x00",
  [0x30d3] = "\x4e\x01", [0x30d4] = "\x4e\x02", [0x30d5] = "\x4e\x03",
  [0x30d6] = "\x4e\x04", [0x30d7] = "\x4e\x05", [0x30d8] = "\x4e\x06",
  [0x30d9] = "\x4e\x07", [0x30da] = "\x4e\x08", [0x30db] = "\x4e\x09",
  [0x30dc] = "\x4e\x0a", [0x30dd] = "\x4e\x0b", [0x30de] = "\x4e\x0c",
  [0x30df] = "\x4e\x0d", [0x30e0] = "\x4e\x0e", [0x30e1] = "\x4e\x0f",
  [0x30e2] = "\x4e\x10", [0x30e3] = "\x4e\x11", [0x30e4] = "\x4e\x12",
  [0x30e5] = "\x4e\x13", [0x30e6] = "\x4e\x14", [0x30e7] = "\x4e\x15",
  [0x30e8] = "\x4e\x16", [0x30e9] = "\x4e\x17", [0x30ea] = "\x4e\x18",
  [0x30eb] = "\x4e\x19", [0x30ec] = "\x4e\x1a", [0x30ed] = "\x4e\x1b",
  [0x30ee] = "\x4e\x1c", [0x30ef] = "\x4e\x1d", [0x30f0] = "\x4e\x1e",
  [0x30f1] = "\x4e\x1f", [0x30f2] = "\x4e\x20", [0x30f3] = "\x4e\x21",
  [0x30f4] = "\x4e\x22", [0x30f5] = "\x4e\x23", [0x30f6] = "\x4e\x24",
  [0x30f7] = "\x4e\x25", [0x30f8] = "\x4e\x26", [0x30f9] = "\x4e\x27",
  [0x30fa] = "\x4e\x28", [0x30fb] = "\x4e\x29", [0x30fc] = "\x4e\x2a",
  [0x30fd] = "\x4e\x2b", [0x30fe] = "\x4e\x2c", [0x30ff] = "\x4e\x2d",
  [0x3100] = "\x4e\x2e", [0x3101] = "\x4e\x2f", [0x3102] = "\x4e\x30",
  [0x3103] = "\x4e\x31", [0x3104] = "\x4e\x32", [0x3105] = "\x4e\x33",
  [0x3106] = "\x4e\x34", [0x3107] = "\x4e\x35", [0x3108] = "\x4e\x36",
  [0x3109] = "\x4e\x37", [0x310a] = "\x4e\x38", [0x310b] = "\x4e\x39",
  [0x310c] = "\x4e\x3a", [0x310d] = "\x4e\x3b", [0x310e] = "\x4e\x3c",
  [0x3110] = "\x4e\x3e", [0x3111] = "\x4e\x3f", [0x3112] = "\x4e\x40",
  [0x3113] = "\x4e\x41", [0x3114] = "\x4e\x42", [0x3115] = "\x4e\x43",
  [0x3116] = "\x4e\x44", [0x3117] = "\x4e\x45", [0x3118] = "\x4e\x46",
  [0x3119] = "\x4e\x47", [0x311a] = "\x4e\x48", [0x311b] = "\x4e\x49",
  [0x311c] = "\x4e\x4a", [0x311d] = "\x4e\x4b", [0x311e] = "\x4e\x4c",
  [0x311f] = "\x4e\x4d", [0x3120] = "\x4e\x4e", [0x3121] = "\x4e\x4f",
  [0x3122] = "\x4e\x50", [0x3123] = "\x4e\x51", [0x3124] = "\x4e\x52",
  [0x3125] = "\x4e\x53", [0x3126] = "\x4e\x54", [0x3127] = "\x4e\x55",
  [0x3128] = "\x4e\x56", [0x3129] = "\x4e\x57", [0x312a] = "\x4e\x58",
  [0x312b] = "\x4e\x59", [0x312c] = "\x4e\x5a", [0x312d] = "\x4e\x5b",
  [0x312e] = "\x4e\x5c", [0x312f] = "\x4e\x5d", [0x3130] = "\x4e\x5e",
  [0x3131] = "\x4e\x5f", [0x3132] = "\x4e\x60", [0x3133] = "\x4e\x61",
  [0x3134] = "\x4e\x62", [0x3135] = "\x4e\x63", [0x3136] = "\x4e\x64",
  [0x3137] = "\x4e\x65", [0x3138] = "\x4e\x66", [0x3139] = "\x4e\x67",
  [0x313a] = "\x4e\x68", [0x3140] = "\x4e\x6e", [0x3141] = "\x4e\x6f",
  [0x3142] = "\x4e\x70", [0x3143] = "\x4e\x71", [0x3144] = "\x4e\x72",
  [0x3145] = "\x4e\x73", [0x3146] = "\x4e\x74", [0x3147] = "\x4e\x75",
  [0x3148] = "\x4e\x76", [0x3149] = "\x4e\x77", [0x314a] = "\x4e\x78",
  [0x314b] = "\x4e\x79", [0x314c] = "\x4e\x7a", [0x314d] = "\x4e\x7b",
  [0x314e] = "\x4e\x7c", [0x314f] = "\x4e\x7d", [0x3150] = "\x4e\x7e",
  [0x3151] = "\x4e\x7f", [0x3152] = "\x4e\x80", [0x3153] = "\x4e\x81",
  [0x3154] = "\x4e\x82", [0x3155] = "\x4e\x83", [0x3156] = "\x4e\x84",
  [0x3157] = "\x4e\x85", [0x3158] = "\x4e\x86", [0x3159] = "\x4e\x87",
  [0x315a] = "\x4e\x88", [0x315b] = "\x4e\x89", [0x315c] = "\x4e\x8a",
  [0x315d] = "\x4e\x8b", [0x315e] = "\x4e\x8c", [0x315f] = "\x4e\x8d",
  [0x3160] = "\x4e\x8e", [0x3161] = "\x4e\x8f", [0x3162] = "\x4e\x90",
  [0x3163] = "\x4e\x91", [0x3170] = "\x4e\x9e", [0x3171] = "\x4e\x9f",
  [0x3172] = "\x4e\xa0", [0x3173] = "\x4e\xa1", [0x3174] = "\x4e\xa2",
  [0x3175] = "\x4e\xa3", [0x3176] = "\x4e\xa4", [0x3177] = "\x4e\xa5",
  [0x3178] = "\x4e\xa6", [0x3179] = "\x4e\xa7", [0x317a] = "\x4e\xa8",
  [0x317b] = "\x4e\xa9", [0x317c] = "\x4e\xaa", [0x317d] = "\x4e\xab",
  [0x317e] = "\x4e\xac", [0x317f] = "\x4e\xad", [0x3180] = "\x4e\xae",
  [0x3181] = "\x4e\xaf", [0x3182] = "\x4e\xb0", [0x3183] = "\x4e\xb1",
  [0x3184] = "\x4e\xb2", [0x3185] = "\x4e\xb3", [0x3186] = "\x4e\xb4",
  [0x3187] = "\x4e\xb5", [0x3188] = "\x4e\xb6", [0x3189] = "\x4e\xb7",
  [0x318a] = "\x4e\xb8", [0x318b] = "\x4e\xb9", [0x318c] = "\x4e\xba",
  [0x318d] = "\x4e\xbb", [0x318e] = "\x4e\xbc", [0x318f] = "\x4e\xbd",
  [0x3190] = "\x4e\xbe", [0x3191] = "\x4e\xbf", [0x3192] = "\x4e\xc0",
  [0x3193] = "\x4e\xc1", [0x3194] = "\x4e\xc2", [0x3195] = "\x4e\xc3",
  [0x3196] = "\x4e\xc4", [0x3197] = "\x4e\xc5", [0x3198] = "\x4e\xc6",
  [0x3199] = "\x4e\xc7", [0x319a] = "\x4e\xc8", [0x319b] = "\x4e\xc9",
  [0x319c] = "\x4e\xca", [0x319d] = "\x4e\xcb", [0x319e] = "\x4e\xcc",
  [0x31a0] = "\xa2\xe5", [0x31a1] = "\xa2\xe6", [0x31a2] = "\xa2\xe7",
  [0x31a3] = "\xa2\xe8", [0x31a4] = "\xa2\xe9", [0x31a5] = "\xa2\xea",
  [0x31a6] = "\xa2\xeb", [0x31a7] = "\xa2\xec", [0x31a8] = "\xa2\xed",
  [0x31a9] = "\xa2\xee", [0x31aa] = "\x4e\xce", [0x31ab] = "\x4e\xcf",
  [0x31ac] = "\x4e\xd0", [0x31ad] = "\x4e\xd1", [0x31ae] = "\x4e\xd2",
  [0x31af] = "\x4e\xd3", [0x31b0] = "\x4e\xd4", [0x31b1] = "\xa9\x5a",
  [0x31b2] = "\x4e\xd5", [0x31b3] = "\x4e\xd6", [0x31b4] = "\x4e\xd7",
  [0x31b5] = "\x4e\xd8", [0x31b6] = "\x4e\xd9", [0x31b7] = "\x4e\xda",
  [0x31b8] = "\x4e\xdb", [0x31b9] = "\x4e\xdc", [0x31ba] = "\x4e\xdd",
  [0x31bb] = "\x4e\xde", [0x31bc] = "\x4e\xdf", [0x31bd] = "\x4e\xe0",
  [0x31be] = "\x4e\xe1", [0x31bf] = "\x4e\xe2", [0x31c0] = "\x4e\xe3",
  [0x31c1] = "\x4e\xe4", [0x31c2] = "\x4e\xe5", [0x31c3] = "\x4e\xe6",
  [0x31c4] = "\x4e\xe7", [0x31c5] = "\x4e\xe8", [0x31c6] = "\x4e\xe9",
  [0x31c7] = "\x4e\xea", [0x31c8] = "\x4e\xeb", [0x31c9] = "\x4e\xec",
  [0x31ca] = "\x4e\xed", [0x31cb] = "\x4e\xee", [0x31cc] = "\x4e\xef",
  [0x31cd] = "\x4e\xf0", [0x31ce] = "\x4e\xf1", [0x31cf] = "\x4e\xf2",
  [0x31d0] = "\x4e\xf3", [0x31d1] = "\x4e\xf4", [0x31d2] = "\x4e\xf5",
  [0x31d3] = "\x4e\xf6", [0x31d4] = "\x4e\xf7", [0x31d5] = "\x4e\xf8",
  [0x31d6] = "\x4e\xf9", [0x31d7] = "\x4e\xfa", [0x31d8] = "\x4e\xfb",
  [0x31d9] = "\x4e\xfc", [0x31da] = "\x4e\xfd", [0x31db] = "\x4e\xfe",
  [0x31dc] = "\x4e\xff", [0x31dd] = "\x4f\x00", [0x31de] = "\x4f\x01",
  [0x31df] = "\x4f\x02", [0x31e0] = "\x4f\x03", [0x31e1] = "\x4f\x04",
  [0x31e2] = "\x4f\x05", [0x31e3] = "\x4f\x06", [0x31e4] = "\x4f\x07",
  [0x31e5] = "\x4f\x08", [0x31e6] = "\x4f\x09", [0x31e7] = "\x4f\x0a",
  [0x31e8] = "\x4f\x0b", [0x31e9] = "\x4f\x0c", [0x31ea] = "\x4f\x0d",
  [0x31eb] = "\x4f\x0e", [0x31ec] = "\x4f\x0f", [0x31ed] = "\x4f\x10",
  [0x31ee] = "\x4f\x11", [0x31ef] = "\x4f\x12", [0x31f0] = "\x4f\x13",
  [0x31f1] = "\x4f\x14", [0x31f2] = "\x4f\x15", [0x31f3] = "\x4f\x16",
  [0x31f4] = "\x4f\x17", [0x31f5] = "\x4f\x18", [0x31f6] = "\x4f\x19",
  [0x31f7] = "\x4f\x1a", [0x31f8] = "\x4f\x1b", [0x31f9] = "\x4f\x1c",
  [0x31fa] = "\x4f\x1d", [0x31fb] = "\x4f\x1e", [0x31fc] = "\x4f\x1f",
  [0x31fd] = "\x4f\x20", [0x31fe] = "\x4f\x21", [0x31ff] = "\x4f\x22",
  [0x3200] = "\x4f\x23", [0x3201] = "\x4f\x24", [0x3202] = "\x4f\x25",
  [0x3203] = "\x4f\x26", [0x3204] = "\x4f\x27", [0x3205] = "\x4f\x28",
  [0x3206] = "\x4f\x29", [0x3207] = "\x4f\x2a", [0x3208] = "\x4f\x2b",
  [0x3209] = "\x4f\x2c", [0x320a] = "\x4f\x2d", [0x320b] = "\x4f\x2e",
  [0x320c] = "\x4f\x2f", [0x320d] = "\x4f\x30", [0x320e] = "\x4f\x31",
  [0x320f] = "\x4f\x32", [0x3210] = "\x4f\x33", [0x3211] = "\x4f\x34",
  [0x3212] = "\x4f\x35", [0x3213] = "\x4f\x36", [0x3214] = "\x4f\x37",
  [0x3215] = "\x4f\x38", [0x3216] = "\x4f\x39", [0x3217] = "\x4f\x3a",
  [0x3218] = "\x4f\x3b", [0x3219] = "\x4f\x3c", [0x321a] = "\x4f\x3d",
  [0x321b] = "\x4f\x3e", [0x321c] = "\x4f\x3f", [0x321d] = "\x4f\x40",
  [0x321e] = "\x4f\x41", [0x321f] = "\x4f\x42", [0x3220] = "\x4f\x43",
  [0x3221] = "\x4f\x44", [0x3222] = "\x4f\x45", [0x3223] = "\xa9\x49",
  [0x3224] = "\x4f\x46", [0x3225] = "\x4f\x47", [0x3226] = "\x4f\x48",
  [0x3227] = "\x4f\x49", [0x3228] = "\x4f\x4a", [0x3229] = "\x4f\x4b",
  [0x322a] = "\x4f\x4c", [0x322b] = "\x4f\x4d", [0x322c] = "\x4f\x4e",
  [0x322d] = "\x4f\x4f", [0x322e] = "\x4f\x50", [0x322f] = "\x4f\x51",
  [0x3230] = "\x4f\x52", [0x3231] = "\x4f\x53", [0x3232] = "\x4f\x54",
  [0x3233] = "\x4f\x55", [0x3234] = "\x4f\x56", [0x3235] = "\x4f\x57",
  [0x3236] = "\x4f\x58", [0x3237] = "\x4f\x59", [0x3238] = "\x4f\x5a",
  [0x3239] = "\x4f\x5b", [0x323a] = "\x4f\x5c", [0x323b] = "\x4f\x5d",
  [0x323c] = "\x4f\x5e", [0x323d] = "\x4f\x5f", [0x323e] = "\x4f\x60",
  [0x323f] = "\x4f\x61", [0x3240] = "\x4f\x62", [0x3241] = "\x4f\x63",
  [0x3242] = "\x4f\x64", [0x3243] = "\x4f\x65", [0x3244] = "\x4f\x66",
  [0x3245] = "\x4f\x67", [0x3246] = "\x4f\x68", [0x3247] = "\x4f\x69",
  [0x3248] = "\x4f\x6a", [0x3249] = "\x4f\x6b", [0x324a] = "\x4f\x6c",
  [0x324b] = "\x4f\x6d", [0x324c] = "\x4f\x6e", [0x324d] = "\x4f\x6f",
  [0x324e] = "\x4f\x70", [0x324f] = "\x4f\x71", [0x3250] = "\x4f\x72",
  [0x3251] = "\x4f\x73", [0x3252] = "\x4f\x74", [0x3253] = "\x4f\x75",
  [0x3254] = "\x4f\x76", [0x3255] = "\x4f\x77", [0x3256] = "\x4f\x78",
  [0x3257] = "\x4f\x79", [0x3258] = "\x4f\x7a", [0x3259] = "\x4f\x7b",
  [0x325a] = "\x4f\x7c", [0x325b] = "\x4f\x7d", [0x325c] = "\x4f\x7e",
  [0x325d] = "\x4f\x7f", [0x325e] = "\x4f\x80", [0x325f] = "\x4f\x81",
  [0x3260] = "\x4f\x82", [0x3261] = "\x4f\x83", [0x3262] = "\x4f\x84",
  [0x3263] = "\x4f\x85", [0x3264] = "\x4f\x86", [0x3265] = "\x4f\x87",
  [0x3266] = "\x4f\x88", [0x3267] = "\x4f\x89", [0x3268] = "\x4f\x8a",
  [0x3269] = "\x4f\x8b", [0x326a] = "\x4f\x8c", [0x326b] = "\x4f\x8d",
  [0x326c] = "\x4f\x8e", [0x326d] = "\x4f\x8f", [0x326e] = "\x4f\x90",
  [0x326f] = "\x4f\x91", [0x3270] = "\x4f\x92", [0x3271] = "\x4f\x93",
  [0x3272] = "\x4f\x94", [0x3273] = "\x4f\x95", [0x3274] = "\x4f\x96",
  [0x3275] = "\x4f\x97", [0x3276] = "\x4f\x98", [0x3277] = "\x4f\x99",
  [0x3278] = "\x4f\x9a", [0x3279] = "\x4f\x9b", [0x327a] = "\x4f\x9c",
  [0x327b] = "\x4f\x9d", [0x327c] = "\x4f\x9e", [0x327d] = "\x4f\x9f",
  [0x327e] = "\x4f\xa0", [0x3280] = "\x4f\xa2", [0x3281] = "\x4f\xa3",
  [0x3282] = "\x4f\xa4", [0x3283] = "\x4f\xa5", [0x3284] = "\x4f\xa6",
  [0x3285] = "\x4f\xa7", [0x3286] = "\x4f\xa8", [0x3287] = "\x4f\xa9",
  [0x3288] = "\x4f\xaa", [0x3289] = "\x4f\xab", [0x328a] = "\x4f\xac",
  [0x328b] = "\x4f\xad", [0x328c] = "\x4f\xae", [0x328d] = "\x4f\xaf",
  [0x328e] = "\x4f\xb0", [0x328f] = "\x4f\xb1", [0x3290] = "\x4f\xb2",
  [0x3291] = "\x4f\xb3", [0x3292] = "\x4f\xb4", [0x3293] = "\x4f\xb5",
  [0x3294] = "\x4f\xb6", [0x3295] = "\x4f\xb7", [0x3296] = "\x4f\xb8",
  [0x3297] = "\x4f\xb9", [0x3298] = "\x4f\xba", [0x3299] = "\x4f\xbb",
  [0x329a] = "\x4f\xbc", [0x329b] = "\x4f\xbd", [0x329c] = "\x4f\xbe",
  [0x329d] = "\x4f\xbf", [0x329e] = "\x4f\xc0", [0x329f] = "\x4f\xc1",
  [0x32a0] = "\x4f\xc2", [0x32a1] = "\x4f\xc3", [0x32a2] = "\x4f\xc4",
  [0x32a3] = "\x4f\xc5", [0x32a4] = "\x4f\xc6", [0x32a5] = "\x4f\xc7",
  [0x32a6] = "\x4f\xc8", [0x32a7] = "\x4f\xc9", [0x32a8] = "\x4f\xca",
  [0x32a9] = "\x4f\xcb", [0x32aa] = "\x4f\xcc", [0x32ab] = "\x4f\xcd",
  [0x32ac] = "\x4f\xce", [0x32ad] = "\x4f\xcf", [0x32ae] = "\x4f\xd0",
  [0x32af] = "\x4f\xd1", [0x32b0] = "\x4f\xd2", [0x32b1] = "\x4f\xd3",
  [0x32b2] = "\x4f\xd4", [0x32b3] = "\x4f\xd5", [0x32b4] = "\x4f\xd6",
  [0x32b5] = "\x4f\xd7", [0x32b6] = "\x4f\xd8", [0x32b7] = "\x4f\xd9",
  [0x32b8] = "\x4f\xda", [0x32b9] = "\x4f\xdb", [0x32ba] = "\x4f\xdc",
  [0x32bb] = "\x4f\xdd", [0x32bc] = "\x4f\xde", [0x32bd] = "\x4f\xdf",
  [0x32be] = "\x4f\xe0", [0x32bf] = "\x4f\xe1", [0x32c0] = "\x4f\xe2",
  [0x32c1] = "\x4f\xe3", [0x32c2] = "\x4f\xe4", [0x32c3] = "\x4f\xe5",
  [0x32c4] = "\x4f\xe6", [0x32c5] = "\x4f\xe7", [0x32c6] = "\x4f\xe8",
  [0x32c7] = "\x4f\xe9", [0x32c8] = "\x4f\xea", [0x32c9] = "\x4f\xeb",
  [0x32ca] = "\x4f\xec", [0x32cb] = "\x4f\xed", [0x32cc] = "\x4f\xee",
  [0x32cd] = "\x4f\xef", [0x32ce] = "\x4f\xf0", [0x32cf] = "\x4f\xf1",
  [0x32d0] = "\x4f\xf2", [0x32d1] = "\x4f\xf3", [0x32d2] = "\x4f\xf4",
  [0x32d3] = "\x4f\xf5", [0x32d4] = "\x4f\xf6", [0x32d5] = "\x4f\xf7",
  [0x32d6] = "\x4f\xf8", [0x32d7] = "\x4f\xf9", [0x32d8] = "\x4f\xfa",
  [0x32d9] = "\x4f\xfb", [0x32da] = "\x4f\xfc", [0x32db] = "\x4f\xfd",
  [0x32dc] = "\x4f\xfe", [0x32dd] = "\x4f\xff", [0x32de] = "\x50\x00",
  [0x32df] = "\x50\x01", [0x32e0] = "\x50\x02", [0x32e1] = "\x50\x03",
  [0x32e2] = "\x50\x04", [0x32e3] = "\x50\x05", [0x32e4] = "\x50\x06",
  [0x32e5] = "\x50\x07", [0x32e6] = "\x50\x08", [0x32e7] = "\x50\x09",
  [0x32e8] = "\x50\x0a", [0x32e9] = "\x50\x0b", [0x32ea] = "\x50\x0c",
  [0x32eb] = "\x50\x0d", [0x32ec] = "\x50\x0e", [0x32ed] = "\x50\x0f",
  [0x32ee] = "\x50\x10", [0x32ef] = "\x50\x11", [0x32f0] = "\x50\x12",
  [0x32f1] = "\x50\x13", [0x32f2] = "\x50\x14", [0x32f3] = "\x50\x15",
  [0x32f4] = "\x50\x16", [0x32f5] = "\x50\x17", [0x32f6] = "\x50\x18",
  [0x32f7] = "\x50\x19", [0x32f8] = "\x50\x1a", [0x32f9] = "\x50\x1b",
  [0x32fa] = "\x50\x1c", [0x32fb] = "\x50\x1d", [0x32fc] = "\x50\x1e",
  [0x32fd] = "\x50\x1f", [0x32fe] = "\x50\x20", [0x32ff] = "\x50\x21",
  [0x3300] = "\x50\x22", [0x3301] = "\x50\x23", [0x3302] = "\x50\x24",
  [0x3303] = "\x50\x25", [0x3304] = "\x50\x26", [0x3305] = "\x50\x27",
  [0x3306] = "\x50\x28", [0x3307] = "\x50\x29", [0x3308] = "\x50\x2a",
  [0x3309] = "\x50\x2b", [0x330a] = "\x50\x2c", [0x330b] = "\x50\x2d",
  [0x330c] = "\x50\x2e", [0x330d] = "\x50\x2f", [0x330e] = "\xa9\x4a",
  [0x330f] = "\xa9\x4b", [0x3310] = "\x50\x30", [0x3311] = "\x50\x31",
  [0x3312] = "\x50\x32", [0x3313] = "\x50\x33", [0x3314] = "\x50\x34",
  [0x3315] = "\x50\x35", [0x3316] = "\x50\x36", [0x3317] = "\x50\x37",
  [0x3318] = "\x50\x38", [0x3319] = "\x50\x39", [0x331a] = "\x50\x3a",
  [0x331b] = "\x50\x3b", [0x331c] = "\xa9\x4c", [0x331d] = "\xa9\x4d",
  [0x331e] = "\xa9\x4e", [0x331f] = "\x50\x3c", [0x3320] = "\x50\x3d",
  [0x3321] = "\xa9\x4f", [0x3322] = "\x50\x3e", [0x3323] = "\x50\x3f",
  [0x3324] = "\x50\x40", [0x3325] = "\x50\x41", [0x3326] = "\x50\x42",
  [0x3327] = "\x50\x43", [0x3328] = "\x50\x44", [0x3329] = "\x50\x45",
  [0x332a] = "\x50\x46", [0x332b] = "\x50\x47", [0x332c] = "\x50\x48",
  [0x332d] = "\x50\x49", [0x332e] = "\x50\x4a", [0x332f] = "\x50\x4b",
  [0x3330] = "\x50\x4c", [0x3331] = "\x50\x4d", [0x3332] = "\x50\x4e",
  [0x3333] = "\x50\x4f", [0x3334] = "\x50\x50", [0x3335] = "\x50\x51",
  [0x3336] = "\x50\x52", [0x3337] = "\x50\x53", [0x3338] = "\x50\x54",
  [0x3339] = "\x50\x55", [0x333a] = "\x50\x56", [0x333b] = "\x50\x57",
  [0x333c] = "\x50\x58", [0x333d] = "\x50\x59", [0x333e] = "\x50\x5a",
  [0x333f] = "\x50\x5b", [0x3340] = "\x50\x5c", [0x3341] = "\x50\x5d",
  [0x3342] = "\x50\x5e", [0x3343] = "\x50\x5f", [0x3344] = "\xa9\x50",
  [0x3345] = "\x50\x60", [0x3346] = "\x50\x61", [0x3347] = "\x50\x62",
  [0x3348] = "\x50\x63", [0x3349] = "\x50\x64", [0x334a] = "\x50\x65",
  [0x334b] = "\x50\x66", [0x334c] = "\x50\x67", [0x334d] = "\x50\x68",
  [0x334e] = "\xa9\x51", [0x334f] = "\x50\x69", [0x3350] = "\x50\x6a",
  [0x3351] = "\xa9\x52", [0x3352] = "\xa9\x53", [0x3353] = "\x50\x6b",
  [0x3354] = "\x50\x6c", [0x3355] = "\xa9\x54", [0x3356] = "\x50\x6d",
  [0x3357] = "\x50\x6e", [0x3358] = "\x50\x6f", [0x3359] = "\x50\x70",
  [0x335a] = "\x50\x71", [0x335b] = "\x50\x72", [0x335c] = "\x50\x73",
  [0x335d] = "\x50\x74", [0x335e] = "\x50\x75", [0x335f] = "\x50\x76",
  [0x3360] = "\x50\x77", [0x3361] = "\x50\x78", [0x3362] = "\x50\x79",
  [0x3363] = "\x50\x7a", [0x3364] = "\x50\x7b", [0x3365] = "\x50\x7c",
  [0x3366] = "\x50\x7d", [0x3367] = "\x50\x7e", [0x3368] = "\x50\x7f",
  [0x3369] = "\x50\x80", [0x336a] = "\x50\x81", [0x336b] = "\x50\x82",
  [0x336c] = "\x50\x83", [0x336d] = "\x50\x84", [0x336e] = "\x50\x85",
  [0x336f] = "\x50\x86", [0x3370] = "\x50\x87", [0x3371] = "\x50\x88",
  [0x3372] = "\x50\x89", [0x3373] = "\x50\x8a", [0x3374] = "\x50\x8b",
  [0x3375] = "\x50\x8c", [0x3376] = "\x50\x8d", [0x3377] = "\x50\x8e",
  [0x3378] = "\x50\x8f", [0x3379] = "\x50\x90", [0x337a] = "\x50\x91",
  [0x337b] = "\x50\x92", [0x337c] = "\x50\x93", [0x337d] = "\x50\x94",
  [0x337e] = "\x50\x95", [0x337f] = "\x50\x96", [0x3380] = "\x50\x97",
  [0x3381] = "\x50\x98", [0x3382] = "\x50\x99", [0x3383] = "\x50\x9a",
  [0x3384] = "\x50\x9b", [0x3385] = "\x50\x9c", [0x3386] = "\x50\x9d",
  [0x3387] = "\x50\x9e", [0x3388] = "\x50\x9f", [0x3389] = "\x50\xa0",
  [0x338a] = "\x50\xa1", [0x338b] = "\x50\xa2", [0x338c] = "\x50\xa3",
  [0x338d] = "\x50\xa4", [0x338e] = "\x50\xa5", [0x338f] = "\x50\xa6",
  [0x3390] = "\x50\xa7", [0x3391] = "\x50\xa8", [0x3392] = "\x50\xa9",
  [0x3393] = "\x50\xaa", [0x3394] = "\x50\xab", [0x3395] = "\x50\xac",
  [0x3396] = "\x50\xad", [0x3397] = "\x50\xae", [0x3398] = "\x50\xaf",
  [0x3399] = "\x50\xb0", [0x339a] = "\x50\xb1", [0x339b] = "\x50\xb2",
  [0x339c] = "\x50\xb3", [0x339d] = "\x50\xb4", [0x339e] = "\x50\xb5",
  [0x339f] = "\x50\xb6", [0x33a0] = "\x50\xb7", [0x33a1] = "\x50\xb8",
  [0x33a2] = "\x50\xb9", [0x33a3] = "\x50\xba", [0x33a4] = "\x50\xbb",
  [0x33a5] = "\x50\xbc", [0x33a6] = "\x50\xbd", [0x33a7] = "\x50\xbe",
  [0x33a8] = "\x50\xbf", [0x33a9] = "\x50\xc0", [0x33aa] = "\x50\xc1",
  [0x33ab] = "\x50\xc2", [0x33ac] = "\x50\xc3", [0x33ad] = "\x50\xc4",
  [0x33ae] = "\x50\xc5", [0x33af] = "\x50\xc6", [0x33b0] = "\x50\xc7",
  [0x33b1] = "\x50\xc8", [0x33b2] = "\x50\xc9", [0x33b3] = "\x50\xca",
  [0x33b4] = "\x50\xcb", [0x33b5] = "\x50\xcc", [0x33b6] = "\x50\xcd",
  [0x33b7] = "\x50\xce", [0x33b8] = "\x50\xcf", [0x33b9] = "\x50\xd0",
  [0x33ba] = "\x50\xd1", [0x33bb] = "\x50\xd2", [0x33bc] = "\x50\xd3",
  [0x33bd] = "\x50\xd4", [0x33be] = "\x50\xd5", [0x33bf] = "\x50\xd6",
  [0x33c0] = "\x50\xd7", [0x33c1] = "\x50\xd8", [0x33c2] = "\x50\xd9",
  [0x33c3] = "\x50\xda", [0x33c4] = "\x50\xdb", [0x33c5] = "\x50\xdc",
  [0x33c6] = "\x50\xdd", [0x33c7] = "\xfe\x56", [0x33c8] = "\x50\xde",
  [0x33c9] = "\x50\xdf", [0x33ca] = "\x50\xe0", [0x33cb] = "\x50\xe1",
  [0x33cc] = "\x50\xe2", [0x33cd] = "\x50\xe3", [0x33ce] = "\x50\xe4",
  [0x33cf] = "\x50\xe5", [0x33d0] = "\x50\xe6", [0x33d1] = "\x50\xe7",
  [0x33d2] = "\x50\xe8", [0x33d3] = "\x50\xe9", [0x33d4] = "\x50\xea",
  [0x33d5] = "\x50\xeb", [0x33d6] = "\x50\xec", [0x33d7] = "\x50\xed",
  [0x33d8] = "\x50\xee", [0x33d9] = "\x50\xef", [0x33da] = "\x50\xf0",
  [0x33db] = "\x50\xf1", [0x33dc] = "\x50\xf2", [0x33dd] = "\x50\xf3",
  [0x33de] = "\x50\xf4", [0x33df] = "\x50\xf5", [0x33e0] = "\x50\xf6",
  [0x33e1] = "\x50\xf7", [0x33e2] = "\x50\xf8", [0x33e3] = "\x50\xf9",
  [0x33e4] = "\x50\xfa", [0x33e5] = "\x50\xfb", [0x33e6] = "\x50\xfc",
  [0x33e7] = "\x50\xfd", [0x33e8] = "\x50\xfe", [0x33e9] = "\x50\xff",
  [0x33ea] = "\x51\x00", [0x33eb] = "\x51\x01", [0x33ec] = "\x51\x02",
  [0x33ed] = "\x51\x03", [0x33ee] = "\x51\x04", [0x33ef] = "\x51\x05",
  [0x33f0] = "\x51\x06", [0x33f1] = "\x51\x07", [0x33f2] = "\x51\x08",
  [0x33f3] = "\xfe\x55", [0x33f4] = "\x51\x09", [0x33f5] = "\x51\x0a",
  [0x33f6] = "\x51\x0b", [0x33f7] = "\x51\x0c", [0x33f8] = "\x51\x0d",
  [0x33f9] = "\x51\x0e", [0x33fa] = "\x51\x0f", [0x33fb] = "\x51\x10",
  [0x33fc] = "\x51\x11", [0x33fd] = "\x51\x12", [0x33fe] = "\x51\x13",
  [0x33ff] = "\x51\x14", [0x3400] = "\x51\x15", [0x3401] = "\x51\x16",
  [0x3402] = "\x51\x17", [0x3403] = "\x51\x18", [0x3404] = "\x51\x19",
  [0x3405] = "\x51\x1a", [0x3406] = "\x51\x1b", [0x3407] = "\x51\x1c",
  [0x3408] = "\x51\x1d", [0x3409] = "\x51\x1e", [0x340a] = "\x51\x1f",
  [0x340b] = "\x51\x20", [0x340c] = "\x51\x21", [0x340d] = "\x51\x22",
  [0x340e] = "\x51\x23", [0x340f] = "\x51\x24", [0x3410] = "\x51\x25",
  [0x3411] = "\x51\x26", [0x3412] = "\x51\x27", [0x3413] = "\x51\x28",
  [0x3414] = "\x51\x29", [0x3415] = "\x51\x2a", [0x3416] = "\x51\x2b",
  [0x3417] = "\x51\x2c", [0x3418] = "\x51\x2d", [0x3419] = "\x51\x2e",
  [0x341a] = "\x51\x2f", [0x341b] = "\x51\x30", [0x341c] = "\x51\x31",
  [0x341d] = "\x51\x32", [0x341e] = "\x51\x33", [0x341f] = "\x51\x34",
  [0x3420] = "\x51\x35", [0x3421] = "\x51\x36", [0x3422] = "\x51\x37",
  [0x3423] = "\x51\x38", [0x3424] = "\x51\x39", [0x3425] = "\x51\x3a",
  [0x3426] = "\x51\x3b", [0x3427] = "\x51\x3c", [0x3428] = "\x51\x3d",
  [0x3429] = "\x51\x3e", [0x342a] = "\x51\x3f", [0x342b] = "\x51\x40",
  [0x342c] = "\x51\x41", [0x342d] = "\x51\x42", [0x342e] = "\x51\x43",
  [0x342f] = "\x51\x44", [0x3430] = "\x51\x45", [0x3431] = "\x51\x46",
  [0x3432] = "\x51\x47", [0x3433] = "\x51\x48", [0x3434] = "\x51\x49",
  [0x3435] = "\x51\x4a", [0x3436] = "\x51\x4b", [0x3437] = "\x51\x4c",
  [0x3438] = "\x51\x4d", [0x3439] = "\x51\x4e", [0x343a] = "\x51\x4f",
  [0x343b] = "\x51\x50", [0x343c] = "\x51\x51", [0x343d] = "\x51\x52",
  [0x343e] = "\x51\x53", [0x343f] = "\x51\x54", [0x3440] = "\x51\x55",
  [0x3441] = "\x51\x56", [0x3442] = "\x51\x57", [0x3443] = "\x51\x58",
  [0x3444] = "\x51\x59", [0x3445] = "\x51\x5a", [0x3446] = "\x51\x5b",
  [0x3447] = "\x51\x5c", [0x3448] = "\x51\x5d", [0x3449] = "\x51\x5e",
  [0x344a] = "\x51\x5f", [0x344b] = "\x51\x60", [0x344c] = "\x51\x61",
  [0x344d] = "\x51\x62", [0x344e] = "\x51\x63", [0x344f] = "\x51\x64",
  [0x3450] = "\x51\x65", [0x3451] = "\x51\x66", [0x3452] = "\x51\x67",
  [0x3453] = "\x51\x68", [0x3454] = "\x51\x69", [0x3455] = "\x51\x6a",
  [0x3456] = "\x51\x6b", [0x3457] = "\x51\x6c", [0x3458] = "\x51\x6d",
  [0x3459] = "\x51\x6e", [0x345a] = "\x51\x6f", [0x345b] = "\x51\x70",
  [0x345c] = "\x51\x71", [0x345d] = "\x51\x72", [0x345e] = "\x51\x73",
  [0x345f] = "\x51\x74", [0x3460] = "\x51\x75", [0x3461] = "\x51\x76",
  [0x3462] = "\x51\x77", [0x3463] = "\x51\x78", [0x3464] = "\x51\x79",
  [0x3465] = "\x51\x7a", [0x3466] = "\x51\x7b", [0x3467] = "\x51\x7c",
  [0x3468] = "\x51\x7d", [0x3469] = "\x51\x7e", [0x346a] = "\x51\x7f",
  [0x346b] = "\x51\x80", [0x346c] = "\x51\x81", [0x346d] = "\x51\x82",
  [0x346e] = "\x51\x83", [0x346f] = "\x51\x84", [0x3470] = "\x51\x85",
  [0x3471] = "\x51\x86", [0x3472] = "\x51\x87", [0x3473] = "\x51\x88",
  [0x3474] = "\x51\x89", [0x3475] = "\x51\x8a", [0x3476] = "\x51\x8b",
  [0x3477] = "\x51\x8c", [0x3478] = "\x51\x8d", [0x3479] = "\x51\x8e",
  [0x347a] = "\x51\x8f", [0x347b] = "\x51\x90", [0x347c] = "\x51\x91",
  [0x347d] = "\x51\x92", [0x347e] = "\x51\x93", [0x347f] = "\x51\x94",
  [0x3480] = "\x51\x95", [0x3481] = "\x51\x96", [0x3482] = "\x51\x97",
  [0x3483] = "\x51\x98", [0x3484] = "\x51\x99", [0x3485] = "\x51\x9a",
  [0x3486] = "\x51\x9b", [0x3487] = "\x51\x9c", [0x3488] = "\x51\x9d",
  [0x3489] = "\x51\x9e", [0x348a] = "\x51\x9f", [0x348b] = "\x51\xa0",
  [0x348c] = "\x51\xa1", [0x348d] = "\x51\xa2", [0x348e] = "\x51\xa3",
  [0x348f] = "\x51\xa4", [0x3490] = "\x51\xa5", [0x3491] = "\x51\xa6",
  [0x3492] = "\x51\xa7", [0x3493] = "\x51\xa8", [0x3494] = "\x51\xa9",
  [0x3495] = "\x51\xaa", [0x3496] = "\x51\xab", [0x3497] = "\x51\xac",
  [0x3498] = "\x51\xad", [0x3499] = "\x51\xae", [0x349a] = "\x51\xaf",
  [0x349b] = "\x51\xb0", [0x349c] = "\x51\xb1", [0x349d] = "\x51\xb2",
  [0x349e] = "\x51\xb3", [0x349f] = "\x51\xb4", [0x34a0] = "\x51\xb5",
  [0x34a1] = "\x51\xb6", [0x34a2] = "\x51\xb7", [0x34a3] = "\x51\xb8",
  [0x34a4] = "\x51\xb9", [0x34a5] = "\x51\xba", [0x34a6] = "\x51\xbb",
  [0x34a7] = "\x51\xbc", [0x34a8] = "\x51\xbd", [0x34a9] = "\x51\xbe",
  [0x34aa] = "\x51\xbf", [0x34ab] = "\x51\xc0", [0x34ac] = "\x51\xc1",
  [0x34ad] = "\x51\xc2", [0x34ae] = "\x51\xc3", [0x34af] = "\x51\xc4",
  [0x34b0] = "\x51\xc5", [0x34b1] = "\x51\xc6", [0x34b2] = "\x51\xc7",
  [0x34b3] = "\x51\xc8", [0x34b4] = "\x51\xc9", [0x34b5] = "\x51\xca",
  [0x34b6] = "\x51\xcb", [0x34b7] = "\x51\xcc", [0x34b8] = "\x51\xcd",
  [0x34b9] = "\x51\xce", [0x34ba] = "\x51\xcf", [0x34bb] = "\x51\xd0",
  [0x34bc] = "\x51\xd1", [0x34bd] = "\x51\xd2", [0x34be] = "\x51\xd3",
  [0x34bf] = "\x51\xd4", [0x34c0] = "\x51\xd5", [0x34c1] = "\x51\xd6",
  [0x34c2] = "\x51\xd7", [0x34c3] = "\x51\xd8", [0x34c4] = "\x51\xd9",
  [0x34c5] = "\x51\xda", [0x34c6] = "\x51\xdb", [0x34c7] = "\x51\xdc",
  [0x34c8] = "\x51\xdd", [0x34c9] = "\x51\xde", [0x34ca] = "\x51\xdf",
  [0x34cb] = "\x51\xe0", [0x34cc] = "\x51\xe1", [0x34cd] = "\x51\xe2",
  [0x34ce] = "\x51\xe3", [0x34cf] = "\x51\xe4", [0x34d0] = "\x51\xe5",
  [0x34d1] = "\x51\xe6", [0x34d2] = "\x51\xe7", [0x34d3] = "\x51\xe8",
  [0x34d4] = "\x51\xe9", [0x34d5] = "\x51\xea", [0x34d6] = "\x51\xeb",
  [0x34d7] = "\x51\xec", [0x34d8] = "\x51\xed", [0x34d9] = "\x51\xee",
  [0x34da] = "\x51\xef", [0x34db] = "\x51\xf0", [0x34dc] = "\x51\xf1",
  [0x34dd] = "\x51\xf2", [0x34de] = "\x51\xf3", [0x34df] = "\x51\xf4",
  [0x34e0] = "\x51\xf5", [0x34e1] = "\x51\xf6", [0x34e2] = "\x51\xf7",
  [0x34e3] = "\x51\xf8", [0x34e4] = "\x51\xf9", [0x34e5] = "\x51\xfa",
  [0x34e6] = "\x51\xfb", [0x34e7] = "\x51\xfc", [0x34e8] = "\x51\xfd",
  [0x34e9] = "\x51\xfe", [0x34ea] = "\x51\xff", [0x34eb] = "\x52\x00",
  [0x34ec] = "\x52\x01", [0x34ed] = "\x52\x02", [0x34ee] = "\x52\x03",
  [0x34ef] = "\x52\x04", [0x34f0] = "\x52\x05", [0x34f1] = "\x52\x06",
  [0x34f2] = "\x52\x07", [0x34f3] = "\x52\x08", [0x34f4] = "\x52\x09",
  [0x34f5] = "\x52\x0a", [0x34f6] = "\x52\x0b", [0x34f7] = "\x52\x0c",
  [0x34f8] = "\x52\x0d", [0x34f9] = "\x52\x0e", [0x34fa] = "\x52\x0f",
  [0x34fb] = "\x52\x10", [0x34fc] = "\x52\x11", [0x34fd] = "\x52\x12",
  [0x34fe] = "\x52\x13", [0x34ff] = "\x52\x14", [0x3500] = "\x52\x15",
  [0x3501] = "\x52\x16", [0x3502] = "\x52\x17", [0x3503] = "\x52\x18",
  [0x3504] = "\x52\x19", [0x3505] = "\x52\x1a", [0x3506] = "\x52\x1b",
  [0x3507] = "\x52\x1c", [0x3508] = "\x52\x1d", [0x3509] = "\x52\x1e",
  [0x350a] = "\x52\x1f", [0x350b] = "\x52\x20", [0x350c] = "\x52\x21",
  [0x350d] = "\x52\x22", [0x350e] = "\x52\x23", [0x350f] = "\x52\x24",
  [0x3510] = "\x52\x25", [0x3511] = "\x52\x26", [0x3512] = "\x52\x27",
  [0x3513] = "\x52\x28", [0x3514] = "\x52\x29", [0x3515] = "\x52\x2a",
  [0x3516] = "\x52\x2b", [0x3517] = "\x52\x2c", [0x3518] = "\x52\x2d",
  [0x3519] = "\x52\x2e", [0x351a] = "\x52\x2f", [0x351b] = "\x52\x30",
  [0x351c] = "\x52\x31", [0x351d] = "\x52\x32", [0x351e] = "\xfe\x5a",
  [0x351f] = "\x52\x33", [0x3520] = "\x52\x34", [0x3521] = "\x52\x35",
  [0x3522] = "\x52\x36", [0x3523] = "\x52\x37", [0x3524] = "\x52\x38",
  [0x3525] = "\x52\x39", [0x3526] = "\x52\x3a", [0x3527] = "\x52\x3b",
  [0x3528] = "\x52\x3c", [0x3529] = "\x52\x3d", [0x352a] = "\x52\x3e",
  [0x352b] = "\x52\x3f", [0x352c] = "\x52\x40", [0x352d] = "\x52\x41",
  [0x352e] = "\x52\x42", [0x352f] = "\x52\x43", [0x3530] = "\x52\x44",
  [0x3531] = "\x52\x45", [0x3532] = "\x52\x46", [0x3533] = "\x52\x47",
  [0x3534] = "\x52\x48", [0x3535] = "\x52\x49", [0x3536] = "\x52\x4a",
  [0x3537] = "\x52\x4b", [0x3538] = "\x52\x4c", [0x3539] = "\x52\x4d",
  [0x353a] = "\x52\x4e", [0x353b] = "\x52\x4f", [0x353c] = "\x52\x50",
  [0x353d] = "\x52\x51", [0x353e] = "\x52\x52", [0x353f] = "\x52\x53",
  [0x3540] = "\x52\x54", [0x3541] = "\x52\x55", [0x3542] = "\x52\x56",
  [0x3543] = "\x52\x57", [0x3544] = "\x52\x58", [0x3545] = "\x52\x59",
  [0x3546] = "\x52\x5a", [0x3547] = "\x52\x5b", [0x3548] = "\x52\x5c",
  [0x3549] = "\x52\x5d", [0x354a] = "\x52\x5e", [0x354b] = "\x52\x5f",
  [0x354c] = "\x52\x60", [0x354d] = "\x52\x61", [0x354e] = "\x52\x62",
  [0x354f] = "\x52\x63", [0x3550] = "\x52\x64", [0x3551] = "\x52\x65",
  [0x3552] = "\x52\x66", [0x3553] = "\x52\x67", [0x3554] = "\x52\x68",
  [0x3555] = "\x52\x69", [0x3556] = "\x52\x6a", [0x3557] = "\x52\x6b",
  [0x3558] = "\x52\x6c", [0x3559] = "\x52\x6d", [0x355a] = "\x52\x6e",
  [0x355b] = "\x52\x6f", [0x355c] = "\x52\x70", [0x355d] = "\x52\x71",
  [0x355e] = "\x52\x72", [0x355f] = "\x52\x73", [0x3560] = "\x52\x74",
  [0x3561] = "\x52\x75", [0x3562] = "\x52\x76", [0x3563] = "\x52\x77",
  [0x3564] = "\x52\x78", [0x3565] = "\x52\x79", [0x3566] = "\x52\x7a",
  [0x3567] = "\x52\x7b", [0x3568] = "\x52\x7c", [0x3569] = "\x52\x7d",
  [0x356a] = "\x52\x7e", [0x356b] = "\x52\x7f", [0x356c] = "\x52\x80",
  [0x356d] = "\x52\x81", [0x356e] = "\x52\x82", [0x356f] = "\x52\x83",
  [0x3570] = "\x52\x84", [0x3571] = "\x52\x85", [0x3572] = "\x52\x86",
  [0x3573] = "\x52\x87", [0x3574] = "\x52\x88", [0x3575] = "\x52\x89",
  [0x3576] = "\x52\x8a", [0x3577] = "\x52\x8b", [0x3578] = "\x52\x8c",
  [0x3579] = "\x52\x8d", [0x357a] = "\x52\x8e", [0x357b] = "\x52\x8f",
  [0x357c] = "\x52\x90", [0x357d] = "\x52\x91", [0x357e] = "\x52\x92",
  [0x357f] = "\x52\x93", [0x3580] = "\x52\x94", [0x3581] = "\x52\x95",
  [0x3582] = "\x52\x96", [0x3583] = "\x52\x97", [0x3584] = "\x52\x98",
  [0x3585] = "\x52\x99", [0x3586] = "\x52\x9a", [0x3587] = "\x52\x9b",
  [0x3588] = "\x52\x9c", [0x3589] = "\x52\x9d", [0x358a] = "\x52\x9e",
  [0x358b] = "\x52\x9f", [0x358c] = "\x52\xa0", [0x358d] = "\x52\xa1",
  [0x358e] = "\xfe\x5c", [0x358f] = "\x52\xa2", [0x3590] = "\x52\xa3",
  [0x3591] = "\x52\xa4", [0x3592] = "\x52\xa5", [0x3593] = "\x52\xa6",
  [0x3594] = "\x52\xa7", [0x3595] = "\x52\xa8", [0x3596] = "\x52\xa9",
  [0x3597] = "\x52\xaa", [0x3598] = "\x52\xab", [0x3599] = "\x52\xac",
  [0x359a] = "\xfe\x5b", [0x3898] = "\xfe\x60", [0x3899] = "\x55\xaa",
  [0x389a] = "\x55\xab", [0x389b] = "\x55\xac", [0x389c] = "\x55\xad",
  [0x389d] = "\x55\xae", [0x389e] = "\x55\xaf", [0x389f] = "\x55\xb0",
  [0x38a0] = "\x55\xb1", [0x38a1] = "\x55\xb2", [0x38a2] = "\x55\xb3",
  [0x38a3] = "\x55\xb4", [0x38a4] = "\x55\xb5", [0x38a5] = "\x55\xb6",
  [0x38a6] = "\x55\xb7", [0x38a7] = "\x55\xb8", [0x38a8] = "\x55\xb9",
  [0x38a9] = "\x55\xba", [0x38aa] = "\x55\xbb", [0x38ab] = "\x55\xbc",
  [0x38ac] = "\x55\xbd", [0x38ad] = "\x55\xbe", [0x38ae] = "\x55\xbf",
  [0x38af] = "\x55\xc0", [0x38b0] = "\x55\xc1", [0x38b1] = "\x55\xc2",
  [0x38b2] = "\x55\xc3", [0x38b3] = "\x55\xc4", [0x38b4] = "\x55\xc5",
  [0x38b5] = "\x55\xc6", [0x38b6] = "\x55\xc7", [0x38b7] = "\x55\xc8",
  [0x38b8] = "\x55\xc9", [0x38b9] = "\x55\xca", [0x38ba] = "\x55\xcb",
  [0x38bb] = "\x55\xcc", [0x38bc] = "\x55\xcd", [0x38bd] = "\x55\xce",
  [0x38be] = "\x55\xcf", [0x38bf] = "\x55\xd0", [0x38c0] = "\x55\xd1",
  [0x38c1] = "\x55\xd2", [0x38c2] = "\x55\xd3", [0x38c3] = "\x55\xd4",
  [0x38c4] = "\x55\xd5", [0x38c5] = "\x55\xd6", [0x38c6] = "\x55\xd7",
  [0x38c7] = "\x55\xd8", [0x38c8] = "\x55\xd9", [0x38c9] = "\x55\xda",
  [0x38ca] = "\x55\xdb", [0x38cb] = "\x55\xdc", [0x38cc] = "\x55\xdd",
  [0x38cd] = "\x55\xde", [0x38ce] = "\x55\xdf", [0x38cf] = "\x55\xe0",
  [0x38d0] = "\x55\xe1", [0x38d1] = "\x55\xe2", [0x38d2] = "\x55\xe3",
  [0x38d3] = "\x55\xe4", [0x38d4] = "\x55\xe5", [0x38d5] = "\x55\xe6",
  [0x38d6] = "\x55\xe7", [0x38d7] = "\x55\xe8", [0x38d8] = "\x55\xe9",
  [0x38d9] = "\x55\xea", [0x38da] = "\x55\xeb", [0x38db] = "\x55\xec",
  [0x38dc] = "\x55\xed", [0x38dd] = "\x55\xee", [0x38de] = "\x55\xef",
  [0x38df] = "\x55\xf0", [0x38e0] = "\x55\xf1", [0x38e1] = "\x55\xf2",
  [0x38e2] = "\x55\xf3", [0x38e3] = "\x55\xf4", [0x38e4] = "\x55\xf5",
  [0x38e5] = "\x55\xf6", [0x38e6] = "\x55\xf7", [0x38e7] = "\x55\xf8",
  [0x38e8] = "\x55\xf9", [0x38e9] = "\x55\xfa", [0x38ea] = "\x55\xfb",
  [0x38eb] = "\x55\xfc", [0x38ec] = "\x55\xfd", [0x38ed] = "\x55\xfe",
  [0x38ee] = "\xfe\x5f", [0x38ef] = "\x55\xff", [0x38f0] = "\x56\x00",
  [0x38f1] = "\x56\x01", [0x38f2] = "\x56\x02", [0x38f3] = "\x56\x03",
  [0x38f4] = "\x56\x04", [0x38f5] = "\x56\x05", [0x38f6] = "\x56\x06",
  [0x38f7] = "\x56\x07", [0x38f8] = "\x56\x08", [0x38f9] = "\x56\x09",
  [0x38fa] = "\x56\x0a", [0x38fb] = "\x56\x0b", [0x38fc] = "\x56\x0c",
  [0x38fd] = "\x56\x0d", [0x38fe] = "\x56\x0e", [0x38ff] = "\x56\x0f",
  [0x3900] = "\x56\x10", [0x3901] = "\x56\x11", [0x3902] = "\x56\x12",
  [0x3903] = "\x56\x13", [0x3904] = "\x56\x14", [0x3905] = "\x56\x15",
  [0x3906] = "\x56\x16", [0x3907] = "\x56\x17", [0x3908] = "\x56\x18",
  [0x3909] = "\x56\x19", [0x390a] = "\x56\x1a", [0x390b] = "\x56\x1b",
  [0x390c] = "\x56\x1c", [0x390d] = "\x56\x1d", [0x390e] = "\x56\x1e",
  [0x390f] = "\x56\x1f", [0x3910] = "\x56\x20", [0x3911] = "\x56\x21",
  [0x3912] = "\x56\x22", [0x3913] = "\x56\x23", [0x3914] = "\x56\x24",
  [0x3915] = "\x56\x25", [0x3916] = "\x56\x26", [0x3917] = "\x56\x27",
  [0x3918] = "\x56\x28", [0x3919] = "\x56\x29", [0x391a] = "\x56\x2a",
  [0x391b] = "\x56\x2b", [0x391c] = "\x56\x2c", [0x391d] = "\x56\x2d",
  [0x391e] = "\x56\x2e", [0x391f] = "\x56\x2f", [0x3920] = "\x56\x30",
  [0x3921] = "\x56\x31", [0x3922] = "\x56\x32", [0x3923] = "\x56\x33",
  [0x3924] = "\x56\x34", [0x3925] = "\x56\x35", [0x3926] = "\x56\x36",
  [0x3927] = "\x56\x37", [0x3928] = "\x56\x38", [0x3929] = "\x56\x39",
  [0x392a] = "\x56\x3a", [0x392b] = "\x56\x3b", [0x392c] = "\x56\x3c",
  [0x392d] = "\x56\x3d", [0x392e] = "\x56\x3e", [0x392f] = "\x56\x3f",
  [0x3930] = "\x56\x40", [0x3931] = "\x56\x41", [0x3932] = "\x56\x42",
  [0x3933] = "\x56\x43", [0x3934] = "\x56\x44", [0x3935] = "\x56\x45",
  [0x3936] = "\x56\x46", [0x3937] = "\x56\x47", [0x3938] = "\x56\x48",
  [0x3939] = "\x56\x49", [0x393a] = "\x56\x4a", [0x393b] = "\x56\x4b",
  [0x393c] = "\x56\x4c", [0x393d] = "\x56\x4d", [0x393e] = "\x56\x4e",
  [0x393f] = "\x56\x4f", [0x3940] = "\x56\x50", [0x3941] = "\x56\x51",
  [0x3942] = "\x56\x52", [0x3943] = "\x56\x53", [0x3944] = "\x56\x54",
  [0x3945] = "\x56\x55", [0x3946] = "\x56\x56", [0x3947] = "\x56\x57",
  [0x3948] = "\x56\x58", [0x3949] = "\x56\x59", [0x394a] = "\x56\x5a",
  [0x394b] = "\x56\x5b", [0x394c] = "\x56\x5c", [0x394d] = "\x56\x5d",
  [0x394e] = "\x56\x5e", [0x394f] = "\xfe\x62", [0x3950] = "\xfe\x65",
  [0x3951] = "\x56\x5f", [0x3952] = "\x56\x60", [0x3953] = "\x56\x61",
  [0x3954] = "\x56\x62", [0x3955] = "\x56\x63", [0x3956] = "\x56\x64",
  [0x3957] = "\x56\x65", [0x3958] = "\x56\x66", [0x3959] = "\x56\x67",
  [0x395a] = "\x56\x68", [0x395b] = "\x56\x69", [0x395c] = "\x56\x6a",
  [0x395d] = "\x56\x6b", [0x395e] = "\x56\x6c", [0x395f] = "\xfe\x63",
  [0x3960] = "\x56\x6d", [0x3961] = "\x56\x6e", [0x3962] = "\x56\x6f",
  [0x3963] = "\x56\x70", [0x3964] = "\x56\x71", [0x3965] = "\x56\x72",
  [0x3966] = "\x56\x73", [0x3967] = "\x56\x74", [0x3968] = "\x56\x75",
  [0x3969] = "\x56\x76", [0x396a] = "\x56\x77", [0x396b] = "\x56\x78",
  [0x396c] = "\x56\x79", [0x396d] = "\x56\x7a", [0x396e] = "\x56\x7b",
  [0x396f] = "\x56\x7c", [0x3970] = "\x56\x7d", [0x3971] = "\x56\x7e",
  [0x3972] = "\x56\x7f", [0x3973] = "\x56\x80", [0x3974] = "\x56\x81",
  [0x3975] = "\x56\x82", [0x3976] = "\x56\x83", [0x3977] = "\x56\x84",
  [0x3978] = "\x56\x85", [0x3979] = "\x56\x86", [0x397a] = "\x56\x87",
  [0x397b] = "\x56\x88", [0x397c] = "\x56\x89", [0x397d] = "\x56\x8a",
  [0x397e] = "\x56\x8b", [0x397f] = "\x56\x8c", [0x3980] = "\x56\x8d",
  [0x3981] = "\x56\x8e", [0x3982] = "\x56\x8f", [0x3983] = "\x56\x90",
  [0x3984] = "\x56\x91", [0x3985] = "\x56\x92", [0x3986] = "\x56\x93",
  [0x3987] = "\x56\x94", [0x3988] = "\x56\x95", [0x3989] = "\x56\x96",
  [0x398a] = "\x56\x97", [0x398b] = "\x56\x98", [0x398c] = "\x56\x99",
  [0x398d] = "\x56\x9a", [0x398e] = "\x56\x9b", [0x398f] = "\x56\x9c",
  [0x3990] = "\x56\x9d", [0x3991] = "\x56\x9e", [0x3992] = "\x56\x9f",
  [0x3993] = "\x56\xa0", [0x3994] = "\x56\xa1", [0x3995] = "\x56\xa2",
  [0x3996] = "\x56\xa3", [0x3997] = "\x56\xa4", [0x3998] = "\x56\xa5",
  [0x3999] = "\x56\xa6", [0x399a] = "\x56\xa7", [0x399b] = "\x56\xa8",
  [0x399c] = "\x56\xa9", [0x399d] = "\x56\xaa", [0x399e] = "\x56\xab",
  [0x399f] = "\x56\xac", [0x39a0] = "\x56\xad", [0x39a1] = "\x56\xae",
  [0x39a2] = "\x56\xaf", [0x39a3] = "\x56\xb0", [0x39a4] = "\x56\xb1",
  [0x39a5] = "\x56\xb2", [0x39a6] = "\x56\xb3", [0x39a7] = "\x56\xb4",
  [0x39a8] = "\x56\xb5", [0x39a9] = "\x56\xb6", [0x39aa] = "\x56\xb7",
  [0x39ab] = "\x56\xb8", [0x39ac] = "\x56\xb9", [0x39ad] = "\x56\xba",
  [0x39ae] = "\x56\xbb", [0x39af] = "\x56\xbc", [0x39b0] = "\x56\xbd",
  [0x39b1] = "\x56\xbe", [0x39b2] = "\x56\xbf", [0x39b3] = "\x56\xc0",
  [0x39b4] = "\x56\xc1", [0x39b5] = "\x56\xc2", [0x39b6] = "\x56\xc3",
  [0x39b7] = "\x56\xc4", [0x39b8] = "\x56\xc5", [0x39b9] = "\x56\xc6",
  [0x39ba] = "\x56\xc7", [0x39bb] = "\x56\xc8", [0x39bc] = "\x56\xc9",
  [0x39bd] = "\x56\xca", [0x39be] = "\x56\xcb", [0x39bf] = "\x56\xcc",
  [0x39c0] = "\x56\xcd", [0x39c1] = "\x56\xce", [0x39c2] = "\x56\xcf",
  [0x39c3] = "\x56\xd0", [0x39c4] = "\x56\xd1", [0x39c5] = "\x56\xd2",
  [0x39c6] = "\x56\xd3", [0x39c7] = "\x56\xd4", [0x39c8] = "\x56\xd5",
  [0x39c9] = "\x56\xd6", [0x39ca] = "\x56\xd7", [0x39cb] = "\x56\xd8",
  [0x39cc] = "\x56\xd9", [0x39cd] = "\x56\xda", [0x39ce] = "\x56\xdb",
  [0x39cf] = "\x56\xdc", [0x39d0] = "\x56\xdd", [0x39d1] = "\x56\xde",
  [0x39d2] = "\x56\xdf", [0x39d3] = "\x56\xe0", [0x39d4] = "\x56\xe1",
  [0x39d5] = "\x56\xe2", [0x39d6] = "\x56\xe3", [0x39d7] = "\x56\xe4",
  [0x39d8] = "\x56\xe5", [0x39d9] = "\x56\xe6", [0x39da] = "\x56\xe7",
  [0x39db] = "\x56\xe8", [0x39dc] = "\x56\xe9", [0x39dd] = "\x56\xea",
  [0x39de] = "\x56\xeb", [0x39df] = "\x56\xec", [0x39e0] = "\x56\xed",
  [0x39e1] = "\x56\xee", [0x39e2] = "\x56\xef", [0x39e3] = "\x56\xf0",
  [0x39e4] = "\x56\xf1", [0x39e5] = "\x56\xf2", [0x39e6] = "\x56\xf3",
  [0x39e7] = "\x56\xf4", [0x39e8] = "\x56\xf5", [0x39e9] = "\x56\xf6",
  [0x39ea] = "\x56\xf7", [0x39eb] = "\x56\xf8", [0x39ec] = "\x56\xf9",
  [0x39ed] = "\x56\xfa", [0x39ee] = "\x56\xfb", [0x39ef] = "\x56\xfc",
  [0x39f0] = "\x56\xfd", [0x39f1] = "\x56\xfe", [0x39f2] = "\x56\xff",
  [0x39f3] = "\xfe\x64", [0x39f4] = "\x57\x00", [0x39f5] = "\x57\x01",
  [0x39f6] = "\x57\x02", [0x39f7] = "\x57\x03", [0x39f8] = "\x57\x04",
  [0x39f9] = "\x57\x05", [0x39fa] = "\x57\x06", [0x39fb] = "\x57\x07",
  [0x39fc] = "\x57\x08", [0x39fd] = "\x57\x09", [0x39fe] = "\x57\x0a",
  [0x39ff] = "\x57\x0b", [0x3a00] = "\x57\x0c", [0x3a01] = "\x57\x0d",
  [0x3a02] = "\x57\x0e", [0x3a03] = "\x57\x0f", [0x3a04] = "\x57\x10",
  [0x3a05] = "\x57\x11", [0x3a06] = "\x57\x12", [0x3a07] = "\x57\x13",
  [0x3a08] = "\x57\x14", [0x3a09] = "\x57\x15", [0x3a0a] = "\x57\x16",
  [0x3a0b] = "\x57\x17", [0x3a0c] = "\x57\x18", [0x3a0d] = "\x57\x19",
  [0x3a0e] = "\x57\x1a", [0x3a0f] = "\x57\x1b", [0x3a10] = "\x57\x1c",
  [0x3a11] = "\x57\x1d", [0x3a12] = "\x57\x1e", [0x3a13] = "\x57\x1f",
  [0x3a14] = "\x57\x20", [0x3a15] = "\x57\x21", [0x3a16] = "\x57\x22",
  [0x3a17] = "\x57\x23", [0x3a18] = "\x57\x24", [0x3a19] = "\x57\x25",
  [0x3a1a] = "\x57\x26", [0x3a1b] = "\x57\x27", [0x3a1c] = "\x57\x28",
  [0x3a1d] = "\x57\x29", [0x3a1e] = "\x57\x2a", [0x3a1f] = "\x57\x2b",
  [0x3a20] = "\x57\x2c", [0x3a21] = "\x57\x2d", [0x3a22] = "\x57\x2e",
  [0x3a23] = "\x57\x2f", [0x3a24] = "\x57\x30", [0x3a25] = "\x57\x31",
  [0x3a26] = "\x57\x32", [0x3a27] = "\x57\x33", [0x3a28] = "\x57\x34",
  [0x3a29] = "\x57\x35", [0x3a2a] = "\x57\x36", [0x3a2b] = "\x57\x37",
  [0x3a2c] = "\x57\x38", [0x3a2d] = "\x57\x39", [0x3a2e] = "\x57\x3a",
  [0x3a2f] = "\x57\x3b", [0x3a30] = "\x57\x3c", [0x3a31] = "\x57\x3d",
  [0x3a32] = "\x57\x3e", [0x3a33] = "\x57\x3f", [0x3a34] = "\x57\x40",
  [0x3a35] = "\x57\x41", [0x3a36] = "\x57\x42", [0x3a37] = "\x57\x43",
  [0x3a38] = "\x57\x44", [0x3a39] = "\x57\x45", [0x3a3a] = "\x57\x46",
  [0x3a3b] = "\x57\x47", [0x3a3c] = "\x57\x48", [0x3a3d] = "\x57\x49",
  [0x3a3e] = "\x57\x4a", [0x3a3f] = "\x57\x4b", [0x3a40] = "\x57\x4c",
  [0x3a41] = "\x57\x4d", [0x3a42] = "\x57\x4e", [0x3a43] = "\x57\x4f",
  [0x3a44] = "\x57\x50", [0x3a45] = "\x57\x51", [0x3a46] = "\x57\x52",
  [0x3a47] = "\x57\x53", [0x3a48] = "\x57\x54", [0x3a49] = "\x57\x55",
  [0x3a4a] = "\x57\x56", [0x3a4b] = "\x57\x57", [0x3a4c] = "\x57\x58",
  [0x3a4d] = "\x57\x59", [0x3a4e] = "\x57\x5a", [0x3a4f] = "\x57\x5b",
  [0x3a50] = "\x57\x5c", [0x3a51] = "\x57\x5d", [0x3a52] = "\x57\x5e",
  [0x3a53] = "\x57\x5f", [0x3a54] = "\x57\x60", [0x3a55] = "\x57\x61",
  [0x3a56] = "\x57\x62", [0x3a57] = "\x57\x63", [0x3a58] = "\x57\x64",
  [0x3a59] = "\x57\x65", [0x3a5a] = "\x57\x66", [0x3a5b] = "\x57\x67",
  [0x3a5c] = "\x57\x68", [0x3a5d] = "\x57\x69", [0x3a5e] = "\x57\x6a",
  [0x3a5f] = "\x57\x6b", [0x3a60] = "\x57\x6c", [0x3a61] = "\x57\x6d",
  [0x3a62] = "\x57\x6e", [0x3a63] = "\x57\x6f", [0x3a64] = "\x57\x70",
  [0x3a65] = "\x57\x71", [0x3a66] = "\x57\x72", [0x3a67] = "\x57\x73",
  [0x3a68] = "\x57\x74", [0x3a69] = "\x57\x75", [0x3a6a] = "\x57\x76",
  [0x3a6b] = "\x57\x77", [0x3a6c] = "\x57\x78", [0x3a6d] = "\x57\x79",
  [0x3a6e] = "\x57\x7a", [0x3a6f] = "\x57\x7b", [0x3a70] = "\x57\x7c",
  [0x3a71] = "\x57\x7d", [0x3a72] = "\x57\x7e", [0x3a73] = "\x57\x7f",
  [0x3a74] = "\x57\x80", [0x3a75] = "\x57\x81", [0x3a76] = "\x57\x82",
  [0x3a77] = "\x57\x83", [0x3a78] = "\x57\x84", [0x3a79] = "\x57\x85",
  [0x3a7a] = "\x57\x86", [0x3a7b] = "\x57\x87", [0x3a7c] = "\x57\x88",
  [0x3a7d] = "\x57\x89", [0x3a7e] = "\x57\x8a", [0x3a7f] = "\x57\x8b",
  [0x3a80] = "\x57\x8c", [0x3a81] = "\x57\x8d", [0x3a82] = "\x57\x8e",
  [0x3a83] = "\x57\x8f", [0x3a84] = "\x57\x90", [0x3a85] = "\x57\x91",
  [0x3a86] = "\x57\x92", [0x3a87] = "\x57\x93", [0x3a88] = "\x57\x94",
  [0x3a89] = "\x57\x95", [0x3a8a] = "\x57\x96", [0x3a8b] = "\x57\x97",
  [0x3a8c] = "\x57\x98", [0x3a8d] = "\x57\x99", [0x3a8e] = "\x57\x9a",
  [0x3a8f] = "\x57\x9b", [0x3a90] = "\x57\x9c", [0x3a91] = "\x57\x9d",
  [0x3a92] = "\x57\x9e", [0x3a93] = "\x57\x9f", [0x3a94] = "\x57\xa0",
  [0x3a95] = "\x57\xa1", [0x3a96] = "\x57\xa2", [0x3a97] = "\x57\xa3",
  [0x3a98] = "\x57\xa4", [0x3a99] = "\x57\xa5", [0x3a9a] = "\x57\xa6",
  [0x3a9b] = "\x57\xa7", [0x3a9c] = "\x57\xa8", [0x3a9d] = "\x57\xa9",
  [0x3a9e] = "\x57\xaa", [0x3a9f] = "\x57\xab", [0x3aa0] = "\x57\xac",
  [0x3aa1] = "\x57\xad", [0x3aa2] = "\x57\xae", [0x3aa3] = "\x57\xaf",
  [0x3aa4] = "\x57\xb0", [0x3aa5] = "\x57\xb1", [0x3aa6] = "\x57\xb2",
  [0x3aa7] = "\x57\xb3", [0x3aa8] = "\x57\xb4", [0x3aa9] = "\x57\xb5",
  [0x3aaa] = "\x57\xb6", [0x3aab] = "\x57\xb7", [0x3aac] = "\x57\xb8",
  [0x3aad] = "\x57\xb9", [0x3aae] = "\x57\xba", [0x3aaf] = "\x57\xbb",
  [0x3ab0] = "\x57\xbc", [0x3ab1] = "\x57\xbd", [0x3ab2] = "\x57\xbe",
  [0x3ab3] = "\x57\xbf", [0x3ab4] = "\x57\xc0", [0x3ab5] = "\x57\xc1",
  [0x3ab6] = "\x57\xc2", [0x3ab7] = "\x57\xc3", [0x3ab8] = "\x57\xc4",
  [0x3ab9] = "\x57\xc5", [0x3aba] = "\x57\xc6", [0x3abb] = "\x57\xc7",
  [0x3abc] = "\x57\xc8", [0x3abd] = "\x57\xc9", [0x3abe] = "\x57\xca",
  [0x3abf] = "\x57\xcb", [0x3ac0] = "\x57\xcc", [0x3ac1] = "\x57\xcd",
  [0x3ac2] = "\x57\xce", [0x3ac3] = "\x57\xcf", [0x3ac4] = "\x57\xd0",
  [0x3ac5] = "\x57\xd1", [0x3ac6] = "\x57\xd2", [0x3ac7] = "\x57\xd3",
  [0x3ac8] = "\x57\xd4", [0x3ac9] = "\x57\xd5", [0x3aca] = "\x57\xd6",
  [0x3acb] = "\x57\xd7", [0x3acc] = "\x57\xd8", [0x3acd] = "\x57\xd9",
  [0x3ace] = "\xfe\x68", [0x3acf] = "\x57\xda", [0x3ad0] = "\x57\xdb",
  [0x3ad1] = "\x57\xdc", [0x3ad2] = "\x57\xdd", [0x3ad3] = "\x57\xde",
  [0x3ad4] = "\x57\xdf", [0x3ad5] = "\x57\xe0", [0x3ad6] = "\x57\xe1",
  [0x3ad7] = "\x57\xe2", [0x3ad8] = "\x57\xe3", [0x3ad9] = "\x57\xe4",
  [0x3ada] = "\x57\xe5", [0x3adb] = "\x57\xe6", [0x3adc] = "\x57\xe7",
  [0x3add] = "\x57\xe8", [0x3ade] = "\x57\xe9", [0x3adf] = "\x57\xea",
  [0x3ae0] = "\x57\xeb", [0x3ae1] = "\x57\xec", [0x3ae2] = "\x57\xed",
  [0x3ae3] = "\x57\xee", [0x3ae4] = "\x57\xef", [0x3ae5] = "\x57\xf0",
  [0x3ae6] = "\x57\xf1", [0x3ae7] = "\x57\xf2", [0x3ae8] = "\x57\xf3",
  [0x3ae9] = "\x57\xf4", [0x3aea] = "\x57\xf5", [0x3aeb] = "\x57\xf6",
  [0x3aec] = "\x57\xf7", [0x3aed] = "\x57\xf8", [0x3aee] = "\x57\xf9",
  [0x3aef] = "\x57\xfa", [0x3af0] = "\x57\xfb", [0x3af1] = "\x57\xfc",
  [0x3af2] = "\x57\xfd", [0x3af3] = "\x57\xfe", [0x3af4] = "\x57\xff",
  [0x3af5] = "\x58\x00", [0x3af6] = "\x58\x01", [0x3af7] = "\x58\x02",
  [0x3af8] = "\x58\x03", [0x3af9] = "\x58\x04", [0x3afa] = "\x58\x05",
  [0x3afb] = "\x58\x06", [0x3afc] = "\x58\x07", [0x3afd] = "\x58\x08",
  [0x3afe] = "\x58\x09", [0x3aff] = "\x58\x0a", [0x3b00] = "\x58\x0b",
  [0x3b01] = "\x58\x0c", [0x3b02] = "\x58\x0d", [0x3b03] = "\x58\x0e",
  [0x3b04] = "\x58\x0f", [0x3b05] = "\x58\x10", [0x3b06] = "\x58\x11",
  [0x3b07] = "\x58\x12", [0x3b08] = "\x58\x13", [0x3b09] = "\x58\x14",
  [0x3b0a] = "\x58\x15", [0x3b0b] = "\x58\x16", [0x3b0c] = "\x58\x17",
  [0x3b0d] = "\x58\x18", [0x3b0e] = "\x58\x19", [0x3b0f] = "\x58\x1a",
  [0x3b10] = "\x58\x1b", [0x3b11] = "\x58\x1c", [0x3b12] = "\x58\x1d",
  [0x3b13] = "\x58\x1e", [0x3b14] = "\x58\x1f", [0x3b15] = "\x58\x20",
  [0x3b16] = "\x58\x21", [0x3b17] = "\x58\x22", [0x3b18] = "\x58\x23",
  [0x3b19] = "\x58\x24", [0x3b1a] = "\x58\x25", [0x3b1b] = "\x58\x26",
  [0x3b1c] = "\x58\x27", [0x3b1d] = "\x58\x28", [0x3b1e] = "\x58\x29",
  [0x3b1f] = "\x58\x2a", [0x3b20] = "\x58\x2b", [0x3b21] = "\x58\x2c",
  [0x3b22] = "\x58\x2d", [0x3b23] = "\x58\x2e", [0x3b24] = "\x58\x2f",
  [0x3b25] = "\x58\x30", [0x3b26] = "\x58\x31", [0x3b27] = "\x58\x32",
  [0x3b28] = "\x58\x33", [0x3b29] = "\x58\x34", [0x3b2a] = "\x58\x35",
  [0x3b2b] = "\x58\x36", [0x3b2c] = "\x58\x37", [0x3b2d] = "\x58\x38",
  [0x3b2e] = "\x58\x39", [0x3b2f] = "\x58\x3a", [0x3b30] = "\x58\x3b",
  [0x3b31] = "\x58\x3c", [0x3b32] = "\x58\x3d", [0x3b33] = "\x58\x3e",
  [0x3b34] = "\x58\x3f", [0x3b35] = "\x58\x40", [0x3b36] = "\x58\x41",
  [0x3b37] = "\x58\x42", [0x3b38] = "\x58\x43", [0x3b39] = "\x58\x44",
  [0x3b3a] = "\x58\x45", [0x3b3b] = "\x58\x46", [0x3b3c] = "\x58\x47",
  [0x3b3d] = "\x58\x48", [0x3b3e] = "\x58\x49", [0x3b3f] = "\x58\x4a",
  [0x3b40] = "\x58\x4b", [0x3b41] = "\x58\x4c", [0x3b42] = "\x58\x4d",
  [0x3b43] = "\x58\x4e", [0x3b44] = "\x58\x4f", [0x3b45] = "\x58\x50",
  [0x3b46] = "\x58\x51", [0x3b47] = "\x58\x52", [0x3b48] = "\x58\x53",
  [0x3b49] = "\x58\x54", [0x3b4a] = "\x58\x55", [0x3b4b] = "\x58\x56",
  [0x3b4c] = "\x58\x57", [0x3b4d] = "\x58\x58", [0x3b4e] = "\x58\x59",
  [0x3b4f] = "\x58\x5a", [0x3b50] = "\x58\x5b", [0x3b51] = "\x58\x5c",
  [0x3b52] = "\x58\x5d", [0x3b53] = "\x58\x5e", [0x3b54] = "\x58\x5f",
  [0x3b55] = "\x58\x60", [0x3b56] = "\x58\x61", [0x3b57] = "\x58\x62",
  [0x3b58] = "\x58\x63", [0x3b59] = "\x58\x64", [0x3b5a] = "\x58\x65",
  [0x3b5b] = "\x58\x66", [0x3b5c] = "\x58\x67", [0x3b5d] = "\x58\x68",
  [0x3b5e] = "\x58\x69", [0x3b5f] = "\x58\x6a", [0x3b60] = "\x58\x6b",
  [0x3b61] = "\x58\x6c", [0x3b62] = "\x58\x6d", [0x3b63] = "\x58\x6e",
  [0x3b64] = "\x58\x6f", [0x3b65] = "\x58\x70", [0x3b66] = "\x58\x71",
  [0x3b67] = "\x58\x72", [0x3b68] = "\x58\x73", [0x3b69] = "\x58\x74",
  [0x3b6a] = "\x58\x75", [0x3b6b] = "\x58\x76", [0x3b6c] = "\x58\x77",
  [0x3b6d] = "\x58\x78", [0x3b6e] = "\x58\x79", [0x3b6f] = "\x58\x7a",
  [0x3b70] = "\x58\x7b", [0x3b71] = "\x58\x7c", [0x3b72] = "\x58\x7d",
  [0x3b73] = "\x58\x7e", [0x3b74] = "\x58\x7f", [0x3b75] = "\x58\x80",
  [0x3b76] = "\x58\x81", [0x3b77] = "\x58\x82", [0x3b78] = "\x58\x83",
  [0x3b79] = "\x58\x84", [0x3b7a] = "\x58\x85", [0x3b7b] = "\x58\x86",
  [0x3b7c] = "\x58\x87", [0x3b7d] = "\x58\x88", [0x3b7e] = "\x58\x89",
  [0x3b7f] = "\x58\x8a", [0x3b80] = "\x58\x8b", [0x3b81] = "\x58\x8c",
  [0x3b82] = "\x58\x8d", [0x3b83] = "\x58\x8e", [0x3b84] = "\x58\x8f",
  [0x3b85] = "\x58\x90", [0x3b86] = "\x58\x91", [0x3b87] = "\x58\x92",
  [0x3b88] = "\x58\x93", [0x3b89] = "\x58\x94", [0x3b8a] = "\x58\x95",
  [0x3b8b] = "\x58\x96", [0x3b8c] = "\x58\x97", [0x3b8d] = "\x58\x98",
  [0x3b8e] = "\x58\x99", [0x3b8f] = "\x58\x9a", [0x3b90] = "\x58\x9b",
  [0x3b91] = "\x58\x9c", [0x3b92] = "\x58\x9d", [0x3b93] = "\x58\x9e",
  [0x3b94] = "\x58\x9f", [0x3b95] = "\x58\xa0", [0x3b96] = "\x58\xa1",
  [0x3b97] = "\x58\xa2", [0x3b98] = "\x58\xa3", [0x3b99] = "\x58\xa4",
  [0x3b9a] = "\x58\xa5", [0x3b9b] = "\x58\xa6", [0x3b9c] = "\x58\xa7",
  [0x3b9d] = "\x58\xa8", [0x3b9e] = "\x58\xa9", [0x3b9f] = "\x58\xaa",
  [0x3ba0] = "\x58\xab", [0x3ba1] = "\x58\xac", [0x3ba2] = "\x58\xad",
  [0x3ba3] = "\x58\xae", [0x3ba4] = "\x58\xaf", [0x3ba5] = "\x58\xb0",
  [0x3ba6] = "\x58\xb1", [0x3ba7] = "\x58\xb2", [0x3ba8] = "\x58\xb3",
  [0x3ba9] = "\x58\xb4", [0x3baa] = "\x58\xb5", [0x3bab] = "\x58\xb6",
  [0x3bac] = "\x58\xb7", [0x3bad] = "\x58\xb8", [0x3bae] = "\x58\xb9",
  [0x3baf] = "\x58\xba", [0x3bb0] = "\x58\xbb", [0x3bb1] = "\x58\xbc",
  [0x3bb2] = "\x58\xbd", [0x3bb3] = "\x58\xbe", [0x3bb4] = "\x58\xbf",
  [0x3bb5] = "\x58\xc0", [0x3bb6] = "\x58\xc1", [0x3bb7] = "\x58\xc2",
  [0x3bb8] = "\x58\xc3", [0x3bb9] = "\x58\xc4", [0x3bba] = "\x58\xc5",
  [0x3bbb] = "\x58\xc6", [0x3bbc] = "\x58\xc7", [0x3bbd] = "\x58\xc8",
  [0x3bbe] = "\x58\xc9", [0x3bbf] = "\x58\xca", [0x3bc0] = "\x58\xcb",
  [0x3bc1] = "\x58\xcc", [0x3bc2] = "\x58\xcd", [0x3bc3] = "\x58\xce",
  [0x3bc4] = "\x58\xcf", [0x3bc5] = "\x58\xd0", [0x3bc6] = "\x58\xd1",
  [0x3bc7] = "\x58\xd2", [0x3bc8] = "\x58\xd3", [0x3bc9] = "\x58\xd4",
  [0x3bca] = "\x58\xd5", [0x3bcb] = "\x58\xd6", [0x3bcc] = "\x58\xd7",
  [0x3bcd] = "\x58\xd8", [0x3bce] = "\x58\xd9", [0x3bcf] = "\x58\xda",
  [0x3bd0] = "\x58\xdb", [0x3bd1] = "\x58\xdc", [0x3bd2] = "\x58\xdd",
  [0x3bd3] = "\x58\xde", [0x3bd4] = "\x58\xdf", [0x3bd5] = "\x58\xe0",
  [0x3bd6] = "\x58\xe1", [0x3bd7] = "\x58\xe2", [0x3bd8] = "\x58\xe3",
  [0x3bd9] = "\x58\xe4", [0x3bda] = "\x58\xe5", [0x3bdb] = "\x58\xe6",
  [0x3bdc] = "\x58\xe7", [0x3bdd] = "\x58\xe8", [0x3bde] = "\x58\xe9",
  [0x3bdf] = "\x58\xea", [0x3be0] = "\x58\xeb", [0x3be1] = "\x58\xec",
  [0x3be2] = "\x58\xed", [0x3be3] = "\x58\xee", [0x3be4] = "\x58\xef",
  [0x3be5] = "\x58\xf0", [0x3be6] = "\x58\xf1", [0x3be7] = "\x58\xf2",
  [0x3be8] = "\x58\xf3", [0x3be9] = "\x58\xf4", [0x3bea] = "\x58\xf5",
  [0x3beb] = "\x58\xf6", [0x3bec] = "\x58\xf7", [0x3bed] = "\x58\xf8",
  [0x3bee] = "\xfe\x69", [0x3bef] = "\x58\xf9", [0x3bf0] = "\x58\xfa",
  [0x3bf1] = "\x58\xfb", [0x3bf2] = "\x58\xfc", [0x3bf3] = "\x58\xfd",
  [0x3bf4] = "\x58\xfe", [0x3bf5] = "\x58\xff", [0x3bf6] = "\x59\x00",
  [0x3bf7] = "\x59\x01", [0x3bf8] = "\x59\x02", [0x3bf9] = "\x59\x03",
  [0x3bfa] = "\x59\x04", [0x3bfb] = "\x59\x05", [0x3bfc] = "\x59\x06",
  [0x3bfd] = "\x59\x07", [0x3bfe] = "\x59\x08", [0x3bff] = "\x59\x09",
  [0x3c00] = "\x59\x0a", [0x3c01] = "\x59\x0b", [0x3c02] = "\x59\x0c",
  [0x3c03] = "\x59\x0d", [0x3c04] = "\x59\x0e", [0x3c05] = "\x59\x0f",
  [0x3c06] = "\x59\x10", [0x3c07] = "\x59\x11", [0x3c08] = "\x59\x12",
  [0x3c09] = "\x59\x13", [0x3c0a] = "\x59\x14", [0x3c0b] = "\x59\x15",
  [0x3c0c] = "\x59\x16", [0x3c0d] = "\x59\x17", [0x3c0e] = "\x59\x18",
  [0x3c0f] = "\x59\x19", [0x3c10] = "\x59\x1a", [0x3c11] = "\x59\x1b",
  [0x3c12] = "\x59\x1c", [0x3c13] = "\x59\x1d", [0x3c14] = "\x59\x1e",
  [0x3c15] = "\x59\x1f", [0x3c16] = "\x59\x20", [0x3c17] = "\x59\x21",
  [0x3c18] = "\x59\x22", [0x3c19] = "\x59\x23", [0x3c1a] = "\x59\x24",
  [0x3c1b] = "\x59\x25", [0x3c1c] = "\x59\x26", [0x3c1d] = "\x59\x27",
  [0x3c1e] = "\x59\x28", [0x3c1f] = "\x59\x29", [0x3c20] = "\x59\x2a",
  [0x3c21] = "\x59\x2b", [0x3c22] = "\x59\x2c", [0x3c23] = "\x59\x2d",
  [0x3c24] = "\x59\x2e", [0x3c25] = "\x59\x2f", [0x3c26] = "\x59\x30",
  [0x3c27] = "\x59\x31", [0x3c28] = "\x59\x32", [0x3c29] = "\x59\x33",
  [0x3c2a] = "\x59\x34", [0x3c2b] = "\x59\x35", [0x3c2c] = "\x59\x36",
  [0x3c2d] = "\x59\x37", [0x3c2e] = "\x59\x38", [0x3c2f] = "\x59\x39",
  [0x3c30] = "\x59\x3a", [0x3c31] = "\x59\x3b", [0x3c32] = "\x59\x3c",
  [0x3c33] = "\x59\x3d", [0x3c34] = "\x59\x3e", [0x3c35] = "\x59\x3f",
  [0x3c36] = "\x59\x40", [0x3c37] = "\x59\x41", [0x3c38] = "\x59\x42",
  [0x3c39] = "\x59\x43", [0x3c3a] = "\x59\x44", [0x3c3b] = "\x59\x45",
  [0x3c3c] = "\x59\x46", [0x3c3d] = "\x59\x47", [0x3c3e] = "\x59\x48",
  [0x3c3f] = "\x59\x49", [0x3c40] = "\x59\x4a", [0x3c41] = "\x59\x4b",
  [0x3c42] = "\x59\x4c", [0x3c43] = "\x59\x4d", [0x3c44] = "\x59\x4e",
  [0x3c45] = "\x59\x4f", [0x3c46] = "\x59\x50", [0x3c47] = "\x59\x51",
  [0x3c48] = "\x59\x52", [0x3c49] = "\x59\x53", [0x3c4a] = "\x59\x54",
  [0x3c4b] = "\x59\x55", [0x3c4c] = "\x59\x56", [0x3c4d] = "\x59\x57",
  [0x3c4e] = "\x59\x58", [0x3c4f] = "\x59\x59", [0x3c50] = "\x59\x5a",
  [0x3c51] = "\x59\x5b", [0x3c52] = "\x59\x5c", [0x3c53] = "\x59\x5d",
  [0x3c54] = "\x59\x5e", [0x3c55] = "\x59\x5f", [0x3c56] = "\x59\x60",
  [0x3c57] = "\x59\x61", [0x3c58] = "\x59\x62", [0x3c59] = "\x59\x63",
  [0x3c5a] = "\x59\x64", [0x3c5b] = "\x59\x65", [0x3c5c] = "\x59\x66",
  [0x3c5d] = "\x59\x67", [0x3c5e] = "\x59\x68", [0x3c5f] = "\x59\x69",
  [0x3c60] = "\xfe\x6a", [0x3fd6] = "\xfe\x6f", [0x3fd7] = "\x5c\xdf",
  [0x3fd8] = "\x5c\xe0", [0x3fd9] = "\x5c\xe1", [0x3fda] = "\x5c\xe2",
  [0x3fdb] = "\x5c\xe3", [0x3fdc] = "\x5c\xe4", [0x3fdd] = "\x5c\xe5",
  [0x3fde] = "\x5c\xe6", [0x3fdf] = "\x5c\xe7", [0x3fe0] = "\x5c\xe8",
  [0x3fe1] = "\x5c\xe9", [0x3fe2] = "\x5c\xea", [0x3fe3] = "\x5c\xeb",
  [0x3fe4] = "\x5c\xec", [0x3fe5] = "\x5c\xed", [0x3fe6] = "\x5c\xee",
  [0x3fe7] = "\x5c\xef", [0x3fe8] = "\x5c\xf0", [0x3fe9] = "\x5c\xf1",
  [0x3fea] = "\x5c\xf2", [0x3feb] = "\x5c\xf3", [0x3fec] = "\x5c\xf4",
  [0x3fed] = "\x5c\xf5", [0x3fee] = "\x5c\xf6", [0x3fef] = "\x5c\xf7",
  [0x3ff0] = "\x5c\xf8", [0x3ff1] = "\x5c\xf9", [0x3ff2] = "\x5c\xfa",
  [0x3ff3] = "\x5c\xfb", [0x3ff4] = "\x5c\xfc", [0x3ff5] = "\x5c\xfd",
  [0x3ff6] = "\x5c\xfe", [0x3ff7] = "\x5c\xff", [0x3ff8] = "\x5d\x00",
  [0x3ff9] = "\x5d\x01", [0x3ffa] = "\x5d\x02", [0x3ffb] = "\x5d\x03",
  [0x3ffc] = "\x5d\x04", [0x3ffd] = "\x5d\x05", [0x3ffe] = "\x5d\x06",
  [0x3fff] = "\x5d\x07", [0x4000] = "\x5d\x08", [0x4001] = "\x5d\x09",
  [0x4002] = "\x5d\x0a", [0x4003] = "\x5d\x0b", [0x4004] = "\x5d\x0c",
  [0x4005] = "\x5d\x0d", [0x4006] = "\x5d\x0e", [0x4007] = "\x5d\x0f",
  [0x4008] = "\x5d\x10", [0x4009] = "\x5d\x11", [0x400a] = "\x5d\x12",
  [0x400b] = "\x5d\x13", [0x400c] = "\x5d\x14", [0x400d] = "\x5d\x15",
  [0x400e] = "\x5d\x16", [0x400f] = "\x5d\x17", [0x4010] = "\x5d\x18",
  [0x4011] = "\x5d\x19", [0x4012] = "\x5d\x1a", [0x4013] = "\x5d\x1b",
  [0x4014] = "\x5d\x1c", [0x4015] = "\x5d\x1d", [0x4016] = "\x5d\x1e",
  [0x4017] = "\x5d\x1f", [0x4018] = "\x5d\x20", [0x4019] = "\x5d\x21",
  [0x401a] = "\x5d\x22", [0x401b] = "\x5d\x23", [0x401c] = "\x5d\x24",
  [0x401d] = "\x5d\x25", [0x401e] = "\x5d\x26", [0x401f] = "\x5d\x27",
  [0x4020] = "\x5d\x28", [0x4021] = "\x5d\x29", [0x4022] = "\x5d\x2a",
  [0x4023] = "\x5d\x2b", [0x4024] = "\x5d\x2c", [0x4025] = "\x5d\x2d",
  [0x4026] = "\x5d\x2e", [0x4027] = "\x5d\x2f", [0x4028] = "\x5d\x30",
  [0x4029] = "\x5d\x31", [0x402a] = "\x5d\x32", [0x402b] = "\x5d\x33",
  [0x402c] = "\x5d\x34", [0x402d] = "\x5d\x35", [0x402e] = "\x5d\x36",
  [0x402f] = "\x5d\x37", [0x4030] = "\x5d\x38", [0x4031] = "\x5d\x39",
  [0x4032] = "\x5d\x3a", [0x4033] = "\x5d\x3b", [0x4034] = "\x5d\x3c",
  [0x4035] = "\x5d\x3d", [0x4036] = "\x5d\x3e", [0x4037] = "\x5d\x3f",
  [0x4038] = "\x5d\x40", [0x4039] = "\x5d\x41", [0x403a] = "\x5d\x42",
  [0x403b] = "\x5d\x43", [0x403c] = "\x5d\x44", [0x403d] = "\x5d\x45",
  [0x403e] = "\x5d\x46", [0x403f] = "\x5d\x47", [0x4040] = "\x5d\x48",
  [0x4041] = "\x5d\x49", [0x4042] = "\x5d\x4a", [0x4043] = "\x5d\x4b",
  [0x4044] = "\x5d\x4c", [0x4045] = "\x5d\x4d", [0x4046] = "\x5d\x4e",
  [0x4047] = "\x5d\x4f", [0x4048] = "\x5d\x50", [0x4049] = "\x5d\x51",
  [0x404a] = "\x5d\x52", [0x404b] = "\x5d\x53", [0x404c] = "\x5d\x54",
  [0x404d] = "\x5d\x55", [0x404e] = "\x5d\x56", [0x404f] = "\x5d\x57",
  [0x4050] = "\x5d\x58", [0x4051] = "\x5d\x59", [0x4052] = "\x5d\x5a",
  [0x4053] = "\x5d\x5b", [0x4054] = "\x5d\x5c", [0x4055] = "\x5d\x5d",
  [0x4056] = "\x5d\x5e", [0x4057] = "\x5d\x5f", [0x4058] = "\x5d\x60",
  [0x4059] = "\x5d\x61", [0x405a] = "\x5d\x62", [0x405b] = "\x5d\x63",
  [0x405c] = "\x5d\x64", [0x405d] = "\x5d\x65", [0x405e] = "\x5d\x66",
  [0x405f] = "\x5d\x67", [0x4060] = "\x5d\x68", [0x4061] = "\x5d\x69",
  [0x4062] = "\x5d\x6a", [0x4063] = "\x5d\x6b", [0x4064] = "\x5d\x6c",
  [0x4065] = "\x5d\x6d", [0x4066] = "\x5d\x6e", [0x4067] = "\x5d\x6f",
  [0x4068] = "\x5d\x70", [0x4069] = "\x5d\x71", [0x406a] = "\x5d\x72",
  [0x406b] = "\x5d\x73", [0x406c] = "\x5d\x74", [0x406d] = "\x5d\x75",
  [0x406e] = "\x5d\x76", [0x406f] = "\x5d\x77", [0x4070] = "\x5d\x78",
  [0x4071] = "\x5d\x79", [0x4072] = "\x5d\x7a", [0x4073] = "\x5d\x7b",
  [0x4074] = "\x5d\x7c", [0x4075] = "\x5d\x7d", [0x4076] = "\x5d\x7e",
  [0x4077] = "\x5d\x7f", [0x4078] = "\x5d\x80", [0x4079] = "\x5d\x81",
  [0x407a] = "\x5d\x82", [0x407b] = "\x5d\x83", [0x407c] = "\x5d\x84",
  [0x407d] = "\x5d\x85", [0x407e] = "\x5d\x86", [0x407f] = "\x5d\x87",
  [0x4080] = "\x5d\x88", [0x4081] = "\x5d\x89", [0x4082] = "\x5d\x8a",
  [0x4083] = "\x5d\x8b", [0x4084] = "\x5d\x8c", [0x4085] = "\x5d\x8d",
  [0x4086] = "\x5d\x8e", [0x4087] = "\x5d\x8f", [0x4088] = "\x5d\x90",
  [0x4089] = "\x5d\x91", [0x408a] = "\x5d\x92", [0x408b] = "\x5d\x93",
  [0x408c] = "\x5d\x94", [0x408d] = "\x5d\x95", [0x408e] = "\x5d\x96",
  [0x408f] = "\x5d\x97", [0x4090] = "\x5d\x98", [0x4091] = "\x5d\x99",
  [0x4092] = "\x5d\x9a", [0x4093] = "\x5d\x9b", [0x4094] = "\x5d\x9c",
  [0x4095] = "\x5d\x9d", [0x4096] = "\x5d\x9e", [0x4097] = "\x5d\x9f",
  [0x4098] = "\x5d\xa0", [0x4099] = "\x5d\xa1", [0x409a] = "\x5d\xa2",
  [0x409b] = "\x5d\xa3", [0x409c] = "\x5d\xa4", [0x409d] = "\x5d\xa5",
  [0x409e] = "\x5d\xa6", [0x409f] = "\x5d\xa7", [0x40a0] = "\x5d\xa8",
  [0x40a1] = "\x5d\xa9", [0x40a2] = "\x5d\xaa", [0x40a3] = "\x5d\xab",
  [0x40a4] = "\x5d\xac", [0x40a5] = "\x5d\xad", [0x40a6] = "\x5d\xae",
  [0x40a7] = "\x5d\xaf", [0x40a8] = "\x5d\xb0", [0x40a9] = "\x5d\xb1",
  [0x40aa] = "\x5d\xb2", [0x40ab] = "\x5d\xb3", [0x40ac] = "\x5d\xb4",
  [0x40ad] = "\x5d\xb5", [0x40ae] = "\x5d\xb6", [0x40af] = "\x5d\xb7",
  [0x40b0] = "\x5d\xb8", [0x40b1] = "\x5d\xb9", [0x40b2] = "\x5d\xba",
  [0x40b3] = "\x5d\xbb", [0x40b4] = "\x5d\xbc", [0x40b5] = "\x5d\xbd",
  [0x40b6] = "\x5d\xbe", [0x40b7] = "\x5d\xbf", [0x40b8] = "\x5d\xc0",
  [0x40b9] = "\x5d\xc1", [0x40ba] = "\x5d\xc2", [0x40bb] = "\x5d\xc3",
  [0x40bc] = "\x5d\xc4", [0x40bd] = "\x5d\xc5", [0x40be] = "\x5d\xc6",
  [0x40bf] = "\x5d\xc7", [0x40c0] = "\x5d\xc8", [0x40c1] = "\x5d\xc9",
  [0x40c2] = "\x5d\xca", [0x40c3] = "\x5d\xcb", [0x40c4] = "\x5d\xcc",
  [0x40c5] = "\x5d\xcd", [0x40c6] = "\x5d\xce", [0x40c7] = "\x5d\xcf",
  [0x40c8] = "\x5d\xd0", [0x40c9] = "\x5d\xd1", [0x40ca] = "\x5d\xd2",
  [0x40cb] = "\x5d\xd3", [0x40cc] = "\x5d\xd4", [0x40cd] = "\x5d\xd5",
  [0x40ce] = "\x5d\xd6", [0x40cf] = "\x5d\xd7", [0x40d0] = "\x5d\xd8",
  [0x40d1] = "\x5d\xd9", [0x40d2] = "\x5d\xda", [0x40d3] = "\x5d\xdb",
  [0x40d4] = "\x5d\xdc", [0x40d5] = "\x5d\xdd", [0x40d6] = "\x5d\xde",
  [0x40d7] = "\x5d\xdf", [0x40d8] = "\x5d\xe0", [0x40d9] = "\x5d\xe1",
  [0x40da] = "\x5d\xe2", [0x40db] = "\x5d\xe3", [0x40dc] = "\x5d\xe4",
  [0x40dd] = "\x5d\xe5", [0x40de] = "\x5d\xe6", [0x40df] = "\xfe\x70",
  [0x42b7] = "\xfe\x72", [0x42b8] = "\x5f\xbe", [0x42b9] = "\x5f\xbf",
  [0x42ba] = "\x5f\xc0", [0x42bb] = "\x5f\xc1", [0x42bc] = "\x5f\xc2",
  [0x42bd] = "\x5f\xc3", [0x42be] = "\x5f\xc4", [0x42bf] = "\x5f\xc5",
  [0x42c0] = "\x5f\xc6", [0x42c1] = "\x5f\xc7", [0x42c2] = "\x5f\xc8",
  [0x42c3] = "\x5f\xc9", [0x42c4] = "\x5f\xca", [0x42c5] = "\x5f\xcb",
  [0x42c6] = "\x5f\xcc", [0x42c7] = "\x5f\xcd", [0x42c8] = "\x5f\xce",
  [0x42c9] = "\x5f\xcf", [0x42ca] = "\x5f\xd0", [0x42cb] = "\x5f\xd1",
  [0x42cc] = "\x5f\xd2", [0x42cd] = "\x5f\xd3", [0x42ce] = "\x5f\xd4",
  [0x42cf] = "\x5f\xd5", [0x42d0] = "\x5f\xd6", [0x42d1] = "\x5f\xd7",
  [0x42d2] = "\x5f\xd8", [0x42d3] = "\x5f\xd9", [0x42d4] = "\x5f\xda",
  [0x42d5] = "\x5f\xdb", [0x42d6] = "\x5f\xdc", [0x42d7] = "\x5f\xdd",
  [0x42d8] = "\x5f\xde", [0x42d9] = "\x5f\xdf", [0x42da] = "\x5f\xe0",
  [0x42db] = "\x5f\xe1", [0x42dc] = "\x5f\xe2", [0x42dd] = "\x5f\xe3",
  [0x42de] = "\x5f\xe4", [0x42df] = "\x5f\xe5", [0x42e0] = "\x5f\xe6",
  [0x42e1] = "\x5f\xe7", [0x42e2] = "\x5f\xe8", [0x42e3] = "\x5f\xe9",
  [0x42e4] = "\x5f\xea", [0x42e5] = "\x5f\xeb", [0x42e6] = "\x5f\xec",
  [0x42e7] = "\x5f\xed", [0x42e8] = "\x5f\xee", [0x42e9] = "\x5f\xef",
  [0x42ea] = "\x5f\xf0", [0x42eb] = "\x5f\xf1", [0x42ec] = "\x5f\xf2",
  [0x42ed] = "\x5f\xf3", [0x42ee] = "\x5f\xf4", [0x42ef] = "\x5f\xf5",
  [0x42f0] = "\x5f\xf6", [0x42f1] = "\x5f\xf7", [0x42f2] = "\x5f\xf8",
  [0x42f3] = "\x5f\xf9", [0x42f4] = "\x5f\xfa", [0x42f5] = "\x5f\xfb",
  [0x42f6] = "\x5f\xfc", [0x42f7] = "\x5f\xfd", [0x42f8] = "\x5f\xfe",
  [0x42f9] = "\x5f\xff", [0x42fa] = "\x60\x00", [0x42fb] = "\x60\x01",
  [0x42fc] = "\x60\x02", [0x42fd] = "\x60\x03", [0x42fe] = "\x60\x04",
  [0x42ff] = "\x60\x05", [0x4300] = "\x60\x06", [0x4301] = "\x60\x07",
  [0x4302] = "\x60\x08", [0x4303] = "\x60\x09", [0x4304] = "\x60\x0a",
  [0x4305] = "\x60\x0b", [0x4306] = "\x60\x0c", [0x4307] = "\x60\x0d",
  [0x4308] = "\x60\x0e", [0x4309] = "\x60\x0f", [0x430a] = "\x60\x10",
  [0x430b] = "\x60\x11", [0x430c] = "\x60\x12", [0x430d] = "\x60\x13",
  [0x430e] = "\x60\x14", [0x430f] = "\x60\x15", [0x4310] = "\x60\x16",
  [0x4311] = "\x60\x17", [0x4312] = "\x60\x18", [0x4313] = "\x60\x19",
  [0x4314] = "\x60\x1a", [0x4315] = "\x60\x1b", [0x4316] = "\x60\x1c",
  [0x4317] = "\x60\x1d", [0x4318] = "\x60\x1e", [0x4319] = "\x60\x1f",
  [0x431a] = "\x60\x20", [0x431b] = "\x60\x21", [0x431c] = "\x60\x22",
  [0x431d] = "\x60\x23", [0x431e] = "\x60\x24", [0x431f] = "\x60\x25",
  [0x4320] = "\x60\x26", [0x4321] = "\x60\x27", [0x4322] = "\x60\x28",
  [0x4323] = "\x60\x29", [0x4324] = "\x60\x2a", [0x4325] = "\x60\x2b",
  [0x4326] = "\x60\x2c", [0x4327] = "\x60\x2d", [0x4328] = "\x60\x2e",
  [0x4329] = "\x60\x2f", [0x432a] = "\x60\x30", [0x432b] = "\x60\x31",
  [0x432c] = "\xfe\x78", [0x432d] = "\x60\x32", [0x432e] = "\x60\x33",
  [0x432f] = "\x60\x34", [0x4330] = "\x60\x35", [0x4331] = "\xfe\x77",
  [0x4332] = "\x60\x36", [0x4333] = "\x60\x37", [0x4334] = "\x60\x38",
  [0x4335] = "\x60\x39", [0x4336] = "\x60\x3a", [0x4337] = "\x60\x3b",
  [0x4338] = "\x60\x3c", [0x4339] = "\x60\x3d", [0x433a] = "\x60\x3e",
  [0x433b] = "\x60\x3f", [0x433c] = "\x60\x40", [0x433d] = "\x60\x41",
  [0x433e] = "\x60\x42", [0x433f] = "\x60\x43", [0x4340] = "\x60\x44",
  [0x4341] = "\x60\x45", [0x4342] = "\x60\x46", [0x4343] = "\x60\x47",
  [0x4344] = "\x60\x48", [0x4345] = "\x60\x49", [0x4346] = "\x60\x4a",
  [0x4347] = "\x60\x4b", [0x4348] = "\x60\x4c", [0x4349] = "\x60\x4d",
  [0x434a] = "\x60\x4e", [0x434b] = "\x60\x4f", [0x434c] = "\x60\x50",
  [0x434d] = "\x60\x51", [0x434e] = "\x60\x52", [0x434f] = "\x60\x53",
  [0x4350] = "\x60\x54", [0x4351] = "\x60\x55", [0x4352] = "\x60\x56",
  [0x4353] = "\x60\x57", [0x4354] = "\x60\x58", [0x4355] = "\x60\x59",
  [0x4356] = "\x60\x5a", [0x4357] = "\x60\x5b", [0x4358] = "\x60\x5c",
  [0x4359] = "\x60\x5d", [0x435a] = "\x60\x5e", [0x435b] = "\x60\x5f",
  [0x435c] = "\x60\x60", [0x435d] = "\xfe\x7a", [0x435e] = "\x60\x61",
  [0x435f] = "\x60\x62", [0x4360] = "\x60\x63", [0x4361] = "\x60\x64",
  [0x4362] = "\x60\x65", [0x4363] = "\x60\x66", [0x4364] = "\x60\x67",
  [0x4365] = "\x60\x68", [0x4366] = "\x60\x69", [0x4367] = "\x60\x6a",
  [0x4368] = "\x60\x6b", [0x4369] = "\x60\x6c", [0x436a] = "\x60\x6d",
  [0x436b] = "\x60\x6e", [0x436c] = "\x60\x6f", [0x436d] = "\x60\x70",
  [0x436e] = "\x60\x71", [0x436f] = "\x60\x72", [0x4370] = "\x60\x73",
  [0x4371] = "\x60\x74", [0x4372] = "\x60\x75", [0x4373] = "\x60\x76",
  [0x4374] = "\x60\x77", [0x4375] = "\x60\x78", [0x4376] = "\x60\x79",
  [0x4377] = "\x60\x7a", [0x4378] = "\x60\x7b", [0x4379] = "\x60\x7c",
  [0x437a] = "\x60\x7d", [0x437b] = "\x60\x7e", [0x437c] = "\x60\x7f",
  [0x437d] = "\x60\x80", [0x437e] = "\x60\x81", [0x437f] = "\x60\x82",
  [0x4380] = "\x60\x83", [0x4381] = "\x60\x84", [0x4382] = "\x60\x85",
  [0x4383] = "\x60\x86", [0x4384] = "\x60\x87", [0x4385] = "\x60\x88",
  [0x4386] = "\x60\x89", [0x4387] = "\x60\x8a", [0x4388] = "\x60\x8b",
  [0x4389] = "\x60\x8c", [0x438a] = "\x60\x8d", [0x438b] = "\x60\x8e",
  [0x438c] = "\x60\x8f", [0x438d] = "\x60\x90", [0x438e] = "\x60\x91",
  [0x438f] = "\x60\x92", [0x4390] = "\x60\x93", [0x4391] = "\x60\x94",
  [0x4392] = "\x60\x95", [0x4393] = "\x60\x96", [0x4394] = "\x60\x97",
  [0x4395] = "\x60\x98", [0x4396] = "\x60\x99", [0x4397] = "\x60\x9a",
  [0x4398] = "\x60\x9b", [0x4399] = "\x60\x9c", [0x439a] = "\x60\x9d",
  [0x439b] = "\x60\x9e", [0x439c] = "\x60\x9f", [0x439d] = "\x60\xa0",
  [0x439e] = "\x60\xa1", [0x439f] = "\x60\xa2", [0x43a0] = "\x60\xa3",
  [0x43a1] = "\x60\xa4", [0x43a2] = "\x60\xa5", [0x43a3] = "\x60\xa6",
  [0x43a4] = "\x60\xa7", [0x43a5] = "\x60\xa8", [0x43a6] = "\x60\xa9",
  [0x43a7] = "\x60\xaa", [0x43a8] = "\x60\xab", [0x43a9] = "\x60\xac",
  [0x43aa] = "\x60\xad", [0x43ab] = "\x60\xae", [0x43ac] = "\x60\xaf",
  [0x43ad] = "\x60\xb0", [0x43ae] = "\x60\xb1", [0x43af] = "\x60\xb2",
  [0x43b0] = "\x60\xb3", [0x43b1] = "\x60\xb4", [0x43b2] = "\x60\xb5",
  [0x43b3] = "\x60\xb6", [0x43b4] = "\x60\xb7", [0x43b5] = "\x60\xb8",
  [0x43b6] = "\x60\xb9", [0x43b7] = "\x60\xba", [0x43b8] = "\x60\xbb",
  [0x43b9] = "\x60\xbc", [0x43ba] = "\x60\xbd", [0x43bb] = "\x60\xbe",
  [0x43bc] = "\x60\xbf", [0x43bd] = "\x60\xc0", [0x43be] = "\x60\xc1",
  [0x43bf] = "\x60\xc2", [0x43c0] = "\x60\xc3", [0x43c1] = "\x60\xc4",
  [0x43c2] = "\x60\xc5", [0x43c3] = "\x60\xc6", [0x43c4] = "\x60\xc7",
  [0x43c5] = "\x60\xc8", [0x43c6] = "\x60\xc9", [0x43c7] = "\x60\xca",
  [0x43c8] = "\x60\xcb", [0x43c9] = "\x60\xcc", [0x43ca] = "\x60\xcd",
  [0x43cb] = "\x60\xce", [0x43cc] = "\x60\xcf", [0x43cd] = "\x60\xd0",
  [0x43ce] = "\x60\xd1", [0x43cf] = "\x60\xd2", [0x43d0] = "\x60\xd3",
  [0x43d1] = "\x60\xd4", [0x43d2] = "\x60\xd5", [0x43d3] = "\x60\xd6",
  [0x43d4] = "\x60\xd7", [0x43d5] = "\x60\xd8", [0x43d6] = "\x60\xd9",
  [0x43d7] = "\x60\xda", [0x43d8] = "\x60\xdb", [0x43d9] = "\x60\xdc",
  [0x43da] = "\x60\xdd", [0x43db] = "\x60\xde", [0x43dc] = "\x60\xdf",
  [0x43dd] = "\x60\xe0", [0x43de] = "\x60\xe1", [0x43df] = "\x60\xe2",
  [0x43e0] = "\x60\xe3", [0x43e1] = "\x60\xe4", [0x43e2] = "\x60\xe5",
  [0x43e3] = "\x60\xe6", [0x43e4] = "\x60\xe7", [0x43e5] = "\x60\xe8",
  [0x43e6] = "\x60\xe9", [0x43e7] = "\x60\xea", [0x43e8] = "\x60\xeb",
  [0x43e9] = "\x60\xec", [0x43ea] = "\x60\xed", [0x43eb] = "\x60\xee",
  [0x43ec] = "\x60\xef", [0x43ed] = "\x60\xf0", [0x43ee] = "\x60\xf1",
  [0x43ef] = "\x60\xf2", [0x43f0] = "\x60\xf3", [0x43f1] = "\x60\xf4",
  [0x43f2] = "\x60\xf5", [0x43f3] = "\x60\xf6", [0x43f4] = "\x60\xf7",
  [0x43f5] = "\x60\xf8", [0x43f6] = "\x60\xf9", [0x43f7] = "\x60\xfa",
  [0x43f8] = "\x60\xfb", [0x43f9] = "\x60\xfc", [0x43fa] = "\x60\xfd",
  [0x43fb] = "\x60\xfe", [0x43fc] = "\x60\xff", [0x43fd] = "\x61\x00",
  [0x43fe] = "\x61\x01", [0x43ff] = "\x61\x02", [0x4400] = "\x61\x03",
  [0x4401] = "\x61\x04", [0x4402] = "\x61\x05", [0x4403] = "\x61\x06",
  [0x4404] = "\x61\x07", [0x4405] = "\x61\x08", [0x4406] = "\x61\x09",
  [0x4407] = "\x61\x0a", [0x4408] = "\x61\x0b", [0x4409] = "\x61\x0c",
  [0x440a] = "\x61\x0d", [0x440b] = "\x61\x0e", [0x440c] = "\x61\x0f",
  [0x440d] = "\x61\x10", [0x440e] = "\x61\x11", [0x440f] = "\x61\x12",
  [0x4410] = "\x61\x13", [0x4411] = "\x61\x14", [0x4412] = "\x61\x15",
  [0x4413] = "\x61\x16", [0x4414] = "\x61\x17", [0x4415] = "\x61\x18",
  [0x4416] = "\x61\x19", [0x4417] = "\x61\x1a", [0x4418] = "\x61\x1b",
  [0x4419] = "\x61\x1c", [0x441a] = "\x61\x1d", [0x441b] = "\x61\x1e",
  [0x441c] = "\x61\x1f", [0x441d] = "\x61\x20", [0x441e] = "\x61\x21",
  [0x441f] = "\x61\x22", [0x4420] = "\x61\x23", [0x4421] = "\x61\x24",
  [0x4422] = "\x61\x25", [0x4423] = "\x61\x26", [0x4424] = "\x61\x27",
  [0x4425] = "\x61\x28", [0x4426] = "\x61\x29", [0x4427] = "\x61\x2a",
  [0x4428] = "\x61\x2b", [0x4429] = "\x61\x2c", [0x442a] = "\x61\x2d",
  [0x442b] = "\x61\x2e", [0x442c] = "\x61\x2f", [0x442d] = "\x61\x30",
  [0x442e] = "\x61\x31", [0x442f] = "\x61\x32", [0x4430] = "\x61\x33",
  [0x4431] = "\x61\x34", [0x4432] = "\x61\x35", [0x4433] = "\x61\x36",
  [0x4434] = "\x61\x37", [0x4435] = "\x61\x38", [0x4436] = "\x61\x39",
  [0x4437] = "\x61\x3a", [0x4438] = "\x61\x3b", [0x4439] = "\x61\x3c",
  [0x443a] = "\x61\x3d", [0x443b] = "\x61\x3e", [0x443c] = "\x61\x3f",
  [0x443d] = "\x61\x40", [0x443e] = "\x61\x41", [0x443f] = "\x61\x42",
  [0x4440] = "\x61\x43", [0x4441] = "\x61\x44", [0x4442] = "\x61\x45",
  [0x4443] = "\x61\x46", [0x4444] = "\x61\x47", [0x4445] = "\x61\x48",
  [0x4446] = "\x61\x49", [0x4447] = "\x61\x4a", [0x4448] = "\x61\x4b",
  [0x4449] = "\x61\x4c", [0x444a] = "\x61\x4d", [0x444b] = "\x61\x4e",
  [0x444c] = "\x61\x4f", [0x444d] = "\x61\x50", [0x444e] = "\x61\x51",
  [0x444f] = "\x61\x52", [0x4450] = "\x61\x53", [0x4451] = "\x61\x54",
  [0x4452] = "\x61\x55", [0x4453] = "\x61\x56", [0x4454] = "\x61\x57",
  [0x4455] = "\x61\x58", [0x4456] = "\xfe\x7b", [0x45cc] = "\xfe\x7d",
  [0x45cd] = "\x62\xce", [0x45ce] = "\x62\xcf", [0x45cf] = "\x62\xd0",
  [0x45d0] = "\x62\xd1", [0x45d1] = "\x62\xd2", [0x45d2] = "\x62\xd3",
  [0x45d3] = "\x62\xd4", [0x45d4] = "\x62\xd5", [0x45d5] = "\x62\xd6",
  [0x45d6] = "\x62\xd7", [0x45d7] = "\x62\xd8", [0x45d8] = "\x62\xd9",
  [0x45d9] = "\x62\xda", [0x45da] = "\x62\xdb", [0x45db] = "\x62\xdc",
  [0x45dc] = "\x62\xdd", [0x45dd] = "\x62\xde", [0x45de] = "\x62\xdf",
  [0x45df] = "\x62\xe0", [0x45e0] = "\x62\xe1", [0x45e1] = "\xfe\x7c",
  [0x45e2] = "\x62\xe2", [0x45e3] = "\x62\xe3", [0x45e4] = "\x62\xe4",
  [0x45e5] = "\x62\xe5", [0x45e6] = "\x62\xe6", [0x45e7] = "\x62\xe7",
  [0x45e8] = "\x62\xe8", [0x45e9] = "\x62\xe9", [0x45ea] = "\x62\xea",
  [0x45eb] = "\x62\xeb", [0x45ec] = "\x62\xec", [0x45ed] = "\x62\xed",
  [0x45ee] = "\x62\xee", [0x45ef] = "\x62\xef", [0x45f0] = "\x62\xf0",
  [0x45f1] = "\x62\xf1", [0x45f2] = "\x62\xf2", [0x45f3] = "\x62\xf3",
  [0x45f4] = "\x62\xf4", [0x45f5] = "\x62\xf5", [0x45f6] = "\x62\xf6",
  [0x45f7] = "\x62\xf7", [0x45f8] = "\x62\xf8", [0x45f9] = "\x62\xf9",
  [0x45fa] = "\x62\xfa", [0x45fb] = "\x62\xfb", [0x45fc] = "\x62\xfc",
  [0x45fd] = "\x62\xfd", [0x45fe] = "\x62\xfe", [0x45ff] = "\x62\xff",
  [0x4600] = "\x63\x00", [0x4601] = "\x63\x01", [0x4602] = "\x63\x02",
  [0x4603] = "\x63\x03", [0x4604] = "\x63\x04", [0x4605] = "\x63\x05",
  [0x4606] = "\x63\x06", [0x4607] = "\x63\x07", [0x4608] = "\x63\x08",
  [0x4609] = "\x63\x09", [0x460a] = "\x63\x0a", [0x460b] = "\x63\x0b",
  [0x460c] = "\x63\x0c", [0x460d] = "\x63\x0d", [0x460e] = "\x63\x0e",
  [0x460f] = "\x63\x0f", [0x4610] = "\x63\x10", [0x4611] = "\x63\x11",
  [0x4612] = "\x63\x12", [0x4613] = "\x63\x13", [0x4614] = "\x63\x14",
  [0x4615] = "\x63\x15", [0x4616] = "\x63\x16", [0x4617] = "\x63\x17",
  [0x4618] = "\x63\x18", [0x4619] = "\x63\x19", [0x461a] = "\x63\x1a",
  [0x461b] = "\x63\x1b", [0x461c] = "\x63\x1c", [0x461d] = "\x63\x1d",
  [0x461e] = "\x63\x1e", [0x461f] = "\x63\x1f", [0x4620] = "\x63\x20",
  [0x4621] = "\x63\x21", [0x4622] = "\x63\x22", [0x4623] = "\x63\x23",
  [0x4624] = "\x63\x24", [0x4625] = "\x63\x25", [0x4626] = "\x63\x26",
  [0x4627] = "\x63\x27", [0x4628] = "\x63\x28", [0x4629] = "\x63\x29",
  [0x462a] = "\x63\x2a", [0x462b] = "\x63\x2b", [0x462c] = "\x63\x2c",
  [0x462d] = "\x63\x2d", [0x462e] = "\x63\x2e", [0x462f] = "\x63\x2f",
  [0x4630] = "\x63\x30", [0x4631] = "\x63\x31", [0x4632] = "\x63\x32",
  [0x4633] = "\x63\x33", [0x4634] = "\x63\x34", [0x4635] = "\x63\x35",
  [0x4636] = "\x63\x36", [0x4637] = "\x63\x37", [0x4638] = "\x63\x38",
  [0x4639] = "\x63\x39", [0x463a] = "\x63\x3a", [0x463b] = "\x63\x3b",
  [0x463c] = "\x63\x3c", [0x463d] = "\x63\x3d", [0x463e] = "\x63\x3e",
  [0x463f] = "\x63\x3f", [0x4640] = "\x63\x40", [0x4641] = "\x63\x41",
  [0x4642] = "\x63\x42", [0x4643] = "\x63\x43", [0x4644] = "\x63\x44",
  [0x4645] = "\x63\x45", [0x4646] = "\x63\x46", [0x4647] = "\x63\x47",
  [0x4648] = "\x63\x48", [0x4649] = "\x63\x49", [0x464a] = "\x63\x4a",
  [0x464b] = "\x63\x4b", [0x464c] = "\x63\x4c", [0x464d] = "\x63\x4d",
  [0x464e] = "\x63\x4e", [0x464f] = "\x63\x4f", [0x4650] = "\x63\x50",
  [0x4651] = "\x63\x51", [0x4652] = "\x63\x52", [0x4653] = "\x63\x53",
  [0x4654] = "\x63\x54", [0x4655] = "\x63\x55", [0x4656] = "\x63\x56",
  [0x4657] = "\x63\x57", [0x4658] = "\x63\x58", [0x4659] = "\x63\x59",
  [0x465a] = "\x63\x5a", [0x465b] = "\x63\x5b", [0x465c] = "\x63\x5c",
  [0x465d] = "\x63\x5d", [0x465e] = "\x63\x5e", [0x465f] = "\x63\x5f",
  [0x4660] = "\x63\x60", [0x4661] = "\x63\x61", [0x4662] = "\x63\x62",
  [0x4663] = "\x63\x63", [0x4664] = "\x63\x64", [0x4665] = "\x63\x65",
  [0x4666] = "\x63\x66", [0x4667] = "\x63\x67", [0x4668] = "\x63\x68",
  [0x4669] = "\x63\x69", [0x466a] = "\x63\x6a", [0x466b] = "\x63\x6b",
  [0x466c] = "\x63\x6c", [0x466d] = "\x63\x6d", [0x466e] = "\x63\x6e",
  [0x466f] = "\x63\x6f", [0x4670] = "\x63\x70", [0x4671] = "\x63\x71",
  [0x4672] = "\x63\x72", [0x4673] = "\x63\x73", [0x4674] = "\x63\x74",
  [0x4675] = "\x63\x75", [0x4676] = "\x63\x76", [0x4677] = "\x63\x77",
  [0x4678] = "\x63\x78", [0x4679] = "\x63\x79", [0x467a] = "\x63\x7a",
  [0x467b] = "\x63\x7b", [0x467c] = "\x63\x7c", [0x467d] = "\x63\x7d",
  [0x467e] = "\x63\x7e", [0x467f] = "\x63\x7f", [0x4680] = "\x63\x80",
  [0x4681] = "\x63\x81", [0x4682] = "\x63\x82", [0x4683] = "\x63\x83",
  [0x4684] = "\x63\x84", [0x4685] = "\x63\x85", [0x4686] = "\x63\x86",
  [0x4687] = "\x63\x87", [0x4688] = "\x63\x88", [0x4689] = "\x63\x89",
  [0x468a] = "\x63\x8a", [0x468b] = "\x63\x8b", [0x468c] = "\x63\x8c",
  [0x468d] = "\x63\x8d", [0x468e] = "\x63\x8e", [0x468f] = "\x63\x8f",
  [0x4690] = "\x63\x90", [0x4691] = "\x63\x91", [0x4692] = "\x63\x92",
  [0x4693] = "\x63\x93", [0x4694] = "\x63\x94", [0x4695] = "\x63\x95",
  [0x4696] = "\x63\x96", [0x4697] = "\x63\x97", [0x4698] = "\x63\x98",
  [0x4699] = "\x63\x99", [0x469a] = "\x63\x9a", [0x469b] = "\x63\x9b",
  [0x469c] = "\x63\x9c", [0x469d] = "\x63\x9d", [0x469e] = "\x63\x9e",
  [0x469f] = "\x63\x9f", [0x46a0] = "\x63\xa0", [0x46a1] = "\x63\xa1",
  [0x46a2] = "\x63\xa2", [0x46a3] = "\xfe\x80", [0x46a4] = "\x63\xa3",
  [0x46a5] = "\x63\xa4", [0x46a6] = "\x63\xa5", [0x46a7] = "\x63\xa6",
  [0x46a8] = "\x63\xa7", [0x46a9] = "\xfe\x81", [0x46aa] = "\x63\xa8",
  [0x46ab] = "\x63\xa9", [0x46ac] = "\x63\xaa", [0x46ad] = "\x63\xab",
  [0x46ae] = "\x63\xac", [0x46af] = "\x63\xad", [0x46b0] = "\x63\xae",
  [0x46b1] = "\x63\xaf", [0x46b2] = "\x63\xb0", [0x46b3] = "\x63\xb1",
  [0x46b4] = "\x63\xb2", [0x46b5] = "\x63\xb3", [0x46b6] = "\x63\xb4",
  [0x46b7] = "\x63\xb5", [0x46b8] = "\x63\xb6", [0x46b9] = "\x63\xb7",
  [0x46ba] = "\x63\xb8", [0x46bb] = "\x63\xb9", [0x46bc] = "\x63\xba",
  [0x46bd] = "\x63\xbb", [0x46be] = "\x63\xbc", [0x46bf] = "\x63\xbd",
  [0x46c0] = "\x63\xbe", [0x46c1] = "\x63\xbf", [0x46c2] = "\x63\xc0",
  [0x46c3] = "\x63\xc1", [0x46c4] = "\x63\xc2", [0x46c5] = "\x63\xc3",
  [0x46c6] = "\x63\xc4", [0x46c7] = "\x63\xc5", [0x46c8] = "\x63\xc6",
  [0x46c9] = "\x63\xc7", [0x46ca] = "\x63\xc8", [0x46cb] = "\x63\xc9",
  [0x46cc] = "\x63\xca", [0x46cd] = "\x63\xcb", [0x46ce] = "\x63\xcc",
  [0x46cf] = "\x63\xcd", [0x46d0] = "\x63\xce", [0x46d1] = "\x63\xcf",
  [0x46d2] = "\x63\xd0", [0x46d3] = "\x63\xd1", [0x46d4] = "\x63\xd2",
  [0x46d5] = "\x63\xd3", [0x46d6] = "\x63\xd4", [0x46d7] = "\x63\xd5",
  [0x46d8] = "\x63\xd6", [0x46d9] = "\x63\xd7", [0x46da] = "\x63\xd8",
  [0x46db] = "\x63\xd9", [0x46dc] = "\x63\xda", [0x46dd] = "\x63\xdb",
  [0x46de] = "\x63\xdc", [0x46df] = "\x63\xdd", [0x46e0] = "\x63\xde",
  [0x46e1] = "\x63\xdf", [0x46e2] = "\x63\xe0", [0x46e3] = "\x63\xe1",
  [0x46e4] = "\x63\xe2", [0x46e5] = "\x63\xe3", [0x46e6] = "\x63\xe4",
  [0x46e7] = "\x63\xe5", [0x46e8] = "\x63\xe6", [0x46e9] = "\x63\xe7",
  [0x46ea] = "\x63\xe8", [0x46eb] = "\x63\xe9", [0x46ec] = "\x63\xea",
  [0x46ed] = "\x63\xeb", [0x46ee] = "\x63\xec", [0x46ef] = "\x63\xed",
  [0x46f0] = "\x63\xee", [0x46f1] = "\x63\xef", [0x46f2] = "\x63\xf0",
  [0x46f3] = "\x63\xf1", [0x46f4] = "\x63\xf2", [0x46f5] = "\x63\xf3",
  [0x46f6] = "\x63\xf4", [0x46f7] = "\x63\xf5", [0x46f8] = "\x63\xf6",
  [0x46f9] = "\x63\xf7", [0x46fa] = "\x63\xf8", [0x46fb] = "\x63\xf9",
  [0x46fc] = "\xfe\x82", [0x46fd] = "\x63\xfa", [0x46fe] = "\x63\xfb",
  [0x46ff] = "\x63\xfc", [0x4700] = "\x63\xfd", [0x4701] = "\x63\xfe",
  [0x4702] = "\x63\xff", [0x4703] = "\x64\x00", [0x4704] = "\x64\x01",
  [0x4705] = "\x64\x02", [0x4706] = "\x64\x03", [0x4707] = "\x64\x04",
  [0x4708] = "\x64\x05", [0x4709] = "\x64\x06", [0x470a] = "\x64\x07",
  [0x470b] = "\x64\x08", [0x470c] = "\x64\x09", [0x470d] = "\xfe\x83",
  [0x48c7] = "\xfe\x85", [0x48c8] = "\x65\xc3", [0x48c9] = "\x65\xc4",
  [0x48ca] = "\x65\xc5", [0x48cb] = "\x65\xc6", [0x48cc] = "\x65\xc7",
  [0x48cd] = "\x65\xc8", [0x48ce] = "\x65\xc9", [0x48cf] = "\x65\xca",
  [0x48d0] = "\x65\xcb", [0x48d1] = "\x65\xcc", [0x48d2] = "\x65\xcd",
  [0x48d3] = "\x65\xce", [0x48d4] = "\x65\xcf", [0x48d5] = "\x65\xd0",
  [0x48d6] = "\x65\xd1", [0x48d7] = "\x65\xd2", [0x48d8] = "\x65\xd3",
  [0x48d9] = "\x65\xd4", [0x48da] = "\x65\xd5", [0x48db] = "\x65\xd6",
  [0x48dc] = "\x65\xd7", [0x48dd] = "\x65\xd8", [0x48de] = "\x65\xd9",
  [0x48df] = "\x65\xda", [0x48e0] = "\x65\xdb", [0x48e1] = "\x65\xdc",
  [0x48e2] = "\x65\xdd", [0x48e3] = "\x65\xde", [0x48e4] = "\x65\xdf",
  [0x48e5] = "\x65\xe0", [0x48e6] = "\x65\xe1", [0x48e7] = "\x65\xe2",
  [0x48e8] = "\x65\xe3", [0x48e9] = "\x65\xe4", [0x48ea] = "\x65\xe5",
  [0x48eb] = "\x65\xe6", [0x48ec] = "\x65\xe7", [0x48ed] = "\x65\xe8",
  [0x48ee] = "\x65\xe9", [0x48ef] = "\x65\xea", [0x48f0] = "\x65\xeb",
  [0x48f1] = "\x65\xec", [0x48f2] = "\x65\xed", [0x48f3] = "\x65\xee",
  [0x48f4] = "\x65\xef", [0x48f5] = "\x65\xf0", [0x48f6] = "\x65\xf1",
  [0x48f7] = "\x65\xf2", [0x48f8] = "\x65\xf3", [0x48f9] = "\x65\xf4",
  [0x48fa] = "\xfe\x86", [0x48fb] = "\x65\xf5", [0x48fc] = "\x65\xf6",
  [0x48fd] = "\xfe\x87", [0x48fe] = "\x65\xf7", [0x48ff] = "\x65\xf8",
  [0x4900] = "\x65\xf9", [0x4901] = "\x65\xfa", [0x4902] = "\xfe\x88",
  [0x4903] = "\xfe\x89", [0x4904] = "\x65\xfb", [0x4905] = "\xfe\x8a",
  [0x4906] = "\xfe\x8b", [0x4907] = "\x65\xfc", [0x4908] = "\x65\xfd",
  [0x4909] = "\x65\xfe", [0x490a] = "\x65\xff", [0x490b] = "\x66\x00",
  [0x490c] = "\x66\x01", [0x490d] = "\x66\x02", [0x490e] = "\x66\x03",
  [0x490f] = "\x66\x04", [0x4910] = "\x66\x05", [0x4911] = "\x66\x06",
  [0x4912] = "\x66\x07", [0x4913] = "\x66\x08", [0x4914] = "\x66\x09",
  [0x4915] = "\x66\x0a", [0x4916] = "\x66\x0b", [0x4917] = "\x66\x0c",
  [0x4918] = "\x66\x0d", [0x4919] = "\x66\x0e", [0x491a] = "\x66\x0f",
  [0x491b] = "\xfe\x8d", [0x491c] = "\x66\x10", [0x491d] = "\x66\x11",
  [0x491e] = "\x66\x12", [0x491f] = "\xfe\x8c", [0x4920] = "\x66\x13",
  [0x4921] = "\x66\x14", [0x4922] = "\x66\x15", [0x4923] = "\x66\x16",
  [0x4924] = "\x66\x17", [0x4925] = "\x66\x18", [0x4926] = "\x66\x19",
  [0x4927] = "\x66\x1a", [0x4928] = "\x66\x1b", [0x4929] = "\x66\x1c",
  [0x492a] = "\x66\x1d", [0x492b] = "\x66\x1e", [0x492c] = "\x66\x1f",
  [0x492d] = "\x66\x20", [0x492e] = "\x66\x21", [0x492f] = "\x66\x22",
  [0x4930] = "\x66\x23", [0x4931] = "\x66\x24", [0x4932] = "\x66\x25",
  [0x4933] = "\x66\x26", [0x4934] = "\x66\x27", [0x4935] = "\x66\x28",
  [0x4936] = "\xfe\x8f", [0x4937] = "\xfe\x8e", [0x4bf7] = "\xfe\x96",
  [0x4bf8] = "\x68\xe8", [0x4bf9] = "\x68\xe9", [0x4bfa] = "\x68\xea",
  [0x4bfb] = "\x68\xeb", [0x4bfc] = "\x68\xec", [0x4bfd] = "\x68\xed",
  [0x4bfe] = "\x68\xee", [0x4bff] = "\x68\xef", [0x4c00] = "\x68\xf0",
  [0x4c01] = "\x68\xf1", [0x4c02] = "\x68\xf2", [0x4c03] = "\x68\xf3",
  [0x4c04] = "\x68\xf4", [0x4c05] = "\x68\xf5", [0x4c06] = "\x68\xf6",
  [0x4c07] = "\x68\xf7", [0x4c08] = "\x68\xf8", [0x4c09] = "\x68\xf9",
  [0x4c0a] = "\x68\xfa", [0x4c0b] = "\x68\xfb", [0x4c0c] = "\x68\xfc",
  [0x4c0d] = "\x68\xfd", [0x4c0e] = "\x68\xfe", [0x4c0f] = "\x68\xff",
  [0x4c10] = "\x69\x00", [0x4c11] = "\x69\x01", [0x4c12] = "\x69\x02",
  [0x4c13] = "\x69\x03", [0x4c14] = "\x69\x04", [0x4c15] = "\x69\x05",
  [0x4c16] = "\x69\x06", [0x4c17] = "\x69\x07", [0x4c18] = "\x69\x08",
  [0x4c19] = "\x69\x09", [0x4c1a] = "\x69\x0a", [0x4c1b] = "\x69\x0b",
  [0x4c1c] = "\x69\x0c", [0x4c1d] = "\x69\x0d", [0x4c1e] = "\x69\x0e",
  [0x4c1f] = "\xfe\x93", [0x4c20] = "\xfe\x94", [0x4c21] = "\xfe\x95",
  [0x4c22] = "\xfe\x97", [0x4c23] = "\xfe\x92", [0x4c24] = "\x69\x0f",
  [0x4c25] = "\x69\x10", [0x4c26] = "\x69\x11", [0x4c27] = "\x69\x12",
  [0x4c28] = "\x69\x13", [0x4c29] = "\x69\x14", [0x4c2a] = "\x69\x15",
  [0x4c2b] = "\x69\x16", [0x4c2c] = "\x69\x17", [0x4c2d] = "\x69\x18",
  [0x4c2e] = "\x69\x19", [0x4c2f] = "\x69\x1a", [0x4c30] = "\x69\x1b",
  [0x4c31] = "\x69\x1c", [0x4c32] = "\x69\x1d", [0x4c33] = "\x69\x1e",
  [0x4c34] = "\x69\x1f", [0x4c35] = "\x69\x20", [0x4c36] = "\x69\x21",
  [0x4c37] = "\x69\x22", [0x4c38] = "\x69\x23", [0x4c39] = "\x69\x24",
  [0x4c3a] = "\x69\x25", [0x4c3b] = "\x69\x26", [0x4c3c] = "\x69\x27",
  [0x4c3d] = "\x69\x28", [0x4c3e] = "\x69\x29", [0x4c3f] = "\x69\x2a",
  [0x4c40] = "\x69\x2b", [0x4c41] = "\x69\x2c", [0x4c42] = "\x69\x2d",
  [0x4c43] = "\x69\x2e", [0x4c44] = "\x69\x2f", [0x4c45] = "\x69\x30",
  [0x4c46] = "\x69\x31", [0x4c47] = "\x69\x32", [0x4c48] = "\x69\x33",
  [0x4c49] = "\x69\x34", [0x4c4a] = "\x69\x35", [0x4c4b] = "\x69\x36",
  [0x4c4c] = "\x69\x37", [0x4c4d] = "\x69\x38", [0x4c4e] = "\x69\x39",
  [0x4c4f] = "\x69\x3a", [0x4c50] = "\x69\x3b", [0x4c51] = "\x69\x3c",
  [0x4c52] = "\x69\x3d", [0x4c53] = "\x69\x3e", [0x4c54] = "\x69\x3f",
  [0x4c55] = "\x69\x40", [0x4c56] = "\x69\x41", [0x4c57] = "\x69\x42",
  [0x4c58] = "\x69\x43", [0x4c59] = "\x69\x44", [0x4c5a] = "\x69\x45",
  [0x4c5b] = "\x69\x46", [0x4c5c] = "\x69\x47", [0x4c5d] = "\x69\x48",
  [0x4c5e] = "\x69\x49", [0x4c5f] = "\x69\x4a", [0x4c60] = "\x69\x4b",
  [0x4c61] = "\x69\x4c", [0x4c62] = "\x69\x4d", [0x4c63] = "\x69\x4e",
  [0x4c64] = "\x69\x4f", [0x4c65] = "\x69\x50", [0x4c66] = "\x69\x51",
  [0x4c67] = "\x69\x52", [0x4c68] = "\x69\x53", [0x4c69] = "\x69\x54",
  [0x4c6a] = "\x69\x55", [0x4c6b] = "\x69\x56", [0x4c6c] = "\x69\x57",
  [0x4c6d] = "\x69\x58", [0x4c6e] = "\x69\x59", [0x4c6f] = "\x69\x5a",
  [0x4c70] = "\x69\x5b", [0x4c71] = "\x69\x5c", [0x4c72] = "\x69\x5d",
  [0x4c73] = "\x69\x5e", [0x4c74] = "\x69\x5f", [0x4c75] = "\x69\x60",
  [0x4c76] = "\x69\x61", [0x4c77] = "\x69\x62", [0x4c78] = "\x69\x63",
  [0x4c79] = "\x69\x64", [0x4c7a] = "\x69\x65", [0x4c7b] = "\x69\x66",
  [0x4c7c] = "\x69\x67", [0x4c7d] = "\x69\x68", [0x4c7e] = "\x69\x69",
  [0x4c7f] = "\x69\x6a", [0x4c80] = "\x69\x6b", [0x4c81] = "\x69\x6c",
  [0x4c82] = "\x69\x6d", [0x4c83] = "\x69\x6e", [0x4c84] = "\x69\x6f",
  [0x4c85] = "\x69\x70", [0x4c86] = "\x69\x71", [0x4c87] = "\x69\x72",
  [0x4c88] = "\x69\x73", [0x4c89] = "\x69\x74", [0x4c8a] = "\x69\x75",
  [0x4c8b] = "\x69\x76", [0x4c8c] = "\x69\x77", [0x4c8d] = "\x69\x78",
  [0x4c8e] = "\x69\x79", [0x4c8f] = "\x69\x7a", [0x4c90] = "\x69\x7b",
  [0x4c91] = "\x69\x7c", [0x4c92] = "\x69\x7d", [0x4c93] = "\xfe\x98",
  [0x4c94] = "\xfe\x99", [0x4c95] = "\xfe\x9a", [0x4c96] = "\xfe\x9b",
  [0x4c97] = "\xfe\x9c", [0x4c98] = "\xfe\x9d", [0x4c99] = "\xfe\x9e",
  [0x4c9a] = "\x69\x7e", [0x4c9b] = "\x69\x7f", [0x4c9c] = "\x69\x80",
  [0x4c9d] = "\x69\x81", [0x4c9e] = "\x69\x82", [0x4c9f] = "\x69\x83",
  [0x4ca0] = "\x69\x84", [0x4ca1] = "\x69\x85", [0x4ca2] = "\x69\x86",
  [0x4ca3] = "\x69\x87", [0x4ca4] = "\x69\x88", [0x4ca5] = "\x69\x89",
  [0x4ca6] = "\x69\x8a", [0x4ca7] = "\x69\x8b", [0x4ca8] = "\x69\x8c",
  [0x4ca9] = "\x69\x8d", [0x4caa] = "\x69\x8e", [0x4cab] = "\x69\x8f",
  [0x4cac] = "\x69\x90", [0x4cad] = "\x69\x91", [0x4cae] = "\x69\x92",
  [0x4caf] = "\x69\x93", [0x4cb0] = "\x69\x94", [0x4cb1] = "\x69\x95",
  [0x4cb2] = "\x69\x96", [0x4cb3] = "\x69\x97", [0x4cb4] = "\x69\x98",
  [0x4cb5] = "\x69\x99", [0x4cb6] = "\x69\x9a", [0x4cb7] = "\x69\x9b",
  [0x4cb8] = "\x69\x9c", [0x4cb9] = "\x69\x9d", [0x4cba] = "\x69\x9e",
  [0x4cbb] = "\x69\x9f", [0x4cbc] = "\x69\xa0", [0x4cbd] = "\x69\xa1",
  [0x4cbe] = "\x69\xa2", [0x4cbf] = "\x69\xa3", [0x4cc0] = "\x69\xa4",
  [0x4cc1] = "\x69\xa5", [0x4cc2] = "\x69\xa6", [0x4cc3] = "\x69\xa7",
  [0x4cc4] = "\x69\xa8", [0x4cc5] = "\x69\xa9", [0x4cc6] = "\x69\xaa",
  [0x4cc7] = "\x69\xab", [0x4cc8] = "\x69\xac", [0x4cc9] = "\x69\xad",
  [0x4cca] = "\x69\xae", [0x4ccb] = "\x69\xaf", [0x4ccc] = "\x69\xb0",
  [0x4ccd] = "\x69\xb1", [0x4cce] = "\x69\xb2", [0x4ccf] = "\x69\xb3",
  [0x4cd0] = "\x69\xb4", [0x4cd1] = "\x69\xb5", [0x4cd2] = "\x69\xb6",
  [0x4cd3] = "\x69\xb7", [0x4cd4] = "\x69\xb8", [0x4cd5] = "\x69\xb9",
  [0x4cd6] = "\x69\xba", [0x4cd7] = "\x69\xbb", [0x4cd8] = "\x69\xbc",
  [0x4cd9] = "\x69\xbd", [0x4cda] = "\x69\xbe", [0x4cdb] = "\x69\xbf",
  [0x4cdc] = "\x69\xc0", [0x4cdd] = "\x69\xc1", [0x4cde] = "\x69\xc2",
  [0x4cdf] = "\x69\xc3", [0x4ce0] = "\x69\xc4", [0x4ce1] = "\x69\xc5",
  [0x4ce2] = "\x69\xc6", [0x4ce3] = "\x69\xc7", [0x4ce4] = "\x69\xc8",
  [0x4ce5] = "\x69\xc9", [0x4ce6] = "\x69\xca", [0x4ce7] = "\x69\xcb",
  [0x4ce8] = "\x69\xcc", [0x4ce9] = "\x69\xcd", [0x4cea] = "\x69\xce",
  [0x4ceb] = "\x69\xcf", [0x4cec] = "\x69\xd0", [0x4ced] = "\x69\xd1",
  [0x4cee] = "\x69\xd2", [0x4cef] = "\x69\xd3", [0x4cf0] = "\x69\xd4",
  [0x4cf1] = "\x69\xd5", [0x4cf2] = "\x69\xd6", [0x4cf3] = "\x69\xd7",
  [0x4cf4] = "\x69\xd8", [0x4cf5] = "\x69\xd9", [0x4cf6] = "\x69\xda",
  [0x4cf7] = "\x69\xdb", [0x4cf8] = "\x69\xdc", [0x4cf9] = "\x69\xdd",
  [0x4cfa] = "\x69\xde", [0x4cfb] = "\x69\xdf", [0x4cfc] = "\x69\xe0",
  [0x4cfd] = "\x69\xe1", [0x4cfe] = "\x69\xe2", [0x4cff] = "\x69\xe3",
  [0x4d00] = "\x69\xe4", [0x4d01] = "\x69\xe5", [0x4d02] = "\x69\xe6",
  [0x4d03] = "\x69\xe7", [0x4d04] = "\x69\xe8", [0x4d05] = "\x69\xe9",
  [0x4d06] = "\x69\xea", [0x4d07] = "\x69\xeb", [0x4d08] = "\x69\xec",
  [0x4d09] = "\x69\xed", [0x4d0a] = "\x69\xee", [0x4d0b] = "\x69\xef",
  [0x4d0c] = "\x69\xf0", [0x4d0d] = "\x69\xf1", [0x4d0e] = "\x69\xf2",
  [0x4d0f] = "\x69\xf3", [0x4d10] = "\x69\xf4", [0x4d11] = "\x69\xf5",
  [0x4d12] = "\x69\xf6", [0x4d13] = "\x69\xf7", [0x4d14] = "\x69\xf8",
  [0x4d15] = "\x69\xf9", [0x4d16] = "\x69\xfa", [0x4d17] = "\x69\xfb",
  [0x4d18] = "\x69\xfc", [0x4d19] = "\x69\xfd", [0x4d1a] = "\x69\xfe",
  [0x4d1b] = "\x69\xff", [0x4d1c] = "\x6a\x00", [0x4d1d] = "\x6a\x01",
  [0x4d1e] = "\x6a\x02", [0x4d1f] = "\x6a\x03", [0x4d20] = "\x6a\x04",
  [0x4d21] = "\x6a\x05", [0x4d22] = "\x6a\x06", [0x4d23] = "\x6a\x07",
  [0x4d24] = "\x6a\x08", [0x4d25] = "\x6a\x09", [0x4d26] = "\x6a\x0a",
  [0x4d27] = "\x6a\x0b", [0x4d28] = "\x6a\x0c", [0x4d29] = "\x6a\x0d",
  [0x4d2a] = "\x6a\x0e", [0x4d2b] = "\x6a\x0f", [0x4d2c] = "\x6a\x10",
  [0x4d2d] = "\x6a\x11", [0x4d2e] = "\xfe\x9f", [0x4d2f] = "\x6a\x12",
  [0x4d30] = "\x6a\x13", [0x4d31] = "\x6a\x14", [0x4d32] = "\x6a\x15",
  [0x4d33] = "\x6a\x16", [0x4d34] = "\x6a\x17", [0x4d35] = "\x6a\x18",
  [0x4d40] = "\x6a\x23", [0x4d41] = "\x6a\x24", [0x4d42] = "\x6a\x25",
  [0x4d43] = "\x6a\x26", [0x4d44] = "\x6a\x27", [0x4d45] = "\x6a\x28",
  [0x4d46] = "\x6a\x29", [0x4d47] = "\x6a\x2a", [0x4d48] = "\x6a\x2b",
  [0x4d49] = "\x6a\x2c", [0x4d4a] = "\x6a\x2d", [0x4d4b] = "\x6a\x2e",
  [0x4d4c] = "\x6a\x2f", [0x4d4d] = "\x6a\x30", [0x4d4e] = "\x6a\x31",
  [0x4d4f] = "\x6a\x32", [0x4d50] = "\x6a\x33", [0x4d51] = "\x6a\x34",
  [0x4d52] = "\x6a\x35", [0x4d53] = "\x6a\x36", [0x4d54] = "\x6a\x37",
  [0x4d55] = "\x6a\x38", [0x4d56] = "\x6a\x39", [0x4d57] = "\x6a\x3a",
  [0x4d58] = "\x6a\x3b", [0x4d59] = "\x6a\x3c", [0x4d5a] = "\x6a\x3d",
  [0x4d5b] = "\x6a\x3e", [0x4d5c] = "\x6a\x3f", [0x4d5d] = "\x6a\x40",
  [0x4d5e] = "\x6a\x41", [0x4d5f] = "\x6a\x42", [0x4d60] = "\x6a\x43",
  [0x4d61] = "\x6a\x44", [0x4d62] = "\x6a\x45", [0x4d63] = "\x6a\x46",
  [0x4d64] = "\x6a\x47", [0x4d65] = "\x6a\x48", [0x4d66] = "\x6a\x49",
  [0x4d67] = "\x6a\x4a", [0x4d68] = "\x6a\x4b", [0x4d69] = "\x6a\x4c",
  [0x4d6a] = "\x6a\x4d", [0x4d6b] = "\x6a\x4e", [0x4d6c] = "\x6a\x4f",
  [0x4d6d] = "\x6a\x50", [0x4d6e] = "\x6a\x51", [0x4d6f] = "\x6a\x52",
  [0x4d70] = "\x6a\x53", [0x4d71] = "\x6a\x54", [0x4d72] = "\x6a\x55",
  [0x4d73] = "\x6a\x56", [0x4d74] = "\x6a\x57", [0x4d75] = "\x6a\x58",
  [0x4d76] = "\x6a\x59", [0x4d77] = "\x6a\x5a", [0x4d78] = "\x6a\x5b",
  [0x4d79] = "\x6a\x5c", [0x4d7a] = "\x6a\x5d", [0x4d7b] = "\x6a\x5e",
  [0x4d7c] = "\x6a\x5f", [0x4d7d] = "\x6a\x60", [0x4d7e] = "\x6a\x61",
  [0x4d7f] = "\x6a\x62", [0x4d80] = "\xd2\xbb", [0x4d81] = "\xb6\xa1",
  [0x4d82] = "\x81\x40", [0x4d83] = "\xc6\xdf", [0x4d84] = "\x81\x41",
  [0x4d85] = "\x81\x42", [0x4d86] = "\x81\x43", [0x4d87] = "\xcd\xf2",
  [0x4d88] = "\xd5\xc9", [0x4d89] = "\xc8\xfd", [0x4d8a] = "\xc9\xcf",
  [0x4d8b] = "\xcf\xc2", [0x4d8c] = "\xd8\xa2", [0x4d8d] = "\xb2\xbb",
  [0x4d8e] = "\xd3\xeb", [0x4d8f] = "\x81\x44", [0x4d90] = "\xd8\xa4",
  [0x4d91] = "\xb3\xf3", [0x4d92] = "\x81\x45", [0x4d93] = "\xd7\xa8",
  [0x4d94] = "\xc7\xd2", [0x4d95] = "\xd8\xa7", [0x4d96] = "\xca\xc0",
  [0x4d97] = "\x81\x46", [0x4d98] = "\xc7\xf0", [0x4d99] = "\xb1\xfb",
  [0x4d9a] = "\xd2\xb5", [0x4d9b] = "\xb4\xd4", [0x4d9c] = "\xb6\xab",
  [0x4d9d] = "\xcb\xbf", [0x4d9e] = "\xd8\xa9", [0x4d9f] = "\x81\x47",
  [0x4da0] = "\x81\x48", [0x4da1] = "\x81\x49", [0x4da2] = "\xb6\xaa",
  [0x4da3] = "\x81\x4a", [0x4da4] = "\xc1\xbd", [0x4da5] = "\xd1\xcf",
  [0x4da6] = "\x81\x4b", [0x4da7] = "\xc9\xa5", [0x4da8] = "\xd8\xad",
  [0x4da9] = "\x81\x4c", [0x4daa] = "\xb8\xf6", [0x4dab] = "\xd1\xbe",
  [0x4dac] = "\xe3\xdc", [0x4dad] = "\xd6\xd0", [0x4dae] = "\x81\x4d",
  [0x4daf] = "\x81\x4e", [0x4db0] = "\xb7\xe1", [0x4db1] = "\x81\x4f",
  [0x4db2] = "\xb4\xae", [0x4db3] = "\x81\x50", [0x4db4] = "\xc1\xd9",
  [0x4db5] = "\x81\x51", [0x4db6] = "\xd8\xbc", [0x4db7] = "\x81\x52",
  [0x4db8] = "\xcd\xe8", [0x4db9] = "\xb5\xa4", [0x4dba] = "\xce\xaa",
  [0x4dbb] = "\xd6\xf7", [0x4dbc] = "\x81\x53", [0x4dbd] = "\xc0\xf6",
  [0x4dbe] = "\xbe\xd9", [0x4dbf] = "\xd8\xaf", [0x4dc0] = "\x81\x54",
  [0x4dc1] = "\x81\x55", [0x4dc2] = "\x81\x56", [0x4dc3] = "\xc4\xcb",
  [0x4dc4] = "\x81\x57", [0x4dc5] = "\xbe\xc3", [0x4dc6] = "\x81\x58",
  [0x4dc7] = "\xd8\xb1", [0x4dc8] = "\xc3\xb4", [0x4dc9] = "\xd2\xe5",
  [0x4dca] = "\x81\x59", [0x4dcb] = "\xd6\xae", [0x4dcc] = "\xce\xda",
  [0x4dcd] = "\xd5\xa7", [0x4dce] = "\xba\xf5", [0x4dcf] = "\xb7\xa6",
  [0x4dd0] = "\xc0\xd6", [0x4dd1] = "\x81\x5a", [0x4dd2] = "\xc6\xb9",
  [0x4dd3] = "\xc5\xd2", [0x4dd4] = "\xc7\xc7", [0x4dd5] = "\x81\x5b",
  [0x4dd6] = "\xb9\xd4", [0x4dd7] = "\x81\x5c", [0x4dd8] = "\xb3\xcb",
  [0x4dd9] = "\xd2\xd2", [0x4dda] = "\x81\x5d", [0x4ddb] = "\x81\x5e",
  [0x4ddc] = "\xd8\xbf", [0x4ddd] = "\xbe\xc5", [0x4dde] = "\xc6\xf2",
  [0x4ddf] = "\xd2\xb2", [0x4de0] = "\xcf\xb0", [0x4de1] = "\xcf\xe7",
  [0x4de2] = "\x81\x5f", [0x4de3] = "\x81\x60", [0x4de4] = "\x81\x61",
  [0x4de5] = "\x81\x62", [0x4de6] = "\xca\xe9", [0x4de7] = "\x81\x63",
  [0x4de8] = "\x81\x64", [0x4de9] = "\xd8\xc0", [0x4dea] = "\x81\x65",
  [0x4deb] = "\x81\x66", [0x4dec] = "\x81\x67", [0x4ded] = "\x81\x68",
  [0x4dee] = "\x81\x69", [0x4def] = "\x81\x6a", [0x4df0] = "\xc2\xf2",
  [0x4df1] = "\xc2\xd2", [0x4df2] = "\x81\x6b", [0x4df3] = "\xc8\xe9",
  [0x4df4] = "\x81\x6c", [0x4df5] = "\x81\x6d", [0x4df6] = "\x81\x6e",
  [0x4df7] = "\x81\x6f", [0x4df8] = "\x81\x70", [0x4df9] = "\x81\x71",
  [0x4dfa] = "\x81\x72", [0x4dfb] = "\x81\x73", [0x4dfc] = "\x81\x74",
  [0x4dfd] = "\x81\x75", [0x4dfe] = "\xc7\xac", [0x4dff] = "\x81\x76",
  [0x4e00] = "\x81\x77", [0x4e01] = "\x81\x78", [0x4e02] = "\x81\x79",
  [0x4e03] = "\x81\x7a", [0x4e04] = "\x81\x7b", [0x4e05] = "\x81\x7c",
  [0x4e06] = "\xc1\xcb", [0x4e07] = "\x81\x7d", [0x4e08] = "\xd3\xe8",
  [0x4e09] = "\xd5\xf9", [0x4e0a] = "\x81\x7e", [0x4e0b] = "\xca\xc2",
  [0x4e0c] = "\xb6\xfe", [0x4e0d] = "\xd8\xa1", [0x4e0e] = "\xd3\xda",
  [0x4e0f] = "\xbf\xf7", [0x4e10] = "\x81\x80", [0x4e11] = "\xd4\xc6",
  [0x4e12] = "\xbb\xa5", [0x4e13] = "\xd8\xc1", [0x4e14] = "\xce\xe5",
  [0x4e15] = "\xbe\xae", [0x4e16] = "\x81\x81", [0x4e17] = "\x81\x82",
  [0x4e18] = "\xd8\xa8", [0x4e19] = "\x81\x83", [0x4e1a] = "\xd1\xc7",
  [0x4e1b] = "\xd0\xa9", [0x4e1c] = "\x81\x84", [0x4e1d] = "\x81\x85",
  [0x4e1e] = "\x81\x86", [0x4e1f] = "\xd8\xbd", [0x4e20] = "\xd9\xef",
  [0x4e21] = "\xcd\xf6", [0x4e22] = "\xbf\xba", [0x4e23] = "\x81\x87",
  [0x4e24] = "\xbd\xbb", [0x4e25] = "\xba\xa5", [0x4e26] = "\xd2\xe0",
  [0x4e27] = "\xb2\xfa", [0x4e28] = "\xba\xe0", [0x4e29] = "\xc4\xb6",
  [0x4e2a] = "\x81\x88", [0x4e2b] = "\xcf\xed", [0x4e2c] = "\xbe\xa9",
  [0x4e2d] = "\xcd\xa4", [0x4e2e] = "\xc1\xc1", [0x4e2f] = "\x81\x89",
  [0x4e30] = "\x81\x8a", [0x4e31] = "\x81\x8b", [0x4e32] = "\xc7\xd7",
  [0x4e33] = "\xd9\xf1", [0x4e34] = "\x81\x8c", [0x4e35] = "\xd9\xf4",
  [0x4e36] = "\x81\x8d", [0x4e37] = "\x81\x8e", [0x4e38] = "\x81\x8f",
  [0x4e39] = "\x81\x90", [0x4e3a] = "\xc8\xcb", [0x4e3b] = "\xd8\xe9",
  [0x4e3c] = "\x81\x91", [0x4e3d] = "\x81\x92", [0x4e3e] = "\x81\x93",
  [0x4e3f] = "\xd2\xda", [0x4e40] = "\xca\xb2", [0x4e41] = "\xc8\xca",
  [0x4e42] = "\xd8\xec", [0x4e43] = "\xd8\xea", [0x4e44] = "\xd8\xc6",
  [0x4e45] = "\xbd\xf6", [0x4e46] = "\xc6\xcd", [0x4e47] = "\xb3\xf0",
  [0x4e48] = "\x81\x94", [0x4e49] = "\xd8\xeb", [0x4e4a] = "\xbd\xf1",
  [0x4e4b] = "\xbd\xe9", [0x4e4c] = "\x81\x95", [0x4e4d] = "\xc8\xd4",
  [0x4e4e] = "\xb4\xd3", [0x4e4f] = "\x81\x96", [0x4e50] = "\x81\x97",
  [0x4e51] = "\xc2\xd8", [0x4e52] = "\x81\x98", [0x4e53] = "\xb2\xd6",
  [0x4e54] = "\xd7\xd0", [0x4e55] = "\xca\xcb", [0x4e56] = "\xcb\xfb",
  [0x4e57] = "\xd5\xcc", [0x4e58] = "\xb8\xb6", [0x4e59] = "\xcf\xc9",
  [0x4e5a] = "\x81\x99", [0x4e5b] = "\x81\x9a", [0x4e5c] = "\x81\x9b",
  [0x4e5d] = "\xd9\xda", [0x4e5e] = "\xd8\xf0", [0x4e5f] = "\xc7\xaa",
  [0x4e60] = "\x81\x9c", [0x4e61] = "\xd8\xee", [0x4e62] = "\x81\x9d",
  [0x4e63] = "\xb4\xfa", [0x4e64] = "\xc1\xee", [0x4e65] = "\xd2\xd4",
  [0x4e66] = "\x81\x9e", [0x4e67] = "\x81\x9f", [0x4e68] = "\xd8\xed",
  [0x4e69] = "\x81\xa0", [0x4e6a] = "\xd2\xc7", [0x4e6b] = "\xd8\xef",
  [0x4e6c] = "\xc3\xc7", [0x4e6d] = "\x81\xa1", [0x4e6e] = "\x81\xa2",
  [0x4e6f] = "\x81\xa3", [0x4e70] = "\xd1\xf6", [0x4e71] = "\x81\xa4",
  [0x4e72] = "\xd6\xd9", [0x4e73] = "\xd8\xf2", [0x4e74] = "\x81\xa5",
  [0x4e75] = "\xd8\xf5", [0x4e76] = "\xbc\xfe", [0x4e77] = "\xbc\xdb",
  [0x4e78] = "\x81\xa6", [0x4e79] = "\x81\xa7", [0x4e7a] = "\x81\xa8",
  [0x4e7b] = "\xc8\xce", [0x4e7c] = "\x81\xa9", [0x4e7d] = "\xb7\xdd",
  [0x4e7e] = "\x81\xaa", [0x4e7f] = "\xb7\xc2", [0x4e80] = "\x81\xab",
  [0x4e81] = "\xc6\xf3", [0x4e82] = "\x81\xac", [0x4e83] = "\x81\xad",
  [0x4e84] = "\x81\xae", [0x4e85] = "\x81\xaf", [0x4e86] = "\x81\xb0",
  [0x4e87] = "\x81\xb1", [0x4e88] = "\x81\xb2", [0x4e89] = "\xd8\xf8",
  [0x4e8a] = "\xd2\xc1", [0x4e8b] = "\x81\xb3", [0x4e8c] = "\x81\xb4",
  [0x4e8d] = "\xce\xe9", [0x4e8e] = "\xbc\xbf", [0x4e8f] = "\xb7\xfc",
  [0x4e90] = "\xb7\xa5", [0x4e91] = "\xd0\xdd", [0x4e92] = "\x81\xb5",
  [0x4e93] = "\x81\xb6", [0x4e94] = "\x81\xb7", [0x4e95] = "\x81\xb8",
  [0x4e96] = "\x81\xb9", [0x4e97] = "\xd6\xda", [0x4e98] = "\xd3\xc5",
  [0x4e99] = "\xbb\xef", [0x4e9a] = "\xbb\xe1", [0x4e9b] = "\xd8\xf1",
  [0x4e9c] = "\x81\xba", [0x4e9d] = "\x81\xbb", [0x4e9e] = "\xc9\xa1",
  [0x4e9f] = "\xce\xb0", [0x4ea0] = "\xb4\xab", [0x4ea1] = "\x81\xbc",
  [0x4ea2] = "\xd8\xf3", [0x4ea3] = "\x81\xbd", [0x4ea4] = "\xc9\xcb",
  [0x4ea5] = "\xd8\xf6", [0x4ea6] = "\xc2\xd7", [0x4ea7] = "\xd8\xf7",
  [0x4ea8] = "\x81\xbe", [0x4ea9] = "\x81\xbf", [0x4eaa] = "\xce\xb1",
  [0x4eab] = "\xd8\xf9", [0x4eac] = "\x81\xc0", [0x4ead] = "\x81\xc1",
  [0x4eae] = "\x81\xc2", [0x4eaf] = "\xb2\xae", [0x4eb0] = "\xb9\xc0",
  [0x4eb1] = "\x81\xc3", [0x4eb2] = "\xd9\xa3", [0x4eb3] = "\x81\xc4",
  [0x4eb4] = "\xb0\xe9", [0x4eb5] = "\x81\xc5", [0x4eb6] = "\xc1\xe6",
  [0x4eb7] = "\x81\xc6", [0x4eb8] = "\xc9\xec", [0x4eb9] = "\x81\xc7",
  [0x4eba] = "\xcb\xc5", [0x4ebb] = "\x81\xc8", [0x4ebc] = "\xcb\xc6",
  [0x4ebd] = "\xd9\xa4", [0x4ebe] = "\x81\xc9", [0x4ebf] = "\x81\xca",
  [0x4ec0] = "\x81\xcb", [0x4ec1] = "\x81\xcc", [0x4ec2] = "\x81\xcd",
  [0x4ec3] = "\xb5\xe8", [0x4ec4] = "\x81\xce", [0x4ec5] = "\x81\xcf",
  [0x4ec6] = "\xb5\xab", [0x4ec7] = "\x81\xd0", [0x4ec8] = "\x81\xd1",
  [0x4ec9] = "\x81\xd2", [0x4eca] = "\x81\xd3", [0x4ecb] = "\x81\xd4",
  [0x4ecc] = "\x81\xd5", [0x4ecd] = "\xce\xbb", [0x4ece] = "\xb5\xcd",
  [0x4ecf] = "\xd7\xa1", [0x4ed0] = "\xd7\xf4", [0x4ed1] = "\xd3\xd3",
  [0x4ed2] = "\x81\xd6", [0x4ed3] = "\xcc\xe5", [0x4ed4] = "\x81\xd7",
  [0x4ed5] = "\xba\xce", [0x4ed6] = "\x81\xd8", [0x4ed7] = "\xd9\xa2",
  [0x4ed8] = "\xd9\xdc", [0x4ed9] = "\xd3\xe0", [0x4eda] = "\xd8\xfd",
  [0x4edb] = "\xb7\xf0", [0x4edc] = "\xd7\xf7", [0x4edd] = "\xd8\xfe",
  [0x4ede] = "\xd8\xfa", [0x4edf] = "\xd9\xa1", [0x4ee0] = "\xc4\xe3",
  [0x4ee1] = "\x81\xd9", [0x4ee2] = "\x81\xda", [0x4ee3] = "\xd3\xb6",
  [0x4ee4] = "\xd8\xf4", [0x4ee5] = "\xd9\xdd", [0x4ee6] = "\x81\xdb",
  [0x4ee7] = "\xd8\xfb", [0x4ee8] = "\x81\xdc", [0x4ee9] = "\xc5\xe5",
  [0x4eea] = "\x81\xdd", [0x4eeb] = "\x81\xde", [0x4eec] = "\xc0\xd0",
  [0x4eed] = "\x81\xdf", [0x4eee] = "\x81\xe0", [0x4eef] = "\xd1\xf0",
  [0x4ef0] = "\xb0\xdb", [0x4ef1] = "\x81\xe1", [0x4ef2] = "\x81\xe2",
  [0x4ef3] = "\xbc\xd1", [0x4ef4] = "\xd9\xa6", [0x4ef5] = "\x81\xe3",
  [0x4ef6] = "\xd9\xa5", [0x4ef7] = "\x81\xe4", [0x4ef8] = "\x81\xe5",
  [0x4ef9] = "\x81\xe6", [0x4efa] = "\x81\xe7", [0x4efb] = "\xd9\xac",
  [0x4efc] = "\xd9\xae", [0x4efd] = "\x81\xe8", [0x4efe] = "\xd9\xab",
  [0x4eff] = "\xca\xb9", [0x4f00] = "\x81\xe9", [0x4f01] = "\x81\xea",
  [0x4f02] = "\x81\xeb", [0x4f03] = "\xd9\xa9", [0x4f04] = "\xd6\xb6",
  [0x4f05] = "\x81\xec", [0x4f06] = "\x81\xed", [0x4f07] = "\x81\xee",
  [0x4f08] = "\xb3\xde", [0x4f09] = "\xd9\xa8", [0x4f0a] = "\x81\xef",
  [0x4f0b] = "\xc0\xfd", [0x4f0c] = "\x81\xf0", [0x4f0d] = "\xca\xcc",
  [0x4f0e] = "\x81\xf1", [0x4f0f] = "\xd9\xaa", [0x4f10] = "\x81\xf2",
  [0x4f11] = "\xd9\xa7", [0x4f12] = "\x81\xf3", [0x4f13] = "\x81\xf4",
  [0x4f14] = "\xd9\xb0", [0x4f15] = "\x81\xf5", [0x4f16] = "\x81\xf6",
  [0x4f17] = "\xb6\xb1", [0x4f18] = "\x81\xf7", [0x4f19] = "\x81\xf8",
  [0x4f1a] = "\x81\xf9", [0x4f1b] = "\xb9\xa9", [0x4f1c] = "\x81\xfa",
  [0x4f1d] = "\xd2\xc0", [0x4f1e] = "\x81\xfb", [0x4f1f] = "\x81\xfc",
  [0x4f20] = "\xcf\xc0", [0x4f21] = "\x81\xfd", [0x4f22] = "\x81\xfe",
  [0x4f23] = "\xc2\xc2", [0x4f24] = "\x82\x40", [0x4f25] = "\xbd\xc4",
  [0x4f26] = "\xd5\xec", [0x4f27] = "\xb2\xe0", [0x4f28] = "\xc7\xc8",
  [0x4f29] = "\xbf\xeb", [0x4f2a] = "\xd9\xad", [0x4f2b] = "\x82\x41",
  [0x4f2c] = "\xd9\xaf", [0x4f2d] = "\x82\x42", [0x4f2e] = "\xce\xea",
  [0x4f2f] = "\xba\xee", [0x4f30] = "\x82\x43", [0x4f31] = "\x82\x44",
  [0x4f32] = "\x82\x45", [0x4f33] = "\x82\x46", [0x4f34] = "\x82\x47",
  [0x4f35] = "\xc7\xd6", [0x4f36] = "\x82\x48", [0x4f37] = "\x82\x49",
  [0x4f38] = "\x82\x4a", [0x4f39] = "\x82\x4b", [0x4f3a] = "\x82\x4c",
  [0x4f3b] = "\x82\x4d", [0x4f3c] = "\x82\x4e", [0x4f3d] = "\x82\x4f",
  [0x4f3e] = "\x82\x50", [0x4f3f] = "\xb1\xe3", [0x4f40] = "\x82\x51",
  [0x4f41] = "\x82\x52", [0x4f42] = "\x82\x53", [0x4f43] = "\xb4\xd9",
  [0x4f44] = "\xb6\xed", [0x4f45] = "\xd9\xb4", [0x4f46] = "\x82\x54",
  [0x4f47] = "\x82\x55", [0x4f48] = "\x82\x56", [0x4f49] = "\x82\x57",
  [0x4f4a] = "\xbf\xa1", [0x4f4b] = "\x82\x58", [0x4f4c] = "\x82\x59",
  [0x4f4d] = "\x82\x5a", [0x4f4e] = "\xd9\xde", [0x4f4f] = "\xc7\xce",
  [0x4f50] = "\xc0\xfe", [0x4f51] = "\xd9\xb8", [0x4f52] = "\x82\x5b",
  [0x4f53] = "\x82\x5c", [0x4f54] = "\x82\x5d", [0x4f55] = "\x82\x5e",
  [0x4f56] = "\x82\x5f", [0x4f57] = "\xcb\xd7", [0x4f58] = "\xb7\xfd",
  [0x4f59] = "\x82\x60", [0x4f5a] = "\xd9\xb5", [0x4f5b] = "\x82\x61",
  [0x4f5c] = "\xd9\xb7", [0x4f5d] = "\xb1\xa3", [0x4f5e] = "\xd3\xe1",
  [0x4f5f] = "\xd9\xb9", [0x4f60] = "\x82\x62", [0x4f61] = "\xd0\xc5",
  [0x4f62] = "\x82\x63", [0x4f63] = "\xd9\xb6", [0x4f64] = "\x82\x64",
  [0x4f65] = "\x82\x65", [0x4f66] = "\xd9\xb1", [0x4f67] = "\x82\x66",
  [0x4f68] = "\xd9\xb2", [0x4f69] = "\xc1\xa9", [0x4f6a] = "\xd9\xb3",
  [0x4f6b] = "\x82\x67", [0x4f6c] = "\x82\x68", [0x4f6d] = "\xbc\xf3",
  [0x4f6e] = "\xd0\xde", [0x4f6f] = "\xb8\xa9", [0x4f70] = "\x82\x69",
  [0x4f71] = "\xbe\xe3", [0x4f72] = "\x82\x6a", [0x4f73] = "\xd9\xbd",
  [0x4f74] = "\x82\x6b", [0x4f75] = "\x82\x6c", [0x4f76] = "\x82\x6d",
  [0x4f77] = "\x82\x6e", [0x4f78] = "\xd9\xba", [0x4f79] = "\x82\x6f",
  [0x4f7a] = "\xb0\xb3", [0x4f7b] = "\x82\x70", [0x4f7c] = "\x82\x71",
  [0x4f7d] = "\x82\x72", [0x4f7e] = "\xd9\xc2", [0x4f7f] = "\x82\x73",
  [0x4f80] = "\x82\x74", [0x4f81] = "\x82\x75", [0x4f82] = "\x82\x76",
  [0x4f83] = "\x82\x77", [0x4f84] = "\x82\x78", [0x4f85] = "\x82\x79",
  [0x4f86] = "\x82\x7a", [0x4f87] = "\x82\x7b", [0x4f88] = "\x82\x7c",
  [0x4f89] = "\x82\x7d", [0x4f8a] = "\x82\x7e", [0x4f8b] = "\x82\x80",
  [0x4f8c] = "\xd9\xc4", [0x4f8d] = "\xb1\xb6", [0x4f8e] = "\x82\x81",
  [0x4f8f] = "\xd9\xbf", [0x4f90] = "\x82\x82", [0x4f91] = "\x82\x83",
  [0x4f92] = "\xb5\xb9", [0x4f93] = "\x82\x84", [0x4f94] = "\xbe\xf3",
  [0x4f95] = "\x82\x85", [0x4f96] = "\x82\x86", [0x4f97] = "\x82\x87",
  [0x4f98] = "\xcc\xc8", [0x4f99] = "\xba\xf2", [0x4f9a] = "\xd2\xd0",
  [0x4f9b] = "\x82\x88", [0x4f9c] = "\xd9\xc3", [0x4f9d] = "\x82\x89",
  [0x4f9e] = "\x82\x8a", [0x4f9f] = "\xbd\xe8", [0x4fa0] = "\x82\x8b",
  [0x4fa1] = "\xb3\xab", [0x4fa2] = "\x82\x8c", [0x4fa3] = "\x82\x8d",
  [0x4fa4] = "\x82\x8e", [0x4fa5] = "\xd9\xc5", [0x4fa6] = "\xbe\xeb",
  [0x4fa7] = "\x82\x8f", [0x4fa8] = "\xd9\xc6", [0x4fa9] = "\xd9\xbb",
  [0x4faa] = "\xc4\xdf", [0x4fab] = "\x82\x90", [0x4fac] = "\xd9\xbe",
  [0x4fad] = "\xd9\xc1", [0x4fae] = "\xd9\xc0", [0x4faf] = "\x82\x91",
  [0x4fb0] = "\x82\x92", [0x4fb1] = "\x82\x93", [0x4fb2] = "\x82\x94",
  [0x4fb3] = "\x82\x95", [0x4fb4] = "\x82\x96", [0x4fb5] = "\x82\x97",
  [0x4fb6] = "\x82\x98", [0x4fb7] = "\x82\x99", [0x4fb8] = "\x82\x9a",
  [0x4fb9] = "\x82\x9b", [0x4fba] = "\xd5\xae", [0x4fbb] = "\x82\x9c",
  [0x4fbc] = "\xd6\xb5", [0x4fbd] = "\x82\x9d", [0x4fbe] = "\xc7\xe3",
  [0x4fbf] = "\x82\x9e", [0x4fc0] = "\x82\x9f", [0x4fc1] = "\x82\xa0",
  [0x4fc2] = "\x82\xa1", [0x4fc3] = "\xd9\xc8", [0x4fc4] = "\x82\xa2",
  [0x4fc5] = "\x82\xa3", [0x4fc6] = "\x82\xa4", [0x4fc7] = "\xbc\xd9",
  [0x4fc8] = "\xd9\xca", [0x4fc9] = "\x82\xa5", [0x4fca] = "\x82\xa6",
  [0x4fcb] = "\x82\xa7", [0x4fcc] = "\xd9\xbc", [0x4fcd] = "\x82\xa8",
  [0x4fce] = "\xd9\xcb", [0x4fcf] = "\xc6\xab", [0x4fd0] = "\x82\xa9",
  [0x4fd1] = "\x82\xaa", [0x4fd2] = "\x82\xab", [0x4fd3] = "\x82\xac",
  [0x4fd4] = "\x82\xad", [0x4fd5] = "\xd9\xc9", [0x4fd6] = "\x82\xae",
  [0x4fd7] = "\x82\xaf", [0x4fd8] = "\x82\xb0", [0x4fd9] = "\x82\xb1",
  [0x4fda] = "\xd7\xf6", [0x4fdb] = "\x82\xb2", [0x4fdc] = "\xcd\xa3",
  [0x4fdd] = "\x82\xb3", [0x4fde] = "\x82\xb4", [0x4fdf] = "\x82\xb5",
  [0x4fe0] = "\x82\xb6", [0x4fe1] = "\x82\xb7", [0x4fe2] = "\x82\xb8",
  [0x4fe3] = "\x82\xb9", [0x4fe4] = "\x82\xba", [0x4fe5] = "\xbd\xa1",
  [0x4fe6] = "\x82\xbb", [0x4fe7] = "\x82\xbc", [0x4fe8] = "\x82\xbd",
  [0x4fe9] = "\x82\xbe", [0x4fea] = "\x82\xbf", [0x4feb] = "\x82\xc0",
  [0x4fec] = "\xd9\xcc", [0x4fed] = "\x82\xc1", [0x4fee] = "\x82\xc2",
  [0x4fef] = "\x82\xc3", [0x4ff0] = "\x82\xc4", [0x4ff1] = "\x82\xc5",
  [0x4ff2] = "\x82\xc6", [0x4ff3] = "\x82\xc7", [0x4ff4] = "\x82\xc8",
  [0x4ff5] = "\x82\xc9", [0x4ff6] = "\xc5\xbc", [0x4ff7] = "\xcd\xb5",
  [0x4ff8] = "\x82\xca", [0x4ff9] = "\x82\xcb", [0x4ffa] = "\x82\xcc",
  [0x4ffb] = "\xd9\xcd", [0x4ffc] = "\x82\xcd", [0x4ffd] = "\x82\xce",
  [0x4ffe] = "\xd9\xc7", [0x4fff] = "\xb3\xa5", [0x5000] = "\xbf\xfe",
  [0x5001] = "\x82\xcf", [0x5002] = "\x82\xd0", [0x5003] = "\x82\xd1",
  [0x5004] = "\x82\xd2", [0x5005] = "\xb8\xb5", [0x5006] = "\x82\xd3",
  [0x5007] = "\x82\xd4", [0x5008] = "\xc0\xfc", [0x5009] = "\x82\xd5",
  [0x500a] = "\x82\xd6", [0x500b] = "\x82\xd7", [0x500c] = "\x82\xd8",
  [0x500d] = "\xb0\xf8", [0x500e] = "\x82\xd9", [0x500f] = "\x82\xda",
  [0x5010] = "\x82\xdb", [0x5011] = "\x82\xdc", [0x5012] = "\x82\xdd",
  [0x5013] = "\x82\xde", [0x5014] = "\x82\xdf", [0x5015] = "\x82\xe0",
  [0x5016] = "\x82\xe1", [0x5017] = "\x82\xe2", [0x5018] = "\x82\xe3",
  [0x5019] = "\x82\xe4", [0x501a] = "\x82\xe5", [0x501b] = "\x82\xe6",
  [0x501c] = "\x82\xe7", [0x501d] = "\x82\xe8", [0x501e] = "\x82\xe9",
  [0x501f] = "\x82\xea", [0x5020] = "\x82\xeb", [0x5021] = "\x82\xec",
  [0x5022] = "\x82\xed", [0x5023] = "\xb4\xf6", [0x5024] = "\x82\xee",
  [0x5025] = "\xd9\xce", [0x5026] = "\x82\xef", [0x5027] = "\xd9\xcf",
  [0x5028] = "\xb4\xa2", [0x5029] = "\xd9\xd0", [0x502a] = "\x82\xf0",
  [0x502b] = "\x82\xf1", [0x502c] = "\xb4\xdf", [0x502d] = "\x82\xf2",
  [0x502e] = "\x82\xf3", [0x502f] = "\x82\xf4", [0x5030] = "\x82\xf5",
  [0x5031] = "\x82\xf6", [0x5032] = "\xb0\xc1", [0x5033] = "\x82\xf7",
  [0x5034] = "\x82\xf8", [0x5035] = "\x82\xf9", [0x5036] = "\x82\xfa",
  [0x5037] = "\x82\xfb", [0x5038] = "\x82\xfc", [0x5039] = "\x82\xfd",
  [0x503a] = "\xd9\xd1", [0x503b] = "\xc9\xb5", [0x503c] = "\x82\xfe",
  [0x503d] = "\x83\x40", [0x503e] = "\x83\x41", [0x503f] = "\x83\x42",
  [0x5040] = "\x83\x43", [0x5041] = "\x83\x44", [0x5042] = "\x83\x45",
  [0x5043] = "\x83\x46", [0x5044] = "\x83\x47", [0x5045] = "\x83\x48",
  [0x5046] = "\x83\x49", [0x5047] = "\x83\x4a", [0x5048] = "\x83\x4b",
  [0x5049] = "\x83\x4c", [0x504a] = "\x83\x4d", [0x504b] = "\x83\x4e",
  [0x504c] = "\x83\x4f", [0x504d] = "\x83\x50", [0x504e] = "\x83\x51",
  [0x504f] = "\xcf\xf1", [0x5050] = "\x83\x52", [0x5051] = "\x83\x53",
  [0x5052] = "\x83\x54", [0x5053] = "\x83\x55", [0x5054] = "\x83\x56",
  [0x5055] = "\x83\x57", [0x5056] = "\xd9\xd2", [0x5057] = "\x83\x58",
  [0x5058] = "\x83\x59", [0x5059] = "\x83\x5a", [0x505a] = "\xc1\xc5",
  [0x505b] = "\x83\x5b", [0x505c] = "\x83\x5c", [0x505d] = "\x83\x5d",
  [0x505e] = "\x83\x5e", [0x505f] = "\x83\x5f", [0x5060] = "\x83\x60",
  [0x5061] = "\x83\x61", [0x5062] = "\x83\x62", [0x5063] = "\x83\x63",
  [0x5064] = "\x83\x64", [0x5065] = "\x83\x65", [0x5066] = "\xd9\xd6",
  [0x5067] = "\xc9\xae", [0x5068] = "\x83\x66", [0x5069] = "\x83\x67",
  [0x506a] = "\x83\x68", [0x506b] = "\x83\x69", [0x506c] = "\xd9\xd5",
  [0x506d] = "\xd9\xd4", [0x506e] = "\xd9\xd7", [0x506f] = "\x83\x6a",
  [0x5070] = "\x83\x6b", [0x5071] = "\x83\x6c", [0x5072] = "\x83\x6d",
  [0x5073] = "\xcb\xdb", [0x5074] = "\x83\x6e", [0x5075] = "\xbd\xa9",
  [0x5076] = "\x83\x6f", [0x5077] = "\x83\x70", [0x5078] = "\x83\x71",
  [0x5079] = "\x83\x72", [0x507a] = "\x83\x73", [0x507b] = "\xc6\xa7",
  [0x507c] = "\x83\x74", [0x507d] = "\x83\x75", [0x507e] = "\x83\x76",
  [0x507f] = "\x83\x77", [0x5080] = "\x83\x78", [0x5081] = "\x83\x79",
  [0x5082] = "\x83\x7a", [0x5083] = "\x83\x7b", [0x5084] = "\x83\x7c",
  [0x5085] = "\x83\x7d", [0x5086] = "\xd9\xd3", [0x5087] = "\xd9\xd8",
  [0x5088] = "\x83\x7e", [0x5089] = "\x83\x80", [0x508a] = "\x83\x81",
  [0x508b] = "\xd9\xd9", [0x508c] = "\x83\x82", [0x508d] = "\x83\x83",
  [0x508e] = "\x83\x84", [0x508f] = "\x83\x85", [0x5090] = "\x83\x86",
  [0x5091] = "\x83\x87", [0x5092] = "\xc8\xe5", [0x5093] = "\x83\x88",
  [0x5094] = "\x83\x89", [0x5095] = "\x83\x8a", [0x5096] = "\x83\x8b",
  [0x5097] = "\x83\x8c", [0x5098] = "\x83\x8d", [0x5099] = "\x83\x8e",
  [0x509a] = "\x83\x8f", [0x509b] = "\x83\x90", [0x509c] = "\x83\x91",
  [0x509d] = "\x83\x92", [0x509e] = "\x83\x93", [0x509f] = "\x83\x94",
  [0x50a0] = "\x83\x95", [0x50a1] = "\xc0\xdc", [0x50a2] = "\x83\x96",
  [0x50a3] = "\x83\x97", [0x50a4] = "\x83\x98", [0x50a5] = "\x83\x99",
  [0x50a6] = "\x83\x9a", [0x50a7] = "\x83\x9b", [0x50a8] = "\x83\x9c",
  [0x50a9] = "\x83\x9d", [0x50aa] = "\x83\x9e", [0x50ab] = "\x83\x9f",
  [0x50ac] = "\x83\xa0", [0x50ad] = "\x83\xa1", [0x50ae] = "\x83\xa2",
  [0x50af] = "\x83\xa3", [0x50b0] = "\x83\xa4", [0x50b1] = "\x83\xa5",
  [0x50b2] = "\x83\xa6", [0x50b3] = "\x83\xa7", [0x50b4] = "\x83\xa8",
  [0x50b5] = "\x83\xa9", [0x50b6] = "\x83\xaa", [0x50b7] = "\x83\xab",
  [0x50b8] = "\x83\xac", [0x50b9] = "\x83\xad", [0x50ba] = "\x83\xae",
  [0x50bb] = "\x83\xaf", [0x50bc] = "\x83\xb0", [0x50bd] = "\x83\xb1",
  [0x50be] = "\x83\xb2", [0x50bf] = "\xb6\xf9", [0x50c0] = "\xd8\xa3",
  [0x50c1] = "\xd4\xca", [0x50c2] = "\x83\xb3", [0x50c3] = "\xd4\xaa",
  [0x50c4] = "\xd0\xd6", [0x50c5] = "\xb3\xe4", [0x50c6] = "\xd5\xd7",
  [0x50c7] = "\x83\xb4", [0x50c8] = "\xcf\xc8", [0x50c9] = "\xb9\xe2",
  [0x50ca] = "\x83\xb5", [0x50cb] = "\xbf\xcb", [0x50cc] = "\x83\xb6",
  [0x50cd] = "\xc3\xe2", [0x50ce] = "\x83\xb7", [0x50cf] = "\x83\xb8",
  [0x50d0] = "\x83\xb9", [0x50d1] = "\xb6\xd2", [0x50d2] = "\x83\xba",
  [0x50d3] = "\x83\xbb", [0x50d4] = "\xcd\xc3", [0x50d5] = "\xd9\xee",
  [0x50d6] = "\xd9\xf0", [0x50d7] = "\x83\xbc", [0x50d8] = "\x83\xbd",
  [0x50d9] = "\x83\xbe", [0x50da] = "\xb5\xb3", [0x50db] = "\x83\xbf",
  [0x50dc] = "\xb6\xb5", [0x50dd] = "\x83\xc0", [0x50de] = "\x83\xc1",
  [0x50df] = "\x83\xc2", [0x50e0] = "\x83\xc3", [0x50e1] = "\x83\xc4",
  [0x50e2] = "\xbe\xa4", [0x50e3] = "\x83\xc5", [0x50e4] = "\x83\xc6",
  [0x50e5] = "\xc8\xeb", [0x50e6] = "\x83\xc7", [0x50e7] = "\x83\xc8",
  [0x50e8] = "\xc8\xab", [0x50e9] = "\x83\xc9", [0x50ea] = "\x83\xca",
  [0x50eb] = "\xb0\xcb", [0x50ec] = "\xb9\xab", [0x50ed] = "\xc1\xf9",
  [0x50ee] = "\xd9\xe2", [0x50ef] = "\x83\xcb", [0x50f0] = "\xc0\xbc",
  [0x50f1] = "\xb9\xb2", [0x50f2] = "\x83\xcc", [0x50f3] = "\xb9\xd8",
  [0x50f4] = "\xd0\xcb", [0x50f5] = "\xb1\xf8", [0x50f6] = "\xc6\xe4",
  [0x50f7] = "\xbe\xdf", [0x50f8] = "\xb5\xe4", [0x50f9] = "\xd7\xc8",
  [0x50fa] = "\x83\xcd", [0x50fb] = "\xd1\xf8", [0x50fc] = "\xbc\xe6",
  [0x50fd] = "\xca\xde", [0x50fe] = "\x83\xce", [0x50ff] = "\x83\xcf",
  [0x5100] = "\xbc\xbd", [0x5101] = "\xd9\xe6", [0x5102] = "\xd8\xe7",
  [0x5103] = "\x83\xd0", [0x5104] = "\x83\xd1", [0x5105] = "\xc4\xda",
  [0x5106] = "\x83\xd2", [0x5107] = "\x83\xd3", [0x5108] = "\xb8\xd4",
  [0x5109] = "\xc8\xbd", [0x510a] = "\x83\xd4", [0x510b] = "\x83\xd5",
  [0x510c] = "\xb2\xe1", [0x510d] = "\xd4\xd9", [0x510e] = "\x83\xd6",
  [0x510f] = "\x83\xd7", [0x5110] = "\x83\xd8", [0x5111] = "\x83\xd9",
  [0x5112] = "\xc3\xb0", [0x5113] = "\x83\xda", [0x5114] = "\x83\xdb",
  [0x5115] = "\xc3\xe1", [0x5116] = "\xda\xa2", [0x5117] = "\xc8\xdf",
  [0x5118] = "\x83\xdc", [0x5119] = "\xd0\xb4", [0x511a] = "\x83\xdd",
  [0x511b] = "\xbe\xfc", [0x511c] = "\xc5\xa9", [0x511d] = "\x83\xde",
  [0x511e] = "\x83\xdf", [0x511f] = "\x83\xe0", [0x5120] = "\xb9\xda",
  [0x5121] = "\x83\xe1", [0x5122] = "\xda\xa3", [0x5123] = "\x83\xe2",
  [0x5124] = "\xd4\xa9", [0x5125] = "\xda\xa4", [0x5126] = "\x83\xe3",
  [0x5127] = "\x83\xe4", [0x5128] = "\x83\xe5", [0x5129] = "\x83\xe6",
  [0x512a] = "\x83\xe7", [0x512b] = "\xd9\xfb", [0x512c] = "\xb6\xac",
  [0x512d] = "\x83\xe8", [0x512e] = "\x83\xe9", [0x512f] = "\xb7\xeb",
  [0x5130] = "\xb1\xf9", [0x5131] = "\xd9\xfc", [0x5132] = "\xb3\xe5",
  [0x5133] = "\xbe\xf6", [0x5134] = "\x83\xea", [0x5135] = "\xbf\xf6",
  [0x5136] = "\xd2\xb1", [0x5137] = "\xc0\xe4", [0x5138] = "\x83\xeb",
  [0x5139] = "\x83\xec", [0x513a] = "\x83\xed", [0x513b] = "\xb6\xb3",
  [0x513c] = "\xd9\xfe", [0x513d] = "\xd9\xfd", [0x513e] = "\x83\xee",
  [0x513f] = "\x83\xef", [0x5140] = "\xbe\xbb", [0x5141] = "\x83\xf0",
  [0x5142] = "\x83\xf1", [0x5143] = "\x83\xf2", [0x5144] = "\xc6\xe0",
  [0x5145] = "\x83\xf3", [0x5146] = "\xd7\xbc", [0x5147] = "\xda\xa1",
  [0x5148] = "\x83\xf4", [0x5149] = "\xc1\xb9", [0x514a] = "\x83\xf5",
  [0x514b] = "\xb5\xf2", [0x514c] = "\xc1\xe8", [0x514d] = "\x83\xf6",
  [0x514e] = "\x83\xf7", [0x514f] = "\xbc\xf5", [0x5150] = "\x83\xf8",
  [0x5151] = "\xb4\xd5", [0x5152] = "\x83\xf9", [0x5153] = "\x83\xfa",
  [0x5154] = "\x83\xfb", [0x5155] = "\x83\xfc", [0x5156] = "\x83\xfd",
  [0x5157] = "\x83\xfe", [0x5158] = "\x84\x40", [0x5159] = "\x84\x41",
  [0x515a] = "\x84\x42", [0x515b] = "\xc1\xdd", [0x515c] = "\x84\x43",
  [0x515d] = "\xc4\xfd", [0x515e] = "\x84\x44", [0x515f] = "\x84\x45",
  [0x5160] = "\xbc\xb8", [0x5161] = "\xb7\xb2", [0x5162] = "\x84\x46",
  [0x5163] = "\x84\x47", [0x5164] = "\xb7\xef", [0x5165] = "\x84\x48",
  [0x5166] = "\x84\x49", [0x5167] = "\x84\x4a", [0x5168] = "\x84\x4b",
  [0x5169] = "\x84\x4c", [0x516a] = "\x84\x4d", [0x516b] = "\xd9\xec",
  [0x516c] = "\x84\x4e", [0x516d] = "\xc6\xbe", [0x516e] = "\x84\x4f",
  [0x516f] = "\xbf\xad", [0x5170] = "\xbb\xcb", [0x5171] = "\x84\x50",
  [0x5172] = "\x84\x51", [0x5173] = "\xb5\xca", [0x5174] = "\x84\x52",
  [0x5175] = "\xdb\xc9", [0x5176] = "\xd0\xd7", [0x5177] = "\x84\x53",
  [0x5178] = "\xcd\xb9", [0x5179] = "\xb0\xbc", [0x517a] = "\xb3\xf6",
  [0x517b] = "\xbb\xf7", [0x517c] = "\xdb\xca", [0x517d] = "\xba\xaf",
  [0x517e] = "\x84\x54", [0x517f] = "\xd4\xe4", [0x5180] = "\xb5\xb6",
  [0x5181] = "\xb5\xf3", [0x5182] = "\xd8\xd6", [0x5183] = "\xc8\xd0",
  [0x5184] = "\x84\x55", [0x5185] = "\x84\x56", [0x5186] = "\xb7\xd6",
  [0x5187] = "\xc7\xd0", [0x5188] = "\xd8\xd7", [0x5189] = "\x84\x57",
  [0x518a] = "\xbf\xaf", [0x518b] = "\x84\x58", [0x518c] = "\x84\x59",
  [0x518d] = "\xdb\xbb", [0x518e] = "\xd8\xd8", [0x518f] = "\x84\x5a",
  [0x5190] = "\x84\x5b", [0x5191] = "\xd0\xcc", [0x5192] = "\xbb\xae",
  [0x5193] = "\x84\x5c", [0x5194] = "\x84\x5d", [0x5195] = "\x84\x5e",
  [0x5196] = "\xeb\xbe", [0x5197] = "\xc1\xd0", [0x5198] = "\xc1\xf5",
  [0x5199] = "\xd4\xf2", [0x519a] = "\xb8\xd5", [0x519b] = "\xb4\xb4",
  [0x519c] = "\x84\x5f", [0x519d] = "\xb3\xf5", [0x519e] = "\x84\x60",
  [0x519f] = "\x84\x61", [0x51a0] = "\xc9\xbe", [0x51a1] = "\x84\x62",
  [0x51a2] = "\x84\x63", [0x51a3] = "\x84\x64", [0x51a4] = "\xc5\xd0",
  [0x51a5] = "\x84\x65", [0x51a6] = "\x84\x66", [0x51a7] = "\x84\x67",
  [0x51a8] = "\xc5\xd9", [0x51a9] = "\xc0\xfb", [0x51aa] = "\x84\x68",
  [0x51ab] = "\xb1\xf0", [0x51ac] = "\x84\x69", [0x51ad] = "\xd8\xd9",
  [0x51ae] = "\xb9\xce", [0x51af] = "\x84\x6a", [0x51b0] = "\xb5\xbd",
  [0x51b1] = "\x84\x6b", [0x51b2] = "\x84\x6c", [0x51b3] = "\xd8\xda",
  [0x51b4] = "\x84\x6d", [0x51b5] = "\x84\x6e", [0x51b6] = "\xd6\xc6",
  [0x51b7] = "\xcb\xa2", [0x51b8] = "\xc8\xaf", [0x51b9] = "\xc9\xb2",
  [0x51ba] = "\xb4\xcc", [0x51bb] = "\xbf\xcc", [0x51bc] = "\x84\x6f",
  [0x51bd] = "\xb9\xf4", [0x51be] = "\x84\x70", [0x51bf] = "\xd8\xdb",
  [0x51c0] = "\xd8\xdc", [0x51c1] = "\xb6\xe7", [0x51c2] = "\xbc\xc1",
  [0x51c3] = "\xcc\xea", [0x51c4] = "\x84\x71", [0x51c5] = "\x84\x72",
  [0x51c6] = "\x84\x73", [0x51c7] = "\x84\x74", [0x51c8] = "\x84\x75",
  [0x51c9] = "\x84\x76", [0x51ca] = "\xcf\xf7", [0x51cb] = "\x84\x77",
  [0x51cc] = "\xd8\xdd", [0x51cd] = "\xc7\xb0", [0x51ce] = "\x84\x78",
  [0x51cf] = "\x84\x79", [0x51d0] = "\xb9\xd0", [0x51d1] = "\xbd\xa3",
  [0x51d2] = "\x84\x7a", [0x51d3] = "\x84\x7b", [0x51d4] = "\xcc\xde",
  [0x51d5] = "\x84\x7c", [0x51d6] = "\xc6\xca", [0x51d7] = "\x84\x7d",
  [0x51d8] = "\x84\x7e", [0x51d9] = "\x84\x80", [0x51da] = "\x84\x81",
  [0x51db] = "\x84\x82", [0x51dc] = "\xd8\xe0", [0x51dd] = "\x84\x83",
  [0x51de] = "\xd8\xde", [0x51df] = "\x84\x84", [0x51e0] = "\x84\x85",
  [0x51e1] = "\xd8\xdf", [0x51e2] = "\x84\x86", [0x51e3] = "\x84\x87",
  [0x51e4] = "\x84\x88", [0x51e5] = "\xb0\xfe", [0x51e6] = "\x84\x89",
  [0x51e7] = "\xbe\xe7", [0x51e8] = "\x84\x8a", [0x51e9] = "\xca\xa3",
  [0x51ea] = "\xbc\xf4", [0x51eb] = "\x84\x8b", [0x51ec] = "\x84\x8c",
  [0x51ed] = "\x84\x8d", [0x51ee] = "\x84\x8e", [0x51ef] = "\xb8\xb1",
  [0x51f0] = "\x84\x8f", [0x51f1] = "\x84\x90", [0x51f2] = "\xb8\xee",
  [0x51f3] = "\x84\x91", [0x51f4] = "\x84\x92", [0x51f5] = "\x84\x93",
  [0x51f6] = "\x84\x94", [0x51f7] = "\x84\x95", [0x51f8] = "\x84\x96",
  [0x51f9] = "\x84\x97", [0x51fa] = "\x84\x98", [0x51fb] = "\x84\x99",
  [0x51fc] = "\x84\x9a", [0x51fd] = "\xd8\xe2", [0x51fe] = "\x84\x9b",
  [0x51ff] = "\xbd\xcb", [0x5200] = "\x84\x9c", [0x5201] = "\xd8\xe4",
  [0x5202] = "\xd8\xe3", [0x5203] = "\x84\x9d", [0x5204] = "\x84\x9e",
  [0x5205] = "\x84\x9f", [0x5206] = "\x84\xa0", [0x5207] = "\x84\xa1",
  [0x5208] = "\xc5\xfc", [0x5209] = "\x84\xa2", [0x520a] = "\x84\xa3",
  [0x520b] = "\x84\xa4", [0x520c] = "\x84\xa5", [0x520d] = "\x84\xa6",
  [0x520e] = "\x84\xa7", [0x520f] = "\x84\xa8", [0x5210] = "\xd8\xe5",
  [0x5211] = "\x84\xa9", [0x5212] = "\x84\xaa", [0x5213] = "\xd8\xe6",
  [0x5214] = "\x84\xab", [0x5215] = "\x84\xac", [0x5216] = "\x84\xad",
  [0x5217] = "\x84\xae", [0x5218] = "\x84\xaf", [0x5219] = "\x84\xb0",
  [0x521a] = "\x84\xb1", [0x521b] = "\xc1\xa6", [0x521c] = "\x84\xb2",
  [0x521d] = "\xc8\xb0", [0x521e] = "\xb0\xec", [0x521f] = "\xb9\xa6",
  [0x5220] = "\xbc\xd3", [0x5221] = "\xce\xf1", [0x5222] = "\xdb\xbd",
  [0x5223] = "\xc1\xd3", [0x5224] = "\x84\xb3", [0x5225] = "\x84\xb4",
  [0x5226] = "\x84\xb5", [0x5227] = "\x84\xb6", [0x5228] = "\xb6\xaf",
  [0x5229] = "\xd6\xfa", [0x522a] = "\xc5\xac", [0x522b] = "\xbd\xd9",
  [0x522c] = "\xdb\xbe", [0x522d] = "\xdb\xbf", [0x522e] = "\x84\xb7",
  [0x522f] = "\x84\xb8", [0x5230] = "\x84\xb9", [0x5231] = "\xc0\xf8",
  [0x5232] = "\xbe\xa2", [0x5233] = "\xc0\xcd", [0x5234] = "\x84\xba",
  [0x5235] = "\x84\xbb", [0x5236] = "\x84\xbc", [0x5237] = "\x84\xbd",
  [0x5238] = "\x84\xbe", [0x5239] = "\x84\xbf", [0x523a] = "\x84\xc0",
  [0x523b] = "\x84\xc1", [0x523c] = "\x84\xc2", [0x523d] = "\x84\xc3",
  [0x523e] = "\xdb\xc0", [0x523f] = "\xca\xc6", [0x5240] = "\x84\xc4",
  [0x5241] = "\x84\xc5", [0x5242] = "\x84\xc6", [0x5243] = "\xb2\xaa",
  [0x5244] = "\x84\xc7", [0x5245] = "\x84\xc8", [0x5246] = "\x84\xc9",
  [0x5247] = "\xd3\xc2", [0x5248] = "\x84\xca", [0x5249] = "\xc3\xe3",
  [0x524a] = "\x84\xcb", [0x524b] = "\xd1\xab", [0x524c] = "\x84\xcc",
  [0x524d] = "\x84\xcd", [0x524e] = "\x84\xce", [0x524f] = "\x84\xcf",
  [0x5250] = "\xdb\xc2", [0x5251] = "\x84\xd0", [0x5252] = "\xc0\xd5",
  [0x5253] = "\x84\xd1", [0x5254] = "\x84\xd2", [0x5255] = "\x84\xd3",
  [0x5256] = "\xdb\xc3", [0x5257] = "\x84\xd4", [0x5258] = "\xbf\xb1",
  [0x5259] = "\x84\xd5", [0x525a] = "\x84\xd6", [0x525b] = "\x84\xd7",
  [0x525c] = "\x84\xd8", [0x525d] = "\x84\xd9", [0x525e] = "\x84\xda",
  [0x525f] = "\xc4\xbc", [0x5260] = "\x84\xdb", [0x5261] = "\x84\xdc",
  [0x5262] = "\x84\xdd", [0x5263] = "\x84\xde", [0x5264] = "\xc7\xda",
  [0x5265] = "\x84\xdf", [0x5266] = "\x84\xe0", [0x5267] = "\x84\xe1",
  [0x5268] = "\x84\xe2", [0x5269] = "\x84\xe3", [0x526a] = "\x84\xe4",
  [0x526b] = "\x84\xe5", [0x526c] = "\x84\xe6", [0x526d] = "\x84\xe7",
  [0x526e] = "\x84\xe8", [0x526f] = "\x84\xe9", [0x5270] = "\xdb\xc4",
  [0x5271] = "\x84\xea", [0x5272] = "\x84\xeb", [0x5273] = "\x84\xec",
  [0x5274] = "\x84\xed", [0x5275] = "\x84\xee", [0x5276] = "\x84\xef",
  [0x5277] = "\x84\xf0", [0x5278] = "\x84\xf1", [0x5279] = "\xd9\xe8",
  [0x527a] = "\xc9\xd7", [0x527b] = "\x84\xf2", [0x527c] = "\x84\xf3",
  [0x527d] = "\x84\xf4", [0x527e] = "\xb9\xb4", [0x527f] = "\xce\xf0",
  [0x5280] = "\xd4\xc8", [0x5281] = "\x84\xf5", [0x5282] = "\x84\xf6",
  [0x5283] = "\x84\xf7", [0x5284] = "\x84\xf8", [0x5285] = "\xb0\xfc",
  [0x5286] = "\xb4\xd2", [0x5287] = "\x84\xf9", [0x5288] = "\xd0\xd9",
  [0x5289] = "\x84\xfa", [0x528a] = "\x84\xfb", [0x528b] = "\x84\xfc",
  [0x528c] = "\x84\xfd", [0x528d] = "\xd9\xe9", [0x528e] = "\x84\xfe",
  [0x528f] = "\xde\xcb", [0x5290] = "\xd9\xeb", [0x5291] = "\x85\x40",
  [0x5292] = "\x85\x41", [0x5293] = "\x85\x42", [0x5294] = "\x85\x43",
  [0x5295] = "\xd8\xb0", [0x5296] = "\xbb\xaf", [0x5297] = "\xb1\xb1",
  [0x5298] = "\x85\x44", [0x5299] = "\xb3\xd7", [0x529a] = "\xd8\xce",
  [0x529b] = "\x85\x45", [0x529c] = "\x85\x46", [0x529d] = "\xd4\xd1",
  [0x529e] = "\x85\x47", [0x529f] = "\x85\x48", [0x52a0] = "\xbd\xb3",
  [0x52a1] = "\xbf\xef", [0x52a2] = "\x85\x49", [0x52a3] = "\xcf\xbb",
  [0x52a4] = "\x85\x4a", [0x52a5] = "\x85\x4b", [0x52a6] = "\xd8\xd0",
  [0x52a7] = "\x85\x4c", [0x52a8] = "\x85\x4d", [0x52a9] = "\x85\x4e",
  [0x52aa] = "\xb7\xcb", [0x52ab] = "\x85\x4f", [0x52ac] = "\x85\x50",
  [0x52ad] = "\x85\x51", [0x52ae] = "\xd8\xd1", [0x52af] = "\x85\x52",
  [0x52b0] = "\x85\x53", [0x52b1] = "\x85\x54", [0x52b2] = "\x85\x55",
  [0x52b3] = "\x85\x56", [0x52b4] = "\x85\x57", [0x52b5] = "\x85\x58",
  [0x52b6] = "\x85\x59", [0x52b7] = "\x85\x5a", [0x52b8] = "\x85\x5b",
  [0x52b9] = "\xc6\xa5", [0x52ba] = "\xc7\xf8", [0x52bb] = "\xd2\xbd",
  [0x52bc] = "\x85\x5c", [0x52bd] = "\x85\x5d", [0x52be] = "\xd8\xd2",
  [0x52bf] = "\xc4\xe4", [0x52c0] = "\x85\x5e", [0x52c1] = "\xca\xae",
  [0x52c2] = "\x85\x5f", [0x52c3] = "\xc7\xa7", [0x52c4] = "\x85\x60",
  [0x52c5] = "\xd8\xa6", [0x52c6] = "\x85\x61", [0x52c7] = "\xc9\xfd",
  [0x52c8] = "\xce\xe7", [0x52c9] = "\xbb\xdc", [0x52ca] = "\xb0\xeb",
  [0x52cb] = "\x85\x62", [0x52cc] = "\x85\x63", [0x52cd] = "\x85\x64",
  [0x52ce] = "\xbb\xaa", [0x52cf] = "\xd0\xad", [0x52d0] = "\x85\x65",
  [0x52d1] = "\xb1\xb0", [0x52d2] = "\xd7\xe4", [0x52d3] = "\xd7\xbf",
  [0x52d4] = "\x85\x66", [0x52d5] = "\xb5\xa5", [0x52d6] = "\xc2\xf4",
  [0x52d7] = "\xc4\xcf", [0x52d8] = "\x85\x67", [0x52d9] = "\x85\x68",
  [0x52da] = "\xb2\xa9", [0x52db] = "\x85\x69", [0x52dc] = "\xb2\xb7",
  [0x52dd] = "\x85\x6a", [0x52de] = "\xb1\xe5", [0x52df] = "\xdf\xb2",
  [0x52e0] = "\xd5\xbc", [0x52e1] = "\xbf\xa8", [0x52e2] = "\xc2\xac",
  [0x52e3] = "\xd8\xd5", [0x52e4] = "\xc2\xb1", [0x52e5] = "\x85\x6b",
  [0x52e6] = "\xd8\xd4", [0x52e7] = "\xce\xd4", [0x52e8] = "\x85\x6c",
  [0x52e9] = "\xda\xe0", [0x52ea] = "\x85\x6d", [0x52eb] = "\xce\xc0",
  [0x52ec] = "\x85\x6e", [0x52ed] = "\x85\x6f", [0x52ee] = "\xd8\xb4",
  [0x52ef] = "\xc3\xae", [0x52f0] = "\xd3\xa1", [0x52f1] = "\xce\xa3",
  [0x52f2] = "\x85\x70", [0x52f3] = "\xbc\xb4", [0x52f4] = "\xc8\xb4",
  [0x52f5] = "\xc2\xd1", [0x52f6] = "\x85\x71", [0x52f7] = "\xbe\xed",
  [0x52f8] = "\xd0\xb6", [0x52f9] = "\x85\x72", [0x52fa] = "\xda\xe1",
  [0x52fb] = "\x85\x73", [0x52fc] = "\x85\x74", [0x52fd] = "\x85\x75",
  [0x52fe] = "\x85\x76", [0x52ff] = "\xc7\xe4", [0x5300] = "\x85\x77",
  [0x5301] = "\x85\x78", [0x5302] = "\xb3\xa7", [0x5303] = "\x85\x79",
  [0x5304] = "\xb6\xf2", [0x5305] = "\xcc\xfc", [0x5306] = "\xc0\xfa",
  [0x5307] = "\x85\x7a", [0x5308] = "\x85\x7b", [0x5309] = "\xc0\xf7",
  [0x530a] = "\x85\x7c", [0x530b] = "\xd1\xb9", [0x530c] = "\xd1\xe1",
  [0x530d] = "\xd8\xc7", [0x530e] = "\x85\x7d", [0x530f] = "\x85\x7e",
  [0x5310] = "\x85\x80", [0x5311] = "\x85\x81", [0x5312] = "\x85\x82",
  [0x5313] = "\x85\x83", [0x5314] = "\x85\x84", [0x5315] = "\xb2\xde",
  [0x5316] = "\x85\x85", [0x5317] = "\x85\x86", [0x5318] = "\xc0\xe5",
  [0x5319] = "\x85\x87", [0x531a] = "\xba\xf1", [0x531b] = "\x85\x88",
  [0x531c] = "\x85\x89", [0x531d] = "\xd8\xc8", [0x531e] = "\x85\x8a",
  [0x531f] = "\xd4\xad", [0x5320] = "\x85\x8b", [0x5321] = "\x85\x8c",
  [0x5322] = "\xcf\xe1", [0x5323] = "\xd8\xc9", [0x5324] = "\x85\x8d",
  [0x5325] = "\xd8\xca", [0x5326] = "\xcf\xc3", [0x5327] = "\x85\x8e",
  [0x5328] = "\xb3\xf8", [0x5329] = "\xbe\xc7", [0x532a] = "\x85\x8f",
  [0x532b] = "\x85\x90", [0x532c] = "\x85\x91", [0x532d] = "\x85\x92",
  [0x532e] = "\xd8\xcb", [0x532f] = "\x85\x93", [0x5330] = "\x85\x94",
  [0x5331] = "\x85\x95", [0x5332] = "\x85\x96", [0x5333] = "\x85\x97",
  [0x5334] = "\x85\x98", [0x5335] = "\x85\x99", [0x5336] = "\xdb\xcc",
  [0x5337] = "\x85\x9a", [0x5338] = "\x85\x9b", [0x5339] = "\x85\x9c",
  [0x533a] = "\x85\x9d", [0x533b] = "\xc8\xa5", [0x533c] = "\x85\x9e",
  [0x533d] = "\x85\x9f", [0x533e] = "\x85\xa0", [0x533f] = "\xcf\xd8",
  [0x5340] = "\x85\xa1", [0x5341] = "\xc8\xfe", [0x5342] = "\xb2\xce",
  [0x5343] = "\x85\xa2", [0x5344] = "\x85\xa3", [0x5345] = "\x85\xa4",
  [0x5346] = "\x85\xa5", [0x5347] = "\x85\xa6", [0x5348] = "\xd3\xd6",
  [0x5349] = "\xb2\xe6", [0x534a] = "\xbc\xb0", [0x534b] = "\xd3\xd1",
  [0x534c] = "\xcb\xab", [0x534d] = "\xb7\xb4", [0x534e] = "\x85\xa7",
  [0x534f] = "\x85\xa8", [0x5350] = "\x85\xa9", [0x5351] = "\xb7\xa2",
  [0x5352] = "\x85\xaa", [0x5353] = "\x85\xab", [0x5354] = "\xca\xe5",
  [0x5355] = "\x85\xac", [0x5356] = "\xc8\xa1", [0x5357] = "\xca\xdc",
  [0x5358] = "\xb1\xe4", [0x5359] = "\xd0\xf0", [0x535a] = "\x85\xad",
  [0x535b] = "\xc5\xd1", [0x535c] = "\x85\xae", [0x535d] = "\x85\xaf",
  [0x535e] = "\x85\xb0", [0x535f] = "\xdb\xc5", [0x5360] = "\xb5\xfe",
  [0x5361] = "\x85\xb1", [0x5362] = "\x85\xb2", [0x5363] = "\xbf\xda",
  [0x5364] = "\xb9\xc5", [0x5365] = "\xbe\xe4", [0x5366] = "\xc1\xed",
  [0x5367] = "\x85\xb3", [0x5368] = "\xdf\xb6", [0x5369] = "\xdf\xb5",
  [0x536a] = "\xd6\xbb", [0x536b] = "\xbd\xd0", [0x536c] = "\xd5\xd9",
  [0x536d] = "\xb0\xc8", [0x536e] = "\xb6\xa3", [0x536f] = "\xbf\xc9",
  [0x5370] = "\xcc\xa8", [0x5371] = "\xdf\xb3", [0x5372] = "\xca\xb7",
  [0x5373] = "\xd3\xd2", [0x5374] = "\x85\xb4", [0x5375] = "\xd8\xcf",
  [0x5376] = "\xd2\xb6", [0x5377] = "\xba\xc5", [0x5378] = "\xcb\xbe",
  [0x5379] = "\xcc\xbe", [0x537a] = "\x85\xb5", [0x537b] = "\xdf\xb7",
  [0x537c] = "\xb5\xf0", [0x537d] = "\xdf\xb4", [0x537e] = "\x85\xb6",
  [0x537f] = "\x85\xb7", [0x5380] = "\x85\xb8", [0x5381] = "\xd3\xf5",
  [0x5382] = "\x85\xb9", [0x5383] = "\xb3\xd4", [0x5384] = "\xb8\xf7",
  [0x5385] = "\x85\xba", [0x5386] = "\xdf\xba", [0x5387] = "\x85\xbb",
  [0x5388] = "\xba\xcf", [0x5389] = "\xbc\xaa", [0x538a] = "\xb5\xf5",
  [0x538b] = "\x85\xbc", [0x538c] = "\xcd\xac", [0x538d] = "\xc3\xfb",
  [0x538e] = "\xba\xf3", [0x538f] = "\xc0\xf4", [0x5390] = "\xcd\xc2",
  [0x5391] = "\xcf\xf2", [0x5392] = "\xdf\xb8", [0x5393] = "\xcf\xc5",
  [0x5394] = "\x85\xbd", [0x5395] = "\xc2\xc0", [0x5396] = "\xdf\xb9",
  [0x5397] = "\xc2\xf0", [0x5398] = "\x85\xbe", [0x5399] = "\x85\xbf",
  [0x539a] = "\x85\xc0", [0x539b] = "\xbe\xfd", [0x539c] = "\x85\xc1",
  [0x539d] = "\xc1\xdf", [0x539e] = "\xcd\xcc", [0x539f] = "\xd2\xf7",
  [0x53a0] = "\xb7\xcd", [0x53a1] = "\xdf\xc1", [0x53a2] = "\x85\xc2",
  [0x53a3] = "\xdf\xc4", [0x53a4] = "\x85\xc3", [0x53a5] = "\x85\xc4",
  [0x53a6] = "\xb7\xf1", [0x53a7] = "\xb0\xc9", [0x53a8] = "\xb6\xd6",
  [0x53a9] = "\xb7\xd4", [0x53aa] = "\x85\xc5", [0x53ab] = "\xba\xac",
  [0x53ac] = "\xcc\xfd", [0x53ad] = "\xbf\xd4", [0x53ae] = "\xcb\xb1",
  [0x53af] = "\xc6\xf4", [0x53b0] = "\x85\xc6", [0x53b1] = "\xd6\xa8",
  [0x53b2] = "\xdf\xc5", [0x53b3] = "\x85\xc7", [0x53b4] = "\xce\xe2",
  [0x53b5] = "\xb3\xb3", [0x53b6] = "\x85\xc8", [0x53b7] = "\x85\xc9",
  [0x53b8] = "\xce\xfc", [0x53b9] = "\xb4\xb5", [0x53ba] = "\x85\xca",
  [0x53bb] = "\xce\xc7", [0x53bc] = "\xba\xf0", [0x53bd] = "\x85\xcb",
  [0x53be] = "\xce\xe1", [0x53bf] = "\x85\xcc", [0x53c0] = "\xd1\xbd",
  [0x53c1] = "\x85\xcd", [0x53c2] = "\x85\xce", [0x53c3] = "\xdf\xc0",
  [0x53c4] = "\x85\xcf", [0x53c5] = "\x85\xd0", [0x53c6] = "\xb4\xf4",
  [0x53c7] = "\x85\xd1", [0x53c8] = "\xb3\xca", [0x53c9] = "\x85\xd2",
  [0x53ca] = "\xb8\xe6", [0x53cb] = "\xdf\xbb", [0x53cc] = "\x85\xd3",
  [0x53cd] = "\x85\xd4", [0x53ce] = "\x85\xd5", [0x53cf] = "\x85\xd6",
  [0x53d0] = "\xc4\xc5", [0x53d1] = "\x85\xd7", [0x53d2] = "\xdf\xbc",
  [0x53d3] = "\xdf\xbd", [0x53d4] = "\xdf\xbe", [0x53d5] = "\xc5\xbb",
  [0x53d6] = "\xdf\xbf", [0x53d7] = "\xdf\xc2", [0x53d8] = "\xd4\xb1",
  [0x53d9] = "\xdf\xc3", [0x53da] = "\x85\xd8", [0x53db] = "\xc7\xba",
  [0x53dc] = "\xce\xd8", [0x53dd] = "\x85\xd9", [0x53de] = "\x85\xda",
  [0x53df] = "\x85\xdb", [0x53e0] = "\x85\xdc", [0x53e1] = "\x85\xdd",
  [0x53e2] = "\xc4\xd8", [0x53e3] = "\x85\xde", [0x53e4] = "\xdf\xca",
  [0x53e5] = "\x85\xdf", [0x53e6] = "\xdf\xcf", [0x53e7] = "\x85\xe0",
  [0x53e8] = "\xd6\xdc", [0x53e9] = "\x85\xe1", [0x53ea] = "\x85\xe2",
  [0x53eb] = "\x85\xe3", [0x53ec] = "\x85\xe4", [0x53ed] = "\x85\xe5",
  [0x53ee] = "\x85\xe6", [0x53ef] = "\x85\xe7", [0x53f0] = "\x85\xe8",
  [0x53f1] = "\xdf\xc9", [0x53f2] = "\xdf\xda", [0x53f3] = "\xce\xb6",
  [0x53f4] = "\x85\xe9", [0x53f5] = "\xba\xc7", [0x53f6] = "\xdf\xce",
  [0x53f7] = "\xdf\xc8", [0x53f8] = "\xc5\xde", [0x53f9] = "\x85\xea",
  [0x53fa] = "\x85\xeb", [0x53fb] = "\xc9\xeb", [0x53fc] = "\xba\xf4",
  [0x53fd] = "\xc3\xfc", [0x53fe] = "\x85\xec", [0x53ff] = "\x85\xed",
  [0x5400] = "\xbe\xd7", [0x5401] = "\x85\xee", [0x5402] = "\xdf\xc6",
  [0x5403] = "\x85\xef", [0x5404] = "\xdf\xcd", [0x5405] = "\x85\xf0",
  [0x5406] = "\xc5\xd8", [0x5407] = "\x85\xf1", [0x5408] = "\x85\xf2",
  [0x5409] = "\x85\xf3", [0x540a] = "\x85\xf4", [0x540b] = "\xd5\xa6",
  [0x540c] = "\xba\xcd", [0x540d] = "\x85\xf5", [0x540e] = "\xbe\xcc",
  [0x540f] = "\xd3\xbd", [0x5410] = "\xb8\xc0", [0x5411] = "\x85\xf6",
  [0x5412] = "\xd6\xe4", [0x5413] = "\x85\xf7", [0x5414] = "\xdf\xc7",
  [0x5415] = "\xb9\xbe", [0x5416] = "\xbf\xa7", [0x5417] = "\x85\xf8",
  [0x5418] = "\x85\xf9", [0x5419] = "\xc1\xfc", [0x541a] = "\xdf\xcb",
  [0x541b] = "\xdf\xcc", [0x541c] = "\x85\xfa", [0x541d] = "\xdf\xd0",
  [0x541e] = "\x85\xfb", [0x541f] = "\x85\xfc", [0x5420] = "\x85\xfd",
  [0x5421] = "\x85\xfe", [0x5422] = "\x86\x40", [0x5423] = "\xdf\xdb",
  [0x5424] = "\xdf\xe5", [0x5425] = "\x86\x41", [0x5426] = "\xdf\xd7",
  [0x5427] = "\xdf\xd6", [0x5428] = "\xd7\xc9", [0x5429] = "\xdf\xe3",
  [0x542a] = "\xdf\xe4", [0x542b] = "\xe5\xeb", [0x542c] = "\xd2\xa7",
  [0x542d] = "\xdf\xd2", [0x542e] = "\x86\x42", [0x542f] = "\xbf\xa9",
  [0x5430] = "\x86\x43", [0x5431] = "\xd4\xdb", [0x5432] = "\x86\x44",
  [0x5433] = "\xbf\xc8", [0x5434] = "\xdf\xd4", [0x5435] = "\x86\x45",
  [0x5436] = "\x86\x46", [0x5437] = "\x86\x47", [0x5438] = "\xcf\xcc",
  [0x5439] = "\x86\x48", [0x543a] = "\x86\x49", [0x543b] = "\xdf\xdd",
  [0x543c] = "\x86\x4a", [0x543d] = "\xd1\xca", [0x543e] = "\x86\x4b",
  [0x543f] = "\xdf\xde", [0x5440] = "\xb0\xa7", [0x5441] = "\xc6\xb7",
  [0x5442] = "\xdf\xd3", [0x5443] = "\x86\x4c", [0x5444] = "\xba\xe5",
  [0x5445] = "\x86\x4d", [0x5446] = "\xb6\xdf", [0x5447] = "\xcd\xdb",
  [0x5448] = "\xb9\xfe", [0x5449] = "\xd4\xd5", [0x544a] = "\x86\x4e",
  [0x544b] = "\x86\x4f", [0x544c] = "\xdf\xdf", [0x544d] = "\xcf\xec",
  [0x544e] = "\xb0\xa5", [0x544f] = "\xdf\xe7", [0x5450] = "\xdf\xd1",
  [0x5451] = "\xd1\xc6", [0x5452] = "\xdf\xd5", [0x5453] = "\xdf\xd8",
  [0x5454] = "\xdf\xd9", [0x5455] = "\xdf\xdc", [0x5456] = "\x86\x50",
  [0x5457] = "\xbb\xa9", [0x5458] = "\x86\x51", [0x5459] = "\xdf\xe0",
  [0x545a] = "\xdf\xe1", [0x545b] = "\x86\x52", [0x545c] = "\xdf\xe2",
  [0x545d] = "\xdf\xe6", [0x545e] = "\xdf\xe8", [0x545f] = "\xd3\xb4",
  [0x5460] = "\x86\x53", [0x5461] = "\x86\x54", [0x5462] = "\x86\x55",
  [0x5463] = "\x86\x56", [0x5464] = "\x86\x57", [0x5465] = "\xb8\xe7",
  [0x5466] = "\xc5\xb6", [0x5467] = "\xdf\xea", [0x5468] = "\xc9\xda",
  [0x5469] = "\xc1\xa8", [0x546a] = "\xc4\xc4", [0x546b] = "\x86\x58",
  [0x546c] = "\x86\x59", [0x546d] = "\xbf\xde", [0x546e] = "\xcf\xf8",
  [0x546f] = "\x86\x5a", [0x5470] = "\x86\x5b", [0x5471] = "\x86\x5c",
  [0x5472] = "\xd5\xdc", [0x5473] = "\xdf\xee", [0x5474] = "\x86\x5d",
  [0x5475] = "\x86\x5e", [0x5476] = "\x86\x5f", [0x5477] = "\x86\x60",
  [0x5478] = "\x86\x61", [0x5479] = "\x86\x62", [0x547a] = "\xb2\xb8",
  [0x547b] = "\x86\x63", [0x547c] = "\xba\xdf", [0x547d] = "\xdf\xec",
  [0x547e] = "\x86\x64", [0x547f] = "\xdb\xc1", [0x5480] = "\x86\x65",
  [0x5481] = "\xd1\xe4", [0x5482] = "\x86\x66", [0x5483] = "\x86\x67",
  [0x5484] = "\x86\x68", [0x5485] = "\x86\x69", [0x5486] = "\xcb\xf4",
  [0x5487] = "\xb4\xbd", [0x5488] = "\x86\x6a", [0x5489] = "\xb0\xa6",
  [0x548a] = "\x86\x6b", [0x548b] = "\x86\x6c", [0x548c] = "\x86\x6d",
  [0x548d] = "\x86\x6e", [0x548e] = "\x86\x6f", [0x548f] = "\xdf\xf1",
  [0x5490] = "\xcc\xc6", [0x5491] = "\xdf\xf2", [0x5492] = "\x86\x70",
  [0x5493] = "\x86\x71", [0x5494] = "\xdf\xed", [0x5495] = "\x86\x72",
  [0x5496] = "\x86\x73", [0x5497] = "\x86\x74", [0x5498] = "\x86\x75",
  [0x5499] = "\x86\x76", [0x549a] = "\x86\x77", [0x549b] = "\xdf\xe9",
  [0x549c] = "\x86\x78", [0x549d] = "\x86\x79", [0x549e] = "\x86\x7a",
  [0x549f] = "\x86\x7b", [0x54a0] = "\xdf\xeb", [0x54a1] = "\x86\x7c",
  [0x54a2] = "\xdf\xef", [0x54a3] = "\xdf\xf0", [0x54a4] = "\xbb\xbd",
  [0x54a5] = "\x86\x7d", [0x54a6] = "\x86\x7e", [0x54a7] = "\xdf\xf3",
  [0x54a8] = "\x86\x80", [0x54a9] = "\x86\x81", [0x54aa] = "\xdf\xf4",
  [0x54ab] = "\x86\x82", [0x54ac] = "\xbb\xa3", [0x54ad] = "\x86\x83",
  [0x54ae] = "\xca\xdb", [0x54af] = "\xce\xa8", [0x54b0] = "\xe0\xa7",
  [0x54b1] = "\xb3\xaa", [0x54b2] = "\x86\x84", [0x54b3] = "\xe0\xa6",
  [0x54b4] = "\x86\x85", [0x54b5] = "\x86\x86", [0x54b6] = "\x86\x87",
  [0x54b7] = "\xe0\xa1", [0x54b8] = "\x86\x88", [0x54b9] = "\x86\x89",
  [0x54ba] = "\x86\x8a", [0x54bb] = "\x86\x8b", [0x54bc] = "\xdf\xfe",
  [0x54bd] = "\x86\x8c", [0x54be] = "\xcd\xd9", [0x54bf] = "\xdf\xfc",
  [0x54c0] = "\x86\x8d", [0x54c1] = "\xdf\xfa", [0x54c2] = "\x86\x8e",
  [0x54c3] = "\xbf\xd0", [0x54c4] = "\xd7\xc4", [0x54c5] = "\x86\x8f",
  [0x54c6] = "\xc9\xcc", [0x54c7] = "\x86\x90", [0x54c8] = "\x86\x91",
  [0x54c9] = "\xdf\xf8", [0x54ca] = "\xb0\xa1", [0x54cb] = "\x86\x92",
  [0x54cc] = "\x86\x93", [0x54cd] = "\x86\x94", [0x54ce] = "\x86\x95",
  [0x54cf] = "\x86\x96", [0x54d0] = "\xdf\xfd", [0x54d1] = "\x86\x97",
  [0x54d2] = "\x86\x98", [0x54d3] = "\x86\x99", [0x54d4] = "\x86\x9a",
  [0x54d5] = "\xdf\xfb", [0x54d6] = "\xe0\xa2", [0x54d7] = "\x86\x9b",
  [0x54d8] = "\x86\x9c", [0x54d9] = "\x86\x9d", [0x54da] = "\x86\x9e",
  [0x54db] = "\x86\x9f", [0x54dc] = "\xe0\xa8", [0x54dd] = "\x86\xa0",
  [0x54de] = "\x86\xa1", [0x54df] = "\x86\xa2", [0x54e0] = "\x86\xa3",
  [0x54e1] = "\xb7\xc8", [0x54e2] = "\x86\xa4", [0x54e3] = "\x86\xa5",
  [0x54e4] = "\xc6\xa1", [0x54e5] = "\xc9\xb6", [0x54e6] = "\xc0\xb2",
  [0x54e7] = "\xdf\xf5", [0x54e8] = "\x86\xa6", [0x54e9] = "\x86\xa7",
  [0x54ea] = "\xc5\xbe", [0x54eb] = "\x86\xa8", [0x54ec] = "\xd8\xc4",
  [0x54ed] = "\xdf\xf9", [0x54ee] = "\xc4\xf6", [0x54ef] = "\x86\xa9",
  [0x54f0] = "\x86\xaa", [0x54f1] = "\x86\xab", [0x54f2] = "\x86\xac",
  [0x54f3] = "\x86\xad", [0x54f4] = "\x86\xae", [0x54f5] = "\xe0\xa3",
  [0x54f6] = "\xe0\xa4", [0x54f7] = "\xe0\xa5", [0x54f8] = "\xd0\xa5",
  [0x54f9] = "\x86\xaf", [0x54fa] = "\x86\xb0", [0x54fb] = "\xe0\xb4",
  [0x54fc] = "\xcc\xe4", [0x54fd] = "\x86\xb1", [0x54fe] = "\xe0\xb1",
  [0x54ff] = "\x86\xb2", [0x5500] = "\xbf\xa6", [0x5501] = "\xe0\xaf",
  [0x5502] = "\xce\xb9", [0x5503] = "\xe0\xab", [0x5504] = "\xc9\xc6",
  [0x5505] = "\x86\xb3", [0x5506] = "\x86\xb4", [0x5507] = "\xc0\xae",
  [0x5508] = "\xe0\xae", [0x5509] = "\xba\xed", [0x550a] = "\xba\xb0",
  [0x550b] = "\xe0\xa9", [0x550c] = "\x86\xb5", [0x550d] = "\x86\xb6",
  [0x550e] = "\x86\xb7", [0x550f] = "\xdf\xf6", [0x5510] = "\x86\xb8",
  [0x5511] = "\xe0\xb3", [0x5512] = "\x86\xb9", [0x5513] = "\x86\xba",
  [0x5514] = "\xe0\xb8", [0x5515] = "\x86\xbb", [0x5516] = "\x86\xbc",
  [0x5517] = "\x86\xbd", [0x5518] = "\xb4\xad", [0x5519] = "\xe0\xb9",
  [0x551a] = "\x86\xbe", [0x551b] = "\x86\xbf", [0x551c] = "\xcf\xb2",
  [0x551d] = "\xba\xc8", [0x551e] = "\x86\xc0", [0x551f] = "\xe0\xb0",
  [0x5520] = "\x86\xc1", [0x5521] = "\x86\xc2", [0x5522] = "\x86\xc3",
  [0x5523] = "\x86\xc4", [0x5524] = "\x86\xc5", [0x5525] = "\x86\xc6",
  [0x5526] = "\x86\xc7", [0x5527] = "\xd0\xfa", [0x5528] = "\x86\xc8",
  [0x5529] = "\x86\xc9", [0x552a] = "\x86\xca", [0x552b] = "\x86\xcb",
  [0x552c] = "\x86\xcc", [0x552d] = "\x86\xcd", [0x552e] = "\x86\xce",
  [0x552f] = "\x86\xcf", [0x5530] = "\x86\xd0", [0x5531] = "\xe0\xac",
  [0x5532] = "\x86\xd1", [0x5533] = "\xd4\xfb", [0x5534] = "\x86\xd2",
  [0x5535] = "\xdf\xf7", [0x5536] = "\x86\xd3", [0x5537] = "\xc5\xe7",
  [0x5538] = "\x86\xd4", [0x5539] = "\xe0\xad", [0x553a] = "\x86\xd5",
  [0x553b] = "\xd3\xf7", [0x553c] = "\x86\xd6", [0x553d] = "\xe0\xb6",
  [0x553e] = "\xe0\xb7", [0x553f] = "\x86\xd7", [0x5540] = "\x86\xd8",
  [0x5541] = "\x86\xd9", [0x5542] = "\x86\xda", [0x5543] = "\x86\xdb",
  [0x5544] = "\xe0\xc4", [0x5545] = "\xd0\xe1", [0x5546] = "\x86\xdc",
  [0x5547] = "\x86\xdd", [0x5548] = "\x86\xde", [0x5549] = "\xe0\xbc",
  [0x554a] = "\x86\xdf", [0x554b] = "\x86\xe0", [0x554c] = "\xe0\xc9",
  [0x554d] = "\xe0\xca", [0x554e] = "\x86\xe1", [0x554f] = "\x86\xe2",
  [0x5550] = "\x86\xe3", [0x5551] = "\xe0\xbe", [0x5552] = "\xe0\xaa",
  [0x5553] = "\xc9\xa4", [0x5554] = "\xe0\xc1", [0x5555] = "\x86\xe4",
  [0x5556] = "\xe0\xb2", [0x5557] = "\x86\xe5", [0x5558] = "\x86\xe6",
  [0x5559] = "\x86\xe7", [0x555a] = "\x86\xe8", [0x555b] = "\x86\xe9",
  [0x555c] = "\xca\xc8", [0x555d] = "\xe0\xc3", [0x555e] = "\x86\xea",
  [0x555f] = "\xe0\xb5", [0x5560] = "\x86\xeb", [0x5561] = "\xce\xcb",
  [0x5562] = "\x86\xec", [0x5563] = "\xcb\xc3", [0x5564] = "\xe0\xcd",
  [0x5565] = "\xe0\xc6", [0x5566] = "\xe0\xc2", [0x5567] = "\x86\xed",
  [0x5568] = "\xe0\xcb", [0x5569] = "\x86\xee", [0x556a] = "\xe0\xba",
  [0x556b] = "\xe0\xbf", [0x556c] = "\xe0\xc0", [0x556d] = "\x86\xef",
  [0x556e] = "\x86\xf0", [0x556f] = "\xe0\xc5", [0x5570] = "\x86\xf1",
  [0x5571] = "\x86\xf2", [0x5572] = "\xe0\xc7", [0x5573] = "\xe0\xc8",
  [0x5574] = "\x86\xf3", [0x5575] = "\xe0\xcc", [0x5576] = "\x86\xf4",
  [0x5577] = "\xe0\xbb", [0x5578] = "\x86\xf5", [0x5579] = "\x86\xf6",
  [0x557a] = "\x86\xf7", [0x557b] = "\x86\xf8", [0x557c] = "\x86\xf9",
  [0x557d] = "\xcb\xd4", [0x557e] = "\xe0\xd5", [0x557f] = "\x86\xfa",
  [0x5580] = "\xe0\xd6", [0x5581] = "\xe0\xd2", [0x5582] = "\x86\xfb",
  [0x5583] = "\x86\xfc", [0x5584] = "\x86\xfd", [0x5585] = "\x86\xfe",
  [0x5586] = "\x87\x40", [0x5587] = "\x87\x41", [0x5588] = "\xe0\xd0",
  [0x5589] = "\xbc\xce", [0x558a] = "\x87\x42", [0x558b] = "\x87\x43",
  [0x558c] = "\xe0\xd1", [0x558d] = "\x87\x44", [0x558e] = "\xb8\xc2",
  [0x558f] = "\xd8\xc5", [0x5590] = "\x87\x45", [0x5591] = "\x87\x46",
  [0x5592] = "\x87\x47", [0x5593] = "\x87\x48", [0x5594] = "\x87\x49",
  [0x5595] = "\x87\x4a", [0x5596] = "\x87\x4b", [0x5597] = "\x87\x4c",
  [0x5598] = "\xd0\xea", [0x5599] = "\x87\x4d", [0x559a] = "\x87\x4e",
  [0x559b] = "\xc2\xef", [0x559c] = "\x87\x4f", [0x559d] = "\x87\x50",
  [0x559e] = "\xe0\xcf", [0x559f] = "\xe0\xbd", [0x55a0] = "\x87\x51",
  [0x55a1] = "\x87\x52", [0x55a2] = "\x87\x53", [0x55a3] = "\xe0\xd4",
  [0x55a4] = "\xe0\xd3", [0x55a5] = "\x87\x54", [0x55a6] = "\x87\x55",
  [0x55a7] = "\xe0\xd7", [0x55a8] = "\x87\x56", [0x55a9] = "\x87\x57",
  [0x55aa] = "\x87\x58", [0x55ab] = "\x87\x59", [0x55ac] = "\xe0\xdc",
  [0x55ad] = "\xe0\xd8", [0x55ae] = "\x87\x5a", [0x55af] = "\x87\x5b",
  [0x55b0] = "\x87\x5c", [0x55b1] = "\xd6\xf6", [0x55b2] = "\xb3\xb0",
  [0x55b3] = "\x87\x5d", [0x55b4] = "\xd7\xec", [0x55b5] = "\x87\x5e",
  [0x55b6] = "\xcb\xbb", [0x55b7] = "\x87\x5f", [0x55b8] = "\x87\x60",
  [0x55b9] = "\xe0\xda", [0x55ba] = "\x87\x61", [0x55bb] = "\xce\xfb",
  [0x55bc] = "\x87\x62", [0x55bd] = "\x87\x63", [0x55be] = "\x87\x64",
  [0x55bf] = "\xba\xd9", [0x55c0] = "\x87\x65", [0x55c1] = "\x87\x66",
  [0x55c2] = "\x87\x67", [0x55c3] = "\x87\x68", [0x55c4] = "\x87\x69",
  [0x55c5] = "\x87\x6a", [0x55c6] = "\x87\x6b", [0x55c7] = "\x87\x6c",
  [0x55c8] = "\x87\x6d", [0x55c9] = "\x87\x6e", [0x55ca] = "\x87\x6f",
  [0x55cb] = "\x87\x70", [0x55cc] = "\xe0\xe1", [0x55cd] = "\xe0\xdd",
  [0x55ce] = "\xd2\xad", [0x55cf] = "\x87\x71", [0x55d0] = "\x87\x72",
  [0x55d1] = "\x87\x73", [0x55d2] = "\x87\x74", [0x55d3] = "\x87\x75",
  [0x55d4] = "\xe0\xe2", [0x55d5] = "\x87\x76", [0x55d6] = "\x87\x77",
  [0x55d7] = "\xe0\xdb", [0x55d8] = "\xe0\xd9", [0x55d9] = "\xe0\xdf",
  [0x55da] = "\x87\x78", [0x55db] = "\x87\x79", [0x55dc] = "\xe0\xe0",
  [0x55dd] = "\x87\x7a", [0x55de] = "\x87\x7b", [0x55df] = "\x87\x7c",
  [0x55e0] = "\x87\x7d", [0x55e1] = "\x87\x7e", [0x55e2] = "\xe0\xde",
  [0x55e3] = "\x87\x80", [0x55e4] = "\xe0\xe4", [0x55e5] = "\x87\x81",
  [0x55e6] = "\x87\x82", [0x55e7] = "\x87\x83", [0x55e8] = "\xc6\xf7",
  [0x55e9] = "\xd8\xac", [0x55ea] = "\xd4\xeb", [0x55eb] = "\xe0\xe6",
  [0x55ec] = "\xca\xc9", [0x55ed] = "\x87\x84", [0x55ee] = "\x87\x85",
  [0x55ef] = "\x87\x86", [0x55f0] = "\x87\x87", [0x55f1] = "\xe0\xe5",
  [0x55f2] = "\x87\x88", [0x55f3] = "\x87\x89", [0x55f4] = "\x87\x8a",
  [0x55f5] = "\x87\x8b", [0x55f6] = "\xb8\xc1", [0x55f7] = "\x87\x8c",
  [0x55f8] = "\x87\x8d", [0x55f9] = "\x87\x8e", [0x55fa] = "\x87\x8f",
  [0x55fb] = "\xe0\xe7", [0x55fc] = "\xe0\xe8", [0x55fd] = "\x87\x90",
  [0x55fe] = "\x87\x91", [0x55ff] = "\x87\x92", [0x5600] = "\x87\x93",
  [0x5601] = "\x87\x94", [0x5602] = "\x87\x95", [0x5603] = "\x87\x96",
  [0x5604] = "\x87\x97", [0x5605] = "\xe0\xe9", [0x5606] = "\xe0\xe3",
  [0x5607] = "\x87\x98", [0x5608] = "\x87\x99", [0x5609] = "\x87\x9a",
  [0x560a] = "\x87\x9b", [0x560b] = "\x87\x9c", [0x560c] = "\x87\x9d",
  [0x560d] = "\x87\x9e", [0x560e] = "\xba\xbf", [0x560f] = "\xcc\xe7",
  [0x5610] = "\x87\x9f", [0x5611] = "\x87\xa0", [0x5612] = "\x87\xa1",
  [0x5613] = "\xe0\xea", [0x5614] = "\x87\xa2", [0x5615] = "\x87\xa3",
  [0x5616] = "\x87\xa4", [0x5617] = "\x87\xa5", [0x5618] = "\x87\xa6",
  [0x5619] = "\x87\xa7", [0x561a] = "\x87\xa8", [0x561b] = "\x87\xa9",
  [0x561c] = "\x87\xaa", [0x561d] = "\x87\xab", [0x561e] = "\x87\xac",
  [0x561f] = "\x87\xad", [0x5620] = "\x87\xae", [0x5621] = "\x87\xaf",
  [0x5622] = "\x87\xb0", [0x5623] = "\xcf\xf9", [0x5624] = "\x87\xb1",
  [0x5625] = "\x87\xb2", [0x5626] = "\x87\xb3", [0x5627] = "\x87\xb4",
  [0x5628] = "\x87\xb5", [0x5629] = "\x87\xb6", [0x562a] = "\x87\xb7",
  [0x562b] = "\x87\xb8", [0x562c] = "\x87\xb9", [0x562d] = "\x87\xba",
  [0x562e] = "\x87\xbb", [0x562f] = "\xe0\xeb", [0x5630] = "\x87\xbc",
  [0x5631] = "\x87\xbd", [0x5632] = "\x87\xbe", [0x5633] = "\x87\xbf",
  [0x5634] = "\x87\xc0", [0x5635] = "\x87\xc1", [0x5636] = "\x87\xc2",
  [0x5637] = "\xc8\xc2", [0x5638] = "\x87\xc3", [0x5639] = "\x87\xc4",
  [0x563a] = "\x87\xc5", [0x563b] = "\x87\xc6", [0x563c] = "\xbd\xc0",
  [0x563d] = "\x87\xc7", [0x563e] = "\x87\xc8", [0x563f] = "\x87\xc9",
  [0x5640] = "\x87\xca", [0x5641] = "\x87\xcb", [0x5642] = "\x87\xcc",
  [0x5643] = "\x87\xcd", [0x5644] = "\x87\xce", [0x5645] = "\x87\xcf",
  [0x5646] = "\x87\xd0", [0x5647] = "\x87\xd1", [0x5648] = "\x87\xd2",
  [0x5649] = "\x87\xd3", [0x564a] = "\xc4\xd2", [0x564b] = "\x87\xd4",
  [0x564c] = "\x87\xd5", [0x564d] = "\x87\xd6", [0x564e] = "\x87\xd7",
  [0x564f] = "\x87\xd8", [0x5650] = "\x87\xd9", [0x5651] = "\x87\xda",
  [0x5652] = "\x87\xdb", [0x5653] = "\x87\xdc", [0x5654] = "\xe0\xec",
  [0x5655] = "\x87\xdd", [0x5656] = "\x87\xde", [0x5657] = "\xe0\xed",
  [0x5658] = "\x87\xdf", [0x5659] = "\x87\xe0", [0x565a] = "\xc7\xf4",
  [0x565b] = "\xcb\xc4", [0x565c] = "\x87\xe1", [0x565d] = "\xe0\xee",
  [0x565e] = "\xbb\xd8", [0x565f] = "\xd8\xb6", [0x5660] = "\xd2\xf2",
  [0x5661] = "\xe0\xef", [0x5662] = "\xcd\xc5", [0x5663] = "\x87\xe2",
  [0x5664] = "\xb6\xda", [0x5665] = "\x87\xe3", [0x5666] = "\x87\xe4",
  [0x5667] = "\x87\xe5", [0x5668] = "\x87\xe6", [0x5669] = "\x87\xe7",
  [0x566a] = "\x87\xe8", [0x566b] = "\xe0\xf1", [0x566c] = "\x87\xe9",
  [0x566d] = "\xd4\xb0", [0x566e] = "\x87\xea", [0x566f] = "\x87\xeb",
  [0x5670] = "\xc0\xa7", [0x5671] = "\xb4\xd1", [0x5672] = "\x87\xec",
  [0x5673] = "\x87\xed", [0x5674] = "\xce\xa7", [0x5675] = "\xe0\xf0",
  [0x5676] = "\x87\xee", [0x5677] = "\x87\xef", [0x5678] = "\x87\xf0",
  [0x5679] = "\xe0\xf2", [0x567a] = "\xb9\xcc", [0x567b] = "\x87\xf1",
  [0x567c] = "\x87\xf2", [0x567d] = "\xb9\xfa", [0x567e] = "\xcd\xbc",
  [0x567f] = "\xe0\xf3", [0x5680] = "\x87\xf3", [0x5681] = "\x87\xf4",
  [0x5682] = "\x87\xf5", [0x5683] = "\xc6\xd4", [0x5684] = "\xe0\xf4",
  [0x5685] = "\x87\xf6", [0x5686] = "\xd4\xb2", [0x5687] = "\x87\xf7",
  [0x5688] = "\xc8\xa6", [0x5689] = "\xe0\xf6", [0x568a] = "\xe0\xf5",
  [0x568b] = "\x87\xf8", [0x568c] = "\x87\xf9", [0x568d] = "\x87\xfa",
  [0x568e] = "\x87\xfb", [0x568f] = "\x87\xfc", [0x5690] = "\x87\xfd",
  [0x5691] = "\x87\xfe", [0x5692] = "\x88\x40", [0x5693] = "\x88\x41",
  [0x5694] = "\x88\x42", [0x5695] = "\x88\x43", [0x5696] = "\x88\x44",
  [0x5697] = "\x88\x45", [0x5698] = "\x88\x46", [0x5699] = "\x88\x47",
  [0x569a] = "\x88\x48", [0x569b] = "\x88\x49", [0x569c] = "\xe0\xf7",
  [0x569d] = "\x88\x4a", [0x569e] = "\x88\x4b", [0x569f] = "\xcd\xc1",
  [0x56a0] = "\x88\x4c", [0x56a1] = "\x88\x4d", [0x56a2] = "\x88\x4e",
  [0x56a3] = "\xca\xa5", [0x56a4] = "\x88\x4f", [0x56a5] = "\x88\x50",
  [0x56a6] = "\x88\x51", [0x56a7] = "\x88\x52", [0x56a8] = "\xd4\xda",
  [0x56a9] = "\xdb\xd7", [0x56aa] = "\xdb\xd9", [0x56ab] = "\x88\x53",
  [0x56ac] = "\xdb\xd8", [0x56ad] = "\xb9\xe7", [0x56ae] = "\xdb\xdc",
  [0x56af] = "\xdb\xdd", [0x56b0] = "\xb5\xd8", [0x56b1] = "\x88\x54",
  [0x56b2] = "\x88\x55", [0x56b3] = "\xdb\xda", [0x56b4] = "\x88\x56",
  [0x56b5] = "\x88\x57", [0x56b6] = "\x88\x58", [0x56b7] = "\x88\x59",
  [0x56b8] = "\x88\x5a", [0x56b9] = "\xdb\xdb", [0x56ba] = "\xb3\xa1",
  [0x56bb] = "\xdb\xdf", [0x56bc] = "\x88\x5b", [0x56bd] = "\x88\x5c",
  [0x56be] = "\xbb\xf8", [0x56bf] = "\x88\x5d", [0x56c0] = "\xd6\xb7",
  [0x56c1] = "\x88\x5e", [0x56c2] = "\xdb\xe0", [0x56c3] = "\x88\x5f",
  [0x56c4] = "\x88\x60", [0x56c5] = "\x88\x61", [0x56c6] = "\x88\x62",
  [0x56c7] = "\xbe\xf9", [0x56c8] = "\x88\x63", [0x56c9] = "\x88\x64",
  [0x56ca] = "\xb7\xbb", [0x56cb] = "\x88\x65", [0x56cc] = "\xdb\xd0",
  [0x56cd] = "\xcc\xae", [0x56ce] = "\xbf\xb2", [0x56cf] = "\xbb\xb5",
  [0x56d0] = "\xd7\xf8", [0x56d1] = "\xbf\xd3", [0x56d2] = "\x88\x66",
  [0x56d3] = "\x88\x67", [0x56d4] = "\x88\x68", [0x56d5] = "\x88\x69",
  [0x56d6] = "\x88\x6a", [0x56d7] = "\xbf\xe9", [0x56d8] = "\x88\x6b",
  [0x56d9] = "\x88\x6c", [0x56da] = "\xbc\xe1", [0x56db] = "\xcc\xb3",
  [0x56dc] = "\xdb\xde", [0x56dd] = "\xb0\xd3", [0x56de] = "\xce\xeb",
  [0x56df] = "\xb7\xd8", [0x56e0] = "\xd7\xb9", [0x56e1] = "\xc6\xc2",
  [0x56e2] = "\x88\x6d", [0x56e3] = "\x88\x6e", [0x56e4] = "\xc0\xa4",
  [0x56e5] = "\x88\x6f", [0x56e6] = "\xcc\xb9", [0x56e7] = "\x88\x70",
  [0x56e8] = "\xdb\xe7", [0x56e9] = "\xdb\xe1", [0x56ea] = "\xc6\xba",
  [0x56eb] = "\xdb\xe3", [0x56ec] = "\x88\x71", [0x56ed] = "\xdb\xe8",
  [0x56ee] = "\x88\x72", [0x56ef] = "\xc5\xf7", [0x56f0] = "\x88\x73",
  [0x56f1] = "\x88\x74", [0x56f2] = "\x88\x75", [0x56f3] = "\xdb\xea",
  [0x56f4] = "\x88\x76", [0x56f5] = "\x88\x77", [0x56f6] = "\xdb\xe9",
  [0x56f7] = "\xbf\xc0", [0x56f8] = "\x88\x78", [0x56f9] = "\x88\x79",
  [0x56fa] = "\x88\x7a", [0x56fb] = "\xdb\xe6", [0x56fc] = "\xdb\xe5",
  [0x56fd] = "\x88\x7b", [0x56fe] = "\x88\x7c", [0x56ff] = "\x88\x7d",
  [0x5700] = "\x88\x7e", [0x5701] = "\x88\x80", [0x5702] = "\xb4\xb9",
  [0x5703] = "\xc0\xac", [0x5704] = "\xc2\xa2", [0x5705] = "\xdb\xe2",
  [0x5706] = "\xdb\xe4", [0x5707] = "\x88\x81", [0x5708] = "\x88\x82",
  [0x5709] = "\x88\x83", [0x570a] = "\x88\x84", [0x570b] = "\xd0\xcd",
  [0x570c] = "\xdb\xed", [0x570d] = "\x88\x85", [0x570e] = "\x88\x86",
  [0x570f] = "\x88\x87", [0x5710] = "\x88\x88", [0x5711] = "\x88\x89",
  [0x5712] = "\xc0\xdd", [0x5713] = "\xdb\xf2", [0x5714] = "\x88\x8a",
  [0x5715] = "\x88\x8b", [0x5716] = "\x88\x8c", [0x5717] = "\x88\x8d",
  [0x5718] = "\x88\x8e", [0x5719] = "\x88\x8f", [0x571a] = "\x88\x90",
  [0x571b] = "\xb6\xe2", [0x571c] = "\x88\x91", [0x571d] = "\x88\x92",
  [0x571e] = "\x88\x93", [0x571f] = "\x88\x94", [0x5720] = "\xdb\xf3",
  [0x5721] = "\xdb\xd2", [0x5722] = "\xb9\xb8", [0x5723] = "\xd4\xab",
  [0x5724] = "\xdb\xec", [0x5725] = "\x88\x95", [0x5726] = "\xbf\xd1",
  [0x5727] = "\xdb\xf0", [0x5728] = "\x88\x96", [0x5729] = "\xdb\xd1",
  [0x572a] = "\x88\x97", [0x572b] = "\xb5\xe6", [0x572c] = "\x88\x98",
  [0x572d] = "\xdb\xeb", [0x572e] = "\xbf\xe5", [0x572f] = "\x88\x99",
  [0x5730] = "\x88\x9a", [0x5731] = "\x88\x9b", [0x5732] = "\xdb\xee",
  [0x5733] = "\x88\x9c", [0x5734] = "\xdb\xf1", [0x5735] = "\x88\x9d",
  [0x5736] = "\x88\x9e", [0x5737] = "\x88\x9f", [0x5738] = "\xdb\xf9",
  [0x5739] = "\x88\xa0", [0x573a] = "\x88\xa1", [0x573b] = "\x88\xa2",
  [0x573c] = "\x88\xa3", [0x573d] = "\x88\xa4", [0x573e] = "\x88\xa5",
  [0x573f] = "\x88\xa6", [0x5740] = "\x88\xa7", [0x5741] = "\x88\xa8",
  [0x5742] = "\xb9\xa1", [0x5743] = "\xb0\xa3", [0x5744] = "\x88\xa9",
  [0x5745] = "\x88\xaa", [0x5746] = "\x88\xab", [0x5747] = "\x88\xac",
  [0x5748] = "\x88\xad", [0x5749] = "\x88\xae", [0x574a] = "\x88\xaf",
  [0x574b] = "\xc2\xf1", [0x574c] = "\x88\xb0", [0x574d] = "\x88\xb1",
  [0x574e] = "\xb3\xc7", [0x574f] = "\xdb\xef", [0x5750] = "\x88\xb2",
  [0x5751] = "\x88\xb3", [0x5752] = "\xdb\xf8", [0x5753] = "\x88\xb4",
  [0x5754] = "\xc6\xd2", [0x5755] = "\xdb\xf4", [0x5756] = "\x88\xb5",
  [0x5757] = "\x88\xb6", [0x5758] = "\xdb\xf5", [0x5759] = "\xdb\xf7",
  [0x575a] = "\xdb\xf6", [0x575b] = "\x88\xb7", [0x575c] = "\x88\xb8",
  [0x575d] = "\xdb\xfe", [0x575e] = "\x88\xb9", [0x575f] = "\xd3\xf2",
  [0x5760] = "\xb2\xba", [0x5761] = "\x88\xba", [0x5762] = "\x88\xbb",
  [0x5763] = "\x88\xbc", [0x5764] = "\xdb\xfd", [0x5765] = "\x88\xbd",
  [0x5766] = "\x88\xbe", [0x5767] = "\x88\xbf", [0x5768] = "\x88\xc0",
  [0x5769] = "\x88\xc1", [0x576a] = "\x88\xc2", [0x576b] = "\x88\xc3",
  [0x576c] = "\x88\xc4", [0x576d] = "\xdc\xa4", [0x576e] = "\x88\xc5",
  [0x576f] = "\xdb\xfb", [0x5770] = "\x88\xc6", [0x5771] = "\x88\xc7",
  [0x5772] = "\x88\xc8", [0x5773] = "\x88\xc9", [0x5774] = "\xdb\xfa",
  [0x5775] = "\x88\xca", [0x5776] = "\x88\xcb", [0x5777] = "\x88\xcc",
  [0x5778] = "\xdb\xfc", [0x5779] = "\xc5\xe0", [0x577a] = "\xbb\xf9",
  [0x577b] = "\x88\xcd", [0x577c] = "\x88\xce", [0x577d] = "\xdc\xa3",
  [0x577e] = "\x88\xcf", [0x577f] = "\x88\xd0", [0x5780] = "\xdc\xa5",
  [0x5781] = "\x88\xd1", [0x5782] = "\xcc\xc3", [0x5783] = "\x88\xd2",
  [0x5784] = "\x88\xd3", [0x5785] = "\x88\xd4", [0x5786] = "\xb6\xd1",
  [0x5787] = "\xdd\xc0", [0x5788] = "\x88\xd5", [0x5789] = "\x88\xd6",
  [0x578a] = "\x88\xd7", [0x578b] = "\xdc\xa1", [0x578c] = "\x88\xd8",
  [0x578d] = "\xdc\xa2", [0x578e] = "\x88\xd9", [0x578f] = "\x88\xda",
  [0x5790] = "\x88\xdb", [0x5791] = "\xc7\xb5", [0x5792] = "\x88\xdc",
  [0x5793] = "\x88\xdd", [0x5794] = "\x88\xde", [0x5795] = "\xb6\xe9",
  [0x5796] = "\x88\xdf", [0x5797] = "\x88\xe0", [0x5798] = "\x88\xe1",
  [0x5799] = "\xdc\xa7", [0x579a] = "\x88\xe2", [0x579b] = "\x88\xe3",
  [0x579c] = "\x88\xe4", [0x579d] = "\x88\xe5", [0x579e] = "\xdc\xa6",
  [0x579f] = "\x88\xe6", [0x57a0] = "\xdc\xa9", [0x57a1] = "\xb1\xa4",
  [0x57a2] = "\x88\xe7", [0x57a3] = "\x88\xe8", [0x57a4] = "\xb5\xcc",
  [0x57a5] = "\x88\xe9", [0x57a6] = "\x88\xea", [0x57a7] = "\x88\xeb",
  [0x57a8] = "\x88\xec", [0x57a9] = "\x88\xed", [0x57aa] = "\xbf\xb0",
  [0x57ab] = "\x88\xee", [0x57ac] = "\x88\xef", [0x57ad] = "\x88\xf0",
  [0x57ae] = "\x88\xf1", [0x57af] = "\x88\xf2", [0x57b0] = "\xd1\xdf",
  [0x57b1] = "\x88\xf3", [0x57b2] = "\x88\xf4", [0x57b3] = "\x88\xf5",
  [0x57b4] = "\x88\xf6", [0x57b5] = "\xb6\xc2", [0x57b6] = "\x88\xf7",
  [0x57b7] = "\x88\xf8", [0x57b8] = "\x88\xf9", [0x57b9] = "\x88\xfa",
  [0x57ba] = "\x88\xfb", [0x57bb] = "\x88\xfc", [0x57bc] = "\x88\xfd",
  [0x57bd] = "\x88\xfe", [0x57be] = "\x89\x40", [0x57bf] = "\x89\x41",
  [0x57c0] = "\x89\x42", [0x57c1] = "\x89\x43", [0x57c2] = "\x89\x44",
  [0x57c3] = "\x89\x45", [0x57c4] = "\xdc\xa8", [0x57c5] = "\x89\x46",
  [0x57c6] = "\x89\x47", [0x57c7] = "\x89\x48", [0x57c8] = "\x89\x49",
  [0x57c9] = "\x89\x4a", [0x57ca] = "\x89\x4b", [0x57cb] = "\x89\x4c",
  [0x57cc] = "\xcb\xfa", [0x57cd] = "\xeb\xf3", [0x57ce] = "\x89\x4d",
  [0x57cf] = "\x89\x4e", [0x57d0] = "\x89\x4f", [0x57d1] = "\xcb\xdc",
  [0x57d2] = "\x89\x50", [0x57d3] = "\x89\x51", [0x57d4] = "\xcb\xfe",
  [0x57d5] = "\x89\x52", [0x57d6] = "\x89\x53", [0x57d7] = "\x89\x54",
  [0x57d8] = "\xcc\xc1", [0x57d9] = "\x89\x55", [0x57da] = "\x89\x56",
  [0x57db] = "\x89\x57", [0x57dc] = "\x89\x58", [0x57dd] = "\x89\x59",
  [0x57de] = "\xc8\xfb", [0x57df] = "\x89\x5a", [0x57e0] = "\x89\x5b",
  [0x57e1] = "\x89\x5c", [0x57e2] = "\x89\x5d", [0x57e3] = "\x89\x5e",
  [0x57e4] = "\x89\x5f", [0x57e5] = "\xdc\xaa", [0x57e6] = "\x89\x60",
  [0x57e7] = "\x89\x61", [0x57e8] = "\x89\x62", [0x57e9] = "\x89\x63",
  [0x57ea] = "\x89\x64", [0x57eb] = "\xcc\xee", [0x57ec] = "\xdc\xab",
  [0x57ed] = "\x89\x65", [0x57ee] = "\x89\x66", [0x57ef] = "\x89\x67",
  [0x57f0] = "\x89\x68", [0x57f1] = "\x89\x69", [0x57f2] = "\x89\x6a",
  [0x57f3] = "\x89\x6b", [0x57f4] = "\x89\x6c", [0x57f5] = "\x89\x6d",
  [0x57f6] = "\x89\x6e", [0x57f7] = "\x89\x6f", [0x57f8] = "\x89\x70",
  [0x57f9] = "\x89\x71", [0x57fa] = "\x89\x72", [0x57fb] = "\x89\x73",
  [0x57fc] = "\x89\x74", [0x57fd] = "\x89\x75", [0x57fe] = "\xdb\xd3",
  [0x57ff] = "\x89\x76", [0x5800] = "\xdc\xaf", [0x5801] = "\xdc\xac",
  [0x5802] = "\x89\x77", [0x5803] = "\xbe\xb3", [0x5804] = "\x89\x78",
  [0x5805] = "\xca\xfb", [0x5806] = "\x89\x79", [0x5807] = "\x89\x7a",
  [0x5808] = "\x89\x7b", [0x5809] = "\xdc\xad", [0x580a] = "\x89\x7c",
  [0x580b] = "\x89\x7d", [0x580c] = "\x89\x7e", [0x580d] = "\x89\x80",
  [0x580e] = "\x89\x81", [0x580f] = "\x89\x82", [0x5810] = "\x89\x83",
  [0x5811] = "\x89\x84", [0x5812] = "\xc9\xca", [0x5813] = "\xc4\xb9",
  [0x5814] = "\x89\x85", [0x5815] = "\x89\x86", [0x5816] = "\x89\x87",
  [0x5817] = "\x89\x88", [0x5818] = "\x89\x89", [0x5819] = "\xc7\xbd",
  [0x581a] = "\xdc\xae", [0x581b] = "\x89\x8a", [0x581c] = "\x89\x8b",
  [0x581d] = "\x89\x8c", [0x581e] = "\xd4\xf6", [0x581f] = "\xd0\xe6",
  [0x5820] = "\x89\x8d", [0x5821] = "\x89\x8e", [0x5822] = "\x89\x8f",
  [0x5823] = "\x89\x90", [0x5824] = "\x89\x91", [0x5825] = "\x89\x92",
  [0x5826] = "\x89\x93", [0x5827] = "\x89\x94", [0x5828] = "\xc4\xab",
  [0x5829] = "\xb6\xd5", [0x582a] = "\x89\x95", [0x582b] = "\x89\x96",
  [0x582c] = "\x89\x97", [0x582d] = "\x89\x98", [0x582e] = "\x89\x99",
  [0x582f] = "\x89\x9a", [0x5830] = "\x89\x9b", [0x5831] = "\x89\x9c",
  [0x5832] = "\x89\x9d", [0x5833] = "\x89\x9e", [0x5834] = "\x89\x9f",
  [0x5835] = "\x89\xa0", [0x5836] = "\x89\xa1", [0x5837] = "\x89\xa2",
  [0x5838] = "\x89\xa3", [0x5839] = "\x89\xa4", [0x583a] = "\x89\xa5",
  [0x583b] = "\x89\xa6", [0x583c] = "\xdb\xd4", [0x583d] = "\x89\xa7",
  [0x583e] = "\x89\xa8", [0x583f] = "\x89\xa9", [0x5840] = "\x89\xaa",
  [0x5841] = "\xb1\xda", [0x5842] = "\x89\xab", [0x5843] = "\x89\xac",
  [0x5844] = "\x89\xad", [0x5845] = "\xdb\xd5", [0x5846] = "\x89\xae",
  [0x5847] = "\x89\xaf", [0x5848] = "\x89\xb0", [0x5849] = "\x89\xb1",
  [0x584a] = "\x89\xb2", [0x584b] = "\x89\xb3", [0x584c] = "\x89\xb4",
  [0x584d] = "\x89\xb5", [0x584e] = "\x89\xb6", [0x584f] = "\x89\xb7",
  [0x5850] = "\x89\xb8", [0x5851] = "\xdb\xd6", [0x5852] = "\x89\xb9",
  [0x5853] = "\x89\xba", [0x5854] = "\x89\xbb", [0x5855] = "\xba\xbe",
  [0x5856] = "\x89\xbc", [0x5857] = "\x89\xbd", [0x5858] = "\x89\xbe",
  [0x5859] = "\x89\xbf", [0x585a] = "\x89\xc0", [0x585b] = "\x89\xc1",
  [0x585c] = "\x89\xc2", [0x585d] = "\x89\xc3", [0x585e] = "\x89\xc4",
  [0x585f] = "\x89\xc5", [0x5860] = "\x89\xc6", [0x5861] = "\x89\xc7",
  [0x5862] = "\x89\xc8", [0x5863] = "\x89\xc9", [0x5864] = "\xc8\xc0",
  [0x5865] = "\x89\xca", [0x5866] = "\x89\xcb", [0x5867] = "\x89\xcc",
  [0x5868] = "\x89\xcd", [0x5869] = "\x89\xce", [0x586a] = "\x89\xcf",
  [0x586b] = "\xca\xbf", [0x586c] = "\xc8\xc9", [0x586d] = "\x89\xd0",
  [0x586e] = "\xd7\xb3", [0x586f] = "\x89\xd1", [0x5870] = "\xc9\xf9",
  [0x5871] = "\x89\xd2", [0x5872] = "\x89\xd3", [0x5873] = "\xbf\xc7",
  [0x5874] = "\x89\xd4", [0x5875] = "\x89\xd5", [0x5876] = "\xba\xf8",
  [0x5877] = "\x89\xd6", [0x5878] = "\x89\xd7", [0x5879] = "\xd2\xbc",
  [0x587a] = "\x89\xd8", [0x587b] = "\x89\xd9", [0x587c] = "\x89\xda",
  [0x587d] = "\x89\xdb", [0x587e] = "\x89\xdc", [0x587f] = "\x89\xdd",
  [0x5880] = "\x89\xde", [0x5881] = "\x89\xdf", [0x5882] = "\xe2\xba",
  [0x5883] = "\x89\xe0", [0x5884] = "\xb4\xa6", [0x5885] = "\x89\xe1",
  [0x5886] = "\x89\xe2", [0x5887] = "\xb1\xb8", [0x5888] = "\x89\xe3",
  [0x5889] = "\x89\xe4", [0x588a] = "\x89\xe5", [0x588b] = "\x89\xe6",
  [0x588c] = "\x89\xe7", [0x588d] = "\xb8\xb4", [0x588e] = "\x89\xe8",
  [0x588f] = "\xcf\xc4", [0x5890] = "\x89\xe9", [0x5891] = "\x89\xea",
  [0x5892] = "\x89\xeb", [0x5893] = "\x89\xec", [0x5894] = "\xd9\xe7",
  [0x5895] = "\xcf\xa6", [0x5896] = "\xcd\xe2", [0x5897] = "\x89\xed",
  [0x5898] = "\x89\xee", [0x5899] = "\xd9\xed", [0x589a] = "\xb6\xe0",
  [0x589b] = "\x89\xef", [0x589c] = "\xd2\xb9", [0x589d] = "\x89\xf0",
  [0x589e] = "\x89\xf1", [0x589f] = "\xb9\xbb", [0x58a0] = "\x89\xf2",
  [0x58a1] = "\x89\xf3", [0x58a2] = "\x89\xf4", [0x58a3] = "\x89\xf5",
  [0x58a4] = "\xe2\xb9", [0x58a5] = "\xe2\xb7", [0x58a6] = "\x89\xf6",
  [0x58a7] = "\xb4\xf3", [0x58a8] = "\x89\xf7", [0x58a9] = "\xcc\xec",
  [0x58aa] = "\xcc\xab", [0x58ab] = "\xb7\xf2", [0x58ac] = "\x89\xf8",
  [0x58ad] = "\xd8\xb2", [0x58ae] = "\xd1\xeb", [0x58af] = "\xba\xbb",
  [0x58b0] = "\x89\xf9", [0x58b1] = "\xca\xa7", [0x58b2] = "\x89\xfa",
  [0x58b3] = "\x89\xfb", [0x58b4] = "\xcd\xb7", [0x58b5] = "\x89\xfc",
  [0x58b6] = "\x89\xfd", [0x58b7] = "\xd2\xc4", [0x58b8] = "\xbf\xe4",
  [0x58b9] = "\xbc\xd0", [0x58ba] = "\xb6\xe1", [0x58bb] = "\x89\xfe",
  [0x58bc] = "\xde\xc5", [0x58bd] = "\x8a\x40", [0x58be] = "\x8a\x41",
  [0x58bf] = "\x8a\x42", [0x58c0] = "\x8a\x43", [0x58c1] = "\xde\xc6",
  [0x58c2] = "\xdb\xbc", [0x58c3] = "\x8a\x44", [0x58c4] = "\xd1\xd9",
  [0x58c5] = "\x8a\x45", [0x58c6] = "\x8a\x46", [0x58c7] = "\xc6\xe6",
  [0x58c8] = "\xc4\xce", [0x58c9] = "\xb7\xee", [0x58ca] = "\x8a\x47",
  [0x58cb] = "\xb7\xdc", [0x58cc] = "\x8a\x48", [0x58cd] = "\x8a\x49",
  [0x58ce] = "\xbf\xfc", [0x58cf] = "\xd7\xe0", [0x58d0] = "\x8a\x4a",
  [0x58d1] = "\xc6\xf5", [0x58d2] = "\x8a\x4b", [0x58d3] = "\x8a\x4c",
  [0x58d4] = "\xb1\xbc", [0x58d5] = "\xde\xc8", [0x58d6] = "\xbd\xb1",
  [0x58d7] = "\xcc\xd7", [0x58d8] = "\xde\xca", [0x58d9] = "\x8a\x4d",
  [0x58da] = "\xde\xc9", [0x58db] = "\x8a\x4e", [0x58dc] = "\x8a\x4f",
  [0x58dd] = "\x8a\x50", [0x58de] = "\x8a\x51", [0x58df] = "\x8a\x52",
  [0x58e0] = "\xb5\xec", [0x58e1] = "\x8a\x53", [0x58e2] = "\xc9\xdd",
  [0x58e3] = "\x8a\x54", [0x58e4] = "\x8a\x55", [0x58e5] = "\xb0\xc2",
  [0x58e6] = "\x8a\x56", [0x58e7] = "\x8a\x57", [0x58e8] = "\x8a\x58",
  [0x58e9] = "\x8a\x59", [0x58ea] = "\x8a\x5a", [0x58eb] = "\x8a\x5b",
  [0x58ec] = "\x8a\x5c", [0x58ed] = "\x8a\x5d", [0x58ee] = "\x8a\x5e",
  [0x58ef] = "\x8a\x5f", [0x58f0] = "\x8a\x60", [0x58f1] = "\x8a\x61",
  [0x58f2] = "\x8a\x62", [0x58f3] = "\xc5\xae", [0x58f4] = "\xc5\xab",
  [0x58f5] = "\x8a\x63", [0x58f6] = "\xc4\xcc", [0x58f7] = "\x8a\x64",
  [0x58f8] = "\xbc\xe9", [0x58f9] = "\xcb\xfd", [0x58fa] = "\x8a\x65",
  [0x58fb] = "\x8a\x66", [0x58fc] = "\x8a\x67", [0x58fd] = "\xba\xc3",
  [0x58fe] = "\x8a\x68", [0x58ff] = "\x8a\x69", [0x5900] = "\x8a\x6a",
  [0x5901] = "\xe5\xf9", [0x5902] = "\xc8\xe7", [0x5903] = "\xe5\xfa",
  [0x5904] = "\xcd\xfd", [0x5905] = "\x8a\x6b", [0x5906] = "\xd7\xb1",
  [0x5907] = "\xb8\xbe", [0x5908] = "\xc2\xe8", [0x5909] = "\x8a\x6c",
  [0x590a] = "\xc8\xd1", [0x590b] = "\x8a\x6d", [0x590c] = "\x8a\x6e",
  [0x590d] = "\xe5\xfb", [0x590e] = "\x8a\x6f", [0x590f] = "\x8a\x70",
  [0x5910] = "\x8a\x71", [0x5911] = "\x8a\x72", [0x5912] = "\xb6\xca",
  [0x5913] = "\xbc\xcb", [0x5914] = "\x8a\x73", [0x5915] = "\x8a\x74",
  [0x5916] = "\xd1\xfd", [0x5917] = "\xe6\xa1", [0x5918] = "\x8a\x75",
  [0x5919] = "\xc3\xee", [0x591a] = "\x8a\x76", [0x591b] = "\x8a\x77",
  [0x591c] = "\x8a\x78", [0x591d] = "\x8a\x79", [0x591e] = "\xe6\xa4",
  [0x591f] = "\x8a\x7a", [0x5920] = "\x8a\x7b", [0x5921] = "\x8a\x7c",
  [0x5922] = "\x8a\x7d", [0x5923] = "\xe5\xfe", [0x5924] = "\xe6\xa5",
  [0x5925] = "\xcd\xd7", [0x5926] = "\x8a\x7e", [0x5927] = "\x8a\x80",
  [0x5928] = "\xb7\xc1", [0x5929] = "\xe5\xfc", [0x592a] = "\xe5\xfd",
  [0x592b] = "\xe6\xa3", [0x592c] = "\x8a\x81", [0x592d] = "\x8a\x82",
  [0x592e] = "\xc4\xdd", [0x592f] = "\xe6\xa8", [0x5930] = "\x8a\x83",
  [0x5931] = "\x8a\x84", [0x5932] = "\xe6\xa7", [0x5933] = "\x8a\x85",
  [0x5934] = "\x8a\x86", [0x5935] = "\x8a\x87", [0x5936] = "\x8a\x88",
  [0x5937] = "\x8a\x89", [0x5938] = "\x8a\x8a", [0x5939] = "\xc3\xc3",
  [0x593a] = "\x8a\x8b", [0x593b] = "\xc6\xde", [0x593c] = "\x8a\x8c",
  [0x593d] = "\x8a\x8d", [0x593e] = "\xe6\xaa", [0x593f] = "\x8a\x8e",
  [0x5940] = "\x8a\x8f", [0x5941] = "\x8a\x90", [0x5942] = "\x8a\x91",
  [0x5943] = "\x8a\x92", [0x5944] = "\x8a\x93", [0x5945] = "\x8a\x94",
  [0x5946] = "\xc4\xb7", [0x5947] = "\x8a\x95", [0x5948] = "\x8a\x96",
  [0x5949] = "\x8a\x97", [0x594a] = "\xe6\xa2", [0x594b] = "\xca\xbc",
  [0x594c] = "\x8a\x98", [0x594d] = "\x8a\x99", [0x594e] = "\x8a\x9a",
  [0x594f] = "\x8a\x9b", [0x5950] = "\xbd\xe3", [0x5951] = "\xb9\xc3",
  [0x5952] = "\xe6\xa6", [0x5953] = "\xd0\xd5", [0x5954] = "\xce\xaf",
  [0x5955] = "\x8a\x9c", [0x5956] = "\x8a\x9d", [0x5957] = "\xe6\xa9",
  [0x5958] = "\xe6\xb0", [0x5959] = "\x8a\x9e", [0x595a] = "\xd2\xa6",
  [0x595b] = "\x8a\x9f", [0x595c] = "\xbd\xaa", [0x595d] = "\xe6\xad",
  [0x595e] = "\x8a\xa0", [0x595f] = "\x8a\xa1", [0x5960] = "\x8a\xa2",
  [0x5961] = "\x8a\xa3", [0x5962] = "\x8a\xa4", [0x5963] = "\xe6\xaf",
  [0x5964] = "\x8a\xa5", [0x5965] = "\xc0\xd1", [0x5966] = "\x8a\xa6",
  [0x5967] = "\x8a\xa7", [0x5968] = "\xd2\xcc", [0x5969] = "\x8a\xa8",
  [0x596a] = "\x8a\xa9", [0x596b] = "\x8a\xaa", [0x596c] = "\xbc\xa7",
  [0x596d] = "\x8a\xab", [0x596e] = "\x8a\xac", [0x596f] = "\x8a\xad",
  [0x5970] = "\x8a\xae", [0x5971] = "\x8a\xaf", [0x5972] = "\x8a\xb0",
  [0x5973] = "\x8a\xb1", [0x5974] = "\x8a\xb2", [0x5975] = "\x8a\xb3",
  [0x5976] = "\x8a\xb4", [0x5977] = "\x8a\xb5", [0x5978] = "\x8a\xb6",
  [0x5979] = "\xe6\xb1", [0x597a] = "\x8a\xb7", [0x597b] = "\xd2\xf6",
  [0x597c] = "\x8a\xb8", [0x597d] = "\x8a\xb9", [0x597e] = "\x8a\xba",
  [0x597f] = "\xd7\xcb", [0x5980] = "\x8a\xbb", [0x5981] = "\xcd\xfe",
  [0x5982] = "\x8a\xbc", [0x5983] = "\xcd\xde", [0x5984] = "\xc2\xa6",
  [0x5985] = "\xe6\xab", [0x5986] = "\xe6\xac", [0x5987] = "\xbd\xbf",
  [0x5988] = "\xe6\xae", [0x5989] = "\xe6\xb3", [0x598a] = "\x8a\xbd",
  [0x598b] = "\x8a\xbe", [0x598c] = "\xe6\xb2", [0x598d] = "\x8a\xbf",
  [0x598e] = "\x8a\xc0", [0x598f] = "\x8a\xc1", [0x5990] = "\x8a\xc2",
  [0x5991] = "\xe6\xb6", [0x5992] = "\x8a\xc3", [0x5993] = "\xe6\xb8",
  [0x5994] = "\x8a\xc4", [0x5995] = "\x8a\xc5", [0x5996] = "\x8a\xc6",
  [0x5997] = "\x8a\xc7", [0x5998] = "\xc4\xef", [0x5999] = "\x8a\xc8",
  [0x599a] = "\x8a\xc9", [0x599b] = "\x8a\xca", [0x599c] = "\xc4\xc8",
  [0x599d] = "\x8a\xcb", [0x599e] = "\x8a\xcc", [0x599f] = "\xbe\xea",
  [0x59a0] = "\xc9\xef", [0x59a1] = "\x8a\xcd", [0x59a2] = "\x8a\xce",
  [0x59a3] = "\xe6\xb7", [0x59a4] = "\x8a\xcf", [0x59a5] = "\xb6\xf0",
  [0x59a6] = "\x8a\xd0", [0x59a7] = "\x8a\xd1", [0x59a8] = "\x8a\xd2",
  [0x59a9] = "\xc3\xe4", [0x59aa] = "\x8a\xd3", [0x59ab] = "\x8a\xd4",
  [0x59ac] = "\x8a\xd5", [0x59ad] = "\x8a\xd6", [0x59ae] = "\x8a\xd7",
  [0x59af] = "\x8a\xd8", [0x59b0] = "\x8a\xd9", [0x59b1] = "\xd3\xe9",
  [0x59b2] = "\xe6\xb4", [0x59b3] = "\x8a\xda", [0x59b4] = "\xe6\xb5",
  [0x59b5] = "\x8a\xdb", [0x59b6] = "\xc8\xa2", [0x59b7] = "\x8a\xdc",
  [0x59b8] = "\x8a\xdd", [0x59b9] = "\x8a\xde", [0x59ba] = "\x8a\xdf",
  [0x59bb] = "\x8a\xe0", [0x59bc] = "\xe6\xbd", [0x59bd] = "\x8a\xe1",
  [0x59be] = "\x8a\xe2", [0x59bf] = "\x8a\xe3", [0x59c0] = "\xe6\xb9",
  [0x59c1] = "\x8a\xe4", [0x59c2] = "\x8a\xe5", [0x59c3] = "\x8a\xe6",
  [0x59c4] = "\x8a\xe7", [0x59c5] = "\x8a\xe8", [0x59c6] = "\xc6\xc5",
  [0x59c7] = "\x8a\xe9", [0x59c8] = "\x8a\xea", [0x59c9] = "\xcd\xf1",
  [0x59ca] = "\xe6\xbb", [0x59cb] = "\x8a\xeb", [0x59cc] = "\x8a\xec",
  [0x59cd] = "\x8a\xed", [0x59ce] = "\x8a\xee", [0x59cf] = "\x8a\xef",
  [0x59d0] = "\x8a\xf0", [0x59d1] = "\x8a\xf1", [0x59d2] = "\x8a\xf2",
  [0x59d3] = "\x8a\xf3", [0x59d4] = "\x8a\xf4", [0x59d5] = "\xe6\xbc",
  [0x59d6] = "\x8a\xf5", [0x59d7] = "\x8a\xf6", [0x59d8] = "\x8a\xf7",
  [0x59d9] = "\x8a\xf8", [0x59da] = "\xbb\xe9", [0x59db] = "\x8a\xf9",
  [0x59dc] = "\x8a\xfa", [0x59dd] = "\x8a\xfb", [0x59de] = "\x8a\xfc",
  [0x59df] = "\x8a\xfd", [0x59e0] = "\x8a\xfe", [0x59e1] = "\x8b\x40",
  [0x59e2] = "\xe6\xbe", [0x59e3] = "\x8b\x41", [0x59e4] = "\x8b\x42",
  [0x59e5] = "\x8b\x43", [0x59e6] = "\x8b\x44", [0x59e7] = "\xe6\xba",
  [0x59e8] = "\x8b\x45", [0x59e9] = "\x8b\x46", [0x59ea] = "\xc0\xb7",
  [0x59eb] = "\x8b\x47", [0x59ec] = "\x8b\x48", [0x59ed] = "\x8b\x49",
  [0x59ee] = "\x8b\x4a", [0x59ef] = "\x8b\x4b", [0x59f0] = "\x8b\x4c",
  [0x59f1] = "\x8b\x4d", [0x59f2] = "\x8b\x4e", [0x59f3] = "\x8b\x4f",
  [0x59f4] = "\xd3\xa4", [0x59f5] = "\xe6\xbf", [0x59f6] = "\xc9\xf4",
  [0x59f7] = "\xe6\xc3", [0x59f8] = "\x8b\x50", [0x59f9] = "\x8b\x51",
  [0x59fa] = "\xe6\xc4", [0x59fb] = "\x8b\x52", [0x59fc] = "\x8b\x53",
  [0x59fd] = "\x8b\x54", [0x59fe] = "\x8b\x55", [0x59ff] = "\xd0\xf6",
  [0x5a00] = "\x8b\x56", [0x5a01] = "\x8b\x57", [0x5a02] = "\x8b\x58",
  [0x5a03] = "\x8b\x59", [0x5a04] = "\x8b\x5a", [0x5a05] = "\x8b\x5b",
  [0x5a06] = "\x8b\x5c", [0x5a07] = "\x8b\x5d", [0x5a08] = "\x8b\x5e",
  [0x5a09] = "\x8b\x5f", [0x5a0a] = "\x8b\x60", [0x5a0b] = "\x8b\x61",
  [0x5a0c] = "\x8b\x62", [0x5a0d] = "\x8b\x63", [0x5a0e] = "\x8b\x64",
  [0x5a0f] = "\x8b\x65", [0x5a10] = "\x8b\x66", [0x5a11] = "\x8b\x67",
  [0x5a12] = "\xc3\xbd", [0x5a13] = "\x8b\x68", [0x5a14] = "\x8b\x69",
  [0x5a15] = "\x8b\x6a", [0x5a16] = "\x8b\x6b", [0x5a17] = "\x8b\x6c",
  [0x5a18] = "\x8b\x6d", [0x5a19] = "\x8b\x6e", [0x5a1a] = "\xc3\xc4",
  [0x5a1b] = "\xe6\xc2", [0x5a1c] = "\x8b\x6f", [0x5a1d] = "\x8b\x70",
  [0x5a1e] = "\x8b\x71", [0x5a1f] = "\x8b\x72", [0x5a20] = "\x8b\x73",
  [0x5a21] = "\x8b\x74", [0x5a22] = "\x8b\x75", [0x5a23] = "\x8b\x76",
  [0x5a24] = "\x8b\x77", [0x5a25] = "\x8b\x78", [0x5a26] = "\x8b\x79",
  [0x5a27] = "\x8b\x7a", [0x5a28] = "\x8b\x7b", [0x5a29] = "\x8b\x7c",
  [0x5a2a] = "\xe6\xc1", [0x5a2b] = "\x8b\x7d", [0x5a2c] = "\x8b\x7e",
  [0x5a2d] = "\x8b\x80", [0x5a2e] = "\x8b\x81", [0x5a2f] = "\x8b\x82",
  [0x5a30] = "\x8b\x83", [0x5a31] = "\x8b\x84", [0x5a32] = "\xe6\xc7",
  [0x5a33] = "\xcf\xb1", [0x5a34] = "\x8b\x85", [0x5a35] = "\xeb\xf4",
  [0x5a36] = "\x8b\x86", [0x5a37] = "\x8b\x87", [0x5a38] = "\xe6\xca",
  [0x5a39] = "\x8b\x88", [0x5a3a] = "\x8b\x89", [0x5a3b] = "\x8b\x8a",
  [0x5a3c] = "\x8b\x8b", [0x5a3d] = "\x8b\x8c", [0x5a3e] = "\xe6\xc5",
  [0x5a3f] = "\x8b\x8d", [0x5a40] = "\x8b\x8e", [0x5a41] = "\xbc\xde",
  [0x5a42] = "\xc9\xa9", [0x5a43] = "\x8b\x8f", [0x5a44] = "\x8b\x90",
  [0x5a45] = "\x8b\x91", [0x5a46] = "\x8b\x92", [0x5a47] = "\x8b\x93",
  [0x5a48] = "\x8b\x94", [0x5a49] = "\xbc\xb5", [0x5a4a] = "\x8b\x95",
  [0x5a4b] = "\x8b\x96", [0x5a4c] = "\xcf\xd3", [0x5a4d] = "\x8b\x97",
  [0x5a4e] = "\x8b\x98", [0x5a4f] = "\x8b\x99", [0x5a50] = "\x8b\x9a",
  [0x5a51] = "\x8b\x9b", [0x5a52] = "\xe6\xc8", [0x5a53] = "\x8b\x9c",
  [0x5a54] = "\xe6\xc9", [0x5a55] = "\x8b\x9d", [0x5a56] = "\xe6\xce",
  [0x5a57] = "\x8b\x9e", [0x5a58] = "\xe6\xd0", [0x5a59] = "\x8b\x9f",
  [0x5a5a] = "\x8b\xa0", [0x5a5b] = "\x8b\xa1", [0x5a5c] = "\xe6\xd1",
  [0x5a5d] = "\x8b\xa2", [0x5a5e] = "\x8b\xa3", [0x5a5f] = "\x8b\xa4",
  [0x5a60] = "\xe6\xcb", [0x5a61] = "\xb5\xd5", [0x5a62] = "\x8b\xa5",
  [0x5a63] = "\xe6\xcc", [0x5a64] = "\x8b\xa6", [0x5a65] = "\x8b\xa7",
  [0x5a66] = "\xe6\xcf", [0x5a67] = "\x8b\xa8", [0x5a68] = "\x8b\xa9",
  [0x5a69] = "\xc4\xdb", [0x5a6a] = "\x8b\xaa", [0x5a6b] = "\xe6\xc6",
  [0x5a6c] = "\x8b\xab", [0x5a6d] = "\x8b\xac", [0x5a6e] = "\x8b\xad",
  [0x5a6f] = "\x8b\xae", [0x5a70] = "\x8b\xaf", [0x5a71] = "\xe6\xcd",
  [0x5a72] = "\x8b\xb0", [0x5a73] = "\x8b\xb1", [0x5a74] = "\x8b\xb2",
  [0x5a75] = "\x8b\xb3", [0x5a76] = "\x8b\xb4", [0x5a77] = "\x8b\xb5",
  [0x5a78] = "\x8b\xb6", [0x5a79] = "\x8b\xb7", [0x5a7a] = "\x8b\xb8",
  [0x5a7b] = "\x8b\xb9", [0x5a7c] = "\x8b\xba", [0x5a7d] = "\x8b\xbb",
  [0x5a7e] = "\x8b\xbc", [0x5a7f] = "\x8b\xbd", [0x5a80] = "\x8b\xbe",
  [0x5a81] = "\x8b\xbf", [0x5a82] = "\x8b\xc0", [0x5a83] = "\x8b\xc1",
  [0x5a84] = "\x8b\xc2", [0x5a85] = "\x8b\xc3", [0x5a86] = "\x8b\xc4",
  [0x5a87] = "\x8b\xc5", [0x5a88] = "\x8b\xc6", [0x5a89] = "\xe6\xd2",
  [0x5a8a] = "\x8b\xc7", [0x5a8b] = "\x8b\xc8", [0x5a8c] = "\x8b\xc9",
  [0x5a8d] = "\x8b\xca", [0x5a8e] = "\x8b\xcb", [0x5a8f] = "\x8b\xcc",
  [0x5a90] = "\x8b\xcd", [0x5a91] = "\x8b\xce", [0x5a92] = "\x8b\xcf",
  [0x5a93] = "\x8b\xd0", [0x5a94] = "\x8b\xd1", [0x5a95] = "\x8b\xd2",
  [0x5a96] = "\xe6\xd4", [0x5a97] = "\xe6\xd3", [0x5a98] = "\x8b\xd3",
  [0x5a99] = "\x8b\xd4", [0x5a9a] = "\x8b\xd5", [0x5a9b] = "\x8b\xd6",
  [0x5a9c] = "\x8b\xd7", [0x5a9d] = "\x8b\xd8", [0x5a9e] = "\x8b\xd9",
  [0x5a9f] = "\x8b\xda", [0x5aa0] = "\x8b\xdb", [0x5aa1] = "\x8b\xdc",
  [0x5aa2] = "\x8b\xdd", [0x5aa3] = "\x8b\xde", [0x5aa4] = "\x8b\xdf",
  [0x5aa5] = "\x8b\xe0", [0x5aa6] = "\x8b\xe1", [0x5aa7] = "\x8b\xe2",
  [0x5aa8] = "\x8b\xe3", [0x5aa9] = "\x8b\xe4", [0x5aaa] = "\x8b\xe5",
  [0x5aab] = "\x8b\xe6", [0x5aac] = "\x8b\xe7", [0x5aad] = "\x8b\xe8",
  [0x5aae] = "\x8b\xe9", [0x5aaf] = "\x8b\xea", [0x5ab0] = "\x8b\xeb",
  [0x5ab1] = "\x8b\xec", [0x5ab2] = "\xe6\xd5", [0x5ab3] = "\x8b\xed",
  [0x5ab4] = "\xd9\xf8", [0x5ab5] = "\x8b\xee", [0x5ab6] = "\x8b\xef",
  [0x5ab7] = "\xe6\xd6", [0x5ab8] = "\x8b\xf0", [0x5ab9] = "\x8b\xf1",
  [0x5aba] = "\x8b\xf2", [0x5abb] = "\x8b\xf3", [0x5abc] = "\x8b\xf4",
  [0x5abd] = "\x8b\xf5", [0x5abe] = "\x8b\xf6", [0x5abf] = "\x8b\xf7",
  [0x5ac0] = "\xe6\xd7", [0x5ac1] = "\x8b\xf8", [0x5ac2] = "\x8b\xf9",
  [0x5ac3] = "\x8b\xfa", [0x5ac4] = "\x8b\xfb", [0x5ac5] = "\x8b\xfc",
  [0x5ac6] = "\x8b\xfd", [0x5ac7] = "\x8b\xfe", [0x5ac8] = "\x8c\x40",
  [0x5ac9] = "\x8c\x41", [0x5aca] = "\x8c\x42", [0x5acb] = "\x8c\x43",
  [0x5acc] = "\x8c\x44", [0x5acd] = "\x8c\x45", [0x5ace] = "\x8c\x46",
  [0x5acf] = "\x8c\x47", [0x5ad0] = "\xd7\xd3", [0x5ad1] = "\xe6\xdd",
  [0x5ad2] = "\x8c\x48", [0x5ad3] = "\xe6\xde", [0x5ad4] = "\xbf\xd7",
  [0x5ad5] = "\xd4\xd0", [0x5ad6] = "\x8c\x49", [0x5ad7] = "\xd7\xd6",
  [0x5ad8] = "\xb4\xe6", [0x5ad9] = "\xcb\xef", [0x5ada] = "\xe6\xda",
  [0x5adb] = "\xd8\xc3", [0x5adc] = "\xd7\xce", [0x5add] = "\xd0\xa2",
  [0x5ade] = "\x8c\x4a", [0x5adf] = "\xc3\xcf", [0x5ae0] = "\x8c\x4b",
  [0x5ae1] = "\x8c\x4c", [0x5ae2] = "\xe6\xdf", [0x5ae3] = "\xbc\xbe",
  [0x5ae4] = "\xb9\xc2", [0x5ae5] = "\xe6\xdb", [0x5ae6] = "\xd1\xa7",
  [0x5ae7] = "\x8c\x4d", [0x5ae8] = "\x8c\x4e", [0x5ae9] = "\xba\xa2",
  [0x5aea] = "\xc2\xcf", [0x5aeb] = "\x8c\x4f", [0x5aec] = "\xd8\xab",
  [0x5aed] = "\x8c\x50", [0x5aee] = "\x8c\x51", [0x5aef] = "\x8c\x52",
  [0x5af0] = "\xca\xeb", [0x5af1] = "\xe5\xee", [0x5af2] = "\x8c\x53",
  [0x5af3] = "\xe6\xdc", [0x5af4] = "\x8c\x54", [0x5af5] = "\xb7\xf5",
  [0x5af6] = "\x8c\x55", [0x5af7] = "\x8c\x56", [0x5af8] = "\x8c\x57",
  [0x5af9] = "\x8c\x58", [0x5afa] = "\xc8\xe6", [0x5afb] = "\x8c\x59",
  [0x5afc] = "\x8c\x5a", [0x5afd] = "\xc4\xf5", [0x5afe] = "\x8c\x5b",
  [0x5aff] = "\x8c\x5c", [0x5b00] = "\xe5\xb2", [0x5b01] = "\xc4\xfe",
  [0x5b02] = "\x8c\x5d", [0x5b03] = "\xcb\xfc", [0x5b04] = "\xe5\xb3",
  [0x5b05] = "\xd5\xac", [0x5b06] = "\x8c\x5e", [0x5b07] = "\xd3\xee",
  [0x5b08] = "\xca\xd8", [0x5b09] = "\xb0\xb2", [0x5b0a] = "\x8c\x5f",
  [0x5b0b] = "\xcb\xce", [0x5b0c] = "\xcd\xea", [0x5b0d] = "\x8c\x60",
  [0x5b0e] = "\x8c\x61", [0x5b0f] = "\xba\xea", [0x5b10] = "\x8c\x62",
  [0x5b11] = "\x8c\x63", [0x5b12] = "\x8c\x64", [0x5b13] = "\xe5\xb5",
  [0x5b14] = "\x8c\x65", [0x5b15] = "\xe5\xb4", [0x5b16] = "\x8c\x66",
  [0x5b17] = "\xd7\xda", [0x5b18] = "\xb9\xd9", [0x5b19] = "\xd6\xe6",
  [0x5b1a] = "\xb6\xa8", [0x5b1b] = "\xcd\xf0", [0x5b1c] = "\xd2\xcb",
  [0x5b1d] = "\xb1\xa6", [0x5b1e] = "\xca\xb5", [0x5b1f] = "\x8c\x67",
  [0x5b20] = "\xb3\xe8", [0x5b21] = "\xc9\xf3", [0x5b22] = "\xbf\xcd",
  [0x5b23] = "\xd0\xfb", [0x5b24] = "\xca\xd2", [0x5b25] = "\xe5\xb6",
  [0x5b26] = "\xbb\xc2", [0x5b27] = "\x8c\x68", [0x5b28] = "\x8c\x69",
  [0x5b29] = "\x8c\x6a", [0x5b2a] = "\xcf\xdc", [0x5b2b] = "\xb9\xac",
  [0x5b2c] = "\x8c\x6b", [0x5b2d] = "\x8c\x6c", [0x5b2e] = "\x8c\x6d",
  [0x5b2f] = "\x8c\x6e", [0x5b30] = "\xd4\xd7", [0x5b31] = "\x8c\x6f",
  [0x5b32] = "\x8c\x70", [0x5b33] = "\xba\xa6", [0x5b34] = "\xd1\xe7",
  [0x5b35] = "\xcf\xfc", [0x5b36] = "\xbc\xd2", [0x5b37] = "\x8c\x71",
  [0x5b38] = "\xe5\xb7", [0x5b39] = "\xc8\xdd", [0x5b3a] = "\x8c\x72",
  [0x5b3b] = "\x8c\x73", [0x5b3c] = "\x8c\x74", [0x5b3d] = "\xbf\xed",
  [0x5b3e] = "\xb1\xf6", [0x5b3f] = "\xcb\xde", [0x5b40] = "\x8c\x75",
  [0x5b41] = "\x8c\x76", [0x5b42] = "\xbc\xc5", [0x5b43] = "\x8c\x77",
  [0x5b44] = "\xbc\xc4", [0x5b45] = "\xd2\xfa", [0x5b46] = "\xc3\xdc",
  [0x5b47] = "\xbf\xdc", [0x5b48] = "\x8c\x78", [0x5b49] = "\x8c\x79",
  [0x5b4a] = "\x8c\x7a", [0x5b4b] = "\x8c\x7b", [0x5b4c] = "\xb8\xbb",
  [0x5b4d] = "\x8c\x7c", [0x5b4e] = "\x8c\x7d", [0x5b4f] = "\x8c\x7e",
  [0x5b50] = "\xc3\xc2", [0x5b51] = "\x8c\x80", [0x5b52] = "\xba\xae",
  [0x5b53] = "\xd4\xa2", [0x5b54] = "\x8c\x81", [0x5b55] = "\x8c\x82",
  [0x5b56] = "\x8c\x83", [0x5b57] = "\x8c\x84", [0x5b58] = "\x8c\x85",
  [0x5b59] = "\x8c\x86", [0x5b5a] = "\x8c\x87", [0x5b5b] = "\x8c\x88",
  [0x5b5c] = "\x8c\x89", [0x5b5d] = "\xc7\xde", [0x5b5e] = "\xc4\xaf",
  [0x5b5f] = "\xb2\xec", [0x5b60] = "\x8c\x8a", [0x5b61] = "\xb9\xd1",
  [0x5b62] = "\x8c\x8b", [0x5b63] = "\x8c\x8c", [0x5b64] = "\xe5\xbb",
  [0x5b65] = "\xc1\xc8", [0x5b66] = "\x8c\x8d", [0x5b67] = "\x8c\x8e",
  [0x5b68] = "\xd5\xaf", [0x5b69] = "\x8c\x8f", [0x5b6a] = "\x8c\x90",
  [0x5b6b] = "\x8c\x91", [0x5b6c] = "\x8c\x92", [0x5b6d] = "\x8c\x93",
  [0x5b6e] = "\xe5\xbc", [0x5b6f] = "\x8c\x94", [0x5b70] = "\xe5\xbe",
  [0x5b71] = "\x8c\x95", [0x5b72] = "\x8c\x96", [0x5b73] = "\x8c\x97",
  [0x5b74] = "\x8c\x98", [0x5b75] = "\x8c\x99", [0x5b76] = "\x8c\x9a",
  [0x5b77] = "\x8c\x9b", [0x5b78] = "\xb4\xe7", [0x5b79] = "\xb6\xd4",
  [0x5b7a] = "\xcb\xc2", [0x5b7b] = "\xd1\xb0", [0x5b7c] = "\xb5\xbc",
  [0x5b7d] = "\x8c\x9c", [0x5b7e] = "\x8c\x9d", [0x5b7f] = "\xca\xd9",
  [0x5b80] = "\x8c\x9e", [0x5b81] = "\xb7\xe2", [0x5b82] = "\x8c\x9f",
  [0x5b83] = "\x8c\xa0", [0x5b84] = "\xc9\xe4", [0x5b85] = "\x8c\xa1",
  [0x5b86] = "\xbd\xab", [0x5b87] = "\x8c\xa2", [0x5b88] = "\x8c\xa3",
  [0x5b89] = "\xce\xbe", [0x5b8a] = "\xd7\xf0", [0x5b8b] = "\x8c\xa4",
  [0x5b8c] = "\x8c\xa5", [0x5b8d] = "\x8c\xa6", [0x5b8e] = "\x8c\xa7",
  [0x5b8f] = "\xd0\xa1", [0x5b90] = "\x8c\xa8", [0x5b91] = "\xc9\xd9",
  [0x5b92] = "\x8c\xa9", [0x5b93] = "\x8c\xaa", [0x5b94] = "\xb6\xfb",
  [0x5b95] = "\xe6\xd8", [0x5b96] = "\xbc\xe2", [0x5b97] = "\x8c\xab",
  [0x5b98] = "\xb3\xbe", [0x5b99] = "\x8c\xac", [0x5b9a] = "\xc9\xd0",
  [0x5b9b] = "\x8c\xad", [0x5b9c] = "\xe6\xd9", [0x5b9d] = "\xb3\xa2",
  [0x5b9e] = "\x8c\xae", [0x5b9f] = "\x8c\xaf", [0x5ba0] = "\x8c\xb0",
  [0x5ba1] = "\x8c\xb1", [0x5ba2] = "\xde\xcc", [0x5ba3] = "\x8c\xb2",
  [0x5ba4] = "\xd3\xc8", [0x5ba5] = "\xde\xcd", [0x5ba6] = "\x8c\xb3",
  [0x5ba7] = "\xd2\xa2", [0x5ba8] = "\x8c\xb4", [0x5ba9] = "\x8c\xb5",
  [0x5baa] = "\x8c\xb6", [0x5bab] = "\x8c\xb7", [0x5bac] = "\xde\xce",
  [0x5bad] = "\x8c\xb8", [0x5bae] = "\x8c\xb9", [0x5baf] = "\x8c\xba",
  [0x5bb0] = "\x8c\xbb", [0x5bb1] = "\xbe\xcd", [0x5bb2] = "\x8c\xbc",
  [0x5bb3] = "\x8c\xbd", [0x5bb4] = "\xde\xcf", [0x5bb5] = "\x8c\xbe",
  [0x5bb6] = "\x8c\xbf", [0x5bb7] = "\x8c\xc0", [0x5bb8] = "\xca\xac",
  [0x5bb9] = "\xd2\xfc", [0x5bba] = "\xb3\xdf", [0x5bbb] = "\xe5\xea",
  [0x5bbc] = "\xc4\xe1", [0x5bbd] = "\xbe\xa1", [0x5bbe] = "\xce\xb2",
  [0x5bbf] = "\xc4\xf2", [0x5bc0] = "\xbe\xd6", [0x5bc1] = "\xc6\xa8",
  [0x5bc2] = "\xb2\xe3", [0x5bc3] = "\x8c\xc1", [0x5bc4] = "\x8c\xc2",
  [0x5bc5] = "\xbe\xd3", [0x5bc6] = "\x8c\xc3", [0x5bc7] = "\x8c\xc4",
  [0x5bc8] = "\xc7\xfc", [0x5bc9] = "\xcc\xeb", [0x5bca] = "\xbd\xec",
  [0x5bcb] = "\xce\xdd", [0x5bcc] = "\x8c\xc5", [0x5bcd] = "\x8c\xc6",
  [0x5bce] = "\xca\xba", [0x5bcf] = "\xc6\xc1", [0x5bd0] = "\xe5\xec",
  [0x5bd1] = "\xd0\xbc", [0x5bd2] = "\x8c\xc7", [0x5bd3] = "\x8c\xc8",
  [0x5bd4] = "\x8c\xc9", [0x5bd5] = "\xd5\xb9", [0x5bd6] = "\x8c\xca",
  [0x5bd7] = "\x8c\xcb", [0x5bd8] = "\x8c\xcc", [0x5bd9] = "\xe5\xed",
  [0x5bda] = "\x8c\xcd", [0x5bdb] = "\x8c\xce", [0x5bdc] = "\x8c\xcf",
  [0x5bdd] = "\x8c\xd0", [0x5bde] = "\xca\xf4", [0x5bdf] = "\x8c\xd1",
  [0x5be0] = "\xcd\xc0", [0x5be1] = "\xc2\xc5", [0x5be2] = "\x8c\xd2",
  [0x5be3] = "\xe5\xef", [0x5be4] = "\x8c\xd3", [0x5be5] = "\xc2\xc4",
  [0x5be6] = "\xe5\xf0", [0x5be7] = "\x8c\xd4", [0x5be8] = "\x8c\xd5",
  [0x5be9] = "\x8c\xd6", [0x5bea] = "\x8c\xd7", [0x5beb] = "\x8c\xd8",
  [0x5bec] = "\x8c\xd9", [0x5bed] = "\x8c\xda", [0x5bee] = "\xe5\xf8",
  [0x5bef] = "\xcd\xcd", [0x5bf0] = "\x8c\xdb", [0x5bf1] = "\xc9\xbd",
  [0x5bf2] = "\x8c\xdc", [0x5bf3] = "\x8c\xdd", [0x5bf4] = "\x8c\xde",
  [0x5bf5] = "\x8c\xdf", [0x5bf6] = "\x8c\xe0", [0x5bf7] = "\x8c\xe1",
  [0x5bf8] = "\x8c\xe2", [0x5bf9] = "\xd2\xd9", [0x5bfa] = "\xe1\xa8",
  [0x5bfb] = "\x8c\xe3", [0x5bfc] = "\x8c\xe4", [0x5bfd] = "\x8c\xe5",
  [0x5bfe] = "\x8c\xe6", [0x5bff] = "\xd3\xec", [0x5c00] = "\x8c\xe7",
  [0x5c01] = "\xcb\xea", [0x5c02] = "\xc6\xf1", [0x5c03] = "\x8c\xe8",
  [0x5c04] = "\x8c\xe9", [0x5c05] = "\x8c\xea", [0x5c06] = "\x8c\xeb",
  [0x5c07] = "\x8c\xec", [0x5c08] = "\xe1\xac", [0x5c09] = "\x8c\xed",
  [0x5c0a] = "\x8c\xee", [0x5c0b] = "\x8c\xef", [0x5c0c] = "\xe1\xa7",
  [0x5c0d] = "\xe1\xa9", [0x5c0e] = "\x8c\xf0", [0x5c0f] = "\x8c\xf1",
  [0x5c10] = "\xe1\xaa", [0x5c11] = "\xe1\xaf", [0x5c12] = "\x8c\xf2",
  [0x5c13] = "\x8c\xf3", [0x5c14] = "\xb2\xed", [0x5c15] = "\x8c\xf4",
  [0x5c16] = "\xe1\xab", [0x5c17] = "\xb8\xda", [0x5c18] = "\xe1\xad",
  [0x5c19] = "\xe1\xae", [0x5c1a] = "\xe1\xb0", [0x5c1b] = "\xb5\xba",
  [0x5c1c] = "\xe1\xb1", [0x5c1d] = "\x8c\xf5", [0x5c1e] = "\x8c\xf6",
  [0x5c1f] = "\x8c\xf7", [0x5c20] = "\x8c\xf8", [0x5c21] = "\x8c\xf9",
  [0x5c22] = "\xe1\xb3", [0x5c23] = "\xe1\xb8", [0x5c24] = "\x8c\xfa",
  [0x5c25] = "\x8c\xfb", [0x5c26] = "\x8c\xfc", [0x5c27] = "\x8c\xfd",
  [0x5c28] = "\x8c\xfe", [0x5c29] = "\xd1\xd2", [0x5c2a] = "\x8d\x40",
  [0x5c2b] = "\xe1\xb6", [0x5c2c] = "\xe1\xb5", [0x5c2d] = "\xc1\xeb",
  [0x5c2e] = "\x8d\x41", [0x5c2f] = "\x8d\x42", [0x5c30] = "\x8d\x43",
  [0x5c31] = "\xe1\xb7", [0x5c32] = "\x8d\x44", [0x5c33] = "\xd4\xc0",
  [0x5c34] = "\x8d\x45", [0x5c35] = "\xe1\xb2", [0x5c36] = "\x8d\x46",
  [0x5c37] = "\xe1\xba", [0x5c38] = "\xb0\xb6", [0x5c39] = "\x8d\x47",
  [0x5c3a] = "\x8d\x48", [0x5c3b] = "\x8d\x49", [0x5c3c] = "\x8d\x4a",
  [0x5c3d] = "\xe1\xb4", [0x5c3e] = "\x8d\x4b", [0x5c3f] = "\xbf\xf9",
  [0x5c40] = "\x8d\x4c", [0x5c41] = "\xe1\xb9", [0x5c42] = "\x8d\x4d",
  [0x5c43] = "\x8d\x4e", [0x5c44] = "\xe1\xbb", [0x5c45] = "\x8d\x4f",
  [0x5c46] = "\x8d\x50", [0x5c47] = "\x8d\x51", [0x5c48] = "\x8d\x52",
  [0x5c49] = "\x8d\x53", [0x5c4a] = "\x8d\x54", [0x5c4b] = "\xe1\xbe",
  [0x5c4c] = "\x8d\x55", [0x5c4d] = "\x8d\x56", [0x5c4e] = "\x8d\x57",
  [0x5c4f] = "\x8d\x58", [0x5c50] = "\x8d\x59", [0x5c51] = "\x8d\x5a",
  [0x5c52] = "\xe1\xbc", [0x5c53] = "\x8d\x5b", [0x5c54] = "\x8d\x5c",
  [0x5c55] = "\x8d\x5d", [0x5c56] = "\x8d\x5e", [0x5c57] = "\x8d\x5f",
  [0x5c58] = "\x8d\x60", [0x5c59] = "\xd6\xc5", [0x5c5a] = "\x8d\x61",
  [0x5c5b] = "\x8d\x62", [0x5c5c] = "\x8d\x63", [0x5c5d] = "\x8d\x64",
  [0x5c5e] = "\x8d\x65", [0x5c5f] = "\x8d\x66", [0x5c60] = "\x8d\x67",
  [0x5c61] = "\xcf\xbf", [0x5c62] = "\x8d\x68", [0x5c63] = "\x8d\x69",
  [0x5c64] = "\xe1\xbd", [0x5c65] = "\xe1\xbf", [0x5c66] = "\xc2\xcd",
  [0x5c67] = "\x8d\x6a", [0x5c68] = "\xb6\xeb", [0x5c69] = "\x8d\x6b",
  [0x5c6a] = "\xd3\xf8", [0x5c6b] = "\x8d\x6c", [0x5c6c] = "\x8d\x6d",
  [0x5c6d] = "\xc7\xcd", [0x5c6e] = "\x8d\x6e", [0x5c6f] = "\x8d\x6f",
  [0x5c70] = "\xb7\xe5", [0x5c71] = "\x8d\x70", [0x5c72] = "\x8d\x71",
  [0x5c73] = "\x8d\x72", [0x5c74] = "\x8d\x73", [0x5c75] = "\x8d\x74",
  [0x5c76] = "\x8d\x75", [0x5c77] = "\x8d\x76", [0x5c78] = "\x8d\x77",
  [0x5c79] = "\x8d\x78", [0x5c7a] = "\x8d\x79", [0x5c7b] = "\xbe\xfe",
  [0x5c7c] = "\x8d\x7a", [0x5c7d] = "\x8d\x7b", [0x5c7e] = "\x8d\x7c",
  [0x5c7f] = "\x8d\x7d", [0x5c80] = "\x8d\x7e", [0x5c81] = "\x8d\x80",
  [0x5c82] = "\xe1\xc0", [0x5c83] = "\xe1\xc1", [0x5c84] = "\x8d\x81",
  [0x5c85] = "\x8d\x82", [0x5c86] = "\xe1\xc7", [0x5c87] = "\xb3\xe7",
  [0x5c88] = "\x8d\x83", [0x5c89] = "\x8d\x84", [0x5c8a] = "\x8d\x85",
  [0x5c8b] = "\x8d\x86", [0x5c8c] = "\x8d\x87", [0x5c8d] = "\x8d\x88",
  [0x5c8e] = "\xc6\xe9", [0x5c8f] = "\x8d\x89", [0x5c90] = "\x8d\x8a",
  [0x5c91] = "\x8d\x8b", [0x5c92] = "\x8d\x8c", [0x5c93] = "\x8d\x8d",
  [0x5c94] = "\xb4\xde", [0x5c95] = "\x8d\x8e", [0x5c96] = "\xd1\xc2",
  [0x5c97] = "\x8d\x8f", [0x5c98] = "\x8d\x90", [0x5c99] = "\x8d\x91",
  [0x5c9a] = "\x8d\x92", [0x5c9b] = "\xe1\xc8", [0x5c9c] = "\x8d\x93",
  [0x5c9d] = "\x8d\x94", [0x5c9e] = "\xe1\xc6", [0x5c9f] = "\x8d\x95",
  [0x5ca0] = "\x8d\x96", [0x5ca1] = "\x8d\x97", [0x5ca2] = "\x8d\x98",
  [0x5ca3] = "\x8d\x99", [0x5ca4] = "\xe1\xc5", [0x5ca5] = "\x8d\x9a",
  [0x5ca6] = "\xe1\xc3", [0x5ca7] = "\xe1\xc2", [0x5ca8] = "\x8d\x9b",
  [0x5ca9] = "\xb1\xc0", [0x5caa] = "\x8d\x9c", [0x5cab] = "\x8d\x9d",
  [0x5cac] = "\x8d\x9e", [0x5cad] = "\xd5\xb8", [0x5cae] = "\xe1\xc4",
  [0x5caf] = "\x8d\x9f", [0x5cb0] = "\x8d\xa0", [0x5cb1] = "\x8d\xa1",
  [0x5cb2] = "\x8d\xa2", [0x5cb3] = "\x8d\xa3", [0x5cb4] = "\xe1\xcb",
  [0x5cb5] = "\x8d\xa4", [0x5cb6] = "\x8d\xa5", [0x5cb7] = "\x8d\xa6",
  [0x5cb8] = "\x8d\xa7", [0x5cb9] = "\x8d\xa8", [0x5cba] = "\x8d\xa9",
  [0x5cbb] = "\x8d\xaa", [0x5cbc] = "\x8d\xab", [0x5cbd] = "\xe1\xcc",
  [0x5cbe] = "\xe1\xca", [0x5cbf] = "\x8d\xac", [0x5cc0] = "\x8d\xad",
  [0x5cc1] = "\x8d\xae", [0x5cc2] = "\x8d\xaf", [0x5cc3] = "\x8d\xb0",
  [0x5cc4] = "\x8d\xb1", [0x5cc5] = "\x8d\xb2", [0x5cc6] = "\x8d\xb3",
  [0x5cc7] = "\xef\xfa", [0x5cc8] = "\x8d\xb4", [0x5cc9] = "\x8d\xb5",
  [0x5cca] = "\xe1\xd3", [0x5ccb] = "\xe1\xd2", [0x5ccc] = "\xc7\xb6",
  [0x5ccd] = "\x8d\xb6", [0x5cce] = "\x8d\xb7", [0x5ccf] = "\x8d\xb8",
  [0x5cd0] = "\x8d\xb9", [0x5cd1] = "\x8d\xba", [0x5cd2] = "\x8d\xbb",
  [0x5cd3] = "\x8d\xbc", [0x5cd4] = "\x8d\xbd", [0x5cd5] = "\x8d\xbe",
  [0x5cd6] = "\x8d\xbf", [0x5cd7] = "\x8d\xc0", [0x5cd8] = "\xe1\xc9",
  [0x5cd9] = "\x8d\xc1", [0x5cda] = "\x8d\xc2", [0x5cdb] = "\xe1\xce",
  [0x5cdc] = "\x8d\xc3", [0x5cdd] = "\xe1\xd0", [0x5cde] = "\x8d\xc4",
  [0x5cdf] = "\x8d\xc5", [0x5ce0] = "\x8d\xc6", [0x5ce1] = "\x8d\xc7",
  [0x5ce2] = "\x8d\xc8", [0x5ce3] = "\x8d\xc9", [0x5ce4] = "\x8d\xca",
  [0x5ce5] = "\x8d\xcb", [0x5ce6] = "\x8d\xcc", [0x5ce7] = "\x8d\xcd",
  [0x5ce8] = "\x8d\xce", [0x5ce9] = "\xe1\xd4", [0x5cea] = "\x8d\xcf",
  [0x5ceb] = "\xe1\xd1", [0x5cec] = "\xe1\xcd", [0x5ced] = "\x8d\xd0",
  [0x5cee] = "\x8d\xd1", [0x5cef] = "\xe1\xcf", [0x5cf0] = "\x8d\xd2",
  [0x5cf1] = "\x8d\xd3", [0x5cf2] = "\x8d\xd4", [0x5cf3] = "\x8d\xd5",
  [0x5cf4] = "\xe1\xd5", [0x5cf5] = "\x8d\xd6", [0x5cf6] = "\x8d\xd7",
  [0x5cf7] = "\x8d\xd8", [0x5cf8] = "\x8d\xd9", [0x5cf9] = "\x8d\xda",
  [0x5cfa] = "\x8d\xdb", [0x5cfb] = "\x8d\xdc", [0x5cfc] = "\x8d\xdd",
  [0x5cfd] = "\x8d\xde", [0x5cfe] = "\x8d\xdf", [0x5cff] = "\x8d\xe0",
  [0x5d00] = "\x8d\xe1", [0x5d01] = "\x8d\xe2", [0x5d02] = "\xe1\xd6",
  [0x5d03] = "\x8d\xe3", [0x5d04] = "\x8d\xe4", [0x5d05] = "\x8d\xe5",
  [0x5d06] = "\x8d\xe6", [0x5d07] = "\x8d\xe7", [0x5d08] = "\x8d\xe8",
  [0x5d09] = "\x8d\xe9", [0x5d0a] = "\x8d\xea", [0x5d0b] = "\x8d\xeb",
  [0x5d0c] = "\x8d\xec", [0x5d0d] = "\x8d\xed", [0x5d0e] = "\x8d\xee",
  [0x5d0f] = "\x8d\xef", [0x5d10] = "\x8d\xf0", [0x5d11] = "\x8d\xf1",
  [0x5d12] = "\x8d\xf2", [0x5d13] = "\x8d\xf3", [0x5d14] = "\x8d\xf4",
  [0x5d15] = "\x8d\xf5", [0x5d16] = "\x8d\xf6", [0x5d17] = "\x8d\xf7",
  [0x5d18] = "\x8d\xf8", [0x5d19] = "\xe1\xd7", [0x5d1a] = "\x8d\xf9",
  [0x5d1b] = "\x8d\xfa", [0x5d1c] = "\x8d\xfb", [0x5d1d] = "\xe1\xd8",
  [0x5d1e] = "\x8d\xfc", [0x5d1f] = "\x8d\xfd", [0x5d20] = "\x8d\xfe",
  [0x5d21] = "\x8e\x40", [0x5d22] = "\x8e\x41", [0x5d23] = "\x8e\x42",
  [0x5d24] = "\x8e\x43", [0x5d25] = "\x8e\x44", [0x5d26] = "\x8e\x45",
  [0x5d27] = "\x8e\x46", [0x5d28] = "\x8e\x47", [0x5d29] = "\x8e\x48",
  [0x5d2a] = "\x8e\x49", [0x5d2b] = "\x8e\x4a", [0x5d2c] = "\x8e\x4b",
  [0x5d2d] = "\x8e\x4c", [0x5d2e] = "\x8e\x4d", [0x5d2f] = "\x8e\x4e",
  [0x5d30] = "\x8e\x4f", [0x5d31] = "\x8e\x50", [0x5d32] = "\x8e\x51",
  [0x5d33] = "\x8e\x52", [0x5d34] = "\x8e\x53", [0x5d35] = "\x8e\x54",
  [0x5d36] = "\x8e\x55", [0x5d37] = "\xe1\xda", [0x5d38] = "\x8e\x56",
  [0x5d39] = "\x8e\x57", [0x5d3a] = "\x8e\x58", [0x5d3b] = "\x8e\x59",
  [0x5d3c] = "\x8e\x5a", [0x5d3d] = "\x8e\x5b", [0x5d3e] = "\x8e\x5c",
  [0x5d3f] = "\x8e\x5d", [0x5d40] = "\x8e\x5e", [0x5d41] = "\x8e\x5f",
  [0x5d42] = "\x8e\x60", [0x5d43] = "\x8e\x61", [0x5d44] = "\x8e\x62",
  [0x5d45] = "\xe1\xdb", [0x5d46] = "\x8e\x63", [0x5d47] = "\x8e\x64",
  [0x5d48] = "\x8e\x65", [0x5d49] = "\x8e\x66", [0x5d4a] = "\x8e\x67",
  [0x5d4b] = "\x8e\x68", [0x5d4c] = "\x8e\x69", [0x5d4d] = "\xce\xa1",
  [0x5d4e] = "\x8e\x6a", [0x5d4f] = "\x8e\x6b", [0x5d50] = "\x8e\x6c",
  [0x5d51] = "\x8e\x6d", [0x5d52] = "\x8e\x6e", [0x5d53] = "\x8e\x6f",
  [0x5d54] = "\x8e\x70", [0x5d55] = "\x8e\x71", [0x5d56] = "\x8e\x72",
  [0x5d57] = "\x8e\x73", [0x5d58] = "\x8e\x74", [0x5d59] = "\x8e\x75",
  [0x5d5a] = "\x8e\x76", [0x5d5b] = "\xe7\xdd", [0x5d5c] = "\x8e\x77",
  [0x5d5d] = "\xb4\xa8", [0x5d5e] = "\xd6\xdd", [0x5d5f] = "\x8e\x78",
  [0x5d60] = "\x8e\x79", [0x5d61] = "\xd1\xb2", [0x5d62] = "\xb3\xb2",
  [0x5d63] = "\x8e\x7a", [0x5d64] = "\x8e\x7b", [0x5d65] = "\xb9\xa4",
  [0x5d66] = "\xd7\xf3", [0x5d67] = "\xc7\xc9", [0x5d68] = "\xbe\xde",
  [0x5d69] = "\xb9\xae", [0x5d6a] = "\x8e\x7c", [0x5d6b] = "\xce\xd7",
  [0x5d6c] = "\x8e\x7d", [0x5d6d] = "\x8e\x7e", [0x5d6e] = "\xb2\xee",
  [0x5d6f] = "\xdb\xcf", [0x5d70] = "\x8e\x80", [0x5d71] = "\xbc\xba",
  [0x5d72] = "\xd2\xd1", [0x5d73] = "\xcb\xc8", [0x5d74] = "\xb0\xcd",
  [0x5d75] = "\x8e\x81", [0x5d76] = "\x8e\x82", [0x5d77] = "\xcf\xef",
  [0x5d78] = "\x8e\x83", [0x5d79] = "\x8e\x84", [0x5d7a] = "\x8e\x85",
  [0x5d7b] = "\x8e\x86", [0x5d7c] = "\x8e\x87", [0x5d7d] = "\xd9\xe3",
  [0x5d7e] = "\xbd\xed", [0x5d7f] = "\x8e\x88", [0x5d80] = "\x8e\x89",
  [0x5d81] = "\xb1\xd2", [0x5d82] = "\xca\xd0", [0x5d83] = "\xb2\xbc",
  [0x5d84] = "\x8e\x8a", [0x5d85] = "\xcb\xa7", [0x5d86] = "\xb7\xab",
  [0x5d87] = "\x8e\x8b", [0x5d88] = "\xca\xa6", [0x5d89] = "\x8e\x8c",
  [0x5d8a] = "\x8e\x8d", [0x5d8b] = "\x8e\x8e", [0x5d8c] = "\xcf\xa3",
  [0x5d8d] = "\x8e\x8f", [0x5d8e] = "\x8e\x90", [0x5d8f] = "\xe0\xf8",
  [0x5d90] = "\xd5\xca", [0x5d91] = "\xe0\xfb", [0x5d92] = "\x8e\x91",
  [0x5d93] = "\x8e\x92", [0x5d94] = "\xe0\xfa", [0x5d95] = "\xc5\xc1",
  [0x5d96] = "\xcc\xfb", [0x5d97] = "\x8e\x93", [0x5d98] = "\xc1\xb1",
  [0x5d99] = "\xe0\xf9", [0x5d9a] = "\xd6\xe3", [0x5d9b] = "\xb2\xaf",
  [0x5d9c] = "\xd6\xc4", [0x5d9d] = "\xb5\xdb", [0x5d9e] = "\x8e\x94",
  [0x5d9f] = "\x8e\x95", [0x5da0] = "\x8e\x96", [0x5da1] = "\x8e\x97",
  [0x5da2] = "\x8e\x98", [0x5da3] = "\x8e\x99", [0x5da4] = "\x8e\x9a",
  [0x5da5] = "\x8e\x9b", [0x5da6] = "\xb4\xf8", [0x5da7] = "\xd6\xa1",
  [0x5da8] = "\x8e\x9c", [0x5da9] = "\x8e\x9d", [0x5daa] = "\x8e\x9e",
  [0x5dab] = "\x8e\x9f", [0x5dac] = "\x8e\xa0", [0x5dad] = "\xcf\xaf",
  [0x5dae] = "\xb0\xef", [0x5daf] = "\x8e\xa1", [0x5db0] = "\x8e\xa2",
  [0x5db1] = "\xe0\xfc", [0x5db2] = "\x8e\xa3", [0x5db3] = "\x8e\xa4",
  [0x5db4] = "\x8e\xa5", [0x5db5] = "\x8e\xa6", [0x5db6] = "\x8e\xa7",
  [0x5db7] = "\xe1\xa1", [0x5db8] = "\xb3\xa3", [0x5db9] = "\x8e\xa8",
  [0x5dba] = "\x8e\xa9", [0x5dbb] = "\xe0\xfd", [0x5dbc] = "\xe0\xfe",
  [0x5dbd] = "\xc3\xb1", [0x5dbe] = "\x8e\xaa", [0x5dbf] = "\x8e\xab",
  [0x5dc0] = "\x8e\xac", [0x5dc1] = "\x8e\xad", [0x5dc2] = "\xc3\xdd",
  [0x5dc3] = "\x8e\xae", [0x5dc4] = "\xe1\xa2", [0x5dc5] = "\xb7\xf9",
  [0x5dc6] = "\x8e\xaf", [0x5dc7] = "\x8e\xb0", [0x5dc8] = "\x8e\xb1",
  [0x5dc9] = "\x8e\xb2", [0x5dca] = "\x8e\xb3", [0x5dcb] = "\x8e\xb4",
  [0x5dcc] = "\xbb\xcf", [0x5dcd] = "\x8e\xb5", [0x5dce] = "\x8e\xb6",
  [0x5dcf] = "\x8e\xb7", [0x5dd0] = "\x8e\xb8", [0x5dd1] = "\x8e\xb9",
  [0x5dd2] = "\x8e\xba", [0x5dd3] = "\x8e\xbb", [0x5dd4] = "\xe1\xa3",
  [0x5dd5] = "\xc4\xbb", [0x5dd6] = "\x8e\xbc", [0x5dd7] = "\x8e\xbd",
  [0x5dd8] = "\x8e\xbe", [0x5dd9] = "\x8e\xbf", [0x5dda] = "\x8e\xc0",
  [0x5ddb] = "\xe1\xa4", [0x5ddc] = "\x8e\xc1", [0x5ddd] = "\x8e\xc2",
  [0x5dde] = "\xe1\xa5", [0x5ddf] = "\x8e\xc3", [0x5de0] = "\x8e\xc4",
  [0x5de1] = "\xe1\xa6", [0x5de2] = "\xb4\xb1", [0x5de3] = "\x8e\xc5",
  [0x5de4] = "\x8e\xc6", [0x5de5] = "\x8e\xc7", [0x5de6] = "\x8e\xc8",
  [0x5de7] = "\x8e\xc9", [0x5de8] = "\x8e\xca", [0x5de9] = "\x8e\xcb",
  [0x5dea] = "\x8e\xcc", [0x5deb] = "\x8e\xcd", [0x5dec] = "\x8e\xce",
  [0x5ded] = "\x8e\xcf", [0x5dee] = "\x8e\xd0", [0x5def] = "\x8e\xd1",
  [0x5df0] = "\x8e\xd2", [0x5df1] = "\x8e\xd3", [0x5df2] = "\xb8\xc9",
  [0x5df3] = "\xc6\xbd", [0x5df4] = "\xc4\xea", [0x5df5] = "\x8e\xd4",
  [0x5df6] = "\xb2\xa2", [0x5df7] = "\x8e\xd5", [0x5df8] = "\xd0\xd2",
  [0x5df9] = "\x8e\xd6", [0x5dfa] = "\xe7\xdb", [0x5dfb] = "\xbb\xc3",
  [0x5dfc] = "\xd3\xd7", [0x5dfd] = "\xd3\xc4", [0x5dfe] = "\x8e\xd7",
  [0x5dff] = "\xb9\xe3", [0x5e00] = "\xe2\xcf", [0x5e01] = "\x8e\xd8",
  [0x5e02] = "\x8e\xd9", [0x5e03] = "\x8e\xda", [0x5e04] = "\xd7\xaf",
  [0x5e05] = "\x8e\xdb", [0x5e06] = "\xc7\xec", [0x5e07] = "\xb1\xd3",
  [0x5e08] = "\x8e\xdc", [0x5e09] = "\x8e\xdd", [0x5e0a] = "\xb4\xb2",
  [0x5e0b] = "\xe2\xd1", [0x5e0c] = "\x8e\xde", [0x5e0d] = "\x8e\xdf",
  [0x5e0e] = "\x8e\xe0", [0x5e0f] = "\xd0\xf2", [0x5e10] = "\xc2\xae",
  [0x5e11] = "\xe2\xd0", [0x5e12] = "\x8e\xe1", [0x5e13] = "\xbf\xe2",
  [0x5e14] = "\xd3\xa6", [0x5e15] = "\xb5\xd7", [0x5e16] = "\xe2\xd2",
  [0x5e17] = "\xb5\xea", [0x5e18] = "\x8e\xe2", [0x5e19] = "\xc3\xed",
  [0x5e1a] = "\xb8\xfd", [0x5e1b] = "\x8e\xe3", [0x5e1c] = "\xb8\xae",
  [0x5e1d] = "\x8e\xe4", [0x5e1e] = "\xc5\xd3", [0x5e1f] = "\xb7\xcf",
  [0x5e20] = "\xe2\xd4", [0x5e21] = "\x8e\xe5", [0x5e22] = "\x8e\xe6",
  [0x5e23] = "\x8e\xe7", [0x5e24] = "\x8e\xe8", [0x5e25] = "\xe2\xd3",
  [0x5e26] = "\xb6\xc8", [0x5e27] = "\xd7\xf9", [0x5e28] = "\x8e\xe9",
  [0x5e29] = "\x8e\xea", [0x5e2a] = "\x8e\xeb", [0x5e2b] = "\x8e\xec",
  [0x5e2c] = "\x8e\xed", [0x5e2d] = "\xcd\xa5", [0x5e2e] = "\x8e\xee",
  [0x5e2f] = "\x8e\xef", [0x5e30] = "\x8e\xf0", [0x5e31] = "\x8e\xf1",
  [0x5e32] = "\x8e\xf2", [0x5e33] = "\xe2\xd8", [0x5e34] = "\x8e\xf3",
  [0x5e35] = "\xe2\xd6", [0x5e36] = "\xca\xfc", [0x5e37] = "\xbf\xb5",
  [0x5e38] = "\xd3\xb9", [0x5e39] = "\xe2\xd5", [0x5e3a] = "\x8e\xf4",
  [0x5e3b] = "\x8e\xf5", [0x5e3c] = "\x8e\xf6", [0x5e3d] = "\x8e\xf7",
  [0x5e3e] = "\xe2\xd7", [0x5e3f] = "\x8e\xf8", [0x5e40] = "\x8e\xf9",
  [0x5e41] = "\x8e\xfa", [0x5e42] = "\x8e\xfb", [0x5e43] = "\x8e\xfc",
  [0x5e44] = "\x8e\xfd", [0x5e45] = "\x8e\xfe", [0x5e46] = "\x8f\x40",
  [0x5e47] = "\x8f\x41", [0x5e48] = "\x8f\x42", [0x5e49] = "\xc1\xae",
  [0x5e4a] = "\xc0\xc8", [0x5e4b] = "\x8f\x43", [0x5e4c] = "\x8f\x44",
  [0x5e4d] = "\x8f\x45", [0x5e4e] = "\x8f\x46", [0x5e4f] = "\x8f\x47",
  [0x5e50] = "\x8f\x48", [0x5e51] = "\xe2\xdb", [0x5e52] = "\xe2\xda",
  [0x5e53] = "\xc0\xaa", [0x5e54] = "\x8f\x49", [0x5e55] = "\x8f\x4a",
  [0x5e56] = "\xc1\xce", [0x5e57] = "\x8f\x4b", [0x5e58] = "\x8f\x4c",
  [0x5e59] = "\x8f\x4d", [0x5e5a] = "\x8f\x4e", [0x5e5b] = "\xe2\xdc",
  [0x5e5c] = "\x8f\x4f", [0x5e5d] = "\x8f\x50", [0x5e5e] = "\x8f\x51",
  [0x5e5f] = "\x8f\x52", [0x5e60] = "\x8f\x53", [0x5e61] = "\x8f\x54",
  [0x5e62] = "\x8f\x55", [0x5e63] = "\x8f\x56", [0x5e64] = "\x8f\x57",
  [0x5e65] = "\x8f\x58", [0x5e66] = "\x8f\x59", [0x5e67] = "\x8f\x5a",
  [0x5e68] = "\xe2\xdd", [0x5e69] = "\x8f\x5b", [0x5e6a] = "\xe2\xde",
  [0x5e6b] = "\x8f\x5c", [0x5e6c] = "\x8f\x5d", [0x5e6d] = "\x8f\x5e",
  [0x5e6e] = "\x8f\x5f", [0x5e6f] = "\x8f\x60", [0x5e70] = "\x8f\x61",
  [0x5e71] = "\x8f\x62", [0x5e72] = "\x8f\x63", [0x5e73] = "\x8f\x64",
  [0x5e74] = "\xdb\xc8", [0x5e75] = "\x8f\x65", [0x5e76] = "\xd1\xd3",
  [0x5e77] = "\xcd\xa2", [0x5e78] = "\x8f\x66", [0x5e79] = "\x8f\x67",
  [0x5e7a] = "\xbd\xa8", [0x5e7b] = "\x8f\x68", [0x5e7c] = "\x8f\x69",
  [0x5e7d] = "\x8f\x6a", [0x5e7e] = "\xde\xc3", [0x5e7f] = "\xd8\xa5",
  [0x5e80] = "\xbf\xaa", [0x5e81] = "\xdb\xcd", [0x5e82] = "\xd2\xec",
  [0x5e83] = "\xc6\xfa", [0x5e84] = "\xc5\xaa", [0x5e85] = "\x8f\x6b",
  [0x5e86] = "\x8f\x6c", [0x5e87] = "\x8f\x6d", [0x5e88] = "\xde\xc4",
  [0x5e89] = "\x8f\x6e", [0x5e8a] = "\xb1\xd7", [0x5e8b] = "\xdf\xae",
  [0x5e8c] = "\x8f\x6f", [0x5e8d] = "\x8f\x70", [0x5e8e] = "\x8f\x71",
  [0x5e8f] = "\xca\xbd", [0x5e90] = "\x8f\x72", [0x5e91] = "\xdf\xb1",
  [0x5e92] = "\x8f\x73", [0x5e93] = "\xb9\xad", [0x5e94] = "\x8f\x74",
  [0x5e95] = "\xd2\xfd", [0x5e96] = "\x8f\x75", [0x5e97] = "\xb8\xa5",
  [0x5e98] = "\xba\xeb", [0x5e99] = "\x8f\x76", [0x5e9a] = "\x8f\x77",
  [0x5e9b] = "\xb3\xda", [0x5e9c] = "\x8f\x78", [0x5e9d] = "\x8f\x79",
  [0x5e9e] = "\x8f\x7a", [0x5e9f] = "\xb5\xdc", [0x5ea0] = "\xd5\xc5",
  [0x5ea1] = "\x8f\x7b", [0x5ea2] = "\x8f\x7c", [0x5ea3] = "\x8f\x7d",
  [0x5ea4] = "\x8f\x7e", [0x5ea5] = "\xc3\xd6", [0x5ea6] = "\xcf\xd2",
  [0x5ea7] = "\xbb\xa1", [0x5ea8] = "\x8f\x80", [0x5ea9] = "\xe5\xf3",
  [0x5eaa] = "\xe5\xf2", [0x5eab] = "\x8f\x81", [0x5eac] = "\x8f\x82",
  [0x5ead] = "\xe5\xf4", [0x5eae] = "\x8f\x83", [0x5eaf] = "\xcd\xe4",
  [0x5eb0] = "\x8f\x84", [0x5eb1] = "\xc8\xf5", [0x5eb2] = "\x8f\x85",
  [0x5eb3] = "\x8f\x86", [0x5eb4] = "\x8f\x87", [0x5eb5] = "\x8f\x88",
  [0x5eb6] = "\x8f\x89", [0x5eb7] = "\x8f\x8a", [0x5eb8] = "\x8f\x8b",
  [0x5eb9] = "\xb5\xaf", [0x5eba] = "\xc7\xbf", [0x5ebb] = "\x8f\x8c",
  [0x5ebc] = "\xe5\xf6", [0x5ebd] = "\x8f\x8d", [0x5ebe] = "\x8f\x8e",
  [0x5ebf] = "\x8f\x8f", [0x5ec0] = "\xec\xb0", [0x5ec1] = "\x8f\x90",
  [0x5ec2] = "\x8f\x91", [0x5ec3] = "\x8f\x92", [0x5ec4] = "\x8f\x93",
  [0x5ec5] = "\x8f\x94", [0x5ec6] = "\x8f\x95", [0x5ec7] = "\x8f\x96",
  [0x5ec8] = "\x8f\x97", [0x5ec9] = "\x8f\x98", [0x5eca] = "\x8f\x99",
  [0x5ecb] = "\x8f\x9a", [0x5ecc] = "\x8f\x9b", [0x5ecd] = "\x8f\x9c",
  [0x5ece] = "\x8f\x9d", [0x5ecf] = "\x8f\x9e", [0x5ed0] = "\xe5\xe6",
  [0x5ed1] = "\x8f\x9f", [0x5ed2] = "\xb9\xe9", [0x5ed3] = "\xb5\xb1",
  [0x5ed4] = "\x8f\xa0", [0x5ed5] = "\xc2\xbc", [0x5ed6] = "\xe5\xe8",
  [0x5ed7] = "\xe5\xe7", [0x5ed8] = "\xe5\xe9", [0x5ed9] = "\x8f\xa1",
  [0x5eda] = "\x8f\xa2", [0x5edb] = "\x8f\xa3", [0x5edc] = "\x8f\xa4",
  [0x5edd] = "\xd2\xcd", [0x5ede] = "\x8f\xa5", [0x5edf] = "\x8f\xa6",
  [0x5ee0] = "\x8f\xa7", [0x5ee1] = "\xe1\xea", [0x5ee2] = "\xd0\xce",
  [0x5ee3] = "\x8f\xa8", [0x5ee4] = "\xcd\xae", [0x5ee5] = "\x8f\xa9",
  [0x5ee6] = "\xd1\xe5", [0x5ee7] = "\x8f\xaa", [0x5ee8] = "\x8f\xab",
  [0x5ee9] = "\xb2\xca", [0x5eea] = "\xb1\xeb", [0x5eeb] = "\x8f\xac",
  [0x5eec] = "\xb1\xf2", [0x5eed] = "\xc5\xed", [0x5eee] = "\x8f\xad",
  [0x5eef] = "\x8f\xae", [0x5ef0] = "\xd5\xc3", [0x5ef1] = "\xd3\xb0",
  [0x5ef2] = "\x8f\xaf", [0x5ef3] = "\xe1\xdc", [0x5ef4] = "\x8f\xb0",
  [0x5ef5] = "\x8f\xb1", [0x5ef6] = "\x8f\xb2", [0x5ef7] = "\xe1\xdd",
  [0x5ef8] = "\x8f\xb3", [0x5ef9] = "\xd2\xdb", [0x5efa] = "\x8f\xb4",
  [0x5efb] = "\xb3\xb9", [0x5efc] = "\xb1\xcb", [0x5efd] = "\x8f\xb5",
  [0x5efe] = "\x8f\xb6", [0x5eff] = "\x8f\xb7", [0x5f00] = "\xcd\xf9",
  [0x5f01] = "\xd5\xf7", [0x5f02] = "\xe1\xde", [0x5f03] = "\x8f\xb8",
  [0x5f04] = "\xbe\xb6", [0x5f05] = "\xb4\xfd", [0x5f06] = "\x8f\xb9",
  [0x5f07] = "\xe1\xdf", [0x5f08] = "\xba\xdc", [0x5f09] = "\xe1\xe0",
  [0x5f0a] = "\xbb\xb2", [0x5f0b] = "\xc2\xc9", [0x5f0c] = "\xe1\xe1",
  [0x5f0d] = "\x8f\xba", [0x5f0e] = "\x8f\xbb", [0x5f0f] = "\x8f\xbc",
  [0x5f10] = "\xd0\xec", [0x5f11] = "\x8f\xbd", [0x5f12] = "\xcd\xbd",
  [0x5f13] = "\x8f\xbe", [0x5f14] = "\x8f\xbf", [0x5f15] = "\xe1\xe2",
  [0x5f16] = "\x8f\xc0", [0x5f17] = "\xb5\xc3", [0x5f18] = "\xc5\xc7",
  [0x5f19] = "\xe1\xe3", [0x5f1a] = "\x8f\xc1", [0x5f1b] = "\x8f\xc2",
  [0x5f1c] = "\xe1\xe4", [0x5f1d] = "\x8f\xc3", [0x5f1e] = "\x8f\xc4",
  [0x5f1f] = "\x8f\xc5", [0x5f20] = "\x8f\xc6", [0x5f21] = "\xd3\xf9",
  [0x5f22] = "\x8f\xc7", [0x5f23] = "\x8f\xc8", [0x5f24] = "\x8f\xc9",
  [0x5f25] = "\x8f\xca", [0x5f26] = "\x8f\xcb", [0x5f27] = "\x8f\xcc",
  [0x5f28] = "\xe1\xe5", [0x5f29] = "\x8f\xcd", [0x5f2a] = "\xd1\xad",
  [0x5f2b] = "\x8f\xce", [0x5f2c] = "\x8f\xcf", [0x5f2d] = "\xe1\xe6",
  [0x5f2e] = "\xce\xa2", [0x5f2f] = "\x8f\xd0", [0x5f30] = "\x8f\xd1",
  [0x5f31] = "\x8f\xd2", [0x5f32] = "\x8f\xd3", [0x5f33] = "\x8f\xd4",
  [0x5f34] = "\x8f\xd5", [0x5f35] = "\xe1\xe7", [0x5f36] = "\x8f\xd6",
  [0x5f37] = "\xb5\xc2", [0x5f38] = "\x8f\xd7", [0x5f39] = "\x8f\xd8",
  [0x5f3a] = "\x8f\xd9", [0x5f3b] = "\x8f\xda", [0x5f3c] = "\xe1\xe8",
  [0x5f3d] = "\xbb\xd5", [0x5f3e] = "\x8f\xdb", [0x5f3f] = "\x8f\xdc",
  [0x5f40] = "\x8f\xdd", [0x5f41] = "\x8f\xde", [0x5f42] = "\x8f\xdf",
  [0x5f43] = "\xd0\xc4", [0x5f44] = "\xe2\xe0", [0x5f45] = "\xb1\xd8",
  [0x5f46] = "\xd2\xe4", [0x5f47] = "\x8f\xe0", [0x5f48] = "\x8f\xe1",
  [0x5f49] = "\xe2\xe1", [0x5f4a] = "\x8f\xe2", [0x5f4b] = "\x8f\xe3",
  [0x5f4c] = "\xbc\xc9", [0x5f4d] = "\xc8\xcc", [0x5f4e] = "\x8f\xe4",
  [0x5f4f] = "\xe2\xe3", [0x5f50] = "\xec\xfe", [0x5f51] = "\xec\xfd",
  [0x5f52] = "\xdf\xaf", [0x5f53] = "\x8f\xe5", [0x5f54] = "\x8f\xe6",
  [0x5f55] = "\x8f\xe7", [0x5f56] = "\xe2\xe2", [0x5f57] = "\xd6\xbe",
  [0x5f58] = "\xcd\xfc", [0x5f59] = "\xc3\xa6", [0x5f5a] = "\x8f\xe8",
  [0x5f5b] = "\x8f\xe9", [0x5f5c] = "\x8f\xea", [0x5f5d] = "\xe3\xc3",
  [0x5f5e] = "\x8f\xeb", [0x5f5f] = "\x8f\xec", [0x5f60] = "\xd6\xd2",
  [0x5f61] = "\xe2\xe7", [0x5f62] = "\x8f\xed", [0x5f63] = "\x8f\xee",
  [0x5f64] = "\xe2\xe8", [0x5f65] = "\x8f\xef", [0x5f66] = "\x8f\xf0",
  [0x5f67] = "\xd3\xc7", [0x5f68] = "\x8f\xf1", [0x5f69] = "\x8f\xf2",
  [0x5f6a] = "\xe2\xec", [0x5f6b] = "\xbf\xec", [0x5f6c] = "\x8f\xf3",
  [0x5f6d] = "\xe2\xed", [0x5f6e] = "\xe2\xe5", [0x5f6f] = "\x8f\xf4",
  [0x5f70] = "\x8f\xf5", [0x5f71] = "\xb3\xc0", [0x5f72] = "\x8f\xf6",
  [0x5f73] = "\x8f\xf7", [0x5f74] = "\x8f\xf8", [0x5f75] = "\xc4\xee",
  [0x5f76] = "\x8f\xf9", [0x5f77] = "\x8f\xfa", [0x5f78] = "\xe2\xee",
  [0x5f79] = "\x8f\xfb", [0x5f7a] = "\x8f\xfc", [0x5f7b] = "\xd0\xc3",
  [0x5f7c] = "\x8f\xfd", [0x5f7d] = "\xba\xf6", [0x5f7e] = "\xe2\xe9",
  [0x5f7f] = "\xb7\xde", [0x5f80] = "\xbb\xb3", [0x5f81] = "\xcc\xac",
  [0x5f82] = "\xcb\xcb", [0x5f83] = "\xe2\xe4", [0x5f84] = "\xe2\xe6",
  [0x5f85] = "\xe2\xea", [0x5f86] = "\xe2\xeb", [0x5f87] = "\x8f\xfe",
  [0x5f88] = "\x90\x40", [0x5f89] = "\x90\x41", [0x5f8a] = "\xe2\xf7",
  [0x5f8b] = "\x90\x42", [0x5f8c] = "\x90\x43", [0x5f8d] = "\xe2\xf4",
  [0x5f8e] = "\xd4\xf5", [0x5f8f] = "\xe2\xf3", [0x5f90] = "\x90\x44",
  [0x5f91] = "\x90\x45", [0x5f92] = "\xc5\xad", [0x5f93] = "\x90\x46",
  [0x5f94] = "\xd5\xfa", [0x5f95] = "\xc5\xc2", [0x5f96] = "\xb2\xc0",
  [0x5f97] = "\x90\x47", [0x5f98] = "\x90\x48", [0x5f99] = "\xe2\xef",
  [0x5f9a] = "\x90\x49", [0x5f9b] = "\xe2\xf2", [0x5f9c] = "\xc1\xaf",
  [0x5f9d] = "\xcb\xbc", [0x5f9e] = "\x90\x4a", [0x5f9f] = "\x90\x4b",
  [0x5fa0] = "\xb5\xa1", [0x5fa1] = "\xe2\xf9", [0x5fa2] = "\x90\x4c",
  [0x5fa3] = "\x90\x4d", [0x5fa4] = "\x90\x4e", [0x5fa5] = "\xbc\xb1",
  [0x5fa6] = "\xe2\xf1", [0x5fa7] = "\xd0\xd4", [0x5fa8] = "\xd4\xb9",
  [0x5fa9] = "\xe2\xf5", [0x5faa] = "\xb9\xd6", [0x5fab] = "\xe2\xf6",
  [0x5fac] = "\x90\x4f", [0x5fad] = "\x90\x50", [0x5fae] = "\x90\x51",
  [0x5faf] = "\xc7\xd3", [0x5fb0] = "\x90\x52", [0x5fb1] = "\x90\x53",
  [0x5fb2] = "\x90\x54", [0x5fb3] = "\x90\x55", [0x5fb4] = "\x90\x56",
  [0x5fb5] = "\xe2\xf0", [0x5fb6] = "\x90\x57", [0x5fb7] = "\x90\x58",
  [0x5fb8] = "\x90\x59", [0x5fb9] = "\x90\x5a", [0x5fba] = "\x90\x5b",
  [0x5fbb] = "\xd7\xdc", [0x5fbc] = "\xed\xa1", [0x5fbd] = "\x90\x5c",
  [0x5fbe] = "\x90\x5d", [0x5fbf] = "\xe2\xf8", [0x5fc0] = "\x90\x5e",
  [0x5fc1] = "\xed\xa5", [0x5fc2] = "\xe2\xfe", [0x5fc3] = "\xca\xd1",
  [0x5fc4] = "\x90\x5f", [0x5fc5] = "\x90\x60", [0x5fc6] = "\x90\x61",
  [0x5fc7] = "\x90\x62", [0x5fc8] = "\x90\x63", [0x5fc9] = "\x90\x64",
  [0x5fca] = "\x90\x65", [0x5fcb] = "\xc1\xb5", [0x5fcc] = "\x90\x66",
  [0x5fcd] = "\xbb\xd0", [0x5fce] = "\x90\x67", [0x5fcf] = "\x90\x68",
  [0x5fd0] = "\xbf\xd6", [0x5fd1] = "\x90\x69", [0x5fd2] = "\xba\xe3",
  [0x5fd3] = "\x90\x6a", [0x5fd4] = "\x90\x6b", [0x5fd5] = "\xcb\xa1",
  [0x5fd6] = "\x90\x6c", [0x5fd7] = "\x90\x6d", [0x5fd8] = "\x90\x6e",
  [0x5fd9] = "\xed\xa6", [0x5fda] = "\xed\xa3", [0x5fdb] = "\x90\x6f",
  [0x5fdc] = "\x90\x70", [0x5fdd] = "\xed\xa2", [0x5fde] = "\x90\x71",
  [0x5fdf] = "\x90\x72", [0x5fe0] = "\x90\x73", [0x5fe1] = "\x90\x74",
  [0x5fe2] = "\xbb\xd6", [0x5fe3] = "\xed\xa7", [0x5fe4] = "\xd0\xf4",
  [0x5fe5] = "\x90\x75", [0x5fe6] = "\x90\x76", [0x5fe7] = "\xed\xa4",
  [0x5fe8] = "\xba\xde", [0x5fe9] = "\xb6\xf7", [0x5fea] = "\xe3\xa1",
  [0x5feb] = "\xb6\xb2", [0x5fec] = "\xcc\xf1", [0x5fed] = "\xb9\xa7",
  [0x5fee] = "\x90\x77", [0x5fef] = "\xcf\xa2", [0x5ff0] = "\xc7\xa1",
  [0x5ff1] = "\x90\x78", [0x5ff2] = "\x90\x79", [0x5ff3] = "\xbf\xd2",
  [0x5ff4] = "\x90\x7a", [0x5ff5] = "\x90\x7b", [0x5ff6] = "\xb6\xf1",
  [0x5ff7] = "\x90\x7c", [0x5ff8] = "\xe2\xfa", [0x5ff9] = "\xe2\xfb",
  [0x5ffa] = "\xe2\xfd", [0x5ffb] = "\xe2\xfc", [0x5ffc] = "\xc4\xd5",
  [0x5ffd] = "\xe3\xa2", [0x5ffe] = "\x90\x7d", [0x5fff] = "\xd3\xc1",
  [0x6000] = "\x90\x7e", [0x6001] = "\x90\x80", [0x6002] = "\x90\x81",
  [0x6003] = "\xe3\xa7", [0x6004] = "\xc7\xc4", [0x6005] = "\x90\x82",
  [0x6006] = "\x90\x83", [0x6007] = "\x90\x84", [0x6008] = "\x90\x85",
  [0x6009] = "\xcf\xa4", [0x600a] = "\x90\x86", [0x600b] = "\x90\x87",
  [0x600c] = "\xe3\xa9", [0x600d] = "\xba\xb7", [0x600e] = "\x90\x88",
  [0x600f] = "\x90\x89", [0x6010] = "\x90\x8a", [0x6011] = "\x90\x8b",
  [0x6012] = "\xe3\xa8", [0x6013] = "\x90\x8c", [0x6014] = "\xbb\xda",
  [0x6015] = "\x90\x8d", [0x6016] = "\xe3\xa3", [0x6017] = "\x90\x8e",
  [0x6018] = "\x90\x8f", [0x6019] = "\x90\x90", [0x601a] = "\xe3\xa4",
  [0x601b] = "\xe3\xaa", [0x601c] = "\x90\x91", [0x601d] = "\xe3\xa6",
  [0x601e] = "\x90\x92", [0x601f] = "\xce\xf2", [0x6020] = "\xd3\xc6",
  [0x6021] = "\x90\x93", [0x6022] = "\x90\x94", [0x6023] = "\xbb\xbc",
  [0x6024] = "\x90\x95", [0x6025] = "\x90\x96", [0x6026] = "\xd4\xc3",
  [0x6027] = "\x90\x97", [0x6028] = "\xc4\xfa", [0x6029] = "\x90\x98",
  [0x602a] = "\x90\x99", [0x602b] = "\xed\xa8", [0x602c] = "\xd0\xfc",
  [0x602d] = "\xe3\xa5", [0x602e] = "\x90\x9a", [0x602f] = "\xc3\xf5",
  [0x6030] = "\x90\x9b", [0x6031] = "\xe3\xad", [0x6032] = "\xb1\xaf",
  [0x6033] = "\x90\x9c", [0x6034] = "\xe3\xb2", [0x6035] = "\x90\x9d",
  [0x6036] = "\x90\x9e", [0x6037] = "\x90\x9f", [0x6038] = "\xbc\xc2",
  [0x6039] = "\x90\xa0", [0x603a] = "\x90\xa1", [0x603b] = "\xe3\xac",
  [0x603c] = "\xb5\xbf", [0x603d] = "\x90\xa2", [0x603e] = "\x90\xa3",
  [0x603f] = "\x90\xa4", [0x6040] = "\x90\xa5", [0x6041] = "\x90\xa6",
  [0x6042] = "\x90\xa7", [0x6043] = "\x90\xa8", [0x6044] = "\x90\xa9",
  [0x6045] = "\xc7\xe9", [0x6046] = "\xe3\xb0", [0x6047] = "\x90\xaa",
  [0x6048] = "\x90\xab", [0x6049] = "\x90\xac", [0x604a] = "\xbe\xaa",
  [0x604b] = "\xcd\xef", [0x604c] = "\x90\xad", [0x604d] = "\x90\xae",
  [0x604e] = "\x90\xaf", [0x604f] = "\x90\xb0", [0x6050] = "\x90\xb1",
  [0x6051] = "\xbb\xf3", [0x6052] = "\x90\xb2", [0x6053] = "\x90\xb3",
  [0x6054] = "\x90\xb4", [0x6055] = "\xcc\xe8", [0x6056] = "\x90\xb5",
  [0x6057] = "\x90\xb6", [0x6058] = "\xe3\xaf", [0x6059] = "\x90\xb7",
  [0x605a] = "\xe3\xb1", [0x605b] = "\x90\xb8", [0x605c] = "\xcf\xa7",
  [0x605d] = "\xe3\xae", [0x605e] = "\x90\xb9", [0x605f] = "\xce\xa9",
  [0x6060] = "\xbb\xdd", [0x6061] = "\x90\xba", [0x6062] = "\x90\xbb",
  [0x6063] = "\x90\xbc", [0x6064] = "\x90\xbd", [0x6065] = "\x90\xbe",
  [0x6066] = "\xb5\xeb", [0x6067] = "\xbe\xe5", [0x6068] = "\xb2\xd2",
  [0x6069] = "\xb3\xcd", [0x606a] = "\x90\xbf", [0x606b] = "\xb1\xb9",
  [0x606c] = "\xe3\xab", [0x606d] = "\xb2\xd1", [0x606e] = "\xb5\xac",
  [0x606f] = "\xb9\xdf", [0x6070] = "\xb6\xe8", [0x6071] = "\x90\xc0",
  [0x6072] = "\x90\xc1", [0x6073] = "\xcf\xeb", [0x6074] = "\xe3\xb7",
  [0x6075] = "\x90\xc2", [0x6076] = "\xbb\xcc", [0x6077] = "\x90\xc3",
  [0x6078] = "\x90\xc4", [0x6079] = "\xc8\xc7", [0x607a] = "\xd0\xca",
  [0x607b] = "\x90\xc5", [0x607c] = "\x90\xc6", [0x607d] = "\x90\xc7",
  [0x607e] = "\x90\xc8", [0x607f] = "\x90\xc9", [0x6080] = "\xe3\xb8",
  [0x6081] = "\xb3\xee", [0x6082] = "\x90\xca", [0x6083] = "\x90\xcb",
  [0x6084] = "\x90\xcc", [0x6085] = "\x90\xcd", [0x6086] = "\xed\xa9",
  [0x6087] = "\x90\xce", [0x6088] = "\xd3\xfa", [0x6089] = "\xd3\xe4",
  [0x608a] = "\x90\xcf", [0x608b] = "\x90\xd0", [0x608c] = "\x90\xd1",
  [0x608d] = "\xed\xaa", [0x608e] = "\xe3\xb9", [0x608f] = "\xd2\xe2",
  [0x6090] = "\x90\xd2", [0x6091] = "\x90\xd3", [0x6092] = "\x90\xd4",
  [0x6093] = "\x90\xd5", [0x6094] = "\x90\xd6", [0x6095] = "\xe3\xb5",
  [0x6096] = "\x90\xd7", [0x6097] = "\x90\xd8", [0x6098] = "\x90\xd9",
  [0x6099] = "\x90\xda", [0x609a] = "\xd3\xde", [0x609b] = "\x90\xdb",
  [0x609c] = "\x90\xdc", [0x609d] = "\x90\xdd", [0x609e] = "\x90\xde",
  [0x609f] = "\xb8\xd0", [0x60a0] = "\xe3\xb3", [0x60a1] = "\x90\xdf",
  [0x60a2] = "\x90\xe0", [0x60a3] = "\xe3\xb6", [0x60a4] = "\xb7\xdf",
  [0x60a5] = "\x90\xe1", [0x60a6] = "\xe3\xb4", [0x60a7] = "\xc0\xa2",
  [0x60a8] = "\x90\xe2", [0x60a9] = "\x90\xe3", [0x60aa] = "\x90\xe4",
  [0x60ab] = "\xe3\xba", [0x60ac] = "\x90\xe5", [0x60ad] = "\x90\xe6",
  [0x60ae] = "\x90\xe7", [0x60af] = "\x90\xe8", [0x60b0] = "\x90\xe9",
  [0x60b1] = "\x90\xea", [0x60b2] = "\x90\xeb", [0x60b3] = "\x90\xec",
  [0x60b4] = "\x90\xed", [0x60b5] = "\x90\xee", [0x60b6] = "\x90\xef",
  [0x60b7] = "\x90\xf0", [0x60b8] = "\x90\xf1", [0x60b9] = "\x90\xf2",
  [0x60ba] = "\x90\xf3", [0x60bb] = "\x90\xf4", [0x60bc] = "\x90\xf5",
  [0x60bd] = "\x90\xf6", [0x60be] = "\x90\xf7", [0x60bf] = "\xd4\xb8",
  [0x60c0] = "\x90\xf8", [0x60c1] = "\x90\xf9", [0x60c2] = "\x90\xfa",
  [0x60c3] = "\x90\xfb", [0x60c4] = "\x90\xfc", [0x60c5] = "\x90\xfd",
  [0x60c6] = "\x90\xfe", [0x60c7] = "\x91\x40", [0x60c8] = "\xb4\xc8",
  [0x60c9] = "\x91\x41", [0x60ca] = "\xe3\xbb", [0x60cb] = "\x91\x42",
  [0x60cc] = "\xbb\xc5", [0x60cd] = "\x91\x43", [0x60ce] = "\xc9\xf7",
  [0x60cf] = "\x91\x44", [0x60d0] = "\x91\x45", [0x60d1] = "\xc9\xe5",
  [0x60d2] = "\x91\x46", [0x60d3] = "\x91\x47", [0x60d4] = "\x91\x48",
  [0x60d5] = "\xc4\xbd", [0x60d6] = "\x91\x49", [0x60d7] = "\x91\x4a",
  [0x60d8] = "\x91\x4b", [0x60d9] = "\x91\x4c", [0x60da] = "\x91\x4d",
  [0x60db] = "\x91\x4e", [0x60dc] = "\x91\x4f", [0x60dd] = "\xed\xab",
  [0x60de] = "\x91\x50", [0x60df] = "\x91\x51", [0x60e0] = "\x91\x52",
  [0x60e1] = "\x91\x53", [0x60e2] = "\xc2\xfd", [0x60e3] = "\x91\x54",
  [0x60e4] = "\x91\x55", [0x60e5] = "\x91\x56", [0x60e6] = "\x91\x57",
  [0x60e7] = "\xbb\xdb", [0x60e8] = "\xbf\xae", [0x60e9] = "\x91\x58",
  [0x60ea] = "\x91\x59", [0x60eb] = "\x91\x5a", [0x60ec] = "\x91\x5b",
  [0x60ed] = "\x91\x5c", [0x60ee] = "\x91\x5d", [0x60ef] = "\x91\x5e",
  [0x60f0] = "\xce\xbf", [0x60f1] = "\x91\x5f", [0x60f2] = "\x91\x60",
  [0x60f3] = "\x91\x61", [0x60f4] = "\x91\x62", [0x60f5] = "\xe3\xbc",
  [0x60f6] = "\x91\x63", [0x60f7] = "\xbf\xb6", [0x60f8] = "\x91\x64",
  [0x60f9] = "\x91\x65", [0x60fa] = "\x91\x66", [0x60fb] = "\x91\x67",
  [0x60fc] = "\x91\x68", [0x60fd] = "\x91\x69", [0x60fe] = "\x91\x6a",
  [0x60ff] = "\x91\x6b", [0x6100] = "\x91\x6c", [0x6101] = "\x91\x6d",
  [0x6102] = "\x91\x6e", [0x6103] = "\x91\x6f", [0x6104] = "\x91\x70",
  [0x6105] = "\x91\x71", [0x6106] = "\x91\x72", [0x6107] = "\x91\x73",
  [0x6108] = "\x91\x74", [0x6109] = "\x91\x75", [0x610a] = "\x91\x76",
  [0x610b] = "\xb1\xef", [0x610c] = "\x91\x77", [0x610d] = "\x91\x78",
  [0x610e] = "\xd4\xf7", [0x610f] = "\x91\x79", [0x6110] = "\x91\x7a",
  [0x6111] = "\x91\x7b", [0x6112] = "\x91\x7c", [0x6113] = "\x91\x7d",
  [0x6114] = "\xe3\xbe", [0x6115] = "\x91\x7e", [0x6116] = "\x91\x80",
  [0x6117] = "\x91\x81", [0x6118] = "\x91\x82", [0x6119] = "\x91\x83",
  [0x611a] = "\x91\x84", [0x611b] = "\x91\x85", [0x611c] = "\x91\x86",
  [0x611d] = "\xed\xad", [0x611e] = "\x91\x87", [0x611f] = "\x91\x88",
  [0x6120] = "\x91\x89", [0x6121] = "\x91\x8a", [0x6122] = "\x91\x8b",
  [0x6123] = "\x91\x8c", [0x6124] = "\x91\x8d", [0x6125] = "\x91\x8e",
  [0x6126] = "\x91\x8f", [0x6127] = "\xe3\xbf", [0x6128] = "\xba\xa9",
  [0x6129] = "\xed\xac", [0x612a] = "\x91\x90", [0x612b] = "\x91\x91",
  [0x612c] = "\xe3\xbd", [0x612d] = "\x91\x92", [0x612e] = "\x91\x93",
  [0x612f] = "\x91\x94", [0x6130] = "\x91\x95", [0x6131] = "\x91\x96",
  [0x6132] = "\x91\x97", [0x6133] = "\x91\x98", [0x6134] = "\x91\x99",
  [0x6135] = "\x91\x9a", [0x6136] = "\x91\x9b", [0x6137] = "\xe3\xc0",
  [0x6138] = "\x91\x9c", [0x6139] = "\x91\x9d", [0x613a] = "\x91\x9e",
  [0x613b] = "\x91\x9f", [0x613c] = "\x91\xa0", [0x613d] = "\x91\xa1",
  [0x613e] = "\xba\xb6", [0x613f] = "\x91\xa2", [0x6140] = "\x91\xa3",
  [0x6141] = "\x91\xa4", [0x6142] = "\xb6\xae", [0x6143] = "\x91\xa5",
  [0x6144] = "\x91\xa6", [0x6145] = "\x91\xa7", [0x6146] = "\x91\xa8",
  [0x6147] = "\x91\xa9", [0x6148] = "\xd0\xb8", [0x6149] = "\x91\xaa",
  [0x614a] = "\xb0\xc3", [0x614b] = "\xed\xae", [0x614c] = "\x91\xab",
  [0x614d] = "\x91\xac", [0x614e] = "\x91\xad", [0x614f] = "\x91\xae",
  [0x6150] = "\x91\xaf", [0x6151] = "\xed\xaf", [0x6152] = "\xc0\xc1",
  [0x6153] = "\x91\xb0", [0x6154] = "\xe3\xc1", [0x6155] = "\x91\xb1",
  [0x6156] = "\x91\xb2", [0x6157] = "\x91\xb3", [0x6158] = "\x91\xb4",
  [0x6159] = "\x91\xb5", [0x615a] = "\x91\xb6", [0x615b] = "\x91\xb7",
  [0x615c] = "\x91\xb8", [0x615d] = "\x91\xb9", [0x615e] = "\x91\xba",
  [0x615f] = "\x91\xbb", [0x6160] = "\x91\xbc", [0x6161] = "\x91\xbd",
  [0x6162] = "\x91\xbe", [0x6163] = "\x91\xbf", [0x6164] = "\x91\xc0",
  [0x6165] = "\x91\xc1", [0x6166] = "\xc5\xb3", [0x6167] = "\x91\xc2",
  [0x6168] = "\x91\xc3", [0x6169] = "\x91\xc4", [0x616a] = "\x91\xc5",
  [0x616b] = "\x91\xc6", [0x616c] = "\x91\xc7", [0x616d] = "\x91\xc8",
  [0x616e] = "\x91\xc9", [0x616f] = "\x91\xca", [0x6170] = "\x91\xcb",
  [0x6171] = "\x91\xcc", [0x6172] = "\x91\xcd", [0x6173] = "\x91\xce",
  [0x6174] = "\x91\xcf", [0x6175] = "\xe3\xc2", [0x6176] = "\x91\xd0",
  [0x6177] = "\x91\xd1", [0x6178] = "\x91\xd2", [0x6179] = "\x91\xd3",
  [0x617a] = "\x91\xd4", [0x617b] = "\x91\xd5", [0x617c] = "\x91\xd6",
  [0x617d] = "\x91\xd7", [0x617e] = "\x91\xd8", [0x617f] = "\xdc\xb2",
  [0x6180] = "\x91\xd9", [0x6181] = "\x91\xda", [0x6182] = "\x91\xdb",
  [0x6183] = "\x91\xdc", [0x6184] = "\x91\xdd", [0x6185] = "\x91\xde",
  [0x6186] = "\xed\xb0", [0x6187] = "\x91\xdf", [0x6188] = "\xb8\xea",
  [0x6189] = "\x91\xe0", [0x618a] = "\xce\xec", [0x618b] = "\xea\xa7",
  [0x618c] = "\xd0\xe7", [0x618d] = "\xca\xf9", [0x618e] = "\xc8\xd6",
  [0x618f] = "\xcf\xb7", [0x6190] = "\xb3\xc9", [0x6191] = "\xce\xd2",
  [0x6192] = "\xbd\xe4", [0x6193] = "\x91\xe1", [0x6194] = "\x91\xe2",
  [0x6195] = "\xe3\xde", [0x6196] = "\xbb\xf2", [0x6197] = "\xea\xa8",
  [0x6198] = "\xd5\xbd", [0x6199] = "\x91\xe3", [0x619a] = "\xc6\xdd",
  [0x619b] = "\xea\xa9", [0x619c] = "\x91\xe4", [0x619d] = "\x91\xe5",
  [0x619e] = "\x91\xe6", [0x619f] = "\xea\xaa", [0x61a0] = "\x91\xe7",
  [0x61a1] = "\xea\xac", [0x61a2] = "\xea\xab", [0x61a3] = "\x91\xe8",
  [0x61a4] = "\xea\xae", [0x61a5] = "\xea\xad", [0x61a6] = "\x91\xe9",
  [0x61a7] = "\x91\xea", [0x61a8] = "\x91\xeb", [0x61a9] = "\x91\xec",
  [0x61aa] = "\xbd\xd8", [0x61ab] = "\x91\xed", [0x61ac] = "\xea\xaf",
  [0x61ad] = "\x91\xee", [0x61ae] = "\xc2\xbe", [0x61af] = "\x91\xef",
  [0x61b0] = "\x91\xf0", [0x61b1] = "\x91\xf1", [0x61b2] = "\x91\xf2",
  [0x61b3] = "\xb4\xc1", [0x61b4] = "\xb4\xf7", [0x61b5] = "\x91\xf3",
  [0x61b6] = "\x91\xf4", [0x61b7] = "\xbb\xa7", [0x61b8] = "\x91\xf5",
  [0x61b9] = "\x91\xf6", [0x61ba] = "\x91\xf7", [0x61bb] = "\x91\xf8",
  [0x61bc] = "\x91\xf9", [0x61bd] = "\xec\xe6", [0x61be] = "\xec\xe5",
  [0x61bf] = "\xb7\xbf", [0x61c0] = "\xcb\xf9", [0x61c1] = "\xb1\xe2",
  [0x61c2] = "\x91\xfa", [0x61c3] = "\xec\xe7", [0x61c4] = "\x91\xfb",
  [0x61c5] = "\x91\xfc", [0x61c6] = "\x91\xfd", [0x61c7] = "\xc9\xc8",
  [0x61c8] = "\xec\xe8", [0x61c9] = "\xec\xe9", [0x61ca] = "\x91\xfe",
  [0x61cb] = "\xca\xd6", [0x61cc] = "\xde\xd0", [0x61cd] = "\xb2\xc5",
  [0x61ce] = "\xd4\xfa", [0x61cf] = "\x92\x40", [0x61d0] = "\x92\x41",
  [0x61d1] = "\xc6\xcb", [0x61d2] = "\xb0\xc7", [0x61d3] = "\xb4\xf2",
  [0x61d4] = "\xc8\xd3", [0x61d5] = "\x92\x42", [0x61d6] = "\x92\x43",
  [0x61d7] = "\x92\x44", [0x61d8] = "\xcd\xd0", [0x61d9] = "\x92\x45",
  [0x61da] = "\x92\x46", [0x61db] = "\xbf\xb8", [0x61dc] = "\x92\x47",
  [0x61dd] = "\x92\x48", [0x61de] = "\x92\x49", [0x61df] = "\x92\x4a",
  [0x61e0] = "\x92\x4b", [0x61e1] = "\x92\x4c", [0x61e2] = "\x92\x4d",
  [0x61e3] = "\xbf\xdb", [0x61e4] = "\x92\x4e", [0x61e5] = "\x92\x4f",
  [0x61e6] = "\xc7\xa4", [0x61e7] = "\xd6\xb4", [0x61e8] = "\x92\x50",
  [0x61e9] = "\xc0\xa9", [0x61ea] = "\xde\xd1", [0x61eb] = "\xc9\xa8",
  [0x61ec] = "\xd1\xef", [0x61ed] = "\xc5\xa4", [0x61ee] = "\xb0\xe7",
  [0x61ef] = "\xb3\xb6", [0x61f0] = "\xc8\xc5", [0x61f1] = "\x92\x51",
  [0x61f2] = "\x92\x52", [0x61f3] = "\xb0\xe2", [0x61f4] = "\x92\x53",
  [0x61f5] = "\x92\x54", [0x61f6] = "\xb7\xf6", [0x61f7] = "\x92\x55",
  [0x61f8] = "\x92\x56", [0x61f9] = "\xc5\xfa", [0x61fa] = "\x92\x57",
  [0x61fb] = "\x92\x58", [0x61fc] = "\xb6\xf3", [0x61fd] = "\x92\x59",
  [0x61fe] = "\xd5\xd2", [0x61ff] = "\xb3\xd0", [0x6200] = "\xbc\xbc",
  [0x6201] = "\x92\x5a", [0x6202] = "\x92\x5b", [0x6203] = "\x92\x5c",
  [0x6204] = "\xb3\xad", [0x6205] = "\x92\x5d", [0x6206] = "\x92\x5e",
  [0x6207] = "\x92\x5f", [0x6208] = "\x92\x60", [0x6209] = "\xbe\xf1",
  [0x620a] = "\xb0\xd1", [0x620b] = "\x92\x61", [0x620c] = "\x92\x62",
  [0x620d] = "\x92\x63", [0x620e] = "\x92\x64", [0x620f] = "\x92\x65",
  [0x6210] = "\x92\x66", [0x6211] = "\xd2\xd6", [0x6212] = "\xca\xe3",
  [0x6213] = "\xd7\xa5", [0x6214] = "\x92\x67", [0x6215] = "\xcd\xb6",
  [0x6216] = "\xb6\xb6", [0x6217] = "\xbf\xb9", [0x6218] = "\xd5\xdb",
  [0x6219] = "\x92\x68", [0x621a] = "\xb8\xa7", [0x621b] = "\xc5\xd7",
  [0x621c] = "\x92\x69", [0x621d] = "\x92\x6a", [0x621e] = "\x92\x6b",
  [0x621f] = "\xde\xd2", [0x6220] = "\xbf\xd9", [0x6221] = "\xc2\xd5",
  [0x6222] = "\xc7\xc0", [0x6223] = "\x92\x6c", [0x6224] = "\xbb\xa4",
  [0x6225] = "\xb1\xa8", [0x6226] = "\x92\x6d", [0x6227] = "\x92\x6e",
  [0x6228] = "\xc5\xea", [0x6229] = "\x92\x6f", [0x622a] = "\x92\x70",
  [0x622b] = "\xc5\xfb", [0x622c] = "\xcc\xa7", [0x622d] = "\x92\x71",
  [0x622e] = "\x92\x72", [0x622f] = "\x92\x73", [0x6230] = "\x92\x74",
  [0x6231] = "\xb1\xa7", [0x6232] = "\x92\x75", [0x6233] = "\x92\x76",
  [0x6234] = "\x92\x77", [0x6235] = "\xb5\xd6", [0x6236] = "\x92\x78",
  [0x6237] = "\x92\x79", [0x6238] = "\x92\x7a", [0x6239] = "\xc4\xa8",
  [0x623a] = "\x92\x7b", [0x623b] = "\xde\xd3", [0x623c] = "\xd1\xba",
  [0x623d] = "\xb3\xe9", [0x623e] = "\x92\x7c", [0x623f] = "\xc3\xf2",
  [0x6240] = "\x92\x7d", [0x6241] = "\x92\x7e", [0x6242] = "\xb7\xf7",
  [0x6243] = "\x92\x80", [0x6244] = "\xd6\xf4", [0x6245] = "\xb5\xa3",
  [0x6246] = "\xb2\xf0", [0x6247] = "\xc4\xb4", [0x6248] = "\xc4\xe9",
  [0x6249] = "\xc0\xad", [0x624a] = "\xde\xd4", [0x624b] = "\x92\x81",
  [0x624c] = "\xb0\xe8", [0x624d] = "\xc5\xc4", [0x624e] = "\xc1\xe0",
  [0x624f] = "\x92\x82", [0x6250] = "\xb9\xd5", [0x6251] = "\x92\x83",
  [0x6252] = "\xbe\xdc", [0x6253] = "\xcd\xd8", [0x6254] = "\xb0\xce",
  [0x6255] = "\x92\x84", [0x6256] = "\xcd\xcf", [0x6257] = "\xde\xd6",
  [0x6258] = "\xbe\xd0", [0x6259] = "\xd7\xbe", [0x625a] = "\xde\xd5",
  [0x625b] = "\xd5\xd0", [0x625c] = "\xb0\xdd", [0x625d] = "\x92\x85",
  [0x625e] = "\x92\x86", [0x625f] = "\xc4\xe2", [0x6260] = "\x92\x87",
  [0x6261] = "\x92\x88", [0x6262] = "\xc2\xa3", [0x6263] = "\xbc\xf0",
  [0x6264] = "\x92\x89", [0x6265] = "\xd3\xb5", [0x6266] = "\xc0\xb9",
  [0x6267] = "\xc5\xa1", [0x6268] = "\xb2\xa6", [0x6269] = "\xd4\xf1",
  [0x626a] = "\x92\x8a", [0x626b] = "\x92\x8b", [0x626c] = "\xc0\xa8",
  [0x626d] = "\xca\xc3", [0x626e] = "\xde\xd7", [0x626f] = "\xd5\xfc",
  [0x6270] = "\x92\x8c", [0x6271] = "\xb9\xb0", [0x6272] = "\x92\x8d",
  [0x6273] = "\xc8\xad", [0x6274] = "\xcb\xa9", [0x6275] = "\x92\x8e",
  [0x6276] = "\xde\xd9", [0x6277] = "\xbf\xbd", [0x6278] = "\x92\x8f",
  [0x6279] = "\x92\x90", [0x627a] = "\x92\x91", [0x627b] = "\x92\x92",
  [0x627c] = "\xc6\xb4", [0x627d] = "\xd7\xa7", [0x627e] = "\xca\xb0",
  [0x627f] = "\xc4\xc3", [0x6280] = "\x92\x93", [0x6281] = "\xb3\xd6",
  [0x6282] = "\xb9\xd2", [0x6283] = "\x92\x94", [0x6284] = "\x92\x95",
  [0x6285] = "\x92\x96", [0x6286] = "\x92\x97", [0x6287] = "\xd6\xb8",
  [0x6288] = "\xea\xfc", [0x6289] = "\xb0\xb4", [0x628a] = "\x92\x98",
  [0x628b] = "\x92\x99", [0x628c] = "\x92\x9a", [0x628d] = "\x92\x9b",
  [0x628e] = "\xbf\xe6", [0x628f] = "\x92\x9c", [0x6290] = "\x92\x9d",
  [0x6291] = "\xcc\xf4", [0x6292] = "\x92\x9e", [0x6293] = "\x92\x9f",
  [0x6294] = "\x92\xa0", [0x6295] = "\x92\xa1", [0x6296] = "\xcd\xda",
  [0x6297] = "\x92\xa2", [0x6298] = "\x92\xa3", [0x6299] = "\x92\xa4",
  [0x629a] = "\xd6\xbf", [0x629b] = "\xc2\xce", [0x629c] = "\x92\xa5",
  [0x629d] = "\xce\xce", [0x629e] = "\xcc\xa2", [0x629f] = "\xd0\xae",
  [0x62a0] = "\xc4\xd3", [0x62a1] = "\xb5\xb2", [0x62a2] = "\xde\xd8",
  [0x62a3] = "\xd5\xf5", [0x62a4] = "\xbc\xb7", [0x62a5] = "\xbb\xd3",
  [0x62a6] = "\x92\xa6", [0x62a7] = "\x92\xa7", [0x62a8] = "\xb0\xa4",
  [0x62a9] = "\x92\xa8", [0x62aa] = "\xc5\xb2", [0x62ab] = "\xb4\xec",
  [0x62ac] = "\x92\xa9", [0x62ad] = "\x92\xaa", [0x62ae] = "\x92\xab",
  [0x62af] = "\xd5\xf1", [0x62b0] = "\x92\xac", [0x62b1] = "\x92\xad",
  [0x62b2] = "\xea\xfd", [0x62b3] = "\x92\xae", [0x62b4] = "\x92\xaf",
  [0x62b5] = "\x92\xb0", [0x62b6] = "\x92\xb1", [0x62b7] = "\x92\xb2",
  [0x62b8] = "\x92\xb3", [0x62b9] = "\xde\xda", [0x62ba] = "\xcd\xa6",
  [0x62bb] = "\x92\xb4", [0x62bc] = "\x92\xb5", [0x62bd] = "\xcd\xec",
  [0x62be] = "\x92\xb6", [0x62bf] = "\x92\xb7", [0x62c0] = "\x92\xb8",
  [0x62c1] = "\x92\xb9", [0x62c2] = "\xce\xe6", [0x62c3] = "\xde\xdc",
  [0x62c4] = "\x92\xba", [0x62c5] = "\xcd\xb1", [0x62c6] = "\xc0\xa6",
  [0x62c7] = "\x92\xbb", [0x62c8] = "\x92\xbc", [0x62c9] = "\xd7\xbd",
  [0x62ca] = "\x92\xbd", [0x62cb] = "\xde\xdb", [0x62cc] = "\xb0\xc6",
  [0x62cd] = "\xba\xb4", [0x62ce] = "\xc9\xd3", [0x62cf] = "\xc4\xf3",
  [0x62d0] = "\xbe\xe8", [0x62d1] = "\x92\xbe", [0x62d2] = "\x92\xbf",
  [0x62d3] = "\x92\xc0", [0x62d4] = "\x92\xc1", [0x62d5] = "\xb2\xb6",
  [0x62d6] = "\x92\xc2", [0x62d7] = "\x92\xc3", [0x62d8] = "\x92\xc4",
  [0x62d9] = "\x92\xc5", [0x62da] = "\x92\xc6", [0x62db] = "\x92\xc7",
  [0x62dc] = "\x92\xc8", [0x62dd] = "\x92\xc9", [0x62de] = "\xc0\xcc",
  [0x62df] = "\xcb\xf0", [0x62e0] = "\x92\xca", [0x62e1] = "\xbc\xf1",
  [0x62e2] = "\xbb\xbb", [0x62e3] = "\xb5\xb7", [0x62e4] = "\x92\xcb",
  [0x62e5] = "\x92\xcc", [0x62e6] = "\x92\xcd", [0x62e7] = "\xc5\xf5",
  [0x62e8] = "\x92\xce", [0x62e9] = "\xde\xe6", [0x62ea] = "\x92\xcf",
  [0x62eb] = "\x92\xd0", [0x62ec] = "\x92\xd1", [0x62ed] = "\xde\xe3",
  [0x62ee] = "\xbe\xdd", [0x62ef] = "\x92\xd2", [0x62f0] = "\x92\xd3",
  [0x62f1] = "\xde\xdf", [0x62f2] = "\x92\xd4", [0x62f3] = "\x92\xd5",
  [0x62f4] = "\x92\xd6", [0x62f5] = "\x92\xd7", [0x62f6] = "\xb4\xb7",
  [0x62f7] = "\xbd\xdd", [0x62f8] = "\x92\xd8", [0x62f9] = "\x92\xd9",
  [0x62fa] = "\xde\xe0", [0x62fb] = "\xc4\xed", [0x62fc] = "\x92\xda",
  [0x62fd] = "\x92\xdb", [0x62fe] = "\x92\xdc", [0x62ff] = "\x92\xdd",
  [0x6300] = "\xcf\xc6", [0x6301] = "\x92\xde", [0x6302] = "\xb5\xe0",
  [0x6303] = "\x92\xdf", [0x6304] = "\x92\xe0", [0x6305] = "\x92\xe1",
  [0x6306] = "\x92\xe2", [0x6307] = "\xb6\xde", [0x6308] = "\xca\xda",
  [0x6309] = "\xb5\xf4", [0x630a] = "\xde\xe5", [0x630b] = "\x92\xe3",
  [0x630c] = "\xd5\xc6", [0x630d] = "\x92\xe4", [0x630e] = "\xde\xe1",
  [0x630f] = "\xcc\xcd", [0x6310] = "\xc6\xfe", [0x6311] = "\x92\xe5",
  [0x6312] = "\xc5\xc5", [0x6313] = "\x92\xe6", [0x6314] = "\x92\xe7",
  [0x6315] = "\x92\xe8", [0x6316] = "\xd2\xb4", [0x6317] = "\x92\xe9",
  [0x6318] = "\xbe\xf2", [0x6319] = "\x92\xea", [0x631a] = "\x92\xeb",
  [0x631b] = "\x92\xec", [0x631c] = "\x92\xed", [0x631d] = "\x92\xee",
  [0x631e] = "\x92\xef", [0x631f] = "\x92\xf0", [0x6320] = "\xc2\xd3",
  [0x6321] = "\x92\xf1", [0x6322] = "\xcc\xbd", [0x6323] = "\xb3\xb8",
  [0x6324] = "\x92\xf2", [0x6325] = "\xbd\xd3", [0x6326] = "\x92\xf3",
  [0x6327] = "\xbf\xd8", [0x6328] = "\xcd\xc6", [0x6329] = "\xd1\xda",
  [0x632a] = "\xb4\xeb", [0x632b] = "\x92\xf4", [0x632c] = "\xde\xe4",
  [0x632d] = "\xde\xdd", [0x632e] = "\xde\xe7", [0x632f] = "\x92\xf5",
  [0x6330] = "\xea\xfe", [0x6331] = "\x92\xf6", [0x6332] = "\x92\xf7",
  [0x6333] = "\xc2\xb0", [0x6334] = "\xde\xe2", [0x6335] = "\x92\xf8",
  [0x6336] = "\x92\xf9", [0x6337] = "\xd6\xc0", [0x6338] = "\xb5\xa7",
  [0x6339] = "\x92\xfa", [0x633a] = "\xb2\xf4", [0x633b] = "\x92\xfb",
  [0x633c] = "\xde\xe8", [0x633d] = "\x92\xfc", [0x633e] = "\xde\xf2",
  [0x633f] = "\x92\xfd", [0x6340] = "\x92\xfe", [0x6341] = "\x93\x40",
  [0x6342] = "\x93\x41", [0x6343] = "\x93\x42", [0x6344] = "\xde\xed",
  [0x6345] = "\x93\x43", [0x6346] = "\xde\xf1", [0x6347] = "\x93\x44",
  [0x6348] = "\x93\x45", [0x6349] = "\xc8\xe0", [0x634a] = "\x93\x46",
  [0x634b] = "\x93\x47", [0x634c] = "\x93\x48", [0x634d] = "\xd7\xe1",
  [0x634e] = "\xde\xef", [0x634f] = "\xc3\xe8", [0x6350] = "\xcc\xe1",
  [0x6351] = "\x93\x49", [0x6352] = "\xb2\xe5", [0x6353] = "\x93\x4a",
  [0x6354] = "\x93\x4b", [0x6355] = "\x93\x4c", [0x6356] = "\xd2\xbe",
  [0x6357] = "\x93\x4d", [0x6358] = "\x93\x4e", [0x6359] = "\x93\x4f",
  [0x635a] = "\x93\x50", [0x635b] = "\x93\x51", [0x635c] = "\x93\x52",
  [0x635d] = "\x93\x53", [0x635e] = "\xde\xee", [0x635f] = "\x93\x54",
  [0x6360] = "\xde\xeb", [0x6361] = "\xce\xd5", [0x6362] = "\x93\x55",
  [0x6363] = "\xb4\xa7", [0x6364] = "\x93\x56", [0x6365] = "\x93\x57",
  [0x6366] = "\x93\x58", [0x6367] = "\x93\x59", [0x6368] = "\x93\x5a",
  [0x6369] = "\xbf\xab", [0x636a] = "\xbe\xbe", [0x636b] = "\x93\x5b",
  [0x636c] = "\x93\x5c", [0x636d] = "\xbd\xd2", [0x636e] = "\x93\x5d",
  [0x636f] = "\x93\x5e", [0x6370] = "\x93\x5f", [0x6371] = "\x93\x60",
  [0x6372] = "\xde\xe9", [0x6373] = "\x93\x61", [0x6374] = "\xd4\xae",
  [0x6375] = "\x93\x62", [0x6376] = "\xde\xde", [0x6377] = "\x93\x63",
  [0x6378] = "\xde\xea", [0x6379] = "\x93\x64", [0x637a] = "\x93\x65",
  [0x637b] = "\x93\x66", [0x637c] = "\x93\x67", [0x637d] = "\xc0\xbf",
  [0x637e] = "\x93\x68", [0x637f] = "\xde\xec", [0x6380] = "\xb2\xf3",
  [0x6381] = "\xb8\xe9", [0x6382] = "\xc2\xa7", [0x6383] = "\x93\x69",
  [0x6384] = "\x93\x6a", [0x6385] = "\xbd\xc1", [0x6386] = "\x93\x6b",
  [0x6387] = "\x93\x6c", [0x6388] = "\x93\x6d", [0x6389] = "\x93\x6e",
  [0x638a] = "\x93\x6f", [0x638b] = "\xde\xf5", [0x638c] = "\xde\xf8",
  [0x638d] = "\x93\x70", [0x638e] = "\x93\x71", [0x638f] = "\xb2\xab",
  [0x6390] = "\xb4\xa4", [0x6391] = "\x93\x72", [0x6392] = "\x93\x73",
  [0x6393] = "\xb4\xea", [0x6394] = "\xc9\xa6", [0x6395] = "\x93\x74",
  [0x6396] = "\x93\x75", [0x6397] = "\x93\x76", [0x6398] = "\x93\x77",
  [0x6399] = "\x93\x78", [0x639a] = "\x93\x79", [0x639b] = "\xde\xf6",
  [0x639c] = "\xcb\xd1", [0x639d] = "\x93\x7a", [0x639e] = "\xb8\xe3",
  [0x639f] = "\x93\x7b", [0x63a0] = "\xde\xf7", [0x63a1] = "\xde\xfa",
  [0x63a2] = "\x93\x7c", [0x63a3] = "\x93\x7d", [0x63a4] = "\x93\x7e",
  [0x63a5] = "\x93\x80", [0x63a6] = "\xde\xf9", [0x63a7] = "\x93\x81",
  [0x63a8] = "\x93\x82", [0x63a9] = "\x93\x83", [0x63aa] = "\xcc\xc2",
  [0x63ab] = "\x93\x84", [0x63ac] = "\xb0\xe1", [0x63ad] = "\xb4\xee",
  [0x63ae] = "\x93\x85", [0x63af] = "\x93\x86", [0x63b0] = "\x93\x87",
  [0x63b1] = "\x93\x88", [0x63b2] = "\x93\x89", [0x63b3] = "\x93\x8a",
  [0x63b4] = "\xe5\xba", [0x63b5] = "\x93\x8b", [0x63b6] = "\x93\x8c",
  [0x63b7] = "\x93\x8d", [0x63b8] = "\x93\x8e", [0x63b9] = "\x93\x8f",
  [0x63ba] = "\xd0\xaf", [0x63bb] = "\x93\x90", [0x63bc] = "\x93\x91",
  [0x63bd] = "\xb2\xeb", [0x63be] = "\x93\x92", [0x63bf] = "\xeb\xa1",
  [0x63c0] = "\x93\x93", [0x63c1] = "\xde\xf4", [0x63c2] = "\x93\x94",
  [0x63c3] = "\x93\x95", [0x63c4] = "\xc9\xe3", [0x63c5] = "\xde\xf3",
  [0x63c6] = "\xb0\xda", [0x63c7] = "\xd2\xa1", [0x63c8] = "\xb1\xf7",
  [0x63c9] = "\x93\x96", [0x63ca] = "\xcc\xaf", [0x63cb] = "\x93\x97",
  [0x63cc] = "\x93\x98", [0x63cd] = "\x93\x99", [0x63ce] = "\x93\x9a",
  [0x63cf] = "\x93\x9b", [0x63d0] = "\x93\x9c", [0x63d1] = "\x93\x9d",
  [0x63d2] = "\xde\xf0", [0x63d3] = "\x93\x9e", [0x63d4] = "\xcb\xa4",
  [0x63d5] = "\x93\x9f", [0x63d6] = "\x93\xa0", [0x63d7] = "\x93\xa1",
  [0x63d8] = "\xd5\xaa", [0x63d9] = "\x93\xa2", [0x63da] = "\x93\xa3",
  [0x63db] = "\x93\xa4", [0x63dc] = "\x93\xa5", [0x63dd] = "\x93\xa6",
  [0x63de] = "\xde\xfb", [0x63df] = "\x93\xa7", [0x63e0] = "\x93\xa8",
  [0x63e1] = "\x93\xa9", [0x63e2] = "\x93\xaa", [0x63e3] = "\x93\xab",
  [0x63e4] = "\x93\xac", [0x63e5] = "\x93\xad", [0x63e6] = "\x93\xae",
  [0x63e7] = "\xb4\xdd", [0x63e8] = "\x93\xaf", [0x63e9] = "\xc4\xa6",
  [0x63ea] = "\x93\xb0", [0x63eb] = "\x93\xb1", [0x63ec] = "\x93\xb2",
  [0x63ed] = "\xde\xfd", [0x63ee] = "\x93\xb3", [0x63ef] = "\x93\xb4",
  [0x63f0] = "\x93\xb5", [0x63f1] = "\x93\xb6", [0x63f2] = "\x93\xb7",
  [0x63f3] = "\x93\xb8", [0x63f4] = "\x93\xb9", [0x63f5] = "\x93\xba",
  [0x63f6] = "\x93\xbb", [0x63f7] = "\x93\xbc", [0x63f8] = "\xc3\xfe",
  [0x63f9] = "\xc4\xa1", [0x63fa] = "\xdf\xa1", [0x63fb] = "\x93\xbd",
  [0x63fc] = "\x93\xbe", [0x63fd] = "\x93\xbf", [0x63fe] = "\x93\xc0",
  [0x63ff] = "\x93\xc1", [0x6400] = "\x93\xc2", [0x6401] = "\x93\xc3",
  [0x6402] = "\xc1\xcc", [0x6403] = "\x93\xc4", [0x6404] = "\xde\xfc",
  [0x6405] = "\xbe\xef", [0x6406] = "\x93\xc5", [0x6407] = "\xc6\xb2",
  [0x6408] = "\x93\xc6", [0x6409] = "\x93\xc7", [0x640a] = "\x93\xc8",
  [0x640b] = "\x93\xc9", [0x640c] = "\x93\xca", [0x640d] = "\x93\xcb",
  [0x640e] = "\x93\xcc", [0x640f] = "\x93\xcd", [0x6410] = "\x93\xce",
  [0x6411] = "\xb3\xc5", [0x6412] = "\xc8\xf6", [0x6413] = "\x93\xcf",
  [0x6414] = "\x93\xd0", [0x6415] = "\xcb\xba", [0x6416] = "\xde\xfe",
  [0x6417] = "\x93\xd1", [0x6418] = "\x93\xd2", [0x6419] = "\xdf\xa4",
  [0x641a] = "\x93\xd3", [0x641b] = "\x93\xd4", [0x641c] = "\x93\xd5",
  [0x641d] = "\x93\xd6", [0x641e] = "\xd7\xb2", [0x641f] = "\x93\xd7",
  [0x6420] = "\x93\xd8", [0x6421] = "\x93\xd9", [0x6422] = "\x93\xda",
  [0x6423] = "\x93\xdb", [0x6424] = "\xb3\xb7", [0x6425] = "\x93\xdc",
  [0x6426] = "\x93\xdd", [0x6427] = "\x93\xde", [0x6428] = "\x93\xdf",
  [0x6429] = "\xc1\xc3", [0x642a] = "\x93\xe0", [0x642b] = "\x93\xe1",
  [0x642c] = "\xc7\xcb", [0x642d] = "\xb2\xa5", [0x642e] = "\xb4\xe9",
  [0x642f] = "\x93\xe2", [0x6430] = "\xd7\xab", [0x6431] = "\x93\xe3",
  [0x6432] = "\x93\xe4", [0x6433] = "\x93\xe5", [0x6434] = "\x93\xe6",
  [0x6435] = "\xc4\xec", [0x6436] = "\x93\xe7", [0x6437] = "\xdf\xa2",
  [0x6438] = "\xdf\xa3", [0x6439] = "\x93\xe8", [0x643a] = "\xdf\xa5",
  [0x643b] = "\x93\xe9", [0x643c] = "\xba\xb3", [0x643d] = "\x93\xea",
  [0x643e] = "\x93\xeb", [0x643f] = "\x93\xec", [0x6440] = "\xdf\xa6",
  [0x6441] = "\x93\xed", [0x6442] = "\xc0\xde", [0x6443] = "\x93\xee",
  [0x6444] = "\x93\xef", [0x6445] = "\xc9\xc3", [0x6446] = "\x93\xf0",
  [0x6447] = "\x93\xf1", [0x6448] = "\x93\xf2", [0x6449] = "\x93\xf3",
  [0x644a] = "\x93\xf4", [0x644b] = "\x93\xf5", [0x644c] = "\x93\xf6",
  [0x644d] = "\xb2\xd9", [0x644e] = "\xc7\xe6", [0x644f] = "\x93\xf7",
  [0x6450] = "\xdf\xa7", [0x6451] = "\x93\xf8", [0x6452] = "\xc7\xdc",
  [0x6453] = "\x93\xf9", [0x6454] = "\x93\xfa", [0x6455] = "\x93\xfb",
  [0x6456] = "\x93\xfc", [0x6457] = "\xdf\xa8", [0x6458] = "\xeb\xa2",
  [0x6459] = "\x93\xfd", [0x645a] = "\x93\xfe", [0x645b] = "\x94\x40",
  [0x645c] = "\x94\x41", [0x645d] = "\x94\x42", [0x645e] = "\xcb\xd3",
  [0x645f] = "\x94\x43", [0x6460] = "\x94\x44", [0x6461] = "\x94\x45",
  [0x6462] = "\xdf\xaa", [0x6463] = "\x94\x46", [0x6464] = "\xdf\xa9",
  [0x6465] = "\x94\x47", [0x6466] = "\xb2\xc1", [0x6467] = "\x94\x48",
  [0x6468] = "\x94\x49", [0x6469] = "\x94\x4a", [0x646a] = "\x94\x4b",
  [0x646b] = "\x94\x4c", [0x646c] = "\x94\x4d", [0x646d] = "\x94\x4e",
  [0x646e] = "\x94\x4f", [0x646f] = "\x94\x50", [0x6470] = "\x94\x51",
  [0x6471] = "\x94\x52", [0x6472] = "\x94\x53", [0x6473] = "\x94\x54",
  [0x6474] = "\x94\x55", [0x6475] = "\x94\x56", [0x6476] = "\x94\x57",
  [0x6477] = "\x94\x58", [0x6478] = "\x94\x59", [0x6479] = "\x94\x5a",
  [0x647a] = "\x94\x5b", [0x647b] = "\x94\x5c", [0x647c] = "\x94\x5d",
  [0x647d] = "\x94\x5e", [0x647e] = "\x94\x5f", [0x647f] = "\x94\x60",
  [0x6480] = "\xc5\xca", [0x6481] = "\x94\x61", [0x6482] = "\x94\x62",
  [0x6483] = "\x94\x63", [0x6484] = "\x94\x64", [0x6485] = "\x94\x65",
  [0x6486] = "\x94\x66", [0x6487] = "\x94\x67", [0x6488] = "\x94\x68",
  [0x6489] = "\xdf\xab", [0x648a] = "\x94\x69", [0x648b] = "\x94\x6a",
  [0x648c] = "\x94\x6b", [0x648d] = "\x94\x6c", [0x648e] = "\x94\x6d",
  [0x648f] = "\x94\x6e", [0x6490] = "\x94\x6f", [0x6491] = "\x94\x70",
  [0x6492] = "\xd4\xdc", [0x6493] = "\x94\x71", [0x6494] = "\x94\x72",
  [0x6495] = "\x94\x73", [0x6496] = "\x94\x74", [0x6497] = "\x94\x75",
  [0x6498] = "\xc8\xc1", [0x6499] = "\x94\x76", [0x649a] = "\x94\x77",
  [0x649b] = "\x94\x78", [0x649c] = "\x94\x79", [0x649d] = "\x94\x7a",
  [0x649e] = "\x94\x7b", [0x649f] = "\x94\x7c", [0x64a0] = "\x94\x7d",
  [0x64a1] = "\x94\x7e", [0x64a2] = "\x94\x80", [0x64a3] = "\x94\x81",
  [0x64a4] = "\x94\x82", [0x64a5] = "\xdf\xac", [0x64a6] = "\x94\x83",
  [0x64a7] = "\x94\x84", [0x64a8] = "\x94\x85", [0x64a9] = "\x94\x86",
  [0x64aa] = "\x94\x87", [0x64ab] = "\xbe\xf0", [0x64ac] = "\x94\x88",
  [0x64ad] = "\x94\x89", [0x64ae] = "\xdf\xad", [0x64af] = "\xd6\xa7",
  [0x64b0] = "\x94\x8a", [0x64b1] = "\x94\x8b", [0x64b2] = "\x94\x8c",
  [0x64b3] = "\x94\x8d", [0x64b4] = "\xea\xb7", [0x64b5] = "\xeb\xb6",
  [0x64b6] = "\xca\xd5", [0x64b7] = "\x94\x8e", [0x64b8] = "\xd8\xfc",
  [0x64b9] = "\xb8\xc4", [0x64ba] = "\x94\x8f", [0x64bb] = "\xb9\xa5",
  [0x64bc] = "\x94\x90", [0x64bd] = "\x94\x91", [0x64be] = "\xb7\xc5",
  [0x64bf] = "\xd5\xfe", [0x64c0] = "\x94\x92", [0x64c1] = "\x94\x93",
  [0x64c2] = "\x94\x94", [0x64c3] = "\x94\x95", [0x64c4] = "\x94\x96",
  [0x64c5] = "\xb9\xca", [0x64c6] = "\x94\x97", [0x64c7] = "\x94\x98",
  [0x64c8] = "\xd0\xa7", [0x64c9] = "\xf4\xcd", [0x64ca] = "\x94\x99",
  [0x64cb] = "\x94\x9a", [0x64cc] = "\xb5\xd0", [0x64cd] = "\x94\x9b",
  [0x64ce] = "\x94\x9c", [0x64cf] = "\xc3\xf4", [0x64d0] = "\x94\x9d",
  [0x64d1] = "\xbe\xc8", [0x64d2] = "\x94\x9e", [0x64d3] = "\x94\x9f",
  [0x64d4] = "\x94\xa0", [0x64d5] = "\xeb\xb7", [0x64d6] = "\xb0\xbd",
  [0x64d7] = "\x94\xa1", [0x64d8] = "\x94\xa2", [0x64d9] = "\xbd\xcc",
  [0x64da] = "\x94\xa3", [0x64db] = "\xc1\xb2", [0x64dc] = "\x94\xa4",
  [0x64dd] = "\xb1\xd6", [0x64de] = "\xb3\xa8", [0x64df] = "\x94\xa5",
  [0x64e0] = "\x94\xa6", [0x64e1] = "\x94\xa7", [0x64e2] = "\xb8\xd2",
  [0x64e3] = "\xc9\xa2", [0x64e4] = "\x94\xa8", [0x64e5] = "\x94\xa9",
  [0x64e6] = "\xb6\xd8", [0x64e7] = "\x94\xaa", [0x64e8] = "\x94\xab",
  [0x64e9] = "\x94\xac", [0x64ea] = "\x94\xad", [0x64eb] = "\xeb\xb8",
  [0x64ec] = "\xbe\xb4", [0x64ed] = "\x94\xae", [0x64ee] = "\x94\xaf",
  [0x64ef] = "\x94\xb0", [0x64f0] = "\xca\xfd", [0x64f1] = "\x94\xb1",
  [0x64f2] = "\xc7\xc3", [0x64f3] = "\x94\xb2", [0x64f4] = "\xd5\xfb",
  [0x64f5] = "\x94\xb3", [0x64f6] = "\x94\xb4", [0x64f7] = "\xb7\xf3",
  [0x64f8] = "\x94\xb5", [0x64f9] = "\x94\xb6", [0x64fa] = "\x94\xb7",
  [0x64fb] = "\x94\xb8", [0x64fc] = "\x94\xb9", [0x64fd] = "\x94\xba",
  [0x64fe] = "\x94\xbb", [0x64ff] = "\x94\xbc", [0x6500] = "\x94\xbd",
  [0x6501] = "\x94\xbe", [0x6502] = "\x94\xbf", [0x6503] = "\x94\xc0",
  [0x6504] = "\x94\xc1", [0x6505] = "\x94\xc2", [0x6506] = "\x94\xc3",
  [0x6507] = "\xce\xc4", [0x6508] = "\x94\xc4", [0x6509] = "\x94\xc5",
  [0x650a] = "\x94\xc6", [0x650b] = "\xd5\xab", [0x650c] = "\xb1\xf3",
  [0x650d] = "\x94\xc7", [0x650e] = "\x94\xc8", [0x650f] = "\x94\xc9",
  [0x6510] = "\xec\xb3", [0x6511] = "\xb0\xdf", [0x6512] = "\x94\xca",
  [0x6513] = "\xec\xb5", [0x6514] = "\x94\xcb", [0x6515] = "\x94\xcc",
  [0x6516] = "\x94\xcd", [0x6517] = "\xb6\xb7", [0x6518] = "\x94\xce",
  [0x6519] = "\xc1\xcf", [0x651a] = "\x94\xcf", [0x651b] = "\xf5\xfa",
  [0x651c] = "\xd0\xb1", [0x651d] = "\x94\xd0", [0x651e] = "\x94\xd1",
  [0x651f] = "\xd5\xe5", [0x6520] = "\x94\xd2", [0x6521] = "\xce\xd3",
  [0x6522] = "\x94\xd3", [0x6523] = "\x94\xd4", [0x6524] = "\xbd\xef",
  [0x6525] = "\xb3\xe2", [0x6526] = "\x94\xd5", [0x6527] = "\xb8\xab",
  [0x6528] = "\x94\xd6", [0x6529] = "\xd5\xb6", [0x652a] = "\x94\xd7",
  [0x652b] = "\xed\xbd", [0x652c] = "\x94\xd8", [0x652d] = "\xb6\xcf",
  [0x652e] = "\x94\xd9", [0x652f] = "\xcb\xb9", [0x6530] = "\xd0\xc2",
  [0x6531] = "\x94\xda", [0x6532] = "\x94\xdb", [0x6533] = "\x94\xdc",
  [0x6534] = "\x94\xdd", [0x6535] = "\x94\xde", [0x6536] = "\x94\xdf",
  [0x6537] = "\x94\xe0", [0x6538] = "\x94\xe1", [0x6539] = "\xb7\xbd",
  [0x653a] = "\x94\xe2", [0x653b] = "\x94\xe3", [0x653c] = "\xec\xb6",
  [0x653d] = "\xca\xa9", [0x653e] = "\x94\xe4", [0x653f] = "\x94\xe5",
  [0x6540] = "\x94\xe6", [0x6541] = "\xc5\xd4", [0x6542] = "\x94\xe7",
  [0x6543] = "\xec\xb9", [0x6544] = "\xec\xb8", [0x6545] = "\xc2\xc3",
  [0x6546] = "\xec\xb7", [0x6547] = "\x94\xe8", [0x6548] = "\x94\xe9",
  [0x6549] = "\x94\xea", [0x654a] = "\x94\xeb", [0x654b] = "\xd0\xfd",
  [0x654c] = "\xec\xba", [0x654d] = "\x94\xec", [0x654e] = "\xec\xbb",
  [0x654f] = "\xd7\xe5", [0x6550] = "\x94\xed", [0x6551] = "\x94\xee",
  [0x6552] = "\xec\xbc", [0x6553] = "\x94\xef", [0x6554] = "\x94\xf0",
  [0x6555] = "\x94\xf1", [0x6556] = "\xec\xbd", [0x6557] = "\xc6\xec",
  [0x6558] = "\x94\xf2", [0x6559] = "\x94\xf3", [0x655a] = "\x94\xf4",
  [0x655b] = "\x94\xf5", [0x655c] = "\x94\xf6", [0x655d] = "\x94\xf7",
  [0x655e] = "\x94\xf8", [0x655f] = "\x94\xf9", [0x6560] = "\xce\xde",
  [0x6561] = "\x94\xfa", [0x6562] = "\xbc\xc8", [0x6563] = "\x94\xfb",
  [0x6564] = "\x94\xfc", [0x6565] = "\xc8\xd5", [0x6566] = "\xb5\xa9",
  [0x6567] = "\xbe\xc9", [0x6568] = "\xd6\xbc", [0x6569] = "\xd4\xe7",
  [0x656a] = "\x94\xfd", [0x656b] = "\x94\xfe", [0x656c] = "\xd1\xae",
  [0x656d] = "\xd0\xf1", [0x656e] = "\xea\xb8", [0x656f] = "\xea\xb9",
  [0x6570] = "\xea\xba", [0x6571] = "\xba\xb5", [0x6572] = "\x95\x40",
  [0x6573] = "\x95\x41", [0x6574] = "\x95\x42", [0x6575] = "\x95\x43",
  [0x6576] = "\xca\xb1", [0x6577] = "\xbf\xf5", [0x6578] = "\x95\x44",
  [0x6579] = "\x95\x45", [0x657a] = "\xcd\xfa", [0x657b] = "\x95\x46",
  [0x657c] = "\x95\x47", [0x657d] = "\x95\x48", [0x657e] = "\x95\x49",
  [0x657f] = "\x95\x4a", [0x6580] = "\xea\xc0", [0x6581] = "\x95\x4b",
  [0x6582] = "\xb0\xba", [0x6583] = "\xea\xbe", [0x6584] = "\x95\x4c",
  [0x6585] = "\x95\x4d", [0x6586] = "\xc0\xa5", [0x6587] = "\x95\x4e",
  [0x6588] = "\x95\x4f", [0x6589] = "\x95\x50", [0x658a] = "\xea\xbb",
  [0x658b] = "\x95\x51", [0x658c] = "\xb2\xfd", [0x658d] = "\x95\x52",
  [0x658e] = "\xc3\xf7", [0x658f] = "\xbb\xe8", [0x6590] = "\x95\x53",
  [0x6591] = "\x95\x54", [0x6592] = "\x95\x55", [0x6593] = "\xd2\xd7",
  [0x6594] = "\xce\xf4", [0x6595] = "\xea\xbf", [0x6596] = "\x95\x56",
  [0x6597] = "\x95\x57", [0x6598] = "\x95\x58", [0x6599] = "\xea\xbc",
  [0x659a] = "\x95\x59", [0x659b] = "\x95\x5a", [0x659c] = "\x95\x5b",
  [0x659d] = "\xea\xc3", [0x659e] = "\x95\x5c", [0x659f] = "\xd0\xc7",
  [0x65a0] = "\xd3\xb3", [0x65a1] = "\x95\x5d", [0x65a2] = "\x95\x5e",
  [0x65a3] = "\x95\x5f", [0x65a4] = "\x95\x60", [0x65a5] = "\xb4\xba",
  [0x65a6] = "\x95\x61", [0x65a7] = "\xc3\xc1", [0x65a8] = "\xd7\xf2",
  [0x65a9] = "\x95\x62", [0x65aa] = "\x95\x63", [0x65ab] = "\x95\x64",
  [0x65ac] = "\x95\x65", [0x65ad] = "\xd5\xd1", [0x65ae] = "\x95\x66",
  [0x65af] = "\xca\xc7", [0x65b0] = "\x95\x67", [0x65b1] = "\xea\xc5",
  [0x65b2] = "\x95\x68", [0x65b3] = "\x95\x69", [0x65b4] = "\xea\xc4",
  [0x65b5] = "\xea\xc7", [0x65b6] = "\xea\xc6", [0x65b7] = "\x95\x6a",
  [0x65b8] = "\x95\x6b", [0x65b9] = "\x95\x6c", [0x65ba] = "\x95\x6d",
  [0x65bb] = "\x95\x6e", [0x65bc] = "\xd6\xe7", [0x65bd] = "\x95\x6f",
  [0x65be] = "\xcf\xd4", [0x65bf] = "\x95\x70", [0x65c0] = "\x95\x71",
  [0x65c1] = "\xea\xcb", [0x65c2] = "\x95\x72", [0x65c3] = "\xbb\xce",
  [0x65c4] = "\x95\x73", [0x65c5] = "\x95\x74", [0x65c6] = "\x95\x75",
  [0x65c7] = "\x95\x76", [0x65c8] = "\x95\x77", [0x65c9] = "\x95\x78",
  [0x65ca] = "\x95\x79", [0x65cb] = "\xbd\xfa", [0x65cc] = "\xc9\xce",
  [0x65cd] = "\x95\x7a", [0x65ce] = "\x95\x7b", [0x65cf] = "\xea\xcc",
  [0x65d0] = "\x95\x7c", [0x65d1] = "\x95\x7d", [0x65d2] = "\xc9\xb9",
  [0x65d3] = "\xcf\xfe", [0x65d4] = "\xea\xca", [0x65d5] = "\xd4\xce",
  [0x65d6] = "\xea\xcd", [0x65d7] = "\xea\xcf", [0x65d8] = "\x95\x7e",
  [0x65d9] = "\x95\x80", [0x65da] = "\xcd\xed", [0x65db] = "\x95\x81",
  [0x65dc] = "\x95\x82", [0x65dd] = "\x95\x83", [0x65de] = "\x95\x84",
  [0x65df] = "\xea\xc9", [0x65e0] = "\x95\x85", [0x65e1] = "\xea\xce",
  [0x65e2] = "\x95\x86", [0x65e3] = "\x95\x87", [0x65e4] = "\xce\xee",
  [0x65e5] = "\x95\x88", [0x65e6] = "\xbb\xde", [0x65e7] = "\x95\x89",
  [0x65e8] = "\xb3\xbf", [0x65e9] = "\x95\x8a", [0x65ea] = "\x95\x8b",
  [0x65eb] = "\x95\x8c", [0x65ec] = "\x95\x8d", [0x65ed] = "\x95\x8e",
  [0x65ee] = "\xc6\xd5", [0x65ef] = "\xbe\xb0", [0x65f0] = "\xce\xfa",
  [0x65f1] = "\x95\x8f", [0x65f2] = "\x95\x90", [0x65f3] = "\x95\x91",
  [0x65f4] = "\xc7\xe7", [0x65f5] = "\x95\x92", [0x65f6] = "\xbe\xa7",
  [0x65f7] = "\xea\xd0", [0x65f8] = "\x95\x93", [0x65f9] = "\x95\x94",
  [0x65fa] = "\xd6\xc7", [0x65fb] = "\x95\x95", [0x65fc] = "\x95\x96",
  [0x65fd] = "\x95\x97", [0x65fe] = "\xc1\xc0", [0x65ff] = "\x95\x98",
  [0x6600] = "\x95\x99", [0x6601] = "\x95\x9a", [0x6602] = "\xd4\xdd",
  [0x6603] = "\x95\x9b", [0x6604] = "\xea\xd1", [0x6605] = "\x95\x9c",
  [0x6606] = "\x95\x9d", [0x6607] = "\xcf\xbe", [0x6608] = "\x95\x9e",
  [0x6609] = "\x95\x9f", [0x660a] = "\x95\xa0", [0x660b] = "\x95\xa1",
  [0x660c] = "\xea\xd2", [0x660d] = "\x95\xa2", [0x660e] = "\x95\xa3",
  [0x660f] = "\x95\xa4", [0x6610] = "\x95\xa5", [0x6611] = "\xca\xee",
  [0x6612] = "\x95\xa6", [0x6613] = "\x95\xa7", [0x6614] = "\x95\xa8",
  [0x6615] = "\x95\xa9", [0x6616] = "\xc5\xaf", [0x6617] = "\xb0\xb5",
  [0x6618] = "\x95\xaa", [0x6619] = "\x95\xab", [0x661a] = "\x95\xac",
  [0x661b] = "\x95\xad", [0x661c] = "\x95\xae", [0x661d] = "\xea\xd4",
  [0x661e] = "\x95\xaf", [0x661f] = "\x95\xb0", [0x6620] = "\x95\xb1",
  [0x6621] = "\x95\xb2", [0x6622] = "\x95\xb3", [0x6623] = "\x95\xb4",
  [0x6624] = "\x95\xb5", [0x6625] = "\x95\xb6", [0x6626] = "\x95\xb7",
  [0x6627] = "\xea\xd3", [0x6628] = "\xf4\xdf", [0x6629] = "\x95\xb8",
  [0x662a] = "\x95\xb9", [0x662b] = "\x95\xba", [0x662c] = "\x95\xbb",
  [0x662d] = "\x95\xbc", [0x662e] = "\xc4\xba", [0x662f] = "\x95\xbd",
  [0x6630] = "\x95\xbe", [0x6631] = "\x95\xbf", [0x6632] = "\x95\xc0",
  [0x6633] = "\x95\xc1", [0x6634] = "\xb1\xa9", [0x6635] = "\x95\xc2",
  [0x6636] = "\x95\xc3", [0x6637] = "\x95\xc4", [0x6638] = "\x95\xc5",
  [0x6639] = "\xe5\xdf", [0x663a] = "\x95\xc6", [0x663b] = "\x95\xc7",
  [0x663c] = "\x95\xc8", [0x663d] = "\x95\xc9", [0x663e] = "\xea\xd5",
  [0x663f] = "\x95\xca", [0x6640] = "\x95\xcb", [0x6641] = "\x95\xcc",
  [0x6642] = "\x95\xcd", [0x6643] = "\x95\xce", [0x6644] = "\x95\xcf",
  [0x6645] = "\x95\xd0", [0x6646] = "\x95\xd1", [0x6647] = "\x95\xd2",
  [0x6648] = "\x95\xd3", [0x6649] = "\x95\xd4", [0x664a] = "\x95\xd5",
  [0x664b] = "\x95\xd6", [0x664c] = "\x95\xd7", [0x664d] = "\x95\xd8",
  [0x664e] = "\x95\xd9", [0x664f] = "\x95\xda", [0x6650] = "\x95\xdb",
  [0x6651] = "\x95\xdc", [0x6652] = "\x95\xdd", [0x6653] = "\x95\xde",
  [0x6654] = "\x95\xdf", [0x6655] = "\x95\xe0", [0x6656] = "\x95\xe1",
  [0x6657] = "\x95\xe2", [0x6658] = "\x95\xe3", [0x6659] = "\xca\xef",
  [0x665a] = "\x95\xe4", [0x665b] = "\xea\xd6", [0x665c] = "\xea\xd7",
  [0x665d] = "\xc6\xd8", [0x665e] = "\x95\xe5", [0x665f] = "\x95\xe6",
  [0x6660] = "\x95\xe7", [0x6661] = "\x95\xe8", [0x6662] = "\x95\xe9",
  [0x6663] = "\x95\xea", [0x6664] = "\x95\xeb", [0x6665] = "\x95\xec",
  [0x6666] = "\xea\xd8", [0x6667] = "\x95\xed", [0x6668] = "\x95\xee",
  [0x6669] = "\xea\xd9", [0x666a] = "\x95\xef", [0x666b] = "\x95\xf0",
  [0x666c] = "\x95\xf1", [0x666d] = "\x95\xf2", [0x666e] = "\x95\xf3",
  [0x666f] = "\x95\xf4", [0x6670] = "\xd4\xbb", [0x6671] = "\x95\xf5",
  [0x6672] = "\xc7\xfa", [0x6673] = "\xd2\xb7", [0x6674] = "\xb8\xfc",
  [0x6675] = "\x95\xf6", [0x6676] = "\x95\xf7", [0x6677] = "\xea\xc2",
  [0x6678] = "\x95\xf8", [0x6679] = "\xb2\xdc", [0x667a] = "\x95\xf9",
  [0x667b] = "\x95\xfa", [0x667c] = "\xc2\xfc", [0x667d] = "\x95\xfb",
  [0x667e] = "\xd4\xf8", [0x667f] = "\xcc\xe6", [0x6680] = "\xd7\xee",
  [0x6681] = "\x95\xfc", [0x6682] = "\x95\xfd", [0x6683] = "\x95\xfe",
  [0x6684] = "\x96\x40", [0x6685] = "\x96\x41", [0x6686] = "\x96\x42",
  [0x6687] = "\x96\x43", [0x6688] = "\xd4\xc2", [0x6689] = "\xd3\xd0",
  [0x668a] = "\xeb\xc3", [0x668b] = "\xc5\xf3", [0x668c] = "\x96\x44",
  [0x668d] = "\xb7\xfe", [0x668e] = "\x96\x45", [0x668f] = "\x96\x46",
  [0x6690] = "\xeb\xd4", [0x6691] = "\x96\x47", [0x6692] = "\x96\x48",
  [0x6693] = "\x96\x49", [0x6694] = "\xcb\xb7", [0x6695] = "\xeb\xde",
  [0x6696] = "\x96\x4a", [0x6697] = "\xc0\xca", [0x6698] = "\x96\x4b",
  [0x6699] = "\x96\x4c", [0x669a] = "\x96\x4d", [0x669b] = "\xcd\xfb",
  [0x669c] = "\x96\x4e", [0x669d] = "\xb3\xaf", [0x669e] = "\x96\x4f",
  [0x669f] = "\xc6\xda", [0x66a0] = "\x96\x50", [0x66a1] = "\x96\x51",
  [0x66a2] = "\x96\x52", [0x66a3] = "\x96\x53", [0x66a4] = "\x96\x54",
  [0x66a5] = "\x96\x55", [0x66a6] = "\xeb\xfc", [0x66a7] = "\x96\x56",
  [0x66a8] = "\xc4\xbe", [0x66a9] = "\x96\x57", [0x66aa] = "\xce\xb4",
  [0x66ab] = "\xc4\xa9", [0x66ac] = "\xb1\xbe", [0x66ad] = "\xd4\xfd",
  [0x66ae] = "\x96\x58", [0x66af] = "\xca\xf5", [0x66b0] = "\x96\x59",
  [0x66b1] = "\xd6\xec", [0x66b2] = "\x96\x5a", [0x66b3] = "\x96\x5b",
  [0x66b4] = "\xc6\xd3", [0x66b5] = "\xb6\xe4", [0x66b6] = "\x96\x5c",
  [0x66b7] = "\x96\x5d", [0x66b8] = "\x96\x5e", [0x66b9] = "\x96\x5f",
  [0x66ba] = "\xbb\xfa", [0x66bb] = "\x96\x60", [0x66bc] = "\x96\x61",
  [0x66bd] = "\xd0\xe0", [0x66be] = "\x96\x62", [0x66bf] = "\x96\x63",
  [0x66c0] = "\xc9\xb1", [0x66c1] = "\x96\x64", [0x66c2] = "\xd4\xd3",
  [0x66c3] = "\xc8\xa8", [0x66c4] = "\x96\x65", [0x66c5] = "\x96\x66",
  [0x66c6] = "\xb8\xcb", [0x66c7] = "\x96\x67", [0x66c8] = "\xe8\xbe",
  [0x66c9] = "\xc9\xbc", [0x66ca] = "\x96\x68", [0x66cb] = "\x96\x69",
  [0x66cc] = "\xe8\xbb", [0x66cd] = "\x96\x6a", [0x66ce] = "\xc0\xee",
  [0x66cf] = "\xd0\xd3", [0x66d0] = "\xb2\xc4", [0x66d1] = "\xb4\xe5",
  [0x66d2] = "\x96\x6b", [0x66d3] = "\xe8\xbc", [0x66d4] = "\x96\x6c",
  [0x66d5] = "\x96\x6d", [0x66d6] = "\xd5\xc8", [0x66d7] = "\x96\x6e",
  [0x66d8] = "\x96\x6f", [0x66d9] = "\x96\x70", [0x66da] = "\x96\x71",
  [0x66db] = "\x96\x72", [0x66dc] = "\xb6\xc5", [0x66dd] = "\x96\x73",
  [0x66de] = "\xe8\xbd", [0x66df] = "\xca\xf8", [0x66e0] = "\xb8\xdc",
  [0x66e1] = "\xcc\xf5", [0x66e2] = "\x96\x74", [0x66e3] = "\x96\x75",
  [0x66e4] = "\x96\x76", [0x66e5] = "\xc0\xb4", [0x66e6] = "\x96\x77",
  [0x66e7] = "\x96\x78", [0x66e8] = "\xd1\xee", [0x66e9] = "\xe8\xbf",
  [0x66ea] = "\xe8\xc2", [0x66eb] = "\x96\x79", [0x66ec] = "\x96\x7a",
  [0x66ed] = "\xba\xbc", [0x66ee] = "\x96\x7b", [0x66ef] = "\xb1\xad",
  [0x66f0] = "\xbd\xdc", [0x66f1] = "\x96\x7c", [0x66f2] = "\xea\xbd",
  [0x66f3] = "\xe8\xc3", [0x66f4] = "\x96\x7d", [0x66f5] = "\xe8\xc6",
  [0x66f6] = "\x96\x7e", [0x66f7] = "\xe8\xcb", [0x66f8] = "\x96\x80",
  [0x66f9] = "\x96\x81", [0x66fa] = "\x96\x82", [0x66fb] = "\x96\x83",
  [0x66fc] = "\xe8\xcc", [0x66fd] = "\x96\x84", [0x66fe] = "\xcb\xc9",
  [0x66ff] = "\xb0\xe5", [0x6700] = "\x96\x85", [0x6701] = "\xbc\xab",
  [0x6702] = "\x96\x86", [0x6703] = "\x96\x87", [0x6704] = "\xb9\xb9",
  [0x6705] = "\x96\x88", [0x6706] = "\x96\x89", [0x6707] = "\xe8\xc1",
  [0x6708] = "\x96\x8a", [0x6709] = "\xcd\xf7", [0x670a] = "\x96\x8b",
  [0x670b] = "\xe8\xca", [0x670c] = "\x96\x8c", [0x670d] = "\x96\x8d",
  [0x670e] = "\x96\x8e", [0x670f] = "\x96\x8f", [0x6710] = "\xce\xf6",
  [0x6711] = "\x96\x90", [0x6712] = "\x96\x91", [0x6713] = "\x96\x92",
  [0x6714] = "\x96\x93", [0x6715] = "\xd5\xed", [0x6716] = "\x96\x94",
  [0x6717] = "\xc1\xd6", [0x6718] = "\xe8\xc4", [0x6719] = "\x96\x95",
  [0x671a] = "\xc3\xb6", [0x671b] = "\x96\x96", [0x671c] = "\xb9\xfb",
  [0x671d] = "\xd6\xa6", [0x671e] = "\xe8\xc8", [0x671f] = "\x96\x97",
  [0x6720] = "\x96\x98", [0x6721] = "\x96\x99", [0x6722] = "\xca\xe0",
  [0x6723] = "\xd4\xe6", [0x6724] = "\x96\x9a", [0x6725] = "\xe8\xc0",
  [0x6726] = "\x96\x9b", [0x6727] = "\xe8\xc5", [0x6728] = "\xe8\xc7",
  [0x6729] = "\x96\x9c", [0x672a] = "\xc7\xb9", [0x672b] = "\xb7\xe3",
  [0x672c] = "\x96\x9d", [0x672d] = "\xe8\xc9", [0x672e] = "\x96\x9e",
  [0x672f] = "\xbf\xdd", [0x6730] = "\xe8\xd2", [0x6731] = "\x96\x9f",
  [0x6732] = "\x96\xa0", [0x6733] = "\xe8\xd7", [0x6734] = "\x96\xa1",
  [0x6735] = "\xe8\xd5", [0x6736] = "\xbc\xdc", [0x6737] = "\xbc\xcf",
  [0x6738] = "\xe8\xdb", [0x6739] = "\x96\xa2", [0x673a] = "\x96\xa3",
  [0x673b] = "\x96\xa4", [0x673c] = "\x96\xa5", [0x673d] = "\x96\xa6",
  [0x673e] = "\x96\xa7", [0x673f] = "\x96\xa8", [0x6740] = "\x96\xa9",
  [0x6741] = "\xe8\xde", [0x6742] = "\x96\xaa", [0x6743] = "\xe8\xda",
  [0x6744] = "\xb1\xfa", [0x6745] = "\x96\xab", [0x6746] = "\x96\xac",
  [0x6747] = "\x96\xad", [0x6748] = "\x96\xae", [0x6749] = "\x96\xaf",
  [0x674a] = "\x96\xb0", [0x674b] = "\x96\xb1", [0x674c] = "\x96\xb2",
  [0x674d] = "\x96\xb3", [0x674e] = "\x96\xb4", [0x674f] = "\xb0\xd8",
  [0x6750] = "\xc4\xb3", [0x6751] = "\xb8\xcc", [0x6752] = "\xc6\xe2",
  [0x6753] = "\xc8\xbe", [0x6754] = "\xc8\xe1", [0x6755] = "\x96\xb5",
  [0x6756] = "\x96\xb6", [0x6757] = "\x96\xb7", [0x6758] = "\xe8\xcf",
  [0x6759] = "\xe8\xd4", [0x675a] = "\xe8\xd6", [0x675b] = "\x96\xb8",
  [0x675c] = "\xb9\xf1", [0x675d] = "\xe8\xd8", [0x675e] = "\xd7\xf5",
  [0x675f] = "\x96\xb9", [0x6760] = "\xc4\xfb", [0x6761] = "\x96\xba",
  [0x6762] = "\xe8\xdc", [0x6763] = "\x96\xbb", [0x6764] = "\x96\xbc",
  [0x6765] = "\xb2\xe9", [0x6766] = "\x96\xbd", [0x6767] = "\x96\xbe",
  [0x6768] = "\x96\xbf", [0x6769] = "\xe8\xd1", [0x676a] = "\x96\xc0",
  [0x676b] = "\x96\xc1", [0x676c] = "\xbc\xed", [0x676d] = "\x96\xc2",
  [0x676e] = "\x96\xc3", [0x676f] = "\xbf\xc2", [0x6770] = "\xe8\xcd",
  [0x6771] = "\xd6\xf9", [0x6772] = "\x96\xc4", [0x6773] = "\xc1\xf8",
  [0x6774] = "\xb2\xf1", [0x6775] = "\x96\xc5", [0x6776] = "\x96\xc6",
  [0x6777] = "\x96\xc7", [0x6778] = "\x96\xc8", [0x6779] = "\x96\xc9",
  [0x677a] = "\x96\xca", [0x677b] = "\x96\xcb", [0x677c] = "\x96\xcc",
  [0x677d] = "\xe8\xdf", [0x677e] = "\x96\xcd", [0x677f] = "\xca\xc1",
  [0x6780] = "\xe8\xd9", [0x6781] = "\x96\xce", [0x6782] = "\x96\xcf",
  [0x6783] = "\x96\xd0", [0x6784] = "\x96\xd1", [0x6785] = "\xd5\xa4",
  [0x6786] = "\x96\xd2", [0x6787] = "\xb1\xea", [0x6788] = "\xd5\xbb",
  [0x6789] = "\xe8\xce", [0x678a] = "\xe8\xd0", [0x678b] = "\xb6\xb0",
  [0x678c] = "\xe8\xd3", [0x678d] = "\x96\xd3", [0x678e] = "\xe8\xdd",
  [0x678f] = "\xc0\xb8", [0x6790] = "\x96\xd4", [0x6791] = "\xca\xf7",
  [0x6792] = "\x96\xd5", [0x6793] = "\xcb\xa8", [0x6794] = "\x96\xd6",
  [0x6795] = "\x96\xd7", [0x6796] = "\xc6\xdc", [0x6797] = "\xc0\xf5",
  [0x6798] = "\x96\xd8", [0x6799] = "\x96\xd9", [0x679a] = "\x96\xda",
  [0x679b] = "\x96\xdb", [0x679c] = "\x96\xdc", [0x679d] = "\xe8\xe9",
  [0x679e] = "\x96\xdd", [0x679f] = "\x96\xde", [0x67a0] = "\x96\xdf",
  [0x67a1] = "\xd0\xa3", [0x67a2] = "\x96\xe0", [0x67a3] = "\x96\xe1",
  [0x67a4] = "\x96\xe2", [0x67a5] = "\x96\xe3", [0x67a6] = "\x96\xe4",
  [0x67a7] = "\x96\xe5", [0x67a8] = "\x96\xe6", [0x67a9] = "\xe8\xf2",
  [0x67aa] = "\xd6\xea", [0x67ab] = "\x96\xe7", [0x67ac] = "\x96\xe8",
  [0x67ad] = "\x96\xe9", [0x67ae] = "\x96\xea", [0x67af] = "\x96\xeb",
  [0x67b0] = "\x96\xec", [0x67b1] = "\x96\xed", [0x67b2] = "\xe8\xe0",
  [0x67b3] = "\xe8\xe1", [0x67b4] = "\x96\xee", [0x67b5] = "\x96\xef",
  [0x67b6] = "\x96\xf0", [0x67b7] = "\xd1\xf9", [0x67b8] = "\xba\xcb",
  [0x67b9] = "\xb8\xf9", [0x67ba] = "\x96\xf1", [0x67bb] = "\x96\xf2",
  [0x67bc] = "\xb8\xf1", [0x67bd] = "\xd4\xd4", [0x67be] = "\xe8\xef",
  [0x67bf] = "\x96\xf3", [0x67c0] = "\xe8\xee", [0x67c1] = "\xe8\xec",
  [0x67c2] = "\xb9\xf0", [0x67c3] = "\xcc\xd2", [0x67c4] = "\xe8\xe6",
  [0x67c5] = "\xce\xa6", [0x67c6] = "\xbf\xf2", [0x67c7] = "\x96\xf4",
  [0x67c8] = "\xb0\xb8", [0x67c9] = "\xe8\xf1", [0x67ca] = "\xe8\xf0",
  [0x67cb] = "\x96\xf5", [0x67cc] = "\xd7\xc0", [0x67cd] = "\x96\xf6",
  [0x67ce] = "\xe8\xe4", [0x67cf] = "\x96\xf7", [0x67d0] = "\xcd\xa9",
  [0x67d1] = "\xc9\xa3", [0x67d2] = "\x96\xf8", [0x67d3] = "\xbb\xb8",
  [0x67d4] = "\xbd\xdb", [0x67d5] = "\xe8\xea", [0x67d6] = "\x96\xf9",
  [0x67d7] = "\x96\xfa", [0x67d8] = "\x96\xfb", [0x67d9] = "\x96\xfc",
  [0x67da] = "\x96\xfd", [0x67db] = "\x96\xfe", [0x67dc] = "\x97\x40",
  [0x67dd] = "\x97\x41", [0x67de] = "\x97\x42", [0x67df] = "\x97\x43",
  [0x67e0] = "\xe8\xe2", [0x67e1] = "\xe8\xe3", [0x67e2] = "\xe8\xe5",
  [0x67e3] = "\xb5\xb5", [0x67e4] = "\xe8\xe7", [0x67e5] = "\xc7\xc5",
  [0x67e6] = "\xe8\xeb", [0x67e7] = "\xe8\xed", [0x67e8] = "\xbd\xb0",
  [0x67e9] = "\xd7\xae", [0x67ea] = "\x97\x44", [0x67eb] = "\xe8\xf8",
  [0x67ec] = "\x97\x45", [0x67ed] = "\x97\x46", [0x67ee] = "\x97\x47",
  [0x67ef] = "\x97\x48", [0x67f0] = "\x97\x49", [0x67f1] = "\x97\x4a",
  [0x67f2] = "\x97\x4b", [0x67f3] = "\x97\x4c", [0x67f4] = "\xe8\xf5",
  [0x67f5] = "\x97\x4d", [0x67f6] = "\xcd\xb0", [0x67f7] = "\xe8\xf6",
  [0x67f8] = "\x97\x4e", [0x67f9] = "\x97\x4f", [0x67fa] = "\x97\x50",
  [0x67fb] = "\x97\x51", [0x67fc] = "\x97\x52", [0x67fd] = "\x97\x53",
  [0x67fe] = "\x97\x54", [0x67ff] = "\x97\x55", [0x6800] = "\x97\x56",
  [0x6801] = "\xc1\xba", [0x6802] = "\x97\x57", [0x6803] = "\xe8\xe8",
  [0x6804] = "\x97\x58", [0x6805] = "\xc3\xb7", [0x6806] = "\xb0\xf0",
  [0x6807] = "\x97\x59", [0x6808] = "\x97\x5a", [0x6809] = "\x97\x5b",
  [0x680a] = "\x97\x5c", [0x680b] = "\x97\x5d", [0x680c] = "\x97\x5e",
  [0x680d] = "\x97\x5f", [0x680e] = "\x97\x60", [0x680f] = "\xe8\xf4",
  [0x6810] = "\x97\x61", [0x6811] = "\x97\x62", [0x6812] = "\x97\x63",
  [0x6813] = "\xe8\xf7", [0x6814] = "\x97\x64", [0x6815] = "\x97\x65",
  [0x6816] = "\x97\x66", [0x6817] = "\xb9\xa3", [0x6818] = "\x97\x67",
  [0x6819] = "\x97\x68", [0x681a] = "\x97\x69", [0x681b] = "\x97\x6a",
  [0x681c] = "\x97\x6b", [0x681d] = "\x97\x6c", [0x681e] = "\x97\x6d",
  [0x681f] = "\x97\x6e", [0x6820] = "\x97\x6f", [0x6821] = "\x97\x70",
  [0x6822] = "\xc9\xd2", [0x6823] = "\x97\x71", [0x6824] = "\x97\x72",
  [0x6825] = "\x97\x73", [0x6826] = "\xc3\xce", [0x6827] = "\xce\xe0",
  [0x6828] = "\xc0\xe6", [0x6829] = "\x97\x74", [0x682a] = "\x97\x75",
  [0x682b] = "\x97\x76", [0x682c] = "\x97\x77", [0x682d] = "\xcb\xf3",
  [0x682e] = "\x97\x78", [0x682f] = "\xcc\xdd", [0x6830] = "\xd0\xb5",
  [0x6831] = "\x97\x79", [0x6832] = "\x97\x7a", [0x6833] = "\xca\xe1",
  [0x6834] = "\x97\x7b", [0x6835] = "\xe8\xf3", [0x6836] = "\x97\x7c",
  [0x6837] = "\x97\x7d", [0x6838] = "\x97\x7e", [0x6839] = "\x97\x80",
  [0x683a] = "\x97\x81", [0x683b] = "\x97\x82", [0x683c] = "\x97\x83",
  [0x683d] = "\x97\x84", [0x683e] = "\x97\x85", [0x683f] = "\x97\x86",
  [0x6840] = "\xbc\xec", [0x6841] = "\x97\x87", [0x6842] = "\xe8\xf9",
  [0x6843] = "\x97\x88", [0x6844] = "\x97\x89", [0x6845] = "\x97\x8a",
  [0x6846] = "\x97\x8b", [0x6847] = "\x97\x8c", [0x6848] = "\x97\x8d",
  [0x6849] = "\xc3\xde", [0x684a] = "\x97\x8e", [0x684b] = "\xc6\xe5",
  [0x684c] = "\x97\x8f", [0x684d] = "\xb9\xf7", [0x684e] = "\x97\x90",
  [0x684f] = "\x97\x91", [0x6850] = "\x97\x92", [0x6851] = "\x97\x93",
  [0x6852] = "\xb0\xf4", [0x6853] = "\x97\x94", [0x6854] = "\x97\x95",
  [0x6855] = "\xd7\xd8", [0x6856] = "\x97\x96", [0x6857] = "\x97\x97",
  [0x6858] = "\xbc\xac", [0x6859] = "\x97\x98", [0x685a] = "\xc5\xef",
  [0x685b] = "\x97\x99", [0x685c] = "\x97\x9a", [0x685d] = "\x97\x9b",
  [0x685e] = "\x97\x9c", [0x685f] = "\x97\x9d", [0x6860] = "\xcc\xc4",
  [0x6861] = "\x97\x9e", [0x6862] = "\x97\x9f", [0x6863] = "\xe9\xa6",
  [0x6864] = "\x97\xa0", [0x6865] = "\x97\xa1", [0x6866] = "\x97\xa2",
  [0x6867] = "\x97\xa3", [0x6868] = "\x97\xa4", [0x6869] = "\x97\xa5",
  [0x686a] = "\x97\xa6", [0x686b] = "\x97\xa7", [0x686c] = "\x97\xa8",
  [0x686d] = "\x97\xa9", [0x686e] = "\xc9\xad", [0x686f] = "\x97\xaa",
  [0x6870] = "\xe9\xa2", [0x6871] = "\xc0\xe2", [0x6872] = "\x97\xab",
  [0x6873] = "\x97\xac", [0x6874] = "\x97\xad", [0x6875] = "\xbf\xc3",
  [0x6876] = "\x97\xae", [0x6877] = "\x97\xaf", [0x6878] = "\x97\xb0",
  [0x6879] = "\xe8\xfe", [0x687a] = "\xb9\xd7", [0x687b] = "\x97\xb1",
  [0x687c] = "\xe8\xfb", [0x687d] = "\x97\xb2", [0x687e] = "\x97\xb3",
  [0x687f] = "\x97\xb4", [0x6880] = "\x97\xb5", [0x6881] = "\xe9\xa4",
  [0x6882] = "\x97\xb6", [0x6883] = "\x97\xb7", [0x6884] = "\x97\xb8",
  [0x6885] = "\xd2\xce", [0x6886] = "\x97\xb9", [0x6887] = "\x97\xba",
  [0x6888] = "\x97\xbb", [0x6889] = "\x97\xbc", [0x688a] = "\x97\xbd",
  [0x688b] = "\xe9\xa3", [0x688c] = "\x97\xbe", [0x688d] = "\xd6\xb2",
  [0x688e] = "\xd7\xb5", [0x688f] = "\x97\xbf", [0x6890] = "\xe9\xa7",
  [0x6891] = "\x97\xc0", [0x6892] = "\xbd\xb7", [0x6893] = "\x97\xc1",
  [0x6894] = "\x97\xc2", [0x6895] = "\x97\xc3", [0x6896] = "\x97\xc4",
  [0x6897] = "\x97\xc5", [0x6898] = "\x97\xc6", [0x6899] = "\x97\xc7",
  [0x689a] = "\x97\xc8", [0x689b] = "\x97\xc9", [0x689c] = "\x97\xca",
  [0x689d] = "\x97\xcb", [0x689e] = "\x97\xcc", [0x689f] = "\xe8\xfc",
  [0x68a0] = "\xe8\xfd", [0x68a1] = "\x97\xcd", [0x68a2] = "\x97\xce",
  [0x68a3] = "\x97\xcf", [0x68a4] = "\xe9\xa1", [0x68a5] = "\x97\xd0",
  [0x68a6] = "\x97\xd1", [0x68a7] = "\x97\xd2", [0x68a8] = "\x97\xd3",
  [0x68a9] = "\x97\xd4", [0x68aa] = "\x97\xd5", [0x68ab] = "\x97\xd6",
  [0x68ac] = "\x97\xd7", [0x68ad] = "\xcd\xd6", [0x68ae] = "\x97\xd8",
  [0x68af] = "\x97\xd9", [0x68b0] = "\xd2\xac", [0x68b1] = "\x97\xda",
  [0x68b2] = "\x97\xdb", [0x68b3] = "\x97\xdc", [0x68b4] = "\xe9\xb2",
  [0x68b5] = "\x97\xdd", [0x68b6] = "\x97\xde", [0x68b7] = "\x97\xdf",
  [0x68b8] = "\x97\xe0", [0x68b9] = "\xe9\xa9", [0x68ba] = "\x97\xe1",
  [0x68bb] = "\x97\xe2", [0x68bc] = "\x97\xe3", [0x68bd] = "\xb4\xaa",
  [0x68be] = "\x97\xe4", [0x68bf] = "\xb4\xbb", [0x68c0] = "\x97\xe5",
  [0x68c1] = "\x97\xe6", [0x68c2] = "\xe9\xab", [0x68c3] = "\x97\xe7",
  [0x68c4] = "\x97\xe8", [0x68c5] = "\x97\xe9", [0x68c6] = "\x97\xea",
  [0x68c7] = "\x97\xeb", [0x68c8] = "\x97\xec", [0x68c9] = "\x97\xed",
  [0x68ca] = "\x97\xee", [0x68cb] = "\x97\xef", [0x68cc] = "\x97\xf0",
  [0x68cd] = "\x97\xf1", [0x68ce] = "\x97\xf2", [0x68cf] = "\x97\xf3",
  [0x68d0] = "\x97\xf4", [0x68d1] = "\x97\xf5", [0x68d2] = "\x97\xf6",
  [0x68d3] = "\x97\xf7", [0x68d4] = "\xd0\xa8", [0x68d5] = "\x97\xf8",
  [0x68d6] = "\x97\xf9", [0x68d7] = "\xe9\xa5", [0x68d8] = "\x97\xfa",
  [0x68d9] = "\x97\xfb", [0x68da] = "\xb3\xfe", [0x68db] = "\x97\xfc",
  [0x68dc] = "\x97\xfd", [0x68dd] = "\xe9\xac", [0x68de] = "\xc0\xe3",
  [0x68df] = "\x97\xfe", [0x68e0] = "\xe9\xaa", [0x68e1] = "\x98\x40",
  [0x68e2] = "\x98\x41", [0x68e3] = "\xe9\xb9", [0x68e4] = "\x98\x42",
  [0x68e5] = "\x98\x43", [0x68e6] = "\xe9\xb8", [0x68e7] = "\x98\x44",
  [0x68e8] = "\x98\x45", [0x68e9] = "\x98\x46", [0x68ea] = "\x98\x47",
  [0x68eb] = "\xe9\xae", [0x68ec] = "\x98\x48", [0x68ed] = "\x98\x49",
  [0x68ee] = "\xe8\xfa", [0x68ef] = "\x98\x4a", [0x68f0] = "\x98\x4b",
  [0x68f1] = "\xe9\xa8", [0x68f2] = "\x98\x4c", [0x68f3] = "\x98\x4d",
  [0x68f4] = "\x98\x4e", [0x68f5] = "\x98\x4f", [0x68f6] = "\x98\x50",
  [0x68f7] = "\xbf\xac", [0x68f8] = "\xe9\xb1", [0x68f9] = "\xe9\xba",
  [0x68fa] = "\x98\x51", [0x68fb] = "\x98\x52", [0x68fc] = "\xc2\xa5",
  [0x68fd] = "\x98\x53", [0x68fe] = "\x98\x54", [0x68ff] = "\x98\x55",
  [0x6900] = "\xe9\xaf", [0x6901] = "\x98\x56", [0x6902] = "\xb8\xc5",
  [0x6903] = "\x98\x57", [0x6904] = "\xe9\xad", [0x6905] = "\x98\x58",
  [0x6906] = "\xd3\xdc", [0x6907] = "\xe9\xb4", [0x6908] = "\xe9\xb5",
  [0x6909] = "\xe9\xb7", [0x690a] = "\x98\x59", [0x690b] = "\x98\x5a",
  [0x690c] = "\x98\x5b", [0x690d] = "\xe9\xc7", [0x690e] = "\x98\x5c",
  [0x690f] = "\x98\x5d", [0x6910] = "\x98\x5e", [0x6911] = "\x98\x5f",
  [0x6912] = "\x98\x60", [0x6913] = "\x98\x61", [0x6914] = "\xc0\xc6",
  [0x6915] = "\xe9\xc5", [0x6916] = "\x98\x62", [0x6917] = "\x98\x63",
  [0x6918] = "\xe9\xb0", [0x6919] = "\x98\x64", [0x691a] = "\x98\x65",
  [0x691b] = "\xe9\xbb", [0x691c] = "\xb0\xf1", [0x691d] = "\x98\x66",
  [0x691e] = "\x98\x67", [0x691f] = "\x98\x68", [0x6920] = "\x98\x69",
  [0x6921] = "\x98\x6a", [0x6922] = "\x98\x6b", [0x6923] = "\x98\x6c",
  [0x6924] = "\x98\x6d", [0x6925] = "\x98\x6e", [0x6926] = "\x98\x6f",
  [0x6927] = "\xe9\xbc", [0x6928] = "\xd5\xa5", [0x6929] = "\x98\x70",
  [0x692a] = "\x98\x71", [0x692b] = "\xe9\xbe", [0x692c] = "\x98\x72",
  [0x692d] = "\xe9\xbf", [0x692e] = "\x98\x73", [0x692f] = "\x98\x74",
  [0x6930] = "\x98\x75", [0x6931] = "\xe9\xc1", [0x6932] = "\x98\x76",
  [0x6933] = "\x98\x77", [0x6934] = "\xc1\xf1", [0x6935] = "\x98\x78",
  [0x6936] = "\x98\x79", [0x6937] = "\xc8\xb6", [0x6938] = "\x98\x7a",
  [0x6939] = "\x98\x7b", [0x693a] = "\x98\x7c", [0x693b] = "\xe9\xbd",
  [0x693c] = "\x98\x7d", [0x693d] = "\x98\x7e", [0x693e] = "\x98\x80",
  [0x693f] = "\x98\x81", [0x6940] = "\x98\x82", [0x6941] = "\xe9\xc2",
  [0x6942] = "\x98\x83", [0x6943] = "\x98\x84", [0x6944] = "\x98\x85",
  [0x6945] = "\x98\x86", [0x6946] = "\x98\x87", [0x6947] = "\x98\x88",
  [0x6948] = "\x98\x89", [0x6949] = "\x98\x8a", [0x694a] = "\xe9\xc3",
  [0x694b] = "\x98\x8b", [0x694c] = "\xe9\xb3", [0x694d] = "\x98\x8c",
  [0x694e] = "\xe9\xb6", [0x694f] = "\x98\x8d", [0x6950] = "\xbb\xb1",
  [0x6951] = "\x98\x8e", [0x6952] = "\x98\x8f", [0x6953] = "\x98\x90",
  [0x6954] = "\xe9\xc0", [0x6955] = "\x98\x91", [0x6956] = "\x98\x92",
  [0x6957] = "\x98\x93", [0x6958] = "\x98\x94", [0x6959] = "\x98\x95",
  [0x695a] = "\x98\x96", [0x695b] = "\xbc\xf7", [0x695c] = "\x98\x97",
  [0x695d] = "\x98\x98", [0x695e] = "\x98\x99", [0x695f] = "\xe9\xc4",
  [0x6960] = "\xe9\xc6", [0x6961] = "\x98\x9a", [0x6962] = "\x98\x9b",
  [0x6963] = "\x98\x9c", [0x6964] = "\x98\x9d", [0x6965] = "\x98\x9e",
  [0x6966] = "\x98\x9f", [0x6967] = "\x98\xa0", [0x6968] = "\x98\xa1",
  [0x6969] = "\x98\xa2", [0x696a] = "\x98\xa3", [0x696b] = "\x98\xa4",
  [0x696c] = "\x98\xa5", [0x696d] = "\xe9\xca", [0x696e] = "\x98\xa6",
  [0x696f] = "\x98\xa7", [0x6970] = "\x98\xa8", [0x6971] = "\x98\xa9",
  [0x6972] = "\xe9\xce", [0x6973] = "\x98\xaa", [0x6974] = "\x98\xab",
  [0x6975] = "\x98\xac", [0x6976] = "\x98\xad", [0x6977] = "\x98\xae",
  [0x6978] = "\x98\xaf", [0x6979] = "\x98\xb0", [0x697a] = "\x98\xb1",
  [0x697b] = "\x98\xb2", [0x697c] = "\x98\xb3", [0x697d] = "\xb2\xdb",
  [0x697e] = "\x98\xb4", [0x697f] = "\xe9\xc8", [0x6980] = "\x98\xb5",
  [0x6981] = "\x98\xb6", [0x6982] = "\x98\xb7", [0x6983] = "\x98\xb8",
  [0x6984] = "\x98\xb9", [0x6985] = "\x98\xba", [0x6986] = "\x98\xbb",
  [0x6987] = "\x98\xbc", [0x6988] = "\x98\xbd", [0x6989] = "\x98\xbe",
  [0x698a] = "\xb7\xae", [0x698b] = "\x98\xbf", [0x698c] = "\x98\xc0",
  [0x698d] = "\x98\xc1", [0x698e] = "\x98\xc2", [0x698f] = "\x98\xc3",
  [0x6990] = "\x98\xc4", [0x6991] = "\x98\xc5", [0x6992] = "\x98\xc6",
  [0x6993] = "\x98\xc7", [0x6994] = "\x98\xc8", [0x6995] = "\x98\xc9",
  [0x6996] = "\x98\xca", [0x6997] = "\xe9\xcb", [0x6998] = "\xe9\xcc",
  [0x6999] = "\x98\xcb", [0x699a] = "\x98\xcc", [0x699b] = "\x98\xcd",
  [0x699c] = "\x98\xce", [0x699d] = "\x98\xcf", [0x699e] = "\x98\xd0",
  [0x699f] = "\xd5\xc1", [0x69a0] = "\x98\xd1", [0x69a1] = "\xc4\xa3",
  [0x69a2] = "\x98\xd2", [0x69a3] = "\x98\xd3", [0x69a4] = "\x98\xd4",
  [0x69a5] = "\x98\xd5", [0x69a6] = "\x98\xd6", [0x69a7] = "\x98\xd7",
  [0x69a8] = "\xe9\xd8", [0x69a9] = "\x98\xd8", [0x69aa] = "\xba\xe1",
  [0x69ab] = "\x98\xd9", [0x69ac] = "\x98\xda", [0x69ad] = "\x98\xdb",
  [0x69ae] = "\x98\xdc", [0x69af] = "\xe9\xc9", [0x69b0] = "\x98\xdd",
  [0x69b1] = "\xd3\xa3", [0x69b2] = "\x98\xde", [0x69b3] = "\x98\xdf",
  [0x69b4] = "\x98\xe0", [0x69b5] = "\xe9\xd4", [0x69b6] = "\x98\xe1",
  [0x69b7] = "\x98\xe2", [0x69b8] = "\x98\xe3", [0x69b9] = "\x98\xe4",
  [0x69ba] = "\x98\xe5", [0x69bb] = "\x98\xe6", [0x69bc] = "\x98\xe7",
  [0x69bd] = "\xe9\xd7", [0x69be] = "\xe9\xd0", [0x69bf] = "\x98\xe8",
  [0x69c0] = "\x98\xe9", [0x69c1] = "\x98\xea", [0x69c2] = "\x98\xeb",
  [0x69c3] = "\x98\xec", [0x69c4] = "\xe9\xcf", [0x69c5] = "\x98\xed",
  [0x69c6] = "\x98\xee", [0x69c7] = "\xc7\xc1", [0x69c8] = "\x98\xef",
  [0x69c9] = "\x98\xf0", [0x69ca] = "\x98\xf1", [0x69cb] = "\x98\xf2",
  [0x69cc] = "\x98\xf3", [0x69cd] = "\x98\xf4", [0x69ce] = "\x98\xf5",
  [0x69cf] = "\x98\xf6", [0x69d0] = "\xe9\xd2", [0x69d1] = "\x98\xf7",
  [0x69d2] = "\x98\xf8", [0x69d3] = "\x98\xf9", [0x69d4] = "\x98\xfa",
  [0x69d5] = "\x98\xfb", [0x69d6] = "\x98\xfc", [0x69d7] = "\x98\xfd",
  [0x69d8] = "\xe9\xd9", [0x69d9] = "\xb3\xc8", [0x69da] = "\x98\xfe",
  [0x69db] = "\xe9\xd3", [0x69dc] = "\x99\x40", [0x69dd] = "\x99\x41",
  [0x69de] = "\x99\x42", [0x69df] = "\x99\x43", [0x69e0] = "\x99\x44",
  [0x69e1] = "\xcf\xf0", [0x69e2] = "\x99\x45", [0x69e3] = "\x99\x46",
  [0x69e4] = "\x99\x47", [0x69e5] = "\xe9\xcd", [0x69e6] = "\x99\x48",
  [0x69e7] = "\x99\x49", [0x69e8] = "\x99\x4a", [0x69e9] = "\x99\x4b",
  [0x69ea] = "\x99\x4c", [0x69eb] = "\x99\x4d", [0x69ec] = "\x99\x4e",
  [0x69ed] = "\x99\x4f", [0x69ee] = "\x99\x50", [0x69ef] = "\x99\x51",
  [0x69f0] = "\x99\x52", [0x69f1] = "\xb3\xf7", [0x69f2] = "\x99\x53",
  [0x69f3] = "\x99\x54", [0x69f4] = "\x99\x55", [0x69f5] = "\x99\x56",
  [0x69f6] = "\x99\x57", [0x69f7] = "\x99\x58", [0x69f8] = "\x99\x59",
  [0x69f9] = "\xe9\xd6", [0x69fa] = "\x99\x5a", [0x69fb] = "\x99\x5b",
  [0x69fc] = "\xe9\xda", [0x69fd] = "\x99\x5c", [0x69fe] = "\x99\x5d",
  [0x69ff] = "\x99\x5e", [0x6a00] = "\xcc\xb4", [0x6a01] = "\x99\x5f",
  [0x6a02] = "\x99\x60", [0x6a03] = "\x99\x61", [0x6a04] = "\xcf\xad",
  [0x6a05] = "\x99\x62", [0x6a06] = "\x99\x63", [0x6a07] = "\x99\x64",
  [0x6a08] = "\x99\x65", [0x6a09] = "\x99\x66", [0x6a0a] = "\x99\x67",
  [0x6a0b] = "\x99\x68", [0x6a0c] = "\x99\x69", [0x6a0d] = "\x99\x6a",
  [0x6a0e] = "\xe9\xd5", [0x6a0f] = "\x99\x6b", [0x6a10] = "\xe9\xdc",
  [0x6a11] = "\xe9\xdb", [0x6a12] = "\x99\x6c", [0x6a13] = "\x99\x6d",
  [0x6a14] = "\x99\x6e", [0x6a15] = "\x99\x6f", [0x6a16] = "\x99\x70",
  [0x6a17] = "\xe9\xde", [0x6a18] = "\x99\x71", [0x6a19] = "\x99\x72",
  [0x6a1a] = "\x99\x73", [0x6a1b] = "\x99\x74", [0x6a1c] = "\x99\x75",
  [0x6a1d] = "\x99\x76", [0x6a1e] = "\x99\x77", [0x6a1f] = "\x99\x78",
  [0x6a20] = "\xe9\xd1", [0x6a21] = "\x99\x79", [0x6a22] = "\x99\x7a",
  [0x6a23] = "\x99\x7b", [0x6a24] = "\x99\x7c", [0x6a25] = "\x99\x7d",
  [0x6a26] = "\x99\x7e", [0x6a27] = "\x99\x80", [0x6a28] = "\x99\x81",
  [0x6a29] = "\xe9\xdd", [0x6a2a] = "\x99\x82", [0x6a2b] = "\xe9\xdf",
  [0x6a2c] = "\xc3\xca", [0x6a2d] = "\x99\x83", [0x6a2e] = "\x99\x84",
  [0x6a2f] = "\x99\x85", [0x6a30] = "\x99\x86", [0x6a31] = "\x99\x87",
  [0x6a32] = "\x99\x88", [0x6a33] = "\x99\x89", [0x6a34] = "\x99\x8a",
  [0x6a35] = "\x99\x8b", [0x6a36] = "\x99\x8c", [0x6a37] = "\x99\x8d",
  [0x6a38] = "\x99\x8e", [0x6a39] = "\x99\x8f", [0x6a3a] = "\x99\x90",
  [0x6a3b] = "\x99\x91", [0x6a3c] = "\x99\x92", [0x6a3d] = "\x99\x93",
  [0x6a3e] = "\x99\x94", [0x6a3f] = "\x99\x95", [0x6a40] = "\x99\x96",
  [0x6a41] = "\x99\x97", [0x6a42] = "\x99\x98", [0x6a43] = "\x99\x99",
  [0x6a44] = "\x99\x9a", [0x6a45] = "\x99\x9b", [0x6a46] = "\x99\x9c",
  [0x6a47] = "\x99\x9d", [0x6a48] = "\x99\x9e", [0x6a49] = "\x99\x9f",
  [0x6a4a] = "\x99\xa0", [0x6a4b] = "\x99\xa1", [0x6a4c] = "\x99\xa2",
  [0x6a4d] = "\x99\xa3", [0x6a4e] = "\x99\xa4", [0x6a4f] = "\x99\xa5",
  [0x6a50] = "\x99\xa6", [0x6a51] = "\x99\xa7", [0x6a52] = "\x99\xa8",
  [0x6a53] = "\x99\xa9", [0x6a54] = "\x99\xaa", [0x6a55] = "\x99\xab",
  [0x6a56] = "\x99\xac", [0x6a57] = "\x99\xad", [0x6a58] = "\x99\xae",
  [0x6a59] = "\x99\xaf", [0x6a5a] = "\x99\xb0", [0x6a5b] = "\x99\xb1",
  [0x6a5c] = "\x99\xb2", [0x6a5d] = "\x99\xb3", [0x6a5e] = "\x99\xb4",
  [0x6a5f] = "\x99\xb5", [0x6a60] = "\x99\xb6", [0x6a61] = "\x99\xb7",
  [0x6a62] = "\x99\xb8", [0x6a63] = "\x99\xb9", [0x6a64] = "\x99\xba",
  [0x6a65] = "\x99\xbb", [0x6a66] = "\x99\xbc", [0x6a67] = "\x99\xbd",
  [0x6a68] = "\x99\xbe", [0x6a69] = "\x99\xbf", [0x6a6a] = "\x99\xc0",
  [0x6a6b] = "\x99\xc1", [0x6a6c] = "\x99\xc2", [0x6a6d] = "\x99\xc3",
  [0x6a6e] = "\x99\xc4", [0x6a6f] = "\x99\xc5", [0x6a70] = "\x99\xc6",
  [0x6a71] = "\x99\xc7", [0x6a72] = "\x99\xc8", [0x6a73] = "\x99\xc9",
  [0x6a74] = "\x99\xca", [0x6a75] = "\x99\xcb", [0x6a76] = "\x99\xcc",
  [0x6a77] = "\x99\xcd", [0x6a78] = "\x99\xce", [0x6a79] = "\x99\xcf",
  [0x6a7a] = "\x99\xd0", [0x6a7b] = "\x99\xd1", [0x6a7c] = "\x99\xd2",
  [0x6a7d] = "\x99\xd3", [0x6a7e] = "\x99\xd4", [0x6a7f] = "\x99\xd5",
  [0x6a80] = "\x99\xd6", [0x6a81] = "\x99\xd7", [0x6a82] = "\x99\xd8",
  [0x6a83] = "\x99\xd9", [0x6a84] = "\x99\xda", [0x6a85] = "\x99\xdb",
  [0x6a86] = "\x99\xdc", [0x6a87] = "\x99\xdd", [0x6a88] = "\x99\xde",
  [0x6a89] = "\x99\xdf", [0x6a8a] = "\x99\xe0", [0x6a8b] = "\x99\xe1",
  [0x6a8c] = "\x99\xe2", [0x6a8d] = "\x99\xe3", [0x6a8e] = "\x99\xe4",
  [0x6a8f] = "\x99\xe5", [0x6a90] = "\x99\xe6", [0x6a91] = "\x99\xe7",
  [0x6a92] = "\x99\xe8", [0x6a93] = "\x99\xe9", [0x6a94] = "\x99\xea",
  [0x6a95] = "\x99\xeb", [0x6a96] = "\x99\xec", [0x6a97] = "\x99\xed",
  [0x6a98] = "\x99\xee", [0x6a99] = "\x99\xef", [0x6a9a] = "\x99\xf0",
  [0x6a9b] = "\x99\xf1", [0x6a9c] = "\x99\xf2", [0x6a9d] = "\x99\xf3",
  [0x6a9e] = "\x99\xf4", [0x6a9f] = "\x99\xf5", [0x6aa0] = "\xc7\xb7",
  [0x6aa1] = "\xb4\xce", [0x6aa2] = "\xbb\xb6", [0x6aa3] = "\xd0\xc0",
  [0x6aa4] = "\xec\xa3", [0x6aa5] = "\x99\xf6", [0x6aa6] = "\x99\xf7",
  [0x6aa7] = "\xc5\xb7", [0x6aa8] = "\x99\xf8", [0x6aa9] = "\x99\xf9",
  [0x6aaa] = "\x99\xfa", [0x6aab] = "\x99\xfb", [0x6aac] = "\x99\xfc",
  [0x6aad] = "\x99\xfd", [0x6aae] = "\x99\xfe", [0x6aaf] = "\x9a\x40",
  [0x6ab0] = "\x9a\x41", [0x6ab1] = "\x9a\x42", [0x6ab2] = "\xd3\xfb",
  [0x6ab3] = "\x9a\x43", [0x6ab4] = "\x9a\x44", [0x6ab5] = "\x9a\x45",
  [0x6ab6] = "\x9a\x46", [0x6ab7] = "\xec\xa4", [0x6ab8] = "\x9a\x47",
  [0x6ab9] = "\xec\xa5", [0x6aba] = "\xc6\xdb", [0x6abb] = "\x9a\x48",
  [0x6abc] = "\x9a\x49", [0x6abd] = "\x9a\x4a", [0x6abe] = "\xbf\xee",
  [0x6abf] = "\x9a\x4b", [0x6ac0] = "\x9a\x4c", [0x6ac1] = "\x9a\x4d",
  [0x6ac2] = "\x9a\x4e", [0x6ac3] = "\xec\xa6", [0x6ac4] = "\x9a\x4f",
  [0x6ac5] = "\x9a\x50", [0x6ac6] = "\xec\xa7", [0x6ac7] = "\xd0\xaa",
  [0x6ac8] = "\x9a\x51", [0x6ac9] = "\xc7\xb8", [0x6aca] = "\x9a\x52",
  [0x6acb] = "\x9a\x53", [0x6acc] = "\xb8\xe8", [0x6acd] = "\x9a\x54",
  [0x6ace] = "\x9a\x55", [0x6acf] = "\x9a\x56", [0x6ad0] = "\x9a\x57",
  [0x6ad1] = "\x9a\x58", [0x6ad2] = "\x9a\x59", [0x6ad3] = "\x9a\x5a",
  [0x6ad4] = "\x9a\x5b", [0x6ad5] = "\x9a\x5c", [0x6ad6] = "\x9a\x5d",
  [0x6ad7] = "\x9a\x5e", [0x6ad8] = "\x9a\x5f", [0x6ad9] = "\xec\xa8",
  [0x6ada] = "\x9a\x60", [0x6adb] = "\x9a\x61", [0x6adc] = "\x9a\x62",
  [0x6add] = "\x9a\x63", [0x6ade] = "\x9a\x64", [0x6adf] = "\x9a\x65",
  [0x6ae0] = "\x9a\x66", [0x6ae1] = "\x9a\x67", [0x6ae2] = "\xd6\xb9",
  [0x6ae3] = "\xd5\xfd", [0x6ae4] = "\xb4\xcb", [0x6ae5] = "\xb2\xbd",
  [0x6ae6] = "\xce\xe4", [0x6ae7] = "\xc6\xe7", [0x6ae8] = "\x9a\x68",
  [0x6ae9] = "\x9a\x69", [0x6aea] = "\xcd\xe1", [0x6aeb] = "\x9a\x6a",
  [0x6aec] = "\x9a\x6b", [0x6aed] = "\x9a\x6c", [0x6aee] = "\x9a\x6d",
  [0x6aef] = "\x9a\x6e", [0x6af0] = "\x9a\x6f", [0x6af1] = "\x9a\x70",
  [0x6af2] = "\x9a\x71", [0x6af3] = "\x9a\x72", [0x6af4] = "\x9a\x73",
  [0x6af5] = "\x9a\x74", [0x6af6] = "\x9a\x75", [0x6af7] = "\x9a\x76",
  [0x6af8] = "\x9a\x77", [0x6af9] = "\xb4\xf5", [0x6afa] = "\x9a\x78",
  [0x6afb] = "\xcb\xc0", [0x6afc] = "\xbc\xdf", [0x6afd] = "\x9a\x79",
  [0x6afe] = "\x9a\x7a", [0x6aff] = "\x9a\x7b", [0x6b00] = "\x9a\x7c",
  [0x6b01] = "\xe9\xe2", [0x6b02] = "\xe9\xe3", [0x6b03] = "\xd1\xea",
  [0x6b04] = "\xe9\xe5", [0x6b05] = "\x9a\x7d", [0x6b06] = "\xb4\xf9",
  [0x6b07] = "\xe9\xe4", [0x6b08] = "\x9a\x7e", [0x6b09] = "\xd1\xb3",
  [0x6b0a] = "\xca\xe2", [0x6b0b] = "\xb2\xd0", [0x6b0c] = "\x9a\x80",
  [0x6b0d] = "\xe9\xe8", [0x6b0e] = "\x9a\x81", [0x6b0f] = "\x9a\x82",
  [0x6b10] = "\x9a\x83", [0x6b11] = "\x9a\x84", [0x6b12] = "\xe9\xe6",
  [0x6b13] = "\xe9\xe7", [0x6b14] = "\x9a\x85", [0x6b15] = "\x9a\x86",
  [0x6b16] = "\xd6\xb3", [0x6b17] = "\x9a\x87", [0x6b18] = "\x9a\x88",
  [0x6b19] = "\x9a\x89", [0x6b1a] = "\xe9\xe9", [0x6b1b] = "\xe9\xea",
  [0x6b1c] = "\x9a\x8a", [0x6b1d] = "\x9a\x8b", [0x6b1e] = "\x9a\x8c",
  [0x6b1f] = "\x9a\x8d", [0x6b20] = "\x9a\x8e", [0x6b21] = "\xe9\xeb",
  [0x6b22] = "\x9a\x8f", [0x6b23] = "\x9a\x90", [0x6b24] = "\x9a\x91",
  [0x6b25] = "\x9a\x92", [0x6b26] = "\x9a\x93", [0x6b27] = "\x9a\x94",
  [0x6b28] = "\x9a\x95", [0x6b29] = "\x9a\x96", [0x6b2a] = "\xe9\xec",
  [0x6b2b] = "\x9a\x97", [0x6b2c] = "\x9a\x98", [0x6b2d] = "\x9a\x99",
  [0x6b2e] = "\x9a\x9a", [0x6b2f] = "\x9a\x9b", [0x6b30] = "\x9a\x9c",
  [0x6b31] = "\x9a\x9d", [0x6b32] = "\x9a\x9e", [0x6b33] = "\xec\xaf",
  [0x6b34] = "\xc5\xb9", [0x6b35] = "\xb6\xce", [0x6b36] = "\x9a\x9f",
  [0x6b37] = "\xd2\xf3", [0x6b38] = "\x9a\xa0", [0x6b39] = "\x9a\xa1",
  [0x6b3a] = "\x9a\xa2", [0x6b3b] = "\x9a\xa3", [0x6b3c] = "\x9a\xa4",
  [0x6b3d] = "\x9a\xa5", [0x6b3e] = "\x9a\xa6", [0x6b3f] = "\xb5\xee",
  [0x6b40] = "\x9a\xa7", [0x6b41] = "\xbb\xd9", [0x6b42] = "\xec\xb1",
  [0x6b43] = "\x9a\xa8", [0x6b44] = "\x9a\xa9", [0x6b45] = "\xd2\xe3",
  [0x6b46] = "\x9a\xaa", [0x6b47] = "\x9a\xab", [0x6b48] = "\x9a\xac",
  [0x6b49] = "\x9a\xad", [0x6b4a] = "\x9a\xae", [0x6b4b] = "\xce\xe3",
  [0x6b4c] = "\x9a\xaf", [0x6b4d] = "\xc4\xb8", [0x6b4e] = "\x9a\xb0",
  [0x6b4f] = "\xc3\xbf", [0x6b50] = "\x9a\xb1", [0x6b51] = "\x9a\xb2",
  [0x6b52] = "\xb6\xbe", [0x6b53] = "\xd8\xb9", [0x6b54] = "\xb1\xc8",
  [0x6b55] = "\xb1\xcf", [0x6b56] = "\xb1\xd1", [0x6b57] = "\xc5\xfe",
  [0x6b58] = "\x9a\xb3", [0x6b59] = "\xb1\xd0", [0x6b5a] = "\x9a\xb4",
  [0x6b5b] = "\xc3\xab", [0x6b5c] = "\x9a\xb5", [0x6b5d] = "\x9a\xb6",
  [0x6b5e] = "\x9a\xb7", [0x6b5f] = "\x9a\xb8", [0x6b60] = "\x9a\xb9",
  [0x6b61] = "\xd5\xb1", [0x6b62] = "\x9a\xba", [0x6b63] = "\x9a\xbb",
  [0x6b64] = "\x9a\xbc", [0x6b65] = "\x9a\xbd", [0x6b66] = "\x9a\xbe",
  [0x6b67] = "\x9a\xbf", [0x6b68] = "\x9a\xc0", [0x6b69] = "\x9a\xc1",
  [0x6b6a] = "\xeb\xa4", [0x6b6b] = "\xba\xc1", [0x6b6c] = "\x9a\xc2",
  [0x6b6d] = "\x9a\xc3", [0x6b6e] = "\x9a\xc4", [0x6b6f] = "\xcc\xba",
  [0x6b70] = "\x9a\xc5", [0x6b71] = "\x9a\xc6", [0x6b72] = "\x9a\xc7",
  [0x6b73] = "\xeb\xa5", [0x6b74] = "\x9a\xc8", [0x6b75] = "\xeb\xa7",
  [0x6b76] = "\x9a\xc9", [0x6b77] = "\x9a\xca", [0x6b78] = "\x9a\xcb",
  [0x6b79] = "\xeb\xa8", [0x6b7a] = "\x9a\xcc", [0x6b7b] = "\x9a\xcd",
  [0x6b7c] = "\x9a\xce", [0x6b7d] = "\xeb\xa6", [0x6b7e] = "\x9a\xcf",
  [0x6b7f] = "\x9a\xd0", [0x6b80] = "\x9a\xd1", [0x6b81] = "\x9a\xd2",
  [0x6b82] = "\x9a\xd3", [0x6b83] = "\x9a\xd4", [0x6b84] = "\x9a\xd5",
  [0x6b85] = "\xeb\xa9", [0x6b86] = "\xeb\xab", [0x6b87] = "\xeb\xaa",
  [0x6b88] = "\x9a\xd6", [0x6b89] = "\x9a\xd7", [0x6b8a] = "\x9a\xd8",
  [0x6b8b] = "\x9a\xd9", [0x6b8c] = "\x9a\xda", [0x6b8d] = "\xeb\xac",
  [0x6b8e] = "\x9a\xdb", [0x6b8f] = "\xca\xcf", [0x6b90] = "\xd8\xb5",
  [0x6b91] = "\xc3\xf1", [0x6b92] = "\x9a\xdc", [0x6b93] = "\xc3\xa5",
  [0x6b94] = "\xc6\xf8", [0x6b95] = "\xeb\xad", [0x6b96] = "\xc4\xca",
  [0x6b97] = "\x9a\xdd", [0x6b98] = "\xeb\xae", [0x6b99] = "\xeb\xaf",
  [0x6b9a] = "\xeb\xb0", [0x6b9b] = "\xb7\xd5", [0x6b9c] = "\x9a\xde",
  [0x6b9d] = "\x9a\xdf", [0x6b9e] = "\x9a\xe0", [0x6b9f] = "\xb7\xfa",
  [0x6ba0] = "\x9a\xe1", [0x6ba1] = "\xeb\xb1", [0x6ba2] = "\xc7\xe2",
  [0x6ba3] = "\x9a\xe2", [0x6ba4] = "\xeb\xb3", [0x6ba5] = "\x9a\xe3",
  [0x6ba6] = "\xba\xa4", [0x6ba7] = "\xd1\xf5", [0x6ba8] = "\xb0\xb1",
  [0x6ba9] = "\xeb\xb2", [0x6baa] = "\xeb\xb4", [0x6bab] = "\x9a\xe4",
  [0x6bac] = "\x9a\xe5", [0x6bad] = "\x9a\xe6", [0x6bae] = "\xb5\xaa",
  [0x6baf] = "\xc2\xc8", [0x6bb0] = "\xc7\xe8", [0x6bb1] = "\x9a\xe7",
  [0x6bb2] = "\xeb\xb5", [0x6bb3] = "\x9a\xe8", [0x6bb4] = "\xcb\xae",
  [0x6bb5] = "\xe3\xdf", [0x6bb6] = "\x9a\xe9", [0x6bb7] = "\x9a\xea",
  [0x6bb8] = "\xd3\xc0", [0x6bb9] = "\x9a\xeb", [0x6bba] = "\x9a\xec",
  [0x6bbb] = "\x9a\xed", [0x6bbc] = "\x9a\xee", [0x6bbd] = "\xd9\xdb",
  [0x6bbe] = "\x9a\xef", [0x6bbf] = "\x9a\xf0", [0x6bc0] = "\xcd\xa1",
  [0x6bc1] = "\xd6\xad", [0x6bc2] = "\xc7\xf3", [0x6bc3] = "\x9a\xf1",
  [0x6bc4] = "\x9a\xf2", [0x6bc5] = "\x9a\xf3", [0x6bc6] = "\xd9\xe0",
  [0x6bc7] = "\xbb\xe3", [0x6bc8] = "\x9a\xf4", [0x6bc9] = "\xba\xba",
  [0x6bca] = "\xe3\xe2", [0x6bcb] = "\x9a\xf5", [0x6bcc] = "\x9a\xf6",
  [0x6bcd] = "\x9a\xf7", [0x6bce] = "\x9a\xf8", [0x6bcf] = "\x9a\xf9",
  [0x6bd0] = "\xcf\xab", [0x6bd1] = "\x9a\xfa", [0x6bd2] = "\x9a\xfb",
  [0x6bd3] = "\x9a\xfc", [0x6bd4] = "\xe3\xe0", [0x6bd5] = "\xc9\xc7",
  [0x6bd6] = "\x9a\xfd", [0x6bd7] = "\xba\xb9", [0x6bd8] = "\x9a\xfe",
  [0x6bd9] = "\x9b\x40", [0x6bda] = "\x9b\x41", [0x6bdb] = "\xd1\xb4",
  [0x6bdc] = "\xe3\xe1", [0x6bdd] = "\xc8\xea", [0x6bde] = "\xb9\xaf",
  [0x6bdf] = "\xbd\xad", [0x6be0] = "\xb3\xd8", [0x6be1] = "\xce\xdb",
  [0x6be2] = "\x9b\x42", [0x6be3] = "\x9b\x43", [0x6be4] = "\xcc\xc0",
  [0x6be5] = "\x9b\x44", [0x6be6] = "\x9b\x45", [0x6be7] = "\x9b\x46",
  [0x6be8] = "\xe3\xe8", [0x6be9] = "\xe3\xe9", [0x6bea] = "\xcd\xf4",
  [0x6beb] = "\x9b\x47", [0x6bec] = "\x9b\x48", [0x6bed] = "\x9b\x49",
  [0x6bee] = "\x9b\x4a", [0x6bef] = "\x9b\x4b", [0x6bf0] = "\xcc\xad",
  [0x6bf1] = "\x9b\x4c", [0x6bf2] = "\xbc\xb3", [0x6bf3] = "\x9b\x4d",
  [0x6bf4] = "\xe3\xea", [0x6bf5] = "\x9b\x4e", [0x6bf6] = "\xe3\xeb",
  [0x6bf7] = "\x9b\x4f", [0x6bf8] = "\x9b\x50", [0x6bf9] = "\xd0\xda",
  [0x6bfa] = "\x9b\x51", [0x6bfb] = "\x9b\x52", [0x6bfc] = "\x9b\x53",
  [0x6bfd] = "\xc6\xfb", [0x6bfe] = "\xb7\xda", [0x6bff] = "\x9b\x54",
  [0x6c00] = "\x9b\x55", [0x6c01] = "\xc7\xdf", [0x6c02] = "\xd2\xca",
  [0x6c03] = "\xce\xd6", [0x6c04] = "\x9b\x56", [0x6c05] = "\xe3\xe4",
  [0x6c06] = "\xe3\xec", [0x6c07] = "\x9b\x57", [0x6c08] = "\xc9\xf2",
  [0x6c09] = "\xb3\xc1", [0x6c0a] = "\x9b\x58", [0x6c0b] = "\x9b\x59",
  [0x6c0c] = "\xe3\xe7", [0x6c0d] = "\x9b\x5a", [0x6c0e] = "\x9b\x5b",
  [0x6c0f] = "\xc6\xe3", [0x6c10] = "\xe3\xe5", [0x6c11] = "\x9b\x5c",
  [0x6c12] = "\x9b\x5d", [0x6c13] = "\xed\xb3", [0x6c14] = "\xe3\xe6",
  [0x6c15] = "\x9b\x5e", [0x6c16] = "\x9b\x5f", [0x6c17] = "\x9b\x60",
  [0x6c18] = "\x9b\x61", [0x6c19] = "\xc9\xb3", [0x6c1a] = "\x9b\x62",
  [0x6c1b] = "\xc5\xe6", [0x6c1c] = "\x9b\x63", [0x6c1d] = "\x9b\x64",
  [0x6c1e] = "\x9b\x65", [0x6c1f] = "\xb9\xb5", [0x6c20] = "\x9b\x66",
  [0x6c21] = "\xc3\xbb", [0x6c22] = "\x9b\x67", [0x6c23] = "\xe3\xe3",
  [0x6c24] = "\xc5\xbd", [0x6c25] = "\xc1\xa4", [0x6c26] = "\xc2\xd9",
  [0x6c27] = "\xb2\xd7", [0x6c28] = "\x9b\x68", [0x6c29] = "\xe3\xed",
  [0x6c2a] = "\xbb\xa6", [0x6c2b] = "\xc4\xad", [0x6c2c] = "\x9b\x69",
  [0x6c2d] = "\xe3\xf0", [0x6c2e] = "\xbe\xda", [0x6c2f] = "\x9b\x6a",
  [0x6c30] = "\x9b\x6b", [0x6c31] = "\xe3\xfb", [0x6c32] = "\xe3\xf5",
  [0x6c33] = "\xba\xd3", [0x6c34] = "\x9b\x6c", [0x6c35] = "\x9b\x6d",
  [0x6c36] = "\x9b\x6e", [0x6c37] = "\x9b\x6f", [0x6c38] = "\xb7\xd0",
  [0x6c39] = "\xd3\xcd", [0x6c3a] = "\x9b\x70", [0x6c3b] = "\xd6\xce",
  [0x6c3c] = "\xd5\xd3", [0x6c3d] = "\xb9\xc1", [0x6c3e] = "\xd5\xb4",
  [0x6c3f] = "\xd1\xd8", [0x6c40] = "\x9b\x71", [0x6c41] = "\x9b\x72",
  [0x6c42] = "\x9b\x73", [0x6c43] = "\x9b\x74", [0x6c44] = "\xd0\xb9",
  [0x6c45] = "\xc7\xf6", [0x6c46] = "\x9b\x75", [0x6c47] = "\x9b\x76",
  [0x6c48] = "\x9b\x77", [0x6c49] = "\xc8\xaa", [0x6c4a] = "\xb2\xb4",
  [0x6c4b] = "\x9b\x78", [0x6c4c] = "\xc3\xda", [0x6c4d] = "\x9b\x79",
  [0x6c4e] = "\x9b\x7a", [0x6c4f] = "\x9b\x7b", [0x6c50] = "\xe3\xee",
  [0x6c51] = "\x9b\x7c", [0x6c52] = "\x9b\x7d", [0x6c53] = "\xe3\xfc",
  [0x6c54] = "\xe3\xef", [0x6c55] = "\xb7\xa8", [0x6c56] = "\xe3\xf7",
  [0x6c57] = "\xe3\xf4", [0x6c58] = "\x9b\x7e", [0x6c59] = "\x9b\x80",
  [0x6c5a] = "\x9b\x81", [0x6c5b] = "\xb7\xba", [0x6c5c] = "\x9b\x82",
  [0x6c5d] = "\x9b\x83", [0x6c5e] = "\xc5\xa2", [0x6c5f] = "\x9b\x84",
  [0x6c60] = "\xe3\xf6", [0x6c61] = "\xc5\xdd", [0x6c62] = "\xb2\xa8",
  [0x6c63] = "\xc6\xfc", [0x6c64] = "\x9b\x85", [0x6c65] = "\xc4\xe0",
  [0x6c66] = "\x9b\x86", [0x6c67] = "\x9b\x87", [0x6c68] = "\xd7\xa2",
  [0x6c69] = "\x9b\x88", [0x6c6a] = "\xc0\xe1", [0x6c6b] = "\xe3\xf9",
  [0x6c6c] = "\x9b\x89", [0x6c6d] = "\x9b\x8a", [0x6c6e] = "\xe3\xfa",
  [0x6c6f] = "\xe3\xfd", [0x6c70] = "\xcc\xa9", [0x6c71] = "\xe3\xf3",
  [0x6c72] = "\x9b\x8b", [0x6c73] = "\xd3\xbe", [0x6c74] = "\x9b\x8c",
  [0x6c75] = "\xb1\xc3", [0x6c76] = "\xed\xb4", [0x6c77] = "\xe3\xf1",
  [0x6c78] = "\xe3\xf2", [0x6c79] = "\x9b\x8d", [0x6c7a] = "\xe3\xf8",
  [0x6c7b] = "\xd0\xba", [0x6c7c] = "\xc6\xc3", [0x6c7d] = "\xd4\xf3",
  [0x6c7e] = "\xe3\xfe", [0x6c7f] = "\x9b\x8e", [0x6c80] = "\x9b\x8f",
  [0x6c81] = "\xbd\xe0", [0x6c82] = "\x9b\x90", [0x6c83] = "\x9b\x91",
  [0x6c84] = "\xe4\xa7", [0x6c85] = "\x9b\x92", [0x6c86] = "\x9b\x93",
  [0x6c87] = "\xe4\xa6", [0x6c88] = "\x9b\x94", [0x6c89] = "\x9b\x95",
  [0x6c8a] = "\x9b\x96", [0x6c8b] = "\xd1\xf3", [0x6c8c] = "\xe4\xa3",
  [0x6c8d] = "\x9b\x97", [0x6c8e] = "\xe4\xa9", [0x6c8f] = "\x9b\x98",
  [0x6c90] = "\x9b\x99", [0x6c91] = "\x9b\x9a", [0x6c92] = "\xc8\xf7",
  [0x6c93] = "\x9b\x9b", [0x6c94] = "\x9b\x9c", [0x6c95] = "\x9b\x9d",
  [0x6c96] = "\x9b\x9e", [0x6c97] = "\xcf\xb4", [0x6c98] = "\x9b\x9f",
  [0x6c99] = "\xe4\xa8", [0x6c9a] = "\xe4\xae", [0x6c9b] = "\xc2\xe5",
  [0x6c9c] = "\x9b\xa0", [0x6c9d] = "\x9b\xa1", [0x6c9e] = "\xb6\xb4",
  [0x6c9f] = "\x9b\xa2", [0x6ca0] = "\x9b\xa3", [0x6ca1] = "\x9b\xa4",
  [0x6ca2] = "\x9b\xa5", [0x6ca3] = "\x9b\xa6", [0x6ca4] = "\x9b\xa7",
  [0x6ca5] = "\xbd\xf2", [0x6ca6] = "\x9b\xa8", [0x6ca7] = "\xe4\xa2",
  [0x6ca8] = "\x9b\xa9", [0x6ca9] = "\x9b\xaa", [0x6caa] = "\xba\xe9",
  [0x6cab] = "\xe4\xaa", [0x6cac] = "\x9b\xab", [0x6cad] = "\x9b\xac",
  [0x6cae] = "\xe4\xac", [0x6caf] = "\x9b\xad", [0x6cb0] = "\x9b\xae",
  [0x6cb1] = "\xb6\xfd", [0x6cb2] = "\xd6\xde", [0x6cb3] = "\xe4\xb2",
  [0x6cb4] = "\x9b\xaf", [0x6cb5] = "\xe4\xad", [0x6cb6] = "\x9b\xb0",
  [0x6cb7] = "\x9b\xb1", [0x6cb8] = "\x9b\xb2", [0x6cb9] = "\xe4\xa1",
  [0x6cba] = "\x9b\xb3", [0x6cbb] = "\xbb\xee", [0x6cbc] = "\xcd\xdd",
  [0x6cbd] = "\xc7\xa2", [0x6cbe] = "\xc5\xc9", [0x6cbf] = "\x9b\xb4",
  [0x6cc0] = "\x9b\xb5", [0x6cc1] = "\xc1\xf7", [0x6cc2] = "\x9b\xb6",
  [0x6cc3] = "\xe4\xa4", [0x6cc4] = "\x9b\xb7", [0x6cc5] = "\xc7\xb3",
  [0x6cc6] = "\xbd\xac", [0x6cc7] = "\xbd\xbd", [0x6cc8] = "\xe4\xa5",
  [0x6cc9] = "\x9b\xb8", [0x6cca] = "\xd7\xc7", [0x6ccb] = "\xb2\xe2",
  [0x6ccc] = "\x9b\xb9", [0x6ccd] = "\xe4\xab", [0x6cce] = "\xbc\xc3",
  [0x6ccf] = "\xe4\xaf", [0x6cd0] = "\x9b\xba", [0x6cd1] = "\xbb\xeb",
  [0x6cd2] = "\xe4\xb0", [0x6cd3] = "\xc5\xa8", [0x6cd4] = "\xe4\xb1",
  [0x6cd5] = "\x9b\xbb", [0x6cd6] = "\x9b\xbc", [0x6cd7] = "\x9b\xbd",
  [0x6cd8] = "\x9b\xbe", [0x6cd9] = "\xd5\xe3", [0x6cda] = "\xbf\xa3",
  [0x6cdb] = "\x9b\xbf", [0x6cdc] = "\xe4\xba", [0x6cdd] = "\x9b\xc0",
  [0x6cde] = "\xe4\xb7", [0x6cdf] = "\x9b\xc1", [0x6ce0] = "\xe4\xbb",
  [0x6ce1] = "\x9b\xc2", [0x6ce2] = "\x9b\xc3", [0x6ce3] = "\xe4\xbd",
  [0x6ce4] = "\x9b\xc4", [0x6ce5] = "\x9b\xc5", [0x6ce6] = "\xc6\xd6",
  [0x6ce7] = "\x9b\xc6", [0x6ce8] = "\x9b\xc7", [0x6ce9] = "\xba\xc6",
  [0x6cea] = "\xc0\xcb", [0x6ceb] = "\x9b\xc8", [0x6cec] = "\x9b\xc9",
  [0x6ced] = "\x9b\xca", [0x6cee] = "\xb8\xa1", [0x6cef] = "\xe4\xb4",
  [0x6cf0] = "\x9b\xcb", [0x6cf1] = "\x9b\xcc", [0x6cf2] = "\x9b\xcd",
  [0x6cf3] = "\x9b\xce", [0x6cf4] = "\xd4\xa1", [0x6cf5] = "\x9b\xcf",
  [0x6cf6] = "\x9b\xd0", [0x6cf7] = "\xba\xa3", [0x6cf8] = "\xbd\xfe",
  [0x6cf9] = "\x9b\xd1", [0x6cfa] = "\x9b\xd2", [0x6cfb] = "\x9b\xd3",
  [0x6cfc] = "\xe4\xbc", [0x6cfd] = "\x9b\xd4", [0x6cfe] = "\x9b\xd5",
  [0x6cff] = "\x9b\xd6", [0x6d00] = "\x9b\xd7", [0x6d01] = "\x9b\xd8",
  [0x6d02] = "\xcd\xbf", [0x6d03] = "\x9b\xd9", [0x6d04] = "\x9b\xda",
  [0x6d05] = "\xc4\xf9", [0x6d06] = "\x9b\xdb", [0x6d07] = "\x9b\xdc",
  [0x6d08] = "\xcf\xfb", [0x6d09] = "\xc9\xe6", [0x6d0a] = "\x9b\xdd",
  [0x6d0b] = "\x9b\xde", [0x6d0c] = "\xd3\xbf", [0x6d0d] = "\x9b\xdf",
  [0x6d0e] = "\xcf\xd1", [0x6d0f] = "\x9b\xe0", [0x6d10] = "\x9b\xe1",
  [0x6d11] = "\xe4\xb3", [0x6d12] = "\x9b\xe2", [0x6d13] = "\xe4\xb8",
  [0x6d14] = "\xe4\xb9", [0x6d15] = "\xcc\xe9", [0x6d16] = "\x9b\xe3",
  [0x6d17] = "\x9b\xe4", [0x6d18] = "\x9b\xe5", [0x6d19] = "\x9b\xe6",
  [0x6d1a] = "\x9b\xe7", [0x6d1b] = "\xcc\xce", [0x6d1c] = "\x9b\xe8",
  [0x6d1d] = "\xc0\xd4", [0x6d1e] = "\xe4\xb5", [0x6d1f] = "\xc1\xb0",
  [0x6d20] = "\xe4\xb6", [0x6d21] = "\xce\xd0", [0x6d22] = "\x9b\xe9",
  [0x6d23] = "\xbb\xc1", [0x6d24] = "\xb5\xd3", [0x6d25] = "\x9b\xea",
  [0x6d26] = "\xc8\xf3", [0x6d27] = "\xbd\xa7", [0x6d28] = "\xd5\xc7",
  [0x6d29] = "\xc9\xac", [0x6d2a] = "\xb8\xa2", [0x6d2b] = "\xe4\xca",
  [0x6d2c] = "\x9b\xeb", [0x6d2d] = "\x9b\xec", [0x6d2e] = "\xe4\xcc",
  [0x6d2f] = "\xd1\xc4", [0x6d30] = "\x9b\xed", [0x6d31] = "\x9b\xee",
  [0x6d32] = "\xd2\xba", [0x6d33] = "\x9b\xef", [0x6d34] = "\x9b\xf0",
  [0x6d35] = "\xba\xad", [0x6d36] = "\x9b\xf1", [0x6d37] = "\x9b\xf2",
  [0x6d38] = "\xba\xd4", [0x6d39] = "\x9b\xf3", [0x6d3a] = "\x9b\xf4",
  [0x6d3b] = "\x9b\xf5", [0x6d3c] = "\x9b\xf6", [0x6d3d] = "\x9b\xf7",
  [0x6d3e] = "\x9b\xf8", [0x6d3f] = "\xe4\xc3", [0x6d40] = "\xb5\xed",
  [0x6d41] = "\x9b\xf9", [0x6d42] = "\x9b\xfa", [0x6d43] = "\x9b\xfb",
  [0x6d44] = "\xd7\xcd", [0x6d45] = "\xe4\xc0", [0x6d46] = "\xcf\xfd",
  [0x6d47] = "\xe4\xbf", [0x6d48] = "\x9b\xfc", [0x6d49] = "\x9b\xfd",
  [0x6d4a] = "\x9b\xfe", [0x6d4b] = "\xc1\xdc", [0x6d4c] = "\xcc\xca",
  [0x6d4d] = "\x9c\x40", [0x6d4e] = "\x9c\x41", [0x6d4f] = "\x9c\x42",
  [0x6d50] = "\x9c\x43", [0x6d51] = "\xca\xe7", [0x6d52] = "\x9c\x44",
  [0x6d53] = "\x9c\x45", [0x6d54] = "\x9c\x46", [0x6d55] = "\x9c\x47",
  [0x6d56] = "\xc4\xd7", [0x6d57] = "\x9c\x48", [0x6d58] = "\xcc\xd4",
  [0x6d59] = "\xe4\xc8", [0x6d5a] = "\x9c\x49", [0x6d5b] = "\x9c\x4a",
  [0x6d5c] = "\x9c\x4b", [0x6d5d] = "\xe4\xc7", [0x6d5e] = "\xe4\xc1",
  [0x6d5f] = "\x9c\x4c", [0x6d60] = "\xe4\xc4", [0x6d61] = "\xb5\xad",
  [0x6d62] = "\x9c\x4d", [0x6d63] = "\x9c\x4e", [0x6d64] = "\xd3\xd9",
  [0x6d65] = "\x9c\x4f", [0x6d66] = "\xe4\xc6", [0x6d67] = "\x9c\x50",
  [0x6d68] = "\x9c\x51", [0x6d69] = "\x9c\x52", [0x6d6a] = "\x9c\x53",
  [0x6d6b] = "\xd2\xf9", [0x6d6c] = "\xb4\xe3", [0x6d6d] = "\x9c\x54",
  [0x6d6e] = "\xbb\xb4", [0x6d6f] = "\x9c\x55", [0x6d70] = "\x9c\x56",
  [0x6d71] = "\xc9\xee", [0x6d72] = "\x9c\x57", [0x6d73] = "\xb4\xbe",
  [0x6d74] = "\x9c\x58", [0x6d75] = "\x9c\x59", [0x6d76] = "\x9c\x5a",
  [0x6d77] = "\xbb\xec", [0x6d78] = "\x9c\x5b", [0x6d79] = "\xd1\xcd",
  [0x6d7a] = "\x9c\x5c", [0x6d7b] = "\xcc\xed", [0x6d7c] = "\xed\xb5",
  [0x6d7d] = "\x9c\x5d", [0x6d7e] = "\x9c\x5e", [0x6d7f] = "\x9c\x5f",
  [0x6d80] = "\x9c\x60", [0x6d81] = "\x9c\x61", [0x6d82] = "\x9c\x62",
  [0x6d83] = "\x9c\x63", [0x6d84] = "\x9c\x64", [0x6d85] = "\xc7\xe5",
  [0x6d86] = "\x9c\x65", [0x6d87] = "\x9c\x66", [0x6d88] = "\x9c\x67",
  [0x6d89] = "\x9c\x68", [0x6d8a] = "\xd4\xa8", [0x6d8b] = "\x9c\x69",
  [0x6d8c] = "\xe4\xcb", [0x6d8d] = "\xd7\xd5", [0x6d8e] = "\xe4\xc2",
  [0x6d8f] = "\x9c\x6a", [0x6d90] = "\xbd\xa5", [0x6d91] = "\xe4\xc5",
  [0x6d92] = "\x9c\x6b", [0x6d93] = "\x9c\x6c", [0x6d94] = "\xd3\xe6",
  [0x6d95] = "\x9c\x6d", [0x6d96] = "\xe4\xc9", [0x6d97] = "\xc9\xf8",
  [0x6d98] = "\x9c\x6e", [0x6d99] = "\x9c\x6f", [0x6d9a] = "\xe4\xbe",
  [0x6d9b] = "\x9c\x70", [0x6d9c] = "\x9c\x71", [0x6d9d] = "\xd3\xe5",
  [0x6d9e] = "\x9c\x72", [0x6d9f] = "\x9c\x73", [0x6da0] = "\xc7\xfe",
  [0x6da1] = "\xb6\xc9", [0x6da2] = "\x9c\x74", [0x6da3] = "\xd4\xfc",
  [0x6da4] = "\xb2\xb3", [0x6da5] = "\xe4\xd7", [0x6da6] = "\x9c\x75",
  [0x6da7] = "\x9c\x76", [0x6da8] = "\x9c\x77", [0x6da9] = "\xce\xc2",
  [0x6daa] = "\x9c\x78", [0x6dab] = "\xe4\xcd", [0x6dac] = "\x9c\x79",
  [0x6dad] = "\xce\xbc", [0x6dae] = "\x9c\x7a", [0x6daf] = "\xb8\xdb",
  [0x6db0] = "\x9c\x7b", [0x6db1] = "\x9c\x7c", [0x6db2] = "\xe4\xd6",
  [0x6db3] = "\x9c\x7d", [0x6db4] = "\xbf\xca", [0x6db5] = "\x9c\x7e",
  [0x6db6] = "\x9c\x80", [0x6db7] = "\x9c\x81", [0x6db8] = "\xd3\xce",
  [0x6db9] = "\x9c\x82", [0x6dba] = "\xc3\xec", [0x6dbb] = "\x9c\x83",
  [0x6dbc] = "\x9c\x84", [0x6dbd] = "\x9c\x85", [0x6dbe] = "\x9c\x86",
  [0x6dbf] = "\x9c\x87", [0x6dc0] = "\x9c\x88", [0x6dc1] = "\x9c\x89",
  [0x6dc2] = "\x9c\x8a", [0x6dc3] = "\xc5\xc8", [0x6dc4] = "\xe4\xd8",
  [0x6dc5] = "\x9c\x8b", [0x6dc6] = "\x9c\x8c", [0x6dc7] = "\x9c\x8d",
  [0x6dc8] = "\x9c\x8e", [0x6dc9] = "\x9c\x8f", [0x6dca] = "\x9c\x90",
  [0x6dcb] = "\x9c\x91", [0x6dcc] = "\x9c\x92", [0x6dcd] = "\xcd\xc4",
  [0x6dce] = "\xe4\xcf", [0x6dcf] = "\x9c\x93", [0x6dd0] = "\x9c\x94",
  [0x6dd1] = "\x9c\x95", [0x6dd2] = "\x9c\x96", [0x6dd3] = "\xe4\xd4",
  [0x6dd4] = "\xe4\xd5", [0x6dd5] = "\x9c\x97", [0x6dd6] = "\xba\xfe",
  [0x6dd7] = "\x9c\x98", [0x6dd8] = "\xcf\xe6", [0x6dd9] = "\x9c\x99",
  [0x6dda] = "\x9c\x9a", [0x6ddb] = "\xd5\xbf", [0x6ddc] = "\x9c\x9b",
  [0x6ddd] = "\x9c\x9c", [0x6dde] = "\x9c\x9d", [0x6ddf] = "\xe4\xd2",
  [0x6de0] = "\x9c\x9e", [0x6de1] = "\x9c\x9f", [0x6de2] = "\x9c\xa0",
  [0x6de3] = "\x9c\xa1", [0x6de4] = "\x9c\xa2", [0x6de5] = "\x9c\xa3",
  [0x6de6] = "\x9c\xa4", [0x6de7] = "\x9c\xa5", [0x6de8] = "\x9c\xa6",
  [0x6de9] = "\x9c\xa7", [0x6dea] = "\x9c\xa8", [0x6deb] = "\xe4\xd0",
  [0x6dec] = "\x9c\xa9", [0x6ded] = "\x9c\xaa", [0x6dee] = "\xe4\xce",
  [0x6def] = "\x9c\xab", [0x6df0] = "\x9c\xac", [0x6df1] = "\x9c\xad",
  [0x6df2] = "\x9c\xae", [0x6df3] = "\x9c\xaf", [0x6df4] = "\x9c\xb0",
  [0x6df5] = "\x9c\xb1", [0x6df6] = "\x9c\xb2", [0x6df7] = "\x9c\xb3",
  [0x6df8] = "\x9c\xb4", [0x6df9] = "\x9c\xb5", [0x6dfa] = "\x9c\xb6",
  [0x6dfb] = "\x9c\xb7", [0x6dfc] = "\x9c\xb8", [0x6dfd] = "\x9c\xb9",
  [0x6dfe] = "\xcd\xe5", [0x6dff] = "\xca\xaa", [0x6e00] = "\x9c\xba",
  [0x6e01] = "\x9c\xbb", [0x6e02] = "\x9c\xbc", [0x6e03] = "\xc0\xa3",
  [0x6e04] = "\x9c\xbd", [0x6e05] = "\xbd\xa6", [0x6e06] = "\xe4\xd3",
  [0x6e07] = "\x9c\xbe", [0x6e08] = "\x9c\xbf", [0x6e09] = "\xb8\xc8",
  [0x6e0a] = "\x9c\xc0", [0x6e0b] = "\x9c\xc1", [0x6e0c] = "\x9c\xc2",
  [0x6e0d] = "\x9c\xc3", [0x6e0e] = "\x9c\xc4", [0x6e0f] = "\xe4\xe7",
  [0x6e10] = "\xd4\xb4", [0x6e11] = "\x9c\xc5", [0x6e12] = "\x9c\xc6",
  [0x6e13] = "\x9c\xc7", [0x6e14] = "\x9c\xc8", [0x6e15] = "\x9c\xc9",
  [0x6e16] = "\x9c\xca", [0x6e17] = "\x9c\xcb", [0x6e18] = "\xe4\xdb",
  [0x6e19] = "\x9c\xcc", [0x6e1a] = "\x9c\xcd", [0x6e1b] = "\x9c\xce",
  [0x6e1c] = "\xc1\xef", [0x6e1d] = "\x9c\xcf", [0x6e1e] = "\x9c\xd0",
  [0x6e1f] = "\xe4\xe9", [0x6e20] = "\x9c\xd1", [0x6e21] = "\x9c\xd2",
  [0x6e22] = "\xd2\xe7", [0x6e23] = "\x9c\xd3", [0x6e24] = "\x9c\xd4",
  [0x6e25] = "\xe4\xdf", [0x6e26] = "\x9c\xd5", [0x6e27] = "\xe4\xe0",
  [0x6e28] = "\x9c\xd6", [0x6e29] = "\x9c\xd7", [0x6e2a] = "\xcf\xaa",
  [0x6e2b] = "\x9c\xd8", [0x6e2c] = "\x9c\xd9", [0x6e2d] = "\x9c\xda",
  [0x6e2e] = "\x9c\xdb", [0x6e2f] = "\xcb\xdd", [0x6e30] = "\x9c\xdc",
  [0x6e31] = "\xe4\xda", [0x6e32] = "\xe4\xd1", [0x6e33] = "\x9c\xdd",
  [0x6e34] = "\xe4\xe5", [0x6e35] = "\x9c\xde", [0x6e36] = "\xc8\xdc",
  [0x6e37] = "\xe4\xe3", [0x6e38] = "\x9c\xdf", [0x6e39] = "\x9c\xe0",
  [0x6e3a] = "\xc4\xe7", [0x6e3b] = "\xe4\xe2", [0x6e3c] = "\x9c\xe1",
  [0x6e3d] = "\xe4\xe1", [0x6e3e] = "\x9c\xe2", [0x6e3f] = "\x9c\xe3",
  [0x6e40] = "\x9c\xe4", [0x6e41] = "\xb3\xfc", [0x6e42] = "\xe4\xe8",
  [0x6e43] = "\x9c\xe5", [0x6e44] = "\x9c\xe6", [0x6e45] = "\x9c\xe7",
  [0x6e46] = "\x9c\xe8", [0x6e47] = "\xb5\xe1", [0x6e48] = "\x9c\xe9",
  [0x6e49] = "\x9c\xea", [0x6e4a] = "\x9c\xeb", [0x6e4b] = "\xd7\xcc",
  [0x6e4c] = "\x9c\xec", [0x6e4d] = "\x9c\xed", [0x6e4e] = "\x9c\xee",
  [0x6e4f] = "\xe4\xe6", [0x6e50] = "\x9c\xef", [0x6e51] = "\xbb\xac",
  [0x6e52] = "\x9c\xf0", [0x6e53] = "\xd7\xd2", [0x6e54] = "\xcc\xcf",
  [0x6e55] = "\xeb\xf8", [0x6e56] = "\x9c\xf1", [0x6e57] = "\xe4\xe4",
  [0x6e58] = "\x9c\xf2", [0x6e59] = "\x9c\xf3", [0x6e5a] = "\xb9\xf6",
  [0x6e5b] = "\x9c\xf4", [0x6e5c] = "\x9c\xf5", [0x6e5d] = "\x9c\xf6",
  [0x6e5e] = "\xd6\xcd", [0x6e5f] = "\xe4\xd9", [0x6e60] = "\xe4\xdc",
  [0x6e61] = "\xc2\xfa", [0x6e62] = "\xe4\xde", [0x6e63] = "\x9c\xf7",
  [0x6e64] = "\xc2\xcb", [0x6e65] = "\xc0\xc4", [0x6e66] = "\xc2\xd0",
  [0x6e67] = "\x9c\xf8", [0x6e68] = "\xb1\xf5", [0x6e69] = "\xcc\xb2",
  [0x6e6a] = "\x9c\xf9", [0x6e6b] = "\x9c\xfa", [0x6e6c] = "\x9c\xfb",
  [0x6e6d] = "\x9c\xfc", [0x6e6e] = "\x9c\xfd", [0x6e6f] = "\x9c\xfe",
  [0x6e70] = "\x9d\x40", [0x6e71] = "\x9d\x41", [0x6e72] = "\x9d\x42",
  [0x6e73] = "\x9d\x43", [0x6e74] = "\xb5\xce", [0x6e75] = "\x9d\x44",
  [0x6e76] = "\x9d\x45", [0x6e77] = "\x9d\x46", [0x6e78] = "\x9d\x47",
  [0x6e79] = "\xe4\xef", [0x6e7a] = "\x9d\x48", [0x6e7b] = "\x9d\x49",
  [0x6e7c] = "\x9d\x4a", [0x6e7d] = "\x9d\x4b", [0x6e7e] = "\x9d\x4c",
  [0x6e7f] = "\x9d\x4d", [0x6e80] = "\x9d\x4e", [0x6e81] = "\x9d\x4f",
  [0x6e82] = "\xc6\xaf", [0x6e83] = "\x9d\x50", [0x6e84] = "\x9d\x51",
  [0x6e85] = "\x9d\x52", [0x6e86] = "\xc6\xe1", [0x6e87] = "\x9d\x53",
  [0x6e88] = "\x9d\x54", [0x6e89] = "\xe4\xf5", [0x6e8a] = "\x9d\x55",
  [0x6e8b] = "\x9d\x56", [0x6e8c] = "\x9d\x57", [0x6e8d] = "\x9d\x58",
  [0x6e8e] = "\x9d\x59", [0x6e8f] = "\xc2\xa9", [0x6e90] = "\x9d\x5a",
  [0x6e91] = "\x9d\x5b", [0x6e92] = "\x9d\x5c", [0x6e93] = "\xc0\xec",
  [0x6e94] = "\xd1\xdd", [0x6e95] = "\xe4\xee", [0x6e96] = "\x9d\x5d",
  [0x6e97] = "\x9d\x5e", [0x6e98] = "\x9d\x5f", [0x6e99] = "\x9d\x60",
  [0x6e9a] = "\x9d\x61", [0x6e9b] = "\x9d\x62", [0x6e9c] = "\x9d\x63",
  [0x6e9d] = "\x9d\x64", [0x6e9e] = "\x9d\x65", [0x6e9f] = "\x9d\x66",
  [0x6ea0] = "\xc4\xae", [0x6ea1] = "\x9d\x67", [0x6ea2] = "\x9d\x68",
  [0x6ea3] = "\x9d\x69", [0x6ea4] = "\xe4\xed", [0x6ea5] = "\x9d\x6a",
  [0x6ea6] = "\x9d\x6b", [0x6ea7] = "\x9d\x6c", [0x6ea8] = "\x9d\x6d",
  [0x6ea9] = "\xe4\xf6", [0x6eaa] = "\xe4\xf4", [0x6eab] = "\xc2\xfe",
  [0x6eac] = "\x9d\x6e", [0x6ead] = "\xe4\xdd", [0x6eae] = "\x9d\x6f",
  [0x6eaf] = "\xe4\xf0", [0x6eb0] = "\x9d\x70", [0x6eb1] = "\xca\xfe",
  [0x6eb2] = "\x9d\x71", [0x6eb3] = "\xd5\xc4", [0x6eb4] = "\x9d\x72",
  [0x6eb5] = "\x9d\x73", [0x6eb6] = "\xe4\xf1", [0x6eb7] = "\x9d\x74",
  [0x6eb8] = "\x9d\x75", [0x6eb9] = "\x9d\x76", [0x6eba] = "\x9d\x77",
  [0x6ebb] = "\x9d\x78", [0x6ebc] = "\x9d\x79", [0x6ebd] = "\x9d\x7a",
  [0x6ebe] = "\xd1\xfa", [0x6ebf] = "\x9d\x7b", [0x6ec0] = "\x9d\x7c",
  [0x6ec1] = "\x9d\x7d", [0x6ec2] = "\x9d\x7e", [0x6ec3] = "\x9d\x80",
  [0x6ec4] = "\x9d\x81", [0x6ec5] = "\x9d\x82", [0x6ec6] = "\xe4\xeb",
  [0x6ec7] = "\xe4\xec", [0x6ec8] = "\x9d\x83", [0x6ec9] = "\x9d\x84",
  [0x6eca] = "\x9d\x85", [0x6ecb] = "\xe4\xf2", [0x6ecc] = "\x9d\x86",
  [0x6ecd] = "\xce\xab", [0x6ece] = "\x9d\x87", [0x6ecf] = "\x9d\x88",
  [0x6ed0] = "\x9d\x89", [0x6ed1] = "\x9d\x8a", [0x6ed2] = "\x9d\x8b",
  [0x6ed3] = "\x9d\x8c", [0x6ed4] = "\x9d\x8d", [0x6ed5] = "\x9d\x8e",
  [0x6ed6] = "\x9d\x8f", [0x6ed7] = "\x9d\x90", [0x6ed8] = "\xc5\xcb",
  [0x6ed9] = "\x9d\x91", [0x6eda] = "\x9d\x92", [0x6edb] = "\x9d\x93",
  [0x6edc] = "\xc7\xb1", [0x6edd] = "\x9d\x94", [0x6ede] = "\xc2\xba",
  [0x6edf] = "\x9d\x95", [0x6ee0] = "\x9d\x96", [0x6ee1] = "\x9d\x97",
  [0x6ee2] = "\xe4\xea", [0x6ee3] = "\x9d\x98", [0x6ee4] = "\x9d\x99",
  [0x6ee5] = "\x9d\x9a", [0x6ee6] = "\xc1\xca", [0x6ee7] = "\x9d\x9b",
  [0x6ee8] = "\x9d\x9c", [0x6ee9] = "\x9d\x9d", [0x6eea] = "\x9d\x9e",
  [0x6eeb] = "\x9d\x9f", [0x6eec] = "\x9d\xa0", [0x6eed] = "\xcc\xb6",
  [0x6eee] = "\xb3\xb1", [0x6eef] = "\x9d\xa1", [0x6ef0] = "\x9d\xa2",
  [0x6ef1] = "\x9d\xa3", [0x6ef2] = "\xe4\xfb", [0x6ef3] = "\x9d\xa4",
  [0x6ef4] = "\xe4\xf3", [0x6ef5] = "\x9d\xa5", [0x6ef6] = "\x9d\xa6",
  [0x6ef7] = "\x9d\xa7", [0x6ef8] = "\xe4\xfa", [0x6ef9] = "\x9d\xa8",
  [0x6efa] = "\xe4\xfd", [0x6efb] = "\x9d\xa9", [0x6efc] = "\xe4\xfc",
  [0x6efd] = "\x9d\xaa", [0x6efe] = "\x9d\xab", [0x6eff] = "\x9d\xac",
  [0x6f00] = "\x9d\xad", [0x6f01] = "\x9d\xae", [0x6f02] = "\x9d\xaf",
  [0x6f03] = "\x9d\xb0", [0x6f04] = "\xb3\xce", [0x6f05] = "\x9d\xb1",
  [0x6f06] = "\x9d\xb2", [0x6f07] = "\x9d\xb3", [0x6f08] = "\xb3\xba",
  [0x6f09] = "\xe4\xf7", [0x6f0a] = "\x9d\xb4", [0x6f0b] = "\x9d\xb5",
  [0x6f0c] = "\xe4\xf9", [0x6f0d] = "\xe4\xf8", [0x6f0e] = "\xc5\xec",
  [0x6f0f] = "\x9d\xb6", [0x6f10] = "\x9d\xb7", [0x6f11] = "\x9d\xb8",
  [0x6f12] = "\x9d\xb9", [0x6f13] = "\x9d\xba", [0x6f14] = "\x9d\xbb",
  [0x6f15] = "\x9d\xbc", [0x6f16] = "\x9d\xbd", [0x6f17] = "\x9d\xbe",
  [0x6f18] = "\x9d\xbf", [0x6f19] = "\x9d\xc0", [0x6f1a] = "\x9d\xc1",
  [0x6f1b] = "\x9d\xc2", [0x6f1c] = "\xc0\xbd", [0x6f1d] = "\x9d\xc3",
  [0x6f1e] = "\x9d\xc4", [0x6f1f] = "\x9d\xc5", [0x6f20] = "\x9d\xc6",
  [0x6f21] = "\xd4\xe8", [0x6f22] = "\x9d\xc7", [0x6f23] = "\x9d\xc8",
  [0x6f24] = "\x9d\xc9", [0x6f25] = "\x9d\xca", [0x6f26] = "\x9d\xcb",
  [0x6f27] = "\xe5\xa2", [0x6f28] = "\x9d\xcc", [0x6f29] = "\x9d\xcd",
  [0x6f2a] = "\x9d\xce", [0x6f2b] = "\x9d\xcf", [0x6f2c] = "\x9d\xd0",
  [0x6f2d] = "\x9d\xd1", [0x6f2e] = "\x9d\xd2", [0x6f2f] = "\x9d\xd3",
  [0x6f30] = "\x9d\xd4", [0x6f31] = "\x9d\xd5", [0x6f32] = "\x9d\xd6",
  [0x6f33] = "\xb0\xc4", [0x6f34] = "\x9d\xd7", [0x6f35] = "\x9d\xd8",
  [0x6f36] = "\xe5\xa4", [0x6f37] = "\x9d\xd9", [0x6f38] = "\x9d\xda",
  [0x6f39] = "\xe5\xa3", [0x6f3a] = "\x9d\xdb", [0x6f3b] = "\x9d\xdc",
  [0x6f3c] = "\x9d\xdd", [0x6f3d] = "\x9d\xde", [0x6f3e] = "\x9d\xdf",
  [0x6f3f] = "\x9d\xe0", [0x6f40] = "\xbc\xa4", [0x6f41] = "\x9d\xe1",
  [0x6f42] = "\xe5\xa5", [0x6f43] = "\x9d\xe2", [0x6f44] = "\x9d\xe3",
  [0x6f45] = "\x9d\xe4", [0x6f46] = "\x9d\xe5", [0x6f47] = "\x9d\xe6",
  [0x6f48] = "\x9d\xe7", [0x6f49] = "\xe5\xa1", [0x6f4a] = "\x9d\xe8",
  [0x6f4b] = "\x9d\xe9", [0x6f4c] = "\x9d\xea", [0x6f4d] = "\x9d\xeb",
  [0x6f4e] = "\x9d\xec", [0x6f4f] = "\x9d\xed", [0x6f50] = "\x9d\xee",
  [0x6f51] = "\xe4\xfe", [0x6f52] = "\xb1\xf4", [0x6f53] = "\x9d\xef",
  [0x6f54] = "\x9d\xf0", [0x6f55] = "\x9d\xf1", [0x6f56] = "\x9d\xf2",
  [0x6f57] = "\x9d\xf3", [0x6f58] = "\x9d\xf4", [0x6f59] = "\x9d\xf5",
  [0x6f5a] = "\x9d\xf6", [0x6f5b] = "\x9d\xf7", [0x6f5c] = "\x9d\xf8",
  [0x6f5d] = "\x9d\xf9", [0x6f5e] = "\xe5\xa8", [0x6f5f] = "\x9d\xfa",
  [0x6f60] = "\xe5\xa9", [0x6f61] = "\xe5\xa6", [0x6f62] = "\x9d\xfb",
  [0x6f63] = "\x9d\xfc", [0x6f64] = "\x9d\xfd", [0x6f65] = "\x9d\xfe",
  [0x6f66] = "\x9e\x40", [0x6f67] = "\x9e\x41", [0x6f68] = "\x9e\x42",
  [0x6f69] = "\x9e\x43", [0x6f6a] = "\x9e\x44", [0x6f6b] = "\x9e\x45",
  [0x6f6c] = "\x9e\x46", [0x6f6d] = "\x9e\x47", [0x6f6e] = "\xe5\xa7",
  [0x6f6f] = "\xe5\xaa", [0x6f70] = "\x9e\x48", [0x6f71] = "\x9e\x49",
  [0x6f72] = "\x9e\x4a", [0x6f73] = "\x9e\x4b", [0x6f74] = "\x9e\x4c",
  [0x6f75] = "\x9e\x4d", [0x6f76] = "\x9e\x4e", [0x6f77] = "\x9e\x4f",
  [0x6f78] = "\x9e\x50", [0x6f79] = "\x9e\x51", [0x6f7a] = "\x9e\x52",
  [0x6f7b] = "\x9e\x53", [0x6f7c] = "\x9e\x54", [0x6f7d] = "\x9e\x55",
  [0x6f7e] = "\x9e\x56", [0x6f7f] = "\x9e\x57", [0x6f80] = "\x9e\x58",
  [0x6f81] = "\x9e\x59", [0x6f82] = "\x9e\x5a", [0x6f83] = "\x9e\x5b",
  [0x6f84] = "\x9e\x5c", [0x6f85] = "\x9e\x5d", [0x6f86] = "\x9e\x5e",
  [0x6f87] = "\x9e\x5f", [0x6f88] = "\x9e\x60", [0x6f89] = "\x9e\x61",
  [0x6f8a] = "\x9e\x62", [0x6f8b] = "\x9e\x63", [0x6f8c] = "\x9e\x64",
  [0x6f8d] = "\x9e\x65", [0x6f8e] = "\x9e\x66", [0x6f8f] = "\x9e\x67",
  [0x6f90] = "\x9e\x68", [0x6f91] = "\xc6\xd9", [0x6f92] = "\x9e\x69",
  [0x6f93] = "\x9e\x6a", [0x6f94] = "\x9e\x6b", [0x6f95] = "\x9e\x6c",
  [0x6f96] = "\x9e\x6d", [0x6f97] = "\x9e\x6e", [0x6f98] = "\x9e\x6f",
  [0x6f99] = "\x9e\x70", [0x6f9a] = "\xe5\xab", [0x6f9b] = "\xe5\xad",
  [0x6f9c] = "\x9e\x71", [0x6f9d] = "\x9e\x72", [0x6f9e] = "\x9e\x73",
  [0x6f9f] = "\x9e\x74", [0x6fa0] = "\x9e\x75", [0x6fa1] = "\x9e\x76",
  [0x6fa2] = "\x9e\x77", [0x6fa3] = "\xe5\xac", [0x6fa4] = "\x9e\x78",
  [0x6fa5] = "\x9e\x79", [0x6fa6] = "\x9e\x7a", [0x6fa7] = "\x9e\x7b",
  [0x6fa8] = "\x9e\x7c", [0x6fa9] = "\x9e\x7d", [0x6faa] = "\x9e\x7e",
  [0x6fab] = "\x9e\x80", [0x6fac] = "\x9e\x81", [0x6fad] = "\x9e\x82",
  [0x6fae] = "\x9e\x83", [0x6faf] = "\x9e\x84", [0x6fb0] = "\x9e\x85",
  [0x6fb1] = "\x9e\x86", [0x6fb2] = "\x9e\x87", [0x6fb3] = "\x9e\x88",
  [0x6fb4] = "\x9e\x89", [0x6fb5] = "\xe5\xaf", [0x6fb6] = "\x9e\x8a",
  [0x6fb7] = "\x9e\x8b", [0x6fb8] = "\x9e\x8c", [0x6fb9] = "\xe5\xae",
  [0x6fba] = "\x9e\x8d", [0x6fbb] = "\x9e\x8e", [0x6fbc] = "\x9e\x8f",
  [0x6fbd] = "\x9e\x90", [0x6fbe] = "\x9e\x91", [0x6fbf] = "\x9e\x92",
  [0x6fc0] = "\x9e\x93", [0x6fc1] = "\x9e\x94", [0x6fc2] = "\x9e\x95",
  [0x6fc3] = "\x9e\x96", [0x6fc4] = "\x9e\x97", [0x6fc5] = "\x9e\x98",
  [0x6fc6] = "\x9e\x99", [0x6fc7] = "\x9e\x9a", [0x6fc8] = "\x9e\x9b",
  [0x6fc9] = "\x9e\x9c", [0x6fca] = "\x9e\x9d", [0x6fcb] = "\x9e\x9e",
  [0x6fcc] = "\xb9\xe0", [0x6fcd] = "\x9e\x9f", [0x6fce] = "\x9e\xa0",
  [0x6fcf] = "\xe5\xb0", [0x6fd0] = "\x9e\xa1", [0x6fd1] = "\x9e\xa2",
  [0x6fd2] = "\x9e\xa3", [0x6fd3] = "\x9e\xa4", [0x6fd4] = "\x9e\xa5",
  [0x6fd5] = "\x9e\xa6", [0x6fd6] = "\x9e\xa7", [0x6fd7] = "\x9e\xa8",
  [0x6fd8] = "\x9e\xa9", [0x6fd9] = "\x9e\xaa", [0x6fda] = "\x9e\xab",
  [0x6fdb] = "\x9e\xac", [0x6fdc] = "\x9e\xad", [0x6fdd] = "\x9e\xae",
  [0x6fde] = "\xe5\xb1", [0x6fdf] = "\x9e\xaf", [0x6fe0] = "\x9e\xb0",
  [0x6fe1] = "\x9e\xb1", [0x6fe2] = "\x9e\xb2", [0x6fe3] = "\x9e\xb3",
  [0x6fe4] = "\x9e\xb4", [0x6fe5] = "\x9e\xb5", [0x6fe6] = "\x9e\xb6",
  [0x6fe7] = "\x9e\xb7", [0x6fe8] = "\x9e\xb8", [0x6fe9] = "\x9e\xb9",
  [0x6fea] = "\x9e\xba", [0x6feb] = "\xbb\xf0", [0x6fec] = "\xec\xe1",
  [0x6fed] = "\xc3\xf0", [0x6fee] = "\x9e\xbb", [0x6fef] = "\xb5\xc6",
  [0x6ff0] = "\xbb\xd2", [0x6ff1] = "\x9e\xbc", [0x6ff2] = "\x9e\xbd",
  [0x6ff3] = "\x9e\xbe", [0x6ff4] = "\x9e\xbf", [0x6ff5] = "\xc1\xe9",
  [0x6ff6] = "\xd4\xee", [0x6ff7] = "\x9e\xc0", [0x6ff8] = "\xbe\xc4",
  [0x6ff9] = "\x9e\xc1", [0x6ffa] = "\x9e\xc2", [0x6ffb] = "\x9e\xc3",
  [0x6ffc] = "\xd7\xc6", [0x6ffd] = "\x9e\xc4", [0x6ffe] = "\xd4\xd6",
  [0x6fff] = "\xb2\xd3", [0x7000] = "\xec\xbe", [0x7001] = "\x9e\xc5",
  [0x7002] = "\x9e\xc6", [0x7003] = "\x9e\xc7", [0x7004] = "\x9e\xc8",
  [0x7005] = "\xea\xc1", [0x7006] = "\x9e\xc9", [0x7007] = "\x9e\xca",
  [0x7008] = "\x9e\xcb", [0x7009] = "\xc2\xaf", [0x700a] = "\xb4\xb6",
  [0x700b] = "\x9e\xcc", [0x700c] = "\x9e\xcd", [0x700d] = "\x9e\xce",
  [0x700e] = "\xd1\xd7", [0x700f] = "\x9e\xcf", [0x7010] = "\x9e\xd0",
  [0x7011] = "\x9e\xd1", [0x7012] = "\xb3\xb4", [0x7013] = "\x9e\xd2",
  [0x7014] = "\xc8\xb2", [0x7015] = "\xbf\xbb", [0x7016] = "\xec\xc0",
  [0x7017] = "\x9e\xd3", [0x7018] = "\x9e\xd4", [0x7019] = "\xd6\xcb",
  [0x701a] = "\x9e\xd5", [0x701b] = "\x9e\xd6", [0x701c] = "\xec\xbf",
  [0x701d] = "\xec\xc1", [0x701e] = "\x9e\xd7", [0x701f] = "\x9e\xd8",
  [0x7020] = "\x9e\xd9", [0x7021] = "\x9e\xda", [0x7022] = "\x9e\xdb",
  [0x7023] = "\x9e\xdc", [0x7024] = "\x9e\xdd", [0x7025] = "\x9e\xde",
  [0x7026] = "\x9e\xdf", [0x7027] = "\x9e\xe0", [0x7028] = "\x9e\xe1",
  [0x7029] = "\x9e\xe2", [0x702a] = "\x9e\xe3", [0x702b] = "\xec\xc5",
  [0x702c] = "\xbe\xe6", [0x702d] = "\xcc\xbf", [0x702e] = "\xc5\xda",
  [0x702f] = "\xbe\xbc", [0x7030] = "\x9e\xe4", [0x7031] = "\xec\xc6",
  [0x7032] = "\x9e\xe5", [0x7033] = "\xb1\xfe", [0x7034] = "\x9e\xe6",
  [0x7035] = "\x9e\xe7", [0x7036] = "\x9e\xe8", [0x7037] = "\xec\xc4",
  [0x7038] = "\xd5\xa8", [0x7039] = "\xb5\xe3", [0x703a] = "\x9e\xe9",
  [0x703b] = "\xec\xc2", [0x703c] = "\xc1\xb6", [0x703d] = "\xb3\xe3",
  [0x703e] = "\x9e\xea", [0x703f] = "\x9e\xeb", [0x7040] = "\xec\xc3",
  [0x7041] = "\xcb\xb8", [0x7042] = "\xc0\xc3", [0x7043] = "\xcc\xfe",
  [0x7044] = "\x9e\xec", [0x7045] = "\x9e\xed", [0x7046] = "\x9e\xee",
  [0x7047] = "\x9e\xef", [0x7048] = "\xc1\xd2", [0x7049] = "\x9e\xf0",
  [0x704a] = "\xec\xc8", [0x704b] = "\x9e\xf1", [0x704c] = "\x9e\xf2",
  [0x704d] = "\x9e\xf3", [0x704e] = "\x9e\xf4", [0x704f] = "\x9e\xf5",
  [0x7050] = "\x9e\xf6", [0x7051] = "\x9e\xf7", [0x7052] = "\x9e\xf8",
  [0x7053] = "\x9e\xf9", [0x7054] = "\x9e\xfa", [0x7055] = "\x9e\xfb",
  [0x7056] = "\x9e\xfc", [0x7057] = "\x9e\xfd", [0x7058] = "\xba\xe6",
  [0x7059] = "\xc0\xd3", [0x705a] = "\x9e\xfe", [0x705b] = "\xd6\xf2",
  [0x705c] = "\x9f\x40", [0x705d] = "\x9f\x41", [0x705e] = "\x9f\x42",
  [0x705f] = "\xd1\xcc", [0x7060] = "\x9f\x43", [0x7061] = "\x9f\x44",
  [0x7062] = "\x9f\x45", [0x7063] = "\x9f\x46", [0x7064] = "\xbf\xbe",
  [0x7065] = "\x9f\x47", [0x7066] = "\xb7\xb3", [0x7067] = "\xc9\xd5",
  [0x7068] = "\xec\xc7", [0x7069] = "\xbb\xe2", [0x706a] = "\x9f\x48",
  [0x706b] = "\xcc\xcc", [0x706c] = "\xbd\xfd", [0x706d] = "\xc8\xc8",
  [0x706e] = "\x9f\x49", [0x706f] = "\xcf\xa9", [0x7070] = "\x9f\x4a",
  [0x7071] = "\x9f\x4b", [0x7072] = "\x9f\x4c", [0x7073] = "\x9f\x4d",
  [0x7074] = "\x9f\x4e", [0x7075] = "\x9f\x4f", [0x7076] = "\x9f\x50",
  [0x7077] = "\xcd\xe9", [0x7078] = "\x9f\x51", [0x7079] = "\xc5\xeb",
  [0x707a] = "\x9f\x52", [0x707b] = "\x9f\x53", [0x707c] = "\x9f\x54",
  [0x707d] = "\xb7\xe9", [0x707e] = "\x9f\x55", [0x707f] = "\x9f\x56",
  [0x7080] = "\x9f\x57", [0x7081] = "\x9f\x58", [0x7082] = "\x9f\x59",
  [0x7083] = "\x9f\x5a", [0x7084] = "\x9f\x5b", [0x7085] = "\x9f\x5c",
  [0x7086] = "\x9f\x5d", [0x7087] = "\x9f\x5e", [0x7088] = "\x9f\x5f",
  [0x7089] = "\xd1\xc9", [0x708a] = "\xba\xb8", [0x708b] = "\x9f\x60",
  [0x708c] = "\x9f\x61", [0x708d] = "\x9f\x62", [0x708e] = "\x9f\x63",
  [0x708f] = "\x9f\x64", [0x7090] = "\xec\xc9", [0x7091] = "\x9f\x65",
  [0x7092] = "\x9f\x66", [0x7093] = "\xec\xca", [0x7094] = "\x9f\x67",
  [0x7095] = "\xbb\xc0", [0x7096] = "\xec\xcb", [0x7097] = "\x9f\x68",
  [0x7098] = "\xec\xe2", [0x7099] = "\xb1\xba", [0x709a] = "\xb7\xd9",
  [0x709b] = "\x9f\x69", [0x709c] = "\x9f\x6a", [0x709d] = "\x9f\x6b",
  [0x709e] = "\x9f\x6c", [0x709f] = "\x9f\x6d", [0x70a0] = "\x9f\x6e",
  [0x70a1] = "\x9f\x6f", [0x70a2] = "\x9f\x70", [0x70a3] = "\x9f\x71",
  [0x70a4] = "\x9f\x72", [0x70a5] = "\x9f\x73", [0x70a6] = "\xbd\xb9",
  [0x70a7] = "\x9f\x74", [0x70a8] = "\x9f\x75", [0x70a9] = "\x9f\x76",
  [0x70aa] = "\x9f\x77", [0x70ab] = "\x9f\x78", [0x70ac] = "\x9f\x79",
  [0x70ad] = "\x9f\x7a", [0x70ae] = "\x9f\x7b", [0x70af] = "\xec\xcc",
  [0x70b0] = "\xd1\xe6", [0x70b1] = "\xec\xcd", [0x70b2] = "\x9f\x7c",
  [0x70b3] = "\x9f\x7d", [0x70b4] = "\x9f\x7e", [0x70b5] = "\x9f\x80",
  [0x70b6] = "\xc8\xbb", [0x70b7] = "\x9f\x81", [0x70b8] = "\x9f\x82",
  [0x70b9] = "\x9f\x83", [0x70ba] = "\x9f\x84", [0x70bb] = "\x9f\x85",
  [0x70bc] = "\x9f\x86", [0x70bd] = "\x9f\x87", [0x70be] = "\x9f\x88",
  [0x70bf] = "\x9f\x89", [0x70c0] = "\x9f\x8a", [0x70c1] = "\x9f\x8b",
  [0x70c2] = "\x9f\x8c", [0x70c3] = "\x9f\x8d", [0x70c4] = "\x9f\x8e",
  [0x70c5] = "\xec\xd1", [0x70c6] = "\x9f\x8f", [0x70c7] = "\x9f\x90",
  [0x70c8] = "\x9f\x91", [0x70c9] = "\x9f\x92", [0x70ca] = "\xec\xd3",
  [0x70cb] = "\x9f\x93", [0x70cc] = "\xbb\xcd", [0x70cd] = "\x9f\x94",
  [0x70ce] = "\xbc\xe5", [0x70cf] = "\x9f\x95", [0x70d0] = "\x9f\x96",
  [0x70d1] = "\x9f\x97", [0x70d2] = "\x9f\x98", [0x70d3] = "\x9f\x99",
  [0x70d4] = "\x9f\x9a", [0x70d5] = "\x9f\x9b", [0x70d6] = "\x9f\x9c",
  [0x70d7] = "\x9f\x9d", [0x70d8] = "\x9f\x9e", [0x70d9] = "\x9f\x9f",
  [0x70da] = "\x9f\xa0", [0x70db] = "\x9f\xa1", [0x70dc] = "\xec\xcf",
  [0x70dd] = "\x9f\xa2", [0x70de] = "\xc9\xb7", [0x70df] = "\x9f\xa3",
  [0x70e0] = "\x9f\xa4", [0x70e1] = "\x9f\xa5", [0x70e2] = "\x9f\xa6",
  [0x70e3] = "\x9f\xa7", [0x70e4] = "\xc3\xba", [0x70e5] = "\x9f\xa8",
  [0x70e6] = "\xec\xe3", [0x70e7] = "\xd5\xd5", [0x70e8] = "\xec\xd0",
  [0x70e9] = "\x9f\xa9", [0x70ea] = "\x9f\xaa", [0x70eb] = "\x9f\xab",
  [0x70ec] = "\x9f\xac", [0x70ed] = "\x9f\xad", [0x70ee] = "\xd6\xf3",
  [0x70ef] = "\x9f\xae", [0x70f0] = "\x9f\xaf", [0x70f1] = "\x9f\xb0",
  [0x70f2] = "\xec\xd2", [0x70f3] = "\xec\xce", [0x70f4] = "\x9f\xb1",
  [0x70f5] = "\x9f\xb2", [0x70f6] = "\x9f\xb3", [0x70f7] = "\x9f\xb4",
  [0x70f8] = "\xec\xd4", [0x70f9] = "\x9f\xb5", [0x70fa] = "\xec\xd5",
  [0x70fb] = "\x9f\xb6", [0x70fc] = "\x9f\xb7", [0x70fd] = "\xc9\xbf",
  [0x70fe] = "\x9f\xb8", [0x70ff] = "\x9f\xb9", [0x7100] = "\x9f\xba",
  [0x7101] = "\x9f\xbb", [0x7102] = "\x9f\xbc", [0x7103] = "\x9f\xbd",
  [0x7104] = "\xcf\xa8", [0x7105] = "\x9f\xbe", [0x7106] = "\x9f\xbf",
  [0x7107] = "\x9f\xc0", [0x7108] = "\x9f\xc1", [0x7109] = "\x9f\xc2",
  [0x710a] = "\xd0\xdc", [0x710b] = "\x9f\xc3", [0x710c] = "\x9f\xc4",
  [0x710d] = "\x9f\xc5", [0x710e] = "\x9f\xc6", [0x710f] = "\xd1\xac",
  [0x7110] = "\x9f\xc7", [0x7111] = "\x9f\xc8", [0x7112] = "\x9f\xc9",
  [0x7113] = "\x9f\xca", [0x7114] = "\xc8\xdb", [0x7115] = "\x9f\xcb",
  [0x7116] = "\x9f\xcc", [0x7117] = "\x9f\xcd", [0x7118] = "\xec\xd6",
  [0x7119] = "\xce\xf5", [0x711a] = "\x9f\xce", [0x711b] = "\x9f\xcf",
  [0x711c] = "\x9f\xd0", [0x711d] = "\x9f\xd1", [0x711e] = "\x9f\xd2",
  [0x711f] = "\xca\xec", [0x7120] = "\xec\xda", [0x7121] = "\x9f\xd3",
  [0x7122] = "\x9f\xd4", [0x7123] = "\x9f\xd5", [0x7124] = "\x9f\xd6",
  [0x7125] = "\x9f\xd7", [0x7126] = "\x9f\xd8", [0x7127] = "\x9f\xd9",
  [0x7128] = "\xec\xd9", [0x7129] = "\x9f\xda", [0x712a] = "\x9f\xdb",
  [0x712b] = "\x9f\xdc", [0x712c] = "\xb0\xbe", [0x712d] = "\x9f\xdd",
  [0x712e] = "\x9f\xde", [0x712f] = "\x9f\xdf", [0x7130] = "\x9f\xe0",
  [0x7131] = "\x9f\xe1", [0x7132] = "\x9f\xe2", [0x7133] = "\xec\xd7",
  [0x7134] = "\x9f\xe3", [0x7135] = "\xec\xd8", [0x7136] = "\x9f\xe4",
  [0x7137] = "\x9f\xe5", [0x7138] = "\x9f\xe6", [0x7139] = "\xec\xe4",
  [0x713a] = "\x9f\xe7", [0x713b] = "\x9f\xe8", [0x713c] = "\x9f\xe9",
  [0x713d] = "\x9f\xea", [0x713e] = "\x9f\xeb", [0x713f] = "\x9f\xec",
  [0x7140] = "\x9f\xed", [0x7141] = "\x9f\xee", [0x7142] = "\x9f\xef",
  [0x7143] = "\xc8\xbc", [0x7144] = "\x9f\xf0", [0x7145] = "\x9f\xf1",
  [0x7146] = "\x9f\xf2", [0x7147] = "\x9f\xf3", [0x7148] = "\x9f\xf4",
  [0x7149] = "\x9f\xf5", [0x714a] = "\x9f\xf6", [0x714b] = "\x9f\xf7",
  [0x714c] = "\x9f\xf8", [0x714d] = "\x9f\xf9", [0x714e] = "\xc1\xc7",
  [0x714f] = "\x9f\xfa", [0x7150] = "\x9f\xfb", [0x7151] = "\x9f\xfc",
  [0x7152] = "\x9f\xfd", [0x7153] = "\x9f\xfe", [0x7154] = "\xec\xdc",
  [0x7155] = "\xd1\xe0", [0x7156] = "\xa0\x40", [0x7157] = "\xa0\x41",
  [0x7158] = "\xa0\x42", [0x7159] = "\xa0\x43", [0x715a] = "\xa0\x44",
  [0x715b] = "\xa0\x45", [0x715c] = "\xa0\x46", [0x715d] = "\xa0\x47",
  [0x715e] = "\xa0\x48", [0x715f] = "\xa0\x49", [0x7160] = "\xec\xdb",
  [0x7161] = "\xa0\x4a", [0x7162] = "\xa0\x4b", [0x7163] = "\xa0\x4c",
  [0x7164] = "\xa0\x4d", [0x7165] = "\xd4\xef", [0x7166] = "\xa0\x4e",
  [0x7167] = "\xec\xdd", [0x7168] = "\xa0\x4f", [0x7169] = "\xa0\x50",
  [0x716a] = "\xa0\x51", [0x716b] = "\xa0\x52", [0x716c] = "\xa0\x53",
  [0x716d] = "\xa0\x54", [0x716e] = "\xdb\xc6", [0x716f] = "\xa0\x55",
  [0x7170] = "\xa0\x56", [0x7171] = "\xa0\x57", [0x7172] = "\xa0\x58",
  [0x7173] = "\xa0\x59", [0x7174] = "\xa0\x5a", [0x7175] = "\xa0\x5b",
  [0x7176] = "\xa0\x5c", [0x7177] = "\xa0\x5d", [0x7178] = "\xa0\x5e",
  [0x7179] = "\xec\xde", [0x717a] = "\xa0\x5f", [0x717b] = "\xa0\x60",
  [0x717c] = "\xa0\x61", [0x717d] = "\xa0\x62", [0x717e] = "\xa0\x63",
  [0x717f] = "\xa0\x64", [0x7180] = "\xa0\x65", [0x7181] = "\xa0\x66",
  [0x7182] = "\xa0\x67", [0x7183] = "\xa0\x68", [0x7184] = "\xa0\x69",
  [0x7185] = "\xa0\x6a", [0x7186] = "\xb1\xac", [0x7187] = "\xa0\x6b",
  [0x7188] = "\xa0\x6c", [0x7189] = "\xa0\x6d", [0x718a] = "\xa0\x6e",
  [0x718b] = "\xa0\x6f", [0x718c] = "\xa0\x70", [0x718d] = "\xa0\x71",
  [0x718e] = "\xa0\x72", [0x718f] = "\xa0\x73", [0x7190] = "\xa0\x74",
  [0x7191] = "\xa0\x75", [0x7192] = "\xa0\x76", [0x7193] = "\xa0\x77",
  [0x7194] = "\xa0\x78", [0x7195] = "\xa0\x79", [0x7196] = "\xa0\x7a",
  [0x7197] = "\xa0\x7b", [0x7198] = "\xa0\x7c", [0x7199] = "\xa0\x7d",
  [0x719a] = "\xa0\x7e", [0x719b] = "\xa0\x80", [0x719c] = "\xa0\x81",
  [0x719d] = "\xec\xdf", [0x719e] = "\xa0\x82", [0x719f] = "\xa0\x83",
  [0x71a0] = "\xa0\x84", [0x71a1] = "\xa0\x85", [0x71a2] = "\xa0\x86",
  [0x71a3] = "\xa0\x87", [0x71a4] = "\xa0\x88", [0x71a5] = "\xa0\x89",
  [0x71a6] = "\xa0\x8a", [0x71a7] = "\xa0\x8b", [0x71a8] = "\xec\xe0",
  [0x71a9] = "\xa0\x8c", [0x71aa] = "\xd7\xa6", [0x71ab] = "\xa0\x8d",
  [0x71ac] = "\xc5\xc0", [0x71ad] = "\xa0\x8e", [0x71ae] = "\xa0\x8f",
  [0x71af] = "\xa0\x90", [0x71b0] = "\xeb\xbc", [0x71b1] = "\xb0\xae",
  [0x71b2] = "\xa0\x91", [0x71b3] = "\xa0\x92", [0x71b4] = "\xa0\x93",
  [0x71b5] = "\xbe\xf4", [0x71b6] = "\xb8\xb8", [0x71b7] = "\xd2\xaf",
  [0x71b8] = "\xb0\xd6", [0x71b9] = "\xb5\xf9", [0x71ba] = "\xa0\x94",
  [0x71bb] = "\xd8\xb3", [0x71bc] = "\xa0\x95", [0x71bd] = "\xcb\xac",
  [0x71be] = "\xa0\x96", [0x71bf] = "\xe3\xdd", [0x71c0] = "\xa0\x97",
  [0x71c1] = "\xa0\x98", [0x71c2] = "\xa0\x99", [0x71c3] = "\xa0\x9a",
  [0x71c4] = "\xa0\x9b", [0x71c5] = "\xa0\x9c", [0x71c6] = "\xa0\x9d",
  [0x71c7] = "\xc6\xac", [0x71c8] = "\xb0\xe6", [0x71c9] = "\xa0\x9e",
  [0x71ca] = "\xa0\x9f", [0x71cb] = "\xa0\xa0", [0x71cc] = "\xc5\xc6",
  [0x71cd] = "\xeb\xb9", [0x71ce] = "\xa0\xa1", [0x71cf] = "\xa0\xa2",
  [0x71d0] = "\xa0\xa3", [0x71d1] = "\xa0\xa4", [0x71d2] = "\xeb\xba",
  [0x71d3] = "\xa0\xa5", [0x71d4] = "\xa0\xa6", [0x71d5] = "\xa0\xa7",
  [0x71d6] = "\xeb\xbb", [0x71d7] = "\xa0\xa8", [0x71d8] = "\xa0\xa9",
  [0x71d9] = "\xd1\xc0", [0x71da] = "\xa0\xaa", [0x71db] = "\xc5\xa3",
  [0x71dc] = "\xa0\xab", [0x71dd] = "\xea\xf2", [0x71de] = "\xa0\xac",
  [0x71df] = "\xc4\xb2", [0x71e0] = "\xa0\xad", [0x71e1] = "\xc4\xb5",
  [0x71e2] = "\xc0\xce", [0x71e3] = "\xa0\xae", [0x71e4] = "\xa0\xaf",
  [0x71e5] = "\xa0\xb0", [0x71e6] = "\xea\xf3", [0x71e7] = "\xc4\xc1",
  [0x71e8] = "\xa0\xb1", [0x71e9] = "\xce\xef", [0x71ea] = "\xa0\xb2",
  [0x71eb] = "\xa0\xb3", [0x71ec] = "\xa0\xb4", [0x71ed] = "\xa0\xb5",
  [0x71ee] = "\xea\xf0", [0x71ef] = "\xea\xf4", [0x71f0] = "\xa0\xb6",
  [0x71f1] = "\xa0\xb7", [0x71f2] = "\xc9\xfc", [0x71f3] = "\xa0\xb8",
  [0x71f4] = "\xa0\xb9", [0x71f5] = "\xc7\xa3", [0x71f6] = "\xa0\xba",
  [0x71f7] = "\xa0\xbb", [0x71f8] = "\xa0\xbc", [0x71f9] = "\xcc\xd8",
  [0x71fa] = "\xce\xfe", [0x71fb] = "\xa0\xbd", [0x71fc] = "\xa0\xbe",
  [0x71fd] = "\xa0\xbf", [0x71fe] = "\xea\xf5", [0x71ff] = "\xea\xf6",
  [0x7200] = "\xcf\xac", [0x7201] = "\xc0\xe7", [0x7202] = "\xa0\xc0",
  [0x7203] = "\xa0\xc1", [0x7204] = "\xea\xf7", [0x7205] = "\xa0\xc2",
  [0x7206] = "\xa0\xc3", [0x7207] = "\xa0\xc4", [0x7208] = "\xa0\xc5",
  [0x7209] = "\xa0\xc6", [0x720a] = "\xb6\xbf", [0x720b] = "\xea\xf8",
  [0x720c] = "\xa0\xc7", [0x720d] = "\xea\xf9", [0x720e] = "\xa0\xc8",
  [0x720f] = "\xea\xfa", [0x7210] = "\xa0\xc9", [0x7211] = "\xa0\xca",
  [0x7212] = "\xea\xfb", [0x7213] = "\xa0\xcb", [0x7214] = "\xa0\xcc",
  [0x7215] = "\xa0\xcd", [0x7216] = "\xa0\xce", [0x7217] = "\xa0\xcf",
  [0x7218] = "\xa0\xd0", [0x7219] = "\xa0\xd1", [0x721a] = "\xa0\xd2",
  [0x721b] = "\xa0\xd3", [0x721c] = "\xa0\xd4", [0x721d] = "\xa0\xd5",
  [0x721e] = "\xa0\xd6", [0x721f] = "\xea\xf1", [0x7220] = "\xa0\xd7",
  [0x7221] = "\xa0\xd8", [0x7222] = "\xa0\xd9", [0x7223] = "\xa0\xda",
  [0x7224] = "\xa0\xdb", [0x7225] = "\xa0\xdc", [0x7226] = "\xa0\xdd",
  [0x7227] = "\xa0\xde", [0x7228] = "\xa0\xdf", [0x7229] = "\xa0\xe0",
  [0x722a] = "\xa0\xe1", [0x722b] = "\xa0\xe2", [0x722c] = "\xc8\xae",
  [0x722d] = "\xe1\xeb", [0x722e] = "\xa0\xe3", [0x722f] = "\xb7\xb8",
  [0x7230] = "\xe1\xec", [0x7231] = "\xa0\xe4", [0x7232] = "\xa0\xe5",
  [0x7233] = "\xa0\xe6", [0x7234] = "\xe1\xed", [0x7235] = "\xa0\xe7",
  [0x7236] = "\xd7\xb4", [0x7237] = "\xe1\xee", [0x7238] = "\xe1\xef",
  [0x7239] = "\xd3\xcc", [0x723a] = "\xa0\xe8", [0x723b] = "\xa0\xe9",
  [0x723c] = "\xa0\xea", [0x723d] = "\xa0\xeb", [0x723e] = "\xa0\xec",
  [0x723f] = "\xa0\xed", [0x7240] = "\xa0\xee", [0x7241] = "\xe1\xf1",
  [0x7242] = "\xbf\xf1", [0x7243] = "\xe1\xf0", [0x7244] = "\xb5\xd2",
  [0x7245] = "\xa0\xef", [0x7246] = "\xa0\xf0", [0x7247] = "\xa0\xf1",
  [0x7248] = "\xb1\xb7", [0x7249] = "\xa0\xf2", [0x724a] = "\xa0\xf3",
  [0x724b] = "\xa0\xf4", [0x724c] = "\xa0\xf5", [0x724d] = "\xe1\xf3",
  [0x724e] = "\xe1\xf2", [0x724f] = "\xa0\xf6", [0x7250] = "\xba\xfc",
  [0x7251] = "\xa0\xf7", [0x7252] = "\xe1\xf4", [0x7253] = "\xa0\xf8",
  [0x7254] = "\xa0\xf9", [0x7255] = "\xa0\xfa", [0x7256] = "\xa0\xfb",
  [0x7257] = "\xb9\xb7", [0x7258] = "\xa0\xfc", [0x7259] = "\xbe\xd1",
  [0x725a] = "\xa0\xfd", [0x725b] = "\xa0\xfe", [0x725c] = "\xaa\x40",
  [0x725d] = "\xaa\x41", [0x725e] = "\xc4\xfc", [0x725f] = "\xaa\x42",
  [0x7260] = "\xba\xdd", [0x7261] = "\xbd\xc6", [0x7262] = "\xaa\x43",
  [0x7263] = "\xaa\x44", [0x7264] = "\xaa\x45", [0x7265] = "\xaa\x46",
  [0x7266] = "\xaa\x47", [0x7267] = "\xaa\x48", [0x7268] = "\xe1\xf5",
  [0x7269] = "\xe1\xf7", [0x726a] = "\xaa\x49", [0x726b] = "\xaa\x4a",
  [0x726c] = "\xb6\xc0", [0x726d] = "\xcf\xc1", [0x726e] = "\xca\xa8",
  [0x726f] = "\xe1\xf6", [0x7270] = "\xd5\xf8", [0x7271] = "\xd3\xfc",
  [0x7272] = "\xe1\xf8", [0x7273] = "\xe1\xfc", [0x7274] = "\xe1\xf9",
  [0x7275] = "\xaa\x4b", [0x7276] = "\xaa\x4c", [0x7277] = "\xe1\xfa",
  [0x7278] = "\xc0\xea", [0x7279] = "\xaa\x4d", [0x727a] = "\xe1\xfe",
  [0x727b] = "\xe2\xa1", [0x727c] = "\xc0\xc7", [0x727d] = "\xaa\x4e",
  [0x727e] = "\xaa\x4f", [0x727f] = "\xaa\x50", [0x7280] = "\xaa\x51",
  [0x7281] = "\xe1\xfb", [0x7282] = "\xaa\x52", [0x7283] = "\xe1\xfd",
  [0x7284] = "\xaa\x53", [0x7285] = "\xaa\x54", [0x7286] = "\xaa\x55",
  [0x7287] = "\xaa\x56", [0x7288] = "\xaa\x57", [0x7289] = "\xaa\x58",
  [0x728a] = "\xe2\xa5", [0x728b] = "\xaa\x59", [0x728c] = "\xaa\x5a",
  [0x728d] = "\xaa\x5b", [0x728e] = "\xc1\xd4", [0x728f] = "\xaa\x5c",
  [0x7290] = "\xaa\x5d", [0x7291] = "\xaa\x5e", [0x7292] = "\xaa\x5f",
  [0x7293] = "\xe2\xa3", [0x7294] = "\xaa\x60", [0x7295] = "\xe2\xa8",
  [0x7296] = "\xb2\xfe", [0x7297] = "\xe2\xa2", [0x7298] = "\xaa\x61",
  [0x7299] = "\xaa\x62", [0x729a] = "\xaa\x63", [0x729b] = "\xc3\xcd",
  [0x729c] = "\xb2\xc2", [0x729d] = "\xe2\xa7", [0x729e] = "\xe2\xa6",
  [0x729f] = "\xaa\x64", [0x72a0] = "\xaa\x65", [0x72a1] = "\xe2\xa4",
  [0x72a2] = "\xe2\xa9", [0x72a3] = "\xaa\x66", [0x72a4] = "\xaa\x67",
  [0x72a5] = "\xe2\xab", [0x72a6] = "\xaa\x68", [0x72a7] = "\xaa\x69",
  [0x72a8] = "\xaa\x6a", [0x72a9] = "\xd0\xc9", [0x72aa] = "\xd6\xed",
  [0x72ab] = "\xc3\xa8", [0x72ac] = "\xe2\xac", [0x72ad] = "\xaa\x6b",
  [0x72ae] = "\xcf\xd7", [0x72af] = "\xaa\x6c", [0x72b0] = "\xaa\x6d",
  [0x72b1] = "\xe2\xae", [0x72b2] = "\xaa\x6e", [0x72b3] = "\xaa\x6f",
  [0x72b4] = "\xba\xef", [0x72b5] = "\xaa\x70", [0x72b6] = "\xaa\x71",
  [0x72b7] = "\xe9\xe0", [0x72b8] = "\xe2\xad", [0x72b9] = "\xe2\xaa",
  [0x72ba] = "\xaa\x72", [0x72bb] = "\xaa\x73", [0x72bc] = "\xaa\x74",
  [0x72bd] = "\xaa\x75", [0x72be] = "\xbb\xab", [0x72bf] = "\xd4\xb3",
  [0x72c0] = "\xaa\x76", [0x72c1] = "\xaa\x77", [0x72c2] = "\xaa\x78",
  [0x72c3] = "\xaa\x79", [0x72c4] = "\xaa\x7a", [0x72c5] = "\xaa\x7b",
  [0x72c6] = "\xaa\x7c", [0x72c7] = "\xaa\x7d", [0x72c8] = "\xaa\x7e",
  [0x72c9] = "\xaa\x80", [0x72ca] = "\xaa\x81", [0x72cb] = "\xaa\x82",
  [0x72cc] = "\xaa\x83", [0x72cd] = "\xe2\xb0", [0x72ce] = "\xaa\x84",
  [0x72cf] = "\xaa\x85", [0x72d0] = "\xe2\xaf", [0x72d1] = "\xaa\x86",
  [0x72d2] = "\xe9\xe1", [0x72d3] = "\xaa\x87", [0x72d4] = "\xaa\x88",
  [0x72d5] = "\xaa\x89", [0x72d6] = "\xaa\x8a", [0x72d7] = "\xe2\xb1",
  [0x72d8] = "\xaa\x8b", [0x72d9] = "\xaa\x8c", [0x72da] = "\xaa\x8d",
  [0x72db] = "\xaa\x8e", [0x72dc] = "\xaa\x8f", [0x72dd] = "\xaa\x90",
  [0x72de] = "\xaa\x91", [0x72df] = "\xaa\x92", [0x72e0] = "\xe2\xb2",
  [0x72e1] = "\xaa\x93", [0x72e2] = "\xaa\x94", [0x72e3] = "\xaa\x95",
  [0x72e4] = "\xaa\x96", [0x72e5] = "\xaa\x97", [0x72e6] = "\xaa\x98",
  [0x72e7] = "\xaa\x99", [0x72e8] = "\xaa\x9a", [0x72e9] = "\xaa\x9b",
  [0x72ea] = "\xaa\x9c", [0x72eb] = "\xaa\x9d", [0x72ec] = "\xe2\xb3",
  [0x72ed] = "\xcc\xa1", [0x72ee] = "\xaa\x9e", [0x72ef] = "\xe2\xb4",
  [0x72f0] = "\xaa\x9f", [0x72f1] = "\xaa\xa0", [0x72f2] = "\xab\x40",
  [0x72f3] = "\xab\x41", [0x72f4] = "\xab\x42", [0x72f5] = "\xab\x43",
  [0x72f6] = "\xab\x44", [0x72f7] = "\xab\x45", [0x72f8] = "\xab\x46",
  [0x72f9] = "\xab\x47", [0x72fa] = "\xab\x48", [0x72fb] = "\xab\x49",
  [0x72fc] = "\xab\x4a", [0x72fd] = "\xab\x4b", [0x72fe] = "\xe2\xb5",
  [0x72ff] = "\xab\x4c", [0x7300] = "\xab\x4d", [0x7301] = "\xab\x4e",
  [0x7302] = "\xab\x4f", [0x7303] = "\xab\x50", [0x7304] = "\xd0\xfe",
  [0x7305] = "\xab\x51", [0x7306] = "\xab\x52", [0x7307] = "\xc2\xca",
  [0x7308] = "\xab\x53", [0x7309] = "\xd3\xf1", [0x730a] = "\xab\x54",
  [0x730b] = "\xcd\xf5", [0x730c] = "\xab\x55", [0x730d] = "\xab\x56",
  [0x730e] = "\xe7\xe0", [0x730f] = "\xab\x57", [0x7310] = "\xab\x58",
  [0x7311] = "\xe7\xe1", [0x7312] = "\xab\x59", [0x7313] = "\xab\x5a",
  [0x7314] = "\xab\x5b", [0x7315] = "\xab\x5c", [0x7316] = "\xbe\xc1",
  [0x7317] = "\xab\x5d", [0x7318] = "\xab\x5e", [0x7319] = "\xab\x5f",
  [0x731a] = "\xab\x60", [0x731b] = "\xc2\xea", [0x731c] = "\xab\x61",
  [0x731d] = "\xab\x62", [0x731e] = "\xab\x63", [0x731f] = "\xe7\xe4",
  [0x7320] = "\xab\x64", [0x7321] = "\xab\x65", [0x7322] = "\xe7\xe3",
  [0x7323] = "\xab\x66", [0x7324] = "\xab\x67", [0x7325] = "\xab\x68",
  [0x7326] = "\xab\x69", [0x7327] = "\xab\x6a", [0x7328] = "\xab\x6b",
  [0x7329] = "\xcd\xe6", [0x732a] = "\xab\x6c", [0x732b] = "\xc3\xb5",
  [0x732c] = "\xab\x6d", [0x732d] = "\xab\x6e", [0x732e] = "\xe7\xe2",
  [0x732f] = "\xbb\xb7", [0x7330] = "\xcf\xd6", [0x7331] = "\xab\x6f",
  [0x7332] = "\xc1\xe1", [0x7333] = "\xe7\xe9", [0x7334] = "\xab\x70",
  [0x7335] = "\xab\x71", [0x7336] = "\xab\x72", [0x7337] = "\xe7\xe8",
  [0x7338] = "\xab\x73", [0x7339] = "\xab\x74", [0x733a] = "\xe7\xf4",
  [0x733b] = "\xb2\xa3", [0x733c] = "\xab\x75", [0x733d] = "\xab\x76",
  [0x733e] = "\xab\x77", [0x733f] = "\xab\x78", [0x7340] = "\xe7\xea",
  [0x7341] = "\xab\x79", [0x7342] = "\xe7\xe6", [0x7343] = "\xab\x7a",
  [0x7344] = "\xab\x7b", [0x7345] = "\xab\x7c", [0x7346] = "\xab\x7d",
  [0x7347] = "\xab\x7e", [0x7348] = "\xe7\xec", [0x7349] = "\xe7\xeb",
  [0x734a] = "\xc9\xba", [0x734b] = "\xab\x80", [0x734c] = "\xab\x81",
  [0x734d] = "\xd5\xe4", [0x734e] = "\xab\x82", [0x734f] = "\xe7\xe5",
  [0x7350] = "\xb7\xa9", [0x7351] = "\xe7\xe7", [0x7352] = "\xab\x83",
  [0x7353] = "\xab\x84", [0x7354] = "\xab\x85", [0x7355] = "\xab\x86",
  [0x7356] = "\xab\x87", [0x7357] = "\xab\x88", [0x7358] = "\xab\x89",
  [0x7359] = "\xe7\xee", [0x735a] = "\xab\x8a", [0x735b] = "\xab\x8b",
  [0x735c] = "\xab\x8c", [0x735d] = "\xab\x8d", [0x735e] = "\xe7\xf3",
  [0x735f] = "\xab\x8e", [0x7360] = "\xd6\xe9", [0x7361] = "\xab\x8f",
  [0x7362] = "\xab\x90", [0x7363] = "\xab\x91", [0x7364] = "\xab\x92",
  [0x7365] = "\xe7\xed", [0x7366] = "\xab\x93", [0x7367] = "\xe7\xf2",
  [0x7368] = "\xab\x94", [0x7369] = "\xe7\xf1", [0x736a] = "\xab\x95",
  [0x736b] = "\xab\x96", [0x736c] = "\xab\x97", [0x736d] = "\xb0\xe0",
  [0x736e] = "\xab\x98", [0x736f] = "\xab\x99", [0x7370] = "\xab\x9a",
  [0x7371] = "\xab\x9b", [0x7372] = "\xe7\xf5", [0x7373] = "\xab\x9c",
  [0x7374] = "\xab\x9d", [0x7375] = "\xab\x9e", [0x7376] = "\xab\x9f",
  [0x7377] = "\xab\xa0", [0x7378] = "\xac\x40", [0x7379] = "\xac\x41",
  [0x737a] = "\xac\x42", [0x737b] = "\xac\x43", [0x737c] = "\xac\x44",
  [0x737d] = "\xac\x45", [0x737e] = "\xac\x46", [0x737f] = "\xac\x47",
  [0x7380] = "\xac\x48", [0x7381] = "\xac\x49", [0x7382] = "\xac\x4a",
  [0x7383] = "\xc7\xf2", [0x7384] = "\xac\x4b", [0x7385] = "\xc0\xc5",
  [0x7386] = "\xc0\xed", [0x7387] = "\xac\x4c", [0x7388] = "\xac\x4d",
  [0x7389] = "\xc1\xf0", [0x738a] = "\xe7\xf0", [0x738b] = "\xac\x4e",
  [0x738c] = "\xac\x4f", [0x738d] = "\xac\x50", [0x738e] = "\xac\x51",
  [0x738f] = "\xe7\xf6", [0x7390] = "\xcb\xf6", [0x7391] = "\xac\x52",
  [0x7392] = "\xac\x53", [0x7393] = "\xac\x54", [0x7394] = "\xac\x55",
  [0x7395] = "\xac\x56", [0x7396] = "\xac\x57", [0x7397] = "\xac\x58",
  [0x7398] = "\xac\x59", [0x7399] = "\xac\x5a", [0x739a] = "\xe8\xa2",
  [0x739b] = "\xe8\xa1", [0x739c] = "\xac\x5b", [0x739d] = "\xac\x5c",
  [0x739e] = "\xac\x5d", [0x739f] = "\xac\x5e", [0x73a0] = "\xac\x5f",
  [0x73a1] = "\xac\x60", [0x73a2] = "\xd7\xc1", [0x73a3] = "\xac\x61",
  [0x73a4] = "\xac\x62", [0x73a5] = "\xe7\xfa", [0x73a6] = "\xe7\xf9",
  [0x73a7] = "\xac\x63", [0x73a8] = "\xe7\xfb", [0x73a9] = "\xac\x64",
  [0x73aa] = "\xe7\xf7", [0x73ab] = "\xac\x65", [0x73ac] = "\xe7\xfe",
  [0x73ad] = "\xac\x66", [0x73ae] = "\xe7\xfd", [0x73af] = "\xac\x67",
  [0x73b0] = "\xe7\xfc", [0x73b1] = "\xac\x68", [0x73b2] = "\xac\x69",
  [0x73b3] = "\xc1\xd5", [0x73b4] = "\xc7\xd9", [0x73b5] = "\xc5\xfd",
  [0x73b6] = "\xc5\xc3", [0x73b7] = "\xac\x6a", [0x73b8] = "\xac\x6b",
  [0x73b9] = "\xac\x6c", [0x73ba] = "\xac\x6d", [0x73bb] = "\xac\x6e",
  [0x73bc] = "\xc7\xed", [0x73bd] = "\xac\x6f", [0x73be] = "\xac\x70",
  [0x73bf] = "\xac\x71", [0x73c0] = "\xac\x72", [0x73c1] = "\xe8\xa3",
  [0x73c2] = "\xac\x73", [0x73c3] = "\xac\x74", [0x73c4] = "\xac\x75",
  [0x73c5] = "\xac\x76", [0x73c6] = "\xac\x77", [0x73c7] = "\xac\x78",
  [0x73c8] = "\xac\x79", [0x73c9] = "\xac\x7a", [0x73ca] = "\xac\x7b",
  [0x73cb] = "\xac\x7c", [0x73cc] = "\xac\x7d", [0x73cd] = "\xac\x7e",
  [0x73ce] = "\xac\x80", [0x73cf] = "\xac\x81", [0x73d0] = "\xac\x82",
  [0x73d1] = "\xac\x83", [0x73d2] = "\xac\x84", [0x73d3] = "\xac\x85",
  [0x73d4] = "\xac\x86", [0x73d5] = "\xe8\xa6", [0x73d6] = "\xac\x87",
  [0x73d7] = "\xe8\xa5", [0x73d8] = "\xac\x88", [0x73d9] = "\xe8\xa7",
  [0x73da] = "\xba\xf7", [0x73db] = "\xe7\xf8", [0x73dc] = "\xe8\xa4",
  [0x73dd] = "\xac\x89", [0x73de] = "\xc8\xf0", [0x73df] = "\xc9\xaa",
  [0x73e0] = "\xac\x8a", [0x73e1] = "\xac\x8b", [0x73e2] = "\xac\x8c",
  [0x73e3] = "\xac\x8d", [0x73e4] = "\xac\x8e", [0x73e5] = "\xac\x8f",
  [0x73e6] = "\xac\x90", [0x73e7] = "\xac\x91", [0x73e8] = "\xac\x92",
  [0x73e9] = "\xac\x93", [0x73ea] = "\xac\x94", [0x73eb] = "\xac\x95",
  [0x73ec] = "\xac\x96", [0x73ed] = "\xe8\xa9", [0x73ee] = "\xac\x97",
  [0x73ef] = "\xac\x98", [0x73f0] = "\xb9\xe5", [0x73f1] = "\xac\x99",
  [0x73f2] = "\xac\x9a", [0x73f3] = "\xac\x9b", [0x73f4] = "\xac\x9c",
  [0x73f5] = "\xac\x9d", [0x73f6] = "\xd1\xfe", [0x73f7] = "\xe8\xa8",
  [0x73f8] = "\xac\x9e", [0x73f9] = "\xac\x9f", [0x73fa] = "\xac\xa0",
  [0x73fb] = "\xad\x40", [0x73fc] = "\xad\x41", [0x73fd] = "\xad\x42",
  [0x73fe] = "\xe8\xaa", [0x73ff] = "\xad\x43", [0x7400] = "\xe8\xad",
  [0x7401] = "\xe8\xae", [0x7402] = "\xad\x44", [0x7403] = "\xc1\xa7",
  [0x7404] = "\xad\x45", [0x7405] = "\xad\x46", [0x7406] = "\xad\x47",
  [0x7407] = "\xe8\xaf", [0x7408] = "\xad\x48", [0x7409] = "\xad\x49",
  [0x740a] = "\xad\x4a", [0x740b] = "\xe8\xb0", [0x740c] = "\xad\x4b",
  [0x740d] = "\xad\x4c", [0x740e] = "\xe8\xac", [0x740f] = "\xad\x4d",
  [0x7410] = "\xe8\xb4", [0x7411] = "\xad\x4e", [0x7412] = "\xad\x4f",
  [0x7413] = "\xad\x50", [0x7414] = "\xad\x51", [0x7415] = "\xad\x52",
  [0x7416] = "\xad\x53", [0x7417] = "\xad\x54", [0x7418] = "\xad\x55",
  [0x7419] = "\xad\x56", [0x741a] = "\xad\x57", [0x741b] = "\xad\x58",
  [0x741c] = "\xe8\xab", [0x741d] = "\xad\x59", [0x741e] = "\xe8\xb1",
  [0x741f] = "\xad\x5a", [0x7420] = "\xad\x5b", [0x7421] = "\xad\x5c",
  [0x7422] = "\xad\x5d", [0x7423] = "\xad\x5e", [0x7424] = "\xad\x5f",
  [0x7425] = "\xad\x60", [0x7426] = "\xad\x61", [0x7427] = "\xe8\xb5",
  [0x7428] = "\xe8\xb2", [0x7429] = "\xe8\xb3", [0x742a] = "\xad\x62",
  [0x742b] = "\xad\x63", [0x742c] = "\xad\x64", [0x742d] = "\xad\x65",
  [0x742e] = "\xad\x66", [0x742f] = "\xad\x67", [0x7430] = "\xad\x68",
  [0x7431] = "\xad\x69", [0x7432] = "\xad\x6a", [0x7433] = "\xad\x6b",
  [0x7434] = "\xad\x6c", [0x7435] = "\xad\x6d", [0x7436] = "\xad\x6e",
  [0x7437] = "\xad\x6f", [0x7438] = "\xad\x70", [0x7439] = "\xad\x71",
  [0x743a] = "\xe8\xb7", [0x743b] = "\xad\x72", [0x743c] = "\xad\x73",
  [0x743d] = "\xad\x74", [0x743e] = "\xad\x75", [0x743f] = "\xad\x76",
  [0x7440] = "\xad\x77", [0x7441] = "\xad\x78", [0x7442] = "\xad\x79",
  [0x7443] = "\xad\x7a", [0x7444] = "\xad\x7b", [0x7445] = "\xad\x7c",
  [0x7446] = "\xad\x7d", [0x7447] = "\xad\x7e", [0x7448] = "\xad\x80",
  [0x7449] = "\xad\x81", [0x744a] = "\xad\x82", [0x744b] = "\xad\x83",
  [0x744c] = "\xad\x84", [0x744d] = "\xad\x85", [0x744e] = "\xad\x86",
  [0x744f] = "\xad\x87", [0x7450] = "\xad\x88", [0x7451] = "\xad\x89",
  [0x7452] = "\xe8\xb6", [0x7453] = "\xad\x8a", [0x7454] = "\xad\x8b",
  [0x7455] = "\xad\x8c", [0x7456] = "\xad\x8d", [0x7457] = "\xad\x8e",
  [0x7458] = "\xad\x8f", [0x7459] = "\xad\x90", [0x745a] = "\xad\x91",
  [0x745b] = "\xad\x92", [0x745c] = "\xb9\xcf", [0x745d] = "\xad\x93",
  [0x745e] = "\xf0\xac", [0x745f] = "\xad\x94", [0x7460] = "\xf0\xad",
  [0x7461] = "\xad\x95", [0x7462] = "\xc6\xb0", [0x7463] = "\xb0\xea",
  [0x7464] = "\xc8\xbf", [0x7465] = "\xad\x96", [0x7466] = "\xcd\xdf",
  [0x7467] = "\xad\x97", [0x7468] = "\xad\x98", [0x7469] = "\xad\x99",
  [0x746a] = "\xad\x9a", [0x746b] = "\xad\x9b", [0x746c] = "\xad\x9c",
  [0x746d] = "\xad\x9d", [0x746e] = "\xce\xcd", [0x746f] = "\xea\xb1",
  [0x7470] = "\xad\x9e", [0x7471] = "\xad\x9f", [0x7472] = "\xad\xa0",
  [0x7473] = "\xae\x40", [0x7474] = "\xea\xb2", [0x7475] = "\xae\x41",
  [0x7476] = "\xc6\xbf", [0x7477] = "\xb4\xc9", [0x7478] = "\xae\x42",
  [0x7479] = "\xae\x43", [0x747a] = "\xae\x44", [0x747b] = "\xae\x45",
  [0x747c] = "\xae\x46", [0x747d] = "\xae\x47", [0x747e] = "\xae\x48",
  [0x747f] = "\xea\xb3", [0x7480] = "\xae\x49", [0x7481] = "\xae\x4a",
  [0x7482] = "\xae\x4b", [0x7483] = "\xae\x4c", [0x7484] = "\xd5\xe7",
  [0x7485] = "\xae\x4d", [0x7486] = "\xae\x4e", [0x7487] = "\xae\x4f",
  [0x7488] = "\xae\x50", [0x7489] = "\xae\x51", [0x748a] = "\xae\x52",
  [0x748b] = "\xae\x53", [0x748c] = "\xae\x54", [0x748d] = "\xdd\xf9",
  [0x748e] = "\xae\x55", [0x748f] = "\xea\xb4", [0x7490] = "\xae\x56",
  [0x7491] = "\xea\xb5", [0x7492] = "\xae\x57", [0x7493] = "\xea\xb6",
  [0x7494] = "\xae\x58", [0x7495] = "\xae\x59", [0x7496] = "\xae\x5a",
  [0x7497] = "\xae\x5b", [0x7498] = "\xb8\xca", [0x7499] = "\xdf\xb0",
  [0x749a] = "\xc9\xf5", [0x749b] = "\xae\x5c", [0x749c] = "\xcc\xf0",
  [0x749d] = "\xae\x5d", [0x749e] = "\xae\x5e", [0x749f] = "\xc9\xfa",
  [0x74a0] = "\xae\x5f", [0x74a1] = "\xae\x60", [0x74a2] = "\xae\x61",
  [0x74a3] = "\xae\x62", [0x74a4] = "\xae\x63", [0x74a5] = "\xc9\xfb",
  [0x74a6] = "\xae\x64", [0x74a7] = "\xae\x65", [0x74a8] = "\xd3\xc3",
  [0x74a9] = "\xcb\xa6", [0x74aa] = "\xae\x66", [0x74ab] = "\xb8\xa6",
  [0x74ac] = "\xf0\xae", [0x74ad] = "\xb1\xc2", [0x74ae] = "\xae\x67",
  [0x74af] = "\xe5\xb8", [0x74b0] = "\xcc\xef", [0x74b1] = "\xd3\xc9",
  [0x74b2] = "\xbc\xd7", [0x74b3] = "\xc9\xea", [0x74b4] = "\xae\x68",
  [0x74b5] = "\xb5\xe7", [0x74b6] = "\xae\x69", [0x74b7] = "\xc4\xd0",
  [0x74b8] = "\xb5\xe9", [0x74b9] = "\xae\x6a", [0x74ba] = "\xee\xae",
  [0x74bb] = "\xbb\xad", [0x74bc] = "\xae\x6b", [0x74bd] = "\xae\x6c",
  [0x74be] = "\xe7\xde", [0x74bf] = "\xae\x6d", [0x74c0] = "\xee\xaf",
  [0x74c1] = "\xae\x6e", [0x74c2] = "\xae\x6f", [0x74c3] = "\xae\x70",
  [0x74c4] = "\xae\x71", [0x74c5] = "\xb3\xa9", [0x74c6] = "\xae\x72",
  [0x74c7] = "\xae\x73", [0x74c8] = "\xee\xb2", [0x74c9] = "\xae\x74",