/* Opcode table for the H8/300
   Copyright 1991, 1992, 1993, 1994, 1996, 1997, 1998, 2000, 2002, 2003, 2004
   Free Software Foundation, Inc.
   Written by Steve Chamberlain <sac@cygnus.com>.
   
   This file is part of GDB, the GNU Debugger and GAS, the GNU Assembler.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

/* Instructions are stored as a sequence of nibbles.
   If the nibble has value 15 or less than the representation is complete.
   Otherwise, we record what it contains with several flags.  */

typedef int op_type;

enum h8_flags
{
  L_2  =	0x10,
  L_3  =	0x20,
  /* 3 bit constant, zero not accepted.  */
  L_3NZ =	0x30,
  L_4  =	0x40,
  L_5  =	0x50,
  L_8  =	0x60,
  L_8U =	0x70,
  L_16 =	0x80,
  L_16U =	0x90,
  L_24 =	0xA0,
  L_32 =	0xB0,
  L_P  =	0xC0,

  /* Mask to isolate the L_x size bits.  */
  SIZE =	0xF0,

  REG =		0x0100,
  ABS =		0x0200,
  MEMIND =	0x0300,
  IMM =		0x0400,
  DISP =	0x0500,
  IND =		0x0600,
  POSTINC =	0x0700,
  POSTDEC =	0x0800,
  PREINC =	0x0900,
  PREDEC =	0x0A00,
  PCREL =	0x0B00,
  KBIT =	0x0C00,
  DBIT =	0x0D00,
  CONST_2 =     0x0E00,
  CONST_4 =     0x0F00,
  CONST_8 =     0x1000,
  CONST_16 =    0x1100,
  INDEXB =      0x1200,
  INDEXW =      0x1300,
  INDEXL =      0x1400,
  PCIDXB =      0x1500,
  PCIDXW =      0x1600,
  PCIDXL =      0x1700,
  VECIND =      0x1800,
  LOWREG =      0x1900,
  DATA   =      0x2000,

  /* Synonyms.  */
  INC =		POSTINC,
  DEC =		PREDEC,
  /* Control Registers.  */
  CCR =		0x4000,
  EXR =		0x4100,
  MACH = 	0x4200,
  MACL = 	0x4300,
  RESERV1 = 	0x4400,
  RESERV2 = 	0x4500,
  VBR = 	0x4600,
  SBR = 	0x4700,
  MACREG =	0x4800,
  CCR_EXR = 	0x4900,
  VBR_SBR = 	0x4A00,
  CC_EX_VB_SB =	0x4B00,
  RESERV3 =	0x4C00,
  RESERV4 = 	0x4D00,
  RESERV5 = 	0x4E00,
  RESERV6 = 	0x4F00,

  /* Mask to isolate the addressing mode bits (REG .. PREDEC).  */
  MODE =	0x7F00,

  CTRL = 	0x4000,

  NO_SYMBOLS =  0x8000,
  SRC =		0x10000,
  DST =		0x20000,
  OP3 =		0x40000,
  MEMRELAX = 	0x80000,		/* Move insn which may relax.  */

  DISPREG =	0x100000,
  IGNORE =	0x200000,
  ABSJMP =	0x400000,

  B00 =         0x800000, 		/* Bit 0 must be low.   */
  B01 =         0x1000000, 		/* Bit 0 must be high.  */
  B10 = 	0x2000000, 		/* Bit 1 must be low.   */
  B11 = 	0x4000000, 		/* Bit 1 must be high.  */
  B20 = 	0x8000000, 		/* Bit 2 must be low.   */
  B21 = 	0x10000000, 		/* Bit 2 must be high.  */
  B30 =		0x20000000, 		/* Bit 3 must be low.   */
  B31 =		0x40000000,		/* Bit 3 must be high.  */
  E =  		0x80000000,		/* End of nibble sequence.  */

  /* Immediates smaller than 8 bits are always unsigned.  */
  IMM3 =	IMM | L_3,
  IMM4 =	IMM | L_4,
  IMM5 =	IMM | L_5,
  IMM3NZ =	IMM | L_3NZ,
  IMM2 =	IMM | L_2,

  IMM8 =	IMM | SRC | L_8,
  IMM8U =	IMM | SRC | L_8U,
  IMM16 =	IMM | SRC | L_16,
  IMM16U =	IMM | SRC | L_16U,
  IMM32 =	IMM | SRC | L_32,

  IMM3NZ_NS =   IMM3NZ | NO_SYMBOLS,
  IMM4_NS =	IMM4 | NO_SYMBOLS,
  IMM8U_NS =	IMM8U | NO_SYMBOLS,
  IMM16U_NS =   IMM16U | NO_SYMBOLS,

  RD8  =	DST | L_8  | REG,
  RD16 =	DST | L_16 | REG,
  RD32 =	DST | L_32 | REG,
  R3_8  =       OP3 | L_8  | REG,
  R3_16 =       OP3 | L_16 | REG,
  R3_32 =       OP3 | L_32 | REG,
  RS8  =	SRC | L_8  | REG,
  RS16 =	SRC | L_16 | REG,
  RS32 =	SRC | L_32 | REG,

  RSP =		SRC | L_P | REG,
  RDP =		DST | L_P | REG,

  PCREL8 =	PCREL | L_8,
  PCREL16 =	PCREL | L_16,

  OP3PCREL8 =	OP3 | PCREL | L_8,
  OP3PCREL16 =	OP3 | PCREL | L_16,

  INDEXB16  =	INDEXB | L_16,
  INDEXW16  =	INDEXW | L_16,
  INDEXL16  =	INDEXL | L_16,
  INDEXB16D =	INDEXB | L_16 | DST,
  INDEXW16D =	INDEXW | L_16 | DST,
  INDEXL16D =	INDEXL | L_16 | DST,

  INDEXB32  =	INDEXB | L_32,
  INDEXW32  =	INDEXW | L_32,
  INDEXL32  =	INDEXL | L_32,
  INDEXB32D =	INDEXB | L_32 | DST,
  INDEXW32D =	INDEXW | L_32 | DST,
  INDEXL32D =	INDEXL | L_32 | DST,

  DISP2SRC =	DISP | L_2  | SRC,
  DISP16SRC =	DISP | L_16 | SRC,
  DISP32SRC =	DISP | L_32 | SRC,

  DISP2DST =	DISP | L_2  | DST,
  DISP16DST =	DISP | L_16 | DST,
  DISP32DST =	DISP | L_32 | DST,

  DSTDISPREG =  DST | DISPREG,
  SRCDISPREG =  SRC | DISPREG,

  ABS8SRC  = 	SRC | ABS | L_8,
  ABS16SRC = 	SRC | ABS | L_16U,
  ABS24SRC = 	SRC | ABS | L_24,
  ABS32SRC = 	SRC | ABS | L_32,

  ABS8DST  =	DST | ABS | L_8,
  ABS16DST = 	DST | ABS | L_16U,
  ABS24DST = 	DST | ABS | L_24,
  ABS32DST = 	DST | ABS | L_32,

  ABS8OP3  =	OP3 | ABS | L_8,
  ABS16OP3 = 	OP3 | ABS | L_16U,
  ABS24OP3 = 	OP3 | ABS | L_24,
  ABS32OP3 = 	OP3 | ABS | L_32,

  RDDEC =	DST | DEC,
  RSINC =	SRC | INC,
  RDINC =	DST | INC,

  RSPOSTINC =	SRC | POSTINC,
  RDPOSTINC =	DST | POSTINC,
  RSPREINC =	SRC | PREINC,
  RDPREINC =	DST | PREINC,
  RSPOSTDEC =	SRC | POSTDEC,
  RDPOSTDEC =	DST | POSTDEC,
  RSPREDEC =	SRC | PREDEC,
  RDPREDEC =	DST | PREDEC,

  RSIND =	SRC | IND,
  RDIND =	DST | IND,
  R3_IND = 	OP3 | IND,

#define MS32	(SRC | L_32 | MACREG)
#define MD32	(DST | L_32 | MACREG)

#if 1
  OR8  =	RS8,		/* ??? OR as in One Register.  */
  OR16 =	RS16,
  OR32 =	RS32,
#else
  OR8  =	RD8,
  OR16 =	RD16,
  OR32 =	RD32
#endif
};

enum ctrlreg
{
  C_CCR  = 0, 
  C_EXR  = 1, 
  C_MACH = 2, 
  C_MACL = 3, 
  C_VBR  = 6, 
  C_SBR  = 7
};

enum {MAX_CODE_NIBBLES = 33};

struct code 
{
  op_type nib[MAX_CODE_NIBBLES];
};

struct arg 
{
  op_type nib[3];
};

/* Availability of instructions on processor models.  */
enum h8_model
{
  AV_H8,
  AV_H8H,
  AV_H8S,
  AV_H8SX
};

struct h8_opcode 
{
  int how;
  enum h8_model available;
  int time;
  char *name;
  struct arg args;
  struct code data;
};

#ifdef DEFINE_TABLE

#define DATA2   DATA, DATA
#define DATA3   DATA, DATA, DATA
#define DATA5   DATA, DATA, DATA, DATA, DATA
#define DATA7   DATA, DATA, DATA, DATA, DATA, DATA, DATA

#define IMM8LIST	IMM8,   DATA
#define IMM16LIST	IMM16,  DATA3
#define IMM16ULIST	IMM16U, DATA3
#define IMM24LIST	IMM24,  DATA5
#define IMM32LIST	IMM32,  DATA7

#define DISP16LIST DISP | L_16, DATA3
#define DISP24LIST DISP | L_24, DATA5
#define DISP32LIST DISP | L_32, DATA7

#define ABS8LIST   ABS  | L_8,   DATA
#define ABS16LIST  ABS  | L_16U, DATA3
#define ABS24LIST  ABS  | L_24,  DATA5
#define ABS32LIST  ABS  | L_32,  DATA7

#define DSTABS8LIST  DST | ABS | L_8,   DATA
#define DSTABS16LIST DST | ABS | L_16U, DATA3
#define DSTABS24LIST DST | ABS | L_24,  DATA5
#define DSTABS32LIST DST | ABS | L_32,  DATA7

#define OP3ABS8LIST  OP3 | ABS | L_8,  DATA
#define OP3ABS16LIST OP3 | ABS | L_16, DATA3
#define OP3ABS24LIST OP3 | ABS | L_24, DATA5
#define OP3ABS32LIST OP3 | ABS | L_32, DATA7

#define DSTDISP16LIST DST | DISP | L_16, DATA3
#define DSTDISP24LIST DST | DISP | L_24, DATA5
#define DSTDISP32LIST DST | DISP | L_32, DATA7

#define A16LIST   L_16,  DATA3
#define A24LIST   L_24,  DATA5
#define A32LIST   L_32,  DATA7

/* Extended Operand Prefixes:  */

#define PREFIX_010	0x0, 0x1, 0x0
#define PREFIX_015	0x0, 0x1, 0x5
#define PREFIX_017	0x0, 0x1, 0x7

#define PREFIX_0100	0x0, 0x1, 0x0, 0x0
#define PREFIX_010_D2	0x0, 0x1, 0x0, B30 | B21 | DISP2SRC
#define PREFIX_0101	0x0, 0x1, 0x0, 0x1
#define PREFIX_0102	0x0, 0x1, 0x0, 0x2
#define PREFIX_0103	0x0, 0x1, 0x0, 0x3
#define PREFIX_0104	0x0, 0x1, 0x0, 0x4
#define PREFIX_0105	0x0, 0x1, 0x0, 0x5
#define PREFIX_0106	0x0, 0x1, 0x0, 0x6
#define PREFIX_0107	0x0, 0x1, 0x0, 0x7
#define PREFIX_0108	0x0, 0x1, 0x0, 0x8
#define PREFIX_0109	0x0, 0x1, 0x0, 0x9
#define PREFIX_010A	0x0, 0x1, 0x0, 0xa
#define PREFIX_010D	0x0, 0x1, 0x0, 0xd
#define PREFIX_010E	0x0, 0x1, 0x0, 0xe

#define PREFIX_0150	0x0, 0x1, 0x5, 0x0
#define PREFIX_015_D2	0x0, 0x1, 0x5, B30 | B21 | DISP2SRC
#define PREFIX_0151	0x0, 0x1, 0x5, 0x1
#define PREFIX_0152	0x0, 0x1, 0x5, 0x2
#define PREFIX_0153	0x0, 0x1, 0x5, 0x3
#define PREFIX_0154	0x0, 0x1, 0x5, 0x4
#define PREFIX_0155	0x0, 0x1, 0x5, 0x5
#define PREFIX_0156	0x0, 0x1, 0x5, 0x6
#define PREFIX_0157	0x0, 0x1, 0x5, 0x7
#define PREFIX_0158	0x0, 0x1, 0x5, 0x8
#define PREFIX_0159	0x0, 0x1, 0x5, 0x9
#define PREFIX_015A	0x0, 0x1, 0x5, 0xa
#define PREFIX_015D	0x0, 0x1, 0x5, 0xd
#define PREFIX_015E	0x0, 0x1, 0x5, 0xe
#define PREFIX_015F	0x0, 0x1, 0x5, 0xf

#define PREFIX_0170	0x0, 0x1, 0x7, 0x0
#define PREFIX_017_D2S	0x0, 0x1, 0x7, B30 | B21 | DISP2SRC
#define PREFIX_017_D2D	0x0, 0x1, 0x7, B30 | B21 | DISP2DST
#define PREFIX_0171	0x0, 0x1, 0x7, 0x1
#define PREFIX_0172	0x0, 0x1, 0x7, 0x2
#define PREFIX_0173	0x0, 0x1, 0x7, 0x3
#define PREFIX_0174	0x0, 0x1, 0x7, 0x4
#define PREFIX_0175	0x0, 0x1, 0x7, 0x5
#define PREFIX_0176	0x0, 0x1, 0x7, 0x6
#define PREFIX_0177	0x0, 0x1, 0x7, 0x7
#define PREFIX_0178	0x0, 0x1, 0x7, 0x8
#define PREFIX_0179	0x0, 0x1, 0x7, 0x9
#define PREFIX_017A	0x0, 0x1, 0x7, 0xa
#define PREFIX_017D	0x0, 0x1, 0x7, 0xd
#define PREFIX_017E	0x0, 0x1, 0x7, 0xe
#define PREFIX_017F	0x0, 0x1, 0x7, 0xf

#define PREFIX_6A15	0x6, 0xa, 0x1, 0x5
#define PREFIX_6A35	0x6, 0xa, 0x3, 0x5
#define PREFIX_6B15	0x6, 0xb, 0x1, 0x5
#define PREFIX_6B35	0x6, 0xb, 0x3, 0x5

#define PREFIX_78R4	0x7, 0x8, B31 | DISPREG, 0x4
#define PREFIX_78R5	0x7, 0x8, B31 | DISPREG, 0x5
#define PREFIX_78R6	0x7, 0x8, B31 | DISPREG, 0x6
#define PREFIX_78R7	0x7, 0x8, B31 | DISPREG, 0x7

#define PREFIX_78R4W	0x7, 0x8, B30 | DISPREG, 0x4
#define PREFIX_78R5W	0x7, 0x8, B30 | DISPREG, 0x5
#define PREFIX_78R6W	0x7, 0x8, B30 | DISPREG, 0x6
#define PREFIX_78R7W	0x7, 0x8, B30 | DISPREG, 0x7

#define PREFIX_78R4WD	0x7, 0x8, B30 | DSTDISPREG, 0x4
#define PREFIX_78R5WD	0x7, 0x8, B30 | DSTDISPREG, 0x5
#define PREFIX_78R6WD	0x7, 0x8, B30 | DSTDISPREG, 0x6
#define PREFIX_78R7WD	0x7, 0x8, B30 | DSTDISPREG, 0x7

#define PREFIX_7974	0x7, 0x9, 0x7, 0x4
#define PREFIX_7A74	0x7, 0xa, 0x7, 0x4
#define PREFIX_7A7C	0x7, 0xa, 0x7, 0xc


/* Source standard fragment:  */
#define FROM_IND	 0, RSIND
#define FROM_POSTINC	 8, RSPOSTINC
#define FROM_POSTDEC	10, RSPOSTDEC
#define FROM_PREINC	 9, RSPREINC
#define FROM_PREDEC	11, RSPREDEC
#define FROM_DISP2	B30 | B20 | DISP2SRC, DISPREG 
#define FROM_DISP16	12, B30 | DISPREG
#define FROM_DISP32	12, B31 | DISPREG
#define FROM_DISP16B	13, B30 | DISPREG
#define FROM_DISP16W	14, B30 | DISPREG
#define FROM_DISP16L	15, B30 | DISPREG
#define FROM_DISP32B	13, B31 | DISPREG
#define FROM_DISP32W	14, B31 | DISPREG
#define FROM_DISP32L	15, B31 | DISPREG
#define FROM_ABS16	 4, B30 | IGNORE
#define FROM_ABS32	 4, B31 | IGNORE

/* Destination standard fragment:  */
#define TO_IND		 0, RDIND
#define TO_IND_MOV	 0, RDIND | B30
#define TO_POSTINC	 8, RDPOSTINC
#define TO_POSTINC_MOV	 8, RDPOSTINC | B30
#define TO_POSTDEC	10, RDPOSTDEC
#define TO_POSTDEC_MOV	10, RDPOSTDEC | B30
#define TO_PREINC	 9, RDPREINC
#define TO_PREINC_MOV	 9, RDPREINC  | B30
#define TO_PREDEC	11, RDPREDEC
#define TO_PREDEC_MOV	11, RDPREDEC  | B30
#define TO_DISP2	B30 | B20 | DISP2DST, DSTDISPREG
#define TO_DISP2_MOV	B30 | B20 | DISP2DST, DSTDISPREG | B30
#define TO_DISP16	12, B30 | DSTDISPREG
#define TO_DISP32	12, B31 | DSTDISPREG
#define TO_DISP16B	13, B30 | DSTDISPREG
#define TO_DISP16W	14, B30 | DSTDISPREG
#define TO_DISP16L	15, B30 | DSTDISPREG
#define TO_DISP32B	13, B31 | DSTDISPREG
#define TO_DISP32W	14, B31 | DSTDISPREG
#define TO_DISP32L	15, B31 | DSTDISPREG
#define TO_ABS16	 4, B30 | IGNORE
#define TO_ABS32	 4, B31 | IGNORE

/* Source fragment for three-word instruction:  */
#define TFROM_IND	6,  9, B30 | RSIND, 12
#define TFROM_DISP2	6,  9, B30 | DISPREG, 12
#define TFROM_ABS16	6, 11, B30 | B20 | B10 | IGNORE, 12, ABS16LIST
#define TFROM_ABS32	6, 11, B30 | B20 | B11 | IGNORE, 12, ABS32LIST
#define TFROM_POSTINC	6, 13, B30 | RSPOSTINC, 12
#define TFROM_PREINC	6, 13, B30 | RSPREINC, 12
#define TFROM_POSTDEC	6, 13, B30 | RSPOSTDEC, 12
#define TFROM_PREDEC	6, 13, B30 | RSPREDEC, 12
#define TFROM_DISP16	6, 15, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP32	6, 11, 2, 12, DISP32LIST
#define TFROM_DISP16B	6, 15, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP16W	6, 15, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP16L	6, 15, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP32B	6, 11, 2, 12, DISP32LIST
#define TFROM_DISP32W	6, 11, 2, 12, DISP32LIST
#define TFROM_DISP32L	6, 11, 2, 12, DISP32LIST
#define TFROM_ABS16W	6, 11, 1, 12, ABS16LIST
#define TFROM_ABS32W	6, 11, 3, 12, ABS32LIST

/* Source fragment for three-word instruction:  */
#define TFROM_IND_B	6,  8, B30 | RSIND, 12
#define TFROM_ABS16_B	6, 10, B30 | B20 | B10 | IGNORE, 12, ABS16LIST
#define TFROM_ABS32_B	6, 10, B30 | B20 | B11 | IGNORE, 12, ABS32LIST

#define TFROM_DISP2_B	6,  8, B30 | DISPREG, 12
#define TFROM_POSTINC_B	6, 12, B30 | RSPOSTINC, 12
#define TFROM_PREINC_B	6, 12, B30 | RSPREINC, 12
#define TFROM_POSTDEC_B	6, 12, B30 | RSPOSTDEC, 12
#define TFROM_PREDEC_B	6, 12, B30 | RSPREDEC, 12
#define TFROM_DISP16_B	6, 14, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP32_B	6, 10, 2, 12, DISP32LIST
#define TFROM_DISP16B_B	6, 14, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP16W_B	6, 14, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP16L_B	6, 14, B30 | DISPREG, 12, DISP16LIST
#define TFROM_DISP32B_B	6, 10, 2, 12, DISP32LIST
#define TFROM_DISP32W_B	6, 10, 2, 12, DISP32LIST
#define TFROM_DISP32L_B	6, 10, 2, 12, DISP32LIST

#define TFROM_ABS16W_B	6, 10, 1, 12, ABS16LIST
#define TFROM_ABS32W_B	6, 10, 3, 12, ABS32LIST

/* Extended Operand Class Expanders:  */

#define MOVFROM_STD(CODE, PREFIX, NAME, SRC, SRC_INFIX) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDIND,     E}},  {{PREFIX, SRC_INFIX, TO_IND_MOV,     E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}},  {{PREFIX, SRC_INFIX, TO_POSTINC_MOV, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}},  {{PREFIX, SRC_INFIX, TO_POSTDEC_MOV, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}},  {{PREFIX, SRC_INFIX, TO_PREINC_MOV,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}},  {{PREFIX, SRC_INFIX, TO_PREDEC_MOV,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}},  {{PREFIX, SRC_INFIX, TO_DISP2_MOV,   E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}},  {{PREFIX, SRC_INFIX, TO_DISP16,  DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}},  {{PREFIX, SRC_INFIX, TO_DISP32,  DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}},  {{PREFIX, SRC_INFIX, TO_DISP16B, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}},  {{PREFIX, SRC_INFIX, TO_DISP16W, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}},  {{PREFIX, SRC_INFIX, TO_DISP16L, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}},  {{PREFIX, SRC_INFIX, TO_DISP32B, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}},  {{PREFIX, SRC_INFIX, TO_DISP32W, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}},  {{PREFIX, SRC_INFIX, TO_DISP32L, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS16DST,  E}},  {{PREFIX, SRC_INFIX, TO_ABS16,   DSTABS16LIST,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS32DST,  E}},  {{PREFIX, SRC_INFIX, TO_ABS32,   DSTABS32LIST,  E}}}

#define MOVFROM_AD(CODE, PREFIX, NAME, SRC, SRC_INFIX, SRC_SUFFIX) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDIND,     E}},  {{PREFIX, SRC_INFIX, TO_IND_MOV,     SRC_SUFFIX, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}},  {{PREFIX, SRC_INFIX, TO_POSTINC_MOV, SRC_SUFFIX, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}},  {{PREFIX, SRC_INFIX, TO_POSTDEC_MOV, SRC_SUFFIX, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}},  {{PREFIX, SRC_INFIX, TO_PREINC_MOV,  SRC_SUFFIX, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}},  {{PREFIX, SRC_INFIX, TO_PREDEC_MOV,  SRC_SUFFIX, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}},  {{PREFIX, SRC_INFIX, TO_DISP2_MOV,   SRC_SUFFIX, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}},  {{PREFIX, SRC_INFIX, TO_DISP16,  SRC_SUFFIX, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}},  {{PREFIX, SRC_INFIX, TO_DISP32,  SRC_SUFFIX, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}},  {{PREFIX, SRC_INFIX, TO_DISP16B, SRC_SUFFIX, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}},  {{PREFIX, SRC_INFIX, TO_DISP16W, SRC_SUFFIX, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}},  {{PREFIX, SRC_INFIX, TO_DISP16L, SRC_SUFFIX, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}},  {{PREFIX, SRC_INFIX, TO_DISP32B, SRC_SUFFIX, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}},  {{PREFIX, SRC_INFIX, TO_DISP32W, SRC_SUFFIX, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}},  {{PREFIX, SRC_INFIX, TO_DISP32L, SRC_SUFFIX, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS16DST,  E}},  {{PREFIX, SRC_INFIX, TO_ABS16,   SRC_SUFFIX, DSTABS16LIST,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS32DST,  E}},  {{PREFIX, SRC_INFIX, TO_ABS32,   SRC_SUFFIX, DSTABS32LIST,  E}}}

#define MOVFROM_IMM8(CODE, PREFIX, NAME, SRC) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDIND,     E}},  {{PREFIX,  0, RDIND,     IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}},  {{PREFIX,  8, RDPOSTINC, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}},  {{PREFIX, 10, RDPOSTDEC, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}},  {{PREFIX,  9, RDPREINC,  IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}},  {{PREFIX, 11, RDPREDEC,  IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}},  {{PREFIX, B30 | B20 | DISP2DST, DSTDISPREG, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}},  {{PREFIX, 12, B30 | DSTDISPREG, IMM8LIST, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}},  {{PREFIX, 12, B31 | DSTDISPREG, IMM8LIST, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}},  {{PREFIX, 13, B30 | DSTDISPREG, IMM8LIST, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}},  {{PREFIX, 14, B30 | DSTDISPREG, IMM8LIST, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}},  {{PREFIX, 15, B30 | DSTDISPREG, IMM8LIST, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}},  {{PREFIX, 13, B31 | DSTDISPREG, IMM8LIST, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}},  {{PREFIX, 14, B31 | DSTDISPREG, IMM8LIST, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}},  {{PREFIX, 15, B31 | DSTDISPREG, IMM8LIST, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS16DST,  E}},  {{PREFIX,  4, B30 | IGNORE,     IMM8LIST, DSTABS16LIST,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS32DST,  E}},  {{PREFIX,  4, B31 | IGNORE,     IMM8LIST, DSTABS32LIST,  E}}}

#define MOVFROM_IMM(CODE, PREFIX, NAME, SRC, LIST) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDIND,     E}},  {{PREFIX, LIST,  0, RDIND,     DATA2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}},  {{PREFIX, LIST,  8, RDPOSTINC, DATA2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}},  {{PREFIX, LIST, 10, RDPOSTDEC, DATA2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}},  {{PREFIX, LIST,  9, RDPREINC,  DATA2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}},  {{PREFIX, LIST, 11, RDPREDEC,  DATA2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}},  {{PREFIX, LIST, B30 | B20 | DISP2DST, DSTDISPREG, DATA2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}},  {{PREFIX, LIST, 12, B30 | DSTDISPREG, DATA2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}},  {{PREFIX, LIST, 12, B31 | DSTDISPREG, DATA2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}},  {{PREFIX, LIST, 13, B30 | DSTDISPREG, DATA2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}},  {{PREFIX, LIST, 14, B30 | DSTDISPREG, DATA2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}},  {{PREFIX, LIST, 15, B30 | DSTDISPREG, DATA2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}},  {{PREFIX, LIST, 13, B31 | DSTDISPREG, DATA2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}},  {{PREFIX, LIST, 14, B31 | DSTDISPREG, DATA2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}},  {{PREFIX, LIST, 15, B31 | DSTDISPREG, DATA2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS16DST,  E}},  {{PREFIX, LIST,  4, B30 | IGNORE,     DATA2, DSTABS16LIST,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS32DST,  E}},  {{PREFIX, LIST,  4, B31 | IGNORE,     DATA2, DSTABS32LIST,  E}}}

#define MOVFROM_REG_BW(CODE, NAME, SRC, PREFIX, OP1, OP2, OP3, OP4, RELAX16) \
  {CODE, AV_H8,   4, NAME, {{SRC, RDIND,     E}}, {{                              6, OP1, B31 | RDIND,      SRC,                E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}}, {{PREFIX, 3,                    6, OP3, B31 | RDPOSTINC,  SRC,                E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}}, {{PREFIX, 1,                    6, OP3, B31 | RDPOSTDEC,  SRC,                E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}}, {{PREFIX, 2,                    6, OP3, B31 | RDPREINC,   SRC,                E}}}, \
  {CODE, AV_H8,   6, NAME, {{SRC, RDPREDEC,  E}}, {{                              6, OP3, B31 | RDPREDEC,   SRC,                E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}}, {{PREFIX, B30 | B20 | DISP2DST, 6, OP1, B31 | DSTDISPREG, SRC,                E}}}, \
  {CODE, AV_H8,   6, NAME, {{SRC, DISP16DST, E}}, {{                              6, OP4, B31 | DSTDISPREG, SRC, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8,   6, NAME, {{SRC, DISP32DST, E}}, {{7, 8, B30 | DSTDISPREG, 0,    6, OP2, 10,               SRC, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}}, {{PREFIX, 1,                    6, OP4, B31 | DSTDISPREG, SRC, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}}, {{PREFIX, 2,                    6, OP4, B31 | DSTDISPREG, SRC, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}}, {{PREFIX, 3,                    6, OP4, B31 | DSTDISPREG, SRC, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}}, {{7, 8, B30 | DSTDISPREG, 1,    6, OP2, 10,               SRC, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}}, {{7, 8, B30 | DSTDISPREG, 2,    6, OP2, 10,               SRC, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}}, {{7, 8, B30 | DSTDISPREG, 3,    6, OP2, 10,               SRC, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8,   4, NAME, {{SRC, ABS16DST,  E}}, {{                              6, OP2, 8,                SRC, RELAX16  | DSTABS16LIST, E}}}, \
  {CODE, AV_H8,   6, NAME, {{SRC, ABS32DST,  E}}, {{                              6, OP2, 10,               SRC, MEMRELAX | DSTABS32LIST, E}}}

#define MOVTO_REG_BW(CODE, NAME, DST, PREFIX, OP1, OP2, OP3, OP4, RELAX16) \
  {CODE, AV_H8,   4, NAME, {{RSIND,     DST, E}}, {{                              6, OP1, B30 | RSIND,     DST,             E}}}, \
  {CODE, AV_H8,   6, NAME, {{RSPOSTINC, DST, E}}, {{                              6, OP3, B30 | RSPOSTINC, DST,             E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTDEC, DST, E}}, {{PREFIX, 2,                    6, OP3, B30 | RSPOSTDEC, DST,             E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREINC,  DST, E}}, {{PREFIX, 1,                    6, OP3, B30 | RSPREINC,  DST,             E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREDEC,  DST, E}}, {{PREFIX, 3,                    6, OP3, B30 | RSPREDEC,  DST,             E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP2SRC,  DST, E}}, {{PREFIX, B30 | B20 | DISP2SRC, 6, OP1, B30 | DISPREG,   DST,             E}}}, \
  {CODE, AV_H8,   6, NAME, {{DISP16SRC, DST, E}}, {{                              6, OP4, B30 | DISPREG,   DST, DISP16LIST, E}}}, \
  {CODE, AV_H8,   6, NAME, {{DISP32SRC, DST, E}}, {{7, 8, B30 | DISPREG, 0,       6, OP2, 2,               DST, DISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB16,  DST, E}}, {{PREFIX, 1,                    6, OP4, B30 | DISPREG,   DST, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW16,  DST, E}}, {{PREFIX, 2,                    6, OP4, B30 | DISPREG,   DST, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL16,  DST, E}}, {{PREFIX, 3,                    6, OP4, B30 | DISPREG,   DST, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB32,  DST, E}}, {{7, 8, B30 | DISPREG, 1,       6, OP2, 2,               DST, DISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW32,  DST, E}}, {{7, 8, B30 | DISPREG, 2,       6, OP2, 2,               DST, DISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL32,  DST, E}}, {{7, 8, B30 | DISPREG, 3,       6, OP2, 2,               DST, DISP32LIST, E}}}, \
  {CODE, AV_H8,   4, NAME, {{ABS16SRC,  DST, E}}, {{                              6, OP2, 0,               DST, RELAX16  | ABS16LIST, E}}}, \
  {CODE, AV_H8,   6, NAME, {{ABS32SRC,  DST, E}}, {{                              6, OP2, 2,               DST, MEMRELAX | ABS32LIST, E}}}

/* Expansion macros for two-word (plus data) instructions.  */

/* Expansion from one source to "standard" destinations.  */
#define EXPAND2_STD_SRC(CODE, WEIGHT, NAME, SRC, PREFIX, NIB1, NIB2) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}}, {{PREFIX, TO_POSTINC, NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}}, {{PREFIX, TO_POSTDEC, NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}}, {{PREFIX, TO_PREINC,  NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}}, {{PREFIX, TO_PREDEC,  NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}}, {{PREFIX, TO_DISP2,   NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}}, {{PREFIX, TO_DISP16,  NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}}, {{PREFIX, TO_DISP32,  NIB1, NIB2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}}, {{PREFIX, TO_DISP16B, NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}}, {{PREFIX, TO_DISP16W, NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}}, {{PREFIX, TO_DISP16L, NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}}, {{PREFIX, TO_DISP32B, NIB1, NIB2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}}, {{PREFIX, TO_DISP32W, NIB1, NIB2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}}, {{PREFIX, TO_DISP32L, NIB1, NIB2, DSTDISP32LIST, E}}}

/* Expansion from one destination to "standard" sources.  */
#define EXPAND2_STD_DST(CODE, WEIGHT, NAME, DST, PREFIX, NIB1, NIB2) \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTINC, DST, E}}, {{PREFIX, FROM_POSTINC, NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTDEC, DST, E}}, {{PREFIX, FROM_POSTDEC, NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREINC,  DST, E}}, {{PREFIX, FROM_PREINC,  NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREDEC,  DST, E}}, {{PREFIX, FROM_PREDEC,  NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP2SRC,  DST, E}}, {{PREFIX, FROM_DISP2,   NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP16SRC, DST, E}}, {{PREFIX, FROM_DISP16,  NIB1, NIB2, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP32SRC, DST, E}}, {{PREFIX, FROM_DISP32,  NIB1, NIB2, DISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB16,  DST, E}}, {{PREFIX, FROM_DISP16B, NIB1, NIB2, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW16,  DST, E}}, {{PREFIX, FROM_DISP16W, NIB1, NIB2, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL16,  DST, E}}, {{PREFIX, FROM_DISP16L, NIB1, NIB2, DISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB32,  DST, E}}, {{PREFIX, FROM_DISP32B, NIB1, NIB2, DISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW32,  DST, E}}, {{PREFIX, FROM_DISP32W, NIB1, NIB2, DISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL32,  DST, E}}, {{PREFIX, FROM_DISP32L, NIB1, NIB2, DISP32LIST, E}}}

/* Expansion from immediate source to "standard" destinations.  */
#define EXPAND2_STD_IMM(CODE, WEIGHT, NAME, SRC, PREFIX, OPCODE, IGN, IMMLIST) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}}, {{PREFIX, TO_POSTINC, OPCODE, IGN, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}}, {{PREFIX, TO_POSTDEC, OPCODE, IGN, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}}, {{PREFIX, TO_PREINC,  OPCODE, IGN, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}}, {{PREFIX, TO_PREDEC,  OPCODE, IGN, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}}, {{PREFIX, TO_DISP2,   OPCODE, IGN, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}}, {{PREFIX, TO_DISP16,  OPCODE, IGN, DSTDISP16LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}}, {{PREFIX, TO_DISP32,  OPCODE, IGN, DSTDISP32LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}}, {{PREFIX, TO_DISP16B, OPCODE, IGN, DSTDISP16LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}}, {{PREFIX, TO_DISP16W, OPCODE, IGN, DSTDISP16LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}}, {{PREFIX, TO_DISP16L, OPCODE, IGN, DSTDISP16LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}}, {{PREFIX, TO_DISP32B, OPCODE, IGN, DSTDISP32LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}}, {{PREFIX, TO_DISP32W, OPCODE, IGN, DSTDISP32LIST, IMMLIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}}, {{PREFIX, TO_DISP32L, OPCODE, IGN, DSTDISP32LIST, IMMLIST, E}}}

/* Expansion from abs/disp source to "standard" destinations.  */
#define EXPAND2_STD_ABSDISP(CODE, WEIGHT, NAME, SRC, PREFIX, DSTLIST, NIB1, NIB2) \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}}, {{PREFIX, DSTLIST, TO_POSTINC, NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}}, {{PREFIX, DSTLIST, TO_POSTDEC, NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}}, {{PREFIX, DSTLIST, TO_PREINC,  NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}}, {{PREFIX, DSTLIST, TO_PREDEC,  NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}}, {{PREFIX, DSTLIST, TO_DISP2,   NIB1, NIB2, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}}, {{PREFIX, DSTLIST, TO_DISP16,  NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}}, {{PREFIX, DSTLIST, TO_DISP32,  NIB1, NIB2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}}, {{PREFIX, DSTLIST, TO_DISP16B, NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}}, {{PREFIX, DSTLIST, TO_DISP16W, NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}}, {{PREFIX, DSTLIST, TO_DISP16L, NIB1, NIB2, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}}, {{PREFIX, DSTLIST, TO_DISP32B, NIB1, NIB2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}}, {{PREFIX, DSTLIST, TO_DISP32W, NIB1, NIB2, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}}, {{PREFIX, DSTLIST, TO_DISP32L, NIB1, NIB2, DSTDISP32LIST, E}}}

/* Expansion from ind source to "standard" destinations.  */
#define EXPAND2_STD_IND(CODE, WEIGHT, NAME, OPCODE, BIT) \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RDPOSTINC, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_POSTINC, OPCODE, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RDPOSTDEC, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_POSTDEC, OPCODE, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RDPREINC,  E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_PREINC,  OPCODE, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RDPREDEC,  E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_PREDEC,  OPCODE, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, DISP2DST,  E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP2,   OPCODE, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, DISP16DST, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP16,  OPCODE, IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, DISP32DST, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP32,  OPCODE, IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, INDEXB16D, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP16B, OPCODE, IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, INDEXW16D, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP16W, OPCODE, IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, INDEXL16D, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP16L, OPCODE, IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, INDEXB32D, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP32B, OPCODE, IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, INDEXW32D, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP32W, OPCODE, IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, INDEXL32D, E}}, {{0x7, 0xc, BIT | RSIND, 0x5, TO_DISP32L, OPCODE, IGNORE, DSTDISP32LIST, E}}}

/* Expansion macros for three word (plus data) instructions.  */

#define EXPAND3_STD_SRC(CODE, WEIGHT, NAME, SRC, PREFIX, INFIX, OPCODE)  \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}}, {{PREFIX, INFIX,  8, RDPOSTINC, OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}}, {{PREFIX, INFIX, 10, RDPOSTDEC, OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}}, {{PREFIX, INFIX,  9, RDPREINC,  OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}}, {{PREFIX, INFIX, 11, RDPREDEC,  OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}}, {{PREFIX, INFIX, B30 | B20 | DISP2DST, DSTDISPREG, OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}}, {{PREFIX, INFIX, 12, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}}, {{PREFIX, INFIX, 12, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}}, {{PREFIX, INFIX, 13, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}}, {{PREFIX, INFIX, 14, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}}, {{PREFIX, INFIX, 15, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}}, {{PREFIX, INFIX, 13, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}}, {{PREFIX, INFIX, 14, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}}, {{PREFIX, INFIX, 15, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}

#define EXPAND3_L_SRC(CODE, WEIGHT, NAME, SRC, PREFIX, INFIX, OPCODE)  \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDIND,     E}}, {{PREFIX, INFIX,  0, RDIND,     OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTINC, E}}, {{PREFIX, INFIX,  8, RDPOSTINC, OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPOSTDEC, E}}, {{PREFIX, INFIX, 10, RDPOSTDEC, OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREINC,  E}}, {{PREFIX, INFIX,  9, RDPREINC,  OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, RDPREDEC,  E}}, {{PREFIX, INFIX, 11, RDPREDEC,  OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP2DST,  E}}, {{PREFIX, INFIX, B30 | B20 | DISP2DST, DSTDISPREG, OPCODE, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP16DST, E}}, {{PREFIX, INFIX, 12, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, DISP32DST, E}}, {{PREFIX, INFIX, 12, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB16D, E}}, {{PREFIX, INFIX, 13, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW16D, E}}, {{PREFIX, INFIX, 14, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL16D, E}}, {{PREFIX, INFIX, 15, B30 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXB32D, E}}, {{PREFIX, INFIX, 13, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXW32D, E}}, {{PREFIX, INFIX, 14, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, INDEXL32D, E}}, {{PREFIX, INFIX, 15, B31 | DSTDISPREG, OPCODE, B30 | IGNORE, DSTDISP32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS16DST,  E}}, {{PREFIX, INFIX,  4, B30 | IGNORE,     OPCODE, B30 | IGNORE, DSTABS16LIST,  E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{SRC, ABS32DST,  E}}, {{PREFIX, INFIX,  4, B31 | IGNORE,     OPCODE, B30 | IGNORE, DSTABS32LIST,  E}}}


#define EXPAND_STD_MATRIX_L(CODE, NAME, OPCODE) \
  EXPAND3_L_SRC (CODE, 6, NAME, RSIND,     PREFIX_0104,   TFROM_IND,     OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, RSPOSTINC, PREFIX_0104,   TFROM_POSTINC, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, RSPOSTDEC, PREFIX_0106,   TFROM_POSTDEC, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, RSPREINC,  PREFIX_0105,   TFROM_PREINC,  OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, RSPREDEC,  PREFIX_0107,   TFROM_PREDEC,  OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, DISP2SRC,  PREFIX_010_D2, TFROM_DISP2,   OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, DISP16SRC, PREFIX_0104,   TFROM_DISP16,  OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, DISP32SRC, PREFIX_78R4,   TFROM_DISP32,  OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, INDEXB16,  PREFIX_0105,   TFROM_DISP16B, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, INDEXW16,  PREFIX_0106,   TFROM_DISP16W, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, INDEXL16,  PREFIX_0107,   TFROM_DISP16L, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, INDEXB32,  PREFIX_78R5,   TFROM_DISP32B, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, INDEXW32,  PREFIX_78R6,   TFROM_DISP32W, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, INDEXL32,  PREFIX_78R7,   TFROM_DISP32L, OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, ABS16SRC,  PREFIX_0104,   TFROM_ABS16,   OPCODE), \
  EXPAND3_L_SRC (CODE, 6, NAME, ABS32SRC,  PREFIX_0104,   TFROM_ABS32,   OPCODE)


#define EXPAND_STD_MATRIX_W(CODE, NAME, OPCODE) \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPOSTINC, PREFIX_0154,   TFROM_POSTINC, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPOSTDEC, PREFIX_0156,   TFROM_POSTDEC, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPREINC,  PREFIX_0155,   TFROM_PREINC,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPREDEC,  PREFIX_0157,   TFROM_PREDEC,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, DISP2SRC,  PREFIX_015_D2, TFROM_DISP2,   OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, DISP16SRC, PREFIX_0154,   TFROM_DISP16,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, DISP32SRC, PREFIX_78R4W,  TFROM_DISP32,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXB16,  PREFIX_0155,   TFROM_DISP16B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXW16,  PREFIX_0156,   TFROM_DISP16W, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXL16,  PREFIX_0157,   TFROM_DISP16L, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXB32,  PREFIX_78R5W,  TFROM_DISP32B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXW32,  PREFIX_78R6W,  TFROM_DISP32W, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXL32,  PREFIX_78R7W,  TFROM_DISP32L, OPCODE)

#define EXPAND_STD_MATRIX_B(CODE, NAME, OPCODE) \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPOSTINC, PREFIX_0174,    TFROM_POSTINC_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPOSTDEC, PREFIX_0176,    TFROM_POSTDEC_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPREINC,  PREFIX_0175,    TFROM_PREINC_B,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, RSPREDEC,  PREFIX_0177,    TFROM_PREDEC_B,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, DISP2SRC,  PREFIX_017_D2S, TFROM_DISP2_B,   OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, DISP16SRC, PREFIX_0174,    TFROM_DISP16_B,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, DISP32SRC, PREFIX_78R4W,   TFROM_DISP32_B,  OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXB16,  PREFIX_0175,    TFROM_DISP16B_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXW16,  PREFIX_0176,    TFROM_DISP16W_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXL16,  PREFIX_0177,    TFROM_DISP16L_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXB32,  PREFIX_78R5W,   TFROM_DISP32B_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXW32,  PREFIX_78R6W,   TFROM_DISP32W_B, OPCODE), \
  EXPAND3_L_SRC (CODE, 4, NAME, INDEXL32,  PREFIX_78R7W,   TFROM_DISP32L_B, OPCODE)


/* Use the expansion macros to fill out the opcode table.  */

#define EXPAND_FROM_REG8(CODE, NAME, OP1, OP2, OP3) \
  {CODE, AV_H8SX, 0, NAME, {{RS8, RDIND,     E}}, {{0x7, 0xd, B30 | RDIND, IGNORE,             OP1, OP2, RS8, IGNORE, E}}}, \
  EXPAND2_STD_SRC (CODE, 2, NAME, RS8, PREFIX_0179, OP3, RS8), \
  {CODE, AV_H8SX, 0, NAME, {{RS8, ABS8DST,   E}}, {{0x7, 0xf, DSTABS8LIST,                     OP1, OP2, RS8, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RS8, ABS16DST,  E}}, {{0x6, 0xa, 0x1, B31 | IGNORE, DSTABS16LIST, OP1, OP2, RS8, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RS8, ABS32DST,  E}}, {{0x6, 0xa, 0x3, B31 | IGNORE, DSTABS32LIST, OP1, OP2, RS8, IGNORE, E}}}

#define EXPAND_TO_REG8(CODE, NAME, OP1, OP2, OP3) \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RD8,     E}}, {{0x7, 0xc, B30 | RSIND, IGNORE,          OP1, OP2, IGNORE, RD8, E}}}, \
  EXPAND2_STD_DST (CODE, 2, NAME, RD8, PREFIX_017A, OP3, RD8), \
  {CODE, AV_H8SX, 0, NAME, {{ABS8SRC,  RD8,  E}}, {{0x7, 0xe, ABS8LIST,                     OP1, OP2, IGNORE, RD8, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, RD8,  E}}, {{0x6, 0xa, 0x1, B30 | IGNORE, ABS16LIST, OP1, OP2, IGNORE, RD8, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, RD8,  E}}, {{0x6, 0xa, 0x3, B30 | IGNORE, ABS32LIST, OP1, OP2, IGNORE, RD8, E}}}

#define EXPAND_FROM_IND8(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RDIND,    E}}, {{0x7, 0xc, B30 | RSIND, 0x5, TO_IND,   OPCODE, IGNORE, E}}}, \
  EXPAND2_STD_IND (CODE, 2, NAME, OPCODE, B30), \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, ABS16DST, E}}, {{0x7, 0xc, B30 | RSIND, 0x5, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, ABS32DST, E}}, {{0x7, 0xc, B30 | RSIND, 0x5, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, E}}}

#define EXPAND_FROM_ABS16_B(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, RDIND,    E}}, {{PREFIX_6A15, ABS16LIST, TO_IND,   OPCODE, IGNORE, E}}}, \
  EXPAND2_STD_ABSDISP (CODE, 2, NAME, ABS16SRC, PREFIX_6A15, ABS16LIST, OPCODE, IGNORE), \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, ABS16DST, E}}, {{PREFIX_6A15, ABS16LIST, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, ABS32DST, E}}, {{PREFIX_6A15, ABS16LIST, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, E}}}

#define EXPAND_FROM_ABS32_B(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, RDIND,    E}}, {{PREFIX_6A35, ABS32LIST, TO_IND,   OPCODE, IGNORE, E}}}, \
  EXPAND2_STD_ABSDISP (CODE, 2, NAME, ABS32SRC, PREFIX_6A35, ABS32LIST, OPCODE, IGNORE), \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, ABS16DST, E}}, {{PREFIX_6A35, ABS32LIST, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, ABS32DST, E}}, {{PREFIX_6A35, ABS32LIST, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, E}}}

#define EXPAND_FROM_IMM16_W(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{IMM16, RDIND,    E}}, {{PREFIX_015E, TO_IND, OPCODE, IGNORE, IMM16LIST, E}}}, \
  EXPAND2_STD_IMM (CODE, 2, NAME, IMM16, PREFIX_015E, OPCODE, IGNORE, IMM16LIST), \
  {CODE, AV_H8SX, 0, NAME, {{IMM16, ABS16DST, E}}, {{PREFIX_015E, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, IMM16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM16, ABS32DST, E}}, {{PREFIX_015E, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, IMM16LIST, E}}}

#define EXPAND_FROM_REG16(CODE, NAME, OP1, OP2, OP3) \
  {CODE, AV_H8,   2, NAME, {{RS16, RDIND,    E}}, {{0x7, 0xd, B31 | RDIND, IGNORE,             OP1, OP2, RS16, IGNORE, E}}}, \
  EXPAND2_STD_SRC (CODE, 2, NAME, RS16, PREFIX_0159, OP3, RS16), \
  {CODE, AV_H8SX, 0, NAME, {{RS16, ABS16DST, E}}, {{0x6, 0xb, 0x1, B31 | IGNORE, DSTABS16LIST, OP1, OP2, RS16, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RS16, ABS32DST, E}}, {{0x6, 0xb, 0x3, B31 | IGNORE, DSTABS32LIST, OP1, OP2, RS16, IGNORE, E}}}

#define EXPAND_TO_REG16(CODE, NAME, OP1, OP2, OP3) \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RD16,    E}}, {{0x7, 0xc, B31 | RSIND, IGNORE,          OP1, OP2, IGNORE, RD16, E}}}, \
  EXPAND2_STD_DST (CODE, 2, NAME, RD16, PREFIX_015A, OP3, RD16), \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, RD16, E}}, {{0x6, 0xb, 0x1, B30 | IGNORE, ABS16LIST, OP1, OP2, IGNORE, RD16, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, RD16, E}}, {{0x6, 0xb, 0x3, B30 | IGNORE, ABS32LIST, OP1, OP2, IGNORE, RD16, E}}}

#define EXPAND_FROM_IND16(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RDIND,    E}}, {{0x7, 0xc, B31 | RSIND, 0x5, TO_IND, OPCODE, IGNORE, E}}}, \
  EXPAND2_STD_IND (CODE, 2, NAME, OPCODE, B31), \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, ABS16DST, E}}, {{0x7, 0xc, B31 | RSIND, 0x5, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, ABS32DST, E}}, {{0x7, 0xc, B31 | RSIND, 0x5, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, E}}}

#define EXPAND_FROM_ABS16_W(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, RDIND,    E}}, {{PREFIX_6B15, ABS16LIST, TO_IND,   OPCODE, IGNORE, E}}}, \
  EXPAND2_STD_ABSDISP (CODE, 2, NAME, ABS16SRC, PREFIX_6B15, ABS16LIST, OPCODE, IGNORE), \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, ABS16DST, E}}, {{PREFIX_6B15, ABS16LIST, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, ABS32DST, E}}, {{PREFIX_6B15, ABS16LIST, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, E}}}

#define EXPAND_FROM_ABS32_W(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, RDIND,    E}}, {{PREFIX_6B35, ABS32LIST, TO_IND,   OPCODE, IGNORE, E}}}, \
  EXPAND2_STD_ABSDISP (CODE, 2, NAME, ABS32SRC, PREFIX_6B35, ABS32LIST, OPCODE, IGNORE), \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, ABS16DST, E}}, {{PREFIX_6B35, ABS32LIST, TO_ABS16, OPCODE, IGNORE, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, ABS32DST, E}}, {{PREFIX_6B35, ABS32LIST, TO_ABS32, OPCODE, IGNORE, DSTABS32LIST, E}}}

#define EXPAND_FROM_IMM16_L(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{IMM16U_NS, RDIND,    E}}, {{PREFIX_010E, TO_IND, OPCODE, B30 | IGNORE, IMM16ULIST, E}}}, \
  EXPAND2_STD_IMM (CODE, 2, NAME, IMM16U_NS, PREFIX_010E, OPCODE, B30 | IGNORE, IMM16ULIST), \
  {CODE, AV_H8SX, 0, NAME, {{IMM16U_NS, ABS16DST, E}}, {{PREFIX_010E, TO_ABS16, OPCODE, B30 | IGNORE, DSTABS16LIST, IMM16ULIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM16U_NS, ABS32DST, E}}, {{PREFIX_010E, TO_ABS32, OPCODE, B30 | IGNORE, DSTABS32LIST, IMM16ULIST, E}}}

#define EXPAND_FROM_IMM32_L(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{IMM32, RDIND,    E}}, {{PREFIX_010E, TO_IND, OPCODE, B31 | IGNORE, IMM32LIST, E}}}, \
  EXPAND2_STD_IMM (CODE, 2, NAME, IMM32, PREFIX_010E, OPCODE, B31 | IGNORE, IMM32LIST), \
  {CODE, AV_H8SX, 0, NAME, {{IMM32, ABS16DST, E}}, {{PREFIX_010E, TO_ABS16, OPCODE, B31 | IGNORE, DSTABS16LIST, IMM32LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM32, ABS32DST, E}}, {{PREFIX_010E, TO_ABS32, OPCODE, B31 | IGNORE, DSTABS32LIST, IMM32LIST, E}}}

#define EXPAND_FROM_REG32(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{RS32, RDIND,    E}}, {{PREFIX_0109, TO_IND,   OPCODE, B30 | RS32, E}}}, \
  EXPAND2_STD_SRC (CODE, 2, NAME, RS32, PREFIX_0109, OPCODE, B30 | RS32), \
  {CODE, AV_H8SX, 0, NAME, {{RS32, ABS16DST, E}}, {{PREFIX_0109, TO_ABS16, OPCODE, B30 | RS32, DSTABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RS32, ABS32DST, E}}, {{PREFIX_0109, TO_ABS32, OPCODE, B30 | RS32, DSTABS32LIST, E}}}

#define EXPAND_TO_REG32(CODE, NAME, OPCODE) \
  {CODE, AV_H8SX, 0, NAME, {{RSIND, RD32,    E}}, {{PREFIX_010A, FROM_IND,   OPCODE, B30 | RD32, E}}}, \
  EXPAND2_STD_DST (CODE, 2, NAME, RD32, PREFIX_010A, OPCODE, B30 | RD32), \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC, RD32, E}}, {{PREFIX_010A, FROM_ABS16, OPCODE, B30 | RD32, ABS16LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC, RD32, E}}, {{PREFIX_010A, FROM_ABS32, OPCODE, B30 | RD32, ABS32LIST, E}}}


#define EXPAND_TWOOP_B(CODE, NAME, OP1, OP2, OP3, OP4, BIT) \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, RDIND,     E}}, {{0x7, 0xd,                 B30 | RDIND,            IGNORE,                      OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, RDPOSTINC, E}}, {{PREFIX_0174,    0x6, 0xc, B30 | RDPOSTINC,  B31 | B20 | IGNORE,                OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, RDPOSTDEC, E}}, {{PREFIX_0176,    0x6, 0xc, B30 | RDPOSTDEC,  B31 | B20 | IGNORE,                OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, RDPREINC,  E}}, {{PREFIX_0175,    0x6, 0xc, B30 | RDPREINC,   B31 | B20 | IGNORE,                OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, RDPREDEC,  E}}, {{PREFIX_0177,    0x6, 0xc, B30 | RDPREDEC,   B31 | B20 | IGNORE,                OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, DISP2DST,  E}}, {{PREFIX_017_D2D, 0x6, 0x8, B30 | DSTDISPREG, B31 | B20 | IGNORE,                OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, DISP16DST, E}}, {{PREFIX_0174,    0x6, 0xe, B30 | DSTDISPREG, B31 | B20 | IGNORE, DSTDISP16LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, DISP32DST, E}}, {{PREFIX_78R4WD,  0x6, 0xa, 2,                B31 | B20 | IGNORE, DSTDISP32LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, INDEXB16D, E}}, {{PREFIX_0175,    0x6, 0xe, B30 | DSTDISPREG, B31 | B20 | IGNORE, DSTDISP16LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, INDEXW16D, E}}, {{PREFIX_0176,    0x6, 0xe, B30 | DSTDISPREG, B31 | B20 | IGNORE, DSTDISP16LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, INDEXL16D, E}}, {{PREFIX_0177,    0x6, 0xe, B30 | DSTDISPREG, B31 | B20 | IGNORE, DSTDISP16LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, INDEXB32D, E}}, {{PREFIX_78R5WD,  0x6, 0xa, 2,                B31 | B20 | IGNORE, DSTDISP32LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, INDEXW32D, E}}, {{PREFIX_78R6WD,  0x6, 0xa, 2,                B31 | B20 | IGNORE, DSTDISP32LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, INDEXL32D, E}}, {{PREFIX_78R7WD,  0x6, 0xa, 2,                B31 | B20 | IGNORE, DSTDISP32LIST, OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, ABS8DST,   E}}, {{0x7, 0xf,                                                       DSTABS8LIST,   OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, ABS16DST,  E}}, {{0x6, 0xa, 0x1,                              B31 | B20 | IGNORE, DSTABS16LIST,  OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{IMM8, ABS32DST,  E}}, {{0x6, 0xa, 0x3,                              B31 | B20 | IGNORE, DSTABS32LIST,  OP1, BIT | IGNORE, IMM8LIST, E}}}, \
  {CODE, AV_H8,   2, NAME, {{RS8,  RD8,       E}}, {{OP2, OP3, RS8, RD8, E}}}, \
  EXPAND_FROM_REG8      (CODE, NAME, OP2, OP3, OP4), \
  EXPAND_TO_REG8        (CODE, NAME, OP2, OP3, OP4), \
  EXPAND_FROM_IND8      (CODE, NAME, OP4), \
  EXPAND_STD_MATRIX_B   (CODE, NAME, OP4), \
  EXPAND_FROM_ABS16_B   (CODE, NAME, OP4), \
  EXPAND_FROM_ABS32_B   (CODE, NAME, OP4)

#define EXPAND_TWOOP_W(CODE, NAME, OP1, OP2, OP3) \
  {CODE, AV_H8H,  6, NAME, {{IMM16, RD16, E}}, {{0x7, 0x9, OP3, RD16, IMM16LIST, E}}}, \
  EXPAND_FROM_IMM16_W   (CODE, NAME, OP3), \
  EXPAND_FROM_REG16     (CODE, NAME, OP1, OP2, OP3), \
  EXPAND_TO_REG16       (CODE, NAME, OP1, OP2, OP3), \
  EXPAND_FROM_IND16     (CODE, NAME, OP3), \
  EXPAND_STD_MATRIX_W   (CODE, NAME, OP3), \
  EXPAND_FROM_ABS16_W   (CODE, NAME, OP3), \
  EXPAND_FROM_ABS32_W   (CODE, NAME, OP3)

#define EXPAND_TWOOP_L(CODE, NAME, OP1)  \
  {CODE, AV_H8SX, 0, NAME, {{IMM16U_NS, RD32, E}}, {{0x7, 0xa, OP1, B31 | RD32, IMM16ULIST, E}}}, \
  {CODE, AV_H8H,  6, NAME, {{IMM32,     RD32, E}}, {{0x7, 0xa, OP1, B30 | RD32, IMM32LIST,  E}}}, \
  EXPAND_FROM_IMM16_L   (CODE, NAME, OP1), \
  EXPAND_FROM_IMM32_L   (CODE, NAME, OP1), \
  EXPAND_FROM_REG32     (CODE, NAME, OP1), \
  EXPAND_TO_REG32       (CODE, NAME, OP1), \
  EXPAND_STD_MATRIX_L   (CODE, NAME, OP1)


/* Old expanders:  */

#define BITOP(code, imm, name, op00, op01, op10,  op11, op20, op21, op30, op4) \
  {code, AV_H8,  2, name, {{imm, RD8,      E}}, {{op00, op01, imm,  RD8,   E}}}, \
  {code, AV_H8,  6, name, {{imm, RDIND,    E}}, {{op10, op11, B30 | RDIND, 0,  op00, op01, imm, 0, E}}}, \
  {code, AV_H8,  6, name, {{imm, ABS8DST,  E}}, {{op20, op21, DSTABS8LIST,     op00, op01, imm, 0, E}}}, \
  {code, AV_H8S, 6, name, {{imm, ABS16DST, E}}, {{0x6,  0xa,  0x1,  op30, DST | MEMRELAX | ABS16LIST , op00, op01, imm, op4, E}}}, \
  {code, AV_H8S, 6, name, {{imm, ABS32DST, E}}, {{0x6,  0xa,  0x3,  op30, DST | MEMRELAX | ABS32LIST , op00, op01, imm, op4, E}}}

#define BITOP_B(code, imm, name, op00, op01, op10, op11, op20, op21, op30, op4) \
  {code, AV_H8SX, 0, name, {{imm, RDIND,    E}}, {{op10, op11, B30 | RDIND, 0,  op00, op01, imm, op4, E}}}, \
  {code, AV_H8SX, 0, name, {{imm, ABS8DST,  E}}, {{op20, op21, DSTABS8LIST,     op00, op01, imm, op4, E}}}, \
  {code, AV_H8SX, 0, name, {{imm, ABS16DST, E}}, {{0x6,  0xa,  0x1,  op30, DST | ABS16LIST, op00, op01, imm, op4, E}}}, \
  {code, AV_H8SX, 0, name, {{imm, ABS32DST, E}}, {{0x6,  0xa,  0x3,  op30, DST | ABS32LIST, op00, op01, imm, op4, E}}}

#define EBITOP(code, imm, name, op00, op01, op10, op11, op20, op21, op30, op4) \
  BITOP(code, imm, name, op00+1, op01, op10, op11, op20, op21, op30, op4), \
  BITOP(code, RS8, name, op00,   op01, op10, op11, op20, op21, op30, op4)

#define EBITOP_B(code, imm, name, op00, op01, op10, op11, op20, op21, op30, op4) \
  BITOP_B(code, imm, name, op00+1, op01, op10, op11, op20, op21, op30, op4), \
  BITOP_B(code, RS8, name, op00,   op01, op10, op11, op20, op21, op30, op4)

#define WTWOP(code, name, op1, op2) \
  {code, AV_H8, 2, name, {{RS16, RD16, E}}, {{op1, op2, RS16, RD16, E}}}

#define BRANCH(code, name, op) \
  {code, AV_H8H, 6, name, {{PCREL16, E}}, {{0x5, 0x8, op, 0x0, PCREL16, DATA3 | B00, E}}}, \
  {code, AV_H8,  4, name, {{PCREL8,  E}}, {{0x4, op,           PCREL8,  DATA  | B00, E}}}


#define UNOP(code, name, op1, op2) \
  {code, AV_H8, 2, name, {{OR8, E}}, {{op1, op2, 0, OR8, E}}}

#define EXPAND_UNOP_STD_B(CODE, NAME, PREFIX, OP1, OP2, OP3) \
  {CODE, AV_H8,   2, NAME, {{OR8,       E}}, {{                                                                  OP1, OP2, OP3, OR8,    E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND,     E}}, {{           7, 13, B30 | RSIND,             IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTINC, E}}, {{PREFIX, 4, 6, 12, B30 | RSPOSTINC,   B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTDEC, E}}, {{PREFIX, 6, 6, 12, B30 | RSPOSTDEC,   B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREINC,  E}}, {{PREFIX, 5, 6, 12, B30 | RSPREINC,    B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREDEC,  E}}, {{PREFIX, 7, 6, 12, B30 | RSPREDEC,    B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP2SRC,  E}}, {{PREFIX, B30 | B21 | DISP2SRC, 6,  8, B30 | DISPREG, B31 | IGNORE, OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP16SRC, E}}, {{PREFIX, 4, 6, 14, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP32SRC, E}}, {{7, 8, B30 | DISPREG, 4,    6, 10, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB16,  E}}, {{PREFIX, 5, 6, 14, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW16,  E}}, {{PREFIX, 6, 6, 14, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL16,  E}}, {{PREFIX, 7, 6, 14, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB32,  E}}, {{7, 8, B30 | DISPREG, 5,    6, 10, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW32,  E}}, {{7, 8, B30 | DISPREG, 6,    6, 10, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL32,  E}}, {{7, 8, B30 | DISPREG, 7,    6, 10, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS8SRC,   E}}, {{                           7, 15,                   ABS8LIST,     OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC,  E}}, {{                           6, 10, 1, B31 | IGNORE,  ABS16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC,  E}}, {{                           6, 10, 3, B31 | IGNORE,  ABS32LIST,    OP1, OP2, OP3, IGNORE, E}}}

#define EXPAND_UNOP_STD_W(CODE, NAME, PREFIX, OP1, OP2, OP3) \
  {CODE, AV_H8H,  2, NAME, {{OR16,      E}}, {{                                                                  OP1, OP2, OP3, OR16,   E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND,     E}}, {{           7, 13, B31 | RSIND,             IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTINC, E}}, {{PREFIX, 4, 6, 13, B30 | RSPOSTINC,   B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTDEC, E}}, {{PREFIX, 6, 6, 13, B30 | RSPOSTDEC,   B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREINC,  E}}, {{PREFIX, 5, 6, 13, B30 | RSPREINC,    B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREDEC,  E}}, {{PREFIX, 7, 6, 13, B30 | RSPREDEC,    B31 | IGNORE,                OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP2SRC,  E}}, {{PREFIX, B30 | B21 | DISP2SRC, 6,  9, B30 | DISPREG, B31 | IGNORE, OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP16SRC, E}}, {{PREFIX, 4, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP32SRC, E}}, {{7, 8, B30 | DISPREG, 4,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB16,  E}}, {{PREFIX, 5, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW16,  E}}, {{PREFIX, 6, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL16,  E}}, {{PREFIX, 7, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB32,  E}}, {{7, 8, B30 | DISPREG, 5,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW32,  E}}, {{7, 8, B30 | DISPREG, 6,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL32,  E}}, {{7, 8, B30 | DISPREG, 7,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC,  E}}, {{                           6, 11, 1, B31 | IGNORE,  ABS16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC,  E}}, {{                           6, 11, 3, B31 | IGNORE,  ABS32LIST,    OP1, OP2, OP3, IGNORE, E}}}

#define EXPAND_UNOP_STD_L(CODE, NAME, PREFIX, OP1, OP2, OP3) \
  {CODE, AV_H8H,  2, NAME, {{OR32,      E}}, {{                                                                  OP1, OP2, OP3, B30 | OR32,   E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSIND,     E}}, {{PREFIX, 4, 6,  9, B30 | RSIND,       B31 | IGNORE,                OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTINC, E}}, {{PREFIX, 4, 6, 13, B30 | RSPOSTINC,   B31 | IGNORE,                OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPOSTDEC, E}}, {{PREFIX, 6, 6, 13, B30 | RSPOSTDEC,   B31 | IGNORE,                OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREINC,  E}}, {{PREFIX, 5, 6, 13, B30 | RSPREINC,    B31 | IGNORE,                OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{RSPREDEC,  E}}, {{PREFIX, 7, 6, 13, B30 | RSPREDEC,    B31 | IGNORE,                OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP2SRC,  E}}, {{PREFIX, B30 | B21 | DISP2SRC, 6,  9, B30 | DISPREG, B31 | IGNORE, OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP16SRC, E}}, {{PREFIX, 4, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{DISP32SRC, E}}, {{7, 8, B31 | DISPREG, 4,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB16,  E}}, {{PREFIX, 5, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW16,  E}}, {{PREFIX, 6, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL16,  E}}, {{PREFIX, 7, 6, 15, B30 | DISPREG,     B31 | IGNORE,  DISP16LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXB32,  E}}, {{7, 8, B31 | DISPREG, 5,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXW32,  E}}, {{7, 8, B31 | DISPREG, 6,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{INDEXL32,  E}}, {{7, 8, B31 | DISPREG, 7,    6, 11, 2, B31 | IGNORE,  DISP32LIST,   OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS16SRC,  E}}, {{PREFIX, 4,                 6, 11, 0, B31 | IGNORE,  ABS16LIST,    OP1, OP2, OP3, B30 | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{ABS32SRC,  E}}, {{PREFIX, 4,                 6, 11, 2, B31 | IGNORE,  ABS32LIST,    OP1, OP2, OP3, B30 | IGNORE, E}}}

#define EXPAND_UNOP_EXTENDED_B(CODE, NAME, CONST, PREFIX, OP1, OP2, OP3) \
  {CODE, AV_H8,   2, NAME, {{CONST, RD8,       E}}, {{                                                                     OP1, OP2, OP3, RD8,    E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDIND,     E}}, {{           7, 13, B30 | RDIND,             IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPOSTINC, E}}, {{PREFIX, 4, 6, 12, B30 | RDPOSTINC,   B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPOSTDEC, E}}, {{PREFIX, 6, 6, 12, B30 | RDPOSTDEC,   B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPREINC,  E}}, {{PREFIX, 5, 6, 12, B30 | RDPREINC,    B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPREDEC,  E}}, {{PREFIX, 7, 6, 12, B30 | RDPREDEC,    B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP2DST,  E}}, {{PREFIX, B30 | B21 | DISP2DST, 6,  8, B30 | DSTDISPREG, B31 | IGNORE, OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP16DST, E}}, {{PREFIX, 4, 6, 14, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP32DST, E}}, {{7, 8, B30 | DSTDISPREG, 4, 6, 10, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXB16D, E}}, {{PREFIX, 5, 6, 14, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXW16D, E}}, {{PREFIX, 6, 6, 14, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXL16D, E}}, {{PREFIX, 7, 6, 14, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXB32D, E}}, {{7, 8, B30 | DSTDISPREG, 5, 6, 10, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXW32D, E}}, {{7, 8, B30 | DSTDISPREG, 6, 6, 10, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXL32D, E}}, {{7, 8, B30 | DSTDISPREG, 7, 6, 10, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS8DST,   E}}, {{                           7, 15,                  DSTABS8LIST,      OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS16DST,  E}}, {{                           6, 10, 1, B31 | IGNORE, DSTABS16LIST,     OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS32DST,  E}}, {{                           6, 10, 3, B31 | IGNORE, DSTABS32LIST,     OP1, OP2, OP3, IGNORE, E}}}

#define EXPAND_UNOP_EXTENDED_W(CODE, NAME, CONST, PREFIX, OP1, OP2, OP3) \
  {CODE, AV_H8,   2, NAME, {{CONST, RD16,      E}}, {{                                                                     OP1, OP2, OP3, RD16,   E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDIND,     E}}, {{           7, 13, B31 | RDIND,             IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPOSTINC, E}}, {{PREFIX, 4, 6, 13, B30 | RDPOSTINC,   B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPOSTDEC, E}}, {{PREFIX, 6, 6, 13, B30 | RDPOSTDEC,   B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPREINC,  E}}, {{PREFIX, 5, 6, 13, B30 | RDPREINC,    B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPREDEC,  E}}, {{PREFIX, 7, 6, 13, B30 | RDPREDEC,    B31 | IGNORE,                   OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP2DST,  E}}, {{PREFIX, B30 | B21 | DISP2DST, 6,  9, B30 | DSTDISPREG, B31 | IGNORE, OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP16DST, E}}, {{PREFIX, 4, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP32DST, E}}, {{7, 8, B30 | DSTDISPREG, 4, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXB16D, E}}, {{PREFIX, 5, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXW16D, E}}, {{PREFIX, 6, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXL16D, E}}, {{PREFIX, 7, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXB32D, E}}, {{7, 8, B30 | DSTDISPREG, 5, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXW32D, E}}, {{7, 8, B30 | DSTDISPREG, 6, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXL32D, E}}, {{7, 8, B30 | DSTDISPREG, 7, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS16DST,  E}}, {{                           6, 11, 1, B31 | IGNORE, DSTABS16LIST,     OP1, OP2, OP3, IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS32DST,  E}}, {{                           6, 11, 3, B31 | IGNORE, DSTABS32LIST,     OP1, OP2, OP3, IGNORE, E}}}

#define EXPAND_UNOP_EXTENDED_L(CODE, NAME, CONST, PREFIX, OP1, OP2, OP3, BIT) \
  {CODE, AV_H8,   2, NAME, {{CONST, RD32,      E}}, {{                                                                     OP1, OP2, OP3, BIT | RD32,   E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDIND,     E}}, {{PREFIX, 4, 6,  9, B30 | RDIND,       B31 | IGNORE,                   OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPOSTINC, E}}, {{PREFIX, 4, 6, 13, B30 | RDPOSTINC,   B31 | IGNORE,                   OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPOSTDEC, E}}, {{PREFIX, 6, 6, 13, B30 | RDPOSTDEC,   B31 | IGNORE,                   OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPREINC,  E}}, {{PREFIX, 5, 6, 13, B30 | RDPREINC,    B31 | IGNORE,                   OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, RDPREDEC,  E}}, {{PREFIX, 7, 6, 13, B30 | RDPREDEC,    B31 | IGNORE,                   OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP2DST,  E}}, {{PREFIX, B30 | B21 | DISP2DST, 6,  9, B30 | DSTDISPREG, B31 | IGNORE, OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP16DST, E}}, {{PREFIX, 4, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, DISP32DST, E}}, {{7, 8, B31 | DSTDISPREG, 4, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXB16D, E}}, {{PREFIX, 5, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXW16D, E}}, {{PREFIX, 6, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXL16D, E}}, {{PREFIX, 7, 6, 15, B30 | DSTDISPREG,  B31 | IGNORE, DSTDISP16LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXB32D, E}}, {{7, 8, B31 | DSTDISPREG, 5, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXW32D, E}}, {{7, 8, B31 | DSTDISPREG, 6, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, INDEXL32D, E}}, {{7, 8, B31 | DSTDISPREG, 7, 6, 11, 2, B31 | IGNORE, DSTDISP32LIST,    OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS16DST,  E}}, {{PREFIX, 4,                 6, 11, 0, B31 | IGNORE, DSTABS16LIST,     OP1, OP2, OP3, BIT | IGNORE, E}}}, \
  {CODE, AV_H8SX, 0, NAME, {{CONST, ABS32DST,  E}}, {{PREFIX, 4,                 6, 11, 2, B31 | IGNORE, DSTABS32LIST,     OP1, OP2, OP3, BIT | IGNORE, E}}}

#define PREFIXLDC 0x0, 0x1, 0x4, B30 | CCR_EXR | DST
#define PREFIXSTC 0x0, 0x1, 0x4, B30 | CCR_EXR | SRC

#define O(op, size)  (op * 4 + size)
#define OP_SIZE(HOW) (HOW % 4)
#define OP_KIND(HOW) (HOW / 4)

enum h8_asm_codes
{
  O_RECOMPILE =	 0,
  O_ADD,
  O_ADDX,
  O_AND,
  O_BAND,
  O_BRA,
  O_BRAB,
  O_BRAW,
  O_BRAL,
  O_BRAS,
  O_BRABC,
  O_BRABS,
  O_BSRBC,
  O_BSRBS,
  O_BRN,
  O_BHI,
  O_BLS,
  O_BCC,
  O_BCS,
  O_BNE,
  O_BVC,
  O_BVS,
  O_BPL,
  O_BMI,
  O_BGE,
  O_BLT,
  O_BGT,
  O_BLE,
  O_ANDC,
  O_BEQ,
  O_BCLR,
  O_BCLREQ,
  O_BCLRNE,
  O_BSETEQ,
  O_BSETNE,
  O_BFLD,
  O_BFST,
  O_BIAND,
  O_BILD,
  O_BIOR,
  O_BIXOR,
  O_BIST,
  O_BISTZ,
  O_BLD,
  O_BNOT,
  O_BOR,
  O_BSET,
  O_BSR,
  O_BXOR,
  O_CMP,
  O_DAA,
  O_DAS,
  O_DEC,
  O_DIVU,
  O_DIVS,
  O_DIVXU,
  O_DIVXS,
  O_INC,
  O_LDC,
  O_MOV,
  O_MOVAB,
  O_MOVAW,
  O_MOVAL,
  O_MOVMD,
  O_MOVSD,
  O_OR,
  O_ROTL,
  O_ROTR,
  O_ROTXL,
  O_ROTXR,
  O_BPT,
  O_SHAL,
  O_SHAR,
  O_SHLL,
  O_SHLR,
  O_SUB,
  O_SUBS,
  O_TRAPA,
  O_XOR,
  O_XORC,
  O_BST,
  O_BSTZ,
  O_BTST,
  O_EEPMOV,
  O_EXTS,
  O_EXTU,
  O_JMP,
  O_JSR,
  O_MULU,
  O_MULUU,
  O_MULS,
  O_MULSU,
  O_MULXU,
  O_MULXS,
  O_NOP,
  O_NOT,
  O_ORC,
  O_RTE,
  O_RTEL,
  O_STC,
  O_SUBX,
  O_NEG,
  O_RTS,
  O_RTSL,
  O_SLEEP,
  O_ILL,
  O_ADDS,
  O_SYSCALL,
  O_TAS,
  O_CLRMAC,
  O_LDMAC,
  O_MAC,
  O_LDM,
  O_STM,
  O_STMAC,
  O_LAST,
  /* Change made for System Call processing.  */
  O_SYS_CREAT,
  O_SYS_OPEN,
  O_SYS_READ,
  O_SYS_WRITE,
  O_SYS_LSEEK,
  O_SYS_CLOSE,
  O_SYS_STAT,
  O_SYS_FSTAT,
/* Space reserved for future file I/O system calls.  */
 O_SYS_CMDLINE
  /* End of System Call specific Changes.  */
};

enum h8_size
{
  SB =	 0,
  SW =	 1,
  SL =	 2,
  SN =	 3
};


/* FIXME: Lots of insns have "E, 0, 0, 0, 0" in the nibble code sequences.
   Methinks the zeroes aren't necessary.  Once confirmed, nuke 'em.  */

struct h8_opcode h8_opcodes[] = 
{
  {O (O_ADD, SB), AV_H8,   2, "add.b", {{IMM8,      RD8,      E}}, {{0x8, RD8, IMM8LIST, E}}}, 
  EXPAND_TWOOP_B (O (O_ADD, SB), "add.b", 0x8, 0x0, 0x8, 0x1, 0), 

  {O (O_ADD, SW), AV_H8,   6, "add.w", {{RS16,      RD16,     E}}, {{0x0, 0x9, RS16,         RD16,   E}}}, 
  {O (O_ADD, SW), AV_H8SX, 0, "add.w", {{IMM3NZ_NS, RD16,     E}}, {{0x0, 0xa, B30 | IMM3NZ, RD16,   E}}}, 
  {O (O_ADD, SW), AV_H8SX, 0, "add.w", {{IMM3NZ_NS, RDIND,    E}}, {{0x7, 0xd,      B31 | RDIND,  IGNORE,       0x0, 0xa, B30 | IMM3NZ, IGNORE, E}}}, 
  {O (O_ADD, SW), AV_H8SX, 0, "add.w", {{IMM3NZ_NS, ABS16DST, E}}, {{0x6, 0xb, 0x1, B31 | IGNORE, DSTABS16LIST, 0x0, 0xa, B30 | IMM3NZ, IGNORE, E}}}, 
  {O (O_ADD, SW), AV_H8SX, 0, "add.w", {{IMM3NZ_NS, ABS32DST, E}}, {{0x6, 0xb, 0x3, B31 | IGNORE, DSTABS32LIST, 0x0, 0xa, B30 | IMM3NZ, IGNORE, E}}}, 
  EXPAND_TWOOP_W (O (O_ADD, SW), "add.w", 0x0, 0x9, 0x1), 

  {O (O_ADD, SL), AV_H8H,  6, "add.l", {{RS32,      RD32,     E}}, {{0x0, 0xa, B31 | RS32,   B30 | RD32, E}}}, 
  {O (O_ADD, SL), AV_H8SX, 0, "add.l", {{IMM3NZ_NS, RD32,     E}}, {{0x0, 0xa, B31 | IMM3NZ, B31 | RD32, E}}}, 
  EXPAND_TWOOP_L (O (O_ADD, SL), "add.l", 0x1), 

  {O (O_ADDS, SL), AV_H8,  2, "adds",  {{KBIT,  RDP,  E}}, {{0x0, 0xB,KBIT, RDP, E}}},

  {O (O_ADDX, SB), AV_H8,   2, "addx",   {{IMM8,      RD8,       E}}, {{0x9, RD8, IMM8LIST, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{IMM8,      RDIND,     E}}, {{0x7, 0xd, B30 | RDIND, IGNORE,                        0x9, IGNORE, IMM8LIST, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{IMM8,      RDPOSTDEC, E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RDPOSTDEC, B31 | IGNORE, 0x9, IGNORE, IMM8LIST, E}}},
  {O (O_ADDX, SB), AV_H8,   2, "addx",   {{RS8,       RD8,       E}}, {{0x0, 0xe, RS8,  RD8,    E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{RS8,       RDIND,     E}}, {{0x7, 0xd, B30 | RDIND, IGNORE,                              0x0, 0xe, RS8, IGNORE, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{RS8,       RDPOSTDEC, E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RDPOSTDEC, B31       | IGNORE, 0x0, 0xe, RS8, IGNORE, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{RSIND,     RD8,       E}}, {{0x7, 0xc, B30 | RSIND, IGNORE,                              0x0, 0xe, IGNORE, RD8, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{RSPOSTDEC, RD8,       E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RSPOSTDEC, B30 | B20 | IGNORE, 0x0, 0xe, IGNORE, RD8, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{RSIND,     RDIND,     E}}, {{PREFIX_0174, 0x6, 0x8, B30 | RSIND,     0xd, 0x0, RDIND,     0x1, IGNORE, E}}},
  {O (O_ADDX, SB), AV_H8SX, 0, "addx.b", {{RSPOSTDEC, RDPOSTDEC, E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RSPOSTDEC, 0xd, 0xa, RDPOSTDEC, 0x1, IGNORE, E}}},

  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{IMM16,     RD16,      E}}, {{PREFIX_0151,                         0x7, 0x9, 0x1, RD16,   IMM16LIST, E}}}, 
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{IMM16,     RDIND,     E}}, {{0x7, 0xd, B31 | RDIND, B01 | IGNORE, 0x7, 0x9, 0x1, IGNORE, IMM16LIST, E}}}, 
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{IMM16,     RDPOSTDEC, E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x7, 0x9, 0x1, IGNORE, IMM16LIST, E}}}, 
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RS16,      RD16,      E}}, {{PREFIX_0151, 0x0, 0x9, RS16,  RD16,    E}}},
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RS16,      RDIND,     E}}, {{0x7, 0xd, B31 | RDIND, B01 | IGNORE, 0x0, 0x9, RS16, IGNORE, E}}},
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RS16,      RDPOSTDEC, E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x0, 0x9, RS16, IGNORE, E}}},
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RSIND,     RD16,      E}}, {{0x7, 0xc, B31 | RSIND, B01 | IGNORE, 0x0, 0x9, IGNORE, RD16, E}}},
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RSPOSTDEC, RD16,      E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RSPOSTDEC, B30 | B20 | B01 | IGNORE, 0x0, 0x9, IGNORE, RD16, E}}},
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RSIND,     RDIND,     E}}, {{PREFIX_0154, 0x6, 0x9, B30 | RSIND,     0xd, 0x0, RDIND,     0x1, IGNORE, E}}},
  {O (O_ADDX, SW), AV_H8SX, 0, "addx.w", {{RSPOSTDEC, RDPOSTDEC, E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RSPOSTDEC, 0xd, 0xa, RDPOSTDEC, 0x1, IGNORE, E}}},

  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{IMM32,     RD32,      E}}, {{PREFIX_0101, 0x7, 0xa, 0x1,  RD32, IMM32LIST, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{IMM32,     RDIND,     E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RDIND,     B31 | B20 | B01 | IGNORE, 0x7, 0xa, 0x1, IGNORE, IMM32LIST, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{IMM32,     RDPOSTDEC, E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x7, 0xa, 0x1, IGNORE, IMM32LIST, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RS32,      RD32,      E}}, {{PREFIX_0101, 0x0, 0xa, B31 | RS32,  B30 | RD32,    E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RS32,      RDIND,     E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RDIND,     B31 | B20 | B01 | IGNORE, 0x0, 0xa, B31 | RS32, B30 | IGNORE, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RS32,      RDPOSTDEC, E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x0, 0xa, B31 | RS32, B30 | IGNORE, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RSIND,     RD32,      E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RSIND,     B30 | B20 | B01 | IGNORE, 0x0, 0xa, B31 | IGNORE, B30 | RD32, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RSPOSTDEC, RD32,      E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RSPOSTDEC, B30 | B20 | B01 | IGNORE, 0x0, 0xa, B31 | IGNORE, B30 | RD32, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RSIND,     RDIND,     E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RSIND,     0xd, 0x0, RDIND,     0x1, IGNORE, E}}},
  {O (O_ADDX, SL), AV_H8SX, 0, "addx.l", {{RSPOSTDEC, RDPOSTDEC, E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RSPOSTDEC, 0xd, 0xa, RDPOSTDEC, 0x1, IGNORE, E}}},

  {O (O_AND, SB), AV_H8,   2, "and.b", {{IMM8, RD8,  E}}, {{0xe, RD8, IMM8LIST, E}}}, 
  EXPAND_TWOOP_B (O (O_AND, SB), "and.b", 0xe, 0x1, 0x6, 0x6, 0), 

  {O (O_AND, SW), AV_H8,   2, "and.w", {{RS16, RD16, E}}, {{0x6, 0x6, RS16, RD16,   E}}}, 
  EXPAND_TWOOP_W (O (O_AND, SW), "and.w", 0x6, 0x6, 0x6), 

  {O (O_AND, SL), AV_H8H,  2, "and.l", {{RS32,  RD32, E}}, {{0x0, 0x1, 0xF, 0x0, 0x6, 0x6, B30 | RS32, B30 | RD32, E}}},
  EXPAND_TWOOP_L (O (O_AND, SL), "and.l", 0x6), 

  {O (O_ANDC, SB), AV_H8,  2, "andc", {{IMM8,  CCR | DST, E}}, {{0x0, 0x6, IMM8LIST, E}}},
  {O (O_ANDC, SB), AV_H8S, 2, "andc", {{IMM8,  EXR | DST, E}}, {{0x0, 0x1, 0x4, EXR | DST, 0x0, 0x6, IMM8LIST, E}}},

  BRANCH (O (O_BRA, SB), "bra", 0x0),

  {O (O_BRAB, SB), AV_H8SX, 0, "bra", {{LOWREG | L_8,  E}}, {{0x5, 0x9, LOWREG | L_8  | B30, 0x5, E}}},
  {O (O_BRAW, SW), AV_H8SX, 0, "bra", {{LOWREG | L_16, E}}, {{0x5, 0x9, LOWREG | L_16 | B30, 0x6, E}}},
  {O (O_BRAL, SL), AV_H8SX, 0, "bra", {{RS32, E}}, {{0x5, 0x9, RS32 | B30, 0x7, E}}},

  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, RDIND,    OP3PCREL8}},  {{0x7, 0xC, B30 | RDIND, 0x0,            0x4, B30 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, ABS8DST,  OP3PCREL8}},  {{0x7, 0xE, DSTABS8LIST,                 0x4, B30 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, ABS16DST, OP3PCREL8}},  {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST,      0x4, B30 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, ABS32DST, OP3PCREL8}},  {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST,      0x4, B30 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, RDIND,    OP3PCREL8}},  {{0x7, 0xC, B30 | RDIND, 0x0,            0x4, B31 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, ABS8DST,  OP3PCREL8}},  {{0x7, 0xE, DSTABS8LIST,                 0x4, B31 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, ABS16DST, OP3PCREL8}},  {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST,      0x4, B31 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, ABS32DST, OP3PCREL8}},  {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST,      0x4, B31 | IMM3,      OP3PCREL8,  DATA,  E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, RDIND,    OP3PCREL16}}, {{0x7, 0xC, B30 | RDIND, 0x0,       0x5, 0x8, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, ABS8DST,  OP3PCREL16}}, {{0x7, 0xE, DSTABS8LIST,            0x5, 0x8, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, ABS16DST, OP3PCREL16}}, {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST, 0x5, 0x8, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABC, SB), AV_H8SX, 0, "bra/bc", {{IMM3, ABS32DST, OP3PCREL16}}, {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST, 0x5, 0x8, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, RDIND,    OP3PCREL16}}, {{0x7, 0xC, B30 | RDIND, 0x0,       0x5, 0x8, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, ABS8DST,  OP3PCREL16}}, {{0x7, 0xE, DSTABS8LIST,            0x5, 0x8, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, ABS16DST, OP3PCREL16}}, {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST, 0x5, 0x8, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BRABS, SB), AV_H8SX, 0, "bra/bs", {{IMM3, ABS32DST, OP3PCREL16}}, {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST, 0x5, 0x8, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  
  {O (O_BRAS,  SB), AV_H8SX, 0, "bra/s",  {{PCREL8, E}}, {{0x4, 0x0, PCREL8, DATA | B01, E}}},

  {O (O_BSRBC, SB), AV_H8SX, 0, "bsr/bc", {{IMM3, RDIND,    OP3PCREL16}}, {{0x7, 0xC, B30 | RDIND, 0x0,       0x5, 0xC, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBC, SB), AV_H8SX, 0, "bsr/bc", {{IMM3, ABS8DST,  OP3PCREL16}}, {{0x7, 0xE, DSTABS8LIST,            0x5, 0xC, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBC, SB), AV_H8SX, 0, "bsr/bc", {{IMM3, ABS16DST, OP3PCREL16}}, {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST, 0x5, 0xC, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBC, SB), AV_H8SX, 0, "bsr/bc", {{IMM3, ABS32DST, OP3PCREL16}}, {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST, 0x5, 0xC, B30 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBS, SB), AV_H8SX, 0, "bsr/bs", {{IMM3, RDIND,    OP3PCREL16}}, {{0x7, 0xC, B30 | RDIND, 0x0,       0x5, 0xC, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBS, SB), AV_H8SX, 0, "bsr/bs", {{IMM3, ABS8DST,  OP3PCREL16}}, {{0x7, 0xE, DSTABS8LIST,            0x5, 0xC, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBS, SB), AV_H8SX, 0, "bsr/bs", {{IMM3, ABS16DST, OP3PCREL16}}, {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST, 0x5, 0xC, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},
  {O (O_BSRBS, SB), AV_H8SX, 0, "bsr/bs", {{IMM3, ABS32DST, OP3PCREL16}}, {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST, 0x5, 0xC, B31 | IMM3, 0x0, OP3PCREL16, DATA3, E}}},

  BRANCH (O (O_BRA, SB), "bt",  0x0),
  BRANCH (O (O_BRN, SB), "brn", 0x1),
  BRANCH (O (O_BRN, SB), "bf",  0x1),
  BRANCH (O (O_BHI, SB), "bhi", 0x2),
  BRANCH (O (O_BLS, SB), "bls", 0x3),
  BRANCH (O (O_BCC, SB), "bcc", 0x4),
  BRANCH (O (O_BCC, SB), "bhs", 0x4),
  BRANCH (O (O_BCS, SB), "bcs", 0x5),
  BRANCH (O (O_BCS, SB), "blo", 0x5),
  BRANCH (O (O_BNE, SB), "bne", 0x6),
  BRANCH (O (O_BEQ, SB), "beq", 0x7),
  BRANCH (O (O_BVC, SB), "bvc", 0x8),
  BRANCH (O (O_BVS, SB), "bvs", 0x9),
  BRANCH (O (O_BPL, SB), "bpl", 0xA),
  BRANCH (O (O_BMI, SB), "bmi", 0xB),
  BRANCH (O (O_BGE, SB), "bge", 0xC),
  BRANCH (O (O_BLT, SB), "blt", 0xD),
  BRANCH (O (O_BGT, SB), "bgt", 0xE),
  BRANCH (O (O_BLE, SB), "ble", 0xF),

  EBITOP (O (O_BCLR,  SB), IMM3 | B30, "bclr",  0x6, 0x2, 0x7, 0xD, 0x7, 0xF, 0x8, 0),
  BITOP  (O (O_BAND,  SB), IMM3 | B30, "band",  0x7, 0x6, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  BITOP  (O (O_BIAND, SB), IMM3 | B31, "biand", 0x7, 0x6, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  BITOP  (O (O_BILD,  SB), IMM3 | B31, "bild",  0x7, 0x7, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  BITOP  (O (O_BIOR,  SB), IMM3 | B31, "bior",  0x7, 0x4, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  BITOP  (O (O_BIST,  SB), IMM3 | B31, "bist",  0x6, 0x7, 0x7, 0xD, 0x7, 0xF, 0x8, 0),
  BITOP  (O (O_BIXOR, SB), IMM3 | B31, "bixor", 0x7, 0x5, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  BITOP  (O (O_BLD,   SB), IMM3 | B30, "bld",   0x7, 0x7, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  EBITOP (O (O_BNOT,  SB), IMM3 | B30, "bnot",  0x6, 0x1, 0x7, 0xD, 0x7, 0xF, 0x8, 0),
  BITOP  (O (O_BOR,   SB), IMM3 | B30, "bor",   0x7, 0x4, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  EBITOP (O (O_BSET,  SB), IMM3 | B30, "bset",  0x6, 0x0, 0x7, 0xD, 0x7, 0xF, 0x8, 0),
  BITOP  (O (O_BST,   SB), IMM3 | B30, "bst",   0x6, 0x7, 0x7, 0xD, 0x7, 0xF, 0x8, 0),
  EBITOP (O (O_BTST,  SB), IMM3 | B30, "btst",  0x6, 0x3, 0x7, 0xC, 0x7, 0xE, 0x0, 0),
  BITOP  (O (O_BXOR,  SB), IMM3 | B30, "bxor",  0x7, 0x5, 0x7, 0xC, 0x7, 0xE, 0x0, 0),

  EBITOP_B (O (O_BCLREQ,  SB), IMM3 | B30, "bclr/eq",  0x6, 0x2, 0x7, 0xD, 0x7, 0xF, 0x8, 0x7),
  EBITOP_B (O (O_BCLRNE,  SB), IMM3 | B30, "bclr/ne",  0x6, 0x2, 0x7, 0xD, 0x7, 0xF, 0x8, 0x6),
  EBITOP_B (O (O_BSETEQ,  SB), IMM3 | B30, "bset/eq",  0x6, 0x0, 0x7, 0xD, 0x7, 0xF, 0x8, 0x7),
  EBITOP_B (O (O_BSETNE,  SB), IMM3 | B30, "bset/ne",  0x6, 0x0, 0x7, 0xD, 0x7, 0xF, 0x8, 0x6),
  BITOP_B  (O (O_BISTZ,   SB), IMM3 | B31, "bistz",    0x6, 0x7, 0x7, 0xD, 0x7, 0xF, 0x8, 0x7),
  BITOP_B  (O (O_BSTZ,    SB), IMM3 | B30, "bstz",     0x6, 0x7, 0x7, 0xD, 0x7, 0xF, 0x8, 0x7),

  {O (O_BFLD, SB), AV_H8SX, 0, "bfld", {{IMM8, RDIND,    R3_8}},  {{0x7, 0xC,           B30 | RDIND, 0x0, 0xF, R3_8, IMM8LIST, E}}},
  {O (O_BFLD, SB), AV_H8SX, 0, "bfld", {{IMM8, ABS8DST,  R3_8}},  {{0x7, 0xE,           DSTABS8LIST,      0xF, R3_8, IMM8LIST, E}}},
  {O (O_BFLD, SB), AV_H8SX, 0, "bfld", {{IMM8, ABS16DST, R3_8}},  {{0x6, 0xA, 0x1, 0x0, DSTABS16LIST,     0xF, R3_8, IMM8LIST, E}}},
  {O (O_BFLD, SB), AV_H8SX, 0, "bfld", {{IMM8, ABS32DST, R3_8}},  {{0x6, 0xA, 0x3, 0x0, DSTABS32LIST,     0xF, R3_8, IMM8LIST, E}}},

  /* Because the assembler treats SRC, DST and OP3 as ordinals, 
     I must designate the second argument, an immediate value, as DST.
     May God have mercy on my soul.  */
  {O (O_BFST, SB), AV_H8SX, 0, "bfst", {{RS8, DST | IMM8, R3_IND}},   {{0x7, 0xD,           B30 | R3_IND, 0x0, 0xF, RS8, DST | IMM8LIST, E}}},
  {O (O_BFST, SB), AV_H8SX, 0, "bfst", {{RS8, DST | IMM8, ABS8OP3}},  {{0x7, 0xF,           OP3ABS8LIST,       0xF, RS8, DST | IMM8LIST, E}}},
  {O (O_BFST, SB), AV_H8SX, 0, "bfst", {{RS8, DST | IMM8, ABS16OP3}}, {{0x6, 0xA, 0x1, 0x8, OP3ABS16LIST,      0xF, RS8, DST | IMM8LIST, E}}},
  {O (O_BFST, SB), AV_H8SX, 0, "bfst", {{RS8, DST | IMM8, ABS32OP3}}, {{0x6, 0xA, 0x3, 0x8, OP3ABS32LIST,      0xF, RS8, DST | IMM8LIST, E}}},

  {O (O_BSR, SB), AV_H8,   6, "bsr", {{PCREL8,  E}}, {{0x5, 0x5,           PCREL8,  DATA,  E}}},
  {O (O_BSR, SB), AV_H8,   6, "bsr", {{PCREL16, E}}, {{0x5, 0xC, 0x0, 0x0, PCREL16, DATA3, E}}},
  {O (O_BSR, SB), AV_H8SX, 0, "bsr", {{LOWREG | L_8,    E}}, {{0x5, 0xd, B30 | LOWREG | L_8,  0x5, E}}},
  {O (O_BSR, SW), AV_H8SX, 0, "bsr", {{LOWREG | L_16,   E}}, {{0x5, 0xd, B30 | LOWREG | L_16, 0x6, E}}},
  {O (O_BSR, SL), AV_H8SX, 0, "bsr", {{OR32,   E}}, {{0x5, 0xd, B30 | OR32, 0x7, E}}},

  {O (O_CMP, SB), AV_H8,   2, "cmp.b", {{IMM8, RD8, E}}, {{0xa, RD8, IMM8LIST, E}}}, 
  EXPAND_TWOOP_B (O (O_CMP, SB), "cmp.b", 0xa, 0x1, 0xc, 0x2, B00), 

  {O (O_CMP, SW), AV_H8,   2, "cmp.w", {{RS16,      RD16,     E}}, {{0x1, 0xd, RS16,         RD16,   E}}}, 
  {O (O_CMP, SW), AV_H8SX, 0, "cmp.w", {{IMM3NZ_NS, RD16,     E}}, {{0x1, 0xf, B30 | IMM3NZ, RD16,   E}}}, 
  {O (O_CMP, SW), AV_H8SX, 0, "cmp.w", {{IMM3NZ_NS, RDIND,    E}}, {{0x7, 0xd,      B31 | RDIND,  IGNORE,       0x1, 0xf, B30 | IMM3NZ, IGNORE, E}}}, 
  {O (O_CMP, SW), AV_H8SX, 0, "cmp.w", {{IMM3NZ_NS, ABS16DST, E}}, {{0x6, 0xb, 0x1, B31 | IGNORE, DSTABS16LIST, 0x1, 0xf, B30 | IMM3NZ, IGNORE, E}}}, 
  {O (O_CMP, SW), AV_H8SX, 0, "cmp.w", {{IMM3NZ_NS, ABS32DST, E}}, {{0x6, 0xb, 0x3, B31 | IGNORE, DSTABS32LIST, 0x1, 0xf, B30 | IMM3NZ, IGNORE, E}}}, 
  EXPAND_TWOOP_W (O (O_CMP, SW), "cmp.w", 0x1, 0xd, 0x2), 

  {O (O_CMP, SL), AV_H8H,  6, "cmp.l", {{RS32,      RD32,     E}}, {{0x1, 0xf, B31 | RS32,   B30 | RD32, E}}}, 
  {O (O_CMP, SL), AV_H8SX, 0, "cmp.l", {{IMM3NZ_NS, RD32,     E}}, {{0x1, 0xf, B31 | IMM3NZ, B31 | RD32, E}}}, 
  EXPAND_TWOOP_L (O (O_CMP, SL), "cmp.l", 0x2), 

  UNOP (O (O_DAA, SB), "daa",   0x0, 0xF),
  UNOP (O (O_DAS, SB), "das",   0x1, 0xF),
  UNOP (O (O_DEC, SB), "dec.b", 0x1, 0xA),

  {O (O_DEC, SW), AV_H8H, 2, "dec.w", {{DBIT, RD16, E}}, {{0x1, 0xB, 0x5 | DBIT, RD16,       E}}},
  {O (O_DEC, SL), AV_H8H, 2, "dec.l", {{DBIT, RD32, E}}, {{0x1, 0xB, 0x7 | DBIT, RD32 | B30, E}}},

  {O (O_DIVS, SW), AV_H8SX, 0, "divs.w", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xd, 0x6, 0x5, 0x1, IMM4, RD16, E}}},
  {O (O_DIVS, SW), AV_H8SX, 0, "divs.w", {{RS16, RD16, E}}, {{0x0, 0x1, 0xd, 0x2, 0x5, 0x1, RS16, RD16, E}}},
  {O (O_DIVS, SL), AV_H8SX, 0, "divs.l", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xd, 0x6, 0x5, 0x3, IMM4,       B30 | RD32, E}}},
  {O (O_DIVS, SL), AV_H8SX, 0, "divs.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xd, 0x2, 0x5, 0x3, B30 | RS32, B30 | RD32, E}}},

  {O (O_DIVU, SW), AV_H8SX, 0, "divu.w", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xd, 0xe, 0x5, 0x1, IMM4, RD16, E}}},
  {O (O_DIVU, SW), AV_H8SX, 0, "divu.w", {{RS16, RD16, E}}, {{0x0, 0x1, 0xd, 0xa, 0x5, 0x1, RS16, RD16, E}}},
  {O (O_DIVU, SL), AV_H8SX, 0, "divu.l", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xd, 0xe, 0x5, 0x3, IMM4,       B30 | RD32, E}}},
  {O (O_DIVU, SL), AV_H8SX, 0, "divu.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xd, 0xa, 0x5, 0x3, B30 | RS32, B30 | RD32, E}}},

  {O (O_DIVXS, SB), AV_H8SX, 0,  "divxs.b", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xD, 0x4, 0x5, 0x1, IMM4, RD16, E}}},
  {O (O_DIVXS, SB), AV_H8H,  13, "divxs.b", {{RS8,  RD16, E}}, {{0x0, 0x1, 0xD, 0x0, 0x5, 0x1, RS8,  RD16, E}}},
  {O (O_DIVXS, SW), AV_H8SX, 0,  "divxs.w", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xD, 0x4, 0x5, 0x3, IMM4, B30 | RD32, E}}},
  {O (O_DIVXS, SW), AV_H8H,  21, "divxs.w", {{RS16, RD32, E}}, {{0x0, 0x1, 0xD, 0x0, 0x5, 0x3, RS16, B30 | RD32, E}}},

  {O (O_DIVXU, SB), AV_H8SX, 0,  "divxu.b", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xD, 0xC, 0x5, 0x1, IMM4, RD16, E}}},
  {O (O_DIVXU, SB), AV_H8,   13, "divxu.b", {{RS8,  RD16, E}}, {{0x5, 0x1,                     RS8,  RD16, E}}},
  {O (O_DIVXU, SW), AV_H8SX, 0,  "divxu.w", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xD, 0xC, 0x5, 0x3, IMM4, B30 | RD32, E}}},
  {O (O_DIVXU, SW), AV_H8H,  21, "divxu.w", {{RS16, RD32, E}}, {{0x5, 0x3,                     RS16, B30 | RD32, E}}},

  {O (O_EEPMOV, SB), AV_H8,  4, "eepmov.b", {{E}}, {{0x7, 0xB, 0x5, 0xC, 0x5, 0x9, 0x8, 0xF, E}}},
  {O (O_EEPMOV, SW), AV_H8H, 4, "eepmov.w", {{E}}, {{0x7, 0xB, 0xD, 0x4, 0x5, 0x9, 0x8, 0xF, E}}},

  EXPAND_UNOP_STD_W      (O (O_EXTS, SW), "exts.w",          PREFIX_015, 0x1, 0x7, 0xd),
  EXPAND_UNOP_STD_L      (O (O_EXTS, SL), "exts.l",          PREFIX_010, 0x1, 0x7, 0xf),
  EXPAND_UNOP_EXTENDED_L (O (O_EXTS, SL), "exts.l", CONST_2, PREFIX_010, 0x1, 0x7, 0xe, 0),
  EXPAND_UNOP_STD_W      (O (O_EXTU, SW), "extu.w",          PREFIX_015, 0x1, 0x7, 0x5),
  EXPAND_UNOP_STD_L      (O (O_EXTU, SL), "extu.l",          PREFIX_010, 0x1, 0x7, 0x7),
  EXPAND_UNOP_EXTENDED_L (O (O_EXTU, SL), "extu.l", CONST_2, PREFIX_010, 0x1, 0x7, 0x6, 0),

  UNOP (O (O_INC, SB), "inc", 0x0, 0xA),

  {O (O_INC, SW), AV_H8H,  2, "inc.w", {{DBIT, RD16, E}},    {{0x0, 0xB, 0x5 | DBIT, RD16,       E}}},
  {O (O_INC, SL), AV_H8H,  2, "inc.l", {{DBIT, RD32, E}},    {{0x0, 0xB, 0x7 | DBIT, RD32 | B30, E}}},

  {O (O_JMP, SN), AV_H8,   4, "jmp", {{RSIND, E}},         {{0x5, 0x9, B30 | RSIND, 0x0, E}}},
  {O (O_JMP, SN), AV_H8,   6, "jmp", {{ABSJMP | L_24, E}}, {{0x5, 0xA, SRC | ABSJMP | L_24, DATA5, E}}},

  {O (O_JMP, SN), AV_H8SX, 0, "jmp", {{ABSJMP | L_32, E}}, {{0x5, 0x9, 0x0, 0x8, ABSJMP | L_32, DATA7, E}}},

  {O (O_JMP, SN), AV_H8,   8, "jmp", {{MEMIND, E}}, {{0x5, 0xB, SRC | MEMIND, DATA, E}}},
  {O (O_JMP, SN), AV_H8SX, 0, "jmp", {{VECIND, E}}, {{0x5, 0x9, B31 | SRC | VECIND, DATA, E}}},

  {O (O_JSR, SN), AV_H8,   6, "jsr", {{RSIND, E}},         {{0x5, 0xD, B30 | RSIND, 0x0, E}}},
  {O (O_JSR, SN), AV_H8,   8, "jsr", {{ABSJMP | L_24, E}}, {{0x5, 0xE, SRC | ABSJMP | L_24, DATA5, E}}},

  {O (O_JSR, SN), AV_H8SX, 0, "jsr", {{ABSJMP | L_32, E}}, {{0x5, 0xD, 0x0, 0x8, ABSJMP | L_32, DATA7, E}}},

  {O (O_JSR, SN), AV_H8,   8, "jsr", {{MEMIND, E}}, {{0x5, 0xF, SRC | MEMIND, DATA, E}}},
  {O (O_JSR, SN), AV_H8SX, 8, "jsr", {{VECIND, E}}, {{0x5, 0xD, SRC | VECIND, DATA, E}}},

  {O (O_LDC, SB), AV_H8,   2, "ldc", {{IMM8,       CCR     | DST, E}}, {{                           0x0, 0x7, IMM8LIST, E}}},
  {O (O_LDC, SB), AV_H8S,  2, "ldc", {{IMM8,       EXR     | DST, E}}, {{0x0, 0x1, 0x4,  EXR | DST, 0x0, 0x7, IMM8LIST, E}}},
  {O (O_LDC, SB), AV_H8,   2, "ldc", {{RS8,        CCR     | DST, E}}, {{0x0, 0x3, B30 | CCR | DST, RS8, E}}},
  {O (O_LDC, SB), AV_H8S,  2, "ldc", {{RS8,        EXR     | DST, E}}, {{0x0, 0x3, B30 | EXR | DST, RS8, E}}},
  {O (O_LDC, SW), AV_H8H,  2, "ldc", {{RSIND,      CCR     | DST, E}}, {{PREFIXLDC, 0x6, 0x9, B30 | RSIND,     IGNORE, E}}},
  {O (O_LDC, SW), AV_H8S,  2, "ldc", {{RSIND,      EXR     | DST, E}}, {{PREFIXLDC, 0x6, 0x9, B30 | RSIND,     IGNORE, E}}},
  {O (O_LDC, SW), AV_H8H,  2, "ldc", {{RSPOSTINC,  CCR     | DST, E}}, {{PREFIXLDC, 0x6, 0xD, B30 | RSPOSTINC, IGNORE, E}}},
  {O (O_LDC, SW), AV_H8S,  2, "ldc", {{RSPOSTINC,  EXR     | DST, E}}, {{PREFIXLDC, 0x6, 0xD, B30 | RSPOSTINC, IGNORE, E}}},
  {O (O_LDC, SW), AV_H8H,  2, "ldc", {{DISP16SRC,  CCR     | DST, E}}, {{PREFIXLDC, 0x6, 0xF, B30 | DISPREG,                     IGNORE, SRC | DISP16LIST, E}}},
  {O (O_LDC, SW), AV_H8S,  2, "ldc", {{DISP16SRC,  EXR     | DST, E}}, {{PREFIXLDC, 0x6, 0xF, B30 | DISPREG,                     IGNORE, SRC | DISP16LIST, E}}},
  {O (O_LDC, SW), AV_H8H,  2, "ldc", {{DISP32SRC,  CCR     | DST, E}}, {{PREFIXLDC, 0x7, 0x8, B30 | DISPREG, 0x0, 0x6, 0xB, 0x2, IGNORE, SRC | DISP32LIST, E}}},
  {O (O_LDC, SW), AV_H8S,  2, "ldc", {{DISP32SRC,  EXR     | DST, E}}, {{PREFIXLDC, 0x7, 0x8, B30 | DISPREG, 0x0, 0x6, 0xB, 0x2, IGNORE, SRC | DISP32LIST, E}}},
  {O (O_LDC, SW), AV_H8H,  2, "ldc", {{ABS16SRC,   CCR     | DST, E}}, {{PREFIXLDC, 0x6, 0xB, 0x0, IGNORE, SRC | ABS16LIST, E}}},
  {O (O_LDC, SW), AV_H8S,  2, "ldc", {{ABS16SRC,   EXR     | DST, E}}, {{PREFIXLDC, 0x6, 0xB, 0x0, IGNORE, SRC | ABS16LIST, E}}},
  {O (O_LDC, SW), AV_H8H,  2, "ldc", {{ABS32SRC,   CCR     | DST, E}}, {{PREFIXLDC, 0x6, 0xB, 0x2, IGNORE, SRC | MEMRELAX | ABS32LIST, E}}},
  {O (O_LDC, SW), AV_H8S,  2, "ldc", {{ABS32SRC,   EXR     | DST, E}}, {{PREFIXLDC, 0x6, 0xB, 0x2, IGNORE, SRC | MEMRELAX | ABS32LIST, E}}},

  {O (O_LDC, SL), AV_H8SX, 0, "ldc", {{RS32, B30 | VBR_SBR | DST, E}}, {{0x0, 0x3, B30 | VBR_SBR | DST, RS32, E}}},


  {O (O_MOV, SB), AV_H8,   2, "mov.b", {{IMM8, RD8,      E}}, {{0xF, RD8,            IMM8LIST,     E}}},
  {O (O_MOV, SB), AV_H8SX, 0, "mov.b", {{IMM4_NS, ABS16DST, E}}, {{0x6, 0xa, 0xd, IMM4, DSTABS16LIST, E}}},
  {O (O_MOV, SB), AV_H8SX, 0, "mov.b", {{IMM4_NS, ABS32DST, E}}, {{0x6, 0xa, 0xf, IMM4, DSTABS32LIST, E}}},
  MOVFROM_IMM8 (O (O_MOV, SB), PREFIX_017D, "mov.b", IMM8),

  {O (O_MOV, SB), AV_H8,   2,    "mov.b", {{RS8, RD8,     E}}, {{0x0, 0xC, RS8, RD8,    E}}},
  MOVFROM_REG_BW (O (O_MOV, SB), "mov.b", RS8, PREFIX_017, 8, 10, 12, 14, MEMRELAX),
  {O (O_MOV, SB), AV_H8,   4,    "mov.b", {{RS8, ABS8DST, E}}, {{0x3, RS8, DSTABS8LIST, E}}},
  MOVTO_REG_BW (O (O_MOV, SB),   "mov.b", RD8, PREFIX_017, 8, 10, 12, 14, MEMRELAX),
  {O (O_MOV, SB), AV_H8,   4,    "mov.b", {{ABS8SRC, RD8, E}}, {{0x2, RD8, ABS8LIST,    E}}},

  MOVFROM_STD (O (O_MOV, SB), PREFIX_0178, "mov.b", RSIND,     FROM_IND), 
  MOVFROM_STD (O (O_MOV, SB), PREFIX_0178, "mov.b", RSPOSTINC, FROM_POSTINC), 
  MOVFROM_STD (O (O_MOV, SB), PREFIX_0178, "mov.b", RSPOSTDEC, FROM_POSTDEC), 
  MOVFROM_STD (O (O_MOV, SB), PREFIX_0178, "mov.b", RSPREINC,  FROM_PREINC), 
  MOVFROM_STD (O (O_MOV, SB), PREFIX_0178, "mov.b", RSPREDEC,  FROM_PREDEC), 
  MOVFROM_STD (O (O_MOV, SB), PREFIX_0178, "mov.b", DISP2SRC,  FROM_DISP2), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", DISP16SRC, FROM_DISP16,  DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", DISP32SRC, FROM_DISP32,  DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", INDEXB16,  FROM_DISP16B, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", INDEXW16,  FROM_DISP16W, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", INDEXL16,  FROM_DISP16L, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", INDEXB32,  FROM_DISP32B, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", INDEXW32,  FROM_DISP32W, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", INDEXL32,  FROM_DISP32L, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", ABS16SRC,  FROM_ABS16,   ABS16LIST), 
  MOVFROM_AD  (O (O_MOV, SB), PREFIX_0178, "mov.b", ABS32SRC,  FROM_ABS32,   ABS32LIST), 

  {O (O_MOV, SW), AV_H8SX, 0, "mov.w", {{IMM3NZ_NS, RD16, E}}, {{0x0, 0xf, B30 | IMM3NZ, RD16,   E}}},
  {O (O_MOV, SW), AV_H8,   4, "mov.w", {{IMM16,     RD16, E}}, {{0x7, 0x9, 0x0, RD16, IMM16LIST, E}}},
  {O (O_MOV, SW), AV_H8SX, 0, "mov.w", {{IMM4_NS,  ABS16DST, E}}, {{0x6, 0xb, 0xd, IMM4, DSTABS16LIST, E}}},
  {O (O_MOV, SW), AV_H8SX, 0, "mov.w", {{IMM4_NS,  ABS32DST, E}}, {{0x6, 0xb, 0xf, IMM4, DSTABS32LIST, E}}},

  MOVFROM_IMM8 (O (O_MOV, SW), PREFIX_015D,   "mov.w", IMM8U_NS),
  MOVFROM_IMM  (O (O_MOV, SW), PREFIX_7974,   "mov.w", IMM16, IMM16LIST),

  {O (O_MOV, SW), AV_H8,   2, "mov.w", {{RS16, RD16,      E}}, {{0x0, 0xD, RS16, RD16, E}}},
  MOVFROM_REG_BW (O (O_MOV, SW), "mov.w", RS16, PREFIX_015, 9, 11, 13, 15, 0),
  MOVTO_REG_BW   (O (O_MOV, SW), "mov.w", RD16, PREFIX_015, 9, 11, 13, 15, 0),

  MOVFROM_STD (O (O_MOV, SW), PREFIX_0158, "mov.w", RSIND,     FROM_IND), 
  MOVFROM_STD (O (O_MOV, SW), PREFIX_0158, "mov.w", RSPOSTINC, FROM_POSTINC), 
  MOVFROM_STD (O (O_MOV, SW), PREFIX_0158, "mov.w", RSPOSTDEC, FROM_POSTDEC), 
  MOVFROM_STD (O (O_MOV, SW), PREFIX_0158, "mov.w", RSPREINC,  FROM_PREINC), 
  MOVFROM_STD (O (O_MOV, SW), PREFIX_0158, "mov.w", RSPREDEC,  FROM_PREDEC), 
  MOVFROM_STD (O (O_MOV, SW), PREFIX_0158, "mov.w", DISP2SRC,  FROM_DISP2), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", DISP16SRC, FROM_DISP16,  DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", DISP32SRC, FROM_DISP32,  DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", INDEXB16,  FROM_DISP16B, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", INDEXW16,  FROM_DISP16W, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", INDEXL16,  FROM_DISP16L, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", INDEXB32,  FROM_DISP32B, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", INDEXW32,  FROM_DISP32W, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", INDEXL32,  FROM_DISP32L, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", ABS16SRC,  FROM_ABS16,   ABS16LIST), 
  MOVFROM_AD  (O (O_MOV, SW), PREFIX_0158, "mov.w", ABS32SRC,  FROM_ABS32,   ABS32LIST), 

  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{IMM3NZ_NS, RD32, E}}, {{0x0, 0xf, B31 | IMM3NZ, B31 | RD32, E}}},

  MOVFROM_IMM8 (O (O_MOV, SL), PREFIX_010D, "mov.l", IMM8U_NS),
  MOVFROM_IMM  (O (O_MOV, SL), PREFIX_7A7C, "mov.l", IMM16U_NS, IMM16ULIST),

  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{IMM16U_NS, RD32, E}}, {{0x7, 0xa, 0x0, B31 | RD32, IMM16ULIST, E}}},
  {O (O_MOV, SL), AV_H8H,  4, "mov.l", {{IMM32,     RD32, E}}, {{0x7, 0xa, 0x0, B30 | RD32, IMM32LIST,  E}}},

  MOVFROM_IMM (O (O_MOV, SL), PREFIX_7A74, "mov.l", IMM32, IMM32LIST),

  {O (O_MOV, SL), AV_H8H,  2, "mov.l", {{RS32, RD32,      E}}, {{0x0, 0xf, B31 | RS32, B30 | RD32, E}}},

  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RS32, RDIND,     E}}, {{PREFIX_0100,                       0x6, 0x9, B31 | RDIND, B30 | RS32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, RDPOSTINC, E}}, {{PREFIX_0103,                       0x6, 0xd, B31 | RDPOSTINC,   RS32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, RDPOSTDEC, E}}, {{PREFIX_0101,                       0x6, 0xd, B31 | RDPOSTDEC,   RS32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, RDPREINC,  E}}, {{PREFIX_0102,                       0x6, 0xd, B31 | RDPREINC,    RS32, E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RS32, RDPREDEC,  E}}, {{PREFIX_0100,                       0x6, 0xd, B31 | RDPREDEC,    RS32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, DISP2DST,  E}}, {{PREFIX_010,  B30 | B20 | DISP2DST, 0x6, 0x9, B31 | DSTDISPREG,  RS32, E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RS32, DISP16DST, E}}, {{PREFIX_0100,                       0x6, 0xf, B31 | DSTDISPREG,  RS32, DSTDISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 6, "mov.l", {{RS32, DISP32DST, E}}, {{0x7, 0x8, B31 | DSTDISPREG, 0x0,   0x6, 0xb, 0xa,               RS32, DSTDISP32LIST, E}}},
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RS32, DISP32DST, E}}, {{PREFIX_0100,                       0x7, 0x8, B31 | DSTDISPREG, 0x0,   0x6, 0xb, 0xa,               RS32, DSTDISP32LIST, E}}},
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, INDEXB16D, E}}, {{PREFIX_0101,                       0x6, 0xf, B31 | DSTDISPREG,  RS32, DSTDISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, INDEXW16D, E}}, {{PREFIX_0102,                       0x6, 0xf, B31 | DSTDISPREG,  RS32, DSTDISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, INDEXL16D, E}}, {{PREFIX_0103,                       0x6, 0xf, B31 | DSTDISPREG,  RS32, DSTDISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, INDEXB32D, E}}, {{0x7, 0x8, B31 | DSTDISPREG, 0x1,   0x6, 0xb, 0xa,               RS32, DSTDISP32LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, INDEXW32D, E}}, {{0x7, 0x8, B31 | DSTDISPREG, 0x2,   0x6, 0xb, 0xa,               RS32, DSTDISP32LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RS32, INDEXL32D, E}}, {{0x7, 0x8, B31 | DSTDISPREG, 0x3,   0x6, 0xb, 0xa,               RS32, DSTDISP32LIST, E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RS32, ABS16DST,  E}}, {{PREFIX_0100,                       0x6, 0xb, 0x8,            RS32, DSTABS16LIST,            E}}},
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RS32, ABS32DST,  E}}, {{PREFIX_0100,                       0x6, 0xb, 0xa,            RS32, MEMRELAX | DSTABS32LIST, E}}},

  {O (O_MOV, SL), AV_H8H,  4, "mov.l", {{RSIND,     RD32, E}}, {{PREFIX_0100,                       0x6, 0x9, B30 | RSIND,     RD32, E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{RSPOSTINC, RD32, E}}, {{PREFIX_0100,                       0x6, 0xd, B30 | RSPOSTINC, RD32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RSPOSTDEC, RD32, E}}, {{PREFIX_0102,                       0x6, 0xd, B30 | RSPOSTDEC, RD32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RSPREINC,  RD32, E}}, {{PREFIX_0101,                       0x6, 0xd, B30 | RSPREINC,  RD32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{RSPREDEC,  RD32, E}}, {{PREFIX_0103,                       0x6, 0xd, B30 | RSPREDEC,  RD32, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{DISP2SRC,  RD32, E}}, {{PREFIX_010,  B30 | B20 | DISP2SRC, 0x6, 0x9, B30 | DISPREG,   RD32, E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{DISP16SRC, RD32, E}}, {{PREFIX_0100,                       0x6, 0xf, B30 | DISPREG,   RD32, SRC | DISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 6, "mov.l", {{DISP32SRC, RD32, E}}, {{0x7, 0x8, B31 | DISPREG, 0x0,      0x6, 0xb, 0x2,             RD32, SRC | DISP32LIST, E}}},
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{DISP32SRC, RD32, E}}, {{PREFIX_0100,                       0x7, 0x8, B30 | DISPREG, 0x0, 0x6, 0xb, 0x2,             RD32, SRC | DISP32LIST, E}}},
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{INDEXB16,  RD32, E}}, {{PREFIX_0101,                       0x6, 0xf, B30 | DISPREG,   RD32, SRC | DISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{INDEXW16,  RD32, E}}, {{PREFIX_0102,                       0x6, 0xf, B30 | DISPREG,   RD32, SRC | DISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{INDEXL16,  RD32, E}}, {{PREFIX_0103,                       0x6, 0xf, B30 | DISPREG,   RD32, SRC | DISP16LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{INDEXB32,  RD32, E}}, {{0x7, 0x8, B31 | DISPREG, 0x1,      0x6, 0xb, 0x2,             RD32, SRC | DISP32LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{INDEXW32,  RD32, E}}, {{0x7, 0x8, B31 | DISPREG, 0x2,      0x6, 0xb, 0x2,             RD32, SRC | DISP32LIST, E}}}, 
  {O (O_MOV, SL), AV_H8SX, 0, "mov.l", {{INDEXL32,  RD32, E}}, {{0x7, 0x8, B31 | DISPREG, 0x3,      0x6, 0xb, 0x2,             RD32, SRC | DISP32LIST, E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{ABS16SRC,  RD32, E}}, {{PREFIX_0100,                       0x6, 0xb, 0x0,             RD32, SRC | ABS16LIST,  E}}}, 
  {O (O_MOV, SL), AV_H8H,  6, "mov.l", {{ABS32SRC,  RD32, E}}, {{PREFIX_0100,                       0x6, 0xb, 0x2,             RD32, SRC | MEMRELAX | ABS32LIST, E}}}, 

  MOVFROM_STD (O (O_MOV, SL), PREFIX_0108, "mov.l", RSIND,     FROM_IND), 
  MOVFROM_STD (O (O_MOV, SL), PREFIX_0108, "mov.l", RSPOSTINC, FROM_POSTINC), 
  MOVFROM_STD (O (O_MOV, SL), PREFIX_0108, "mov.l", RSPOSTDEC, FROM_POSTDEC), 
  MOVFROM_STD (O (O_MOV, SL), PREFIX_0108, "mov.l", RSPREINC,  FROM_PREINC), 
  MOVFROM_STD (O (O_MOV, SL), PREFIX_0108, "mov.l", RSPREDEC,  FROM_PREDEC), 
  MOVFROM_STD (O (O_MOV, SL), PREFIX_0108, "mov.l", DISP2SRC,  FROM_DISP2), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", DISP16SRC, FROM_DISP16,  DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", DISP32SRC, FROM_DISP32,  DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", INDEXB16,  FROM_DISP16B, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", INDEXW16,  FROM_DISP16W, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", INDEXL16,  FROM_DISP16L, DISP16LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", INDEXB32,  FROM_DISP32B, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", INDEXW32,  FROM_DISP32W, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", INDEXL32,  FROM_DISP32L, DISP32LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", ABS16SRC,  FROM_ABS16,   ABS16LIST), 
  MOVFROM_AD  (O (O_MOV, SL), PREFIX_0108, "mov.l", ABS32SRC,  FROM_ABS32,   ABS32LIST), 

#define DO_MOVA1(TYPE, OP0, OP1) \
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB16, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0x8, B30 | R3_32, DISP16LIST, E}}}, \
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW16, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0x9, B30 | R3_32, DISP16LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB16, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xA, B30 | R3_32, DISP16LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW16, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xB, B30 | R3_32, DISP16LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB16, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xC, B30 | R3_32, DISP16LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW16, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xD, B30 | R3_32, DISP16LIST, E}}}, \
\
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB32, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0x8, B31 | R3_32, DISP32LIST, E}}}, \
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW32, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0x9, B31 | R3_32, DISP32LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB32, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xA, B31 | R3_32, DISP32LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW32, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xB, B31 | R3_32, DISP32LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB32, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xC, B31 | R3_32, DISP32LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW32, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xD, B31 | R3_32, DISP32LIST, E}}}

#define DO_MOVA2(TYPE, OP0, OP1, OP2) \
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB16, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0x8, B30 | R3_32, OP2, DISP16LIST, E}}}, \
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW16, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0x9, B30 | R3_32, OP2, DISP16LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB16, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xA, B30 | R3_32, OP2, DISP16LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW16, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xB, B30 | R3_32, OP2, DISP16LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB16, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xC, B30 | R3_32, OP2, DISP16LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW16, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xD, B30 | R3_32, OP2, DISP16LIST, E}}}, \
\
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB32, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0x8, B31 | R3_32, OP2, DISP32LIST, E}}}, \
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW32, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0x9, B31 | R3_32, OP2, DISP32LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB32, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xA, B31 | R3_32, OP2, DISP32LIST, E}}}, \
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW32, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xB, B31 | R3_32, OP2, DISP32LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB32, TYPE, R3_32}}, {{PREFIX_017F, OP0, OP1, 0xC, B31 | R3_32, OP2, DISP32LIST, E}}}, \
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW32, TYPE, R3_32}}, {{PREFIX_015F, OP0, OP1, 0xD, B31 | R3_32, OP2, DISP32LIST, E}}}

  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB16, E}}, {{0x7, 0xA, 0x8, B31 | DISPREG, DISP16LIST, E}}},
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW16, E}}, {{0x7, 0xA, 0x9, B31 | DISPREG, DISP16LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB16, E}}, {{0x7, 0xA, 0xA, B31 | DISPREG, DISP16LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW16, E}}, {{0x7, 0xA, 0xB, B31 | DISPREG, DISP16LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB16, E}}, {{0x7, 0xA, 0xC, B31 | DISPREG, DISP16LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW16, E}}, {{0x7, 0xA, 0xD, B31 | DISPREG, DISP16LIST, E}}},

  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB32, E}}, {{0x7, 0xA, 0x8, B30 | DISPREG, DISP32LIST, E}}},
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW32, E}}, {{0x7, 0xA, 0x9, B30 | DISPREG, DISP32LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB32, E}}, {{0x7, 0xA, 0xA, B30 | DISPREG, DISP32LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW32, E}}, {{0x7, 0xA, 0xB, B30 | DISPREG, DISP32LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB32, E}}, {{0x7, 0xA, 0xC, B30 | DISPREG, DISP32LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW32, E}}, {{0x7, 0xA, 0xD, B30 | DISPREG, DISP32LIST, E}}},

  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB16, RD8,  R3_32}}, {{0x7, 0x8, RD8,  0x8, 0x7, 0xA, 0x8, B31 | R3_32, DISP16LIST, E}}},
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW16, RD16, R3_32}}, {{0x7, 0x8, RD16, 0x9, 0x7, 0xA, 0x9, B31 | R3_32, DISP16LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB16, RD8,  R3_32}}, {{0x7, 0x8, RD8,  0x8, 0x7, 0xA, 0xA, B31 | R3_32, DISP16LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW16, RD16, R3_32}}, {{0x7, 0x8, RD16, 0x9, 0x7, 0xA, 0xB, B31 | R3_32, DISP16LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB16, RD8,  R3_32}}, {{0x7, 0x8, RD8,  0x8, 0x7, 0xA, 0xC, B31 | R3_32, DISP16LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW16, RD16, R3_32}}, {{0x7, 0x8, RD16, 0x9, 0x7, 0xA, 0xD, B31 | R3_32, DISP16LIST, E}}},

  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXB32, RD8,  R3_32}}, {{0x7, 0x8, RD8,  0x8, 0x7, 0xA, 0x8, B30 | R3_32, DISP32LIST, E}}},
  {O (O_MOVAB, SL), AV_H8SX, 0, "mova/b.l", {{INDEXW32, RD16, R3_32}}, {{0x7, 0x8, RD16, 0x9, 0x7, 0xA, 0x9, B30 | R3_32, DISP32LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXB32, RD8,  R3_32}}, {{0x7, 0x8, RD8,  0x8, 0x7, 0xA, 0xA, B30 | R3_32, DISP32LIST, E}}},
  {O (O_MOVAW, SL), AV_H8SX, 0, "mova/w.l", {{INDEXW32, RD16, R3_32}}, {{0x7, 0x8, RD16, 0x9, 0x7, 0xA, 0xB, B30 | R3_32, DISP32LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXB32, RD8,  R3_32}}, {{0x7, 0x8, RD8,  0x8, 0x7, 0xA, 0xC, B30 | R3_32, DISP32LIST, E}}},
  {O (O_MOVAL, SL), AV_H8SX, 0, "mova/l.l", {{INDEXW32, RD16, R3_32}}, {{0x7, 0x8, RD16, 0x9, 0x7, 0xA, 0xD, B30 | R3_32, DISP32LIST, E}}},

  DO_MOVA1 (RDIND,     0x0, B30 | RDIND),
  DO_MOVA1 (RDPOSTINC, 0x8, B30 | RDPOSTINC),
  DO_MOVA1 (RDPOSTDEC, 0xA, B30 | RDPOSTDEC),
  DO_MOVA1 (RDPREINC,  0x9, B30 | RDPREINC),
  DO_MOVA1 (RDPREDEC,  0xB, B30 | RDPREDEC),
  DO_MOVA1 (DISP2DST,  B30 | B20 | DISP2DST,  B30 | DSTDISPREG),
  DO_MOVA2 (DISP16DST, 0xC, B30 | DSTDISPREG, DSTDISP16LIST),
  DO_MOVA2 (DISP32DST, 0xC, B31 | DSTDISPREG, DSTDISP32LIST),
  DO_MOVA2 (INDEXB16D, 0xD, B30 | DSTDISPREG, DSTDISP16LIST),
  DO_MOVA2 (INDEXW16D, 0xE, B30 | DSTDISPREG, DSTDISP16LIST),
  DO_MOVA2 (INDEXL16D, 0xF, B30 | DSTDISPREG, DSTDISP16LIST),
  DO_MOVA2 (INDEXB32D, 0xD, B31 | DSTDISPREG, DSTDISP32LIST),
  DO_MOVA2 (INDEXW32D, 0xE, B31 | DSTDISPREG, DSTDISP32LIST),
  DO_MOVA2 (INDEXL32D, 0xF, B31 | DSTDISPREG, DSTDISP32LIST),
  DO_MOVA2 (ABS16DST,  0x4, 0x0,              DSTABS16LIST),
  DO_MOVA2 (ABS32DST,  0x4, 0x8,              DSTABS32LIST),

  {O (O_MOV, SB), AV_H8, 10, "movfpe", {{ABS16SRC, RD8, E}}, {{0x6, 0xA, 0x4, RD8, ABS16SRC, DATA3, E}}},
  {O (O_MOV, SB), AV_H8, 10, "movtpe", {{RS8, ABS16DST, E}}, {{0x6, 0xA, 0xC, RS8, ABS16DST, DATA3, E}}},

  {O (O_MOVMD, SB), AV_H8SX, 0, "movmd.b", {{E}},         {{0x7, 0xb, 0x9, 0x4, E}}},
  {O (O_MOVMD, SW), AV_H8SX, 0, "movmd.w", {{E}},         {{0x7, 0xb, 0xa, 0x4, E}}},
  {O (O_MOVMD, SL), AV_H8SX, 0, "movmd.l", {{E}},         {{0x7, 0xb, 0xb, 0x4, E}}},
  {O (O_MOVSD, SB), AV_H8SX, 0, "movsd.b", {{PCREL16, E}}, {{0x7, 0xb, 0x8, 0x4, PCREL16, DATA3, E}}},

  {O (O_MULS, SW), AV_H8SX, 0, "muls.w", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xc, 0x6, 0x5, 0x0, IMM4, RD16, E}}},
  {O (O_MULS, SW), AV_H8SX, 0, "muls.w", {{RS16, RD16, E}}, {{0x0, 0x1, 0xc, 0x2, 0x5, 0x0, RS16, RD16, E}}},
  {O (O_MULS, SL), AV_H8SX, 0, "muls.l", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xc, 0x6, 0x5, 0x2, IMM4,       B30 | RD32, E}}},
  {O (O_MULS, SL), AV_H8SX, 0, "muls.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xc, 0x2, 0x5, 0x2, B30 | RS32, B30 | RD32, E}}},

  {O (O_MULU, SW), AV_H8SX, 0, "mulu.w", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xc, 0xe, 0x5, 0x0, IMM4, RD16, E}}},
  {O (O_MULU, SW), AV_H8SX, 0, "mulu.w", {{RS16, RD16, E}}, {{0x0, 0x1, 0xc, 0xa, 0x5, 0x0, RS16, RD16, E}}},
  {O (O_MULU, SL), AV_H8SX, 0, "mulu.l", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xc, 0xe, 0x5, 0x2, IMM4,       B30 | RD32, E}}},
  {O (O_MULU, SL), AV_H8SX, 0, "mulu.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xc, 0xa, 0x5, 0x2, B30 | RS32, B30 | RD32, E}}},

  {O (O_MULSU, SL), AV_H8SX, 0, "muls/u.l", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xc, 0x7, 0x5, 0x2, IMM4,       B30 | RD32, E}}},
  {O (O_MULSU, SL), AV_H8SX, 0, "muls/u.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xc, 0x3, 0x5, 0x2, B30 | RS32, B30 | RD32, E}}},
  {O (O_MULUU, SL), AV_H8SX, 0, "mulu/u.l", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xc, 0xf, 0x5, 0x2, IMM4,       B30 | RD32, E}}},
  {O (O_MULUU, SL), AV_H8SX, 0, "mulu/u.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xc, 0xb, 0x5, 0x2, B30 | RS32, B30 | RD32, E}}},

  {O (O_MULXS, SB), AV_H8SX, 0,  "mulxs.b", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xc, 0x4, 0x5, 0x0, IMM4, RD16,       E}}},
  {O (O_MULXS, SB), AV_H8H,  20, "mulxs.b", {{RS8,  RD16, E}}, {{0x0, 0x1, 0xc, 0x0, 0x5, 0x0, RS8,  RD16,       E}}},
  {O (O_MULXS, SW), AV_H8SX, 0,  "mulxs.w", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xc, 0x4, 0x5, 0x2, IMM4, B30 | RD32, E}}},
  {O (O_MULXS, SW), AV_H8H,  20, "mulxs.w", {{RS16, RD32, E}}, {{0x0, 0x1, 0xc, 0x0, 0x5, 0x2, RS16, B30 | RD32, E}}},

  {O (O_MULXU, SB), AV_H8SX, 0,  "mulxu.b", {{IMM4, RD16, E}}, {{0x0, 0x1, 0xc, 0xc, 0x5, 0x0, IMM4, RD16, E}}},
  {O (O_MULXU, SB), AV_H8,   14, "mulxu.b", {{RS8,  RD16, E}}, {{0x5, 0x0,                     RS8,  RD16, E}}},
  {O (O_MULXU, SW), AV_H8SX, 0,  "mulxu.w", {{IMM4, RD32, E}}, {{0x0, 0x1, 0xc, 0xc, 0x5, 0x2, IMM4, B30 | RD32, E}}},
  {O (O_MULXU, SW), AV_H8H,  14, "mulxu.w", {{RS16, RD32, E}}, {{0x5, 0x2,                     RS16, B30 | RD32, E}}},

  EXPAND_UNOP_STD_B (O (O_NEG,  SB), "neg.b", PREFIX_017, 0x1, 0x7, 0x8),
  EXPAND_UNOP_STD_W (O (O_NEG,  SW), "neg.w", PREFIX_015, 0x1, 0x7, 0x9),
  EXPAND_UNOP_STD_L (O (O_NEG,  SL), "neg.l", PREFIX_010, 0x1, 0x7, 0xb),

  {O (O_NOP, SN), AV_H8,  2, "nop",   {{E}},  {{0x0, 0x0, 0x0, 0x0, E}}},

  EXPAND_UNOP_STD_B (O (O_NOT,  SB), "not.b", PREFIX_017, 0x1, 0x7, 0x0),
  EXPAND_UNOP_STD_W (O (O_NOT,  SW), "not.w", PREFIX_015, 0x1, 0x7, 0x1),
  EXPAND_UNOP_STD_L (O (O_NOT,  SL), "not.l", PREFIX_010, 0x1, 0x7, 0x3),

  {O (O_OR,  SB), AV_H8,  2, "or.b",  {{IMM8, RD8,  E}}, {{0xc, RD8, IMM8LIST, E}}}, 
  EXPAND_TWOOP_B (O (O_OR, SB), "or.b", 0xc, 0x1, 0x4, 0x4, 0), 

  {O (O_OR,  SW), AV_H8,  2, "or.w",  {{RS16, RD16, E}}, {{0x6, 0x4, RS16, RD16, E}}}, 
  EXPAND_TWOOP_W (O (O_OR, SW), "or.w", 0x6, 0x4, 0x4), 

  {O (O_OR,  SL), AV_H8H, 2, "or.l",  {{RS32, RD32, E}}, {{0x0, 0x1, 0xF, 0x0, 0x6, 0x4, B30 | RS32, B30 | RD32, E}}},
  EXPAND_TWOOP_L (O (O_OR, SL), "or.l", 0x4), 

  {O (O_ORC, SB), AV_H8,  2, "orc", {{IMM8, CCR | DST, E}}, {{0x0, 0x4, IMM8LIST, E}}},
  {O (O_ORC, SB), AV_H8S, 2, "orc", {{IMM8, EXR | DST, E}}, {{0x0, 0x1, 0x4, EXR | DST, 0x0, 0x4, IMM8LIST, E}}},

  {O (O_MOV, SW), AV_H8,  6, "pop.w",  {{OR16, E}}, {{0x6, 0xD, 0x7, OR16, E}}},
  {O (O_MOV, SL), AV_H8H, 6, "pop.l",  {{OR32, E}}, {{PREFIX_0100, 0x6, 0xD, 0x7, OR32 | B30, E}}},
  {O (O_MOV, SW), AV_H8,  6, "push.w", {{OR16, E}}, {{0x6, 0xD, 0xF, OR16, E}}},
  {O (O_MOV, SL), AV_H8H, 6, "push.l", {{OR32, E}}, {{PREFIX_0100, 0x6, 0xD, 0xF, OR32 | B30, E}}},

  EXPAND_UNOP_STD_B      (O (O_ROTL,  SB), "rotl.b",           PREFIX_017, 0x1, 0x2, 0x8),
  EXPAND_UNOP_EXTENDED_B (O (O_ROTL,  SB), "rotl.b",  CONST_2, PREFIX_017, 0x1, 0x2, 0xc),
  EXPAND_UNOP_STD_W      (O (O_ROTL,  SW), "rotl.w",           PREFIX_015, 0x1, 0x2, 0x9),
  EXPAND_UNOP_EXTENDED_W (O (O_ROTL,  SW), "rotl.w",  CONST_2, PREFIX_015, 0x1, 0x2, 0xd),
  EXPAND_UNOP_STD_L      (O (O_ROTL,  SL), "rotl.l",           PREFIX_010, 0x1, 0x2, 0xb),
  EXPAND_UNOP_EXTENDED_L (O (O_ROTL,  SL), "rotl.l",  CONST_2, PREFIX_010, 0x1, 0x2, 0xf, B30),
  EXPAND_UNOP_STD_B      (O (O_ROTR,  SB), "rotr.b",           PREFIX_017, 0x1, 0x3, 0x8),
  EXPAND_UNOP_EXTENDED_B (O (O_ROTR,  SB), "rotr.b",  CONST_2, PREFIX_017, 0x1, 0x3, 0xc),
  EXPAND_UNOP_STD_W      (O (O_ROTR,  SW), "rotr.w",           PREFIX_015, 0x1, 0x3, 0x9),
  EXPAND_UNOP_EXTENDED_W (O (O_ROTR,  SW), "rotr.w",  CONST_2, PREFIX_015, 0x1, 0x3, 0xd),
  EXPAND_UNOP_STD_L      (O (O_ROTR,  SL), "rotr.l",           PREFIX_010, 0x1, 0x3, 0xb),
  EXPAND_UNOP_EXTENDED_L (O (O_ROTR,  SL), "rotr.l",  CONST_2, PREFIX_010, 0x1, 0x3, 0xf, B30),
  EXPAND_UNOP_STD_B      (O (O_ROTXL, SB), "rotxl.b",          PREFIX_017, 0x1, 0x2, 0x0),
  EXPAND_UNOP_EXTENDED_B (O (O_ROTXL, SB), "rotxl.b", CONST_2, PREFIX_017, 0x1, 0x2, 0x4),
  EXPAND_UNOP_STD_W      (O (O_ROTXL, SW), "rotxl.w",          PREFIX_015, 0x1, 0x2, 0x1),
  EXPAND_UNOP_EXTENDED_W (O (O_ROTXL, SW), "rotxl.w", CONST_2, PREFIX_015, 0x1, 0x2, 0x5),
  EXPAND_UNOP_STD_L      (O (O_ROTXL, SL), "rotxl.l",          PREFIX_010, 0x1, 0x2, 0x3),
  EXPAND_UNOP_EXTENDED_L (O (O_ROTXL, SL), "rotxl.l", CONST_2, PREFIX_010, 0x1, 0x2, 0x7, B30),
  EXPAND_UNOP_STD_B      (O (O_ROTXR, SB), "rotxr.b",          PREFIX_017, 0x1, 0x3, 0x0),
  EXPAND_UNOP_EXTENDED_B (O (O_ROTXR, SB), "rotxr.b", CONST_2, PREFIX_017, 0x1, 0x3, 0x4),
  EXPAND_UNOP_STD_W      (O (O_ROTXR, SW), "rotxr.w",          PREFIX_015, 0x1, 0x3, 0x1),
  EXPAND_UNOP_EXTENDED_W (O (O_ROTXR, SW), "rotxr.w", CONST_2, PREFIX_015, 0x1, 0x3, 0x5),
  EXPAND_UNOP_STD_L      (O (O_ROTXR, SL), "rotxr.l",          PREFIX_010, 0x1, 0x3, 0x3),
  EXPAND_UNOP_EXTENDED_L (O (O_ROTXR, SL), "rotxr.l", CONST_2, PREFIX_010, 0x1, 0x3, 0x7, B30),


  {O (O_BPT,  SN), AV_H8,  10, "bpt",   {{E}}, {{0x7, 0xA, 0xF, 0xF, E}}},
  {O (O_RTE,  SN), AV_H8,  10, "rte",   {{E}}, {{0x5, 0x6, 0x7, 0x0, E}}},
  {O (O_RTS,  SN), AV_H8,   8, "rts",   {{E}}, {{0x5, 0x4, 0x7, 0x0, E}}},
  {O (O_RTEL, SN), AV_H8SX, 0, "rte/l", {{RS32, RD32, E}}, {{0x5, 0x6, RS32 | B30, RD32 | B30, E}}},
  {O (O_RTSL, SN), AV_H8SX, 0, "rts/l", {{RS32, RD32, E}}, {{0x5, 0x4, RS32 | B30, RD32 | B30, E}}},

  EXPAND_UNOP_STD_B      (O (O_SHAL, SB), "shal.b",          PREFIX_017, 0x1, 0x0, 0x8),
  EXPAND_UNOP_EXTENDED_B (O (O_SHAL, SB), "shal.b", CONST_2, PREFIX_017, 0x1, 0x0, 0xc),
  EXPAND_UNOP_STD_W      (O (O_SHAL, SW), "shal.w",          PREFIX_015, 0x1, 0x0, 0x9),
  EXPAND_UNOP_EXTENDED_W (O (O_SHAL, SW), "shal.w", CONST_2, PREFIX_015, 0x1, 0x0, 0xd),
  EXPAND_UNOP_STD_L      (O (O_SHAL, SL), "shal.l",          PREFIX_010, 0x1, 0x0, 0xb),
  EXPAND_UNOP_EXTENDED_L (O (O_SHAL, SL), "shal.l", CONST_2, PREFIX_010, 0x1, 0x0, 0xf, B30),
  EXPAND_UNOP_STD_B      (O (O_SHAR, SB), "shar.b",          PREFIX_017, 0x1, 0x1, 0x8),
  EXPAND_UNOP_EXTENDED_B (O (O_SHAR, SB), "shar.b", CONST_2, PREFIX_017, 0x1, 0x1, 0xc),
  EXPAND_UNOP_STD_W      (O (O_SHAR, SW), "shar.w",          PREFIX_015, 0x1, 0x1, 0x9),
  EXPAND_UNOP_EXTENDED_W (O (O_SHAR, SW), "shar.w", CONST_2, PREFIX_015, 0x1, 0x1, 0xd),
  EXPAND_UNOP_STD_L      (O (O_SHAR, SL), "shar.l",          PREFIX_010, 0x1, 0x1, 0xb),
  EXPAND_UNOP_EXTENDED_L (O (O_SHAR, SL), "shar.l", CONST_2, PREFIX_010, 0x1, 0x1, 0xf, B30),

  EXPAND_UNOP_STD_B      (O (O_SHLL, SB), "shll.b",          PREFIX_017, 0x1, 0x0, 0x0),

  {O (O_SHLL, SB), AV_H8SX, 0, "shll.b", {{RS8,  RD8,  E}}, {{0x7, 0x8, RS8,        0x8,  0x1, 0x0, 0x0, RD8,  E}}},

  EXPAND_UNOP_EXTENDED_B (O (O_SHLL, SB), "shll.b", CONST_2, PREFIX_017, 0x1, 0x0, 0x4),
  EXPAND_UNOP_EXTENDED_B (O (O_SHLL, SB), "shll.b", CONST_4, PREFIX_017, 0x1, 0x0, 0xa),
  {O (O_SHLL, SB), AV_H8SX, 0, "shll.b", {{IMM5, RD8,  E}}, {{0x0, 0x3, B31 | IMM5, DATA, 0x1, 0x0, 0x0, RD8,  E}}},

  EXPAND_UNOP_STD_W      (O (O_SHLL, SW), "shll.w",          PREFIX_015, 0x1, 0x0, 0x1),

  {O (O_SHLL, SW), AV_H8SX, 0, "shll.w", {{RS8,  RD16, E}}, {{0x7, 0x8, RS8,        0x8,  0x1, 0x0, 0x1, RD16, E}}},

  EXPAND_UNOP_EXTENDED_W (O (O_SHLL, SW), "shll.w", CONST_2, PREFIX_015, 0x1, 0x0, 0x5),
  EXPAND_UNOP_EXTENDED_W (O (O_SHLL, SW), "shll.w", CONST_4, PREFIX_015, 0x1, 0x0, 0x2),
  EXPAND_UNOP_EXTENDED_W (O (O_SHLL, SW), "shll.w", CONST_8, PREFIX_015, 0x1, 0x0, 0x6),
  {O (O_SHLL, SW), AV_H8SX, 0, "shll.w", {{IMM5, RD16, E}}, {{0x0, 0x3, B31 | IMM5, DATA, 0x1, 0x0, 0x1, RD16, E}}},

  EXPAND_UNOP_STD_L      (O (O_SHLL, SL), "shll.l",           PREFIX_010, 0x1, 0x0, 0x3),

  {O (O_SHLL, SL), AV_H8SX, 0, "shll.l", {{RS8,  RD32, E}}, {{0x7, 0x8, RS8,        0x8,  0x1, 0x0, 0x3, B30 | RD32, E}}},

  EXPAND_UNOP_EXTENDED_L (O (O_SHLL, SL), "shll.l", CONST_2,  PREFIX_010, 0x1, 0x0, 0x7, B30),
  EXPAND_UNOP_EXTENDED_L (O (O_SHLL, SL), "shll.l", CONST_4,  PREFIX_010, 0x1, 0x0, 0x3, B31),
  EXPAND_UNOP_EXTENDED_L (O (O_SHLL, SL), "shll.l", CONST_8,  PREFIX_010, 0x1, 0x0, 0x7, B31),
  EXPAND_UNOP_EXTENDED_L (O (O_SHLL, SL), "shll.l", CONST_16, PREFIX_010, 0x1, 0x0, 0xf, B31),
  {O (O_SHLL, SL), AV_H8SX, 0, "shll.l", {{IMM5, RD32, E}}, {{0x0, 0x3, B31 | IMM5, DATA, 0x1, 0x0, 0x3, B30 | RD32, E}}},

  EXPAND_UNOP_STD_B      (O (O_SHLR, SB), "shlr.b",          PREFIX_017, 0x1, 0x1, 0x0),

  {O (O_SHLR, SB), AV_H8SX, 0, "shlr.b", {{RS8,  RD8,  E}}, {{0x7, 0x8, RS8,        0x8,  0x1, 0x1, 0x0, RD8,  E}}},

  EXPAND_UNOP_EXTENDED_B (O (O_SHLR, SB), "shlr.b", CONST_2, PREFIX_017, 0x1, 0x1, 0x4),
  EXPAND_UNOP_EXTENDED_B (O (O_SHLR, SB), "shlr.b", CONST_4, PREFIX_017, 0x1, 0x1, 0xa),
  {O (O_SHLR, SB), AV_H8SX, 0, "shlr.b", {{IMM5, RD8,  E}}, {{0x0, 0x3, B31 | IMM5, DATA, 0x1, 0x1, 0x0, RD8,  E}}},

  EXPAND_UNOP_STD_W      (O (O_SHLR, SW), "shlr.w",          PREFIX_015, 0x1, 0x1, 0x1),

  {O (O_SHLR, SW), AV_H8SX, 0, "shlr.w", {{RS8,  RD16, E}}, {{0x7, 0x8, RS8,        0x8,  0x1, 0x1, 0x1, RD16, E}}},

  EXPAND_UNOP_EXTENDED_W (O (O_SHLR, SW), "shlr.w", CONST_2, PREFIX_015, 0x1, 0x1, 0x5),
  EXPAND_UNOP_EXTENDED_W (O (O_SHLR, SW), "shlr.w", CONST_4, PREFIX_015, 0x1, 0x1, 0x2),
  EXPAND_UNOP_EXTENDED_W (O (O_SHLR, SW), "shlr.w", CONST_8, PREFIX_015, 0x1, 0x1, 0x6),
  {O (O_SHLR, SW), AV_H8SX, 0, "shlr.w", {{IMM5, RD16, E}}, {{0x0, 0x3, B31 | IMM5, DATA, 0x1, 0x1, 0x1, RD16, E}}},

  EXPAND_UNOP_STD_L      (O (O_SHLR, SL), "shlr.l",           PREFIX_010, 0x1, 0x1, 0x3),

  {O (O_SHLR, SL), AV_H8SX, 0, "shlr.l", {{RS8,  RD32, E}}, {{0x7, 0x8, RS8,        0x8,  0x1, 0x1, 0x3, B30 | RD32, E}}},

  EXPAND_UNOP_EXTENDED_L (O (O_SHLR, SL), "shlr.l", CONST_2,  PREFIX_010, 0x1, 0x1, 0x7, B30),
  EXPAND_UNOP_EXTENDED_L (O (O_SHLR, SL), "shlr.l", CONST_4,  PREFIX_010, 0x1, 0x1, 0x3, B31),
  EXPAND_UNOP_EXTENDED_L (O (O_SHLR, SL), "shlr.l", CONST_8,  PREFIX_010, 0x1, 0x1, 0x7, B31),
  EXPAND_UNOP_EXTENDED_L (O (O_SHLR, SL), "shlr.l", CONST_16, PREFIX_010, 0x1, 0x1, 0xf, B31),
  {O (O_SHLR, SL), AV_H8SX, 0, "shlr.l", {{IMM5, RD32, E}}, {{0x0, 0x3, B31 | IMM5, DATA, 0x1, 0x1, 0x3, B30 | RD32, E}}},

  {O (O_SLEEP, SN), AV_H8, 2, "sleep", {{E}}, {{0x0, 0x1, 0x8, 0x0, E}}},

  {O (O_STC, SB), AV_H8,   2, "stc", {{CCR | SRC, RD8,            E}}, {{0x0, 0x2, B30 | CCR | SRC,           RD8,    E}}},
  {O (O_STC, SB), AV_H8S,  2, "stc", {{EXR | SRC, RD8,            E}}, {{0x0, 0x2, B30 | EXR | SRC,           RD8,    E}}},
  {O (O_STC, SW), AV_H8H,  2, "stc", {{CCR | SRC, RDIND,          E}}, {{PREFIXSTC, 0x6, 0x9, B31 | RDIND,    IGNORE, E}}},
  {O (O_STC, SW), AV_H8S,  2, "stc", {{EXR | SRC, RDIND,          E}}, {{PREFIXSTC, 0x6, 0x9, B31 | RDIND,    IGNORE, E}}},
  {O (O_STC, SW), AV_H8H,  2, "stc", {{CCR | SRC, RDPREDEC,       E}}, {{PREFIXSTC, 0x6, 0xD, B31 | RDPREDEC, IGNORE, E}}},
  {O (O_STC, SW), AV_H8S,  2, "stc", {{EXR | SRC, RDPREDEC,       E}}, {{PREFIXSTC, 0x6, 0xD, B31 | RDPREDEC, IGNORE, E}}},
  {O (O_STC, SW), AV_H8H,  2, "stc", {{CCR | SRC, DISP16DST,      E}}, {{PREFIXSTC, 0x6, 0xF, B31 | DSTDISPREG,                   IGNORE, DSTDISP16LIST, E}}},
  {O (O_STC, SW), AV_H8S,  2, "stc", {{EXR | SRC, DISP16DST,      E}}, {{PREFIXSTC, 0x6, 0xF, B31 | DSTDISPREG,                   IGNORE, DSTDISP16LIST, E}}},
  {O (O_STC, SW), AV_H8H,  2, "stc", {{CCR | SRC, DISP32DST,      E}}, {{PREFIXSTC, 0x7, 0x8, B30 | DSTDISPREG, 0, 0x6, 0xB, 0xA, IGNORE, DSTDISP32LIST, E}}},
  {O (O_STC, SW), AV_H8S,  2, "stc", {{EXR | SRC, DISP32DST,      E}}, {{PREFIXSTC, 0x7, 0x8, B30 | DSTDISPREG, 0, 0x6, 0xB, 0xA, IGNORE, DSTDISP32LIST, E}}},
  {O (O_STC, SW), AV_H8H,  2, "stc", {{CCR | SRC, ABS16DST,       E}}, {{PREFIXSTC, 0x6, 0xB, 0x8, IGNORE, DST | ABS16LIST, E}}},
  {O (O_STC, SW), AV_H8S,  2, "stc", {{EXR | SRC, ABS16DST,       E}}, {{PREFIXSTC, 0x6, 0xB, 0x8, IGNORE, DST | ABS16LIST, E}}},
  {O (O_STC, SW), AV_H8H,  2, "stc", {{CCR | SRC, ABS32DST,       E}}, {{PREFIXSTC, 0x6, 0xB, 0xA, IGNORE, DST | MEMRELAX | ABS32LIST, E}}},
  {O (O_STC, SW), AV_H8S,  2, "stc", {{EXR | SRC, ABS32DST,       E}}, {{PREFIXSTC, 0x6, 0xB, 0xA, IGNORE, DST | MEMRELAX | ABS32LIST, E}}},
  {O (O_STC, SL), AV_H8SX, 0, "stc", {{B30 | VBR_SBR | SRC, RD32, E}}, {{0x0, 0x2, B30 | VBR_SBR | SRC, RD32, E}}},


  EXPAND_TWOOP_B (O (O_SUB, SB), "sub.b", 0xa, 0x1, 0x8, 0x3, B01), 

  {O (O_SUB, SW), AV_H8,   2, "sub.w", {{RS16,      RD16,     E}}, {{0x1, 0x9, RS16,         RD16,   E}}}, 
  {O (O_SUB, SW), AV_H8SX, 0, "sub.w", {{IMM3NZ_NS, RD16,     E}}, {{0x1, 0xa, B30 | IMM3NZ, RD16,   E}}}, 
  {O (O_SUB, SW), AV_H8SX, 0, "sub.w", {{IMM3NZ_NS, RDIND,    E}}, {{0x7, 0xd,      B31 | RDIND,  IGNORE,       0x1, 0xa, B30 | IMM3NZ, IGNORE, E}}}, 
  {O (O_SUB, SW), AV_H8SX, 0, "sub.w", {{IMM3NZ_NS, ABS16DST, E}}, {{0x6, 0xb, 0x1, B31 | IGNORE, DSTABS16LIST, 0x1, 0xa, B30 | IMM3NZ, IGNORE, E}}}, 
  {O (O_SUB, SW), AV_H8SX, 0, "sub.w", {{IMM3NZ_NS, ABS32DST, E}}, {{0x6, 0xb, 0x3, B31 | IGNORE, DSTABS32LIST, 0x1, 0xa, B30 | IMM3NZ, IGNORE, E}}}, 
  EXPAND_TWOOP_W (O (O_SUB, SW), "sub.w", 0x1, 0x9, 0x3), 

  {O (O_SUB, SL), AV_H8H,  6, "sub.l", {{RS32,      RD32,     E}}, {{0x1, 0xa, B31 | RS32,   B30 | RD32, E}}}, 
  {O (O_SUB, SL), AV_H8SX, 0, "sub.l", {{IMM3NZ_NS, RD32,     E}}, {{0x1, 0xa, B31 | IMM3NZ, B31 | RD32, E}}}, 
  EXPAND_TWOOP_L (O (O_SUB, SL), "sub.l", 0x3), 

  {O (O_SUBS, SL), AV_H8, 2, "subs", {{KBIT, RDP, E}}, {{0x1, 0xB,KBIT, RDP, E}}},

  {O (O_SUBX, SB), AV_H8,   2, "subx",   {{IMM8,      RD8,       E}}, {{0xb, RD8, IMM8LIST, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{IMM8,      RDIND,     E}}, {{0x7, 0xd, B30 | RDIND, IGNORE,                        0xb, IGNORE, IMM8LIST, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{IMM8,      RDPOSTDEC, E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RDPOSTDEC, B31 | IGNORE, 0xb, IGNORE, IMM8LIST, E}}},
  {O (O_SUBX, SB), AV_H8,   2, "subx",   {{RS8,       RD8,       E}}, {{0x1, 0xe, RS8,  RD8,    E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{RS8,       RDIND,     E}}, {{0x7, 0xd, B30 | RDIND, IGNORE,                              0x1, 0xe, RS8, IGNORE, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{RS8,       RDPOSTDEC, E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RDPOSTDEC, B31       | IGNORE, 0x1, 0xe, RS8, IGNORE, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{RSIND,     RD8,       E}}, {{0x7, 0xc, B30 | RSIND, IGNORE,                              0x1, 0xe, IGNORE, RD8, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{RSPOSTDEC, RD8,       E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RSPOSTDEC, B30 | B20 | IGNORE, 0x1, 0xe, IGNORE, RD8, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{RSIND,     RDIND,     E}}, {{PREFIX_0174, 0x6, 0x8, B30 | RSIND,     0xd, 0x0, RDIND,     0x3, IGNORE, E}}},
  {O (O_SUBX, SB), AV_H8SX, 0, "subx.b", {{RSPOSTDEC, RDPOSTDEC, E}}, {{PREFIX_0176, 0x6, 0xc, B30 | RSPOSTDEC, 0xd, 0xa, RDPOSTDEC, 0x3, IGNORE, E}}},

  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{IMM16,     RD16,      E}}, {{PREFIX_0151,                         0x7, 0x9, 0x3, RD16,   IMM16LIST, E}}}, 
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{IMM16,     RDIND,     E}}, {{0x7, 0xd, B31 | RDIND, B01 | IGNORE, 0x7, 0x9, 0x3, IGNORE, IMM16LIST, E}}}, 
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{IMM16,     RDPOSTDEC, E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x7, 0x9, 0x3, IGNORE, IMM16LIST, E}}}, 
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RS16,      RD16,      E}}, {{PREFIX_0151, 0x1, 0x9, RS16,  RD16,    E}}},
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RS16,      RDIND,     E}}, {{0x7, 0xd, B31 | RDIND, B01 | IGNORE, 0x1, 0x9, RS16, IGNORE, E}}},
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RS16,      RDPOSTDEC, E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x1, 0x9, RS16, IGNORE, E}}}, 
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RSIND,     RD16,      E}}, {{0x7, 0xc, B31 | RSIND, B01 | IGNORE, 0x1, 0x9, IGNORE, RD16, E}}},
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RSPOSTDEC, RD16,      E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RSPOSTDEC, B30 | B20 | B01 | IGNORE, 0x1, 0x9, IGNORE, RD16, E}}},
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RSIND,     RDIND,     E}}, {{PREFIX_0154, 0x6, 0x9, B30 | RSIND,     0xd, 0x0, RDIND,     0x3, IGNORE, E}}},
  {O (O_SUBX, SW), AV_H8SX, 0, "subx.w", {{RSPOSTDEC, RDPOSTDEC, E}}, {{PREFIX_0156, 0x6, 0xd, B30 | RSPOSTDEC, 0xd, 0xa, RDPOSTDEC, 0x3, IGNORE, E}}},

  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{IMM32,     RD32,      E}}, {{PREFIX_0101, 0x7, 0xa, 0x3,  RD32, IMM32LIST, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{IMM32,     RDIND,     E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RDIND,     B31 | B20 | B01 | IGNORE, 0x7, 0xa, 0x3, IGNORE, IMM32LIST, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{IMM32,     RDPOSTDEC, E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x7, 0xa, 0x3, IGNORE, IMM32LIST, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RS32,      RD32,      E}}, {{PREFIX_0101, 0x1, 0xa, B31 | RS32,  B30 | RD32,    E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RS32,      RDIND,     E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RDIND,     B31 | B20 | B01 | IGNORE, 0x1, 0xa, B31 | RS32, B30 | IGNORE, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RS32,      RDPOSTDEC, E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RDPOSTDEC, B31 | B20 | B01 | IGNORE, 0x1, 0xa, B31 | RS32, B30 | IGNORE, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RSIND,     RD32,      E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RSIND,     B30 | B20 | B01 | IGNORE, 0x1, 0xa, B31 | IGNORE, B30 | RD32, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RSPOSTDEC, RD32,      E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RSPOSTDEC, B30 | B20 | B01 | IGNORE, 0x1, 0xa, B31 | IGNORE, B30 | RD32, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RSIND,     RDIND,     E}}, {{PREFIX_0104, 0x6, 0x9, B30 | RSIND,     0xd, 0x0, RDIND,     0x3, IGNORE, E}}},
  {O (O_SUBX, SL), AV_H8SX, 0, "subx.l", {{RSPOSTDEC, RDPOSTDEC, E}}, {{PREFIX_0106, 0x6, 0xd, B30 | RSPOSTDEC, 0xd, 0xa, RDPOSTDEC, 0x3, IGNORE, E}}},

  {O (O_TRAPA, SB), AV_H8H, 2, "trapa", {{IMM2,  E}}, {{0x5, 0x7, IMM2, IGNORE, E}}},
  {O (O_TAS,   SB), AV_H8H, 2, "tas",   {{RSIND, E}}, {{0x0, 0x1, 0xe, 0x0, 0x7, 0xb, B30 | RSIND, 0xc, E}}},

  {O (O_XOR,   SB), AV_H8,  2, "xor.b", {{IMM8, RD8,  E}}, {{0xd, RD8, IMM8LIST, E}}}, 
  EXPAND_TWOOP_B (O (O_XOR, SB), "xor.b", 0xd, 0x1, 0x5, 0x5, 0), 

  {O (O_XOR,   SW), AV_H8,  2, "xor.w", {{RS16, RD16, E}}, {{0x6, 0x5, RS16, RD16, E}}}, 
  EXPAND_TWOOP_W (O (O_XOR, SW), "xor.w", 0x6, 0x5, 0x5), 

  {O (O_XOR,   SL), AV_H8H, 2, "xor.l", {{RS32, RD32, E}}, {{0x0, 0x1, 0xF, 0x0, 0x6, 0x5, B30 | RS32, B30 | RD32, E}}},
  EXPAND_TWOOP_L (O (O_XOR, SL), "xor.l", 0x5), 

  {O (O_XORC, SB), AV_H8,  2, "xorc", {{IMM8, CCR | DST, E}}, {{0x0, 0x5, IMM8LIST, E}}},
  {O (O_XORC, SB), AV_H8S, 2, "xorc", {{IMM8, EXR | DST, E}}, {{0x0, 0x1, 0x4, EXR | DST, 0x0, 0x5, IMM8LIST, E}}},

  {O (O_CLRMAC, SN), AV_H8S, 2, "clrmac", {{E}}, {{0x0, 0x1, 0xa, 0x0, E}}},
  {O (O_MAC,    SW), AV_H8S, 2, "mac",    {{RSPOSTINC, RDPOSTINC, E}}, {{0x0, 0x1, 0x6, 0x0, 0x6, 0xd, B30 | RSPOSTINC, B30 | RDPOSTINC, E}}},
  {O (O_LDMAC,  SL), AV_H8S, 2, "ldmac",  {{RS32, MD32, E}}, {{0x0, 0x3, MD32, RS32, E}}},
  {O (O_STMAC,  SL), AV_H8S, 2, "stmac",  {{MS32, RD32, E}}, {{0x0, 0x2, MS32, RD32, E}}},
  {O (O_LDM,    SL), AV_H8H, 6, "ldm.l",  {{RSPOSTINC, RD32, E}}, {{0x0, 0x1, DATA, 0x0, 0x6, 0xD, 0x7, B30 | RD32, E}}},
  {O (O_STM,    SL), AV_H8H, 6, "stm.l",  {{RS32, RDPREDEC,  E}}, {{0x0, 0x1, DATA, 0x0, 0x6, 0xD, 0xF, B30 | RS32, E}}},
  {0, 0, 0, NULL, {{0, 0, 0}}, {{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}}
};
#else
extern const struct h8_opcode h8_opcodes[];
#endif

