;;;; ange-ftp-dired.el - ange-ftp support for tree dired
;;;; $Id: ange-ftp-dired.el,v 1.3 90/11/23 12:39:46 sk Exp $

;; provide for remote ls:

(defun ange-ftp-dired-ls (file &optional switches wildcard full-directory-p)
  "Insert ls output of FILE, optionally formatted with SWITCHES.
Optional third arg WILDCARD means treat FILE as shell wildcard.
Optional fourth arg FULL-DIRECTORY-P means file is a directory and
switches do not contain \"d\".

SWITCHES default to dired-listing-switches.
Uses dired-ls-program and maybe shell-file-name to do the work.

This is an emulation of dired-ls with support for remote files using ftp."
  (setq switches (or switches dired-listing-switches))
  (let ((parsed (ange-ftp-ftp-path file)))
    (if parsed
	(insert
	 (if wildcard
	     ;; Prevent ls from inserting subdirs, as the
	     ;; subdir header line format would be wrong
	     ;; (it would have no "/user@host:" prefix)
	     (ange-ftp-ls file (concat switches "d"))
	   ;; quoting the path part of the file name seems to be a
	   ;; good idea (using dired.el's shell-quote function), but
	   ;; ftpd always globs ls args before passing them to
	   ;; /bin/ls or even doing the ls formatting itself.
	   ;; --> So wildcard characters in FILE lose.  Sigh...
	   (ange-ftp-ls file switches)))
      (ange-ftp-real-dired-ls file switches wildcard))))

(defvar ange-ftp-remote-shell-file-name
  (if (memq system-type '(hpux usg-unix-v)) ; hope that's right
      "remsh"
    "rsh")
  "Remote shell used by ange-ftp.")

(defun ange-ftp-dired-run-shell-command (command &optional in-background)
  "Run shell COMMAND, optionally IN-BACKGROUND.
If COMMAND is longer than shell-maximum-command-length, you are asked
for confirmation.

This command has support for remote files using a remote shell as
specified in ange-ftp-remote-shell-file-name."
  (let* ((parsed (ange-ftp-ftp-path default-directory))
	 (host (nth 0 parsed))
	 (user (nth 1 parsed))
	 (path (nth 2 parsed)))
    (if (not parsed)
	(ange-ftp-real-dired-run-shell-command command in-background)
      (if (> (length path) 0)		; else it's $HOME
	  (setq command (concat "cd " path "; " command)))
      (setq command
	    (format  "%s %s \"%s\""	; remsh -l USER does not work well
					; on a hp-ux machine I tried
		     ange-ftp-remote-shell-file-name host command))
      (message "Remote command '%s' ..." command)
      ;; Cannot call ange-ftp-real-dired-run-shell-command here as it
      ;; would prepend "cd default-directory" --- which bombs because
      ;; default-directory is in ange-ftp syntax for remote path names.
      (if (or (and shell-maximum-command-length
		   (< (length command) shell-maximum-command-length))
	      (yes-or-no-p
	       (format
		"Dired shell command is %d bytes long - execute anyway? "
		(length command))))
	  (if in-background
	      (background command)
	    (shell-command command))))))

