;; -*-Emacs-Lisp-*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; File:         ange-ftp.el
;; RCS:          $Header: /home/sk/lib/emacs/lisp/RCS/ange-ftp.el,v 3.72 1991/01/11 10:56:04 ange Exp $
;; Description:  simple ftp access to files from GNU Emacs
;; Author:       Andy Norman, ange@hplb.hpl.hp.com
;; Created:      Thu Oct 12 14:00:05 1989
;; Modified:     Fri Jan 11 10:54:38 1991 (Ange) ange@anorman
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (C) 1990 Andy Norman.
;;;
;;; Author: Andy Norman (ange@hplb.hpl.hp.com)
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 1, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; A copy of the GNU General Public License can be obtained from this
;;; program's author (send electronic mail to ange@hplb.hpl.hp.com) or from
;;; the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
;;; 02139, USA.

;;; This package attempts to make accessing files / directories using ftp from
;;; within GNU Emacs as simple as possible.  A subset of the normal
;;; file-handling routines are extended to understand ftp.
;;;
;;; To read or write a file using ftp, or to read a directory using ftp, the
;;; only thing that a user needs to do is to specify the filename using a 
;;; slighly extended syntax.
;;;
;;; The default syntax of ftp files is /user@host:path.  This is customizable.
;;; See the variable ange-ftp-path-format for more details.
;;;
;;; A password is required for each host/user pair.  This will be prompted for
;;; when needed, unless already set by calling ange-ftp-set-passwd, or
;;; specified in a valid ~/.netrc file.
;;;
;;; Ftp processes are left running for speed.  They can easily be killed by
;;; killing their associated buffers.
;;;
;;; Full file name completion is supported on remote files.
;;;
;;; File transfers can be done in binary mode. See the documentation for the
;;; variable ange-ftp-binary-file-name-regexp for more details.
;;;
;;; The ftp process can be either be run locally, or run on a different machine.
;;; Sometimes this is neccessary when the local machine does not have full internet
;;; access.  See the documentation for the variables ange-ftp-gateway-host,
;;; ange-ftp-local-host-regexp, ange-ftp-gateway-tmp-name-template, 
;;; ange-ftp-gateway-program and ange-ftp-gateway-program-interactive for more
;;; details.
;;;
;;; WARNING, the following GNU Emacs functions are replaced by this program:
;;;
;;;   write-region
;;;   insert-file-contents
;;;   dired-readin
;;;   delete-file
;;;   read-file-name-internal
;;;   verify-visited-file-modtime
;;;   directory-files
;;;   backup-buffer
;;;   file-directory-p
;;;   file-writable-p
;;;   file-exists-p
;;;   file-readable-p
;;;   file-attributes
;;;   copy-file
;;;   file-name-as-directory
;;;   file-name-directory
;;;   file-name-nondirectory
;;;   directory-file-name
;;;   expand-file-name
;;;
;;; If you find any bugs or problems with this package, please e-mail the
;;; above author.  Constructive comments are especially welcome.
;;;
;;; Many thanks to Roland McGrath <roland@ai.mit.edu> for improving the
;;; filename syntax handling, for suggesting many enhancements and for
;;; numerous cleanups to the code.
;;;
;;; Thanks to Jamie Zawinski <jwz@lucid.com> for bugfixes and for ideas such
;;; as gateways.
;;;
;;; Thanks to Ken Laprade <laprade@trantor.harris-atd.com> for improved .netrc
;;; parsing and password reading.
;;;
;;; Finally, thanks to Keith Waclena, Mark D. Baushke, Terence Kelleher, Ping
;;; Zhou, Edward Vielmetti, Jack Repenning, Mike Balenger, Todd Kaufmann and
;;; many others whose names I've forgotten who have helped to debug and fix
;;; problems in ange-ftp.el.

;;;; ------------------------------------------------------------
;;;; User customization variables.
;;;; ------------------------------------------------------------

(defvar ange-ftp-path-format
  '("^/\\(\\([^@/:]*\\)@\\)?\\([^@/:]*\\):\\(.*\\)" . (3 2 4))
  "*Format of a fully expanded remote pathname.  This is a cons
\(REGEXP . \(HOST USER PATH\)\), where REGEXP is a regular expression matching
the full remote pathname, and HOST, USER, and PATH are the numbers of
parenthesized expressions in REGEXP for the components (in that order).")

(defvar ange-ftp-multi-msgs
  "^[0-9][0-9][0-9]-"
  "*Regular expression matching messages from the ftp process that start
a multiline reply.")

(defvar ange-ftp-good-msgs
  "^220 \\|^230 \\|^226 \\|^25. \\|^221 \\|^200 "
  "*Regular expression matching messages from the ftp process that indicate
that the action that was initiated has completed successfully.")

(defvar ange-ftp-skip-msgs
  (concat "^200 PORT \\|^331 \\|^2.0-\\|^150 \\|^[0-9]+ bytes \\|"
	  "^Connected \\|^$\\|^Remote system\\|^Using\\|^ \\|Password:\\|"
	  "^local:")
  "*Regular expression matching messages from the ftp process that can be
ignored.")

(defvar ange-ftp-fatal-msgs
  (concat "^ftp: \\|^Not connected\\|^530 \\|^421 \\|rcmd: \\|"
	  "^No control connection")
  "*Regular expression matching messages from the ftp process that indicate
something has gone drastically wrong attempting the action that was
initiated.")

(defvar ange-ftp-ls-follow-symbolic-links t
  "*If non-nil, tell ls to always follow symbolic links.")

(defvar ange-ftp-tmp-name-template "/tmp/ange-ftp"
  "*Template given to make-temp-name to create temporary files.")

(defvar ange-ftp-gateway-tmp-name-template "/tmp/ange-ftp"
  "*Template given to make-temp-name to create temporary files when
ftp-ing through a gateway.  Files starting with this prefix need to
be accessible from BOTH the local machine and the gateway machine, 
and need to have the SAME name on both machines, that is, /tmp is probably
NOT what you want, since that is rarely cross-mounted.")

(defvar ange-ftp-copy-tmp-name-template "/tmp/ange-ftp-copy"
  "*Template given to make-temp-name to to create temporary files when
copying files between one remote machine and another.
This should be different from \`ange-ftp-tmp-name-template\' and
\'ange-ftp-gateway-tmp-name-template\'.")

(defvar ange-ftp-netrc-filename "~/.netrc"
  "*File in .netrc format to search for passwords.")

(defvar ange-ftp-default-user nil
  "*User name to use when none is specied in a pathname.
If nil, then the name under which the user is logged in is used.
If non-nil but not a string, the user is prompted for the name.")

(defvar ange-ftp-generate-anonymous-password nil
  "*If non-nil, by default use a password of user@host when logging
in as the anonymous user.")

(defvar ange-ftp-dumb-host-regexp nil
  "*If non-nil, if the host being ftp'd to matches this regexp then the ftp
process uses the \'dir\' command to get directory information.")

(defvar ange-ftp-binary-file-name-regexp
  (concat "\\.Z$\\|\\.lzh$\\|\\.arc$\\|\\.zip$\\|\\.zoo$\\|\\.tar$\\|"
	  "\\.dvi$\\|\\.ps$\\|\\.elc$")
  "*If a file matches this regexp then it is transferred in binary mode.")

(defvar ange-ftp-gateway-host nil
  "*Name of host to use as gateway machine when local ftp isn't possible.")

(defvar ange-ftp-local-host-regexp ".*"
  "*If a host being ftp'd to matches this regexp then the ftp process is started
locally, otherwise the ftp process is started on \`ange-ftp-gateway-host\'
instead.")

(defvar ange-ftp-gateway-program-interactive nil
  "*If non-nil then the gateway program is expected to connect to the gateway
machine and eventually give a shell prompt.  Both telnet and rlogin do something
like this.")

(defvar ange-ftp-gateway-program (if (eq system-type 'hpux) "remsh" "rsh")
  "*Name of program to spawn a shell on the gateway machine.  Valid candidates
are remsh (rsh on hp-ux), telnet and rlogin.  See also the gateway variable
above.")

(defvar ange-ftp-gateway-prompt-pattern "^[^#$%>;]*[#$%>;] *"
  "*Regexp used to detect that the logging-in sequence is completed on the
gateway machine and that the shell is now awaiting input.")

(defvar ange-ftp-gateway-setup-term-command
  (if (eq system-type 'hpux)
      "stty -onlcr -echo\n"
    "stty -echo nl\n")
  "*Command to use after logging in to the gateway machine to stop the terminal
echoing each command and to strip out trailing ^M characters.")

(defvar ange-ftp-smart-gateway nil
  "*If the gateway ftp is smart enough to use proxy server, then don't bother
telnetting etc, just issue a user@host command instead.")

(defvar ange-ftp-smart-gateway-port "21"
  "*Port on gateway machine to use when smart gateway is in operation.")

;;;; ------------------------------------------------------------
;;;; Hash table support.
;;;; ------------------------------------------------------------

(defun ange-ftp-make-hashtable (&optional size)
  "Make an obarray suitable for use as a hashtable.
SIZE, if supplied, should be a prime number."
  (make-vector (or size 511) 0))

(defun ange-ftp-map-hashtable (fun tbl)
  "Call FUNCTION on each key and value in HASHTABLE."
  (mapatoms
   (function 
    (lambda (sym)
      (and (get sym 'active)
	   (funcall fun (get sym 'key) (get sym 'val)))))
   tbl))

(defmacro ange-ftp-make-hash-key (key)
  "Convert KEY into a suitable key for a hashtable."
  (` (if (stringp (, key))
	 (, key)
       (prin1-to-string (, key)))))

(defun ange-ftp-get-hash-entry (key tbl)
  "Return the value associated with KEY in HASHTABLE."
  (let ((sym (intern-soft (ange-ftp-make-hash-key key) tbl)))
    (and sym
	 (get sym 'active)
	 (get sym 'val))))

(defun ange-ftp-put-hash-entry (key val tbl)
  "Record an association between KEY and VALUE in HASHTABLE."
  (let ((sym (intern (ange-ftp-make-hash-key key) tbl)))
    (put sym 'val val)
    (put sym 'key key)
    (put sym 'active t)))

(defun ange-ftp-del-hash-entry (key tbl)
  "Delete KEY from HASHTABLE."
  (let ((sym (intern-soft (ange-ftp-make-hash-key key) tbl)))
    (and sym (put sym 'active nil))))

(defun ange-ftp-hash-entry-exists-p (key tbl)
  "Return whether there is an association for KEY in TABLE."
  (let ((sym (intern-soft (ange-ftp-make-hash-key key) tbl)))
    (and sym (get sym 'active))))

(defun ange-ftp-hash-table-keys (tbl)
  "Return a sorted list of all the active keys in the hashtable, as strings."
  (sort (all-completions ""
			 tbl
			 (function (lambda (x) (get x 'active))))
	(function string-lessp)))

;;;; ------------------------------------------------------------
;;;; Internal variables.
;;;; ------------------------------------------------------------

(defvar ange-ftp-data-buffer-name "*ftp data*"
  "Buffer name to hold data received from ftp process.")

(defvar ange-ftp-process-string ""
  "Currently unprocessed output from the ftp process.")

(defvar ange-ftp-process-running nil
  "Boolean indicates whether the ftp process is currently handling
an action.")

(defvar ange-ftp-process-status nil
  "Set to t if an action sent to the ftp process succeeds.")

(defvar ange-ftp-process-multi-skip nil
  "Set to t while skipping a multi-line reply.")

(defvar ange-ftp-have-read-netrc nil
  "Boolean indicating whether the user's .netrc file has been read yet.")

(defvar ange-ftp-user-hashtable (ange-ftp-make-hashtable)
  "Hash table holding associations between HOST, USER pairs.")

(defvar ange-ftp-passwd-hashtable (ange-ftp-make-hashtable)
  "Mapping between a HOST, USER pair and a PASSWORD for it.")

(defvar ange-ftp-files-hashtable (ange-ftp-make-hashtable)
  "Hash table for storing directories and their respective files.")

;;;; ------------------------------------------------------------
;;;; Password support.
;;;; ------------------------------------------------------------

(defun ange-ftp-read-passwd (prompt &optional default)
  "Read a password from the user. Echos a . for each character typed.
End with RET, LFD, or ESC. DEL or C-h rubs out.  ^U kills line.
Optional DEFAULT is password to start with."
  (let ((pass (if default default ""))
	(c 0)
	(echo-keystrokes 0)
	(cursor-in-echo-area t))
    (while (and (/= c ?\r) (/= c ?\n) (/= c ?\e))
      (message "%s%s"
	       prompt
	       (make-string (length pass) ?.))
      (setq c (read-char))
      (if (= c ?\C-u)
	  (setq pass "")
	(if (and (/= c ?\b) (/= c ?\177))
	    (setq pass (concat pass (char-to-string c)))
	  (if (> (length pass) 0)
	      (setq pass (substring pass 0 -1))))))
    (message "")
    (substring pass 0 -1)))

(defun ange-ftp-set-user (host user)
  "For a given HOST, set or change the default USER."
  (interactive "sHost: \nsUser: ")
  (ange-ftp-put-hash-entry host user ange-ftp-user-hashtable))

(defun ange-ftp-get-user (host)
  "Given a HOST, return the default USER."
  (or ange-ftp-have-read-netrc (ange-ftp-parse-netrc))
  (let ((user (ange-ftp-get-hash-entry host ange-ftp-user-hashtable)))
    (or user
	(cond ((stringp ange-ftp-default-user)
	       ;; We have a default name.  Use it.
	       ange-ftp-default-user)
	      (ange-ftp-default-user
	       ;; Ask the user and remember the response.
	       (let* ((enable-recursive-minibuffers t)
		      (user (read-string (format "User for %s: " host)
					 (user-login-name))))
		 (ange-ftp-set-user host user)
		 user))
	      ;; Default to the user's login name.
	      (t (user-login-name))))))

(defun ange-ftp-set-passwd (host user passwd)
  "For a given HOST and USER, set or change the associated PASSWD."
  (interactive (list (read-string "Host: ")
		     (read-string "User: ")
		     (ange-ftp-read-passwd "Password: ")))
  (ange-ftp-put-hash-entry (concat host "/" user)
			   passwd
			   ange-ftp-passwd-hashtable))

(defun ange-ftp-get-host-with-passwd (user)
  "Given a USER, return a host we know the password for."
  (or ange-ftp-have-read-netrc (ange-ftp-parse-netrc))
  (catch 'found-one
    (ange-ftp-map-hashtable
     (function (lambda (host val)
		 (if (ange-ftp-get-hash-entry (concat host "/" user)
					      ange-ftp-passwd-hashtable)
		     (throw 'found-one host))))
     ange-ftp-user-hashtable)
    nil))

(defun ange-ftp-get-passwd (host user)
  "Given a HOST and USER, return the ftp password,
prompting if it was not previously set."
  (or ange-ftp-have-read-netrc (ange-ftp-parse-netrc))
  (let ((passwd (ange-ftp-get-hash-entry (concat host "/" user)
					 ange-ftp-passwd-hashtable)))
    (or passwd
	(and ange-ftp-generate-anonymous-password
	     (string-equal user "anonymous")
	     (concat (user-login-name) "@" (system-name)))
	(let* ((other (ange-ftp-get-host-with-passwd user))
	       (passwd (if other
			   (ange-ftp-read-passwd
			    (format "passwd for %s@%s (same as %s@%s): "
				    user host user other)
			    (ange-ftp-get-hash-entry (concat other "/" user)
						     ange-ftp-passwd-hashtable))
			 (ange-ftp-read-passwd
			  (format "Password for %s@%s: " user host)))))
	  (ange-ftp-set-passwd host user passwd)
	  passwd))))

;;;; ------------------------------------------------------------
;;;; ~/.netrc support
;;;; ------------------------------------------------------------

(defun ange-ftp-parse-field (field limit)
  "Move along current line looking for the value of the FIELD.  Valid
separators between FIELD and its value are commas and whitespace.
Second arg LIMIT is a limit for the search."
  (if (search-forward field limit t)
      (let (beg)
	(skip-chars-forward ", \t\r\n" limit)
	(if (looking-at "\"")		;quoted field value
	    (progn (forward-char 1)
		   (setq beg (point))
		   (skip-chars-forward "^\"" limit)
		   (forward-char 1)
		   (buffer-substring beg (1- (point))))
	  (setq beg (point))
	  (skip-chars-forward "^, \t\r\n" limit)
	  (buffer-substring beg (point))))))

(defun ange-ftp-parse-line ()
  "Extract the values of the fields \`machine\', \`login\' and \`password\'
from the current line of the buffer.  If successful, call ange-ftp-set-passwd
with the values found."
  (beginning-of-line)
  (let ((bol (point))
	(eol (progn (re-search-forward "machine" (point-max) 'end 2) (point)))
	machine login password)
    (goto-char bol)
    (setq machine (ange-ftp-parse-field "machine" eol)
	  login (ange-ftp-parse-field "login" eol)
	  password (ange-ftp-parse-field "password" eol))
    (and machine login
	 (progn
	   (ange-ftp-set-user machine login)
	   (ange-ftp-set-passwd machine login password)))
    (goto-char eol)))

(defun ange-ftp-chase-symlinks (file)
  "Return the filename that FILENAME references, following all symbolic links."
  (let (temp)
    (while (setq temp (file-symlink-p file))
      (setq file
	    (if (file-name-absolute-p temp)
		temp
	      (concat (file-name-directory file) temp)))))
  file)

(defun ange-ftp-parse-netrc ()
  "If ~/.netrc file exists and has the correct security then extract the
\`machine\', \`login\' and \`password\' information from each line." 
  ;; We set this before actually doing it to avoid the possibility
  ;; of an infinite loop if ange-ftp-netrc-filename is an ftp file.
  (setq ange-ftp-have-read-netrc t)
  (let* ((file (ange-ftp-chase-symlinks (expand-file-name ange-ftp-netrc-filename)))
	 (attr (file-attributes file)))
    (if attr				; File exits.
	(if (and (eq (nth 2 attr) (user-uid)) ; Same uids.
		 (string-match ".r..------" (nth 8 attr))) ; Readable by user only.
	    (save-excursion
	      (set-buffer (generate-new-buffer "*ftp-.netrc*"))
	      (insert-file-contents file)
	      (goto-char (point-min))
	      (while (not (eobp))
		(ange-ftp-parse-line))
	      (kill-buffer (current-buffer)))
	  (message "skipping badly configured .netrc file")))))

;;;; ------------------------------------------------------------
;;;; Miscellaneous utils.
;;;; ------------------------------------------------------------

(defun ange-ftp-ftp-process-buffer (host user)
  "Return the name of the buffer that collects output from the ftp process
connected to the given HOST and USER pair."
  (concat "*ftp " user "@" host "*"))

(defun ange-ftp-error (host user msg)
  "Display the last chunk of output from the ftp process for the given HOST
USER pair, and signal an error including MSG in the text."
  (let ((cur (selected-window))
	(pop-up-windows t))
    (pop-to-buffer
     (get-buffer-create
      (ange-ftp-ftp-process-buffer host user)))
    (goto-char (point-max))
    (select-window cur))
  (error "ange-ftp: %s" msg))

(defun ange-ftp-set-buffer-mode ()
  "Set the correct modes for the current buffer if it is visiting a remote
file."
  (if (and (stringp buffer-file-name)
	   (ange-ftp-ftp-path buffer-file-name))
      (progn
	(auto-save-mode 0)
	(make-variable-buffer-local 'revert-buffer-function)
	(setq revert-buffer-function 'ange-ftp-revert-buffer))))

(defun ange-ftp-kill-ftp-process (buffer)
  "If the BUFFER's visited filename or default-directory is an ftp filename
then kill the related ftp process."
  (interactive "bKill FTP process associated with buffer: ")
  (if (null buffer)
      (setq buffer (current-buffer)))
  (let ((file (or (buffer-file-name) default-directory)))
    (if file
	(let ((parsed (ange-ftp-ftp-path (expand-file-name file))))
	  (if parsed
	      (let ((host (nth 0 parsed))
		    (user (nth 1 parsed)))
		(kill-buffer (ange-ftp-ftp-process-buffer host user))))))))


;;;; ------------------------------------------------------------
;;;; FTP process filter support.
;;;; ------------------------------------------------------------

(defun ange-ftp-process-handle-line (line)
  "Look at the given LINE from the ftp process.  Try to catagorize it
into one of four categories: good, skip, fatal, or unknown."
  (cond ((string-match ange-ftp-skip-msgs line)
	 t)
	((string-match ange-ftp-good-msgs line)
	 (setq ange-ftp-process-running nil
	       ange-ftp-process-status t))
	((string-match ange-ftp-fatal-msgs line)
	 (delete-process proc)
	 (setq ange-ftp-process-running nil))
	((string-match ange-ftp-multi-msgs line)
	 (setq ange-ftp-process-multi-skip t))
	(ange-ftp-process-multi-skip
	 t)
	(t
	 (setq ange-ftp-process-running nil))))

(defun ange-ftp-process-log-string (proc str)
  "For a given PROCESS, log the given STRING at the end of its
associated buffer."
  (let ((old-buffer (current-buffer)))
    (unwind-protect
	(let (moving)
	  (set-buffer (process-buffer proc))
	  (setq moving (= (point) (process-mark proc)))
	  (save-excursion
	    ;; Insert the text, moving the process-marker.
	    (goto-char (process-mark proc))
	    (insert str)
	    (set-marker (process-mark proc) (point)))
	  (if moving (goto-char (process-mark proc))))
      (set-buffer old-buffer))))

(defun ange-ftp-process-filter (proc str)
  "Build up a complete line of output from the ftp PROCESS and pass it
on to ange-ftp-process-handle-line to deal with."
  (ange-ftp-process-log-string proc str)
  (if ange-ftp-process-running
      (progn
	(setq ange-ftp-process-string (concat ange-ftp-process-string str)))
    ;; if we gave an empty password to the USER command earlier then we
    ;; should send a null password now.
    (if (string-match "Password: *$" ange-ftp-process-string)
	(send-string proc "\n")))
  (while (and ange-ftp-process-running
	      (string-match "\n" ange-ftp-process-string))
    (let ((line (substring ange-ftp-process-string 0 (match-beginning 0))))
      (setq ange-ftp-process-string (substring ange-ftp-process-string
					       (match-end 0)))
      (while (string-match "^ftp> *" line)
	(setq line (substring line (match-end 0))))
      (ange-ftp-process-handle-line line))))

(defun ange-ftp-process-sentinel (proc str)
  "When ftp process changes state, nuke all file-entries in cache."
  (let ((name (process-name proc)))
    (if (string-match "\\*ftp \\([^@]+\\)@\\([^*]+\\)*" name)
      (let ((user (substring name (match-beginning 1) (match-end 1)))
	    (host (substring name (match-beginning 2) (match-end 2))))
	(ange-ftp-wipe-file-entries host user))))
  (setq ange-ftp-ls-cache-file nil))

;;;; ------------------------------------------------------------
;;;; Gateway support.
;;;; ------------------------------------------------------------

(defun ange-ftp-use-gateway-p (host)
  (not (string-match ange-ftp-local-host-regexp host)))

(defun ange-ftp-make-tmp-name (host)
  (make-temp-name (if (ange-ftp-use-gateway-p host)
		      ange-ftp-gateway-tmp-name-template
		    ange-ftp-tmp-name-template)))


;;;; ------------------------------------------------------------
;;;; Interactive gateway program support.
;;;; ------------------------------------------------------------

(defvar ange-ftp-gwp-running t)
(defvar ange-ftp-gwp-status nil)

(defun ange-ftp-gwp-sentinel (proc str)
  (setq ange-ftp-gwp-running nil))

(defun ange-ftp-gwp-filter (proc str)
  (ange-ftp-process-log-string proc str)
  (cond ((string-match "login: *$" str)
	 (send-string proc
		      (concat
		       (let ((ange-ftp-default-user t))
			 (ange-ftp-get-user ange-ftp-gateway-host))
		       "\n")))
	((string-match "Password: *$" str)
	 (send-string proc
		      (concat
		       (ange-ftp-get-passwd ange-ftp-gateway-host
					    (ange-ftp-get-user ange-ftp-gateway-host))
		       "\n")))
	((string-match "No route to host\\|Connection closed\\|No such host" str)
	 (delete-process proc)
	 (setq ange-ftp-gwp-running nil))
	((string-match ange-ftp-gateway-prompt-pattern str)
	 (setq ange-ftp-gwp-running nil
	       ange-ftp-gwp-status t))))

(defun ange-ftp-gwp-start (host user name args)
  "Login to the gateway machine and fire up an ftp process."
  (let* ((gw-user (ange-ftp-get-user ange-ftp-gateway-host))
	 (proc (start-process name name 
			      ange-ftp-gateway-program
			      ange-ftp-gateway-host))
	 (ftp (mapconcat (function (lambda (x) x)) args " ")))
    (process-kill-without-query proc)
    (set-process-sentinel proc (function ange-ftp-gwp-sentinel))
    (set-process-filter proc (function ange-ftp-gwp-filter))
    (set-marker (process-mark proc) (point))
    (setq ange-ftp-gwp-running t
	  ange-ftp-gwp-status nil)
    (message "Connecting to gateway %s..." ange-ftp-gateway-host)
    (while ange-ftp-gwp-running		;perform login sequence
      (accept-process-output proc))
    (if (not ange-ftp-gwp-status)
	(ange-ftp-error host user "unable to login to gateway"))
    (message "Connecting to gateway %s...done" ange-ftp-gateway-host)
    (setq ange-ftp-gwp-running t
	  ange-ftp-gwp-status nil)
    (process-send-string proc ange-ftp-gateway-setup-term-command)
    (while ange-ftp-gwp-running		;zap ^M's and double echoing.
      (accept-process-output proc))
    (if (not ange-ftp-gwp-status)
	(ange-ftp-error host user "unable to set terminal modes on gateway"))
    (setq ange-ftp-gwp-running t
	  ange-ftp-gwp-status nil)
    (process-send-string proc (concat "exec " ftp "\n")) ;spawn ftp process
    proc))

;;;; ------------------------------------------------------------
;;;; Support for sending commands to the ftp process.
;;;; ------------------------------------------------------------

(defun ange-ftp-raw-send-cmd (proc cmd)
  "Low-level routine to send the given ftp CMD to the ftp PROCESS.
Returns non-nil if successful."
  (if (eq (process-status proc) 'run)
      (save-excursion
	(setq ange-ftp-process-string ""
	      ange-ftp-process-running t
	      ange-ftp-process-status nil
	      ange-ftp-process-multi-skip nil
	      cmd (concat cmd "\n"))
	(set-buffer (process-buffer proc))
	(goto-char (point-max))
	(move-marker last-input-start (point))
	;; don't insert the password into the buffer on the USER command.
	(if (string-match "^user \"[^\"]*\"" cmd)
	    (insert (substring cmd 0 (match-end 0)) " Turtle Power!\n")
	  (insert cmd))
	(move-marker last-input-end (point))
	(send-string proc cmd)
	(set-marker (process-mark proc) (point))
	(while ange-ftp-process-running
	  (accept-process-output proc))
	ange-ftp-process-status)))

(defun ange-ftp-start-process (host user name)
  "Spawn a new ftp process ready to connect to machine HOST and give it NAME.
If HOST is only ftp-able through a gateway machine then spawn a shell
on the gateway machine to do the ftp instead."
  (let ((args '("ftp" "-i" "-n" "-g" "-v"))
	proc)
    (if (and (ange-ftp-use-gateway-p host)
	     (not ange-ftp-smart-gateway))
	(if ange-ftp-gateway-program-interactive
	    (setq proc (ange-ftp-gwp-start host user name args))
	  (setq proc (apply 'start-process name name
			    (append (list ange-ftp-gateway-program
					  ange-ftp-gateway-host)
				    args))))
      (setq proc (apply 'start-process name name args)))
    (process-kill-without-query proc)
    (set-process-sentinel proc (function ange-ftp-process-sentinel))
    (set-process-filter proc (function ange-ftp-process-filter))
    (save-excursion
      (set-buffer (process-buffer proc))
      (ange-ftp-shell-mode))
    (accept-process-output proc)	;wait for ftp startup message
    proc))

(defun ange-ftp-quote-passwd (pass)
  "Return a properly quoted password that is acceptable to the ftp process."
  (apply (function concat)
	 (mapcar
	   (function
	     (lambda (char)
		   (vector ?\\ char)))
	   pass)))

(defun ange-ftp-get-process (host user)
  "Return the process object for a ftp process connected to HOST and
logged in as USER.  Create a new proces if needed."
  (let* ((name (ange-ftp-ftp-process-buffer host user))
	 (proc (get-process name)))
    (if (and proc (eq (process-status proc) 'run))
	proc
      (let ((pass (ange-ftp-quote-passwd
		   (ange-ftp-get-passwd host user))))
	(setq proc (ange-ftp-start-process host user name))
	(if (and ange-ftp-smart-gateway
		 (ange-ftp-use-gateway-p host))
	    (progn
	      (message "Opening FTP connection to %s via %s..." host
		       ange-ftp-gateway-host)
	      (or (ange-ftp-raw-send-cmd proc
					 (format "open %s %s"
						 ange-ftp-gateway-host
						 ange-ftp-smart-gateway-port))
		  (ange-ftp-error host user "OPEN request failed"))
	      (message "Logging in as user %s@%s..." user host)
	      (or (ange-ftp-raw-send-cmd proc (format "user \"%s\"@%s %s"
						      user host pass))
		  (progn
		    (ange-ftp-set-passwd host user nil) ;reset password.
		    (ange-ftp-error host user "USER request failed")))
	      (message "Logging in as user %s@%s...done" user host))
	  (message "Opening FTP connection to %s..." host)
	  (or (ange-ftp-raw-send-cmd proc (format "open %s" host))
	      (ange-ftp-error host user "OPEN request failed"))
	  (message "Logging in as user %s@%s..." user host)
	  (or (ange-ftp-raw-send-cmd proc (format "user \"%s\" %s" user pass))
	      (progn
		(ange-ftp-set-passwd host user nil) ;reset password.
		(ange-ftp-error host user "USER request failed")))
	  (message "Logging in as user %s@%s...done" user host))
	proc))))

(defun ange-ftp-send-cmd (host user cmd)
  "Find an ftp process connected to HOST logged in as USER and send it CMD.
Returns whether successful."
  (let ((proc (ange-ftp-get-process host user)))
    (or (ange-ftp-raw-send-cmd proc cmd)
	;; Failed, try once more.
	(and (setq proc (ange-ftp-get-process host user))
	     (ange-ftp-raw-send-cmd proc cmd)))))

;;;; ------------------------------------------------------------
;;;; Simple FTP process shell support.
;;;; ------------------------------------------------------------

(require 'shell)

(defvar ange-ftp-shell-mode-map nil)

(defun ange-ftp-shell-mode ()
  "Major mode for interacting with an FTP process.
Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.

The following commands imitate the usual Unix interrupt and editing
control characters:
\\{ange-ftp-shell-mode-map}
Runs ange-ftp-shell-mode-hook if not nil."
  (interactive)
  (let ((proc (get-buffer-process (current-buffer))))
    (kill-all-local-variables)
    (if (not ange-ftp-shell-mode-map)
	(progn
	  (setq ange-ftp-shell-mode-map (copy-keymap shell-mode-map))
	  (define-key ange-ftp-shell-mode-map "\C-m" 'ange-ftp-shell-send-input)))
    (use-local-map ange-ftp-shell-mode-map)
    (setq major-mode 'ange-ftp-shell-mode)
    (setq mode-name "ange-ftp")
    (setq mode-line-process '(": %s"))
    (make-local-variable 'last-input-start)
    (setq last-input-start (make-marker))
    (make-local-variable 'last-input-end)
    (setq last-input-end (make-marker))
    (goto-char (point-max))
    (set-marker (process-mark proc) (point))
    (run-hooks 'ange-ftp-shell-mode-hook)))

(defun ange-ftp-shell-send-input ()
  "Send input to FTP process.
At end of buffer, sends all text after last output as input to the subshell,
including a newline inserted at the end.  When not at end, copies current line
to the end of the buffer and sends it, after first attempting to discard any
prompt at the beginning of the line."
  (interactive)
  (let ((process (get-buffer-process (current-buffer))))
    (or process
	(error "Current buffer has no process"))
    (end-of-line)
    (if (eobp)
	(progn
	  (move-marker last-input-start
		       (process-mark process))
	  (insert ?\n)
	  (move-marker last-input-end (point)))
      (beginning-of-line)
      (re-search-forward "ftp> *"
			 (save-excursion (end-of-line) (point))
			 t)
      (let ((copy (buffer-substring (point)
				    (progn (forward-line 1) (point)))))
	(goto-char (point-max))
	(move-marker last-input-start (point))
	(insert copy)
	(move-marker last-input-end (point))))
    (process-send-region process last-input-start last-input-end)
    (set-marker (process-mark process) (point))))

;;;; ------------------------------------------------------------
;;;; Remote pathname syntax support.
;;;; ------------------------------------------------------------

(defmacro ange-ftp-ftp-path-component (n)
  "Extract the Nth ftp path component."
  (` (let ((elt (nth (, n) ns)))
       (substring path (match-beginning elt) (match-end elt)))))

(defun ange-ftp-ftp-path (path)
  "Parse PATH according to ange-ftp-path-format (which see).
Returns a list (HOST USER PATH), or nil if PATH does not match the format."
  (let ((data (match-data)))
    (unwind-protect
	(if (string-match (car ange-ftp-path-format) path)
	    (let* ((ns (cdr ange-ftp-path-format))
		   (host (ange-ftp-ftp-path-component 0))
		   (user (ange-ftp-ftp-path-component 1))
		   (path (ange-ftp-ftp-path-component 2)))
	      (if (zerop (length user))
		  (setq user (ange-ftp-get-user host)))
	      (list host user path))
	  nil)
      (store-match-data data))))

(defun ange-ftp-replace-path-component (fullpath path)
  "Take a FULLPATH that matches according to ange-ftp-path-format and
replace the path component with PATH."
  (if (string-match (car ange-ftp-path-format) fullpath)
      (let* ((ns (cdr ange-ftp-path-format))
	     (elt (nth 2 ns)))
	(concat (substring fullpath 0 (match-beginning elt))
		path
		(substring fullpath (match-end elt))))))

;;;; ------------------------------------------------------------
;;;; Remote file and directory listing support.
;;;; ------------------------------------------------------------

(defun ange-ftp-dumb-host (host)
  "Returns whether HOST's ftp daemon doesn't like \'ls\' or \'dir\' commands
to take switch arguments."
  (and ange-ftp-dumb-host-regexp
       (string-match ange-ftp-dumb-host-regexp host)))

(defun ange-ftp-add-dumb-host (host)
  "Interactively adds a given HOST to ange-ftp-dumb-host-regexp."
  (interactive "sHost: ")
  (if (not (ange-ftp-dumb-host host))
      (setq ange-ftp-dumb-host-regexp
	    (concat "^" (regexp-quote host) "$\\|"
		    ange-ftp-dumb-host-regexp))))

(defvar ange-ftp-ls-cache-cmd nil
  "Last `ls' command issued by ange-ftp-ls.")

(defvar ange-ftp-ls-cache-file nil
  "Last file passed to ange-ftp-ls.")

(defvar ange-ftp-ls-cache-res nil
  "Last result returned from ange-ftp-ls.")

(defun ange-ftp-ls (file lsargs &optional want-buffer)
  "Return the output of an `ls' command done on a remote machine using ftp.
The first argument FILE is the full name of the remote file, the second arg
LSARGS is any args to pass to the `ls' command, and the optional third arg
WANT-BUFFER indicates that a buffer object should be returned rather than
a string object."
  (let ((parsed (ange-ftp-ftp-path file)))
    (if parsed
	(let* ((host (nth 0 parsed))
	       (user (nth 1 parsed))
	       (path (nth 2 parsed))
	       (temp (ange-ftp-make-tmp-name host))
	       lscmd)
	  (if (string-equal path "")
	      (setq path "."))
	  (if (ange-ftp-dumb-host host)
	      (setq lscmd (concat "dir " path " " temp))
	    (if ange-ftp-ls-follow-symbolic-links
		(if (> (length lsargs) 0)
		    (setq lsargs (concat lsargs "L"))
		  (setq lsargs "-L")))
	    (setq lscmd (format "ls \"%s %s\" %s" lsargs path temp)))
	  (if (and ange-ftp-ls-cache-file
		   (string-equal file ange-ftp-ls-cache-file)
		   (string-equal lscmd ange-ftp-ls-cache-cmd))
	      (if (not want-buffer)
		  ange-ftp-ls-cache-res
		(set-buffer (get-buffer-create ange-ftp-data-buffer-name))
		(erase-buffer)
		(insert ange-ftp-ls-cache-res))
	    (message "Listing %s..." file)
	    (if (ange-ftp-send-cmd host user lscmd)
		(let (data)
		  (save-excursion
		    (set-buffer (get-buffer-create ange-ftp-data-buffer-name))
		    (erase-buffer)
		    (if (file-readable-p temp)
			(insert-file-contents temp)
		      (ange-ftp-error host user
				      (format "list data file %s not readable"
					      temp)))
		    (setq ange-ftp-ls-cache-file file
			  ange-ftp-ls-cache-cmd lscmd
			  ange-ftp-ls-cache-res (buffer-string))
		    (if want-buffer
			(setq data (current-buffer))
		      (setq data ange-ftp-ls-cache-res)
		      (kill-buffer (current-buffer)))
		    (condition-case () (delete-file temp) (error nil)))
		  (message "Listing %s...done" file)
		  data)
	      (ange-ftp-error host user "Unable to get a remote ls")))))))

;;;; ------------------------------------------------------------
;;;; Directory information caching support.
;;;; ------------------------------------------------------------

(defun ange-ftp-parse-filename ()
  "Extract the filename from the current line of a dired-like listing."
  (save-excursion
    (let ((eol (progn (end-of-line) (point))))
      (beginning-of-line)
      (if (re-search-forward
	   "\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\)[ ]+[0-9]+"
	   eol t)
	  (progn (skip-chars-forward " ")
		 (skip-chars-forward "^ " eol)
		 (skip-chars-forward " " eol)
		 (let ((beg (point)))
		   (skip-chars-forward "^ \n")
;;	           (skip-chars-backward "*/@")
		   (buffer-substring beg (point))))))))

(defun ange-ftp-parse-dired-listing ()
  "Parse the current buffer which is assumed to be in a dired-like listing
format, and return a hashtable as the result."
  (let ((tbl (ange-ftp-make-hashtable)))
    (goto-char (point-min))
    (if (re-search-forward "total [0-9]+" nil t) ;total not always first line.
	(progn
	  (forward-line 1)			;Skip over total byte count.
	  (let (file)
	    (while (setq file (ange-ftp-parse-filename))
	      (beginning-of-line)
;;	      (skip-chars-forward "\t 0-9")
	      (ange-ftp-put-hash-entry file (looking-at "d") tbl)
	      (forward-line 1)))
	  (ange-ftp-put-hash-entry "." t tbl)
	  (ange-ftp-put-hash-entry ".." t tbl)))
    tbl))

(defun ange-ftp-set-files (directory files)
  "For a given DIRECTORY, set or change the associated FILES hashtable."
  (ange-ftp-put-hash-entry directory files ange-ftp-files-hashtable))

(defun ange-ftp-get-files (directory)
  "Given a given DIRECTORY, return a hashtable of file entries."
  (setq directory (file-name-as-directory directory)) ;normalize
  (let ((files (ange-ftp-get-hash-entry directory ange-ftp-files-hashtable)))
    (or files
	(save-excursion
	  (set-buffer (ange-ftp-ls directory "-al" t))
	  (let ((files (ange-ftp-parse-dired-listing)))
	    (ange-ftp-put-hash-entry directory
				     files
				     ange-ftp-files-hashtable)
	    (kill-buffer (current-buffer))
	    files)))))

(defmacro ange-ftp-get-file-part (path)
  "Given PATH, return the file part that can be used for looking up the file's
entry in a hashtable."
  (` (let ((file (file-name-nondirectory (, path))))
       (if (string-equal file "")
	   "."
	 file))))

(defun ange-ftp-get-file-entry (path)
  "Given PATH, return whether the given file entry.  At the moment
this returns whether PATH is a directory or not."
  (ange-ftp-get-hash-entry (ange-ftp-get-file-part path)
			   (ange-ftp-get-files (file-name-directory path))))

(defun ange-ftp-file-entry-p (path)
  "Given PATH, return whether there is a file entry for it."
  (ange-ftp-hash-entry-exists-p (ange-ftp-get-file-part path)
				(ange-ftp-get-files (file-name-directory path))))

(defun ange-ftp-delete-file-entry (path)
  "Given a PATH, delete the file entry for it, if it exists."
  (let ((files (ange-ftp-get-hash-entry (file-name-directory path)
					ange-ftp-files-hashtable)))
    (if files
	(ange-ftp-del-hash-entry (ange-ftp-get-file-part path)
				 files)))
  (setq ange-ftp-ls-cache-file nil))

(defun ange-ftp-add-file-entry (path &optional dir-p)
  "Given a PATH, add the file entry for it, if its directory info exists."
  (let ((files (ange-ftp-get-hash-entry (file-name-directory path)
					ange-ftp-files-hashtable)))
    (if files
	(ange-ftp-put-hash-entry (ange-ftp-get-file-part path)
				 dir-p
				 files)))
  (setq ange-ftp-ls-cache-file nil))

(defun ange-ftp-wipe-file-entries (host user)
  "Remove all file entry information for the given HOST, USER pair."
  (ange-ftp-map-hashtable
   (function
    (lambda (key val)
      (let ((parsed (ange-ftp-ftp-path key)))
	(if parsed
	    (let ((h (nth 0 parsed))
		  (u (nth 1 parsed)))
	      (if (and (equal host h) (equal user u))
		  (ange-ftp-del-hash-entry key
					   ange-ftp-files-hashtable)))))))
   ange-ftp-files-hashtable))

;;;; ------------------------------------------------------------
;;;; File transfer mode support.
;;;; ------------------------------------------------------------

(defun ange-ftp-set-binary-mode (host user)
  "Tell the ftp process for the given HOST & USER to switch to binary mode."
  (ange-ftp-send-cmd host user "type binary"))

(defun ange-ftp-set-ascii-mode (host user)
  "Tell the ftp process for the given HOST & USER to switch to ascii mode."
  (ange-ftp-send-cmd host user "type ascii"))

;;;; ------------------------------------------------------------
;;;; Redefinitions of standard GNU Emacs functions.
;;;; ------------------------------------------------------------

(defun ange-ftp-canonize-filename (n)
  "Take a list of characters and short-circuit //, /. and /.."
  (if (string-match "//" n)
      (setq n (substring n (1- (match-end 0)))))
  (while (cond ((string-match "/\\./\\|/\\.$" n)
		(setq n (concat (substring n 0 (match-beginning 0))
				(substring n (+ 2 (match-beginning 0))))))
	       ((string-match "/[^/]+/\\.\\.\\(/\\|$\\)" n)
		(let ((end (match-end 0))
		      (beg (match-beginning 0))
		      (slash (= (match-beginning 1) (match-end 1))))
		  (if (or (and (zerop beg) (not slash))
			  (and (not (zerop beg)) slash))
		      (setq end (1- end)))
		  (setq n (concat (substring n 0 (1+ beg))
				  (substring n end)))))))
  n)

(defun ange-ftp-expand-file-name (name &optional default)
  "Convert FILENAME to absolute, and canonicalize it.
Second arg DEFAULT is directory to start with if FILENAME is relative
 \(does not start with slash\); if DEFAULT is nil or missing,
the current buffer's value of default-directory is used.
Filenames containing . or .. as components are simplified;
initial ~ is expanded.  See also the function  substitute-in-file-name.

Note that this function has been extended to deal with remote filename syntax."
  (let (lose)
    (if (string-match "^/" name)
	(progn
	  (while (cond ((string-match "//" name)
			(setq name (substring name (1- (match-end 0)))))
		       ((string-match "/~" name)
			(setq name (substring name (1- (match-end 0)))
			      lose t))))
	  (setq lose
		(or lose
		    (string-match "/\\./\\|/\\.$\\|/\\.\\./\\|/\\.\\.$" name))))
      (setq lose t))
    (if lose
	(if (string-match "^~" name)
	    (ange-ftp-real-expand-file-name name)
	  (if (string-match "^/" name)
	      (ange-ftp-canonize-filename name)
	    (ange-ftp-canonize-filename
	     (concat (file-name-as-directory (or default default-directory))
			  name))))
      name)))

(defun ange-ftp-file-name-as-directory (name)
  "Return a string representing file FILENAME interpreted as a directory.
This string can be used as the value of default-directory
or passed as second argument to expand-file-name.
For a Unix-syntax file name, just appends a slash.
On VMS, converts \"[X]FOO.DIR\" to \"[X.FOO]\", etc.

Note that this function has been extended to deal with remote filename syntax."
  (let ((parsed (ange-ftp-ftp-path name)))
    (if (and parsed (string-equal (nth 2 parsed) ""))
	name
      (ange-ftp-real-file-name-as-directory name))))
	
(defun ange-ftp-file-name-directory (name)
  "Return the directory component in file name NAME.
Return nil if NAME does not include a directory.
Otherwise returns a directory spec.
Given a Unix syntax file name, returns a string ending in slash;
on VMS, perhaps instead a string ending in :, ] or >.

Note that this function has been extended to deal with remote filename syntax."
  (let ((parsed (ange-ftp-ftp-path name)))
    (if parsed
	(let ((path (nth 2 parsed)))
	  (if (string-match "^~[^/]*$" path)
	      name
	    (ange-ftp-replace-path-component
	        name
		(ange-ftp-real-file-name-directory path))))
      (ange-ftp-real-file-name-directory name))))

(defun ange-ftp-file-name-nondirectory (name)
  "Return file name NAME sans its directory.
For example, in a Unix-syntax file name,
this is everything after the last slash,
or the entire name if it contains no slash.

Note that this function has been extended to deal with remote filename syntax."
  (let ((parsed (ange-ftp-ftp-path name)))
    (if parsed
	(let ((path (nth 2 parsed)))
	  (if (string-match "^~[^/]*$" path)
	      ""
	    (ange-ftp-real-file-name-nondirectory path)))
      (ange-ftp-real-file-name-nondirectory name))))

(defun ange-ftp-directory-file-name (dir)
  "Returns the file name of the directory named DIR.
This is the name of the file that holds the data for the directory DIR.
In Unix-syntax, this just removes the final slash.
On VMS, given a VMS-syntax directory name such as \"[X.Y]\",
returns a file name such as \"[X]Y.DIR.1\".

Note that this function has been extended to deal with remote filename syntax."
  (let ((parsed (ange-ftp-ftp-path dir)))
    (if parsed
	(ange-ftp-replace-path-component
	   dir
	   (ange-ftp-real-directory-file-name (nth 2 parsed)))
      (ange-ftp-real-directory-file-name dir))))

(defun ange-ftp-binary-file (file)
  "Returns whether the given FILE is to be considered as a binary file for
ftp transfers."
  (string-match ange-ftp-binary-file-name-regexp file))

(defun ange-ftp-write-region (start end filename &optional append visit)
  "Write current region into specified file.
When called from a program, takes three arguments:
START, END and FILENAME.  START and END are buffer positions.
Optional fourth argument APPEND if non-nil means
  append to existing file contents (if any).
Optional fifth argument VISIT if t means
  set last-save-file-modtime of buffer to this file's modtime
  and mark buffer not modified.
If VISIT is neither t nor nil, it means do not print
  the \"Wrote file\" message.

Note that this function has been extended to deal with remote files using ftp."
  (interactive "r\nFWrite region to file: ")
  (setq filename (expand-file-name filename))
  (let ((parsed (ange-ftp-ftp-path filename)))
    (if parsed
	(let* ((host (nth 0 parsed))
	       (user (nth 1 parsed))
	       (path (nth 2 parsed))
	       (temp (ange-ftp-make-tmp-name host))
	       (binary (ange-ftp-binary-file filename))
	       (cmd (if append "append" "put")))
	  (ange-ftp-real-write-region start end temp nil 'nomsg)
	  (message "Writing %s..." filename)
	  (unwind-protect
	      (progn
		(if binary
		    (ange-ftp-set-binary-mode host user))
		(or (ange-ftp-send-cmd host user
				       (format "%s %s %s" cmd temp path))
		    (signal 'file-error
			    (list
			     "Opening output file"
			     (format "Unable to %s remote file" (upcase cmd))
			     filename))))
	    (delete-file temp)
	    (if binary 
		(ange-ftp-set-ascii-mode host user)))
	  (if (eq visit t)
	      (progn
		(ange-ftp-set-buffer-mode)
		(setq buffer-file-name filename)
		(set-buffer-modified-p nil)))
	  (message "Wrote %s" filename)
	  (ange-ftp-add-file-entry filename))
      (ange-ftp-real-write-region start end filename append visit))))

(defun ange-ftp-insert-file-contents (filename &optional visit)
  "Insert contents of file FILENAME after point.
Returns list of absolute pathname and length of data inserted.
If second argument VISIT is non-nil, the buffer's visited filename
and last save file modtime are set, and it is marked unmodified.
If visiting and the file does not exist, visiting is completed
before the error is signaled.

Note this function has been extended to deal with remote files using ftp."
  (barf-if-buffer-read-only)
  (setq filename (expand-file-name filename))
  (let ((parsed (ange-ftp-ftp-path filename)))
    (if parsed
	(let* ((host (nth 0 parsed))
	       (user (nth 1 parsed))
	       (path (nth 2 parsed))
	       (temp (ange-ftp-make-tmp-name host))
	       (binary (ange-ftp-binary-file filename))
	       result)
	  (if visit
	      (setq buffer-file-name filename))
	  (unwind-protect
	      (progn
		(if binary
		    (ange-ftp-set-binary-mode host user))
		(message "Retrieving %s..." filename)
		(or (ange-ftp-send-cmd host user
				       (format "get %s %s" path temp))
		    (signal 'file-error
			    (list
			     "Opening input file"
			     "Unable to GET remote file"
			     filename)))
		(setq result (ange-ftp-real-insert-file-contents temp visit))
		(message "Retrieving %s...done" filename))
	    (condition-case () (delete-file temp) (error nil))
	    (if binary
		(ange-ftp-set-ascii-mode host user)))
	  (if visit
	      (setq buffer-file-name filename))
	  result)
      (ange-ftp-real-insert-file-contents filename visit))))

(defun ange-ftp-revert-buffer (arg noconfirm)
  "Revert this buffer from a remote file using ftp."
  (let ((opoint (point)))
    (cond ((null buffer-file-name)
	   (error "Buffer does not seem to be associated with any file"))
	  ((or noconfirm
	       (yes-or-no-p (format "Revert buffer from file %s? "
				    buffer-file-name)))
	   (let ((buffer-read-only nil))
	     ;; Set buffer-file-name to nil
	     ;; so that we don't try to lock the file.
	     (let ((buffer-file-name nil))
	       (unlock-buffer)
	       (erase-buffer))
	     (insert-file-contents buffer-file-name t))
	   (goto-char (min opoint (point-max)))
	   (after-find-file nil)
	   t))))

(defun ange-ftp-file-exists-p (file)
  "Return t if FILE exists."
  (setq file (expand-file-name file))
  (if (ange-ftp-ftp-path file)
      (ange-ftp-file-entry-p file)
    (ange-ftp-real-file-exists-p file)))

(defun ange-ftp-file-directory-p (file)
  "Return t if FILENAME is the name of a directory as a file.
A directory name spec may be given instead; then the value is t
if the directory so specified exists and really is a directory.

Note that this function has been extended to deal with remote files using ftp."
  (setq file (expand-file-name file))
  (if (ange-ftp-ftp-path file)
      (ange-ftp-get-file-entry file)
    (ange-ftp-real-file-directory-p file)))

(defun ange-ftp-directory-files (directory &optional full match)
  "Return a list of names of files in DIRECTORY.
If FULL is non-NIL, absolute pathnames of the files are returned.
If MATCH is non-NIL, only pathnames containing that regexp are returned.

Note that this function has been extended to deal with remote files using ftp."
  (setq directory (expand-file-name directory))
  (if (ange-ftp-ftp-path directory)
      (let (files)
	(setq directory (file-name-as-directory directory))
	(mapcar (function
		 (lambda (f)
		   (if full
		       (setq f (concat directory f)))
		   (if match
		       (if (string-match match f)
			   (setq files (cons f files)))
		     (setq files (cons f files)))))
		(ange-ftp-hash-table-keys (ange-ftp-get-files directory)))
	(nreverse files))
    (ange-ftp-real-directory-files directory full match)))

(defun ange-ftp-file-attributes (file)
  "Return a list of attributes of file FILENAME.
Value is nil if specified file cannot be opened.
Otherwise, list elements are:
 0. t for directory, string (name linked to) for symbolic link, or nil.
 1. Number of links to file.
 2. File uid.
 3. File gid.
 4. Last access time, as a list of two integers.
  First integer has high-order 16 bits of time, second has low 16 bits.
 5. Last modification time, likewise.
 6. Last status change time, likewise.
 7. Size in bytes.
 8. File modes, as a string of ten letters or dashes as in ls -l.
 9. t iff file's gid would change if file were deleted and recreated.
10. inode number.

Note that this function has been extended to deal with remote files using ftp."
  (setq file (expand-file-name file))
  (let ((parsed (ange-ftp-ftp-path file)))
    (if parsed
	(if (ange-ftp-file-entry-p file)
	    (let ((host (nth 0 parsed))
		  (user (nth 1 parsed))
		  (path (nth 2 parsed)))
	      (list (ange-ftp-get-file-entry file) ;0
		    -1			           ;1
		    -1				   ;2
		    -1				   ;3
		    '(0 0)		           ;4
		    '(0 0)			   ;5
		    '(0 0)			   ;6
		    -1				   ;7
		    "??????????"		   ;8
		    nil				   ;9
		    ;; Hack to give remote files a unique "inode number".
		    ;; It's actually the sum of the characters in its name.
		    (apply '+ (nconc (mapcar 'identity host)
				     (mapcar 'identity user)
				     (mapcar 'identity path))))))
      (ange-ftp-real-file-attributes file))))

(defun ange-ftp-file-writable-p (file)
  "Return t if file FILENAME can be written or created by you.

Note that this function has been extended to deal with remote files using ftp."
  (setq file (expand-file-name file))
  (or (ange-ftp-ftp-path file)
      (ange-ftp-real-file-writable-p file)))

(defun ange-ftp-file-readable-p (file)
  "Return t if file FILENAME exists and can be read by you.

Note that this function has been extended to deal with remote files using ftp."
  (setq file (expand-file-name file))
  (if (ange-ftp-ftp-path file)
      (ange-ftp-file-entry-p file)
    (ange-ftp-real-file-readable-p file)))

(defun ange-ftp-delete-file (file)
  "Delete specified file.  One argument, a file name string.
If file has multiple names, it continues to exist with the other names.

Note that this function has been extended to deal with remote files using ftp."
  (interactive "fDelete file: ")
  (setq file (expand-file-name file))
  (let ((parsed (ange-ftp-ftp-path file)))
    (if parsed
	(let ((host (nth 0 parsed))
	      (user (nth 1 parsed))
	      (path (nth 2 parsed)))
	  (message "Deleting %s..." file)
	  (or (ange-ftp-send-cmd host user (concat "delete " path))
	      (signal 'file-error
		      (list
		       "Removing old name"
		       "Unable to execute remote DELETE command"
		       path)))
	  (message "Deleting %s...done" file)
	  (ange-ftp-delete-file-entry file))
      (ange-ftp-real-delete-file file))))

(defun ange-ftp-verify-visited-file-modtime (buf)
  "Return t if last mod time of BUF's visited file matches what BUF records.
This means that the file has not been changed since it was visited or saved.

Note that this function has been extended to deal with remote files using ftp."
  (let ((name (buffer-file-name buf)))
    (if (and (stringp name) (ange-ftp-ftp-path name))
	t
      (ange-ftp-real-verify-visited-file-modtime buf))))

(defun ange-ftp-backup-buffer ()
  "Make a backup of the disk file visited by the current buffer, if appropriate.
This is normally done before saving the buffer the first time.
If the value is non-nil, it is the result of `file-modes' on the original file;
this means that the caller, after saving the buffer, should change the modes
of the new file to agree with the old modes.

Note that this function has been extended to deal with remote files using ftp."
  (if (and (stringp buffer-file-name) (ange-ftp-ftp-path buffer-file-name))
      nil
    (ange-ftp-real-backup-buffer)))

;;;; ------------------------------------------------------------
;;;; File copying support.
;;;; ------------------------------------------------------------

(defun ange-ftp-barf-or-query-if-file-exists (absname querystring interactive)
  (if (file-exists-p absname)
      (if (not interactive)
	  (signal 'file-already-exists (list absname))
	(if (not (yes-or-no-p (format "File %s already exists; %s anyway? "
				      absname querystring)))
	    (signal 'file-already-exists (list absname))))))

(defun ange-ftp-copy-remote-to-local (remote local parsed binary)
  "Copy REMOTE file to LOCAL file, where the former is on a remote machine."
  (let ((host (nth 0 parsed))
	(user (nth 1 parsed))
	(path (nth 2 parsed))
	temp
	cmd)
    (if (or ange-ftp-smart-gateway 
	    (not (ange-ftp-use-gateway-p host)))
	(setq cmd (format "get %s %s" path local))
      (setq temp (ange-ftp-make-tmp-name host))
      (setq cmd (format "get %s %s" path temp)))
    (unwind-protect
	(progn
	  (if binary
	      (ange-ftp-set-binary-mode host user))
	  (message "Copying %s to %s..." remote local)
	  (or (ange-ftp-send-cmd host user cmd)
	      (signal 'file-error
		      (list
		       "Opening output file"
		       "Unable to GET remote file"
		       remote)))
	  (if temp (copy-file temp local t))
	  (message "Copying %s to %s...done" remote local))
      (if binary
	  (ange-ftp-set-ascii-mode host user))
      (if temp (delete-file temp)))))

(defun ange-ftp-copy-local-to-remote (local remote parsed binary)
  "Copy LOCAL file to REMOTE file where the latter is a file on a remote machine."
  (let ((host (nth 0 parsed))
	(user (nth 1 parsed))
	(path (nth 2 parsed))
	temp
	cmd)
    (if (or ange-ftp-smart-gateway
	    (not (ange-ftp-use-gateway-p host)))
	(setq cmd (format "put %s %s" local path))
      (setq temp (ange-ftp-make-tmp-name host))
      (setq cmd (format "put %s %s" temp path)))
    (unwind-protect
	(progn
	  (if binary
	      (ange-ftp-set-binary-mode host user))
	  (message "Copying %s to %s..." local remote)
	  (if temp (copy-file local temp t))
	  (or (ange-ftp-send-cmd host user cmd)
	      (signal 'file-error
		      (list
		       "Opening output file"
		       "Unable to PUT remote file"
		       remote)))
	  (message "Copying %s to %s...done" local remote))
      (if binary
	  (ange-ftp-set-ascii-mode host user))
      (if temp (delete-file temp)))
    (ange-ftp-add-file-entry remote)))

(defun ange-ftp-copy-remote-to-remote (f-file t-file f-parsed t-parsed binary)
  "Copy F-FILE to T-FILE, where both files are on remote machines."
  (let ((temp (make-temp-name ange-ftp-copy-tmp-name-template)))
    (unwind-protect
	(progn
	  (ange-ftp-copy-remote-to-local f-file temp f-parsed binary)
	  (ange-ftp-copy-local-to-remote temp t-file t-parsed binary))
      (delete-file temp))))

(defun ange-ftp-copy-file (filename newname &optional ok-if-already-exists
				    keep-date)
  "Copy FILE to NEWNAME.  Both args strings.
Signals a  file-already-exists  error if NEWNAME already exists,
unless a third argument OK-IF-ALREADY-EXISTS is supplied and non-nil.
A number as third arg means request confirmation if NEWNAME already exists.
This is what happens in interactive use with M-x.
Fourth arg non-nil means give the new file the same last-modified time
that the old one has.  (This works on only some systems.)

Note this function has been extended to deal with remote files using ftp."
  (interactive "fCopy file: \nFCopy %s to file: \np")
  (setq filename (expand-file-name filename)
	newname (expand-file-name newname))
  (let ((f-parsed (ange-ftp-ftp-path filename))
	(t-parsed (ange-ftp-ftp-path newname))
	(binary (ange-ftp-binary-file filename)))
    (if (and (or f-parsed t-parsed)
	     (or (not ok-if-already-exists)
		 (numberp ok-if-already-exists)))
	(ange-ftp-barf-or-query-if-file-exists newname "copy to it"
					       (numberp ok-if-already-exists)))
    (if f-parsed
	(if t-parsed
	    (ange-ftp-copy-remote-to-remote filename newname
					    f-parsed t-parsed binary)
	  (ange-ftp-copy-remote-to-local filename newname f-parsed binary))
      (if t-parsed
	  (ange-ftp-copy-local-to-remote filename newname t-parsed binary)
	(ange-ftp-real-copy-file filename newname ok-if-already-exists keep-date)))))

;;;; ------------------------------------------------------------
;;;; Simple Dired support.
;;;; ------------------------------------------------------------

(require 'dired)

(defun ange-ftp-dired-readin (dirname buffer)
  "Emulation of dired-readin with support for remote files using ftp."
  (save-excursion
    (message "Reading directory %s..." dirname)
    (set-buffer buffer)
    (let ((buffer-read-only nil))
      (widen)
      (erase-buffer)
      (setq dirname (expand-file-name dirname))
      (if (ange-ftp-ftp-path dirname)
	  (progn (insert (ange-ftp-ls dirname dired-listing-switches))
		 (ange-ftp-set-files dirname (ange-ftp-parse-dired-listing)))
	(if (file-directory-p dirname)
	    (call-process "ls" nil buffer nil
			  dired-listing-switches dirname)
	  (let ((default-directory (file-name-directory dirname)))
	    (call-process shell-file-name nil buffer nil
			  "-c" (concat "ls " dired-listing-switches " "
				       (file-name-nondirectory dirname))))))
      (goto-char (point-min))
      (while (not (eobp))
	(insert "  ")
	(forward-line 1))
      (goto-char (point-min))))
  (message "Reading directory %s...done" dirname))

;;;; ------------------------------------------------------------
;;;; File name completion support.
;;;; ------------------------------------------------------------

(defun ange-ftp-get-files-for-completion (dir)
  "Return a list of files in the given directory.  Each filename is wrapped
in a singleton list and has a trailing slash if it is a directory."
  (let (res)
    (ange-ftp-map-hashtable
     (function (lambda (key val)
		 (setq res (cons (list (if val
					   (concat key "/")
					 key))
				 res))))
     (ange-ftp-get-files dir))
    (or res '(()))))

(defun ange-ftp-file-name-all-completions (file dir)
  "Return a list of all completions of file name FILE in directory DIR."
  (if (ange-ftp-ftp-path dir)
      (all-completions file (ange-ftp-get-files-for-completion dir))
    (file-name-all-completions file dir)))

(defun ange-ftp-file-name-completion (file dir)
  "Complete file name FILE in directory DIR.
Returns the longest string common to all filenames in DIR that start with FILE.
If there is only one and FILE matches it exactly, returns t.
Returns nil if DIR contains no name starting with FILE."
  (if (ange-ftp-ftp-path dir)
      (try-completion file (ange-ftp-get-files-for-completion dir))
    (file-name-completion file dir)))

(defun ange-ftp-quote-filename (file)
  "Quote $ as $$ to get it past substitute-in-file-name."
  (apply (function concat)
	 (mapcar (function
		   (lambda (char)
		     (if (= char ?$)
			 "$$"
			 (vector char))))
		 file)))

(defun ange-ftp-read-file-name-internal (string dir action)
  "Emulates read-file-name-internal for ftp."
  (let (name realdir)
    (if (eq action 'lambda)
	(if (> (length string) 0)
	    (ange-ftp-file-exists-p (substitute-in-file-name string)))
      (if (zerop (length string))
	  (setq name string realdir dir)
	(setq string (substitute-in-file-name string)
	      name (file-name-nondirectory string)
	      realdir (file-name-directory string))
	(setq realdir (if realdir (expand-file-name realdir dir) dir)))
      (if action
	  (ange-ftp-file-name-all-completions name realdir)
	(let ((specdir (file-name-directory string))
	      (val (ange-ftp-file-name-completion name realdir)))
	  (if (and specdir (stringp val))
	      (ange-ftp-quote-filename (concat specdir val))
	    val))))))


;;;; ------------------------------------------------------------
;;;; Bits and bobs to bolt ange-ftp into GNU Emacs.
;;;; ------------------------------------------------------------

(defun ange-ftp-overwrite-fn (fun)
  "Replace FUN's function definition with ange-ftp-FUN's, saving the
original definition as ange-ftp-real-FUN."
  (let* ((name (symbol-name fun))
	 (saved (intern (concat "ange-ftp-real-" name)))
	 (new (intern (concat "ange-ftp-" name))))
    (or (fboundp saved)
	(fset saved (symbol-function fun)))
    (fset fun new)))

(ange-ftp-overwrite-fn 'insert-file-contents)
(if (not (fboundp 'dired-ls))		; dired has been loaded by now
    (ange-ftp-overwrite-fn 'dired-readin) ; 18.55 dired
  ;; tree dired
  (load "ange-ftp-dired")
  (ange-ftp-overwrite-fn 'dired-run-shell-command)
  (ange-ftp-overwrite-fn 'dired-ls))
(ange-ftp-overwrite-fn 'directory-files)
(ange-ftp-overwrite-fn 'file-directory-p)
(ange-ftp-overwrite-fn 'file-writable-p)
(ange-ftp-overwrite-fn 'file-readable-p)
(ange-ftp-overwrite-fn 'delete-file)
(ange-ftp-overwrite-fn 'read-file-name-internal)
(ange-ftp-overwrite-fn 'verify-visited-file-modtime)
(ange-ftp-overwrite-fn 'file-exists-p)
(ange-ftp-overwrite-fn 'write-region)
(ange-ftp-overwrite-fn 'backup-buffer)
(ange-ftp-overwrite-fn 'copy-file)
(ange-ftp-overwrite-fn 'file-attributes)
(ange-ftp-overwrite-fn 'file-name-directory)
(ange-ftp-overwrite-fn 'file-name-nondirectory)
(ange-ftp-overwrite-fn 'file-name-as-directory)
(ange-ftp-overwrite-fn 'directory-file-name)
(ange-ftp-overwrite-fn 'expand-file-name)


(or (memq 'ange-ftp-set-buffer-mode find-file-hooks)
    (setq find-file-hooks
	  (cons 'ange-ftp-set-buffer-mode find-file-hooks)))


;;;; ------------------------------------------------------------
;;;; Finally provide package.
;;;; ------------------------------------------------------------

(provide 'ange-ftp)
