;; DIRED commands for Emacs.  $Revision: 5.30 $
;; Copyright (C) 1985, 1986, 1990 Free Software Foundation, Inc.
;; Enhanced from Emacs 18.55 dired by Sebastian Kremer <sk@thp.uni-koeln.de>

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(provide 'dired)

(defconst dired-version (substring "$Revision: 5.30 $" 11 -2)
  "The revision number of dired (as string).  The complete RCS id is:

  $Id: dired.el,v 5.30 1991/02/03 19:29:42 sk Exp $

Don't forget to mention this when reporting bugs.")

;; compatibility package when using Emacs 18.55
(require 'emacs-19)			;;;>>> install (delete for Emacs 19)

;;>>> install (change loaddefs.el)
;; can now contain even `F',`i' and `s'.
;In loaddefs.el
;(defvar dired-listing-switches "-al"
;  "Switches passed to ls for dired. MUST contain the `l' option.")

(defvar dired-chown-program
  (if (memq system-type '(hpux usg-unix-v)) "/bin/chown" "/etc/chown")
  "Pathname of chown command.")

(defvar dired-ls-program "ls"
  ;; GNU ls has no way to suppress the group, so one might prefer /bin/ls.
  "*Absolute or relative name of the ls program used by dired.")

(defvar dired-ls-F-marks-symlinks nil
  "*Set this to t if dired-ls-program with -lF marks the symbolic link
itself with a trailing @ (usually the case under Ultrix).

Example: if `ln -s foo bar; ls -F bar' gives `bar -> foo', set it to
nil (the default), if it gives `bar@ -> foo', set it to t.

Dired checks if there is really a @ appended.  Thus, if you have a
marking ls program on one host and a non-marking on another host, and
don't care about symbolic links which really contain a trailing @, you
can always set this variable to t.")

(defvar dired-directory nil
  "The directory name or shell wildcard passed as argument to ls.
Local to each dired buffer.")

(defvar dired-actual-switches nil
  "The actual (buffer-local) value of dired-listing-switches.")

;; This makes matches rather slow - perhaps -is should be forbidden.
;; If you don't use -is, you can set this to "".
(defvar dired-re-inode-size ;;"\\(\\s *[0-9]*\\s *[0-9]* \\)?"
  "\\s *[0-9]*\\s *[0-9]* ?" ; this seems to be slightly faster
  "Regexp for optional initial inode and file size as produced
by ls' -i and -s flags."
)

;; These regexps must be tested at beginning-of-line, but are also
;; used to search for next matches, so omitting "^" won't do.
;; Replacing "^" by "\n" might be faster, but fails on the first line,
;; thus excluding the possibility to mark subdir lines.

(defconst dired-re-mark "^[^ \n]");; "\n[^ \n]" faster?
;; "Regexp matching a marked line.
;; Important: the match ends just after the marker."
(defconst dired-re-maybe-mark "^. ")
(defconst dired-re-dir (concat dired-re-maybe-mark dired-re-inode-size "d"))
(defconst dired-re-sym (concat dired-re-maybe-mark dired-re-inode-size "l"))
(defconst dired-re-exe
  (mapconcat (function
	      (lambda (x)
		(concat dired-re-maybe-mark dired-re-inode-size x)))
	     '("-[-r][-w][xs][-r][-w].[-r][-w]."
	       "-[-r][-w].[-r][-w][xs][-r][-w]."
	       "-[-r][-w].[-r][-w].[-r][-w][xst]")
	     "\\|"))
(defconst dired-re-dot "^.* \\.\\.?$")

;;; Customizable variables:

;; Might use {,} for bash or csh:
(defvar dired-mark-prefix ""
  "*Prepended to marked files in dired shell commands.")
(defvar dired-mark-postfix ""
  "*Appended to marked files in dired shell commands..")
(defvar dired-mark-separator " "
  "*Separates marked files in dired shell commands..")

(defvar shell-maximum-command-length 10000
  "*If non-nil, maximum number of bytes a dired shell command can have
before the user is asked for confirmation.")

(defvar dired-process-maxargs 7000
  ;; NCARGS=10K is the smallest value I know of.  SunOS has 1MB.
  "*If non-nil, maximum size (in characters) of the argv your system
allows.
You should subtract 2-3K from your NCARGS or {ARG_MAX} parameter to
allow some space for the environment.")

(defvar dired-print-command "print %s"
  "Format string for shell command to print files in dired.
Can actually be used for any special purpose shell command to be run
by \\[dired-mark-print].")

(defvar dired-trivial-filenames "^\\.\\.?$\\|^#"
  "*Regexp of files to skip when moving point to the first file of a
new directory listing.
Nil means move to the subdir line, t means move to first file.")

(defvar dired-basename-regexp "\\(.+\\)\\.\\(.+\\)$")

(defvar dired-traditional t
  "If t, marking and flagging for deletion are separate features.
Use `d' to flag with D and `x' to delete the D files.

Else deletion is just a special case of the general file marking
feature.  Use `m' to mark with * (as usual) and `d' to delete the *
\(or next ARG) files.  If you don't like that `d' defaults to the
current file if there are no marks, run the following inside your
dired-mode-hook:
    (define-key dired-mode-map \"d\" 'dired-do-deletions)
"
)

(defvar dired-marker-char ?*		; the answer is 42
  ;; so that you can write things like
  ;; (let ((dired-marker-char ?X))
  ;;    ;; great code using X markers ...
  ;;    )
  ;; For example, commands operating on two sets of files, A and B.
  ;; Or marking files with digits 0-9.  This could implicate
  ;; concentric sets or an order for the marked files.
  ;; The code depends on dynamic scoping on the marker char.
  "In dired, character used to mark files for later commands.")

(defvar dired-del-marker (if dired-traditional ?\D dired-marker-char)
  "*Character used to flag files for deletion.")

(defvar dired-shrink-to-fit (> (baud-rate) 1200)
  "*Whether dired shrinks the display buffer to fit the marked files.")

(defvar dired-confirm nil
  "*Whether dired should confirm those commands that don't require
additional arguments and thus cannot be aborted from a prompt.
Besides nil or t, it can also be a sublist of
  '(compress uncompress byte-recompile load)
to have confirmation for only those commands.")

(defvar dired-pop-up '(deletions shell)
  "*Whether dired should pop up a window with marked files before
operating on them.
Besides nil or t, it can also be a sublist of
  '(deletions shell print cp-or-mv chmod chgrp chown)
to have pop-up buffers for only those commands.
The elements of dired-confirm are also allowed.")

;;; Hook variables

(defvar dired-load-hook nil
  "Run after loading dired.
You can customize key bindings or load extensions with this.")

(defvar dired-mode-hook nil
  "Run each time dired is called.")

(defvar dired-readin-hook nil
  "After each listing of a file or directory, this hook is run
with the buffer narrowed to the listing.")

;; An example filter to squeeze spaces:
;(setq dired-readin-hook
;      '(lambda () (goto-char (point-min))
;	 (while (re-search-forward " +" nil t) (replace-match " "))))
;
;  See dired-extra.el for an example on how to use it for sorting on
;  file size.   It also supports use of several different markers
;  (other than `D' and `*') in parallel and a minibuffer history for
;  shell commands.  Email if you want to try it.  It is about 20K.

;;; Global internal variables

(defvar dired-flagging-regexp nil
  "Last regexp used in flagging files.")

;;; Macros must be defined before they are used - for the byte compiler.

(defmacro dired-count-up ()
  ;; Increment variable dired-mark-count.
  '(setq dired-mark-count (1+ dired-mark-count)))

;; This was a macro, but nested macros are not expanded when
;; byte-compiled (?).
(defun dired-plural-s (&optional count)
  (if (= 1 (or count dired-mark-count)) "" "s"))

(defmacro dired-mark-if (predicate msg)
  (` (let (buffer-read-only)
       (save-excursion
	 (setq dired-mark-count 0)
	 (if (, msg) (message "0 %ss..." (, msg)))
	 (goto-char (point-min))
	 (while (not (eobp))
	   (if (, predicate)
	       (progn
		 (delete-char 1)
		 (insert dired-marker-char)
		 (setq dired-mark-count (1+ dired-mark-count))))
	   (forward-line 1))
	 (if (, msg) (message "%s %s%s %s%s."
			  dired-mark-count
			  (, msg)
			  (dired-plural-s)
			  (if (eq dired-marker-char ?\ ) "un" "")
			  (if (eq dired-marker-char dired-del-marker)
			      "flagged" "marked")))))))

(defmacro dired-mark-wrapper (body)
  "Eval BODY with dired-mark-count and dired-mark-files locally let-bound."
  (` (let (dired-mark-files dired-mark-count)
       (, body))))

(defmacro dired-mark-map (body arg)
  "Macro: Perform BODY with point somewhere on each marked line
  and return a list of BODY's results.
If no file was marked, execute BODY on the current line.
  If ARG is an integer, use the next ARG (or previous -ARG, if ARG<0)
  files instead of the marked files.
  If ARG is otherwise non-nil, use current file instead.
No guarantee is made about the position on the marked line.
  BODY must ensure this itself if it depends on this.
Search starts at the beginning of the buffer, thus the car of the list
  corresponds to the line nearest to the buffer's bottom.  This
  is also true for (positive and negative) integer values of ARG.
BODY should not be too long as it is expanded four times.

Warning: BODY must not add new lines before point - this may cause and
endless loop."
  ;; One might argue that the user being able to reverse the order by
  ;; moving to the end and using -ARG would be a good thing.
  ;; On the other hand, reversing the order only sometimes may cause
  ;; unexpected behaviour.
  (` (prog1
	 (let (buffer-read-only found results)
	   (if (, arg)
	       (if (integerp (, arg))
		   (save-excursion
		     (dired-repeat-over-lines
		      (, arg)
		      (function (lambda ()
				  (setq results (cons (, body) results)))))
		     (if (< (, arg) 0)
			 (nreverse results)
		       results))
		 (list (, body)))
	     (let (opoint (regexp (dired-marker-regexp)))
	       (save-excursion
		 (goto-char (point-min))
		 (while (re-search-forward regexp nil t)
		   ;; If body contains dired-redisplay, the deletion (and
		   ;; new insertion) of the line confuses save-excursion.
		   (setq opoint (point)) ; column 1 stays, however
		   (setq results (cons (, body) results))
		   (goto-char opoint)
		   (setq found t)))
	       (if found
		   results
		 (list (, body))))))
       ;; save-excursion loses, again
       (dired-move-to-filename))))

;; The following functions are redefinable for VMS or ange-ftp
;; - or for customization.

(defun dired-ls (file &optional switches wildcard full-directory-p)
;  "Insert ls output of FILE, optionally formatted with SWITCHES.
;Optional third arg WILDCARD means treat FILE as shell wildcard.
;Optional fourth arg FULL-DIRECTORY-P means file is a directory and
;switches do not contain `d'.
;
;SWITCHES default to dired-listing-switches.
;Uses dired-ls-program and maybe shell-file-name to do the work."
  (or switches (setq switches dired-listing-switches))
  (if wildcard
      (let ((default-directory (file-name-directory file)))
	(call-process shell-file-name nil t nil
		      "-c" (concat dired-ls-program " -d " switches " "
				   (file-name-nondirectory file))))
    (call-process dired-ls-program nil t nil switches file)))

(defun dired-call-process (program discard &rest arguments)
;  "Run PROGRAM with output to current buffer unless DISCARD is t.
;Remaining arguments are strings passed as command arguments to PROGRAM."
  (apply 'call-process program nil (not discard) nil arguments))

(defconst dired-log-buf "*Dired log*")

(defun dired-why ()
  "Pop up a buffer with error log output from Dired.
All mark-using commands log errors there."
  (interactive)
  (pop-to-buffer dired-log-buf)
  (goto-char (point-max)))

(defun dired-log (log)
  ;; Log a message or the contents of a buffer.
  ;; Usage like (dired-log "Foo failed:\n") (dired-log foo-error-buffer).
  (save-excursion
    (set-buffer (get-buffer-create dired-log-buf))
    (goto-char (point-max))
    (cond ((stringp log) (insert "\n\t" (current-time-string) "\n\t"  log))
	  ((bufferp log) (insert-buffer log)))))

(defun dired-check-process-handler (program msg arguments)
  ;;"Run from function dired-check-process if there is output.
  ;; Insert output in a log buffer and return a non-nil error indication.
  ;; Caller can cons up a list of failed args."
  (dired-log (concat program " " (prin1-to-string arguments) "\n"))
  (dired-log err-buffer)
  (or arguments msg program))

(defun dired-check-process (program msg &rest arguments)
;  "Run PROGRAM, display MSG while running, and check for output.
;Remaining arguments are strings passed as command arguments to PROGRAM.
; On output call dired-check-process-handler and return its value.
;Else returns nil for success."
  (if (and dired-process-maxargs	; (apply '+ nil)==(+) yields 0.
	   (> (apply (function +) (mapcar (function length) arguments))
	      dired-process-maxargs))
      (error "%s argument list larger than dired-process-maxargs." program))
  (let (err-buffer err)
    (message "%s..." msg)
    (save-excursion
      ;; Get a clean buffer for error output:
      (setq err-buffer (get-buffer-create " *dired-check-process output*"))
      (set-buffer err-buffer)
      (erase-buffer)
      (apply 'dired-call-process program nil arguments)
      ;; In Emacs 19 the exit status should be checked instead.
      ;; The following is not The Right Thing as some compress
      ;; programs are verbose by default
      (setq err (/= 0 (buffer-size))))
    ;; Check for errors and display them:
    (if err
	(dired-check-process-handler program msg arguments)
      (kill-buffer err-buffer)
      (message "%s... done." msg)
      nil)))

(defun dired-insert-headerline (dir)
  ;; No trailing slash, like ls does:
  (insert "  " (directory-file-name dir) ":")
  ;; put cursor on root subdir line:
  (save-excursion (insert "\n")))

(defun dired-readin (dirname buffer)
  (save-excursion
    (message "Reading directory %s..." dirname)
    (set-buffer buffer)
    (let (buffer-read-only)
      (widen)
      (erase-buffer)
      (setq dirname (expand-file-name dirname))
      (if (eq system-type 'vax-vms)
	  (vms-read-directory dirname dired-actual-switches buffer)
	(if (file-directory-p dirname)
	    (dired-ls dirname dired-actual-switches nil t)
	  (if (not (file-readable-p
		    (directory-file-name (file-name-directory dirname))))
	      (insert "Directory " dirname " inaccessible or nonexistent.\n")
	    ;; else assume it contains wildcards:
	    (dired-ls dirname dired-actual-switches t))))
      (goto-char (point-min))
      (indent-rigidly (point-min) (point-max) 2)
      (run-hooks 'dired-readin-hook)
      ;; We need this to make the root dir have a header line as all
      ;; other subdirs have:
      (goto-char (point-min))
      (dired-insert-headerline default-directory))
    (set-buffer-modified-p nil)
    (message "Reading directory %s...done" dirname)))

;; This differs from dired-buffers in that it does not consider
;; subdirs of default-directory and searches for the _first_ match
(defun dired-find-buffer (dirname)
  (let ((blist (buffer-list))
	found)
    (while blist
      (save-excursion
        (set-buffer (car blist))
	(if (and (eq major-mode 'dired-mode)
		 (equal dired-directory dirname))
	    (setq found (car blist)
		  blist nil)
	  (setq blist (cdr blist)))))
    (or found
	(create-file-buffer (directory-file-name dirname)))))

(defun dired-read-dir-and-switches (str)
  ;; For use in interactive.
  (list
   (read-file-name (format "Dired %s (directory): " str)
		   nil default-directory nil)
   (if current-prefix-arg
       (read-string "Dired listing switches: "
		    dired-listing-switches))))

(defun dired (dirname &optional switches)
  "`Edit' directory DIRNAME--delete, rename, print, etc. some files in it.
Prefix arg lets you change the buffer local value of dired-actual-switches.
Dired displays a list of files in DIRNAME (which may also have
  shell wildcards appended to select certain files).
You can move around in it with the usual commands.
You can flag files for deletion with C-d and then delete them by
  typing `x'.
Type `h' after entering dired for more info."
  ;; Cannot use (interactive "D") because of wildcards.
  (interactive (dired-read-dir-and-switches ""))
  (switch-to-buffer (dired-noselect dirname switches)))

(defun dired-other-window (dirname &optional switches)
  "`Edit' directory DIRNAME.  Like M-x dired but selects in another window."
  (interactive (dired-read-dir-and-switches "in other window "))
  (switch-to-buffer-other-window (dired-noselect dirname switches)))

(defun dired-noselect (dirname &optional switches)
  ;; Like M-x dired but returns the dired buffer as value, does not
  ;; select it.
  (or dirname (setq dirname default-directory))
  ;; This loses the distinction between "/foo/*/" and "/foo/*" that
  ;; some shells make:
  (setq dirname (expand-file-name (directory-file-name dirname)))
  (if (file-directory-p dirname)
      (setq dirname (file-name-as-directory dirname)))
  (dired-internal-noselect dirname switches))

(defun dired-internal-noselect (dirname &optional switches)
  (let ((buffer (dired-find-buffer dirname))
	(old-buf (current-buffer)))
    (or switches (setq switches dired-listing-switches))
    (save-excursion
      (set-buffer buffer)
      ;; must be set before dired-readin inserts the root line:
      (setq default-directory (if (file-directory-p dirname)
				  dirname (file-name-directory dirname)))
      (let ((dired-actual-switches switches))
	(dired-readin dirname buffer))
      (dired-mode dirname switches))
    ;; changing point inside a save-excursion is rather pointless...
    (unwind-protect
	(progn
	  (set-buffer buffer)
	  (goto-char (point-min))
	  (dired-initial-position))
      (set-buffer old-buf))
    buffer))

(defun dired-remember-marks ()
  ;; Return alist of files and their marks, from point to eob.
  (let (fil chr alist)
    (while (re-search-forward dired-re-mark nil t)
      (if (setq fil (dired-get-filename nil t))
	  (setq chr (preceding-char)
		alist (cons (cons fil chr) alist))))
    alist))

(defun dired-mark-remembered (alist)
  ;; Mark all files remembered in ALIST.
  (let (elt fil chr)
    (while alist
      (setq elt (car alist)
	    alist (cdr alist)
	    fil (car elt)
	    chr (cdr elt))
      (if (dired-goto-file fil)
	  (save-excursion
	    (beginning-of-line)
	    (delete-char 1)
	    (insert chr))))))

(defun dired-revert (&optional arg noconfirm)
  ;; Reread the dired buffer.  Should not fail even on completely
  ;; garbaged buffers.
  ;; All marks/flags are preserved.
  (let ((opoint (point))
	(ofile (dired-get-filename nil t))
	(mark-alist nil)		; save marked files
	;; Save old alist except default-directory:
	(old-subdir-alist (cdr (reverse dired-subdir-alist)))
	case-fold-search		; we check for F and R ls flags
	buffer-read-only)
    ;; Remember all marks/flags.  Must unhide to make this work.
    (if selective-display
	(subst-char-in-region (point-min) (point-max) ?\r ?\n))
    (goto-char 1)
    (setq mark-alist (dired-remember-marks))
    (dired-readin dired-directory (current-buffer))
    (dired-advertise)			; no harm if already called
    (setq dired-used-F			; ls switches may have changed
	  (string-match "F" dired-actual-switches))
    (dired-build-subdir-alist)		; moving/retrieval cmds work now

    ;; Try to insert all subdirs that were displayed before
    (or (string-match "R" dired-actual-switches)
	(let (elt dir)
	  (while old-subdir-alist
	    (setq elt (car old-subdir-alist)
		  old-subdir-alist (cdr old-subdir-alist)
		  dir (car elt))
	    (condition-case ()
		(dired-insert-subdir dir)
	      (error nil)))))

    ;; Mark files that were marked before
    (dired-mark-remembered mark-alist)

    ;; Move cursor to where it was before
    (or (and ofile (dired-goto-file ofile))
	(goto-char opoint))
    (dired-move-to-filename))

  ;; outside of the let scope:
  (setq buffer-read-only t)		; gets sometimes out of sync
)

(defvar dired-mode-map nil "Local keymap for dired-mode buffers.")
(if dired-mode-map
    nil
  (setq dired-mode-map (make-keymap))
  (suppress-keymap dired-mode-map)
  (define-key dired-mode-map " "  'dired-next-line)
  (define-key dired-mode-map "!" 'dired-mark-shell-command)
  (define-key dired-mode-map "#" 'dired-flag-auto-save-files)
  (define-key dired-mode-map "$" 'dired-hide-subdir)
  (define-key dired-mode-map "&" 'dired-mark-background-shell-command)
  (define-key dired-mode-map "*" 'dired-mark-executables)
  (define-key dired-mode-map "+" 'dired-create-directory)
  (define-key dired-mode-map "." 'dired-clean-directory)
  (define-key dired-mode-map "/" 'dired-mark-dirlines)
  (define-key dired-mode-map "<" 'dired-prev-dirline)
  (define-key dired-mode-map "=" 'dired-hide-all)
  (define-key dired-mode-map ">" 'dired-next-dirline)
  (define-key dired-mode-map "?" 'dired-summary)
  (define-key dired-mode-map "@" 'dired-mark-symlinks)
  (define-key dired-mode-map "B" 'dired-mark-byte-recompile)
  (define-key dired-mode-map "C" 'dired-mark-compress)
  (define-key dired-mode-map "D" 'dired-diff)
  (define-key dired-mode-map "F" 'dired-flag-regexp-files)
  (define-key dired-mode-map "G" 'dired-mark-chgrp)
  (define-key dired-mode-map "K" 'dired-kill-subdir)
  (define-key dired-mode-map "L" 'dired-mark-load)
  (define-key dired-mode-map "M" 'dired-mark-chmod)
  (define-key dired-mode-map "O" 'dired-mark-chown)
  (define-key dired-mode-map "P" 'dired-mark-print)
  (define-key dired-mode-map "R" 'dired-rename-regexp)
  (define-key dired-mode-map "S" 'dired-sort-other)
  (define-key dired-mode-map "U" 'dired-mark-uncompress)
  (define-key dired-mode-map "W" 'dired-why)
;;  (define-key dired-mode-map "X" 'dired-mark-delete)
  (define-key dired-mode-map "\177" 'dired-backup-unflag)
  (define-key dired-mode-map "\C-_" 'dired-undo)
  (define-key dired-mode-map "\C-d" 'dired-flag-file-deleted)
  (define-key dired-mode-map "\C-n" 'dired-next-line)
  (define-key dired-mode-map "\C-p" 'dired-previous-line)
  (define-key dired-mode-map "\C-xu" 'dired-undo)
  (define-key dired-mode-map "\M-\C-?" 'dired-unflag-all-files)
  (define-key dired-mode-map "\M-g" 'dired-goto-file)
  (define-key dired-mode-map "\M-d" 'dired-down-subdir)
  (define-key dired-mode-map "\M-G" 'dired-goto-subdir)
  (define-key dired-mode-map "\M-k" 'dired-mark-kill)
  (define-key dired-mode-map "\M-m" 'dired-mark-files)
  (define-key dired-mode-map "\M-n" 'dired-next-subdir)
  (define-key dired-mode-map "\M-p" 'dired-prev-subdir)
  (define-key dired-mode-map "\M-u" 'dired-up-subdir)
  (define-key dired-mode-map "\M-~" 'dired-backup-diff)
  (define-key dired-mode-map "^" 'dired-up-directory)
  (define-key dired-mode-map "c" 'dired-mark-copy)
  (define-key dired-mode-map "d" 'dired-flag-file-deleted)
  (define-key dired-mode-map "e" 'dired-find-file)
  (define-key dired-mode-map "f" 'dired-find-file)
  (define-key dired-mode-map "g" 'revert-buffer)
  (define-key dired-mode-map "h" 'describe-mode)
  (define-key dired-mode-map "i" 'dired-insert-subdir)
  (define-key dired-mode-map "k" 'dired-kill-line)
  (define-key dired-mode-map "l" 'dired-mark-redisplay)
  (define-key dired-mode-map "m" 'dired-mark-subdir-or-file)
  (define-key dired-mode-map "n" 'dired-next-line)
  (define-key dired-mode-map "o" 'dired-find-file-other-window)
  (define-key dired-mode-map "p" 'dired-previous-line)
  (define-key dired-mode-map "q" 'dired-bury-or-kill-buffer)
  (define-key dired-mode-map "r" 'dired-mark-move)
  (define-key dired-mode-map "s" 'dired-sort-toggle)
;;  (define-key dired-mode-map "u" 'dired-unflag)
  (define-key dired-mode-map "u" 'dired-unmark-subdir-or-file)
  (define-key dired-mode-map "v" 'dired-view-file)
  (define-key dired-mode-map "w" 'dired-copy-filename-as-kill)
  (define-key dired-mode-map "x" 'dired-do-deletions)
  ;;(define-key dired-mode-map "z" 'bury-buffer)
  (define-key dired-mode-map "~" 'dired-flag-backup-files)
  (if dired-traditional
      (define-key dired-mode-map "X" 'dired-mark-delete)
    (define-key dired-mode-map "d" 'dired-mark-delete))
)


;; Dired mode is suitable only for specially formatted data.
(put 'dired-mode 'mode-class 'special)

(defun dired-mode (&optional dirname switches)
  "Mode for `editing' directory listings.
In dired, you are `editing' a list of the files in a directory and
  \(optionally) its subdirectories, in the format of `ls -lR'.
You can move using the usual cursor motion commands.
Letters no longer insert themselves.  Digits are prefix arguments.
Instead, type d to flag a file for Deletion.
Type m to Mark a file or subdirectory for later commands.
  Most commands operate on the marked files and use the current file
  if no files are marked.  Use a numeric prefix argument to operate on
  the next ARG (or previous -ARG if ARG<0) files, or just \\[universal-argument]
  to operate on the current file only.  Prefix arguments override marks.
  Mark-using commands display a list of failures afterwards.  Type \\[dired-why] to see
  why something went wrong.
Type u to Unflag a file (remove its D flag or any mark) or all files
  of a subdirectory.  Type DEL to back up one line and unflag.
Type x to eXecute the deletions requested.
Type f to Find the current line's file
  (or dired it in another buffer, if it is a directory).
Type o to find file or dired directory in Other window.
Type i to Insert a subdirectory in situ and K to kill it again or ^ to
  go back up.  Type v to view a file or go down to its in situ subdirectory.
Type ^ to go to or dired the parent directory.
Type < and > to move to files that are directories.
Type M-n, M-p, M-u, M-d to move to in situ subdirectory headerlines.
Type M-g to Go to a file's line, M-G to go to a subdir headerline.
Type # to flag temporary files (names beginning with #) for deletion.
Type ~ to flag backup files (names ending with ~) for deletion.
Type . to flag numerical backups for deletion.
  (Spares dired-kept-versions (or prefix argument) recent versions.)
Type + to create a new directory.
Type r to Rename a file or move the marked files to another directory.
Type c to Copy files.
Type R to rename by Regexp, \\[dired-upcase] and \\[dired-downcase] to
  change case of marked files.
Type D to Diff a file, M-~ to diff it with its backup.
Type l to reList marked files or a subdirectory.
Type s to toggle sorting by name/date, S to set dired-actual-switches.
Type g to read all currently expanded directories again.
  This retains all marks.
Space and Rubout can be used to move down and up by lines.
Also:
 C 	 -- compress files		  U -- uncompress files
 ! 	 -- run shell command on files    & -- background shell command
 M, G, O -- change mode, group or owner of files
 L, B 	 -- load or byte-compile emacs lisp files
 F, M-m  -- flag (`D') or mark (`*') files matching a regexp
 *, @, / -- (un)mark executables, symbolic links, directories
 $, = 	 -- (un)hide this or all subdirectories
 X       -- delete marked (as opposed to `D'-flagged) files

If dired ever gets confused, you can either type \\[revert-buffer] \
to read the
directories again, type \\[dired-mark-redisplay] \
to relist a single or the marked files or subdirectory, or
type \\[dired-build-subdir-alist] to parse the buffer again for the
directory tree.

Important customization variables: dired-ls-program,
  dired-listing-switches, dired-ls-F-marks-symlinks,
  dired-traditional, dired-shrink-to-fit, dired-confirm
  dired-pop-up, dired-trivial-filenames (q.v.)

Hooks: dired-load-hook, dired-mode-hook, dired-readin-hook (q.v.)

Keybindings:
\\{dired-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (make-local-variable 'revert-buffer-function)
  (setq revert-buffer-function 'dired-revert)
  (setq major-mode 'dired-mode)
  (setq mode-name "Dired")
  (make-local-variable 'dired-directory)
  (setq dired-directory (or dirname default-directory))
  (make-local-variable 'list-buffers-directory)
  (setq list-buffers-directory dired-directory)	; never used!?
  (make-local-variable 'dired-actual-switches)
  (setq dired-actual-switches (or switches
				  dired-listing-switches))
  (setq case-fold-search nil)
  (set (make-local-variable 'dired-used-F)
       (string-match "F" dired-actual-switches))
  (setq mode-line-buffer-identification;; '("Dired: %17b")
	;;>>> install (display revlevel in beta release only):
	(list (concat "Dired " dired-version ": %17b")))
  (setq buffer-read-only t)
  (use-local-map dired-mode-map)
  (make-local-variable 'minor-mode-alist)
  (setq selective-display t)		; for subdirectory hiding
  (dired-advertise)
  (make-local-variable 'dired-subdir-alist)
  (setq dired-subdir-alist nil)
  (dired-build-subdir-alist)
  (make-local-variable 'dired-sort-mode)
  (dired-sort-mode)
  (setq minor-mode-alist
	(cons '(dired-sort-mode dired-sort-mode)
	      minor-mode-alist))
  (run-hooks 'dired-mode-hook))


(defun dired-repeat-over-lines (arg function)
  ;; This version skips non-file lines.
  (beginning-of-line)
  (while (and (> arg 0) (not (eobp)))
    (setq arg (1- arg))
    (beginning-of-line)
    (while (and (not (eobp)) (dired-between-files)) (forward-line 1))
    (save-excursion (funcall function))
    (forward-line 1)
    (dired-move-to-filename))
  (while (and (< arg 0) (not (bobp)))
    (setq arg (1+ arg))
    (forward-line -1)
    (while (and (not (bobp)) (dired-between-files)) (forward-line -1))
    (beginning-of-line)
    (save-excursion (funcall function))
    (dired-move-to-filename)))

(defun dired-flag-file-deleted (arg)
  "In dired, flag the current line's file for deletion.
With arg, repeat over several lines."
  (interactive "p")
  (dired-repeat-over-lines arg
    '(lambda ()
       (let (buffer-read-only)
	 (delete-char 1)
	 (insert dired-del-marker)
	 nil))))

(defun dired-read-regexp (prompt)
;; This is an extra function so that gmhist can redefine it.
  (setq dired-flagging-regexp
	(read-string prompt dired-flagging-regexp)))

(defun dired-flag-regexp-files (regexp &optional arg marker-char)
  "In dired, flag all files containing the specified REGEXP for deletion.
Use `^' and `$' if the match should span the whole (non-directory
  part) of the filename.   Exclude subdirs by hiding them.
Directories are not flagged unless a prefix argument is given.
`.' and `..' are never marked."
  (interactive (list (dired-read-regexp "Flagging regexp: ")
		     current-prefix-arg))
  (let ((dired-marker-char (or marker-char dired-del-marker)))
    (dired-mark-if
     (and (or arg (not (looking-at dired-re-dir)))
	  (not (looking-at dired-re-dot))
	  (not (eolp))
	  (dired-this-file-matches regexp))
     "matching file")))

(defun dired-bury-or-kill-buffer (arg)
  "Bury the current dired buffer.  With a prefix argument, kill it instead."
  (interactive "P")
  (let (answer)
    (setq answer
	  (save-excursion
	    (goto-char (point-min))
	    (or (not (re-search-forward dired-re-mark nil t))
		(y-or-n-p
		 (format "There are pending marks - %s buffer anyway? "
			 (if arg "kill" "bury"))))))
    (if answer
	(if arg
	    (kill-buffer (buffer-name))
	  (bury-buffer)))))

(defun dired-summary ()
  (interactive)
  ;>> this should check the key-bindings and use substitute-command-keys if non-standard
  (message
   (if dired-traditional
       "d-elete, u-ndelete, x-ecute, f-ind, o-ther window, r-ename, c-opy, h-elp"
     "m-ark, u-nmark, d-elete, f-ind, o-ther window, r-ename, c-opy, h-elp")))

(defun dired-unflag (arg)
  "In dired, remove the current line's delete flag then move to next line."
  (interactive "p")
  (dired-repeat-over-lines arg
    '(lambda ()
       (let (buffer-read-only)
	 (delete-char 1)
	 (insert " ")
	 (forward-char -1)
	 nil))))

(defun dired-backup-unflag (arg)
  "In dired, move up a line and remove deletion flag there."
  (interactive "p")
  (dired-unflag (- arg)))

(defun dired-next-line (arg)
  "Move down ARG lines then position at filename."
  (interactive "p")
  (next-line arg)
  (dired-move-to-filename))

(defun dired-previous-line (arg)
  "Move up ARG lines then position at filename."
  (interactive "p")
  (previous-line arg)
  (dired-move-to-filename))

(defun dired-up-directory ()
  "Dired parent directory.  Tries first to find it in this buffer."
  (interactive)
  (let ((fn "..")
	(dir (dired-current-directory)))
    (setq fn (file-name-as-directory (expand-file-name fn dir)))
    (or (dired-goto-file (directory-file-name dir))
	(dired (expand-file-name	; give user a chance to abort
		(read-file-name "Dired: " fn fn t))))))

(defun dired-find-file ()
  "In dired, visit the file or directory named on this line."
  (interactive)
  (find-file (dired-get-filename)))

(defun dired-view-file ()
  "In dired, examine a file in view mode, returning to dired when done.
When file is a directory, tries to go to its in situ subdirectory."
  (interactive)
  (if (file-directory-p (dired-get-filename))
      (or (dired-goto-subdir (dired-get-filename))
	  (message "Directory %s not inserted - type i to insert or f to dired."
		   (dired-get-filename t)))
    (view-file (dired-get-filename))))

(defun dired-find-file-other-window ()
  "In dired, visit this file or directory in another window."
  (interactive)
  (find-file-other-window (dired-get-filename)))

; Now that there is dired-move-to-end-of-filename,
; use it in dired-get-filename.
(defun dired-get-filename (&optional localp no-error-if-not-filep)
  "In dired, return name of file mentioned on this line.
Value returned normally includes the directory name.
A non-nil 1st argument means use path name relative to
  default-directory, which may contain slashes if in a subdirectory.
A non-nil 2nd argument says return nil if no filename on this line,
  otherwise an error occurs."
  (let (case-fold-search file p1 p2)
    (save-excursion
      (if (setq p1 (dired-move-to-filename (not no-error-if-not-filep)))
	  (setq p2 (dired-move-to-end-of-filename no-error-if-not-filep))))
    ;; nil if no file on this line, but no-error-if-not-filep is t:
    (setq file (and p1 p2 (buffer-substring p1 p2)))
    (and file (dired-make-absolute file (dired-current-directory localp)))))

(defun dired-move-to-filename (&optional raise-error eol)
  "In dired, move to first char of filename on this line.
Returns position (point) or nil if no filename on this line."
  (or eol (setq eol (progn (end-of-line) (point))))
  (beginning-of-line)
  (if (eq system-type 'vax-vms)
      (if (re-search-forward ". [][.A-Z-0-9_$;<>]" eol t)
	  (backward-char 1)
	(if raise-error
	    (error "No file on this line.")
	  nil))
    ;; Unix case
    (if (re-search-forward
	 "\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\)[ ]+[0-9]+"
	 eol t)
	(progn
	  (skip-chars-forward " ")	; there is one SPC after day of month
	  (skip-chars-forward "^ " eol)	; move after time of day (or year)
	  (skip-chars-forward " " eol)	; there is one SPC before the file name
	  (point))
      (if raise-error
	  (error "No file on this line.")
	nil))))

(defun dired-move-to-end-of-filename (&optional no-error eol)
  ;; Assumes point is at beginning of filename,
  ;; thus the rwx bit re-search-backward below will succeed in *this* line.
  ;; So, it should be called only after (dired-move-to-filename t).
  ;; case-fold-search must be nil, at least for VMS.
  ;; On failure, signals an error or returns nil.
  (let (opoint flag ex sym hidden case-fold-search)
    (setq opoint (point))
    (or eol (setq eol (save-excursion (end-of-line) (point))))
    (setq hidden (and selective-display
		      (save-excursion (search-forward "\r" eol t))))
    (if hidden
	nil
      (if (eq system-type 'vax-vms)
	  ;; Non-filename lines don't match
	  ;; because they have lower case letters.
	  (re-search-forward "[][.A-Z-0-9_$;<>]+" eol t)
	;; Unix case
	(save-excursion
	  (or (re-search-backward
	       ;; Restrict perm bits to be non-blank,
	       ;; otherwise this matches one char to early (looking backward):
	       ;; "l---------" (some systems make symlinks that way)
	       ;; "----------" (plain file with zero perms)
	       "\\([^ ]\\)[-r][-w]\\([^ ]\\)[-r][-w]\\([^ ]\\)[-r][-w]\\([^ ]\\)"
	       nil t)
	      no-error
	      (error "No file on this line."))
	  (setq flag (buffer-substring (match-beginning 1) (match-end 1))
		sym (string= flag "l")
		;; ex is actually only needed when dired-used-F is t.
		ex (string-match
		    "[xst]";; execute bit set anywhere?
		    (concat
		     (buffer-substring (match-beginning 2) (match-end 2))
		     (buffer-substring (match-beginning 3) (match-end 3))
		     (buffer-substring (match-beginning 4) (match-end 4))))))
	(if sym
	    (if (re-search-forward " ->" eol t)
		(progn
		  (forward-char -3)
		  ;; we check that ls -lF really marks the link
		  (if (and dired-ls-F-marks-symlinks (eq (preceding-char) ?@))
		      (forward-char -1))))
	  (goto-char eol))
	(if (and dired-used-F
		 (or (string= flag "d")
		     (string= flag "s")
		     (and (not sym) ex))) ; ls -lF ignores x bits on symlinks
	    (forward-char -1))))
    (or no-error
	(not (eq opoint (point)))
	(error (if hidden
		   "File line is hidden, type $ to unhide."
		 "No file on this line.")))
    (if (eq opoint (point))
	nil
      (point))))

(defun dired-map-dired-file-lines (fn)
  ;; perform fn with point at the end of each non-directory line:
  ;; arguments are the short and long filename
  (save-excursion
    (let (filename longfilename buffer-read-only)
      (goto-char (point-min))
      (while (not (eobp))
	(save-excursion
	  (and (not (looking-at dired-re-dir))
	       (not (eolp))
	       (setq filename (dired-get-filename t t)
		     longfilename (dired-get-filename nil t))
	       (progn (end-of-line)
		      (funcall fn filename longfilename))))
	(forward-line 1)))))

;; Perhaps something could be done to handle VMS' own backups.

(defun dired-clean-directory (keep)
  "Flag numerical backups for deletion.
Spares dired-kept-versions latest versions, and kept-old-versions oldest.
Positive numeric arg overrides dired-kept-versions;
negative numeric arg overrides kept-old-versions with minus the arg.

To clear the flags on these files, you can use \\[dired-flag-backup-files]
with a prefix argument."
  (interactive "P")
  (setq keep (if keep (prefix-numeric-value keep) dired-kept-versions))
  (let ((early-retention (if (< keep 0) (- keep) kept-old-versions))
	(late-retention (if (<= keep 0) dired-kept-versions keep))
	(file-version-assoc-list ()))
    ;; Look at each file.
    ;; If the file has numeric backup versions,
    ;; put on file-version-assoc-list an element of the form
    ;; (FILENAME . VERSION-NUMBER-LIST)
    (dired-map-dired-file-lines 'dired-collect-file-versions)
    ;; Sort each VERSION-NUMBER-LIST,
    ;; and remove the versions not to be deleted.
    (let ((fval file-version-assoc-list))
      (while fval
	(let* ((sorted-v-list (cons 'q (sort (cdr (car fval)) '<)))
	       (v-count (length sorted-v-list)))
	  (if (> v-count (+ early-retention late-retention))
	      (rplacd (nthcdr early-retention sorted-v-list)
		      (nthcdr (- v-count late-retention)
			      sorted-v-list)))
	  (rplacd (car fval)
		  (cdr sorted-v-list)))
	(setq fval (cdr fval))))
    ;; Look at each file.  If it is a numeric backup file,
    ;; find it in a VERSION-NUMBER-LIST and maybe flag it for deletion.
    (dired-map-dired-file-lines 'dired-trample-file-versions)))

(defun dired-collect-file-versions (ignore fn)
  ;; If it looks like fn has versions, we make a list of the versions.
  ;; We may want to flag some for deletion.
    (let* ((base-versions
	    (concat (file-name-nondirectory fn) ".~"))
	   (bv-length (length base-versions))
	   (possibilities (file-name-all-completions
			   base-versions
			   (file-name-directory fn)))
	   (versions (mapcar 'backup-extract-version possibilities)))
      (if versions
	  (setq file-version-assoc-list (cons (cons fn versions)
					      file-version-assoc-list)))))

(defun dired-trample-file-versions (ignore fn)
  (let* ((start-vn (string-match "\\.~[0-9]+~$" fn))
	 base-version-list)
    (and start-vn
	 (setq base-version-list	; there was a base version to which
	       (assoc (substring fn 0 start-vn)	; this looks like a
		      file-version-assoc-list))	; subversion
	 (not (memq (string-to-int (substring fn (+ 2 start-vn)))
		    base-version-list))	; this one doesn't make the cut
	 (progn (beginning-of-line)
		(delete-char 1)
		(insert dired-del-marker)))))

(defun dired-flag-backup-and-auto-save-files ()
  "Flag all backup and temporary files for deletion.
Backup files have names ending in `~'.  Auto save file names usually
start with `#'."
  (interactive)
  (dired-flag-backup-files)
  (dired-flag-auto-save-files))

(defun dired-create-directory (directory)
  "Create a directory called DIRECTORY"
  (interactive
   (list (read-file-name "Create directory: " (dired-current-directory))))
  (let ((expanded (directory-file-name (expand-file-name directory))))
    (make-directory expanded)
    (dired-add-entry-all-buffers (file-name-directory expanded)
				 (file-name-nondirectory expanded))
  (dired-next-line 1)))


(defun dired-buffers (dir)
;; Return a list of buffers that dired DIR (possibly as subdir).
;; As a side effect, killed dired buffers for DIR are removed from
;; dired-buffers.
  (setq dir (file-name-as-directory dir))
  (let ((alist dired-buffers) result elt)
    (while alist
      (setq elt (car alist))
      (if (dired-in-this-tree dir (car elt))
	  (let ((buf (cdr elt)))
	    (if (buffer-name buf)
		(setq result (cons buf result))
	      ;; else buffer is killed - clean up:
	      (setq dired-buffers (delq elt dired-buffers)))))
      (setq alist (cdr alist)))
    result))

(defun dired-fun-in-all-buffers (directory fun)
  ;; In all buffers dired'ing DIRECTORY, run FUN.
  ;; FUN returns t for success, nil else.
  (let ((buf-list (dired-buffers directory)) buf success-list)
    (while buf-list
      (setq buf (car buf-list)
	    buf-list (cdr buf-list))
      (save-excursion
	(set-buffer buf)
	(if (funcall fun)
	    (setq success-list (cons (buffer-name buf) success-list)))))
    success-list))

(defun dired-add-entry-all-buffers (directory filename)
  (dired-fun-in-all-buffers
   directory
   (function (lambda () (dired-add-entry directory filename)))))

(defun dired-add-entry (directory filename)
  ;; Note that this adds the entry `out of order' if files sorted by
  ;; time, etc.
  ;; At least this version tries to insert in the right subdirectory.
  ;; And it skips "." or ".." (dired-trivial-filenames).
  ;; Hidden subdirs are exposed if a file is added there.
  (setq directory (file-name-as-directory directory))
  (let*
      ((opoint (point))
       (cur-dir (dired-current-directory))
       (reason
	(catch 'not-found
	  (if (string= directory cur-dir)
	      (progn;; unhide if necessary
		(if (dired-subdir-hidden-p cur-dir) (dired-unhide-subdir))
		;; We are already where we should be, except in one case:
		;; If point is before the *root* subdir line or its
		;; total line, inserting there is ugly.
		;; (Everything *before* the rootline is considered as
		;; belonging to the root dir, too - in contrast to other
		;; subdirs)
		(if (string= default-directory cur-dir)
		    (let ((p (save-excursion
			       (dired-goto-next-file)
			       (point))))
		      (if (<= (point) p)
			  (goto-char p)))))
	    ;; else try to find correct place to insert
	    (if (dired-goto-subdir directory)
		(progn;; unhide if necessary
		  (if (looking-at "\r");; point is at end of subdir line
		      (dired-unhide-subdir))
		  ;; found - skip subdir and `total' line
		  ;; and uninteresting files like . and ..
		  (dired-goto-next-nontrivial-file))
	      ;; not found
	      (throw 'not-found "Subdir not found")))
	  ;; found and point is at The Right Place:
	  (let (buffer-read-only)
	    (beginning-of-line)
	    (insert "  ")
	    (dired-ls (dired-make-absolute filename directory)
		      (concat dired-actual-switches "d"))
	    (forward-line -1)
	    (dired-move-to-filename t)	; raise an error if ls output
					; is strange
	    (let* ((beg (point))
		   (end (progn (dired-move-to-end-of-filename) (point))))
	      (setq filename (buffer-substring beg end))
	      (delete-region beg end)
	      (insert (file-name-nondirectory filename)))
	    (beginning-of-line)
	    (if dired-readin-hook
		(save-restriction
		  (narrow-to-region (point)
				    (save-excursion (forward-line 1) (point)))
		  (run-hooks 'dired-readin-hook)))
	    )
	  ;; return nil if all went well
	  nil)))
    (if reason
	(progn
	  (goto-char opoint)		; don't move away on failure
	  ;;-(message "Couldn't add %s%s: %s" directory filename reason)
	  ))
    (not reason)			; return t on succes, nil else
    ))

(defun dired-remove-entry-all-buffers (file)
  (dired-fun-in-all-buffers
   (file-name-directory file)
   (function (lambda () (dired-remove-entry file)))))

(defun dired-remove-entry (file)
  (save-excursion
    (and (dired-goto-file file)
	 (let (buffer-read-only)
	   (delete-region (progn (beginning-of-line) (point))
			  (save-excursion (forward-line 1) (point)))))))


(defun dired-diff (file &optional arg)
  "Compare this file with another (default: file at mark), by running `diff'.
The other file is the first file given to `diff'.
Prefix arg lets you edit the diff switches.  See the command `diff'."
  (interactive
   (let ((default (if (mark)
		      (save-excursion (goto-char (mark))
				      (dired-get-filename t t)))))
     (list (read-file-name (format "Diff %s with: %s"
				   (dired-get-filename t)
				   (if default
				       (concat "(default " default ") ")
				     ""))
			   (dired-current-directory) default t)
	   current-prefix-arg)))
  (diff file (dired-get-filename t) arg))

(defun dired-backup-diff (&optional arg)
  "Diff this file with its backup file.
Uses the latest backup, if there are several numerical backups.
If this file is a backup, diff it with its original.
The backup file is the first file given to `diff'.
Prefix arg lets you edit the diff switches.  See the command `diff'."
  (interactive "P")
  (let (bak ori (file (dired-get-filename)))
    (if (backup-file-name-p file)
	(setq bak file
	      ori (file-name-sans-versions file))
      (setq bak (or (latest-backup-file file)
		    (error "File has no backup: %s" file))
	    ori file))
    (diff bak ori arg)))

;;>>> install (move this function into files.el)
(defun latest-backup-file (fn)	; actually belongs into files.el
  "Return the latest existing backup of FILE, or nil."
  ;; First try simple backup, then the highest numbered of the
  ;; numbered backups.
  ;; Ignore the value of version-control because we look for existing
  ;; backups, which maybe were made earlier or by another user with
  ;; a different value of version-control.
  (or
   (let ((bak (make-backup-file-name fn)))
     (if (file-exists-p bak) bak))
   (let* ((base-versions (concat (file-name-nondirectory fn) ".~"))
	  (bv-length (length base-versions)))
     (car (sort
	   (file-name-all-completions base-versions (file-name-directory fn))
	   ;; bv-length is a fluid var for backup-extract-version:
	   (function
	    (lambda (fn1 fn2)
	      (> (backup-extract-version fn1)
		 (backup-extract-version fn2)))))))))

(defun dired-compress ()
  ;; Return nil for success, offending filename else.
  (let* (buffer-read-only
	 (from-file (dired-get-filename))
	 (to-file (concat from-file ".Z")))
    (if (save-excursion
	  (beginning-of-line)
	  (looking-at dired-re-sym))
	(progn
	  (dired-log (concat
		      "Attempt to compress a symbolic link:\n"
		      from-file))
	  from-file)
      (or (dired-check-process
	   "compress" (format "Compressing %s" from-file) "-f" from-file)
	  (progn (dired-redisplay to-file) nil)))))

(defun dired-uncompress ()
  (let* (buffer-read-only
	 (from-file (dired-get-filename))
	 (to-file (substring from-file 0 -2)))
    (or (dired-check-process
	 "uncompress" (format "Uncompressing %s" from-file) from-file)
	(progn (dired-redisplay to-file) nil))))

; The (un)compress functions are just mapped over all marked files
; It is not very effective to call many processes if one would suffice,
; but you can use dired-mark-shell-command if necessary,
; This version has the advantage of redisplaying after each
; (un)compress the corresponding (different!) filename.
; And it does not stop if a single file cannot be compressed.

(defun dired-mark-map-check (fun1 arg fmt type question)
;  "Run FUN1 on marked files (with second ARG like in dired-mark-map)
;and display failures using third arg FMT.  FMT must contain %d, %s, %s
;for failure-count, total-count, plural-s, failure-list.
;Fourth arg TYPE is a symbol describing the operation performed.
;Fifth arg QUESTION is a format string to pass to dired-yes."
  (if (or (not dired-confirm)
	  (and (listp dired-confirm)
	       (not (memq type dired-confirm)))
	  (dired-mark-wrapper
	   (progn
	    ;; just for the prompt and the pop up
	    (dired-mark-get-files t arg)
	    (dired-mark-pop-up
	     '(lambda ()
		;; This is strange: if dired-yes gets the EASY
		;; flag, calling y-or-n-p instead of yes-or-no-p, the
		;; cursor seems to be in the *Marked Files* buffer
		;; instead of in the minibuffer.
		;; (let ((cursor-in-echo-area t)) ...)
		;;; doesn't help either.
		  (dired-yes
		   (format question (dired-mark-prompt arg))))
	     nil type))))
      (let* ((total-list  (dired-mark-map (funcall fun1) arg))
	     (total (length total-list))
	     (failures (delq nil total-list))
	     (count (length failures)))
	(if failures
	    (message fmt count total (dired-plural-s count) failures)))))

(defun dired-mark-compress (&optional arg)
  "Compress marked (or next ARG) files."
  (interactive "P")
  (dired-mark-map-check
   'dired-compress arg "%d of %d compression%s failed - type W to see why %s"
   'compress "Compress %s "))

(defun dired-mark-uncompress (&optional arg)
  "Uncompress marked (or next ARG) files."
  (interactive "P")
  (dired-mark-map-check
   'dired-uncompress arg "%d of %d uncompression%s failed - type W to see why %s"
   'uncompress "Uncompress %s "))

;; Elisp commands on files

(defun dired-byte-recompile ()
  (let* (buffer-read-only
	 (from-file (dired-get-filename))
	 (new-file (concat from-file "c")))
    (if (not (string-match "\\.el$" from-file))
	(dired-make-relative from-file)	; non-nil for error
      (byte-compile-file from-file)
      (dired-remove-entry-all-buffers new-file)
      ;; This is to circumvent dired-mark-map's inability to cope with
      ;; a BODY that inserts additional lines *before* point.
      ;; This only works because dired-mark-map's searches *forward*.
      (forward-line)			; kludge kludge
      (dired-add-entry-all-buffers (file-name-directory new-file)
				   (file-name-nondirectory new-file))
      nil)))

(defun dired-mark-byte-recompile (&optional arg)
  "Byte recompile marked (or next ARG) Emacs lisp files."
  (interactive "P")
  (dired-mark-map-check
   'dired-byte-recompile arg "Skipped %d of %d non `.el' file%s %s"
   'byte-recompile "Byte recompile %s "))

(defun dired-load ()
  (let ((file (dired-get-filename)))
    (if (not (string-match "\\.elc?$\\|/\\.emacs$" file))
	(dired-make-relative file)	; non-nil for error
      (progn (load file nil nil t) nil))))

(defun dired-mark-load (&optional arg)
  "Load the marked (or next ARG) Emacs lisp files."
  (interactive "P")
  (dired-mark-map-check
   'dired-load arg "Skipped %d of %d non `.el' file%s %s"
   'load "Load %s "))

;; Change file modes.

; Don't use absolute path for ch{mod,grp} as /bin should be in
; any PATH.  However, chown is special: dired-chown-program.

(defun dired-mark-chxxx (string prg prg-path sym arg)
  (dired-mark-wrapper
   (let* ((files (dired-mark-get-files t arg))
	  (xxx (dired-mark-read-string
		(concat "Change " string " of %s to: ") nil sym arg))
	  (msg (format (concat prg " %s ") xxx))
	  (failure (apply 'dired-check-process prg-path msg xxx files)))
     (dired-mark-redisplay arg)
     (if failure (message "%s... error - type W to see why." msg)))))

(defun dired-mark-chmod (&optional arg)
  "Change mode of marked (or next ARG) files.
This calls chmod, thus symbolic modes like `g+w' are allowed."
  (interactive "P")
  (dired-mark-chxxx "Mode" "chmod" "chmod" 'chmod arg))

(defun dired-mark-chgrp (&optional arg)
  "Change group of marked (or next ARG) files."
  (interactive "P")
  (dired-mark-chxxx "Group" "chgrp" "chgrp" 'chgrp arg))

(defun dired-mark-chown (&optional arg)
  "Change owner of marked (or next ARG) files."
  (interactive "P")
  (dired-mark-chxxx "Owner" "chown" dired-chown-program 'chown arg))

(defun dired-redisplay (file)
  ;; Redisplay the file on this line.
  ;; Keeps any marks that may be present in column one.
  ;; Does not bother to update other dired buffers.
  (beginning-of-line)
  (let ((char (following-char)) (opoint (point)))
    (delete-region (point) (progn (forward-line 1) (point)))
    (if file
	(progn
	  (dired-add-entry (file-name-directory    file)
			   (file-name-nondirectory file))
	  ;; Replace space by old marker without moving point.
	  ;; Faster than goto+insdel inside a save-excursion?
	  (subst-char-in-region opoint (1+ opoint) ?\040 char))))
  (dired-move-to-filename))

(defun dired-mark-redisplay (&optional arg)
  "Redisplay all marked (or next ARG) files.

If on a subdir line, redisplay that subdirectory.  In that case,
a prefix arg lets you edit the ls switches used for the new listing."
  (interactive "P")
  (if (dired-get-subdir)
      (dired-insert-subdir
       (dired-get-subdir)
       (if arg (read-string "Switches for listing: " dired-actual-switches)))
    (message "Redisplaying...")
    (dired-mark-map (dired-redisplay (dired-get-filename)) arg)
    (dired-move-to-filename)
    (message "Redisplaying... done.")))

(defun dired-mark-kill (&optional arg)
  "Kill all marked lines (not files).
With a prefix arg, kill all lines not marked or flagged."
  (interactive "P")
  (save-excursion
    (goto-char (point-min))
    (let (buffer-read-only)
      (if (not arg)
	  (flush-lines (dired-marker-regexp))
	(while (not (eobp))
	  (if (or (dired-between-files)
		  (not (looking-at "^  ")))
	      (forward-line 1)
	    (delete-region (point) (save-excursion
				     (forward-line 1)
				     (point)))))))))

(defun dired-do-deletions ()
  "In dired, delete the files flagged for deletion."
  ;; The traditional way of deleting files in dired.
  (interactive)
  (let ((dired-marker-char dired-del-marker))
    (let ((regexp (dired-marker-regexp)) case-fold-search delete-list)
      (save-excursion
	(goto-char 1)
	(while (re-search-forward regexp nil t)
	  (setq delete-list
		(cons (cons (dired-get-filename t) (1- (point)))
		      delete-list))))
      (if (null delete-list)
	  (message "(No deletions requested)")
	(dired-internal-do-deletions delete-list nil)))))

(defun dired-mark-delete (&optional arg)
  "Delete all marked (or next ARG) files."
;; This is more consistent with the new file marking feature.
  (interactive "P")
  (dired-internal-do-deletions
   (dired-mark-map (cons (dired-get-filename t) (point)) arg)
   arg))

(defun dired-internal-do-deletions (l arg)
  ;; L is an alist of files to delete, with their buffer positions.
  ;; (car L) MUST be the LAST (bottommost) file in the dired buffer.
  ;; ARG is the prefix arg.
  (dired-mark-wrapper
   (progn
     (setq dired-mark-count (length l)
	   dired-mark-files (mapcar (function car) l))
     (if (dired-mark-pop-up
	  (function
	   (lambda ()
	     (dired-yes (format "Delete %s " (dired-mark-prompt arg)) nil)))
	  " *Deletions*" 'deletions)
	 (save-excursion
	   (let (failures)
	     ;; Files better be in reverse order for this loop!
	     ;; That way as changes are made in the buffer
	     ;; they do not shift the lines still to be changed.
	     ;; So the (point) values in l stay valid.
	     ;; Also, for subdirs in natural order, a subdir's files
	     ;; are deleted before the subdir itself - the other way
	     ;; around would not work.
	     (while l
	       (goto-char (cdr (car l)))
	       (let (buffer-read-only)
		 (condition-case err
		     (let ((fn (dired-make-absolute (car (car l))
						    default-directory)))
		       (if (and (file-directory-p fn)
				(not (file-symlink-p fn)))
			   ;; above might be slow, esp. with ange-ftp
			   ;; look into buffer instead?
			   (remove-directory fn)
			 (delete-file fn))
		       ;; if we get here, removing worked
		       (delete-region (progn (beginning-of-line) (point))
				      (progn (forward-line 1) (point)))
		       ;; remove expanded subdir of deleted dir, if any
		       (save-excursion
			 (if (dired-goto-subdir fn)
			     (dired-kill-subdir))))
		   (error
		    (dired-log (format "%s" err))
		    (setq failures (cons (car (car l)) failures)))))
	       (setq l (cdr l)))
	     (if failures
		 (message "Deletions failed: %s"
			  (prin1-to-string failures))
	       (message "%d deletion%s done."
			dired-mark-count (dired-plural-s)))))
       (message "(No deletions performed)"))))
  (dired-move-to-filename))

(defun dired-replace-in-string (regexp to string)
  ;; Replace REGEXP with TO in STRING and return result.
  ;; No \\DIGIT escapes will be recognized in TO.
  (let ((result "") (start 0) mb me)
    (while (string-match regexp string start)
      (setq mb (match-beginning 0)
	    me (match-end 0)
	    result (concat result (substring string start mb) to)
	    start me))
    (concat result (substring string start))))

(defun dired-next-dirline (arg)
  "Goto ARG'th next directory file line."
  (interactive "p")
  (if (if (> arg 0)
	  (re-search-forward dired-re-dir nil t arg)
	(re-search-backward dired-re-dir nil t
			    (if (save-excursion (beginning-of-line)
						(looking-at dired-re-dir))
				(- 1 arg)
			      (- arg))))
      (dired-move-to-filename)		; user may type `i' or `f'
    (error "No more subdirectories.")))

(defun dired-prev-dirline (arg)
  "Goto ARG'th previous directory file line."
  (interactive "p")
  (dired-next-dirline (- arg)))

(defun dired-unflag-all-files (flag &optional arg)
  "Remove a specific or all flags from every file.
With an arg, queries for each marked file.
Type \\[help-command] at that time for help."
  (interactive "sRemove flag: (default: all flags) \nP")
  (let ((count 0)
	(re (if (zerop (length flag)) dired-re-mark
	      (concat "^" (regexp-quote flag)))))
    (save-excursion
      (let (buffer-read-only case-fold-search dired-query
			     (help-form "\
Type Space or `y' to unflag one file, Delete or `n' to skip to next,
! to unflag all remaining files with no more questions."))
	(goto-char (point-min))
	(while (re-search-forward re nil t)
	  (if (or (not arg)
		  (dired-query "Unflag file `%s' ? " (dired-get-filename t)))
	      (progn (delete-char -1) (insert " ") (setq count (1+ count))))
	  (forward-line 1))))
    (message (format "Flags removed: %d %s" count flag) )))


(defun dired-kill-line (arg)
  "Kill this line (but not this file).
If file is displayed as in situ subdir, kill that as well.
In that case, a prefix arg means to kill just the subdir, not this line."
  (interactive "P")
  (let (buffer-read-only (file (dired-get-filename nil t)))
    (save-excursion (and file (dired-goto-subdir file) (dired-kill-subdir)))
    (or arg (delete-region (progn (beginning-of-line) (point))
			   (progn (forward-line 1) (point))))))

;;>>> install (move this into simple.el)
;; This function is missing in simple.el:
(defun copy-string-as-kill (string)
  "Save STRING as if killed in a buffer."
  (setq kill-ring (cons string kill-ring))
  (if (> (length kill-ring) kill-ring-max)
	(setcdr (nthcdr (1- kill-ring-max) kill-ring) nil))
  (setq kill-ring-yank-pointer kill-ring))

(defun dired-copy-filename-as-kill (&optional arg)
  "Copy this file (or subdir) name into the kill ring.
With a prefix arg, use the complete pathname of file.
Subdirs are always complete pathnames."
  (interactive "P")
  (copy-string-as-kill
   (or (dired-get-subdir)
       (if arg;; dired-get-filename's localp is not what we usually
	   (dired-get-filename);; want, esp. deep in a tree
	 (file-name-nondirectory (dired-get-filename)))))
  (message "%s" (car kill-ring)))

;; file marking

(defun dired-marker-regexp ()
  (concat "^" (regexp-quote (char-to-string dired-marker-char))))

(defun dired-mark-prompt (arg)
  ;; Either the current file name or the marker and a count of marked
  ;; files for use in a prompt.
  ;; Uses two fluid vars from dired-mark-wrapper: dired-mark-files and
  ;;   dired-mark-count==(length dired-mark-files)
  (if (eq dired-mark-count 1)
      (file-name-nondirectory (car dired-mark-files))
    ;; more than 1 file:
    (if (integerp arg)
	;; abs(arg) = dired-mark-count
	;; Perhaps this is nicer, but it also takes more screen space:
	;;(format "[%s %d files]" (if (> arg 0) "next" "previous")
	;;                        dired-mark-count)
	(format "[next %d files]" arg)
      (format "%c [%d files]" dired-marker-char dired-mark-count))))

(defun dired-yes (question &optional easy)
  ;;"Ask user a yes/no QUESTION, optionally taking it EASY."
  (if easy (y-or-n-p question) (yes-or-no-p question)))

(defun dired-query (prompt &rest args)
  ;; Format PROMPT with ARGS.  Query user and return nil or t.
  ;; Depends on var dired-query to be defined, e.g., let-bound.
  ;; Binding variable help-form will help the user who types C-h
  ;; Should perhaps recognize more keys.
  (if (eq ?! dired-query)
      t					; no more questions asked
    (let (;;(cursor-in-echo-area t)
	  ;; actually it looks nicer without - you can look at the
	  ;; dired buffer instead of at the prompt to decide
	  )
      (apply 'message prompt args)
      (setq dired-query (read-char)))
    (cond ((eq ?y dired-query) t)
	  ((eq ?\040 dired-query) t)	; SPC
	  ((eq ?n dired-query) nil)
	  ((eq ?\177 dired-query) nil)	; DEL
	  ((eq ?! dired-query) t))))

(defun dired-pop-up (buf)
  ;; Pop up buffer BUF.
  ;; If dired-shrink-to-fit is t, make its window fit its contents.
  ;; Could use Electric-pop-up-window instead.
  (set-buffer (get-buffer-create buf))
  (pop-to-buffer (current-buffer))
;  (funcall (if (> (length list) (* (window-height) 2))
;	       'switch-to-buffer 'switch-to-buffer-other-window)
;	   (current-buffer))
  ;; let window shrink to fit:
  (if dired-shrink-to-fit
      (let* ((window (selected-window))
	     (start (window-start window))
	     (window-lines (window-height window)))
	(goto-char (point-min))
	(enlarge-window (- (max (+ 2 (vertical-motion window-lines))
				window-min-height)
			   window-lines))
	(set-window-start (selected-window) start)))  )

(defun dired-mark-pop-up (function bufname type)
  ;;"Return FUNCTION's result after popping up a window (in a buffer
  ;;named BUFNAME, nil gives " *Marked Files*") showing the marked
  ;;files.  Uses function dired-pop-up.
  ;;The window is not shown if there is just one file or
  ;;dired-pop-up is a list and third arg TYPE is not a member
  ;;of this list."
  ;;dired-mark-files should be in reverse buffer order,
  ;;i.e., its car is the last marked file in the buffer.
  ;;Consing in dired-mark-map yields this order, but
  ;;dired-mark-get-files returns it sorted in ascending buffer order.
  (or bufname (setq bufname  " *Marked Files*"))
  (let ((list dired-mark-files))
    (if (or (not dired-pop-up)
	    (and (listp dired-pop-up)
		 (not (memq type dired-pop-up)))
	    (= dired-mark-count 1))
	(funcall function)
      (save-excursion
	(set-buffer (get-buffer-create bufname))
	(erase-buffer)
	(setq fill-column 70)
	(let ((l (reverse list)))
	  ;; Files should be in forward order for this loop.
	  (while l
	    (if (> (current-column) 59)
		(insert ?\n)
	      (or (bobp)
		  (indent-to (* (/ (+ (current-column) 19) 20) 20) 1)))
	    (insert (car l))
	    (setq l (cdr l)))))
      (save-window-excursion
	(dired-pop-up bufname)
	(funcall function)))))

;; Read arguments for a mark command of type TYPE,
;; perhaps popping up the list of marked files.
;; ARG is the prefix arg and indicates whether the files came from
;; marks (ARG=nil) or a repeat factor (integerp ARG).
;; If the current file was used, the list has but one element and ARG
;; does not matter. (It is non-nil, non-integer in that case, namely '(4)).

(defun dired-mark-read-string (prompt initial type arg)
  ;; PROMPT for a string, with INITIAL input.
  (dired-mark-pop-up
   (function
    (lambda ()
      (read-string (format prompt (dired-mark-prompt arg)) initial)))
   nil type))

(defun dired-mark-read-file-name (prompt dir type arg)
  (dired-mark-pop-up
   (function
    (lambda ()
      (read-file-name (format prompt (dired-mark-prompt arg)) dir)))
   nil type))

(defun dired-mark-file (arg)
  "In dired, mark the current line's file for later commands.
With arg, repeat over several lines.
Use \\[dired-unflag-all-files] to remove all flags."
  (interactive "p")
  (let (buffer-read-only)
    (dired-repeat-over-lines
     arg
     (function (lambda () (delete-char 1) (insert dired-marker-char))))))

(defun dired-mark-files (regexp &optional arg)
  "Mark all files matching REGEXP for use in later commands.
Directories are not flagged unless a prefix argument is given.
`.' and `..' are never marked.

This is an Emacs regexp, not a shell wildcard.	E.g., use \\.o$ for
object files - just .o will mark more than you might think.

An empty string will match all files except directories."
  (interactive
   (list (dired-read-regexp "Mark files (regexp): ")
	 current-prefix-arg))
  (dired-flag-regexp-files regexp arg dired-marker-char))

(defun dired-mark-symlinks (unflag-p)
  "Mark all symbolic links.
With prefix argument, unflag all those files."
  (interactive "P")
  (let ((dired-marker-char (if unflag-p ?\  dired-marker-char)))
    (dired-mark-if (looking-at dired-re-sym) "symbolic link")))

(defun dired-mark-dirlines (unflag-p)
  "Mark all directory file lines except `.' and `..'.
With prefix argument, unflag all those files."
  (interactive "P")
  (let ((dired-marker-char (if unflag-p ?\  dired-marker-char)))
    (dired-mark-if (and (looking-at dired-re-dir)
			(not (looking-at dired-re-dot)))
		   "directory file")))

(defun dired-mark-executables (unflag-p)
  "Mark all executable files.
With prefix argument, unflag all those files."
  (interactive "P")
  (let ((dired-marker-char (if unflag-p ?\  dired-marker-char)))
    (dired-mark-if (looking-at dired-re-exe) "executable file")))

(defun dired-flag-auto-save-files (&optional unflag-p)
  "Flag for deletion files whose names suggest they are auto save files.
A prefix argument says to unflag those files instead."
  (interactive "P")
  (let ((dired-marker-char (if unflag-p ?\  dired-del-marker))
	(bound (fboundp 'auto-save-file-name-p)))
    (dired-mark-if
       (and (not (looking-at dired-re-dir))
	    (if bound
		(let ((fn (dired-get-filename t t)))
		  (if fn (auto-save-file-name-p
			  (file-name-nondirectory fn))))
	      (if (save-excursion
		    (dired-move-to-filename)
		    (looking-at "#")))))
       "auto save file")))

(defun dired-flag-backup-files (&optional unflag-p)
  "Flag all backup files (names ending with `~') for deletion.
With prefix argument, unflag all those files."
  (interactive "P")
  (let ((dired-marker-char (if unflag-p ?\  dired-del-marker))
	(bound (fboundp 'backup-file-name-p)))
    (dired-mark-if
     (and (not (looking-at dired-re-dir))
	  (if bound
	      (let ((fn (dired-get-filename t t)))
		(if fn (backup-file-name-p fn)))
	    (save-excursion
	      (end-of-line)		; symlinks are never backups
	      (forward-char -1)
	      (looking-at "~"))))
     "backup file")))

(defun dired-mark-get-files (&optional localp arg)
  "Return the marked files as list of strings.
The list is in the opposite order as the buffer, that is, the car is the
  first marked file.
Values returned normally do include the directory name.
A non-nil first argument LOCALP means do not include it.
A non-nil second argument ARG forces to use other files.  If ARG is an
  integer, use the next ARG files.  If ARG is otherwise non-nil, use
  current file.

Sets the variables  dired-mark-count and  dired-mark-files."
  (setq dired-mark-files (dired-mark-map (dired-get-filename localp) arg)
	dired-mark-count (length dired-mark-files))
  dired-mark-files)


;;; Generic renaming, with special cases upcase, downcase, by regexp.

(defun dired-generic-rename (modify &optional predicate)
;;  "Rename all marked files to what MODIFY makes out of the filename
;;in the buffer.
;;Optional second arg PREDICATE restricts the files operated upon.
;;Both arguments are functions of zero arguments.
;;Displays and returns a list of failures (those files for which MODIFY
;;signalled an error)."
  (save-excursion
    (goto-char (point-min))
    (let ((total 0)
	  (re (dired-marker-regexp))
	  old new failures)
      (while (and (let (case-fold-search) (re-search-forward re nil t))
		  (setq old (dired-get-filename)))
	;;?(undo-boundary)
	(if (if predicate (funcall predicate) t)
	    (let (buffer-read-only)
	      (setq total (1+ total))
	      (funcall modify)
	      (undo-boundary)		; prepare to undo
	      (setq new (dired-get-filename))
	      (dired-count-up)
	      (condition-case err
		  (rename-file old new)
		(error
		 (progn
		   (undo)		; undo buffer change
		   (dired-log (format "%s" err))
		   (setq failures
			 (cons (dired-make-relative new) failures))))))))
      (if failures
	  (let ((num (length failures)))
	    (dired-log
	     (message "Rename failed [%d of %d] on existing file%s %s"
		      num total (dired-plural-s num) failures)))
	(message "%d file%s renamed." total (dired-plural-s)))
      ;; return failures or nil
      failures)))

(defun dired-generic-rename-region (fun)
;;  "Generic rename using FUN, a function of two args operating on the region.
;;The two args delimit the filename in a dired buffer."
  (dired-generic-rename
   (function (lambda ()
	       (dired-move-to-filename)
	       (funcall fun
			(point)
			(save-excursion
			  (dired-move-to-end-of-filename)
			  (point)))))))

(defun dired-upcase ()
  "Rename all marked files to upper case."
  (interactive)
  (dired-generic-rename-region'upcase-region))

(defun dired-downcase ()
  "Rename all marked files to lower case."
  (interactive)
  (dired-generic-rename-region 'downcase-region))

(defun dired-rename-regexp (regexp newname)
  "Rename all marked files containing REGEXP to NEWNAME.
As each match is found, the user must type a character saying
  what to do with it.  For directions, type \\[help-command] at that time.
NEWNAME may contain \\=\\<n> or \\& as in query-replace-regexp.
REGEXP defaults to the last regexp used.
But with a prefix arg, dired-basename-regexp is provided.  This makes
  the basename as \\1 and the extension as \\2 available in NEWNAME.
See command \\[dired-flag-regexp-files] for more info on how REGEXP is matched against filenames.
Skips to next file when a NEWNAME file is found that already exists."
  (interactive
   (let ((a1 (read-string "Rename from (regexp): "
			  (if current-prefix-arg
			      dired-basename-regexp
			    dired-flagging-regexp))))
     (list a1 (read-string (format "Rename %s to: " a1)))))
  (let (dired-query
	(help-form "\
Type Space or `y' to rename one match, Delete or `n' to skip to next,
! to rename all remaining matches with no more questions."))
    (dired-generic-rename
     (function (lambda ()		; MODIFY
		 (replace-match newname t)))
     ;; MODIFY depends on PREDICATE to run a match function 
     (function (lambda ()
		 (and (dired-query "Regexp rename file `%s' ?"
				   (dired-get-filename t))
		      (dired-this-file-matches regexp)))))))

(defun dired-this-file-matches (regexp)
; The obvious approach
;	  (let ((fn (dired-get-filename t t)))
;	    (if fn (string-match regexp fn)))
; fails in subdirs.
; But much worse, we can not use (replace-match) for renaming by
; regexp unless the match was in a buffer (not a string)
  (save-excursion
    (let ((beg (dired-move-to-filename)) end)
      (and beg
	   (setq end (dired-move-to-end-of-filename t))
      (save-restriction			; so that "^" in the
	(narrow-to-region beg end)	; regexp works.
	(goto-char beg)
	;; search is better than looking-at because then it is easy to
	;; replace "frizzle" by "frozzle" _anywhere_ in a name.
	;; "^" and "$" can still be used to anchor a match.
	;; caller decides about case-fold-search's value
	(re-search-forward regexp end t))))))

;;; Shell commands
;;>>> install (move this function into simple.el)
(defun shell-quote (filename)		; actually belongs into simple.el
  "Quote a file name for inferior shell (see variable shell-file-name)."
  ;; Quote everything except POSIX filename characters.
  ;; This should be safe enough even for really wierd shells.
  (let ((result "") (start 0) end)
    (while (string-match "[^---0-9a-zA-Z_./]" filename start)
      (setq end (match-beginning 0)
	    result (concat result (substring filename start end)
			   "\\" (substring filename end (1+ end)))
	    start (1+ end)))
    (concat result (substring filename start))))

(defun dired-read-shell-command (prompt arg)
  "Read a dired shell command prompting with PROMPT (using read-string).
ARG is the prefix arg and may be used to indicate in the prompt which
  files are affected.
This is an extra function so that you can redefine it, e.g., to use gmhist."
  (dired-mark-read-string prompt nil 'shell arg))

(defun dired-mark-background-shell-command (&optional arg)
  "Like \\[dired-mark-shell-command], but starts command in background.
This requires background.el to work."
  (interactive "P")
  (require 'background)
  (dired-mark-shell-command arg t))

(defun dired-mark-shell-command (&optional arg in-background)
  "Run a shell command on the marked files.
If there is output, it goes to a separate buffer.
The list of marked files is appended to the command string unless asterisks
  `*' indicate the place(s) where the list should go.  See variables
  dired-mark-prefix, -separator, -postfix.  If you have a curly brace
  expanding shell, you may want to set these to \"{\", \",\" and \"}\"
  if you really use commands like `mv *~ bak; compress bak/*~'.
If no files are marked or a specific numeric prefix arg is given, uses
  next ARG files.  As always, a raw arg (\\[universal-argument]) means the current file.
  The prompt mentions the file(s) or the marker, as appropriate.  With
  a zero argument, run command on each marked file separately: `cmd *
  foo' results in `cmd F1 foo; ...; cmd Fn foo'.
No automatic redisplay is attempted, as the file names may have
  changed.  Type \\[dired-mark-redisplay] to redisplay the marked files."
;;Functions dired-run-shell-command and dired-shell-stuff-it do the
;;actual work and can be redefined for customization.
  (interactive "P")
  (dired-mark-wrapper
    (let* ((on-each (equal arg 0))
	   (prompt (concat (if in-background "& on " "! on ")
			   (if on-each "each " "")
			   "%s: "))
	   (file-list (dired-mark-get-files t (if on-each nil arg)))
	   ;; Want to give feedback whether this file or marked files are used:
	   (command (dired-read-shell-command prompt (if on-each nil arg)))
	   (result
	    ;; It is probably not important to pass the the file list
	    ;; in the same order as in the buffer (topmost left).
	    ;; For `tar cvf foo.tar' it might be appropriate.
	    ;; But if we do it, reverse file-list only *after*
	    ;; displaying it in a pop-up buffer because in the pop-up
	    ;; it looks just right.
	    (dired-shell-stuff-it command
				  (nreverse file-list) ; buffer order
				  on-each arg)))
      ;; execute the shell command
      (dired-run-shell-command result in-background))))

(defun dired-shell-stuff-it (command file-list on-each &optional raw-arg)
;; "Make up a shell command line from COMMAND and FILE-LIST.
;; If ON-EACH is t, COMMAND should be applied to each file, else
;; simply concat all files and apply COMMAND to this.
;; FILE-LIST's elements will be quoted for the shell."
;; Might be redefined for smarter things and could then use RAW-ARG
;; (coming from interactive P and currently ignored) to decide what to do.
;; Smart would be a way to access basename or extension of file names.
;; See dired-trans.el for an approach to this.
  ;; Bug: There is no way to quote a *
  ;; On the other hand, you can never accidentally get a * into your cmd.
  (let ((stuff-it
	 (if (string-match "\\*" command)
	     (function (lambda (x)
			 (dired-replace-in-string "\\*" x command)))
	   (function (lambda (x) (concat command " " x))))))
    (if on-each
	(mapconcat stuff-it (mapcar (function shell-quote) file-list) ";")
      (let ((fns (mapconcat (function shell-quote)
			    file-list dired-mark-separator)))
	(if (> (length file-list) 1)
	    (setq fns (concat dired-mark-prefix fns dired-mark-postfix)))
	(funcall stuff-it fns)))))

;; This is an extra function so that it can be redefined for remote
;; shells or whatever.
(defun dired-run-shell-command (command &optional in-background)
  "Run shell COMMAND, optionally IN-BACKGROUND.
If COMMAND is longer than shell-maximum-command-length, you are asked
for confirmation."
  (if in-background
      (setq command (concat "cd " default-directory "; " command)))
  (if (or (and shell-maximum-command-length
	       (< (length command) shell-maximum-command-length))
	  (yes-or-no-p
	   (format
	    "Dired shell command is %d bytes long - execute anyway? "
	    (length command))))
      (if in-background
	  (background command)
	(shell-command command))))

(defun dired-mark-print (&optional arg)
  "Print the marked (or next ARG) files.
Uses the shell command in variable dired-print-command as default."
  (interactive "P")
  (dired-mark-wrapper
   (let* ((files (mapconcat (function shell-quote)
			    (dired-mark-get-files t arg)
			    " "))
	  (command (dired-mark-read-string
		    "Print %s with cmd: "  dired-print-command 'print arg)))
     (setq dired-print-command command)
     (dired-run-shell-command (format command files)))))


;;; Copy, move and rename

(defun dired-rename-visited (filename to-file)
  ;; Optionally rename the visited file of any buffer visiting this file.
  (and (get-file-buffer filename)
	 (y-or-n-p (message "Change visited file name of buffer %s too? "
			    (buffer-name (get-file-buffer filename))))
	 (save-excursion
	   (set-buffer (get-file-buffer filename))
	   (let ((modflag (buffer-modified-p)))
	     (set-visited-file-name to-file)
	     (set-buffer-modified-p modflag)))))

(defun dired-mark-cp-or-mv (fun msg msg1 &optional arg)
  ;; FUN has three args: file newname ok-if-already-exists
  ;; It is called for each marked file.
  ;; MSG describes the operation performed.
  ;; MSG1 is an alternate form for MSG if there is only one file.
  ;; ARG as in dired-mark-get-files.
  (dired-mark-wrapper
   (let* ((fn-list (dired-mark-get-files nil arg))
	  ;; this depends on dired-mark-get-files to be run first:
	  (target (expand-file-name
		   (dired-mark-read-file-name
		    (format "%s %%s to: " (if (= 1 dired-mark-count) msg1 msg))
		    (dired-current-directory) 'cp-or-mv arg)))
	  failures
	  (is-dir (file-directory-p target)))
     (if (and (> dired-mark-count 1)
	      (not is-dir))
	 (error "Marked %s: target must be a directory: %s" msg target))
     (let (to overwrite buffer-read-only)
       (or is-dir (setq to target))
       (or is-dir			; paranoid
	   (= 1 (length fn-list))
	   (error "Internal error: non-dir and more than 1 file: %s" fn-list))
       (mapcar
	(function
	 (lambda (from)
	   (if is-dir			; else to = target
	       (setq to (expand-file-name
			 (file-name-nondirectory from) target)))
	   (setq overwrite (file-exists-p to))
	   (condition-case err
	       (progn
		 (funcall fun from to 0)
		 (if overwrite;; if we get here, fun hasn't been aborted
		     ;; and the old entry has to be deleted
		     ;; before adding the new entry
		     (dired-remove-entry-all-buffers to))
		 (dired-add-entry-all-buffers (file-name-directory to)
					      (file-name-nondirectory to)))
	     (file-error		; aborted cp or mv
	      (progn
		(setq failures (cons (dired-make-relative from) failures))
		(dired-log (format "%s\n" err)))))))
	fn-list))
     (if failures
	 (let ((count (length failures)))
	   (dired-log (message "%s failed for %d file%s %s"
			       msg count (dired-plural-s count) failures))))))
  (dired-move-to-filename))

(defun dired-mark-copy (&optional arg)
 "Copy all marked (or next ARG) files."
  (interactive "P")
  (dired-mark-cp-or-mv 'copy-file "Copy" "Copy" arg))

(defun dired-mark-move (&optional arg)
  "Move all marked (or next ARG) files into a directory
\(or rename this file if none are marked or raw prefix given).

Note that directory is prompted for in the current dired subdirectory,
so moving point into the target subdir beforehand may save some typing."
  (interactive "P")
  (dired-mark-cp-or-mv
   (function (lambda (from to ok-flag)
	       (rename-file from to ok-flag)
	       (dired-remove-entry-all-buffers from)
	       (dired-rename-visited from to)))
   "Move" "Rename" arg))

;; tree dired

;;---------------------------------------------------------------------

(defvar dired-buffers nil
  ;; Enlarged/modified by dired-mode and dired-revert
  ;; Queried by function dired-buffers. When this detects a
  ;; killed buffer, it is removed from this list.
  "Alist of directories and their associated dired buffers.")

;;---------------------------------------------------------------------

;;; utility functions

(defconst dired-subdir-regexp "^. \\([^ ]*\\)\\(:\\)[\n\r]"
  "Regexp matching a maybe hidden subdirectory line in ls -lR output.
Subexpression 1 is subdirectory proper, no trailing slash.
The match starts at the beginning of the line and ends after the end
of the line (\\n or \\r).
Subexpression 2 must end right before the \\n or \\r.")

(defun dired-relative-path-p (file)
  ;;"Return t iff FILE is a relative path name.
  ;;Dired uses dired-make-absolute to convert it to an absolute pathname."
  ;; Only used in dired-normalize-subdir, but might perhaps be
  ;; redefined (for VMS?)
  (not (file-name-absolute-p file)))

(defun dired-make-absolute (file &optional dir)
  ;;"Convert FILE (a pathname relative to DIR) to an absolute pathname."
  ;; This should be good enough for ange-ftp, but might easily be
  ;; redefined (for VMS?).
  ;; It should be reasonably fast, though, as it is called in
  ;; dired-get-filename.
  (or dir (setq dir default-directory))
  (concat dir file))

(defun dired-make-relative (file &optional dir)
  ;;"Convert FILE (an absolute pathname) to a pathname relative to DIR.
  ;;Else error."
  ;; DIR must be file-name-as-directory, as with all directory args in
  ;; elisp code.
  (or dir (setq dir default-directory))
  (if (string-match (concat "^" (regexp-quote dir)) file)
      (substring file (match-end 0))
    (error  "%s: not in directory tree growing at %s." file dir)))

(defun dired-in-this-tree (file dir)
  ;;"Is FILE part of the directory tree starting at DIR?"
  (let (case-fold-search)
    (string-match (concat "^" (regexp-quote dir)) file)))

(defun dired-normalize-subdir (dir)
  ;; prepend default-directory if relative path name
  ;; and make sure it ends in a slash, like default-directory does
  ;; Make this "end in a slash or a colon" for ange-ftp.  The point is
  ;; that dired-make-absolute (i.e. concat) must suffice in
  ;; dired-get-filename to make a valid filename from a file and its
  ;; directory.
  (file-name-as-directory
   (if (dired-relative-path-p dir)
       (dired-make-absolute dir default-directory)
     dir)))

(defun dired-between-files ()
  ;; Point must be at beginning of line
  ;; Should be equivalent to (save-excursion (not (dired-move-to-filename)))
  ;; but faster.
  (or (looking-at "^$")
      (looking-at "^. *$")		; should not be marked
      (looking-at "^. total")		; but may be
      (looking-at dired-subdir-regexp)))

(defun dired-get-subdir ()
  ;;"Return the subdir name on this line, or nil."
  (save-excursion
    (beginning-of-line)
    (if (looking-at dired-subdir-regexp)
	(file-name-as-directory
	 (buffer-substring (match-beginning 1)
			   (match-end 1))))))

;;; We use an alist of directories for speed.

(defconst dired-subdir-alist nil
  "Association list of subdirectories and their buffer positions:

  \((lastdir . lastmarker) ... (default-directory . firstmarker)).

The markers point right at the end of the line, so that the cursor
looks at either \\n or \\r, the latter for a hidden subdir.")

(defun dired-clear-alist ()
  (while dired-subdir-alist
    (set-marker (cdr (car dired-subdir-alist)) nil)
    (setq dired-subdir-alist (cdr dired-subdir-alist))))

(defun dired-build-subdir-alist ()
  "Build dired-subdir-alist anew and return it's new value."
  (interactive)
  (dired-clear-alist)
  (save-excursion
    (let ((count 0))
      (goto-char (point-min))
      (setq dired-subdir-alist nil)
      (while (re-search-forward dired-subdir-regexp nil t)
	(setq count (1+ count))
	(message "%d" count)
	(dired-alist-add (buffer-substring (match-beginning 1)
					   (match-end 1))
			 (progn
			   (goto-char (match-end 2))
			   (point-marker))))
      (message "%d director%s." count (if (= 1 count) "y" "ies"))
      ;; return new alist:
      dired-subdir-alist)))

(defun dired-alist-add (dir new-marker)
  ;; Add new DIR at NEW-MARKER (at end of buffer, but beginning of alist!)
  ;; Should perhaps use setcar for speed?
  (setq dired-subdir-alist
	(cons (cons (dired-normalize-subdir dir) new-marker)
	      dired-subdir-alist)))

(defun dired-unsubdir (dir)
  ;; Remove DIR from the alist
  (setq dired-subdir-alist
	(delq (assoc dir dired-subdir-alist) dired-subdir-alist)))

(defun dired-goto-next-nontrivial-file ()
  ;; Position point on first nontrivial file after point.
  (dired-goto-next-file);; so there is a file to compare with
  (if (stringp dired-trivial-filenames)
      (while (and (not (eobp))
		  (string-match dired-trivial-filenames
				(file-name-nondirectory
				 (or (dired-get-filename nil t) ""))))
	(forward-line 1)
	(dired-move-to-filename))))

(defun dired-goto-next-file ()
  (while (and (not (dired-move-to-filename)) (not (eobp)))
    (forward-line 1)))

(defun dired-goto-subdir (dir)
  "Goto header line of DIR in this dired buffer."
  ;; Search for DIR (an absolute pathname) in alist and move to it.
  ;; Return buffer position on success, otherwise return nil.
  (interactive (list (expand-file-name
		      ;;(read-file-name "Goto directory: ")
		      (completing-read "Goto directory: " ; prompt
				       dired-subdir-alist ; table
				       nil ; predicate
				       t ; require-match
				       (dired-current-directory)))))
  (let ((elt (assoc (file-name-as-directory dir) dired-subdir-alist)))
    (and elt (goto-char (cdr elt)))))

(defun dired-goto-file (file)
  "Goto file line of FILE in this dired buffer."
  (interactive (list (expand-file-name
		      (read-file-name "Goto file: "
				      (dired-current-directory)))))
  (setq file (directory-file-name file)) ; does no harm if no directory
  (let (found case-fold-search)
    (save-excursion
      (if (dired-goto-subdir (file-name-directory file))
	  (let ((keep-going t)
		(match nil)
		(string (file-name-nondirectory file))
		(boundary (dired-subdir-max)))
	    (while keep-going
	      (setq keep-going
		    (and (< (point) boundary)
			 (setq match (search-forward string nil 'move))))
	      (if (and match (equal file (dired-get-filename nil t)))
		  (setq found (point) keep-going nil)))
	    )))
    (and found (prog1 (goto-char found) (dired-move-to-filename)))))

(defun dired-initial-position ()
  ;; Where point should go in new listings.
  ;; Point assumed at beginning of new subdir line.
  (end-of-line)
  (if dired-trivial-filenames (dired-goto-next-nontrivial-file)))

;;; moving by subdirectories

(defun dired-subdir-index (dir)
  ;; Return an index into alist for use with nth
  ;; for the sake of subdir moving commands.
  (let (found (index 0) (alist dired-subdir-alist))
    (while alist
      (if (string= dir (car (car alist)))
	  (setq alist nil found t)
	(setq alist (cdr alist) index (1+ index))))
    ;; (message "%s %s" dir (nth index dired-subdir-alist))
    (if found index nil)))

(defun dired-next-subdir (arg &optional no-error-if-not-found)
  "Go to next subdirectory, regardless of level.
Use 0 prefix argument to go to this directory's header line."
  (interactive "p")
  (let ((this-dir (dired-current-directory))
	pos index)
    ;; nth with negative arg does not return nil but the first element
    (setq index (- (dired-subdir-index this-dir) arg))
    (setq pos (if (>= index 0) (cdr (nth index dired-subdir-alist)) nil))
    (if pos
	(goto-char pos)			; exit with non-nil return value
      (if no-error-if-not-found
	  nil				; return nil if not found
	(error "No more directories.")))))

(defun dired-prev-subdir (arg &optional no-error-if-not-found)
  "Go to previous subdirectory, regardless of level.
When called interactively and not on a subdir line, go to subdir line."
  (interactive
   (list (if current-prefix-arg
	     (prefix-numeric-value current-prefix-arg)
	   (if (and (dired-get-subdir) (not (bolp))) 1 0))))
  (dired-next-subdir (- arg) no-error-if-not-found))

(defun dired-up-subdir (arg)
  "Go up ARG levels in the dired tree."
  (interactive "p")
  (let ((dir (concat (dired-current-directory) "..")))
    (while (> arg 1) (setq arg (1- arg) dir (concat dir "/..")))
    (setq dir (expand-file-name dir))
    (or (dired-goto-subdir dir)
	(error "Cannot go up to %s - not in this tree." dir))))

(defun dired-down-subdir (arg)
  "Go down ARG levels in the dired tree."
  (interactive "p")
  (let ((dir (dired-current-directory)) ; has slash
	case-fold-search		; filenames are case sensitive
	(tail "[^/]+"))			; at least one more path name component
    (while (> arg 1) (setq arg (1- arg) tail (concat tail "/[^/]+")))
    (if (re-search-forward		; can't use $ searches when
	 (concat "^. " dir tail ":[\n\r]") nil t) ; dir is hidden
	(backward-char 1)
      (error "At the bottom."))))

;;; hiding

(defun dired-subdir-hidden-p (dir)
  (save-excursion
    (and selective-display
	 (dired-goto-subdir dir)
	 (looking-at "\r"))))

(defun dired-unhide-subdir ()
  (let (buffer-read-only)
    (subst-char-in-region (dired-subdir-min) (dired-subdir-max) ?\r ?\n)))

(defun dired-hide-check ()
  (or selective-display
      (error "selective-display must be t for subdir hiding to work!")))

(defun dired-hide-subdir (arg)
  "Hide or unhide the current subdirectory and move to next directory.
Optional prefix arg is a repeat factor.
Use \\[dired-hide-all] to (un)hide all directories."
  (interactive "p")
  (dired-hide-check)
  (let (from-char to-char end-pos buffer-read-only)
    (dired-next-subdir 0)	; to end of subdir line
    (while (> arg 0)
      (setq arg (1- arg))
      (if (looking-at "\n")
	  (setq from-char ?\n to-char ?\r) ; hide
	(setq to-char ?\n from-char ?\r)) ; unhide
      (subst-char-in-region
       (point)
       (save-excursion
	 (or (setq end-pos (dired-next-subdir 1 t))
	     (goto-char (point-max)))
	 ;;(forward-line -1) does work only with \n, not \r
	 ;; search backward for \n or \r:
	 (skip-chars-backward (concat "^" (char-to-string from-char)))
	 ;; this is necessary, else blank lines will be deleted:
	 (if (= from-char ?\n) (backward-char 1))
	 (point))
       from-char to-char)
      (if end-pos (goto-char end-pos)))))

(defun dired-hide-all (arg)
  "Hide all subdirectories, leaving only their header lines.
If there is already something hidden, make everything visible again.
Use \\[dired-hide-subdir] to (un)hide a particular subdirectory."
  (interactive "P")
  (dired-hide-check)
  (let (buffer-read-only)
    (if (save-excursion
	  (goto-char (point-min))
	  (search-forward "\r" nil t))
	;; unhide - bombs on \r in filenames
	(subst-char-in-region (point-min) (point-max) ?\r ?\n)
      ;; hide
      (let ((pos (point-max))		; pos of end of last directory
	    (alist dired-subdir-alist))
	(while alist			; while there are dirs before pos
	  (subst-char-in-region (cdr (car alist)) ; pos of prev dir
				(save-excursion
				  (goto-char pos) ; current dir
				  (forward-line -1)
				  (point))
				?\n ?\r)
	  (setq pos (cdr (car alist)))	; prev dir gets current dir
	  (setq alist (cdr alist)))))))

(defun dired-undo ()
  "Undo in a dired buffer.
This doesn't recover lost files, it is just normal undo with temporarily
writeable buffer.  You can use it to recover killed lines or subdirs.
You might have to do \\[dired-build-subdir-alist] to parse the buffer again."
  (interactive)
  (let (buffer-read-only)
    (undo)))

(defun dired-advertise ()
  ;;"Advertise in  dired-buffers  what directory we dired."
  (if (memq (current-buffer) (dired-buffers default-directory))
      t					; we have already advertised ourselves
    (setq dired-buffers
	  (cons (cons default-directory (current-buffer))
		dired-buffers))))

; unused:
;(defun dired-unadvertise (dir)
;  ;; Remove DIR from the buffer alist in variable dired-buffers.
;  (setq dired-buffers
;      (delq (assoc dir dired-buffers) dired-buffers)))

;; This function is the heart of tree dired
(defun dired-current-directory (&optional relative)
  "Get the subdirectory to which this line belongs.
This returns a string with trailing slash, like default-directory.
Optional argument means return a name relative to default-directory."
  (let (elt
	dir
	(here (point))
	;; Under strange circumstances, when dired-revert calls
	;; dired-get-filename and thus this function, the alist is not
	;; defined.  I don't understand how this can happen.
	(alist (or dired-subdir-alist (dired-build-subdir-alist))))
    (while alist
      (setq elt (car alist)
	    dir (car elt))
      (if (<= (cdr elt) here)		; subdir line is part of subdir
	  ;; found - exit while
	  (setq alist nil)
	;; else have to loop once more
	(setq alist (cdr alist))))
    (if relative
	(dired-make-relative dir default-directory)
      dir)))

;; Subdirs start at the beginning of their header lines and end just
;; before the beginning of the next header line (or end of buffer).
;;
;; This is slightly different from dired-current-directory and
;; subdir-moving commands which place the boundary at the buffer
;; position contained in dired-subdir-alist, usually right after the
;; colon and before the end of the line.
;;
;; It looks nicer if point lands there instead at the beginning of the
;; line, more like dired-move-to-filename's way of positioning the
;; cursor.

(defun dired-subdir-min ()
  (save-excursion
    (end-of-line);;  necessary if on a subdir line
    (if (not (dired-prev-subdir 0 t))
	(error "Not in a subdir!")
      (beginning-of-line)
      (point))))

(defun dired-subdir-max ()
  (save-excursion
    (end-of-line);;  necessary if on a subdir line
    (if (not (dired-next-subdir 1 t))
	(point-max)
      (beginning-of-line)
      (point))))

(defun dired-kill-subdir (&optional no-build)
  "Remove all lines of current subdirectory.
Lower levels are unaffected."
  (interactive)
  (let (buffer-read-only)
    (end-of-line);;  necessary if on a subdir line
    (if (and (interactive-p)
	     (equal (dired-current-directory) default-directory))
	(error "Cannot kill top level directory."))
    (delete-region (dired-subdir-min) (dired-subdir-max))
    ;; leave one blank line when between directories:
    (skip-chars-backward " \n")
    (or (eobp) (forward-char 1))
    (while (and (not (eobp))
		(looking-at "[ \n]"))
      (delete-char 1))
    ;;(insert "\n")
    (or (eobp) (insert "\n  "))
    (or no-build (dired-unsubdir (dired-current-directory)))))

(defun dired-mark-files-in-region (start end &optional arg)
  (let (buffer-read-only)
    (if (> start end)
	(error "start > end"))
    (goto-char start)			; assumed at beginning of line
    (while (< (point) end)
      ;; Skip subdir line and following garbage like the `total' line:
      (while (and (< (point) end) (dired-between-files))
	(forward-line 1))
      (if (and (or arg (not (looking-at dired-re-dir)))
	       (not (looking-at dired-re-dot))
	       (dired-get-filename nil t))
	  (progn
	    (delete-char 1)
	    (insert dired-marker-char)))
      (forward-line 1))))

(defun dired-mark-subdir-files (&optional arg)
  "Mark all files except directories in this subdir.
With prefix arg, mark even directories (except `.' and `..')."
  (interactive "P")
  (let (buffer-read-only
	(p-min (dired-subdir-min)))
    (dired-mark-files-in-region p-min (dired-subdir-max) arg)))

(defun dired-mark-subdir-or-file (arg)
  "Mark the current (or next ARG) files.
If looking at a subdir, mark all its files except directories.
With prefix arg, mark even directories (except `.' and `..').

Use \\[dired-unflag-all-files] to remove all marks
and \\[dired-unmark-subdir-or-file] on a subdir to remove the marks in
this subdir."
  (interactive "P")
  (if (dired-get-subdir)
      (save-excursion
	(end-of-line)
	(dired-mark-subdir-files arg))
    (dired-mark-file (prefix-numeric-value arg))))

(defun dired-unmark-subdir-or-file (arg)
  "Unmark the current (or next ARG) files.
If looking at a subdir, unmark all its files except directories.
With prefix arg, unmark even directories (except `.' and `..')."
  (interactive "P")
  (let ((dired-marker-char ?\ )) (dired-mark-subdir-or-file arg)))

(defun dired-insert-subdir (dirname &optional switches)
  "Insert this subdirectory into the same dired buffer.
If subdirectory is already present, overwrites previous entry, else
appends at end of buffer.
With a prefix arg, you may edit the ls switches used for this listing.
This function takes some pains to conform to ls -lR output."
  (interactive
   (list (dired-get-filename)
	 (if current-prefix-arg
	     (read-string "Switches for listing: " dired-actual-switches))))
  (setq dirname (file-name-as-directory (expand-file-name dirname)))
  (dired-make-relative dirname default-directory) ; error on failure
  (let (beg end index old-marker new-marker mark-alist buffer-read-only case-fold-search)
    (or (file-directory-p dirname) (error  "Not a directory: %s" dirname))
    (if (setq index (dired-subdir-index dirname))
	;; subdir is already present - must erase it first
	(progn
	  (setq old-marker (cdr (nth index dired-subdir-alist)))
	  (goto-char old-marker)
	  (beginning-of-line)
	  (skip-chars-backward " \n")
	  ;; if previous subdir is hidden, there has to be no blank line
	  ;; else exactly one.
	  (if (and (not (bobp))
		   (not (= ?\r (preceding-char))))
	      (forward-char 1))
	  (setq beg (point))
	  (goto-char old-marker)
	  (setq end (dired-subdir-max))
	  (save-restriction
	    (narrow-to-region old-marker end)
	    ;; Must unhide to make remembering work:
	    (subst-char-in-region (point-min) (point-max) ?\r ?\n)
	    (setq mark-alist (dired-remember-marks)))
	  (delete-region beg end)
	  ;; must make an empty line to
	  ;; separate it from next subdir (if any)
	  (if (not (eobp))
	      (save-excursion (insert "\n"))))
      ;; else new subdir - append it at end of buffer
      (goto-char (point-max)))
    (or (bobp) (insert "\n"))
    (setq beg (point))
    (message "Reading directory %s..." dirname)
    (dired-ls dirname
	      (or switches
		  (dired-replace-in-string "R" "" dired-actual-switches))
	      nil t)
    (message "Reading directory %s...done" dirname)
    (indent-rigidly beg (point) 2)
    (if dired-readin-hook
	(save-restriction
	  (narrow-to-region beg (point))
	  (run-hooks 'dired-readin-hook)))
    ;;  call dired-insert-headerline afterwards, as under VMS dired-ls
    ;;  does insert the headerline itself and the insert function just
    ;;  moves point.
    (goto-char beg)
    (dired-insert-headerline dirname)	; must put point where
    (setq new-marker (point-marker))	; dired-build-subdir-alist
					; would
    (if index (set-marker old-marker new-marker))

    (if index				; if already present,
	(set-marker new-marker nil)	; new-marker is unused
      (dired-alist-add dirname new-marker))
    (if (and switches (string-match "R" switches))
	(dired-build-subdir-alist))
    (dired-initial-position)
    (save-excursion
      (goto-char beg)
      (dired-mark-remembered mark-alist))))

;; sorting

(defvar dired-sort-by-date-regexp "^-altR?$"
  "Regexp recognized by dired-sort-mode to set by date mode.")

(defvar dired-sort-by-name-regexp "^-alR?$"
  "Regexp recognized by dired-sort-mode to set by name mode.")

(defun dired-sort-mode ()
  "Set dired-sort-mode according to dired-actual-switches."
  (let (case-fold-search)
    (cond ((string-match dired-sort-by-date-regexp dired-actual-switches)
	   (dired-sort-by-date))
	  ((string-match dired-sort-by-name-regexp dired-actual-switches)
	   (dired-sort-by-name))
	  (t (dired-sort-other dired-actual-switches t)))))

(defun dired-sort-toggle ()
  "Toggle between sort by date/name."
  (interactive)
  (let (case-fold-search)
    (if (string-match dired-sort-by-date-regexp dired-actual-switches)
	(dired-sort-by-name)
      (dired-sort-by-date)))
  (revert-buffer))

;; We can't preserve arbitrary ls switches because they may override
;; the presence or absence of the `t' option.
;; And we have to make sure to set dired-actual-switches to a legal
;; value.
;; And when displaying `by name' or `by date' in the modeline, this
;; should correspond to a definite listing format.
;; All other formats are displayed literally.

(defun dired-sort-by-date ()
  ;; Force sort by date, but preserve `R' and `a' ls switches.
  (setq dired-actual-switches
	(let (case-fold-search)
	  (concat "-" (if (string-match "a" dired-actual-switches) "a" "")
		  "lt" (if (string-match "R" dired-actual-switches) "R" ""))))
  (setq dired-sort-mode " by date")
  (set-buffer-modified-p (buffer-modified-p))) ;No-op, but updates mode line.

(defun dired-sort-by-name ()
  ;; Force sort by name, but preserve `R' and `a' ls switches.
  (setq dired-actual-switches
	(let (case-fold-search)
	  (concat "-" (if (string-match "a" dired-actual-switches) "a" "")
		  "l" (if (string-match "R" dired-actual-switches) "R" ""))))
  (setq dired-sort-mode " by name")
  (set-buffer-modified-p (buffer-modified-p))) ;No-op, but updates mode line.

(defun dired-sort-other (switches &optional no-revert)
  "Specify dired-actual-switches for dired-mode.
Values matching dired-sort-by-date-regexp or dired-sort-by-name-regexp
set the minor mode accordingly, others appear literally in the mode line.
With prefix arg, don't revert the buffer afterwards."
  (interactive
   (list (read-string "ls switches (must contain -l): "
		      dired-actual-switches)
	 current-prefix-arg))
  (setq dired-actual-switches switches)
  (setq dired-sort-mode (concat " " dired-actual-switches))
  ;; might really be by name or by date
  (let (case-fold-search)
    (if (string-match dired-sort-by-date-regexp dired-actual-switches)
	(dired-sort-by-date)
      (if (string-match dired-sort-by-name-regexp dired-actual-switches)
	  (dired-sort-by-name))))
    (set-buffer-modified-p (buffer-modified-p)) ;No-op, but updates mode line.
    (or no-revert (revert-buffer)))

(if (eq system-type 'vax-vms)
    (load "dired-vms"))

(run-hooks 'dired-load-hook)		; for your customizations
