;;;; gmhist.el - Provide generic minibuffer history for commands
(defconst gmhist-version
  "$Id: gmhist.el,v 3.12 1991/02/01 14:48:12 sk Exp $")

;; Copyright (C) 1990 by Sebastian Kremer <sk@thp.uni-koeln.de>

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; LISPDIR ENTRY for the Elisp Archive ===============================
;; 
;;    gmhist|Sebastian Kremer|sk@thp.uni-koeln.de
;;    |Generic minibuffer history package.
;;    |$Date: 1991/02/01 14:48:12 $|$Revision: 3.12 $|

;; INSTALLATION ======================================================
;; 
;; Put this file into your load-path and the following in your
;; ~/.emacs or default.el:
;; 
;;   (autoload 'read-with-history-in "gmhist")
;;   (autoload 'read-file-name-with-history-in "gmhist")
;;   (autoload 'gmhist-make-magic "gmhist")

;; USAGE =============================================================
;;
;;   - as an Elisp programmer: use functions read-with-history-in
;;     and read-file-name-with-history-in inside the interactive
;;     clause of your functions instead of a string specification.
;;
;;   - as an Emacs user: To provide `simple' functions with history,
;;     just type M-x gmhist-make-magic and enter the name of the
;;     function, e.g., `eval-expression'.
;;     Type M-x gmhist-remove-magic to restore the function's old
;;     interactive behaviour.
;;     `Simple' functions are those that prompt for strings, file
;;     names or lisp objects and perhaps use prefix args.  Functions
;;     that operate on the region are *not* simple.
;;     See the file gmhist-app.el for examples with simple and
;;     other functions.

(provide 'gmhist)

(defvar gmhist-default-format "[%s] "	; saves screen space, too
  "Format used by gmhist to indicate the presence of a default value.
Set this to \"(default %s) \" to get the standard format.")

(defvar gmhist-search-history nil "History of history searches.")

;; The main entry points

(defun read-with-history-in (GMHIST-SYMBOL prompt &optional
		initial-contents GMHIST-READ)
  "\
Read a string, maintaining minibuffer history across calls in GMHIST-SYMBOL,
  prompting with PROMPT, with optional INITIAL-CONTENTS.
If optional fourth arg GMHIST-READ is non-nil, then interpret the
  result as a lisp object and return that object.
See variable gmhist-map for history commands available during edit.
Example:
    (defun foo-command (cmd)
      (interactive (list (read-with-history-in 'foo-history \"Foo: \" )))
      (message \"Fooing %s...\" cmd))

See function gmhist-make-magic to give an existing function history.

These properties (see function put) of GMHIST-SYMBOL are supported:

cursor-end    Put cursor at end of a newly retrieved history line.
cursor-pos    A regexp to put the cursor on.
keep-dups     If t, duplicate commands are remembered, too.
initial-hist  Initial value of the history list.
hist-ignore   Regexp of commands that are not to be added to the history.
default       An empty string as input will default to the last
	      command (whether the last command was added to the
	      history or not).  The default is stored in this
	      property, thus its initial value is the first default.
dangerous     Commands matching this regexp will never be the default.
no-default    If you don't want defaults at all, set this to t.
              This is the only way to allow empty strings as input.

Use the following only if you know what you are doing:

hist-map      Minibuffer key map to use instead of gmhist-map.
hist-function Name of a function to call instead of doing normal
              history processing.  read-with-history-in becomes
              effectively an alias for this function.
completion-table
completion-predicate
	      Used in completion on history strings, when the hist-map
	      property has gmhist-completion-map as value.
	      They default to the history list and nil, respectively.
	      Thus, to get completion on history items just do
		(put 'foo-history 'hist-map gmhist-completion-map)
"
  ;; We don't use property names prefixed with 'ghmist-' because the
  ;;   caller has freedom to use anything for GMHIST-SYMBOL.
  ;; The history list is never truncated, but I don't think this will
  ;;   cause problems.  All histories together have at most a few k.
  ;; You can use 'initial-hist to save (part of) the history in a file
  ;;   and provide it at next startup.  [Is there an exit-emacs-hook?]
  ;; You can use 'hist-function to implement a completely different
  ;;   history mechanism, e.g., a ring instead of a list, without having
  ;;   to modify existing gmhist applications.
  (let ((hist-function (get GMHIST-SYMBOL 'hist-function)))
    (if (fboundp hist-function)		; must be name, not lambda
	(funcall hist-function
		 GMHIST-SYMBOL prompt initial-contents GMHIST-READ)
      ;; else do the usual history processing simply using lists:
      (let* ((GMHIST-POSITION 0)	; fluid var for motion commands
	     command ignore history
	     (hist-map (or (get GMHIST-SYMBOL 'hist-map)
			   gmhist-map))
	     ;; Command is an s-exp when GMHIST-READ->t.  In this case,
	     ;; cannot have empty input:
	     (no-default (or GMHIST-READ (get GMHIST-SYMBOL 'no-default)))
	     (dangerous (if no-default nil (get GMHIST-SYMBOL 'dangerous)))
	     (default (if no-default nil (get GMHIST-SYMBOL 'default))))
	(run-hooks 'gmhist-hook)
	;; create history list if not already done
	(or (boundp GMHIST-SYMBOL)	; history list defaults to nil
	    (set GMHIST-SYMBOL (get GMHIST-SYMBOL 'initial-hist)))
	;; We will need the history list several times:
	(setq history (eval GMHIST-SYMBOL))
	;; Read the command from minibuffer, providing history motion
	;; key map and minibuffer completion
	(let ((minibuffer-completion-table
	       (or (get GMHIST-SYMBOL 'completion-table)
		   (mapcar 'list history)))
	      (minibuffer-completion-predicate ; defaults to nil:
	       (get GMHIST-SYMBOL 'completion-predicate)))
	  ;; let caller decide on minibuffer-completion-confirm's value
	  (setq command
		(read-from-minibuffer
		 (concat prompt
			 (if default (format gmhist-default-format default) ""))
		 initial-contents hist-map GMHIST-READ)))
	;; Care about default values unless forbidden:
	(if no-default
	    nil
	  (if (string= "" command)
	      (if default (setq command default)
		;; Empty commands may be useful for the caller,
		;; then we actually should not raise an error...
		;; Caller must use the no-default property in this case!
		(error "No default command for %s" GMHIST-SYMBOL)))
	  ;; Set default value unless it is dangerous:
	  (or (and (stringp dangerous)
		   (string-match dangerous (gmhist-stringify command)))
	      (put GMHIST-SYMBOL 'default command)))
	;; Add to history if first command, or not a dup, or not to be ignored
	(or (and history
		 (or (if (get GMHIST-SYMBOL 'keep-dups)
			 nil
		       (equal command (car history)))
		     (if (stringp (setq ignore (get GMHIST-SYMBOL 'hist-ignore)))
			 (string-match ignore (gmhist-stringify (car history))))))
	    (set GMHIST-SYMBOL (cons command history)))
	;; Return command's value to caller:
	command))))

;; low level function called by read-file-name-with-history-in
(defun gmhist-read-file-name
  (GMHIST-SYMBOL prompt &optional initial dir default mustmatch)
  "Args GMHIST-SYMBOL PROMPT &optional INITIAL DIR DEFAULT MUSTMATCH
Read file name, maintaining history in GMHIST-SYMBOL, prompting
  with PROMPT, with optional INITIAL input and completing in directory DIR. 
Value is not expanded!  You must call expand-file-name yourself.
Default name to fifth arg DEFAULT if user enters a null string.
Sixth arg MUSTMATCH non-nil means require existing file's name.
 Non-nil and non-t means also require confirmation after completion.
DIR defaults to current buffer's default-directory.

This function differs from read-file-name in providing a history of
filenames bound to GMHIST-SYMBOL and in providing an argument INITIAL
not present in read-file-name.

You should set INITIAL to default-directory's value for maximum
compatibility between these two functions.  Function
read-file-name-with-history-in does exactly this, so you probably want
to use that function.

This will yield a complete (but unexpanded) pathname for the file.
However, setting INITIAL eliminates the possibility of a default
value.  The last input may still be accessed by the user by typing
M-p.  See function read-with-history-in for more info on default
values.

Not setting INITIAL (or setting it to nil) will yield a basename for
the file, relative to default-directory."
  (put GMHIST-SYMBOL 'cursor-end t)
  (put GMHIST-SYMBOL 'hist-map
       (if (not mustmatch)
	   gmhist-completion-map
	 gmhist-must-match-map))
  (put GMHIST-SYMBOL 'completion-table 'read-file-name-internal)
  (put GMHIST-SYMBOL 'completion-predicate (or dir default-directory))
  (if default (put GMHIST-SYMBOL 'default default))
  (if initial
      (progn
	(put GMHIST-SYMBOL 'no-default t)
	;; Make prompt look nicer by un-expanding home dir
	;; read-file-name does this, too
	(let (start end (home (expand-file-name "~/")))
	  (and (setq start (string-match (regexp-quote home) initial))
	       (setq end (match-end 0))
	       (setq initial (concat "~/" (substring initial end)))))))
  (let ((minibuffer-completion-confirm (if (eq mustmatch t) nil t)))
    (read-with-history-in GMHIST-SYMBOL prompt initial)))

(defun read-file-name-with-history-in
  (GMHIST-SYMBOL prompt &optional dir default mustmatch)
  "Read file name, maintaining history in GMHIST-SYMBOL, prompting
with PROMPT, completing in directory DIR. 
Value is not expanded!  You must call expand-file-name yourself.
Default name to fourth arg DEFAULT if user enters a null string.
Fifth arg MUSTMATCH non-nil means require existing file's name.
 Non-nil and non-t means also require confirmation after completion.
DIR defaults to current buffer's default-directory.

This function differs from read-file-name in providing a history of
filenames bound to GMHIST-SYMBOL.  Also, the directory names in the
minibuffer are alway expanded, e.g, not with initial `~'.  See also
functions gmhist-read-file-name and read-with-history-in."
  (gmhist-read-file-name
   GMHIST-SYMBOL prompt default-directory dir default mustmatch))

;; Minibuffer key maps to implement history

(defvar gmhist-map nil
  "Key map for generic minibuffer history.
\\<gmhist-map>\\[gmhist-previous], \\[gmhist-next], \
\\[gmhist-beginning], \\[gmhist-end] move through, \
\\[gmhist-search-backward] and \\[gmhist-search-forward] search,
\\[gmhist-show] displays the history:
\\{gmhist-map}")

(defun gmhist-define-keys (map)
  "Bind the standard history commands in MAP, a key map."
  (define-key map "\M-p" 'gmhist-previous)
  (define-key map "\M-n" 'gmhist-next)
  (define-key map "\M-r" 'gmhist-search-backward)
  (define-key map "\M-s" 'gmhist-search-forward)
  (define-key map "\M-<" 'gmhist-beginning)
  (define-key map "\M->" 'gmhist-end)
  ;; Last two for bash/readline compatibility. Better M-a and M-e ?
  ;; In  query-replace, multi-line text together with next-line's
  ;; misfeature of adding blank lines really lets you lose without M-<
  ;; and M->.
  ;; I have a next-line-safe function take care of this.
  (define-key map "\M-?" 'gmhist-show))

(if gmhist-map
    nil
  (setq gmhist-map (copy-keymap minibuffer-local-map))
  (gmhist-define-keys gmhist-map))

(defvar gmhist-completion-map nil
  "Key map for generic minibuffer history with completion, see gmhist-map.")

(if gmhist-completion-map
    nil
  ;; If you use D. Gillespie's complete.el, you get it in gmhist, too:
  (setq gmhist-completion-map (copy-keymap minibuffer-local-completion-map))
  (gmhist-define-keys gmhist-completion-map))

(defvar gmhist-must-match-map nil
  "Key map for generic minibuffer history with completion that must match, see gmhist-map.") 

(if gmhist-must-match-map
    nil
  (setq gmhist-must-match-map (copy-keymap minibuffer-local-must-match-map))
  (gmhist-define-keys gmhist-must-match-map))

;; Minibuffer commands to implement history

(defun gmhist-goto (n)
  ;; Go to history position N, 1 <= N <= length of history
  ;; N<0 means the future and inserts an empty string
  ;; N=0 means initial-contents (fluid-var from read-with-history-in)
  (erase-buffer)
  (if (< n 0)
      nil
    (setq elt (if (= n 0)
		  (or initial-contents "")
		(nth (1- n) (eval GMHIST-SYMBOL))))
    (insert (gmhist-stringify elt))
    (goto-char (if (get GMHIST-SYMBOL 'cursor-end)
		   (point-max)
		 (point-min)))
    (let ((pos (get GMHIST-SYMBOL 'cursor-pos)))
      (if (stringp pos)
	  (if (eobp)
	      (re-search-backward pos nil t)
	    (re-search-forward pos nil t))))))

(defun gmhist-beginning ()
  "Go to the oldest command in the history."
  (interactive)
  (gmhist-goto (setq GMHIST-POSITION (length (eval GMHIST-SYMBOL)))))

(defun gmhist-end ()
  "Position before the most recent command in the history."
  (interactive)
  (gmhist-goto (setq GMHIST-POSITION 0)))

(defun gmhist-next (n)
  "Go to next history position."
  ;; fluid vars: GMHIST-SYMBOL GMHIST-POSITION GMHIST-READ
  ;; Inserts the next element of GMHIST-SYMBOL's value into the minibuffer.
  ;; GMHIST-POSITION is the current history position
  (interactive "p")
  ;; clip the new history position to the valid range:
  (let (elt
	(narg (min (max 0 (- GMHIST-POSITION n))
		   (length (eval GMHIST-SYMBOL)))))
    (if (= GMHIST-POSITION narg)
	(error "No %s item in %s"
	       (if (= 0 GMHIST-POSITION) "following" "preceding")
	       GMHIST-SYMBOL)
      (gmhist-goto (setq GMHIST-POSITION narg)))))

(defun gmhist-previous (n)
  "Go to previous history position."
  (interactive "p")
  (gmhist-next (- n)))

;; Searching the history

(defun gmhist-search-backward (regexp &optional forward)
  "Search backward in the history list for REGEXP."
  (interactive
   (let ((enable-recursive-minibuffers t))
     (list (read-with-history-in 'gmhist-search-history
				 "History search (regexp): "))))
  (let* (found
	 (direction (if forward -1 1))
	 (pos (+ GMHIST-POSITION direction)) ; find _next_ match!
	 (history (eval GMHIST-SYMBOL))
	 (len (length history)))
    (while (and (if forward (> pos 0) (<= pos len))
		(not (setq found
			   (string-match
			    regexp
			    (gmhist-stringify (nth (1- pos) history))))))
      (setq pos (+ pos direction)))
    (or found (error "%s not found in %s" regexp GMHIST-SYMBOL))
    (gmhist-goto (setq GMHIST-POSITION pos))))

(defun gmhist-search-forward (regexp)
  "Search forward in the history list for REGEXP."
  (interactive
   (let ((enable-recursive-minibuffers t))
     (list (read-with-history-in 'gmhist-search-history
				 "History search forward (regexp): "))))
  (gmhist-search-backward regexp t))

;; Misc.

(defun gmhist-stringify (elt)
  ;; If ELT is not a string, convert it to one.
  (if (stringp elt) elt (prin1-to-string elt)))

(defun gmhist-show ()
  "Show the history list in another buffer.
Use \\[scroll-other-window] to scroll, with negative arg to scroll back."
  (interactive)
  (let ((count 0))
  (with-output-to-temp-buffer "*History*"
    (mapcar
     (function
      (lambda (x)
	(princ (format "%4d: %s\n" (setq count (1+ count)) x))))
     (eval GMHIST-SYMBOL)))))

;; Hack up interactive specifications of existing functions

(defun gmhist-copy-function (fun)
  ;; copy-sequence does not copy recursively.
  ;; Iteration is faster than recursion, and we need just two levels
  ;; to be able to use setcdr to mung the interactive spec.
  (let (old new elt)
    (setq old (symbol-function fun))
    (while old
      (setq elt (car old)
	    old (cdr old)
	    new (cons (if (sequencep elt)
			  (copy-sequence elt)
			elt)
		      new)))
    (nreverse new)))

(defun gmhist-check-autoload (fun)
  "If FUN is an autoload, load its definition."
  (let ((lis (symbol-function fun)))
    (if (eq 'autoload (car lis))
	(load (nth 1 lis)))))

(defun gmhist-replace-spec (fun new-spec &optional copy-first)
  "Replace the interactive specification of FUN with NEW-SPEC.
FUN must be a symbol with a function definition.
Autoload functions are taken care of by loading the appropriate file first.
If FUN is a pure storage function (one dumped into Emacs) it is first
  copied onto itself, because pure storage cannot be modified.
  Optional non-nil third arg COPY-FIRST is used internally for this.
The old spec is put on FUN's gmhist-old-interactive-spec property.  
  That property is never overwritten by this function.  It is used by
  function gmhist-remove-magic."
  (gmhist-check-autoload fun)
  (if copy-first			; copy (from pure storage)
      (fset fun (gmhist-copy-function fun)))
  (let ((flambda (symbol-function fun)) fint old-spec)
    (setq fint (nth 2 flambda))
    (if (or (stringp fint)
	    (integerp fint))
	(setq fint (nth 3 flambda)))
    (or (eq (car-safe fint) 'interactive)
	(error "Not an interactive command: %s!" fun))
    ;; Save old interactive spec as property of FUN:
    (setq old-spec (nth 1 fint))
    (or (get fun 'gmhist-old-interactive-spec)
	(put fun 'gmhist-old-interactive-spec old-spec))
    ;; Replace '(interactive OLD-SPEC) with '(interactive NEW-SPEC)
    (if copy-first
	;; This should not fail - if it does, we must abort.
	(setcdr fint (list new-spec))
      ;; else prepare for a second try
      (condition-case err
	  (setcdr fint (list new-spec))
	;; Setcdr bombs on preloaded functions:
	;;     (error "Attempt to modify read-only object")
	;; There seems to be no simple way to test whether an object
	;; resides in pure storage, so we let it bomb and try again.
	(error (gmhist-replace-spec fun new-spec t))))))

(defun gmhist-spec (fun)
  "Get the current interactive specification for FUN (a symbol).
Signal an error if FUN is not interactive."
  (let ((flambda (symbol-function fun)) fint)
    (setq fint (nth 2 flambda))
    (if (or (stringp fint)
	    (integerp fint))
	(setq fint (nth 3 flambda)))
    (or (eq (car-safe fint) 'interactive)
	(error "Cannot get spec of a non-interactive command: %s!" fun))
    (nth 1 fint)))

;; Automagic gmhistification

;; There should be a builtin split function - inverse to mapconcat.
(defun gmhist-split (pat str &optional limit)
  "Splitting on regexp PAT, turn string STR into a list of substrings.
Optional third arg LIMIT (>= 1) is a limit to the length of the
resulting list.
Thus, if SEP is a regexp that only matches itself,

   (mapconcat 'identity (gmhist-split SEP STRING) SEP)

is always equal to STRING."
  (let* ((start (string-match pat str))
	 (result (list (substring str 0 start)))
	 (count 1)
	 (end (if start (match-end 0))))
    (if end				; else nothing left
	(while (and (or (not (integerp limit))
			(< count limit))
		    (string-match pat str end))
	  (setq start (match-beginning 0)
		count (1+ count)
		result (cons (substring str end start) result)
		end (match-end 0)
		start end)
	  ))
    (if (and (or (not (integerp limit))
		 (< count limit))
	     end)			; else nothing left
	(setq result
	      (cons (substring str end) result)))
    (nreverse result)))

(defun gmhist-new-spec (fun &optional hist)
  "Return a new interactive specification for FUN, suitable for use
with setcdr in function gmhist-replace-spec.
Use symbol HIST to store the history.  HIST defaults to `FUN-history'.
The returned spec does the same as the old one, only with history in HIST.

Currently recognized key letters are: `s', `x', `f' , `F', `p' and
  `P'.
 
If FUN is an autoload object, its file is loaded first.

Signals an error if FUN's interactive string contains other key
  letters than those mentioned above or has no interactive string." 
  ;; Need more code letters.
  ;; Does not handle `*' as first char in interactive string.
  (or hist (setq hist (intern (concat (symbol-name fun) "-history"))))
  (gmhist-check-autoload fun)
  (let ((spec (gmhist-spec fun)) spec-list elt new-spec new-elt)
    (if (stringp spec)
	(setq spec-list (mapcar '(lambda (x)
				   (cons (aref x 0) (substring x 1)))
				(gmhist-split "\n" spec)))
      (error "Can't gmhistify %s's spec: %s" fun spec))
  ;; spec-list is an alist with elements (CHAR . PROMPT)
  ;; CHAR is an interactive key letter, PROMPT the associated prompt,
  ;; e.g., "sPrompt" -> (?s . "Prompt").
    (while spec-list
      (setq elt (car spec-list)
	    spec-list (cdr spec-list)
	    char (car elt)
	    prompt (cdr elt))
      (cond ((eq char '?s)		; string
	     (setq new-elt (list 'read-with-history-in
				 (list 'quote hist)
				 prompt)))
	    ((eq char ?f)	; existing file name
	     (setq new-elt (list 'read-file-name-with-history-in
				 (list 'quote hist)
				 prompt
				 nil	; dir
				 nil	; default
				 '(quote 'confirm))))
	    ((eq char ?F)	; possibly nonexistent file name
	     (setq new-elt (list 'read-file-name-with-history-in
				 (list 'quote hist)
				 prompt)))
	    ((memq char '(?x))		; lisp expression, unevaluated
	     (setq new-elt (list 'read-with-history-in
				 (list 'quote hist)
				 prompt nil
				 ;; have to tell gmhist to read s-exps
				 ;; instead of strings:
				 t)))
	    ((eq char ?p)		; cooked prefix arg
	     (setq new-elt '(prefix-numeric-value current-prefix-arg)))
	    ((eq char ?P)		; raw prefix arg
	     (setq new-elt 'current-prefix-arg))
	    (t
	     (error "Can't gmhistify interactive spec `%c'!" char)))
      (setq new-spec (cons new-elt new-spec)))
    (setq new-spec (cons 'list (nreverse new-spec)))))

(defun gmhist-make-magic (fun &optional hist)
  "Make FUN magically maintain minibuffer history in symbol HIST.
HIST defaults to `FUN-history'.
This works by modifying the interactive specification.  See also
  function gmhist-replace-spec.
The magic goes away when you call gmhist-remove-magic on FUN."
  (interactive "CPut gmhist magic on command: ")
  (gmhist-replace-spec fun (gmhist-new-spec fun hist)))

(defun gmhist-remove-magic (fun)
  "Remove the magic that gmhist-make-magic put on FUN,
restoring the old interactive spec." 
  (interactive "CRemove gmhist magic from command: ")
  (gmhist-replace-spec
   fun
   (or (get fun 'gmhist-old-interactive-spec)
       (error "Can't find %s's old interactive spec!"))))

(run-hooks 'gmhist-load-hook)

;; Examples:
;;
;;(defun foo-command (cmd)
;;  (interactive (list
;;	       (read-with-history-in 'foo-history "Foo: ")))
;;  (message "Foo %s" cmd))
;;
;;
;;;(put 'foo-history 'hist-map minibuffer-local-map) ; disable motion ...
;;;(put 'foo-history 'hist-function 'gmhist-read-nohistory) ; and history
;;
;;(put 'foo-history 'hist-function nil) ; enable history ...
;;(put 'foo-history 'hist-map nil) ; and motion again
;;
;;(defun gmhist-read-nohistory (symbol prompt initial-input read)
;;  "An example function to put on the hist-function property."
;;  (message "read-nohistory...")
;;  (sit-for 2)
;;  (read-string prompt initial-input))
;;
;; Example for reading file names:
;;(defun bar-command (cmd)
;;  (interactive
;;   (list
;;    (read-file-name-with-history-in
;;     ;; HIST-SYM  PROMPT  DIR DFLT MUSTMATCH
;;     'bar-history "Bar: " nil nil 'confirm)))
;;  (message "Bar %s" cmd))
;;
;; Example function to apply gmhist-make-magic to.
;; Compare the missing initial input in bar to the magic version of zod.
;;(defun zod-command (cmd)
;;  (interactive "fZod: ")
;;  (message "Zod %s" cmd))
