/*
 *     double strtod (str, endptr);
 *     const char *str;
 *     const char **endptr;
 *		if !NULL, on return, points to char in str where conv. stopped
 *
 *     double atof (str)
 *     const char *str;
 *
 * recognizes:
 *   [spaces] [sign] digits [ [.] [ [digits] [ [e|E|d|D] [space|sign] [int]]]]
 *
 * returns:
 *	the number
 *		on overflow: HUGE_VAL and errno = ERANGE
 *		on underflow: -HUGE_VAL and errno = ERANGE
 *
 * bugs:
 *   naive over/underflow detection
 *
 *	++jrb bammi@dsrgsun.ces.cwru.edu
 *
 * 10/30/89:
 *      took stab at overflow detection problem,
 *      use <math.h>,
 *      added \f \r \n \v to local "isspace" macro,
 *      and removed naughty code setting errno to 0. [atw]
 *      Also changed strtod to return original pointer if no
 *      conversion performed.
 * 10/02/89:
 *      in-line 68K code removed, replaced with pure C code. [atw]
 * 07/29/89:
 *	ok, before you beat me over the head, here is a hopefully
 *	much improved one, with proper regard for rounding/precision etc
 *	(had to read up on everything i did'nt want to know in K. V2)
 *	the old naive coding is at the end bracketed by ifdef __OLD__.
 *	modeled after peter housels posting on comp.os.minix.
 *	thanks peter!
 *		++jrb
 */

/* define to turn off assertion checking */
#define NDEBUG
/* define to get a self-test routine */
/* #define TEST */

#include "lib.h"

#include <math.h>
#include <assert.h>

#ifndef NULL
#define NULL (void *)0
#endif

#define Ise(c)		((c == 'e') || (c == 'E') || (c == 'd') || (c == 'D'))
#define Isdigit(c)	((c <= '9') && (c >= '0'))
#define Isspace(c)	((c == ' ') || (c == '\t') || (c=='\n') || (c=='\v') \
                         || (c == '\r') || (c == '\f'))
#define Issign(c)	((c == '-') || (c == '+'))
#define Val(c)		((c - '0'))

#define MAXDOUBLE	DBL_MAX
#define MINDOUBLE	DBL_MIN

#define MAXF  1.797693134862316
#define MINF  2.225073858507201
#define MAXE  308
#define MINE  (-308)

/*
 * [atw] multiply 64 bit accumulator by 10 and add digit.
 * The KA/CA way to do this should be to use
 * a 64-bit integer internally and use "adjust" to
 * convert it to float at the end of processing.
 */
static int __ten_mul(acc, digit)
     double *acc;
     int digit;
{
  /* [atw] Crude, but effective (at least on a KB)...
   */
  *acc *= 10;
  *acc += digit;
  
  return 0;	/* no overflow */
}

/*
 * compute 10**x by successive squaring.
 */
static const double
  exp10(x)
unsigned x;
{
  static double powtab[] = {1.0,
                            10.0,
			    100.0,
			    1000.0,
			    10000.0};

  if (x < (sizeof(powtab)/sizeof(double)))
    return powtab[x];
  else if (x & 1)
    return 10.0 * exp10(x-1);
  else
    return exp10(x/2) * exp10(x/2);
}

/*
 * return (*acc) scaled by 10**dexp.
 */
double __adjust(acc, dexp, sign)
     double *acc;	/* the 64 bit accumulator */
     int     dexp;	/* decimal exponent       */
     int	sign;	/* sign flag		  */
{
  double r, mult;

  if (dexp > MAXE)
    {
      errno = ERANGE;
      return (sign) ? -HUGE_VAL : HUGE_VAL;
    }
  else if (dexp < MINE)
    {
      errno = ERANGE;
      return 0.0;
    }

  r = *acc;
  if (sign)
    r = -r;
  if (dexp==0)
    return r;

  if (dexp < 0)
    return r / exp10(abs(dexp));
  else
    return r * exp10(dexp);
}

/* flags */
#define SIGN	0x01
#define ESIGN	0x02
#define DECP	0x04

double strtod (s, endptr)
     const char *s;
     char **endptr;
{
  const char *start=s;
  double accum = 0.0;
  int flags = 0;
  int texp  = 0;
  int e     = 0;
  int conv_done = 0;
#ifdef __STDC__
  int __ten_mul(double *, int);
#else
  extern int __ten_mul();
#endif
  
  assert ((s != NULL));  
  while(Isspace(*s)) s++;
  if(*s == '\0')
    {	/* just leading spaces */
      if(endptr != NULL) *endptr = start;
      return 0.0;
    }
  
  
  if(Issign(*s))
    {
      if(*s == '-') flags = SIGN;
      if(*++s == '\0')
	{   /* "+|-" : should be an error ? */
	  if(endptr != NULL) *endptr = start;
	  return 0.0;
	}
    }
  
  for(; (Isdigit(*s) || (*s == '.')); s++)
    {
      conv_done = 1;
      if(*s == '.')
	flags |= DECP;
      else
	{
	  if( __ten_mul(&accum, Val(*s)) ) texp++;
	  if(flags & DECP) texp--;
	}
    }
  
  if(Ise(*s))
    {
      conv_done = 1;
      if(*++s != '\0') /* skip e|E|d|D */
	{  /* ! ([s]xxx[.[yyy]]e)  */
	  
	  while(Isspace(*s)) s++; /* Ansi allows spaces after e */
	  if(*s != '\0')
	    { /*  ! ([s]xxx[.[yyy]]e[space])  */
	      
	      if(Issign(*s))
		if(*s++ == '-') flags |= ESIGN;
	      
	      if(*s != '\0')
		{ /*  ! ([s]xxx[.[yyy]]e[s])  -- error?? */
		  
		  for(; Isdigit(*s); s++)
		    if (e < MAXE) /* prevent from grossly overflowing */
		      e = e*10 + Val(*s);
		  
		  /* dont care what comes after this */
		  if(flags & ESIGN)
		    texp -= e;
		  else
		    texp += e;
		}
	    }
	}
    }
  
  if(endptr != NULL) 
    *endptr = (conv_done) ? s : start;
  
  return __adjust(&accum, (int)texp, (int)(flags & SIGN));
  
}

double atof(s)
     const char *s;
{
  return strtod(s, NULL);
}

#ifdef TEST
#ifdef __MSHORT__
#error "please run this test in 32 bit int mode"
#endif

#define NTEST 10000L

atof_test()
{
  
  double expected, result, e, max_abs_err;
  char buf[128];
  register long i, errs;
  register int s;
#ifdef __STDC__
  double atof(const char *);
  int rand(void);
#else
  extern double atof();
  extern int rand();
#endif
  
#if 0
  expected = atof("3.14159265358979e23");
  expected = atof("3.141");
  expected = atof(".31415"); 
  printf("%f\n\n", expected);
  expected = atof("3.1415"); 
  printf("%f\n\n", expected);
  expected = atof("31.415"); 
  printf("%f\n\n", expected);
  expected = atof("314.15"); 
  printf("%f\n\n", expected);
  
  expected = atof(".31415"); 
  printf("%f\n\n", expected);
  expected = atof(".031415"); 
  printf("%f\n\n", expected);
  expected = atof(".0031415"); 
  printf("%f\n\n", expected);
  expected = atof(".00031415"); 
  printf("%f\n\n", expected);
  expected = atof(".000031415"); 
  printf("%f\n\n", expected);
  
  expected = atof("-3.1415e-9"); 
  printf("%20.15e\n\n", expected);
  
  expected = atof("+3.1415e+009"); 
  printf("%20.15e\n\n", expected);
#endif
  
  expected = atof("+3.123456789123456789"); 
  printf("%30.25e\n\n", expected);
  
  expected = atof(".000003123456789123456789"); 
  printf("%30.25e\n\n", expected);
  
  expected = atof("3.1234567891234567890000000000"); 
  printf("%30.25e\n\n", expected);
  
#if 0
  expected = atof("1.7e+308");
  if(errno != 0)
    {
      printf("%d\n", errno);
    }
  else    printf("1.7e308 OK %g\n", expected);
  expected = atof("1.797693e308");	/* anything gt looses */
  if(errno != 0)
    {
      printf("%d\n", errno);
    }
  else    printf("Max OK %g\n", expected);
  expected = atof("2.225073858507201E-307");
  if(errno != 0)
    {
      printf("%d\n", errno, expected);
    }
  else    printf("Min OK %g\n", expected);
#endif
  
  max_abs_err = 0.0;
  for(errs = 0, i = 0; i < NTEST; i++)
    {
      expected = (double)(s = rand()) / (double)rand();
      if(s > (RAND_MAX >> 1)) expected = -expected;
      sprintf(buf, "%.14e", expected);
      result = atof(buf);
      e = (expected == 0.0) ? result : (result - expected)/expected;
      if(e < 0) e = (-e);
      if(e > 1.0e-6) 
	{
	  errs++; printf("%.14e %s %.14e (%.14e)\n", expected, buf, result, e);
	}
      if (e > max_abs_err) max_abs_err = e;
    }
  printf("%ld Error(s), Max abs err %.14e\n", errs, max_abs_err);
}
#endif /* TEST */
