/******************************************************************/
/* 		Copyright (c) 1989, Intel Corporation

   Intel hereby grants you permission to copy, modify, and 
   distribute this software and its documentation.  Intel grants
   this permission provided that the above copyright notice 
   appears in all copies and that both the copyright notice and
   this permission notice appear in supporting documentation.  In
   addition, Intel grants this permission provided that you
   prominently mark as not part of the original any modifications
   made to this software or documentation, and that the name of 
   Intel Corporation not be used in advertising or publicity 
   pertaining to distribution of the software or the documentation 
   without specific, written prior permission.  

   Intel Corporation does not warrant, guarantee or make any 
   representations regarding the use of, or the results of the use
   of, the software and documentation in terms of correctness, 
   accuracy, reliability, currentness, or otherwise; and you rely
   on the software, documentation and results solely at your own 
   risk.							  */
/******************************************************************/
/********************************************************
 * This comment attempts to explain the structure used 
 * for system calls.  
 *
 *  - Serial I/O calls (read and write to STDIN, STDIO,
 * STDERR) are made through a 960 system call to console().
 *
 *  - Block I/O calls (file manipulation and all other 
 * system calls) are made through a 960 system call to 
 * files().  All the arguments from the calling procedure 
 * are passed directly through, with the name of the 
 * calling procedure inserted as the first argument.
 * 
 *  - Laser I/O calls (read from LPT1, no write error messages
 * are made through a 960 system call to laser().
 *
 * These calls go through the System Procedure Table as 
 * calls 0, calls 1 and calls 2, and are then routed to the 
 * routines on the NINDY side, console_io(), buffer_io(), and
 * lpt_io() respectively.  The arguments passed along
 * from the original calling procedure are held in the 
 * global variables until they reach their final destination,
 * the procedure on the NINDY side which actually does the 
 * work.
 *
 * This indirection was chosen so that only the hardware 
 * dependent pieces of NINDY would have to be changed for
 * I/O to different devices.  This also allows NINDY to 
 * handle any errors which may occur.
********************************************************/
#include "block_io.h"
#include <errno.h>
#define STDIN 	0
#define STDOUT 	1
#define STDERR 	2
#define ERROR 	-1
#define FALSE 	0
#define TRUE 	1
#define CI 	0
#define CO 	1
#define LF	0x0a
#define CR	0x0d
#define DEL	0x7f
#define	BS	0x08
#define	SPACE	0x20

extern char *stack_start;  /* address label from linker */
extern int end;		/* linker reserved word - end of .bss */

struct {		/* IOCTL structure */
	short lpt;	/* read from lpt port */
	short crlf;	/* turn on CR/LF translation */
} iostruct = {0,1};

#define LPTON 	0x53 	/* chosen randomly to not interfere with */
#define LPTOFF	0x33 	/* standard IOCTL definitions */

/* Following is the current break address (the high address of the
 * heap).
 *
 * This variable should NOT be initialized:  that places it in the
 * .data, which does not necessarily get re-initialized before the
 * program is run.  If the program is run out of RAM several times
 * in succession without re-downloading, the break address will 
 * creep steadily up until mallocs fail.
 *
 * However, crtnindy.o guarantees that .bss is always zeroed before
 * the user program is run.  So the brk() and sbrk() routines check
 * this value;  if it's 0, they assume it's the first call and 
 * initialize it to '&end', the end of the .bss segment.
 */

static char *current_brk;

/***************************************/
/* Access                              */
/*                                     */
/* determines accessibility of a file  */
/***************************************/
access(path, mode)
char *path;
int mode;
{
	return (files(BS_ACCESS, path, mode));
}

/***************************************/
/* Close                               */
/*                                     */
/* closes a file descriptor            */
/***************************************/
close(fd)
int fd;
{
	return (files(BS_CLOSE,fd));
}

/***************************************/
/* Creat                               */
/*                                     */
/***************************************/
creat(path, mode)
char *path;
int mode;
{
	return (files(BS_CREAT, path, mode));
}

/***************************************/
/* Lseek                               */
/*                                     */
/* moves the file pointer w/in the file*/
/***************************************/
lseek(fd, offset, how)
int fd;
long offset;
int how;
{
	return (files(BS_LSEEK, fd, offset, how));
}

/***************************************/
/* Open                                */
/*                                     */
/* opens a file for read/write         */
/***************************************/
open(path, flag, mode)
char *path;
unsigned int flag;
int mode;
{
	return (files(BS_OPEN, path, flag, mode));
}

/***************************************/
/* Spawnd                              */
/*                                     */
/***************************************/
spawnd(path, argv)
char *path;
char *argv[];
{
	return(ERROR);
}

/***************************************/
/* Stat                                */
/*                                     */
/* get status on file		       */
/***************************************/
stat (path, buff)
char *path;
struct st *buff;
{
	return (files(BS_STAT, path, buff));
}
/***************************************/
/* System                              */
/*                                     */
/***************************************/
systemd(string)
char *string;
{
	return (files(BS_SYSTEMD, string));
}

/***************************************/
/* Time                                */
/*                                     */
/***************************************/
time(t)
long *t;
{
long tmp;

	tmp = files(BS_TIME);
	if (t != 0)
		*t = tmp;
	return (tmp);
}

/***************************************/
/* Unmask                              */
/*                                     */
/***************************************/
unmask(mode)
int mode;
{
	return (files(BS_UNMASK, mode));
}

/***************************************/
/* Unlink                              */
/*                                     */
/***************************************/
unlink(path)
char *path;
{
	return (files(BS_UNLINK, path));
}

/***************************************/
/* Read                                */
/*                                     */
/***************************************/
read(fd, buff, bytes)
int fd;
unsigned char *buff;
unsigned bytes;
{
/* for now, assume that CR/LF is translated to LF.  this
	should really be controlled by ioctl */

unsigned char temp_char;
int i, buffnum;

	if (fd != STDIN) {
		i = 0;
		buffnum = 1;
		if (bytes <= 512)
			i = files(BS_READ, fd, buff, bytes, buffnum);
		else {
			while (bytes > 512) {
				i += files(BS_READ, fd, buff, 512, buffnum);
				buff += 512;
				bytes -= 512;
				if (buffnum == 1)
					buffnum = 2;
				else
					buffnum = 1;
			}
			i += files(BS_READ, fd, buff, bytes, buffnum);
		}
	}
	else
	{
		i = 0; 
		while (i<bytes) {
			if (iostruct.lpt) {
				if (laser(CI, &temp_char)) 
					break; /* no character */
				*buff++ = temp_char;
				i++;
			}
			else  {
			    if (console(CI, &temp_char))
					break; /* no character */

				switch (temp_char) {
				
				case CR:
					*buff++ = LF;
					i++;
					console(CO, CR);
					console(CO, LF);
					return (i);
				
				case DEL:
				case BS:
				/* remove previous char, if there */
					/* ignore otherwise */
					if (i > 0) { 
						i--;
						buff--;
						console(CO, BS);
						console(CO, SPACE);
						console(CO, BS);
					}
					break;
				
				default:
					console(CO,temp_char);
					*buff++ = temp_char;
					i++;
				}
			}
		}
	}
	return(i);
}

/***************************************/
/* Write                               */
/*                                     */
/***************************************/
write(fd, buff, bytes)
int fd;
unsigned char *buff;
unsigned bytes;
{
/*   for now (first revision) this routine is assumed to do
	translation of LF to CR/LF sequence (for ttys).  This
	should be made controllable through ioctl() someday.
*/
	
int i, buffnum;

	if ((fd != STDOUT) && (fd != STDERR)) {
		i = 0;
		buffnum = 1;
		if (bytes <= 512)
			i = files(BS_WRITE, fd, buff, bytes, buffnum);
		else {
			while (bytes > 512) {
				i += files(BS_WRITE, fd, buff, 512, buffnum);
				buff += 512;
				bytes -= 512;
				if (buffnum == 1)
					buffnum = 2;
				else
					buffnum = 1;
			}
			i += files(BS_WRITE, fd, buff, bytes, buffnum);
		}
	}
	else
		for (i=0; i<bytes; i++)  {
			if (! iostruct.lpt) {  
				if (*buff == LF) { 
					if (console (CO, CR))	
						break;
					}
				if (console (CO, *buff++))	
					break;
			}
			else {
				if (laser (CO, *buff++))
					break;
			}
		}
	return(i);
}

/***************************************/
/* ioctl			       */
/*                                     */
/***************************************/
ioctl(fd, request)
long fd;
unsigned long request;
{
	if ((fd != STDIN) && (fd != STDOUT) && (fd != STDERR)) {
		errno = EBADF;
	/*	return (ERROR); 	*/
		return (EBADF);  /* should be the above return, but
				 changed for ic960 compatability */
	}

/*	if ((request != LPTON) || (request != LPTOFF)) {
		errno = EINVAL;
		return (ERROR);  
	} 
should be able to use the above, but left out for ic960 compatability 
*/
	if (request == LPTON)
		iostruct.lpt = TRUE;
	else
		iostruct.lpt = FALSE;
	return(0);
}

/***************************************/
/* brk                                 */
/*                                     */
/***************************************/
int
brk(endds)
char *endds;
{
	if ( (endds < (char *)&end) || (endds >= stack_start) ){
		return ERROR;
	}
	current_brk = endds;
	return 0;
}

/***************************************/
/* Sbrk                                */
/*                                     */
/***************************************/
char *
sbrk(incr)
int incr;
{
	char *new_brk;
	char *junk;
	
	if ( current_brk == 0 ){
		/* First call: initialize break address */
		current_brk = (char *) &end;
	}

	new_brk = current_brk + incr;

	if ( (new_brk < (char *)&end) || (new_brk >= stack_start) ){
		return (char *) ERROR;
	}

	for ( junk = current_brk; junk < new_brk; junk++ ){
		*junk = 0;	/* zero memory */
	}
	current_brk = new_brk;
	return new_brk;	
}
