/*
 *	Reformat assembly output of 960 compilers, inserting C source code
 *	into the assembly language as comments, based on line numbers left
 *
 *	mcg - 1/27/89 based on memories of the same program written
 *		in 1986 based on an original for the Tek 8086 compiler
 *		by Mike Zuhl
 */


#include <stdio.h>
#include <ctype.h>

#ifdef USG
#	include "sysv.h"
#else
#	include <strings.h>
#endif

#define	LINELEN		512
#define	COMMENT_CHAR	'#'
#define	PSEUDO_CHAR	'.'
#define	INITIAL_CONTEXT	2
#define	CONTEXT_SIZE	20
#define	NDIRPATHS	10

#define	atoi(p)	strtol(p,(char **) 0, 0)

char	cur_source_name[100];	/* name of current C source file */
FILE	*cur_source_f = 0;	/* current C source file to merge */
char	*dirpath[NDIRPATHS];
int	ndirs = 0;
int	src_line = 1;
int	last_line = -1;
char	*myname;

int	Aflag = 0;		/* copy everything */
int	gflag = 0;		/* look for '.ln' directives, not comments */
int	ggflag = 0;		/* look for '.gdbline' directives */
int	Cflag = 0;
int	nflag = 0;		/* number source lines */
int	sflag = 0;		/* don't strip gdb directives */

extern char fmt960_ver[];

main(argc, argv)
int argc;
char **argv;
{
	register char *p, *q, *begin;
	char line[LINELEN];
	int first;

	if ((myname = rindex(argv[0],'/')) == 0) {
		myname = argv[0];
	} else {
		myname++;
	}
	--argc;
	++argv;
	while (argc > 0 && **argv == '-') {
		switch (argv[0][1]) {
		case 'A':	/* copy everything */
			Aflag++;
			break;
		case 'n':
			nflag++;
			break;
		case 'g':	/* look for STABS directives, not comments */
			if (argv[0][2] == 'g') {
				ggflag++;
			}
			gflag++;
			break;
		case 'C':	/* look for comments, but not stabs */
			Cflag++;
			break;
		case 'd':	/* directory to look for source in */
			--argc;
			++argv;
			if (argc < 1 || **argv == '-') {
				error("missing directory argument","");
				exit(1);
			}
			if (ndirs < NDIRPATHS) {
				dirpath[ndirs++] = *argv;
			} else {
				error("too many directories to search","");
			}
			break;
		case 'm':
		case 'f':	/* override source file */
			--argc;
			++argv;
			if (argc < 1 || **argv == '-') {
				error("missing C source argument","");
				exit(1);
			}
			strcpy(cur_source_name, *argv);
			if ((cur_source_f = fopen(*argv, "r")) == NULL) {
				error("cannot open source file %s\n", *argv);
				exit(1);
			}
			break;

		case 's':	/* don't strip gdb info */
			sflag++;
			break;

		case 'v':
			if ( !strcmp(*argv,"-v960") ){
				puts( fmt960_ver );
				exit(0);
			}
			/* FALL THROUGH TO DEFAULT */
		default:
			error("unknown flag: -%c\n", argv[0][1]);
			exit(1);
		}
		argc--;
		argv++;
	}
	if (argc > 0 && freopen(*argv, "r", stdin) == NULL) {
		error("cannot open assembly input file %s", *argv);
		exit(1);
	}
	--argc;
	++argv;
	if (argc > 0 && freopen(*argv, "w", stdout) == NULL) {
		error("cannot create output file %s", *argv);
		exit(1);
	}

	while (fgets(line,LINELEN,stdin) != NULL) {
		first = 0;
		for (begin = p = line; *p != '\0'; p++) {
			if (*p == '\0' || *p == '\n') {
				break;
			}
			if (*p == ';') {
				first = 0;
				continue;
			}
			if (isspace(*p)) {
				continue;
			}
			if ((first == 0) && (*p == PSEUDO_CHAR)) {
				if (strncmp(p,".asci",5) == 0) {
					if ((q = index(p,';')) != 0) {
						p = q-1;
						continue;
					}
					break;
				}
				if (!Cflag && strncmp(p,".file",5) == 0) {
					open_file(p+5);
					p += 5;
					continue;
				}
				if (!Cflag && strncmp(p,".ln",3) == 0) {
					expand_dotln(p+3);
					p += 3;
					if (!sflag)
						*begin = '\0';
					break;
				}

				if (!Cflag && strncmp(p,".def",4) == 0)  {
					if (!sflag)
						*begin = '\0';
					break;
				}
				/*	.stabd	68,0,line		*/
				/* 	68 can be octal, hex, or decimal */

				if (!Cflag &&
				    ((strncmp(p,".stabd 68,", 10) == 0) ||
				    (strncmp(p,".stabd\t68,", 10) == 0))) {
					if (q = index(p+10,',')) {
						expand_dotln(q+1);
					}
					if (!sflag)
						*begin = '\0';
					break;
				}
				if (!Cflag &&
				    ((strncmp(p,".stabd", 6) == 0) ||
				     (strncmp(p,".stabn", 6) == 0))) {
					if (!sflag)
						*begin = '\0';
					break;
				}

				/*	.stabs "file.c",100		*/
				/*	.stabs "file.c",132		*/
				/*	specifies filename		*/

				if (!Cflag && strncmp(p,".stabs",6) == 0) {
					if ((q = index(p+6,'"')) &&
					    (p = index(q+1,'"')) &&
					    (p = index(p,',')) &&
					    ((strncmp(p+1,"100",3) == 0) ||
					     (strncmp(p+1,"132",3) == 0))) {
						open_file(q);
					}
					if (!sflag)
						*begin ='\0';
					break;
				}

				/* this gdb stuff is obsolete */

				if (!Cflag && strncmp(p,".gdbline",8) == 0) {
					if (q = index(p+8,',')) {
						expand_dotln(q+1);
						p += 8;
					}
					*begin = '\0';
					break;
				}
				if (!Cflag && strncmp(p,".gdb",4) == 0) {
					/* just delete them */
					*begin = '\0';
					break;
				}
			}
			if (!gflag && *p == COMMENT_CHAR) {
				expand_comment(++p);
				*begin = '\0';
				break;
			}
		}
		if (begin) {
			fputs(begin,stdout);
		}
	}
	exit(0);
}


expand_dotln(p)
register char *p;
{
	register int line;

	while (isspace(*p) && *p != '\0') {
		p++;
	}
	if ((line = atoi(p)) <= 0) {
		return;
	}
	if (last_line == -1) {
		/* this is the first line number seen - don't copy */
		/* all of the header information, etc, unless asked */
		if (Aflag) {
			copysrc(1,line);
		} else {
			copysrc(line - INITIAL_CONTEXT,line);
		}
		last_line = line;
	} else if (line < last_line) {
		/* this is an out-of-order line - just copy one line */
		/* don't change last_line */
		copysrc(line,line);
	} else if (line > last_line + CONTEXT_SIZE) {
		/* probably also out-of-order - just copy one line */
		if (Aflag) {
			copysrc(last_line+1,line);
			last_line = line;
		} else {
			copysrc(line,line);
		}
	} else {
		copysrc(last_line+1,line);
		last_line = line;
	}
}

expand_comment(p)
register char *p;
{
	char c;
	register char *file;
	register int line;

	while (isspace(*p) && *p != '\0') {
		p++;
	}

	file = p;
	while (!isspace(*p) && *p != '\0') {
		p++;
	}
	if (*p == '\0') {
		return;
	}

	while (isspace(*p) && *p != '\0') {
		p++;
	}

	if ((line = atoi(p)) <= 0) {
		return;
	}
	if (last_line == -1) {
		/* this is the first line number seen - don't copy */
		/* all of the header information, etc, unless asked */
		if (Aflag) {
			copysrc(1,line);
		} else {
			copysrc(line - INITIAL_CONTEXT,line);
		}
		last_line = line;
	} else if (line < last_line) {
		/* this is an out-of-order line - just copy one line */
		/* don't change last_line */
		copysrc(line,line);
	} else if (line > last_line + CONTEXT_SIZE) {
		/* probably also out-of-order - just copy one line */
		if (Aflag) {
			copysrc(last_line+1,line);
			last_line = line;
		} else {
			copysrc(line,line);
		}
	} else {
		copysrc(last_line+1,line);
		last_line = line;
	}
}

open_file(file)
register char *file;
{
	FILE *f;
	char filename[1000];
	char buf[1000];
	int i;
	register char *p = filename;

	while(isspace(*file) && *file != '\0') {
		file++;
	}
	if (*file == '"') {
		file++;
	}
	while(*file != '\0' && *file != '"' && !isspace(*file)) {
		*p++ = *file++;
	}
	*p = '\0';

	if (strcmp(filename,cur_source_name) == 0) {
		return 1;
	}

	i = ndirs;
	strcpy(buf,filename);
	do {
		if ((f = fopen(buf,"r")) != NULL) {
			if (cur_source_f)
				fclose(cur_source_f);
			cur_source_f = f;
			strcpy(cur_source_name, filename);
			src_line = 1;
			last_line = -1;
			return 1;
		} else {
			if (*filename == '/')
				break;
			strcpy(buf,dirpath[i]);
			strcat(buf,"/");
			strcat(buf,filename);
		}
	} while (i--);

	error("cannot open source file %s",filename);
	if (cur_source_f)
		fclose(cur_source_f);
	cur_source_f = 0;
	return 0;
}

copysrc(begin,end)
register int begin;
register int end;
{
	char buf[LINELEN];
	register char *p;
	int blank = 0;

	if (!cur_source_f || feof(cur_source_f)) {
		return(0);
	}
	if (begin <= 0) {
		begin = 1;
	}
	if (begin < src_line) {
		rewind(cur_source_f);
		src_line = 1;
	}
	while (src_line < begin) {
		if (fgets(buf,LINELEN,cur_source_f) == NULL) {
			return(0);
		}
		src_line++;
	}
	printf("\n");
	if (nflag) printf("\t%c (%d)\n", COMMENT_CHAR,src_line);
	while (src_line <= end) {
		if (fgets(buf,LINELEN,cur_source_f) == NULL) {
			return(0);
		}

		/* skip second and successive blank lines */
		for (p = buf; p && isspace(*p) && (*p != '\n'); p++)
			;
		if ((buf[0] == '\n') || (*p == '\n')) {
			if (blank)
				continue;
			blank++;
		} else {
			blank = 0;
		}
		printf("\t%c ",COMMENT_CHAR);
		if (nflag) {
			printf("%4d: ",src_line);
		}
		fputs(buf,stdout);
		src_line++;
	}
	return(1);
}

error(s1,s2) {
	fprintf(stderr,"%s: ", myname);
	fprintf(stderr,s1,s2);
	fprintf(stderr,"\n");
}
