/* Basic 80960 instruction formats.
 *
 * The 'COJ' instructions are actually COBR instructions with the 'b' in
 * the mnemonic replaced by a 'j';  they are ALWAYS "de-optimized" if necessary:
 * if the displacement will not fit in 13 bits, the assembler will replace them
 * with the corresponding compare and branch instructions.
 *
 * All of the 'MEMn' instructions are the same format; the 'n' in the name
 * indicates the default index scale factor (the size of the datum operated on).
 *
 * The FBRA formats are not actually an instruction format.  They are the
 * "convenience directives" for branching on floating-point comparisons,
 * each of which generates 2 instructions (a 'bno' and one other branch).
 *
 * The CALLJ format is not actually an instruction format.  It indicates that
 * the instruction generated (a CTRL-format 'call') should have its relocation
 * specially flagged for link-time replacement with a 'bal' or 'calls' if
 * appropriate.
 */ 
#define CTRL	0
#define COBR	1
#define COJ	2
#define REG	3
#define MEM1	4
#define MEM2	5
#define MEM4	6
#define MEM8	7
#define MEM12	8
#define MEM16	9
#define FBRA	10
#define CALLJ	11

/* Masks for the mode bits in REG format instructions */
#define M1		0x0800
#define M2		0x1000
#define M3		0x2000

/* Generate the 12-bit opcode for a REG format instruction by placing the 
 * high 8 bits in instruction bits 24-31, the low 4 bits in instruction bits
 * 7-10.
 */

#define REG_OPC(opc)	((opc & 0xff0) << 20) | ((opc & 0xf) << 7)

/* Generate a template for a REG format instruction:  place the opcode bits
 * in the appropriate fields and OR in mode bits for the operands that will not
 * be used.  I.e.,
 *		set m1=1, if src1 will not be used
 *		set m2=1, if src2 will not be used
 *		set m3=1, if dst  will not be used
 *
 * Setting the "unused" mode bits to 1 speeds up instruction execution(!).
 * The information is also useful to us because some 1-operand REG instructions
 * use the src1 field, others the dst field; and some 2-operand REG instructions
 * use src1/src2, others src1/dst.  The set mode bits enable us to distinguish.
 */
#define R_0(opc)	( REG_OPC(opc) | M1 | M2 | M3 )	/* No operands      */
#define R_1(opc)	( REG_OPC(opc) | M2 | M3 )	/* 1 operand: src1  */
#define R_1D(opc)	( REG_OPC(opc) | M1 | M2 )	/* 1 operand: dst   */
#define R_2(opc)	( REG_OPC(opc) | M3 )		/* 2 ops: src1/src2 */
#define R_2D(opc)	( REG_OPC(opc) | M2 )		/* 2 ops: src1/dst  */
#define R_3(opc)	( REG_OPC(opc) )		/* 3 operands       */

/* DESCRIPTOR BYTES FOR REGISTER OPERANDS
 *
 * Interpret names as follows:
 *	R:   global or local register only
 *	RS:  global, local, or (if target allows) special-function register only
 *	RL:  global or local register, or integer literal
 *	RSL: global, local, or (if target allows) special-function register;
 *		or integer literal
 *	F:   global, local, or floating-point register
 *	FL:  global, local, or floating-point register; or literal (including
 *		floating point)
 *
 * A number appended to a name indicates that registers must be aligned,
 * as follows:
 *	2: register number must be multiple of 2
 *	4: register number must be multiple of 4
 */

#define SFR	0x10		/* Mask for the "sfr-OK" bit */
#define LIT	0x08		/* Mask for the "literal-OK" bit */
#define FP	0x04		/* Mask for "floating-point-OK" bit */

/* This macro ors the bits together.  Note that 'align' is a mask
 * for the low 0, 1, or 2 bits of the register number, as appropriate.
 */
#define OP(align,lit,fp,sfr)	( align | lit | fp | sfr )

#define R	OP( 0, 0,   0,  0   )
#define RS	OP( 0, 0,   0,  SFR )
#define RL	OP( 0, LIT, 0,  0   )
#define RSL	OP( 0, LIT, 0,  SFR )
#define F	OP( 0, 0,   FP, 0   )
#define FL	OP( 0, LIT, FP, 0   )
#define R2	OP( 1, 0,   0,  0   )
#define RL2	OP( 1, LIT, 0,  0   )
#define F2	OP( 1, 0,   FP, 0   )
#define FL2	OP( 1, LIT, FP, 0   )
#define R4	OP( 3, 0,   0,  0   )
#define RL4	OP( 3, LIT, 0,  0   )
#define F4	OP( 3, 0,   FP, 0   )
#define FL4	OP( 3, LIT, FP, 0   )

#define M	0x7f	/* Memory operand (MEMA & MEMB format instructions) */

/* Macros to extract info from the register operand descriptor byte 'od'.
 */
#define SFR_OK(od)	(od & SFR)	/* TRUE if sfr operand allowed */
#define LIT_OK(od)	(od & LIT)	/* TRUE if literal operand allowed */
#define FP_OK(od)	(od & FP)	/* TRUE if floating-point op allowed */
#define REG_ALIGN(od,n)	((od & 0x3 & n) == 0)
					/* TRUE if reg #n is properly aligned */
#define MEMOP(od)	(od == M)	/* TRUE if operand is a memory operand*/

/* Description of a single i80960 instruction */
struct i960_opcode {
	long opcode;	/* 32 bits, constant fields filled in, rest zeroed */
	char *name;	/* Assembler mnemonic				   */
	short iclass;	/* Class: see #defines below			   */
	char format;	/* REG, COBR, CTRL, MEMn, COJ, FBRA, or CALLJ	   */
	char num_ops;	/* Number of operands				   */
	char operand[3];/* Operand descriptors; same order as assembler instr */
};

/* Classes of 960 intructions:
 *	- each instruction falls into one class.
 *	- each target architecture supports one or more classes.
 *
 * EACH CONSTANT MUST CONTAIN 1 AND ONLY 1 SET BIT!:  see targ_has_iclass().
 */
#define I_BASE	0x01	/* 80960 base instruction set	*/
#define I_CX	0x02	/* 80960Cx instruction		*/
#define I_DEC	0x04	/* Decimal instruction		*/
#define I_FP	0x08	/* Floating point instruction	*/
#define I_KX	0x10	/* 80960Kx instruction		*/
#define I_MIL	0x20	/* Military instruction		*/

extern struct i960_opcode i960_opcodes[];
