/*****************************************************************************
 *
 * $Header: /m2/j/tools/g960/src/gld960/common/RCS/i960.c,v 1.1 89/10/23 18:36:51 chrisb Exp $
 *
 * This file contains i960-specific routines for the GNU linker (gld960).
 *
 *****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef USG
#	include "sysv.h"
#else
#	include <strings.h>
#endif

extern char *getenv();

/* Things in ld.c
 */
extern char *concat();
extern char **search_dirs;
extern int n_search_dirs;


/* Which i960 architecture we're linking for -- used when there's more than
 * one (architecture-dependent) library to choose from.
 */
#define ARCH_KA	0
#define ARCH_KB	1
#define ARCH_MC 2
#define ARCH_CA 3

char i960_arch = ARCH_KB;	/* Default is KB -- changeable by user via
				 * invocation line switch.
				 */

/* The following table must be in the same order as the #defines for ARCH_xx.
 * The value of an entry is the string that should come after the basename
 * but before the suffix when trying to convert a library name to an
 * architecture-specific name.
 *
 *	e.g., libc.a -> libcka.a for the KA.
 */
static
char *lib_infix[] = {
	"ka.",	/* ARCH_KA */
	"kb.",	/* ARCH_KB */
	"kb.",	/* ARCH_MC -- uses KB libraries */
	"ca.",	/* ARCH_CA */
};


/******************************************************************************
 * i960_lib_path:
 *	Create a library directory path by concatenating two name components
 *	onto base name, optionally looking up the base name as a runtime
 *	environment variable.  Check to see if the resulting directory
 *	actually exists.  If so, add it to the end of the list of directories
 *	to be searched for libraries.
 *
 *****************************************************************************/
i960_lib_path( base, c1, c2, env )
    char *base;		/* Base directory of path */
    char *c1, *c2;	/* Components (subdirectories and/or file name) to be
			 *	appended onto the base directory name.
			 */
    int env;		/* If 1, '*base' is the name of an environment variable
			 *	to be examined for the base directory name;
			 *	otherwise, '*base' is the actual base directory
			 *	name.
			 */
{
	struct stat buf;	/* For call to 'stat' -- never examined */
	char *path;		/* Pointer to full pathname (malloc'd memory) */

	if ( env ){
		base = getenv( base );
		if ( base == NULL ){
			return;
		}
	}
	path = concat( base, c1, c2 );

	if ( stat(path,&buf) != 0 ){
		free( path );
	} else {
		n_search_dirs++;
		search_dirs = (char **)
			xrealloc( search_dirs, n_search_dirs * sizeof(char*) );
		search_dirs[n_search_dirs - 1] = path;
	}
}

/******************************************************************************
 * i960_set_arch:
 *	Set i80960 arcitecture flag, according to type specified on invocation
 *	line.
 *	
 *****************************************************************************/
i960_set_arch( swt )
    char *swt;		/* Invocation line switch (portion followin "-a") */
{
	if ( !strcmp(swt,"KA") ){
		i960_arch = ARCH_KA;

	} else if ( !strcmp(swt,"KB") ){
		i960_arch = ARCH_KB;

	} else if ( !strcmp(swt,"MC") ){
		i960_arch = ARCH_MC;

	} else if ( !strcmp(swt,"CA") ){
		i960_arch = ARCH_CA;

	} else {
		error( "Unknown architecture: %s", swt );
	}
}


/*****************************************************************************
 * i960_lib_open:
 *	Here's the scoop:  sometimes i960 libraries have to come in
 *	multiple flavors to accommodate the different i960 architectures.
 *	But sometimes not.  This routine tries to open a library under the
 *	name passed to it, but if that fails it sticks an architecture-
 *	identifying string before the ".a" and tries again.  It returns
 *	the file descriptor (<0, if no library is found), and sets *fullpath
 *	to the last full pathname tried.
 *
 * WARNING:
 *	*fullpath MUST be left pointing to malloc'd memory, even if no library
 *	is opened, because the caller tries to free the memory when it's done.
 *
 *****************************************************************************/
int
i960_lib_open( dirname, libname, fullpath )
    char *dirname;	/* Directory in which to look	*/
    char *libname;	/* Name of library		*/
    char **fullpath;	/* Leave pointer to last full path tried here */
{
	int desc;	/* File descriptor from 'open'		*/
	char *tmpname;	/* libname, massaged to an architecture-specific name*/
	char *p;	/* Pointer to the "." in libname	*/

	*fullpath = concat( dirname, "/", libname );
	desc = open( *fullpath, O_RDONLY, 0 );
	if ( desc < 0 ){
		p = rindex( libname, '.' );
		if ( p != NULL ){
			*p = '\0';
			tmpname = concat( libname, lib_infix[i960_arch], p+1);
			*p = '.';
			*fullpath = concat( dirname, "/", tmpname );
			free( tmpname );
			desc = open( *fullpath, O_RDONLY, 0 );
		}
	}
	return desc;
}


/*****************************************************************************
 * i960_align:
 *	Align an address by rounding it up to the specified boundary.
 *****************************************************************************/
long
i960_align( addr, alignment )
    long addr;		/* Address to be rounded up	*/
    int alignment;	/* Alignment as a power of 2 (e.g., 1 = 2-byte boundary,
			 *	2 = 4-byte boundary, etc.
			 */
{
	return (addr + (1<<alignment) - 1) & (-1 << alignment );
}
