char rcsid[] = "$Id: gswab960.c,v 1.7 89/11/07 12:27:06 chrisb Exp $";

/* Convert b.out format object files and/or libraries from little-endian byte
 * ordering to big-endian ordering, or vice versa.
 *
 * See 'usage()' function for invocation; or invoke without arguments for help.
 *
 * Note that actual i960 code is always preserved in little-endian order,
 * but the file header, symbol information, relocation directives, and string
 * table length must be in host order so the cross-tools will work.
 * See comments in "b.out.h".
 *
 * WARNING:
 *	This code is HIGHLY dependent on format of b.out file, with especially
 *	subtle dependencies on relocation directives: see comments for
 *	cvt_reloc().
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "b.out.h"

/* Under USG Unix, more than one archive format is supported, via
 * conditional compilation.  The following one is compatible with BSD
 * (and the code here).
 */
#define PORTAR	1
#include <ar.h>

#ifdef USG
#	include	<fcntl.h>		/* needed for O_RDWR */
#	include	<unistd.h>		/* needed for SEEK_SET */
#	define	L_SET			SEEK_SET
#	define	bcopy(src,dst,len)	memcpy(dst,src,len)
#	define	bcmp(b1,b2,len)		memcmp(b1,b2,len)
#endif

extern char *malloc();
extern char *mktemp();
char *xmalloc();

extern char gswab960_ver[];

#define FILE_OBJ	0
#define FILE_LIB	1
#define FILE_ERR	2


char	*progname;	/* Name by which we were invoked	*/
int	exit_code;	/* 0 if no errors occurred, else 1	*/
char	quiet = 0;	/* TRUE if "-q" switch specified	*/

/* BYTE ORDERS
 *	Legal values of these variables are BIGEND and LITEND.
 *	HUH is used to record if we ever saw an invocation switch for desired
 *		order.
 */
#define LITEND	0
#define BIGEND	1
#define HUH	2
char host_order;	/* Byte order of host			*/
char in_order;		/* Byte order of current input file	*/
char out_order = HUH;	/* Desired byte order of output file	*/
char *out_string;	/* "big" or "little"			*/


/* FILE CURRENTLY BEING PROCESSED
 */
char *curfn;		/* Name of file					*/
char *curfile;		/* Pointer to in-core file image		*/
struct exec curhdr;	/* File header, in host byte-order; for finding
			 *	file components in the in-core image
			 */


usage()
{
	printf( "Usage:\n" );
	printf( "\t%s [-b] [-h] [-l] [-q] files...\n", progname );
	printf( "where\n" );
	printf( "\t   -b: convert to big-endian byte order\n" );
	printf( "\t   -h: convert to byte order of host on which we're running\n" );
	printf( "\t   -l: convert to little-endian byte order\n" );
	printf( "\t   -q: quiet\n" );
	printf( "\tfiles: b.out format object or library files\n" );
	printf( "\n" );
	printf( "\tOnly one of -b/-h/-l can be specified; default is -h\n" );

	exit(1);
}


main(argc,argv)
    int argc;
    char **argv;
{
	int fd;		/* Current input file descriptor */
	int type;
	int i;
	char *order;
	
	progname = argv[0];
	check_host();
	curfile = 0;
	fd = -1;
	exit_code = 0;

	if ( argc == 1 ){
		usage();	/* No return */
	}

	/* First process switches.
	 */
	for ( i = 1; i < argc; i++ ){

		if ( argv[i][0] == '-' ){
			switch ( argv[i][1] ){
			case 'l':
				if ( out_order != HUH ){
					usage();	/* No return */
				}
				out_order = LITEND;
				break;
			case 'b':
				if ( out_order != HUH ){
					usage();	/* No return */
				}
				out_order = BIGEND;
				break;
			case 'h':
				if ( out_order != HUH ){
					usage();	/* No return */
				}
				out_order = host_order;
				break;
			case 'q':
				quiet = 1;
				break;
			case 'v':
				if ( !strcmp(argv[i],"-v960") ){
					puts( gswab960_ver );
					exit(0);
				}
				/* FALL THROUGH TO DEFAULT */
			default:
				printf( "Unknown switch: %s\n", argv[i] );
				usage();	/* No return */
				break;
			}
		}
	}

	if ( out_order == HUH ){
		out_order = host_order;
	}
	out_string = (out_order == BIGEND) ? "big" : "little";


	for ( argv++, argc--; argc; argv++, argc-- ){

		if ( *argv[0] == '-' ){
			continue;
		}

		curfn = *argv;

		/* CLEAN UP AFTER PREVIOUS FILE.
		 *	Doing it here makes error-handling simpler.
		 */
		if ( fd >= 0 ){
			close(fd);
			fd = -1;
		}
		if ( curfile ){
			free( curfile );
			curfile = NULL;
		}


		fd = open( curfn, O_RDWR, 0 );
		if ( fd < 0 ){
			perr( "Can't open" );
			continue;
		}

		if ( (type = file_type(fd)) == FILE_ERR ){
			/* Error message already issued by file_type() */
			continue;
		}

		if ( in_order == out_order ){
			if ( !quiet ){
				printf( "'%s' already in %s-endian order\n",
							curfn, out_string );
			}
			continue;
		}

		if ( !quiet ){
			printf( "Converting '%s' to %s-endian order\n",
							curfn, out_string );
		}

		lseek( fd, 0, L_SET );		/* Rewind */

		if ( type == FILE_OBJ ){
			do_obj_file( fd );
		} else { 	/*  FILE_LIB */
			do_lib( fd );
		}
	}
	exit( exit_code );
}


/* Determine type of file whose descriptor is fd.
 * If it's an object or a library, set the global "in_order" to byte order
 *	of input file.
 */
int		/* Return FILE_OBJ, FILE_LIB, or FILE_ERR */
file_type( fd )
    int fd;
{
	struct exec ehdr;
	char armagic[SARMAG];
	struct ar_hdr arhdr;
	int member_size;

	if ( read(fd,&ehdr,sizeof(ehdr)) != sizeof(ehdr) ){
		perr( "Can't read" );
		return FILE_ERR;
	}

	if ( is_obj(ehdr.a_magic) ){
		return FILE_OBJ;
	}

	/* It's either a library or garbage
	 */
	lseek( fd, 0, L_SET );	/* Rewind */
	read( fd, armagic, SARMAG );
	if ( bcmp(armagic,ARMAG,SARMAG) ){
		err( "invalid object format" );
		return FILE_ERR;
	}

	/* Find the first object member of the library so we can check
	 * it's byte ordering.  Skip the __.SYMDEF member, if there is one,
	 * because there's no magic number there to cue us.
	 */
	if ( (member_size = read_member(fd,&arhdr)) <= 0 ){
		if ( member_size == 0 ){
			err( "empty archive" );
		}
		return FILE_ERR;
	}
	if ( !strncmp(arhdr.ar_name,"__.SYMDEF",9) ){
		if ( (member_size = read_member(fd,&arhdr)) <= 0 ){
			if ( member_size == 0 ){
				err( "archive contains __.SYMDEF only" );
			}
			return FILE_ERR;
		}
	}
	if ( !is_obj( ((struct exec*)curfile)->a_magic ) ){
		perr( "Garbage first member" );
		return FILE_ERR;
	}
	return FILE_LIB;
}


/* is_obj:	check for valid object file
 *
 *	Return true if magic number is valid (in either byte ordering),
 *	false otherwise.  Set global 'in_order' according to byte ordering
 *	of magic number, if it's valid.
 */
int
is_obj( magic )
    unsigned long magic;	/* Magic number extracted from object file */
{
	in_order = host_order;

	/* Check magic number in both host byte-order and swapped byte-order.
	 */
	if ( N_BADMAG( *((struct exec*)&magic) ) ){
		in_order = (host_order == BIGEND) ? LITEND : BIGEND;
		byteswap( &magic, 4 );
		if ( N_BADMAG( *((struct exec*)&magic) ) ){
			return 0;
		}
	}
	return 1;
}


/* do_obj_file:		Convert an object (as opposed to archive) file
 *
 *	Assumes header has been checked, input pointer re-wound
 *	to beginning of file, input file is known to be writable.
 */
do_obj_file( fd )
    int fd;	/* Descriptor of file to be converted */
{
	struct stat buf;

	/* Allocate storage for file image and read it in
	 */
	if ( fstat(fd,&buf) == -1 ){
		perr( "Can't stat" );
		return;
	}
	curfile = xmalloc( buf.st_size );

	if ( read(fd,curfile,buf.st_size) != buf.st_size ){
		perr( "Can't read" );
		return;
	}

	/* Convert in-core image, rewind input file, write out converted image
	 */
	cvt_obj();
	lseek( fd, 0, L_SET );	/* Rewind */
	write( fd, curfile, buf.st_size );
}


/* do_lib:	Convert a library (archive)
 *
 *	Assumes archive header has been checked, input pointer re-wound
 *	to beginning of file, input file is known to be writable.
 */
do_lib( fd )
    int fd;	/* I/O descriptor for library file */
{
	/* Temporary output file -- do conversion there in case of errors,
	 * copy over input file when successfully done.
	 */
	static char template[] = "/tmp/gswabXXXXXX";
	char *tmpname;
	int tmpfd;

	/* Current archive member (object file in library)
	 */
	struct ar_hdr mem_hdr;	/* Member header			*/
	char *mem_name;		/* Member name				*/
	int mem_size;		/* Member size, in bytes, not including header*/
	int len;		/* Length of mem_name			*/

	char *libname;		/* Save name of input (lib) file here	*/
	char armag[SARMAG];	/* Read archive header here		*/
	char *p;		/* Pointer to command to send to system() */
	int ok;			/* TRUE iff conversion completes w/o errors */


	ok = 0;			/* Expect the worst */
	mem_name = "";
	libname = xmalloc( strlen(curfn) + 1 );
	strcpy( libname, curfn );

	/* Open temp output file
	 */
	tmpname = mktemp( template );
	tmpfd = open( tmpname, O_WRONLY|O_CREAT|O_TRUNC, 0644 );
	if ( tmpfd < 0 ){
		perr( "Can't open temp file" );
		return;
	}

	/* Archive header was already confirmed, or we wouldn't be here.
	 * Just copy it out.
	 */
	read( fd, armag, SARMAG );
	write( tmpfd, armag, SARMAG );

	while ( 1 ){
		mem_size = read_member( fd, &mem_hdr );
		if ( mem_size <= 0 ){
			/* 0 => no more members -- we're done;
			 * otherwise we had an i/o error
			 */
			if ( mem_size == 0 ){
				ok = 1;
			}
			break;
		}


		/* Pick up member name,
		 * after freeing previous one if necessary
		 */
		if ( *mem_name ){
			free( mem_name );
		}
		for ( len = 0; mem_hdr.ar_name[len] != ' '; len++ ){
			if ( len == sizeof(mem_hdr.ar_name) ){
				break;
			}
		}
		mem_name = xmalloc( strlen(libname) + len + 10 );
		sprintf( mem_name, "(%s)", libname );
		strncat( mem_name, mem_hdr.ar_name, len );

		/* Process member
		 */
		if ( !strncmp(mem_hdr.ar_name,"__.SYMDEF",9) ){
			if ( !cvt_symdef() ){
				break;
			}
		} else {
			unsigned long magic = ((struct exec*)curfile)->a_magic;

			if ( host_order != in_order ){
				byteswap( &magic, 4 );
			}
			if ( N_BADMAG( *((struct exec*)&magic) ) ){
				err( "Garbage b.out header in library" );
				break;
			}

			cvt_obj();
		}
		write( tmpfd, &mem_hdr, sizeof(mem_hdr));/* Write header */
		write( tmpfd, curfile, mem_size );	/* Write member */
	}

	/* Clean up
	 */
	close( tmpfd );
	if ( ok ){
		/* Copy temp file to lib file
		 */
		p = xmalloc( strlen(tmpname) + strlen(libname) + 20 );
		sprintf( p, "cp %s %s", tmpname, libname );
		if ( system(p) ){
			err( "Temp file copy failed: library not modified" );
		}
		free( p );
	} else {
		err( "Library not modified" );
	}
	unlink( tmpname );
	free( libname );
	if ( *mem_name ){
		free( mem_name );
	}
}


/* read_member:  read library member
 *
 *	Read member header into *hp.
 *	Allocate memory for member body, read it, point global 'curfile' at it.
 *	Return size of member body, 0 on end of file, -1 on read error.
 */
read_member( fd, hp )
    int fd;		/* File descriptor	*/
    struct ar_hdr *hp;	/* Member header	*/
{
	int mem_size;		/* Size of member (exclusive of header)	*/
	long n;			/* Number of bytes returned by read	*/

	n = read( fd, hp, sizeof(struct ar_hdr) );
	if ( n != sizeof(struct ar_hdr) ){
		if ( n != 0 ){
			err( "read error" );
			return -1;
		} else {
			return 0;
		}
	}

	/* Verify member header
	 */
	if ( strncmp(hp->ar_fmag,ARFMAG,sizeof(hp->ar_fmag)) ){
		err( "Garbage member header in library" );
		return -1;
	}

	/* Pick up member size;  round it up to a multiple of 2
	 * (members are padded to even size in library).
	 */
	sscanf( hp->ar_size, "%d", &mem_size );
	mem_size++;
	mem_size &= ~1;

	/* Allocate memory for core image;  read member in
	 */
	if ( curfile ){
		free( curfile );
		curfile = NULL;
	}
	curfile = xmalloc( mem_size );

	if ( read(fd,curfile,mem_size) != mem_size ){
		err( "read error" );
		return -1;
	}
	return mem_size;
}


/* cvt_symdef:	Convert in-core image of __.SYMDEF member of library
 *
 *	Format of __.SYMDEF:
 *	o int: number of bytes of symbols
 *	o symbols (0 or more):
 *		int:  offset of string in symdef string table
 *		int:  offset into archive of object module that
 *			contains the symbol
 *	o string table:
 *		int: length of string table, in bytes
 *		0 or more '\0'-terminated strings
 */
cvt_symdef()
{
	int *p;		/* -> Next in integer in __SYMDEF to byte-swap	*/
	int symbytes;	/* Total number of bytes of symbols in __.SYMDEF */

	/* Find out how big symbol table is
	 */
	p = (int*) curfile;
	symbytes = *p;
	if ( in_order != host_order ){
		byteswap( &symbytes, 4 );
	}
	if ( symbytes % 8 ){
		err( "Number of symbol bytes in __.SYMDEF not a multiple of 8");
		return 0;
	}

	/* Do the byte-swapping
	 */
	byteswap( p++, 4 );		/* # bytes of symbols	*/
	for ( ; symbytes; symbytes -= 8 ){
		byteswap( p++, 4 );	/* String offset	*/
		byteswap( p++, 4 );	/* Member offset	*/
	}
	byteswap( p, 4 );		/* String table size	*/
	return 1;
}


/* cvt_obj:	Convert in-core image of object file
 */
cvt_obj()
{
	/* Make a copy of the header in host order, for information purposes */

	if ( in_order == host_order ){
		bcopy( curfile, &curhdr, sizeof(struct exec) );
	}

	cvt_hdr();	/* Convert header	*/

	if ( in_order != host_order ){
		bcopy( curfile, &curhdr, sizeof(struct exec) );
	}

	cvt_reloc();	/* Convert relocation directives	*/
	cvt_syms();	/* Convert symbol information		*/

	/* Convert string table size */
	byteswap( curfile + N_STROFF(curhdr), 4 );
}


/* cvt_hdr: Convert file header
 */
cvt_hdr()
{
	struct exec *ep;	/* Pointer to header in core file image	*/

	ep = (struct exec *)curfile;

	byteswap( &ep->a_magic, 4 );
	byteswap( &ep->a_text, 4 );
	byteswap( &ep->a_data, 4 );
	byteswap( &ep->a_bss, 4 );
	byteswap( &ep->a_syms, 4 );
	byteswap( &ep->a_entry, 4 );
	byteswap( &ep->a_trsize, 4 );
	byteswap( &ep->a_drsize, 4 );
	byteswap( &ep->a_tload, 4 );
	byteswap( &ep->a_dload, 4 );
}

/* cvt_reloc:	Convert relocation information
 *
 *	This is the most difficult part of the conversion itself:  a relocation
 *	directive consists of a int address of the item to be relocated (no
 *	problem: we reverse byte order) and (oh, God) a series of bit fields.
 *
 *	There is one 24-bit field (the symbol number) and a 2-bit field, and a
 *	bunch of 1-bitters.  If the the symbol number is 0x123456 and if xx
 *	represents the other fields, we see the following string of bytes in
 *	memory (from low mem address to high):
 *
 *		big-endian host:	12 34 56 xx	(at least, on a Sun 3)
 *		little-endian host:	56 34 12 xx
 *
 *	So we just swap the first three bytes, then deal with the "xx" byte.
 *	The "xx" byte differs from big- to little-endian as follows:
 *		o the bits in the byte come in the opposite order, from high-
 *			to low-end of the byte, EXCEPT THAT
 *		o the bits of the 2-bit field must not be swapped relative to
 *			each other.
 *
 *	We do the "xx" conversion with a table look-up.  The algorithm for
 *	generating a byte in one of the tables follows;  its scheme is to
 *	reverse ALL of the bits in the byte, then find and swap the bits of
 *	the 2-bit field.
 *
 *	unsigned char
 *	bitswap( old )
 *	    unsigned char old;
 *	{
 *		unsigned char new;
 *		int i;
 *		static char swap_5_6[] = { 0x00, 0x40, 0x20, 0x60 };
 *		static char swap_1_2[] = { 0x00, 0x04, 0x02, 0x06 };
 *
 *		for ( new = 0, i = 0; i < 8; i++ ){	>>> reverse bit order<<<
 *			new <<= 1;
 *			new |= old & 1;
 *			old >>= 1;
 *		}
 *		if ( in_order == LITEND && out_order == BIGEND ){
 *							>>> swap bits 5 & 6 <<<
 *			return (new & 0x9f) | swap_5_6[ (new>>5) & 3 ];
 *		} else {				>>> swap bits 1 & 2 <<<
 *			return (new & 0xf9) | swap_1_2[ (new>>1) & 3 ];
 *		}
 *	}
 */

/* reloc_btol[n] is the little-endian version of the big-endian byte 'n' */
unsigned char reloc_btol[] = {
	0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0, 
	0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0, 
	0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8, 
	0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8, 
	0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2, 
	0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2, 
	0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea, 
	0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa, 
	0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4, 
	0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4, 
	0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec, 
	0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc, 
	0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6, 
	0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6, 
	0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee, 
	0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe, 
	0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1, 
	0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1, 
	0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9, 
	0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9, 
	0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3, 
	0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3, 
	0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb, 
	0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb, 
	0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5, 
	0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5, 
	0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed, 
	0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd, 
	0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7, 
	0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7, 
	0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef, 
	0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff, 
};

/* reloc_ltob[n] is the big-endian version of the little-endian byte 'n' */
unsigned char reloc_ltob[] = {
	0x00, 0x80, 0x20, 0xa0, 0x40, 0xc0, 0x60, 0xe0, 
	0x10, 0x90, 0x30, 0xb0, 0x50, 0xd0, 0x70, 0xf0, 
	0x08, 0x88, 0x28, 0xa8, 0x48, 0xc8, 0x68, 0xe8, 
	0x18, 0x98, 0x38, 0xb8, 0x58, 0xd8, 0x78, 0xf8, 
	0x04, 0x84, 0x24, 0xa4, 0x44, 0xc4, 0x64, 0xe4, 
	0x14, 0x94, 0x34, 0xb4, 0x54, 0xd4, 0x74, 0xf4, 
	0x0c, 0x8c, 0x2c, 0xac, 0x4c, 0xcc, 0x6c, 0xec, 
	0x1c, 0x9c, 0x3c, 0xbc, 0x5c, 0xdc, 0x7c, 0xfc, 
	0x02, 0x82, 0x22, 0xa2, 0x42, 0xc2, 0x62, 0xe2, 
	0x12, 0x92, 0x32, 0xb2, 0x52, 0xd2, 0x72, 0xf2, 
	0x0a, 0x8a, 0x2a, 0xaa, 0x4a, 0xca, 0x6a, 0xea, 
	0x1a, 0x9a, 0x3a, 0xba, 0x5a, 0xda, 0x7a, 0xfa, 
	0x06, 0x86, 0x26, 0xa6, 0x46, 0xc6, 0x66, 0xe6, 
	0x16, 0x96, 0x36, 0xb6, 0x56, 0xd6, 0x76, 0xf6, 
	0x0e, 0x8e, 0x2e, 0xae, 0x4e, 0xce, 0x6e, 0xee, 
	0x1e, 0x9e, 0x3e, 0xbe, 0x5e, 0xde, 0x7e, 0xfe, 
	0x01, 0x81, 0x21, 0xa1, 0x41, 0xc1, 0x61, 0xe1, 
	0x11, 0x91, 0x31, 0xb1, 0x51, 0xd1, 0x71, 0xf1, 
	0x09, 0x89, 0x29, 0xa9, 0x49, 0xc9, 0x69, 0xe9, 
	0x19, 0x99, 0x39, 0xb9, 0x59, 0xd9, 0x79, 0xf9, 
	0x05, 0x85, 0x25, 0xa5, 0x45, 0xc5, 0x65, 0xe5, 
	0x15, 0x95, 0x35, 0xb5, 0x55, 0xd5, 0x75, 0xf5, 
	0x0d, 0x8d, 0x2d, 0xad, 0x4d, 0xcd, 0x6d, 0xed, 
	0x1d, 0x9d, 0x3d, 0xbd, 0x5d, 0xdd, 0x7d, 0xfd, 
	0x03, 0x83, 0x23, 0xa3, 0x43, 0xc3, 0x63, 0xe3, 
	0x13, 0x93, 0x33, 0xb3, 0x53, 0xd3, 0x73, 0xf3, 
	0x0b, 0x8b, 0x2b, 0xab, 0x4b, 0xcb, 0x6b, 0xeb, 
	0x1b, 0x9b, 0x3b, 0xbb, 0x5b, 0xdb, 0x7b, 0xfb, 
	0x07, 0x87, 0x27, 0xa7, 0x47, 0xc7, 0x67, 0xe7, 
	0x17, 0x97, 0x37, 0xb7, 0x57, 0xd7, 0x77, 0xf7, 
	0x0f, 0x8f, 0x2f, 0xaf, 0x4f, 0xcf, 0x6f, 0xef, 
	0x1f, 0x9f, 0x3f, 0xbf, 0x5f, 0xdf, 0x7f, 0xff, 
};

cvt_reloc()
{
	unsigned long nrelocs;
	struct relocation_info *relocp;
	unsigned char *rp;

	nrelocs = (curhdr.a_trsize + curhdr.a_drsize)
					/ sizeof(struct relocation_info );
	relocp = (struct relocation_info *) (curfile + N_TROFF(curhdr));

	for ( ; nrelocs--; relocp++ ){
		byteswap( (char *) &relocp->r_address, 4 );
		byteswap( ((char*)(&relocp->r_address)) + 4, 3 );
		rp = ((unsigned char*)(&relocp->r_address)) + 7;
		*rp = (out_order==BIGEND) ? reloc_ltob[*rp] : reloc_btol[*rp];
	}
}

/* cvt_syms:	Convert symbol information
 */
cvt_syms()
{
	int nsyms;		/* Number of symbols in object file */
	struct nlist *symp;	/* Pointer to next symbol to convert */

	nsyms = curhdr.a_syms / sizeof( struct nlist );
	symp = (struct nlist *) (curfile + N_SYMOFF(curhdr));
	for ( ; nsyms--; symp++ ){
		byteswap( &symp->n_un.n_strx, 4 );
		byteswap( &symp->n_desc, 2 );
		byteswap( &symp->n_value, 4 );
	}
}

/* byteswap:	Reverse the order of a string of bytes.
 */
byteswap( p, n )
    char *p;	/* Pointer to source/destination string	*/
    int n;	/* Number of bytes in string		*/
{
	int i;	/* Index into string			*/
	char c;	/* Temp holding place during 2-byte swap*/

	for ( n--, i = 0; i < n; i++, n-- ){
		c = p[i];
		p[i] = p[n];
		p[n] = c;
	}
}


/* perr:
 *	Print an error, including
 *		o name by which we were invoked
 *		o name of file we were processing
 *		o error-specific string
 *		o last system error message (should tell why error occurred)
 *
 *	Set the exit code to indicate failure.
 */
perr( s )
    char *s;	/* Error message string	*/
{
	extern int errno;
	extern int sys_nerr;
	extern char *sys_errlist[];
	char *errmsg;

	exit_code = 1;
	errmsg = errno < sys_nerr ? sys_errlist[errno] : "";
	fprintf( stderr, "%s: file '%s': %s: %s\n", progname, curfn, s, errmsg);
}

/* err:
 *	Print an error, including
 *		o name by which we were invoked
 *		o name of file we were processing
 *		o error message
 *
 *	Set the exit code to indicate failure.
 */
err( s )
    char *s;	/* Error message string	*/
{
	exit_code = 1;
	fprintf( stderr, "%s: file '%s': %s\n", progname, curfn, s );
}


/* check_host:
 *	Make sure essential data types of host correspond to those of i960.
 *	Determine byte-ordering used by host.
 */
check_host()
{
	int i;
	union { int i; char c; } test;

	static struct typestruct {
		int hostsize;		/* Size of type on host		*/
		int i960size;		/* Size of type on i960		*/
		char *typename;		/* Name of type, for error msg	*/
	} types[] = {
		{ sizeof(short),  2, "short" },
		{ sizeof(int),    4, "int" },
		{ sizeof(long),   4, "long" },
		{ sizeof(char *), 4, "pointer" },
	};
#	define TYPELEN	(sizeof(types) / sizeof(struct typestruct))

	/* Make sure that host type sizes are same as i960
	 */
	for ( i = 0; i < TYPELEN; i++ ){
		if ( types[i].hostsize != types[i].i960size ){
			fprintf(stderr,"%s: sizeof(%s) != %d: can't continue\n",
				progname, types[i].typename, types[i].i960size);
			exit(99);
		}
	}

	/* Check for big-endian host, set global flag if it is one
	 */
	test.i = 1;
	host_order = (test.c != 1) ? BIGEND : LITEND;
}


char *
xmalloc( n )
    int n;
{
	char *p;

	p = malloc( n );
	if ( !p ){
		perr( "Malloc failed" );
		exit(1);
	}
	return p;
}
