/****************************************************************************
 *
 * Class:  RegistrarClient implementation
 * Author: Mark Roseman
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 02/17/92 MR        initial version
 * 06/08/92 MR        many, many changes to do with string handling.
 *                    also added routine (createReaderAndWriter(fd) to
 *                    provide a remote coordinator with the address of
 *                    a particular conference (via the local coordinator)
 * 08/14/92 MR        clean up
 * 09/20/92 MR        updated to use host/port to find registrar
 * 09/20/92 MR        fixed up a bug with keeping the user and conf lists
 *                      up to date
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/reader.h>
#include <gk/writer.h>
#include <gk-reg/regclient.h>
#include <OS/host.h>
#include <pwd.h>
#include <libc.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <Dispatch/rpcstream.h>
#include <Dispatch/dispatcher.h>
#include <gk/straction.h>
#include <gk/groupkit.h>
#include <gk/infoconn.h>
#include <gk/connaction.h>
#include <gk-reg/coordinator.h>
#include <gk/groupsession.h>

declareConnActionCallback(Coordinator)

declareStrActionCallback(RegistrarClient);
implementStrActionCallback(RegistrarClient)

declareList(IntLst,int);
implementList(IntLst,int);

declarePtrList(AttrListList, AttributeList);
implementPtrList(AttrListList, AttributeList);



/****************************************************************************
 *
 * Constructor.  Try to hook up to the registrar.
 *
 ****************************************************************************/

RegistrarClient::RegistrarClient(const char* host, int port, 
				 Coordinator* coord) :
     RpcPeer("/dev/null"), writer_(nil), reader_(nil), coord_(coord) 
{
  if(createReaderAndWriter(host,port)) {
    startListening();
    conference_tbl_ = new AttrListTable(30);
    users_tbl_ = new UserListTbl(30);
    PollConferences();
  } else {
    fprintf(stderr, "Registrar client exiting\n");
    exit(-1);
  }
}


/****************************************************************************
 *
 * We've received an up to date conference list.  Update
 * our data structure.  First, mark everything as "invalid".
 * Then parse the conference list description string we got
 * from the registrar, extracting one conference at a time.
 * If we can find the conference with the same ID in our
 * structure already, just mark it as "valid".  If we can't
 * find it, its a new conference, so insert it and call
 * "foundNewConference".  When done, check to see if any
 * conferences are still "invalid", in which case call
 * "foundDeletedConference".  Note that its up to the 
 * RegistrarClient subclasses to actually delete the conference
 * from the table (if they want to).
 *
 ****************************************************************************/

void RegistrarClient::UpdateConferenceList(char *s) {
  AttrListTable* tbl = AttrListTable::read(s, "confnum", 30);
  AttributeList* al;
  
  /*
   * check for new conferences
   */

  AttrListList newlist;
  for (TableIterator(AttrLstTbl) i(*tbl); i.more(); i.next()) {
    if( !conference_tbl_->find( al, i.cur_key())) 
      newlist.append( i.cur_value());
  }

  /*
   * check for deleted conferences
   */

  IntLst gonelist;
  for (TableIterator(AttrLstTbl) j(*conference_tbl_); j.more(); j.next()) 
    if (!tbl->find( al, j.cur_key()))
      gonelist.append( j.cur_key());

  /*  
   * announce new and deleted conferences
   */

  delete tbl;
  for (ListItr(AttrListList) ni(newlist); ni.more(); ni.next()) 
    foundNewConference( ni.cur() );
  for (ListItr(IntLst) gi(gonelist); gi.more(); gi.next())
    foundDeletedConference( gi.cur());
}


/****************************************************************************
 *
 * We've received an up to date list of users for a conference.
 * Update our data structure.  Do the same kind of thing we did
 * with UpdateConferenceList.
 *
 ****************************************************************************/

void RegistrarClient::UpdateUserList(char *s) {
  AttrListTable* tbl = AttrListTable::read(s, "usernum", 30);
  AttributeList* al;
  AttrListTable* usrs;
  int confnum = -1; char conf_num[80];

  /*
   * look up list of users for the conference
   */

  for (TableIterator(AttrLstTbl) i(*tbl); i.more()&&(confnum==-1); i.next()) 
    if(i.cur_value()->find_attribute("confnum", conf_num))
      confnum = atoi(conf_num);

  if(users_tbl_->find( usrs, confnum)) {

    /*
     * check for new users
     */

    AttrListList newlist;
    for (TableIterator(AttrLstTbl) j(*tbl); j.more(); j.next()) 
      if( (j.cur_key()!=-1) && (!usrs->find( al, j.cur_key())))
	newlist.append( j.cur_value() );
    
    /*
     * check for deleted users 
     */

    IntLst gonelist;
    for (TableIterator(AttrLstTbl) k(*usrs); k.more(); k.next()) {
      if (!tbl->find( al, k.cur_key()))
	gonelist.append( k.cur_key());
    }
    
    /*
     * announce new and deleted users
     */

    delete tbl;
    for (ListItr(AttrListList) ni(newlist); ni.more(); ni.next())
      foundNewUser( ni.cur() );
    for (ListItr(IntLst) gi(gonelist); gi.more(); gi.next())
      foundDeletedUser( confnum, gi.cur());
  }
}


/****************************************************************************
 *
 * This routine converts a password string to a proper name.  Its
 * legal for the password file to store a name as e.g. "Mark &", which
 * equates to "Mark Roseman" if the userid is "roseman".  
 *
 ****************************************************************************/

char* cvt_gecos(char* gecos, char* uid) {
  char *s = malloc(80), *t = s, *x;
  while (*gecos) {
    if(*gecos=='&') {
      x = uid;
      *s++ = (*x++ - 32);
      while(*x)
	*s++ = *x++;
      gecos++;
    } else 
      *s++ = *gecos++;
  }
  *s = 0;
  return t;
}


/****************************************************************************
 *
 * This routine tells the registrar to join the current user to the
 * indicated conference.
 *
 ****************************************************************************/

void RegistrarClient::callJoinConference(int conference) 
{
  char s[500];

  AttributeList al;
  al.attribute("confnum", sprintf(s, "%d", conference));
  al.attribute("userid", cuserid(0));
  al.attribute("username", cvt_gecos(getpwnam(cuserid(0))->pw_gecos, 
				     cuserid(0)));
  al.attribute("host", GroupSession::host_name());
  al.attribute("port", sprintf(s, "%d", lPort()));
  al.write(s);
  writer_->sendMsg(ADDUSER, s);
  PollUsers(conference);
}


/****************************************************************************
 *
 * This routine provides a more generic join facility, which will be needed
 * if there are clients who join remote users to conferences
 *
 ****************************************************************************/

void RegistrarClient::callJoinConference(AttributeList* al ) 
{
  char s[500], confnum[80];
  al->find_attribute("confnum", confnum);
  al->write(s);
  writer_->sendMsg(ADDUSER, s);
  PollUsers(atoi(confnum));
}

/****************************************************************************
 *
 * This routine tells the registrar to pull the indicated user out of
 * the indicated conference.
 *
 ****************************************************************************/

void RegistrarClient::callLeaveConference( int conference, int user) {
  char s[30];
  sprintf(s,"confnum=%d:usernum=%d", conference, user);
  writer_->sendMsg(DELUSER, s);
  PollUsers( conference );
}


/****************************************************************************
 *
 * This routine tells the registrar to create a new conference.
 *
 ****************************************************************************/

void RegistrarClient::callNewConference(AttributeList* al) {
  char s[200];
  al->write(s);
  writer_->sendMsg(NEWCONF, s);
  PollConferences();
}


/****************************************************************************
 *
 * This routine tells the registrar to delete the indicated conference.
 *
 ****************************************************************************/

void RegistrarClient::callDeleteConference( int conference) {
  char s[30];
  sprintf(s, "confnum=%d", conference);
  writer_->sendMsg(DELCONF, s);
  PollConferences();
}


/****************************************************************************
 *
 * This routine asks the registrar to send the current list of conferences.
 *
 ****************************************************************************/

void RegistrarClient::PollConferences() {
  writer_->sendMsg(DISPCONF, nil);
}


/****************************************************************************
 *
 * This routine asks the registrar to send the user list for the given
 * conference.
 *
 ****************************************************************************/

void RegistrarClient::PollUsers(int conference) {
  char s[30];
  sprintf(s, "confnum=%d", conference);
  writer_->sendMsg(DISPUSER, s);
}


/****************************************************************************
 *
 * This routine creates the reader and writer for the conference, which
 * should hook up to the registrar.
 *
 ****************************************************************************/

boolean RegistrarClient::createReaderAndWriter(const char* rHost, int rPort) {
  writer_ = new Writer(rHost, rPort);
  if (writer_->server()) {
    reader_ = new CallbackRpcReader(&writer_->server());
    reader_->registerCallback( new StrActionCallback(RegistrarClient)
		  (this, &RegistrarClient::UpdateConferenceList), CONFLIST);
    reader_->registerCallback( new StrActionCallback(RegistrarClient)
		  (this, &RegistrarClient::UpdateUserList), USERLIST);
    return true;
  } else {
    cerr << "RegistrarClient: error opening connection to registrar at ";
    cerr << rHost << "." << rPort << "\n";
    delete writer_;
    writer_ = nil;
    return false;
  }
}


/****************************************************************************
 *
 * Return the local port number.  _lPort is a protected instance variable.
 *
 ****************************************************************************/

int RegistrarClient::lPort() {
  return _lPort;
}


/****************************************************************************
 *
 * Override the default method to not do some mucking around with recording.
 * This one simply calls listen on the socket.
 *
 ****************************************************************************/

void RegistrarClient::startListening() {
  _service = new rpcbuf;
  
  rpcbuf* ok = _service->listen(_lPort);
  _lPort = _service->port();
  if (!ok) {
    abort();
  }
  Dispatcher::instance().link(_service->fd(), Dispatcher::ReadMask, this);
}




/****************************************************************************
 *
 * People can connect up to me to ask me about locations of specific 
 * conferences.  The coordinator knows how to respond to these though.
 *
 ****************************************************************************/

void RegistrarClient::createReaderAndWriter(int fd) {
  InfoConnection* in = new InfoConnection( fd, -1, nil, ADDRREQ, new ConnActionCallback(Coordinator)(coord_, &Coordinator::addrReq));
}



