/***************************************************************************
 *
 * Class:  GroupSession implementation
 * Author: Mark Roseman
 * 
 * Provide a Session object specifically for GroupKit conferences.  The 
 * main duties here are setting up some default parameters, parsing some
 * GroupKit specific command line options, and making it easier to 
 * instantiate some GroupKit objects such as a Conference.
 *
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 09/20/92 MR        initial version
 * 10/12/92 MR        added internet domain information
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/groupsession.h>
#include <gk/conference.h>
#include <malloc.h>
#include <OS/string.h>
#include <InterViews/style.h>
#include <OS/host.h>
#include <string.h>


static PropertyData gkProps[] = {
  { "*CursorOverlay-foreground",          "black" },
  { "*CursorOverlay-localBitmapCursor",   "off" },
  { "*CursorOverlay-localRealCursor",     "on" },
  { "*CursorOverlay-bitmap",              "" },                  // not used
  { "*CursorOverlay-annotateWithNames",   "off" },               // not used
  { "*SketchOverlay-foreground",          "black" },             // not used
  { "*SketchOverlay-lineWidth",           "1" },                 // not used
  { "*SketchOverlay-button",              "1" },                 // not used
  { "*RegistrarHost",                     "janu.cpsc.ucalgary.ca" },
  { "*RegistrarPort",                     "1500" },
  { "*GroupKitBinDir",                    "/home/grouplab/mark/test/gk1.0/bin/SUN4" },
    { "*DomainRoot",                      "cpsc.ucalgary.ca" },
  { nil }
};

static OptionDesc gkOptions[] = {
  { "-confname",    "*ConferenceName",       OptionValueNext },
  { "-coordhost",   "*CoordinatorHost",      OptionValueNext },
  { "-coordport",   "*CoordinatorPort",      OptionValueNext },
  { "-confid",      "*ConferenceID",         OptionValueNext },
  { "-attrs",       "*ConferenceAttributes", OptionValueNext },
  { "domain=",      "*DomainRoot",           OptionValueAfter },
  { nil }
};



/****************************************************************************
 *
 * GroupSession constructor.  Augment the option and property lists we're
 * passed with the GroupKit defaults, and then call the Session constructor.
 *
 ****************************************************************************/

GroupSession::GroupSession( const char* classname, int& argc, char** argv,
			    const OptionDesc* opts, const PropertyData* initprops) : Session(classname, argc, argv, AugmentOpts(opts), AugmentProps(initprops))
{

}

/****************************************************************************
 *
 * Instantiate a new conference object, based upon the information we
 * obtained from command line arguments.
 *
 ****************************************************************************/

Conference* GroupSession::conference() {
  String *name = new String(), *host = new String();
  long port, id;
  if (!style()->find_attribute("ConferenceName", *name)) return nil;
  if (!style()->find_attribute("CoordinatorHost", *host)) return nil;
  if (!style()->find_attribute("CoordinatorPort", port)) return nil;
  if (!style()->find_attribute("ConferenceID", id)) return nil;
  return new Conference( (char*)name->string(), (char*)host->string(), (int)port, (int)id );
}


/****************************************************************************
 *
 * Create a new set of Options, combining the passed in set with the GroupKit
 * set.  Put the user passed in ones first so they'll override any we have
 * defined.
 *
 ****************************************************************************/

OptionDesc* GroupSession::AugmentOpts(const OptionDesc* opts) {
  const OptionDesc *o; OptionDesc *op;
  int i;
  if(opts==nil) return gkOptions;
  for ( o=&opts[0],i=0; o->name != nil; o++, i++) ;
  for (o = &gkOptions[0]; o->name != nil; o++, i++);
  OptionDesc* newOpts = (OptionDesc*)calloc( i+1, sizeof(OptionDesc));
  for(op = &newOpts[0], o = &opts[0]; o->name != nil; o++,op++) {
    op->name = o->name; op->path = o->path; op->style = o->style; op->value = o->value; 
  }
  for( o = &gkOptions[0]; o->name != nil; o++, op++) {
    op->name = o->name; op->path = o->path; op->style = o->style; op->value = o->value; 
  }
  op->name = nil;
  return newOpts;
}


/****************************************************************************
 *
 * Create a new set of Properties, combining the passed in set with the
 * GroupKit set.
 *
 ****************************************************************************/

PropertyData* GroupSession::AugmentProps(const PropertyData* props) {
  const PropertyData* p; PropertyData *pr;
  int i;
  if (props==nil) return gkProps;
  for (p = &props[0], i=0; p->path != nil; p++, i++) ;
  for (p = &gkProps[0]; p->path != nil; p++, i++);
  PropertyData* newProps = (PropertyData*)calloc( i+1, sizeof(PropertyData));
  for(pr = &newProps[0], p = &props[0]; p->path != nil; p++,pr++) {
    pr->path = p->path; pr->value = p->value; 
  }
  for( p = &gkProps[0]; p->path != nil; p++, pr++) {
    pr->path = p->path; pr->value = p->value; 
  }
  pr->path = nil;
  return newProps;
}


extern "C" {
  int gethostname(char*, int);
}

char GroupSession::host_name_[256];

/****************************************************************************
 *
 * Return the host name of the machine, qualified with the domain name,
 * e.g. janu.cpsc.ucalgary.ca
 *
 ****************************************************************************/

const char* GroupSession::host_name() {
  String domain_name;
  if(host_name_[0] == '\0') {
    gethostname(host_name_, sizeof(host_name_));
    if ((Session::instance()->style()->find_attribute("DomainRoot", domain_name)) 
      && (strlen( domain_name.string()) > 1)) {
      strcat(host_name_, ".");
      strcat(host_name_, domain_name.string());
    }
  }
  return host_name_;
}

