/****************************************************************************
 *
 * Class:  CallbackReader Implementation
 * Author: Mark Roseman
 * 
 * Much of the code is based on the RpcReader implementation
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 02/17/92 MR        initial version
 * 05/28/92 MR        changed to do callbacks from a table, eliminates
 *                    inheritance stuff..
 * 09/17/92 MR        added hook to do callback on connection closed
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/reader.h>
#include <gk/groupkit.h>
#include <Dispatch/dispatcher.h>
#include <Dispatch/rpchdr.h>
#include <Dispatch/rpcstream.h>

implementTable(ReaderCallbackTable, int, StrAction*);


/****************************************************************************
 * 
 * Various constructors and destructors
 *
 ****************************************************************************/

CallbackRpcReader::CallbackRpcReader(rpcstream* client, 
				     ReaderCallbackTable* tbl, int id, 
				     RpcAction* closeCB) : 
        RpcReader(client, 0), id_(id), closeCallback_(closeCB) 
{
  callbacks_ = ( tbl == nil ? new ReaderCallbackTable(FUNCTBLSIZE) : tbl );
}

CallbackRpcReader::CallbackRpcReader(int fd, boolean binary, 
				     ReaderCallbackTable* tbl, int id, 
				     RpcAction* closeCB) : 
        RpcReader(fd,0,binary), id_(id), closeCallback_(closeCB) 
{
  callbacks_ = ( tbl == nil ? new ReaderCallbackTable(FUNCTBLSIZE) : tbl );
}

CallbackRpcReader::~CallbackRpcReader()  {}


/****************************************************************************
 * 
 * This is called when the connection is broken.  If there was a 'close
 * callback' specified, call it.
 * 
 ****************************************************************************/

void CallbackRpcReader::connectionClosed(int fd) {
  if (closeCallback_ != nil)
    closeCallback_->execute( this, fd );
}


/****************************************************************************
 * 
 * Add a callback function to the hash table.
 * 
 ****************************************************************************/

void CallbackRpcReader::registerCallback(StrAction* act, int val) {
	callbacks_->insert(val, act);
}


/****************************************************************************
 * 
 * Handle receiving a message from the socket.  If all checks out okay,
 * call myexecute (below).
 * 
 ****************************************************************************/

int CallbackRpcReader::inputReady(int fd) {
    RpcHdr hdr;

    client() >> hdr;
    if (client().good() && !client().incomplete_request()) {
        CallbackRpcReader* reader = (CallbackRpcReader*)map(hdr.reader());
        if (!myexecute(reader, hdr)) {
            client().ignore(hdr.ndata());
        }
    }

    if (client().eof() || client().fail()) {
        connectionClosed(fd);
        return -1;              // don't ever call me again (i.e., detach me)
    } else if (client().incomplete_request()) {
        return 0;               // call me only when more input arrives
    } else {
        return 1;               // call me again as soon as possible
    }
}


/****************************************************************************
 * 
 * This actually calls the callback function.
 * 
 ****************************************************************************/

boolean CallbackRpcReader::myexecute(CallbackRpcReader* reader, RpcHdr& hdr) {
    if ((!reader) || (hdr.request() < 0)) 
        return false;

    StrAction* act;
    if(callbacks_ == nil) 
      return false;
    callbacks_->find(act, hdr.request());
    if (!act) 
        return false;

    char s[16000];
    client() >> s;
    act->execute(s);
    return true;
}


/****************************************************************************
 * 
 * return the id field
 *
 ****************************************************************************/

int CallbackRpcReader::id(void) {
  return id_;
}


/****************************************************************************
 * 
 * set the connection closed callback
 *
 ****************************************************************************/

void CallbackRpcReader::closeCallback( RpcAction* a) {
  closeCallback_ = a;
}
