
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/******************************************************************
*
*  File: zoom.c
*
*  Contents:  Routines to zoom in on vertex.
*
*/

#include "include.h"

/******************************************************************
*
*  Function: zoom_vertex()
*
*  Purpose:  Zoom in on a vertex by chopping off everything beyond
*            a given distance and fixing in place the jagged edges.
*/

void zoom_vertex(v_id,radius)
vertex_id v_id;  /* vertex to zoom on */
REAL radius;     /* cutoff distance from v_id */
{
  vertex_id vv_id,free_v;
  edge_id e_id,free_e;
  facet_id f_id,free_f;
  facetedge_id fe_id,free_fe,next,prev;
  int i;

  if ( web.dimension != SOAPFILM )
    { fprintf(stderr,"Zoom is only implemented for SOAPFILM model.\n");
      return;
    }

  if ( web.bodycount != 0 )
    { fprintf(stderr,"Zoom is not implemented for bodies.\n");
      return;
    }

  if ( radius <= 0.0 )
    { fprintf(stderr,"Must have positive cut-off radius.\n");
      return;
    }

  if ( !valid_id(v_id) || !valid_element(v_id) )
    { fprintf(stderr,"Vertex %d is not valid.\n",ordinal(v_id)+1);
      return;
    }

  
  /* eliminate all vertices beyond cutoff */
  free_v = vv_id = NULLVERTEX;
  while ( generate_all(VERTEX,&vv_id) )
    { /* have to be careful not to free a vertex before we use
	 it to generate the next one in the list */
      if ( valid_id(free_v) ) free_element(free_v);
      if ( distance(v_id,vv_id) > radius )
        free_v = vv_id;
      else  free_v = NULLVERTEX;
    }
  if ( valid_id(free_v) ) free_element(free_v);

  /* eliminate all edges connected to gone vertices */
  free_e = e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
    {
      if ( valid_id(free_e) ) free_element(free_e);
      if ( !valid_element(get_edge_tailv(e_id))
	   || !valid_element(get_edge_headv(e_id)) )
        free_e = e_id;
      else free_e = NULLEDGE;
    }
  if ( valid_id(free_e) ) free_element(free_e);

  /* eliminate all facets verging on gone edges */
  free_f = f_id = NULLFACET;
  while ( generate_all(FACET,&f_id) )
    { if ( valid_id(free_f) ) free_element(free_f);
      free_f = NULLFACET;
      fe_id = get_facet_fe(f_id);
      for ( i = 0 ; i < FACET_EDGES ; i++ )
	{ if ( !valid_element(get_fe_edge(fe_id)) )
	    { free_f = f_id;
	      break;
            }
	  fe_id = get_next_edge(fe_id);
	}
    }
  if ( valid_id(free_f) ) free_element(free_f);
    
  /* eliminate all facet-edges on gone facets */
  free_fe = fe_id = NULLFACETEDGE;
  while ( generate_all(FACETEDGE,&fe_id) )
    {
      if ( valid_id(free_fe) ) free_element(free_fe);
      if ( !valid_element(get_fe_facet(fe_id)) )
        { 
  	  e_id = get_fe_edge(fe_id);
	  if ( valid_element(e_id) )
	   {
             /* patch up ragged edges */
             next = get_next_facet(fe_id);
	     prev = get_prev_facet(fe_id);
	     if ( equal_id(next,fe_id) )
	      { /* was only facet on edge */
	        set_edge_fe(e_id,NULLFACETEDGE);
	      }
	     else
	      { /* close ranks */
	        set_next_facet(prev,next);
	        set_prev_facet(next,prev);
		set_edge_fe(e_id,next);
	      }
	 
	     /* fix edge in place */
	     set_attr(e_id,FIXED);
	     set_attr(get_edge_tailv(e_id),FIXED);
	     set_attr(get_edge_headv(e_id),FIXED);
	   }
          free_fe = fe_id;
        }
      else free_fe = NULLFACETEDGE;
    }
  if ( valid_id(free_fe) ) free_element(free_fe);
}

