;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: GARNET-GADGETS; Base: 10 -*-
;;;___________________________________________________________________
;;; The Garnet User Interface Development Environment
;;; Copyright (c) 1989, 1990 Carnegie Mellon University
;;; All rights reserved.  The CMU software License Agreement specifies
;;; the terms and conditions for use and redistribution.
;;;
;;; If you want to use this code or anything developed as part of the
;;; Garnet project, please contact Brad Myers (Brad.Myers@CS.CMU.EDU).
;;;___________________________________________________________________
;;;
;;;  Menu
;;;
;;;  Features and operation of Menu:
;;;     1)  The Menu object is a vertical list of strings framed by a white box.
;;;         An optional title may appear at the top of the menu in inverse
;;;         video.
;;;     2)  Click the left mouse button on a menu item to select the item.
;;;     3)  A box will be drawn around the selected item momentarily.
;;;     4)  The top level :value slot points to the string of the currently
;;;         selected item.
;;;     5)  The top level :value-obj slot points to the currently selected
;;;         item object, and can be set directly with S-VALUE to select an item.
;;;     6)  The :items slot may contain functions to be executed as each
;;;         item is selected, and :selection-function may contain a function
;;;         to be executed when any item selected.
;;;
;;;  Customizable slots:
;;;     1)  Left, top
;;;     2)  V-spacing -- distance between menu items
;;;     3)  H-align -- how to justify the items (:left, :center, or :right)
;;;     3)  Shadow-offset -- the amount of shadow that shows under the menu
;;;     4)  Text-offset -- the distance from the longest text to the menu frame
;;;     4)  Title -- a string to appear in inverse at the top of the menu
;;;                  (a value of NIL implies no title will appear)
;;;     5)  Title-Font and Item-Font
;;;     6)  Items -- This can be: 
;;;                  A list of strings, as in '("Large" ...), or
;;;                  a list of atoms, as in '(:center ...), or
;;;                  a list of string/function pairs, '(("Cut" Cut-FN) ...), or
;;;                  a list of atom/function pairs, '((:center Center-FN) ...).
;;;                  Each function will be executed when the associated button
;;;                  becomes selected.  The parameters are the top-level
;;;                  GADGET and the ITEM-STRING.
;;;     7)  Selection-function -- Global function to be executed when any button
;;;                               is selected.  Parameters are the top-level
;;;                               GADGET and the ITEM-STRING.
;;;
;;;     NOTE:  When the menu object is exported (as in the demo function
;;;            below), slots can be changed on the fly as well when the objects
;;;            are created.
;;;
;;;  Menu demo:
;;;     This module contains a function which creates a window and a menu.
;;;     To run it, enter (GARNET-GADGETS:menu-go).
;;;     To stop, enter (GARNET-GADGETS:menu-stop).
;;;
;;;  NOTE:  This module requires several schemata defined in GAD-button-parts.
;;;         Thus, GAD-button-parts.fasl must be loaded before this module.
;;;
;;;  Designed by Brad Myers
;;;  Written by Andrew Mickish

;;; CHANGE LOG
;;; 04/17/91 Andrew Mickish - Changed MENU's :frame-width formula to look at
;;;            the :width of the aggrelist instead of the :max-width
;;;

(in-package "GARNET-GADGETS" :use '("LISP" "KR"))

(export '(Menu
	  Menu-Go Menu-Stop Menu-Obj))


(create-instance 'MENU-SHADOW-RECT opal:rectangle
   (:left (o-formula (+ (gv (path 0 :parent) :left)
			(gv (path 0 :parent) :shadow-offset))))
   (:top (o-formula (+ (gv (path 0 :parent) :top)
		       (gv (path 0 :parent) :shadow-offset))))
   (:width (o-formula (gv (path 0 :parent :frame) :width)))
   (:height (o-formula (gv (path 0 :parent :frame) :height)))
   (:filling-style opal:black-fill))


(create-instance 'MENU-FRAME opal:rectangle
   (:filling-style opal:white-fill)
   (:left (o-formula (gv (path 0 :parent) :left)))
   (:top (o-formula (gv (path 0 :parent) :top)))
   (:width (o-formula (gv (path 0 :parent) :frame-width)))
   (:height (o-formula (gv (path 0 :parent) :frame-height))))


(create-instance 'MENU-FEEDBACK-RECT opal:rectangle
   (:left (o-formula (+ 1 (gvl :obj-over :left))))
   (:top (o-formula (+ 1 (gvl :obj-over :top))))
   (:width (o-formula (- (gvl :obj-over :width) 2)))
   (:height (o-formula (- (gvl :obj-over :height) 2)))
   (:visible (o-formula (gvl :obj-over))))


(create-instance 'MENU-TITLE opal:aggregadget
   (:text-offset (o-formula (gv (path 0 :parent) :text-offset)))
   (:text-offset2 (o-formula (gv (path 0 :parent) :text-offset2)))
   (:left (o-formula (gv (path 0 :parent) :left)))
   (:top (o-formula (gv (path 0 :parent) :top)))
   (:title (o-formula (gv (path 0 :parent) :title)))
   (:font (o-formula (gv (path 0 :parent) :title-font)))
   (:string-width (o-formula (if (gvl :title)
				 (+ (gvl :text :width) (gvl :text-offset2))
				 0)))
   (:width (o-formula (gv (path 0 :parent) :frame-width)))
   (:height (o-formula (if (gvl :title)
			   (+ (gvl :text :height) (gvl :text-offset2))
			   0)))
   (:visible (o-formula (gvl :title)))

   (:parts
    `((:text ,opal:text
	     (:left ,(o-formula (- (+ (gv (path 0 :parent) :left)
				      (floor (gv (path 0 :parent) :width) 2))
				   (floor (gvl :width) 2))))
	     (:top ,(o-formula (+ (gv (path 0 :parent) :top)
				  (gv (path 0 :parent) :text-offset))))
	     (:string ,(o-formula (gv (path 0 :parent) :title)))
	     (:font ,(o-formula (gv (path 0 :parent) :font)))
	     (:visible ,(o-formula (gv (path 0 :parent) :visible))))
      (:rect ,opal:rectangle
	     (:left ,(o-formula (+ 1 (gv (path 0 :parent) :left))))
	     (:top ,(o-formula (+ 1 (gv (path 0 :parent) :top))))
	     (:width ,(o-formula (- (gv (path 0 :parent) :width) 2)))
	     (:height ,(o-formula (- (gv (path 0 :parent) :height) 2)))
	     (:draw-function :xor)
	     (:filling-style ,opal:black-fill)
	     (:visible ,(o-formula (gv (path 0 :parent) :visible)))))))

(create-instance 'MENU-ITEM opal:aggregadget
   ;; Conditional formulas are required to allow either a list of strings or
   ;; a list of string/function pairs in the :items slot.
   (:string (o-formula (if (gv (path 0 :parent :parent) :actions-p)
			   (first (nth (gvl :rank)
				       (gv (path 1 :parent) :items)))
			   (nth (gvl :rank) (gv (path 0 :parent) :items)))))
   (:action (o-formula (when (gv (path 0 :parent :parent) :actions-p)
			 (second (nth (gvl :rank)
				      (gv (path 1 :parent) :items))))))

   (:font (o-formula (gv (path 0 :parent :parent) :item-font)))
   (:text-offset (o-formula (gv (path 0 :parent :parent) :text-offset)))
   (:text-offset2 (o-formula (gv (path 0 :parent :parent) :text-offset2)))
   (:h-align (o-formula (gv (path 0 :parent :parent) :h-align)))
   (:max-text-width-thus-far
       (o-formula (if (gvl :prev-visible)
		      (MAX (gvl :prev-visible :max-text-width-thus-far)
			   (gvl :text :width))
		      (gvl :text :width))))

   ;; These slots are used by the parent aggrelist in calculating its own
   ;; :max-width and :height slots.
   (:height (o-formula (+ (gvl :text :height) (gvl :text-offset2))))
   (:width (o-formula (MAX (gv (path 0 :parent :parent :menu-title)
			       :string-width)
			   (+ (gvl :parent :tail :max-text-width-thus-far)
			      (gvl :text-offset2)))))


   ;; An aggregadget is implemented for each text item so that there is no
   ;; "dead" space between menu items.
   (:parts
    `((:text ,user::pixmap
	     (:left ,(o-formula (case (gv (path 0 :parent) :h-align)
				  (:left (+ (gv (path 0 :parent) :left)
					    (gv (path 0 :parent) :text-offset)))
				  (:center (- (+ (gv (path 0 :parent) :left)
						 (floor (gv (path 0 :parent)
							    :width) 2))
					      (floor (gvl :width) 2)))
				  (:right (- (+ (gv (path 0 :parent) :left)
						(gv (path 0 :parent) :width))
					     (gvl :width)
					     (gv (path 0 :parent)
						 :text-offset))))))
	     (:top ,(o-formula (+ (gv (path 0 :parent) :top)
				  (gv (path 0 :parent) :text-offset))))
	     (:filename
	      ,(o-formula (let ((s (gv (path 0 :parent) :string)))
			    (if (stringp s)
				s
				(string-capitalize (string-trim ":" s))))))
	     (:font ,(o-formula (gv (path 0 :parent) :font)))))))

(create-instance 'MENU-ITEM-LIST opal:aggrelist
   (:left (o-formula (gv (path 0 :parent) :left)))
   (:top (o-formula (gv (path 0 :parent) :items-top)))
   (:v-spacing (o-formula (gv (path 0 :parent) :v-spacing)))
   (:items (o-formula (gv (path 0 :parent) :items)))
   (:item-prototype menu-item))


(create-instance 'MENU opal:aggregadget

   ;; Customizable slots
   ;;
   (:left 0) (:top 0) 
   (:v-spacing 0)
   (:h-align :left)     ; Implemented in MENU-ITEM code, not through aggrelists
   (:shadow-offset 5)
   (:text-offset 4)
   (:title NIL)
   (:title-font (create-instance NIL opal:font
		   (:family :serif)
		   (:size :large)
		   (:face :roman)))
   (:items '("Item 1" "Item 2" "Item 3" "Item 4"))
   (:item-font opal:default-font)
   (:selection-function NIL)

   (:value-obj NIL)
   (:value (o-formula (gvl :value-obj :string)))

   ;; Generally non-customizable slots
   ;;
   (:actions-p (o-formula (listp (first (gvl :items)))))
   (:items-top (o-formula (+ (gvl :top) (gvl :menu-title :height))))
   (:text-offset2 (o-formula (* 2 (gvl :text-offset))))
   (:frame-width (o-formula (gvl :menu-item-list :width)))
   (:frame-height (o-formula (+ (gvl :menu-title :height)
				(gvl :menu-item-list :height))))
   (:parts 
    `((:shadow ,menu-shadow-rect)
      (:frame ,menu-frame)
      (:feedback ,menu-feedback-rect)
      (:menu-title ,menu-title)
      (:menu-item-list ,menu-item-list)))
   (:interactors
    `((:selector ,inter:menu-interactor
		 (:window ,(o-formula (gv-local :self :operates-on :window)))
		 (:start-where ,(o-formula (list :element-of
						 (gvl :operates-on
						      :menu-item-list))))
		 (:running-where ,(o-formula (list :element-of
						   (gvl :operates-on
							:menu-item-list))))
		 (:how-set NIL)
		 (:feedback-obj ,(o-formula (gvl :operates-on :feedback)))
		 (:stop-action
		    (lambda (interactor obj-under-mouse)
		      (let ((action (g-value obj-under-mouse :action))
			    (gadget (g-value interactor :operates-on))
			    (string (g-value obj-under-mouse :string)))

			(s-value (g-value gadget :feedback) :obj-over NIL)
			(s-value gadget :value-obj obj-under-mouse)
			
			;; Global function for all items
			(kr-send gadget :selection-function gadget string)

			;; Local function assigned to item
			(when action
			  (funcall action gadget string)))))))))


;;;
;;;  MENU-GO
;;;

(defparameter Menu-win NIL)
(defparameter Menu-top-agg NIL)
(defparameter Menu-Obj NIL)

(defun Report-Item (top-level-obj string)
  (let ((value-obj (g-value top-level-obj :value-obj)))
    (format t "Menu-item object ~S selected with string ~S.~%"
	    value-obj string)))

(defun Menu-Go ()

  (create-instance 'menu-win inter:interactor-window
     (:top 5)(:left 700)(:height 360)(:width 300))

  (s-value Menu-win
	   :aggregate
	   (create-instance 'menu-top-agg opal:aggregate
	      (:overlapping NIL)))


  (create-instance 'menu-obj Menu
     (:left 20) (:top 20)
     (:selection-function #'Report-Item)
     (:title "Menu")
     (:items '(("eye1.xpm" my-cut) ("eye2.xpm" my-copy)
	       ("eye3.xpm" my-paste) ("eye4.xpm" my-undo) ("eye5.xpm" my-cancel))))
  (opal:add-components Menu-top-agg Menu-Obj)

  (format t "Leftdown on a menu item causes a box to be drawn around the~%")
  (format t "button, executes the function locally assigned to the item~%")
  (format t "(if there is one), and executes the function specified in~%")
  (format t ":selection-function (if there is one).~%")

  (opal:update Menu-win)

  NIL)


;;;
;;;  MENU-STOP
;;;

(defun Menu-Stop ()
  (opal:destroy Menu-win))
  

;;;  These functions are included to show that selection of one of the
;;;  menu items causes the associated local function to be called.

(defun my-cut (gadget item-string)
  (declare (ignore gadget item-string))
  (format t "Function CUT called~%~%"))
(defun my-copy (gadget item-string)
  (declare (ignore gadget item-string))
  (format t "Function COPY called~%~%"))
(defun my-paste (gadget item-string)
  (declare (ignore gadget item-string))
  (format t "Function PASTE called~%~%"))
(defun my-undo (gadget item-string)
  (declare (ignore gadget item-string))
  (format t "Function UNDO called~%~%"))
(defun my-cancel (gadget item-string)
  (declare (ignore gadget item-string))
  (format t "Function CANCEL called~%~%"))


