/*
// Abstract:
//	SEARCH---Searching
//
//	The Searhing module supports searching portions of memory for
//	specified data.
//
// Author:
//	Derek S. Nickel
//
// Creation date:
//	6 January 1991
//
// History:
// V01-001	Derek S. Nickel		 6-JAN-1991
//	Extracted into separte module.
//
*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include "memory.h"
#include "modes.h"
#include "pager.h"

static void search_port(int port_no, bin5_t start, bin5_t end,
long *matches, char *rest);

static void search_port_prefix(int port_no, bin5_t start, bin5_t end,
long *matches );

static void search_port_entry(int port_no, bin5_t start, bin5_t end,
long *matches );

/***********************************************************************
	Messages.
***********************************************************************/

#define vgr__nomatches \
"%%VOYAGER-I-NOMATCHES, no matches found\n"

#define vgr__matched \
"%%VOYAGER-I-MATCHED, found %ld matches\n"

/***********************************************************************
	search_memory
***********************************************************************/

void search_memory(bin5_t start, bin5_t end, char *data)
{
	long matches = 0;

	data = strupr(data);

	if (strncmp(data,"PREFIX",strlen(data)) == 0) {

		if (ports[0].loaded)
			search_port_prefix(0, start, end, &matches);

		if (ports[1].loaded)
			search_port_prefix(1, start, end, &matches);

		if (ports[2].loaded)
			search_port_prefix(2, start, end, &matches);

	} else if (strncmp(data,"ENTRY",strlen(data)) == 0) {

		if (ports[0].loaded)
			search_port_entry(0, start, end, &matches);

		if (ports[1].loaded)
			search_port_entry(1, start, end, &matches);

		if (ports[2].loaded)
			search_port_entry(2, start, end, &matches);

	} else {

		if (*data == '\'') {
			data++;
		} else {
			strrev(data);
		}

		if (ports[0].loaded)
			search_port(0, start, end, &matches, data);

		if (ports[1].loaded)
			search_port(1, start, end, &matches, data);

		if (ports[2].loaded)
			search_port(2, start, end, &matches, data);

	}

	/*
	// Finish last line and print summary.
	*/

	if (matches % 10 != 0) putchar('\n');

	pager(0);
	if (matches == 0)
		printf(vgr__nomatches);
	else
		printf(vgr__matched, matches);
}

/***********************************************************************
	search_port
***********************************************************************/

static void search_port(int port_no, bin5_t start, bin5_t end,
long *matches, char *data)
{
	char buf[513];
	char *pos;
	bin5_t found_at;
	bin5_t min_adr = 0;
	bin5_t max_adr = 0;
	size_t len_data;
	size_t offset;
	int nibs;

	/*
	// Define allowable search range for this port.
	*/

	min_adr = ports[port_no].min_adr;
	max_adr = ports[port_no].max_adr;

	/*
	// Force start amd end into range.
	*/

	if (start < min_adr) start = min_adr;
	if (start > max_adr) start = max_adr;
	if (end < min_adr) end = min_adr;
	if (end > max_adr) end = max_adr;

	/*
	// Verify correct sequencing.
	*/

	if (start > end) {
		bin5_t tmp;
		tmp = start;
		start = end;
		end = tmp;
	}

	/*
	// Search memory in chunks.
	*/

	len_data = strlen(data);

	while (start < end - len_data) {

		/* putchar('.'); */

		SetWorkPtr(start);
		nibs = min(512L, end - start + 1L);
		GetNNibbles(buf,nibs);
		pos = buf;

		while (pos = strstr(pos,data)) {
			if (*matches % 10 == 0) pager(0);
			(*matches)++;
			offset = pos - buf;
			found_at = start + offset;
			printf(" %05lX ", found_at);
			if (*matches % 10 == 0) putchar('\n');
			pos++;
		}

		offset = nibs - len_data + 1;
		start += offset;
	}
}

/***********************************************************************
	search_port_prefix
***********************************************************************/

static void search_port_prefix(int port_no, bin5_t start, bin5_t end,
long *matches)
{
	bin5_t min_adr = 0;
	bin5_t max_adr = 0;
	bin5_t prefix;

	/*
	// Define allowable search range for this port.
	*/

	min_adr = ports[port_no].min_adr;
	max_adr = ports[port_no].max_adr;

	/*
	// Force start amd end into range.
	*/

	if (start < min_adr) start = min_adr;
	if (start > max_adr) start = max_adr;
	if (end < min_adr) end = min_adr;
	if (end > max_adr) end = max_adr;

	/*
	// Verify correct sequencing.
	*/

	if (start > end) {
		bin5_t tmp;
		tmp = start;
		start = end;
		end = tmp;
	}

	/*
	// Search memory.
	*/

	while (start < end - 5) {

		/* putchar('.'); */

		SetWorkPtr(start);
		prefix = get_bin5();

		if (prefix == start + 5) {
			if (*matches % 10 == 0) pager(0);
			(*matches)++;
			printf(" %05lX ", start);
			if (*matches % 10 == 0) putchar('\n');
			start += 5;
		} else {
			start += 1;
		}
	}
}

/***********************************************************************
	search_port_entry
***********************************************************************/

static void search_port_entry(int port_no, bin5_t start, bin5_t end,
long *matches)
{
	bin5_t min_adr = 0;
	bin5_t max_adr = 0;
	bin5_t prefix;

	/*
	// Define allowable search range for this port.
	*/

	min_adr = ports[port_no].min_adr;
	max_adr = ports[port_no].max_adr;

	/*
	// Force start amd end into range.
	*/

	if (start < min_adr) start = min_adr;
	if (start > max_adr) start = max_adr;
	if (end < min_adr) end = min_adr;
	if (end > max_adr) end = max_adr;

	/*
	// Verify correct sequencing.
	*/

	if (start > end) {
		bin5_t tmp;
		tmp = start;
		start = end;
		end = tmp;
	}

	/*
	// Search memory.
	*/

	while (start < end - 5) {

		/* putchar('.'); */

		SetWorkPtr(start);
		prefix = get_bin5();

		if ((prefix == start + 5) || (prefix == 0x02D9D)) {
			if (*matches % 10 == 0) pager(0);
			(*matches)++;
			printf(" %05lX ", start);
			if (*matches % 10 == 0) putchar('\n');
			start += 5;
		} else {
			start += 1;
		}
	}
}
