/*
 *  Copyright 2004 the mime4j project
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
using System;
using System.IO;
using System.Collections;
using System.Text;

using NI.Email.Mime.Message;
using NI.Email.Mime.Field;
using NI.Email.Mime.Field.Address;
using NI.Email.Mime.Util;
using NUnit.Framework;

namespace NI.Tests.Email.Mime.Message {

	/**
	* 
	* 
	*
	* @author Niklas Therning
	* @version $Id: MimeMessageParserTest.java,v 1.4 2004/10/25 07:26:47 ntherning Exp $
	*/
	[TestFixture]
	public class MimeMessageParserTest {

		public MimeMessageParserTest() { }

		[Test]
		public void test_MimeField() {
			string[] headers = new string[] {
				"Subject: (=?ISO-8859-1?Q?a?=)",
				"Subject: (=?ISO-8859-1?Q?a?= b)",
				"Subject: (=?ISO-8859-1?Q?a?= =?ISO-8859-1?Q?b?=)",
				"Subject: (=?ISO-8859-1?Q?a?=  =?ISO-8859-1?Q?b?=)",
				@"Subject: (=?ISO-8859-1?Q?a?=
	=?ISO-8859-1?Q?b?=)",
				@"Subject: (=?ISO-8859-1?Q?a_b?=)",
				@"Subject: (=?ISO-8859-1?Q?a?= =?ISO-8859-2?Q?_b?=)"
			};
			string[] decodedValues = new string[] {
				"(a)", "(a b)", "(ab)", "(ab)", "(ab)", "(a b)", "(a b)"
			};

			for (int i=0; i<headers.Length; i++) {
				MimeField fld = MimeField.Parse( headers[i] );
				Assert.AreEqual( decodedValues[i], ((UnstructuredField)fld).Value, "Decode failed" );
			}

			// special test for MailBox field
			MimeField fromFld = MimeField.Parse("From: (=?ISO-8859-1?Q?a?= =?ISO-8859-2?Q?_b?=) <ojarnef@admin.kth.se>");
			string fromName = ((NamedMailbox) ((MailboxListField)fromFld).MailboxList[0]).Name;
			Assert.AreEqual( fromName, "(a b)" );

			// special test for ContentType field
			MimeField contentTypeField = MimeField.Parse(@"Content-Type: text/plain; charset=""US-ASCII""; name==?koi8-r?B?89DJ08/LLmRvYw==?=");
			Assert.AreEqual( ((ContentTypeField)contentTypeField).Parameters["name"].ToString(), ".doc" );
		}

		[Test]
		public void test_DateTimeField() {

			string[] strDateTime = new string[] {
				"29 Mar 2007 15:38:39 +0000",
				"29 Mar 2007 15:38:39 GMT",
				"29 Mar 2007 9:38:39 CST",
				"29 Mar 2007 16:38:39 N",
				"29 Mar 2007 15:38:39 +0000 (GMT)",
				"29 Mar 2007 10:38:39 -0500 (EST)",
			};

			for (int i=0; i<strDateTime.Length; i++)
				Assert.AreEqual( DateTime.Parse("29 Mar 2007 15:38:39 GMT"), ((DateTimeField)MimeField.Parse("Date", strDateTime[i])).Date );
		}

		[Test]
		public void test_Parse() {
			string[] fileNames = Directory.GetFiles("../../testmsgs", "*.msg");
			
			for (int i = 0; i < fileNames.Length; i++) {
	            
				Console.WriteLine("Parsing " + fileNames[i]);
				
				String result;
				try {
					MimeMessage m = new MimeMessage( new FileStream(fileNames[i], FileMode.Open, FileAccess.Read) );
				
					result = GetStructure(m,
						Path.GetFileNameWithoutExtension(fileNames[i]), "1");
				} catch (Exception ex) {
					throw new Exception("While parsing "+fileNames[i], ex);
				}
	            
				String xmlFilePath = 
					Path.Combine(
					Path.GetDirectoryName(fileNames[i]),
					Path.GetFileNameWithoutExtension(fileNames[i]) + "_decoded.xml" );
				String expected = new StreamReader(xmlFilePath,
					Encoding.GetEncoding("ISO8859-1") ).ReadToEnd();
	            
				Assert.AreEqual(result, expected, "Error parsing " + fileNames[i]);
			}
		}

        [Test]
        public void test_TempFileParse()
        {
            string[] fileNames = Directory.GetFiles("../../testmsgs", "*.msg");

            for (int i = 0; i < fileNames.Length; i++)
            {

                Console.WriteLine("Parsing " + fileNames[i]);

                String result;
                try
                {
                    MimeMessage m = new MimeMessage(new FileStream(fileNames[i], FileMode.Open, FileAccess.Read), true);

                    result = GetStructure(m,
                        Path.GetFileNameWithoutExtension(fileNames[i]), "1");
                }
                catch (Exception ex)
                {
                    throw new Exception("While parsing " + fileNames[i], ex);
                }

                String xmlFilePath =
                    Path.Combine(
                    Path.GetDirectoryName(fileNames[i]),
                    Path.GetFileNameWithoutExtension(fileNames[i]) + "_decoded.xml");
                String expected = new StreamReader(xmlFilePath,
                    Encoding.GetEncoding("ISO8859-1")).ReadToEnd();

                Assert.AreEqual(result, expected, "Error parsing " + fileNames[i]);
            }
        }
		
			    
		private String Escape(String s) {
			s = s.Replace("&", "&amp;");
			s = s.Replace("<", "&lt;");
			return s.Replace(">", "&gt;");
		}

		private String GetStructure(Entity e, String prefix, String id)  {
	        
			StringBuilder sb = new StringBuilder();
	        
			if (e is MimeMessage) {
				sb.Append("<message>\r\n");
			} else {
				sb.Append("<body-part>\r\n");
			}            
	        
			sb.Append("<header>\r\n");
			foreach (MimeField fld in e.Header.Fields) {
				sb.Append("<field>\r\n" + Escape(fld.Raw) 
						+ "</field>\r\n");
			}
			sb.Append("</header>\r\n");
	        
			if (e.Body is Multipart) {
				sb.Append("<multipart>\r\n");
	            
				Multipart multipart =(Multipart) e.Body; 
				IList parts =multipart.BodyParts;

				sb.Append("<preamble>\r\n");
				sb.Append( Escape(multipart.Preamble));
				sb.Append("</preamble>\r\n");
	            
				for (int i=0; i<parts.Count; i++) {
					sb.Append( GetStructure( (Entity)parts[i], prefix, id + "_" + (i+1).ToString() ) );
				}

				sb.Append("<epilogue>\r\n");
				sb.Append( Escape(multipart.Epilogue));
				sb.Append("</epilogue>\r\n");
	            
				sb.Append("</multipart>\r\n");
	            
			} else if (e.Body is MimeMessage) {
				sb.Append( GetStructure( (MimeMessage) e.Body, prefix, id + "_1"));
			} else {
				IBody b = e.Body;
				String name = prefix + "_decoded_" + id 
								+ (b is ITextBody ? ".txt" : ".bin");
				String tag = b is ITextBody ? "text-body" : "binary-body";
				sb.Append("<" + tag + " name=\"" + name + "\"/>\r\n");
	                
	            string fullAttachedFileName = Path.Combine("../../testmsgs",name);
				FileStream expectedPerlFile = new FileStream( 
					fullAttachedFileName,
					FileMode.Open, FileAccess.Read);
	            
				if (b is ITextBody) {
					String charset = CharsetUtil.ToPlatformCharset(e.Charset);
					if (charset == null) charset = "ISO8859-1";
	                
	                string s1 = new StreamReader(expectedPerlFile, Encoding.GetEncoding(charset) ).ReadToEnd().Replace("\r", "");
                    StreamReader reader = ((ITextBody)b).Reader;
	                string s2 = reader.ReadToEnd().Replace("\r", "");
                    reader.Close();
					
					Assert.AreEqual( s1, s2, fullAttachedFileName );
				} 
                else 
                {
                    BinaryReader reader = ((IBinaryBody)b).Reader;					
					Assert.AreEqual(
						EqualBinary(reader, new BinaryReader(expectedPerlFile) ),
						true, fullAttachedFileName);
 				}
			}
	        
	        
			if (e is MimeMessage) {
				sb.Append("</message>\r\n");
			} else {
				sb.Append("</body-part>\r\n");
			}            
	        
			return sb.ToString();
		}

		private bool EqualBinary(BinaryReader br1, BinaryReader br2) {
			do {
				int b1, b2;
				
				try {
					b1 = br1.ReadByte();
				} catch (EndOfStreamException e) { b1 = -1; }

				try {
					b2 = br2.ReadByte();
				} catch (EndOfStreamException e) { b2 = -1; }

                if (b1 != b2)
                {
                    br1.Close();
                    br2.Close();
                    return false;
                }
                if (b1 == -1)
                {
                    br1.Close();
                    br2.Close();
                    return true;
                }
			} while (true);
			
		}

	}


}