/*
 *  Copyright 2004 the mime4j project
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
using System;
using System.IO;
using System.Collections;
using System.Text;

using NI.Email.Mime.Message;
using NI.Email.Mime.Field;
using NUnit.Framework;

namespace NI.Tests.Email.Mime.Message {

	/**
	* 
	*
	* @author Niklas Therning
	* @version $Id: MimeMessageTest.java,v 1.4 2004/10/02 12:41:11 ntherning Exp $
	*/
	[TestFixture]
	public class MimeMessageTest {
	
		private Header headerTextPlain = null;
		private Header headerMimeMessageRFC822 = null;
		private Header headerEmpty = null;
		private Header headerMultipartMixed = null;
		private Header headerMultipartDigest = null;

		public MimeMessageTest() {
			headerTextPlain = new Header();
			headerMimeMessageRFC822 = new Header();
			headerEmpty = new Header();
			headerMultipartMixed = new Header();
			headerMultipartDigest = new Header();
	        
			headerTextPlain.AddField(
					MimeField.Parse("Content-Type: text/plain"));
			headerMimeMessageRFC822.AddField(
					MimeField.Parse("Content-Type: message/RFC822"));
			headerMultipartMixed.AddField(
					MimeField.Parse("Content-Type: multipart/mixed; boundary=foo"));
			headerMultipartDigest.AddField(
					MimeField.Parse("Content-Type: multipart/digest; boundary=foo"));
		}

		[Test]
		public void test_GetMimeType() {
			MimeMessage parent = null;
			MimeMessage child = null;
	        
			parent = new MimeMessage();
			child = new MimeMessage();
			child.Parent = parent;
			parent.Header = headerMultipartDigest;
			child.Header = headerEmpty;
			Assert.AreEqual("message/rfc822", child.MimeType, "multipart/digest, empty");
			
			child.Header = headerTextPlain;
			Assert.AreEqual("text/plain", child.MimeType, "multipart/digest, text/plain");
			child.Header = headerMimeMessageRFC822;
			Assert.AreEqual("message/rfc822", child.MimeType, "multipart/digest, message/rfc822");
	        
			parent = new MimeMessage();
			child = new MimeMessage();
			child.Parent = parent;
			parent.Header = headerMultipartMixed;
			child.Header = headerEmpty;
			Assert.AreEqual("text/plain", child.MimeType, "multipart/mixed, empty");
			child.Header = headerTextPlain;
			Assert.AreEqual("text/plain", 	child.MimeType, "multipart/mixed, text/plain");
			child.Header = headerMimeMessageRFC822;
			Assert.AreEqual("message/rfc822", child.MimeType, "multipart/mixed, message/rfc822");
	        
			child = new MimeMessage();
			child.Header = headerEmpty;
			Assert.AreEqual("text/plain", child.MimeType, "null, empty");
			child.Header = headerTextPlain;
			Assert.AreEqual("text/plain", child.MimeType, "null, text/plain");
			child.Header = headerMimeMessageRFC822;
			Assert.AreEqual("message/rfc822", child.MimeType, "null, message/rfc822");
		}

		[Test]
		public void test_IsMultipart() {
			MimeMessage m = new MimeMessage();
	        
			m.Header = headerEmpty;
			Assert.AreEqual(!m.IsMultipart, true, "empty");
	        
			m.Header = headerTextPlain;
			Assert.AreEqual(!m.IsMultipart, true, "text/plain");
	        
			m.Header = headerMultipartDigest;
			Assert.AreEqual(m.IsMultipart, true, "multipart/digest");
	        
			m.Header = headerMultipartMixed;
			Assert.AreEqual(m.IsMultipart, true, "multipart/mixed");
		}

	}

}