/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMOT.C - SNMP object functions
 *
 * $Revision:   1.3  $ $Date:   08 Jul 1992 17:33:06  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMOT.C_V  $
 * 
 *    Rev 1.3   08 Jul 1992 17:33:06   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.2   29 Jun 1992 19:47:28   gfoster
 * Changed the incorrect escape sequence in the fprintf
 * statement from "\%s\"..." to "\"%s\"...".
 * 
 *    Rev 1.1   19 Jun 1992 16:14:56   gfoster
 * Copyright text was reformated.
 * 
 * Fixed bug so that situation is correctly reported when
 * duplicate OBJECT-TYPE objects are defined.
 * 
 * Fixed size/range compatibility checking.
 * 
 *    Rev 1.0   27 May 1992 16:02:46   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <string.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/** finishOT - finish definition of an SNMP object
*
* call with:
*   pOid - object
*   pSyn - syntax info
*   usAccess - access for object
*   usStatus - status for object
*   pDesc - description for object
*   pRefer - reference for object
*   pNaPar - parent for object
*   ulVal - last OID component value for object
*
* returns:
*   ptr to object or NULL for error
*/
    MIBSYM *
#ifdef __STDC__
finishOT(MIBSYM *pOid, MIBSYN *pSyn, USHORT usAccess,
         USHORT usStatus, STRTAB *pDesc, STRTAB *pRefer,
         STRTAB *pNaPar, ULONG ulVal)
#else
finishOT(pOid, pSyn, usAccess, usStatus, pDesc, pRefer, pNaPar, ulVal)
    MIBSYM *pOid;
    MIBSYN *pSyn;
    USHORT usAccess;
    USHORT usStatus;
    STRTAB *pDesc;
    STRTAB *pRefer;
    STRTAB *pNaPar;
    ULONG ulVal;
#endif /* __STDC__ */
{
    MIBSYM *pPar;
    MIBSYM *pSymPar;
    MIBSYM *pT;
    MIBSYM *pOT;
    USHORT usSyntax;
    MIBSYM *pMod;
    MIBSYM *pSeq;
    BOOL fCheckCirc = FALSE;


    /* check parent info */
    /* check if parent already defined */
    pSymPar = pNaPar->pSym;
    pMod = pOid->pMod;
    if ((pSymPar != NULL) && (pSymPar->pMod == pMod)) {
        /* parent defined */
        /* check if import */
        if (pSymPar->usType == MIBSYMimp) {
            pSymPar->ut.imp.cUse++;
            pSymPar = pSymPar->ut.imp.pImpSym;
        } else {
            fCheckCirc = TRUE;
        }

        /* check if parent is an OID type */
        if ((pSymPar->usType & MIBSYMmask) != MIBSYMoid) {
            yyerror("parent \"%s\" of \"%s\" must be OID object",
                    pSymPar->pszName, pOid->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }
        pPar = pSymPar;
        if (fCheckCirc) {
            /* parent already defined and not import */
            /* check for circular reference */
            for (pT = pPar; pT != NULL; pT = pT->ut.oid.pPar) {
                if (pT == pOid) {
                    yyerror("\"%s\" is circular defined", pOid->pszName);
#ifdef OLD
                    yyterm();
#endif
                    return(NULL);
                }
            }
        }
    } else {
        /* parent not defined, allocate one */
        pPar = newOID(pNaPar, pMod);
    }


    /* store values */
    pOid->usType = MIBSYMoid;
    pOid->ut.oid.ulVal = ulVal;
    pOid->ut.oid.pPar = pPar;
    /* insert in children list for parent */
    for (pOT = NULL, pT = pPar->ut.oid.pChild; pT != NULL;
            pOT = pT, pT = pT->ut.oid.pSib) {
        if (ulVal < pT->ut.oid.ulVal) {
            /* found place to insert */
            break;
        }
        if (ulVal == pT->ut.oid.ulVal) {
            /* duplicate values */
            if (pT->pMod == pMod) {
                if (strcmp(pOid->pszName, pT->pszName) == 0)
                    yyerror("\"%s\" is already defined",
                            pOid->pszName);
                else
                    yyerror("\"%s\" and \"%s\" have same registration value",
                            pOid->pszName, pT->pszName);
            } else {
                if (strcmp(pOid->pszName, pT->pszName) == 0)
                    yyerror("\"%s\" is already defined in module \"%s\"",
                            pT->pszName, (pT->pMod)->pszName);
                else
                    yyerror(
                "\"%s\" and \"%s\" from \"%s\" have same registration value",
                            pOid->pszName, pT->pszName,
                            (pT->pMod)->pszName);
            }
#ifdef OLD
            yyterm();
            return(NULL);
#endif
            break;      /* insert item */
        }
    }
    if (pOT == NULL) {
        /* insert as first child */
        pPar->ut.oid.pChild = pOid;
    } else {
        /* insert in list */
        pOT->ut.oid.pSib = pOid;
    }
    pOid->ut.oid.pSib = pT;

    /* get syntax for object */
    pOid->ut.oid.syn = *pSyn;   /* copy syntax info */

    /* get resolved syntax */
    if (pSyn->usSyntax == MIBSYNtc) {
        /* a TC, so get resolved syntax from TC's resolved syntax */
        pOid->ut.oid.rsyn = (pSyn->usi.pTC)->ut.tc.rsyn;
        /* add size/range if present */
        if ((pSyn->usSizeRange != MIBSRno) &&
                (pSyn->usSizeRange != MIBSRbad)) {
            pOid->ut.oid.rsyn.usSizeRange = pSyn->usSizeRange;
            pOid->ut.oid.rsyn.usr = pSyn->usr;
        }
    } else
        pOid->ut.oid.rsyn = *pSyn; /* syntax and resolved syn are the same */


    /* determine object type */
    usSyntax = pOid->ut.oid.syn.usSyntax;

    if (usSyntax == MIBSYNseqOf) {
        pOid->ut.oid.usOType = MIBOTtab;   /* table */
    } else if (usSyntax == MIBSYNseq) {
        pOid->ut.oid.usOType = MIBOTrow;   /* row */
        pSeq = pOid->ut.oid.syn.usi.pSeq;
        if (pSeq != NULL) {
            if (pSeq->ut.seq.pRow != NULL) {
                yyerror("Row \"%s\" uses sequences \"%s\" which is already used by row \"%s\"",
                        pOid->pszName, pSeq->pszName,
                        (pSeq->ut.seq.pRow)->pszName);
#ifdef OLD
                yyterm();
                return(NULL);
#endif
            } else
                pSeq->ut.seq.pRow = pOid;
        }
    } else if (usSyntax == MIBSYNbad) {
        /* syntax was bad, most likely a row with
         *   bad sequence name or object with bad
         *   syntax name */

        /* check if parent a table */
        if (pPar->ut.oid.usOType == MIBOTtab)
            pOid->ut.oid.usOType = MIBOTrow; /* row */
        else
            pOid->ut.oid.usOType = MIBOTobj; /* object */
    } else
        pOid->ut.oid.usOType = MIBOTobj;   /* object */

    /* fill in remaining fields */
    pOid->ut.oid.usAccess = usAccess;
    pOid->usStatus = usStatus;
    if ((pDesc != NULL) && (*(pDesc->pszVal) != 0))
        pOid->pszDesc = pDesc->pszVal;
    if ((pRefer != NULL) && (*(pRefer->pszVal) != 0))
        pOid->pszRefer = pRefer->pszVal;

    return(pOid);

} /* finishOT */


/** checkSyntax - check and store syntax info
*
* call with:
*   pSyn - syntax info
*   pNa - textual convention or sequence name
*   usSyntax - type of syntax
*   usSizeRange - type of size/range
*   ulFirst - first size/range
*   ulSecond - second size/range
*   pMod - current module
*
* returns:
*   TRUE - values OK
*   FALSE - problem with values
*/
    BOOL
#ifdef __STDC__
checkSyntax(MIBSYN *pSyn, STRTAB *pNa, USHORT usSyntax,
            USHORT usSizeRange, ULONG ulFirst,
            ULONG ulSecond, MIBSYM *pMod)
#else
checkSyntax(pSyn, pNa, usSyntax, usSizeRange, ulFirst, ulSecond, pMod)
    MIBSYN *pSyn;
    STRTAB *pNa;
    USHORT usSyntax;
    USHORT usSizeRange;
    ULONG ulFirst;
    ULONG ulSecond;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    BOOL fOk = TRUE;


    memset(pSyn, 0, sizeof(MIBSYN)); /* init syntax */

    pSyn->usSyntax = usSyntax;
    pSyn->usSizeRange = usSizeRange;

    /* handle SEQUENCE OF */
    if (usSyntax == MIBSYNseqOf) {
        /* sequence of */
        /* check if name known in current module */
        pSym = pNa->pSym;
        if ((pSym == NULL) || (pSym->pMod != pMod)) {
            /* symbol not known */
            /* allocate new sequence */
            pSym = newSEQ(pNa, pMod);
        } else {
            /* symbol in current module */
            /* check for import */
            if (pSym->usType == MIBSYMimp) {
                /* an import */
                pSym->ut.imp.cUse++;
                pSym = pSym->ut.imp.pImpSym;
            }
            /* check symbol type */
            if ((pSym->usType & MIBSYMmask) != MIBSYMseq) {
                /* symbol is some other type */
                yyerror("Object \"%s\" should be the name of a sequence",
                        pSym->pszName);
#ifdef OLD
                yyterm();
                return(FALSE);
#endif
                pSyn->usSyntax = MIBSYNbad; /* change syntax to bad */
                return(FALSE);
            }
        }    
        /* everyting OK */
        pSyn->usi.pSeq = pSym;
        pSym->ut.seq.cUse++;
        return(TRUE);
    }


    /* NOTE: semantics for size/range must be checked later */
    switch(usSizeRange) {
    case MIBSRno:   /* none */
        break;
    case MIBSRfs:   /* fixed size */
    case MIBSRvs:   /* variable size */
        pSyn->usr.usSize[0] = (USHORT)ulFirst;
        pSyn->usr.usSize[1] = (USHORT)ulSecond;
        if (ulFirst > ulSecond) {
            yyerror("Invalid size values specifed, smallest greater than largest");
            fOk = FALSE;
            pSyn->usSizeRange = MIBSRno; /* set to no size/range */
        }
        if ((ulFirst == 0L) && (ulSecond == 0L)) {
            yyerror("Invalid size values specifed, both sizes cannot tbe zero");
            fOk = FALSE;
            pSyn->usSizeRange = MIBSRno; /* set to no size/range */
        }
        break;
    case MIBSRpp:   /* positive to positive */
    case MIBSRnn:   /* negative to negative */
    case MIBSRnp:   /* negative to positive */
        pSyn->usr.ulRange[0] = ulFirst;
        pSyn->usr.ulRange[1] = ulSecond;
        if (((usSizeRange == MIBSRpp) &&
                    (ulFirst > ulSecond)) ||
                ((usSizeRange == MIBSRnn) &&
                    (ulFirst < ulSecond))) {
            yyerror("Invalid range values specified, smallest greater than largest");
            fOk = FALSE;
            pSyn->usSizeRange = MIBSRno; /* set to no size/range */
        }
        break;
    default:
        yyerror("checkSyntax: invalid size/range type of %d",
                usSizeRange);
        yyterm();
        pSyn->usSizeRange = MIBSRno; /* set to no size/range */
        return(FALSE);
    }


    /* check for textual convention (or sequence) */
    if (usSyntax == MIBSYNtc) {
        /* check if name known in current module */
        pSym = pNa->pSym;
        if ((pSym == NULL) || (pSym->pMod != pMod)) {
            /* symbol not known */
            /* textual conventions must be defined, so must be sequence */
            /* But, a properly written MIB will have seq already referenced */
            yyerror("Syntax \"%s\" not known in current module",
                    pNa->pszVal);
            pSyn->usSyntax = MIBSYNbad; /* change syntax to BAD */
            pSyn->usSizeRange = MIBSRno; /* change to no size/Range */
#ifdef OLD
            yyterm();
#endif
            return(FALSE);
        }
        /* symbol in current module */
        /* check for import */
        if (pSym->usType == MIBSYMimp) {
            pSym->ut.imp.cUse++;
            pSym = pSym->ut.imp.pImpSym;
        }
        /* check symbol type */
        if ((pSym->usType & MIBSYMmask) == MIBSYMseq) {
            /* sequence */
            /* check for size/range */
            if (usSizeRange != MIBSRno) {
                yyerror("size/range must not be specified with sequence \"%s\"",
                        pSym->pszName);
#ifdef OLD
                yyterm();
                return(FALSE);
#endif
                fOk = FALSE;
                pSyn->usSizeRange = MIBSRno; /* change to no size/range */
            }
            pSyn->usSyntax = MIBSYNseq; /* change syntax to sequence */
            pSyn->usi.pSeq = pSym;
            pSym->ut.seq.cUse++;
            return(fOk);
        } else if (pSym->usType != MIBSYMtc) {
            /* symbol is some other type */
            yyerror("Object \"%s\" specified where a syntax should be",
                    pSym->pszName);
#ifdef OLD
            yyterm();
#endif
            pSyn->usSyntax = MIBSYNbad; /* change syntax to bad */
            pSyn->usSizeRange = MIBSRno; /* change to no size/Range */
            return(FALSE);
        }
        /* textual convention */
        pSyn->usi.pTC = pSym;
        pSym->ut.tc.cUse++;

        /* check for compatible size/range */
        if ((pSyn->usSizeRange != MIBSRno) &&
                (pSym->ut.tc.rsyn.usSizeRange != MIBSRno)) {
            yyerror("size/range can not be specified since \"%s\" already has this specified",
                    pSym->pszName);
            pSyn->usSizeRange = MIBSRno; /* change to no size/range */
        }
        /* get size/range */
        if (pSyn->usSizeRange == MIBSRno)
            usSizeRange = pSym->ut.tc.rsyn.usSizeRange;

        /* continue check */
        usSyntax = pSym->ut.tc.rsyn.usSyntax;
    }


    /* check that size/range compatible with type */

    if ((usSizeRange != MIBSRno) &&
            (usSizeRange != MIBSRbad)) {
        if (usSyntax == MIBSYNoctstr) {
            /* octet string - only size valid */
            if ((usSizeRange != MIBSRfs) &&
                    (usSizeRange != MIBSRvs)) {
                yyerror("An OCTET STRING must have SIZE, not range specified");
                pSyn->usSizeRange = MIBSRno; /* set to none */
                fOk = FALSE;
            }
        } else if (usSyntax == MIBSYNint) {
            /* integer - only range valid */
            if ((usSizeRange != MIBSRpp) &&
                    (usSizeRange != MIBSRnp) &&
                    (usSizeRange != MIBSRnn)) {
                yyerror("An INTEGER must have range, not SIZE specified");
                pSyn->usSizeRange = MIBSRno; /* set to none */
                fOk = FALSE;
            }
        } else if (usSyntax == MIBSYNopaque) {
            /* an OPAQUE - only size valid */
            if ((usSizeRange != MIBSRfs) &&
                    (usSizeRange != MIBSRvs)) {
                yyerror("An Opaque must have SIZE, not range specified");
                pSyn->usSizeRange = MIBSRno; /* set to none */
                fOk = FALSE;
            }
        } else {
            /* all other types, size/range not allowed */
            yyerror("Given syntax type may not have SIZE or range specified");
            pSyn->usSizeRange = MIBSRno; /* set to none */
            fOk = FALSE;
        }
    }
    

    return(fOk);

} /* checkSyntax */


/** bstrToVal - convert binary string to integer
*
* NOTE: this is a special conversion for size/ranges
*
* call with:
*   pVal - value
*
* returns:
*   value
*/
    ULONG
#ifdef __STDC__
bstrToVal(STRTAB *pVal)
#else
bstrToVal(pVal)
    STRTAB *pVal;
#endif /* __STDC__ */
{
    USHORT i;
    ULONG ul;
    PSZ psz;

    if (*(pVal->pszVal) == 0) {
        yyerror("size/range should not be expressed as zero length binary string");
        return(0L);
    }

    for (ul = 0L, psz = pVal->pszVal; *psz == '0'; psz++) {
        /* skip leading zeros */
    }

    for (i = 0; *psz != 0; psz++) {
        if (i++ == 32) {
            yyerror("Binary value for size/range is too long");
            return(0L);
        }
        ul = (ul << 1) + ((*psz == '1') ? 1L : 0L);
    }
#ifdef OLD
    yywarning("size/range should not be expressed as binary string");
#endif

    return(ul);

} /* bstrToVal */


/** hstrToVal - convert hex string to integer
*
* NOTE: this is a special conversion for size/ranges
*
* call with:
*   pVal - value
*
* returns:
*   value
*/
    ULONG
#ifdef __STDC__
hstrToVal(STRTAB *pVal)
#else
hstrToVal(pVal)
    STRTAB *pVal;
#endif /* __STDC__ */
{
    USHORT i;
    ULONG ul;
    PSZ psz;
    USHORT us;


    if (*(pVal->pszVal) == 0) {
        yyerror("size/range should not be expressed as zero length hex string");
        return(0L);
    }

    for (ul = 0L, psz = pVal->pszVal; *psz == '0'; psz++) {
        /* skip leading zeros */
    }

    for (i = 0; *psz != 0; psz++) {
        if (i++ == 8) {
            yyerror("Hex value for size/range is too long");
            return(0L);
        }
        if ((*psz >= '0') && (*psz <= '9')) {
            us = *psz - '0';
        } else if ((*psz >= 'A') && (*psz <= 'F')) {
            us = *psz - 'A' + 10;
        } else {
            us = *psz - 'a' + 10;
        }
        ul = (ul << 4) + (ULONG)us;
    }
#ifdef OLD
    yywarning("size/range should not be expressed as hex string");
#endif

    return(ul);

} /* bstrToVal */


/* end of file: SMOT.C */
