#ifndef DOUBLEVEC_H
#define DOUBLEVEC_H
#pragma once

/*
 *	Declarations for Double Precision Vectors
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)DoubleVec.h	2.2	9/18/89
 */

/*	This code is designed to be as compatible as possible with the
 *	NIH Vector classes, while preserving efficiency.  These Vectors
 *	are NOT based on the NIH "Object" class, making them much
 *	smaller.  They also implement reference counting, making them
 *	faster. 
 */


#include "vdefs.h"
#include <math.h>

class istream;
class ostream;
class DoubleVec;

class DoubleBlock {
  unsigned short  	refs;		// Number of references
  unsigned  		npts;		// Number of elements
  double    		array[1];	// The data
  friend		DoubleVec;
public:
  DoubleBlock(unsigned n);
  DoubleBlock(unsigned n, double val);
  DoubleBlock(unsigned n, double val, double by);
  ~DoubleBlock();

  void			add_reference()	{refs++;}
  unsigned		references()	{return refs;}
  double*		data()		{return array;}
};

class DoubleVec {
  DoubleBlock*		block;
  double*		begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine; // For printing
  DoubleVec(DoubleVec&, int, unsigned, int); // For slices
protected:
  void			boundsErr(int);
  void			boundsCheck(int);
  void			lengthErr(int);
  void			lengthCheck(int i)	{if(npts!=i) lengthErr(i);}
  void			emptyErr(const char* fname);
  void			sliceErr(unsigned, int, unsigned, int);
public:
  DoubleVec();
  DoubleVec(unsigned n);
  DoubleVec(unsigned n, double val);
  DoubleVec(unsigned n, double val, double by);
  DoubleVec(const DoubleVec& a);
  DoubleVec(const double* dat, unsigned n);  // Copy of dat will be made
  ~DoubleVec();
  
  DoubleVec		slice(int start, unsigned lgt, int strider=1);
  
  double*		data()		{return begin;}
  unsigned		length()	{return npts;}
  int			stride()	{return step;}

  DoubleVec&		reference(DoubleVec& v);	// Reference self to v
  DoubleVec		deepCopy();	// copy of self with distinct instance variables 
  DoubleVec		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Insures only 1 reference to data
  void			resize(unsigned); 	// Will pad with zeroes if necessary

  void			scanFrom(istream& s); // Read to eof or delimit with []
  void			printOn(ostream& s);  // Pretty print
  void			setFormatting(int);   // Change # items per line
  void			readFrom(istream&);   // Internal ASCII formatting
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);// Internal binary formatting
  void			storeOn(fileDescTy&);

  // Indexing:
  double&		operator[](int i);	// With bounds checking
  double&		operator()(int i);	// With optional bounds checking
  
  // Assignment:
  DoubleVec&		operator=(const DoubleVec& v); // Must be same length as v
  DoubleVec&		operator=(double);
  
  // Arithmetic operators:
  DoubleVec&		operator++();
  DoubleVec&		operator--();
  DoubleVec&		operator+=(const DoubleVec&);
  DoubleVec&		operator+=(double);
  DoubleVec&		operator-=(const DoubleVec&);
  DoubleVec&		operator-=(double);
  DoubleVec&		operator*=(const DoubleVec&);
  DoubleVec&		operator*=(double);
  DoubleVec&		operator/=(const DoubleVec&);
  DoubleVec&		operator/=(double);
  
  // Friendly arithmetic operators:
  friend DoubleVec	operator-(const DoubleVec&);
  friend DoubleVec	operator+(const DoubleVec&);
  friend DoubleVec	operator*(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator/(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator+(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator-(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator*(const DoubleVec&,double);
  friend DoubleVec	operator*(double,const DoubleVec&);
  friend DoubleVec	operator/(const DoubleVec&,double);
  friend DoubleVec	operator/(double,const DoubleVec&);
  friend DoubleVec	operator+(const DoubleVec&,double);
  friend DoubleVec	operator+(double,const DoubleVec&);
  friend DoubleVec	operator-(const DoubleVec&,double);
  friend DoubleVec	operator-(double,const DoubleVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  // Math functions:
  DoubleVec	apply(mathFunTy);
  friend	DoubleVec	abs(const DoubleVec&);
  friend	DoubleVec	acos(const DoubleVec&);
  friend	DoubleVec	asin(const DoubleVec&);
  friend	DoubleVec	atan(const DoubleVec&);
  friend	DoubleVec	atan2(const DoubleVec&,const DoubleVec&);
  friend	DoubleVec	ceil(const DoubleVec&);
  friend	DoubleVec	cos(const DoubleVec&);
  friend	DoubleVec	cosh(const DoubleVec&);
  friend	DoubleVec	cumsum(const DoubleVec&);
  friend	DoubleVec	delta(const DoubleVec&);
  friend	double		dot(const DoubleVec&,const DoubleVec&);
  friend	DoubleVec	exp(const DoubleVec&); 
  friend	DoubleVec	floor(const DoubleVec&);
  friend	DoubleVec	log(const DoubleVec&);
  friend	int		max(const DoubleVec&);
  friend	int		min(const DoubleVec&);
  friend	double		mean(const DoubleVec&);
  friend	double		prod(const DoubleVec&);
  friend	DoubleVec	pow(const DoubleVec&,const DoubleVec&);
  friend	DoubleVec	reverse(const DoubleVec&);
  friend	DoubleVec	rint(const DoubleVec&);
  friend	DoubleVec	sin(const DoubleVec&);
  friend	DoubleVec	sinh(const DoubleVec&);
  friend	DoubleVec	sqrt(const DoubleVec&);
  friend	double		sum(const DoubleVec&);
  friend	DoubleVec	tan(const DoubleVec&);
  friend	DoubleVec	tanh(const DoubleVec&);
  friend	double		variance(const DoubleVec&);
#endif
  
};

// Other (related) declarations:
DoubleVec	expandEven(const DoubleVec&);
DoubleVec	expandOdd(const DoubleVec&);
ostream&	operator<<(ostream&, const DoubleVec&);
istream&	operator>>(istream&, DoubleVec&);

/******************* I N L I N E S **************************/

Inline void	DoubleVec::setFormatting(int i){numberPerLine = i;}

Inline void	DoubleVec::boundsCheck(int i){
  if(i<0 || i>npts) boundsErr(i);
}
Inline double&	DoubleVec::operator[](int i){
  boundsCheck(i); return begin[i*step];
}
Inline double&	DoubleVec::operator()(int i) {
#if BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

Inline DoubleVec	operator+(const DoubleVec& a)		{return a;}
Inline DoubleVec	operator*(double a, const DoubleVec& b)	{return b*a;}
Inline DoubleVec	operator+(double a, const DoubleVec& b)	{return b+a;}

#ifndef NO_VECTOR_MATHFUN
Inline DoubleVec acos(const DoubleVec& V)	{ return V.apply(::acos); }
Inline DoubleVec asin(const DoubleVec& V)	{ return V.apply(::asin); }
Inline DoubleVec atan(const DoubleVec& V)	{ return V.apply(::atan); }
Inline DoubleVec ceil(const DoubleVec& V)	{ return V.apply(::ceil); }
Inline DoubleVec cos(const DoubleVec& V)	{ return V.apply(::cos); }
Inline DoubleVec cosh(const DoubleVec& V)	{ return V.apply(::cosh); }
Inline DoubleVec exp(const DoubleVec& V)	{ return V.apply(::exp); }
Inline DoubleVec floor(const DoubleVec& V )  	{ return V.apply(::floor); }
Inline DoubleVec log(const DoubleVec& V)	{ return V.apply(::log); }
Inline DoubleVec rint(const DoubleVec& V)	{ return V.apply(::rint); }
Inline DoubleVec sin(const DoubleVec& V)	{ return V.apply(::sin); }
Inline DoubleVec sinh(const DoubleVec& V)	{ return V.apply(::sinh); }
Inline DoubleVec sqrt(const DoubleVec& V)	{ return V.apply(::sqrt); }
Inline DoubleVec tan(const DoubleVec& V)	{ return V.apply(::tan); }
Inline DoubleVec tanh(const DoubleVec& V)	{ return V.apply(::tanh); }
Inline double mean(const DoubleVec& V)		{ return sum(V)/V.length(); }
#endif
 
#endif
