/*
 *	Definitions for <T>Vec I/O
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)xvecio.cc	2.2	9/18/89
 */

#define NO_VECTOR_MATHFUN
#include "rw/<T>Vec.h"
#include <stream.h>
#include <ctype.h>

#define DEFAULT_RESIZE 128
#define DEFAULT_PERLINE 5

static const char SCCSid[] = "@(#)xvecio.cc	2.2 9/18/89";

static int default_resize = DEFAULT_RESIZE;
static char typeID[] = "<T>Vec2.2";

void
<T>Vec::boundsErr(int i)
{
  char msg[120];
  sprintf(msg, "Index (%d) out of range [0->%d].", i, int(length()-1));
  RWnote("<T>Vec::operator[]", msg);
  RWerror(DEFAULT);
}

void
<T>Vec::lengthErr(int i)
{
  char msg[120];
  sprintf(msg, "Lengths do not match: %d versus %d", length(), i);
  RWnote("<T>Vec::", msg);
  RWerror(DEFAULT);
}

void
<T>Vec::sliceErr(unsigned vecmax, int pos, unsigned n, int s)
{
  char msg[120];
  sprintf(msg,
	  "slice(%d, %u, %d) cannot be constructed from vector %u elements long.",
	  pos, n, s, vecmax);
  RWnote("<T>Vec::slice()", msg);
  RWerror(DEFAULT);
}

void
<T>Vec::emptyErr(const char* fname)
{
  cerr << "** <T>Vec::\n";
  RWnote(fname, "Vector empty.");
  RWerror(DEFAULT);
}

void
<T>Vec::scanFrom(istream& s)
{
  <T> item;
  register int nextspace = 0;
  char c = 0;		// The first character read from the stream

  s >> c;
  while(isspace(c)) s >> c;
  if (c != '[') {
    // Scan input stream, resizing as necessary.
    // Keep scanning till we can't scan no more
    s.putback(c);
    while(s >> item){
      if(nextspace >= length())resize(length()+default_resize);
      (*this)(nextspace++) = item;
    }
  }
  else {  // Scan input stream, stop scanning at the matching ']' character
    s >> c;
    while( s && (c != ']') ) {
      s.putback(c);
      if (s >> item) {
        if(nextspace >= length())resize(length()+default_resize);
        (*this)(nextspace++) = item;
      }
      s >> c;
    }
  }
  // Trim to fit
  if(nextspace != length())resize(nextspace);
}

void
<T>Vec::printOn(ostream& s)
{
  if(numberPerLine<=0)numberPerLine = DEFAULT_PERLINE;
  for(register int n = 0; n < length(); n++){
    if (n) { if (n%numberPerLine) {s<<" ";} else {s<<"\n";} }
    <T> item = (*this)(n);	// Can't use s<<(*this)(n) due to cfront bug
    s << item;
  }
}

ostream&
operator<<(ostream& s, const <T>Vec& v)
{
  v.printOn(s); return s;
}
istream&
operator>>(istream& s, <T>Vec& v)
{
  v.scanFrom(s); return s;
}

void
<T>Vec::readFrom(istream& s)
{
  RWreadID(s,typeID,"<T>Vec::readFrom");
  int len;
  s >> len;
  resize(len);
  for(register int n=0; n<len; n++)
    s >> (*this)[n];  //Use [] to guarantee bounds check in case resize fails
}

void
<T>Vec::storeOn(ostream& s)
{
  RWstoreID(s,typeID,"<T>Vec::storeOn");
  s << length() << "\n";
  for(register int n = 0; n < length(); n++){
    if (n) { if (n%5) {s<<" ";} else {s<<"\n";} }
    s << (*this)(n);
  }
  s << "\n";
}

void
<T>Vec::readFrom(fileDescTy& fd)
{
  static const char routine[] = "<T>Vec::readFrom";
  RWreadID(fd,typeID,routine);
  unsigned len;
  RWreadBin( fd, &len, sizeof(unsigned), routine );
  resize(len);
  RWreadBin( fd, data(), len*sizeof(<T>), routine );
}

void
<T>Vec::storeOn(fileDescTy& fd)
{
  static const char routine[] = "<T>Vec::storeOn";
  RWstoreID(fd,typeID,routine);
  unsigned len = length();
  RWstoreBin( fd, &len, sizeof(unsigned), routine );
  RWstoreBin( fd, data(), len*sizeof(<T>), routine );
}
