
#ifndef __MCOMM
#define __MCOMM

#include <sys/types.h>
#if defined(intelnx)
struct	iovec {
	void    *iov_base;
	int	iov_len;
	};
#else
#include <sys/uio.h>
#endif
/* 
   An initial attempt at this defined separate routines for each of the
   different data arrangements.  After discussions within the MPI group,
   I've decided to change this to a single, extensible routine that handles
   all non-contiguous data arrangements.

   These include
   readv/writev-like
   non-unit-stride
   indexed (scatter/gather)

   In addition, we add features for heterogeneity by including the datatype.

   
   In-order to be compatible with readv/writev definitions, the
   format for the send/receive routines is:
 
   SENDV....(type,bvec,bcnt,to,bdatatype[,id])
   RECVV....(type,bvec,bcnt,bdatatype[,id])

   where bvec is a pointer to structure iovec, usually defined as

   struct iovec { caddr_t iov_base; int iov_len; }

   bdatatype is simply a vector of datatypes; if NULL, MSG_OTHER is used.

   In fact, use instead:
        bvec = MSGALLOCBVEC(maxcnt)
	MSGBVECSET(bvec,i,pointer,nbytes,datatype)
   then
   SENDV...(type,bvec,to[,id])
   RECVV...(type,bvec[,id])
   and finally
        MSGBVECFREE(bvec)

   This allows any structure to be used for this purpose, including the
   Unix one of iovec + a separate data array, or anything else.
   For a vector of structures, 

   For vector ops, we have (ala CM-5)
   SENDVEC...(type,buf,elmlen,stride,elmcount,to,datatype)
   RECVVEC...(type,buf,elmlen,stride,elmcout,datatype)

   For scatter/gather, we have

   SENDI...(type,buf,ibuf,elmlen,elmcount,to,datatype)
   RECVI...(type,buf,ibuf,elmlen,maxelmcount,datatype)

   We'd ALSO like to be able to send arrays of structures.  For this,
   we need a description of the structure and the datatypes.  An easy way
   to do this may be to use the V version, but provide a repeat count
   and a stride (just sizeof(structure) for a contiguous array).
 */

/* Async message structures */
typedef struct {
     ASYNCSendId_t id;
     char       *buf;
     } PIAsyncSendMsg_t;
typedef struct {
     ASYNCRecvId_t id;
     char       *buf;
     } PIAsyncRecvMsg_t;

/* Default implementation: Block sends */
typedef struct {
    int          maxcnt, cnt;
    struct iovec *BV;
    int          *datatype;
    } MSGBVEC;

extern MSGBVEC *PIGetBvec();
extern void    PIFreeBvec();

#if !defined(MSG_HAS_BVEC)
#define MSGALLOCBVEC(maxcnt) PIGetBvec( maxcnt )
#define MSGFREEBVEC(bvec)    PIFreeBvec( bvec )
#define MSGBVECSET(bvec,i,pointer,nbytes,dtype) {\
    if ((bvec)->maxcnt >= i) {SETERRC(1,"Exceeded size of MSGBVEC");}\
    else {(bvec)->BV[i].iov_base = pointer;\
          (bvec)->BV[i].iov_len  = nbytes;\
          (bvec)->datatype[i]    = dtype;\
          if ((bvec)->cnt < i) (bvec)->cnt = i;}}
#define SENDVSYNC(type,bvec,to)     PISendVSync(type,bvec,to,0)
#define RECVVSYNC(type,bvec)        PIRecvVSync(type,bvec,0)
#define SENDVASYNC(type,bvec,to,id) PISendVASync(type,bvec,to,&(id),0)
#define RECVVASYNC(type,bvec,id)    PIRecvVASync(type,bvec,&(id),0)
#define SENDVWAIT(type,bvec,to,id)  PISendVWait(type,bvec,to,&(id),0)
#define RECVVWAIT(type,bvec,id)     PIRecvVWait(type,bvec,&(id),0)

#define SENDVSYNCFORCE(type,bvec,to)     PISendVSync(type,bvec,to,1)
#define RECVVSYNCFORCE(type,bvec)        PIRecvVSync(type,bvec,1)
#define SENDVASYNCFORCE(type,bvec,to,id) PISendVASync(type,bvec,to,&(id),1)
#define RECVVASYNCFORCE(type,bvec,id)    PIRecvVASync(type,bvec,&(id),1)
#define SENDVWAITFORCE(type,bvec,to,id)  PISendVWait(type,bvec,to,&(id),1)
#define RECVVWAITFORCE(type,bvec,id)     PIRecvVWait(type,bvec,&(id),1)
#endif

/* Default implementation: Vector sends.
   These are done entirely by the support routines; there are no
   local datastructures 
 */
#if !defined(MSG_HAS_VECTORS)
#define SENDVECSYNC(type,buf,elmlen,stride,elmcount,to,datatype) \
          PISendVecSync(type,buf,elmlen,stride,elmcount,to,datatype,0) 
#define RECVVECSYNC(type,buf,elmlen,stride,elmcount,datatype) \
          PIRecvVecSync(type,buf,elmlen,stride,elmcount,datatype,0) 
#define SENDVECASYNC(type,buf,elmlen,stride,elmcount,to,datatype,id) \
          PISendVecASync(type,buf,elmlen,stride,elmcount,to,datatype,&(id),0) 
#define RECVVECASYNC(type,buf,elmlen,stride,elmcount,datatype,id) \
          PIRecvVecASync(type,buf,elmlen,stride,elmcount,datatype,&(id),0) 
#define SENDVECWAIT(type,buf,elmlen,stride,elmcount,to,datatype,id) \
          PIASendVecWait(type,buf,elmlen,stride,elmcount,to,datatype,&(id),0) 
#define RECVVECWAIT(type,buf,elmlen,stride,elmcount,datatype,id) \
          PIRecvVecWait(type,buf,elmlen,stride,elmcount,datatype,&(id),0) 

#define SENDVECSYNCFORCE(type,buf,elmlen,stride,elmcount,to,datatype) \
        PISendVecSync(type,buf,elmlen,stride,elmcount,to,datatype,1) 
#define RECVVECSYNCFORCE(type,buf,elmlen,stride,elmcount,datatype) \
        PIRecvVecSync(type,buf,elmlen,stride,elmcount,datatype,1) 
#define SENDVECASYNCFORCE(type,buf,elmlen,stride,elmcount,to,datatype,id) \
        PISendVecASync(type,buf,elmlen,stride,elmcount,to,datatype,&(id),1) 
#define RECVVECASYNCFORCE(type,buf,elmlen,stride,elmcount,datatype,id) \
        PIRecvVecASync(type,buf,elmlen,stride,elmcount,datatype,&(id),1) 
#define SENDVECWAITFORCE(type,buf,elmlen,stride,elmcount,to,datatype,id) \
        PIASendVecWait(type,buf,elmlen,stride,elmcount,to,datatype,&(id),1) 
#define RECVVECWAITFORCE(type,buf,elmlen,stride,elmcount,datatype,id) \
        PIRecvVecWait(type,buf,elmlen,stride,elmcount,datatype,&(id),1) 
#endif

/* 
   Default implementation for indexed ops 
   (These still need work)
 */
#if !defined(MSG_HAS_INDEXED)
#define SENDISYNC(type,buf,bufi,elmlen,elmcount,to,datatype) \
        PISendISync(type,buf,bufi,elmlen,elmcount,to,datatype,0)
#define RECVISYNC(type,buf,bufi,elmlen,maxelms,datatype) \
        PIRecvISync(type,buf,bufi,elmlen,maxelms,datatype,0)
#define SENDISYNCFORCE(type,buf,bufi,elmlen,elmcount,to,datatype) \
        PISendISync(type,buf,bufi,elmlen,elmcount,to,datatype,1)
#define RECVISYNCFORCE(type,buf,bufi,elmlen,maxelms,datatype) \
        PIRecvISync(type,buf,bufi,elmlen,maxelms,datatype,1)
#define SENDIASYNC(type,buf,bufi,elmlen,elmcount,to,datatype,id) \
        PISendIAsync(type,buf,bufi,elmlen,elmcount,to,datatype,&(id),0)
#define RECVIASYNC(type,buf,bufi,elmlen,maxelms,datatype,id) \
        PIRecvIAsync(type,buf,bufi,elmlen,maxelms,datatype,&(id),0)
#define SENDIASYNCFORCE(type,buf,bufi,elmlen,elmcount,to,datatype,id) \
        PISendIAsync(type,buf,bufi,elmlen,elmcount,to,datatype,&(id),1)
#define RECVIASYNCFORCE(type,buf,bufi,elmlen,maxelms,datatype,id) \
        PIRecvIAsync(type,buf,bufi,elmlen,maxelms,datatype,&(id),1)
#define SENDIWAIT(type,buf,bufi,elmlen,elmcount,to,datatype,id) \
        PISendIWait(type,buf,bufi,elmlen,elmcount,to,datatype,&(id),0)
#define RECVIWAIT(type,buf,bufi,elmlen,maxelms,datatype,id) \
        PIRecvIWait(type,buf,bufi,elmlen,maxelms,datatype,&(id),0)
#define SENDIWAITFORCE(type,buf,bufi,elmlen,elmcount,to,datatype,id) \
        PISendIWait(type,buf,bufi,elmlen,elmcount,to,datatype,&(id),1)
#define RECVIWAITFORCE(type,buf,bufi,elmlen,maxelms,datatype,id) \
        PIRecvIWait(type,buf,bufi,elmlen,maxelms,datatype,&(id),1)
#endif

/*
   We also need to have special routines for the probes
 */
#define ASYNCDONEVEC(id)
#define ASYNCDONEI(id)
#define ASYNCDONEV(id)

/* Macro man pages */
/* Block send/receive */
/* MC
     MSGALLOCBVEC - Create a bvec structure for general send/receives

     Input Parameter:
.    maxcnt - Maximum number of blocks
     
     Description:
     Allocate a MSGBVEC for maxcnt blocks

     Synopsis:
     MSGBVEC *MSGALLOCBVEC(maxcnt)
     int maxcnt;
M*/

/* MC
     MSGFREEBVEC - Free a bvec structure created with MSGALLOCBVEC

     Input Parameter:
.    bvec - MSGBVEC to free

     Synopsis:
     void MSGFREEBVEC( bvec )
     MSGBVEC *bvec;

M*/

/* MC
     MSGBVECSET - Set a element of a bvector

     Input Parameters:
.    bvec - bvector to set
.    i    - component of bvec (set bvec[i])
.    pointer - pointer value
.    nbytes  - number of bytes that pointer points to
.    dtype   - data type

     Synopsis:
     MSGBVECSET( bvec, i, pointer, nbytes, dtype )
     MSGBVEC *bvec;
     int     i, nbytes, dtype;
     char    *pointer;
M*/

/* M
     SENDVSYNC - blocking send of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector
.    to   - destination processor

     Synopsis:
     void SENDVSYNC(type,bvec,to) 
     int     type, to;
     MSGBVEC *bvec;
M*/

/* M
     RECVVSYNC - locking receive of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector

     Synopsis:
     void RECVVSYNC(type,bvec) 
     int     type;
     MSGBVEC *bvec;
M*/

/* M
     SENDVASYNC - nonblocking send of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector
.    to   - destination processor
.    id   - id for operation (use in SENDVWAIT)

     Synopsis:
     void SENDVASYNC(type,bvec,to,id) 
     int              type, to;
     MSGBVEC          *bvec;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVASYNC - nonblocking receive of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector
.    it   - id for operation (use in RECVVWAIT)

     Synopsis:
     void RECVVASYNC(type,bvec,id) 
     int              type;
     MSGBVEC          *bvec;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVWAIT - blocking wait for data described by a bvector to be sent

     Input Parameters:
.    type - message type
.    bvec - bvector
.    to   - destination processor
.    id   - id for operation (used in SENDVASYNC)

     Synopsis:
     void SENDVWAIT(type,bvec,to,id) 
     int              type, to;
     MSGBVEC          *bvec;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVWAIT - blocking wait for data described by a bvector to be received

     Input Parameters:
.    type - message type
.    bvec - bvector
.    it   - id for operation (use in RECVVASYNC)

     Synopsis:
     void RECVVWAIT(type,bvec,id) 
     int              type;
     MSGBVEC          *bvec;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVSYNCFORCE - blocking send of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector
.    to   - destination processor

     Synopsis:
     void SENDVSYNCFORCE(type,bvec,to) 
     int     type, to;
     MSGBVEC *bvec;
M*/

/* M
     RECVVSYNCFORCE - locking receive of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector

     Synopsis:
     void RECVVSYNCFORCE(type,bvec) 
     int     type;
     MSGBVEC *bvec;
M*/

/* M
     SENDVASYNCFORCE - nonblocking send of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector
.    to   - destination processor
.    id   - id for operation (use in SENDVWAITFORCE)

     Synopsis:
     void SENDVASYNCFORCE(type,bvec,to,id) 
     int              type, to;
     MSGBVEC          *bvec;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVASYNCFORCE - nonblocking receive of data described by a bvector

     Input Parameters:
.    type - message type
.    bvec - bvector
.    it   - id for operation (use in RECVVWAITFORCE)

     Synopsis:
     void RECVVASYNCFORCE(type,bvec,id) 
     int              type;
     MSGBVEC          *bvec;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVWAITFORCE - blocking wait for data described by a bvector to be sent

     Input Parameters:
.    type - message type
.    bvec - bvector
.    to   - destination processor
.    id   - id for operation (used in SENDVASYNCFORCE)

     Synopsis:
     void SENDVWAITFORCE(type,bvec,to,id) 
     int              type, to;
     MSGBVEC          *bvec;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVWAITFORCE - blocking wait for data described by a bvector to be 
                      received

     Input Parameters:
.    type - message type
.    bvec - bvector
.    it   - id for operation (use in RECVVASYNCFORCE)

     Synopsis:
     void RECVVWAITFORCE(type,bvec,id) 
     int              type;
     MSGBVEC          *bvec;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVECSYNC - Blocking send of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to send
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to send, in bytes
.    elmcount - number of elements to send
.    to - destination processor
.    datatype - type of data

     Synopsis:
     void SENDVECSYNC(type,buf,elmlen,stride,elmcount,to,datatype)
     int type, elmlen, stride, elmcount, to, datatype;
     void *buf;
M*/

/* M
     RECVVECSYNC - Blocking receive of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to receive
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to receive, in bytes
.    elmcount - maximum number of elements to receive
.    datatype - type of data

     Synopsis:
     void RECVVECSYNC(type,buf,elmlen,stride,elmcount,datatype)
     int type, elmlen, stride, elmcount, datatype;
     void *buf;
M*/

/* M
     SENDVECASYNC - Nonblocking send of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to send
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to send, in bytes
.    elmcount - number of elements to send
.    to - destination processor
.    datatype - type of data
.    id       - id of message (use with SENDVECWAIT)

     Synopsis:
     void SENDVECASYNC(type,buf,elmlen,stride,elmcount,to,datatype,id)
     int              type, elmlen, stride, elmcount, to, datatype;
     void             *buf;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVECASYNC - Nonblocking receive of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to receive
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to receive, in bytes
.    elmcount - maximum number of elements to receive
.    datatype - type of data
.    id       - id of message (use with RECVVECWAIT)

     Synopsis:
     void RECVVECASYNC(type,buf,elmlen,stride,elmcount,datatype,id)
     int              type, elmlen, stride, elmcount, datatype;
     void             *buf;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVECWAIT - Blocking wait for send of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to send
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to send, in bytes
.    elmcount - number of elements to send
.    to - destination processor
.    datatype - type of data
.    id       - id of message (use with SENDVECASYNC)

     Synopsis:
     void SENDVECWAIT(type,buf,elmlen,stride,elmcount,to,datatype,id)
     int              type, elmlen, stride, elmcount, to, datatype;
     void             *buf;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVECWAIT - Blocking wait for 
                   receive of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to receive
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to receive, in bytes
.    elmcount - maximum number of elements to receive
.    datatype - type of data
.    id       - id of message (use with RECVVECASYNC)

     Synopsis:
     void RECVVECWAIT(type,buf,elmlen,stride,elmcount,datatype,id)
     int              type, elmlen, stride, elmcount, datatype;
     void             *buf;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVECSYNCFORCE - Blocking send of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to send
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to send, in bytes
.    elmcount - number of elements to send
.    to - destination processor
.    datatype - type of data

     Synopsis:
     void SENDVECSYNCFORCE(type,buf,elmlen,stride,elmcount,to,datatype)
     int type, elmlen, stride, elmcount, to, datatype;
     void *buf;
M*/

/* M
     RECVVECSYNCFORCE - Blocking receive of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to receive
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to receive, in bytes
.    elmcount - maximum number of elements to receive
.    datatype - type of data

     Synopsis:
     void RECVVECSYNCFORCE(type,buf,elmlen,stride,elmcount,datatype)
     int type, elmlen, stride, elmcount, datatype;
     void *buf;
M*/

/* M
     SENDVECASYNCFORCE - Nonblocking send of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to send
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to send, in bytes
.    elmcount - number of elements to send
.    to - destination processor
.    datatype - type of data
.    id       - id of message (use with SENDVECWAITFORCE)

     Synopsis:
     void SENDVECASYNCFORCE(type,buf,elmlen,stride,elmcount,to,datatype,id)
     int              type, elmlen, stride, elmcount, to, datatype;
     void             *buf;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVECASYNCFORCE - Nonblocking receive of data separated by constant 
                         strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to receive
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to receive, in bytes
.    elmcount - maximum number of elements to receive
.    datatype - type of data
.    id       - id of message (use with RECVVECWAITFORCE)

     Synopsis:
     void RECVVECASYNCFORCE(type,buf,elmlen,stride,elmcount,datatype,id)
     int              type, elmlen, stride, elmcount, datatype;
     void             *buf;
     PIAsyncRecvMsg_t id;
M*/

/* M
     SENDVECWAITFORCE - Blocking wait for send of data separated by 
                        constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to send
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to send, in bytes
.    elmcount - number of elements to send
.    to - destination processor
.    datatype - type of data
.    id       - id of message (use with SENDVECASYNCFORCE)

     Synopsis:
     void SENDVECWAITFORCE(type,buf,elmlen,stride,elmcount,to,datatype,id)
     int              type, elmlen, stride, elmcount, to, datatype;
     void             *buf;
     PIAsyncSendMsg_t id;
M*/

/* M
     RECVVECWAITFORCE - Blocking wait for 
                        receive of data separated by constant strides

     Input Parameters:
.    type - message type
.    buf  - points to first element to receive
.    elmlen - length of each element in bytes
.    stride - distance between successive elements to receive, in bytes
.    elmcount - maximum number of elements to receive
.    datatype - type of data
.    id       - id of message (use with RECVVECASYNCFORCE)

     Synopsis:
     void RECVVECWAITFORCE(type,buf,elmlen,stride,elmcount,datatype,id)
     int              type, elmlen, stride, elmcount, datatype;
     void             *buf;
     PIAsyncRecvMsg_t id;
M*/

#endif
