#ifndef lint
static char SCCSid[] = "@(#) ./set/iset.c 07/23/93";
#endif

#include "tools.h"
#include "set/iset.h"           /*I "set/iset.h" I*/

/*
    Routines for operations on sets, stored as SORTED indices.  We can
    often generate the sets in this way, either all at once, or with
    a separate sort step.

    This needs to change to include a more object-oriented framework and to
    include operations with the index sets (scatter and gather).  To start
    with, we should include a simple range or perhaps a multi-dimensional
    direct product.
 */

/*
   In many applications, there is a considerable amount of time spent 
   allocating and deallocating index sets.  We set up here a cache containing
   a single index set.

   We also cache a larger number of headers, since they are small and not
   efficiently allocated by MALLOC.
 */

static int  cacheEnabled = 0;
static ISet *cachedISet  = 0;

#define MAXHEADERS 20
static int  header=0;
static ISet *(cachedHeaders[MAXHEADERS]);

/*@
  ISEnableCache - Enables the ISet cache.

  Input Parameter:
. n - size of index set to cache (my be 0).

  Returns:
  1 if the cache was already enabled, 0 otherwise
@*/
int ISEnableCache( n )
int n;
{
int old = cacheEnabled;

if (!old) {
    cachedISet = ISAlloc( n );
    }
cacheEnabled = 1;
} 

/*@
  ISDisableCache - Disables the ISet cache and flushes the cache.
@*/
void ISDisableCache( )
{
cacheEnabled = 0;
if (cachedISet) 
    ISDestroy( cachedISet );
cachedISet = 0;
}

/*@
    ISAlloc - Allocate an index set of a given size

    Input Parameter:
.   n - maximum number of indices in the set    
@*/
ISet *ISAlloc( n )
int n;
{
ISet *is;

if (cacheEnabled && cachedISet && cachedISet->na >= n) {
    is         = cachedISet;
    cachedISet = 0;
    is->n      = 0;
    return is;
    }
if (header > 0) {
    is = cachedHeaders[--header];
    }
else {
    is      = NEW(ISet);                          CHKPTRN(is);
    }
is->idx = (int *)MALLOC( n * sizeof(int) );   CHKPTRN(is->idx);
is->na  = n;
is->n   = 0;

return is;
}

/*@
    ISCreateFromData - Create an index set from user data

    Input Parameters:
.   n   - number of elements in set
.   idx - (sorted) list of elements

    Returns:
    ISet structure
@*/
ISet *ISCreateFromData( n, idx )
int n, *idx;
{
ISet *is;

is      = NEW(ISet);                           CHKPTRN(is);
is->idx = idx;
is->na  = is->n = n;

return is;
}

/*@
    ISSetFromData - Modify an index set with user data.

    Input Parameters:
.   is  - Index set to modify
.   n   - number of elements in set
.   idx - (sorted) list of elements
@*/
void ISSetFromData( is, n, idx )
ISet *is;
int  n, *idx;
{
is->idx = idx;
is->na  = is->n = n;
}

/*@
    ISDestroy - Deallocate an index set

    Input Parameter:
.   is  - pointer to index set
@*/    
void ISDestroy( is )
ISet *is;
{
if (cacheEnabled) {
    if (!cachedISet) {
	cachedISet = is;
	return;
	}
    else if (cachedISet->na < is->na) {
	/* Free the old one and cache the larger one */
	ISet *itmp;
	itmp       = cachedISet;
	cachedISet = is;
	is         = itmp;
	/* fall through to delete the formerly cached ISet */
	}
    }
if (is->idx) {
    FREE( is->idx );
    }
if (header < MAXHEADERS) {
    cachedHeaders[header++] = is;
    }
else {
    FREE( is );
    }
}

/*@
    ISUnion - Form the union of two index sets

    Input Parameters:
.   a,b   - sets to union
.   c      - set to place union in (should be preallocated)
@*/
void ISUnion( a, b, c )
ISet *a, *b, *c;
{
int          n1, n2, i1, i2, i3, na3;
register int *a1, *b1, *c1;
register int ta, tb, bt;

a1 = a->idx;
n1 = a->n;
i1 = 0;
b1 = b->idx;
n2 = b->n;
i2 = 0;
c1 = c->idx;
i3 = 0;
na3= c->na;

ta = a1[0]; 
tb = b1[0];

if (n1 > 0 && n2 >= 0) {
    while (i3 < na3) {
	/* We could optimize for many common members by testing for equality
	   first; a while (ta == tb) ... (with an appropriate test for end of
	   sets) might be appropriate as well... */
	if (ta < tb) { 
	    c1[i3++] = ta; ta = a1[++i1]; if (i1 >= n1) break; }
	else if (ta > tb) { 
	    c1[i3++] = tb; tb = b1[++i2]; if (i2 >= n2) break; }
	else {
	    bt = 0;
	    while (ta == tb) {
		c1[i3++] = ta; tb = b1[++i2]; ta = a1[++i1]; 
		if (i1 >= n1 || i2 >= n2 || i3 >= na3) { bt = 1; break; }
		}
	    if (bt) break;
	    }
	}
    }
/* At most one of these is true */    
#ifdef FOO
if (i3 < na3) {
    c1 += i3;
    l3 = na3 - i3 + 1;
    if (i1 < n1) {
	l1 = n1 - i1 + 1;
	if (l1 > l3) l1 = l3;
	a1 += i1;
	for(i=0; i<l1; i++) c1[i] = a1[i];
	}
    else if (i2 < n2) {
	l1 = n2 - i2 + 1;
	if (l1 > l3) l1 = l3;
	b1 += i2;
	for(i=0; i<l1; i++) c1[i] = b1[i];
	}
    }
#endif	
while (i1 < n1 && i3 < na3) c1[i3++] = a1[i1++];
while (i2 < n2 && i3 < na3) c1[i3++] = b1[i2++];
c->n = i3;
}

/*@
    ISIntersection - Form the intersection of two index sets

    Input Parameters:
.   a,b   - sets to intersect
.   c      - set to place intersection in (should be preallocated)
@*/
void ISIntersection( a, b, c )
ISet *a, *b, *c;
{
int n1, n2, i1, i2, i3, na3;
register int *a1, *b1, *c1;
register int ta, tb;

a1 = a->idx;
n1 = a->n;
i1 = 0;
b1 = b->idx;
n2 = b->n;
i2 = 0;
c1 = c->idx;
i3 = 0;
na3= c->na;

ta = a1[0];
tb = b1[0];
if (n1 > 0 && n2 > 0) {
    while (1) {
	if (ta < tb)      { ta = a1[++i1]; if (i1 >= n1) break; }
	else if (ta > tb) { tb = b1[++i2]; if (i2 >= n2) break; }
	else {
	    c1[i3++] = ta; ta = a1[++i1]; tb = b1[++i2];
	    if (i1 >= n1 || i2 >= n2 || i3 >= na3) break;
	    }
	}
    }
c->n = i3;
}

/*@
    ISIntersectionNotNull - Without forming the intersection, return whether 
    the two sets intersect.

    Input Parameters:
.   a,b   - sets to intersect

    Returns:
    1 if the sets intersect, 0 otherwise.
@*/
int ISIntersectionNotNull( a, b )
ISet *a, *b;
{
int n1, n2, i1, i2, ta, tb;
int *a1, *b1;

a1 = a->idx;
n1 = a->n;
i1 = 0;
b1 = b->idx;
n2 = b->n;
i2 = 0;

ta = a1[0];
tb = b1[0];

/* A special case tests for non-overlapping sets */
if (ta > b->idx[n2-1] || tb > a->idx[n1-1]) return 0;

while (i1 < n1 && i2 < n2) {
    if (ta < tb)      ta = a1[++i1];
    else if (ta > tb) tb = b1[++i2];
    else {
	return 1;
        }
    }
return 0;
}

/*@
    ISCompIntersection - Intersect one set with the complement of another 

    Input Parameters:
.   a,b   - for a intersection complement(b)
.   c      - set to place result in (must be preallocated)
@*/
void ISCompIntersection( a, b, c )
ISet *a, *b, *c;
{
int n1, n2, i1, i2, i3, na3, ta, tb;
int *a1, *b1, *c1;

a1 = a->idx;
n1 = a->n;
i1 = 0;
b1 = b->idx;
n2 = b->n;
i2 = 0;
c1 = c->idx;
i3 = 0;
na3= c->na;

ta = a1[0];
tb = b1[0];
while (i1 < n1 && i2 < n2 && i3 < na3) {
    if (ta < tb)      { c1[i3++] = ta; ta = a1[++i1]; }
    else if (ta > tb)  tb = b1[++i2];
    else {
    	ta = a1[++i1]; tb = b1[++i2];
        }
    }
while (i1 < n1 && i3 < na3) c1[i3++] = a1[i1++];
c->n = i3;
}

/*@
    ISCompress - Compress the storage for an index set

    Input Parameter:
.   is - set to compress
@*/    
void ISCompress( is )
ISet *is;
{
int i, n, *np, *op;

n  = is->n;
op = is->idx;
np = (int *)MALLOC( n * sizeof(int) );

for(i=0; i<n; i++) np[i] = op[i];
FREE( op );
is->idx = np;
is->na  = n;
}

/* Return 1 if i is a contiguous index set, 0 otherwise */
int ISiIsContig( n, i )
register int n, *i;
{
register int j;

for (j=1; j<n; j++) 
    if (i[j] != i[j-1] + 1) return 0;
return 1;
}


#include <stdio.h>
/*
   Internal routine to print out an index set.

   We try to catch the following cases:
       i i+1 ... i+n       (write as i-i+n)
   and
       j j j .. j k        (write as p*j)
 */
void ISiPrint( fp, idx, n )
FILE *fp;
int  *idx, n;
{
int *cp, val, j, k;

if (fp == 0) fp = stdout;

j = 0;
while (j < n) {
    val = idx[j];
    k   = 1;
    /* Find runs of a single value */
    while (j + k < n && idx[j+k] == val) k++;
    if (k > 1) {
	fprintf( fp, "%d*%d%c", k-1, val, (j + k >= n) ? '\n' : ',' );
	j += k;
	if (j >= n) break;
	continue;
	}
    k   = 0;
    val = idx[j];
    /* Find runs of consequtive values */
    while (j + k < n && idx[j+k] == val + k) k++;
    if (k > 1) {
	fprintf( fp, "%d-%d%c", val, val + k - 1, (j + k >= n) ? '\n' : ',' );
	j  += k;
	if (j >= n) break;
	continue;
	}
    val = idx[j++];
    fprintf( fp, "%d%c", val, (j == n) ? '\n' : ',' );
    }

}
