#ifndef lint
static char SCCSid[] = "@(#) ./sparse/aij/aijcret.c 07/23/93";
#endif

/*
   This file contains routines for representing and manipulating matrices
   in aij format.  
   This is RAW aij format; all column indices are 1, rather than 0, origin.
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"

extern void SpAIJMult();
void   SpAIJDestroy();
SpMat  *SpAIJCreate();
SpMatSplit *SpAIJCreateSplit();
int    SpAIJFactor();
void   SpAIJSolve();
void   SpAIJScatterFromRow(), SpAIJGatherToRow();
static SpOps _SpAIJOps = { SpAIJMult, 0, 0,
			   SpAIJSolve, 0, 
			   0, 0, 
			   SpAIJDestroy, 
			   (void *(*)())SpAIJCreate, 
			   (void *(*)())SpAIJCreateSplit, 
			   0, 0, SpAIJFactor, 
			   0, 0, 
			   SpAIJGatherToRow, 0, SpAIJScatterFromRow, 0, 
			   0, 
			   0, 1, 1 };

/*
   SpAIJMult - Form the matrix-vector product vout = mat * vin.

   Input Parameters:
.  nmat  - matrix (in diagonal format)
.  vin   - vector to multiply

   Output Parameters:
.  vout - result vector
*/
void SpAIJMult( nmat, vin, vout )
SpMat  *nmat;
double *vin, *vout;
{
SpAIJ    *mat = (SpAIJ *)nmat->data;
int      *ia = mat->ia, *ja = mat->ja, row;
double   *a = mat->a;
int      nr = nmat->rows;
register double sum, *p, *vvin = vin;
register int    j, nz, *c;

/* Shift the origins */
vin -= 1;
ja  -= 1;
a   -= 1;
for (row=0; row<nr; row++) {
    c   = ja + ia[row];
    p   = a + ia[row];
    nz  = ia[row+1] - ia[row];
    sum = 0.0;
#ifdef FOO
    SPARSEDENSEDOT(sum,vvin,p,c,nz);
#else
    for (j=0; j<nz; j++) {
	sum += p[j] * vin[c[j]];
	}
#endif
    vout[row] = sum;
    }
}

/*@
   SpAIJCreateFromData - Creates an AIJ matrix descriptor, given an existing
   AIJ format matrix.

   Input Parameters:
.  nr - number of rows
.  nc - number of columns
.  ia - row pointers
.  ja - column values
.  a  - entries
.  maxnz - maximum number of non-zeros (allocated size of ja and a)
 @*/
SpMat *SpAIJCreateFromData( nr, nc, ia, ja, a, maxnz )
int    nr, nc, *ia, *ja, maxnz;
double *a;
{
SpAIJ *mat;
SpMat *nmat;

mat         = NEW(SpAIJ);      CHKPTRV(mat,0);
mat->ia     = ia;
mat->ja     = ja;
mat->a      = a;
mat->dloc   = 0;
mat->quser  = 1;
mat->maxnz  = maxnz;
mat->maxrow = nr;

/* Allocate and initialize the SpMat structure (there should be a separate
   routine for this) */
nmat                 = NEW(SpMat);         CHKPTRV(nmat,0);
nmat->type           = MATAIJ;
nmat->ops            = &_SpAIJOps;
nmat->rows           = nr;
nmat->cols           = nc;
nmat->is_sorted      = 0;
nmat->element_length = 8;
nmat->nz             = ia[nr] - ia[0];
nmat->pool.pool.ptr  = 0;
nmat->pool.pool.n    = 0;
nmat->pool.pool.next = 0;
nmat->pool.alloc     = 0;
nmat->pool.free      = 0;
nmat->alloc_together = 0;
nmat->alloc_incr     = 0;
nmat->map            = 0;
nmat->data           = (void *)mat;

return nmat;
}

/*@
   SpAIJCreate - Creates an AIJ matrix descriptor.

   Input Parameters:
.  nr - number of rows
.  nc - number of columns
.  maxnz - maximum number of non-zeros
 @*/
SpMat *SpAIJCreate( nr, nc, maxnz )
int   nr, nc, maxnz;
{
SpMat      *nmat;
int        *ia, *ja, k;
double     *a;

ia = 0;
ja = 0;
a  = 0;
if (maxnz > 0) {
    ia = (int *)MALLOC( (nr + 1) * sizeof(int) );    CHKPTRN(ia);
    for (k=0; k<nr+1; k++)
	ia[k] = 1;
    ja = (int *)MALLOC( maxnz * sizeof(int) );       CHKPTRN(ja);
    a  = (double *)MALLOC( maxnz * sizeof(double) ); CHKPTRN(a);
    }

nmat = SpAIJCreateFromData( nr, nc, ia, ja, a, maxnz );
CHKERRV(1,0);
((SpAIJ *)(nmat->data))->quser = 0;
return nmat;
}

/*
  SpAIJDestroy - Free storage for a AIJ matrix 

  Input Parameter:
. nmat - AIJ format matrix to free
 */
void SpAIJDestroy( nmat )
SpMat     *nmat;
{
SpAIJ *mat = (SpAIJ *)nmat->data;

/* Free the storage if not user defined */
if (!mat->quser) {
    if (mat->ia) {
	FREE( mat->ia );
	FREE( mat->ja );
	FREE( mat->a );
	}
    }

FREE( mat );
FREE( nmat );
}

/* These are factor and solve routines; they should use YSMP */
void SpAIJSolve( m, b, x )
SpMatSplit *m;
double *b, *x;
{
}

#ifdef FOO
void SpAIJSolvePermBase( m, b, x, Tmp )
SpMatSplit *m;
double     *b, *x, *Tmp;
{
SpMat    *mat = m->factor;
SpAIJ    *dmat = (SpAIJ *)mat->data;
int      *ia = dmat->ia, *ja = dmat->ja, row;
double   *a = dmat->a, *p, sum;
int      nr = mat->rows;
register double sum, *p;
register int    j, nz, *c;
int      *r = mat->map->rowmap, *c = mat->map->colmap;

/* Shift the origins */
b   -= 1;
x   -= 1;
tmp -= 1;
ja  -= 1;
a   -= 1;
tmp[1] = b[*r++];
for (row=1; row<=nr; row++) {
    c  = ja + ia[row];
    p  = a + ia[row];
    nz = ia[row+1] - ia[row];
    sum = 0.0;
    for (j=0; j<nz; j++) {
	sum += vin[c[j]] * p[j];
	}
    vout[row] += sum;
    }

}
#endif 

int SpAIJFactor( m, mb )
SpMat      *m;
SpMatSplit *mb;
{
SpMat *mat = mb->factor;
SpAIJ *dmat = (SpAIJ *)mat->data;

return 0;
}

/* These are used to hold scattered values */
static int    *Jcols;
static int    Jsize = 0;

void SpAIJScatterFromRow( mat, row, nz, i, v )
SpMat  *mat;
int    *nz, **i, row;
double **v;
{
SpAIJ        *dmat = (SpAIJ *)mat->data;
register int *ia = dmat->ia;
register int *JC, j, *ja, Nz;

Nz = ia[row+1] - ia[row];
if (nz) *nz = Nz;
if (i) {
    if(Nz > Jsize) {
	if (Jsize > 0) {
	    FREE( Jcols );
	    }
	Jcols  = (int *)MALLOC( Nz * sizeof(int) );   CHKPTR(Jcols);
	Jsize  = Nz;
	}
    ja = dmat->ja;
    *i = Jcols;
    JC = Jcols;
    ja += ia[row] - 1;
    for (j=0; j<Nz; j++) 
	JC[j] = ja[j] - 1;
    }
if (v) {
    *v = dmat->a + ia[row] - 1;
    }
}

/* 
   This one is difficult, since we have to make room for the new values 
   For starters, we'll use the following algorithm:

   Expand the existing row
   Insert the elements
   Compute the new row size and insert.

   As a special case (only one implemented so far), if the elements
   are being added at the end of the matrix, and there is room, just
   insert them.
 */   
void SpAIJGatherToRow( mat, row, nz, i, v )
SpMat  *mat;
int    nz, *i, row;
double *v;
{
SpAIJ      *dmat = (SpAIJ *)mat->data;
double     *a    = dmat->a;
int        *ia   = dmat->ia, *ja = dmat->ja;
int        j, k, nr = mat->rows;

if (ia[row+1] == ia[row]) {
    /* Row is empty */
    for (k=row+2; k<nr+1; k++) 
	if (ia[k] != ia[row]) break;
    if (k == nr+1) {
	/* At the end of the current matrix */
	if (ia[row] + nz < dmat->maxnz) {
	    /* There is room to add the data at the end */
	    ja += ia[row] - 1;
	    a  += ia[row] - 1;
	    for (k=0; k<nz; k++) {
		ja[k] = i[k] + 1;
		a[k]  = v[k];
		}
	    for (k=row+1; k<nr+1; k++) 
		ia[k] = ia[row] + nz;
	    return;
	    }
	/* else, need to extend matrix */
	SETERRC(1,"Extension of AIJ matrix not implemented");
	return;
	}
    }
SETERRC(1,"Insertion into AIJ matrix not implemented");
}

/*+
   SpAIJCreateSplit - Allocate an n x m AIJ matrix factor.

   Input Parameters:
.   model - model matrix   
.   mmax - estimated number of elements in each row (this many elements
          will be pre-allocated; this value may be set to zero, in which
	  case space will be allocated as required.)
 +*/
SpMatSplit *SpAIJCreateSplit( model, mmax )
SpMat      *model;
int        mmax;
{
int  n = model->rows, m = model->cols;
SpMatSplit *mat;

TRPUSH(SPTRID + TRIDCREATE + 1);
mat         = NEW(SpMatSplit);  CHKPTRV(mat,0);
mat->type   = MATSPLIT;
mat->factor = SpAIJCreate( n, m, mmax ); CHKERRV(mat->factor,0);
mat->nzl    = (int *) MALLOC( n*sizeof(int) ); CHKPTRV(mat->nzl,0);
TRPOP;
return mat;
}
