#ifndef lint
static char SCCSid[] = "@(#) ./sparse/toysmp.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"

/*@
  SpToAIJ - Convert a sparse matrix to AIJ format. 
            No space is allocated by this routine.

    Description:
    Take the data in the row-oriented sparse storage and build the
    AIJ data for the Matrix.  Return 0 on success, number of additional
    elements needed on failure.

    Input parameters:
.   Matrix - matrix to convert
.   maxj   - number of words in ja (and a) to hold the matrix values

    Output parameters:
.   ia,ja,a - pointers to storage used for conventional aij form.
    ia must be at least (Matrix->rows + 1) in size.

    Returns:
    0 on success, or the number of additional storage elements in ja and a
    that are needed (that is, maxj needs to be this much larger).
 @*/
int SpToAIJ( Matrix, ia, ja, a, maxj )
SpMat           *Matrix;
int             *ia, maxj;
register int    *ja;
register double *a;
{
double          *pp;
register double *p;
int             n, row, *jj, nnz;
register int    *j, nz;

SPLITTOMAT(Matrix);
n  = Matrix->rows;

/* If Matrix->nz is known, can use that the check for adequate storage
   in advance */

/* transcribe the data */
ia[0] = 1;
for (row = 0; row < n; row++) {
    SpScatterFromRow( Matrix, row, &nnz, &jj, &pp );
    ia[row+1] = ia[row] + nnz;
    maxj      -= nnz;
    if (maxj < 0) {
	/* Figure out how much we needed */
	while (row < n) {
	    SpScatterFromRow( Matrix, row, &nnz, (int **)0, (double**)0 );
	    maxj -= nnz;
	    }
	return -maxj;
	}
    j  = jj;
    p  = pp;
    nz = nnz;
    while (nz--) {
        *ja++ = 1 + *j++;   /* We use zero offset, ysmp uses 1 */
        *a++  = *p++;
        }
    }
return 0;
}

/*@
  SpFromAIJ - Convert a sparse matrix in AIJ format to Sp Format.

  Input parameters:
.  ia,ja,a - Conventional aij format
.  nrow    - number of rows in matrix
  
  Returns:
  matrix
@*/
SpMat *SpFromAIJ( ia, ja, a, nrow )
int             *ia, nrow;
register int    *ja;
register double *a;
{
SpMat             *mat;
SpVec             *r;
SpRowMat          *R;
int               row, matNz = 0;
register int      nz, *j, i;
register double   *p;

mat = SpCreate( nrow, nrow, 0 ); CHKERRV(1,0);
R   = (SpRowMat *)mat->data;
for (row = 0; row < nrow; row++) {
    r    = R->rs[row];
    nz   = ia[row+1] - ia[row];
    if (nz < 0) {
	SETERRC(1,"Invalid data in ia array in SpFromAIJ");
	return 0;
	}
    matNz += nz;
    /* allocate this many elements */
    SPMallocNV( mat, nz, &r->v, &r->i ); CHKERRV(1,0);

    /* copy into the storage */
    r->nz     = nz;
    p         = r->v;
    j         = r->i;
    while (nz--) {
        *j++  = *ja++ - 1;   /* We use zero offset, ysmp uses 1 */
        *p++  = *a++;
        }
    /* We really should sort them.  Eventually, we should make this
       an option.  For now, since sorting is expensive, we first check to
       see if we should sort... */
    j  = r->i;
    nz = r->nz;
    for (i=1; i<nz; i++)
	if (j[i] < j[i-1]) break;
    if (i < nz)
	SpSort( r->i, r->v, r->nz );
    }

mat->nz = matNz;    
return mat;
}

/*@
  SpToIJ - Convert a sparse matrix to IJ format (ignore the "A" part)
           No space is allocated by this routine.

    Description:
    Take the data in the row-oriented sparse storage and build the
    IJ data for the Matrix.  Return 0 on success, row + 1 on failure
    at that row.

    Input Parameters:
.   Matrix - matrix to convert
.   maxj   - maximum number of elements in ja    

    Output Parameters:
.   ia     - ia part of IJ representation (row information)
.   ja     - ja part (column indices)
 @*/
int SpToIJ( Matrix, ia, ja, maxj )
SpMat  *Matrix;
int    *ia, *ja, maxj;
{
int    *j, nz, n, row;
SpVec  *r;
SpRowMat *R;

SPLITTOMAT(Matrix);
R  = GETROWMAT(Matrix);
n  = Matrix->rows;

/* transcribe the data */
ia[0] = 1;
for (row = 0; row < n; row++) {
    r    = R->rs[row];
    nz   = r->nz;
    ia[row+1] = ia[row] + nz;
    j         = r->i;
    maxj      -= nz;
    if (maxj < 0) return row + 1;
    while (nz--) {
        *ja++ = 1 + *j++;   /* We use zero offset, ysmp uses 1 */
        }
    }
return 0;
}

/*@
  SpFromIJA - Convert a sparse matrix in IJA format to Sp Format.

  Input parameters:
.  ia,ja,a - Conventional aij format
.  nrow    - number of rows in matrix
  
  Returns:
  matrix

  Notes:
  IJA format is different from AIJ format.  In IJA, the diagonals
  are stored in a(i), and the off-diagonals are stored separately.
  The format is
  ija(row)    = offset into ija of non-diagonal column indices
  ija(offset) = column
  a(offset)   = value for that column
  a(row)      = value for mat(row,row)
@*/
SpMat *SpFromIJA( ija, a, nrow )
int    *ija, nrow;
double *a;
{
SpMat    *mat;
SpVec    *r;
SpRowMat *R;
int      k, row, nz, *j, matNz = 0, offset;
double   *p;

mat = SpCreate( nrow, nrow, 0 ); CHKERRV(1,0);
R   = (SpRowMat *)mat->data;
for (row = 0; row < nrow; row++) {
    r      = R->rs[row];
    offset = ija[row] - 1;
    nz     = ija[row+1] - ija[row] + 1;
    matNz += nz;
    /* allocate this many elements */
    SPMallocNV( mat, nz, &r->v, &r->i ); CHKERRV(1,0);

    /* copy into the storage */
    r->nz     = nz;
    p         = r->v;
    j         = r->i;
    /* Add in diagonal element */
    *j++      = row;
    *p++      = a[row];
    nz--;
    for (k=0; k<nz; k++) {
        j[k]  = ija[offset+k] - 1;   /* We use zero offset, ija uses 1 */
        p[k]  = a[offset+k];
        }
    }

mat->nz = matNz;    
return mat;
}
