#ifndef lint
static char *SCCSid = "@(#)rgsun.c	1.7	(NCSA)	9/9/87";
#endif

/*
**
** rgsun.c
** Steve Alexander, based on
** rge.c by Aaron Contorer for NCSA
** Copyright 1987, board of trustees, University of Illinois
** 
** graphics routines for drawing in Sun Window
** Input coordinate space = 0..4095 by 0..4095
** 
** Prefix	Meaning
** ------	-----------------------------------
** RG	Real Graphics routines in general
** RGS	Real Graphics for Sun specifically
** SUN	Internal calls for Sun specifically
** VG	Virtual Graphics driver
** 
*/

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sun/fbio.h>
#include <sys/time.h>
#include <suntool/sunview.h>
#include <suntool/window.h>
#include <suntool/scrollbar.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include <pixrect/pixrect.h>
#include <sunwindow/pixwin.h>
#include <stdio.h>
#include "defs.h"
#include "session.h"

static short icon_image[] = {
#include "Icons/rgsun.icon"
};
mpr_static(icon_pixrect, 64, 64, 1, icon_image);

/*
** these should be included from vgtek.h, but for now
** this is ok.  since we can save Tek now, we may not
** need to add many more of these...
*/

#define NULL 0
#define SUN 1
#define PS 2
#define HP 3

#define TRUE 1
#define FALSE 0

#define INXMAX 4096
#define INYMAX 4096
#define DEFXMAX 640
#define DEFYMAX 500

/*
** these two must be one smaller than the two above
*/

#define SCRNXHI 639
#define SCRNYHI 499

#define MAXZOOM 1000			/* a nice big round number  */

static float aratio = 1.28;		/* 1152 / 900 aspect ratio	*/
extern char *malloc();

static char *Sunname = "SunView Window (640 x 500)";
static int outxmax = DEFXMAX;
static int outymax = DEFYMAX;
static int outbytes = 80;		/* number of bytes per line */
static int cursx, cursy;		/* graphics cursor */

/*
** Current status of a SUN window
*/

struct SUNWIN {
	int			VGnum;			/* virtual window number */
	Icon		icon;			/* Icon for the frame	 */
	Frame		frame;			/* SunView Frame		 */
	Canvas		canvas;			/* "  "    Canvas		 */
	Pixwin		*pixwin;		/* Pixwin from Canvas	 */	
	Scrollbar	scrollx;
	Scrollbar	scrolly;
	Panel		panel;			/* panel for zoom control*/
	Panel_item	slider;			/* zoom control			 */
	char		inuse;			/* active/inactive flag	 */
	int			xmax;
	int			ymax;
	int			pencolor;		/* use PIX_COLOR(this)	 */
	int			rotation;
	int			size;
	int			winbot;
	int			winleft;
	int			wintall;
	int			winwide;
	int			curzoom;		/* current zoom factor 	 */
	u_char		red[2];			/* used for colormap	 */
	u_char		green[2];
	u_char		blue[2];
	int			x1, x2, y1, y2;
};

static struct SUNWIN window[MAXTEK];

/*
** mode switch routines are dummies
*/

RGSgmode()
{
}

RGStmode()
{
}

/*
** draw a line from (x0,y0) to (x1,y1)
*/

static SUNrasline(w,x0,y0,x1,y1)
int w,x0,y0,x1,y1;
{
	char buf[200];

	sprintf(buf,"%d %d %d %d %d\r\n",w,x0,y0,x1,y1);
	debug(8,buf);

	x0 = (int) ((long)x0 * window[w].xmax / INXMAX);
	y0 = window[w].ymax-(int) ((long)y0 * window[w].ymax / INYMAX);
	x1 = (int) ((long)x1 * window[w].xmax / INXMAX);
	y1 = window[w].ymax-(int) ((long)y1 * window[w].ymax / INYMAX);

	sprintf(buf,"%d %d %d %d %d\r\n",w,x0,y0,x1,y1);
	debug(8,buf);

	pw_batch_on(window[w].pixwin);
	pw_vector(window[w].pixwin, x0, y0, x1, y1, PIX_COLOR(1) | PIX_SRC, 1);
	pw_batch_off(window[w].pixwin);

} /* end SUNrasline() */

/* 
**	Clear the screen.
*/

RGSclrscr(w)
int w;
{
	debug(5,"In RGclrscr\r\n");
	pw_writebackground(window[w].pixwin, 0, 0, window[w].xmax,
					   window[w].ymax, PIX_SRC);
}

/* 
**	Set up a new window; return its number.
**	Returns -1 if cannot create window.
*/

int RGSnewwin()
{

	extern Notify_value		kbdhandler();
	Notify_value			RGSslider();
	Notify_value			RGSdestroy();
	int						RGSresize();
	Pixwin					*pw;
	Menu					new_menu;
	extern char				hostname[];
	Notify_value			RGSscroll();
	char					flabel[200];
	char					ilabel[20];
	char					ihname[8];
	Rect					irect;
	extern struct pixfont	*ifont;
	int						w = 0;

	debug(5, "In RGSnewwin\r\n");

	while (w < MAXTEK && window[w].inuse)
		w++;
	if (w == MAXTEK)
		return(-1); /* no windows available */

	/*
	** the following snotty comment means that if VGnewwin
	** doesn't call VGgiveinfo to tell us what VGnum really
	** is, we're in big trouble.  Fortunately, it does.
	*/

	window[w].VGnum = 0;			/* uh-huh?  yeah... */
	window[w].pencolor = 1;
	window[w].winbot = 0;
	window[w].wintall = 3072;
	window[w].winleft = 0;
	window[w].winwide = 4096;
	window[w].inuse = TRUE;
	window[w].curzoom = MAXZOOM;

	sprintf(flabel,"Tektronix Window #%d - %s",w + 1,hostname);

	/*
	** create the icon
	*/

	window[w].icon = icon_create(ICON_IMAGE, &icon_pixrect, 0);
	strncpy(ihname, hostname, 8);
	ihname[7] = '\0';
	sprintf(ilabel, "#%d %s",w + 1, ihname);
	irect.r_width = 62;
	irect.r_height = 9;
	irect.r_left = 1;
	irect.r_top = 54;
	icon_set(window[w].icon, ICON_FONT, ifont, 0);
	icon_set(window[w].icon, ICON_LABEL, ilabel, 0);
	icon_set(window[w].icon, ICON_LABEL_RECT, &irect, 0);

	/*
	** create the frame
	*/

	window[w].frame = window_create(0, FRAME,WIN_SHOW,TRUE,
									FRAME_ICON, window[w].icon,
									WIN_WIDTH,outxmax + 24,
									WIN_HEIGHT,outymax + 23 + 25 + 18,
									FRAME_LABEL,flabel,0);

	/*
	** create the slider panel
	*/

	window[w].panel = window_create(window[w].frame, PANEL,
									WIN_HEIGHT,25,
									WIN_X,0,0);

	/*
	** create the slider itself
	*/

	window[w].slider = panel_create_item(window[w].panel,
										 PANEL_SLIDER,
										 PANEL_LABEL_STRING,"Zoom:",
										 PANEL_VALUE, MAXZOOM,
										 PANEL_MIN_VALUE,1,
										 PANEL_MAX_VALUE,MAXZOOM,
										 PANEL_SHOW_RANGE,FALSE,
										 PANEL_SHOW_VALUE,FALSE,
										 PANEL_SLIDER_WIDTH,outxmax - 80,
										 PANEL_NOTIFY_PROC,RGSslider,
										 0);

	/*
	** set up so that we can tell when the user Frame->Quits on
	** us...
	*/

	notify_interpose_destroy_func(window[w].frame, RGSdestroy);

	window[w].xmax = outxmax;
	window[w].ymax = outymax;

	window_fit_height(window[w].panel);

	window[w].canvas = window_create(window[w].frame, CANVAS,
									 WIN_BELOW,window[w].panel,
									 WIN_CONSUME_KBD_EVENTS,
									 WIN_ASCII_EVENTS,
									 WIN_RIGHT_KEYS,WIN_LEFT_KEYS,
									 WIN_TOP_KEYS,
									 WIN_MOUSE_BUTTONS,
									 0,
									 WIN_IGNORE_KBD_EVENTS,
									 WIN_UP_EVENTS,
									 WIN_UP_ASCII_EVENTS,
									 0,
									 CANVAS_RETAINED,FALSE,
									 CANVAS_WIDTH, DEFXMAX,
									 CANVAS_HEIGHT, DEFYMAX,
									 WIN_SHOW,TRUE,
									 CANVAS_FAST_MONO,TRUE,	/* 3/110 hack */
									 CANVAS_AUTO_CLEAR,FALSE,
									 WIN_EVENT_PROC,kbdhandler,
									 CANVAS_RESIZE_PROC,RGSresize,
									 0);

	/*
	** set canvas menu
	*/

	init_tek_menu(w,window[w].canvas);

	/*
	** set colormap
	*/

	pw = canvas_pixwin(window[w].canvas);

	window[w].red[0] = window[w].green[0] = window[w].blue[0] = 255;
	window[w].red[1] = window[w].green[1] = window[w].blue[1] = 0;
#ifdef colorstufffromhell
	/*
	** red
	*/
	window[w].red[2] = 255;
	window[w].green[2] = window[w].blue[2] = 0;
	/*
	** green
	*/
	window[w].red[3] = 0;
	window[w].green[3] = 255;
	window[w].blue[3] = 0;
	/*
	** blue
	*/
	window[w].red[4] = window[w].green[4] = 0;
	window[w].blue[4] = 255;
	/*
	** cyan
	*/
	window[w].red[5] = 0;
	window[w].green[5] = window[w].blue[5] = 255;
	/*
	** magenta
	*/
	window[w].green[6] = 0;
	window[w].red[6] = window[w].blue[6] = 255;
	/*
	** yellow
	*/
	window[w].red[7] = window[w].green[7] = 255;
	window[w].blue[7] = 0;

	pw_setcmsname(pw, "tkcms");

	pw_putcolormap(pw, 0, 8, window[w].red, window[w].green, window[w].blue);

#endif colorstufffromhell

	pw_setcmsname(pw, "bwcms");

	pw_putcolormap(pw, 0, 2, window[w].red, window[w].green, window[w].blue);

	window_set(window[w].canvas, CANVAS_RETAINED,TRUE,CANVAS_AUTO_CLEAR,
				TRUE,CANVAS_FIXED_IMAGE,FALSE,0);

	/*
	** set up the scroll bars
	*/

	window[w].scrollx = scrollbar_create(SCROLL_PLACEMENT,SCROLL_SOUTH,0);
	window[w].scrolly = scrollbar_create(0);

	scrollbar_set(window[w].scrollx, SCROLL_NOTIFY_CLIENT, window[w].canvas,0);
	scrollbar_set(window[w].scrolly, SCROLL_NOTIFY_CLIENT, window[w].canvas,0);

	window_set(window[w].canvas, WIN_HORIZONTAL_SCROLLBAR, window[w].scrollx,0);
	window_set(window[w].canvas, WIN_VERTICAL_SCROLLBAR, window[w].scrolly,0);

	scrollbar_set(window[w].scrollx, SCROLL_OBJECT, window[w].canvas, 0);
	scrollbar_set(window[w].scrolly, SCROLL_OBJECT, window[w].canvas, 0);

	scrollbar_set(window[w].scrollx, SCROLL_OBJECT_LENGTH, 4096,0);
	scrollbar_set(window[w].scrolly, SCROLL_OBJECT_LENGTH, 4096,0);

	scrollbar_set(window[w].scrollx, SCROLL_NORMALIZE, FALSE,0);
	scrollbar_set(window[w].scrolly, SCROLL_NORMALIZE, FALSE,0);

	scrollbar_set(window[w].scrollx, SCROLL_ADVANCED_MODE, TRUE,0);
	scrollbar_set(window[w].scrolly, SCROLL_ADVANCED_MODE, TRUE,0);

	scrollbar_set(window[w].scrollx, SCROLL_LINE_HEIGHT, 1, 0);
	scrollbar_set(window[w].scrolly, SCROLL_LINE_HEIGHT, 1, 0);
	update_scroll(w);

	/*
	** snarf the pixwin
	*/

	window[w].pixwin = pw;

	/*
	** clear screen
	*/

	RGSclrscr(w);

	debug(5, "Leaving RGSnewwin\r\n");

	return(w);
}

/*
** resize handler
*/

RGSresize(canvas, width, height)
Canvas canvas;
int width;
int height;
{
	char buf[60];
	int w ;

	w = RGSgetwin(canvas);

	sprintf(buf,"In RGSResize ; w = %d, h = %d\r\n", width, height);
	debug(5,buf);

	/*
	** rescale
	*/

	if ((float)width / (float)height < aratio) {
		window[w].ymax = (int)((float)width / aratio);
		window[w].xmax = width;
	}
	else {
		window[w].xmax = (int)(aratio *(float)height);
		window[w].ymax = height;
	}

	sprintf(buf,"RGSResize scaled ; w = %d, h = %d\r\n",window[w].xmax,
			window[w].ymax);
	debug(5,buf);

	/*
	** clear off old image 
	*/

	pw_batch_on(window[w].pixwin);
	pw_writebackground(window[w].pixwin, 0, 0, width,
						   height, PIX_SRC);
	pw_batch_off(window[w].pixwin);

	/*
	** fix slider
	*/

	panel_set(window[w].slider, PANEL_SLIDER_WIDTH, 
						(width < 80) ? 0 : width - 80, 0);

	Session.s_tekwin = window[w].VGnum;
	Session.s_termstate = TEKTYPE;

	/*
	** now redraw
	*/

	sprintf(buf,"w = %d, vgnum = %d\n",w, window[w].VGnum);
	debug(5,buf);

	pw_batch_on(window[w].pixwin);
	VGredraw(window[w].VGnum,window[w].VGnum);
	pw_batch_off(window[w].pixwin);

}

/*
** RGSgetcanvas -- return the canvas for a window
*/

Canvas RGSgetcanvas(w)
int w;
{
	return window[w].canvas;
}

/*
** get the window number for a frame
*/

RGSgetwinbyframe(frame)
Frame frame;
{
	int i;

	for (i = 0; i < MAXTEK; i++) {
		char buf[200];
		sprintf(buf,"i = %d, frame= %X, i.frame= %X\r\n",i, frame,
				window[i].frame);
		debug(5,buf);
		if (window[i].frame == frame)
			return i;
	}
	return -1;
}

/*
** get the window number for a canvas
*/

RGSgetwin(canvas)
Canvas canvas;
{
	int i;

	for (i = 0; i < MAXTEK; i++) {
		char buf[200];
		sprintf(buf,"i = %d, canvas= %X, i.canvas= %X\r\n",i, canvas,
				window[i].canvas);
		debug(5,buf);
		if (window[i].canvas == canvas)
			return i;
	}
	return -1;
}

/*
** get the window number for a slider
*/

RGSgetwinpan(item)
Panel_item item;
{
	int i;

	for (i = 0; i < MAXTEK; i++)
		if (window[i].slider == item)
			return i;
	return -1;
}

/*
** destroy a window... lots of notifier-subterfuging here...
*/

RGSclose(vw)
int vw;
{
	Notify_value RGSdestroy();

	Session.s_termstate = VTTYPE;
	if (!window[vw].inuse)
		return;
	window[vw].inuse = FALSE;

	/*
	** make it so window doesn't ask whether we want to do this
	*/

	window_set(window[vw].frame,FRAME_NO_CONFIRM,TRUE,0);

	/*
	** get our destroy function out of the way, and then
	** blow the window away.
	*/

	notify_remove_destroy_func(window[vw].frame,RGSdestroy);
	window_destroy(window[vw].frame, DESTROY_CLEANUP);
}

/*
** set pixel at location (x,y) -- no range checking performed
*/

RGSpoint(w,x,y)
int w,x,y;
{
	int x2, y2;

	Pixwin *pw = window[w].pixwin;

	x2=(int) ((long)x*window[w].xmax/INXMAX);
	y2=SCRNYHI-(int) ((long)y*window[w].ymax/INYMAX);
	pw_put(pw, x2, y2, 1);			/* need pencolor here for color stuff */
}  

/*
** new call  -- Aaron says life is easier this way...
*/

RGSdrawline(w,x0,y0, x1, y1)
int w,x0,y0, x1, y1;
{
	SUNrasline(w,x0,y0,x1,y1);
}

/*
**	Do whatever has to be done when the drawing is all done.
**	(For printers, that means eject page.)
*/

RGSpagedone(w)
int w;
{
	/* do nothing for SUN */
}

/*
** Copy 'count' bytes of data to screen starting at current
** cursor location.
*/

RGSdataline(w,data,count)
int w,count;
char *data;
{
	/* Function not supported yet. */
}

/*
** Change pen color to the specified color. -- this is not too tough.
*/

RGSpencolor(w,color)
int w,color;
{
	window[w].pencolor = color;
}

/*
**	Set description of future device-supported graphtext.
**	Rotation=quadrant:
**
**  1 xxxx   2 xxxxxxxxx  3 x  x  4 xxxxxxxxx
**  x  x     x    x       xxxx       x    x
**  x  x     xxxxxxxxx    x  x    xxxxxxxxx
**  xxxx                  x  x    
**  x  x                  xxxx
**
*/

RGScharmode(w,rotation,size)
{
	/* No rotatable device-supported graphtext is available on SUN. */
}

/* Not yet supported: */
RGSshowcur() {}
RGSlockcur() {}
RGShidecur() {}

/*
** Ring bell in window w
*/

RGSbell(w)
int w;
{
	struct timeval tv;
	Pixwin *pw = window[w].pixwin;
	Canvas cw = window[w].canvas;

	tv.tv_sec = 0;
	tv.tv_usec = 45500;		/* microseconds */
	win_bell(window_get(cw, WIN_FD), tv, pw);
}

/*
** return name of device that this RG supports
*/

char *RGSdevname()
{
	return(Sunname);
}

/*
** get info from VG
*/

RGSinfo(w, v, a, b, c, d)
int w, v, a, b, c, d;
{
	window[w].VGnum = v;
	window[w].winbot = a;
	window[w].winleft = b;
	window[w].wintall = c - a + 1;
	window[w].winwide = d - b + 1;
}

/*
** initialize all RGS variables
*/

RGSinit()
{
	int				i;
	int				f;
	struct fbtype	fbstruct;

	for (i=0; i < MAXTEK; i++) {
		window[i].inuse = FALSE;
	}

	/*
	** compute the aspect ratio
	*/

	f = open("/dev/fb",0);

	if (f < 0)
		return;

	ioctl(f, FBIOGTYPE, &fbstruct);

	if (fbstruct.fb_height == 0 || fbstruct.fb_width == 0)
		return;
	aratio = (float)fbstruct.fb_width / (float)fbstruct.fb_height;
	outymax = (int)((float)outxmax / aratio);

	close(f);

}

/*
** Make this window visible, hiding all others.
** Caller should follow this with clrscr and redraw to show the current
** contents of the window.
*/
RGSuncover(w)
int w;
{
}

/*
** menu of things to do
*/

Menu tek_menu;
static int current = -1;

init_tek_menu(vw, canvas)
int vw;
Canvas canvas;
{

	caddr_t tek_pw();
	caddr_t tek_pb();
	caddr_t tek_hp();
	caddr_t tek_sav();
	caddr_t tek_uz();
	caddr_t tek_pg();
	Menu_item uzgen();

	extern struct Pixfont *zzbold;

	tek_menu = menu_create(MENU_FONT,zzbold,
						   MENU_ITEM, MENU_STRING,
						   "UnZoom",
						   MENU_GEN_PROC, uzgen,
						   MENU_INACTIVE, TRUE,
						   MENU_ACTION_PROC,tek_uz,
						   0,
						   MENU_ITEM,MENU_STRING,
						   "Clear Page",MENU_ACTION_PROC,
						   tek_pg,
						   0,
						   MENU_ITEM,MENU_STRING,
						   "Print PostScript",
						   MENU_ACTION_PROC,tek_pw,
						   0,
						   MENU_ITEM,MENU_STRING,
						   "Print HP-GL",
						   MENU_ACTION_PROC,tek_hp,0,
						   MENU_ITEM,MENU_STRING,
						   "Save Tek Image",
						   MENU_ACTION_PROC,tek_sav,0,
						   MENU_ITEM,MENU_STRING,
						   "Print Bitmap",
						   MENU_ACTION_PROC,tek_pb,
						   0,
						   0);

}

/*
** display the menu
*/

display_tek_menu(vw, event)
int vw;
Event *event;
{

	current = vw;

	if (event_is_up(event))
		return;
	menu_show(tek_menu,window[vw].canvas, event, 0);
}

/*
** unzoom if not zoomed
*/

caddr_t tek_uz(m, mi)
Menu m;
Menu_item mi;
{
	int vw = current;

	window[vw].curzoom = MAXZOOM;

	VGzoom(window[vw].VGnum, 0, 0, 4095, 3071);

	update_scroll(vw);
	panel_set_value(window[vw].slider, window[vw].curzoom);

	pw_batch_on(window[vw].pixwin);
	pw_writebackground(window[vw].pixwin, 0, 0, window[vw].xmax,
						   window[vw].ymax, PIX_SRC);

	VGredraw(window[vw].VGnum,window[vw].VGnum);
	pw_batch_off(window[vw].pixwin);

}

/*
** generate proc for unzoom
*/

Menu_item uzgen(mi, op)
Menu_item mi;
Menu_generate op;
{
	switch (op) {
		case MENU_DISPLAY:
			if (window[current].curzoom == MAXZOOM)
				menu_set(mi, MENU_INACTIVE, TRUE, 0);
			else
				menu_set(mi, MENU_INACTIVE, FALSE, 0);
			break;

		case MENU_DISPLAY_DONE:
		case MENU_NOTIFY:
		case MENU_NOTIFY_DONE:
			break;
	}
	return mi;
}

/*
** page the screen
*/

caddr_t tek_pg(m, mi)
Menu m;
Menu_item mi;
{
	int vw = current;
	VGwrite(window[vw].VGnum, "\033\014",2);
	VGclrstor(window[vw].VGnum);
}

/*
** print something to the postscript device
*/

caddr_t tek_pw(m, mi)
Menu m;
Menu_item mi;
{
	int ow = current;
	int nw;
	char data;

	hourglasscursor(window[ow].canvas);
	if (open_ps() < 0) {
		errmsg("Error during printing","");
		return;
	}
	nw = VGnewwin(PS);
	if (nw < 0) {
		errmsg("Error during printing","");
		return;
	}

	VGzcpy(window[ow].VGnum, nw);
	VGredraw(window[ow].VGnum, nw);
	VGclose(nw);
	close_ps();

	normalcursor(window[ow].canvas);

	return mi;
}

/*
** capture a tektronix image
*/

caddr_t tek_sav(m, mi)
Menu m;
Menu_item mi;
{
	int			ow = current;
	extern int	tek_outfunc();

	hourglasscursor(window[ow].canvas);
	if (open_tek() < 0) {
		errmsg("Error during saving","");
		return;
	}

	VGdumpstore(ow,tek_outfunc);
	close_tek();

	normalcursor(window[ow].canvas);

	return mi;
}

/*
** print something to the HP-GL device
*/

caddr_t tek_hp(m, mi)
Menu m;
Menu_item mi;
{
	int ow = current;
	int nw;
	char data;

	hourglasscursor(window[ow].canvas);
	if (open_hp() < 0) {
		errmsg("Error during plotting","");
		return;
	}
	nw = VGnewwin(HP);
	if (nw < 0) {
		errmsg("Error during plotting","");
		return;
	}

	VGzcpy(window[ow].VGnum, nw);
	VGredraw(window[ow].VGnum, nw);
	VGclose(nw);
	close_hp();

	normalcursor(window[ow].canvas);

	return mi;
}

/*
** print a bitmap of the window
*/

caddr_t tek_pb(m, mi)
Menu m;
Menu_item mi;
{
	int 		vw;
	Pixrect		*tpr;
	struct pixrect *mem_create();
	colormap_t	cm;
	u_char		red[2], green[2], blue[2];
	FILE		*pout;
	int			pflag;
	extern char bfname[];

	vw = current;
	hourglasscursor(window[vw].canvas);

	/*
	** make a place for the data
	*/

	if ((tpr = mem_create(window[vw].xmax, window[vw].ymax, 1)) == NULL) {
		errmsg("Memory Allocation Failure","");
		return;
	}

	/*
	** grab the window's pixrect
	*/

	pw_read(tpr,0,0,window[vw].xmax, window[vw].ymax, PIX_SRC, 
			window[vw].pixwin, 0, 0);

	/*
	** grab the colormap thing
	*/

	pw_getcolormap(window[vw].pixwin, 0, 2, red, green, blue);
	cm.map[0] = red;
	cm.map[1] = green;
	cm.map[2] = blue;
	cm.type = RMT_EQUAL_RGB;
	cm.length = 2;

	/*
	** open the output file
	*/

	if (bfname[0] == '|') {
		pflag = 1;
		pout = popen(&bfname[1],"w");
	}
	else {
		pflag = 0;
		pout = fopen(bfname,"w");
	}

	if (pout == NULL) {
		syserr("Can't open output device:");
		return;
	}

	/*
	** pump out the output
	*/

	if (pr_dump(tpr, pout, &cm, RT_BYTE_ENCODED, 1) != 0)
		syserr("Write Failed");

	if (pflag)
		pclose(pout);
	else
		fclose(pout);

	pr_close(tpr);

	normalcursor(window[vw].canvas);
	return;
}

/*
** Graphics input thingy
*/

Cursor	wcurs;

RGSgin(vw)
int vw;
{

	wcurs = cursor_create(CURSOR_SHOW_CROSSHAIRS, TRUE, 
		CURSOR_SHOW_CURSOR, FALSE, 
		CURSOR_HORIZ_HAIR_LENGTH, window[vw].xmax - 18,
		CURSOR_VERT_HAIR_LENGTH, window[vw].ymax - 18, 0);

	window_set(window[vw].canvas, WIN_CURSOR, wcurs, 0);
	Session.s_tekgin = TRUE;
}

RGSginoff(canvas, x, y, c)
int		x;
int		y;
Canvas	canvas;
char	c;
{
	int		vw;
	char	a[5];
	float	xx;
	float	yy;

	vw = RGSgetwin(canvas);

	xx = (float)x / (float)window[vw].xmax;
	xx *= (float)window[vw].winwide;
	xx += (float)window[vw].winleft;

	yy = (float)(window[vw].ymax - y) / (float)window[vw].ymax;
	yy *= (float)window[vw].wintall;
	yy += (float)window[vw].winbot;

	x = (int)xx;
	y = (int)yy;

	VGgindata(window[vw].VGnum, x, y, c, a);
	netwrite(Session.s_winval, a, 5);

	cursor_destroy(wcurs);
	normalcursor(window[vw].canvas);
	Session.s_tekgin = FALSE;
}

/*
** handle the slider
*/

Notify_value RGSslider(item, value, event)
Panel_item item;
int value;
Event *event;
{

	int vw;
	float xzoom, yzoom;
	int zoomfac;
	int x, y, w, h;
	char buf[200];

	vw = RGSgetwinpan(item);

	x = window[vw].winleft;
	w = window[vw].winwide;
	y = window[vw].winbot;
	h = window[vw].wintall;

	xzoom = 4.096;
	yzoom = 3.072;

	zoomfac = value - window[vw].curzoom;

	sprintf(buf,"old %d %d %d %d\r\n",x,y,w,h);
	debug(6,buf);
	sprintf(buf,"%d, %f, %f\r\n",zoomfac,xzoom,yzoom);
	debug(6,buf);

	/*
	** special case for maximum zoom
	*/

	if (value == MAXZOOM) {
		x = 0;
		y = 0;
		w = 4096;
		h = 3072;
	}

	else {
		if (zoomfac > 0) {

			/*
			** zooming out
			*/

			x -= (int)((xzoom / 2.0) * (float)zoomfac);	
			y -= (int)((yzoom / 2.0) * (float)zoomfac);	
			w += (int)(xzoom * (float)zoomfac);	
			h += (int)(yzoom * (float)zoomfac);	
		}

		else {

			/*
			** zooming in
			*/

			x += (int)((xzoom / 2.0) * (float) (-zoomfac));	
			y += (int)((yzoom / 2.0) * (float) (-zoomfac));	
			w -= (int)(xzoom * (float) (-zoomfac));	
			h -= (int)(yzoom * (float) (-zoomfac));	
		}
	}

	window[vw].curzoom = value;

	if (x < 0)
		x = 0;

	if (y < 0)
		y = 0;

	sprintf(buf,"new %d %d %d %d\r\n",x,y,w,h);
	debug(6,buf);

	VGzoom(window[vw].VGnum, x, y, x + w, y + h);

	update_scroll(vw);

	pw_batch_on(window[vw].pixwin);
	pw_writebackground(window[vw].pixwin, 0, 0, window[vw].xmax,
						   window[vw].ymax, PIX_SRC);
	VGredraw(window[vw].VGnum,window[vw].VGnum);
	pw_batch_off(window[vw].pixwin);

}

/*
** rubber band for zoom/pan
*/

static int sqd = FALSE;
static int lbd = FALSE;
static int fp = FALSE;
static int sx, sy, ex, ey;
static Pixwin *pw;

RGSrband(canvas, event)
Canvas	canvas;
Event	*event;
{

	int			vw;
	int			ev;
	char		buf[100];

	/*
	** check on left button
	*/

	ev = event_id(event);

	if (ev == MS_LEFT)
		lbd = event_is_down(event);

	/*
	** get window number and box coords
	*/

	vw = RGSgetwin(canvas);
	pw = window[vw].pixwin;

	if (sqd == FALSE) {
		if ((ev == LOC_RGNEXIT) && (fp == TRUE)) {
			fp = FALSE;
			draw_square(sx, sy, ex, ey);
			lbd = FALSE;
		}
		else 
			get_square(event);
	}

	/*
	** square already drawn
	*/

	else {
		if (lbd) {
			sqd = FALSE;
			draw_square(sx, sy, ex, ey);
		}
	}

	/*
	** now map back to useful coords
	*/

	if (sqd) {

		float minx, miny, maxx, maxy;
		float area;

		sprintf(buf,"%d %d %d %d\r\n",sx, sy, ex, ey);
		debug(5,buf);

		if ((sx == ex) && (sy == ey))
			return;

		if (sx > ex) {
			int t;
			t = sx;
			sx = ex;
			ex = t;
		}
		if (sy > ey) {
			int t;
			t = sy;
			sy = ey;
			ey = t;
		}

		minx = (float)sx / (float)window[vw].xmax;
		minx *= (float)window[vw].winwide;
		minx += (float)window[vw].winleft;

		miny = (float)(window[vw].ymax - sy) / (float)window[vw].ymax;
		miny *= (float)window[vw].wintall;
		miny += (float)window[vw].winbot;

		maxx = (float)ex / (float)window[vw].xmax;
		maxx *= (float)window[vw].winwide;
		maxx += (float)window[vw].winleft;

		maxy = (float)(window[vw].ymax - ey) / (float)window[vw].ymax;
		maxy *= (float)window[vw].wintall;
		maxy += (float)window[vw].winbot;

		sprintf(buf,"%f %f %f %f\r\n",minx, miny, maxx, maxy);
		debug(5,buf);

		/*
		** y coordinates are reversed for tek 
		*/

		pw_batch_on(window[vw].pixwin);
		pw_writebackground(window[vw].pixwin, 0, 0, window[vw].xmax,
						   window[vw].ymax, PIX_SRC);
		pw_batch_off(window[vw].pixwin);

		VGzoom(window[vw].VGnum, (int)minx, (int)maxy, (int)maxx, (int)miny);

		pw_batch_on(window[vw].pixwin);
		VGredraw(window[vw].VGnum,window[vw].VGnum);
		pw_batch_off(window[vw].pixwin);

		/*
		** update slider control by computing percentage of total image
		** 4096.0 * 3072.0 = 12582912.0 
		*/

		area = ((maxx - minx) * (maxy - miny));
		if (area < 0)
			area = -area;

		window[vw].curzoom = (int)((float)MAXZOOM * (area / 12582912.0));

		panel_set_value(window[vw].slider, window[vw].curzoom);

		update_scroll(vw);

		sqd = FALSE;
	}	
}

/*
** get the coordinates as the user moves
*/

get_square(event)
Event *event;
{

	int	nx, ny;

	if (fp == FALSE) {
		if (lbd) {
			sx = event_x(event);
			sy = event_y(event);
			ex = sx;
			ey = sy;
			fp = TRUE;
		}
		return;
	}

	if (lbd) {
		nx = event_x(event);
		ny = event_y(event);
		if (nx != ex || ny != ey) {
			draw_square(sx, sy, ex, ey);
			ex = nx;
			ey = ny;
			draw_square(sx, sy, ex, ey);
		}
		return;
	}

	else {
		sqd = TRUE;
		fp = FALSE;
	}
}

/*
** actually draw the box
*/

draw_square(stx,sty,edx,edy)
int stx,sty,edx,edy;
{
	int	x1;
	int	x2;
	int y1;
	int	y2; 

	if (stx < edx) {
		x1 = stx;
		x2 = edx;
	}
	else {
		x1 = edx;
		x2 = stx;
	}

	if (sty < edy) {
		y1 = sty;
		y2 = edy;
	}
	else {
		y1 = edy;
		y2 = sty;
	}

	pw_batch_on(pw);
	pw_vector(pw,x1,y1,x1,y2,PIX_SRC ^ PIX_DST, 1);
	pw_vector(pw,x1,y1,x2,y1,PIX_SRC ^ PIX_DST, 1);
	pw_vector(pw,x1,y2,x2,y2,PIX_SRC ^ PIX_DST, 1);
	pw_vector(pw,x2,y1,x2,y2,PIX_SRC ^ PIX_DST, 1);
	pw_batch_off(pw);

}

/*
** scroll bar handler
*/
Notify_value RGSscroll(client, event, arg, type)
Notify_client		client;
Event				*event;
Notify_arg			arg;
Notify_event_type	type;
{

	int		vw;
	float	diff;
	int		motion;
	int		offset;
	int		sh;
	int		sw;

	sh = (int)scrollbar_get(arg, SCROLL_HEIGHT);
	sw = (int)scrollbar_get(arg, SCROLL_WIDTH);

	vw = RGSgetwin(client);

	if (vw < 0)
		return NOTIFY_DONE;

	switch (event_id(event)) {

		case SCROLL_EXIT:
			return;

		case SCROLL_ENTER:

			/*
			** if entering a scrollbar, make sure values are ok
			*/

			if (arg == window[vw].scrollx)
				scrollbar_set(window[vw].scrollx,
					SCROLL_OBJECT_LENGTH, 4096,
					SCROLL_VIEW_START, window[vw].winleft,
					SCROLL_VIEW_LENGTH, window[vw].winwide,
					0);

			if (arg == window[vw].scrolly)
				scrollbar_set(window[vw].scrolly,
					SCROLL_OBJECT_LENGTH, 4096,
					SCROLL_VIEW_START, window[vw].winbot,
					SCROLL_VIEW_LENGTH, window[vw].wintall,
					0);

			return;

		case SCROLL_REQUEST:

			/*
			** figure out which one's moving
			*/

			motion = (int)scrollbar_get(arg,SCROLL_REQUEST_MOTION);
			offset = (int)scrollbar_get(arg,SCROLL_REQUEST_OFFSET);

			switch ((int)scrollbar_get(arg, SCROLL_DIRECTION)) {

				/*
				** y direction
				*/

				case SCROLL_VERTICAL:
					switch (motion) {
						case SCROLL_POINT_TO_MIN:
						case SCROLL_LINE_FORWARD:
							if (window[vw].winbot == 0)
								return;
							window[vw].winbot -= 41;
							break;

						case SCROLL_MIN_TO_POINT:
						case SCROLL_LINE_BACKWARD:
							if (window[vw].winbot == 4095)
								return;
							window[vw].winbot += 41;
							break;

						case SCROLL_MAX_TO_POINT:
						case SCROLL_PAGE_BACKWARD:
							if (window[vw].winbot == 0)
								return;
							window[vw].winbot -= window[vw].wintall;
							break;

						case SCROLL_POINT_TO_MAX:
						case SCROLL_PAGE_FORWARD:
							if (window[vw].winbot == 4095)
								return;
							window[vw].winbot += window[vw].wintall;
							break;

						case SCROLL_ABSOLUTE:
							diff = (float) offset / (float) sh;
							diff *= 4096.0;

							if (window[vw].winbot == (int)diff)
								return;
							window[vw].winbot = (int)diff;

							break;
							
						default:
							break;
					}

					break;

				/*
				** x direction
				*/

				case SCROLL_HORIZONTAL:
					switch (motion) {
						case SCROLL_POINT_TO_MIN:
						case SCROLL_LINE_FORWARD:
							if (window[vw].winleft == 0)
								return;
							window[vw].winleft -= 41;
							break;

						case SCROLL_MIN_TO_POINT:
						case SCROLL_LINE_BACKWARD:
							if (window[vw].winleft == 4095)
								return;
							window[vw].winleft += 41;
							break;

						case SCROLL_MAX_TO_POINT:
						case SCROLL_PAGE_BACKWARD:
							if (window[vw].winleft == 0)
								return;
							window[vw].winleft -= window[vw].winwide;
							break;

						case SCROLL_POINT_TO_MAX:
						case SCROLL_PAGE_FORWARD:
							if (window[vw].winleft == 4095)
								return;
							window[vw].winleft += window[vw].winwide;
							break;

						case SCROLL_ABSOLUTE:
							diff = (float) offset / (float) sw;
							diff *= 4096.0;

							if (window[vw].winleft == (int)diff)
								return;
							window[vw].winleft = (int)diff;
							break;
							
						default:
							break;
					}

					break;

				/*
				** huh?
				*/

				default:
					break;
			}

			default:
				break;
	}

	/*
	** force coords in range
	*/

	if (window[vw].winleft < 0)
		window[vw].winleft = 0;

	if (window[vw].winbot < 0)
		window[vw].winbot = 0;

	if (window[vw].winleft > 4095)
		window[vw].winleft = 4095;

	if (window[vw].winbot > 4095)
		window[vw].winbot = 4095;

	/*
	** update the scroll bars
	*/

	VGzoom(window[vw].VGnum, window[vw].winleft, window[vw].winbot,
		   window[vw].winleft + window[vw].winwide - 1,
		   window[vw].winbot + window[vw].wintall - 1);

	update_scroll(vw);

	pw_batch_on(window[vw].pixwin);
	VGpage(window[vw].VGnum);
	VGredraw(window[vw].VGnum,window[vw].VGnum);
	pw_batch_off(window[vw].pixwin);

	return NOTIFY_DONE;
}

/*
** update the scroll bars when we zoom
*/

update_scroll(vw)
int vw;
{

	scrollbar_set(window[vw].scrollx,
				  SCROLL_VIEW_START, window[vw].winleft, 0);
	scrollbar_set(window[vw].scrolly,
				  SCROLL_VIEW_START,window[vw].winbot,0);

	scrollbar_set(window[vw].scrollx,
				  SCROLL_VIEW_LENGTH, window[vw].winwide, 0);
	scrollbar_set(window[vw].scrolly,
				  SCROLL_VIEW_LENGTH, window[vw].wintall, 0);

	scrollbar_paint(window[vw].scrollx);
	scrollbar_paint(window[vw].scrolly);
}

/*
** destroy the frame
*/

Notify_value RGSdestroy(frame, status)
Frame			frame;
Destroy_status	status;
{
	int i;

	if (status == DESTROY_CHECKING) {
		Session.s_termstate = VTTYPE;
		return NOTIFY_DONE;
	}

	if (window[RGSgetwinbyframe(frame)].inuse)
		VGclose(window[RGSgetwinbyframe(frame)].VGnum);

	(void) notify_next_destroy_func(frame, status);
	return NOTIFY_DONE;
}
