 /*
  * Khoros: $Id: writeimage.c,v 1.2 1991/07/15 06:07:03 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: writeimage.c,v 1.2 1991/07/15 06:07:03 khoros Exp $";
#endif

 /*
  * $Log: writeimage.c,v $
 * Revision 1.2  1991/07/15  06:07:03  khoros
 * HellPatch1
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.

 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

/*

    WRITEIMAGE - Write an KHOROS image to the file pointed to by the
                 supplied file descriptor.

    Written: Scott Wilson
    Date:     3-Mar-87

    Returns:  1 if successful
	      0 if unsuccessful

    Modified: Scott Wilson and Tom Sauer 11/23/88
              Scott Wilson 27-Feb-89 - Updated to XV III
              Scott Wilson 22-Apr-89 - Finished encode/decode support
	      Scott Wilson 10-Jul-89 - Added return value
              Scott Wilson 7-Jan-90  - Added image cache support
              Scott Wilson 22-Jun-90 - Modified image cache support
              Scott Wilson 11-Dec-90 - Added normal file locking
              Scott Wilson 14-Feb-91 - Added sanity check on data pointers
              Scott Wilson 23-Feb-91 - Added cache write disable support
              Scott Wilson 26-Feb-91 - Changed cache write disable support so
                that if you try to write with the cache turned on you are
                allowed to write to the cache ONLY if there is already an
                image of the same name in there. Otherwise you write to disk.
                Effectively, this means that WDISABLE controls the *creation*
                of new files in cache, not the writing-over of old ones.
              Scott Wilson 26-Feb-91 - Write in small chunks instead of one
                huge one. This should help out with NFS for large images.
              Scott Wilson 30-May-91 - Fix "-" and "#" as output files
*/

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "vinclude.h"
#include "vencode.h"

#define VWRITE_SIZE 65536		/* Write 64KB at a time */

writeimage(filename,imageptr)
struct xvimage *imageptr;
char *filename;
  {
    char buf[512];
    int imgsize,locsize,mapsize,imgcnt,loccnt,mapcnt,index,n,totalsize;
    char *temp1,*temp3;
    float *temp2;
    int file;
    int cfd;

    /* Check the image pointer */
    if (imageptr == NULL)
      {
        fprintf(stderr,"writeimage: NULL image pointer, no image written!\n");
        return(NULL);
      }

    /* Open the output file */
    if (!vfullpath(filename,NULL,buf)) return(NULL);

#ifdef CACHE
    if (khoros_cache != 0) khlock();
#endif

#ifdef CACHE
    if (khoros_cache == 0)
#else
    if(1)
#endif
      {
        if ( (file = open_output(buf)) < 0)
          {
            fprintf(stderr,"writeimage: unable to access %s for writing\n",
                           filename);
            return(NULL);
          }
        flock(file,LOCK_EX);
      }
#ifdef CACHE
    else
      {
        file = -1000; /* Fake like we have already been assigned a cache slot */
      }
#endif

    if (! imagesize(imageptr,&imgsize,&imgcnt,&mapsize,&mapcnt,&locsize,
	  &loccnt))
      {
        fprintf(stderr,"writeimage: Uninterpretable image specification\n");
        kwclose(file);
        return(0);
      }
    totalsize = sizeof(struct xvimage)+imgsize+mapsize+locsize;
    /* Sanity check */
    if (imgsize != 0 && imageptr->imagedata == NULL)
      {
        fprintf(stderr,"writeimage: Bogus image - no image data but nonzero image size!\n");
        fprintf(stderr,"writeimage: Image not written!\n");
        kwclose(file);
        return(0);
      }
    if (mapsize != 0 && imageptr->maps == NULL)
      {
        fprintf(stderr,"writeimage: Bogus image - no map data but nonzero map size!\n");
        fprintf(stderr,"writeimage: Image not written!\n");
        kwclose(file);
        return(0);
      }
    if (locsize != 0 && imageptr->location == NULL)
      {
        fprintf(stderr,"writeimage: Bogus image - no location data but nonzero location size!\n");
        fprintf(stderr,"writeimage: Image not written!\n");
        kwclose(file);
        return(0);
      }

    /* Write the image header */
    /* Note that the pointer fields in the header must be written out
       as NULLs in order that an image file on the disk can be
       compared using diff or cmp! */
    temp1 = imageptr->maps;      /* Save pointers to data areas */
    temp2 = imageptr->location;
    temp3 = imageptr->imagedata;
    imageptr->maps = NULL;       /* Set 'em to NULL */
    imageptr->location = NULL;
    imageptr->imagedata = NULL;
    if (file <= -1000)
      {
#ifdef CACHE
        /* Try to write into the image cache. This only occurs if the
           file descriptor has been buggered up to act like a cache
           slot id. */
        holepath(buf);  /* Get the REAL full path */
        cfd = -1;
        if (khoros_cache_wdisable != 0) cfd = find_image_in_cache(buf);
        if (khoros_cache_wdisable != 0)
          { 
            /* Must do this to insure that the temp directory entry
               is made properly. Additionally, the new image may be
               much larger than the old image with the same name and
               thusly won't fit in the cache. Calling allocate_cache_slot
               fixes these possibilities. */
            if (cfd != -1) cfd = allocate_cache_slot(totalsize,buf);
          }
        else
          cfd = allocate_cache_slot(totalsize,buf);
        if (cfd == -1)
          {
            /* Image is larger than the cache - cannot write to cache
               so write back to disk */
            khunlock(); /* Don't need to mess with the cache anymore! */
            file = open_output(buf);
            if (file == -1)
              {
                fprintf(stderr,"writeimage: Cannot access \"%s\" for writing\n",
                                filename);
                    exit(-5);
              }
            flock(file,LOCK_EX);
            if (vwrite(file,imageptr,sizeof(struct xvimage)) != sizeof(struct xvimage))
              {
                fprintf(stderr,"writeimage: Unable to write image header\n");
                kwclose(file);
                return(0);
              }

          }
        else
          {
            /* Write to image cache */
            file = cfd;
            if (cache_write(file,imageptr,sizeof(struct xvimage)) != sizeof(struct xvimage))
              {
                fprintf(stderr,"writeimage: Unable to cache_write image header\n");
                kwclose(file);
                return(0);
              }
          }
#else
        /* Something is totally bogus here, so give up */
        fprintf(stderr, "writeimage: Cache support not compiled!\n");
        exit(1);
#endif
      }
#ifdef CACHE
    else if (vwrite(file,imageptr,sizeof(struct xvimage)) != sizeof(struct xvimage))
#else
    if (vwrite(file,imageptr,sizeof(struct xvimage)) != sizeof(struct xvimage))
#endif
      {
        fprintf(stderr,"writeimage: Unable to write image header\n");
        kwclose(file);
        return(0);
      }
    imageptr->maps = temp1;      /* Restore the previous values */
    imageptr->location = temp2;
    imageptr->imagedata = temp3;

    /* Write maps */
    index = imageptr->data_encode_scheme;
#ifdef CACHE
    if (file <= -1000)
      {
        /* Write to the image cache  - no encoding is performed */
        n = cache_write(file,imageptr->maps,mapsize);
	if (n != mapsize)
          {
	    fprintf(stderr,"writeimage: Unable to cache_write maps\n");
            kwclose(file);
            return(0);
          }
      }
    else
#endif
      {
        /* Write to normal file descriptor */
        switch(imageptr->data_encode_scheme)
          {
            case VFF_DES_RAW:
              n = vwrite(file,imageptr->maps,mapsize);
	      if (n != mapsize)
                {
	          fprintf(stderr,"writeimage: Unable to write maps\n");
                  kwclose(file);
                  return(0);
                }
              break;
            case VFF_DES_COMPRESS:
            case VFF_DES_RLE:
            case VFF_DES_TRANSFORM:
            case VFF_DES_CCITT:
            case VFF_DES_ADPCM:
            case VFF_DES_GENERIC:
              n = write_compressed(file,imageptr->maps,mapsize,
                  compr_cmd[index]);
              if (n != mapsize)
                {
                  fprintf(stderr,"writeimage: Unable to write maps\n");
                  kwclose(file);
                  return(0);
                }
	      break;
            default:
              fprintf(stderr,
                  "writeimage: Unknown encoding scheme: %d maps not written!\n",
                  imageptr->data_encode_scheme);
                  kwclose(file);
              return(0);
              break;
          }
      }
                 
    /* Write the location data */
#ifdef CACHE
    if (file <= -1000)
      {
        /* Write to image cache */
        n = cache_write(file,imageptr->location,locsize);
        if (n != locsize)
          {
            fprintf(stderr,"writeimage: Unable to cache_write location data\n");
            kwclose(file);
            return(0);
          }
      }
    else
#endif
      {
        switch(imageptr->data_encode_scheme)
          {
            case VFF_DES_RAW: 
              n = vwrite(file,imageptr->location,locsize);
              if (n != locsize)
                {
                  fprintf(stderr,"writeimage: Unable to write location data\n");
                  kwclose(file);
                  return(0);
                }
              break;
            case VFF_DES_COMPRESS:
            case VFF_DES_RLE:
            case VFF_DES_TRANSFORM:
            case VFF_DES_CCITT:
            case VFF_DES_ADPCM:
            case VFF_DES_GENERIC:
              n = write_compressed(file,imageptr->location,locsize,
                                   compr_cmd[index]);
              if (n != locsize)
                {
                  fprintf(stderr,"writeimage: Unable to write location data\n");
                  kwclose(file);
                  return(0);
                }
	      break;
            default:
              fprintf(stderr,
             "writeimage: Unknown encoding scheme: %d locations not written!\n",
                      imageptr->data_encode_scheme);
                  kwclose(file);
              return(0);
              break;
          }
      }
                 
    /* write image data */
#ifdef CACHE
    if (file <= -1000)
      {
        /* Write to image cache */
        n = cache_write(file,imageptr->imagedata,imgsize);
        if (n != imgsize)
          {
            fprintf(stderr,"writeimage: Unable to cache_write image data\n");
            kwclose(file);
            return(0);
          }
      }
    else
#endif
      {
        switch(imageptr->data_encode_scheme)
          {
            case VFF_DES_RAW:
              n = vwrite(file,imageptr->imagedata,imgsize);
              if (n != imgsize)
                {
                  fprintf(stderr,"writeimage: Unable to write image data\n");
                  kwclose(file);
                  return(0);
                }
              break;
            case VFF_DES_COMPRESS:
            case VFF_DES_RLE:
            case VFF_DES_TRANSFORM:
            case VFF_DES_CCITT:
            case VFF_DES_ADPCM:
            case VFF_DES_GENERIC:
              n = write_compressed(file,imageptr->imagedata,imgsize,                                               compr_cmd[index]);
              if (n != imgsize)
                {
                  fprintf(stderr,"writeimage: Unable to write image data\n");
                  kwclose(file);
                  return(0);
                }
	      break;
            default:
              fprintf(stderr,"writeimage: Unknown encoding scheme: %d ",
                      "imagedata not written!\n",index);
                  kwclose(file);
               return(0);
               break;
          }
      }

    kwclose(file);
    return(1);
  }

kwclose(file)
int file;
  {

#ifdef CACHE
    /* If the file is being written to cache then reset the "file
       pointers" so that subsequent operations (namely reads) will
       begin at the right place */
    if (file <= -1000) 
      {
        xvcache_reset_ptrs(file);
        khunlock();
      }
    else
#endif
      {
        close(file);
        flock(file,LOCK_UN); 
      }
  }

static int
vwrite(fd,data,count) 
int fd,count;
char *data;
  {
    /* Write out data in chunks of VWRITE_SIZE to keep some sanity
       during I/O, especially with NFS or network transport. */

    int i,k,n,extra,written;
    char *ptr;

    n = count/VWRITE_SIZE;
    extra = count-(n*VWRITE_SIZE);

    written = 0;
    ptr = data;

    for (i=0; i<n; i++)
      {
        k = write(fd,ptr,VWRITE_SIZE);        
        if (k != VWRITE_SIZE)
          {
            fprintf(stderr, "vwrite: Unable to write full byte count!\n");
            return(written);
          }
        written += VWRITE_SIZE;
        ptr += VWRITE_SIZE;
      }
    if (extra != 0)
      {
        k = write(fd,ptr,extra);        
        if (k != extra)
          {
            fprintf(stderr, "vwrite: Unable to write full byte count!\n");
            return(written);
          }
        written += extra;
      }
    return(written);
  }

