/****************************************************************************
 *
 * Class:  RegistrarClient
 * Author: Mark Roseman
 *
 * The RegistrarClient provides the user's view of the Registrar.  This
 * is an abstract base class.  A subclass must provide a specific 
 * registration policy.  A registration policy must provide specific
 * semantics to both requests from the user and information from the 
 * Registrar.  The policy must convert application specific semantics
 * into the "open communications protocol" between the Registrar and
 * the Clients.  Subclasses must also provide a user interface, suitable
 * for the chosen registration semantics.
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 02/17/92 MR        initial version
 * 06/08/92 MR        added routine createReaderAndWriter(fd) to
 *                    provide a remote coordinator with the address of
 *                    a particular conference (via the local coordinator)
 * 08/14/92 MR        clean up
 * 09/20/92 MR        updated to use host/port to find registrar
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#ifndef __regclient_h
#define __regclient_h

#include <Dispatch/rpcpeer.h>
#include <gk/attrlist.h>



class RegistrarClient : public RpcPeer {
public:

  /*
  ** constructor 
  */
  RegistrarClient(const char* host, int port, class Coordinator*);


  /*
  ** routines to send messages to the registrar
  */
  void callJoinConference(int conf_id);
  void callJoinConference(AttributeList* al);
  void callLeaveConference(int conf_id, int user_id);
  void callNewConference(AttributeList*);
  void callDeleteConference(int conf_id);
  void PollConferences();
  void PollUsers(int conf_id);

  /*
  ** hook from the coordinator - subclass must define
  */
  virtual void userLeft(int conf_id, int user_id) = 0;

protected:

  /*
  ** hooks activated via info from registrar - subclasses must define
  */
  virtual void foundNewConference(AttributeList*) = 0;
  virtual void foundDeletedConference(int conf_id) = 0;
  virtual void foundNewUser(AttributeList*) = 0;
  virtual void foundDeletedUser(int conf_id, int user_id) = 0;


  virtual void UpdateConferenceList(char *info);
  virtual void UpdateUserList(char *info);
  virtual boolean createReaderAndWriter(const char* host, int port);
  virtual void createReaderAndWriter(int fd);
  virtual void startListening(void);
  int lPort(void);

  class CallbackRpcReader* reader_;
  class Writer* writer_;
  class Coordinator* coord_;
  int currentConference;
  AttrListTable* conference_tbl_;
  UserListTbl* users_tbl_;
};


#endif
