/**************************************************************************
 * 
 * Class:  RegClientDisplay implementation
 * Author: Mark Roseman
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 05/28/92 MR        initial version
 * 06/08/92 MR        added support for reading conference types from resources
 * 09/13/92 MR        changed to work with iv3.1b2 interface
 * 09/21/92 MR        took out the leave button for now
 *
 **************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <InterViews/action.h>
#include <gk-ui/DialogMgr.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <InterViews/background.h>
#include <InterViews/layout.h>
#include <IV-look/kit.h>
#include <InterViews/glue.h>
#include <InterViews/box.h>
#include <IV-look/button.h>
#include <stdio.h>
#include "openregclient.h"
#include "rcdisplay.h"
#include <InterViews/window.h>
#include <InterViews/scrbox.h>
#include <gk-ui/strbrowser.h>
#include <IV-look/choice.h>
#include <InterViews/telltale.h>
#include <IV-look/field.h>
#include <IV-look/dialogs.h>
#include <malloc.h>
#include <OS/string.h>
#include <gk-ui/labscrlst.h>

declareActionCallback(RegClientDisplay);
implementActionCallback(RegClientDisplay);


/**************************************************************************
 * 
 * Constructor.
 *
 **************************************************************************/

RegClientDisplay::RegClientDisplay(OpenRegClient* rc, WidgetKit* kit) : 
      MonoGlyph(nil), rc_(rc), kit_(kit) 
{
  body ( makeGlyph() );
}


/**************************************************************************
 * 
 * Create the glyph, which primarily contains two scrollable string
 * browsers (one for conferences, one for users), and buttons for new
 * conference, join conference, and leave conference.
 *
 **************************************************************************/

Glyph* RegClientDisplay::makeGlyph(void) {
  Style* style = kit_->style();

  LayoutKit& layout = *LayoutKit::instance();
  confs = new StringBrowser(kit_,
			    new ActionCallback(RegClientDisplay)(this, 
				 &RegClientDisplay::requestJoinConference),
			    new ActionCallback(RegClientDisplay)(this, 
                                 &RegClientDisplay::requestViewConference));
  users = new StringBrowser( kit_, nil, nil );

  Button* newbutton =   kit_->push_button("New",  
                            new ActionCallback(RegClientDisplay)(this, 
                                &RegClientDisplay::requestNewConference));
  Button* joinbutton =  kit_->push_button("Join", 
                            new ActionCallback(RegClientDisplay)(this, 
                                &RegClientDisplay::requestJoinConference));
  return kit_->inset_frame(
    layout.vbox (
      layout.hbox(
	layout.hglue(5.0),
	new LabelledScrollList(kit_, "Conferences", confs, 200, 70),
	layout.hglue(5.0),
	new LabelledScrollList(kit_, "Users", users, 200, 70),
        layout.hglue(5.0)
      ),
      layout.vglue(5.0),
      layout.hbox(
	layout.hglue(5.0),
	newbutton,
	layout.hglue(5.0),
	joinbutton,
	layout.hglue(5.0)
      ),
      layout.vglue(5.0)
   )
 );
 
}


/**************************************************************************
 * 
 * The user asked to create a new conference.  Pop up a dialog box asking
 * for the name.  Types are read in from XDefaults.
 *
 **************************************************************************/

declareFieldEditorCallback(RegClientDisplay)
implementFieldEditorCallback(RegClientDisplay)

void RegClientDisplay::requestNewConference(void) {
  long types;
  Button** buttons;
  
  LayoutKit& layout = *LayoutKit::instance();
  DialogKit& dlgkit = *DialogKit::instance();
  FieldEditor* fld_ = dlgkit.field_editor( 
    "Conference name", 
    kit_->style(),
    new FieldEditorCallback(RegClientDisplay)
         (this, &RegClientDisplay::accept1, &RegClientDisplay::cancel1));
  TelltaleGroup* tt_ = new TelltaleGroup;

  /*
  ** Top of dialog box -- get name of new conference 
  */

  PolyGlyph* g =
    layout.vbox(
      layout.vglue(10), 
      kit_->label("Conference name"), 
      layout.vglue(5), 
      fld_, 
      layout.vglue(20)
  );

  /*
  ** Read available types of conferences from XDefaults.  For each
  ** type available, append a radio button to the dialog box.
  */

  if (kit_->style()->find_attribute("conferenceTypes", types)) {
    buttons = (Button**)malloc( (int)types * sizeof (Button*));
    for ( int i = 1; i <= types; i++ ) {
      String desc;
      char att[50];
      sprintf(att, "conf%d-desc", i);
      if ( kit_->style()->find_attribute( att, desc )) {
	buttons[i-1] = kit_->radio_button( tt_, desc, (Action*)nil );
	g->append( buttons[i-1] );
      } else
	printf("couldn't find description resource %d\n", i);
    }
  }

  /*
  ** Put the OK and Cancel buttons at the bottom of the dialog box.
  */

  g->append( layout.vglue(10) );
  g->append( 
    layout.hbox( 
      layout.hglue(100),
      kit_->default_button("OK", new ActionCallback(RegClientDisplay)(this, 
                                                  &RegClientDisplay::accept)),
      layout.hglue(10),
      kit_->push_button("Cancel", new ActionCallback(RegClientDisplay)(this, 
                                                   &RegClientDisplay::cancel))
    )
  );
  g->append( layout.vglue(10) );

  /*
  ** Post the dialog, and if OK is selected, ask the registrar client
  ** to start up a new conference of the selected type.
  */

  dialog_ = new Dialog(kit_->inset_frame(layout.margin( g, 12 )), 
		       kit_->style());
  dialog_->append_input_handler( fld_);
  dialog_->next_focus();
  if (dialog_->post_for ( win_ ) ) {
    for(int i=0;i<types;i++)
      if( buttons[i]->state()->test(TelltaleState::is_chosen)) {
	String desc;
	char att[50];
	sprintf(att, "conf%d-desc", i+1);
	if( kit_->style()->find_attribute( att, desc))
	  rc_->requestNewConference( (char*)fld_->text()->string(), 
				     (char*)desc.string() );
      }
  }
  free((char*)buttons);
}


/**************************************************************************
 * 
 * Callbacks from the dialog for OK and Cancel (as well as hitting Enter
 * in the name field, which is equivalent to clicking OK).
 *
 **************************************************************************/

void RegClientDisplay::accept1(FieldEditor*) { dialog_->dismiss(true); }
void RegClientDisplay::cancel1(FieldEditor*) { dialog_->dismiss(false); }
void RegClientDisplay::accept() { dialog_->dismiss(true); }
void RegClientDisplay::cancel() { dialog_->dismiss(false); }



/**************************************************************************
 * 
 * Callbacks for join, leave, and view conferences, which all call the
 * appropriate methods in the registrar client.
 *
 **************************************************************************/

void RegClientDisplay::requestJoinConference(void) {
  rc_->requestJoinConference( conf_id[ confs->selected() ]);
}


void RegClientDisplay::requestViewConference(void) {
  rc_->requestViewConference( conf_id[ confs->selected() ]);
}


/**************************************************************************
 * 
 * Update the list of conferences.  We should add support later for 
 * maintaining selected conferences.
 *
 **************************************************************************/

void RegClientDisplay::updateConference(void) {
  char name[80], conf[80];
  confs->Clear();
  for (TableIterator(AttrLstTbl) i(*rc_->conference_tbl_);i.more();i.next()) {
    i.cur_value()->find_attribute( "name", name );
    confs->Append( name);
    i.cur_value()->find_attribute( "confnum", conf );
    conf_id[ confs->Count()-1] = atoi(conf);
  }
  confs->refresh();
}


/**************************************************************************
 * 
 * Update the list of users.
 *
 **************************************************************************/

void RegClientDisplay::updateUsers(int conf) {
  char s[80], name[80], userid[80], host[80];
  users->Clear();

  AttrListTable* usrs;
  if(rc_->users_tbl_->find( usrs, conf )) 
    for( TableIterator(AttrLstTbl) i(*usrs); i.more(); i.next()) {
      i.cur_value()->find_attribute("username", name );
      i.cur_value()->find_attribute("userid", userid);
      i.cur_value()->find_attribute("host", host);
      sprintf(s, "%s <%s@%s>", name, userid, host);
      users->Append( s );
    }
  users->refresh();
}




